from __future__ import annotations

import mimetypes
from enum import Enum
from enum import unique
from functools import cached_property
from pathlib import Path
from urllib.parse import quote
from urllib.parse import urljoin
from urllib.request import getproxies

import requests
from bs4 import BeautifulSoup
from faker import Faker
from pathvalidate import sanitize_filepath
from requests.adapters import HTTPAdapter
from tqdm import tqdm
from tqdm import trange
from urllib3.util.retry import Retry

mimetypes.add_type("image/webp", ".webp")
fake = Faker()
MULTPORN_HOME = "https://multporn.net/"


def initializeSession(proxies: dict = {}):
    """Initialize a requests.Session object for persistent cookies and proxies"""
    session = requests.Session()
    session.proxies.update(proxies or getproxies())
    session.headers.update({
        "User-Agent": fake.chrome(
            version_from=80,
            version_to=86,
            build_from=4100,
            build_to=4200,
        ),
    })
    retries = Retry(
        total=5,
        backoff_factor=1,
        status_forcelist=[413, 429, 500, 502, 503, 504],
    )
    retry_strategy = HTTPAdapter(max_retries=retries)
    session.mount("http://", retry_strategy)
    session.mount("https://", retry_strategy)
    session.hooks.update({
        "response": lambda r, *args, **
        kwargs: r.raise_for_status(),
    })
    return session


class Multporn:
    """
    A `multporn <https://multporn.net>`__ comic class

    Basic Usage:
        >>> from multporn import Multporn
        >>> comic = Multporn("https://multporn.net/comics/between_friends")
        >>> print(comic)
        'Between Friends'
    """

    def __init__(self, url: str, proxies: dict = {}, download: bool = False, requestsParams: dict = {}):
        """
        Start a request session and load soup from <https://multporn.net> for this link.
        url is the url of the comic, proxies is an optional dict in the form of dict(https="https://123.456.78.90", http="http://123.456.78.90")
        """
        self.session = initializeSession(proxies)
        self.url = urljoin(MULTPORN_HOME, url)
        self.requestsParams = requestsParams
        self.__response = self.session.get(self.url, **self.requestsParams)
        self.__soup = BeautifulSoup(self.__response.text, "html.parser")
        if (download):
            self.downloadContent()

    def __str__(self):
        """
        returns the name of the Multporn object
        """
        return self.name

    @cached_property
    def contentUrls(self) -> list[str]:
        """
        Return the content url of the Multporn object
        for picture albums and comics will return a list of images
        for videos will return an array with the video file link in the first index
        """
        if (self.contentType == "video"):
            return [self.__soup.find("video").source["src"]]
        else:
            return [image.find("img")["src"] for image in self.__soup.find_all("p", "jb-image")]

    @cached_property
    def tags(self) -> list[str]:
        """
        Returns a list of tags empty if non found
        """
        try:
            return [i.next.text for i in self.__soup.find(text="Tags: ").find_next().contents]
        except AttributeError:
            return []

    @cached_property
    def ongoing(self) -> bool:
        """
        Returns true if the Multporn object is ongoing
        Only use with comics and mangas as the concept is meaningless for videos
        """
        try:
            return "ongoing" in self.__soup.find(text="Section: ").find_next().text.lower()
        except AttributeError:
            return False

    @cached_property
    def name(self) -> str:
        """
        Returns the name of the Multporn object
        """
        return self.__soup.find("meta", attrs={"name": "dcterms.title"})["content"]

    @cached_property
    def sanitizedName(self) -> str:
        """
        Return the sanitized name of the Multporn object
        """
        return sanitize_filepath(self.name)

    @cached_property
    def thumbnail(self) -> str | None:
        """
        Returns the thumbnail of the Multporn object
        For picture album such as comics and mangas, it's the first page of the album
        For videos, a screenshot generated by the website itself
        """
        if (self.contentType == "video"):
            return self.__soup.find("video")["poster"]
        else:
            try:
                return self.contentUrls[0]
            except:
                return None

    @cached_property
    def pageCount(self) -> int:
        """
        Return the number of pages
        Always 1 for videos
        """
        return len(self.contentUrls)

    @cached_property
    def artists(self) -> list[str]:
        """
        Return a list of artists
        only present for comics
        most likely a single artist but multiple artists are possible
        """
        return [i.next.text for i in self.__soup.find(text="Author: ").find_next().contents]

    @cached_property
    def sections(self) -> list[str]:
        """
        Returns a list of sections that this comic is present in
        only present for comics
        Most likely a single section but multiple sections are possible
        """
        return [i.next.text for i in self.__soup.find(text="Section: ").find_next().contents]

    @cached_property
    def characters(self) -> list[str]:
        """
        Returns a list of characters listed in the comic
        Only present for comics
        May be empty even for comics
        """
        return [i.next.text for i in self.__soup.find(text="Characters: ").find_next().contents]

    @cached_property
    def exists(self) -> bool:
        """
        Returns the existence status of the Multporn object
        """
        try:
            return self.pageCount > 0
        except:
            return False

    @cached_property
    def contentType(self) -> str:
        """
        Returns the content type of the Multporn object as a string
        """
        return self.url.split("/")[3]

    def downloadContent(self, root: Path | str = Path("Albums"), printProgress: bool = True):
        """
        Downloads all comic pages that don't already exist in the directory
        Logging can be disabled by passing false to printProgress
        """
        paths = []
        root = Path(root)
        root = root.joinpath(sanitize_filepath(self.sanitizedName))
        root.mkdir(parents=True, exist_ok=True)
        if (self.contentType == "video"):
            url = self.contentUrls[0]
            fpath = root.joinpath(self.sanitizedName)
            printName = self.name
            r = self.session.get(
                url, timeout=5, stream=True, **self.requestsParams,
            )
            fpath = fpath.with_name(
                fpath.name+str(
                    mimetypes.guess_extension(
                        r.headers['content-type'],
                    ),
                ),
            )
            total_size_in_bytes = int(
                r.headers.get('content-length', 0),
            )
            with tqdm(total=total_size_in_bytes, disable=not printProgress, unit='iB', unit_scale=True, desc=self.name) as tq:
                with open(sanitize_filepath(fpath), 'wb') as file:
                    for data in r.iter_content(1024):
                        tq.update(len(data))
                        file.write(data)
                if total_size_in_bytes != 0 and tq.n != total_size_in_bytes:
                    sanitize_filepath(
                        fpath.with_name(
                            fpath.name + "_SKIPPED",
                        ),
                    ).touch(exist_ok=True)
                    tq.set_description(f'{printName} skipped')
                    paths.append(fpath)
                else:
                    paths.append(fpath)
        else:
            with trange(len(self.contentUrls), disable=not printProgress, desc=self.name) as tq:
                for i in tq:
                    fpath = root.joinpath(
                        f"{self.sanitizedName}_{str(i).zfill(len(str(self.pageCount-1)))}",
                    )
                    printName = f'"{self.name}" page {i+1}/{self.pageCount}'
                    globResult = list(root.glob(f"{fpath.name}*"))
                    if (globResult):
                        tq.set_description(f"{printName} exists")
                        paths.append(globResult[0])
                        continue
                    else:
                        try:
                            r = self.session.get(
                                self.contentUrls[i], timeout=5, **self.requestsParams,
                            )
                            fpath = fpath.with_name(
                                fpath.name+str(
                                    mimetypes.guess_extension(
                                        r.headers['content-type'],
                                    ),
                                ),
                            )
                            with open(sanitize_filepath(fpath), "wb") as f:
                                f.write(r.content)
                            tq.set_description(f'{printName} done')
                            paths.append(fpath)
                        except requests.exceptions.HTTPError as e:
                            if r.status_code == 404:
                                sanitize_filepath(
                                    fpath.with_name(
                                        fpath.name + "_SKIPPED",
                                    ),
                                ).touch(exist_ok=True)
                            tq.set_description(
                                f'{printName} skipped because {e}',
                            )
                            paths.append(fpath)
        return paths


class Webpage:

    """
    A Webpage class that bundles together everything related to <https://multporn.net>
    If you're confused what I mean by "webpage", this is an example (obviously NSFW): <https://multporn.net/category/cosplay>
    """

    def __init__(self, url, proxies: dict = {}, requestParams: dict = {}):
        """
        initializing the webpage object
        """
        self.session = initializeSession(proxies)
        self.url = urljoin(MULTPORN_HOME, url)
        self.__response = self.session.get(self.url, **requestParams)
        self.__soup = BeautifulSoup(self.__response.text, "html.parser")

    @cached_property
    def links(self) -> list[str]:
        """
        return all links found in this webpage
        """
        table = self.__soup.find(
            "table", "views-view-grid",
        ) or self.__soup.find("div", "view-content")
        return [urljoin(MULTPORN_HOME, i.a['href']) for i in table.find_all("strong") if i.a]

    @cached_property
    def name(self) -> str:
        """
        Return the name of this webpage
        usually is a category, character, author, etc
        """
        return self.__soup.find(
            "meta", attrs={"name": "dcterms.title"},
        )["content"]


@unique
class Sort(Enum):
    """
    Known search sort options. Defaults to `Relevant`.
    """
    Relevant = "search_api_relevance"
    Author = "Author"


@unique
class Types(Enum):
    """
    Known types of content
    """
    All = "All"
    Comics = "1"
    HentaiManga = "2"
    GayComics = "3"
    CartoonPictures = "4"
    HentaiPictures = "5"
    Games = "6"
    Flash = "7"
    CartoonVideos = "8"
    HentaiVideos = "9"
    GIFAnimations = "10"
    Rule63 = "11"
    AuthorsAlbums = "12"
    Humor = "13"


class Utils:
    """
    A class used to help with various multporn related tasks
    """

    @staticmethod
    def Search(query: str, page: int = 1, queryType: Types = Types.All, sort: Sort = Sort.Relevant, proxies: dict = {}, requestsParams: dict = {}):
        """
        Return a dict with 2 keys link, thumb and name
        searches on page `page` that match this search `query` sorted by `sort`
        filter by type with `queryType`
        """
        session = initializeSession(proxies)
        searchHome = urljoin(MULTPORN_HOME, "/search/")
        searchUrl = urljoin(
            searchHome, f"?views_fulltext={quote(query)}&type={queryType.value}&sort_by={sort.value}&page={page-1}",
        )
        Response = session.get(searchUrl, timeout=5, **requestsParams)
        soup = BeautifulSoup(Response.text, "html.parser")
        # links
        try:
            links = [
                urljoin(MULTPORN_HOME, i.a['href']) for i in soup.find(
                    "div", attrs={"class": "view-content"},
                ).find_all("strong")
            ]
        except AttributeError:
            return []
        # thumbs
        thumbs = soup.find(
            "div", attrs={"class": "view-content"},
        ).find_all("img")
        try:
            thumbs = [i['src'] for i in thumbs]
        except:
            thumbs = thumbs[1::2]
            thumbs = [i['src'] for i in thumbs]
        # names
        names = [
            i.string for i in soup.find(
                "div", attrs={"class": "view-content"},
            ).find_all("strong")
        ]
        r = []
        for i in range(len(links)):
            r.append({"link": links[i], "thumb": thumbs[i], "name": names[i]})
        return r
