"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClusterRole = exports.Role = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const k8s = require("./imports/k8s");
const rb = require("./role-binding");
const utils_1 = require("./utils");
class ImportedRole extends constructs_1.Construct {
    constructor(scope, id, name) {
        super(scope, id);
        this.resourceType = 'roles';
        this._name = name;
    }
    get name() {
        return this._name;
    }
    get apiVersion() {
        return k8s.KubeRole.GVK.apiVersion;
    }
    get apiGroup() {
        return 'rbac.authorization.k8s.io';
    }
    get kind() {
        return k8s.KubeRole.GVK.kind;
    }
    get resourceName() {
        return this.name;
    }
}
/**
 * Role is a namespaced, logical grouping of PolicyRules that can be referenced
 * as a unit by a RoleBinding.
 */
class Role extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'roles';
        this._rules = [];
        this.apiObject = new k8s.KubeRole(this, 'Resource', {
            metadata: props.metadata,
            rules: cdk8s_1.Lazy.any({ produce: () => this.synthesizeRules() }),
        });
    }
    /**
     * Imports a role from the cluster as a reference.
     */
    static fromRoleName(scope, id, name) {
        return new ImportedRole(scope, id, name);
    }
    /**
     * Rules associaated with this Role.
     * Returns a copy, use `allow` to add rules.
     */
    get rules() {
        return [...this._rules];
    }
    /**
     * Add permission to perform a list of HTTP verbs on a collection of
     * resources.
     *
     * @param resources The resource(s) to apply to
     * @see https://kubernetes.io/docs/reference/access-authn-authz/authorization/#determine-the-request-verb
     */
    allow(verbs, ...resources) {
        this._rules.push({ verbs, resources });
    }
    /**
     * Add "create" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowCreate(...resources) {
        this.allow(['create'], ...resources);
    }
    /**
     * Add "get" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowGet(...resources) {
        this.allow(['get'], ...resources);
    }
    /**
     * Add "list" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowList(...resources) {
        this.allow(['list'], ...resources);
    }
    /**
     * Add "watch" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowWatch(...resources) {
        this.allow(['watch'], ...resources);
    }
    /**
     * Add "update" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowUpdate(...resources) {
        this.allow(['update'], ...resources);
    }
    /**
     * Add "patch" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowPatch(...resources) {
        this.allow(['patch'], ...resources);
    }
    /**
     * Add "delete" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowDelete(...resources) {
        this.allow(['delete'], ...resources);
    }
    /**
     * Add "deletecollection" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowDeleteCollection(...resources) {
        this.allow(['deletecollection'], ...resources);
    }
    /**
     * Add "get", "list", and "watch" permissions for the resources.
     * @param resources The resource(s) to apply to
     */
    allowRead(...resources) {
        this.allow(['get', 'list', 'watch'], ...resources);
    }
    /**
     * Add "get", "list", "watch", "create", "update", "patch", "delete", and
     * "deletecollection" permissions for the resources.
     *
     * @param resources The resource(s) to apply to
     */
    allowReadWrite(...resources) {
        this.allow(['get', 'list', 'watch', 'create', 'update', 'patch', 'delete', 'deletecollection'], ...resources);
    }
    /**
     * Create a RoleBinding that binds the permissions in this Role
     * to a list of subjects, that will only apply this role's namespace.
     * @param subjects a list of subjects to bind to
     */
    bind(...subjects) {
        const subjectsAddress = utils_1.address(...subjects);
        const binding = new rb.RoleBinding(this, `RoleBinding${subjectsAddress}`, {
            metadata: {
                namespace: this.metadata.namespace,
            },
            role: this,
        });
        binding.addSubjects(...subjects);
        return binding;
    }
    synthesizeRules() {
        const rules = [];
        for (const rule of this._rules) {
            for (const resource of rule.resources) {
                rules.push({
                    verbs: rule.verbs,
                    apiGroups: [resource.apiGroup === 'core' ? '' : resource.apiGroup],
                    resourceNames: resource.resourceName ? [resource.resourceName] : undefined,
                    resources: resource.resourceType ? [resource.resourceType] : undefined,
                });
            }
        }
        return rules;
    }
}
exports.Role = Role;
_a = JSII_RTTI_SYMBOL_1;
Role[_a] = { fqn: "cdk8s-plus-23.Role", version: "2.3.26" };
class ImportedClusterRole extends constructs_1.Construct {
    constructor(scope, id, name) {
        super(scope, id);
        this.resourceType = 'clusterroles';
        this._name = name;
    }
    get name() {
        return this._name;
    }
    get apiVersion() {
        return k8s.KubeClusterRole.GVK.apiVersion;
    }
    get apiGroup() {
        return 'rbac.authorization.k8s.io';
    }
    get kind() {
        return k8s.KubeClusterRole.GVK.kind;
    }
    get resourceName() {
        return this.name;
    }
}
/**
 * ClusterRole is a cluster level, logical grouping of PolicyRules that can be
 * referenced as a unit by a RoleBinding or ClusterRoleBinding.
 */
class ClusterRole extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'clusterroles';
        this._labelSelector = {};
        this._rules = [];
        this.apiObject = new k8s.KubeClusterRole(this, 'Resource', {
            metadata: props.metadata,
            rules: cdk8s_1.Lazy.any({ produce: () => this.synthesizeRules() }),
            aggregationRule: cdk8s_1.Lazy.any({ produce: () => this.synthesizeAggregationRules() }),
        });
        for (const rule of props.rules ?? []) {
            this.allow(rule.verbs, ...rule.endpoints);
        }
        for (const [key, value] of Object.entries(props.aggregationLabels ?? {})) {
            this.aggregate(key, value);
        }
    }
    /**
     * Imports a role from the cluster as a reference.
     */
    static fromClusterRoleName(scope, id, name) {
        return new ImportedClusterRole(scope, id, name);
    }
    /**
     * Rules associaated with this Role.
     * Returns a copy, use `allow` to add rules.
     */
    get rules() {
        return [...this._rules];
    }
    /**
     * Add permission to perform a list of HTTP verbs on a collection of
     * resources.
     *
     * @param endpoints The endpoints(s) to apply to
     * @see https://kubernetes.io/docs/reference/access-authn-authz/authorization/#determine-the-request-verb
     */
    allow(verbs, ...endpoints) {
        this._rules.push({ verbs, endpoints });
    }
    /**
     * Add "create" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowCreate(...endpoints) {
        this.allow(['create'], ...endpoints);
    }
    /**
     * Add "get" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowGet(...endpoints) {
        this.allow(['get'], ...endpoints);
    }
    /**
     * Add "list" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowList(...endpoints) {
        this.allow(['list'], ...endpoints);
    }
    /**
     * Add "watch" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowWatch(...endpoints) {
        this.allow(['watch'], ...endpoints);
    }
    /**
     * Add "update" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowUpdate(...endpoints) {
        this.allow(['update'], ...endpoints);
    }
    /**
     * Add "patch" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowPatch(...endpoints) {
        this.allow(['patch'], ...endpoints);
    }
    /**
     * Add "delete" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowDelete(...endpoints) {
        this.allow(['delete'], ...endpoints);
    }
    /**
     * Add "deletecollection" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowDeleteCollection(...endpoints) {
        this.allow(['deletecollection'], ...endpoints);
    }
    /**
     * Add "get", "list", and "watch" permissions for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowRead(...endpoints) {
        this.allow(['get', 'list', 'watch'], ...endpoints);
    }
    /**
     * Add "get", "list", "watch", "create", "update", "patch", "delete", and
     * "deletecollection" permissions for the resources.
     *
     * @param endpoints The resource(s) to apply to
     */
    allowReadWrite(...endpoints) {
        this.allow(['get', 'list', 'watch', 'create', 'update', 'patch', 'delete', 'deletecollection'], ...endpoints);
    }
    /**
     * Aggregate rules from roles matching this label selector.
     */
    aggregate(key, value) {
        this._labelSelector[key] = value;
    }
    /**
     * Combines the rules of the argument ClusterRole into this ClusterRole
     * using aggregation labels.
     * @param rol
     */
    combine(rol) {
        const key = `cdk8s.cluster-role/aggregate-to-${cdk8s_1.Names.toLabelValue(this)}`;
        const value = 'true';
        rol.metadata.addLabel(key, value);
        this.aggregate(key, value);
    }
    /**
     * Create a RoleBinding that binds the permissions in this ClusterRole
     * to a list of subjects, that will only apply to the given namespace.
     * @param namespace the namespace to limit permissions to.
     * @param subjects a list of subjects to bind to
     */
    bindInNamespace(namespace, ...subjects) {
        const binding = new rb.RoleBinding(this, `RoleBinding-${namespace}`, {
            metadata: {
                namespace,
            },
            role: this,
        });
        binding.addSubjects(...subjects);
        return binding;
    }
    /**
     * Create a ClusterRoleBinding that binds the permissions in this
     * ClusterRole to a list of subjects, without namespace restrictions.
     * @param subjects a list of subjects to bind to
     */
    bind(...subjects) {
        const binding = new rb.ClusterRoleBinding(this, 'ClusterRoleBinding', {
            role: this,
        });
        binding.addSubjects(...subjects);
        return binding;
    }
    synthesizeRules() {
        const rules = [];
        for (const rule of this._rules) {
            for (const endpoint of rule.endpoints) {
                const resource = endpoint.asApiResource();
                const nonResource = endpoint.asNonApiResource();
                if (resource && nonResource) {
                    throw new Error('Endpoint must be either resource or non resource. not both.');
                }
                if (!resource && !nonResource) {
                    throw new Error('Endpoint must be either resource or non resource. not neither.');
                }
                if (resource) {
                    rules.push({
                        apiGroups: [resource.apiGroup === 'core' ? '' : resource.apiGroup],
                        resources: [resource.resourceType],
                        resourceNames: resource.resourceName ? [resource.resourceName] : [],
                        verbs: rule.verbs,
                    });
                }
                if (nonResource) {
                    rules.push({ verbs: rule.verbs, nonResourceUrLs: [nonResource] });
                }
            }
        }
        return rules;
    }
    synthesizeAggregationRules() {
        if (Object.keys(this._labelSelector).length === 0) {
            return undefined;
        }
        return { clusterRoleSelectors: [{ matchLabels: this._labelSelector }] };
    }
}
exports.ClusterRole = ClusterRole;
_b = JSII_RTTI_SYMBOL_1;
ClusterRole[_b] = { fqn: "cdk8s-plus-23.ClusterRole", version: "2.3.26" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicm9sZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9yb2xlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsaUNBQStDO0FBQy9DLDJDQUF1QztBQUV2QywrQkFBK0I7QUFDL0IscUNBQXFDO0FBQ3JDLHFDQUFxQztBQUNyQyxtQ0FBa0M7QUFzQ2xDLE1BQU0sWUFBYSxTQUFRLHNCQUFTO0lBTWxDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsSUFBWTtRQUNwRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBSEgsaUJBQVksR0FBRyxPQUFPLENBQUM7UUFJckMsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLENBQUM7SUFDcEIsQ0FBQztJQUVELElBQVcsSUFBSTtRQUNiLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQztJQUNwQixDQUFDO0lBRUQsSUFBVyxVQUFVO1FBQ25CLE9BQU8sR0FBRyxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDO0lBQ3JDLENBQUM7SUFFRCxJQUFXLFFBQVE7UUFDakIsT0FBTywyQkFBMkIsQ0FBQztJQUNyQyxDQUFDO0lBRUQsSUFBVyxJQUFJO1FBQ2IsT0FBTyxHQUFHLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUM7SUFDL0IsQ0FBQztJQUVELElBQVcsWUFBWTtRQUNyQixPQUFPLElBQUksQ0FBQyxJQUFJLENBQUM7SUFDbkIsQ0FBQztDQUVGO0FBRUQ7OztHQUdHO0FBQ0gsTUFBYSxJQUFLLFNBQVEsSUFBSSxDQUFDLFFBQVE7SUFrQnJDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBbUIsRUFBRTtRQUM3RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBTEgsaUJBQVksR0FBRyxPQUFPLENBQUM7UUFFdEIsV0FBTSxHQUEwQixFQUFFLENBQUM7UUFLbEQsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNsRCxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7WUFDeEIsS0FBSyxFQUFFLFlBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGVBQWUsRUFBRSxFQUFFLENBQUM7U0FDM0QsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQXZCRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxZQUFZLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsSUFBWTtRQUNuRSxPQUFPLElBQUksWUFBWSxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDM0MsQ0FBQztJQW9CRDs7O09BR0c7SUFDSCxJQUFXLEtBQUs7UUFDZCxPQUFPLENBQUMsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDMUIsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLEtBQUssQ0FBQyxLQUFlLEVBQUUsR0FBRyxTQUF5QjtRQUN4RCxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLEtBQUssRUFBRSxTQUFTLEVBQUUsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7O09BR0c7SUFDSSxXQUFXLENBQUMsR0FBRyxTQUF5QjtRQUM3QyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsUUFBUSxDQUFDLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksUUFBUSxDQUFDLEdBQUcsU0FBeUI7UUFDMUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLEtBQUssQ0FBQyxFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFNBQVMsQ0FBQyxHQUFHLFNBQXlCO1FBQzNDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxNQUFNLENBQUMsRUFBRSxHQUFHLFNBQVMsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7O09BR0c7SUFDSSxVQUFVLENBQUMsR0FBRyxTQUF5QjtRQUM1QyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksV0FBVyxDQUFDLEdBQUcsU0FBeUI7UUFDN0MsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLFFBQVEsQ0FBQyxFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFVBQVUsQ0FBQyxHQUFHLFNBQXlCO1FBQzVDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUMsRUFBRSxHQUFHLFNBQVMsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7O09BR0c7SUFDSSxXQUFXLENBQUMsR0FBRyxTQUF5QjtRQUM3QyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsUUFBUSxDQUFDLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7OztPQUdHO0lBQ0kscUJBQXFCLENBQUMsR0FBRyxTQUF5QjtRQUN2RCxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsRUFBRSxHQUFHLFNBQVMsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7O09BR0c7SUFDSSxTQUFTLENBQUMsR0FBRyxTQUF5QjtRQUMzQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxPQUFPLENBQUMsRUFBRSxHQUFHLFNBQVMsQ0FBQyxDQUFDO0lBQ3JELENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLGNBQWMsQ0FBQyxHQUFHLFNBQXlCO1FBQ2hELElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLE9BQU8sRUFBRSxRQUFRLEVBQUUsUUFBUSxFQUFFLE9BQU8sRUFBRSxRQUFRLEVBQUUsa0JBQWtCLENBQUMsRUFBRSxHQUFHLFNBQVMsQ0FBQyxDQUFDO0lBQ2hILENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksSUFBSSxDQUFDLEdBQUcsUUFBdUI7UUFDcEMsTUFBTSxlQUFlLEdBQUcsZUFBTyxDQUFDLEdBQUcsUUFBUSxDQUFDLENBQUM7UUFDN0MsTUFBTSxPQUFPLEdBQUcsSUFBSSxFQUFFLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSxjQUFjLGVBQWUsRUFBRSxFQUFFO1lBQ3hFLFFBQVEsRUFBRTtnQkFDUixTQUFTLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxTQUFTO2FBQ25DO1lBQ0QsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxPQUFPLENBQUMsV0FBVyxDQUFDLEdBQUcsUUFBUSxDQUFDLENBQUM7UUFDakMsT0FBTyxPQUFPLENBQUM7SUFDakIsQ0FBQztJQUVPLGVBQWU7UUFDckIsTUFBTSxLQUFLLEdBQXFCLEVBQUUsQ0FBQztRQUNuQyxLQUFLLE1BQU0sSUFBSSxJQUFJLElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDOUIsS0FBSyxNQUFNLFFBQVEsSUFBSSxJQUFJLENBQUMsU0FBUyxFQUFFO2dCQUNyQyxLQUFLLENBQUMsSUFBSSxDQUFDO29CQUNULEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSztvQkFDakIsU0FBUyxFQUFFLENBQUMsUUFBUSxDQUFDLFFBQVEsS0FBSyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQztvQkFDbEUsYUFBYSxFQUFFLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO29CQUMxRSxTQUFTLEVBQUUsUUFBUSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7aUJBQ3ZFLENBQUMsQ0FBQzthQUNKO1NBQ0Y7UUFDRCxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7O0FBOUpILG9CQStKQzs7O0FBNkNELE1BQU0sbUJBQW9CLFNBQVEsc0JBQVM7SUFNekMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxJQUFZO1FBQ3BELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFISCxpQkFBWSxHQUFXLGNBQWMsQ0FBQztRQUlwRCxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksQ0FBQztJQUNwQixDQUFDO0lBRUQsSUFBVyxJQUFJO1FBQ2IsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDO0lBQ3BCLENBQUM7SUFFRCxJQUFXLFVBQVU7UUFDbkIsT0FBTyxHQUFHLENBQUMsZUFBZSxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUM7SUFDNUMsQ0FBQztJQUVELElBQVcsUUFBUTtRQUNqQixPQUFPLDJCQUEyQixDQUFDO0lBQ3JDLENBQUM7SUFFRCxJQUFXLElBQUk7UUFDYixPQUFPLEdBQUcsQ0FBQyxlQUFlLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQztJQUN0QyxDQUFDO0lBRUQsSUFBVyxZQUFZO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQztJQUNuQixDQUFDO0NBRUY7QUFFRDs7O0dBR0c7QUFDSCxNQUFhLFdBQVksU0FBUSxJQUFJLENBQUMsUUFBUTtJQW1CNUMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUEwQixFQUFFO1FBQ3BFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFOSCxpQkFBWSxHQUFHLGNBQWMsQ0FBQztRQUU3QixtQkFBYyxHQUEyQixFQUFFLENBQUM7UUFDNUMsV0FBTSxHQUFpQyxFQUFFLENBQUM7UUFLekQsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUN6RCxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7WUFDeEIsS0FBSyxFQUFFLFlBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGVBQWUsRUFBRSxFQUFFLENBQUM7WUFDMUQsZUFBZSxFQUFFLFlBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLDBCQUEwQixFQUFFLEVBQUUsQ0FBQztTQUNoRixDQUFDLENBQUM7UUFFSCxLQUFLLE1BQU0sSUFBSSxJQUFJLEtBQUssQ0FBQyxLQUFLLElBQUksRUFBRSxFQUFFO1lBQ3BDLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztTQUMzQztRQUVELEtBQUssTUFBTSxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsSUFBSSxFQUFFLENBQUMsRUFBRTtZQUN4RSxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsQ0FBQztTQUM1QjtJQUNILENBQUM7SUFqQ0Q7O09BRUc7SUFDSSxNQUFNLENBQUMsbUJBQW1CLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsSUFBWTtRQUMxRSxPQUFPLElBQUksbUJBQW1CLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBOEJEOzs7T0FHRztJQUNILElBQVcsS0FBSztRQUNkLE9BQU8sQ0FBQyxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUMxQixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksS0FBSyxDQUFDLEtBQWUsRUFBRSxHQUFHLFNBQXlCO1FBQ3hELElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBRSxDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFdBQVcsQ0FBQyxHQUFHLFNBQXlCO1FBQzdDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLENBQUMsRUFBRSxHQUFHLFNBQVMsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7O09BR0c7SUFDSSxRQUFRLENBQUMsR0FBRyxTQUF5QjtRQUMxQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsS0FBSyxDQUFDLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksU0FBUyxDQUFDLEdBQUcsU0FBeUI7UUFDM0MsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLE1BQU0sQ0FBQyxFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFVBQVUsQ0FBQyxHQUFHLFNBQXlCO1FBQzVDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUMsRUFBRSxHQUFHLFNBQVMsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7O09BR0c7SUFDSSxXQUFXLENBQUMsR0FBRyxTQUF5QjtRQUM3QyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsUUFBUSxDQUFDLEVBQUUsR0FBRyxTQUFTLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksVUFBVSxDQUFDLEdBQUcsU0FBeUI7UUFDNUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLE9BQU8sQ0FBQyxFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFdBQVcsQ0FBQyxHQUFHLFNBQXlCO1FBQzdDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLENBQUMsRUFBRSxHQUFHLFNBQVMsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7O09BR0c7SUFDSSxxQkFBcUIsQ0FBQyxHQUFHLFNBQXlCO1FBQ3ZELElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFNBQVMsQ0FBQyxHQUFHLFNBQXlCO1FBQzNDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLE9BQU8sQ0FBQyxFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDckQsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksY0FBYyxDQUFDLEdBQUcsU0FBeUI7UUFDaEQsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUUsT0FBTyxFQUFFLFFBQVEsRUFBRSxRQUFRLEVBQUUsT0FBTyxFQUFFLFFBQVEsRUFBRSxrQkFBa0IsQ0FBQyxFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUM7SUFDaEgsQ0FBQztJQUVEOztPQUVHO0lBQ0ksU0FBUyxDQUFDLEdBQVcsRUFBRSxLQUFhO1FBQ3pDLElBQUksQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLEdBQUcsS0FBSyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksT0FBTyxDQUFDLEdBQWdCO1FBQzdCLE1BQU0sR0FBRyxHQUFHLG1DQUFtQyxhQUFLLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUM7UUFDMUUsTUFBTSxLQUFLLEdBQUcsTUFBTSxDQUFDO1FBQ3JCLEdBQUcsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUNsQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUM3QixDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxlQUFlLENBQUMsU0FBaUIsRUFBRSxHQUFHLFFBQXVCO1FBQ2xFLE1BQU0sT0FBTyxHQUFHLElBQUksRUFBRSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsZUFBZSxTQUFTLEVBQUUsRUFBRTtZQUNuRSxRQUFRLEVBQUU7Z0JBQ1IsU0FBUzthQUNWO1lBQ0QsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxPQUFPLENBQUMsV0FBVyxDQUFDLEdBQUcsUUFBUSxDQUFDLENBQUM7UUFDakMsT0FBTyxPQUFPLENBQUM7SUFDakIsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxJQUFJLENBQUMsR0FBRyxRQUF1QjtRQUNwQyxNQUFNLE9BQU8sR0FBRyxJQUFJLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsb0JBQW9CLEVBQUU7WUFDcEUsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxPQUFPLENBQUMsV0FBVyxDQUFDLEdBQUcsUUFBUSxDQUFDLENBQUM7UUFDakMsT0FBTyxPQUFPLENBQUM7SUFDakIsQ0FBQztJQUVPLGVBQWU7UUFDckIsTUFBTSxLQUFLLEdBQXFCLEVBQUUsQ0FBQztRQUNuQyxLQUFLLE1BQU0sSUFBSSxJQUFJLElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDOUIsS0FBSyxNQUFNLFFBQVEsSUFBSSxJQUFJLENBQUMsU0FBUyxFQUFFO2dCQUNyQyxNQUFNLFFBQVEsR0FBRyxRQUFRLENBQUMsYUFBYSxFQUFFLENBQUM7Z0JBQzFDLE1BQU0sV0FBVyxHQUFHLFFBQVEsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDO2dCQUVoRCxJQUFJLFFBQVEsSUFBSSxXQUFXLEVBQUU7b0JBQzNCLE1BQU0sSUFBSSxLQUFLLENBQUMsNkRBQTZELENBQUMsQ0FBQztpQkFDaEY7Z0JBRUQsSUFBSSxDQUFDLFFBQVEsSUFBSSxDQUFDLFdBQVcsRUFBRTtvQkFDN0IsTUFBTSxJQUFJLEtBQUssQ0FBQyxnRUFBZ0UsQ0FBQyxDQUFDO2lCQUNuRjtnQkFFRCxJQUFJLFFBQVEsRUFBRTtvQkFDWixLQUFLLENBQUMsSUFBSSxDQUFDO3dCQUNULFNBQVMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxRQUFRLEtBQUssTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUM7d0JBQ2xFLFNBQVMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUM7d0JBQ2xDLGFBQWEsRUFBRSxRQUFRLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTt3QkFDbkUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLO3FCQUNsQixDQUFDLENBQUM7aUJBQ0o7Z0JBQ0QsSUFBSSxXQUFXLEVBQUU7b0JBQ2YsS0FBSyxDQUFDLElBQUksQ0FBQyxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRSxDQUFDLFdBQVcsQ0FBQyxFQUFFLENBQUMsQ0FBQztpQkFDbkU7YUFDRjtTQUVGO1FBQ0QsT0FBTyxLQUFLLENBQUM7SUFDZixDQUFDO0lBRU8sMEJBQTBCO1FBQ2hDLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNqRCxPQUFPLFNBQVMsQ0FBQztTQUNsQjtRQUVELE9BQU8sRUFBRSxvQkFBb0IsRUFBRSxDQUFDLEVBQUUsV0FBVyxFQUFFLElBQUksQ0FBQyxjQUFjLEVBQUUsQ0FBQyxFQUFFLENBQUM7SUFDMUUsQ0FBQzs7QUFqT0gsa0NBa09DIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQXBpT2JqZWN0LCBMYXp5LCBOYW1lcyB9IGZyb20gJ2NkazhzJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgSUFwaVJlc291cmNlLCBJQXBpRW5kcG9pbnQgfSBmcm9tICcuL2FwaS1yZXNvdXJjZS5nZW5lcmF0ZWQnO1xuaW1wb3J0ICogYXMgYmFzZSBmcm9tICcuL2Jhc2UnO1xuaW1wb3J0ICogYXMgazhzIGZyb20gJy4vaW1wb3J0cy9rOHMnO1xuaW1wb3J0ICogYXMgcmIgZnJvbSAnLi9yb2xlLWJpbmRpbmcnO1xuaW1wb3J0IHsgYWRkcmVzcyB9IGZyb20gJy4vdXRpbHMnO1xuXG4vKipcbiAqIEEgcmVmZXJlbmNlIHRvIGFueSBSb2xlIG9yIENsdXN0ZXJSb2xlLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElSb2xlIGV4dGVuZHMgYmFzZS5JUmVzb3VyY2Uge1xuXG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYFJvbGVgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFJvbGVQcm9wcyBleHRlbmRzIGJhc2UuUmVzb3VyY2VQcm9wcyB7XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBydWxlcyB0aGUgcm9sZSBzaG91bGQgYWxsb3cuXG4gICAqXG4gICAqIEBkZWZhdWx0IFtdXG4gICAqL1xuICByZWFkb25seSBydWxlcz86IFJvbGVQb2xpY3lSdWxlW107XG59XG5cbi8qKlxuICogUG9saWN5IHJ1bGUgb2YgYSBgUm9sZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBSb2xlUG9saWN5UnVsZSB7XG5cbiAgLyoqXG4gICAqIFZlcmJzIHRvIGFsbG93LiAoZS5nIFsnZ2V0JywgJ3dhdGNoJ10pXG4gICAqL1xuICByZWFkb25seSB2ZXJiczogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFJlc291cmNlcyB0aGlzIHJ1bGUgYXBwbGllcyB0by5cbiAgICovXG4gIHJlYWRvbmx5IHJlc291cmNlczogSUFwaVJlc291cmNlW107XG59XG5cbmNsYXNzIEltcG9ydGVkUm9sZSBleHRlbmRzIENvbnN0cnVjdCBpbXBsZW1lbnRzIElSb2xlIHtcblxuICBwcml2YXRlIHJlYWRvbmx5IF9uYW1lOiBzdHJpbmc7XG5cbiAgcHVibGljIHJlYWRvbmx5IHJlc291cmNlVHlwZSA9ICdyb2xlcyc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgbmFtZTogc3RyaW5nKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICB0aGlzLl9uYW1lID0gbmFtZTtcbiAgfVxuXG4gIHB1YmxpYyBnZXQgbmFtZSgpOiBzdHJpbmcge1xuICAgIHJldHVybiB0aGlzLl9uYW1lO1xuICB9XG5cbiAgcHVibGljIGdldCBhcGlWZXJzaW9uKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIGs4cy5LdWJlUm9sZS5HVksuYXBpVmVyc2lvbjtcbiAgfVxuXG4gIHB1YmxpYyBnZXQgYXBpR3JvdXAoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gJ3JiYWMuYXV0aG9yaXphdGlvbi5rOHMuaW8nO1xuICB9XG5cbiAgcHVibGljIGdldCBraW5kKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIGs4cy5LdWJlUm9sZS5HVksua2luZDtcbiAgfVxuXG4gIHB1YmxpYyBnZXQgcmVzb3VyY2VOYW1lKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHRoaXMubmFtZTtcbiAgfVxuXG59XG5cbi8qKlxuICogUm9sZSBpcyBhIG5hbWVzcGFjZWQsIGxvZ2ljYWwgZ3JvdXBpbmcgb2YgUG9saWN5UnVsZXMgdGhhdCBjYW4gYmUgcmVmZXJlbmNlZFxuICogYXMgYSB1bml0IGJ5IGEgUm9sZUJpbmRpbmcuXG4gKi9cbmV4cG9ydCBjbGFzcyBSb2xlIGV4dGVuZHMgYmFzZS5SZXNvdXJjZSBpbXBsZW1lbnRzIElSb2xlIHtcblxuICAvKipcbiAgICogSW1wb3J0cyBhIHJvbGUgZnJvbSB0aGUgY2x1c3RlciBhcyBhIHJlZmVyZW5jZS5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVJvbGVOYW1lKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIG5hbWU6IHN0cmluZyk6IElSb2xlIHtcbiAgICByZXR1cm4gbmV3IEltcG9ydGVkUm9sZShzY29wZSwgaWQsIG5hbWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIEBzZWUgYmFzZS5SZXNvdXJjZS5hcGlPYmplY3RcbiAgICovXG4gIHByb3RlY3RlZCByZWFkb25seSBhcGlPYmplY3Q6IEFwaU9iamVjdDtcblxuICBwdWJsaWMgcmVhZG9ubHkgcmVzb3VyY2VUeXBlID0gJ3JvbGVzJztcblxuICBwcml2YXRlIHJlYWRvbmx5IF9ydWxlczogQXJyYXk8Um9sZVBvbGljeVJ1bGU+ID0gW107XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFJvbGVQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMuYXBpT2JqZWN0ID0gbmV3IGs4cy5LdWJlUm9sZSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBtZXRhZGF0YTogcHJvcHMubWV0YWRhdGEsXG4gICAgICBydWxlczogTGF6eS5hbnkoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLnN5bnRoZXNpemVSdWxlcygpIH0pLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFJ1bGVzIGFzc29jaWFhdGVkIHdpdGggdGhpcyBSb2xlLlxuICAgKiBSZXR1cm5zIGEgY29weSwgdXNlIGBhbGxvd2AgdG8gYWRkIHJ1bGVzLlxuICAgKi9cbiAgcHVibGljIGdldCBydWxlcygpOiBSb2xlUG9saWN5UnVsZVtdIHtcbiAgICByZXR1cm4gWy4uLnRoaXMuX3J1bGVzXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgcGVybWlzc2lvbiB0byBwZXJmb3JtIGEgbGlzdCBvZiBIVFRQIHZlcmJzIG9uIGEgY29sbGVjdGlvbiBvZlxuICAgKiByZXNvdXJjZXMuXG4gICAqXG4gICAqIEBwYXJhbSByZXNvdXJjZXMgVGhlIHJlc291cmNlKHMpIHRvIGFwcGx5IHRvXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvcmVmZXJlbmNlL2FjY2Vzcy1hdXRobi1hdXRoei9hdXRob3JpemF0aW9uLyNkZXRlcm1pbmUtdGhlLXJlcXVlc3QtdmVyYlxuICAgKi9cbiAgcHVibGljIGFsbG93KHZlcmJzOiBzdHJpbmdbXSwgLi4ucmVzb3VyY2VzOiBJQXBpUmVzb3VyY2VbXSk6IHZvaWQge1xuICAgIHRoaXMuX3J1bGVzLnB1c2goeyB2ZXJicywgcmVzb3VyY2VzIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBcImNyZWF0ZVwiIHBlcm1pc3Npb24gZm9yIHRoZSByZXNvdXJjZXMuXG4gICAqIEBwYXJhbSByZXNvdXJjZXMgVGhlIHJlc291cmNlKHMpIHRvIGFwcGx5IHRvXG4gICAqL1xuICBwdWJsaWMgYWxsb3dDcmVhdGUoLi4ucmVzb3VyY2VzOiBJQXBpUmVzb3VyY2VbXSk6IHZvaWQge1xuICAgIHRoaXMuYWxsb3coWydjcmVhdGUnXSwgLi4ucmVzb3VyY2VzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgXCJnZXRcIiBwZXJtaXNzaW9uIGZvciB0aGUgcmVzb3VyY2VzLlxuICAgKiBAcGFyYW0gcmVzb3VyY2VzIFRoZSByZXNvdXJjZShzKSB0byBhcHBseSB0b1xuICAgKi9cbiAgcHVibGljIGFsbG93R2V0KC4uLnJlc291cmNlczogSUFwaVJlc291cmNlW10pOiB2b2lkIHtcbiAgICB0aGlzLmFsbG93KFsnZ2V0J10sIC4uLnJlc291cmNlcyk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIFwibGlzdFwiIHBlcm1pc3Npb24gZm9yIHRoZSByZXNvdXJjZXMuXG4gICAqIEBwYXJhbSByZXNvdXJjZXMgVGhlIHJlc291cmNlKHMpIHRvIGFwcGx5IHRvXG4gICAqL1xuICBwdWJsaWMgYWxsb3dMaXN0KC4uLnJlc291cmNlczogSUFwaVJlc291cmNlW10pOiB2b2lkIHtcbiAgICB0aGlzLmFsbG93KFsnbGlzdCddLCAuLi5yZXNvdXJjZXMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBcIndhdGNoXCIgcGVybWlzc2lvbiBmb3IgdGhlIHJlc291cmNlcy5cbiAgICogQHBhcmFtIHJlc291cmNlcyBUaGUgcmVzb3VyY2UocykgdG8gYXBwbHkgdG9cbiAgICovXG4gIHB1YmxpYyBhbGxvd1dhdGNoKC4uLnJlc291cmNlczogSUFwaVJlc291cmNlW10pOiB2b2lkIHtcbiAgICB0aGlzLmFsbG93KFsnd2F0Y2gnXSwgLi4ucmVzb3VyY2VzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgXCJ1cGRhdGVcIiBwZXJtaXNzaW9uIGZvciB0aGUgcmVzb3VyY2VzLlxuICAgKiBAcGFyYW0gcmVzb3VyY2VzIFRoZSByZXNvdXJjZShzKSB0byBhcHBseSB0b1xuICAgKi9cbiAgcHVibGljIGFsbG93VXBkYXRlKC4uLnJlc291cmNlczogSUFwaVJlc291cmNlW10pOiB2b2lkIHtcbiAgICB0aGlzLmFsbG93KFsndXBkYXRlJ10sIC4uLnJlc291cmNlcyk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIFwicGF0Y2hcIiBwZXJtaXNzaW9uIGZvciB0aGUgcmVzb3VyY2VzLlxuICAgKiBAcGFyYW0gcmVzb3VyY2VzIFRoZSByZXNvdXJjZShzKSB0byBhcHBseSB0b1xuICAgKi9cbiAgcHVibGljIGFsbG93UGF0Y2goLi4ucmVzb3VyY2VzOiBJQXBpUmVzb3VyY2VbXSk6IHZvaWQge1xuICAgIHRoaXMuYWxsb3coWydwYXRjaCddLCAuLi5yZXNvdXJjZXMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBcImRlbGV0ZVwiIHBlcm1pc3Npb24gZm9yIHRoZSByZXNvdXJjZXMuXG4gICAqIEBwYXJhbSByZXNvdXJjZXMgVGhlIHJlc291cmNlKHMpIHRvIGFwcGx5IHRvXG4gICAqL1xuICBwdWJsaWMgYWxsb3dEZWxldGUoLi4ucmVzb3VyY2VzOiBJQXBpUmVzb3VyY2VbXSk6IHZvaWQge1xuICAgIHRoaXMuYWxsb3coWydkZWxldGUnXSwgLi4ucmVzb3VyY2VzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgXCJkZWxldGVjb2xsZWN0aW9uXCIgcGVybWlzc2lvbiBmb3IgdGhlIHJlc291cmNlcy5cbiAgICogQHBhcmFtIHJlc291cmNlcyBUaGUgcmVzb3VyY2UocykgdG8gYXBwbHkgdG9cbiAgICovXG4gIHB1YmxpYyBhbGxvd0RlbGV0ZUNvbGxlY3Rpb24oLi4ucmVzb3VyY2VzOiBJQXBpUmVzb3VyY2VbXSk6IHZvaWQge1xuICAgIHRoaXMuYWxsb3coWydkZWxldGVjb2xsZWN0aW9uJ10sIC4uLnJlc291cmNlcyk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIFwiZ2V0XCIsIFwibGlzdFwiLCBhbmQgXCJ3YXRjaFwiIHBlcm1pc3Npb25zIGZvciB0aGUgcmVzb3VyY2VzLlxuICAgKiBAcGFyYW0gcmVzb3VyY2VzIFRoZSByZXNvdXJjZShzKSB0byBhcHBseSB0b1xuICAgKi9cbiAgcHVibGljIGFsbG93UmVhZCguLi5yZXNvdXJjZXM6IElBcGlSZXNvdXJjZVtdKTogdm9pZCB7XG4gICAgdGhpcy5hbGxvdyhbJ2dldCcsICdsaXN0JywgJ3dhdGNoJ10sIC4uLnJlc291cmNlcyk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIFwiZ2V0XCIsIFwibGlzdFwiLCBcIndhdGNoXCIsIFwiY3JlYXRlXCIsIFwidXBkYXRlXCIsIFwicGF0Y2hcIiwgXCJkZWxldGVcIiwgYW5kXG4gICAqIFwiZGVsZXRlY29sbGVjdGlvblwiIHBlcm1pc3Npb25zIGZvciB0aGUgcmVzb3VyY2VzLlxuICAgKlxuICAgKiBAcGFyYW0gcmVzb3VyY2VzIFRoZSByZXNvdXJjZShzKSB0byBhcHBseSB0b1xuICAgKi9cbiAgcHVibGljIGFsbG93UmVhZFdyaXRlKC4uLnJlc291cmNlczogSUFwaVJlc291cmNlW10pOiB2b2lkIHtcbiAgICB0aGlzLmFsbG93KFsnZ2V0JywgJ2xpc3QnLCAnd2F0Y2gnLCAnY3JlYXRlJywgJ3VwZGF0ZScsICdwYXRjaCcsICdkZWxldGUnLCAnZGVsZXRlY29sbGVjdGlvbiddLCAuLi5yZXNvdXJjZXMpO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIFJvbGVCaW5kaW5nIHRoYXQgYmluZHMgdGhlIHBlcm1pc3Npb25zIGluIHRoaXMgUm9sZVxuICAgKiB0byBhIGxpc3Qgb2Ygc3ViamVjdHMsIHRoYXQgd2lsbCBvbmx5IGFwcGx5IHRoaXMgcm9sZSdzIG5hbWVzcGFjZS5cbiAgICogQHBhcmFtIHN1YmplY3RzIGEgbGlzdCBvZiBzdWJqZWN0cyB0byBiaW5kIHRvXG4gICAqL1xuICBwdWJsaWMgYmluZCguLi5zdWJqZWN0czogcmIuSVN1YmplY3RbXSk6IHJiLlJvbGVCaW5kaW5nIHtcbiAgICBjb25zdCBzdWJqZWN0c0FkZHJlc3MgPSBhZGRyZXNzKC4uLnN1YmplY3RzKTtcbiAgICBjb25zdCBiaW5kaW5nID0gbmV3IHJiLlJvbGVCaW5kaW5nKHRoaXMsIGBSb2xlQmluZGluZyR7c3ViamVjdHNBZGRyZXNzfWAsIHtcbiAgICAgIG1ldGFkYXRhOiB7XG4gICAgICAgIG5hbWVzcGFjZTogdGhpcy5tZXRhZGF0YS5uYW1lc3BhY2UsXG4gICAgICB9LFxuICAgICAgcm9sZTogdGhpcyxcbiAgICB9KTtcbiAgICBiaW5kaW5nLmFkZFN1YmplY3RzKC4uLnN1YmplY3RzKTtcbiAgICByZXR1cm4gYmluZGluZztcbiAgfVxuXG4gIHByaXZhdGUgc3ludGhlc2l6ZVJ1bGVzKCk6IGs4cy5Qb2xpY3lSdWxlW10ge1xuICAgIGNvbnN0IHJ1bGVzOiBrOHMuUG9saWN5UnVsZVtdID0gW107XG4gICAgZm9yIChjb25zdCBydWxlIG9mIHRoaXMuX3J1bGVzKSB7XG4gICAgICBmb3IgKGNvbnN0IHJlc291cmNlIG9mIHJ1bGUucmVzb3VyY2VzKSB7XG4gICAgICAgIHJ1bGVzLnB1c2goe1xuICAgICAgICAgIHZlcmJzOiBydWxlLnZlcmJzLFxuICAgICAgICAgIGFwaUdyb3VwczogW3Jlc291cmNlLmFwaUdyb3VwID09PSAnY29yZScgPyAnJyA6IHJlc291cmNlLmFwaUdyb3VwXSxcbiAgICAgICAgICByZXNvdXJjZU5hbWVzOiByZXNvdXJjZS5yZXNvdXJjZU5hbWUgPyBbcmVzb3VyY2UucmVzb3VyY2VOYW1lXSA6IHVuZGVmaW5lZCxcbiAgICAgICAgICByZXNvdXJjZXM6IHJlc291cmNlLnJlc291cmNlVHlwZSA/IFtyZXNvdXJjZS5yZXNvdXJjZVR5cGVdIDogdW5kZWZpbmVkLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHJ1bGVzO1xuICB9XG59XG5cbi8qKlxuICogUmVwcmVzZW50cyBhIGNsdXN0ZXItbGV2ZWwgcm9sZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJQ2x1c3RlclJvbGUgZXh0ZW5kcyBiYXNlLklSZXNvdXJjZSB7XG5cbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBgQ2x1c3RlclJvbGVgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENsdXN0ZXJSb2xlUHJvcHMgZXh0ZW5kcyBiYXNlLlJlc291cmNlUHJvcHMge1xuXG4gIC8qKlxuICAgICogQSBsaXN0IG9mIHJ1bGVzIHRoZSByb2xlIHNob3VsZCBhbGxvdy5cbiAgICAqXG4gICAgKiBAZGVmYXVsdCBbXVxuICAgICovXG4gIHJlYWRvbmx5IHJ1bGVzPzogQ2x1c3RlclJvbGVQb2xpY3lSdWxlW107XG5cbiAgLyoqXG4gICAgKiBTcGVjaWZ5IGxhYmVscyB0aGF0IHNob3VsZCBiZSB1c2VkIHRvIGxvY2F0ZSBDbHVzdGVyUm9sZXMsIHdob3NlIHJ1bGVzXG4gICAgKiB3aWxsIGJlIGF1dG9tYXRpY2FsbHkgZmlsbGVkIGludG8gdGhpcyBDbHVzdGVyUm9sZSdzIHJ1bGVzLlxuICAgICovXG4gIHJlYWRvbmx5IGFnZ3JlZ2F0aW9uTGFiZWxzPzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcbn1cblxuLyoqXG4gKiBQb2xpY3kgcnVsZSBvZiBhIGBDbHVzdGVyUm9sZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDbHVzdGVyUm9sZVBvbGljeVJ1bGUge1xuXG4gIC8qKlxuICAgKiBWZXJicyB0byBhbGxvdy4gKGUuZyBbJ2dldCcsICd3YXRjaCddKVxuICAgKi9cbiAgcmVhZG9ubHkgdmVyYnM6IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBFbmRwb2ludHMgdGhpcyBydWxlIGFwcGxpZXMgdG8uIENhbiBiZSBlaXRoZXIgYXBpIHJlc291cmNlc1xuICAgKiBvciBub24gYXBpIHJlc291cmNlcy5cbiAgICovXG4gIHJlYWRvbmx5IGVuZHBvaW50czogSUFwaUVuZHBvaW50W107XG59XG5cbmNsYXNzIEltcG9ydGVkQ2x1c3RlclJvbGUgZXh0ZW5kcyBDb25zdHJ1Y3QgaW1wbGVtZW50cyBJQ2x1c3RlclJvbGUge1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgX25hbWU6IHN0cmluZztcblxuICBwdWJsaWMgcmVhZG9ubHkgcmVzb3VyY2VUeXBlOiBzdHJpbmcgPSAnY2x1c3RlcnJvbGVzJztcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBuYW1lOiBzdHJpbmcpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuICAgIHRoaXMuX25hbWUgPSBuYW1lO1xuICB9XG5cbiAgcHVibGljIGdldCBuYW1lKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHRoaXMuX25hbWU7XG4gIH1cblxuICBwdWJsaWMgZ2V0IGFwaVZlcnNpb24oKTogc3RyaW5nIHtcbiAgICByZXR1cm4gazhzLkt1YmVDbHVzdGVyUm9sZS5HVksuYXBpVmVyc2lvbjtcbiAgfVxuXG4gIHB1YmxpYyBnZXQgYXBpR3JvdXAoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gJ3JiYWMuYXV0aG9yaXphdGlvbi5rOHMuaW8nO1xuICB9XG5cbiAgcHVibGljIGdldCBraW5kKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIGs4cy5LdWJlQ2x1c3RlclJvbGUuR1ZLLmtpbmQ7XG4gIH1cblxuICBwdWJsaWMgZ2V0IHJlc291cmNlTmFtZSgpOiBzdHJpbmcge1xuICAgIHJldHVybiB0aGlzLm5hbWU7XG4gIH1cblxufVxuXG4vKipcbiAqIENsdXN0ZXJSb2xlIGlzIGEgY2x1c3RlciBsZXZlbCwgbG9naWNhbCBncm91cGluZyBvZiBQb2xpY3lSdWxlcyB0aGF0IGNhbiBiZVxuICogcmVmZXJlbmNlZCBhcyBhIHVuaXQgYnkgYSBSb2xlQmluZGluZyBvciBDbHVzdGVyUm9sZUJpbmRpbmcuXG4gKi9cbmV4cG9ydCBjbGFzcyBDbHVzdGVyUm9sZSBleHRlbmRzIGJhc2UuUmVzb3VyY2UgaW1wbGVtZW50cyBJQ2x1c3RlclJvbGUsIElSb2xlIHtcblxuICAvKipcbiAgICogSW1wb3J0cyBhIHJvbGUgZnJvbSB0aGUgY2x1c3RlciBhcyBhIHJlZmVyZW5jZS5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUNsdXN0ZXJSb2xlTmFtZShzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBuYW1lOiBzdHJpbmcpOiBJQ2x1c3RlclJvbGUge1xuICAgIHJldHVybiBuZXcgSW1wb3J0ZWRDbHVzdGVyUm9sZShzY29wZSwgaWQsIG5hbWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIEBzZWUgYmFzZS5SZXNvdXJjZS5hcGlPYmplY3RcbiAgICovXG4gIHByb3RlY3RlZCByZWFkb25seSBhcGlPYmplY3Q6IEFwaU9iamVjdDtcblxuICBwdWJsaWMgcmVhZG9ubHkgcmVzb3VyY2VUeXBlID0gJ2NsdXN0ZXJyb2xlcyc7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBfbGFiZWxTZWxlY3RvcjogUmVjb3JkPHN0cmluZywgc3RyaW5nPiA9IHt9O1xuICBwcml2YXRlIHJlYWRvbmx5IF9ydWxlczogQXJyYXk8Q2x1c3RlclJvbGVQb2xpY3lSdWxlPiA9IFtdO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDbHVzdGVyUm9sZVByb3BzID0ge30pIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5hcGlPYmplY3QgPSBuZXcgazhzLkt1YmVDbHVzdGVyUm9sZSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBtZXRhZGF0YTogcHJvcHMubWV0YWRhdGEsXG4gICAgICBydWxlczogTGF6eS5hbnkoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLnN5bnRoZXNpemVSdWxlcygpIH0pLFxuICAgICAgYWdncmVnYXRpb25SdWxlOiBMYXp5LmFueSh7IHByb2R1Y2U6ICgpID0+IHRoaXMuc3ludGhlc2l6ZUFnZ3JlZ2F0aW9uUnVsZXMoKSB9KSxcbiAgICB9KTtcblxuICAgIGZvciAoY29uc3QgcnVsZSBvZiBwcm9wcy5ydWxlcyA/PyBbXSkge1xuICAgICAgdGhpcy5hbGxvdyhydWxlLnZlcmJzLCAuLi5ydWxlLmVuZHBvaW50cyk7XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBba2V5LCB2YWx1ZV0gb2YgT2JqZWN0LmVudHJpZXMocHJvcHMuYWdncmVnYXRpb25MYWJlbHMgPz8ge30pKSB7XG4gICAgICB0aGlzLmFnZ3JlZ2F0ZShrZXksIHZhbHVlKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogUnVsZXMgYXNzb2NpYWF0ZWQgd2l0aCB0aGlzIFJvbGUuXG4gICAqIFJldHVybnMgYSBjb3B5LCB1c2UgYGFsbG93YCB0byBhZGQgcnVsZXMuXG4gICAqL1xuICBwdWJsaWMgZ2V0IHJ1bGVzKCk6IENsdXN0ZXJSb2xlUG9saWN5UnVsZVtdIHtcbiAgICByZXR1cm4gWy4uLnRoaXMuX3J1bGVzXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgcGVybWlzc2lvbiB0byBwZXJmb3JtIGEgbGlzdCBvZiBIVFRQIHZlcmJzIG9uIGEgY29sbGVjdGlvbiBvZlxuICAgKiByZXNvdXJjZXMuXG4gICAqXG4gICAqIEBwYXJhbSBlbmRwb2ludHMgVGhlIGVuZHBvaW50cyhzKSB0byBhcHBseSB0b1xuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL3JlZmVyZW5jZS9hY2Nlc3MtYXV0aG4tYXV0aHovYXV0aG9yaXphdGlvbi8jZGV0ZXJtaW5lLXRoZS1yZXF1ZXN0LXZlcmJcbiAgICovXG4gIHB1YmxpYyBhbGxvdyh2ZXJiczogc3RyaW5nW10sIC4uLmVuZHBvaW50czogSUFwaUVuZHBvaW50W10pOiB2b2lkIHtcbiAgICB0aGlzLl9ydWxlcy5wdXNoKHsgdmVyYnMsIGVuZHBvaW50cyB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgXCJjcmVhdGVcIiBwZXJtaXNzaW9uIGZvciB0aGUgcmVzb3VyY2VzLlxuICAgKiBAcGFyYW0gZW5kcG9pbnRzIFRoZSByZXNvdXJjZShzKSB0byBhcHBseSB0b1xuICAgKi9cbiAgcHVibGljIGFsbG93Q3JlYXRlKC4uLmVuZHBvaW50czogSUFwaUVuZHBvaW50W10pOiB2b2lkIHtcbiAgICB0aGlzLmFsbG93KFsnY3JlYXRlJ10sIC4uLmVuZHBvaW50cyk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIFwiZ2V0XCIgcGVybWlzc2lvbiBmb3IgdGhlIHJlc291cmNlcy5cbiAgICogQHBhcmFtIGVuZHBvaW50cyBUaGUgcmVzb3VyY2UocykgdG8gYXBwbHkgdG9cbiAgICovXG4gIHB1YmxpYyBhbGxvd0dldCguLi5lbmRwb2ludHM6IElBcGlFbmRwb2ludFtdKTogdm9pZCB7XG4gICAgdGhpcy5hbGxvdyhbJ2dldCddLCAuLi5lbmRwb2ludHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBcImxpc3RcIiBwZXJtaXNzaW9uIGZvciB0aGUgcmVzb3VyY2VzLlxuICAgKiBAcGFyYW0gZW5kcG9pbnRzIFRoZSByZXNvdXJjZShzKSB0byBhcHBseSB0b1xuICAgKi9cbiAgcHVibGljIGFsbG93TGlzdCguLi5lbmRwb2ludHM6IElBcGlFbmRwb2ludFtdKTogdm9pZCB7XG4gICAgdGhpcy5hbGxvdyhbJ2xpc3QnXSwgLi4uZW5kcG9pbnRzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgXCJ3YXRjaFwiIHBlcm1pc3Npb24gZm9yIHRoZSByZXNvdXJjZXMuXG4gICAqIEBwYXJhbSBlbmRwb2ludHMgVGhlIHJlc291cmNlKHMpIHRvIGFwcGx5IHRvXG4gICAqL1xuICBwdWJsaWMgYWxsb3dXYXRjaCguLi5lbmRwb2ludHM6IElBcGlFbmRwb2ludFtdKTogdm9pZCB7XG4gICAgdGhpcy5hbGxvdyhbJ3dhdGNoJ10sIC4uLmVuZHBvaW50cyk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIFwidXBkYXRlXCIgcGVybWlzc2lvbiBmb3IgdGhlIHJlc291cmNlcy5cbiAgICogQHBhcmFtIGVuZHBvaW50cyBUaGUgcmVzb3VyY2UocykgdG8gYXBwbHkgdG9cbiAgICovXG4gIHB1YmxpYyBhbGxvd1VwZGF0ZSguLi5lbmRwb2ludHM6IElBcGlFbmRwb2ludFtdKTogdm9pZCB7XG4gICAgdGhpcy5hbGxvdyhbJ3VwZGF0ZSddLCAuLi5lbmRwb2ludHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBcInBhdGNoXCIgcGVybWlzc2lvbiBmb3IgdGhlIHJlc291cmNlcy5cbiAgICogQHBhcmFtIGVuZHBvaW50cyBUaGUgcmVzb3VyY2UocykgdG8gYXBwbHkgdG9cbiAgICovXG4gIHB1YmxpYyBhbGxvd1BhdGNoKC4uLmVuZHBvaW50czogSUFwaUVuZHBvaW50W10pOiB2b2lkIHtcbiAgICB0aGlzLmFsbG93KFsncGF0Y2gnXSwgLi4uZW5kcG9pbnRzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgXCJkZWxldGVcIiBwZXJtaXNzaW9uIGZvciB0aGUgcmVzb3VyY2VzLlxuICAgKiBAcGFyYW0gZW5kcG9pbnRzIFRoZSByZXNvdXJjZShzKSB0byBhcHBseSB0b1xuICAgKi9cbiAgcHVibGljIGFsbG93RGVsZXRlKC4uLmVuZHBvaW50czogSUFwaUVuZHBvaW50W10pOiB2b2lkIHtcbiAgICB0aGlzLmFsbG93KFsnZGVsZXRlJ10sIC4uLmVuZHBvaW50cyk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIFwiZGVsZXRlY29sbGVjdGlvblwiIHBlcm1pc3Npb24gZm9yIHRoZSByZXNvdXJjZXMuXG4gICAqIEBwYXJhbSBlbmRwb2ludHMgVGhlIHJlc291cmNlKHMpIHRvIGFwcGx5IHRvXG4gICAqL1xuICBwdWJsaWMgYWxsb3dEZWxldGVDb2xsZWN0aW9uKC4uLmVuZHBvaW50czogSUFwaUVuZHBvaW50W10pOiB2b2lkIHtcbiAgICB0aGlzLmFsbG93KFsnZGVsZXRlY29sbGVjdGlvbiddLCAuLi5lbmRwb2ludHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBcImdldFwiLCBcImxpc3RcIiwgYW5kIFwid2F0Y2hcIiBwZXJtaXNzaW9ucyBmb3IgdGhlIHJlc291cmNlcy5cbiAgICogQHBhcmFtIGVuZHBvaW50cyBUaGUgcmVzb3VyY2UocykgdG8gYXBwbHkgdG9cbiAgICovXG4gIHB1YmxpYyBhbGxvd1JlYWQoLi4uZW5kcG9pbnRzOiBJQXBpRW5kcG9pbnRbXSk6IHZvaWQge1xuICAgIHRoaXMuYWxsb3coWydnZXQnLCAnbGlzdCcsICd3YXRjaCddLCAuLi5lbmRwb2ludHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBcImdldFwiLCBcImxpc3RcIiwgXCJ3YXRjaFwiLCBcImNyZWF0ZVwiLCBcInVwZGF0ZVwiLCBcInBhdGNoXCIsIFwiZGVsZXRlXCIsIGFuZFxuICAgKiBcImRlbGV0ZWNvbGxlY3Rpb25cIiBwZXJtaXNzaW9ucyBmb3IgdGhlIHJlc291cmNlcy5cbiAgICpcbiAgICogQHBhcmFtIGVuZHBvaW50cyBUaGUgcmVzb3VyY2UocykgdG8gYXBwbHkgdG9cbiAgICovXG4gIHB1YmxpYyBhbGxvd1JlYWRXcml0ZSguLi5lbmRwb2ludHM6IElBcGlFbmRwb2ludFtdKTogdm9pZCB7XG4gICAgdGhpcy5hbGxvdyhbJ2dldCcsICdsaXN0JywgJ3dhdGNoJywgJ2NyZWF0ZScsICd1cGRhdGUnLCAncGF0Y2gnLCAnZGVsZXRlJywgJ2RlbGV0ZWNvbGxlY3Rpb24nXSwgLi4uZW5kcG9pbnRzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZ2dyZWdhdGUgcnVsZXMgZnJvbSByb2xlcyBtYXRjaGluZyB0aGlzIGxhYmVsIHNlbGVjdG9yLlxuICAgKi9cbiAgcHVibGljIGFnZ3JlZ2F0ZShrZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyk6IHZvaWQge1xuICAgIHRoaXMuX2xhYmVsU2VsZWN0b3Jba2V5XSA9IHZhbHVlO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbWJpbmVzIHRoZSBydWxlcyBvZiB0aGUgYXJndW1lbnQgQ2x1c3RlclJvbGUgaW50byB0aGlzIENsdXN0ZXJSb2xlXG4gICAqIHVzaW5nIGFnZ3JlZ2F0aW9uIGxhYmVscy5cbiAgICogQHBhcmFtIHJvbFxuICAgKi9cbiAgcHVibGljIGNvbWJpbmUocm9sOiBDbHVzdGVyUm9sZSk6IHZvaWQge1xuICAgIGNvbnN0IGtleSA9IGBjZGs4cy5jbHVzdGVyLXJvbGUvYWdncmVnYXRlLXRvLSR7TmFtZXMudG9MYWJlbFZhbHVlKHRoaXMpfWA7XG4gICAgY29uc3QgdmFsdWUgPSAndHJ1ZSc7XG4gICAgcm9sLm1ldGFkYXRhLmFkZExhYmVsKGtleSwgdmFsdWUpO1xuICAgIHRoaXMuYWdncmVnYXRlKGtleSwgdmFsdWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIFJvbGVCaW5kaW5nIHRoYXQgYmluZHMgdGhlIHBlcm1pc3Npb25zIGluIHRoaXMgQ2x1c3RlclJvbGVcbiAgICogdG8gYSBsaXN0IG9mIHN1YmplY3RzLCB0aGF0IHdpbGwgb25seSBhcHBseSB0byB0aGUgZ2l2ZW4gbmFtZXNwYWNlLlxuICAgKiBAcGFyYW0gbmFtZXNwYWNlIHRoZSBuYW1lc3BhY2UgdG8gbGltaXQgcGVybWlzc2lvbnMgdG8uXG4gICAqIEBwYXJhbSBzdWJqZWN0cyBhIGxpc3Qgb2Ygc3ViamVjdHMgdG8gYmluZCB0b1xuICAgKi9cbiAgcHVibGljIGJpbmRJbk5hbWVzcGFjZShuYW1lc3BhY2U6IHN0cmluZywgLi4uc3ViamVjdHM6IHJiLklTdWJqZWN0W10pOiByYi5Sb2xlQmluZGluZyB7XG4gICAgY29uc3QgYmluZGluZyA9IG5ldyByYi5Sb2xlQmluZGluZyh0aGlzLCBgUm9sZUJpbmRpbmctJHtuYW1lc3BhY2V9YCwge1xuICAgICAgbWV0YWRhdGE6IHtcbiAgICAgICAgbmFtZXNwYWNlLFxuICAgICAgfSxcbiAgICAgIHJvbGU6IHRoaXMsXG4gICAgfSk7XG4gICAgYmluZGluZy5hZGRTdWJqZWN0cyguLi5zdWJqZWN0cyk7XG4gICAgcmV0dXJuIGJpbmRpbmc7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgQ2x1c3RlclJvbGVCaW5kaW5nIHRoYXQgYmluZHMgdGhlIHBlcm1pc3Npb25zIGluIHRoaXNcbiAgICogQ2x1c3RlclJvbGUgdG8gYSBsaXN0IG9mIHN1YmplY3RzLCB3aXRob3V0IG5hbWVzcGFjZSByZXN0cmljdGlvbnMuXG4gICAqIEBwYXJhbSBzdWJqZWN0cyBhIGxpc3Qgb2Ygc3ViamVjdHMgdG8gYmluZCB0b1xuICAgKi9cbiAgcHVibGljIGJpbmQoLi4uc3ViamVjdHM6IHJiLklTdWJqZWN0W10pOiByYi5DbHVzdGVyUm9sZUJpbmRpbmcge1xuICAgIGNvbnN0IGJpbmRpbmcgPSBuZXcgcmIuQ2x1c3RlclJvbGVCaW5kaW5nKHRoaXMsICdDbHVzdGVyUm9sZUJpbmRpbmcnLCB7XG4gICAgICByb2xlOiB0aGlzLFxuICAgIH0pO1xuICAgIGJpbmRpbmcuYWRkU3ViamVjdHMoLi4uc3ViamVjdHMpO1xuICAgIHJldHVybiBiaW5kaW5nO1xuICB9XG5cbiAgcHJpdmF0ZSBzeW50aGVzaXplUnVsZXMoKTogazhzLlBvbGljeVJ1bGVbXSB7XG4gICAgY29uc3QgcnVsZXM6IGs4cy5Qb2xpY3lSdWxlW10gPSBbXTtcbiAgICBmb3IgKGNvbnN0IHJ1bGUgb2YgdGhpcy5fcnVsZXMpIHtcbiAgICAgIGZvciAoY29uc3QgZW5kcG9pbnQgb2YgcnVsZS5lbmRwb2ludHMpIHtcbiAgICAgICAgY29uc3QgcmVzb3VyY2UgPSBlbmRwb2ludC5hc0FwaVJlc291cmNlKCk7XG4gICAgICAgIGNvbnN0IG5vblJlc291cmNlID0gZW5kcG9pbnQuYXNOb25BcGlSZXNvdXJjZSgpO1xuXG4gICAgICAgIGlmIChyZXNvdXJjZSAmJiBub25SZXNvdXJjZSkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcignRW5kcG9pbnQgbXVzdCBiZSBlaXRoZXIgcmVzb3VyY2Ugb3Igbm9uIHJlc291cmNlLiBub3QgYm90aC4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGlmICghcmVzb3VyY2UgJiYgIW5vblJlc291cmNlKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdFbmRwb2ludCBtdXN0IGJlIGVpdGhlciByZXNvdXJjZSBvciBub24gcmVzb3VyY2UuIG5vdCBuZWl0aGVyLicpO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKHJlc291cmNlKSB7XG4gICAgICAgICAgcnVsZXMucHVzaCh7XG4gICAgICAgICAgICBhcGlHcm91cHM6IFtyZXNvdXJjZS5hcGlHcm91cCA9PT0gJ2NvcmUnID8gJycgOiByZXNvdXJjZS5hcGlHcm91cF0sXG4gICAgICAgICAgICByZXNvdXJjZXM6IFtyZXNvdXJjZS5yZXNvdXJjZVR5cGVdLFxuICAgICAgICAgICAgcmVzb3VyY2VOYW1lczogcmVzb3VyY2UucmVzb3VyY2VOYW1lID8gW3Jlc291cmNlLnJlc291cmNlTmFtZV0gOiBbXSxcbiAgICAgICAgICAgIHZlcmJzOiBydWxlLnZlcmJzLFxuICAgICAgICAgIH0pO1xuICAgICAgICB9XG4gICAgICAgIGlmIChub25SZXNvdXJjZSkge1xuICAgICAgICAgIHJ1bGVzLnB1c2goeyB2ZXJiczogcnVsZS52ZXJicywgbm9uUmVzb3VyY2VVckxzOiBbbm9uUmVzb3VyY2VdIH0pO1xuICAgICAgICB9XG4gICAgICB9XG5cbiAgICB9XG4gICAgcmV0dXJuIHJ1bGVzO1xuICB9XG5cbiAgcHJpdmF0ZSBzeW50aGVzaXplQWdncmVnYXRpb25SdWxlcygpOiBrOHMuQWdncmVnYXRpb25SdWxlIHwgdW5kZWZpbmVkIHtcbiAgICBpZiAoT2JqZWN0LmtleXModGhpcy5fbGFiZWxTZWxlY3RvcikubGVuZ3RoID09PSAwKSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cblxuICAgIHJldHVybiB7IGNsdXN0ZXJSb2xlU2VsZWN0b3JzOiBbeyBtYXRjaExhYmVsczogdGhpcy5fbGFiZWxTZWxlY3RvciB9XSB9O1xuICB9XG59Il19