import os
from selenium import webdriver
from robot.libraries.BuiltIn import BuiltIn, RobotNotRunningError
from selenium.webdriver.remote.webdriver import WebDriver as RemoteWebDriver
from SeleniumLibrary.locators.elementfinder import ElementFinder
from SeleniumLibrary.errors import ElementNotFound


class UtilityWebdriverSetup:

    @classmethod
    def create_utility_webdrivers(cls, executor_url, session_id):
        """Creates up to two utility webdrivers: a Python Selenium WebDriver named "browser" that attaches itself to an existing web session
           and, if a Robot session is running, also creates a Robot Selenium WebDriver
        """

        # Save the original function, so we can revert our patch
        org_command_execute = RemoteWebDriver.execute

        def new_command_execute(self, command, params=None):
            if command == "newSession":
                # Mock the response
                return {'success': 0, 'value': None, 'sessionId': session_id}
            else:
                return org_command_execute(self, command, params)

        # Patch the function before creating the driver object
        RemoteWebDriver.execute = new_command_execute

        new_driver = webdriver.Remote(command_executor=executor_url, desired_capabilities={})
        new_driver.session_id = session_id

        try:
            BuiltIn().run_keyword('create_webdriver', 'Remote', 'None', f'command_executor={executor_url}')
        except RobotNotRunningError:
            pass

        # Replace the patched function with original function
        RemoteWebDriver.execute = org_command_execute

        cls.browser = new_driver

        cls.change_find(cls.browser)

        cls.update_local_session_info()

    @classmethod
    def update_local_session_info(cls):
        """
        Creates a file named session_info.py at the same location of this file and writes the given executor_url and session_id
           to the file for later use in the use_current_persistent_browser method
        """
        remote_url = cls.browser.command_executor._url
        session_id = cls.browser.session_id

        session_info_file = open(os.path.dirname(os.path.realpath(__file__))+'\\session_info.py', 'w')
        session_info_file.write('session_info=["'+remote_url+'", "'+session_id+'"]')
        
    @classmethod
    def change_find(cls, driver):
        """Changes the find method within the ElementFinder class to use a WebDriver of choice rather than the WebDriver generated by Robot
        """

        def new_find(self, locator, tag=None, first_only=True, required=True,
             parent=None):
            element_type = 'Element' if not tag else tag.capitalize()
            if parent and not self._is_webelement(parent):
                raise ValueError('Parent must be Selenium WebElement but it '
                                'was {}.'.format(type(parent)))
            if self._is_webelement(locator):
                return locator
            prefix, criteria = self._parse_locator(locator)
            strategy = self._strategies[prefix]
            tag, constraints = self._get_tag_and_constraints(tag)
            elements = strategy(criteria, tag, constraints,
                                parent=parent or driver)
            if required and not elements:
                raise ElementNotFound("%s with locator '%s' not found."
                                    % (element_type, locator))
            if first_only:
                if not elements:
                    return None
                return elements[0]
            return elements

        ElementFinder.find = new_find