# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

# Pretty-printing and other formatting utilities for Arrow data structures

import pyarrow.scalar as scalar


def array_format(arr, window=None):
    values = []

    if window is None or window * 2 >= len(arr):
        for x in arr:
            values.append(value_format(x, 0))
        contents = _indent(',\n'.join(values), 2)
    else:
        for i in range(window):
            values.append(value_format(arr[i], 0) + ',')
        values.append('...')
        for i in range(len(arr) - window, len(arr)):
            formatted = value_format(arr[i], 0)
            if i < len(arr) - 1:
                formatted += ','
            values.append(formatted)
        contents = _indent('\n'.join(values), 2)

    return '[\n{0}\n]'.format(contents)


def value_format(x, indent_level=0):
    if isinstance(x, scalar.ListValue):
        contents = ',\n'.join(value_format(item) for item in x)
        return '[{0}]'.format(_indent(contents, 1).strip())
    else:
        return repr(x)


def _indent(text, spaces):
    if spaces == 0:
        return text
    block = ' ' * spaces
    return '\n'.join(block + x for x in text.split('\n'))
