"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToKendra = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const defaults = require("@aws-solutions-constructs/core");
const iam = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
/**
 * @summary The LambdaToKendra class.
 */
class LambdaToKendra extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToKendra class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToKendraProps} props - user provided props for the construct.
     * @since 1.120.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckProps(props);
        if (props.kendraIndexProps && props.existingKendraIndexObj) {
            throw new Error('You may not provide both kendraIndexProps and existingKendraIndexObj');
        }
        if (props.kendraIndexProps && props.kendraDataSourcesProps) {
            throw new Error('You may not provide both kendraDataSourcesProps and existingKendraIndexObj');
        }
        if (props.deployVpc || props.existingVpc) {
            this.vpc = defaults.buildVpc(scope, {
                defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
                existingVpc: props.existingVpc,
                userVpcProps: props.vpcProps,
                constructVpcProps: {
                    enableDnsHostnames: true,
                    enableDnsSupport: true,
                },
            });
            defaults.AddAwsServiceEndpoint(scope, this.vpc, defaults.ServiceEndpointTypes.KENDRA);
        }
        // Setup the Lambda function
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps,
            vpc: this.vpc,
        });
        this.kendraIndex = defaults.buildKendraIndex(this, id, {
            kendraIndexProps: props.kendraIndexProps,
            existingIndexObj: props.existingKendraIndexObj
        });
        this.kendraDataSources = defaults.AddMultipleKendraDataSources(this, id, this.kendraIndex, props.kendraDataSourcesProps);
        // Update Lambda function IAM policy with correct privileges to Kendra index
        const normalizedPermissions = props.indexPermissions ? defaults.normalizeKendraPermissions(props.indexPermissions) : undefined;
        if (!normalizedPermissions || normalizedPermissions.includes("READ")) {
            // Add policy with query permissions
            this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    "kendra:Query",
                    "kendra:Describe*",
                    "kendra:Get*",
                    "kendra:BatchGet*",
                    "kendra:List*",
                    "kendra:Retrieve"
                ],
                resources: [this.kendraIndex.attrArn]
            }));
        }
        if (!normalizedPermissions || normalizedPermissions.includes("SUBMITFEEDBACK")) {
            // Add policy with query permissions
            this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    "kendra:SubmitFeedback"
                ],
                resources: [this.kendraIndex.attrArn]
            }));
        }
        if (normalizedPermissions?.includes("WRITE")) {
            // Add policy with query permissions
            this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    "kendra:Associate*",
                    "kendra:BatchPut*",
                    "kendra:Clear",
                    "kendra:Create*",
                    "kendra:Delete*",
                    "kendra:Disassociate*",
                    "kendra:Put*",
                    "kendra:Update*",
                    "kendra:Start*",
                    "kendra:Submit*",
                    "kendra:Stop*",
                    "kendra:TagResource",
                    "kendra:UntagResource"
                ],
                resources: [this.kendraIndex.attrArn]
            }));
        }
        // Configure environment variables
        const indexIdEnvironmentVariableName = props.indexIdEnvironmentVariableName || 'KENDRA_INDEX_ID';
        this.lambdaFunction.addEnvironment(indexIdEnvironmentVariableName, this.kendraIndex.attrId);
    }
}
exports.LambdaToKendra = LambdaToKendra;
_a = JSII_RTTI_SYMBOL_1;
LambdaToKendra[_a] = { fqn: "@aws-solutions-constructs/aws-lambda-kendra.LambdaToKendra", version: "2.44.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBOzs7Ozs7Ozs7OztHQVdHO0FBRUgsVUFBVTtBQUNWLDJEQUEyRDtBQUUzRCwyQ0FBMkM7QUFHM0MsMkNBQXVDO0FBb0V2Qzs7R0FFRztBQUNILE1BQWEsY0FBZSxTQUFRLHNCQUFTO0lBTTNDOzs7Ozs7O09BT0c7SUFDSCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTBCO1FBQ2xFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDakIsUUFBUSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUUzQixJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxLQUFLLENBQUMsc0JBQXNCLEVBQUU7WUFDMUQsTUFBTSxJQUFJLEtBQUssQ0FBQyxzRUFBc0UsQ0FBQyxDQUFDO1NBQ3pGO1FBRUQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLElBQUksS0FBSyxDQUFDLHNCQUFzQixFQUFFO1lBQzFELE1BQU0sSUFBSSxLQUFLLENBQUMsNEVBQTRFLENBQUMsQ0FBQztTQUMvRjtRQUVELElBQUksS0FBSyxDQUFDLFNBQVMsSUFBSSxLQUFLLENBQUMsV0FBVyxFQUFFO1lBQ3hDLElBQUksQ0FBQyxHQUFHLEdBQUcsUUFBUSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUU7Z0JBQ2xDLGVBQWUsRUFBRSxRQUFRLENBQUMsdUJBQXVCLEVBQUU7Z0JBQ25ELFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztnQkFDOUIsWUFBWSxFQUFFLEtBQUssQ0FBQyxRQUFRO2dCQUM1QixpQkFBaUIsRUFBRTtvQkFDakIsa0JBQWtCLEVBQUUsSUFBSTtvQkFDeEIsZ0JBQWdCLEVBQUUsSUFBSTtpQkFDdkI7YUFDRixDQUFDLENBQUM7WUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLG9CQUFvQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ3ZGO1FBRUQsNEJBQTRCO1FBQzVCLElBQUksQ0FBQyxjQUFjLEdBQUcsUUFBUSxDQUFDLG1CQUFtQixDQUFDLElBQUksRUFBRTtZQUN2RCxpQkFBaUIsRUFBRSxLQUFLLENBQUMsaUJBQWlCO1lBQzFDLG1CQUFtQixFQUFFLEtBQUssQ0FBQyxtQkFBbUI7WUFDOUMsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO1NBQ2QsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFdBQVcsR0FBRyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUNyRCxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCO1lBQ3hDLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxzQkFBc0I7U0FDL0MsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGlCQUFpQixHQUFHLFFBQVEsQ0FBQyw0QkFBNEIsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFLElBQUksQ0FBQyxXQUFXLEVBQUUsS0FBSyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFFekgsNEVBQTRFO1FBQzVFLE1BQU0scUJBQXFCLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsMEJBQTBCLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUMvSCxJQUFJLENBQUMscUJBQXFCLElBQUkscUJBQXFCLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQ3BFLG9DQUFvQztZQUNwQyxJQUFJLENBQUMsY0FBYyxDQUFDLGVBQWUsQ0FDakMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUN0QixPQUFPLEVBQUU7b0JBQ1AsY0FBYztvQkFDZCxrQkFBa0I7b0JBQ2xCLGFBQWE7b0JBQ2Isa0JBQWtCO29CQUNsQixjQUFjO29CQUNkLGlCQUFpQjtpQkFBQztnQkFDcEIsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUM7YUFDdEMsQ0FBQyxDQUNILENBQUM7U0FDSDtRQUVELElBQUksQ0FBQyxxQkFBcUIsSUFBSSxxQkFBcUIsQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsRUFBRTtZQUM5RSxvQ0FBb0M7WUFDcEMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxlQUFlLENBQ2pDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDdEIsT0FBTyxFQUFFO29CQUNQLHVCQUF1QjtpQkFBQztnQkFDMUIsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUM7YUFDdEMsQ0FBQyxDQUNILENBQUM7U0FDSDtRQUVELElBQUkscUJBQXFCLEVBQUUsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQzVDLG9DQUFvQztZQUNwQyxJQUFJLENBQUMsY0FBYyxDQUFDLGVBQWUsQ0FDakMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUN0QixPQUFPLEVBQUU7b0JBQ1AsbUJBQW1CO29CQUNuQixrQkFBa0I7b0JBQ2xCLGNBQWM7b0JBQ2QsZ0JBQWdCO29CQUNoQixnQkFBZ0I7b0JBQ2hCLHNCQUFzQjtvQkFDdEIsYUFBYTtvQkFDYixnQkFBZ0I7b0JBQ2hCLGVBQWU7b0JBQ2YsZ0JBQWdCO29CQUNoQixjQUFjO29CQUNkLG9CQUFvQjtvQkFDcEIsc0JBQXNCO2lCQUN2QjtnQkFDRCxTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQzthQUN0QyxDQUFDLENBQ0gsQ0FBQztTQUNIO1FBRUQsa0NBQWtDO1FBQ2xDLE1BQU0sOEJBQThCLEdBQUcsS0FBSyxDQUFDLDhCQUE4QixJQUFJLGlCQUFpQixDQUFDO1FBQ2pHLElBQUksQ0FBQyxjQUFjLENBQUMsY0FBYyxDQUFDLDhCQUE4QixFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLENBQUM7SUFFOUYsQ0FBQzs7QUEvR0gsd0NBaUhDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuLy8gSW1wb3J0c1xuaW1wb3J0ICogYXMgZGVmYXVsdHMgZnJvbSBcIkBhd3Mtc29sdXRpb25zLWNvbnN0cnVjdHMvY29yZVwiO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtbGFtYmRhXCI7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1pYW1cIjtcbmltcG9ydCAqIGFzIGtlbmRyYSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWtlbmRyYVwiO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtZWMyXCI7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tIFwiY29uc3RydWN0c1wiO1xuXG4vKipcbiAqIEBzdW1tYXJ5IFRoZSBwcm9wZXJ0aWVzIGZvciB0aGUgTGFtYmRhVG9LZW5kcmEgY2xhc3MuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTGFtYmRhVG9LZW5kcmFQcm9wcyB7XG4gIC8qKlxuICAgKlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE9wdGlvbmFsIHVzZXIgcHJvdmlkZWQgcHJvcHMgdG8gb3ZlcnJpZGUgdGhlIGRlZmF1bHQgcHJvcHMgZm9yIHRoZSBLZW5kcmEgaW5kZXguIElzIHRoaXMgcmVxdWlyZWQ/XG4gICAqL1xuICByZWFkb25seSBrZW5kcmFJbmRleFByb3BzPzoga2VuZHJhLkNmbkluZGV4UHJvcHM7XG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgZGF0YSBzb3VyY2VzIHRoYXQgd2lsbCBwcm92aWRlIGRhdGEgdG8gdGhlIEtlbmRyYSBpbmRleC4gP0F0IGxlYXN0IDEgbXVzdCBiZSBzcGVjaWZpZWQuIFdlIHdpbGwgZG8gbWFqb3JpdHkgb2ZcbiAgICogcHJvY2Vzc2luZyBmb3Igc29tZSBkYXRhIHNvdXJjZXMgKFMzIGNyYXdsZXIgaW5pdGlhbGx5KSwgYnV0IGZvciBvdGhlcnMgdGhlIHByb3BzIG11c3QgYmUgY29tcGxldGUgKGUuZy4gcHJvcGVyIHJvbGVBcm4sIGV0Yy4pXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZW1wdHkgbGlzdCAobm8gZGF0YSBzb3VyY2VzKVxuICAgKi9cbiAgcmVhZG9ubHkga2VuZHJhRGF0YVNvdXJjZXNQcm9wczogQXJyYXk8a2VuZHJhLkNmbkRhdGFTb3VyY2VQcm9wcyB8IGFueT47XG4gIC8qKlxuICAgKiBPcHRpb25hbCAtIGluZGV4IHBlcm1pc3Npb25zIHRvIGdyYW50IHRvIHRoZSBMYW1iZGEgZnVuY3Rpb24uIE9uZSBvciBtb3JlIG9mIHRoZSBmb2xsb3dpbmdcbiAgICogbWF5IGJlIHNwZWNpZmllZDrCoGBSZWFkYCwgYFN1Ym1pdEZlZWRiYWNrYCBhbmQgYFdyaXRlYC4gRGVmYXVsdCBpcyBgW1wiUmVhZFwiLCBcIlN1Ym1pdEZlZWRiYWNrXCJdYC4gUmVhZCBpc1xuICAgKiBhbGwgdGhlIG9wZXJhdGlvbnMgSUFNIGRlZmluZXMgYXMgUmVhZCBhbmQgTGlzdC4gU3VibWl0RmVlZGJhY2sgaXMgb25seSB0aGUgU3VibWl0RmVlZGJhY2sgYWN0aW9uLiBXcml0ZSBpcyBhbGwgdGhlXG4gICAqIG9wZXJhdGlvbnMgSUFNIGRlZmluZXMgYXMgV3JpdGUgYW5kIFRhZy4gVGhpcyBmdW5jdGlvbmFsaXR5IG1heSBiZSBvdmVycmlkZGVuIGJ5IHByb3ZpZGluZyBhIHNwZWNpZmljIHJvbGUgYXJuIGluIGxhbWJkYUZ1bmN0aW9uUHJvcHNcbiAgICpcbiAgICogQGRlZmF1bHQgLSBbXCJSZWFkXCIsIFwiU3VibWl0RmVlZGJhY2tcIl1cbiAgICovXG4gIHJlYWRvbmx5IGluZGV4UGVybWlzc2lvbnM/OiBzdHJpbmdbXTtcbiAgLyoqXG4gICAqIEV4aXN0aW5nIGluc3RhbmNlIG9mIGEgS2VuZHJhIEluZGV4LiBQcm92aWRpbmcgYm90aCB0aGlzIGFuZCBrZW5kcmFJbmRleFByb3BzIHdpbGwgY2F1c2UgYW4gZXJyb3IuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgZXhpc3RpbmdLZW5kcmFJbmRleE9iaj86IGtlbmRyYS5DZm5JbmRleDtcbiAgLyoqXG4gICAqIEV4aXN0aW5nIGluc3RhbmNlIG9mIExhbWJkYSBGdW5jdGlvbiBvYmplY3QsIHByb3ZpZGluZyBib3RoIHRoaXMgYW5kIGBsYW1iZGFGdW5jdGlvblByb3BzYCB3aWxsIGNhdXNlIGFuIGVycm9yLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGV4aXN0aW5nTGFtYmRhT2JqPzogbGFtYmRhLkZ1bmN0aW9uO1xuICAvKipcbiAgICogVXNlciBwcm92aWRlZCBwcm9wcyB0byBvdmVycmlkZSB0aGUgZGVmYXVsdCBwcm9wcyBmb3IgdGhlIExhbWJkYSBmdW5jdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBEZWZhdWx0IHByb3BlcnRpZXMgYXJlIHVzZWQuXG4gICAqL1xuICByZWFkb25seSBsYW1iZGFGdW5jdGlvblByb3BzPzogbGFtYmRhLkZ1bmN0aW9uUHJvcHM7XG4gIC8qKlxuICAgKiBBbiBleGlzdGluZyBWUEMgZm9yIHRoZSBjb25zdHJ1Y3QgdG8gdXNlIChjb25zdHJ1Y3Qgd2lsbCBOT1QgY3JlYXRlIGEgbmV3IFZQQyBpbiB0aGlzIGNhc2UpXG4gICAqL1xuICByZWFkb25seSBleGlzdGluZ1ZwYz86IGVjMi5JVnBjO1xuICAvKipcbiAgICogUHJvcGVydGllcyB0byBvdmVycmlkZSBkZWZhdWx0IHByb3BlcnRpZXMgaWYgZGVwbG95VnBjIGlzIHRydWVcbiAgICovXG4gIHJlYWRvbmx5IHZwY1Byb3BzPzogZWMyLlZwY1Byb3BzO1xuICAvKipcbiAgICogV2hldGhlciB0byBkZXBsb3kgYSBuZXcgVlBDXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGRlcGxveVZwYz86IGJvb2xlYW47XG4gIC8qKlxuICAgKiBPcHRpb25hbCBOYW1lIGZvciB0aGUgTGFtYmRhIGZ1bmN0aW9uIGVudmlyb25tZW50IHZhcmlhYmxlIHNldCB0byB0aGUgaW5kZXggaWQgZm9yIHRoZSBLZW5kcmEgaW5kZXguXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gS0VORFJBX0lOREVYX0lEXG4gICAqL1xuICByZWFkb25seSBpbmRleElkRW52aXJvbm1lbnRWYXJpYWJsZU5hbWU/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogQHN1bW1hcnkgVGhlIExhbWJkYVRvS2VuZHJhIGNsYXNzLlxuICovXG5leHBvcnQgY2xhc3MgTGFtYmRhVG9LZW5kcmEgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICBwdWJsaWMgcmVhZG9ubHkgbGFtYmRhRnVuY3Rpb246IGxhbWJkYS5GdW5jdGlvbjtcbiAgcHVibGljIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xuICBwdWJsaWMgcmVhZG9ubHkga2VuZHJhSW5kZXg6IGtlbmRyYS5DZm5JbmRleDtcbiAgcHVibGljIHJlYWRvbmx5IGtlbmRyYURhdGFTb3VyY2VzOiBrZW5kcmEuQ2ZuRGF0YVNvdXJjZVtdO1xuXG4gIC8qKlxuICAgKiBAc3VtbWFyeSBDb25zdHJ1Y3RzIGEgbmV3IGluc3RhbmNlIG9mIHRoZSBMYW1iZGFUb0tlbmRyYSBjbGFzcy5cbiAgICogQHBhcmFtIHtjZGsuQXBwfSBzY29wZSAtIHJlcHJlc2VudHMgdGhlIHNjb3BlIGZvciBhbGwgdGhlIHJlc291cmNlcy5cbiAgICogQHBhcmFtIHtzdHJpbmd9IGlkIC0gdGhpcyBpcyBhIGEgc2NvcGUtdW5pcXVlIGlkLlxuICAgKiBAcGFyYW0ge0xhbWJkYVRvS2VuZHJhUHJvcHN9IHByb3BzIC0gdXNlciBwcm92aWRlZCBwcm9wcyBmb3IgdGhlIGNvbnN0cnVjdC5cbiAgICogQHNpbmNlIDEuMTIwLjBcbiAgICogQGFjY2VzcyBwdWJsaWNcbiAgICovXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBMYW1iZGFUb0tlbmRyYVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICBkZWZhdWx0cy5DaGVja1Byb3BzKHByb3BzKTtcblxuICAgIGlmIChwcm9wcy5rZW5kcmFJbmRleFByb3BzICYmIHByb3BzLmV4aXN0aW5nS2VuZHJhSW5kZXhPYmopIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignWW91IG1heSBub3QgcHJvdmlkZSBib3RoIGtlbmRyYUluZGV4UHJvcHMgYW5kIGV4aXN0aW5nS2VuZHJhSW5kZXhPYmonKTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMua2VuZHJhSW5kZXhQcm9wcyAmJiBwcm9wcy5rZW5kcmFEYXRhU291cmNlc1Byb3BzKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1lvdSBtYXkgbm90IHByb3ZpZGUgYm90aCBrZW5kcmFEYXRhU291cmNlc1Byb3BzIGFuZCBleGlzdGluZ0tlbmRyYUluZGV4T2JqJyk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmRlcGxveVZwYyB8fCBwcm9wcy5leGlzdGluZ1ZwYykge1xuICAgICAgdGhpcy52cGMgPSBkZWZhdWx0cy5idWlsZFZwYyhzY29wZSwge1xuICAgICAgICBkZWZhdWx0VnBjUHJvcHM6IGRlZmF1bHRzLkRlZmF1bHRJc29sYXRlZFZwY1Byb3BzKCksXG4gICAgICAgIGV4aXN0aW5nVnBjOiBwcm9wcy5leGlzdGluZ1ZwYyxcbiAgICAgICAgdXNlclZwY1Byb3BzOiBwcm9wcy52cGNQcm9wcyxcbiAgICAgICAgY29uc3RydWN0VnBjUHJvcHM6IHtcbiAgICAgICAgICBlbmFibGVEbnNIb3N0bmFtZXM6IHRydWUsXG4gICAgICAgICAgZW5hYmxlRG5zU3VwcG9ydDogdHJ1ZSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuXG4gICAgICBkZWZhdWx0cy5BZGRBd3NTZXJ2aWNlRW5kcG9pbnQoc2NvcGUsIHRoaXMudnBjLCBkZWZhdWx0cy5TZXJ2aWNlRW5kcG9pbnRUeXBlcy5LRU5EUkEpO1xuICAgIH1cblxuICAgIC8vIFNldHVwIHRoZSBMYW1iZGEgZnVuY3Rpb25cbiAgICB0aGlzLmxhbWJkYUZ1bmN0aW9uID0gZGVmYXVsdHMuYnVpbGRMYW1iZGFGdW5jdGlvbih0aGlzLCB7XG4gICAgICBleGlzdGluZ0xhbWJkYU9iajogcHJvcHMuZXhpc3RpbmdMYW1iZGFPYmosXG4gICAgICBsYW1iZGFGdW5jdGlvblByb3BzOiBwcm9wcy5sYW1iZGFGdW5jdGlvblByb3BzLFxuICAgICAgdnBjOiB0aGlzLnZwYyxcbiAgICB9KTtcblxuICAgIHRoaXMua2VuZHJhSW5kZXggPSBkZWZhdWx0cy5idWlsZEtlbmRyYUluZGV4KHRoaXMsIGlkLCB7XG4gICAgICBrZW5kcmFJbmRleFByb3BzOiBwcm9wcy5rZW5kcmFJbmRleFByb3BzLFxuICAgICAgZXhpc3RpbmdJbmRleE9iajogcHJvcHMuZXhpc3RpbmdLZW5kcmFJbmRleE9ialxuICAgIH0pO1xuXG4gICAgdGhpcy5rZW5kcmFEYXRhU291cmNlcyA9IGRlZmF1bHRzLkFkZE11bHRpcGxlS2VuZHJhRGF0YVNvdXJjZXModGhpcywgaWQsIHRoaXMua2VuZHJhSW5kZXgsIHByb3BzLmtlbmRyYURhdGFTb3VyY2VzUHJvcHMpO1xuXG4gICAgLy8gVXBkYXRlIExhbWJkYSBmdW5jdGlvbiBJQU0gcG9saWN5IHdpdGggY29ycmVjdCBwcml2aWxlZ2VzIHRvIEtlbmRyYSBpbmRleFxuICAgIGNvbnN0IG5vcm1hbGl6ZWRQZXJtaXNzaW9ucyA9IHByb3BzLmluZGV4UGVybWlzc2lvbnMgPyBkZWZhdWx0cy5ub3JtYWxpemVLZW5kcmFQZXJtaXNzaW9ucyhwcm9wcy5pbmRleFBlcm1pc3Npb25zKSA6IHVuZGVmaW5lZDtcbiAgICBpZiAoIW5vcm1hbGl6ZWRQZXJtaXNzaW9ucyB8fCBub3JtYWxpemVkUGVybWlzc2lvbnMuaW5jbHVkZXMoXCJSRUFEXCIpKSB7XG4gICAgICAvLyBBZGQgcG9saWN5IHdpdGggcXVlcnkgcGVybWlzc2lvbnNcbiAgICAgIHRoaXMubGFtYmRhRnVuY3Rpb24uYWRkVG9Sb2xlUG9saWN5KFxuICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgXCJrZW5kcmE6UXVlcnlcIixcbiAgICAgICAgICAgIFwia2VuZHJhOkRlc2NyaWJlKlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6R2V0KlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6QmF0Y2hHZXQqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpMaXN0KlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6UmV0cmlldmVcIl0sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbdGhpcy5rZW5kcmFJbmRleC5hdHRyQXJuXVxuICAgICAgICB9KVxuICAgICAgKTtcbiAgICB9XG5cbiAgICBpZiAoIW5vcm1hbGl6ZWRQZXJtaXNzaW9ucyB8fCBub3JtYWxpemVkUGVybWlzc2lvbnMuaW5jbHVkZXMoXCJTVUJNSVRGRUVEQkFDS1wiKSkge1xuICAgICAgLy8gQWRkIHBvbGljeSB3aXRoIHF1ZXJ5IHBlcm1pc3Npb25zXG4gICAgICB0aGlzLmxhbWJkYUZ1bmN0aW9uLmFkZFRvUm9sZVBvbGljeShcbiAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgIFwia2VuZHJhOlN1Ym1pdEZlZWRiYWNrXCJdLFxuICAgICAgICAgIHJlc291cmNlczogW3RoaXMua2VuZHJhSW5kZXguYXR0ckFybl1cbiAgICAgICAgfSlcbiAgICAgICk7XG4gICAgfVxuXG4gICAgaWYgKG5vcm1hbGl6ZWRQZXJtaXNzaW9ucz8uaW5jbHVkZXMoXCJXUklURVwiKSkge1xuICAgICAgLy8gQWRkIHBvbGljeSB3aXRoIHF1ZXJ5IHBlcm1pc3Npb25zXG4gICAgICB0aGlzLmxhbWJkYUZ1bmN0aW9uLmFkZFRvUm9sZVBvbGljeShcbiAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgIFwia2VuZHJhOkFzc29jaWF0ZSpcIixcbiAgICAgICAgICAgIFwia2VuZHJhOkJhdGNoUHV0KlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6Q2xlYXJcIixcbiAgICAgICAgICAgIFwia2VuZHJhOkNyZWF0ZSpcIixcbiAgICAgICAgICAgIFwia2VuZHJhOkRlbGV0ZSpcIixcbiAgICAgICAgICAgIFwia2VuZHJhOkRpc2Fzc29jaWF0ZSpcIixcbiAgICAgICAgICAgIFwia2VuZHJhOlB1dCpcIixcbiAgICAgICAgICAgIFwia2VuZHJhOlVwZGF0ZSpcIixcbiAgICAgICAgICAgIFwia2VuZHJhOlN0YXJ0KlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6U3VibWl0KlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6U3RvcCpcIixcbiAgICAgICAgICAgIFwia2VuZHJhOlRhZ1Jlc291cmNlXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpVbnRhZ1Jlc291cmNlXCJcbiAgICAgICAgICBdLFxuICAgICAgICAgIHJlc291cmNlczogW3RoaXMua2VuZHJhSW5kZXguYXR0ckFybl1cbiAgICAgICAgfSlcbiAgICAgICk7XG4gICAgfVxuXG4gICAgLy8gQ29uZmlndXJlIGVudmlyb25tZW50IHZhcmlhYmxlc1xuICAgIGNvbnN0IGluZGV4SWRFbnZpcm9ubWVudFZhcmlhYmxlTmFtZSA9IHByb3BzLmluZGV4SWRFbnZpcm9ubWVudFZhcmlhYmxlTmFtZSB8fCAnS0VORFJBX0lOREVYX0lEJztcbiAgICB0aGlzLmxhbWJkYUZ1bmN0aW9uLmFkZEVudmlyb25tZW50KGluZGV4SWRFbnZpcm9ubWVudFZhcmlhYmxlTmFtZSwgdGhpcy5rZW5kcmFJbmRleC5hdHRySWQpO1xuXG4gIH1cblxufSJdfQ==