"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const ec2 = require("aws-cdk-lib/aws-ec2");
const kendra = require("aws-cdk-lib/aws-kendra");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const defaults = require("@aws-solutions-constructs/core");
test.only('Launch with minimal code and check  structure', () => {
    const stack = new cdk.Stack();
    const testFunctionName = 'test-function-name24334';
    const testBucketName = 'test-bucket-name12344';
    const lambdaProps = {
        functionName: testFunctionName,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: testFunctionName,
        Environment: {
            Variables: {
                KENDRA_INDEX_ID: {
                    "Fn::GetAtt": ["samplekendraindexsample8A81A6C2", "Id"]
                }
            }
        },
    });
    template.hasResourceProperties("AWS::Kendra::Index", {
        RoleArn: {
            "Fn::GetAtt": [
                "samplekendraindexrolesample4F9E7B66",
                "Arn",
            ],
        },
    });
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                BucketName: testBucketName
            },
        },
        RoleArn: {
            "Fn::GetAtt": ["sampledatasourcerolesample05A05F8BD", "Arn"]
        },
    });
    // Confirm policy for Kendra index
    template.hasResourceProperties("AWS::IAM::Role", {
        Description: "Allow Kendra index to write CloudWatch Logs",
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "cloudwatch:PutMetricData",
                            Condition: {
                                StringEquals: {
                                    "cloudwatch:namespace": "AWS/Kendra"
                                }
                            },
                            Effect: "Allow",
                            Resource: "*"
                        },
                        {
                            Action: "logs:CreateLogGroup",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/kendra/*"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: "logs:DescribeLogGroups",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/kendra/*"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: [
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                                "logs:DescribeLogStream"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/kendra/*:log-stream:*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "AllowLogging"
            }
        ],
    });
    // Confirm policy for Kendra index
    template.hasResourceProperties("AWS::IAM::Role", {
        Description: "Policy for Kendra S3 Data Source",
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "kendra.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "s3:GetObject",
                            Effect: "Allow",
                            Resource: `arn:aws:s3:::test-bucket-name12344/*`
                        },
                        {
                            Action: "s3:ListBucket",
                            Effect: "Allow",
                            Resource: `arn:aws:s3:::test-bucket-name12344`
                        },
                        {
                            Action: [
                                "kendra:BatchPutDocument",
                                "kendra:BatchDeleteDocument"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::GetAtt": [
                                    "samplekendraindexsample8A81A6C2",
                                    "Arn"
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "s3CrawlPolicy"
            }
        ]
    });
    // Confirm that Lambda function has QUERY access
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "kendra:Query",
                        "kendra:Describe*",
                        "kendra:Get*",
                        "kendra:BatchGet*",
                        "kendra:List*",
                        "kendra:Retrieve",
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "kendra:SubmitFeedback",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ],
        },
        Roles: [
            {
                Ref: "sampletestfunctionname24334ServiceRole99395A01"
            }
        ]
    });
});
test('Check pattern properties on minimal launch', () => {
    const stack = new cdk.Stack();
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    const newConstruct = new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: 'your-bucket-name',
                    }
                }
            }
        ],
    });
    expect(newConstruct.lambdaFunction).toBeDefined();
    expect(newConstruct.kendraDataSources).toBeDefined();
    expect(newConstruct.kendraDataSources.length).toEqual(1);
    expect(newConstruct.kendraIndex).toBeDefined();
});
test('Launch with VPC', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name12539';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        deployVpc: true,
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    // Check the VPC
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: "10.0.0.0/16",
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
        InstanceTenancy: "default",
    });
    // Is the Lambda function associated with the VPC
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "sampleReplaceDefaultSecurityGroupsecuritygroupE5725669",
                        "GroupId"
                    ]
                }
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                }
            ]
        }
    });
    // Check that the Lambda function Policy has proper network access
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "ec2:CreateNetworkInterface",
                        "ec2:DescribeNetworkInterfaces",
                        "ec2:DeleteNetworkInterface",
                        "ec2:AssignPrivateIpAddresses",
                        "ec2:UnassignPrivateIpAddresses"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {},
                {},
                {}
            ],
        },
        Roles: [
            {
                Ref: "sampleLambdaFunctionServiceRole7A3C4AF5"
            }
        ]
    });
    // Check for the Kendra endpoint in the VPC
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        ServiceName: {
            "Fn::Join": [
                "",
                [
                    "com.amazonaws.",
                    {
                        Ref: "AWS::Region"
                    },
                    ".kendra"
                ]
            ]
        },
        VpcId: {
            Ref: "Vpc8378EB38"
        },
        PrivateDnsEnabled: true,
        SecurityGroupIds: [
            {
                "Fn::GetAtt": [
                    "DefaultKENDRAsecuritygroup34536A79",
                    "GroupId"
                ]
            }
        ],
        SubnetIds: [
            {
                Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
            },
            {
                Ref: "VpcisolatedSubnet2Subnet39217055"
            }
        ],
        VpcEndpointType: "Interface"
    });
});
test('Launch with existing lambda', () => {
    const stack = new cdk.Stack();
    const testTimeout = 17;
    const testFunctionName = 'test-name';
    const existingFunction = new lambda.Function(stack, 'existing-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler',
        timeout: cdk.Duration.seconds(testTimeout),
        functionName: testFunctionName
    });
    new lib_1.LambdaToKendra(stack, 'sample', {
        existingLambdaObj: existingFunction,
        kendraDataSourcesProps: [],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Lambda::Function", 1);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Timeout: testTimeout,
        FunctionName: testFunctionName,
    });
});
test('Confirm error with existing vpc and vpc props', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    const app = () => {
        new lib_1.LambdaToKendra(stack, 'sample', {
            existingVpc: defaults.getTestVpc(stack),
            deployVpc: true,
            vpcProps: {
                ipAddresses: ec2.IpAddresses.cidr('10.0.0.0/16')
            },
            lambdaFunctionProps: lambdaProps,
            kendraDataSourcesProps: [{
                    type: 'S3',
                    dataSourceConfiguration: {
                        s3Configuration: {
                            bucketName: testBucketName,
                        }
                    }
                }
            ],
        });
    };
    expect(app).toThrowError(/Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n/);
});
test('Confirm error with data source with no bucket name', () => {
    const stack = new cdk.Stack();
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    const app = () => {
        new lib_1.LambdaToKendra(stack, 'sample', {
            lambdaFunctionProps: lambdaProps,
            kendraDataSourcesProps: [{
                    type: 'S3',
                    dataSourceConfiguration: {
                        s3Configuration: {}
                    }
                }
            ],
        });
    };
    expect(app).toThrowError(/Error - an S3 Kendra DataSource requires the DataSourceCofiguration.S3Configuration.bucketName prop/);
});
test('Launch with existing vpc', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        existingVpc: defaults.getTestVpc(stack),
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::EC2::VPC", 1);
    template.resourceCountIs("AWS::EC2::VPCEndpoint", 1);
    template.hasResourceProperties("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
test('Launch with Read/Write permissions on kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["ReaD", "SubmitFeedBack", "wrITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "kendra:Query",
                        "kendra:Describe*",
                        "kendra:Get*",
                        "kendra:BatchGet*",
                        "kendra:List*",
                        "kendra:Retrieve",
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "kendra:SubmitFeedback",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: [
                        "kendra:Associate*",
                        "kendra:BatchPut*",
                        "kendra:Clear",
                        "kendra:Create*",
                        "kendra:Delete*",
                        "kendra:Disassociate*",
                        "kendra:Put*",
                        "kendra:Update*",
                        "kendra:Start*",
                        "kendra:Submit*",
                        "kendra:Stop*",
                        "kendra:TagResource",
                        "kendra:UntagResource"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Launch with Write permissions on kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "kendra:Associate*",
                        "kendra:BatchPut*",
                        "kendra:Clear",
                        "kendra:Create*",
                        "kendra:Delete*",
                        "kendra:Disassociate*",
                        "kendra:Put*",
                        "kendra:Update*",
                        "kendra:Start*",
                        "kendra:Submit*",
                        "kendra:Stop*",
                        "kendra:TagResource",
                        "kendra:UntagResource"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Launch with Read permissions on kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["READ"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "kendra:Query",
                        "kendra:Describe*",
                        "kendra:Get*",
                        "kendra:BatchGet*",
                        "kendra:List*",
                        "kendra:Retrieve",
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Launch with SubmitFeedback permissions on kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["SUBMITFEEDBACK"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: "kendra:SubmitFeedback",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Launch with existing kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    const existingRole = new iam.Role(stack, 'existing-role', {
        assumedBy: new iam.ServicePrincipal('kendra.amazonaws.com'),
    });
    const existingIndex = new kendra.CfnIndex(stack, 'existing-index', {
        edition: 'ENTERPRISE',
        name: 'existingIndex',
        roleArn: existingRole.roleArn
    });
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        existingKendraIndexObj: existingIndex,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    // Make sure we didn't create an index anyway
    template.resourceCountIs("AWS::Kendra::Index", 1);
    template.hasResourceProperties("AWS::Kendra::Index", {
        Edition: 'ENTERPRISE'
    });
});
test('Launch with S3 data source with overridden defaults', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name223423';
    const testInclusionPattern = 'this-folder';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        inclusionPatterns: [testInclusionPattern],
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Kendra::DataSource", 1);
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                InclusionPatterns: [testInclusionPattern],
                BucketName: testBucketName
            },
        }
    });
});
test('Launch with S3 data source and unsupported data source', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const nonImplementedSourceType = "WEBCRAWLER";
    const nonImplementedSourceName = "test-other-source";
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    // Create a role
    const fakeRole = new iam.Role(stack, 'fakeRole', {
        assumedBy: new iam.ServicePrincipal('kendra.amazonaws.com'),
        roleName: 'externalFakeRole'
    });
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            },
            {
                name: nonImplementedSourceName,
                roleArn: fakeRole.roleArn,
                type: nonImplementedSourceType,
            }],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Kendra::DataSource", 2);
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                BucketName: testBucketName
            },
        },
        RoleArn: {
            "Fn::GetAtt": ["sampledatasourcerolesample05A05F8BD", "Arn"]
        },
    });
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Name: nonImplementedSourceName,
        Type: nonImplementedSourceType,
    });
});
test('Launch with multiple S3 data sources', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22342';
    const secondBucketName = 'second-bucket-name22342342';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'two-sources', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            },
            {
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: secondBucketName,
                    }
                }
            }],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Kendra::DataSource", 2);
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                BucketName: testBucketName
            },
        },
        RoleArn: {
            "Fn::GetAtt": ["twosourcesdatasourceroletwosources0B8E24996", "Arn"]
        },
    });
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                BucketName: secondBucketName
            },
        },
        RoleArn: {
            "Fn::GetAtt": ["twosourcesdatasourceroletwosources164176C5E", "Arn"]
        },
    });
});
test('Test with custom environment variable name', () => {
    const stack = new cdk.Stack();
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: 'your-bucket-name',
                    }
                }
            }
        ],
        indexIdEnvironmentVariableName: "MY_VAR_NAME",
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                MY_VAR_NAME: {
                    "Fn::GetAtt": ["samplekendraindexsample8A81A6C2", "Id"]
                }
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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