"use strict";
const assert_1 = require("@aws-cdk/assert");
const s3n = require("@aws-cdk/aws-s3-notifications");
const cdk = require("@aws-cdk/cdk");
const cdk_1 = require("@aws-cdk/cdk");
const s3 = require("../lib");
const notification_dests_1 = require("./notification-dests");
module.exports = {
    'bucket without notifications'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket');
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain"
                }
            }
        });
        test.done();
    },
    'when notification are added, a custom resource is provisioned + a lambda handler for it'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket');
        const topic = new notification_dests_1.Topic(stack, 'MyTopic');
        bucket.onEvent(s3.EventType.ObjectCreated, topic);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', { Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)' }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications'));
        test.done();
    },
    'when notification are added, you can tag the lambda'(test) {
        const stack = new cdk.Stack();
        stack.node.apply(new cdk.Tag('Lambda', 'AreTagged'));
        const bucket = new s3.Bucket(stack, 'MyBucket');
        const topic = new notification_dests_1.Topic(stack, 'MyTopic');
        bucket.onEvent(s3.EventType.ObjectCreated, topic);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            Tags: [{ Key: 'Lambda', Value: 'AreTagged' }],
            Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)'
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications'));
        test.done();
    },
    'bucketNotificationTarget is not called during synthesis'(test) {
        const stack = new cdk.Stack();
        // notice the order here - topic is defined before bucket
        // but this shouldn't impact the fact that the topic policy includes
        // the bucket information
        const topic = new notification_dests_1.Topic(stack, 'Topic');
        const bucket = new s3.Bucket(stack, 'MyBucket');
        bucket.onObjectCreated(topic);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::TopicPolicy', {
            "Topics": [
                {
                    "Ref": "TopicBFC7AF6E"
                }
            ],
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": "sns:Publish",
                        "Condition": {
                            "ArnLike": {
                                "aws:SourceArn": {
                                    "Fn::GetAtt": [
                                        "MyBucketF68F3FF0",
                                        "Arn"
                                    ]
                                }
                            }
                        },
                        "Effect": "Allow",
                        "Principal": {
                            "Service": { "Fn::Join": ["", ["s3.", { Ref: "AWS::URLSuffix" }]] }
                        },
                        "Resource": {
                            "Ref": "TopicBFC7AF6E"
                        },
                        "Sid": "sid0"
                    }
                ],
                "Version": "2012-10-17"
            }
        }));
        test.done();
    },
    'subscription types'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'TestBucket');
        const queueTarget = {
            asBucketNotificationDestination: _ => ({
                type: s3n.BucketNotificationDestinationType.Queue,
                arn: 'arn:aws:sqs:...'
            })
        };
        const lambdaTarget = {
            asBucketNotificationDestination: _ => ({
                type: s3n.BucketNotificationDestinationType.Lambda,
                arn: 'arn:aws:lambda:...'
            })
        };
        const topicTarget = {
            asBucketNotificationDestination: _ => ({
                type: s3n.BucketNotificationDestinationType.Topic,
                arn: 'arn:aws:sns:...'
            })
        };
        bucket.onEvent(s3.EventType.ObjectCreated, queueTarget);
        bucket.onEvent(s3.EventType.ObjectCreated, lambdaTarget);
        bucket.onObjectRemoved(topicTarget, { prefix: 'prefix' });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications', {
            "ServiceToken": {
                "Fn::GetAtt": [
                    "BucketNotificationsHandler050a0587b7544547bf325f094a3db8347ECC3691",
                    "Arn"
                ]
            },
            "BucketName": {
                "Ref": "TestBucket560B80BC"
            },
            "NotificationConfiguration": {
                "LambdaFunctionConfigurations": [
                    {
                        "Events": [
                            "s3:ObjectCreated:*"
                        ],
                        "LambdaFunctionArn": "arn:aws:lambda:..."
                    }
                ],
                "QueueConfigurations": [
                    {
                        "Events": [
                            "s3:ObjectCreated:*"
                        ],
                        "QueueArn": "arn:aws:sqs:..."
                    }
                ],
                "TopicConfigurations": [
                    {
                        "Events": [
                            "s3:ObjectRemoved:*"
                        ],
                        "TopicArn": "arn:aws:sns:...",
                        "Filter": {
                            "Key": {
                                "FilterRules": [
                                    {
                                        "Name": "prefix",
                                        "Value": "prefix"
                                    }
                                ]
                            }
                        }
                    }
                ]
            }
        }));
        test.done();
    },
    'multiple subscriptions of the same type'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'TestBucket');
        bucket.onEvent(s3.EventType.ObjectRemovedDelete, {
            asBucketNotificationDestination: _ => ({
                type: s3n.BucketNotificationDestinationType.Queue,
                arn: 'arn:aws:sqs:...:queue1'
            })
        });
        bucket.onEvent(s3.EventType.ObjectRemovedDelete, {
            asBucketNotificationDestination: _ => ({
                type: s3n.BucketNotificationDestinationType.Queue,
                arn: 'arn:aws:sqs:...:queue2'
            })
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications', {
            "ServiceToken": {
                "Fn::GetAtt": [
                    "BucketNotificationsHandler050a0587b7544547bf325f094a3db8347ECC3691",
                    "Arn"
                ]
            },
            "BucketName": {
                "Ref": "TestBucket560B80BC"
            },
            "NotificationConfiguration": {
                "QueueConfigurations": [
                    {
                        "Events": [
                            "s3:ObjectRemoved:Delete"
                        ],
                        "QueueArn": "arn:aws:sqs:...:queue1"
                    },
                    {
                        "Events": [
                            "s3:ObjectRemoved:Delete"
                        ],
                        "QueueArn": "arn:aws:sqs:...:queue2"
                    }
                ]
            }
        }));
        test.done();
    },
    'prefix/suffix filters'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'TestBucket');
        const bucketNotificationTarget = {
            type: s3n.BucketNotificationDestinationType.Queue,
            arn: 'arn:aws:sqs:...'
        };
        bucket.onEvent(s3.EventType.ObjectRemovedDelete, { asBucketNotificationDestination: _ => bucketNotificationTarget }, { prefix: 'images/', suffix: '.jpg' });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications', {
            "ServiceToken": {
                "Fn::GetAtt": [
                    "BucketNotificationsHandler050a0587b7544547bf325f094a3db8347ECC3691",
                    "Arn"
                ]
            },
            "BucketName": {
                "Ref": "TestBucket560B80BC"
            },
            "NotificationConfiguration": {
                "QueueConfigurations": [
                    {
                        "Events": [
                            "s3:ObjectRemoved:Delete"
                        ],
                        "Filter": {
                            "Key": {
                                "FilterRules": [
                                    {
                                        "Name": "suffix",
                                        "Value": ".jpg"
                                    },
                                    {
                                        "Name": "prefix",
                                        "Value": "images/"
                                    }
                                ]
                            }
                        },
                        "QueueArn": "arn:aws:sqs:..."
                    }
                ]
            }
        }));
        test.done();
    },
    'a notification destination can specify a set of dependencies that must be resolved before the notifications resource is created'(test) {
        const stack = new cdk_1.Stack();
        const bucket = new s3.Bucket(stack, 'Bucket');
        const dependent = new cdk.CfnResource(stack, 'Dependent', { type: 'DependOnMe' });
        const dest = {
            asBucketNotificationDestination: () => ({
                arn: 'arn',
                type: s3n.BucketNotificationDestinationType.Queue,
                dependencies: [dependent]
            })
        };
        bucket.onObjectCreated(dest);
        stack.node.prepareTree();
        test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Resources.BucketNotifications8F2E257D, {
            Type: 'Custom::S3BucketNotifications',
            Properties: {
                ServiceToken: { 'Fn::GetAtt': ['BucketNotificationsHandler050a0587b7544547bf325f094a3db8347ECC3691', 'Arn'] },
                BucketName: { Ref: 'Bucket83908E77' },
                NotificationConfiguration: { QueueConfigurations: [{ Events: ['s3:ObjectCreated:*'], QueueArn: 'arn' }] }
            },
            DependsOn: ['Dependent']
        });
        test.done();
    },
    'CloudWatch Events': {
        'onPutItem contains the Bucket ARN itself when path is undefined'(test) {
            const stack = new cdk.Stack();
            const bucket = s3.Bucket.import(stack, 'Bucket', {
                bucketName: 'MyBucket',
            });
            bucket.onPutObject('PutRule');
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "aws.s3",
                    ],
                    "detail": {
                        "eventSource": [
                            "s3.amazonaws.com",
                        ],
                        "eventName": [
                            "PutObject",
                        ],
                        "resources": {
                            "ARN": [
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition",
                                            },
                                            ":s3:::MyBucket",
                                        ],
                                    ],
                                },
                            ],
                        },
                    },
                },
                "State": "ENABLED",
            }));
            test.done();
        },
        "onPutItem contains the path when it's provided"(test) {
            const stack = new cdk.Stack();
            const bucket = s3.Bucket.import(stack, 'Bucket', {
                bucketName: 'MyBucket',
            });
            bucket.onPutObject('PutRule', undefined, 'my/path.zip');
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "aws.s3",
                    ],
                    "detail": {
                        "eventSource": [
                            "s3.amazonaws.com",
                        ],
                        "eventName": [
                            "PutObject",
                        ],
                        "resources": {
                            "ARN": [
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition",
                                            },
                                            ":s3:::MyBucket/my/path.zip"
                                        ],
                                    ],
                                },
                            ],
                        },
                    },
                },
                "State": "ENABLED",
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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