"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const policy_statement_1 = require("./policy-statement");
/**
 * Result of a grant() operation
 *
 * This class is not instantiable by consumers on purpose, so that they will be
 * required to call the Grant factory functions.
 */
class Grant {
    /**
     * Grant the given permissions to the principal
     *
     * The permissions will be added to the principal policy primarily, falling
     * back to the resource policy if necessary. The permissions must be granted
     * somewhere.
     *
     * - Trying to grant permissions to a principal that does not admit adding to
     *   the principal policy while not providing a resource with a resource policy
     *   is an error.
     * - Trying to grant permissions to an absent principal (possible in the
     *   case of imported resources) leads to a warning being added to the
     *   resource construct.
     */
    static addToPrincipalOrResource(options) {
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource
        });
        if (result.success) {
            return result;
        }
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.grantee.grantPrincipal]
        });
        options.resource.addToResourcePolicy(statement);
        return new Grant({ resourceStatement: statement, options });
    }
    /**
     * Try to grant the given permissions to the given principal
     *
     * Absence of a principal leads to a warning, but failing to add
     * the permissions to a present principal is not an error.
     */
    static addToPrincipal(options) {
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: options.resourceArns
        });
        const addedToPrincipal = options.grantee.grantPrincipal.addToPolicy(statement);
        return new Grant({ principalStatement: addedToPrincipal ? statement : undefined, options });
    }
    /**
     * Add a grant both on the principal and on the resource
     *
     * As long as any principal is given, granting on the pricipal may fail (in
     * case of a non-identity principal), but granting on the resource will
     * never fail.
     *
     * Statement will be the resource statement.
     */
    static addToPrincipalAndResource(options) {
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.grantee.grantPrincipal]
        });
        options.resource.addToResourcePolicy(statement);
        return new Grant({ principalStatement: statement, resourceStatement: result.resourceStatement, options });
    }
    /**
     * Returns a "no-op" `Grant` object which represents a "dropped grant".
     *
     * This can be used for e.g. imported resources where you may not be able to modify
     * the resource's policy or some underlying policy which you don't know about.
     *
     * @param grantee The intended grantee
     * @param _intent The user's intent (will be ignored at the moment)
     */
    static drop(grantee, _intent) {
        return new Grant({
            options: { grantee, actions: [], resourceArns: [] }
        });
    }
    constructor(props) {
        this.options = props.options;
        this.principalStatement = props.principalStatement;
        this.resourceStatement = props.resourceStatement;
    }
    /**
     * Whether the grant operation was successful
     */
    get success() {
        return this.principalStatement !== undefined || this.resourceStatement !== undefined;
    }
    /**
     * Throw an error if this grant wasn't successful
     */
    assertSuccess() {
        if (!this.success) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`${describeGrant(this.options)} could not be added on either identity or resource policy.`);
        }
    }
}
exports.Grant = Grant;
function describeGrant(options) {
    return `Permissions for '${options.grantee}' to call '${options.actions}' on '${options.resourceArns}'`;
}
//# sourceMappingURL=data:application/json;base64,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