"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const cdk_1 = require("@aws-cdk/cdk");
/**
 * A PolicyDocument is a collection of statements
 */
class PolicyDocument {
    constructor(props = {}) {
        this.statements = new Array();
        this.creationStack = cdk_1.captureStackTrace();
        this.autoAssignSids = !!props.assignSids;
        this.addStatements(...props.statements || []);
    }
    resolve(context) {
        context.registerPostProcessor(new RemoveDuplicateStatements(this.autoAssignSids));
        return this.render();
    }
    get isEmpty() {
        return this.statements.length === 0;
    }
    /**
     * The number of statements already added to this policy.
     * Can be used, for example, to generate uniuqe "sid"s within the policy.
     */
    get statementCount() {
        return this.statements.length;
    }
    /**
     * Adds a statement to the policy document.
     *
     * @param statement the statement to add.
     */
    addStatements(...statement) {
        this.statements.push(...statement);
    }
    /**
     * Encode the policy document as a string
     */
    toString() {
        return cdk.Token.asString(this, {
            displayHint: 'PolicyDocument'
        });
    }
    /**
     * JSON-ify the document
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return this.render();
    }
    render() {
        if (this.isEmpty) {
            return undefined;
        }
        const doc = {
            Statement: this.statements.map(s => s.toStatementJson()),
            Version: '2012-10-17'
        };
        return doc;
    }
}
exports.PolicyDocument = PolicyDocument;
/**
 * Removes duplicate statements and assign Sids if necessary
 */
class RemoveDuplicateStatements {
    constructor(autoAssignSids) {
        this.autoAssignSids = autoAssignSids;
    }
    postProcess(input, _context) {
        if (!input || !input.Statement) {
            return input;
        }
        const jsonStatements = new Set();
        const uniqueStatements = [];
        for (const statement of input.Statement) {
            const jsonStatement = JSON.stringify(statement);
            if (!jsonStatements.has(jsonStatement)) {
                uniqueStatements.push(statement);
                jsonStatements.add(jsonStatement);
            }
        }
        // assign unique SIDs (the statement index) if `autoAssignSids` is enabled
        const statements = uniqueStatements.map((s, i) => {
            if (this.autoAssignSids && !s.Sid) {
                s.Sid = i.toString();
            }
            return s;
        });
        return {
            ...input,
            Statement: statements
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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