"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const cdk_1 = require("@aws-cdk/cdk");
const region_info_1 = require("@aws-cdk/region-info");
const util_1 = require("./util");
/**
 * Base class for policy principals
 */
class PrincipalBase {
    constructor() {
        this.grantPrincipal = this;
        /**
         * When this Principal is used in an AssumeRole policy, the action to use.
         */
        this.assumeRoleAction = 'sts:AssumeRole';
    }
    addToPolicy(_statement) {
        // This base class is used for non-identity principals. None of them
        // have a PolicyDocument to add to.
        return false;
    }
    toString() {
        // This is a first pass to make the object readable. Descendant principals
        // should return something nicer.
        return JSON.stringify(this.policyFragment.principalJson);
    }
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
}
exports.PrincipalBase = PrincipalBase;
/**
 * A collection of the fields in a PolicyStatement that can be used to identify a principal.
 *
 * This consists of the JSON used in the "Principal" field, and optionally a
 * set of "Condition"s that need to be applied to the policy.
 */
class PrincipalPolicyFragment {
    constructor(principalJson, conditions = {}) {
        this.principalJson = principalJson;
        this.conditions = conditions;
    }
}
exports.PrincipalPolicyFragment = PrincipalPolicyFragment;
class ArnPrincipal extends PrincipalBase {
    constructor(arn) {
        super();
        this.arn = arn;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: [this.arn] });
    }
    toString() {
        return `ArnPrincipal(${this.arn})`;
    }
}
exports.ArnPrincipal = ArnPrincipal;
class AccountPrincipal extends ArnPrincipal {
    constructor(accountId) {
        super(new StackDependentToken(stack => `arn:${stack.partition}:iam::${accountId}:root`).toString());
        this.accountId = accountId;
    }
    toString() {
        return `AccountPrincipal(${this.accountId})`;
    }
}
exports.AccountPrincipal = AccountPrincipal;
/**
 * An IAM principal that represents an AWS service (i.e. sqs.amazonaws.com).
 */
class ServicePrincipal extends PrincipalBase {
    constructor(service, opts = {}) {
        super();
        this.service = service;
        this.opts = opts;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({
            Service: [
                new ServicePrincipalToken(this.service, this.opts).toString()
            ]
        }, this.opts.conditions);
    }
    toString() {
        return `ServicePrincipal(${this.service})`;
    }
}
exports.ServicePrincipal = ServicePrincipal;
/**
 * A principal that represents an AWS Organization
 */
class OrganizationPrincipal extends PrincipalBase {
    constructor(organizationId) {
        super();
        this.organizationId = organizationId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: ['*'] }, { StringEquals: { 'aws:PrincipalOrgID': this.organizationId } });
    }
    toString() {
        return `OrganizationPrincipal(${this.organizationId})`;
    }
}
exports.OrganizationPrincipal = OrganizationPrincipal;
/**
 * A policy prinicipal for canonicalUserIds - useful for S3 bucket policies that use
 * Origin Access identities.
 *
 * See https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html
 *
 * and
 *
 * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 *
 * for more details.
 *
 */
class CanonicalUserPrincipal extends PrincipalBase {
    constructor(canonicalUserId) {
        super();
        this.canonicalUserId = canonicalUserId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ CanonicalUser: [this.canonicalUserId] });
    }
    toString() {
        return `CanonicalUserPrincipal(${this.canonicalUserId})`;
    }
}
exports.CanonicalUserPrincipal = CanonicalUserPrincipal;
class FederatedPrincipal extends PrincipalBase {
    constructor(federated, conditions, assumeRoleAction = 'sts:AssumeRole') {
        super();
        this.federated = federated;
        this.conditions = conditions;
        this.assumeRoleAction = assumeRoleAction;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `FederatedPrincipal(${this.federated})`;
    }
}
exports.FederatedPrincipal = FederatedPrincipal;
class AccountRootPrincipal extends AccountPrincipal {
    constructor() {
        super(new StackDependentToken(stack => stack.account).toString());
    }
    toString() {
        return `AccountRootPrincipal()`;
    }
}
exports.AccountRootPrincipal = AccountRootPrincipal;
/**
 * A principal representing all identities in all accounts
 */
class AnyPrincipal extends ArnPrincipal {
    constructor() {
        super('*');
    }
    toString() {
        return `AnyPrincipal()`;
    }
}
exports.AnyPrincipal = AnyPrincipal;
/**
 * A principal representing all identities in all accounts
 * @deprecated use `AnyPrincipal`
 */
class Anyone extends AnyPrincipal {
}
exports.Anyone = Anyone;
class CompositePrincipal extends PrincipalBase {
    constructor(...principals) {
        super();
        this.principals = new Array();
        if (principals.length === 0) {
            throw new Error('CompositePrincipals must be constructed with at least 1 Principal but none were passed.');
        }
        this.assumeRoleAction = principals[0].assumeRoleAction;
        this.addPrincipals(...principals);
    }
    addPrincipals(...principals) {
        for (const p of principals) {
            if (p.assumeRoleAction !== this.assumeRoleAction) {
                throw new Error(`Cannot add multiple principals with different "assumeRoleAction". ` +
                    `Expecting "${this.assumeRoleAction}", got "${p.assumeRoleAction}"`);
            }
            const fragment = p.policyFragment;
            if (fragment.conditions && Object.keys(fragment.conditions).length > 0) {
                throw new Error(`Components of a CompositePrincipal must not have conditions. ` +
                    `Tried to add the following fragment: ${JSON.stringify(fragment)}`);
            }
            this.principals.push(p);
        }
        return this;
    }
    get policyFragment() {
        const principalJson = {};
        for (const p of this.principals) {
            util_1.mergePrincipal(principalJson, p.policyFragment.principalJson);
        }
        return new PrincipalPolicyFragment(principalJson);
    }
    toString() {
        return `CompositePrincipal(${this.principals})`;
    }
}
exports.CompositePrincipal = CompositePrincipal;
/**
 * A lazy token that requires an instance of Stack to evaluate
 */
class StackDependentToken {
    constructor(fn) {
        this.fn = fn;
        this.creationStack = cdk_1.captureStackTrace();
    }
    resolve(context) {
        return this.fn(cdk_1.Stack.of(context.scope));
    }
    toString() {
        return cdk.Token.asString(this);
    }
    toJSON() {
        return `<unresolved-token>`;
    }
}
class ServicePrincipalToken {
    constructor(service, opts) {
        this.service = service;
        this.opts = opts;
        this.creationStack = cdk_1.captureStackTrace();
    }
    resolve(ctx) {
        const region = this.opts.region || cdk_1.Stack.of(ctx.scope).region;
        const fact = region_info_1.RegionInfo.get(region).servicePrincipal(this.service);
        return fact || region_info_1.Default.servicePrincipal(this.service, region, cdk_1.Stack.of(ctx.scope).urlSuffix);
    }
    toString() {
        return cdk.Token.asString(this, {
            displayHint: this.service
        });
    }
    toJSON() {
        return `<${this.service}>`;
    }
}
//# sourceMappingURL=data:application/json;base64,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