"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk_1 = require("@aws-cdk/cdk");
/**
 * A principal for use in resources that need to have a role but it's unknown
 *
 * Some resources have roles associated with them which they assume, such as
 * Lambda Functions, CodeBuild projects, StepFunctions machines, etc.
 *
 * When those resources are imported, their actual roles are not always
 * imported with them. When that happens, we use an instance of this class
 * instead, which will add user warnings when statements are attempted to be
 * added to it.
 */
class UnknownPrincipal {
    constructor(props) {
        this.assumeRoleAction = 'sts:AssumeRole';
        this.resource = props.resource;
        this.grantPrincipal = this;
    }
    get policyFragment() {
        throw new Error(`Cannot get policy fragment of ${this.resource.node.path}, resource imported without a role`);
    }
    addToPolicy(statement) {
        const stack = cdk_1.Stack.of(this.resource);
        const repr = JSON.stringify(stack.resolve(statement));
        this.resource.node.addWarning(`Add statement to this resource's role: ${repr}`);
        return true; // Pretend we did the work. The human will do it for us, eventually.
    }
}
exports.UnknownPrincipal = UnknownPrincipal;
//# sourceMappingURL=data:application/json;base64,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