"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk_1 = require("@aws-cdk/cdk");
const iam_generated_1 = require("./iam.generated");
const policy_1 = require("./policy");
const principals_1 = require("./principals");
const util_1 = require("./util");
class User extends cdk_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.grantPrincipal = this;
        this.assumeRoleAction = 'sts:AssumeRole';
        this.groups = new Array();
        this.managedPolicies = new Array();
        this.attachedPolicies = new util_1.AttachedPolicies();
        const user = new iam_generated_1.CfnUser(this, 'Resource', {
            userName: props.userName,
            groups: util_1.undefinedIfEmpty(() => this.groups),
            managedPolicyArns: cdk_1.Lazy.listValue({ produce: () => this.managedPolicies.map(p => p.managedPolicyArn) }, { omitEmpty: true }),
            path: props.path,
            loginProfile: this.parseLoginProfile(props)
        });
        this.userName = user.refAsString;
        this.userArn = user.attrArn;
        this.policyFragment = new principals_1.ArnPrincipal(this.userArn).policyFragment;
        if (props.groups) {
            props.groups.forEach(g => this.addToGroup(g));
        }
    }
    /**
     * Adds this user to a group.
     */
    addToGroup(group) {
        this.groups.push(group.groupName);
    }
    /**
     * Attaches a managed policy to the user.
     * @param policy The managed policy to attach.
     */
    addManagedPolicy(policy) {
        this.managedPolicies.push(policy);
    }
    /**
     * Attaches a policy to this user.
     */
    attachInlinePolicy(policy) {
        this.attachedPolicies.attach(policy);
        policy.attachToUser(this);
    }
    /**
     * Adds an IAM statement to the default policy.
     *
     * @returns true
     */
    addToPolicy(statement) {
        if (!this.defaultPolicy) {
            this.defaultPolicy = new policy_1.Policy(this, 'DefaultPolicy');
            this.defaultPolicy.attachToUser(this);
        }
        this.defaultPolicy.addStatements(statement);
        return true;
    }
    parseLoginProfile(props) {
        if (props.password) {
            return {
                password: props.password.toString(),
                passwordResetRequired: props.passwordResetRequired
            };
        }
        if (props.passwordResetRequired) {
            throw new Error('Cannot set "passwordResetRequired" without specifying "initialPassword"');
        }
        return undefined; // no console access
    }
}
exports.User = User;
//# sourceMappingURL=data:application/json;base64,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