"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk_1 = require("@aws-cdk/cdk");
const MAX_POLICY_NAME_LEN = 128;
function undefinedIfEmpty(f) {
    return cdk_1.Lazy.listValue({ produce: () => {
            const array = f();
            return (array && array.length > 0) ? array : undefined;
        } });
}
exports.undefinedIfEmpty = undefinedIfEmpty;
/**
 * Used to generate a unique policy name based on the policy resource construct.
 * The logical ID of the resource is a great candidate as long as it doesn't exceed
 * 128 characters, so we take the last 128 characters (in order to make sure the hash
 * is there).
 */
function generatePolicyName(logicalId) {
    return logicalId.substring(Math.max(logicalId.length - MAX_POLICY_NAME_LEN, 0), logicalId.length);
}
exports.generatePolicyName = generatePolicyName;
/**
 * Helper class that maintains the set of attached policies for a principal.
 */
class AttachedPolicies {
    constructor() {
        this.policies = new Array();
    }
    /**
     * Adds a policy to the list of attached policies.
     *
     * If this policy is already, attached, returns false.
     * If there is another policy attached with the same name, throws an exception.
     */
    attach(policy) {
        if (this.policies.find(p => p === policy)) {
            return; // already attached
        }
        if (this.policies.find(p => p.policyName === policy.policyName)) {
            throw new Error(`A policy named "${policy.policyName}" is already attached`);
        }
        this.policies.push(policy);
    }
}
exports.AttachedPolicies = AttachedPolicies;
/**
 * Merge two dictionaries that represent IAM principals
 */
function mergePrincipal(target, source) {
    for (const key of Object.keys(source)) {
        target[key] = target[key] || [];
        const value = source[key];
        if (!Array.isArray(value)) {
            throw new Error(`Principal value must be an array (it will be normalized later): ${value}`);
        }
        target[key].push(...value);
    }
    return target;
}
exports.mergePrincipal = mergePrincipal;
//# sourceMappingURL=data:application/json;base64,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