"use strict";
const cdk_1 = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
const lib_2 = require("../lib");
module.exports = {
    'the Permission class is a programming model for iam'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addActions('sqs:SendMessage');
        p.addActions('dynamodb:CreateTable', 'dynamodb:DeleteTable');
        p.addResources('myQueue');
        p.addResources('yourQueue');
        p.addAllResources();
        p.addAwsAccountPrincipal(`my${cdk_1.Token.asString({ account: 'account' })}name`);
        p.addAccountCondition('12221121221');
        test.deepEqual(stack.resolve(p.toStatementJson()), { Action: ['sqs:SendMessage',
                'dynamodb:CreateTable',
                'dynamodb:DeleteTable'],
            Resource: ['myQueue', 'yourQueue', '*'],
            Effect: 'Allow',
            Principal: { AWS: { 'Fn::Join': ['',
                        ['arn:',
                            { Ref: 'AWS::Partition' },
                            ':iam::my',
                            { account: 'account' },
                            'name:root']] } },
            Condition: { StringEquals: { 'sts:ExternalId': '12221121221' } } });
        test.done();
    },
    'the PolicyDocument class is a dom for iam policy documents'(test) {
        const stack = new cdk_1.Stack();
        const doc = new lib_1.PolicyDocument();
        const p1 = new lib_1.PolicyStatement();
        p1.addActions('sqs:SendMessage');
        p1.addResources('*');
        const p2 = new lib_1.PolicyStatement();
        p2.effect = lib_1.Effect.Deny;
        p2.addActions('cloudformation:CreateStack');
        doc.addStatements(p1);
        doc.addStatements(p2);
        test.deepEqual(stack.resolve(doc), {
            Version: '2012-10-17',
            Statement: [{ Effect: 'Allow', Action: 'sqs:SendMessage', Resource: '*' },
                { Effect: 'Deny', Action: 'cloudformation:CreateStack' }]
        });
        test.done();
    },
    'Permission allows specifying multiple actions upon construction'(test) {
        const stack = new cdk_1.Stack();
        const perm = new lib_1.PolicyStatement();
        perm.addResources('MyResource');
        perm.addActions('Action1', 'Action2', 'Action3');
        test.deepEqual(stack.resolve(perm.toStatementJson()), {
            Effect: 'Allow',
            Action: ['Action1', 'Action2', 'Action3'],
            Resource: 'MyResource'
        });
        test.done();
    },
    'PolicyDoc resolves to undefined if there are no permissions'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyDocument();
        test.deepEqual(stack.resolve(p), undefined);
        test.done();
    },
    'canonicalUserPrincipal adds a principal to a policy with the passed canonical user id'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyStatement();
        const canoncialUser = "averysuperduperlongstringfor";
        p.addPrincipals(new lib_1.CanonicalUserPrincipal(canoncialUser));
        test.deepEqual(stack.resolve(p.toStatementJson()), {
            Effect: "Allow",
            Principal: {
                CanonicalUser: canoncialUser
            }
        });
        test.done();
    },
    'addAccountRootPrincipal adds a principal with the current account root'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addAccountRootPrincipal();
        test.deepEqual(stack.resolve(p.toStatementJson()), {
            Effect: "Allow",
            Principal: {
                AWS: {
                    "Fn::Join": [
                        "",
                        [
                            "arn:",
                            { Ref: "AWS::Partition" },
                            ":iam::",
                            { Ref: "AWS::AccountId" },
                            ":root"
                        ]
                    ]
                }
            }
        });
        test.done();
    },
    'addFederatedPrincipal adds a Federated principal with the passed value'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addFederatedPrincipal("com.amazon.cognito", { StringEquals: { key: 'value' } });
        test.deepEqual(stack.resolve(p.toStatementJson()), {
            Effect: "Allow",
            Principal: {
                Federated: "com.amazon.cognito"
            },
            Condition: {
                StringEquals: { key: 'value' }
            }
        });
        test.done();
    },
    'addAwsAccountPrincipal can be used multiple times'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addAwsAccountPrincipal('1234');
        p.addAwsAccountPrincipal('5678');
        test.deepEqual(stack.resolve(p.toStatementJson()), {
            Effect: 'Allow',
            Principal: {
                AWS: [
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::1234:root']] },
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::5678:root']] }
                ]
            }
        });
        test.done();
    },
    'hasResource': {
        'false if there are no resources'(test) {
            test.equal(new lib_1.PolicyStatement().hasResource, false, 'hasResource should be false for an empty permission');
            test.done();
        },
        'true if there is one resource'(test) {
            test.equal(new lib_1.PolicyStatement({ resources: ['one-resource'] }).hasResource, true, 'hasResource is true when there is one resource');
            test.done();
        },
        'true for multiple resources'(test) {
            const p = new lib_1.PolicyStatement();
            p.addResources('r1');
            p.addResources('r2');
            test.equal(p.hasResource, true, 'hasResource is true when there are multiple resource');
            test.done();
        },
    },
    'hasPrincipal': {
        'false if there is no principal'(test) {
            test.equal(new lib_1.PolicyStatement().hasPrincipal, false);
            test.done();
        },
        'true if there is a principal'(test) {
            const p = new lib_1.PolicyStatement();
            p.addArnPrincipal('bla');
            test.equal(p.hasPrincipal, true);
            test.done();
        }
    },
    'statementCount returns the number of statement in the policy document'(test) {
        const p = new lib_1.PolicyDocument();
        test.equal(p.statementCount, 0);
        p.addStatements(new lib_1.PolicyStatement({ actions: ['action1'] }));
        test.equal(p.statementCount, 1);
        p.addStatements(new lib_1.PolicyStatement({ actions: ['action2'] }));
        test.equal(p.statementCount, 2);
        test.done();
    },
    '{ AWS: "*" } principal': {
        'is represented as `Anyone`'(test) {
            const stack = new cdk_1.Stack();
            const p = new lib_1.PolicyDocument();
            p.addStatements(new lib_1.PolicyStatement({ principals: [new lib_1.Anyone()] }));
            test.deepEqual(stack.resolve(p), {
                Statement: [
                    { Effect: 'Allow', Principal: '*' }
                ],
                Version: '2012-10-17'
            });
            test.done();
        },
        'is represented as `AnyPrincipal`'(test) {
            const stack = new cdk_1.Stack();
            const p = new lib_1.PolicyDocument();
            p.addStatements(new lib_1.PolicyStatement({ principals: [new lib_1.AnyPrincipal()] }));
            test.deepEqual(stack.resolve(p), {
                Statement: [
                    { Effect: 'Allow', Principal: '*' }
                ],
                Version: '2012-10-17'
            });
            test.done();
        },
        'is represented as `addAnyPrincipal`'(test) {
            const stack = new cdk_1.Stack();
            const p = new lib_1.PolicyDocument();
            const s = new lib_1.PolicyStatement();
            s.addAnyPrincipal();
            p.addStatements(s);
            test.deepEqual(stack.resolve(p), {
                Statement: [
                    { Effect: 'Allow', Principal: '*' }
                ],
                Version: '2012-10-17'
            });
            test.done();
        }
    },
    'addResources() will not break a list-encoded Token'(test) {
        const stack = new cdk_1.Stack();
        const statement = new lib_1.PolicyStatement();
        statement.addActions(...cdk_1.Lazy.listValue({ produce: () => ['a', 'b', 'c'] }));
        statement.addResources(...cdk_1.Lazy.listValue({ produce: () => ['x', 'y', 'z'] }));
        test.deepEqual(stack.resolve(statement.toStatementJson()), {
            Effect: 'Allow',
            Action: ['a', 'b', 'c'],
            Resource: ['x', 'y', 'z'],
        });
        test.done();
    },
    'addCanonicalUserPrincipal can be used to add cannonical user principals'(test) {
        const stack = new cdk_1.Stack();
        const p = new lib_1.PolicyDocument();
        const s1 = new lib_1.PolicyStatement();
        s1.addCanonicalUserPrincipal('cannonical-user-1');
        const s2 = new lib_1.PolicyStatement();
        s2.addPrincipals(new lib_1.CanonicalUserPrincipal('cannonical-user-2'));
        p.addStatements(s1);
        p.addStatements(s2);
        test.deepEqual(stack.resolve(p), {
            Statement: [
                { Effect: 'Allow', Principal: { CanonicalUser: 'cannonical-user-1' } },
                { Effect: 'Allow', Principal: { CanonicalUser: 'cannonical-user-2' } }
            ],
            Version: '2012-10-17'
        });
        test.done();
    },
    'addPrincipal correctly merges array in'(test) {
        const stack = new cdk_1.Stack();
        const arrayPrincipal = {
            get grantPrincipal() { return this; },
            assumeRoleAction: 'sts:AssumeRole',
            policyFragment: new lib_2.PrincipalPolicyFragment({ AWS: ['foo', 'bar'] }),
            addToPolicy() { return false; }
        };
        const s = new lib_1.PolicyStatement();
        s.addAccountRootPrincipal();
        s.addPrincipals(arrayPrincipal);
        test.deepEqual(stack.resolve(s.toStatementJson()), {
            Effect: 'Allow',
            Principal: {
                AWS: [
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::', { Ref: 'AWS::AccountId' }, ':root']] },
                    'foo', 'bar'
                ]
            }
        });
        test.done();
    },
    // https://github.com/awslabs/aws-cdk/issues/1201
    'policy statements with multiple principal types can be created using multiple addPrincipal calls'(test) {
        const stack = new cdk_1.Stack();
        const s = new lib_1.PolicyStatement();
        s.addArnPrincipal('349494949494');
        s.addServicePrincipal('test.service');
        s.addResources('resource');
        s.addActions('action');
        test.deepEqual(stack.resolve(s.toStatementJson()), {
            Action: 'action',
            Effect: 'Allow',
            Principal: { AWS: '349494949494', Service: 'test.service' },
            Resource: 'resource'
        });
        test.done();
    },
    'Service principals': {
        'regional service principals resolve appropriately'(test) {
            const stack = new cdk_1.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
            const s = new lib_1.PolicyStatement();
            s.addActions('test:Action');
            s.addServicePrincipal('codedeploy.amazonaws.com');
            test.deepEqual(stack.resolve(s.toStatementJson()), {
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'codedeploy.cn-north-1.amazonaws.com.cn' }
            });
            test.done();
        },
        'regional service principals resolve appropriately (with user-set region)'(test) {
            const stack = new cdk_1.Stack(undefined, undefined, { env: { region: 'cn-northeast-1' } });
            const s = new lib_1.PolicyStatement();
            s.addActions('test:Action');
            s.addServicePrincipal('codedeploy.amazonaws.com', { region: 'cn-north-1' });
            test.deepEqual(stack.resolve(s.toStatementJson()), {
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'codedeploy.cn-north-1.amazonaws.com.cn' }
            });
            test.done();
        },
        'obscure service principals resolve to the user-provided value'(test) {
            const stack = new cdk_1.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
            const s = new lib_1.PolicyStatement();
            s.addActions('test:Action');
            s.addServicePrincipal('test.service-principal.dev');
            test.deepEqual(stack.resolve(s.toStatementJson()), {
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'test.service-principal.dev' }
            });
            test.done();
        },
    },
    'CompositePrincipal can be used to represent a principal that has multiple types': {
        'with a single principal'(test) {
            const stack = new cdk_1.Stack();
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('i:am:an:arn'));
            const statement = new lib_1.PolicyStatement();
            statement.addPrincipals(p);
            test.deepEqual(stack.resolve(statement.toStatementJson()), { Effect: 'Allow', Principal: { AWS: 'i:am:an:arn' } });
            test.done();
        },
        'conditions are not allowed on individual principals of a composite'(test) {
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('i:am'));
            test.throws(() => p.addPrincipals(new lib_2.FederatedPrincipal('federated', { condition: 1 })), /Components of a CompositePrincipal must not have conditions/);
            test.done();
        },
        'principals and conditions are a big nice merge'(test) {
            const stack = new cdk_1.Stack();
            // add via ctor
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('i:am:an:arn'), new lib_2.ServicePrincipal('amazon.com'));
            // add via `addPrincipals` (with condition)
            p.addPrincipals(new lib_1.Anyone(), new lib_2.ServicePrincipal('another.service'));
            const statement = new lib_1.PolicyStatement();
            statement.addPrincipals(p);
            // add via policy statement
            statement.addArnPrincipal('aws-principal-3');
            statement.addCondition('cond2', { boom: 123 });
            test.deepEqual(stack.resolve(statement.toStatementJson()), {
                Condition: {
                    cond2: { boom: 123 }
                },
                Effect: 'Allow',
                Principal: {
                    AWS: ['i:am:an:arn', '*', 'aws-principal-3'],
                    Service: ['amazon.com', 'another.service'],
                }
            });
            test.done();
        },
        'cannot mix types of assumeRoleAction in a single composite'(test) {
            // GIVEN
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('arn')); // assumeRoleAction is "sts:AssumeRule"
            // THEN
            test.throws(() => p.addPrincipals(new lib_2.FederatedPrincipal('fed', {}, 'sts:Boom')), /Cannot add multiple principals with different "assumeRoleAction". Expecting "sts:AssumeRole", got "sts:Boom"/);
            test.done();
        }
    },
    'duplicate statements': {
        'without tokens'(test) {
            // GIVEN
            const stack = new cdk_1.Stack();
            const p = new lib_1.PolicyDocument();
            const statement = new lib_1.PolicyStatement();
            statement.addResources('resource1', 'resource2');
            statement.addActions('action1', 'action2');
            statement.addServicePrincipal('service');
            statement.addConditions({
                a: {
                    b: 'c'
                },
                d: {
                    e: 'f'
                }
            });
            // WHEN
            p.addStatements(statement);
            p.addStatements(statement);
            p.addStatements(statement);
            // THEN
            test.equal(stack.resolve(p).Statement.length, 1);
            test.done();
        },
        'with tokens'(test) {
            // GIVEN
            const stack = new cdk_1.Stack();
            const p = new lib_1.PolicyDocument();
            const statement1 = new lib_1.PolicyStatement();
            statement1.addResources(cdk_1.Lazy.stringValue({ produce: () => 'resource' }));
            statement1.addActions(cdk_1.Lazy.stringValue({ produce: () => 'action' }));
            const statement2 = new lib_1.PolicyStatement();
            statement2.addResources(cdk_1.Lazy.stringValue({ produce: () => 'resource' }));
            statement2.addActions(cdk_1.Lazy.stringValue({ produce: () => 'action' }));
            // WHEN
            p.addStatements(statement1);
            p.addStatements(statement2);
            // THEN
            test.equal(stack.resolve(p).Statement.length, 1);
            test.done();
        },
    },
    'autoAssignSids enables auto-assignment of a unique SID for each statement'(test) {
        // GIVEN
        const doc = new lib_1.PolicyDocument({
            assignSids: true
        });
        // WHEN
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['action2'], resources: ['resource2'] }));
        // THEN
        const stack = new cdk_1.Stack();
        test.deepEqual(stack.resolve(doc), {
            Version: '2012-10-17',
            Statement: [
                { Action: 'action1', Effect: 'Allow', Resource: 'resource1', Sid: '0' },
                { Action: 'action2', Effect: 'Allow', Resource: 'resource2', Sid: '1' }
            ],
        });
        test.done();
    },
    'constructor args are equivalent to mutating in-place'(test) {
        const stack = new cdk_1.Stack();
        const s = new lib_1.PolicyStatement();
        s.addActions('action1', 'action2');
        s.addAllResources();
        s.addArnPrincipal('arn');
        s.addCondition('key', { equals: 'value' });
        const doc1 = new lib_1.PolicyDocument();
        doc1.addStatements(s);
        const doc2 = new lib_1.PolicyDocument();
        doc2.addStatements(new lib_1.PolicyStatement({
            actions: ['action1', 'action2'],
            resources: ['*'],
            principals: [new lib_2.ArnPrincipal('arn')],
            conditions: {
                key: { equals: 'value' }
            }
        }));
        test.deepEqual(stack.resolve(doc1), stack.resolve(doc2));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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