"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk_1 = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
const util_1 = require("../lib/util");
module.exports = {
    'fails when policy is empty'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        new lib_1.Policy(stack, 'MyPolicy');
        test.throws(() => app.synth(), /Policy is empty/);
        test.done();
    },
    'policy with statements'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy', { policyName: 'MyPolicyName' });
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['sqs:SendMessage'] }));
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['arn'], actions: ['sns:Subscribe'] }));
        const group = new lib_1.Group(stack, 'MyGroup');
        group.attachInlinePolicy(policy);
        assert_1.expect(stack).toMatch({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'MyGroupCBA54B1B' }],
                        PolicyDocument: { Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicyName' } },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group' } } });
        test.done();
    },
    'policy name can be omitted, in which case the logical id will be used'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy');
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['sqs:SendMessage'] }));
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['arn'], actions: ['sns:Subscribe'] }));
        const user = new lib_1.User(stack, 'MyUser');
        user.attachInlinePolicy(policy);
        assert_1.expect(stack).toMatch({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { PolicyDocument: { Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' } } });
        test.done();
    },
    'policy can be attached users, groups and roles and added permissions via props'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const user1 = new lib_1.User(stack, 'User1');
        const group1 = new lib_1.Group(stack, 'Group1');
        const role1 = new lib_1.Role(stack, 'Role1', {
            assumedBy: new lib_1.ServicePrincipal('test.service')
        });
        new lib_1.Policy(stack, 'MyTestPolicy', {
            policyName: 'Foo',
            users: [user1],
            groups: [group1],
            roles: [role1],
            statements: [new lib_1.PolicyStatement({ resources: ['*'], actions: ['dynamodb:PutItem'] })],
        });
        assert_1.expect(stack).toMatch({ Resources: { User1E278A736: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } },
                MyTestPolicy316BDB50: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'Group1BEBD4686' }],
                        PolicyDocument: { Statement: [{ Action: 'dynamodb:PutItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'Foo',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }] } } } });
        test.done();
    },
    'idempotent if a principal (user/group/role) is attached twice'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const p = new lib_1.Policy(stack, 'MyPolicy');
        p.addStatements(new lib_1.PolicyStatement({ actions: ['*'], resources: ['*'] }));
        const user = new lib_1.User(stack, 'MyUser');
        p.attachToUser(user);
        p.attachToUser(user);
        assert_1.expect(stack).toMatch({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { PolicyDocument: { Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' } } });
        test.done();
    },
    'users, groups, roles and permissions can be added using methods'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const p = new lib_1.Policy(stack, 'MyTestPolicy', {
            policyName: 'Foo',
        });
        p.attachToUser(new lib_1.User(stack, 'User1'));
        p.attachToUser(new lib_1.User(stack, 'User2'));
        p.attachToGroup(new lib_1.Group(stack, 'Group1'));
        p.attachToRole(new lib_1.Role(stack, 'Role1', { assumedBy: new lib_1.ServicePrincipal('test.service') }));
        p.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['dynamodb:GetItem'] }));
        assert_1.expect(stack).toMatch({ Resources: { MyTestPolicy316BDB50: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'Group1BEBD4686' }],
                        PolicyDocument: { Statement: [{ Action: 'dynamodb:GetItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'Foo',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }, { Ref: 'User21F1486D1' }] } },
                User1E278A736: { Type: 'AWS::IAM::User' },
                User21F1486D1: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } } } });
        test.done();
    },
    'policy can be attached to users, groups or role via methods on the principal'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy');
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('test.service') });
        user.attachInlinePolicy(policy);
        group.attachInlinePolicy(policy);
        role.attachInlinePolicy(policy);
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['*'] }));
        assert_1.expect(stack).toMatch({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'MyGroupCBA54B1B' }],
                        PolicyDocument: { Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Roles: [{ Ref: 'MyRoleF48FFE04' }],
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group' },
                MyRoleF48FFE04: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } } } });
        test.done();
    },
    'fails if policy name is not unique within a user/group/role'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        // create two policies named Foo and attach them both to the same user/group/role
        const p1 = new lib_1.Policy(stack, 'P1', { policyName: 'Foo' });
        const p2 = new lib_1.Policy(stack, 'P2', { policyName: 'Foo' });
        const p3 = new lib_1.Policy(stack, 'P3'); // uses logicalID as name
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
        p1.attachToUser(user);
        p1.attachToGroup(group);
        p1.attachToRole(role);
        // try to attach p2 to all of these and expect to fail
        test.throws(() => p2.attachToUser(user), /A policy named "Foo" is already attached/);
        test.throws(() => p2.attachToGroup(group), /A policy named "Foo" is already attached/);
        test.throws(() => p2.attachToRole(role), /A policy named "Foo" is already attached/);
        p3.attachToUser(user);
        p3.attachToGroup(group);
        p3.attachToRole(role);
        test.done();
    },
    'fails if policy is not attached to a principal'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'MyStack');
        new lib_1.Policy(stack, 'MyPolicy');
        test.throws(() => app.synth(), /Policy must be attached to at least one principal: user, group or role/);
        test.done();
    },
    'generated policy name only uses the last 128 characters of the logical id'(test) {
        test.equal(util_1.generatePolicyName('Foo'), 'Foo');
        const logicalId50 = '[' + dup(50 - 2) + ']';
        test.equal(util_1.generatePolicyName(logicalId50), logicalId50);
        const logicalId128 = '[' + dup(128 - 2) + ']';
        test.equal(util_1.generatePolicyName(logicalId128), logicalId128);
        const withPrefix = 'PREFIX' + logicalId128;
        test.equal(util_1.generatePolicyName(withPrefix), logicalId128, 'ensure prefix is omitted');
        test.done();
        function dup(count) {
            let r = '';
            for (let i = 0; i < count; ++i) {
                r += 'x';
            }
            return r;
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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