from propylean.generic_equipment_classes import _PressureChangers
from propylean import streams
from propylean.settings import Settings
from propylean.constants import Constants
import propylean.properties as prop
import fluids.compressible as compressible_fluid
from math import pow

# Start of final classes of pumps.
class CentrifugalPump(_PressureChangers):
    items = []    
    def __init__(self, **inputs) -> None:
        """ 
        DESCRIPTION:
            Final class for creating objects to represent Centrifugal Pump.
        
        PARAMETERS:
            Read _PressureChangers class for more arguments for this class
            min_flow:
                Required: No
                Type: int/float or tuple(value, unit) or VolumetricFlowrate(recommended)
                Acceptable values: Non-negative values
                Default value: None # TODO: Add industry standard
                Description: Minimum flow requirement of the pump

            NPSHr:
                Required: No
                Type: int/float or tuple(value, unit) or Length(recommended)
                Acceptable values: Non-negative values
                Default value: None
                Description: NPSHr of the pump
        
        PROPERTIES:
            NPSHa:
                Type: Length
                Description: NPSH available to the pump.
            
            head:
                Type: Length
                Description: Differential head generated by the pump.

            hydraulic_power:
                Type: Power
                Description: Hydraulic power generated by the pump.
            
            power/energy_in:
                Type: Power
                Description: Power required by the pump.

        
        RETURN VALUE:
            Type: CentrifugalPump
            Description: Returns an object of type CentrifugalPump with all properties of
                         a centrifugal pump used in process industries.
        
        ERROR RAISED:
            Type:
            Description:
        
        SAMPLE USE CASES:
            >>> pump_1 = CentrifugalPump(tag="P1")
            >>> print(pump_1)
            Centrifugal Pump with tag: P1
        """
        self._index = len(CentrifugalPump.items)
        super().__init__( **inputs)
        self._NPSHr = prop.Length()
        self._NPSHa = prop.Length()
        self._min_flow = prop.VolumetricFlowRate()
        del self.energy_out
        
        if 'min_flow' in inputs:
            self.min_flow = inputs['min_flow']
        if "NPSHr" in inputs:
            self.NPSHr = inputs['NPSHr']
    
        CentrifugalPump.items.append(self)
    
    def __repr__(self):
        return "Centrifugal Pump with tag: " + self.tag
    def __hash__(self):
        return hash(self.__repr__())

    @property
    def min_flow(self):
        self = self._get_equipment_object(self)
        return self._min_flow
    @min_flow.setter
    def min_flow(self, value):
        self = self._get_equipment_object(self)
        value, unit = self._tuple_property_value_unit_returner(value, prop.VolumetricFlowRate)
        if unit is None:
            unit = self._min_flow.unit
        self._min_flow = prop.VolumetricFlowRate(value, unit)
        self._update_equipment_object(self)

    @property
    def NPSHr(self):
        self = self._get_equipment_object(self)
        return self._NPSHr
    @NPSHr.setter
    def NPSHr(self, value):
        self = self._get_equipment_object(self)
        value, unit = self._tuple_property_value_unit_returner(value, prop.Pressure)
        if unit is None:
            unit = self._NPSHr.unit
        self._NPSHr = prop.Length(value, unit)
        self._update_equipment_object(self)
    
    @property
    def NPSHa(self):
        self = self._get_equipment_object(self)
        if self._inlet_material_stream_tag is None:
            raise Exception("Pump should be connected with MaterialStream at the inlet")
        density = self._connected_stream_property_getter(True, "material", "density")
        density.unit = "kg/m^3"
        old_p_unit = self.inlet_pressure.unit
        self.inlet_pressure.unit = 'Pa'
        value = self.inlet_pressure.value/(9.8 * density.value)
        self.inlet_pressure.unit = old_p_unit
        return prop.Length(value, "m")

    @property
    def head(self):
        self = self._get_equipment_object(self)
        if (self._outlet_material_stream_tag is None and
            self._inlet_material_stream_tag is None):
            raise Exception("Pump should be connected with MaterialStream either at inlet or outlet")
        is_inlet = False if self._inlet_material_stream_index is None else True
        density = self._connected_stream_property_getter(is_inlet, "material", "density")
        density.unit = "kg/m^3"
        dp = self.differential_pressure
        dp.unit = "Pa"
        value = dp.value / (9.8 * density.value)
        return prop.Length(value, "m")
    @property
    def hydraulic_power(self):
        self = self._get_equipment_object(self)
        if (self._outlet_material_stream_tag is None and
            self._inlet_material_stream_tag is None):
            raise Exception("Centrifugal Pump should be connected with MaterialStream either at inlet or outlet")
        is_inlet = False if self._inlet_material_stream_index is None else True
        vol_flowrate = self._connected_stream_property_getter(is_inlet, "material", "vol_flowrate")
        vol_flowrate.unit = "m^3/h"
        dp = self.differential_pressure
        dp.unit = "Pa"
        value = vol_flowrate.value * dp.value / (3.6e3)
        return prop.Power(value, 'W')
    @property
    def power(self):
        self = self._get_equipment_object(self)
        self.hydraulic_power.unit = "W"
        value = self.hydraulic_power.value / self.efficiency
        return prop.Power(value, "W")
    @power.setter
    def power(self, value):
        #TODO Proived setting feature for power
        raise Exception("Pump power value setting is not yet supported. Modify differential pressure to get required power.")
    @property
    def energy_in(self):
        return self.power
    @energy_in.setter
    def energy_in(self, value):
        self = self._get_equipment_object(self)
        value, unit = self._tuple_property_value_unit_returner(value, prop.Power)
        if unit is None:
            unit = self.energy_in.unit
        self._energy_in = prop.Power(value, unit)
        self._update_equipment_object(self)
    
    @classmethod
    def list_objects(cls):
        return cls.items
    
    def connect_stream(self, 
                       stream_object=None, 
                       direction=None, 
                       stream_tag=None, 
                       stream_type=None,
                       stream_governed=True):
        if ((stream_object is not None and 
            isinstance(stream_object, streams.EnergyStream)) or
            stream_type in ['energy', 'power', 'e', 'p']):
            direction = 'in'
            stream_governed = False
        return super().connect_stream(direction=direction, 
                                      stream_object=stream_object, 
                                      stream_tag=stream_tag, 
                                      stream_type=stream_type,
                                      stream_governed=stream_governed)
    
    def disconnect_stream(self, stream_object=None, direction=None, stream_tag=None, stream_type=None):
        if ((stream_object is not None and 
            isinstance(stream_object, streams.EnergyStream)) or
            stream_type in ['energy', 'power', 'e', 'p']):
            direction = 'in'
        return super().disconnect_stream(stream_object, direction, stream_tag, stream_type)

class PositiveDisplacementPump(_PressureChangers):
    items = []
    def __init__(self, **inputs) -> None:
        """ 
        DESCRIPTION:
            Final class for creating objects to represent a Positive Displacement Pump.
        
        PARAMETERS:
            Read _PressureChangers class for more arguments for this class
            min_flow:
                Required: No
                Type: int/float or tuple(value, unit) or VolumetricFlowrate(recommended)
                Acceptable values: Non-negative values
                Default value: None # TODO: Add industry standard
                Description: Minimum flow requirement of the pump

            NPSHr:
                Required: No
                Type: int/float or tuple(value, unit) or Length(recommended)
                Acceptable values: Non-negative values
                Default value: None
                Description: NPSHr of the pump
        
        PROPERTIES:
            NPSHa:
                Type: Length
                Description: NPSH available to the pump.
            
            head:
                Type: Length
                Description: Differential head generated by the pump.

            accel_head:
                Type: Length
                Description: Acceleration head loss at the inlet of pump.
            
            power/energy_in:
                Type: Power
                Description: Power required by the pump.

        RETURN VALUE:
            Type: PositiveDisplacementPump
            Description: Returns an object of type PositiveDisplacementPump with all properties of
                         a positive displacement pump used in process industries.
        
        ERROR RAISED:
            Type:
            Description:
        
        SAMPLE USE CASES:
            >>> pump_1 = PositiveDisplacementlPump(tag="P1")
            >>> print(pump_1)
            Positive Displacement Pump with tag: P1
        """
        self._index = len(PositiveDisplacementPump.items)
        super().__init__( **inputs)
        self._speed = prop.Frequency()
        self._NPSHr = prop.Length()
        if "NPSHr" in inputs:
            self.NPSHr = inputs['NPSHr']
        del self.energy_out


        PositiveDisplacementPump.items.append(self)
    
    def __repr__(self):
        return "Positive Displacement Pump with tag: " + self.tag
    def __hash__(self):
        return hash(self.__repr__())
    
    @property
    def NPSHa(self):
        self = self._get_equipment_object(self)
        if self._inlet_material_stream_tag is None:
            raise Exception("Pump should be connected with MaterialStream at the inlet")
        density = self._connected_stream_property_getter(True, "material", "density")
        density.unit = "kg/m^3"
        old_p_unit = self.inlet_pressure.unit
        old_acc_head_unit = self.accel_head.unit
        self.inlet_pressure.unit = self.accel_head = 'Pa'
        
        value = (self.inlet_pressure.value - self.accel_head.value)/(9.8 * density.value)
        self.inlet_pressure.unit = old_p_unit
        self.accel_head.unit = old_acc_head_unit
        return prop.Length(value, "m")
    @property
    def NPSHr(self):
        self = self._get_equipment_object(self)
        return self._NPSHr
    @NPSHr.setter
    def NPSHr(self, value):
        self = self._get_equipment_object(self)
        value, unit = self._tuple_property_value_unit_returner(value, prop.Pressure)
        if unit is None:
            unit = self._NPSHr.unit
        self._NPSHr = prop.Length(value, unit)
        self._update_equipment_object(self)
    
    @property
    def head(self):
        self = self._get_equipment_object(self)
        if (self._outlet_material_stream_tag is None and
            self._inlet_material_stream_tag is None):
            raise Exception("Pump should be connected with MaterialStream either at inlet or outlet")
        is_inlet = False if self._inlet_material_stream_index is None else True
        density = self._connected_stream_property_getter(is_inlet, "material", "density")
        density.unit = "kg/m^3"
        dp = self.differential_pressure
        dp.unit = "Pa"
        value = dp.value / (9.8 * density.value)
        return prop.Length(value, "m")
    
    @property
    def accel_head(self):
        L, V = self._get_suction_length_velocity()
        density = self._connected_stream_property_getter(True, "material", "density")
        density.unit = "kg/m^3"
        SG = density.value/1000
        k = 1
        head_loss = L.value * V.value * self.speed * SG / (k * 9.8) 
        return prop.Length(head_loss, 'm')
    
    @property
    def power(self):
        self = self._get_equipment_object(self)
        is_inlet = False if self._inlet_material_stream_index is None else True
        vol_flow = self._connected_stream_property_getter(is_inlet, "material", "vol_flowrate")
        vol_flow.unit = "gal/min"
        old_dp_unit = self.differential_pressure.unit
        self.differential_pressure.unit = 'psi'
        value = vol_flow.value * self.differential_pressure.value /(1714 * self.efficiency)
        return prop.Power(value, "hp")
    @power.setter
    def power(self, value):
        #TODO Proived setting feature for power
        raise Exception("Pump power value setting is not yet supported. Modify differential pressure to get required power.")
    @property
    def energy_in(self):
        return self.power
    @energy_in.setter
    def energy_in(self, value):
        self = self._get_equipment_object(self)
        value, unit = self._tuple_property_value_unit_returner(value, prop.Power)
        if unit is None:
            unit = self.energy_in.unit
        self._energy_in = prop.Power(value, unit)
        self._update_equipment_object(self)
        
    
    @classmethod
    def list_objects(cls):
        return cls.items

    def connect_stream(self, 
                       stream_object=None, 
                       direction=None, 
                       stream_tag=None, 
                       stream_type=None,
                       stream_governed=True):
        if ((stream_object is not None and 
            isinstance(stream_object, streams.EnergyStream)) or
            stream_type in ['energy', 'power', 'e', 'p']):
            direction = 'in'
        return super().connect_stream(direction=direction, 
                                      stream_object=stream_object, 
                                      stream_tag=stream_tag, 
                                      stream_type=stream_type,
                                      stream_governed=stream_governed)
    
    def disconnect_stream(self, stream_object=None, direction=None, stream_tag=None, stream_type=None):
        if ((stream_object is not None and 
            isinstance(stream_object, streams.EnergyStream)) or
            stream_type in ['energy', 'power', 'e', 'p']):
            direction = 'in'
        return super().disconnect_stream(stream_object, direction, stream_tag, stream_type)
# End of final classes of pumps

# Start of final classes of Compressors and Expanders.
class CentrifugalCompressor(_PressureChangers):
    items = []
    def __init__(self, **inputs) -> None:
        """ 
        DESCRIPTION:
            Final class for creating objects to represent a Centrifugal Compressors.
        
        PARAMETERS:
            Read _PressureChangers class for more arguments for this class           
            polytropic_exponent:
                Required: No
                Type: int or float (recommended)
                Acceptable values: Non-negative values
                Default value: 1.4
                Description: Polytropic exponent of the gas.
        
        PROPERTIES:
            power:
                Type: Power
                Description: Power required by the compressor.

        
        RETURN VALUE:
            Type: CentrifugalCompressor
            Description: Returns an object of type CentrifugalCompressor with all properties of
                         a Centrifugal Compressor used in process industries.
        
        ERROR RAISED:
            Type:
            Description:
        
        SAMPLE USE CASES:
            >>> CC_1 = CentrifugalCompressor(tag="P1")
            >>> print(CC_1)
            Centrifugal Compressor with tag: P1
        """
        self._index = len(CentrifugalCompressor.items)
        super().__init__( **inputs)
        self.adiabatic_efficiency = 0.7 if 'efficiency' not in inputs else inputs['efficiency']
        self.polytropic_exponent = 1.4 if 'polytropic_exponent' not in inputs else inputs['polytropic_exponent']
        del self.energy_out
        CentrifugalCompressor.items.append(self)
    
    def __repr__(self):
        return "Centrifugal Compressor with tag: " + self.tag
    def __hash__(self):
        return hash(self.__repr__())
    
    @property
    def temperature_increase(self):
        self = self._get_equipment_object(self)
        k = self.polytropic_exponent
        if (self._inlet_material_stream_index is not None or
           self._outlet_material_stream_index is not None):
            is_inlet = False if self._inlet_material_stream_index is None else True
            k = self._connected_stream_property_getter(is_inlet, "material", "isentropic_exponent")
            if Settings.compressor_process == "Polytropic":
                k = compressible_fluid.polytropic_exponent(k=k, eta_p=self.polytropic_efficiency)

        T1 = self.inlet_temperature
        P1 = self.inlet_pressure
        P2 = self.outlet_pressure
        T1.unit = "K"
        P1.unit = P2.unit ="Pa"
        eta = self.efficiency
        value = prop.Temperature(compressible_fluid.isentropic_T_rise_compression(T1.value, P1.value, P2.value, k, eta))
        value = value - T1
        value.unit = self.inlet_temperature.unit
        print("temp change: ", value)
        return value

    @property
    def efficiency(self):
        self = self._get_equipment_object(self)
        if Settings.compressor_process == "Polytropic":
            return self.polytropic_efficiency
        else:
            return self.adiabatic_efficiency
    @efficiency.setter
    def efficiency(self, value):
        self = self._get_equipment_object(self)
        if value < 0:
            raise Exception("Please enter a positive value for efficiency")
        elif value <= 1:
            value = value
        else:
            value = value/100
        if Settings.compressor_process.lower() in ["adiabatic", "isentropic"]:
            self.adiabatic_efficiency = value
        else:
            self.polytropic_efficiency = value
            
        super().efficiency(value)

    @property
    def adiabatic_efficiency(self):
        self = self._get_equipment_object(self)
        return self._adiabatic_efficiency
    @adiabatic_efficiency.setter
    def adiabatic_efficiency(self, value):
        self = self._get_equipment_object(self)
        if value ==  None:
            value = 0.7
        self._adiabatic_efficiency = value
        self._update_equipment_object(self)
    
    @property
    def polytropic_efficiency(self):
        self = self._get_equipment_object(self)
        if (self._inlet_material_stream_index is not None or
           self._outlet_material_stream_index is not None):
            is_inlet = False if self._inlet_material_stream_index is None else True
            isentropic_exponent = self._connected_stream_property_getter(is_inlet, "material", "isentropic_exponent")
            return compressible_fluid.isentropic_efficiency(P1 = self._inlet_pressure.value,
                                                            P2 = self._outlet_pressure.value,
                                                            k = isentropic_exponent,
                                                            eta_s = self.adiabatic_efficiency)
        else:
            return self.efficiency

    @polytropic_efficiency.setter
    def polytropic_efficiency(self, value):
        self = self._get_equipment_object(self)
        is_inlet = False if self._intlet_material_stream_index is None else True
        isentropic_exponent = self._connected_stream_property_getter(is_inlet, "material", "isentropic_exponent")
        self.adiabatic_efficiency = compressible_fluid.isentropic_efficiency(P1 = self._inlet_pressure.value,
                                                                             P2 = self._outlet_pressure.value,
                                                                             k = isentropic_exponent,
                                                                             eta_p = value)
        self._update_equipment_object(self)
    
    @property
    def polytropic_exponent(self):
        self = self._get_equipment_object(self)
        if (self._inlet_material_stream_index is None and
            self._outlet_material_stream_index is None):
            return self._polytropic_exponent
        is_inlet = False if self._inlet_material_stream_index is None else True
        k = self._connected_stream_property_getter(is_inlet, "material", "isentropic_exponent")
        return compressible_fluid.polytropic_exponent(k=k, eta_p=self.polytropic_efficiency)
    @polytropic_exponent.setter
    def polytropic_exponent(self, value):
        self = self._get_equipment_object(self)
        if (self._inlet_material_stream_index is None and
            self._outlet_material_stream_index is None):
            self._polytropic_exponent = value
        else:
            raise Exception("""Polytropic Exponent cannot be set as Compressor is connected to a MaterialStream object.\n
                               Update Isentropic Exponent of the stream object instead.""")

    @property
    def power(self):
        self = self._get_equipment_object(self)
        is_inlet = False if self._inlet_material_stream_index is None else True
        isentropic_exponent = self._connected_stream_property_getter(is_inlet, "material", "isentropic_exponent")
        Z = self._connected_stream_property_getter(is_inlet, "material", "Z_g")
        MW = self._connected_stream_property_getter(is_inlet, "material", "molecular_weight")
        work = compressible_fluid.isentropic_work_compression(T1 = self.inlet_temperature.value,
                                                              k = isentropic_exponent,
                                                              Z = Z,
                                                              P1 = self._inlet_pressure.value,
                                                              P2 = self._outlet_pressure.value,
                                                              eta = self.adiabatic_efficiency)
        return prop.Power(work * self.inlet_mass_flowrate.value / MW.value)
    
    @property
    def head(self):
        if (self._inlet_material_stream_index is None or
            self._outlet_material_stream_index is None):
            raise Exception("Head calculations only supported when Compressor is connected to a MaterialStream object.")
        if Settings.compressor_process.lower() in ["adiabatic", "isentropic"]:
            isentropic_exponent = self._connected_stream_property_getter(True, "material", "isentropic_exponent")
            ratio = (isentropic_exponent - 1)/isentropic_exponent
        else:
            ratio = (self.polytropic_exponent - 1)/self.polytropic_exponent
        
        P1 = self.inlet_pressure
        P2 = self.outlet_pressure
        P1.unit = "Pa"
        P2.unit = "Pa"
        Zi = self._connected_stream_property_getter(True, "material", "Z_g")
        Zo = self._connected_stream_property_getter(False, "material", "Z_g")
        Z = (Zi + Zo)/2
        T1 = self.inlet_temperature
        T1.unit = "K"
        MW = self._connected_stream_property_getter(True, "material", "molecular_weight")
        head = Z * Constants.R * T1.value * (pow((P2.value/P1.value), ratio) - 1)/(ratio * MW.value)
        return prop.Length(head)
        
    @property
    def energy_in(self):
        return self.power
    @energy_in.setter
    def energy_in(self, value):
        self = self._get_equipment_object(self)
        value, unit = self._tuple_property_value_unit_returner(value, prop.Power)
        if unit is None:
            unit = self.energy_in.unit
        self._energy_in = prop.Power(value, unit)
        self._update_equipment_object(self)

    @classmethod
    def list_objects(cls):
        return cls.items
    
    def connect_stream(self, 
                       stream_object=None, 
                       direction=None, 
                       stream_tag=None, 
                       stream_type=None,
                       stream_governed=True):
        if ((stream_object is not None and 
            isinstance(stream_object, streams.EnergyStream)) or
            stream_type in ['energy', 'power', 'e', 'p']):
            direction = 'in'
            stream_governed = False
        return super().connect_stream(direction=direction, 
                                      stream_object=stream_object, 
                                      stream_tag=stream_tag, 
                                      stream_type=stream_type,
                                      stream_governed=stream_governed)
    
    def disconnect_stream(self, stream_object=None, direction=None, stream_tag=None, stream_type=None):
        if ((stream_object is not None and 
            isinstance(stream_object, streams.EnergyStream)) or
            stream_type in ['energy', 'power', 'e', 'p']):
            direction = 'in'
        return super().disconnect_stream(stream_object, direction, stream_tag, stream_type)

class Expander(_PressureChangers):
    items = []
    def __init__(self, **inputs) -> None:
        self._index = len(Expander.items)
        super().__init__( **inputs)
        del self.energy_in
        Expander.items.append(self)
    
    def __repr__(self):
        return "Expander with tag: " + self.tag
    def __hash__(self):
        return hash(self.__repr__())

    @classmethod
    def list_objects(cls):
        return cls.items
    
    def connect_stream(self,
                       stream_object=None, 
                       direction=None, 
                       stream_tag=None, 
                       stream_type=None,
                       stream_governed=True):
        if ((stream_object is not None and 
            isinstance(stream_object, streams.EnergyStream)) or
            stream_type in ['energy', 'power', 'e', 'p']):
            direction = 'out'
        return super().connect_stream(direction=direction, 
                                      stream_object=stream_object, 
                                      stream_tag=stream_tag, 
                                      stream_type=stream_type,
                                      stream_governed=stream_governed)
    
    def disconnect_stream(self, stream_object=None, direction=None, stream_tag=None, stream_type=None):
        if ((stream_object is not None and 
            isinstance(stream_object, streams.EnergyStream)) or
            stream_type in ['energy', 'power', 'e', 'p']):
            direction = 'out'
        return super().disconnect_stream(stream_object, direction, stream_tag, stream_type)

# End of final classes of compressors