#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""A simple Python 3 CLI to read your Things app data."""

from __future__ import print_function

__author__ = "Alexander Willner"
__copyright__ = "2021 Alexander Willner"
__credits__ = ["Alexander Willner"]
__license__ = "Apache License 2.0"
__version__ = "0.0.3"
__maintainer__ = "Alexander Willner"
__email__ = "alex@willner.ws"
__status__ = "Development"

import sys
import argparse
import json
import csv
import webbrowser
import argcomplete  # type: ignore
import xml.etree.ElementTree as ET
from xml.etree.ElementTree import Element, SubElement
from xml.dom import minidom
from io import StringIO

import things as api


class ThingsCLI:
    """A simple Python 3 CLI to read your Things app data."""

    print_json = False
    print_csv = False
    print_opml = False
    # anonymize = False
    database = None
    recursive = False

    def __init__(self, database=None):
        self.database = database

    def print_tasks(self, tasks):
        """Print a task."""
        if self.print_json:
            print(json.dumps(tasks))
        elif self.print_opml:
            print(self.opml_dumps(tasks))
        elif self.print_csv:
            print(self.csv_dumps(tasks))
        else:
            print(self.txt_dumps(tasks))

    def csv_dumps(self, tasks):
        fieldnames = []
        self.csv_header(tasks, fieldnames)
        if 'items' in fieldnames:
            fieldnames.remove('items')
        if 'checklist' in fieldnames:
            fieldnames.remove('checklist')

        output = StringIO()
        writer = csv.DictWriter(output, fieldnames=fieldnames, delimiter=";")
        writer.writeheader()

        self.csv_converter(tasks, writer)

        return output.getvalue()

    def csv_header(self, tasks, fieldnames):
        for task in tasks:
            fieldnames.extend(field for field in task if field not in fieldnames)
            self.csv_header(task.get('items', []), fieldnames)

    def csv_converter(self, tasks, writer):
        if tasks is True:
            return
        for task in tasks:
            self.csv_converter(task.get('items', []), writer)
            task.pop('items', [])
            self.csv_converter(task.get('checklist', []), writer)
            task.pop('checklist', [])
            writer.writerow(task)

    def opml_dumps(self, tasks):
        top = Element('opml')
        head = SubElement(top, 'head')
        SubElement(head, 'title').text = 'Things 3 Database'
        body = SubElement(top, 'body')

        self.opml_convert(tasks, body)

        return minidom.parseString(
            ET.tostring(top)).toprettyxml(indent="   ")

    def opml_convert(self, tasks, top):
        """Print pretty OPML of selected tasks."""

        if tasks is True:
            return
        for task in tasks:
            area = SubElement(top, 'outline')
            area.set('text', task['title'])
            self.opml_convert(task.get('items', []), area)
            task.pop('items', [])
            self.opml_convert(task.get('checklist', []), area)
            task.pop('checklist', [])

    def txt_dumps(self, tasks, indentation="", result=""):
        """Print pretty text version of selected tasks."""

        if tasks is True:
            return result
        for task in tasks:
            title = task["title"]
            context = task.get("project_title", None) or \
                task.get("area_title", None) or \
                task.get("heading_title", None) or \
                task.get("start", None)
            start = task.get("start_date", None)
            details = " | ".join(filter(None, [start, context]))
            result = result + f"{indentation}- {title} ({details})\n"
            result = self.txt_dumps(task.get('items', []), indentation + "  ", result)
            task.pop('items', [])
            result = self.txt_dumps(task.get('checklist', []), indentation + "  ", result)

        return result

    @classmethod
    def print_unimplemented(cls, command):
        """Show warning that method is not yet implemented."""
        print("command '%s' not implemented yet" % command, file=sys.stderr)

    @classmethod
    def get_parser(cls):
        """Create command line argument parser"""
        parser = argparse.ArgumentParser(description="Simple read-only Thing 3 CLI.")

        subparsers = parser.add_subparsers(
            help="", metavar="command", required=True, dest="command"
        )

        ################################
        # Core database methods
        ################################
        subparsers.add_parser("inbox", help="Shows inbox tasks")
        subparsers.add_parser("today", help="Shows todays tasks")
        subparsers.add_parser("upcoming", help="Shows upcoming tasks")
        subparsers.add_parser("anytime", help="Shows anytime tasks")
        subparsers.add_parser("completed", help="Shows completed tasks")
        subparsers.add_parser("canceled", help="Shows canceled tasks")
        subparsers.add_parser("all", help="Shows all tasks")
        subparsers.add_parser("areas", help="Shows all areas")
        subparsers.add_parser("projects", help="Shows all projects")
        subparsers.add_parser("logbook", help="Shows tasks completed today")
        subparsers.add_parser("tags", help="Shows all tags ordered by their usage")
        subparsers.add_parser("deadlines", help="Shows tasks with due dates")

        ################################
        # Additional functions
        ################################
        subparsers.add_parser("feedback", help="Give feedback")
        subparsers.add_parser(
            "search", help="Searches for a specific task"
        ).add_argument("string", help="String to search for")

        ################################
        # To be implemented in things.py
        ################################
        # subparsers.add_parser("repeating", help="Shows all repeating tasks")
        # subparsers.add_parser("trashed", help="Shows trashed tasks")
        # subparsers.add_parser("subtasks", help="Shows all subtasks")
        # subparsers.add_parser("headings", help="Shows headings")

        ################################
        # To be converted from https://github.com/alexanderwillner/things.sh
        ################################
        # subparsers.add_parser("backlog", help="Shows backlog tasks")
        # subparsers.add_parser("empty", help="Shows projects that are empty")
        # subparsers.add_parser("hours", help="Shows hours planned today")
        # subparsers.add_parser("ical", help="Shows tasks ordered by due date as iCal")
        # subparsers.add_parser("lint", help="Shows tasks that float around")
        # subparsers.add_parser(
        #     "mostClosed", help="Shows days when most tasks were closed"
        # )
        # subparsers.add_parser(
        #     "mostCancelled", help="Shows days when most tasks were cancelled"
        # )
        # subparsers.add_parser(
        #     "mostTrashed", help="Shows days when most tasks were trashed"
        # )
        # subparsers.add_parser(
        #     "mostCreated", help="Shows days when most tasks were created"
        # )
        # subparsers.add_parser("mostTasks", help="Shows projects that have most tasks")
        # subparsers.add_parser(
        #     "mostCharacters", help="Shows tasks that have most characters"
        # )
        # subparsers.add_parser("nextish", help="Shows all nextish tasks")
        # subparsers.add_parser("old", help="Shows all old tasks")
        # subparsers.add_parser("schedule", help="Schedules an event using a template")
        # subparsers.add_parser("stat", help="Provides a number of statistics")
        # subparsers.add_parser("statcsv", help="Exports some statistics as CSV")
        # subparsers.add_parser("tag", help="Shows all tasks with the waiting for tag")
        # subparsers.add_parser(
        #     "waiting", help="Shows all tasks with the waiting for tag"
        # )

        ################################
        # To be converted from https://github.com/alexanderwillner/things.sh
        ################################
        # parser.add_argument("-a", "--anonymize",
        #                     action="store_true", default=False,
        #                     help="anonymize output", dest="anonymize")

        parser.add_argument(
            "-o",
            "--opml",
            action="store_true",
            default=False,
            help="output as OPML",
            dest="opml")

        parser.add_argument(
            "-j",
            "--json",
            action="store_true",
            default=False,
            help="output as JSON",
            dest="json",
        )

        parser.add_argument(
            "-c",
            "--csv",
            action="store_true",
            default=False,
            help="output as CSV",
            dest="csv",
        )

        parser.add_argument(
            "-r", "--recursive", help="in-depth output", dest="recursive", default=False, action="store_true"
        )

        parser.add_argument(
            "-d", "--database", help="set path to database", dest="database"
        )

        parser.add_argument(
            "--version",
            "-v",
            action="version",
            version="%(prog)s (version {version})".format(version=__version__),
        )

        argcomplete.autocomplete(parser)

        return parser

    def main(self, args=None):
        """ Main entry point of the app """

        if args is None:
            self.main(ThingsCLI.get_parser().parse_args())
        else:
            command = args.command
            self.print_json = args.json
            self.print_csv = args.csv
            self.print_opml = args.opml
            self.database = (
                args.database if args.database is not None else self.database
            )
            self.recursive = args.recursive
            # self.anonymize = args.anonymize
            # self.things3.anonymize = self.anonymize ## not implemented

            if command == "all":
                inbox = api.inbox(filepath=self.database, include_items=self.recursive)
                today = api.today(filepath=self.database, include_items=self.recursive)
                upcoming = api.upcoming(filepath=self.database, include_items=self.recursive)
                anytime = api.anytime(filepath=self.database, include_items=self.recursive)
                someday = api.someday(filepath=self.database, include_items=self.recursive)
                logbook = api.logbook(filepath=self.database, include_items=self.recursive)
                no_area = api.projects(area=False, filepath=self.database, include_items=self.recursive)
                areas = api.areas(filepath=self.database, include_items=self.recursive)
                structure = [{"title": "Inbox",
                              "items": inbox},
                             {"title": "Today",
                              "items": today},
                             {"title": "Upcoming",
                              "items": upcoming},
                             {"title": "Anytime",
                              "items": anytime},
                             {"title": "Someday",
                              "items": someday},
                             {"title": "Logbook",
                              "items": logbook},
                             {"title": "No Area",
                              "items": no_area},
                             {"title": "Areas",
                              "items": areas}
                             ]
                self.print_tasks(structure)
            elif command == "upcoming":
                result = getattr(api, command)(filepath=self.database,
                                               include_items=self.recursive)
                result.sort(key=lambda task: task["start_date"], reverse=False)
                self.print_tasks(result)
            elif command == "search":
                self.print_tasks(api.search(args.string, filepath=self.database,
                                            include_items=self.recursive))
            elif command == "feedback":  # pragma: no cover
                webbrowser.open("https://github.com/thingsapi/things-cli/issues")
            elif command in dir(api):
                self.print_tasks(
                    getattr(api, command)(filepath=self.database,
                                          include_items=self.recursive))
            else:  # pragma: no cover
                ThingsCLI.print_unimplemented(command)
                sys.exit(3)


def main():
    """Main entry point for CLI installation"""
    ThingsCLI().main()


if __name__ == "__main__":
    main()
