"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnGuardValidator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const os = require("os");
const path = require("path");
const check_1 = require("./check");
const utils_1 = require("./utils");
/**
 * A validation plugin using CFN Guard
 */
class CfnGuardValidator {
    constructor(props = {}) {
        this.rulesPaths = [];
        this.executionConfig = [];
        this.name = 'cdk-validator-cfnguard';
        this.disabledRules = props.disabledRules ?? [];
        if (props.controlTowerRulesEnabled ?? true) {
            this.rulesPaths.push(path.join(__dirname, '..', 'rules/control-tower/cfn-guard'));
        }
        this.rulesPaths.push(...props.rules ?? []);
        const osPlatform = os.platform();
        // guard calls it ubuntu but seems to apply to all linux
        // https://github.com/aws-cloudformation/cloudformation-guard/blob/184002cdfc0ae9e29c61995aae41b7d1f1d3b26c/install-guard.sh#L43-L46
        const platform = osPlatform === 'linux'
            ? 'ubuntu'
            : osPlatform === 'darwin' ? 'macos' : undefined;
        if (!platform) {
            throw new Error(`${os.platform()} not supported, must be either 'darwin' or 'linux'`);
        }
        this.guard = path.join(__dirname, '..', 'bin', platform, 'cfn-guard');
    }
    /**
     * This is (hopefully) a temporary solution to https://github.com/aws-cloudformation/cloudformation-guard/issues/180
     * Rather than try and parse the output and split out the JSON entries we'll just
     * invoke guard separately for each rule.
     */
    generateGuardExecutionConfig(filePath, templatePaths) {
        const stat = fs.statSync(filePath);
        if (stat.isDirectory()) {
            const dir = fs.readdirSync(filePath);
            dir.forEach(d => this.generateGuardExecutionConfig(path.join(filePath, d), templatePaths));
        }
        else {
            templatePaths.forEach(template => {
                if (!this.disabledRules.includes(path.parse(filePath).name)) {
                    this.executionConfig.push({ rulePath: filePath, templatePath: template });
                }
            });
        }
    }
    validate(context) {
        const templatePaths = context.templatePaths;
        this.rulesPaths.forEach(rule => this.generateGuardExecutionConfig(rule, templatePaths));
        const result = this.executionConfig.reduce((acc, config) => {
            const report = this.execGuard(config);
            return {
                violations: [...acc.violations, ...report.violations],
                success: acc.success === false ? false : report.success,
            };
        }, { violations: [], success: true });
        return {
            ...result,
        };
    }
    execGuard(config) {
        const flags = [
            'validate',
            '--rules',
            config.rulePath,
            '--data',
            config.templatePath,
            '--output-format',
            'json',
            '--show-summary',
            'none',
        ];
        const violations = [];
        let success;
        try {
            const result = (0, utils_1.exec)([this.guard, ...flags], {
                json: true,
            });
            const guardResult = JSON.parse(JSON.stringify(result), reviver);
            if (!guardResult.not_compliant || guardResult.not_compliant.length === 0) {
                return { success: true, violations: [] };
            }
            success = false;
            guardResult.not_compliant.forEach((check) => {
                const violationCheck = new check_1.ViolationCheck(check, config.templatePath, config.rulePath);
                const violation = violationCheck.processCheck();
                violations.push(...violation);
            });
        }
        catch (e) {
            success = false;
            throw new Error(`
        CfnGuardValidator plugin failed processing cfn-guard results.
        Please create an issue https://github.com/cdklabs/cdk-validator-cfnguard/issues/new
        Rule: ${path.basename(config.rulePath)}
        Error: ${e}`);
        }
        return {
            success,
            violations: violations,
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
CfnGuardValidator[_a] = { fqn: "@cdklabs/cdk-validator-cfnguard.CfnGuardValidator", version: "0.0.44" };
exports.CfnGuardValidator = CfnGuardValidator;
/**
 * Guard does not have a standard JSON schema and the schema
 * that is returned can be dependent on the type of rule or type
 * of check that was executed. The results are very much an attempt to
 * display the internals of guard to the user. Trying to make sense of that
 * can be difficult.
 *
 * The result structure can depend on the way that the rule was written. For example
 * I could write a rule like this:
 *
 *     rule MY_RULE {
 *       # This is a "check" and is a `Clause` type check
 *       Properties.SomeProp == true
 *     }
 *
 * Or I could write a rule like this:
 *
 *     rule MY_RULE {
 *       #  This is a "check" and is a `Rule` type check
 *       check(Properties)
 *     }
 *     rule check(properties) {
 *       properties.SomeProp == true
 *     }
 *
 * Both of the above examples are checking the same thing
 * but the schema that is returned is different because the
 * way the rule was written is different
 *
 * This reviver function is executed bottom up and is essentially
 * creating a new object with a well known schema that the rest of the
 * plugin can work with. It looks for certain fields that always appear in
 * the guard results, but appear in different locations. It finds those fields
 * and then pulls them up the object, dropping any fields that we don't
 * care about. For example guard may return
 *
 * {
 *   Clause: {
 *     Unary: {
 *       check: {
 *         UnResolved: {
 *           value: {
 *             traversed_to: {...} // we only care about this!!!
 *           }
 *         }
 *       }
 *     }
 *   }
 * }
 *
 * Or it may return
 *
 * {
 *   Rule: {
 *     checks: [{
 *       Block: {
 *         unresolved: {
 *           traversed_to: {...} // we only care about this!!!
 *         }
 *       }
 *     }]
 *   }
 * }
 *
 * In the above example we only care about the 'traversed_to' field,
 * so this reviver function will grab that field and pull it up the object, dropping
 * the fields we don't care about, ending with something like
 * {
 *   checks: [{
 *     resolved: false,
 *     traversed: {...}
 *   }]
 * }
 *
 */
function reviver(key, value) {
    if (key === 'not_compliant') {
        // not_compliant can sometimes be an empty object (but not an Array), so we
        // process this value before diving into other object values to ensure this
        // one is always made into an Array
        return Object.values(value).map((v) => v.Rule);
    }
    else if (value !== null && typeof value === 'object' && !Array.isArray(value)) {
        return extractNestedObject(value);
    }
    else if (key === 'checks' && Array.isArray(value)) {
        return extractNestedChecks(value.flatMap(v => v));
    }
    return value;
}
/**
 * Extract a nested 'checks' object. This also handles checks
 * nested within checks. It will grab the checks at the level below
 * and pull it up to the next level.
 */
function extractNestedChecks(checks) {
    const containsNestedChecks = checks.some(check => Object.values(check).some((value) => {
        return typeof value === 'object' && value.hasOwnProperty('checks');
    }));
    if (containsNestedChecks) {
        return checks.flatMap(check => {
            if (Object.keys(check).includes('traversed')) {
                return check;
            }
            return Object.values(check).flatMap((checkValue) => {
                return Object.values(checkValue.checks ?? checkValue).flatMap((nestedCheckValue) => {
                    return {
                        ...nestedCheckValue,
                        name: checkValue.name,
                        messages: checkValue.messages ?? nestedCheckValue.messages,
                    };
                });
            });
        });
    }
    return checks.flatMap(check => {
        if (Object.keys(check).includes('traversed')) {
            return check;
        }
        return Object.values(check);
    });
}
/**
 * Extract a nested object and pull it up a level
 */
function extractNestedObject(object) {
    let newObject = object;
    Object.entries(object).forEach(([level1NestedKey, level1NestedValue]) => {
        const nestedValue = level1NestedValue;
        switch (level1NestedKey.toLowerCase()) {
            // this should always be found earlier than the rest since it appears
            // within the 'unresolved' and 'resolved' objects. The object
            // is slightly different for each case so here we create
            // a new object with the key 'traversed' with a consistent value
            case 'traversed_to':
                newObject = {
                    traversed: {
                        to: {
                            path: nestedValue.path,
                            value: nestedValue.value,
                        },
                        from: nestedValue.from ? {
                            path: nestedValue.from.path,
                            value: undefined,
                        } : undefined,
                    },
                    messages: nestedValue.messages,
                };
                break;
            // This should be found in the "second" pass after the above
            // 'traversed_to' case has been executed. We take the new
            // object that was created in the `traversed_to` case and
            // a couple other fields, dropping the rest that we don't care about
            case 'unresolved':
                newObject = {
                    resolved: false,
                    traversed: nestedValue.traversed,
                    messages: nestedValue.messages ?? object.messages,
                };
                break;
            // This should be found in the "second" pass after the above
            // 'traversed_to' case has been executed. We take the new
            // object that was created in the `traversed_to` case and
            // a couple other fields, dropping the rest that we don't care about
            // A check can either be resolved or unresolved
            case 'resolved':
            case 'inresolved':
                if ('from' in nestedValue) {
                    newObject = {
                        resolved: true,
                        traversed: {
                            from: nestedValue.from,
                            to: {
                                path: nestedValue.from.path,
                                value: nestedValue.to.value,
                            },
                        },
                        messages: nestedValue.messages,
                    };
                }
                else if ('value' in nestedValue) {
                    newObject = {
                        resolved: true,
                        traversed: {
                            from: nestedValue.value,
                            to: {
                                path: nestedValue.value.path,
                                value: nestedValue.value.value,
                            },
                        },
                        messages: nestedValue.messages,
                    };
                }
                break;
        }
        // this check will be evaluated _after_ the 'traversed_to' check and _before_ the 'resolved'
        // and 'unresolved' checks above. There may be a case where 'traversed' is nested 2 (or 3 or 4) below
        // 'unresolved' or 'resolved' and this will keep pulling it up until it is just one below
        // and the above checks can work
        if (level1NestedValue !== null && typeof level1NestedValue === 'object' && !Array.isArray(level1NestedValue)) {
            Object.entries(level1NestedValue).forEach(([level2NestedKey, level2NestedValue]) => {
                switch (level2NestedKey.toLowerCase()) {
                    case 'traversed':
                        newObject = {
                            traversed: nestedValue.traversed,
                            resolved: nestedValue.resolved,
                            messages: nestedValue.messages ?? level2NestedValue.messages ?? object.messages,
                        };
                        break;
                }
            });
        }
    });
    return newObject;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGx1Z2luLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL3BsdWdpbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLHlCQUF5QjtBQUN6Qix5QkFBeUI7QUFDekIsNkJBQTZCO0FBTzdCLG1DQUFzRDtBQUN0RCxtQ0FBK0I7QUErQy9COztHQUVHO0FBQ0gsTUFBYSxpQkFBaUI7SUFPNUIsWUFBWSxRQUFnQyxFQUFFO1FBTDdCLGVBQVUsR0FBYSxFQUFFLENBQUM7UUFHMUIsb0JBQWUsR0FBMkIsRUFBRSxDQUFDO1FBRzVELElBQUksQ0FBQyxJQUFJLEdBQUcsd0JBQXdCLENBQUM7UUFDckMsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxJQUFJLEVBQUUsQ0FBQztRQUMvQyxJQUFJLEtBQUssQ0FBQyx3QkFBd0IsSUFBSSxJQUFJLEVBQUU7WUFDMUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsSUFBSSxFQUFFLCtCQUErQixDQUFDLENBQUMsQ0FBQztTQUNuRjtRQUNELElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLEdBQUcsS0FBSyxDQUFDLEtBQUssSUFBSSxFQUFFLENBQUMsQ0FBQztRQUMzQyxNQUFNLFVBQVUsR0FBRyxFQUFFLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDakMsd0RBQXdEO1FBQ3hELG9JQUFvSTtRQUNwSSxNQUFNLFFBQVEsR0FBRyxVQUFVLEtBQUssT0FBTztZQUNyQyxDQUFDLENBQUMsUUFBUTtZQUNWLENBQUMsQ0FBQyxVQUFVLEtBQUssUUFBUSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUVsRCxJQUFJLENBQUMsUUFBUSxFQUFFO1lBQ2IsTUFBTSxJQUFJLEtBQUssQ0FBQyxHQUFHLEVBQUUsQ0FBQyxRQUFRLEVBQUUsb0RBQW9ELENBQUMsQ0FBQztTQUN2RjtRQUNELElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsSUFBSSxFQUFFLEtBQUssRUFBRSxRQUFRLEVBQUUsV0FBVyxDQUFDLENBQUM7SUFDeEUsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyw0QkFBNEIsQ0FBQyxRQUFnQixFQUFFLGFBQXVCO1FBQzVFLE1BQU0sSUFBSSxHQUFHLEVBQUUsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDbkMsSUFBSSxJQUFJLENBQUMsV0FBVyxFQUFFLEVBQUU7WUFDdEIsTUFBTSxHQUFHLEdBQUcsRUFBRSxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUNyQyxHQUFHLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLDRCQUE0QixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQyxFQUFFLGFBQWEsQ0FBQyxDQUFDLENBQUM7U0FDNUY7YUFBTTtZQUNMLGFBQWEsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLEVBQUU7Z0JBQy9CLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFO29CQUMzRCxJQUFJLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxFQUFFLFFBQVEsRUFBRSxRQUFRLEVBQUUsWUFBWSxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7aUJBQzNFO1lBQ0gsQ0FBQyxDQUFDLENBQUM7U0FDSjtJQUNILENBQUM7SUFFRCxRQUFRLENBQUMsT0FBc0M7UUFDN0MsTUFBTSxhQUFhLEdBQUcsT0FBTyxDQUFDLGFBQWEsQ0FBQztRQUM1QyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyw0QkFBNEIsQ0FBQyxJQUFJLEVBQUUsYUFBYSxDQUFDLENBQUMsQ0FBQztRQUN4RixNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEdBQUcsRUFBRSxNQUFNLEVBQUUsRUFBRTtZQUN6RCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQ3RDLE9BQU87Z0JBQ0wsVUFBVSxFQUFFLENBQUMsR0FBRyxHQUFHLENBQUMsVUFBVSxFQUFFLEdBQUcsTUFBTSxDQUFDLFVBQVUsQ0FBQztnQkFDckQsT0FBTyxFQUFFLEdBQUcsQ0FBQyxPQUFPLEtBQUssS0FBSyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxPQUFPO2FBQ3hELENBQUM7UUFDSixDQUFDLEVBQUUsRUFBRSxVQUFVLEVBQUUsRUFBRSxFQUFFLE9BQU8sRUFBRSxJQUFJLEVBQXVFLENBQUMsQ0FBQztRQUMzRyxPQUFPO1lBQ0wsR0FBRyxNQUFNO1NBQ1YsQ0FBQztJQUNKLENBQUM7SUFFTyxTQUFTLENBQUMsTUFBNEI7UUFDNUMsTUFBTSxLQUFLLEdBQUc7WUFDWixVQUFVO1lBQ1YsU0FBUztZQUNULE1BQU0sQ0FBQyxRQUFRO1lBQ2YsUUFBUTtZQUNSLE1BQU0sQ0FBQyxZQUFZO1lBQ25CLGlCQUFpQjtZQUNqQixNQUFNO1lBQ04sZ0JBQWdCO1lBQ2hCLE1BQU07U0FDUCxDQUFDO1FBQ0YsTUFBTSxVQUFVLEdBQTJCLEVBQUUsQ0FBQztRQUM5QyxJQUFJLE9BQWdCLENBQUM7UUFDckIsSUFBSTtZQUNGLE1BQU0sTUFBTSxHQUFHLElBQUEsWUFBSSxFQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFO2dCQUMxQyxJQUFJLEVBQUUsSUFBSTthQUNYLENBQUMsQ0FBQztZQUNILE1BQU0sV0FBVyxHQUFnQixJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDN0UsSUFBSSxDQUFDLFdBQVcsQ0FBQyxhQUFhLElBQUksV0FBVyxDQUFDLGFBQWEsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO2dCQUN4RSxPQUFPLEVBQUUsT0FBTyxFQUFFLElBQUksRUFBRSxVQUFVLEVBQUUsRUFBRSxFQUFFLENBQUM7YUFDMUM7WUFDRCxPQUFPLEdBQUcsS0FBSyxDQUFDO1lBQ2hCLFdBQVcsQ0FBQyxhQUFhLENBQUMsT0FBTyxDQUFDLENBQUMsS0FBSyxFQUFFLEVBQUU7Z0JBQzFDLE1BQU0sY0FBYyxHQUFHLElBQUksc0JBQWMsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLFlBQVksRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUM7Z0JBQ3ZGLE1BQU0sU0FBUyxHQUFHLGNBQWMsQ0FBQyxZQUFZLEVBQUUsQ0FBQztnQkFDaEQsVUFBVSxDQUFDLElBQUksQ0FBQyxHQUFHLFNBQVMsQ0FBQyxDQUFDO1lBQ2hDLENBQUMsQ0FBQyxDQUFDO1NBQ0o7UUFBQyxPQUFPLENBQUMsRUFBRTtZQUNWLE9BQU8sR0FBRyxLQUFLLENBQUM7WUFDaEIsTUFBTSxJQUFJLEtBQUssQ0FBQzs7O2dCQUdOLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQztpQkFDN0IsQ0FBQyxFQUFFLENBQUMsQ0FBQztTQUNqQjtRQUNELE9BQU87WUFDTCxPQUFPO1lBQ1AsVUFBVSxFQUFFLFVBQVU7U0FDdkIsQ0FBQztJQUNKLENBQUM7Ozs7QUFyR1UsOENBQWlCO0FBeUc5Qjs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0EwRUc7QUFDSCxTQUFTLE9BQU8sQ0FBQyxHQUFXLEVBQUUsS0FBVTtJQUN0QyxJQUFJLEdBQUcsS0FBSyxlQUFlLEVBQUU7UUFDM0IsMkVBQTJFO1FBQzNFLDJFQUEyRTtRQUMzRSxtQ0FBbUM7UUFDbkMsT0FBTyxNQUFNLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQU0sRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDO0tBQ3JEO1NBQU0sSUFBSSxLQUFLLEtBQUssSUFBSSxJQUFJLE9BQU8sS0FBSyxLQUFLLFFBQVEsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUU7UUFDL0UsT0FBTyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztLQUNuQztTQUFNLElBQUksR0FBRyxLQUFLLFFBQVEsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFO1FBQ25ELE9BQU8sbUJBQW1CLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7S0FDbkQ7SUFDRCxPQUFPLEtBQUssQ0FBQztBQUNmLENBQUM7QUFFRDs7OztHQUlHO0FBQ0gsU0FBUyxtQkFBbUIsQ0FBQyxNQUFhO0lBQ3hDLE1BQU0sb0JBQW9CLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsS0FBVSxFQUFFLEVBQUU7UUFDekYsT0FBTyxPQUFPLEtBQUssS0FBSyxRQUFRLElBQUksS0FBSyxDQUFDLGNBQWMsQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUNyRSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ0osSUFBSSxvQkFBb0IsRUFBRTtRQUN4QixPQUFPLE1BQU0sQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDNUIsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsRUFBRTtnQkFDNUMsT0FBTyxLQUFLLENBQUM7YUFDZDtZQUNELE9BQU8sTUFBTSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxVQUFlLEVBQUUsRUFBRTtnQkFDdEQsT0FBTyxNQUFNLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxNQUFNLElBQUksVUFBVSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsZ0JBQXFCLEVBQUUsRUFBRTtvQkFDdEYsT0FBTzt3QkFDTCxHQUFHLGdCQUFnQjt3QkFDbkIsSUFBSSxFQUFFLFVBQVUsQ0FBQyxJQUFJO3dCQUNyQixRQUFRLEVBQUUsVUFBVSxDQUFDLFFBQVEsSUFBSSxnQkFBZ0IsQ0FBQyxRQUFRO3FCQUMzRCxDQUFDO2dCQUNKLENBQUMsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztLQUNKO0lBQ0QsT0FBTyxNQUFNLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFO1FBQzVCLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLEVBQUU7WUFDNUMsT0FBTyxLQUFLLENBQUM7U0FDZDtRQUNELE9BQU8sTUFBTSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUM5QixDQUFDLENBQUMsQ0FBQztBQUNMLENBQUM7QUFFRDs7R0FFRztBQUNILFNBQVMsbUJBQW1CLENBQUMsTUFBVztJQUN0QyxJQUFJLFNBQVMsR0FBRyxNQUFNLENBQUM7SUFDdkIsTUFBTSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLGVBQWUsRUFBRSxpQkFBaUIsQ0FBQyxFQUFFLEVBQUU7UUFDdEUsTUFBTSxXQUFXLEdBQUcsaUJBQXdCLENBQUM7UUFDN0MsUUFBUSxlQUFlLENBQUMsV0FBVyxFQUFFLEVBQUU7WUFDckMscUVBQXFFO1lBQ3JFLDZEQUE2RDtZQUM3RCx3REFBd0Q7WUFDeEQsZ0VBQWdFO1lBQ2hFLEtBQUssY0FBYztnQkFDakIsU0FBUyxHQUFHO29CQUNWLFNBQVMsRUFBRTt3QkFDVCxFQUFFLEVBQUU7NEJBQ0YsSUFBSSxFQUFFLFdBQVcsQ0FBQyxJQUFJOzRCQUN0QixLQUFLLEVBQUUsV0FBVyxDQUFDLEtBQUs7eUJBQ3pCO3dCQUNELElBQUksRUFBRSxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQzs0QkFDdkIsSUFBSSxFQUFFLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSTs0QkFDM0IsS0FBSyxFQUFFLFNBQVM7eUJBQ2pCLENBQUMsQ0FBQyxDQUFDLFNBQVM7cUJBQ2Q7b0JBQ0QsUUFBUSxFQUFFLFdBQVcsQ0FBQyxRQUFRO2lCQUMvQixDQUFDO2dCQUNGLE1BQU07WUFDUiw0REFBNEQ7WUFDNUQseURBQXlEO1lBQ3pELHlEQUF5RDtZQUN6RCxvRUFBb0U7WUFDcEUsS0FBSyxZQUFZO2dCQUNmLFNBQVMsR0FBRztvQkFDVixRQUFRLEVBQUUsS0FBSztvQkFDZixTQUFTLEVBQUUsV0FBVyxDQUFDLFNBQVM7b0JBQ2hDLFFBQVEsRUFBRSxXQUFXLENBQUMsUUFBUSxJQUFJLE1BQU0sQ0FBQyxRQUFRO2lCQUNsRCxDQUFDO2dCQUNGLE1BQU07WUFDUiw0REFBNEQ7WUFDNUQseURBQXlEO1lBQ3pELHlEQUF5RDtZQUN6RCxvRUFBb0U7WUFDcEUsK0NBQStDO1lBQy9DLEtBQUssVUFBVSxDQUFDO1lBQ2hCLEtBQUssWUFBWTtnQkFDZixJQUFJLE1BQU0sSUFBSSxXQUFXLEVBQUU7b0JBQ3pCLFNBQVMsR0FBRzt3QkFDVixRQUFRLEVBQUUsSUFBSTt3QkFDZCxTQUFTLEVBQUU7NEJBQ1QsSUFBSSxFQUFFLFdBQVcsQ0FBQyxJQUFJOzRCQUN0QixFQUFFLEVBQUU7Z0NBQ0YsSUFBSSxFQUFFLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSTtnQ0FDM0IsS0FBSyxFQUFFLFdBQVcsQ0FBQyxFQUFFLENBQUMsS0FBSzs2QkFDNUI7eUJBQ0Y7d0JBQ0QsUUFBUSxFQUFFLFdBQVcsQ0FBQyxRQUFRO3FCQUMvQixDQUFDO2lCQUNIO3FCQUFNLElBQUksT0FBTyxJQUFJLFdBQVcsRUFBRTtvQkFDakMsU0FBUyxHQUFHO3dCQUNWLFFBQVEsRUFBRSxJQUFJO3dCQUNkLFNBQVMsRUFBRTs0QkFDVCxJQUFJLEVBQUUsV0FBVyxDQUFDLEtBQUs7NEJBQ3ZCLEVBQUUsRUFBRTtnQ0FDRixJQUFJLEVBQUUsV0FBVyxDQUFDLEtBQUssQ0FBQyxJQUFJO2dDQUM1QixLQUFLLEVBQUUsV0FBVyxDQUFDLEtBQUssQ0FBQyxLQUFLOzZCQUMvQjt5QkFDRjt3QkFDRCxRQUFRLEVBQUUsV0FBVyxDQUFDLFFBQVE7cUJBQy9CLENBQUM7aUJBRUg7Z0JBQ0QsTUFBTTtTQUNUO1FBQ0QsNEZBQTRGO1FBQzVGLHFHQUFxRztRQUNyRyx5RkFBeUY7UUFDekYsZ0NBQWdDO1FBQ2hDLElBQUksaUJBQWlCLEtBQUssSUFBSSxJQUFJLE9BQU8saUJBQWlCLEtBQUssUUFBUSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFO1lBQzVHLE1BQU0sQ0FBQyxPQUFPLENBQUUsaUJBQTRCLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLGVBQWUsRUFBRSxpQkFBaUIsQ0FBQyxFQUFFLEVBQUU7Z0JBQzdGLFFBQVEsZUFBZSxDQUFDLFdBQVcsRUFBRSxFQUFFO29CQUNyQyxLQUFLLFdBQVc7d0JBQ2QsU0FBUyxHQUFHOzRCQUNWLFNBQVMsRUFBRSxXQUFXLENBQUMsU0FBUzs0QkFDaEMsUUFBUSxFQUFFLFdBQVcsQ0FBQyxRQUFROzRCQUM5QixRQUFRLEVBQUUsV0FBVyxDQUFDLFFBQVEsSUFBSSxpQkFBaUIsQ0FBQyxRQUFRLElBQUksTUFBTSxDQUFDLFFBQVE7eUJBQ2hGLENBQUM7d0JBQ0YsTUFBTTtpQkFDVDtZQUNILENBQUMsQ0FBQyxDQUFDO1NBQ0o7SUFDSCxDQUFDLENBQUMsQ0FBQztJQUNILE9BQU8sU0FBUyxDQUFDO0FBQ25CLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBmcyBmcm9tICdmcyc7XG5pbXBvcnQgKiBhcyBvcyBmcm9tICdvcyc7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0IHtcbiAgSVBvbGljeVZhbGlkYXRpb25QbHVnaW5CZXRhMSxcbiAgSVBvbGljeVZhbGlkYXRpb25Db250ZXh0QmV0YTEsXG4gIFBvbGljeVZpb2xhdGlvbkJldGExLFxuICBQb2xpY3lWYWxpZGF0aW9uUGx1Z2luUmVwb3J0QmV0YTEsXG59IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IFZpb2xhdGlvbkNoZWNrLCBHdWFyZFJlc3VsdCB9IGZyb20gJy4vY2hlY2snO1xuaW1wb3J0IHsgZXhlYyB9IGZyb20gJy4vdXRpbHMnO1xuXG5leHBvcnQgaW50ZXJmYWNlIENmbkd1YXJkVmFsaWRhdG9yUHJvcHMge1xuICAvKipcbiAgICogRW5hYmxlIHRoZSBkZWZhdWx0IENvbnRyb2wgVG93ZXIgR3VhcmQgcnVsZXNcbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgY29udHJvbFRvd2VyUnVsZXNFbmFibGVkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogTGlzdCBvZiBydWxlIG5hbWVzIHRvIGRpc2FibGVcbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBydWxlcyBhcmUgZGlzYWJsZWRcbiAgICovXG4gIHJlYWRvbmx5IGRpc2FibGVkUnVsZXM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogTG9jYWwgZmlsZSBwYXRocyB0byBlaXRoZXIgYSBkaXJlY3RvcnkgY29udGFpbmluZ1xuICAgKiBndWFyZCBydWxlcywgb3IgdG8gYW4gaW5kaXZpZHVhbCBndWFyZCBydWxlIGZpbGVcbiAgICpcbiAgICogSWYgdGhlIHBhdGggaXMgdG8gYSBkaXJlY3RvcnkgdGhlbiB0aGUgZGlyZWN0b3J5IG11c3RcbiAgICogb25seSBjb250YWluIGd1YXJkIHJ1bGUgYW5kIHRoZSBwbHVnaW4gd2lsbCB1c2VcbiAgICogYWxsIHRoZSBydWxlcyBpbiB0aGUgZGlyZWN0b3J5XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gbG9jYWwgcnVsZXMgd2lsbCBiZSB1c2VkXG4gICAqL1xuICByZWFkb25seSBydWxlcz86IHN0cmluZ1tdO1xufVxuXG4vKipcbiAqIENvbmZpZ3VyYXRpb24gZm9yIHJ1bm5pbmcgZ3VhcmQgd2l0aFxuICogYSBzaW5nbGUgcnVsZSBmaWxlIGFnYWluc3QgYSBzaW5nbGUgdGVtcGxhdGVcbiAqL1xuaW50ZXJmYWNlIEd1YXJkRXhlY3V0aW9uQ29uZmlnIHtcbiAgLyoqXG4gICAqIFRoZSBwYXRoIHRvIHRoZSBDbG91ZEZvcm1hdGlvbiB0ZW1wbGF0ZSB0aGF0IHNob3VsZFxuICAgKiBiZSB2YWxpZGF0ZWRcbiAgICovXG4gIHJlYWRvbmx5IHRlbXBsYXRlUGF0aDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcGF0aCB0byB0aGUgZ3VhcmQgcnVsZSBmaWxlXG4gICAqL1xuICByZWFkb25seSBydWxlUGF0aDogc3RyaW5nO1xufVxuXG4vKipcbiAqIEEgdmFsaWRhdGlvbiBwbHVnaW4gdXNpbmcgQ0ZOIEd1YXJkXG4gKi9cbmV4cG9ydCBjbGFzcyBDZm5HdWFyZFZhbGlkYXRvciBpbXBsZW1lbnRzIElQb2xpY3lWYWxpZGF0aW9uUGx1Z2luQmV0YTEge1xuICBwdWJsaWMgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IHJ1bGVzUGF0aHM6IHN0cmluZ1tdID0gW107XG4gIHByaXZhdGUgcmVhZG9ubHkgZ3VhcmQ6IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSBkaXNhYmxlZFJ1bGVzOiBzdHJpbmdbXTtcbiAgcHJpdmF0ZSByZWFkb25seSBleGVjdXRpb25Db25maWc6IEd1YXJkRXhlY3V0aW9uQ29uZmlnW10gPSBbXTtcblxuICBjb25zdHJ1Y3Rvcihwcm9wczogQ2ZuR3VhcmRWYWxpZGF0b3JQcm9wcyA9IHt9KSB7XG4gICAgdGhpcy5uYW1lID0gJ2Nkay12YWxpZGF0b3ItY2ZuZ3VhcmQnO1xuICAgIHRoaXMuZGlzYWJsZWRSdWxlcyA9IHByb3BzLmRpc2FibGVkUnVsZXMgPz8gW107XG4gICAgaWYgKHByb3BzLmNvbnRyb2xUb3dlclJ1bGVzRW5hYmxlZCA/PyB0cnVlKSB7XG4gICAgICB0aGlzLnJ1bGVzUGF0aHMucHVzaChwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4nLCAncnVsZXMvY29udHJvbC10b3dlci9jZm4tZ3VhcmQnKSk7XG4gICAgfVxuICAgIHRoaXMucnVsZXNQYXRocy5wdXNoKC4uLnByb3BzLnJ1bGVzID8/IFtdKTtcbiAgICBjb25zdCBvc1BsYXRmb3JtID0gb3MucGxhdGZvcm0oKTtcbiAgICAvLyBndWFyZCBjYWxscyBpdCB1YnVudHUgYnV0IHNlZW1zIHRvIGFwcGx5IHRvIGFsbCBsaW51eFxuICAgIC8vIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MtY2xvdWRmb3JtYXRpb24vY2xvdWRmb3JtYXRpb24tZ3VhcmQvYmxvYi8xODQwMDJjZGZjMGFlOWUyOWM2MTk5NWFhZTQxYjdkMWYxZDNiMjZjL2luc3RhbGwtZ3VhcmQuc2gjTDQzLUw0NlxuICAgIGNvbnN0IHBsYXRmb3JtID0gb3NQbGF0Zm9ybSA9PT0gJ2xpbnV4J1xuICAgICAgPyAndWJ1bnR1J1xuICAgICAgOiBvc1BsYXRmb3JtID09PSAnZGFyd2luJyA/ICdtYWNvcycgOiB1bmRlZmluZWQ7XG5cbiAgICBpZiAoIXBsYXRmb3JtKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYCR7b3MucGxhdGZvcm0oKX0gbm90IHN1cHBvcnRlZCwgbXVzdCBiZSBlaXRoZXIgJ2Rhcndpbicgb3IgJ2xpbnV4J2ApO1xuICAgIH1cbiAgICB0aGlzLmd1YXJkID0gcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uJywgJ2JpbicsIHBsYXRmb3JtLCAnY2ZuLWd1YXJkJyk7XG4gIH1cblxuICAvKipcbiAgICogVGhpcyBpcyAoaG9wZWZ1bGx5KSBhIHRlbXBvcmFyeSBzb2x1dGlvbiB0byBodHRwczovL2dpdGh1Yi5jb20vYXdzLWNsb3VkZm9ybWF0aW9uL2Nsb3VkZm9ybWF0aW9uLWd1YXJkL2lzc3Vlcy8xODBcbiAgICogUmF0aGVyIHRoYW4gdHJ5IGFuZCBwYXJzZSB0aGUgb3V0cHV0IGFuZCBzcGxpdCBvdXQgdGhlIEpTT04gZW50cmllcyB3ZSdsbCBqdXN0XG4gICAqIGludm9rZSBndWFyZCBzZXBhcmF0ZWx5IGZvciBlYWNoIHJ1bGUuXG4gICAqL1xuICBwcml2YXRlIGdlbmVyYXRlR3VhcmRFeGVjdXRpb25Db25maWcoZmlsZVBhdGg6IHN0cmluZywgdGVtcGxhdGVQYXRoczogc3RyaW5nW10pOiB2b2lkIHtcbiAgICBjb25zdCBzdGF0ID0gZnMuc3RhdFN5bmMoZmlsZVBhdGgpO1xuICAgIGlmIChzdGF0LmlzRGlyZWN0b3J5KCkpIHtcbiAgICAgIGNvbnN0IGRpciA9IGZzLnJlYWRkaXJTeW5jKGZpbGVQYXRoKTtcbiAgICAgIGRpci5mb3JFYWNoKGQgPT4gdGhpcy5nZW5lcmF0ZUd1YXJkRXhlY3V0aW9uQ29uZmlnKHBhdGguam9pbihmaWxlUGF0aCwgZCksIHRlbXBsYXRlUGF0aHMpKTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGVtcGxhdGVQYXRocy5mb3JFYWNoKHRlbXBsYXRlID0+IHtcbiAgICAgICAgaWYgKCF0aGlzLmRpc2FibGVkUnVsZXMuaW5jbHVkZXMocGF0aC5wYXJzZShmaWxlUGF0aCkubmFtZSkpIHtcbiAgICAgICAgICB0aGlzLmV4ZWN1dGlvbkNvbmZpZy5wdXNoKHsgcnVsZVBhdGg6IGZpbGVQYXRoLCB0ZW1wbGF0ZVBhdGg6IHRlbXBsYXRlIH0pO1xuICAgICAgICB9XG4gICAgICB9KTtcbiAgICB9XG4gIH1cblxuICB2YWxpZGF0ZShjb250ZXh0OiBJUG9saWN5VmFsaWRhdGlvbkNvbnRleHRCZXRhMSk6IFBvbGljeVZhbGlkYXRpb25QbHVnaW5SZXBvcnRCZXRhMSB7XG4gICAgY29uc3QgdGVtcGxhdGVQYXRocyA9IGNvbnRleHQudGVtcGxhdGVQYXRocztcbiAgICB0aGlzLnJ1bGVzUGF0aHMuZm9yRWFjaChydWxlID0+IHRoaXMuZ2VuZXJhdGVHdWFyZEV4ZWN1dGlvbkNvbmZpZyhydWxlLCB0ZW1wbGF0ZVBhdGhzKSk7XG4gICAgY29uc3QgcmVzdWx0ID0gdGhpcy5leGVjdXRpb25Db25maWcucmVkdWNlKChhY2MsIGNvbmZpZykgPT4ge1xuICAgICAgY29uc3QgcmVwb3J0ID0gdGhpcy5leGVjR3VhcmQoY29uZmlnKTtcbiAgICAgIHJldHVybiB7XG4gICAgICAgIHZpb2xhdGlvbnM6IFsuLi5hY2MudmlvbGF0aW9ucywgLi4ucmVwb3J0LnZpb2xhdGlvbnNdLFxuICAgICAgICBzdWNjZXNzOiBhY2Muc3VjY2VzcyA9PT0gZmFsc2UgPyBmYWxzZSA6IHJlcG9ydC5zdWNjZXNzLFxuICAgICAgfTtcbiAgICB9LCB7IHZpb2xhdGlvbnM6IFtdLCBzdWNjZXNzOiB0cnVlIH0gYXMgUGljazxQb2xpY3lWYWxpZGF0aW9uUGx1Z2luUmVwb3J0QmV0YTEsICdzdWNjZXNzJyB8ICd2aW9sYXRpb25zJz4pO1xuICAgIHJldHVybiB7XG4gICAgICAuLi5yZXN1bHQsXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgZXhlY0d1YXJkKGNvbmZpZzogR3VhcmRFeGVjdXRpb25Db25maWcpOiBQaWNrPFBvbGljeVZhbGlkYXRpb25QbHVnaW5SZXBvcnRCZXRhMSwgJ3N1Y2Nlc3MnIHwgJ3Zpb2xhdGlvbnMnPiB7XG4gICAgY29uc3QgZmxhZ3MgPSBbXG4gICAgICAndmFsaWRhdGUnLFxuICAgICAgJy0tcnVsZXMnLFxuICAgICAgY29uZmlnLnJ1bGVQYXRoLFxuICAgICAgJy0tZGF0YScsXG4gICAgICBjb25maWcudGVtcGxhdGVQYXRoLFxuICAgICAgJy0tb3V0cHV0LWZvcm1hdCcsXG4gICAgICAnanNvbicsXG4gICAgICAnLS1zaG93LXN1bW1hcnknLFxuICAgICAgJ25vbmUnLFxuICAgIF07XG4gICAgY29uc3QgdmlvbGF0aW9uczogUG9saWN5VmlvbGF0aW9uQmV0YTFbXSA9IFtdO1xuICAgIGxldCBzdWNjZXNzOiBib29sZWFuO1xuICAgIHRyeSB7XG4gICAgICBjb25zdCByZXN1bHQgPSBleGVjKFt0aGlzLmd1YXJkLCAuLi5mbGFnc10sIHtcbiAgICAgICAganNvbjogdHJ1ZSxcbiAgICAgIH0pO1xuICAgICAgY29uc3QgZ3VhcmRSZXN1bHQ6IEd1YXJkUmVzdWx0ID0gSlNPTi5wYXJzZShKU09OLnN0cmluZ2lmeShyZXN1bHQpLCByZXZpdmVyKTtcbiAgICAgIGlmICghZ3VhcmRSZXN1bHQubm90X2NvbXBsaWFudCB8fCBndWFyZFJlc3VsdC5ub3RfY29tcGxpYW50Lmxlbmd0aCA9PT0gMCkge1xuICAgICAgICByZXR1cm4geyBzdWNjZXNzOiB0cnVlLCB2aW9sYXRpb25zOiBbXSB9O1xuICAgICAgfVxuICAgICAgc3VjY2VzcyA9IGZhbHNlO1xuICAgICAgZ3VhcmRSZXN1bHQubm90X2NvbXBsaWFudC5mb3JFYWNoKChjaGVjaykgPT4ge1xuICAgICAgICBjb25zdCB2aW9sYXRpb25DaGVjayA9IG5ldyBWaW9sYXRpb25DaGVjayhjaGVjaywgY29uZmlnLnRlbXBsYXRlUGF0aCwgY29uZmlnLnJ1bGVQYXRoKTtcbiAgICAgICAgY29uc3QgdmlvbGF0aW9uID0gdmlvbGF0aW9uQ2hlY2sucHJvY2Vzc0NoZWNrKCk7XG4gICAgICAgIHZpb2xhdGlvbnMucHVzaCguLi52aW9sYXRpb24pO1xuICAgICAgfSk7XG4gICAgfSBjYXRjaCAoZSkge1xuICAgICAgc3VjY2VzcyA9IGZhbHNlO1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBcbiAgICAgICAgQ2ZuR3VhcmRWYWxpZGF0b3IgcGx1Z2luIGZhaWxlZCBwcm9jZXNzaW5nIGNmbi1ndWFyZCByZXN1bHRzLlxuICAgICAgICBQbGVhc2UgY3JlYXRlIGFuIGlzc3VlIGh0dHBzOi8vZ2l0aHViLmNvbS9jZGtsYWJzL2Nkay12YWxpZGF0b3ItY2ZuZ3VhcmQvaXNzdWVzL25ld1xuICAgICAgICBSdWxlOiAke3BhdGguYmFzZW5hbWUoY29uZmlnLnJ1bGVQYXRoKX1cbiAgICAgICAgRXJyb3I6ICR7ZX1gKTtcbiAgICB9XG4gICAgcmV0dXJuIHtcbiAgICAgIHN1Y2Nlc3MsXG4gICAgICB2aW9sYXRpb25zOiB2aW9sYXRpb25zLFxuICAgIH07XG4gIH1cbn1cblxuXG4vKipcbiAqIEd1YXJkIGRvZXMgbm90IGhhdmUgYSBzdGFuZGFyZCBKU09OIHNjaGVtYSBhbmQgdGhlIHNjaGVtYVxuICogdGhhdCBpcyByZXR1cm5lZCBjYW4gYmUgZGVwZW5kZW50IG9uIHRoZSB0eXBlIG9mIHJ1bGUgb3IgdHlwZVxuICogb2YgY2hlY2sgdGhhdCB3YXMgZXhlY3V0ZWQuIFRoZSByZXN1bHRzIGFyZSB2ZXJ5IG11Y2ggYW4gYXR0ZW1wdCB0b1xuICogZGlzcGxheSB0aGUgaW50ZXJuYWxzIG9mIGd1YXJkIHRvIHRoZSB1c2VyLiBUcnlpbmcgdG8gbWFrZSBzZW5zZSBvZiB0aGF0XG4gKiBjYW4gYmUgZGlmZmljdWx0LlxuICpcbiAqIFRoZSByZXN1bHQgc3RydWN0dXJlIGNhbiBkZXBlbmQgb24gdGhlIHdheSB0aGF0IHRoZSBydWxlIHdhcyB3cml0dGVuLiBGb3IgZXhhbXBsZVxuICogSSBjb3VsZCB3cml0ZSBhIHJ1bGUgbGlrZSB0aGlzOlxuICpcbiAqICAgICBydWxlIE1ZX1JVTEUge1xuICogICAgICAgIyBUaGlzIGlzIGEgXCJjaGVja1wiIGFuZCBpcyBhIGBDbGF1c2VgIHR5cGUgY2hlY2tcbiAqICAgICAgIFByb3BlcnRpZXMuU29tZVByb3AgPT0gdHJ1ZVxuICogICAgIH1cbiAqXG4gKiBPciBJIGNvdWxkIHdyaXRlIGEgcnVsZSBsaWtlIHRoaXM6XG4gKlxuICogICAgIHJ1bGUgTVlfUlVMRSB7XG4gKiAgICAgICAjICBUaGlzIGlzIGEgXCJjaGVja1wiIGFuZCBpcyBhIGBSdWxlYCB0eXBlIGNoZWNrXG4gKiAgICAgICBjaGVjayhQcm9wZXJ0aWVzKVxuICogICAgIH1cbiAqICAgICBydWxlIGNoZWNrKHByb3BlcnRpZXMpIHtcbiAqICAgICAgIHByb3BlcnRpZXMuU29tZVByb3AgPT0gdHJ1ZVxuICogICAgIH1cbiAqXG4gKiBCb3RoIG9mIHRoZSBhYm92ZSBleGFtcGxlcyBhcmUgY2hlY2tpbmcgdGhlIHNhbWUgdGhpbmdcbiAqIGJ1dCB0aGUgc2NoZW1hIHRoYXQgaXMgcmV0dXJuZWQgaXMgZGlmZmVyZW50IGJlY2F1c2UgdGhlXG4gKiB3YXkgdGhlIHJ1bGUgd2FzIHdyaXR0ZW4gaXMgZGlmZmVyZW50XG4gKlxuICogVGhpcyByZXZpdmVyIGZ1bmN0aW9uIGlzIGV4ZWN1dGVkIGJvdHRvbSB1cCBhbmQgaXMgZXNzZW50aWFsbHlcbiAqIGNyZWF0aW5nIGEgbmV3IG9iamVjdCB3aXRoIGEgd2VsbCBrbm93biBzY2hlbWEgdGhhdCB0aGUgcmVzdCBvZiB0aGVcbiAqIHBsdWdpbiBjYW4gd29yayB3aXRoLiBJdCBsb29rcyBmb3IgY2VydGFpbiBmaWVsZHMgdGhhdCBhbHdheXMgYXBwZWFyIGluXG4gKiB0aGUgZ3VhcmQgcmVzdWx0cywgYnV0IGFwcGVhciBpbiBkaWZmZXJlbnQgbG9jYXRpb25zLiBJdCBmaW5kcyB0aG9zZSBmaWVsZHNcbiAqIGFuZCB0aGVuIHB1bGxzIHRoZW0gdXAgdGhlIG9iamVjdCwgZHJvcHBpbmcgYW55IGZpZWxkcyB0aGF0IHdlIGRvbid0XG4gKiBjYXJlIGFib3V0LiBGb3IgZXhhbXBsZSBndWFyZCBtYXkgcmV0dXJuXG4gKlxuICoge1xuICogICBDbGF1c2U6IHtcbiAqICAgICBVbmFyeToge1xuICogICAgICAgY2hlY2s6IHtcbiAqICAgICAgICAgVW5SZXNvbHZlZDoge1xuICogICAgICAgICAgIHZhbHVlOiB7XG4gKiAgICAgICAgICAgICB0cmF2ZXJzZWRfdG86IHsuLi59IC8vIHdlIG9ubHkgY2FyZSBhYm91dCB0aGlzISEhXG4gKiAgICAgICAgICAgfVxuICogICAgICAgICB9XG4gKiAgICAgICB9XG4gKiAgICAgfVxuICogICB9XG4gKiB9XG4gKlxuICogT3IgaXQgbWF5IHJldHVyblxuICpcbiAqIHtcbiAqICAgUnVsZToge1xuICogICAgIGNoZWNrczogW3tcbiAqICAgICAgIEJsb2NrOiB7XG4gKiAgICAgICAgIHVucmVzb2x2ZWQ6IHtcbiAqICAgICAgICAgICB0cmF2ZXJzZWRfdG86IHsuLi59IC8vIHdlIG9ubHkgY2FyZSBhYm91dCB0aGlzISEhXG4gKiAgICAgICAgIH1cbiAqICAgICAgIH1cbiAqICAgICB9XVxuICogICB9XG4gKiB9XG4gKlxuICogSW4gdGhlIGFib3ZlIGV4YW1wbGUgd2Ugb25seSBjYXJlIGFib3V0IHRoZSAndHJhdmVyc2VkX3RvJyBmaWVsZCxcbiAqIHNvIHRoaXMgcmV2aXZlciBmdW5jdGlvbiB3aWxsIGdyYWIgdGhhdCBmaWVsZCBhbmQgcHVsbCBpdCB1cCB0aGUgb2JqZWN0LCBkcm9wcGluZ1xuICogdGhlIGZpZWxkcyB3ZSBkb24ndCBjYXJlIGFib3V0LCBlbmRpbmcgd2l0aCBzb21ldGhpbmcgbGlrZVxuICoge1xuICogICBjaGVja3M6IFt7XG4gKiAgICAgcmVzb2x2ZWQ6IGZhbHNlLFxuICogICAgIHRyYXZlcnNlZDogey4uLn1cbiAqICAgfV1cbiAqIH1cbiAqXG4gKi9cbmZ1bmN0aW9uIHJldml2ZXIoa2V5OiBzdHJpbmcsIHZhbHVlOiBhbnkpOiBhbnkge1xuICBpZiAoa2V5ID09PSAnbm90X2NvbXBsaWFudCcpIHtcbiAgICAvLyBub3RfY29tcGxpYW50IGNhbiBzb21ldGltZXMgYmUgYW4gZW1wdHkgb2JqZWN0IChidXQgbm90IGFuIEFycmF5KSwgc28gd2VcbiAgICAvLyBwcm9jZXNzIHRoaXMgdmFsdWUgYmVmb3JlIGRpdmluZyBpbnRvIG90aGVyIG9iamVjdCB2YWx1ZXMgdG8gZW5zdXJlIHRoaXNcbiAgICAvLyBvbmUgaXMgYWx3YXlzIG1hZGUgaW50byBhbiBBcnJheVxuICAgIHJldHVybiBPYmplY3QudmFsdWVzKHZhbHVlKS5tYXAoKHY6IGFueSkgPT4gdi5SdWxlKTtcbiAgfSBlbHNlIGlmICh2YWx1ZSAhPT0gbnVsbCAmJiB0eXBlb2YgdmFsdWUgPT09ICdvYmplY3QnICYmICFBcnJheS5pc0FycmF5KHZhbHVlKSkge1xuICAgIHJldHVybiBleHRyYWN0TmVzdGVkT2JqZWN0KHZhbHVlKTtcbiAgfSBlbHNlIGlmIChrZXkgPT09ICdjaGVja3MnICYmIEFycmF5LmlzQXJyYXkodmFsdWUpKSB7XG4gICAgcmV0dXJuIGV4dHJhY3ROZXN0ZWRDaGVja3ModmFsdWUuZmxhdE1hcCh2ID0+IHYpKTtcbiAgfVxuICByZXR1cm4gdmFsdWU7XG59XG5cbi8qKlxuICogRXh0cmFjdCBhIG5lc3RlZCAnY2hlY2tzJyBvYmplY3QuIFRoaXMgYWxzbyBoYW5kbGVzIGNoZWNrc1xuICogbmVzdGVkIHdpdGhpbiBjaGVja3MuIEl0IHdpbGwgZ3JhYiB0aGUgY2hlY2tzIGF0IHRoZSBsZXZlbCBiZWxvd1xuICogYW5kIHB1bGwgaXQgdXAgdG8gdGhlIG5leHQgbGV2ZWwuXG4gKi9cbmZ1bmN0aW9uIGV4dHJhY3ROZXN0ZWRDaGVja3MoY2hlY2tzOiBhbnlbXSk6IGFueVtdIHtcbiAgY29uc3QgY29udGFpbnNOZXN0ZWRDaGVja3MgPSBjaGVja3Muc29tZShjaGVjayA9PiBPYmplY3QudmFsdWVzKGNoZWNrKS5zb21lKCh2YWx1ZTogYW55KSA9PiB7XG4gICAgcmV0dXJuIHR5cGVvZiB2YWx1ZSA9PT0gJ29iamVjdCcgJiYgdmFsdWUuaGFzT3duUHJvcGVydHkoJ2NoZWNrcycpO1xuICB9KSk7XG4gIGlmIChjb250YWluc05lc3RlZENoZWNrcykge1xuICAgIHJldHVybiBjaGVja3MuZmxhdE1hcChjaGVjayA9PiB7XG4gICAgICBpZiAoT2JqZWN0LmtleXMoY2hlY2spLmluY2x1ZGVzKCd0cmF2ZXJzZWQnKSkge1xuICAgICAgICByZXR1cm4gY2hlY2s7XG4gICAgICB9XG4gICAgICByZXR1cm4gT2JqZWN0LnZhbHVlcyhjaGVjaykuZmxhdE1hcCgoY2hlY2tWYWx1ZTogYW55KSA9PiB7XG4gICAgICAgIHJldHVybiBPYmplY3QudmFsdWVzKGNoZWNrVmFsdWUuY2hlY2tzID8/IGNoZWNrVmFsdWUpLmZsYXRNYXAoKG5lc3RlZENoZWNrVmFsdWU6IGFueSkgPT4ge1xuICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAuLi5uZXN0ZWRDaGVja1ZhbHVlLFxuICAgICAgICAgICAgbmFtZTogY2hlY2tWYWx1ZS5uYW1lLFxuICAgICAgICAgICAgbWVzc2FnZXM6IGNoZWNrVmFsdWUubWVzc2FnZXMgPz8gbmVzdGVkQ2hlY2tWYWx1ZS5tZXNzYWdlcyxcbiAgICAgICAgICB9O1xuICAgICAgICB9KTtcbiAgICAgIH0pO1xuICAgIH0pO1xuICB9XG4gIHJldHVybiBjaGVja3MuZmxhdE1hcChjaGVjayA9PiB7XG4gICAgaWYgKE9iamVjdC5rZXlzKGNoZWNrKS5pbmNsdWRlcygndHJhdmVyc2VkJykpIHtcbiAgICAgIHJldHVybiBjaGVjaztcbiAgICB9XG4gICAgcmV0dXJuIE9iamVjdC52YWx1ZXMoY2hlY2spO1xuICB9KTtcbn1cblxuLyoqXG4gKiBFeHRyYWN0IGEgbmVzdGVkIG9iamVjdCBhbmQgcHVsbCBpdCB1cCBhIGxldmVsXG4gKi9cbmZ1bmN0aW9uIGV4dHJhY3ROZXN0ZWRPYmplY3Qob2JqZWN0OiBhbnkpOiBhbnkge1xuICBsZXQgbmV3T2JqZWN0ID0gb2JqZWN0O1xuICBPYmplY3QuZW50cmllcyhvYmplY3QpLmZvckVhY2goKFtsZXZlbDFOZXN0ZWRLZXksIGxldmVsMU5lc3RlZFZhbHVlXSkgPT4ge1xuICAgIGNvbnN0IG5lc3RlZFZhbHVlID0gbGV2ZWwxTmVzdGVkVmFsdWUgYXMgYW55O1xuICAgIHN3aXRjaCAobGV2ZWwxTmVzdGVkS2V5LnRvTG93ZXJDYXNlKCkpIHtcbiAgICAgIC8vIHRoaXMgc2hvdWxkIGFsd2F5cyBiZSBmb3VuZCBlYXJsaWVyIHRoYW4gdGhlIHJlc3Qgc2luY2UgaXQgYXBwZWFyc1xuICAgICAgLy8gd2l0aGluIHRoZSAndW5yZXNvbHZlZCcgYW5kICdyZXNvbHZlZCcgb2JqZWN0cy4gVGhlIG9iamVjdFxuICAgICAgLy8gaXMgc2xpZ2h0bHkgZGlmZmVyZW50IGZvciBlYWNoIGNhc2Ugc28gaGVyZSB3ZSBjcmVhdGVcbiAgICAgIC8vIGEgbmV3IG9iamVjdCB3aXRoIHRoZSBrZXkgJ3RyYXZlcnNlZCcgd2l0aCBhIGNvbnNpc3RlbnQgdmFsdWVcbiAgICAgIGNhc2UgJ3RyYXZlcnNlZF90byc6XG4gICAgICAgIG5ld09iamVjdCA9IHtcbiAgICAgICAgICB0cmF2ZXJzZWQ6IHtcbiAgICAgICAgICAgIHRvOiB7XG4gICAgICAgICAgICAgIHBhdGg6IG5lc3RlZFZhbHVlLnBhdGgsXG4gICAgICAgICAgICAgIHZhbHVlOiBuZXN0ZWRWYWx1ZS52YWx1ZSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBmcm9tOiBuZXN0ZWRWYWx1ZS5mcm9tID8ge1xuICAgICAgICAgICAgICBwYXRoOiBuZXN0ZWRWYWx1ZS5mcm9tLnBhdGgsXG4gICAgICAgICAgICAgIHZhbHVlOiB1bmRlZmluZWQsXG4gICAgICAgICAgICB9IDogdW5kZWZpbmVkLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgbWVzc2FnZXM6IG5lc3RlZFZhbHVlLm1lc3NhZ2VzLFxuICAgICAgICB9O1xuICAgICAgICBicmVhaztcbiAgICAgIC8vIFRoaXMgc2hvdWxkIGJlIGZvdW5kIGluIHRoZSBcInNlY29uZFwiIHBhc3MgYWZ0ZXIgdGhlIGFib3ZlXG4gICAgICAvLyAndHJhdmVyc2VkX3RvJyBjYXNlIGhhcyBiZWVuIGV4ZWN1dGVkLiBXZSB0YWtlIHRoZSBuZXdcbiAgICAgIC8vIG9iamVjdCB0aGF0IHdhcyBjcmVhdGVkIGluIHRoZSBgdHJhdmVyc2VkX3RvYCBjYXNlIGFuZFxuICAgICAgLy8gYSBjb3VwbGUgb3RoZXIgZmllbGRzLCBkcm9wcGluZyB0aGUgcmVzdCB0aGF0IHdlIGRvbid0IGNhcmUgYWJvdXRcbiAgICAgIGNhc2UgJ3VucmVzb2x2ZWQnOlxuICAgICAgICBuZXdPYmplY3QgPSB7XG4gICAgICAgICAgcmVzb2x2ZWQ6IGZhbHNlLFxuICAgICAgICAgIHRyYXZlcnNlZDogbmVzdGVkVmFsdWUudHJhdmVyc2VkLFxuICAgICAgICAgIG1lc3NhZ2VzOiBuZXN0ZWRWYWx1ZS5tZXNzYWdlcyA/PyBvYmplY3QubWVzc2FnZXMsXG4gICAgICAgIH07XG4gICAgICAgIGJyZWFrO1xuICAgICAgLy8gVGhpcyBzaG91bGQgYmUgZm91bmQgaW4gdGhlIFwic2Vjb25kXCIgcGFzcyBhZnRlciB0aGUgYWJvdmVcbiAgICAgIC8vICd0cmF2ZXJzZWRfdG8nIGNhc2UgaGFzIGJlZW4gZXhlY3V0ZWQuIFdlIHRha2UgdGhlIG5ld1xuICAgICAgLy8gb2JqZWN0IHRoYXQgd2FzIGNyZWF0ZWQgaW4gdGhlIGB0cmF2ZXJzZWRfdG9gIGNhc2UgYW5kXG4gICAgICAvLyBhIGNvdXBsZSBvdGhlciBmaWVsZHMsIGRyb3BwaW5nIHRoZSByZXN0IHRoYXQgd2UgZG9uJ3QgY2FyZSBhYm91dFxuICAgICAgLy8gQSBjaGVjayBjYW4gZWl0aGVyIGJlIHJlc29sdmVkIG9yIHVucmVzb2x2ZWRcbiAgICAgIGNhc2UgJ3Jlc29sdmVkJzpcbiAgICAgIGNhc2UgJ2lucmVzb2x2ZWQnOlxuICAgICAgICBpZiAoJ2Zyb20nIGluIG5lc3RlZFZhbHVlKSB7XG4gICAgICAgICAgbmV3T2JqZWN0ID0ge1xuICAgICAgICAgICAgcmVzb2x2ZWQ6IHRydWUsXG4gICAgICAgICAgICB0cmF2ZXJzZWQ6IHtcbiAgICAgICAgICAgICAgZnJvbTogbmVzdGVkVmFsdWUuZnJvbSxcbiAgICAgICAgICAgICAgdG86IHtcbiAgICAgICAgICAgICAgICBwYXRoOiBuZXN0ZWRWYWx1ZS5mcm9tLnBhdGgsXG4gICAgICAgICAgICAgICAgdmFsdWU6IG5lc3RlZFZhbHVlLnRvLnZhbHVlLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIG1lc3NhZ2VzOiBuZXN0ZWRWYWx1ZS5tZXNzYWdlcyxcbiAgICAgICAgICB9O1xuICAgICAgICB9IGVsc2UgaWYgKCd2YWx1ZScgaW4gbmVzdGVkVmFsdWUpIHtcbiAgICAgICAgICBuZXdPYmplY3QgPSB7XG4gICAgICAgICAgICByZXNvbHZlZDogdHJ1ZSxcbiAgICAgICAgICAgIHRyYXZlcnNlZDoge1xuICAgICAgICAgICAgICBmcm9tOiBuZXN0ZWRWYWx1ZS52YWx1ZSxcbiAgICAgICAgICAgICAgdG86IHtcbiAgICAgICAgICAgICAgICBwYXRoOiBuZXN0ZWRWYWx1ZS52YWx1ZS5wYXRoLFxuICAgICAgICAgICAgICAgIHZhbHVlOiBuZXN0ZWRWYWx1ZS52YWx1ZS52YWx1ZSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBtZXNzYWdlczogbmVzdGVkVmFsdWUubWVzc2FnZXMsXG4gICAgICAgICAgfTtcblxuICAgICAgICB9XG4gICAgICAgIGJyZWFrO1xuICAgIH1cbiAgICAvLyB0aGlzIGNoZWNrIHdpbGwgYmUgZXZhbHVhdGVkIF9hZnRlcl8gdGhlICd0cmF2ZXJzZWRfdG8nIGNoZWNrIGFuZCBfYmVmb3JlXyB0aGUgJ3Jlc29sdmVkJ1xuICAgIC8vIGFuZCAndW5yZXNvbHZlZCcgY2hlY2tzIGFib3ZlLiBUaGVyZSBtYXkgYmUgYSBjYXNlIHdoZXJlICd0cmF2ZXJzZWQnIGlzIG5lc3RlZCAyIChvciAzIG9yIDQpIGJlbG93XG4gICAgLy8gJ3VucmVzb2x2ZWQnIG9yICdyZXNvbHZlZCcgYW5kIHRoaXMgd2lsbCBrZWVwIHB1bGxpbmcgaXQgdXAgdW50aWwgaXQgaXMganVzdCBvbmUgYmVsb3dcbiAgICAvLyBhbmQgdGhlIGFib3ZlIGNoZWNrcyBjYW4gd29ya1xuICAgIGlmIChsZXZlbDFOZXN0ZWRWYWx1ZSAhPT0gbnVsbCAmJiB0eXBlb2YgbGV2ZWwxTmVzdGVkVmFsdWUgPT09ICdvYmplY3QnICYmICFBcnJheS5pc0FycmF5KGxldmVsMU5lc3RlZFZhbHVlKSkge1xuICAgICAgT2JqZWN0LmVudHJpZXMoKGxldmVsMU5lc3RlZFZhbHVlIGFzIG9iamVjdCkpLmZvckVhY2goKFtsZXZlbDJOZXN0ZWRLZXksIGxldmVsMk5lc3RlZFZhbHVlXSkgPT4ge1xuICAgICAgICBzd2l0Y2ggKGxldmVsMk5lc3RlZEtleS50b0xvd2VyQ2FzZSgpKSB7XG4gICAgICAgICAgY2FzZSAndHJhdmVyc2VkJzpcbiAgICAgICAgICAgIG5ld09iamVjdCA9IHtcbiAgICAgICAgICAgICAgdHJhdmVyc2VkOiBuZXN0ZWRWYWx1ZS50cmF2ZXJzZWQsXG4gICAgICAgICAgICAgIHJlc29sdmVkOiBuZXN0ZWRWYWx1ZS5yZXNvbHZlZCxcbiAgICAgICAgICAgICAgbWVzc2FnZXM6IG5lc3RlZFZhbHVlLm1lc3NhZ2VzID8/IGxldmVsMk5lc3RlZFZhbHVlLm1lc3NhZ2VzID8/IG9iamVjdC5tZXNzYWdlcyxcbiAgICAgICAgICAgIH07XG4gICAgICAgICAgICBicmVhaztcbiAgICAgICAgfVxuICAgICAgfSk7XG4gICAgfVxuICB9KTtcbiAgcmV0dXJuIG5ld09iamVjdDtcbn1cbiJdfQ==