#!/usr/local/bin/python3
# -*- coding: utf-8 -*-
# By default, terrestrial
#     relative isotopic abundance is assumed.
"""
Reference molecular data. The mol_info dictionary contains:
    (1) terrestrial relative isotopic abundance for each molecule;
    (2) mass of each isotope in atomic mass unit (amu);
    (3) molecular mass in amu assuming terrestrial isotopic abundance;
    (4) coefficients of the 4th order polynomial fit to the 70-300K section
        of the total partition function (vib + rot) distributed with
        the HITRAN-08 database.


The molecules are ranked by their Radtran gas identifiers and isotopes by
their terrestrial relative abundance.

Examples
--------
Type mol_info['1'] to access informaiton of the molecule with Radtran ID 1,
which is H2O. Type mol_info['1']['mmw'] to get its mean molecular weight
in amu assuming terrestrial isotopic abundance. Type mol_info['1']['1']
to access information about the first (most abundant) isotope.

Notes
-----
In most line databases (e.g. HITRAN and GEISA), TERRESTRIAL relative abundances
are assumed and the lines in the databases are scaled by these relative
abundances, such that if you request the lines of all the isotopes of a
given gas the line strengths come 'pre-scaled' by these ratios. If you want to use
different isotope ratios then you should really modify the line strengths in your
database and update the relative abundances in your reference file.

Where no partition function information is listed, for linear molecules we
have set the coeffs. to (0.0,1.0,0.0,0.0) and for polyatomic molecules we have
set the coeffs to (-0.44219E+02, +0.58047E+01, +0.50145E-01, -0.37316E-04),
which is a reasonable approximation to T**1.5 (rotation partition
function of polyatomic rotor). For monatomic gases we have set the
partition function coeffs to (1.0,0.0,0.0,0.0).

Source
------
HITRAN08 release
"""

mol_info = {
    "1": {
        "name": "H2O",
        "isotope": {
            "1": {
                "abun": 0.997317,
                "mass": 18.0106,
                "id": 161,
                "partition": [
                    -3.46229,
                    0.251974,
                    0.00143133,
                    -8.45955e-07
                ]
            },
            "2": {
                "abun": 0.0019998,
                "mass": 20.0148,
                "id": 181,
                "partition": [
                    -4.5017,
                    0.254188,
                    0.00144279,
                    -8.51659e-07
                ]
            },
            "3": {
                "abun": 0.0003719,
                "mass": 19.0148,
                "id": 171,
                "partition": [
                    -27.7882,
                    1.53995,
                    0.0084651,
                    -4.70886e-06
                ]
            },
            "4": {
                "abun": 0.0003107,
                "mass": 19.0167,
                "id": 162,
                "partition": [
                    -24.7416,
                    1.24678,
                    0.00708276,
                    -4.09274e-06
                ]
            },
            "5": {
                "abun": 6e-07,
                "mass": 21.021,
                "id": 182,
                "partition": [
                    -19.0622,
                    1.26371,
                    0.00715754,
                    -4.13404e-06
                ]
            }
        },
        "mmw": 18.01529586265
    },
    "2": {
        "name": "CO2",
        "isotope": {
            "1": {
                "abun": 0.984204,
                "mass": 43.9898,
                "id": 626,
                "partition": [
                    -2.96501,
                    0.983215,
                    -0.000897249,
                    2.98804e-06
                ]
            },
            "2": {
                "abun": 0.0110574,
                "mass": 44.9932,
                "id": 636,
                "partition": [
                    -5.69943,
                    1.96467,
                    -0.00183314,
                    6.29369e-06
                ]
            },
            "3": {
                "abun": 0.0039471,
                "mass": 45.9941,
                "id": 628,
                "partition": [
                    -6.35548,
                    2.08607,
                    -0.00192846,
                    6.453e-06
                ]
            },
            "4": {
                "abun": 0.000734,
                "mass": 44.994,
                "id": 627,
                "partition": [
                    -36.7382,
                    12.1635,
                    -0.0111648,
                    3.73013e-05
                ]
            },
            "5": {
                "abun": 4.43e-05,
                "mass": 46.9974,
                "id": 638,
                "partition": [
                    -12.0751,
                    4.16621,
                    -0.00393008,
                    1.35896e-05
                ]
            },
            "6": {
                "abun": 8.2e-06,
                "mass": 45.9974,
                "id": 637,
                "partition": [
                    -70.6247,
                    24.3071,
                    -0.0228238,
                    7.8622e-05
                ]
            },
            "7": {
                "abun": 4e-06,
                "mass": 47.9983,
                "id": 828,
                "partition": [
                    -3.36,
                    1.10736,
                    -0.00103189,
                    3.47875e-06
                ]
            },
            "8": {
                "abun": 1e-07,
                "mass": 46.9983,
                "id": 727,
                "partition": [
                    0.0,
                    1.0,
                    0.0,
                    0.0
                ]
            },
            "9": {
                "abun": 1e-07,
                "mass": 48.0,
                "id": 838,
                "partition": [
                    0.0,
                    1.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 44.00967449352
    },
    "3": {
        "name": "O3",
        "isotope": {
            "1": {
                "abun": 0.992901,
                "mass": 47.9847,
                "id": 666,
                "partition": [
                    -133.581,
                    6.04917,
                    0.0170067,
                    1.28849e-05
                ]
            },
            "2": {
                "abun": 0.0039819,
                "mass": 49.989,
                "id": 668,
                "partition": [
                    -289.12,
                    13.0276,
                    0.0352366,
                    3.10756e-05
                ]
            },
            "3": {
                "abun": 0.001991,
                "mass": 49.989,
                "id": 686,
                "partition": [
                    -142.445,
                    6.39302,
                    0.0170594,
                    1.54142e-05
                ]
            },
            "4": {
                "abun": 0.0007405,
                "mass": 48.989,
                "id": 667,
                "partition": [
                    -1668.28,
                    75.3972,
                    0.208145,
                    0.000170149
                ]
            },
            "5": {
                "abun": 0.0003702,
                "mass": 48.989,
                "id": 676,
                "partition": [
                    -827.131,
                    37.315,
                    0.102505,
                    8.44186e-05
                ]
            }
        },
        "mmw": 47.99704799509999
    },
    "4": {
        "name": "N2O",
        "isotope": {
            "1": {
                "abun": 0.990333,
                "mass": 44.0011,
                "id": 446,
                "partition": [
                    -36.8881,
                    16.3266,
                    -0.0159404,
                    6.18354e-05
                ]
            },
            "2": {
                "abun": 0.0036409,
                "mass": 44.9981,
                "id": 456,
                "partition": [
                    -21.9223,
                    10.8403,
                    -0.0105753,
                    4.25126e-05
                ]
            },
            "3": {
                "abun": 0.0036409,
                "mass": 44.9981,
                "id": 546,
                "partition": [
                    -25.1325,
                    11.2613,
                    -0.011048,
                    4.31522e-05
                ]
            },
            "4": {
                "abun": 0.0019858,
                "mass": 46.0053,
                "id": 448,
                "partition": [
                    -38.9884,
                    17.2992,
                    -0.0170867,
                    6.67585e-05
                ]
            },
            "5": {
                "abun": 0.0003693,
                "mass": 45.0053,
                "id": 447,
                "partition": [
                    -226.226,
                    100.913,
                    -0.0989411,
                    0.000385688
                ]
            }
        },
        "mmw": 44.01138631291001
    },
    "5": {
        "name": "CO",
        "isotope": {
            "1": {
                "abun": 0.986544,
                "mass": 27.9949,
                "id": 26,
                "partition": [
                    0.334009,
                    0.359976,
                    2.25241e-06,
                    1.26962e-09
                ]
            },
            "2": {
                "abun": 0.0110836,
                "mass": 28.9983,
                "id": 36,
                "partition": [
                    0.677444,
                    0.752962,
                    5.6089e-06,
                    1.38546e-09
                ]
            },
            "3": {
                "abun": 0.0019782,
                "mass": 29.9992,
                "id": 28,
                "partition": [
                    0.333434,
                    0.378025,
                    2.29809e-06,
                    1.46391e-09
                ]
            },
            "4": {
                "abun": 0.0003679,
                "mass": 28.9991,
                "id": 27,
                "partition": [
                    2.00295,
                    2.21582,
                    1.52613e-05,
                    4.34306e-09
                ]
            },
            "5": {
                "abun": 2.22e-05,
                "mass": 31.0025,
                "id": 38,
                "partition": [
                    0.675861,
                    0.792552,
                    5.88599e-06,
                    1.30104e-09
                ]
            },
            "6": {
                "abun": 4.1e-06,
                "mass": 30.0025,
                "id": 37,
                "partition": [
                    4.00014,
                    4.6418,
                    2.90809e-05,
                    1.42774e-08
                ]
            }
        },
        "mmw": 28.010430635559995
    },
    "6": {
        "name": "CH4",
        "isotope": {
            "1": {
                "abun": 0.988274,
                "mass": 16.0313,
                "id": 211,
                "partition": [
                    -16.1194,
                    0.9052,
                    0.00440876,
                    -1.85094e-06
                ]
            },
            "2": {
                "abun": 0.0111031,
                "mass": 17.0347,
                "id": 311,
                "partition": [
                    -32.2742,
                    1.81107,
                    0.00881186,
                    -3.69149e-06
                ]
            },
            "3": {
                "abun": 0.0006158,
                "mass": 17.0375,
                "id": 212,
                "partition": [
                    -132.101,
                    7.30804,
                    0.0357219,
                    -1.49912e-05
                ]
            },
            "4": {
                "abun": 6.9179e-06,
                "mass": 18.0409,
                "id": 312,
                "partition": [
                    -132.101,
                    7.30804,
                    0.0357219,
                    -1.49912e-05
                ]
            }
        },
        "mmw": 16.04307145141211
    },
    "7": {
        "name": "O2",
        "isotope": {
            "1": {
                "abun": 0.995262,
                "mass": 31.9898,
                "id": 66,
                "partition": [
                    0.752687,
                    0.726609,
                    -1.02337e-05,
                    3.21953e-08
                ]
            },
            "2": {
                "abun": 0.0039914,
                "mass": 33.9941,
                "id": 68,
                "partition": [
                    -3.18564,
                    1.54078,
                    -3.41463e-05,
                    9.23627e-08
                ]
            },
            "3": {
                "abun": 0.0007422,
                "mass": 32.994,
                "id": 67,
                "partition": [
                    -18.3703,
                    8.98862,
                    -0.000152297,
                    4.44482e-07
                ]
            }
        },
        "mmw": 31.998404525139996
    },
    "8": {
        "name": "NO",
        "isotope": {
            "1": {
                "abun": 0.993974,
                "mass": 29.998,
                "id": 46,
                "partition": [
                    -14.1684,
                    2.37012,
                    0.00810726,
                    -9.88509e-06
                ]
            },
            "2": {
                "abun": 0.0036543,
                "mass": 30.995,
                "id": 56,
                "partition": [
                    -9.87836,
                    1.6386,
                    0.00559892,
                    -6.82485e-06
                ]
            },
            "3": {
                "abun": 0.0019931,
                "mass": 32.0022,
                "id": 48,
                "partition": [
                    -15.1258,
                    2.50117,
                    0.00853937,
                    -1.04032e-05
                ]
            }
        },
        "mmw": 29.99428066532
    },
    "9": {
        "name": "SO2",
        "isotope": {
            "1": {
                "abun": 0.945678,
                "mass": 63.9619,
                "id": 626,
                "partition": [
                    -205.876,
                    10.142,
                    0.029518,
                    3.68379e-05
                ]
            },
            "2": {
                "abun": 0.0419503,
                "mass": 65.9577,
                "id": 646,
                "partition": [
                    -207.071,
                    10.1929,
                    0.0296328,
                    3.70217e-05
                ]
            }
        },
        "mmw": 63.25430697051
    },
    "10": {
        "name": "NO2",
        "isotope": {
            "1": {
                "abun": 0.991616,
                "mass": 45.9929,
                "id": 646,
                "partition": [
                    -447.916,
                    21.9662,
                    0.0840982,
                    5.69159e-06
                ]
            }
        },
        "mmw": 45.6072955264
    },
    "11": {
        "name": "NH3",
        "isotope": {
            "1": {
                "abun": 0.995872,
                "mass": 17.0265,
                "id": 411,
                "partition": [
                    -47.0458,
                    2.66615,
                    0.0124549,
                    -4.20413e-06
                ]
            },
            "2": {
                "abun": 0.0036613,
                "mass": 18.0236,
                "id": 511,
                "partition": [
                    -31.4668,
                    1.78176,
                    0.00831785,
                    -2.79961e-06
                ]
            }
        },
        "mmw": 17.02220441468
    },
    "12": {
        "name": "HNO3",
        "isotope": {
            "1": {
                "abun": 0.98911,
                "mass": 62.9956,
                "id": 146,
                "partition": [
                    -14336.0,
                    507.726,
                    -1.03819,
                    0.00650804
                ]
            }
        },
        "mmw": 62.30957791600001
    },
    "13": {
        "name": "OH",
        "isotope": {
            "1": {
                "abun": 0.997473,
                "mass": 17.0027,
                "id": 61,
                "partition": [
                    9.8453,
                    0.132445,
                    0.00057262,
                    -7.29821e-07
                ]
            },
            "2": {
                "abun": 0.0020001,
                "mass": 19.007,
                "id": 81,
                "partition": [
                    9.80176,
                    0.134016,
                    0.000574469,
                    -7.32264e-07
                ]
            },
            "3": {
                "abun": 0.0001554,
                "mass": 18.0089,
                "id": 62,
                "partition": [
                    12.0475,
                    0.391224,
                    0.00145614,
                    -1.78393e-06
                ]
            }
        },
        "mmw": 17.00054866086
    },
    "14": {
        "name": "HF",
        "isotope": {
            "1": {
                "abun": 0.999844,
                "mass": 20.0062,
                "id": 19,
                "partition": [
                    1.57821,
                    0.132791,
                    1.06981e-05,
                    -1.37698e-08
                ]
            }
        },
        "mmw": 20.0030790328
    },
    "15": {
        "name": "HCl",
        "isotope": {
            "1": {
                "abun": 0.757587,
                "mass": 35.9767,
                "id": 15,
                "partition": [
                    2.89837,
                    0.530371,
                    1.22006e-05,
                    -1.15908e-08
                ]
            },
            "2": {
                "abun": 0.242257,
                "mass": 37.9737,
                "id": 17,
                "partition": [
                    2.91205,
                    0.530851,
                    1.44892e-05,
                    -1.64577e-08
                ]
            }
        },
        "mmw": 36.4548748638
    },
    "16": {
        "name": "HBr",
        "isotope": {
            "1": {
                "abun": 0.506781,
                "mass": 79.9262,
                "id": 19,
                "partition": [
                    2.85471,
                    0.663967,
                    1.22426e-05,
                    -1.11152e-08
                ]
            },
            "2": {
                "abun": 0.493063,
                "mass": 81.9241,
                "id": 11,
                "partition": [
                    2.87147,
                    0.663835,
                    1.41725e-05,
                    -1.44642e-08
                ]
            }
        },
        "mmw": 80.8988220805
    },
    "17": {
        "name": "HI",
        "isotope": {
            "1": {
                "abun": 0.999844,
                "mass": 127.9123,
                "id": 17,
                "partition": [
                    4.20077,
                    1.29599,
                    1.60701e-05,
                    -8.77552e-09
                ]
            }
        },
        "mmw": 127.8923456812
    },
    "18": {
        "name": "ClO",
        "isotope": {
            "1": {
                "abun": 0.755908,
                "mass": 50.9638,
                "id": 56,
                "partition": [
                    74.8171,
                    7.19926,
                    0.0103672,
                    6.23034e-06
                ]
            },
            "2": {
                "abun": 0.24172,
                "mass": 52.9608,
                "id": 76,
                "partition": [
                    76.027,
                    7.32562,
                    0.0105152,
                    6.46501e-06
                ]
            }
        },
        "mmw": 51.3256287064
    },
    "19": {
        "name": "OCS",
        "isotope": {
            "1": {
                "abun": 0.937395,
                "mass": 59.967,
                "id": 622,
                "partition": [
                    -8.39885,
                    3.77942,
                    -0.0045514,
                    1.96579e-05
                ]
            },
            "2": {
                "abun": 0.0415828,
                "mass": 61.9628,
                "id": 624,
                "partition": [
                    -8.80897,
                    3.87953,
                    -0.00472004,
                    2.03521e-05
                ]
            },
            "3": {
                "abun": 0.0105315,
                "mass": 60.9703,
                "id": 632,
                "partition": [
                    -13.6526,
                    7.52105,
                    -0.00894903,
                    4.07341e-05
                ]
            },
            "4": {
                "abun": 0.0018797,
                "mass": 61.9712,
                "id": 822,
                "partition": [
                    -8.08853,
                    4.01362,
                    -0.00484194,
                    2.15305e-05
                ]
            },
            "5": {
                "abun": 0.007399,
                "mass": 60.9664,
                "id": 623,
                "partition": [
                    -34.308,
                    15.318,
                    -0.0185246,
                    7.99975e-05
                ]
            },
            "6": {
                "abun": 1e-07,
                "mass": 63.0,
                "id": 634,
                "partition": [
                    0.0,
                    1.0,
                    0.0,
                    0.0
                ]
            },
            "7": {
                "abun": 1e-07,
                "mass": 61.0,
                "id": 722,
                "partition": [
                    0.0,
                    1.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 59.99905145753
    },
    "20": {
        "name": "H2CO",
        "isotope": {
            "1": {
                "abun": 0.986237,
                "mass": 30.0106,
                "id": 126,
                "partition": [
                    -80.6693,
                    4.32186,
                    0.0214787,
                    -9.07461e-06
                ]
            },
            "2": {
                "abun": 0.0110802,
                "mass": 31.0139,
                "id": 136,
                "partition": [
                    -164.949,
                    8.85012,
                    0.04413,
                    -1.8772e-05
                ]
            },
            "3": {
                "abun": 0.0019776,
                "mass": 32.0148,
                "id": 128,
                "partition": [
                    -84.4756,
                    4.52897,
                    0.0225631,
                    -9.57127e-06
                ]
            }
        },
        "mmw": 30.00451679546
    },
    "21": {
        "name": "HOCl",
        "isotope": {
            "1": {
                "abun": 0.75579,
                "mass": 51.9716,
                "id": 165,
                "partition": [
                    -639.768,
                    31.3071,
                    0.116209,
                    1.75429e-05
                ]
            },
            "2": {
                "abun": 0.241683,
                "mass": 53.9686,
                "id": 167,
                "partition": [
                    -652.631,
                    31.8954,
                    0.118026,
                    1.83609e-05
                ]
            }
        },
        "mmw": 52.3229087178
    },
    "22": {
        "name": "N2",
        "isotope": {
            "1": {
                "abun": 0.992687,
                "mass": 28.0061,
                "id": 44,
                "partition": [
                    1.50887,
                    1.56547,
                    1.20366e-05,
                    1.42517e-09
                ]
            },
            "2": {
                "abun": 0.007313,
                "mass": 29.0043,
                "id": 45,
                "partition": [
                    1.50887,
                    1.56547,
                    1.20366e-05,
                    1.42517e-09
                ]
            }
        },
        "mmw": 28.0133998366
    },
    "23": {
        "name": "HCN",
        "isotope": {
            "1": {
                "abun": 0.985114,
                "mass": 27.0109,
                "id": 124,
                "partition": [
                    -7.69888,
                    3.09721,
                    -0.00254497,
                    8.06649e-06
                ]
            },
            "2": {
                "abun": 0.0110676,
                "mass": 28.0143,
                "id": 134,
                "partition": [
                    -15.9127,
                    6.36125,
                    -0.00527088,
                    1.67791e-05
                ]
            },
            "3": {
                "abun": 0.0036217,
                "mass": 28.0079,
                "id": 125,
                "partition": [
                    -5.32057,
                    2.14441,
                    -0.00178399,
                    5.71841e-06
                ]
            }
        },
        "mmw": 27.02030302071
    },
    "24": {
        "name": "CH3Cl",
        "isotope": {
            "1": {
                "abun": 0.748937,
                "mass": 49.9923,
                "id": 215,
                "partition": [
                    -4885.53,
                    211.457,
                    0.49796,
                    0.000554523
                ]
            },
            "2": {
                "abun": 0.239491,
                "mass": 51.9894,
                "id": 217,
                "partition": [
                    -4962.51,
                    214.792,
                    0.505896,
                    0.000563292
                ]
            }
        },
        "mmw": 49.8920765805
    },
    "25": {
        "name": "H2O2",
        "isotope": {
            "1": {
                "abun": 0.994952,
                "mass": 34.0055,
                "id": 166,
                "partition": [
                    -204.542,
                    10.2294,
                    0.051462,
                    9.57989e-05
                ]
            }
        },
        "mmw": 33.83384023599999
    },
    "26": {
        "name": "C2H2",
        "isotope": {
            "1": {
                "abun": 0.977599,
                "mass": 26.0157,
                "id": 221,
                "partition": [
                    -8.42596,
                    1.44585,
                    -0.00254357,
                    8.37932e-06
                ]
            },
            "2": {
                "abun": 0.0219663,
                "mass": 27.019,
                "id": 231,
                "partition": [
                    -33.7411,
                    5.78454,
                    -0.0101816,
                    3.35342e-05
                ]
            },
            "3": {
                "abun": 0.0003107,
                "mass": 27.0219,
                "id": 222,
                "partition": [
                    -0.0219322,
                    0.00409836,
                    -7.97705e-06,
                    3.02274e-08
                ]
            }
        },
        "mmw": 26.03482546833
    },
    "27": {
        "name": "C2H6",
        "isotope": {
            "1": {
                "abun": 0.9769,
                "mass": 30.047,
                "id": 226,
                "partition": [
                    -2952.05,
                    119.368,
                    -0.0213804,
                    0.00155277
                ]
            },
            "2": {
                "abun": 0.0222,
                "mass": 31.07,
                "id": 236,
                "partition": [
                    -2952.05,
                    119.368,
                    -0.0213804,
                    0.00155277
                ]
            }
        },
        "mmw": 30.042668300000003
    },
    "28": {
        "name": "PH3",
        "isotope": {
            "1": {
                "abun": 0.999533,
                "mass": 33.9972,
                "id": 131,
                "partition": [
                    -105.413,
                    5.32304,
                    0.0208494,
                    -1.95721e-06
                ]
            }
        },
        "mmw": 33.9813233076
    },
    "29": {
        "name": "C2N2",
        "isotope": {
            "1": {
                "abun": 0.9705,
                "mass": 52.0,
                "id": 224,
                "partition": [
                    -0.0656507,
                    0.00508,
                    -1.86588e-05,
                    1.08991e-07
                ]
            },
            "2": {
                "abun": 0.0222,
                "mass": 52.0,
                "id": 324,
                "partition": [
                    -0.0656507,
                    0.00508,
                    -1.86588e-05,
                    1.08991e-07
                ]
            },
            "3": {
                "abun": 0.0073,
                "mass": 52.0,
                "id": 225,
                "partition": [
                    -0.0656507,
                    0.00508,
                    -1.86588e-05,
                    1.08991e-07
                ]
            }
        },
        "mmw": 52.00000000000001
    },
    "30": {
        "name": "C4H2",
        "isotope": {
            "1": {
                "abun": 0.9777777,
                "mass": 50.0587,
                "id": 211,
                "partition": [
                    -0.334142,
                    0.0116107,
                    -6.99304e-05,
                    2.49526e-07
                ]
            },
            "2": {
                "abun": 0.0222222,
                "mass": 51.0062,
                "id": 311,
                "partition": [
                    -0.334142,
                    0.0116107,
                    -6.99304e-05,
                    2.49526e-07
                ]
            }
        },
        "mmw": 50.07975052863
    },
    "31": {
        "name": "HC3N",
        "isotope": {
            "1": {
                "abun": 0.98525,
                "mass": 51.0109,
                "id": 124,
                "partition": [
                    -0.155444,
                    0.00725791,
                    -3.58075e-05,
                    1.59203e-07
                ]
            },
            "2": {
                "abun": 0.0111,
                "mass": 52.0143,
                "id": 134,
                "partition": [
                    -0.155444,
                    0.00725791,
                    -3.58075e-05,
                    1.59203e-07
                ]
            },
            "3": {
                "abun": 0.00365,
                "mass": 52.0079,
                "id": 125,
                "partition": [
                    -0.155444,
                    0.00725791,
                    -3.58075e-05,
                    1.59203e-07
                ]
            }
        },
        "mmw": 51.02567679
    },
    "32": {
        "name": "C2H4",
        "isotope": {
            "1": {
                "abun": 0.977294,
                "mass": 28.0313,
                "id": 211,
                "partition": [
                    -533.427,
                    21.6816,
                    0.0353904,
                    7.87332e-05
                ]
            },
            "2": {
                "abun": 0.0219595,
                "mass": 29.0347,
                "id": 311,
                "partition": [
                    -2182.96,
                    88.7363,
                    0.14492,
                    0.000322243
                ]
            }
        },
        "mmw": 28.03240879685
    },
    "33": {
        "name": "GeH4",
        "isotope": {
            "1": {
                "abun": 0.3594,
                "mass": 78.0,
                "id": 411,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            },
            "2": {
                "abun": 0.2766,
                "mass": 76.0,
                "id": 211,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            },
            "3": {
                "abun": 0.2123,
                "mass": 74.0,
                "id": 11,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            },
            "4": {
                "abun": 0.0773,
                "mass": 77.0,
                "id": 311,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            },
            "5": {
                "abun": 0.0744,
                "mass": 80.0,
                "id": 611,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            }
        },
        "mmw": 76.6691
    },
    "34": {
        "name": "C3H8",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 44.0,
                "id": 211,
                "partition": [
                    -0.0886492,
                    0.00221492,
                    -6.74559e-06,
                    5.12411e-08
                ]
            }
        },
        "mmw": 44.0
    },
    "35": {
        "name": "HCOOH",
        "isotope": {
            "1": {
                "abun": 0.983898,
                "mass": 46.0055,
                "id": 261,
                "partition": [
                    -1984.17,
                    78.9531,
                    0.0451356,
                    0.000530188
                ]
            }
        },
        "mmw": 45.264719439
    },
    "36": {
        "name": "H2S",
        "isotope": {
            "1": {
                "abun": 0.949884,
                "mass": 33.9877,
                "id": 121,
                "partition": [
                    -12.0797,
                    0.742684,
                    0.00401,
                    -2.16779e-06
                ]
            },
            "2": {
                "abun": 0.0074977,
                "mass": 34.9871,
                "id": 131,
                "partition": [
                    -48.4085,
                    2.97512,
                    0.0160583,
                    -8.67821e-06
                ]
            },
            "3": {
                "abun": 0.0421369,
                "mass": 35.9835,
                "id": 141,
                "partition": [
                    -12.1068,
                    0.744435,
                    0.00402107,
                    -2.17485e-06
                ]
            }
        },
        "mmw": 34.06292834762
    },
    "37": {
        "name": "COF2",
        "isotope": {
            "1": {
                "abun": 0.986544,
                "mass": 65.9917,
                "id": 269,
                "partition": [
                    -3890.09,
                    148.981,
                    -0.0388424,
                    0.0012786
                ]
            }
        },
        "mmw": 65.1037156848
    },
    "38": {
        "name": "SF6",
        "isotope": {
            "1": {
                "abun": 0.95018,
                "mass": 145.9625,
                "id": 29,
                "partition": [
                    -398561.0,
                    10139.6,
                    -70.363,
                    0.199113
                ]
            }
        },
        "mmw": 138.69064825
    },
    "39": {
        "name": "H2",
        "isotope": {
            "1": {
                "abun": 0.999955,
                "mass": 2.016,
                "id": 11,
                "partition": [
                    0.0,
                    1.0,
                    0.0,
                    0.0
                ]
            },
            "2": {
                "abun": 4.5e-05,
                "mass": 3.024,
                "id": 12,
                "partition": [
                    0.0,
                    1.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 2.01604536
    },
    "40": {
        "name": "He",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 4.0021,
                "id": 4,
                "partition": [
                    0.0,
                    1.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 4.0021
    },
    "41": {
        "name": "AsH3",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 78.5,
                "id": 181,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            }
        },
        "mmw": 78.5
    },
    "42": {
        "name": "C3H4",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 40.0,
                "id": 341,
                "partition": [
                    -0.0893543,
                    0.00327791,
                    -1.05124e-05,
                    6.78358e-08
                ]
            }
        },
        "mmw": 40.0
    },
    "43": {
        "name": "ClONO2",
        "isotope": {
            "1": {
                "abun": 0.74957,
                "mass": 96.9567,
                "id": 564,
                "partition": [
                    -555568.0,
                    14641.4,
                    -97.6502,
                    0.367867
                ]
            },
            "2": {
                "abun": 0.239694,
                "mass": 98.9537,
                "id": 764,
                "partition": [
                    -569803.0,
                    15016.1,
                    -100.149,
                    0.377253
                ]
            }
        },
        "mmw": 96.39444178679999
    },
    "44": {
        "name": "HO2",
        "isotope": {
            "1": {
                "abun": 0.995107,
                "mass": 32.9977,
                "id": 166,
                "partition": [
                    -117.959,
                    6.45327,
                    0.0331743,
                    -1.54409e-05
                ]
            }
        },
        "mmw": 32.836242253900004
    },
    "45": {
        "name": "O",
        "isotope": {
            "1": {
                "abun": 0.997628,
                "mass": 15.9949,
                "id": 6,
                "partition": [
                    -1.0,
                    0.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 15.9569600972
    },
    "46": {
        "name": "NO+",
        "isotope": {
            "1": {
                "abun": 0.993974,
                "mass": 29.998,
                "id": 4,
                "partition": [
                    1.04596,
                    1.04827,
                    5.65417e-06,
                    -5.58099e-09
                ]
            }
        },
        "mmw": 29.817232052
    },
    "47": {
        "name": "CH3OH",
        "isotope": {
            "1": {
                "abun": 0.98593,
                "mass": 32.0262,
                "id": 211,
                "partition": [
                    -4437.1,
                    172.219,
                    0.137732,
                    0.000915583
                ]
            }
        },
        "mmw": 31.575591366
    },
    "48": {
        "name": "H",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 1.00794,
                "id": 1,
                "partition": [
                    -1.0,
                    0.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 1.00794
    },
    "49": {
        "name": "C6H6",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 78.11,
                "id": 221,
                "partition": [
                    -0.173943,
                    0.00780648,
                    -3.77745e-05,
                    1.14948e-07
                ]
            }
        },
        "mmw": 78.11
    },
    "50": {
        "name": "CH3CN",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 41.05,
                "id": 241,
                "partition": [
                    -0.0627792,
                    0.00255606,
                    -3.70671e-06,
                    4.36134e-08
                ]
            }
        },
        "mmw": 41.05
    },
    "51": {
        "name": "CH2NH",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 29.0,
                "id": 241,
                "partition": [
                    -44.5427,
                    6.52519,
                    0.0563892,
                    -4.07146e-05
                ]
            },
            "2": {
                "abun": 0.01053,
                "mass": 30.0,
                "id": 341,
                "partition": [
                    -13.1524,
                    2.04989,
                    0.0211016,
                    -1.81248e-05
                ]
            },
            "3": {
                "abun": 0.01538,
                "mass": 30.0,
                "id": 251,
                "partition": [
                    -13.1737,
                    2.0532,
                    0.0211356,
                    -1.81539e-05
                ]
            },
            "4": {
                "abun": 0.000117,
                "mass": 30.0,
                "id": 242,
                "partition": [
                    -28.6027,
                    5.28362,
                    0.0918531,
                    -9.81119e-05
                ]
            }
        },
        "mmw": 29.78081
    },
    "52": {
        "name": "C2H3CN",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 53.0,
                "id": 241,
                "partition": [
                    -154.793,
                    58.8874,
                    0.93928,
                    0.000643523
                ]
            },
            "2": {
                "abun": 0.01053,
                "mass": 54.0,
                "id": 441,
                "partition": [
                    -170.787,
                    26.8203,
                    0.276754,
                    -0.000234522
                ]
            },
            "3": {
                "abun": 0.01053,
                "mass": 54.0,
                "id": 541,
                "partition": [
                    -172.984,
                    27.1647,
                    0.280301,
                    -0.000237552
                ]
            },
            "4": {
                "abun": 0.01053,
                "mass": 54.0,
                "id": 641,
                "partition": [
                    -175.702,
                    27.5884,
                    0.284693,
                    -0.000241259
                ]
            },
            "5": {
                "abun": 0.000117,
                "mass": 54.0,
                "id": 242,
                "partition": [
                    -190.29,
                    30.0907,
                    0.314149,
                    -0.000271996
                ]
            },
            "6": {
                "abun": 0.01538,
                "mass": 54.0,
                "id": 251,
                "partition": [
                    -175.435,
                    27.546,
                    0.284265,
                    -0.000240851
                ]
            }
        },
        "mmw": 55.54269800000001
    },
    "53": {
        "name": "HCP",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 44.0,
                "id": 123,
                "partition": [
                    0.407958,
                    1.03821,
                    6.67295e-05,
                    -2.16079e-07
                ]
            }
        },
        "mmw": 44.0
    },
    "54": {
        "name": "CS",
        "isotope": {
            "1": {
                "abun": 0.947333,
                "mass": 44.0,
                "id": 22,
                "partition": [
                    0.353804,
                    0.849631,
                    1.39364e-05,
                    -3.6623e-08
                ]
            },
            "2": {
                "abun": 0.010643,
                "mass": 44.0,
                "id": 32,
                "partition": [
                    0.368163,
                    0.899182,
                    2.41945e-05,
                    -6.10961e-08
                ]
            },
            "3": {
                "abun": 0.042024,
                "mass": 44.0,
                "id": 24,
                "partition": [
                    0.358717,
                    0.86316,
                    1.65761e-05,
                    -4.21693e-08
                ]
            }
        },
        "mmw": 43.99999999999999
    },
    "55": {
        "name": "HC5N",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 75.0,
                "id": 241,
                "partition": [
                    0.333717,
                    15.6509,
                    1.10945e-05,
                    -2.04631e-11
                ]
            }
        },
        "mmw": 75.0
    },
    "56": {
        "name": "HC7N",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 99.0,
                "id": 241,
                "partition": [
                    -246.6,
                    130.5,
                    -0.27785,
                    0.000216506
                ]
            }
        },
        "mmw": 99.0
    },
    "57": {
        "name": "C2H5CN",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 55.0,
                "id": 241,
                "partition": [
                    -242.048,
                    37.9879,
                    0.391607,
                    -0.000332892
                ]
            },
            "2": {
                "abun": 0.01053,
                "mass": 56.0,
                "id": 441,
                "partition": [
                    -922.629,
                    110.609,
                    1.14061,
                    -0.000839605
                ]
            },
            "3": {
                "abun": 0.01053,
                "mass": 56.0,
                "id": 541,
                "partition": [
                    -932.681,
                    111.813,
                    1.15301,
                    -0.000848733
                ]
            },
            "4": {
                "abun": 0.01053,
                "mass": 56.0,
                "id": 641,
                "partition": [
                    -946.039,
                    113.398,
                    1.16931,
                    -0.000860716
                ]
            },
            "5": {
                "abun": 0.01538,
                "mass": 56.0,
                "id": 251,
                "partition": [
                    6.18308,
                    18.21,
                    0.568471,
                    -0.000674954
                ]
            }
        },
        "mmw": 57.630320000000005
    },
    "58": {
        "name": "CH3NH2",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 31.0,
                "id": 124,
                "partition": [
                    -813.72,
                    126.598,
                    1.30776,
                    -0.00111831
                ]
            }
        },
        "mmw": 31.0
    },
    "59": {
        "name": "HNC",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 27.0,
                "id": 142,
                "partition": [
                    0.349399,
                    0.459311,
                    2.92041e-06,
                    -3.51095e-09
                ]
            },
            "2": {
                "abun": 0.01053,
                "mass": 28.0,
                "id": 143,
                "partition": [
                    0.357422,
                    0.477576,
                    9.6211e-06,
                    -2.27844e-08
                ]
            },
            "3": {
                "abun": 0.01538,
                "mass": 28.0,
                "id": 152,
                "partition": [
                    0.353799,
                    0.46832,
                    6.59346e-06,
                    -1.42053e-08
                ]
            },
            "4": {
                "abun": 0.000117,
                "mass": 28.0,
                "id": 242,
                "partition": [
                    0.34557,
                    0.545889,
                    2.50195e-06,
                    -2.46926e-09
                ]
            }
        },
        "mmw": 27.728756
    },
    "60": {
        "name": "Na",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 22.99,
                "id": 11,
                "partition": [
                    4.39393,
                    -0.00433822,
                    2.39818e-06,
                    -4.12438e-10
                ]
            }
        },
        "mmw": 22.99
    },
    "61": {
        "name": "K",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 39.1,
                "id": 19,
                "partition": [
                    6.01454,
                    -0.00722715,
                    3.97413e-06,
                    -6.80502e-10
                ]
            }
        },
        "mmw": 39.1
    },
    "62": {
        "name": "TiO",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 63.8664,
                "id": 486,
                "partition": [
                    -860.00527,
                    5.25794,
                    0.00411768,
                    4.44797e-07
                ]
            }
        },
        "mmw": 63.8664
    },
    "63": {
        "name": "VO",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 66.9409,
                "id": 516,
                "partition": [
                    -922.20713,
                    4.96349,
                    0.0024269,
                    1.66643e-07
                ]
            }
        },
        "mmw": 66.9409
    },
    "64": {
        "name": "CH2CCH2",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 40.0646,
                "id": 221,
                "partition": [
                    -0.104157,
                    0.00355223,
                    -1.14095e-05,
                    6.36864e-08
                ]
            }
        },
        "mmw": 40.0646
    },
    "65": {
        "name": "C4N2",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 76.0566,
                "id": 224,
                "partition": [
                    -1.92827,
                    0.0572307,
                    -0.000488527,
                    1.72324e-06
                ]
            }
        },
        "mmw": 76.0566
    },
    "66": {
        "name": "C5H5N",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 79.1,
                "id": 241,
                "partition": [
                    -559.899,
                    87.6703,
                    0.901869,
                    -0.000772987
                ]
            }
        },
        "mmw": 79.1
    },
    "67": {
        "name": "C5H4N2",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 80.088,
                "id": 241,
                "partition": [
                    -526.099,
                    82.396,
                    0.847528,
                    -0.000726356
                ]
            }
        },
        "mmw": 80.088
    },
    "68": {
        "name": "C7H8",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 92.14,
                "id": 221,
                "partition": [
                    -1127.29,
                    177.06,
                    1.8376,
                    -0.00159469
                ]
            }
        },
        "mmw": 92.14
    },
    "69": {
        "name": "C8H6",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 102.133,
                "id": 221,
                "partition": [
                    -1655.61,
                    268.028,
                    2.94334,
                    -0.00278395
                ]
            }
        },
        "mmw": 102.133
    },
    "70": {
        "name": "C5H5CN",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 103.12,
                "id": 241,
                "partition": [
                    -7009.17,
                    1095.06,
                    11.2784,
                    -0.0096696
                ]
            }
        },
        "mmw": 103.12
    },
    "71": {
        "name": "HOBr",
        "isotope": {
            "1": {
                "abun": 0.505579,
                "mass": 95.921076,
                "id": 169,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            },
            "2": {
                "abun": 0.491894,
                "mass": 97.919027,
                "id": 161,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            }
        },
        "mmw": 96.661463550142
    },
    "72": {
        "name": "CH3Br",
        "isotope": {
            "1": {
                "abun": 0.500995,
                "mass": 93.941811,
                "id": 219,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            },
            "2": {
                "abun": 0.487433,
                "mass": 95.939764,
                "id": 211,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            }
        },
        "mmw": 93.828584587757
    },
    "73": {
        "name": "CF4",
        "isotope": {
            "1": {
                "abun": 0.98889,
                "mass": 87.993616,
                "id": 29,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            }
        },
        "mmw": 87.01600692624001
    },
    "74": {
        "name": "SO3",
        "isotope": {
            "1": {
                "abun": 0.9434,
                "mass": 79.95682,
                "id": 26,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            }
        },
        "mmw": 75.431263988
    },
    "75": {
        "name": "Ne",
        "isotope": {
            "1": {
                "abun": 0.9048,
                "mass": 19.99244,
                "id": 20,
                "partition": [
                    1.0,
                    0.0,
                    0.0,
                    0.0
                ]
            },
            "2": {
                "abun": 0.0925,
                "mass": 21.991385,
                "id": 22,
                "partition": [
                    1.0,
                    0.0,
                    0.0,
                    0.0
                ]
            },
            "3": {
                "abun": 0.0027,
                "mass": 20.993847,
                "id": 21,
                "partition": [
                    1.0,
                    0.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 20.1800462114
    },
    "76": {
        "name": "Ar",
        "isotope": {
            "1": {
                "abun": 0.996035,
                "mass": 39.962383,
                "id": 40,
                "partition": [
                    1.0,
                    0.0,
                    0.0,
                    0.0
                ]
            },
            "2": {
                "abun": 0.003336,
                "mass": 35.967545,
                "id": 36,
                "partition": [
                    1.0,
                    0.0,
                    0.0,
                    0.0
                ]
            },
            "3": {
                "abun": 0.000629,
                "mass": 37.962732,
                "id": 38,
                "partition": [
                    1.0,
                    0.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 39.947798439953004
    },
    "77": {
        "name": "COCl2",
        "isotope": {
            "1": {
                "abun": 0.566392,
                "mass": 97.93262,
                "id": 2655,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            },
            "2": {
                "abun": 0.362235,
                "mass": 99.92967,
                "id": 2657,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            }
        },
        "mmw": 91.66627651949
    },
    "78": {
        "name": "SO",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 48.064,
                "id": 3616,
                "partition": [
                    0.0,
                    1.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 48.064
    },
    "79": {
        "name": "H2SO4",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 98.079,
                "id": 13616,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            }
        },
        "mmw": 98.079
    },
    "80": {
        "name": "e-",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 0.000545,
                "id": 111,
                "partition": [
                    1.0,
                    0.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 0.000545
    },
    "81": {
        "name": "H3+",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 3.02352,
                "id": 111,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            }
        },
        "mmw": 3.02352
    },
    "82": {
        "name": "FeH",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 56.85284,
                "id": 61,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 56.85284
    },
    "83": {
        "name": "AlO",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 42.980539,
                "id": 76,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 42.980539
    },
    "84": {
        "name": "AlCl",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 62.434539,
                "id": 75,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 62.434539
    },
    "85": {
        "name": "AlF",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 45.979942,
                "id": 79,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 45.979942
    },
    "86": {
        "name": "AlH",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 27.989379,
                "id": 71,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 27.989379
    },
    "87": {
        "name": "BeH",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 10.020022,
                "id": 91,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 10.020022
    },
    "88": {
        "name": "C2",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 24.0214,
                "id": 22,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 24.0214
    },
    "89": {
        "name": "CaF",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 59.076403,
                "id": 409,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 59.076403
    },
    "90": {
        "name": "CaH",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 41.08584,
                "id": 401,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 41.08584
    },
    "91": {
        "name": "H-",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 1.008,
                "id": 111,
                "partition": [
                    1.0,
                    0.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 1.008
    },
    "92": {
        "name": "CaO",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 56.077,
                "id": 406,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 56.077
    },
    "93": {
        "name": "CH",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 13.01854,
                "id": 21,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 13.01854
    },
    "94": {
        "name": "CH3",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 15.03422,
                "id": 211,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            }
        },
        "mmw": 15.03422
    },
    "95": {
        "name": "CH3F",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 34.032623,
                "id": 219,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            }
        },
        "mmw": 34.032623
    },
    "96": {
        "name": "CN",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 26.0174,
                "id": 24,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 26.0174
    },
    "97": {
        "name": "CP",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 42.984462,
                "id": 231,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 42.984462
    },
    "98": {
        "name": "CrH",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 53.00394,
                "id": 521,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 53.00394
    },
    "99": {
        "name": "HD+",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 3.02194178,
                "id": 11,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 3.02194178
    },
    "100": {
        "name": "HeH+",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 5.010442,
                "id": 41,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 5.010442
    },
    "101": {
        "name": "KCl",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 74.5513,
                "id": 395,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 74.5513
    },
    "102": {
        "name": "KF",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 58.096703,
                "id": 399,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 58.096703
    },
    "103": {
        "name": "LiCl",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 42.394,
                "id": 75,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 42.394
    },
    "104": {
        "name": "LiF",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 25.939403,
                "id": 79,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 25.939403
    },
    "105": {
        "name": "LiH",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 7.94884,
                "id": 71,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 7.94884
    },
    "106": {
        "name": "LiH+",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 7.94884,
                "id": 71,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 7.94884
    },
    "107": {
        "name": "MgF",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 43.303403,
                "id": 249,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 43.303403
    },
    "108": {
        "name": "MgH",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 25.31284,
                "id": 241,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 25.31284
    },
    "109": {
        "name": "MgO",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 40.304,
                "id": 246,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 40.304
    },
    "110": {
        "name": "NaCl",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 58.4427693,
                "id": 235,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 58.4427693
    },
    "111": {
        "name": "NaF",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 41.9881723,
                "id": 239,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 41.9881723
    },
    "112": {
        "name": "NaH",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 23.9976093,
                "id": 231,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 23.9976093
    },
    "113": {
        "name": "NH",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 15.01454,
                "id": 41,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 15.01454
    },
    "114": {
        "name": "NS",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 46.0717,
                "id": 42,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 46.0717
    },
    "115": {
        "name": "OH+",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 17.00684,
                "id": 61,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 17.00684
    },
    "116": {
        "name": "cis-P2H2",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 63.963204,
                "id": 311,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 63.963204
    },
    "117": {
        "name": "trans-P2H2",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 63.963204,
                "id": 311,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 63.963204
    },
    "118": {
        "name": "PH",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 31.981602,
                "id": 311,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 31.981602
    },
    "119": {
        "name": "PN",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 44.980462,
                "id": 314,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 44.980462
    },
    "120": {
        "name": "PO",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 46.972762,
                "id": 316,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 46.972762
    },
    "121": {
        "name": "PS",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 63.038762,
                "id": 312,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 63.038762
    },
    "122": {
        "name": "ScH",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 45.963752,
                "id": 451,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 45.963752
    },
    "123": {
        "name": "SH",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 33.07284,
                "id": 21,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 33.07284
    },
    "124": {
        "name": "SiH",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 29.09334,
                "id": 281,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 29.09334
    },
    "125": {
        "name": "SiH2",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 30.10118,
                "id": 2811,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            }
        },
        "mmw": 30.10118
    },
    "126": {
        "name": "SiH4",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 32.11686,
                "id": 2811,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            }
        },
        "mmw": 32.11686
    },
    "127": {
        "name": "SiO",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 44.0845,
                "id": 286,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 44.0845
    },
    "128": {
        "name": "SiO2",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 60.0835,
                "id": 2866,
                "partition": [
                    -44.219,
                    5.8047,
                    0.050145,
                    -3.7316e-05
                ]
            }
        },
        "mmw": 60.0835
    },
    "129": {
        "name": "SiS",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 60.1505,
                "id": 282,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 60.1505
    },
    "130": {
        "name": "TiH",
        "isotope": {
            "1": {
                "abun": 1.0,
                "mass": 48.87484,
                "id": 481,
                "partition": [
                    0.0,
                    10.0,
                    0.0,
                    0.0
                ]
            }
        },
        "mmw": 48.87484
    }
}