from __future__ import annotations
from typing import TYPE_CHECKING, Optional

if TYPE_CHECKING:
    from .servo import Servo


class _EEPROMRegisters:
    def __init__(self, servo: "Servo"):
        self.servo = servo

    def read_firmware_major_version(self) -> Optional[int]:
        """
        Read the firmware major version number.

        Returns:
            int: Major version number, or None if read fails
        """
        return self.servo._read_memory(0x00, 1)

    def read_firmware_minor_version(self) -> Optional[int]:
        """
        Read the firmware minor version number.

        Returns:
            int: Minor version number, or None if read fails
        """
        return self.servo._read_memory(0x01, 1)

    def read_servo_main_version(self) -> Optional[int]:
        """
        Read the servo main version number.

        Returns:
            int: Main version number, or None if read fails
        """
        return self.servo._read_memory(0x03, 1)

    def read_servo_version(self) -> Optional[int]:
        """
        Read the servo version number.

        Returns:
            int: Servo version number, or None if read fails
        """
        return self.servo._read_memory(0x04, 1)

    def read_id(self) -> Optional[int]:
        """
        Read the servo ID.

        Returns:
            int: Servo ID (0-253), or None if read fails
        """
        return self.servo._read_memory(0x05, 1)

    def write_id(self, value: int, reg: bool = False):
        """
        Set the servo ID.

        Args:
            value (int): Servo ID (0-253). 254 is reserved for broadcast.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x05, [value & 0xFF])

    def read_baudrate(self) -> Optional[int]:
        """
        Read the baud rate setting.

        Returns:
            int: Baud rate code (0-7), where:
                 0 = 1,000,000 baud
                 1 = 500,000 baud
                 2 = 250,000 baud
                 3 = 115,200 baud
                 4 = 57,600 baud
                 5 = 38,400 baud
                 6 = 19,200 baud
                 7 = 9,600 baud
            Returns None if read fails
        """
        return self.servo._read_memory(0x06, 1)

    def write_baudrate(self, value: int, reg: bool = False):
        """
        Set the baud rate.

        Args:
            value (int): Baud rate code (0-7). See read_baudrate for mapping.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x06, [value & 0xFF])

    def read_return_delay(self) -> Optional[int]:
        """
        Read the return delay time.

        Returns:
            int: Delay in 2µs units (0-254). Max = 508µs. Returns None if read fails.
        """
        return self.servo._read_memory(0x07, 1)

    def write_return_delay(self, value: int, reg: bool = False):
        """
        Set the return delay time.

        Args:
            value (int): Delay in 2µs units (0-254). Max configurable = 508µs.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x07, [value & 0xFF])

    def read_response_status_level(self) -> Optional[int]:
        """
        Read the response status level.

        Returns:
            int: 0 = No response except for READ/PING commands
                 1 = Response returned for all commands
            Returns None if read fails
        """
        return self.servo._read_memory(0x08, 1)

    def write_response_status_level(self, value: int, reg: bool = False):
        """
        Set the response status level.

        Args:
            value (int): 0 = No response except READ/PING
                        1 = Response for all commands
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x08, [value & 0xFF])

    def read_min_angle_limit(self) -> Optional[int]:
        """
        Read the minimum angle limit.

        Returns:
            int: Minimum position in steps (0-4094), or None if read fails.
                 Must be less than max_angle_limit.
        """
        return self.servo._read_memory(0x09, 2)

    def write_min_angle_limit(self, value: int, reg: bool = False):
        """
        Set the minimum angle limit.

        Args:
            value (int): Minimum position in steps (0-4094).
                        Must be less than max_angle_limit.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        low = value & 0xFF
        high = (value >> 8) & 0xFF
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x09, [low, high])

    def read_max_angle_limit(self) -> Optional[int]:
        """
        Read the maximum angle limit.

        Returns:
            int: Maximum position in steps (1-4095), or None if read fails.
                 Must be greater than min_angle_limit.
        """
        return self.servo._read_memory(0x0B, 2)

    def write_max_angle_limit(self, value: int, reg: bool = False):
        """
        Set the maximum angle limit.

        Args:
            value (int): Maximum position in steps (1-4095).
                        Must be greater than min_angle_limit.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        low = value & 0xFF
        high = (value >> 8) & 0xFF
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x0B, [low, high])

    def read_max_temperature_limit(self) -> Optional[int]:
        """
        Read the maximum temperature limit.

        Returns:
            int: Temperature in °C (0-100), or None if read fails.
        """
        return self.servo._read_memory(0x0D, 1)

    def write_max_temperature_limit(self, value: int, reg: bool = False):
        """
        Set the maximum temperature limit.

        Args:
            value (int): Temperature in °C (0-100).
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x0D, [value & 0xFF])

    def read_max_input_voltage(self) -> Optional[int]:
        """
        Read the maximum input voltage limit.

        Returns:
            int: Voltage in 0.1V units (0-254). Example: 120 = 12.0V.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x0E, 1)

    def write_max_input_voltage(self, value: int, reg: bool = False):
        """
        Set the maximum input voltage limit.

        Args:
            value (int): Voltage in 0.1V units (0-254). Example: 120 = 12.0V.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x0E, [value & 0xFF])

    def read_min_input_voltage(self) -> Optional[int]:
        """
        Read the minimum input voltage limit.

        Returns:
            int: Voltage in 0.1V units (0-254). Example: 60 = 6.0V.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x0F, 1)

    def write_min_input_voltage(self, value: int, reg: bool = False):
        """
        Set the minimum input voltage limit.

        Args:
            value (int): Voltage in 0.1V units (0-254). Example: 60 = 6.0V.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x0F, [value & 0xFF])

    def read_max_torque(self) -> Optional[int]:
        """
        Read the maximum torque limit.

        Returns:
            int: Torque in 0.1% units (0-1000). Example: 500 = 50.0%.
                 This value is assigned to torque_limit on power-up.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x10, 2)

    def write_max_torque(self, value: int, reg: bool = False):
        """
        Set the maximum torque limit.

        Args:
            value (int): Torque in 0.1% units (0-1000). Example: 500 = 50.0%.
                        This value is assigned to torque_limit on power-up.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        low = value & 0xFF
        high = (value >> 8) & 0xFF
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x10, [low, high])

    def read_phase(self) -> Optional[int]:
        """
        Read the phase register (special function).

        Returns:
            int: Phase value (0-254), or None if read fails.
                 Do not modify unless required.
        """
        return self.servo._read_memory(0x12, 1)

    def write_phase(self, value: int, reg: bool = False):
        """
        Set the phase register (special function).

        Args:
            value (int): Phase value (0-254). Do not modify unless required.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x12, [value & 0xFF])

    def read_uninstallation_conditions(self) -> Optional[int]:
        """
        Read the uninstallation conditions (protection settings).

        Returns:
            int: Bitmask (0-254) where bit 1 enables/disables corresponding protection.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x13, 1)

    def write_uninstallation_conditions(self, value: int, reg: bool = False):
        """
        Set the uninstallation conditions (protection settings).

        Args:
            value (int): Bitmask (0-254) where bit 1 enables/disables protection.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x13, [value & 0xFF])

    def read_led_alarm_conditions(self) -> Optional[int]:
        """
        Read the LED alarm conditions.

        Returns:
            int: Bitmask (0-254) where bit 1 enables/disables LED flashing alarm.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x14, 1)

    def write_led_alarm_conditions(self, value: int, reg: bool = False):
        """
        Set the LED alarm conditions.

        Args:
            value (int): Bitmask (0-254) where bit 1 enables/disables LED alarm.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x14, [value & 0xFF])

    def read_position_p(self) -> Optional[int]:
        """
        Read the position control loop P (proportional) coefficient.

        Returns:
            int: P coefficient (0-254), or None if read fails.
        """
        return self.servo._read_memory(0x15, 1)

    def write_position_p(self, value: int, reg: bool = False):
        """
        Set the position control loop P (proportional) coefficient.

        Args:
            value (int): P coefficient (0-254).
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x15, [value & 0xFF])

    def read_position_d(self) -> Optional[int]:
        """
        Read the position control loop D (differential) coefficient.

        Returns:
            int: D coefficient (0-254), or None if read fails.
        """
        return self.servo._read_memory(0x16, 1)

    def write_position_d(self, value: int, reg: bool = False):
        """
        Set the position control loop D (differential) coefficient.

        Args:
            value (int): D coefficient (0-254).
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x16, [value & 0xFF])

    def read_position_i(self) -> Optional[int]:
        """
        Read the position control loop I (integral) coefficient.

        Returns:
            int: I coefficient (0-254), or None if read fails.
        """
        return self.servo._read_memory(0x17, 1)

    def write_position_i(self, value: int, reg: bool = False):
        """
        Set the position control loop I (integral) coefficient.

        Args:
            value (int): I coefficient (0-254).
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x17, [value & 0xFF])

    def read_min_starting_force(self) -> Optional[int]:
        """
        Read the minimum starting force.

        Returns:
            int: Force in 0.1% units (0-254). Example: 10 = 1% of stall torque.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x18, 1)

    def write_min_starting_force(self, value: int, reg: bool = False):
        """
        Set the minimum starting force.

        Args:
            value (int): Force in 0.1% units (0-254). Example: 10 = 1% stall torque.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x18, [value & 0xFF])

    def read_points_limit(self) -> Optional[int]:
        """
        Read the points limit.

        Returns:
            int: Points limit value (0-254). Max score = value * 4.
                 0 disables the limit. Returns None if read fails.
        """
        return self.servo._read_memory(0x19, 1)

    def write_points_limit(self, value: int, reg: bool = False):
        """
        Set the points limit.

        Args:
            value (int): Points limit (0-254). Max score = value * 4.
                        0 disables the limit.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x19, [value & 0xFF])

    def read_cw_insensitive_area(self) -> Optional[int]:
        """
        Read the clockwise insensitive area (deadzone).

        Returns:
            int: Deadzone in steps (0-32). Smallest unit is minimum resolution angle.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x1A, 1)

    def write_cw_insensitive_area(self, value: int, reg: bool = False):
        """
        Set the clockwise insensitive area (deadzone).

        Args:
            value (int): Deadzone in steps (0-32).
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x1A, [value & 0xFF])

    def read_ccw_insensitive_area(self) -> Optional[int]:
        """
        Read the counterclockwise insensitive area (deadzone).

        Returns:
            int: Deadzone in steps (0-32). Smallest unit is minimum resolution angle.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x1B, 1)

    def write_ccw_insensitive_area(self, value: int, reg: bool = False):
        """
        Set the counterclockwise insensitive area (deadzone).

        Args:
            value (int): Deadzone in steps (0-32).
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x1B, [value & 0xFF])

    def read_protective_current(self) -> Optional[int]:
        """
        Read the protective current limit.

        Returns:
            int: Current in 6.5mA units (0-511). Max = 500*6.5mA = 3250mA.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x1C, 2)

    def write_protective_current(self, value: int, reg: bool = False):
        """
        Set the protective current limit.

        Args:
            value (int): Current in 6.5mA units (0-511). Max = 3250mA.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        low = value & 0xFF
        high = (value >> 8) & 0xFF
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x1C, [low, high])

    def read_angular_resolution(self) -> Optional[int]:
        """
        Read the angular resolution setting.

        Returns:
            int: Resolution mode (1-3). Modifies magnification factor for
                 sensor minimum resolution angle. Returns None if read fails.
        """
        return self.servo._read_memory(0x1E, 1)

    def write_angular_resolution(self, value: int, reg: bool = False):
        """
        Set the angular resolution.

        Args:
            value (int): Resolution mode (1-3).
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x1E, [value & 0xFF])

    def read_position_correction(self) -> Optional[int]:
        """
        Read the position correction offset.

        Returns:
            int: Position offset in steps (-2047 to +2047), or None if read fails.
        """
        raw = self.servo._read_memory(0x1F, 2)
        if raw is None:
            return None
        if raw & 0x800:
            return raw - 4096
        return raw

    def write_position_correction(self, value: int, reg: bool = False):
        """
        Set the position correction offset.

        Args:
            value (int): Position offset in steps (-2047 to +2047).
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        if value < 0:
            raw = (abs(value) & 0x7FF) | 0x800
        else:
            raw = value & 0x7FF
        low = raw & 0xFF
        high = (raw >> 8) & 0xFF
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x1F, [low, high])

    def read_operating_mode(self) -> Optional[int]:
        """
        Read the operating mode.

        Returns:
            int: Mode (0-3):
                 0 = Position control mode
                 1 = Constant speed mode
                 2 = PWM open-loop mode
                 3 = Stepper servo mode
            Returns None if read fails.
        """
        return self.servo._read_memory(0x21, 1)

    def write_operating_mode(self, value: int, reg: bool = False):
        """
        Set the operating mode.

        Args:
            value (int): Mode (0-3):
                        0 = Position control mode
                        1 = Constant speed mode
                        2 = PWM open-loop mode
                        3 = Stepper servo mode
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x21, [value & 0xFF])

    def read_protective_torque(self) -> Optional[int]:
        """
        Read the protective torque setting.

        Returns:
            int: Torque in 1.0% units (0-100). Output torque after overload protection.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x22, 1)

    def write_protective_torque(self, value: int, reg: bool = False):
        """
        Set the protective torque.

        Args:
            value (int): Torque in 1.0% units (0-100).
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x22, [value & 0xFF])

    def read_protection_time(self) -> Optional[int]:
        """
        Read the protection time.

        Returns:
            int: Time in 10ms units (0-254). Duration exceeding overload torque
                 before reset. Returns None if read fails.
        """
        return self.servo._read_memory(0x23, 1)

    def write_protection_time(self, value: int, reg: bool = False):
        """
        Set the protection time.

        Args:
            value (int): Time in 10ms units (0-254). Max = 2540ms.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x23, [value & 0xFF])

    def read_overload_torque(self) -> Optional[int]:
        """
        Read the overload torque threshold.

        Returns:
            int: Torque in 1.0% units (0-100). Max torque threshold for
                 overload protection. Returns None if read fails.
        """
        return self.servo._read_memory(0x24, 1)

    def write_overload_torque(self, value: int, reg: bool = False):
        """
        Set the overload torque threshold.

        Args:
            value (int): Torque in 1.0% units (0-100).
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x24, [value & 0xFF])

    def read_speed_p(self) -> Optional[int]:
        """
        Read the speed control loop P (proportional) coefficient.

        Returns:
            int: P coefficient (0-254) for constant speed mode.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x25, 1)

    def write_speed_p(self, value: int, reg: bool = False):
        """
        Set the speed control loop P (proportional) coefficient.

        Args:
            value (int): P coefficient (0-254) for constant speed mode.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x25, [value & 0xFF])

    def read_overcurrent_protection_time(self) -> Optional[int]:
        """
        Read the overcurrent protection time.

        Returns:
            int: Time in 10ms units (0-254). Max = 2540ms.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x26, 1)

    def write_overcurrent_protection_time(self, value: int, reg: bool = False):
        """
        Set the overcurrent protection time.

        Args:
            value (int): Time in 10ms units (0-254). Max = 2540ms.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x26, [value & 0xFF])

    def read_speed_i(self) -> Optional[int]:
        """
        Read the speed control loop I (integral) coefficient.

        Returns:
            int: I coefficient (0-254) in 1/10 units. Reduced by factor 10 vs v3.6.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x27, 1)

    def write_speed_i(self, value: int, reg: bool = False):
        """
        Set the speed control loop I (integral) coefficient.

        Args:
            value (int): I coefficient (0-254) in 1/10 units.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x27, [value & 0xFF])


class SRAMRegisters:
    def __init__(self, servo: "Servo"):
        self.servo = servo

    def read_torque_switch(self) -> Optional[int]:
        """
        Read the torque switch state.

        Returns:
            int: 0 = Torque off
                 1 = Torque on
                 128 = Correct current position to 2048
            Returns None if read fails.
        """
        return self.servo._read_memory(0x28, 1)

    def write_torque_switch(self, value: int, reg: bool = False):
        """
        Set the torque switch state.

        Args:
            value (int): 0 = Off, 1 = On, 128 = Correct position to 2048.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x28, [value & 0xFF])

    def torque_enable(self, reg: bool = False):
        """
        Enable servo torque (motor power on).

        Args:
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        return self.write_torque_switch(1, reg=reg)

    def torque_disable(self, reg: bool = False):
        """
        Disable servo torque (motor power off, free-spinning).

        Args:
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        return self.write_torque_switch(0, reg=reg)

    def correct_position_to_2048(self, reg: bool = False):
        """
        Correct the current position to 2048 (mid-point calibration).

        Args:
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        return self.write_torque_switch(128, reg=reg)

    def read_acceleration(self) -> Optional[int]:
        """
        Read the acceleration setting.

        Returns:
            int: Acceleration in 100 steps/s² units (0-254).
                 Example: 10 = 1000 steps/s².
            Returns None if read fails.
        """
        return self.servo._read_memory(0x29, 1)

    def write_acceleration(self, value: int, reg: bool = False):
        """
        Set the acceleration/deceleration.

        Args:
            value (int): Acceleration in 100 steps/s² units (0-254).
                        Example: 10 = 1000 steps/s².
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x29, [value & 0xFF])

    def read_target_location(self) -> Optional[int]:
        """
        Read the target position.

        Returns:
            int: Target position in steps (-32766 to +32766).
                 Each step = minimum resolvable angle.
            Returns None if read fails.
        """
        raw = self.servo._read_memory(0x2A, 2)
        if raw is None:
            return None
        if raw & 0x8000:
            return raw - 65536
        return raw

    def write_target_location(self, value: int, reg: bool = False):
        """
        Set the target position (absolute position control).

        Args:
            value (int): Target position in steps (-32766 to +32766).
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        if value < 0:
            raw = 65536 + value
        else:
            raw = value
        low = raw & 0xFF
        high = (raw >> 8) & 0xFF
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x2A, [low, high])

    def read_runtime(self) -> Optional[int]:
        """
        Read the runtime setting (PWM open-loop mode).

        Returns:
            int: Runtime value (0-1000). BIT10 indicates direction.
                 Used for PWM open-loop speed control.
            Returns None if read fails.
        """
        return self.servo._read_memory(0x2C, 2)

    def write_runtime(self, value: int, reg: bool = False):
        """
        Set the runtime (PWM open-loop mode).

        Args:
            value (int): Runtime value (0-1000). BIT10 indicates direction.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        low = value & 0xFF
        high = (value >> 8) & 0xFF
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x2C, [low, high])

    def read_running_speed(self) -> Optional[int]:
        """
        Read the running speed setpoint.

        Returns:
            int: Speed in steps/s (-32766 to +32766).
                 Sign indicates direction. 50 steps/s ≈ 0.732 RPM.
            Returns None if read fails.
        """
        raw = self.servo._read_memory(0x2E, 2)
        if raw is None:
            return None
        if raw & 0x8000:
            return raw - 65536
        return raw

    def write_running_speed(self, value: int, reg: bool = False):
        """
        Set the running speed (constant speed mode).

        Args:
            value (int): Speed in steps/s (-32766 to +32766).
                        Sign indicates direction. 50 steps/s ≈ 0.732 RPM.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        if value < 0:
            raw = 65536 + value
        else:
            raw = value
        low = raw & 0xFF
        high = (raw >> 8) & 0xFF
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x2E, [low, high])

    def read_torque_limit(self) -> Optional[int]:
        """
        Read the current torque limit.

        Returns:
            int: Torque limit in 0.1% units (0-1000). Example: 500 = 50.0%.
                 Initial value is set from max_torque (EEPROM) on power-up.
            Returns None if read fails.
        """
        return self.servo._read_memory(0x30, 2)

    def write_torque_limit(self, value: int, reg: bool = False):
        """
        Set the torque limit.

        Args:
            value (int): Torque limit in 0.1% units (0-1000). Example: 500 = 50.0%.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        low = value & 0xFF
        high = (value >> 8) & 0xFF
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x30, [low, high])

    def read_lock_symbol(self) -> Optional[int]:
        """
        Read the EEPROM write lock status.

        Returns:
            int: 0 = Write lock disabled (can save to EEPROM)
                 1 = Write lock enabled (cannot save to EEPROM)
            Returns None if read fails.
        """
        return self.servo._read_memory(0x37, 1)

    def write_lock_symbol(self, value: int, reg: bool = False):
        """
        Set the EEPROM write lock status.

        Args:
            value (int): 0 = Disable write lock, 1 = Enable write lock.
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        write_fn = self.servo._reg_write_memory if reg else self.servo._write_memory
        return write_fn(0x37, [value & 0xFF])

    def lock(self, reg: bool = False):
        """
        Enable EEPROM write lock (prevent saving settings to EEPROM).

        Args:
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        return self.write_lock_symbol(0, reg=reg)

    def unlock(self, reg: bool = False):
        """
        Disable EEPROM write lock (allow saving settings to EEPROM).

        Args:
            reg (bool): If True, use registered write mode.

        Returns:
            Response dict or None
        """
        return self.write_lock_symbol(1, reg=reg)

    def read_current_location(self) -> Optional[int]:
        """
        Read the current position (feedback).

        Returns:
            int: Current position in steps. Returns None if read fails.
        """
        return self.servo._read_memory(0x38, 2)

    def read_current_speed(self) -> Optional[int]:
        """
        Read the current speed (feedback).

        Returns:
            int: Current speed in steps/s. Returns None if read fails.
        """
        return self.servo._read_memory(0x3A, 2)

    def read_current_load(self) -> Optional[int]:
        """
        Read the current load (motor drive duty cycle).

        Returns:
            int: Load in 0.1% units (0-1000). Example: 500 = 50.0%.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x3C, 2)

    def read_current_voltage(self) -> Optional[int]:
        """
        Read the current operating voltage.

        Returns:
            int: Voltage in 0.1V units. Example: 120 = 12.0V.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x3E, 1)

    def read_current_temperature(self) -> Optional[int]:
        """
        Read the current internal temperature.

        Returns:
            int: Temperature in °C. Returns None if read fails.
        """
        return self.servo._read_memory(0x3F, 1)

    def read_async_write_flag(self) -> Optional[int]:
        """
        Read the asynchronous write flag.

        Returns:
            int: Flag value for asynchronous write instructions.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x40, 1)

    def read_servo_status(self) -> Optional[int]:
        """
        Read the servo status (error flags).

        Returns:
            int: Status bitmask. Bit = 1 indicates error, 0 = no error.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x41, 1)

    def read_mobile_sign(self) -> Optional[int]:
        """
        Read the movement flag.

        Returns:
            int: 1 = Servo in motion, 0 = Stationary.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x42, 1)

    def is_moving(self) -> bool:
        """
        Check if the servo is currently moving.

        Returns:
            bool: True if servo is moving, False if stationary or read fails.
        """
        value = self.read_mobile_sign()
        return value == 1 if value is not None else False

    def read_current_current(self) -> Optional[int]:
        """
        Read the current motor current draw.

        Returns:
            int: Current in 6.5mA units (0-500). Max = 500*6.5mA = 3250mA.
                 Returns None if read fails.
        """
        return self.servo._read_memory(0x45, 2)
