# Copyright 2016 Brigham Young University
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
import click

from awsparams import __VERSION__, AWSParams


def sanity_check(param: str, force: bool) -> bool:
    if force:
        return True
    sanity_check = input(f"Remove {param} y/n ")
    return sanity_check == 'y'


@click.group()
@click.version_option(version=__VERSION__)
def main():
    pass


@main.command('ls')
@click.argument('prefix', default='')
@click.option('--profile', type=click.STRING, help='profile to run with')
@click.option('-v', '--values', is_flag=True, help='display values')
@click.option('--decryption/--no-decryption', help='by default display decrypted values', default=True)
def ls(prefix='', profile=None, values=False, decryption=True):
    """
    List Paramters, optional matching a specific prefix
    """
    aws_params = AWSParams(profile)
    if not values:
        decryption = False
    for parm in aws_params.get_all_parameters(prefix=prefix, values=values, decryption=decryption, trim_name=False):
        if values:
            click.echo(f'{parm.Name}: {parm.Value}')
        else:
            click.echo(parm.Name)


@main.command('cp')
@click.argument('src')
@click.argument('dst', default='')
@click.option('--src_profile', type=click.STRING, default='', help="source profile")
@click.option('--dst_profile', type=click.STRING, default='', help="destination profile")
@click.option('--prefix', is_flag=True, help='copy set of parameters based on a prefix')
@click.option('--overwrite', is_flag=True, help='overwrite existing parameters')
@click.option('--key', type=click.STRING, default='', help="kms key to use for new copy")
def cp(src, dst, src_profile, dst_profile, prefix=False, overwrite=False, key=""):
    """
    Copy a parameter, optionally across accounts
    """
    aws_params = AWSParams(src_profile)
    # cross account copy without needing dst
    if dst_profile and src_profile != dst_profile and not dst:
        dst = src
    elif not dst:
        click.echo(
            "dst (Destination) is required when not copying to another profile"
        )
        return
    if prefix:
        params = aws_params.get_all_parameters(prefix=src, trim_name=False)
        for i in params:
            i = i._asdict()
            orignal_name = i["Name"]
            i["Name"] = i["Name"].replace(src, dst)
            if key:
                i['KeyId'] = key
            aws_params.put_parameter(i, overwrite=overwrite, profile=dst_profile)
            click.echo(f'Copied {orignal_name} to {i["Name"]}')
        return True
    else:
        if isinstance(src, str):
            src_param = aws_params.get_parameter(src)
            if not src_param:
                click.echo(f"Parameter: {src} not found")
                return
            src_param = src_param._asdict()
            src_param["Name"] = dst
            if key:
                src_param['KeyId'] = key
            aws_params.put_parameter(src_param, overwrite=overwrite, profile=dst_profile)
            click.echo(f"Copied {src} to {dst}")
            return True


@main.command('mv')
@click.argument('src')
@click.argument('dst')
@click.option('--prefix', is_flag=True, help="move/rename based on prefix")
@click.option('--profile', type=click.STRING, help="alternative profile to use")
@click.pass_context
def mv(ctx, src, dst, prefix=False, profile=None):
    """
    Move or rename a parameter
    """
    if prefix:
        if ctx.invoke(cp, src=src, dst=dst, src_profile=profile, prefix=prefix):
            ctx.invoke(rm, src=src, force=True, prefix=True, profile=profile)
    else:
        if ctx.invoke(cp, src=src, dst=dst, src_profile=profile):
            ctx.invoke(rm, src=src, force=True, profile=profile)


@main.command('rm')
@click.argument('src')
@click.option('-f', '--force', is_flag=True, help='force without confirmation')
@click.option('--prefix', is_flag=True, help='remove/delete based on prefix/path')
@click.option('--profile', type=click.STRING, help='alternative profile to use')
def rm(src, force=False, prefix=False, profile=None):
    """
    Remove/Delete a parameter
    """
    aws_params = AWSParams(profile)
    if prefix:
        params = aws_params.get_all_parameters(prefix=src, trim_name=False)
        if len(params) == 0:
            click.echo(f"No parameters with the {src} prefix found")
        else:
            for param in params:
                if sanity_check(param.Name, force):
                    aws_params.remove_parameter(param.Name)
                    click.echo(
                        f"The {param.Name} parameter has been removed"
                    )
    else:
        param = aws_params.get_parameter(name=src)
        if param and param.Name == src:
            if sanity_check(src, force):
                aws_params.remove_parameter(src)
                click.echo(f"The {src} parameter has been removed")
        else:
            click.echo(f"Parameter {src} not found")


@main.command('new')
@click.option('--name', type=click.STRING, prompt="Parameter Name", help='parameter name')
@click.option('--value', type=click.STRING, help='parameter value')
@click.option('--param_type', type=click.STRING, default='String', help='parameter type one of String(default), StringList, SecureString')
@click.option('--key', type=click.STRING, default='', help='KMS Key used to encrypt the parameter')
@click.option('--description', type=click.STRING, default='', help='parameter description text')
@click.option('--profile', type=click.STRING, help='alternative profile to be used')
@click.option('--overwrite', is_flag=True, help='overwrite exisiting parameters')
def new(name=None, value=None, param_type='String', key='', description='', profile=None, overwrite=False):
    """
    Create a new parameter
    """
    AWSParams(profile).new_param(
        name, value, param_type=param_type, key=key, description=description, overwrite=overwrite)


@main.command('set')
@click.argument('src')
@click.argument('value')
@click.option('--profile', type=click.STRING, default='', help="source profile")
def set(src=None, value=None, profile=None):
    """
    Edit an existing parameter
    """
    result = AWSParams(profile).set_param(src, value)
    if result:
        click.echo(f"updated param '{src}' with value")
    else:
        click.echo(f"not updated, param '{src}' already contains that value")


if __name__ == '__main__':
    main()
