"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnJson = void 0;
const custom_resource_1 = require("./custom-resource");
const cfn_utils_provider_1 = require("./private/cfn-utils-provider");
const stack_1 = require("./stack");
const stack_trace_1 = require("./stack-trace");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("./construct-compat");
/**
 * Captures a synthesis-time JSON object a CloudFormation reference which resolves during deployment to the resolved values of the JSON object.
 *
 * The main use case for this is to overcome a limitation in CloudFormation that
 * does not allow using intrinsic functions as dictionary keys (because
 * dictionary keys in JSON must be strings). Specifically this is common in IAM
 * conditions such as `StringEquals: { lhs: "rhs" }` where you want "lhs" to be
 * a reference.
 *
 * This object is resolvable, so it can be used as a value.
 *
 * This construct is backed by a custom resource.
 */
class CfnJson extends construct_compat_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The creation stack of this resolvable which will be appended to errors thrown during resolution.
         *
         * If this returns an empty array the stack will not be attached.
         */
        this.creationStack = [];
        this.creationStack = stack_trace_1.captureStackTrace();
        // stringify the JSON object in a token-aware way.
        this.jsonString = stack_1.Stack.of(this).toJsonString(props.value);
        const resource = new custom_resource_1.CustomResource(this, 'Resource', {
            serviceToken: cfn_utils_provider_1.CfnUtilsProvider.getOrCreate(this),
            resourceType: "Custom::AWSCDKCfnJson" /* CFN_JSON */,
            properties: {
                Value: this.jsonString,
            },
        });
        this.value = resource.getAtt('Value');
    }
    /**
     * This is required in case someone JSON.stringifys an object which refrences this object. Otherwise, we'll get a cyclic JSON reference.
     */
    toJSON() {
        return this.jsonString;
    }
    /**
     * Produce the Token's value at resolution time.
     */
    resolve(_) {
        return this.value;
    }
}
exports.CfnJson = CfnJson;
//# sourceMappingURL=data:application/json;base64,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