
from __future__ import annotations

import logging
import os
import textwrap
import typing as t
import subprocess as sp
from pathlib import Path

from novella.markdown.preprocessor import MarkdownFiles, MarkdownPreprocessor
from novella.novella import Novella

if t.TYPE_CHECKING:
  from novella.markdown.tagparser import Tag

logger = logging.getLogger(__name__)


class ShellTagProcessor(MarkdownPreprocessor):
  """ Provides a `@shell <command>` tag that runs a shell command from the project directory and inserts its output
  into the file. This is useful if parts of your documentation need to be dynamically generated by another program.

  The environment variable `BUILD_DIR` is set to point to the temporary build directory. The current working directory
  is the project directory in which the Novella build script lies.

  __Example__

      @shell cd .. && slam changelog format --all --markdown

  !!! note The example shows how to embed a changelog generated and formatted by [Slam][].

  [Slam]: https://pypi.org/project/slam-cli/
  """

  def process_files(self, files: MarkdownFiles) -> None:
    from novella.markdown.tagparser import parse_block_tags, replace_tags
    for file in files:
      tags = parse_block_tags(file.content)
      file.content = replace_tags(file.content, tags, lambda t: self._replace_tag(files.novella, file.path, t))

  def _replace_tag(self, novella: Novella, file_path: Path, tag: Tag) -> str | None:
    if tag.name != 'shell':
      return None

    command = tag.args.strip()
    env = os.environ.copy()
    env['BUILD_DIR'] = str(novella.build.directory)

    try:
      output = sp.check_output(command, shell=True, cwd=novella.project_directory, env=env, stderr=sp.PIPE).decode()
    except sp.CalledProcessError as exc:
      logger.exception('@shell command <fg=cyan>%s</fg> exited with return code <fg=red>%s</fg>', command, exc.returncode)
      output = textwrap.indent((exc.stdout or b'').decode() + '' + (exc.stderr or b'').decode(), '    ')
      output = f'    $ {command}  # exited with return code {exc.returncode}\n{output}'

    return output
