"use strict";
/*
 * Copyright 2018-2021 Elyra Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PipelineEditorFactory = exports.PipelineEditor = exports.PipelineEditorWidget = exports.commandIDs = void 0;
const canvas_1 = require("@elyra/canvas");
const ui_components_1 = require("@elyra/ui-components");
const ui_components_2 = require("@elyra/ui-components");
const apputils_1 = require("@jupyterlab/apputils");
const coreutils_1 = require("@jupyterlab/coreutils");
const docregistry_1 = require("@jupyterlab/docregistry");
const algorithm_1 = require("@lumino/algorithm");
const signaling_1 = require("@lumino/signaling");
const core_1 = require("@material-ui/core");
const Close_1 = __importDefault(require("@material-ui/icons/Close"));
const Alert_1 = __importDefault(require("@material-ui/lab/Alert"));
require("carbon-components/css/carbon-components.min.css");
require("@elyra/canvas/dist/styles/common-canvas.min.css");
require("../style/canvas.css");
const React = __importStar(require("react"));
const react_intl_1 = require("react-intl");
const canvas_2 = require("./canvas");
const constants_1 = require("./constants");
const i18nData = __importStar(require("./en.json"));
const formDialogWidget_1 = require("./formDialogWidget");
const palette = __importStar(require("./palette.json"));
const PipelineExportDialog_1 = require("./PipelineExportDialog");
const PipelineService_1 = require("./PipelineService");
const PipelineSubmissionDialog_1 = require("./PipelineSubmissionDialog");
const properties = __importStar(require("./properties.json"));
const StringArrayInput_1 = require("./StringArrayInput");
const utils_1 = __importDefault(require("./utils"));
const validation_1 = require("./validation");
const PIPELINE_CLASS = 'elyra-PipelineEditor';
const NODE_TOOLTIP_CLASS = 'elyra-PipelineNodeTooltip';
const TIP_TYPE_NODE = 'tipTypeNode';
const NodeProperties = (properties) => {
    return (React.createElement("dl", { className: NODE_TOOLTIP_CLASS }, Object.keys(properties).map((key, idx) => {
        let value = properties[key];
        if (Array.isArray(value)) {
            value = value.join('\n');
        }
        else if (typeof value === 'boolean') {
            value = value ? 'Yes' : 'No';
        }
        let tooltipTextClass = '';
        if (key == 'Error') {
            tooltipTextClass = 'elyra-tooltipError';
        }
        return (React.createElement(React.Fragment, { key: idx },
            React.createElement("dd", { className: tooltipTextClass }, key),
            React.createElement("dt", { className: tooltipTextClass }, value)));
    })));
};
exports.commandIDs = {
    openPipelineEditor: 'pipeline-editor:open',
    openMetadata: 'elyra-metadata:open',
    openDocManager: 'docmanager:open',
    newDocManager: 'docmanager:new-untitled',
    submitScript: 'python-editor:submit',
    submitNotebook: 'notebook:submit',
    addFileToPipeline: 'pipeline-editor:add-node'
};
/**
 * Wrapper Class for Common Canvas React Component
 */
class PipelineEditorWidget extends apputils_1.ReactWidget {
    constructor(props) {
        super(props);
        this.shell = props.shell;
        this.commands = props.commands;
        this.browserFactory = props.browserFactory;
        this.context = props.context;
        this.serviceManager = props.serviceManager;
        this.addFileToPipelineSignal = props.addFileToPipelineSignal;
    }
    render() {
        return (React.createElement(PipelineEditor, { shell: this.shell, commands: this.commands, browserFactory: this.browserFactory, widgetContext: this.context, widgetId: this.parent.id, serviceManager: this.serviceManager, addFileToPipelineSignal: this.addFileToPipelineSignal }));
    }
}
exports.PipelineEditorWidget = PipelineEditorWidget;
/**
 * Class for Common Canvas React Component
 */
class PipelineEditor extends React.Component {
    constructor(props) {
        super(props);
        this.position = 10;
        this.shell = props.shell;
        this.commands = props.commands;
        this.browserFactory = props.browserFactory;
        this.serviceManager = props.serviceManager;
        this.canvasController = new canvas_1.CanvasController();
        this.canvasController.setPipelineFlowPalette(palette);
        this.widgetContext = props.widgetContext;
        this.canvasManager = new canvas_2.CanvasManager(this.widgetContext, this.canvasController);
        this.widgetId = props.widgetId;
        this.addFileToPipelineSignal = props.addFileToPipelineSignal;
        this.contextMenuHandler = this.contextMenuHandler.bind(this);
        this.clickActionHandler = this.clickActionHandler.bind(this);
        this.editActionHandler = this.editActionHandler.bind(this);
        this.beforeEditActionHandler = this.beforeEditActionHandler.bind(this);
        this.tipHandler = this.tipHandler.bind(this);
        this.state = {
            showPropertiesDialog: false,
            propertiesInfo: {},
            showValidationError: false,
            validationError: {
                errorMessage: '',
                errorSeverity: 'error'
            },
            emptyPipeline: utils_1.default.isEmptyPipeline(this.canvasController.getPipelineFlow())
        };
        this.applyPropertyChanges = this.applyPropertyChanges.bind(this);
        this.closePropertiesDialog = this.closePropertiesDialog.bind(this);
        this.openPropertiesDialog = this.openPropertiesDialog.bind(this);
        this.propertiesActionHandler = this.propertiesActionHandler.bind(this);
        this.propertiesControllerHandler = this.propertiesControllerHandler.bind(this);
        this.addFileToPipelineSignal.connect((args) => {
            this.handleAddFileToPipelineCanvas();
        });
    }
    render() {
        const validationAlert = (React.createElement(core_1.Collapse, { in: this.state.showValidationError },
            React.createElement(Alert_1.default, { severity: this.state.validationError.errorSeverity, action: React.createElement(core_1.IconButton, { "aria-label": "close", color: "inherit", size: "small", onClick: () => {
                        this.setState({ showValidationError: false });
                    } },
                    React.createElement(Close_1.default, { fontSize: "inherit" })) }, this.state.validationError.errorMessage)));
        const emptyCanvasContent = (React.createElement("div", null,
            React.createElement(ui_components_1.dragDropIcon.react, { tag: "div", elementPosition: "center", height: "120px" }),
            React.createElement("h1", null,
                ' ',
                "Start your new pipeline by dragging files from the file browser pane.",
                ' ')));
        const canvasConfig = {
            enableInternalObjectModel: true,
            emptyCanvasContent: emptyCanvasContent,
            enablePaletteLayout: 'None',
            paletteInitialState: false,
            enableInsertNodeDroppedOnLink: true,
            enableNodeFormatType: 'Horizontal'
        };
        const contextMenuConfig = {
            enableCreateSupernodeNonContiguous: true,
            defaultMenuEntries: {
                saveToPalette: false,
                createSupernode: true
            }
        };
        const pipelineDefinition = this.canvasController.getPipelineFlow();
        const emptyCanvas = utils_1.default.isEmptyCanvas(pipelineDefinition);
        const toolbarConfig = [
            {
                action: 'run',
                label: 'Run Pipeline',
                enable: !this.state.emptyPipeline
            },
            {
                action: 'save',
                label: 'Save Pipeline',
                enable: true,
                iconEnabled: ui_components_1.IconUtil.encode(ui_components_1.savePipelineIcon),
                iconDisabled: ui_components_1.IconUtil.encode(ui_components_1.savePipelineIcon)
            },
            {
                action: 'export',
                label: 'Export Pipeline',
                enable: !this.state.emptyPipeline,
                iconEnabled: ui_components_1.IconUtil.encode(ui_components_1.exportPipelineIcon),
                iconDisabled: ui_components_1.IconUtil.encode(ui_components_1.exportPipelineIcon)
            },
            {
                action: 'clear',
                label: 'Clear Pipeline',
                enable: !this.state.emptyPipeline || !emptyCanvas,
                iconEnabled: ui_components_1.IconUtil.encode(ui_components_1.clearPipelineIcon),
                iconDisabled: ui_components_1.IconUtil.encode(ui_components_1.clearPipelineIcon)
            },
            {
                action: 'openRuntimes',
                label: 'Open Runtimes',
                enable: true,
                iconEnabled: ui_components_1.IconUtil.encode(ui_components_1.runtimesIcon),
                iconDisabled: ui_components_1.IconUtil.encode(ui_components_1.runtimesIcon)
            },
            { divider: true },
            { action: 'undo', label: 'Undo' },
            { action: 'redo', label: 'Redo' },
            { action: 'cut', label: 'Cut' },
            { action: 'copy', label: 'Copy' },
            { action: 'paste', label: 'Paste' },
            { action: 'createAutoComment', label: 'Add Comment', enable: true },
            { action: 'deleteSelectedObjects', label: 'Delete' },
            {
                action: 'arrangeHorizontally',
                label: 'Arrange Horizontally',
                enable: !this.state.emptyPipeline
            },
            {
                action: 'arrangeVertically',
                label: 'Arrange Vertically',
                enable: !this.state.emptyPipeline
            }
        ];
        const propertiesCallbacks = {
            actionHandler: this.propertiesActionHandler,
            controllerHandler: this.propertiesControllerHandler,
            applyPropertyChanges: this.applyPropertyChanges,
            closePropertiesDialog: this.closePropertiesDialog
        };
        const commProps = (React.createElement(react_intl_1.IntlProvider, { key: "IntlProvider2", locale: 'en', messages: i18nData.messages },
            React.createElement(canvas_1.CommonProperties, { ref: (instance) => {
                    this.CommonProperties = instance;
                }, propertiesInfo: this.state.propertiesInfo, propertiesConfig: {
                    containerType: 'Custom',
                    rightFlyout: true,
                    applyOnBlur: true
                }, callbacks: propertiesCallbacks, customControls: [StringArrayInput_1.StringArrayInput] })));
        return (React.createElement(ui_components_2.Dropzone, { onDrop: (e) => {
                this.handleAddFileToPipelineCanvas(e.offsetX, e.offsetY);
            } },
            validationAlert,
            React.createElement(react_intl_1.IntlProvider, { key: "IntlProvider1", locale: 'en', messages: i18nData.messages },
                React.createElement(canvas_1.CommonCanvas, { canvasController: this.canvasController, contextMenuHandler: this.contextMenuHandler, clickActionHandler: this.clickActionHandler, editActionHandler: this.editActionHandler, beforeEditActionHandler: this.beforeEditActionHandler, tipHandler: this.tipHandler, toolbarConfig: toolbarConfig, config: canvasConfig, notificationConfig: { enable: false }, contextMenuConfig: contextMenuConfig, rightFlyoutContent: commProps, showRightFlyout: this.state.showPropertiesDialog }))));
    }
    updateModel() {
        const pipelineFlow = this.canvasController.getPipelineFlow();
        this.widgetContext.model.fromString(JSON.stringify(pipelineFlow, null, 2));
        this.setState({ emptyPipeline: utils_1.default.isEmptyPipeline(pipelineFlow) });
    }
    initPropertiesInfo() {
        return __awaiter(this, void 0, void 0, function* () {
            const runtimeImages = yield PipelineService_1.PipelineService.getRuntimeImages().catch(error => ui_components_1.RequestErrors.serverError(error));
            const imageEnum = [];
            for (const runtimeImage in runtimeImages) {
                imageEnum.push(runtimeImage);
                properties.resources['runtime_image.' + runtimeImage + '.label'] = runtimeImages[runtimeImage];
            }
            properties.parameters[1].enum = imageEnum;
            this.propertiesInfo = {
                parameterDef: properties,
                appData: { id: '' },
                labelEditable: true
            };
        });
    }
    openPropertiesDialog(source) {
        console.log('Opening properties dialog');
        if (!source.targetObject) {
            source.targetObject = this.canvasController.getNode(source.id);
        }
        const node_id = source.targetObject.id;
        const app_data = source.targetObject.app_data;
        const node_props = JSON.parse(JSON.stringify(this.propertiesInfo));
        node_props.appData.id = node_id;
        node_props.parameterDef.current_parameters.filename = app_data.filename;
        node_props.parameterDef.current_parameters.runtime_image =
            app_data.runtime_image;
        node_props.parameterDef.current_parameters.outputs = app_data.outputs;
        node_props.parameterDef.current_parameters.env_vars = app_data.env_vars;
        node_props.parameterDef.current_parameters.dependencies =
            app_data.dependencies;
        node_props.parameterDef.current_parameters.include_subdirectories =
            app_data.include_subdirectories;
        node_props.parameterDef.current_parameters.cpu = app_data.cpu;
        node_props.parameterDef.current_parameters.memory = app_data.memory;
        node_props.parameterDef.current_parameters.gpu = app_data.gpu;
        node_props.parameterDef.titleDefinition = {
            title: this.canvasController.getNode(source.id).label,
            editable: true
        };
        this.setState({
            showValidationError: false,
            showPropertiesDialog: true,
            propertiesInfo: node_props
        });
    }
    applyPropertyChanges(propertySet, appData, additionalData) {
        console.log('Applying changes to properties');
        const pipelineId = this.canvasController.getPrimaryPipelineId();
        let node = this.canvasController.getNode(appData.id, pipelineId);
        // If the node is in a supernode, search supernodes for it
        if (!node) {
            const superNodes = this.canvasController.getSupernodes(pipelineId);
            for (const superNode of superNodes) {
                node = this.canvasController.getNode(appData.id, superNode.subflow_ref.pipeline_id_ref);
                if (node) {
                    break;
                }
            }
        }
        const app_data = node.app_data;
        if (additionalData.title) {
            this.canvasController.setNodeLabel(appData.id, additionalData.title);
        }
        if (app_data.filename !== propertySet.filename) {
            app_data.filename = propertySet.filename;
            this.canvasController.setNodeLabel(appData.id, coreutils_1.PathExt.basename(propertySet.filename));
        }
        app_data.runtime_image = propertySet.runtime_image;
        app_data.outputs = propertySet.outputs;
        app_data.env_vars = propertySet.env_vars;
        app_data.dependencies = propertySet.dependencies;
        app_data.include_subdirectories = propertySet.include_subdirectories;
        app_data.cpu = propertySet.cpu;
        app_data.memory = propertySet.memory;
        app_data.gpu = propertySet.gpu;
        this.canvasController.setNodeProperties(appData.id, { app_data }, pipelineId);
        this.validateAllNodes();
        this.updateModel();
    }
    closePropertiesDialog() {
        console.log('Closing properties dialog');
        const propsInfo = JSON.parse(JSON.stringify(this.propertiesInfo));
        if (this.CommonProperties) {
            this.CommonProperties.applyPropertiesEditing(false);
        }
        this.setState({ showPropertiesDialog: false, propertiesInfo: propsInfo });
    }
    propertiesControllerHandler(propertiesController) {
        this.propertiesController = propertiesController;
    }
    propertiesActionHandler(id, appData, data) {
        const propertyId = { name: data.parameter_ref };
        const filename = PipelineService_1.PipelineService.getWorkspaceRelativeNodePath(this.widgetContext.path, this.propertiesController.getPropertyValue('filename'));
        if (this.CommonProperties) {
            this.CommonProperties.applyPropertiesEditing(false);
        }
        if (id === 'browse_file') {
            const currentExt = coreutils_1.PathExt.extname(filename);
            ui_components_1.showBrowseFileDialog(this.browserFactory.defaultBrowser.model.manager, {
                startPath: coreutils_1.PathExt.dirname(filename),
                filter: (model) => {
                    const ext = coreutils_1.PathExt.extname(model.path);
                    return currentExt === ext;
                }
            }).then((result) => {
                if (result.button.accept && result.value.length) {
                    this.propertiesController.updatePropertyValue(propertyId, PipelineService_1.PipelineService.getPipelineRelativeNodePath(this.widgetContext.path, result.value[0].path));
                }
            });
        }
        else if (id === 'add_dependencies') {
            ui_components_1.showBrowseFileDialog(this.browserFactory.defaultBrowser.model.manager, {
                multiselect: true,
                includeDir: true,
                rootPath: coreutils_1.PathExt.dirname(filename),
                filter: (model) => {
                    // do not include the notebook itself
                    return model.path !== filename;
                }
            }).then((result) => {
                if (result.button.accept && result.value.length) {
                    const dependencies = Array.from(this.propertiesController.getPropertyValue(propertyId));
                    // If multiple files are selected, replace the given index in the dependencies list
                    // and insert the rest of the values after that index.
                    result.value.forEach((val, index) => {
                        if (index === 0) {
                            dependencies[data.index] = val.path;
                        }
                        else {
                            dependencies.splice(data.index, 0, val.path);
                        }
                    });
                    this.propertiesController.updatePropertyValue(propertyId, dependencies);
                }
            });
        }
    }
    /*
     * Add options to the node context menu
     * Pipeline specific context menu items are:
     *  - Enable opening selected notebook(s)
     *  - Enable node properties for single node
     */
    contextMenuHandler(source, defaultMenu) {
        let customMenu = defaultMenu;
        if (source.type === 'node') {
            if (source.selectedObjectIds.length > 1) {
                // multiple nodes selected
                customMenu = customMenu.concat({
                    action: 'openFile',
                    label: 'Open Files'
                });
            }
            else if (source.targetObject.type == 'execution_node') {
                // single node selected
                customMenu = customMenu.concat({
                    action: 'openFile',
                    label: 'Open File'
                }, {
                    action: 'properties',
                    label: 'Properties'
                });
            }
        }
        return customMenu;
    }
    /*
     * Handles mouse click actions
     */
    clickActionHandler(source) {
        return __awaiter(this, void 0, void 0, function* () {
            // opens the Jupyter Notebook associated with a given node
            if (source.clickType === 'DOUBLE_CLICK' && source.objectType === 'node') {
                this.handleOpenFile(source.selectedObjectIds);
            }
            else if (source.clickType === 'SINGLE_CLICK' &&
                source.objectType === 'node' &&
                this.state.showPropertiesDialog) {
                this.closePropertiesDialog();
                this.openPropertiesDialog(source);
            }
        });
    }
    beforeEditActionHandler(data) {
        if (data.editType !== 'linkNodes') {
            return data;
        }
        // Checks validity of links before adding
        const proposedLink = {
            id: 'proposed-link',
            trgNodeId: data.targetNodes[0].id,
            srcNodeId: data.nodes[0].id,
            type: 'nodeLink'
        };
        const links = this.canvasController.getLinks();
        const taintedLinks = validation_1.checkCircularReferences([proposedLink, ...links]);
        if (taintedLinks.length > 0) {
            this.setState({
                validationError: {
                    errorMessage: 'Invalid operation: circular references in pipeline.',
                    errorSeverity: 'error'
                },
                showValidationError: true
            });
            // Don't proceed with adding the link if invalid.
            return null;
        }
        return data;
    }
    /*
     * Handles creating new nodes in the canvas
     */
    editActionHandler(data) {
        this.setState({
            showValidationError: false
        });
        if (data && data.editType) {
            console.log(`Handling action: ${data.editType}`);
            switch (data.editType) {
                case 'run':
                    this.handleRunPipeline();
                    break;
                case 'export':
                    this.handleExportPipeline();
                    break;
                case 'save':
                    this.handleSavePipeline();
                    break;
                case 'clear':
                    this.handleClearPipeline(data);
                    break;
                case 'openRuntimes':
                    this.handleOpenRuntimes();
                    break;
                case 'openFile':
                    if (data.type === 'node') {
                        this.handleOpenFile(data.selectedObjectIds);
                    }
                    break;
                case 'properties':
                    if (data.type === 'node') {
                        if (this.state.showPropertiesDialog) {
                            this.closePropertiesDialog();
                        }
                        this.openPropertiesDialog(data);
                    }
                    break;
            }
        }
        this.validateAllLinks();
        this.updateModel();
    }
    /*
     * Handles displaying node properties
     */
    tipHandler(tipType, data) {
        if (tipType === TIP_TYPE_NODE) {
            const appData = data.node.app_data;
            const propsInfo = this.propertiesInfo.parameterDef.uihints.parameter_info;
            const tooltipProps = {};
            if (appData != null && appData.invalidNodeError != null) {
                tooltipProps['Error'] = appData.invalidNodeError;
            }
            if (data.node.type == 'execution_node') {
                propsInfo.forEach((info) => {
                    tooltipProps[info.label.default] =
                        appData[info.parameter_ref] || '';
                });
            }
            return React.createElement(NodeProperties, Object.assign({}, tooltipProps));
        }
    }
    handleAddFileToPipelineCanvas(x, y) {
        // Only add file to pipeline if it is currently in focus
        if (this.shell.currentWidget.id !== this.widgetId) {
            return;
        }
        let failedAdd = 0;
        let position = 0;
        const missingXY = !(x && y);
        // if either x or y is undefined use the default coordinates
        if (missingXY) {
            position = this.position;
            x = 75;
            y = 85;
        }
        const fileBrowser = this.browserFactory.defaultBrowser;
        algorithm_1.toArray(fileBrowser.selectedItems()).map(item => {
            if (this.canvasManager.isSupportedNode(item)) {
                // read the file contents
                // create a notebook widget to get a string with the node content then dispose of it
                let itemContent;
                if (item.type == 'notebook') {
                    const fileWidget = fileBrowser.model.manager.open(item.path);
                    itemContent = fileWidget.content.model.toString();
                    fileWidget.dispose();
                }
                const success = this.canvasManager.addNode(item, itemContent, x + position, y + position);
                if (success) {
                    position += 20;
                    this.setState({ showValidationError: false });
                }
                else {
                    // handle error
                }
            }
            else {
                failedAdd++;
            }
        });
        // update position if the default coordinates were used
        if (missingXY) {
            this.position = position;
        }
        if (failedAdd) {
            return apputils_1.showDialog({
                title: 'Unsupported File(s)',
                body: 'Currently, only selected notebook and python script files can be added to a pipeline',
                buttons: [apputils_1.Dialog.okButton()]
            });
        }
    }
    /*
     * Open node associated notebook
     */
    handleOpenFile(selectedNodes) {
        for (let i = 0; i < selectedNodes.length; i++) {
            const path = PipelineService_1.PipelineService.getWorkspaceRelativeNodePath(this.widgetContext.path, this.canvasController.getNode(selectedNodes[i]).app_data.filename);
            this.commands.execute(exports.commandIDs.openDocManager, { path });
        }
    }
    cleanNullProperties() {
        // Delete optional fields that have null value
        for (const node of this.canvasController.getPipelineFlow().pipelines[0]
            .nodes) {
            if (node.app_data.cpu === null) {
                delete node.app_data.cpu;
            }
            if (node.app_data.memory === null) {
                delete node.app_data.memory;
            }
            if (node.app_data.gpu === null) {
                delete node.app_data.gpu;
            }
        }
    }
    handleExportPipeline() {
        return __awaiter(this, void 0, void 0, function* () {
            // Warn user if the pipeline has invalid nodes
            const errorMessage = yield this.validatePipeline();
            if (errorMessage) {
                this.setState({
                    showValidationError: true,
                    validationError: {
                        errorMessage: errorMessage,
                        errorSeverity: 'error'
                    }
                });
                return;
            }
            if (this.widgetContext.model.dirty) {
                const dialogResult = yield apputils_1.showDialog({
                    title: 'This pipeline contains unsaved changes. To submit the pipeline the changes need to be saved.',
                    buttons: [
                        apputils_1.Dialog.cancelButton(),
                        apputils_1.Dialog.okButton({ label: 'Save and Submit' })
                    ]
                });
                if (dialogResult.button && dialogResult.button.accept === true) {
                    yield this.widgetContext.save();
                }
                else {
                    // Don't proceed if cancel button pressed
                    return;
                }
            }
            const action = 'export pipeline';
            const runtimes = yield PipelineService_1.PipelineService.getRuntimes(true, action).catch(error => ui_components_1.RequestErrors.serverError(error));
            if (utils_1.default.isDialogResult(runtimes)) {
                // Open the runtimes widget
                runtimes.button.label.includes(PipelineService_1.RUNTIMES_NAMESPACE) &&
                    this.handleOpenRuntimes();
                return;
            }
            const schema = yield PipelineService_1.PipelineService.getRuntimesSchema().catch(error => ui_components_1.RequestErrors.serverError(error));
            const dialogOptions = {
                title: 'Export pipeline',
                body: formDialogWidget_1.formDialogWidget(React.createElement(PipelineExportDialog_1.PipelineExportDialog, { runtimes: runtimes, schema: schema })),
                buttons: [apputils_1.Dialog.cancelButton(), apputils_1.Dialog.okButton()],
                defaultButton: 1,
                focusNodeSelector: '#runtime_config'
            };
            const dialogResult = yield ui_components_1.showFormDialog(dialogOptions);
            if (dialogResult.value == null) {
                // When Cancel is clicked on the dialog, just return
                return;
            }
            // prepare pipeline submission details
            const pipelineFlow = this.canvasController.getPipelineFlow();
            const pipeline_path = this.widgetContext.path;
            const pipeline_dir = coreutils_1.PathExt.dirname(pipeline_path);
            const pipeline_name = coreutils_1.PathExt.basename(pipeline_path, coreutils_1.PathExt.extname(pipeline_path));
            const pipeline_export_format = dialogResult.value.pipeline_filetype;
            let pipeline_export_path = pipeline_name + '.' + pipeline_export_format;
            // only prefix the '/' when pipeline_dir is non-empty
            if (pipeline_dir) {
                pipeline_export_path = pipeline_dir + '/' + pipeline_export_path;
            }
            const overwrite = dialogResult.value.overwrite;
            const runtime_config = dialogResult.value.runtime_config;
            const runtime = PipelineService_1.PipelineService.getRuntimeName(runtime_config, runtimes);
            PipelineService_1.PipelineService.setNodePathsRelativeToWorkspace(pipelineFlow.pipelines[0], this.widgetContext.path);
            this.cleanNullProperties();
            pipelineFlow.pipelines[0]['app_data']['name'] = pipeline_name;
            pipelineFlow.pipelines[0]['app_data']['runtime'] = runtime;
            pipelineFlow.pipelines[0]['app_data']['runtime-config'] = runtime_config;
            pipelineFlow.pipelines[0]['app_data']['source'] = coreutils_1.PathExt.basename(this.widgetContext.path);
            PipelineService_1.PipelineService.exportPipeline(pipelineFlow, pipeline_export_format, pipeline_export_path, overwrite).catch(error => ui_components_1.RequestErrors.serverError(error));
        });
    }
    handleOpenPipeline() {
        return __awaiter(this, void 0, void 0, function* () {
            this.widgetContext.ready.then(() => __awaiter(this, void 0, void 0, function* () {
                let pipelineJson = null;
                try {
                    pipelineJson = this.widgetContext.model.toJSON();
                }
                catch (error) {
                    this.handleJSONError(error);
                }
                if (pipelineJson === null) {
                    // creating new pipeline
                    pipelineJson = this.canvasController.getPipelineFlow();
                    if (utils_1.default.isEmptyPipeline(pipelineJson)) {
                        pipelineJson.pipelines[0]['app_data']['version'] = constants_1.PIPELINE_CURRENT_VERSION;
                        this.canvasController.setPipelineFlow(pipelineJson);
                    }
                }
                else {
                    // opening an existing pipeline
                    const pipelineVersion = +utils_1.default.getPipelineVersion(pipelineJson);
                    if (pipelineVersion !== constants_1.PIPELINE_CURRENT_VERSION) {
                        // pipeline version and current version are divergent
                        if (pipelineVersion > constants_1.PIPELINE_CURRENT_VERSION) {
                            // in this case, pipeline was last edited in a "more recent release" and
                            // the user should update his version of Elyra to consume the pipeline
                            apputils_1.showDialog({
                                title: 'Load pipeline failed!',
                                body: (React.createElement("p", null, "This pipeline corresponds to a more recent version of Elyra and cannot be used until Elyra has been upgraded.")),
                                buttons: [apputils_1.Dialog.okButton()]
                            }).then(() => {
                                this.handleClosePipeline();
                            });
                        }
                        else {
                            // in this case, pipeline was last edited in a "old" version of Elyra and
                            // it needs to be updated/migrated.
                            apputils_1.showDialog({
                                title: 'Migrate pipeline?',
                                body: (React.createElement("p", null,
                                    "This pipeline corresponds to an older version of Elyra and needs to be migrated.",
                                    React.createElement("br", null),
                                    "Although the pipeline can be further edited and/or submitted after its update,",
                                    React.createElement("br", null),
                                    "the migration will not be completed until the pipeline has been saved within the editor.",
                                    React.createElement("br", null),
                                    React.createElement("br", null),
                                    "Proceed with migration?")),
                                buttons: [apputils_1.Dialog.cancelButton(), apputils_1.Dialog.okButton()]
                            }).then(result => {
                                if (result.button.accept) {
                                    // proceed with migration
                                    pipelineJson = PipelineService_1.PipelineService.convertPipeline(pipelineJson, this.widgetContext.path);
                                    this.setAndVerifyPipelineFlow(pipelineJson);
                                }
                                else {
                                    this.handleClosePipeline();
                                }
                            });
                        }
                    }
                    else {
                        yield this.setAndVerifyPipelineFlow(pipelineJson);
                    }
                }
            }));
        });
    }
    setAndVerifyPipelineFlow(pipelineJson) {
        return __awaiter(this, void 0, void 0, function* () {
            this.canvasController.setPipelineFlow(pipelineJson);
            const errorMessage = yield this.validatePipeline();
            if (errorMessage) {
                this.setState({
                    emptyPipeline: utils_1.default.isEmptyPipeline(pipelineJson),
                    showValidationError: true,
                    validationError: {
                        errorMessage: errorMessage,
                        errorSeverity: 'error'
                    }
                });
            }
            else {
                this.setState({
                    emptyPipeline: utils_1.default.isEmptyPipeline(pipelineJson),
                    showValidationError: false
                });
            }
        });
    }
    /**
     * Adds an error decoration if a node has any invalid properties.
     *
     * @param node - canvas node object to validate
     *
     * @returns true if the node is valid.
     */
    validateNode(node, pipelineId) {
        return __awaiter(this, void 0, void 0, function* () {
            let validNode = true;
            let indicatorXPos;
            let indicatorYPos;
            // Check if node is valid
            if (node.type == 'super_node') {
                for (const childNode of this.canvasController.getNodes(node.subflow_ref.pipeline_id_ref)) {
                    validNode =
                        (yield this.validateNode(childNode, node.subflow_ref.pipeline_id_ref)) && validNode;
                }
                if (validNode) {
                    node.app_data.invalidNodeError = null;
                }
                else {
                    if (!node.app_data) {
                        node.app_data = {};
                    }
                    node.app_data.invalidNodeError = 'Supernode contains invalid nodes.';
                }
                indicatorXPos = 15;
                indicatorYPos = 0;
            }
            else if (node.type == 'execution_node') {
                node.app_data.invalidNodeError = yield this.validateProperties(node);
                indicatorXPos = 20;
                indicatorYPos = 3;
            }
            else {
                return true;
            }
            // update app_data with latest invalidNodeError value
            this.canvasController.setNodeProperties(node.id, { app_data: node.app_data }, pipelineId);
            // Add or remove decorations
            if (node.app_data != null && node.app_data.invalidNodeError != null) {
                this.canvasController.setNodeDecorations(node.id, [
                    {
                        id: 'error',
                        image: ui_components_1.IconUtil.encode(ui_components_1.errorIcon),
                        outline: false,
                        position: 'topLeft',
                        x_pos: indicatorXPos,
                        y_pos: indicatorYPos
                    }
                ], pipelineId);
                const stylePipelineObj = {};
                stylePipelineObj[pipelineId] = [node.id];
                const styleSpec = {
                    body: { default: 'stroke: var(--jp-error-color1);' },
                    selection_outline: { default: 'stroke: var(--jp-error-color1);' },
                    label: { default: 'fill: var(--jp-error-color1);' }
                };
                this.canvasController.setObjectsStyle(stylePipelineObj, styleSpec, true);
                return false;
            }
            else {
                // Remove any existing decorations if valid
                const stylePipelineObj = {};
                stylePipelineObj[pipelineId] = [node.id];
                const styleSpec = {
                    body: { default: '' },
                    selection_outline: { default: '' },
                    label: { default: '' }
                };
                this.canvasController.setObjectsStyle(stylePipelineObj, styleSpec, true);
                this.canvasController.setNodeDecorations(node.id, [], pipelineId);
                return true;
            }
        });
    }
    /**
     * Validates the properties of a given node.
     *
     * @param node: node to check properties for
     *
     * @returns a warning message to display in the tooltip
     * if there are invalid properties. If there are none,
     * returns null.
     */
    validateProperties(node) {
        return __awaiter(this, void 0, void 0, function* () {
            const validationErrors = [];
            const notebookValidationErr = yield this.serviceManager.contents
                .get(PipelineService_1.PipelineService.getWorkspaceRelativeNodePath(this.widgetContext.path, node.app_data.filename))
                .then((result) => {
                return null;
            })
                .catch((err) => {
                return 'notebook does not exist';
            });
            if (notebookValidationErr) {
                validationErrors.push(notebookValidationErr);
            }
            if (node.app_data.runtime_image == null ||
                node.app_data.runtime_image == '') {
                validationErrors.push('no runtime image');
            }
            if (node.app_data.cpu != null && node.app_data.cpu <= 0) {
                validationErrors.push('CPU must be greater than 0');
            }
            if (node.app_data.gpu != null && node.app_data.gpu < 0) {
                validationErrors.push('GPU must be greater than or equal to 0');
            }
            if (node.app_data.memory != null && node.app_data.memory <= 0) {
                validationErrors.push('Memory must be greater than 0');
            }
            return validationErrors.length == 0 ? null : validationErrors.join('\n');
        });
    }
    /**
     * Validates the properties of all nodes in the pipeline.
     * Updates the decorations / style of all nodes.
     *
     * @returns null if all nodes are valid, error message if
     * invalid.
     */
    validateAllNodes() {
        return __awaiter(this, void 0, void 0, function* () {
            let errorMessage = null;
            // Reset any existing flagged nodes' style
            const pipelineId = this.canvasController.getPrimaryPipelineId();
            for (const node of this.canvasController.getNodes(pipelineId)) {
                const validNode = yield this.validateNode(node, pipelineId);
                if (!validNode) {
                    errorMessage = 'Some nodes have missing or invalid properties. ';
                }
            }
            return errorMessage;
        });
    }
    /**
     * Validates all links in the pipeline.
     * Updates the decorations / style of links.
     *
     * @returns null if pipeline is valid, error message if not.
     */
    validateAllLinks() {
        const links = this.canvasController.getLinks();
        const taintedLinks = validation_1.checkCircularReferences(links);
        // reset styles.
        const pipelineId = this.canvasController.getPrimaryPipelineId();
        const allSeenLinks = { [pipelineId]: links.map(l => l.id) };
        const defaultStyle = { line: { default: '' } };
        this.canvasController.setLinksStyle(allSeenLinks, defaultStyle, true);
        // set error styles
        const cycleLinks = { [pipelineId]: [...taintedLinks] };
        const errorStyle = {
            line: {
                default: 'stroke-dasharray: 13; stroke: var(--jp-error-color1);'
            }
        };
        this.canvasController.setLinksStyle(cycleLinks, errorStyle, true);
        if (taintedLinks.length > 0) {
            return 'Circular references in pipeline.';
        }
        return null;
    }
    /**
     * Validates all links and nodes in the pipeline.
     * Updates the decorations / style of links and nodes.
     *
     * @returns null if pipeline is valid, error message if not.
     */
    validatePipeline() {
        return __awaiter(this, void 0, void 0, function* () {
            const nodeErrorMessage = yield this.validateAllNodes();
            const linkErrorMessage = this.validateAllLinks();
            if (nodeErrorMessage || linkErrorMessage) {
                return ('Invalid pipeline: ' +
                    (nodeErrorMessage == null ? '' : nodeErrorMessage) +
                    (linkErrorMessage == null ? '' : linkErrorMessage));
            }
            else {
                return null;
            }
        });
    }
    /**
     * Displays a dialog containing a JSON error
     */
    handleJSONError(error) {
        apputils_1.showDialog({
            title: 'The pipeline file is not valid JSON.',
            body: (React.createElement("p", null,
                error.name,
                ": ",
                error.message)),
            buttons: [apputils_1.Dialog.okButton()]
        }).then(result => {
            this.handleClosePipeline();
        });
    }
    handleRunPipeline() {
        return __awaiter(this, void 0, void 0, function* () {
            // Check that all nodes are valid
            const errorMessage = yield this.validatePipeline();
            if (errorMessage) {
                this.setState({
                    showValidationError: true,
                    validationError: {
                        errorMessage: errorMessage,
                        errorSeverity: 'error'
                    }
                });
                return;
            }
            if (this.widgetContext.model.dirty) {
                const dialogResult = yield apputils_1.showDialog({
                    title: 'This pipeline contains unsaved changes. To submit the pipeline the changes need to be saved.',
                    buttons: [
                        apputils_1.Dialog.cancelButton(),
                        apputils_1.Dialog.okButton({ label: 'Save and Submit' })
                    ]
                });
                if (dialogResult.button && dialogResult.button.accept === true) {
                    yield this.widgetContext.save();
                }
                else {
                    // Don't proceed if cancel button pressed
                    return;
                }
            }
            const pipelineName = coreutils_1.PathExt.basename(this.widgetContext.path, coreutils_1.PathExt.extname(this.widgetContext.path));
            const action = 'run pipeline';
            const runtimes = yield PipelineService_1.PipelineService.getRuntimes(false, action).catch(error => ui_components_1.RequestErrors.serverError(error));
            const schema = yield PipelineService_1.PipelineService.getRuntimesSchema().catch(error => ui_components_1.RequestErrors.serverError(error));
            const localRuntime = {
                name: 'local',
                display_name: 'Run in-place locally',
                schema_name: 'local'
            };
            runtimes.unshift(JSON.parse(JSON.stringify(localRuntime)));
            const localSchema = {
                name: 'local',
                display_name: 'Local Runtime'
            };
            schema.unshift(JSON.parse(JSON.stringify(localSchema)));
            const dialogOptions = {
                title: 'Run pipeline',
                body: formDialogWidget_1.formDialogWidget(React.createElement(PipelineSubmissionDialog_1.PipelineSubmissionDialog, { name: pipelineName, runtimes: runtimes, schema: schema })),
                buttons: [apputils_1.Dialog.cancelButton(), apputils_1.Dialog.okButton()],
                defaultButton: 1,
                focusNodeSelector: '#pipeline_name'
            };
            const dialogResult = yield ui_components_1.showFormDialog(dialogOptions);
            if (dialogResult.value === null) {
                // When Cancel is clicked on the dialog, just return
                return;
            }
            // prepare pipeline submission details
            const pipelineFlow = this.canvasController.getPipelineFlow();
            const runtime_config = dialogResult.value.runtime_config;
            const runtime = PipelineService_1.PipelineService.getRuntimeName(runtime_config, runtimes) || 'local';
            PipelineService_1.PipelineService.setNodePathsRelativeToWorkspace(pipelineFlow.pipelines[0], this.widgetContext.path);
            this.cleanNullProperties();
            pipelineFlow.pipelines[0]['app_data']['name'] =
                dialogResult.value.pipeline_name;
            pipelineFlow.pipelines[0]['app_data']['runtime'] = runtime;
            pipelineFlow.pipelines[0]['app_data']['runtime-config'] = runtime_config;
            pipelineFlow.pipelines[0]['app_data']['source'] = coreutils_1.PathExt.basename(this.widgetContext.path);
            PipelineService_1.PipelineService.submitPipeline(pipelineFlow, PipelineService_1.PipelineService.getDisplayName(dialogResult.value.runtime_config, runtimes)).catch(error => ui_components_1.RequestErrors.serverError(error));
        });
    }
    handleSavePipeline() {
        this.updateModel();
        this.widgetContext.save();
    }
    handleClearPipeline(data) {
        return apputils_1.showDialog({
            title: 'Clear Pipeline',
            body: 'Are you sure you want to clear the pipeline?',
            buttons: [apputils_1.Dialog.cancelButton(), apputils_1.Dialog.okButton({ label: 'Clear' })]
        }).then(result => {
            if (result.button.accept) {
                // select all canvas elements
                this.canvasController.selectAll();
                // trigger delete of all selected canvas elements
                this.canvasController.editActionHandler({
                    editType: 'deleteSelectedObjects',
                    editSource: data.editSource,
                    pipelineId: data.pipelineId
                });
            }
        });
    }
    handleOpenRuntimes() {
        this.shell.activateById(`elyra-metadata:${PipelineService_1.RUNTIMES_NAMESPACE}`);
    }
    handleClosePipeline() {
        if (this.shell.currentWidget) {
            this.shell.currentWidget.close();
        }
    }
    componentDidMount() {
        this.initPropertiesInfo().finally(() => {
            this.handleOpenPipeline();
        });
    }
    componentDidUpdate() {
        const inputFields = document.querySelectorAll('.properties-readonly');
        for (const inputField of inputFields) {
            if (inputField.children.length > 1) {
                continue;
            }
            const tooltip = document.createElement('span');
            tooltip.className = 'elyra-Tooltip common-canvas-tooltip';
            tooltip.setAttribute('direction', 'bottom');
            const arrow = document.createElement('div');
            arrow.className = 'elyra-Tooltip-arrow';
            inputField.appendChild(arrow);
            const arrowOutline = document.createElement('div');
            arrowOutline.className =
                'elyra-Tooltip-arrow elyra-Tooltip-arrow-outline';
            inputField.appendChild(arrowOutline);
            tooltip.innerText = inputField.children[0].innerHTML;
            inputField.appendChild(tooltip);
        }
    }
}
exports.PipelineEditor = PipelineEditor;
class PipelineEditorFactory extends docregistry_1.ABCWidgetFactory {
    constructor(options) {
        super(options);
        this.shell = options.shell;
        this.commands = options.commands;
        this.browserFactory = options.browserFactory;
        this.serviceManager = options.serviceManager;
        this.addFileToPipelineSignal = new signaling_1.Signal(this);
    }
    createNewWidget(context) {
        // Creates a blank widget with a DocumentWidget wrapper
        const props = {
            shell: this.shell,
            commands: this.commands,
            browserFactory: this.browserFactory,
            context: context,
            addFileToPipelineSignal: this.addFileToPipelineSignal,
            serviceManager: this.serviceManager
        };
        const content = new PipelineEditorWidget(props);
        const widget = new docregistry_1.DocumentWidget({
            content,
            context,
            node: document.createElement('div')
        });
        widget.addClass(PIPELINE_CLASS);
        widget.title.icon = ui_components_1.pipelineIcon;
        return widget;
    }
}
exports.PipelineEditorFactory = PipelineEditorFactory;
//# sourceMappingURL=PipelineEditorWidget.js.map