"use strict";
/*
 * Copyright 2018-2021 Elyra Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PipelineService = exports.RUNTIMES_NAMESPACE = exports.KFP_SCHEMA = void 0;
const services_1 = require("@elyra/services");
const ui_components_1 = require("@elyra/ui-components");
const apputils_1 = require("@jupyterlab/apputils");
const coreutils_1 = require("@jupyterlab/coreutils");
const React = __importStar(require("react"));
const utils_1 = __importDefault(require("./utils"));
exports.KFP_SCHEMA = 'kfp';
exports.RUNTIMES_NAMESPACE = 'runtimes';
class PipelineService {
    /**
     * Returns a list of external runtime configurations available as
     * `runtimes metadata`. This is used to submit the pipeline to be
     * executed on these runtimes.
     */
    static getRuntimes(showError = true, action) {
        return __awaiter(this, void 0, void 0, function* () {
            return services_1.MetadataService.getMetadata(exports.RUNTIMES_NAMESPACE).then(runtimes => {
                if (showError && Object.keys(runtimes).length === 0) {
                    return ui_components_1.RequestErrors.noMetadataError('runtime', action);
                }
                return runtimes;
            });
        });
    }
    /**
     * Returns a list of runtime schema
     */
    static getRuntimesSchema(showError = true) {
        return __awaiter(this, void 0, void 0, function* () {
            return services_1.MetadataService.getSchema(exports.RUNTIMES_NAMESPACE).then(schema => {
                if (showError && Object.keys(schema).length === 0) {
                    return ui_components_1.RequestErrors.noMetadataError('schema');
                }
                return schema;
            });
        });
    }
    /**
     * Return a list of configured docker images that are used as runtimes environments
     * to run the pipeline nodes.
     */
    static getRuntimeImages() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                let runtimeImages = yield services_1.MetadataService.getMetadata('runtime-images');
                runtimeImages = runtimeImages.sort((a, b) => 0 - (a.name > b.name ? -1 : 1));
                if (Object.keys(runtimeImages).length === 0) {
                    return ui_components_1.RequestErrors.noMetadataError('runtime image');
                }
                const images = {};
                for (const image in runtimeImages) {
                    const imageName = runtimeImages[image]['metadata']['image_name'];
                    images[imageName] = runtimeImages[image]['display_name'];
                }
                return images;
            }
            catch (error) {
                Promise.reject(error);
            }
        });
    }
    static getDisplayName(name, metadataArr) {
        return metadataArr.find(r => r['name'] === name)['display_name'];
    }
    /**
     * The runtime name is currently based on the schema name (one schema per runtime)
     * @param name
     * @param metadataArr
     */
    static getRuntimeName(name, metadataArr) {
        return metadataArr.find(r => r['name'] === name)['schema_name'];
    }
    /**
     * Creates a Dialog for passing to makeServerRequest
     */
    static getWaitDialog(title = 'Making server request...', body = 'This may take some time') {
        return new apputils_1.Dialog({
            title: title,
            body: body,
            buttons: [apputils_1.Dialog.okButton()]
        });
    }
    /**
     * Submit the pipeline to be executed on an external runtime (e.g. Kbeflow Pipelines)
     *
     * @param pipeline
     * @param runtimeName
     */
    static submitPipeline(pipeline, runtimeName) {
        return __awaiter(this, void 0, void 0, function* () {
            return services_1.RequestHandler.makePostRequest('elyra/pipeline/schedule', JSON.stringify(pipeline), this.getWaitDialog('Packaging and submitting pipeline ...')).then(response => {
                let dialogTitle;
                let dialogBody;
                if (response['run_url']) {
                    // pipeline executed remotely in a runtime of choice
                    dialogTitle = 'Job submission to ' + runtimeName + ' succeeded';
                    dialogBody = (React.createElement("p", null,
                        response['platform'] == 'airflow' ? (React.createElement("p", null,
                            "Apache Airflow DAG has been pushed to the",
                            ' ',
                            React.createElement("a", { href: response['git_url'], target: "_blank", rel: "noopener noreferrer" }, "GitHub Repository."),
                            React.createElement("br", null))) : null,
                        "Check the status of your job at",
                        ' ',
                        React.createElement("a", { href: response['run_url'], target: "_blank", rel: "noopener noreferrer" }, "Run Details."),
                        React.createElement("br", null),
                        "The results and outputs are in the ",
                        response['object_storage_path'],
                        ' ',
                        "working directory in",
                        ' ',
                        React.createElement("a", { href: response['object_storage_url'], target: "_blank", rel: "noopener noreferrer" }, "object storage"),
                        "."));
                }
                else {
                    // pipeline executed in-place locally
                    dialogTitle = 'Job execution succeeded';
                    dialogBody = (React.createElement("p", null, "Your job has been executed in-place in your local environment."));
                }
                return apputils_1.showDialog({
                    title: dialogTitle,
                    body: dialogBody,
                    buttons: [apputils_1.Dialog.okButton()]
                });
            });
        });
    }
    /**
     * Export a pipeline to different formats (e.g. DSL, YAML, etc). These formats
     * are understood by a given runtime.
     *
     * @param pipeline
     * @param pipeline_export_format
     * @param pipeline_export_path
     * @param overwrite
     */
    static exportPipeline(pipeline, pipeline_export_format, pipeline_export_path, overwrite) {
        return __awaiter(this, void 0, void 0, function* () {
            console.log('Exporting pipeline to [' + pipeline_export_format + '] format');
            console.log('Overwriting existing file: ' + overwrite);
            const body = {
                pipeline: pipeline,
                export_format: pipeline_export_format,
                export_path: pipeline_export_path,
                overwrite: overwrite
            };
            return services_1.RequestHandler.makePostRequest('elyra/pipeline/export', JSON.stringify(body), this.getWaitDialog('Generating pipeline artifacts ...')).then(response => {
                return apputils_1.showDialog({
                    title: 'Pipeline export succeeded',
                    body: React.createElement("p", null,
                        "Exported file: ",
                        response['export_path'],
                        " "),
                    buttons: [apputils_1.Dialog.okButton()]
                });
            });
        });
    }
    /**
     * Verify if the given pipeline is "current" by looking on it's version, and perform
     * any conversion if needed.
     *
     * @param pipelineDefinition
     */
    static convertPipeline(pipelineDefinition, pipelinePath) {
        let pipelineJSON = JSON.parse(JSON.stringify(pipelineDefinition));
        const currentVersion = utils_1.default.getPipelineVersion(pipelineJSON);
        if (currentVersion < 1) {
            // original pipeline definition without a version
            console.info('Migrating pipeline to version 1.');
            pipelineJSON = this.convertPipelineV0toV1(pipelineJSON);
        }
        if (currentVersion < 2) {
            // adding relative path on the pipeline filenames
            console.info('Migrating pipeline to version 2.');
            pipelineJSON = this.convertPipelineV1toV2(pipelineJSON, pipelinePath);
        }
        if (currentVersion < 3) {
            // Adding python script support
            console.info('Migrating pipeline to version 3 (current version).');
            pipelineJSON = this.convertPipelineV2toV3(pipelineJSON, pipelinePath);
        }
        return pipelineJSON;
    }
    static convertPipelineV0toV1(pipelineJSON) {
        utils_1.default.renamePipelineAppdataField(pipelineJSON.pipelines[0], 'title', 'name');
        utils_1.default.deletePipelineAppdataField(pipelineJSON.pipelines[0], 'export');
        utils_1.default.deletePipelineAppdataField(pipelineJSON.pipelines[0], 'export_format');
        utils_1.default.deletePipelineAppdataField(pipelineJSON.pipelines[0], 'export_path');
        // look into nodes
        for (const nodeKey in pipelineJSON.pipelines[0]['nodes']) {
            const node = pipelineJSON.pipelines[0]['nodes'][nodeKey];
            utils_1.default.renamePipelineAppdataField(node, 'artifact', 'filename');
            utils_1.default.renamePipelineAppdataField(node, 'image', 'runtime_image');
            utils_1.default.renamePipelineAppdataField(node, 'vars', 'env_vars');
            utils_1.default.renamePipelineAppdataField(node, 'file_dependencies', 'dependencies');
            utils_1.default.renamePipelineAppdataField(node, 'recursive_dependencies', 'include_subdirectories');
        }
        pipelineJSON.pipelines[0]['app_data']['version'] = 1;
        return pipelineJSON;
    }
    static convertPipelineV1toV2(pipelineJSON, pipelinePath) {
        pipelineJSON.pipelines[0] = this.setNodePathsRelativeToPipeline(pipelineJSON.pipelines[0], pipelinePath);
        pipelineJSON.pipelines[0]['app_data']['version'] = 2;
        return pipelineJSON;
    }
    static convertPipelineV2toV3(pipelineJSON, pipelinePath) {
        // No-Op this is to disable old versions of Elyra
        // to see a pipeline with Python Script nodes
        pipelineJSON.pipelines[0]['app_data']['version'] = 3;
        return pipelineJSON;
    }
    static getPipelineRelativeNodePath(pipelinePath, nodePath) {
        const relativePath = coreutils_1.PathExt.relative(coreutils_1.PathExt.dirname(pipelinePath), nodePath);
        return relativePath;
    }
    static getWorkspaceRelativeNodePath(pipelinePath, nodePath) {
        // since resolve returns an "absolute" path we need to strip off the leading '/'
        const workspacePath = coreutils_1.PathExt.resolve(coreutils_1.PathExt.dirname(pipelinePath), nodePath);
        return workspacePath;
    }
    static setNodePathsRelativeToPipeline(pipeline, pipelinePath) {
        for (const node of pipeline.nodes) {
            node.app_data.filename = this.getPipelineRelativeNodePath(pipelinePath, node.app_data.filename);
        }
        return pipeline;
    }
    static setNodePathsRelativeToWorkspace(pipeline, pipelinePath) {
        for (const node of pipeline.nodes) {
            node.app_data.filename = this.getWorkspaceRelativeNodePath(pipelinePath, node.app_data.filename);
        }
        return pipeline;
    }
}
exports.PipelineService = PipelineService;
/**
 * Returns a list of external runtime configurations
 * based on the runtimePlatform (Airflow or Kubeflow)
 */
PipelineService.filterRuntimes = (runtimes, runtimePlatform) => runtimes.filter(runtime => runtime.schema_name === runtimePlatform);
/**
 * Returns a list of external schema configurations
 * based a list of runtimes instances
 */
PipelineService.filterValidSchema = (runtimes, schema) => schema.filter(s => runtimes.some(runtime => runtime.schema_name === s.name));
/**
 * Sorts given list of runtimes by the display_name property
 */
PipelineService.sortRuntimesByDisplayName = (runtimes) => {
    runtimes.sort((r1, r2) => r1.display_name.localeCompare(r2.display_name));
};
//# sourceMappingURL=PipelineService.js.map