"use strict";
/*
 * Copyright 2018-2021 Elyra Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const application_1 = require("@jupyterlab/application");
const coreutils_1 = require("@jupyterlab/coreutils");
const v4_1 = __importDefault(require("uuid/v4"));
const constants_1 = require("./constants");
const pipeline_template_json_1 = __importDefault(require("./pipeline-template.json"));
/**
 * A utilities class for static functions.
 */
class Utils {
    static getUUID() {
        return v4_1.default();
    }
    /**
     * Utility to create a one node pipeline to submit a single file as a pipeline
     */
    static generateSingleFilePipeline(filename, runtime_platform, runtime_config, runtimeImage, dependencies, envObject) {
        const template = JSON.parse(JSON.stringify(pipeline_template_json_1.default));
        const generated_uuid = Utils.getUUID();
        const artifactName = coreutils_1.PathExt.basename(filename, coreutils_1.PathExt.extname(filename));
        const envVars = Object.entries(envObject).map(([key, val]) => `${key}=${val}`);
        template.id = generated_uuid;
        template.primary_pipeline = generated_uuid;
        template.pipelines[0].id = generated_uuid;
        template.pipelines[0].nodes[0].id = generated_uuid;
        template.pipelines[0].nodes[0].app_data.filename = filename;
        template.pipelines[0].nodes[0].app_data.runtime_image = runtimeImage;
        template.pipelines[0].nodes[0].app_data.env_vars = envVars;
        template.pipelines[0].nodes[0].app_data.dependencies = dependencies;
        template.pipelines[0].app_data.name = artifactName;
        template.pipelines[0].app_data.runtime = runtime_platform;
        template.pipelines[0].app_data['runtime-config'] = runtime_config;
        template.pipelines[0].app_data.version = constants_1.PIPELINE_CURRENT_VERSION;
        template.pipelines[0].app_data.source =
            'Generated for ' + coreutils_1.PathExt.basename(filename);
        return template;
    }
    /**
     * Check if the provided pipeline is empty (no nodes)
     *
     * @param pipelineDefinition
     */
    static isEmptyPipeline(pipelineDefinition) {
        return Object.keys(pipelineDefinition.pipelines[0].nodes).length === 0;
    }
    /**
     * Check if the provided pipeline is clear of nodes and comments
     *
     * @param pipelineDefinition
     */
    static isEmptyCanvas(pipelineDefinition) {
        return (this.isEmptyPipeline(pipelineDefinition) &&
            pipelineDefinition.pipelines[0].app_data.ui_data.comments.length === 0);
    }
    /**
     * Read the version of a Pipeline. If no version is found return 0
     *
     * @param pipelineDefinition
     */
    static getPipelineVersion(pipelineDefinition) {
        let version = 0;
        if (pipelineDefinition)
            version =
                +this.getPipelineAppdataField(pipelineDefinition.pipelines[0], 'version') || 0;
        return version;
    }
    /**
     * Read an application specific field from the pipeline definition
     * (e.g. pipelines[0][app_data][fieldName])
     */
    static getPipelineAppdataField(node, fieldName) {
        if (this.hasPipelineAppdataField(node, fieldName)) {
            return node['app_data'][fieldName];
        }
        else {
            return null;
        }
    }
    /**
     * Check if an application specific field from the pipeline defintion exists
     * (e.g. pipelines[0][app_data][fieldName])
     */
    static hasPipelineAppdataField(node, fieldName) {
        return (Object.prototype.hasOwnProperty.call(node, 'app_data') &&
            Object.prototype.hasOwnProperty.call(node['app_data'], fieldName));
    }
    /**
     * Delete an application specific field from the pipeline definition
     * (e.g. pipelines[0][app_data][fieldName])
     */
    static deletePipelineAppdataField(node, fieldName) {
        if (this.hasPipelineAppdataField(node, fieldName)) {
            delete node['app_data'][fieldName];
        }
    }
    /**
     * Rename an application specific field from the pepileine definition if it exists by
     * by copying the field value to the new field name and then deleting the previously
     * existing field
     */
    static renamePipelineAppdataField(node, currentFieldName, newFieldName) {
        if (this.hasPipelineAppdataField(node, currentFieldName)) {
            node['app_data'][newFieldName] = node['app_data'][currentFieldName];
            this.deletePipelineAppdataField(node, currentFieldName);
        }
    }
    /**
     * Break an array into an array of "chunks", each "chunk" having "n" elements.
     * The final "chuck" may have less than "n" elements.
     * Example:
     * chunkArray(['a', 'b', 'c', 'd', 'e', 'f', 'g'], 4)
     * -> [['a', 'b', 'c', 'd'], ['e', 'f', 'g']]
     */
    static chunkArray(arr, n) {
        return Array.from(Array(Math.ceil(arr.length / n)), (_, i) => arr.slice(i * n, i * n + n));
    }
    /**
     * Check if the object is not an array, but the result of a Dialog instead
     */
    static isDialogResult(runtimesObj) {
        return runtimesObj && !(runtimesObj instanceof Array) && runtimesObj.button;
    }
}
exports.default = Utils;
/**
 * From a given widget, find the application shell and return it
 */
Utils.getLabShell = (widget) => {
    while (widget !== null && !(widget instanceof application_1.LabShell)) {
        widget = widget.parent;
    }
    return widget;
};
//# sourceMappingURL=utils.js.map