# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, unused-wildcard-import
"""
|lland_knauf_ic| is still under development and likely to change in the future.  Its
goal is to extend |lland_knauf| with processes dealing with the interception of snow in
canopies, similar to the `SCHNEEINTERZEPTION` option of `LARSIM`_.  The publically
available `LARSIM`_ documentation does not provide all information required for
implementing a perfect emulation, but detailed comparisons suggest that the essential
process equations are defined similarly.

.. _`LARSIM`: http://www.larsim.de/en/the-model/

The following list summarises the current status of |lland_knauf_ic| by discussing its
main differences to |lland_knauf|:

 * Snow interception occurs only in tree canopies, which is why the results of
   |lland_knauf_ic| differ from those of |lland_knauf| only for the land-use types
   |LAUBW|, |MISCHW|, and |NADELW|.
 * There is are two additional storages, |SInz| and |STInz|, which handle the total
   and the frozen amount of intercepted snowfall, placed between the "normal"
   interception and the "normal" snow module.
 * If not stated otherwise, |SInz| and |STInz| correspond to |WAeS| and |WATS| and
   are modified by the same or functionally identical methods.
 * If the `LARSIM`_ documentation does not suggest otherwise, we reuse the names
   of the "normal" snow module and suffix "Inz" to it or replace its "Snow" suffix
   (for example, |WaDa| becomes |WaDaInz| and |NetRadiationSnow| becomes
   |NetRadiationInz|).
 * Only a fraction of the water leaving the "normal" interception module reaches the
   snow interception module (see |Calc_NBesInz_V1| and |Calc_SnowIntRate_V1|).
 * The surface temperature of the intercepted snow is always identical with its bulk
   temperature (|TempSInz|), except for the calculation of the latent heat flux
   (see |Return_WSurfInz_V1|).
 * The energy balance of the intercepted snow does not (need to) consider the soil
   heat flux.
 * The radiation exchange with the atmosphere depends on how much the intercepted
   snow shields the ground (see |Calc_NetShortwaveRadiationInz_V1| and
   |Return_NetLongwaveRadiationInz_V1|).
 * During periods with intercepted snow, water evaporates (or sublimates) from the snow
   interception module (|EvSInz|) but not from the "normal" interception module (|EvI|).

:cite:t:`ref-LARSIM` discusses many of these points in more detail.

In our experience, |lland_knauf_ic| tends to calculate thicker snowpacks on the ground
than |lland_knauf| (the highest values of |WAeS| are usually larger when applying
|lland_knauf_ic|).  This counterintuitive behaviour seems to be related to disabling
the "normal" interception evaporation (|EvI|) during snow interception periods and is
still under discussion.

We will add further explanations and references as soon as we can consolidate
|lland_knauf_ic|.

The following integration tests are identical to the ones of |lland_knauf|, except for
the additional snow interception processes.

Integration tests
=================

.. how_to_understand_integration_tests::

daily simulation
________________

>>> from hydpy import pub
>>> pub.timegrids = "1997-08-01", "1997-09-01", "1d"
>>> from hydpy.models.lland_knauf_ic import *
>>> parameterstep("1h")
>>> from hydpy import Node, Element
>>> outlet = Node("outlet")
>>> inlet = Node("inlet")
>>> land = Element("land", inlets=inlet, outlets=outlet)
>>> land.model = model
>>> nhru(1)
>>> ft(1.0)
>>> fhru(1.0)

.. _lland_knauf_ic_acker_summer_daily:

acre (summer)
-------------

>>> lnk(ACKER)
>>> kg(0.94)
>>> kt(0.0)
>>> hinz(0.2)
>>> lai.acker_aug = 3.5
>>> tgr(0.0)
>>> trefn(0.0)
>>> tsp(4.0)
>>> pwmax(1.43)
>>> wmax(309.0)
>>> fk(199.0)
>>> pwp(119.4)
>>> kapgrenz(option="0_WMax/10")
>>> kapmax(0.08)
>>> beta(0.001/24)
>>> fbeta(1.0)
>>> rbeta(False)
>>> dmax(r_dmax=0.55)
>>> dmin(r_dmin=10.0)
>>> bsf(0.3)
>>> volbmax(inf)
>>> gsbmax(1.0)
>>> gsbgrad1(inf)
>>> gsbgrad2(inf)
>>> a1(0.0)
>>> a2(0.5)
>>> tind(0.191056)
>>> eqb(35000.0)
>>> eqi1(400.0)
>>> eqi2(400.0)
>>> eqd1(200.0)
>>> eqd2(50.0)

>>> negq(True)

>>> measuringheightwindspeed(10.0)
>>> wg2z.aug = -2.9
>>> p1strahl(0.5)
>>> p2strahl(1.0/35.0)
>>> p1wind(0.6)
>>> p2wind(1.0/70.0)
>>> turb0(2.0)
>>> turb1(1.6)
>>> albedo0snow(0.9)
>>> snowagingfactor(0.35)
>>> refreezeflag(True)
>>> ktschnee(inf)

>>> fvf(0.3)
>>> bsff(2.0)

>>> p1simax(8.0)
>>> p2simax(1.5)
>>> p1sirate(0.2)
>>> p2sirate(0.02)
>>> p3sirate(0.003)

>>> with model.add_radiationmodel_v4("meteo_psun_sun_glob_io") as submodel_meteo_psun_sun_glob_io:
...     pass

>>> with model.add_aetmodel_v1("evap_aet_morsim"):
...     albedo.acker_aug = 0.24
...     cropheight.acker_aug = 0.4
...     surfaceresistance.acker = 40.0
...     emissivity(0.95)
...     averagesoilheatflux.aug = 2.9
...     soilmoisturelimit(119.4 / 309.0)
...     model.add_radiationmodel_v4(submodel_meteo_psun_sun_glob_io)

>>> from hydpy import IntegrationTest
>>> IntegrationTest.plotting_options.axis1 = inputs.nied
>>> IntegrationTest.plotting_options.axis2 = fluxes.qah
>>> test = IntegrationTest(land)
>>> test.dateformat = "%Y-%d-%m"

>>> test.inits = ((states.inzp, 0.0),
...               (states.stinz, 0.0),
...               (states.sinz, 0.0),
...               (states.esnowinz, 0.0),
...               (states.asinz, 0.0),
...               (states.wats, 0.0),
...               (states.waes, 0.0),
...               (states.esnow, 0.0),
...               (states.taus, 0.0),
...               (states.ebdn, 350.0),
...               (states.bowa, 72.0),
...               (states.sdg1, 0.0),
...               (states.sdg2, 0.0),
...               (states.sig1, 0.0),
...               (states.sig2, 0.0),
...               (states.sbg, 0.0))

>>> inputs.nied.series = (
...     13.3, 0.1, 0.0, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.0, 0.0, 2.7,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.0, 0.3, 0.0, 0.1, 10.5, 6.5, 15.0, 9.2, 0.0,
...     0.0)
>>> inputs.teml.series = (
...     16.1, 17.3, 18.9, 17.8, 18.5, 19.9, 20.8, 20.9, 22.3, 23.0, 23.1, 23.2, 23.2,
...     21.7, 20.8, 19.5, 21.0, 21.6, 21.1, 22.4, 22.5, 19.6, 20.3, 24.1, 23.7, 21.1,
...     20.9, 20.2, 16.1, 17.0, 18.2)
>>> inputs.windspeed.series = (
...     2.4, 1.6, 1.5, 2.8, 3.0, 3.5, 3.8, 3.3, 1.8, 1.9, 3.0, 2.5, 1.6, 2.7, 3.2, 1.3,
...     1.9, 2.0, 2.6, 3.6, 2.5, 2.1, 1.5, 2.2, 1.9, 2.3, 2.1, 2.6, 2.6, 2.1, 2.1)
>>> inputs.relativehumidity.series = (
...     86.2, 85.8, 82.4, 77.6, 74.9, 77.9, 73.5, 80.1, 72.9, 68.5, 68.6, 66.0, 69.0,
...     75.6, 81.4, 79.0, 75.9, 70.0, 70.3, 69.0, 62.1, 84.5, 83.6, 76.5, 76.6, 84.2,
...     85.8, 86.5, 89.6, 78.3, 78.5)

>>> model.radiationmodel.sequences.inputs.sunshineduration.series = (
...     6.3, 1.7, 4.5, 12.4, 13.9, 13.0, 13.8, 12.3, 13.1, 12.8, 13.1, 13.3, 12.7,
...     10.2, 9.4, 10.3, 11.1, 11.0, 8.5, 11.3, 12.4, 0.1, 6.7, 10.4, 6.5, 4.9, 6.6,
...     0.3, 0.1, 5.0, 3.8)

>>> aetinputs = model.aetmodel.sequences.inputs
>>> aetinputs.atmosphericpressure.series = (
...     1007.0, 1013.0, 1016.0, 1017.0, 1018.0, 1018.0, 1019.0, 1019.0, 1019.0, 1020.0,
...     1021.0, 1022.0, 1019.0, 1015.0, 1016.0, 1019.0, 1019.0, 1018.0, 1019.0, 1019.0,
...     1017.0, 1015.0, 1015.0, 1011.0, 1008.0, 1006.0, 1005.0, 1000.0, 1004.0, 1011.0,
...     1014.0)
>>> aetinputs.windspeed.series = inputs.windspeed.series
>>> aetinputs.relativehumidity.series = inputs.relativehumidity.series

>>> inlet.sequences.sim.series = 0.0

>>> model.radiationmodel.sequences.inputs.possiblesunshineduration.series = (
...     15.69078473, 15.62882066, 15.56613383, 15.50275437, 15.43871163, 15.37403418,
...     15.3087498, 15.24288548, 15.17646742, 15.10952106, 15.04207104, 14.97414122,
...     14.90575471, 14.83693386, 14.76770029, 14.69807488, 14.62807779, 14.5577285,
...     14.48704578, 14.41604776, 14.34475191, 14.27317506, 14.20133345, 14.12924271,
...     14.05691791, 13.98437357, 13.91162366, 13.83868165, 13.76556053, 13.69227282,
...     13.61883057)
>>> model.radiationmodel.sequences.inputs.globalradiation.series = (
...     190.2514903, 127.6607822, 164.4012978, 269.0205016, 287.8486672, 274.6742631,
...     284.0478917, 262.9712633, 272.2564135, 267.0192178, 269.6265466, 270.8782909,
...     261.6971638, 227.9621155, 216.3563213, 226.6379036, 235.5285025, 232.8494284,
...     199.7134896, 233.7719021, 246.1212861, 90.58551, 171.9449932, 216.5498126,
...     166.9168296, 145.9856109, 165.5438903, 87.6933456, 84.356885, 142.3092025,
...     126.7220785)

.. integration-test::

    >>> parameters.update()
    >>> test.reset_inits()
    >>> conditions = model.conditions
    >>> conditions_acker_summer = test(
    ...     "lland_knauf_ic_acker_summer_daily",
    ...     axis1=(inputs.nied, fluxes.qah), axis2=states.bowa,
    ...     get_conditions="1997-08-03")
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow |   nbes | sbes | snowintmax | snowintrate | nbesinz | sbesinz | wniedinz | actualalbedoinz | wadainz | schmpotinz | schminz | gefrpotinz | gefrinz | evsinz |   evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatinz | wlatsnow | wsensinz | wsenssnow | wsurfinz | wsurf | sff | fvg |   wada |      qdb | qib1 | qib2 | qbb |     qkap |     qdgz |    qdgz1 | qdgz2 | qigz1 | qigz2 |      qbgz |    qdga1 | qdga2 | qiga1 | qiga2 |      qbga |       qah |        qa | inzp | stinz | sinz | esnowinz | asinz | wats | waes | esnow | taus |       ebdn |      bowa |     sdg1 | sdg2 | sig1 | sig2 |       sbg | inlet |    outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.0 | 0.0 |                   6.3 |                     15.690785 |       190.25149 | 12.502 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                         0.0 |                          0.0 |            15.797694 | 11.733333 |      -26.2 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 11.802 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.7 | 1.720128 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 11.802 | 0.761343 |  0.0 |  0.0 | 0.0 |      0.0 | 0.761343 | 0.761343 |   0.0 |   0.0 |   0.0 |       0.0 | 0.196005 |   0.0 |   0.0 |   0.0 |       0.0 |  0.196005 |  0.002269 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |      373.3 | 81.320529 | 0.565338 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.002269 |
    | 1997-02-08 |  0.1 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.0 | 0.0 |                   1.7 |                     15.628821 |      127.660782 |  0.094 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                         0.0 |                          0.0 |            16.970501 | 18.443733 |     6.8624 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.094 | 1.697173 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.263667 |   0.0 |   0.0 |   0.0 |       0.0 |  0.263667 |  0.003052 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |   363.5376 | 79.623355 |  0.30167 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.003052 |
    | 1997-03-08 |  0.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.0 | 0.0 |                   4.5 |                     15.566134 |      164.401298 |    0.0 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                         0.0 |                          0.0 |            18.022179 | 15.632162 | -19.607027 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 2.329453 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.140696 |   0.0 |   0.0 |   0.0 |       0.0 |  0.140696 |  0.001628 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 380.244627 | 77.293902 | 0.160975 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.001628 |
    | 1997-04-08 |  0.1 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.0 | 0.0 |                  12.4 |                     15.502754 |      269.020502 |  0.094 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                         0.0 |                          0.0 |            15.840774 | 20.443786 |  15.862716 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.094 | 3.314337 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.075077 |   0.0 |   0.0 |   0.0 |       0.0 |  0.075077 |  0.000869 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 361.481911 | 73.979565 | 0.085898 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000869 |
    | 1997-05-08 |  0.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.0 | 0.0 |                  13.9 |                     15.438712 |      287.848667 |    0.0 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                         0.0 |                          0.0 |            15.977013 | 15.040124 | -20.759257 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 |  3.73228 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.040062 |   0.0 |   0.0 |   0.0 |       0.0 |  0.040062 |  0.000464 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 379.341169 | 70.247285 | 0.045836 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000464 |
    | 1997-06-08 |  0.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.0 | 0.0 |                  13.0 |                     15.374034 |      274.674263 |    0.0 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                         0.0 |                          0.0 |            18.131496 |  20.18359 |   1.701539 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 3.652387 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.021377 |   0.0 |   0.0 |   0.0 |       0.0 |  0.021377 |  0.000247 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 374.739629 | 66.594898 | 0.024459 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000247 |
    | 1997-07-08 |  0.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.0 | 0.0 |                  13.8 |                      15.30875 |      284.047892 |    0.0 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                         0.0 |                          0.0 |            18.084931 | 18.858347 | -11.649921 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 4.128687 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.011407 |   0.0 |   0.0 |   0.0 |       0.0 |  0.011407 |  0.000132 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  383.48955 |  62.46621 | 0.013051 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000132 |
    | 1997-08-08 |  0.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.0 | 0.0 |                  12.3 |                     15.242885 |      262.971263 |    0.0 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                         0.0 |                          0.0 |            19.830469 | 21.378324 |   2.869942 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 3.518399 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.006087 |   0.0 |   0.0 |   0.0 |       0.0 |  0.006087 |   0.00007 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 377.719608 | 58.947811 | 0.006964 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |   0.00007 |
    | 1997-09-08 |  0.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.176467 |      272.256414 |    0.0 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                         0.0 |                          0.0 |            19.660875 |  19.71658 | -15.500518 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 4.091259 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.003248 |   0.0 |   0.0 |   0.0 |       0.0 |  0.003248 |  0.000038 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 390.320126 | 54.856552 | 0.003716 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000038 |
    | 1997-10-08 |  0.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.0 | 0.0 |                  12.8 |                     15.109521 |      267.019218 |    0.0 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                         0.0 |                          0.0 |            19.275317 |  23.34553 |   2.073177 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 4.345753 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.001733 |   0.0 |   0.0 |   0.0 |       0.0 |  0.001733 |   0.00002 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 385.346949 | 50.510799 | 0.001983 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |   0.00002 |
    | 1997-11-08 |  0.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.042071 |      269.626547 |    0.0 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                         0.0 |                          0.0 |            19.420506 | 21.913255 |  -7.120473 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 4.547949 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000925 |   0.0 |   0.0 |   0.0 |       0.0 |  0.000925 |  0.000011 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 389.567421 | 45.962849 | 0.001058 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000011 |
    | 1997-12-08 |  0.1 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.0 | 0.0 |                  13.3 |                     14.974141 |      270.878291 |  0.094 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                         0.0 |                          0.0 |             18.79766 | 23.128751 |  -0.427496 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.094 | 4.554398 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000494 |   0.0 |   0.0 |   0.0 |       0.0 |  0.000494 |  0.000006 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 387.094917 | 41.408451 | 0.000565 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000006 |
    | 1997-13-08 |  0.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.0 | 0.0 |                  12.7 |                     14.905755 |      261.697164 |    0.0 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                         0.0 |                          0.0 |              19.6521 | 22.416669 |  -4.699983 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 4.137046 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000263 |   0.0 |   0.0 |   0.0 |       0.0 |  0.000263 |  0.000003 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |   388.8949 | 37.271405 | 0.000301 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000003 |
    | 1997-14-08 |  0.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.0 | 0.0 |                  10.2 |                     14.836934 |      227.962116 |    0.0 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                         0.0 |                          0.0 |            19.656892 | 22.935065 |   7.410388 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 3.428735 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000141 |   0.0 |   0.0 |   0.0 |       0.0 |  0.000141 |  0.000002 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 378.584513 |  33.84267 | 0.000161 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000002 |
    | 1997-15-08 |  2.7 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.0 | 0.0 |                   9.4 |                       14.7677 |      216.356321 |  2.538 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                         0.0 |                          0.0 |            20.028753 | 19.965673 |  -5.005962 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  1.838 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.7 | 2.505678 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  1.838 | 0.049893 |  0.0 |  0.0 | 0.0 |      0.0 | 0.049893 | 0.049893 |   0.0 |   0.0 |   0.0 |       0.0 |  0.01292 |   0.0 |   0.0 |   0.0 |       0.0 |   0.01292 |   0.00015 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 380.690475 | 33.125099 | 0.037134 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |   0.00015 |
    | 1997-16-08 |  0.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.0 | 0.0 |                  10.3 |                     14.698075 |      226.637904 |    0.0 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                         0.0 |                          0.0 |            17.936694 |  20.57219 |   6.433141 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 2.836506 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.017319 |   0.0 |   0.0 |   0.0 |       0.0 |  0.017319 |    0.0002 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 371.357334 | 30.288593 | 0.019815 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |    0.0002 |
    | 1997-17-08 |  0.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.0 | 0.0 |                  11.1 |                     14.628078 |      235.528503 |    0.0 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                         0.0 |                          0.0 |            18.906499 | 17.884246 | -18.694526 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 3.276924 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.03799 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  -0.03799 | 0.009242 |   0.0 |   0.0 |   0.0 | -0.000068 |  0.009173 |  0.000106 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 387.151861 | 27.049659 | 0.010574 |  0.0 |  0.0 |  0.0 | -0.037922 |   0.0 |  0.000106 |
    | 1997-18-08 |  0.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.0 | 0.0 |                  11.0 |                     14.557728 |      232.849428 |    0.0 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                         0.0 |                          0.0 |            18.089927 | 22.433069 |   4.998415 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 3.581415 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.239244 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -0.239244 | 0.004931 |   0.0 |   0.0 |   0.0 | -0.000565 |  0.004367 |  0.000051 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 379.253445 | 23.707488 | 0.005642 |  0.0 |  0.0 |  0.0 | -0.276602 |   0.0 |  0.000051 |
    | 1997-19-08 |  0.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.0 | 0.0 |                   8.5 |                     14.487046 |       199.71349 |    0.0 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                         0.0 |                          0.0 |            17.619414 | 20.158326 |  -5.650046 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 3.296835 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.446913 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -0.446913 | 0.002631 |   0.0 |   0.0 |   0.0 | -0.001792 |  0.000839 |   0.00001 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 382.003492 | 20.857567 | 0.003011 |  0.0 |  0.0 |  0.0 | -0.721723 |   0.0 |   0.00001 |
    | 1997-20-08 |  0.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.0 | 0.0 |                  11.3 |                     14.416048 |      233.771902 |    0.0 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                         0.0 |                          0.0 |            18.722517 | 20.950339 |  -8.697966 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 3.891994 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.623996 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -0.623996 | 0.001404 |   0.0 |   0.0 |   0.0 | -0.003704 |   -0.0023 | -0.000027 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 387.801458 | 17.589568 | 0.001607 |  0.0 |  0.0 |  0.0 | -1.342015 |   0.0 | -0.000027 |
    | 1997-21-08 |  0.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.0 | 0.0 |                  12.4 |                     14.344752 |      246.121286 |    0.0 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                         0.0 |                          0.0 |             16.95292 | 22.620153 |    0.72092 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 4.183176 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.827056 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -0.827056 | 0.000749 |   0.0 |   0.0 |   0.0 |  -0.00629 | -0.005541 | -0.000064 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 384.180539 | 14.233449 | 0.000857 |  0.0 |  0.0 |  0.0 | -2.162781 |   0.0 | -0.000064 |
    | 1997-22-08 |  2.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.0 | 0.0 |                   0.1 |                     14.273175 |        90.58551 |   1.88 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                         0.0 |                          0.0 |            19.305027 | 21.577328 |  11.863971 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   1.18 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.7 | 1.075698 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |   1.18 | 0.013305 |  0.0 |  0.0 | 0.0 | 1.035592 | 0.013305 | 0.013305 |   0.0 |   0.0 |   0.0 | -1.035592 | 0.003825 |   0.0 |   0.0 |   0.0 | -0.009605 | -0.005779 | -0.000067 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 369.416568 | 15.360037 | 0.010337 |  0.0 |  0.0 |  0.0 | -3.188768 |   0.0 | -0.000067 |
    | 1997-23-08 |  0.3 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.0 | 0.0 |                   6.7 |                     14.201333 |      171.944993 |  0.282 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                         0.0 |                          0.0 |            19.945708 | 17.325305 | -17.848171 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.282 | 2.014865 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.96559 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  -0.96559 | 0.004821 |   0.0 |   0.0 |   0.0 | -0.013155 | -0.008334 | -0.000096 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 384.364739 | 14.310761 | 0.005516 |  0.0 |  0.0 |  0.0 | -4.141203 |   0.0 | -0.000096 |
    | 1997-24-08 |  0.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.0 | 0.0 |                  10.4 |                     14.129243 |      216.549813 |    0.0 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                         0.0 |                          0.0 |            23.000686 | 21.630378 | -14.817732 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 3.362119 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.030788 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -1.030788 | 0.002573 |   0.0 |   0.0 |   0.0 | -0.016684 | -0.014111 | -0.000163 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  396.28247 |  11.97943 | 0.002943 |  0.0 |  0.0 |  0.0 | -5.155306 |   0.0 | -0.000163 |
    | 1997-25-08 |  0.1 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.0 | 0.0 |                   6.5 |                     14.056918 |       166.91683 |  0.094 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                         0.0 |                          0.0 |            22.484088 | 25.062685 |   8.176109 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.094 | 2.719352 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.175647 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -1.175647 | 0.001373 |   0.0 |   0.0 |   0.0 | -0.020577 | -0.019204 | -0.000222 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 385.206362 | 10.435725 | 0.001571 |  0.0 |  0.0 |  0.0 | -6.310376 |   0.0 | -0.000222 |
    | 1997-26-08 | 10.5 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.0 | 0.0 |                   4.9 |                     13.984374 |      145.985611 |   9.87 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                         0.0 |                          0.0 |            21.103195 | 21.872765 |   4.636593 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   9.17 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.7 | 1.597373 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |   9.17 | 0.104578 |  0.0 |  0.0 | 0.0 | 1.271567 | 0.104578 | 0.104578 |   0.0 |   0.0 |   0.0 | -1.271567 | 0.027656 |   0.0 |   0.0 |   0.0 | -0.024887 |  0.002769 |  0.000032 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 377.669769 | 19.175341 | 0.078493 |  0.0 |  0.0 |  0.0 | -7.557056 |   0.0 |  0.000032 |
    | 1997-27-08 |  6.5 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.0 | 0.0 |                   6.6 |                     13.911624 |       165.54389 |   6.11 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                         0.0 |                          0.0 |            21.241626 | 19.702227 |   -7.18664 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   5.41 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.7 | 1.722258 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |   5.41 | 0.090745 |  0.0 |  0.0 | 0.0 | 0.728523 | 0.090745 | 0.090745 |   0.0 |   0.0 |   0.0 | -0.728523 |  0.05997 |   0.0 |   0.0 |   0.0 |  -0.02838 |   0.03159 |  0.000366 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 381.956408 | 23.500861 | 0.109268 |  0.0 |  0.0 |  0.0 | -8.257199 |   0.0 |  0.000366 |
    | 1997-28-08 | 15.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.0 | 0.0 |                   0.3 |                     13.838682 |       87.693346 |   14.1 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                         0.0 |                          0.0 |            20.510475 | 20.936779 |   4.420674 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   13.4 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.7 | 1.071953 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |   13.4 | 0.312945 |  0.0 |  0.0 | 0.0 | 0.459752 | 0.312945 | 0.312945 |   0.0 |   0.0 |   0.0 | -0.459752 | 0.131528 |   0.0 |   0.0 |   0.0 | -0.030407 |  0.101121 |   0.00117 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 374.635735 | 35.975715 | 0.290685 |  0.0 |  0.0 |  0.0 | -8.686544 |   0.0 |   0.00117 |
    | 1997-29-08 |  9.2 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.0 | 0.0 |                   0.1 |                     13.765561 |       84.356885 |  8.648 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                         0.0 |                          0.0 |            16.420805 | 18.828425 |   16.37055 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  7.948 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.7 | 0.717164 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  7.948 | 0.249164 |  0.0 |  0.0 | 0.0 |      0.0 | 0.249164 | 0.249164 |   0.0 |   0.0 |   0.0 |       0.0 | 0.199719 |   0.0 |   0.0 |   0.0 | -0.031121 |  0.168598 |  0.001951 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 355.365185 | 42.957387 |  0.34013 |  0.0 |  0.0 |  0.0 | -8.655423 |   0.0 |  0.001951 |
    | 1997-30-08 |  0.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.0 | 0.0 |                   5.0 |                     13.692273 |      142.309202 |    0.0 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                         0.0 |                          0.0 |            15.195625 | 13.278507 |  -22.32896 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 2.035891 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.158633 |   0.0 |   0.0 |   0.0 | -0.031009 |  0.127624 |  0.001477 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 374.794145 | 40.921496 | 0.181497 |  0.0 |  0.0 |  0.0 | -8.624414 |   0.0 |  0.001477 |
    | 1997-31-08 |  0.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.0 | 0.0 |                   3.8 |                     13.618831 |      126.722078 |    0.0 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                         0.0 |                          0.0 |            16.432781 | 18.874047 |   4.044283 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 2.018218 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.084648 |   0.0 |   0.0 |   0.0 | -0.030898 |   0.05375 |  0.000622 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 367.849862 | 38.903278 | 0.096849 |  0.0 |  0.0 |  0.0 | -8.593516 |   0.0 |  0.000622 |

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_ic_acker_routing_daily:

:ref:`acre (routing) <lland_dd_acker_routing>`
----------------------------------------------

.. integration-test::

    >>> inlet.sequences.sim.series = 0.02
    >>> control.negq(False)
    >>> test("lland_knauf_ic_acker_routing_daily",
    ...      axis1=(inputs.nied, fluxes.qah), axis2=states.bowa)
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |   qz |   qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow |   nbes | sbes | snowintmax | snowintrate | nbesinz | sbesinz | wniedinz | actualalbedoinz | wadainz | schmpotinz | schminz | gefrpotinz | gefrinz | evsinz |   evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatinz | wlatsnow | wsensinz | wsenssnow | wsurfinz | wsurf | sff | fvg |   wada |      qdb | qib1 | qib2 | qbb |     qkap |     qdgz |    qdgz1 | qdgz2 | qigz1 | qigz2 |      qbgz |    qdga1 | qdga2 | qiga1 | qiga2 |      qbga |      qah |       qa | inzp | stinz | sinz | esnowinz | asinz | wats | waes | esnow | taus |       ebdn |      bowa |     sdg1 | sdg2 | sig1 | sig2 |       sbg | inlet |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.02 | 1.728 |                   6.3 |                     15.690785 |       190.25149 | 12.502 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                         0.0 |                          0.0 |            15.797694 | 11.733333 |      -26.2 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 11.802 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.7 | 1.720128 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 11.802 | 0.761343 |  0.0 |  0.0 | 0.0 |      0.0 | 0.761343 | 0.761343 |   0.0 |   0.0 |   0.0 |       0.0 | 0.196005 |   0.0 |   0.0 |   0.0 |       0.0 | 1.924005 | 0.022269 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |      373.3 | 81.320529 | 0.565338 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.022269 |
    | 1997-02-08 |  0.1 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.02 | 1.728 |                   1.7 |                     15.628821 |      127.660782 |  0.094 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                         0.0 |                          0.0 |            16.970501 | 18.443733 |     6.8624 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.094 | 1.697173 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.263667 |   0.0 |   0.0 |   0.0 |       0.0 | 1.991667 | 0.023052 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |   363.5376 | 79.623355 |  0.30167 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.023052 |
    | 1997-03-08 |  0.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.02 | 1.728 |                   4.5 |                     15.566134 |      164.401298 |    0.0 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                         0.0 |                          0.0 |            18.022179 | 15.632162 | -19.607027 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 2.329453 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.140696 |   0.0 |   0.0 |   0.0 |       0.0 | 1.868696 | 0.021628 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 380.244627 | 77.293902 | 0.160975 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.021628 |
    | 1997-04-08 |  0.1 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.02 | 1.728 |                  12.4 |                     15.502754 |      269.020502 |  0.094 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                         0.0 |                          0.0 |            15.840774 | 20.443786 |  15.862716 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.094 | 3.314337 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.075077 |   0.0 |   0.0 |   0.0 |       0.0 | 1.803077 | 0.020869 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 361.481911 | 73.979565 | 0.085898 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020869 |
    | 1997-05-08 |  0.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.02 | 1.728 |                  13.9 |                     15.438712 |      287.848667 |    0.0 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                         0.0 |                          0.0 |            15.977013 | 15.040124 | -20.759257 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 |  3.73228 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.040062 |   0.0 |   0.0 |   0.0 |       0.0 | 1.768062 | 0.020464 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 379.341169 | 70.247285 | 0.045836 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020464 |
    | 1997-06-08 |  0.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.02 | 1.728 |                  13.0 |                     15.374034 |      274.674263 |    0.0 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                         0.0 |                          0.0 |            18.131496 |  20.18359 |   1.701539 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 3.652387 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.021377 |   0.0 |   0.0 |   0.0 |       0.0 | 1.749377 | 0.020247 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 374.739629 | 66.594898 | 0.024459 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020247 |
    | 1997-07-08 |  0.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.02 | 1.728 |                  13.8 |                      15.30875 |      284.047892 |    0.0 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                         0.0 |                          0.0 |            18.084931 | 18.858347 | -11.649921 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 4.128687 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.011407 |   0.0 |   0.0 |   0.0 |       0.0 | 1.739407 | 0.020132 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  383.48955 |  62.46621 | 0.013051 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020132 |
    | 1997-08-08 |  0.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.02 | 1.728 |                  12.3 |                     15.242885 |      262.971263 |    0.0 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                         0.0 |                          0.0 |            19.830469 | 21.378324 |   2.869942 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 3.518399 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.006087 |   0.0 |   0.0 |   0.0 |       0.0 | 1.734087 |  0.02007 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 377.719608 | 58.947811 | 0.006964 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 |  0.02007 |
    | 1997-09-08 |  0.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.02 | 1.728 |                  13.1 |                     15.176467 |      272.256414 |    0.0 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                         0.0 |                          0.0 |            19.660875 |  19.71658 | -15.500518 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 4.091259 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.003248 |   0.0 |   0.0 |   0.0 |       0.0 | 1.731248 | 0.020038 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 390.320126 | 54.856552 | 0.003716 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020038 |
    | 1997-10-08 |  0.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.02 | 1.728 |                  12.8 |                     15.109521 |      267.019218 |    0.0 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                         0.0 |                          0.0 |            19.275317 |  23.34553 |   2.073177 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 4.345753 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.001733 |   0.0 |   0.0 |   0.0 |       0.0 | 1.729733 |  0.02002 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 385.346949 | 50.510799 | 0.001983 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 |  0.02002 |
    | 1997-11-08 |  0.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.02 | 1.728 |                  13.1 |                     15.042071 |      269.626547 |    0.0 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                         0.0 |                          0.0 |            19.420506 | 21.913255 |  -7.120473 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 4.547949 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000925 |   0.0 |   0.0 |   0.0 |       0.0 | 1.728925 | 0.020011 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 389.567421 | 45.962849 | 0.001058 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020011 |
    | 1997-12-08 |  0.1 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.02 | 1.728 |                  13.3 |                     14.974141 |      270.878291 |  0.094 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                         0.0 |                          0.0 |             18.79766 | 23.128751 |  -0.427496 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.094 | 4.554398 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000494 |   0.0 |   0.0 |   0.0 |       0.0 | 1.728494 | 0.020006 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 387.094917 | 41.408451 | 0.000565 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020006 |
    | 1997-13-08 |  0.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.02 | 1.728 |                  12.7 |                     14.905755 |      261.697164 |    0.0 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                         0.0 |                          0.0 |              19.6521 | 22.416669 |  -4.699983 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 4.137046 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000263 |   0.0 |   0.0 |   0.0 |       0.0 | 1.728263 | 0.020003 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |   388.8949 | 37.271405 | 0.000301 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020003 |
    | 1997-14-08 |  0.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.02 | 1.728 |                  10.2 |                     14.836934 |      227.962116 |    0.0 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                         0.0 |                          0.0 |            19.656892 | 22.935065 |   7.410388 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 3.428735 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000141 |   0.0 |   0.0 |   0.0 |       0.0 | 1.728141 | 0.020002 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 378.584513 |  33.84267 | 0.000161 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020002 |
    | 1997-15-08 |  2.7 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.02 | 1.728 |                   9.4 |                       14.7677 |      216.356321 |  2.538 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                         0.0 |                          0.0 |            20.028753 | 19.965673 |  -5.005962 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  1.838 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.7 | 2.505678 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  1.838 | 0.049893 |  0.0 |  0.0 | 0.0 |      0.0 | 0.049893 | 0.049893 |   0.0 |   0.0 |   0.0 |       0.0 |  0.01292 |   0.0 |   0.0 |   0.0 |       0.0 |  1.74092 |  0.02015 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 380.690475 | 33.125099 | 0.037134 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 |  0.02015 |
    | 1997-16-08 |  0.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.02 | 1.728 |                  10.3 |                     14.698075 |      226.637904 |    0.0 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                         0.0 |                          0.0 |            17.936694 |  20.57219 |   6.433141 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 2.836506 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.017319 |   0.0 |   0.0 |   0.0 |       0.0 | 1.745319 |   0.0202 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 371.357334 | 30.288593 | 0.019815 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 |   0.0202 |
    | 1997-17-08 |  0.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.02 | 1.728 |                  11.1 |                     14.628078 |      235.528503 |    0.0 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                         0.0 |                          0.0 |            18.906499 | 17.884246 | -18.694526 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 3.276924 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.03799 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  -0.03799 | 0.009242 |   0.0 |   0.0 |   0.0 | -0.000068 | 1.737173 | 0.020106 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 387.151861 | 27.049659 | 0.010574 |  0.0 |  0.0 |  0.0 | -0.037922 |  0.02 | 0.020106 |
    | 1997-18-08 |  0.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.02 | 1.728 |                  11.0 |                     14.557728 |      232.849428 |    0.0 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                         0.0 |                          0.0 |            18.089927 | 22.433069 |   4.998415 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 3.581415 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.239244 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -0.239244 | 0.004931 |   0.0 |   0.0 |   0.0 | -0.000565 | 1.732367 | 0.020051 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 379.253445 | 23.707488 | 0.005642 |  0.0 |  0.0 |  0.0 | -0.276602 |  0.02 | 0.020051 |
    | 1997-19-08 |  0.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.02 | 1.728 |                   8.5 |                     14.487046 |       199.71349 |    0.0 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                         0.0 |                          0.0 |            17.619414 | 20.158326 |  -5.650046 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 3.296835 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.446913 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -0.446913 | 0.002631 |   0.0 |   0.0 |   0.0 | -0.001792 | 1.728839 |  0.02001 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 382.003492 | 20.857567 | 0.003011 |  0.0 |  0.0 |  0.0 | -0.721723 |  0.02 |  0.02001 |
    | 1997-20-08 |  0.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.02 | 1.728 |                  11.3 |                     14.416048 |      233.771902 |    0.0 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                         0.0 |                          0.0 |            18.722517 | 20.950339 |  -8.697966 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 3.891994 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.623996 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -0.623996 | 0.001404 |   0.0 |   0.0 |   0.0 | -0.003704 |   1.7257 | 0.019973 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 387.801458 | 17.589568 | 0.001607 |  0.0 |  0.0 |  0.0 | -1.342015 |  0.02 | 0.019973 |
    | 1997-21-08 |  0.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.02 | 1.728 |                  12.4 |                     14.344752 |      246.121286 |    0.0 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                         0.0 |                          0.0 |             16.95292 | 22.620153 |    0.72092 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 4.183176 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.827056 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -0.827056 | 0.000749 |   0.0 |   0.0 |   0.0 |  -0.00629 | 1.722459 | 0.019936 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 384.180539 | 14.233449 | 0.000857 |  0.0 |  0.0 |  0.0 | -2.162781 |  0.02 | 0.019936 |
    | 1997-22-08 |  2.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.02 | 1.728 |                   0.1 |                     14.273175 |        90.58551 |   1.88 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                         0.0 |                          0.0 |            19.305027 | 21.577328 |  11.863971 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   1.18 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.7 | 1.075698 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |   1.18 | 0.013305 |  0.0 |  0.0 | 0.0 | 1.035592 | 0.013305 | 0.013305 |   0.0 |   0.0 |   0.0 | -1.035592 | 0.003825 |   0.0 |   0.0 |   0.0 | -0.009605 | 1.722221 | 0.019933 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 369.416568 | 15.360037 | 0.010337 |  0.0 |  0.0 |  0.0 | -3.188768 |  0.02 | 0.019933 |
    | 1997-23-08 |  0.3 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.02 | 1.728 |                   6.7 |                     14.201333 |      171.944993 |  0.282 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                         0.0 |                          0.0 |            19.945708 | 17.325305 | -17.848171 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.282 | 2.014865 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.96559 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  -0.96559 | 0.004821 |   0.0 |   0.0 |   0.0 | -0.013155 | 1.719666 | 0.019904 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 384.364739 | 14.310761 | 0.005516 |  0.0 |  0.0 |  0.0 | -4.141203 |  0.02 | 0.019904 |
    | 1997-24-08 |  0.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.02 | 1.728 |                  10.4 |                     14.129243 |      216.549813 |    0.0 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                         0.0 |                          0.0 |            23.000686 | 21.630378 | -14.817732 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 3.362119 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.030788 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -1.030788 | 0.002573 |   0.0 |   0.0 |   0.0 | -0.016684 | 1.713889 | 0.019837 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  396.28247 |  11.97943 | 0.002943 |  0.0 |  0.0 |  0.0 | -5.155306 |  0.02 | 0.019837 |
    | 1997-25-08 |  0.1 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.02 | 1.728 |                   6.5 |                     14.056918 |       166.91683 |  0.094 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                         0.0 |                          0.0 |            22.484088 | 25.062685 |   8.176109 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.094 | 2.719352 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.175647 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -1.175647 | 0.001373 |   0.0 |   0.0 |   0.0 | -0.020577 | 1.708796 | 0.019778 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 385.206362 | 10.435725 | 0.001571 |  0.0 |  0.0 |  0.0 | -6.310376 |  0.02 | 0.019778 |
    | 1997-26-08 | 10.5 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.02 | 1.728 |                   4.9 |                     13.984374 |      145.985611 |   9.87 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                         0.0 |                          0.0 |            21.103195 | 21.872765 |   4.636593 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   9.17 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.7 | 1.597373 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |   9.17 | 0.104578 |  0.0 |  0.0 | 0.0 | 1.271567 | 0.104578 | 0.104578 |   0.0 |   0.0 |   0.0 | -1.271567 | 0.027656 |   0.0 |   0.0 |   0.0 | -0.024887 | 1.730769 | 0.020032 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 377.669769 | 19.175341 | 0.078493 |  0.0 |  0.0 |  0.0 | -7.557056 |  0.02 | 0.020032 |
    | 1997-27-08 |  6.5 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.02 | 1.728 |                   6.6 |                     13.911624 |       165.54389 |   6.11 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                         0.0 |                          0.0 |            21.241626 | 19.702227 |   -7.18664 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   5.41 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.7 | 1.722258 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |   5.41 | 0.090745 |  0.0 |  0.0 | 0.0 | 0.728523 | 0.090745 | 0.090745 |   0.0 |   0.0 |   0.0 | -0.728523 |  0.05997 |   0.0 |   0.0 |   0.0 |  -0.02838 |  1.75959 | 0.020366 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 381.956408 | 23.500861 | 0.109268 |  0.0 |  0.0 |  0.0 | -8.257199 |  0.02 | 0.020366 |
    | 1997-28-08 | 15.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.02 | 1.728 |                   0.3 |                     13.838682 |       87.693346 |   14.1 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                         0.0 |                          0.0 |            20.510475 | 20.936779 |   4.420674 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   13.4 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.7 | 1.071953 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |   13.4 | 0.312945 |  0.0 |  0.0 | 0.0 | 0.459752 | 0.312945 | 0.312945 |   0.0 |   0.0 |   0.0 | -0.459752 | 0.131528 |   0.0 |   0.0 |   0.0 | -0.030407 | 1.829121 |  0.02117 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 374.635735 | 35.975715 | 0.290685 |  0.0 |  0.0 |  0.0 | -8.686544 |  0.02 |  0.02117 |
    | 1997-29-08 |  9.2 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.02 | 1.728 |                   0.1 |                     13.765561 |       84.356885 |  8.648 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                         0.0 |                          0.0 |            16.420805 | 18.828425 |   16.37055 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  7.948 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.7 | 0.717164 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  7.948 | 0.249164 |  0.0 |  0.0 | 0.0 |      0.0 | 0.249164 | 0.249164 |   0.0 |   0.0 |   0.0 |       0.0 | 0.199719 |   0.0 |   0.0 |   0.0 | -0.031121 | 1.896598 | 0.021951 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 355.365185 | 42.957387 |  0.34013 |  0.0 |  0.0 |  0.0 | -8.655423 |  0.02 | 0.021951 |
    | 1997-30-08 |  0.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.02 | 1.728 |                   5.0 |                     13.692273 |      142.309202 |    0.0 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                         0.0 |                          0.0 |            15.195625 | 13.278507 |  -22.32896 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 2.035891 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.158633 |   0.0 |   0.0 |   0.0 | -0.031009 | 1.855624 | 0.021477 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 374.794145 | 40.921496 | 0.181497 |  0.0 |  0.0 |  0.0 | -8.624414 |  0.02 | 0.021477 |
    | 1997-31-08 |  0.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.02 | 1.728 |                   3.8 |                     13.618831 |      126.722078 |    0.0 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                         0.0 |                          0.0 |            16.432781 | 18.874047 |   4.044283 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 2.018218 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.084648 |   0.0 |   0.0 |   0.0 | -0.030898 |  1.78175 | 0.020622 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 367.849862 | 38.903278 | 0.096849 |  0.0 |  0.0 |  0.0 | -8.593516 |  0.02 | 0.020622 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_ic_acker_heavy_rain_daily:

acre (heavy rain)
-----------------

.. integration-test::

    >>> lnk(ACKER)
    >>> inlet.sequences.sim.series = 0.0
    >>> nied = inputs.nied.series.copy()
    >>> inputs.nied.series = 20.0
    >>> conditions_acker_heavy_rain = test(
    ...     "lland_knauf_ic_acker_heavy_rain_daily",
    ...     axis1=(inputs.nied, fluxes.qah), axis2=states.bowa,
    ...     get_conditions="1997-08-03")
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation | nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow | nbes | sbes | snowintmax | snowintrate | nbesinz | sbesinz | wniedinz | actualalbedoinz | wadainz | schmpotinz | schminz | gefrpotinz | gefrinz | evsinz | evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatinz | wlatsnow | wsensinz | wsenssnow | wsurfinz | wsurf | sff | fvg | wada |       qdb |     qib1 |     qib2 |      qbb | qkap |      qdgz |     qdgz1 |    qdgz2 |    qigz1 |    qigz2 |     qbgz |     qdga1 |    qdga2 |    qiga1 |    qiga2 |     qbga |       qah |       qa | inzp | stinz | sinz | esnowinz | asinz | wats | waes | esnow | taus |       ebdn |       bowa |      sdg1 |     sdg2 |     sig1 |     sig2 |      sbg | inlet |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 20.0 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.0 | 0.0 |                   6.3 |                     15.690785 |       190.25149 | 18.8 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                         0.0 |                          0.0 |            15.797694 | 11.733333 |      -26.2 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 1.720128 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  1.217805 |      0.0 |      0.0 |      0.0 |  0.0 |  1.217805 |  1.217805 |      0.0 |      0.0 |      0.0 |      0.0 |   0.31352 |      0.0 |      0.0 |      0.0 |      0.0 |   0.31352 | 0.003629 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |      373.3 |  87.162066 |  0.904285 |      0.0 |      0.0 |      0.0 |      0.0 |   0.0 | 0.003629 |
    | 1997-02-08 | 20.0 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.0 | 0.0 |                   1.7 |                     15.628821 |      127.660782 | 18.8 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                         0.0 |                          0.0 |            16.970501 | 18.443733 |     6.8624 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 1.294457 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |   1.48087 |      0.0 |      0.0 |      0.0 |  0.0 |   1.48087 |   1.48087 |      0.0 |      0.0 |      0.0 |      0.0 |  0.802994 |      0.0 |      0.0 |      0.0 |      0.0 |  0.802994 | 0.009294 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |   363.5376 | 102.486739 |  1.582162 |      0.0 |      0.0 |      0.0 |      0.0 |   0.0 | 0.009294 |
    | 1997-03-08 | 20.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.0 | 0.0 |                   4.5 |                     15.566134 |      164.401298 | 18.8 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                         0.0 |                          0.0 |            18.022179 | 15.632162 | -19.607027 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 1.922357 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  1.761525 |      0.0 |      0.0 |      0.0 |  0.0 |  1.761525 |  1.761525 |      0.0 |      0.0 |      0.0 |      0.0 |  1.191401 |      0.0 |      0.0 |      0.0 |      0.0 |  1.191401 | 0.013789 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 380.244627 | 116.902858 |  2.152285 |      0.0 |      0.0 |      0.0 |      0.0 |   0.0 | 0.013789 |
    | 1997-04-08 | 20.0 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.0 | 0.0 |                  12.4 |                     15.502754 |      269.020502 | 18.8 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                         0.0 |                          0.0 |            15.840774 | 20.443786 |  15.862716 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.397171 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  2.040918 |      0.0 |      0.0 |      0.0 |  0.0 |  2.040918 |  2.040918 |      0.0 |      0.0 |      0.0 |      0.0 |   1.52923 |      0.0 |      0.0 |      0.0 |      0.0 |   1.52923 | 0.017699 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 361.481911 | 129.564768 |  2.663974 |      0.0 |      0.0 |      0.0 |      0.0 |   0.0 | 0.017699 |
    | 1997-05-08 | 20.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.0 | 0.0 |                  13.9 |                     15.438712 |      287.848667 | 18.8 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                         0.0 |                          0.0 |            15.977013 | 15.040124 | -20.759257 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.917105 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  2.300286 | 0.101438 |      0.0 | 0.010165 |  0.0 |  2.300286 |  2.300286 |      0.0 | 0.101438 |      0.0 | 0.010165 |  1.834649 |      0.0 | 0.014384 |      0.0 | 0.000018 |  1.849051 | 0.021401 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 379.341169 | 141.335775 |  3.129611 |      0.0 | 0.087054 |      0.0 | 0.010147 |   0.0 | 0.021401 |
    | 1997-06-08 | 20.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.0 | 0.0 |                  13.0 |                     15.374034 |      274.674263 | 18.8 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                         0.0 |                          0.0 |            18.131496 |  20.18359 |   1.701539 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.913569 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  2.554694 | 0.110654 |      0.0 | 0.021936 |  0.0 |  2.554694 |  2.554694 |      0.0 | 0.110654 |      0.0 | 0.021936 |  2.117313 |      0.0 | 0.039153 |      0.0 | 0.000076 |  2.156542 |  0.02496 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 374.739629 | 152.834923 |  3.566992 |      0.0 | 0.158555 |      0.0 | 0.032007 |   0.0 |  0.02496 |
    | 1997-07-08 | 20.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.0 | 0.0 |                  13.8 |                      15.30875 |      284.047892 | 18.8 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                         0.0 |                          0.0 |            18.084931 | 18.858347 | -11.649921 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 4.551444 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  2.817235 | 0.119656 |      0.0 | 0.033435 |  0.0 |  2.817235 |  2.817235 |      0.0 | 0.119656 |      0.0 | 0.033435 |  2.388893 |      0.0 |   0.0597 |      0.0 | 0.000175 |  2.448767 | 0.028342 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  383.48955 | 163.413153 |  3.995333 |      0.0 | 0.218512 |      0.0 | 0.065267 |   0.0 | 0.028342 |
    | 1997-08-08 | 20.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.0 | 0.0 |                  12.3 |                     15.242885 |      262.971263 | 18.8 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                         0.0 |                          0.0 |            19.830469 | 21.378324 |   2.869942 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 |  3.76789 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.072622 | 0.127938 |      0.0 | 0.044013 |  0.0 |  3.072622 |  3.072622 |      0.0 | 0.127938 |      0.0 | 0.044013 |  2.654416 |      0.0 | 0.077033 |      0.0 | 0.000313 |  2.731761 | 0.031618 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 377.719608 | 174.500689 |   4.41354 |      0.0 | 0.269417 |      0.0 | 0.108967 |   0.0 | 0.031618 |
    | 1997-09-08 | 20.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.176467 |      272.256414 | 18.8 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                         0.0 |                          0.0 |            19.660875 |  19.71658 | -15.500518 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 4.120968 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.356609 | 0.136619 |      0.0 | 0.055101 |  0.0 |  3.356609 |  3.356609 |      0.0 | 0.136619 |      0.0 | 0.055101 |  2.922574 |      0.0 | 0.091984 |      0.0 | 0.000489 |  3.015047 | 0.034896 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 390.320126 | 184.931393 |  4.847575 |      0.0 | 0.314052 |      0.0 | 0.163579 |   0.0 | 0.034896 |
    | 1997-10-08 | 20.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.0 | 0.0 |                  12.8 |                     15.109521 |      267.019218 | 18.8 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                         0.0 |                          0.0 |            19.275317 |  23.34553 |   2.073177 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 4.450324 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.641281 | 0.144785 |      0.0 | 0.065531 |  0.0 |  3.641281 |  3.641281 |      0.0 | 0.144785 |      0.0 | 0.065531 |  3.198291 |      0.0 | 0.105171 |      0.0 | 0.000704 |  3.304166 | 0.038243 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 385.346949 | 194.729472 |  5.290565 |      0.0 | 0.353666 |      0.0 | 0.228407 |   0.0 | 0.038243 |
    | 1997-11-08 | 20.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.042071 |      269.626547 | 18.8 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                         0.0 |                          0.0 |            19.420506 | 21.913255 |  -7.120473 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 4.992996 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.926597 | 0.152456 |      0.0 | 0.075329 |  0.0 |  3.926597 |  3.926597 |      0.0 | 0.152456 |      0.0 | 0.075329 |   3.47835 |      0.0 | 0.116936 |      0.0 | 0.000953 |  3.596239 | 0.041623 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 389.567421 | 203.682094 |  5.738812 |      0.0 | 0.389186 |      0.0 | 0.302783 |   0.0 | 0.041623 |
    | 1997-12-08 | 20.0 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.0 | 0.0 |                  13.3 |                     14.974141 |      270.878291 | 18.8 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                         0.0 |                          0.0 |             18.79766 | 23.128751 |  -0.427496 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 4.979765 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  4.204896 | 0.159465 |  0.00956 | 0.084282 |  0.0 |  4.204896 |  4.204896 |      0.0 | 0.159465 |  0.00956 | 0.084282 |  3.759055 |      0.0 | 0.127503 | 0.001356 | 0.001236 |  3.889149 | 0.045013 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 387.094917 | 212.344126 |  6.184653 |      0.0 | 0.421149 | 0.008204 | 0.385829 |   0.0 | 0.045013 |
    | 1997-13-08 | 20.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.0 | 0.0 |                  12.7 |                     14.905755 |      261.697164 | 18.8 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                         0.0 |                          0.0 |              19.6521 | 22.416669 |  -4.699983 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 4.198392 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  4.492791 | 0.166247 | 0.045997 | 0.092944 |  0.0 |  4.492791 |  4.492791 |      0.0 | 0.166247 | 0.045997 | 0.092944 |  4.041108 |      0.0 | 0.137079 | 0.008733 | 0.001549 |  4.188469 | 0.048478 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |   388.8949 | 221.447755 |  6.636336 |      0.0 | 0.450317 | 0.045468 | 0.477225 |   0.0 | 0.048478 |
    | 1997-14-08 | 20.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.0 | 0.0 |                  10.2 |                     14.836934 |      227.962116 | 18.8 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                         0.0 |                          0.0 |            19.656892 | 22.935065 |   7.410388 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.701991 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  4.818593 | 0.173374 | 0.100359 | 0.102048 |  0.0 |  4.818593 |  4.818593 |      0.0 | 0.173374 | 0.100359 | 0.102048 |  4.335644 |      0.0 | 0.145951 | 0.026485 | 0.001893 |  4.509972 | 0.052199 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 378.584513 |  230.65139 |  7.119285 |      0.0 | 0.477741 | 0.119342 |  0.57738 |   0.0 | 0.052199 |
    | 1997-15-08 | 20.0 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.0 | 0.0 |                   9.4 |                       14.7677 |      216.356321 | 18.8 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                         0.0 |                          0.0 |            20.028753 | 19.965673 |  -5.005962 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.155199 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  5.176976 |  0.18058 | 0.168029 | 0.111251 |  0.0 |  5.176976 |  5.176976 |      0.0 |  0.18058 | 0.168029 | 0.111251 |  4.653151 |      0.0 | 0.154363 |  0.05599 | 0.002268 |  4.865773 | 0.056317 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 380.690475 | 239.959354 |  7.643111 |      0.0 | 0.503957 |  0.23138 | 0.686363 |   0.0 | 0.056317 |
    | 1997-16-08 | 20.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.0 | 0.0 |                  10.3 |                     14.698075 |      226.637904 | 18.8 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                         0.0 |                          0.0 |            17.936694 |  20.57219 |   6.433141 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 2.708353 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  5.575883 | 0.187867 | 0.247357 | 0.120559 |  0.0 |  5.575883 |  5.575883 |      0.0 | 0.187867 | 0.247357 | 0.120559 |  5.000154 |      0.0 | 0.162462 | 0.097435 | 0.002675 |  5.262727 | 0.060911 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 371.357334 | 249.219334 |  8.218839 |      0.0 | 0.529362 | 0.381303 | 0.804247 |   0.0 | 0.060911 |
    | 1997-17-08 | 20.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.0 | 0.0 |                  11.1 |                     14.628078 |      235.528503 | 18.8 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                         0.0 |                          0.0 |            18.906499 | 17.884246 | -18.694526 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.378596 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  6.018762 | 0.195117 | 0.335816 | 0.129819 |  0.0 |  6.018762 |  6.018762 |      0.0 | 0.195117 | 0.335816 | 0.129819 |  5.382685 |      0.0 | 0.170337 | 0.150384 | 0.003114 |  5.706521 | 0.066048 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 387.151861 | 257.261224 |  8.854915 |      0.0 | 0.554142 | 0.566735 | 0.930953 |   0.0 | 0.066048 |
    | 1997-18-08 | 20.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.0 | 0.0 |                  11.0 |                     14.557728 |      232.849428 | 18.8 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                         0.0 |                          0.0 |            18.089927 | 22.433069 |   4.998415 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.779721 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  6.451575 | 0.201413 | 0.419628 | 0.137861 |  0.0 |  6.451575 |  6.451575 |      0.0 | 0.201413 | 0.419628 | 0.137861 |  5.790771 |      0.0 | 0.177909 | 0.212245 | 0.003582 |  6.184507 |  0.07158 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 379.253445 | 264.371025 |   9.51572 |      0.0 | 0.577646 | 0.774118 | 1.065232 |   0.0 |  0.07158 |
    | 1997-19-08 | 20.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.0 | 0.0 |                   8.5 |                     14.487046 |       199.71349 | 18.8 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                         0.0 |                          0.0 |            17.619414 | 20.158326 |  -5.650046 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.628366 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  6.883257 | 0.206979 | 0.498739 | 0.144971 |  0.0 |  6.883257 |  6.883257 |      0.0 | 0.206979 | 0.498739 | 0.144971 |  6.210098 |      0.0 | 0.185033 | 0.279355 | 0.004076 |  6.678562 | 0.077298 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 382.003492 | 271.108713 | 10.188879 |      0.0 | 0.599593 | 0.993502 | 1.206126 |   0.0 | 0.077298 |
    | 1997-20-08 | 20.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.0 | 0.0 |                  11.3 |                     14.416048 |      233.771902 | 18.8 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                         0.0 |                          0.0 |            18.722517 | 20.950339 |  -8.697966 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 4.624935 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  7.348961 | 0.212254 | 0.577799 | 0.151709 |  0.0 |  7.348961 |  7.348961 |      0.0 | 0.212254 | 0.577799 | 0.151709 |  6.643946 |      0.0 | 0.191696 | 0.349693 | 0.004593 |  7.189927 | 0.083217 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 387.801458 | 276.293054 | 10.893894 |      0.0 | 0.620152 | 1.221608 | 1.353242 |   0.0 | 0.083217 |
    | 1997-21-08 | 20.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.0 | 0.0 |                  12.4 |                     14.344752 |      246.121286 | 18.8 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                         0.0 |                          0.0 |             16.95292 | 22.620153 |    0.72092 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 4.756903 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  7.757566 | 0.216313 | 0.641218 | 0.156893 |  0.0 |  7.757566 |  7.757566 |      0.0 | 0.216313 | 0.641218 | 0.156893 |  7.077951 |      0.0 | 0.197812 | 0.420163 | 0.005129 |  7.701056 | 0.089133 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 384.180539 | 280.864161 | 11.573509 |      0.0 | 0.638653 | 1.442663 | 1.505006 |   0.0 | 0.089133 |
    | 1997-22-08 | 20.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.0 | 0.0 |                   0.1 |                     14.273175 |        90.58551 | 18.8 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                         0.0 |                          0.0 |            19.305027 | 21.577328 |  11.863971 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 1.361124 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  8.166722 | 0.219892 | 0.698934 | 0.161464 |  0.0 |  8.166722 |  8.166722 |      0.0 | 0.219892 | 0.698934 | 0.161464 |  7.500252 |      0.0 | 0.203306 | 0.487924 | 0.005681 |  8.197163 | 0.094875 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 369.416568 | 288.356024 |  12.23998 |      0.0 |  0.65524 | 1.653673 | 1.660789 |   0.0 | 0.094875 |
    | 1997-23-08 | 20.0 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.0 | 0.0 |                   6.7 |                     14.201333 |      171.944993 | 18.8 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                         0.0 |                          0.0 |            19.945708 | 17.325305 | -17.848171 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 2.097454 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  8.984347 | 0.225758 | 0.797042 | 0.168956 |  0.0 |  8.984347 |  8.984347 |      0.0 | 0.225758 | 0.797042 | 0.168956 |  8.021581 |      0.0 | 0.208608 | 0.558706 | 0.006253 |  8.795147 | 0.101796 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 384.364739 | 294.182468 | 13.202746 |      0.0 |  0.67239 | 1.892009 | 1.823492 |   0.0 | 0.101796 |
    | 1997-24-08 | 20.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.0 | 0.0 |                  10.4 |                     14.129243 |      216.549813 | 18.8 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                         0.0 |                          0.0 |            23.000686 | 21.630378 | -14.817732 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.649811 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  9.835059 | 0.230319 | 0.876255 | 0.174782 |  0.0 |  9.835059 |  9.835059 |      0.0 | 0.230319 | 0.876255 | 0.174782 |  8.689617 |      0.0 | 0.213877 | 0.634173 | 0.006846 |  9.544513 | 0.110469 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  396.28247 | 297.516241 | 14.348187 |      0.0 | 0.688832 | 2.134091 | 1.991428 |   0.0 | 0.110469 |
    | 1997-25-08 | 20.0 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.0 | 0.0 |                   6.5 |                     14.056918 |       166.91683 | 18.8 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                         0.0 |                          0.0 |            22.484088 | 25.062685 |   8.176109 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 2.895865 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 10.483053 | 0.232929 | 0.922693 | 0.178116 |  0.0 | 10.483053 | 10.483053 |      0.0 | 0.232929 | 0.922693 | 0.178116 |  9.390662 |      0.0 | 0.218678 | 0.706002 | 0.007454 | 10.322796 | 0.119477 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 385.206362 | 300.903585 | 15.440578 |      0.0 | 0.703083 | 2.350782 |  2.16209 |   0.0 | 0.119477 |
    | 1997-26-08 | 20.0 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.0 | 0.0 |                   4.9 |                     13.984374 |      145.985611 | 18.8 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                         0.0 |                          0.0 |            21.103195 | 21.872765 |   4.636593 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 2.071449 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 11.394349 | 0.235581 | 0.970688 | 0.181504 |  0.0 | 11.394349 | 11.394349 |      0.0 | 0.235581 | 0.970688 | 0.181504 | 10.134751 |      0.0 | 0.222895 | 0.771208 | 0.008071 | 11.136926 |   0.1289 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 377.669769 | 304.150015 | 16.700175 |      0.0 | 0.715769 | 2.550261 | 2.335523 |   0.0 |   0.1289 |
    | 1997-27-08 | 20.0 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.0 | 0.0 |                   6.6 |                     13.911624 |       165.54389 | 18.8 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                         0.0 |                          0.0 |            21.241626 | 19.702227 |   -7.18664 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 2.108859 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 13.250015 | 0.238123 | 1.017441 |  0.18475 |  0.0 | 13.250015 |      12.0 | 1.250015 | 0.238123 | 1.017441 |  0.18475 | 10.878137 | 0.792807 | 0.226675 |   0.8316 | 0.008698 | 12.737917 |  0.14743 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 381.956408 | 305.450827 | 17.822039 | 0.457208 | 0.727217 | 2.736102 | 2.511574 |   0.0 |  0.14743 |
    | 1997-28-08 | 20.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.0 | 0.0 |                   0.3 |                     13.838682 |       87.693346 | 18.8 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                         0.0 |                          0.0 |            20.510475 | 20.936779 |   4.420674 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 |  1.33161 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 14.550827 | 0.239141 |  1.03638 | 0.186051 |  0.0 | 14.550827 |      12.0 | 2.550827 | 0.239141 |  1.03638 | 0.186051 | 11.401361 | 2.037971 | 0.229904 | 0.884372 | 0.009332 |  14.56294 | 0.168553 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 374.635735 | 306.206818 | 18.420678 | 0.970064 | 0.736454 | 2.888109 | 2.688293 |   0.0 | 0.168553 |
    | 1997-29-08 | 20.0 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.0 | 0.0 |                   0.1 |                     13.765561 |       84.356885 | 18.8 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                         0.0 |                          0.0 |            16.420805 | 18.828425 |   16.37055 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 0.881773 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 15.306818 | 0.239733 | 1.047439 | 0.186807 |  0.0 | 15.306818 |      12.0 | 3.306818 | 0.239733 | 1.047439 | 0.186807 |  11.68056 | 2.988724 | 0.232478 | 0.926908 | 0.009966 | 15.838636 | 0.183318 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 355.365185 | 306.644248 | 18.740118 | 1.288158 |  0.74371 |  3.00864 | 2.865134 |   0.0 | 0.183318 |
    | 1997-30-08 | 20.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.0 | 0.0 |                   5.0 |                     13.692273 |      142.309202 | 18.8 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                         0.0 |                          0.0 |            15.195625 | 13.278507 |  -22.32896 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 1.913351 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 15.744248 | 0.240076 | 1.053857 | 0.187244 |  0.0 | 15.744248 |      12.0 | 3.744248 | 0.240076 | 1.053857 | 0.187244 | 11.829543 | 3.558462 | 0.234482 | 0.960303 |   0.0106 | 16.593391 | 0.192053 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 374.794145 | 305.605473 | 18.910575 | 1.473943 | 0.749303 | 3.102193 | 3.041778 |   0.0 | 0.192053 |
    | 1997-31-08 | 20.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.0 | 0.0 |                   3.8 |                     13.618831 |      126.722078 | 18.8 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                         0.0 |                          0.0 |            16.432781 | 18.874047 |   4.044283 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 1.882123 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 14.705473 | 0.239262 | 1.038639 | 0.186205 |  0.0 | 14.705473 |      12.0 | 2.705473 | 0.239262 | 1.038639 | 0.186205 | 11.909042 | 3.070354 | 0.235874 | 0.983359 | 0.011231 | 16.209861 | 0.187614 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 367.849862 | 305.653771 | 19.001532 | 1.109062 | 0.752692 | 3.157473 | 3.216752 |   0.0 | 0.187614 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_ic_acker_garto_daily:

acre (GARTO)
------------

>>> from hydpy import pub
>>> with model.add_soilmodel_v1("ga_garto_submodel1"):
...     nmbbins(4)
...     with pub.options.parameterstep("1m"):
...         dt(1.0)
...     sealed(False)
...     soildepth(309.0 / 0.434)
...     residualmoisture(0.027)
...     saturationmoisture(0.434)
...     saturatedconductivity(13.2)
...     poresizedistribution(0.252)
...     airentrypotential(111.5)
...     states.moisture = 72.0 / 309.0 * 0.434
...     states.frontdepth = 0.0
...     states.moisturechange = 0.0

.. integration-test::

    >>> test("lland_knauf_ic_acker_garto_daily",
    ...      axis1=(inputs.nied, fluxes.qah), axis2=states.bowa)
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation | nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow | nbes | sbes | snowintmax | snowintrate | nbesinz | sbesinz | wniedinz | actualalbedoinz | wadainz | schmpotinz | schminz | gefrpotinz | gefrinz | evsinz | evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatinz | wlatsnow | wsensinz | wsenssnow | wsurfinz | wsurf | sff | fvg | wada |      qdb |     qib1 |     qib2 |       qbb | qkap |     qdgz |    qdgz1 | qdgz2 |    qigz1 |    qigz2 |      qbgz |    qdga1 | qdga2 |    qiga1 |    qiga2 |     qbga |      qah |       qa | inzp | stinz | sinz | esnowinz | asinz | wats | waes | esnow | taus |       ebdn |       bowa |     sdg1 | sdg2 |     sig1 |     sig2 |        sbg | inlet |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 20.0 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.0 | 0.0 |                   6.3 |                     15.690785 |       190.25149 | 18.8 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                         0.0 |                          0.0 |            15.797694 | 11.733333 |      -26.2 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 1.720128 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 1.217805 |      0.0 |      0.0 |  0.000003 |  0.0 | 1.217805 | 1.217805 |   0.0 |      0.0 |      0.0 |  0.000003 |  0.31352 |   0.0 |      0.0 |      0.0 |      0.0 |  0.31352 | 0.003629 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |      373.3 |  87.162064 | 0.904285 |  0.0 |      0.0 |      0.0 |   0.000003 |   0.0 | 0.003629 |
    | 1997-02-08 | 20.0 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.0 | 0.0 |                   1.7 |                     15.628821 |      127.660782 | 18.8 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                         0.0 |                          0.0 |            16.970501 | 18.443733 |     6.8624 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 1.294457 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  1.48087 |      0.0 |      0.0 |  0.000003 |  0.0 |  1.48087 |  1.48087 |   0.0 |      0.0 |      0.0 |  0.000003 | 0.802994 |   0.0 |      0.0 |      0.0 |      0.0 | 0.802994 | 0.009294 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |   363.5376 | 102.486734 | 1.582161 |  0.0 |      0.0 |      0.0 |   0.000005 |   0.0 | 0.009294 |
    | 1997-03-08 | 20.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.0 | 0.0 |                   4.5 |                     15.566134 |      164.401298 | 18.8 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                         0.0 |                          0.0 |            18.022179 | 15.632162 | -19.607027 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 1.922357 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 1.761525 |      0.0 |      0.0 |  0.000003 |  0.0 | 1.761525 | 1.761525 |   0.0 |      0.0 |      0.0 |  0.000003 | 1.191401 |   0.0 |      0.0 |      0.0 |      0.0 | 1.191401 | 0.013789 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 380.244627 |  116.90285 | 2.152285 |  0.0 |      0.0 |      0.0 |   0.000008 |   0.0 | 0.013789 |
    | 1997-04-08 | 20.0 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.0 | 0.0 |                  12.4 |                     15.502754 |      269.020502 | 18.8 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                         0.0 |                          0.0 |            15.840774 | 20.443786 |  15.862716 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.397171 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 2.040918 |      0.0 |      0.0 |  0.000003 |  0.0 | 2.040918 | 2.040918 |   0.0 |      0.0 |      0.0 |  0.000003 | 1.529229 |   0.0 |      0.0 |      0.0 |      0.0 | 1.529229 | 0.017699 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 361.481911 | 129.564759 | 2.663974 |  0.0 |      0.0 |      0.0 |    0.00001 |   0.0 | 0.017699 |
    | 1997-05-08 | 20.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.0 | 0.0 |                  13.9 |                     15.438712 |      287.848667 | 18.8 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                         0.0 |                          0.0 |            15.977013 | 15.040124 | -20.759257 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.917105 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 2.300286 | 0.101438 |      0.0 |  0.010167 |  0.0 | 2.300286 | 2.300286 |   0.0 | 0.101438 |      0.0 |  0.010167 | 1.834649 |   0.0 | 0.014384 |      0.0 | 0.000018 | 1.849051 | 0.021401 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 379.341169 | 141.335762 | 3.129611 |  0.0 | 0.087054 |      0.0 |   0.010159 |   0.0 | 0.021401 |
    | 1997-06-08 | 20.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.0 | 0.0 |                  13.0 |                     15.374034 |      274.674263 | 18.8 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                         0.0 |                          0.0 |            18.131496 |  20.18359 |   1.701539 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.913569 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 2.554693 | 0.110654 |      0.0 |  0.021938 |  0.0 | 2.554693 | 2.554693 |   0.0 | 0.110654 |      0.0 |  0.021938 | 2.117313 |   0.0 | 0.039153 |      0.0 | 0.000076 | 2.156542 |  0.02496 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 374.739629 | 152.834908 | 3.566991 |  0.0 | 0.158555 |      0.0 |   0.032022 |   0.0 |  0.02496 |
    | 1997-07-08 | 20.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.0 | 0.0 |                  13.8 |                      15.30875 |      284.047892 | 18.8 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                         0.0 |                          0.0 |            18.084931 | 18.858347 | -11.649921 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 4.551444 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 2.817235 | 0.119656 |      0.0 |  0.033437 |  0.0 | 2.817235 | 2.817235 |   0.0 | 0.119656 |      0.0 |  0.033437 | 2.388893 |   0.0 |   0.0597 |      0.0 | 0.000175 | 2.448767 | 0.028342 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  383.48955 | 163.413136 | 3.995333 |  0.0 | 0.218512 |      0.0 |   0.065285 |   0.0 | 0.028342 |
    | 1997-08-08 | 20.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.0 | 0.0 |                  12.3 |                     15.242885 |      262.971263 | 18.8 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                         0.0 |                          0.0 |            19.830469 | 21.378324 |   2.869942 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 |  3.76789 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 3.072622 | 0.127938 |      0.0 |  0.044016 |  0.0 | 3.072622 | 3.072622 |   0.0 | 0.127938 |      0.0 |  0.044016 | 2.654415 |   0.0 | 0.077033 |      0.0 | 0.000313 | 2.731761 | 0.031618 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 377.719608 |  174.50067 |  4.41354 |  0.0 | 0.269417 |      0.0 |   0.108988 |   0.0 | 0.031618 |
    | 1997-09-08 | 20.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.176467 |      272.256414 | 18.8 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                         0.0 |                          0.0 |            19.660875 |  19.71658 | -15.500518 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 4.120968 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 3.356608 | 0.136619 |      0.0 |  0.055103 |  0.0 | 3.356608 | 3.356608 |   0.0 | 0.136619 |      0.0 |  0.055103 | 2.922574 |   0.0 | 0.091984 |      0.0 | 0.000489 | 3.015046 | 0.034896 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 390.320126 | 184.931372 | 4.847574 |  0.0 | 0.314052 |      0.0 |   0.163602 |   0.0 | 0.034896 |
    | 1997-10-08 | 20.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.0 | 0.0 |                  12.8 |                     15.109521 |      267.019218 | 18.8 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                         0.0 |                          0.0 |            19.275317 |  23.34553 |   2.073177 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 4.450324 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.64128 | 0.144785 |      0.0 |  0.065534 |  0.0 |  3.64128 |  3.64128 |   0.0 | 0.144785 |      0.0 |  0.065534 |  3.19829 |   0.0 | 0.105171 |      0.0 | 0.000704 | 3.304165 | 0.038243 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 385.346949 | 194.729449 | 5.290564 |  0.0 | 0.353666 |      0.0 |   0.228432 |   0.0 | 0.038243 |
    | 1997-11-08 | 20.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.042071 |      269.626547 | 18.8 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                         0.0 |                          0.0 |            19.420506 | 21.913255 |  -7.120473 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 4.992996 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 3.926596 | 0.152456 |      0.0 |  0.075332 |  0.0 | 3.926596 | 3.926596 |   0.0 | 0.152456 |      0.0 |  0.075332 |  3.47835 |   0.0 | 0.116936 |      0.0 | 0.000953 | 3.596238 | 0.041623 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 389.567421 | 203.682069 | 5.738811 |  0.0 | 0.389186 |      0.0 |   0.302811 |   0.0 | 0.041623 |
    | 1997-12-08 | 20.0 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.0 | 0.0 |                  13.3 |                     14.974141 |      270.878291 | 18.8 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                         0.0 |                          0.0 |             18.79766 | 23.128751 |  -0.427496 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 4.979765 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.204895 | 0.159465 |  0.00956 |  0.084285 |  0.0 | 4.204895 | 4.204895 |   0.0 | 0.159465 |  0.00956 |  0.084285 | 3.759054 |   0.0 | 0.127503 | 0.001356 | 0.001236 | 3.889148 | 0.045013 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 387.094917 | 212.344099 | 6.184652 |  0.0 | 0.421149 | 0.008204 |   0.385859 |   0.0 | 0.045013 |
    | 1997-13-08 | 20.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.0 | 0.0 |                  12.7 |                     14.905755 |      261.697164 | 18.8 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                         0.0 |                          0.0 |              19.6521 | 22.416669 |  -4.699983 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 4.198392 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  4.49279 | 0.166247 | 0.045997 |  1.325619 |  0.0 |  4.49279 |  4.49279 |   0.0 | 0.166247 | 0.045997 |  1.325619 | 4.041107 |   0.0 | 0.137079 | 0.008733 | 0.003758 | 4.190677 | 0.048503 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |   388.8949 | 220.215055 | 6.636335 |  0.0 | 0.450317 | 0.045468 |    1.70772 |   0.0 | 0.048503 |
    | 1997-14-08 | 20.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.0 | 0.0 |                  10.2 |                     14.836934 |      227.962116 | 18.8 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                         0.0 |                          0.0 |            19.656892 | 22.935065 |   7.410388 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.701991 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.772928 | 0.172409 | 0.092206 | 10.848312 |  0.0 | 4.772928 | 4.772928 |   0.0 | 0.172409 | 0.092206 | 10.848312 | 4.323887 |   0.0 | 0.145814 | 0.025329 | 0.025563 | 4.520593 | 0.052322 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 378.584513 | 218.727208 | 7.085376 |  0.0 | 0.476912 | 0.112345 |  12.530469 |   0.0 | 0.052322 |
    | 1997-15-08 | 20.0 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.0 | 0.0 |                   9.4 |                       14.7677 |      216.356321 | 18.8 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                         0.0 |                          0.0 |            20.028753 | 19.965673 |  -5.005962 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.155199 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.718485 | 0.171244 | 0.082679 |  9.337522 |  0.0 | 4.718485 | 4.718485 |   0.0 | 0.171244 | 0.082679 |  9.337522 | 4.519299 |   0.0 | 0.152816 | 0.042002 | 0.061629 | 4.775746 | 0.055275 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 380.690475 | 219.362078 | 7.284562 |  0.0 |  0.49534 | 0.153022 |  21.806363 |   0.0 | 0.055275 |
    | 1997-16-08 | 20.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.0 | 0.0 |                  10.3 |                     14.698075 |      226.637904 | 18.8 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                         0.0 |                          0.0 |            17.936694 |  20.57219 |   6.433141 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 2.708353 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.741627 | 0.171741 | 0.086702 |  9.982112 |  0.0 | 4.741627 | 4.741627 |   0.0 | 0.171741 | 0.086702 |  9.982112 | 4.618155 |   0.0 | 0.157853 | 0.053536 | 0.096016 |  4.92556 | 0.057009 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 371.357334 | 219.771543 | 7.408034 |  0.0 | 0.509228 | 0.186188 |  31.692458 |   0.0 | 0.057009 |
    | 1997-17-08 | 20.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.0 | 0.0 |                  11.1 |                     14.628078 |      235.528503 | 18.8 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                         0.0 |                          0.0 |            18.906499 | 17.884246 | -18.694526 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.378596 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.756623 | 0.172062 |  0.08933 | 10.397901 |  0.0 | 4.756623 | 4.756623 |   0.0 | 0.172062 |  0.08933 | 10.397901 | 4.679602 |   0.0 | 0.161642 | 0.062847 |  0.13218 |  5.03627 |  0.05829 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 387.151861 | 219.077031 | 7.485056 |  0.0 | 0.519648 | 0.212671 |  41.958179 |   0.0 |  0.05829 |
    | 1997-18-08 | 20.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.0 | 0.0 |                  11.0 |                     14.557728 |      232.849428 | 18.8 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                         0.0 |                          0.0 |            18.089927 | 22.433069 |   4.998415 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.779721 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.731221 | 0.171518 | 0.084888 |  9.692687 |  0.0 | 4.731221 | 4.731221 |   0.0 | 0.171518 | 0.084888 |  9.692687 | 4.708984 |   0.0 | 0.164373 | 0.069355 | 0.167694 | 5.110406 | 0.059148 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 379.253445 | 218.716996 | 7.507292 |  0.0 | 0.526794 | 0.228204 |  51.483172 |   0.0 | 0.059148 |
    | 1997-19-08 | 20.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.0 | 0.0 |                   8.5 |                     14.487046 |       199.71349 | 18.8 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                         0.0 |                          0.0 |            17.619414 | 20.158326 |  -5.650046 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.628366 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.718114 | 0.171236 | 0.082615 |  9.327154 |  0.0 | 4.718114 | 4.718114 |   0.0 | 0.171236 | 0.082615 |  9.327154 | 4.715981 |   0.0 | 0.166259 | 0.073219 | 0.201164 | 5.156622 | 0.059683 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 382.003492 |  218.88951 | 7.509426 |  0.0 | 0.531771 |   0.2376 |  60.609163 |   0.0 | 0.059683 |
    | 1997-20-08 | 20.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.0 | 0.0 |                  11.3 |                     14.416048 |      233.771902 | 18.8 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                         0.0 |                          0.0 |            18.722517 | 20.950339 |  -8.697966 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 4.624935 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.724389 | 0.171371 | 0.083701 |  9.502299 |  0.0 | 4.724389 | 4.724389 |   0.0 | 0.171371 | 0.083701 |  9.502299 | 4.718591 |   0.0 | 0.167619 | 0.075905 | 0.234173 | 5.196288 | 0.060142 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 387.801458 | 217.882815 | 7.515224 |  0.0 | 0.535523 | 0.245396 |  69.877289 |   0.0 | 0.060142 |
    | 1997-21-08 | 20.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.0 | 0.0 |                  12.4 |                     14.344752 |      246.121286 | 18.8 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                         0.0 |                          0.0 |             16.95292 | 22.620153 |    0.72092 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 4.756903 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.687908 | 0.170583 | 0.077428 |  8.480382 |  0.0 | 4.687908 | 4.687908 |   0.0 | 0.170583 | 0.077428 |  8.480382 | 4.711903 |   0.0 | 0.168519 | 0.077117 | 0.265545 | 5.223084 | 0.060452 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 384.180539 | 217.809611 | 7.491228 |  0.0 | 0.537587 | 0.245707 |  78.092126 |   0.0 | 0.060452 |
    | 1997-22-08 | 20.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.0 | 0.0 |                   0.1 |                     14.273175 |        90.58551 | 18.8 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                         0.0 |                          0.0 |            19.305027 | 21.577328 |  11.863971 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 1.361124 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.685268 | 0.170526 | 0.076978 |  8.406087 |  0.0 | 4.685268 | 4.685268 |   0.0 | 0.170526 | 0.076978 |  8.406087 | 4.700032 |   0.0 | 0.169067 | 0.077137 | 0.294843 | 5.241079 | 0.060661 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 369.416568 | 221.209629 | 7.476464 |  0.0 | 0.539046 | 0.245548 |   86.20337 |   0.0 | 0.060661 |
    | 1997-23-08 | 20.0 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.0 | 0.0 |                   6.7 |                     14.201333 |      171.944993 | 18.8 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                         0.0 |                          0.0 |            19.945708 | 17.325305 | -17.848171 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 2.097454 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.809732 | 0.173188 | 0.098766 | 11.858545 |  0.0 | 4.809732 | 4.809732 |   0.0 | 0.173188 | 0.098766 | 11.858545 | 4.725189 |   0.0 | 0.169838 | 0.080183 | 0.330091 | 5.305301 | 0.061404 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 384.364739 | 220.271945 | 7.561006 |  0.0 | 0.542396 | 0.264131 |  97.731823 |   0.0 | 0.061404 |
    | 1997-24-08 | 20.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.0 | 0.0 |                  10.4 |                     14.129243 |      216.549813 | 18.8 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                         0.0 |                          0.0 |            23.000686 | 21.630378 | -14.817732 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 3.649811 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.775024 | 0.172454 | 0.092578 |  10.90609 |  0.0 | 4.775024 | 4.775024 |   0.0 | 0.172454 | 0.092578 |  10.90609 | 4.755683 |   0.0 | 0.170637 | 0.084314 | 0.369686 |  5.38032 | 0.062272 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  396.28247 | 218.775987 | 7.580347 |  0.0 | 0.544213 | 0.272394 | 108.268228 |   0.0 | 0.062272 |
    | 1997-25-08 | 20.0 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.0 | 0.0 |                   6.5 |                     14.056918 |       166.91683 | 18.8 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                         0.0 |                          0.0 |            22.484088 | 25.062685 |   8.176109 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 2.895865 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.720259 | 0.171282 | 0.082986 |  9.387044 |  0.0 | 4.720259 | 4.720259 |   0.0 | 0.171282 | 0.082986 |  9.387044 | 4.750605 |   0.0 |  0.17096 | 0.085181 | 0.404712 | 5.411457 | 0.062633 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 385.206362 | 219.618551 | 7.550001 |  0.0 | 0.544536 | 0.270199 |  117.25056 |   0.0 | 0.062633 |
    | 1997-26-08 | 20.0 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.0 | 0.0 |                   4.9 |                     13.984374 |      145.985611 | 18.8 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                         0.0 |                          0.0 |            21.103195 | 21.872765 |   4.636593 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 2.071449 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.751014 | 0.171942 | 0.088345 | 10.242541 |  0.0 | 4.751014 | 4.751014 |   0.0 | 0.171942 | 0.088345 | 10.242541 | 4.744369 |   0.0 | 0.171141 | 0.085349 | 0.438425 | 5.439285 | 0.062955 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 377.669769 |  220.39326 | 7.556645 |  0.0 | 0.545337 | 0.273195 | 127.054676 |   0.0 | 0.062955 |
    | 1997-27-08 | 20.0 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.0 | 0.0 |                   6.6 |                     13.911624 |       165.54389 | 18.8 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                         0.0 |                          0.0 |            21.241626 | 19.702227 |   -7.18664 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 2.108859 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.779498 | 0.172549 | 0.093371 | 11.029304 |  0.0 | 4.779498 | 4.779498 |   0.0 | 0.172549 | 0.093371 | 11.029304 | 4.754801 |   0.0 | 0.171443 | 0.086869 |  0.47496 | 5.488074 | 0.063519 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 381.956408 |  220.30968 | 7.581342 |  0.0 | 0.546443 | 0.279696 |  137.60902 |   0.0 | 0.063519 |
    | 1997-28-08 | 20.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.0 | 0.0 |                   0.3 |                     13.838682 |       87.693346 | 18.8 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                         0.0 |                          0.0 |            20.510475 | 20.936779 |   4.420674 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 |  1.33161 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.776415 | 0.172483 | 0.092824 | 10.944416 |  0.0 | 4.776415 | 4.776415 |   0.0 | 0.172483 | 0.092824 | 10.944416 | 4.765526 |   0.0 | 0.171731 | 0.088544 | 0.512621 | 5.538422 | 0.064102 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 374.635735 | 221.091931 | 7.592231 |  0.0 | 0.547195 | 0.283976 | 148.040815 |   0.0 | 0.064102 |
    | 1997-29-08 | 20.0 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.0 | 0.0 |                   0.1 |                     13.765561 |       84.356885 | 18.8 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                         0.0 |                          0.0 |            16.420805 | 18.828425 |   16.37055 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 0.881773 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.805359 | 0.173096 | 0.097982 | 11.738981 |  0.0 | 4.805359 | 4.805359 |   0.0 | 0.173096 | 0.097982 | 11.738981 | 4.778056 |   0.0 | 0.172021 | 0.090429 | 0.551418 | 5.591924 | 0.064721 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 355.365185 | 221.494741 | 7.619534 |  0.0 |  0.54827 | 0.291529 | 159.228378 |   0.0 | 0.064721 |
    | 1997-30-08 | 20.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.0 | 0.0 |                   5.0 |                     13.692273 |      142.309202 | 18.8 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                         0.0 |                          0.0 |            15.195625 | 13.278507 |  -22.32896 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 1.913351 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.820344 | 0.173411 | 0.100674 | 12.148195 |  0.0 | 4.820344 | 4.820344 |   0.0 | 0.173411 | 0.100674 | 12.148195 | 4.794648 |   0.0 | 0.172355 | 0.092846 | 0.592233 | 5.652082 | 0.065418 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 374.794145 | 220.438767 |  7.64523 |  0.0 | 0.549325 | 0.299357 |  170.78434 |   0.0 | 0.065418 |
    | 1997-31-08 | 20.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.0 | 0.0 |                   3.8 |                     13.618831 |      126.722078 | 18.8 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                         0.0 |                          0.0 |            16.432781 | 18.874047 |   4.044283 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 18.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.7 | 1.882123 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.781177 | 0.172584 | 0.093669 | 11.075523 |  0.0 | 4.781177 | 4.781177 |   0.0 | 0.172584 | 0.093669 | 11.075523 | 4.796549 |   0.0 | 0.172523 | 0.093963 | 0.631711 | 5.694745 | 0.065911 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 367.849862 | 220.533691 | 7.629859 |  0.0 | 0.549387 | 0.299063 | 181.228153 |   0.0 | 0.065911 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

>>> del model.soilmodel
>>> inputs.nied.series = nied

.. _lland_knauf_ic_wasser_daily:

water
-----

.. integration-test::

    >>> lnk(WASSER)
    >>> negq(False)
    >>> model.aetmodel.parameters.control.surfaceresistance.wasser_aug = 0.0
    >>> model.aetmodel.parameters.control.cropheight.wasser_aug = 0.05
    >>> model.aetmodel.parameters.control.albedo.wasser_aug = 0.7
    >>> conditions_wasser = test(
    ...     "lland_knauf_ic_wasser_daily",
    ...     axis1=(fluxes.nkor, fluxes.evi, fluxes.qah),
    ...     get_conditions="1997-08-03")
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |  tz |  wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow | nbes | sbes | snowintmax | snowintrate | nbesinz | sbesinz | wniedinz | actualalbedoinz | wadainz | schmpotinz | schminz | gefrpotinz | gefrinz | evsinz |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatinz | wlatsnow | wsensinz | wsenssnow | wsurfinz | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga |       qah |       qa | inzp | stinz | sinz | esnowinz | asinz | wats | waes | esnow | taus | ebdn | bowa | sdg1 | sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.0 | 0.0 |                   6.3 |                     15.690785 |       190.25149 | 12.502 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                         0.0 |                          0.0 |            15.797694 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.077197 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 11.424803 | 0.132232 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.132232 |
    | 1997-02-08 |  0.1 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.0 | 0.0 |                   1.7 |                     15.628821 |      127.660782 |  0.094 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                         0.0 |                          0.0 |            16.970501 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.094 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-03-08 |  0.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.0 | 0.0 |                   4.5 |                     15.566134 |      164.401298 |    0.0 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                         0.0 |                          0.0 |            18.022179 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-04-08 |  0.1 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.0 | 0.0 |                  12.4 |                     15.502754 |      269.020502 |  0.094 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                         0.0 |                          0.0 |            15.840774 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.094 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-05-08 |  0.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.0 | 0.0 |                  13.9 |                     15.438712 |      287.848667 |    0.0 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                         0.0 |                          0.0 |            15.977013 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-06-08 |  0.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.0 | 0.0 |                  13.0 |                     15.374034 |      274.674263 |    0.0 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                         0.0 |                          0.0 |            18.131496 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-07-08 |  0.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.0 | 0.0 |                  13.8 |                      15.30875 |      284.047892 |    0.0 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                         0.0 |                          0.0 |            18.084931 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-08 |  0.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.0 | 0.0 |                  12.3 |                     15.242885 |      262.971263 |    0.0 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                         0.0 |                          0.0 |            19.830469 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-09-08 |  0.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.176467 |      272.256414 |    0.0 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                         0.0 |                          0.0 |            19.660875 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-10-08 |  0.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.0 | 0.0 |                  12.8 |                     15.109521 |      267.019218 |    0.0 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                         0.0 |                          0.0 |            19.275317 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-11-08 |  0.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.042071 |      269.626547 |    0.0 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                         0.0 |                          0.0 |            19.420506 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-12-08 |  0.1 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.0 | 0.0 |                  13.3 |                     14.974141 |      270.878291 |  0.094 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                         0.0 |                          0.0 |             18.79766 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.094 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-13-08 |  0.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.0 | 0.0 |                  12.7 |                     14.905755 |      261.697164 |    0.0 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                         0.0 |                          0.0 |              19.6521 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-14-08 |  0.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.0 | 0.0 |                  10.2 |                     14.836934 |      227.962116 |    0.0 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                         0.0 |                          0.0 |            19.656892 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-15-08 |  2.7 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.0 | 0.0 |                   9.4 |                       14.7677 |      216.356321 |  2.538 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                         0.0 |                          0.0 |            20.028753 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.529542 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  1.008458 | 0.011672 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.011672 |
    | 1997-16-08 |  0.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.0 | 0.0 |                  10.3 |                     14.698075 |      226.637904 |    0.0 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                         0.0 |                          0.0 |            17.936694 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-17-08 |  0.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.0 | 0.0 |                  11.1 |                     14.628078 |      235.528503 |    0.0 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                         0.0 |                          0.0 |            18.906499 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-18-08 |  0.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.0 | 0.0 |                  11.0 |                     14.557728 |      232.849428 |    0.0 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                         0.0 |                          0.0 |            18.089927 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-19-08 |  0.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.0 | 0.0 |                   8.5 |                     14.487046 |       199.71349 |    0.0 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                         0.0 |                          0.0 |            17.619414 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-20-08 |  0.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.0 | 0.0 |                  11.3 |                     14.416048 |      233.771902 |    0.0 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                         0.0 |                          0.0 |            18.722517 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.0 | 0.0 |                  12.4 |                     14.344752 |      246.121286 |    0.0 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                         0.0 |                          0.0 |             16.95292 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.0 | 0.0 |                   0.1 |                     14.273175 |        90.58551 |   1.88 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                         0.0 |                          0.0 |            19.305027 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.806188 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  1.073812 | 0.012428 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.012428 |
    | 1997-23-08 |  0.3 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.0 | 0.0 |                   6.7 |                     14.201333 |      171.944993 |  0.282 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                         0.0 |                          0.0 |            19.945708 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.282 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-24-08 |  0.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.0 | 0.0 |                  10.4 |                     14.129243 |      216.549813 |    0.0 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                         0.0 |                          0.0 |            23.000686 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-25-08 |  0.1 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.0 | 0.0 |                   6.5 |                     14.056918 |       166.91683 |  0.094 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                         0.0 |                          0.0 |            22.484088 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.094 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-26-08 | 10.5 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.0 | 0.0 |                   4.9 |                     13.984374 |      145.985611 |   9.87 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                         0.0 |                          0.0 |            21.103195 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.077096 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  8.792904 |  0.10177 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.10177 |
    | 1997-27-08 |  6.5 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.0 | 0.0 |                   6.6 |                     13.911624 |       165.54389 |   6.11 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                         0.0 |                          0.0 |            21.241626 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  1.09756 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   5.01244 | 0.058014 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.058014 |
    | 1997-28-08 | 15.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.0 | 0.0 |                   0.3 |                     13.838682 |       87.693346 |   14.1 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                         0.0 |                          0.0 |            20.510475 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.794859 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 13.305141 | 0.153995 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.153995 |
    | 1997-29-08 |  9.2 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.0 | 0.0 |                   0.1 |                     13.765561 |       84.356885 |  8.648 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                         0.0 |                          0.0 |            16.420805 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.607776 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  8.040224 | 0.093058 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.093058 |
    | 1997-30-08 |  0.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.0 | 0.0 |                   5.0 |                     13.692273 |      142.309202 |    0.0 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                         0.0 |                          0.0 |            15.195625 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-31-08 |  0.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.0 | 0.0 |                   3.8 |                     13.618831 |      126.722078 |    0.0 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                         0.0 |                          0.0 |            16.432781 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_ic_wasser_routing_daily:

water (routing)
---------------

.. integration-test::

    >>> inlet.sequences.sim.series = 0.02
    >>> test("lland_knauf_ic_wasser_routing_daily",
    ...      axis1=(fluxes.nkor, fluxes.evi, fluxes.qah))
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |   qz |   qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |  tz |  wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow | nbes | sbes | snowintmax | snowintrate | nbesinz | sbesinz | wniedinz | actualalbedoinz | wadainz | schmpotinz | schminz | gefrpotinz | gefrinz | evsinz |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatinz | wlatsnow | wsensinz | wsenssnow | wsurfinz | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga |       qah |       qa | inzp | stinz | sinz | esnowinz | asinz | wats | waes | esnow | taus | ebdn | bowa | sdg1 | sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.02 | 1.728 |                   6.3 |                     15.690785 |       190.25149 | 12.502 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                         0.0 |                          0.0 |            15.797694 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.077197 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 13.152803 | 0.152232 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.152232 |
    | 1997-02-08 |  0.1 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.02 | 1.728 |                   1.7 |                     15.628821 |      127.660782 |  0.094 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                         0.0 |                          0.0 |            16.970501 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.86545 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.95655 | 0.011071 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.011071 |
    | 1997-03-08 |  0.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.02 | 1.728 |                   4.5 |                     15.566134 |      164.401298 |    0.0 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                         0.0 |                          0.0 |            18.022179 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.092365 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.635635 | 0.007357 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.007357 |
    | 1997-04-08 |  0.1 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.02 | 1.728 |                  12.4 |                     15.502754 |      269.020502 |  0.094 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                         0.0 |                          0.0 |            15.840774 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.565894 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.256106 | 0.002964 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.002964 |
    | 1997-05-08 |  0.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.02 | 1.728 |                  13.9 |                     15.438712 |      287.848667 |    0.0 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                         0.0 |                          0.0 |            15.977013 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-06-08 |  0.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.02 | 1.728 |                  13.0 |                     15.374034 |      274.674263 |    0.0 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                         0.0 |                          0.0 |            18.131496 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-07-08 |  0.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.02 | 1.728 |                  13.8 |                      15.30875 |      284.047892 |    0.0 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                         0.0 |                          0.0 |            18.084931 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-08-08 |  0.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.02 | 1.728 |                  12.3 |                     15.242885 |      262.971263 |    0.0 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                         0.0 |                          0.0 |            19.830469 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-09-08 |  0.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.02 | 1.728 |                  13.1 |                     15.176467 |      272.256414 |    0.0 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                         0.0 |                          0.0 |            19.660875 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-10-08 |  0.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.02 | 1.728 |                  12.8 |                     15.109521 |      267.019218 |    0.0 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                         0.0 |                          0.0 |            19.275317 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-11-08 |  0.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.02 | 1.728 |                  13.1 |                     15.042071 |      269.626547 |    0.0 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                         0.0 |                          0.0 |            19.420506 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-12-08 |  0.1 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.02 | 1.728 |                  13.3 |                     14.974141 |      270.878291 |  0.094 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                         0.0 |                          0.0 |             18.79766 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    1.822 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-13-08 |  0.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.02 | 1.728 |                  12.7 |                     14.905755 |      261.697164 |    0.0 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                         0.0 |                          0.0 |              19.6521 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-14-08 |  0.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.02 | 1.728 |                  10.2 |                     14.836934 |      227.962116 |    0.0 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                         0.0 |                          0.0 |            19.656892 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.672736 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.055264 |  0.00064 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |  0.00064 |
    | 1997-15-08 |  2.7 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.02 | 1.728 |                   9.4 |                       14.7677 |      216.356321 |  2.538 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                         0.0 |                          0.0 |            20.028753 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.529542 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  2.736458 | 0.031672 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.031672 |
    | 1997-16-08 |  0.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.02 | 1.728 |                  10.3 |                     14.698075 |      226.637904 |    0.0 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                         0.0 |                          0.0 |            17.936694 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.283901 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.444099 |  0.00514 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |  0.00514 |
    | 1997-17-08 |  0.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.02 | 1.728 |                  11.1 |                     14.628078 |      235.528503 |    0.0 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                         0.0 |                          0.0 |            18.906499 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.510743 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.217257 | 0.002515 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.002515 |
    | 1997-18-08 |  0.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.02 | 1.728 |                  11.0 |                     14.557728 |      232.849428 |    0.0 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                         0.0 |                          0.0 |            18.089927 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.594572 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.133428 | 0.001544 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.001544 |
    | 1997-19-08 |  0.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.02 | 1.728 |                   8.5 |                     14.487046 |       199.71349 |    0.0 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                         0.0 |                          0.0 |            17.619414 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.541567 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.186433 | 0.002158 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.002158 |
    | 1997-20-08 |  0.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.02 | 1.728 |                  11.3 |                     14.416048 |      233.771902 |    0.0 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                         0.0 |                          0.0 |            18.722517 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.02 | 1.728 |                  12.4 |                     14.344752 |      246.121286 |    0.0 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                         0.0 |                          0.0 |             16.95292 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.02 | 1.728 |                   0.1 |                     14.273175 |        90.58551 |   1.88 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                         0.0 |                          0.0 |            19.305027 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.806188 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  2.801812 | 0.032428 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.032428 |
    | 1997-23-08 |  0.3 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.02 | 1.728 |                   6.7 |                     14.201333 |      171.944993 |  0.282 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                         0.0 |                          0.0 |            19.945708 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  1.08421 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.92579 | 0.010715 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.010715 |
    | 1997-24-08 |  0.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.02 | 1.728 |                  10.4 |                     14.129243 |      216.549813 |    0.0 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                         0.0 |                          0.0 |            23.000686 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.672232 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.055768 | 0.000645 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.000645 |
    | 1997-25-08 |  0.1 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.02 | 1.728 |                   6.5 |                     14.056918 |       166.91683 |  0.094 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                         0.0 |                          0.0 |            22.484088 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.374102 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.447898 | 0.005184 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.005184 |
    | 1997-26-08 | 10.5 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.02 | 1.728 |                   4.9 |                     13.984374 |      145.985611 |   9.87 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                         0.0 |                          0.0 |            21.103195 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.077096 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 10.520904 |  0.12177 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |  0.12177 |
    | 1997-27-08 |  6.5 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.02 | 1.728 |                   6.6 |                     13.911624 |       165.54389 |   6.11 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                         0.0 |                          0.0 |            21.241626 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  1.09756 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   6.74044 | 0.078014 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.078014 |
    | 1997-28-08 | 15.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.02 | 1.728 |                   0.3 |                     13.838682 |       87.693346 |   14.1 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                         0.0 |                          0.0 |            20.510475 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.794859 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 15.033141 | 0.173995 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.173995 |
    | 1997-29-08 |  9.2 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.02 | 1.728 |                   0.1 |                     13.765561 |       84.356885 |  8.648 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                         0.0 |                          0.0 |            16.420805 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.607776 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  9.768224 | 0.113058 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.113058 |
    | 1997-30-08 |  0.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.02 | 1.728 |                   5.0 |                     13.692273 |      142.309202 |    0.0 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                         0.0 |                          0.0 |            15.195625 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.90065 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.82735 | 0.009576 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.009576 |
    | 1997-31-08 |  0.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.02 | 1.728 |                   3.8 |                     13.618831 |      126.722078 |    0.0 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                         0.0 |                          0.0 |            16.432781 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.912078 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.815922 | 0.009444 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.009444 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_ic_wasser_negq_daily:

water (negative runoff)
-----------------------

.. integration-test::

    >>> negq(True)
    >>> inlet.sequences.sim.series = 0.0
    >>> test("lland_knauf_ic_wasser_negq_daily",
    ...      axis1=(fluxes.nkor, fluxes.evi, fluxes.qah))
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |  tz |  wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow | nbes | sbes | snowintmax | snowintrate | nbesinz | sbesinz | wniedinz | actualalbedoinz | wadainz | schmpotinz | schminz | gefrpotinz | gefrinz | evsinz |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatinz | wlatsnow | wsensinz | wsenssnow | wsurfinz | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga |       qah |        qa | inzp | stinz | sinz | esnowinz | asinz | wats | waes | esnow | taus | ebdn | bowa | sdg1 | sdg2 | sig1 | sig2 | sbg | inlet |    outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.0 | 0.0 |                   6.3 |                     15.690785 |       190.25149 | 12.502 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                         0.0 |                          0.0 |            15.797694 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.077197 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 11.424803 |  0.132232 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.132232 |
    | 1997-02-08 |  0.1 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.0 | 0.0 |                   1.7 |                     15.628821 |      127.660782 |  0.094 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                         0.0 |                          0.0 |            16.970501 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.86545 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  -0.77145 | -0.008929 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.008929 |
    | 1997-03-08 |  0.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.0 | 0.0 |                   4.5 |                     15.566134 |      164.401298 |    0.0 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                         0.0 |                          0.0 |            18.022179 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.092365 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.092365 | -0.012643 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.012643 |
    | 1997-04-08 |  0.1 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.0 | 0.0 |                  12.4 |                     15.502754 |      269.020502 |  0.094 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                         0.0 |                          0.0 |            15.840774 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.565894 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.471894 | -0.017036 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.017036 |
    | 1997-05-08 |  0.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.0 | 0.0 |                  13.9 |                     15.438712 |      287.848667 |    0.0 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                         0.0 |                          0.0 |            15.977013 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.729733 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.729733 |  -0.02002 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  -0.02002 |
    | 1997-06-08 |  0.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.0 | 0.0 |                  13.0 |                     15.374034 |      274.674263 |    0.0 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                         0.0 |                          0.0 |            18.131496 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.826087 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.826087 | -0.021135 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.021135 |
    | 1997-07-08 |  0.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.0 | 0.0 |                  13.8 |                      15.30875 |      284.047892 |    0.0 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                         0.0 |                          0.0 |            18.084931 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 2.041956 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -2.041956 | -0.023634 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.023634 |
    | 1997-08-08 |  0.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.0 | 0.0 |                  12.3 |                     15.242885 |      262.971263 |    0.0 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                         0.0 |                          0.0 |            19.830469 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.800053 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.800053 | -0.020834 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.020834 |
    | 1997-09-08 |  0.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.176467 |      272.256414 |    0.0 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                         0.0 |                          0.0 |            19.660875 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  1.82186 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  -1.82186 | -0.021086 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.021086 |
    | 1997-10-08 |  0.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.0 | 0.0 |                  12.8 |                     15.109521 |      267.019218 |    0.0 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                         0.0 |                          0.0 |            19.275317 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.891469 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.891469 | -0.021892 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.021892 |
    | 1997-11-08 |  0.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.042071 |      269.626547 |    0.0 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                         0.0 |                          0.0 |            19.420506 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 2.106575 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -2.106575 | -0.024382 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.024382 |
    | 1997-12-08 |  0.1 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.0 | 0.0 |                  13.3 |                     14.974141 |      270.878291 |  0.094 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                         0.0 |                          0.0 |             18.79766 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 2.036929 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.942929 | -0.022488 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.022488 |
    | 1997-13-08 |  0.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.0 | 0.0 |                  12.7 |                     14.905755 |      261.697164 |    0.0 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                         0.0 |                          0.0 |              19.6521 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.808559 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.808559 | -0.020932 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.020932 |
    | 1997-14-08 |  0.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.0 | 0.0 |                  10.2 |                     14.836934 |      227.962116 |    0.0 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                         0.0 |                          0.0 |            19.656892 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.672736 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.672736 |  -0.01936 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  -0.01936 |
    | 1997-15-08 |  2.7 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.0 | 0.0 |                   9.4 |                       14.7677 |      216.356321 |  2.538 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                         0.0 |                          0.0 |            20.028753 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.529542 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  1.008458 |  0.011672 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.011672 |
    | 1997-16-08 |  0.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.0 | 0.0 |                  10.3 |                     14.698075 |      226.637904 |    0.0 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                         0.0 |                          0.0 |            17.936694 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.283901 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.283901 |  -0.01486 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  -0.01486 |
    | 1997-17-08 |  0.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.0 | 0.0 |                  11.1 |                     14.628078 |      235.528503 |    0.0 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                         0.0 |                          0.0 |            18.906499 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.510743 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.510743 | -0.017485 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.017485 |
    | 1997-18-08 |  0.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.0 | 0.0 |                  11.0 |                     14.557728 |      232.849428 |    0.0 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                         0.0 |                          0.0 |            18.089927 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.594572 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.594572 | -0.018456 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.018456 |
    | 1997-19-08 |  0.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.0 | 0.0 |                   8.5 |                     14.487046 |       199.71349 |    0.0 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                         0.0 |                          0.0 |            17.619414 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.541567 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.541567 | -0.017842 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.017842 |
    | 1997-20-08 |  0.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.0 | 0.0 |                  11.3 |                     14.416048 |      233.771902 |    0.0 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                         0.0 |                          0.0 |            18.722517 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.939383 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.939383 | -0.022447 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.022447 |
    | 1997-21-08 |  0.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.0 | 0.0 |                  12.4 |                     14.344752 |      246.121286 |    0.0 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                         0.0 |                          0.0 |             16.95292 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.826811 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.826811 | -0.021144 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.021144 |
    | 1997-22-08 |  2.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.0 | 0.0 |                   0.1 |                     14.273175 |        90.58551 |   1.88 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                         0.0 |                          0.0 |            19.305027 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.806188 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  1.073812 |  0.012428 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.012428 |
    | 1997-23-08 |  0.3 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.0 | 0.0 |                   6.7 |                     14.201333 |      171.944993 |  0.282 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                         0.0 |                          0.0 |            19.945708 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  1.08421 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  -0.80221 | -0.009285 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.009285 |
    | 1997-24-08 |  0.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.0 | 0.0 |                  10.4 |                     14.129243 |      216.549813 |    0.0 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                         0.0 |                          0.0 |            23.000686 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.672232 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.672232 | -0.019355 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.019355 |
    | 1997-25-08 |  0.1 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.0 | 0.0 |                   6.5 |                     14.056918 |       166.91683 |  0.094 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                         0.0 |                          0.0 |            22.484088 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.374102 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.280102 | -0.014816 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.014816 |
    | 1997-26-08 | 10.5 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.0 | 0.0 |                   4.9 |                     13.984374 |      145.985611 |   9.87 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                         0.0 |                          0.0 |            21.103195 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.077096 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  8.792904 |   0.10177 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |   0.10177 |
    | 1997-27-08 |  6.5 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.0 | 0.0 |                   6.6 |                     13.911624 |       165.54389 |   6.11 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                         0.0 |                          0.0 |            21.241626 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  1.09756 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   5.01244 |  0.058014 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.058014 |
    | 1997-28-08 | 15.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.0 | 0.0 |                   0.3 |                     13.838682 |       87.693346 |   14.1 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                         0.0 |                          0.0 |            20.510475 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.794859 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 13.305141 |  0.153995 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.153995 |
    | 1997-29-08 |  9.2 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.0 | 0.0 |                   0.1 |                     13.765561 |       84.356885 |  8.648 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                         0.0 |                          0.0 |            16.420805 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.607776 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  8.040224 |  0.093058 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.093058 |
    | 1997-30-08 |  0.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.0 | 0.0 |                   5.0 |                     13.692273 |      142.309202 |    0.0 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                         0.0 |                          0.0 |            15.195625 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.90065 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  -0.90065 | -0.010424 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.010424 |
    | 1997-31-08 |  0.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.0 | 0.0 |                   3.8 |                     13.618831 |      126.722078 |    0.0 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                         0.0 |                          0.0 |            16.432781 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.912078 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -0.912078 | -0.010556 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.010556 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_ic_see_daily:

lakes
-----

.. integration-test::

    >>> lnk(SEE)
    >>> model.aetmodel.parameters.control.surfaceresistance.see_aug = 0.0
    >>> model.aetmodel.parameters.control.cropheight.see_aug = 0.05
    >>> model.aetmodel.parameters.control.albedo.see_aug = 0.7
    >>> negq(False)
    >>> test("lland_knauf_ic_see_daily",
    ...      axis1=(fluxes.nkor, fluxes.evi, fluxes.qah))
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |  tz |  wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow | nbes | sbes | snowintmax | snowintrate | nbesinz | sbesinz | wniedinz | actualalbedoinz | wadainz | schmpotinz | schminz | gefrpotinz | gefrinz | evsinz |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatinz | wlatsnow | wsensinz | wsenssnow | wsurfinz | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz | qdgz1 | qdgz2 | qigz1 | qigz2 |      qbgz | qdga1 | qdga2 | qiga1 | qiga2 |      qbga |      qah |       qa | inzp | stinz | sinz | esnowinz | asinz | wats | waes | esnow | taus | ebdn | bowa | sdg1 | sdg2 | sig1 | sig2 |        sbg | inlet |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.0 | 0.0 |                   6.3 |                     15.690785 |       190.25149 | 12.502 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                         0.0 |                          0.0 |            15.797694 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.077197 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 11.424803 |   0.0 |   0.0 |   0.0 |   0.0 |  0.020478 | 0.020478 | 0.000237 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  11.404325 |   0.0 | 0.000237 |
    | 1997-02-08 |  0.1 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.0 | 0.0 |                   1.7 |                     15.628821 |      127.660782 |  0.094 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                         0.0 |                          0.0 |            16.970501 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.86545 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.77145 |   0.0 |   0.0 |   0.0 |   0.0 |  0.039475 | 0.039475 | 0.000457 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  10.593401 |   0.0 | 0.000457 |
    | 1997-03-08 |  0.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.0 | 0.0 |                   4.5 |                     15.566134 |      164.401298 |    0.0 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                         0.0 |                          0.0 |            18.022179 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.092365 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.092365 |   0.0 |   0.0 |   0.0 |   0.0 |  0.035994 | 0.035994 | 0.000417 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   9.465042 |   0.0 | 0.000417 |
    | 1997-04-08 |  0.1 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.0 | 0.0 |                  12.4 |                     15.502754 |      269.020502 |  0.094 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                         0.0 |                          0.0 |            15.840774 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.565894 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.471894 |   0.0 |   0.0 |   0.0 |   0.0 |  0.031272 | 0.031272 | 0.000362 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   7.961876 |   0.0 | 0.000362 |
    | 1997-05-08 |  0.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.0 | 0.0 |                  13.9 |                     15.438712 |      287.848667 |    0.0 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                         0.0 |                          0.0 |            15.977013 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.729733 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.729733 |   0.0 |   0.0 |   0.0 |   0.0 |  0.025424 | 0.025424 | 0.000294 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   6.206719 |   0.0 | 0.000294 |
    | 1997-06-08 |  0.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.0 | 0.0 |                  13.0 |                     15.374034 |      274.674263 |    0.0 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                         0.0 |                          0.0 |            18.131496 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.826087 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.826087 |   0.0 |   0.0 |   0.0 |   0.0 |  0.018963 | 0.018963 | 0.000219 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   4.361669 |   0.0 | 0.000219 |
    | 1997-07-08 |  0.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.0 | 0.0 |                  13.8 |                      15.30875 |      284.047892 |    0.0 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                         0.0 |                          0.0 |            18.084931 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 2.041956 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -2.041956 |   0.0 |   0.0 |   0.0 |   0.0 |  0.011966 | 0.011966 | 0.000138 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   2.307747 |   0.0 | 0.000138 |
    | 1997-08-08 |  0.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.0 | 0.0 |                  12.3 |                     15.242885 |      262.971263 |    0.0 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                         0.0 |                          0.0 |            19.830469 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.800053 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.800053 |   0.0 |   0.0 |   0.0 |   0.0 |  0.005041 | 0.005041 | 0.000058 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.502653 |   0.0 | 0.000058 |
    | 1997-09-08 |  0.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.176467 |      272.256414 |    0.0 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                         0.0 |                          0.0 |            19.660875 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.820396 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -1.82186 |   0.0 |   0.0 |   0.0 |   0.0 | -0.001465 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  -1.317743 |   0.0 |      0.0 |
    | 1997-10-08 |  0.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.0 | 0.0 |                  12.8 |                     15.109521 |      267.019218 |    0.0 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                         0.0 |                          0.0 |            19.275317 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.883357 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.891469 |   0.0 |   0.0 |   0.0 |   0.0 | -0.008111 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |    -3.2011 |   0.0 |      0.0 |
    | 1997-11-08 |  0.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.042071 |      269.626547 |    0.0 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                         0.0 |                          0.0 |            19.420506 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 2.091331 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -2.106575 |   0.0 |   0.0 |   0.0 |   0.0 | -0.015244 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  -5.292431 |   0.0 |      0.0 |
    | 1997-12-08 |  0.1 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.0 | 0.0 |                  13.3 |                     14.974141 |      270.878291 |  0.094 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                         0.0 |                          0.0 |             18.79766 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 2.014486 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.942929 |   0.0 |   0.0 |   0.0 |   0.0 | -0.022443 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  -7.212917 |   0.0 |      0.0 |
    | 1997-13-08 |  0.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.0 | 0.0 |                  12.7 |                     14.905755 |      261.697164 |    0.0 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                         0.0 |                          0.0 |              19.6521 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.779476 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.808559 |   0.0 |   0.0 |   0.0 |   0.0 | -0.029083 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  -8.992393 |   0.0 |      0.0 |
    | 1997-14-08 |  0.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.0 | 0.0 |                  10.2 |                     14.836934 |      227.962116 |    0.0 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                         0.0 |                          0.0 |            19.656892 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.637521 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.672736 |   0.0 |   0.0 |   0.0 |   0.0 | -0.035215 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -10.629914 |   0.0 |      0.0 |
    | 1997-15-08 |  2.7 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.0 | 0.0 |                   9.4 |                       14.7677 |      216.356321 |  2.538 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                         0.0 |                          0.0 |            20.028753 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.493266 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  1.008458 |   0.0 |   0.0 |   0.0 |   0.0 | -0.036276 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   -9.58518 |   0.0 |      0.0 |
    | 1997-16-08 |  0.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.0 | 0.0 |                  10.3 |                     14.698075 |      226.637904 |    0.0 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                         0.0 |                          0.0 |            17.936694 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.247259 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.283901 |   0.0 |   0.0 |   0.0 |   0.0 | -0.036642 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -10.832439 |   0.0 |      0.0 |
    | 1997-17-08 |  0.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.0 | 0.0 |                  11.1 |                     14.628078 |      235.528503 |    0.0 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                         0.0 |                          0.0 |            18.906499 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.469226 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.510743 |   0.0 |   0.0 |   0.0 |   0.0 | -0.041517 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -12.301666 |   0.0 |      0.0 |
    | 1997-18-08 |  0.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.0 | 0.0 |                  11.0 |                     14.557728 |      232.849428 |    0.0 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                         0.0 |                          0.0 |            18.089927 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.547641 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.594572 |   0.0 |   0.0 |   0.0 |   0.0 | -0.046931 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -13.849307 |   0.0 |      0.0 |
    | 1997-19-08 |  0.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.0 | 0.0 |                   8.5 |                     14.487046 |       199.71349 |    0.0 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                         0.0 |                          0.0 |            17.619414 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.489186 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.541567 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.05238 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -15.338493 |   0.0 |      0.0 |
    | 1997-20-08 |  0.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.0 | 0.0 |                  11.3 |                     14.416048 |      233.771902 |    0.0 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                         0.0 |                          0.0 |            18.722517 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.880954 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.939383 |   0.0 |   0.0 |   0.0 |   0.0 | -0.058428 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -17.219448 |   0.0 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.0 | 0.0 |                  12.4 |                     14.344752 |      246.121286 |    0.0 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                         0.0 |                          0.0 |             16.95292 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.761846 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.826811 |   0.0 |   0.0 |   0.0 |   0.0 | -0.064965 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -18.981293 |   0.0 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.0 | 0.0 |                   0.1 |                     14.273175 |        90.58551 |   1.88 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                         0.0 |                          0.0 |            19.305027 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.740109 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  1.073812 |   0.0 |   0.0 |   0.0 |   0.0 | -0.066078 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -17.841402 |   0.0 |      0.0 |
    | 1997-23-08 |  0.3 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.0 | 0.0 |                   6.7 |                     14.201333 |      171.944993 |  0.282 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                         0.0 |                          0.0 |            19.945708 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.018853 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.80221 |   0.0 |   0.0 |   0.0 |   0.0 | -0.065357 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -18.578255 |   0.0 |      0.0 |
    | 1997-24-08 |  0.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.0 | 0.0 |                  10.4 |                     14.129243 |      216.549813 |    0.0 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                         0.0 |                          0.0 |            23.000686 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.602675 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.672232 |   0.0 |   0.0 |   0.0 |   0.0 | -0.069557 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -20.180931 |   0.0 |      0.0 |
    | 1997-25-08 |  0.1 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.0 | 0.0 |                   6.5 |                     14.056918 |       166.91683 |  0.094 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                         0.0 |                          0.0 |            22.484088 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.299507 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.280102 |   0.0 |   0.0 |   0.0 |   0.0 | -0.074595 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -21.386437 |   0.0 |      0.0 |
    | 1997-26-08 | 10.5 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.0 | 0.0 |                   4.9 |                     13.984374 |      145.985611 |   9.87 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                         0.0 |                          0.0 |            21.103195 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.016237 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  8.792904 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.06086 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -12.532674 |   0.0 |      0.0 |
    | 1997-27-08 |  6.5 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.0 | 0.0 |                   6.6 |                     13.911624 |       165.54389 |   6.11 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                         0.0 |                          0.0 |            21.241626 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 1.061645 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   5.01244 |   0.0 |   0.0 |   0.0 |   0.0 | -0.035916 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  -7.484319 |   0.0 |      0.0 |
    | 1997-28-08 | 15.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.0 | 0.0 |                   0.3 |                     13.838682 |       87.693346 |   14.1 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                         0.0 |                          0.0 |            20.510475 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.791893 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 13.305141 |   0.0 |   0.0 |   0.0 |   0.0 | -0.002966 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   5.823788 |   0.0 |      0.0 |
    | 1997-29-08 |  9.2 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.0 | 0.0 |                   0.1 |                     13.765561 |       84.356885 |  8.648 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                         0.0 |                          0.0 |            16.420805 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.607776 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  8.040224 |   0.0 |   0.0 |   0.0 |   0.0 |  0.035276 | 0.035276 | 0.000408 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  13.828736 |   0.0 | 0.000408 |
    | 1997-30-08 |  0.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.0 | 0.0 |                   5.0 |                     13.692273 |      142.309202 |    0.0 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                         0.0 |                          0.0 |            15.195625 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.90065 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.90065 |   0.0 |   0.0 |   0.0 |   0.0 |  0.047929 | 0.047929 | 0.000555 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  12.880157 |   0.0 | 0.000555 |
    | 1997-31-08 |  0.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.0 | 0.0 |                   3.8 |                     13.618831 |      126.722078 |    0.0 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                         0.0 |                          0.0 |            16.432781 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.912078 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.912078 |   0.0 |   0.0 |   0.0 |   0.0 |   0.04451 |  0.04451 | 0.000515 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  11.923568 |   0.0 | 0.000515 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_ic_fluss_daily:

streams
-------

.. integration-test::

    >>> lnk(FLUSS)
    >>> model.aetmodel.parameters.control.surfaceresistance.fluss_aug = 0.0
    >>> model.aetmodel.parameters.control.cropheight.fluss_aug = 0.05
    >>> model.aetmodel.parameters.control.albedo.fluss_aug = 0.7
    >>> test("lland_knauf_ic_fluss_daily",
    ...      axis1=(fluxes.nkor, fluxes.evi, fluxes.qah))
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |  tz |  wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow | nbes | sbes | snowintmax | snowintrate | nbesinz | sbesinz | wniedinz | actualalbedoinz | wadainz | schmpotinz | schminz | gefrpotinz | gefrinz | evsinz |       evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatinz | wlatsnow | wsensinz | wsenssnow | wsurfinz | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap |      qdgz |     qdgz1 |    qdgz2 | qigz1 | qigz2 | qbgz |     qdga1 |    qdga2 | qiga1 | qiga2 | qbga |      qah |       qa | inzp | stinz | sinz | esnowinz | asinz | wats | waes | esnow | taus | ebdn | bowa |      sdg1 |     sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.0 | 0.0 |                   6.3 |                     15.690785 |       190.25149 | 12.502 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                         0.0 |                          0.0 |            15.797694 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  1.077197 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 11.424803 | 11.424803 |      0.0 |   0.0 |   0.0 |  0.0 |  2.941276 |      0.0 |   0.0 |   0.0 |  0.0 | 2.941276 | 0.034043 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  8.483526 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.034043 |
    | 1997-02-08 |  0.1 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.0 | 0.0 |                   1.7 |                     15.628821 |      127.660782 |  0.094 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                         0.0 |                          0.0 |            16.970501 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.86545 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.77145 |  -0.77145 |      0.0 |   0.0 |   0.0 |  0.0 |  3.758016 |      0.0 |   0.0 |   0.0 |  0.0 | 3.758016 | 0.043496 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  3.954061 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.043496 |
    | 1997-03-08 |  0.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.0 | 0.0 |                   4.5 |                     15.566134 |      164.401298 |    0.0 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                         0.0 |                          0.0 |            18.022179 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  1.092365 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.092365 | -1.092365 |      0.0 |   0.0 |   0.0 |  0.0 |  1.562905 |      0.0 |   0.0 |   0.0 |  0.0 | 1.562905 | 0.018089 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  1.298791 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.018089 |
    | 1997-04-08 |  0.1 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.0 | 0.0 |                  12.4 |                     15.502754 |      269.020502 |  0.094 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                         0.0 |                          0.0 |            15.840774 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  1.565894 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.471894 | -1.471894 |      0.0 |   0.0 |   0.0 |  0.0 |  0.226808 |      0.0 |   0.0 |   0.0 |  0.0 | 0.226808 | 0.002625 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  -0.39991 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002625 |
    | 1997-05-08 |  0.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.0 | 0.0 |                  13.9 |                     15.438712 |      287.848667 |    0.0 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                         0.0 |                          0.0 |            15.977013 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  1.097905 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.729733 | -1.729733 |      0.0 |   0.0 |   0.0 |  0.0 | -0.631827 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.497815 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-06-08 |  0.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.0 | 0.0 |                  13.0 |                     15.374034 |      274.674263 |    0.0 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                         0.0 |                          0.0 |            18.131496 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.657403 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.826087 | -1.826087 |      0.0 |   0.0 |   0.0 |  0.0 | -1.168684 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.155218 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-07-08 |  0.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.0 | 0.0 |                  13.8 |                      15.30875 |      284.047892 |    0.0 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                         0.0 |                          0.0 |            18.084931 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.511091 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -2.041956 | -2.041956 |      0.0 |   0.0 |   0.0 |  0.0 | -1.530864 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.666309 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-08 |  0.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.0 | 0.0 |                  12.3 |                     15.242885 |      262.971263 |    0.0 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                         0.0 |                          0.0 |            19.830469 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.093098 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.800053 | -1.800053 |      0.0 |   0.0 |   0.0 |  0.0 | -1.706955 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.759408 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-09-08 |  0.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.176467 |      272.256414 |    0.0 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                         0.0 |                          0.0 |            19.660875 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.065871 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -1.82186 |  -1.82186 |      0.0 |   0.0 |   0.0 |  0.0 | -1.755989 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.825279 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-10-08 |  0.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.0 | 0.0 |                  12.8 |                     15.109521 |      267.019218 |    0.0 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                         0.0 |                          0.0 |            19.275317 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.086838 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.891469 | -1.891469 |      0.0 |   0.0 |   0.0 |  0.0 | -1.804631 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.912116 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-11-08 |  0.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.042071 |      269.626547 |    0.0 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                         0.0 |                          0.0 |            19.420506 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.206065 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -2.106575 | -2.106575 |      0.0 |   0.0 |   0.0 |  0.0 | -1.900509 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -3.118182 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-12-08 |  0.1 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.0 | 0.0 |                  13.3 |                     14.974141 |      270.878291 |  0.094 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                         0.0 |                          0.0 |             18.79766 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.082443 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.942929 | -1.942929 |      0.0 |   0.0 |   0.0 |  0.0 | -1.954486 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -3.106625 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-13-08 |  0.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.0 | 0.0 |                  12.7 |                     14.905755 |      261.697164 |    0.0 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                         0.0 |                          0.0 |              19.6521 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.105944 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.808559 | -1.808559 |      0.0 |   0.0 |   0.0 |  0.0 | -1.914503 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -3.000681 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-14-08 |  0.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.0 | 0.0 |                  10.2 |                     14.836934 |      227.962116 |    0.0 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                         0.0 |                          0.0 |            19.656892 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.157388 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.672736 | -1.672736 |      0.0 |   0.0 |   0.0 |  0.0 | -1.830125 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.843292 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-15-08 |  2.7 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.0 | 0.0 |                   9.4 |                       14.7677 |      216.356321 |  2.538 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                         0.0 |                          0.0 |            20.028753 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.463086 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.008458 |  1.008458 |      0.0 |   0.0 |   0.0 |  0.0 | -1.066456 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -0.768378 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-16-08 |  0.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.0 | 0.0 |                  10.3 |                     14.698075 |      226.637904 |    0.0 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                         0.0 |                          0.0 |            17.936694 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.595002 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.283901 | -1.283901 |      0.0 |   0.0 |   0.0 |  0.0 | -0.688899 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  -1.36338 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-17-08 |  0.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.0 | 0.0 |                  11.1 |                     14.628078 |      235.528503 |    0.0 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                         0.0 |                          0.0 |            18.906499 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.485942 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.510743 | -1.510743 |      0.0 |   0.0 |   0.0 |  0.0 | -1.024801 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.849322 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-18-08 |  0.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.0 | 0.0 |                  11.0 |                     14.557728 |      232.849428 |    0.0 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                         0.0 |                          0.0 |            18.089927 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.321551 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.594572 | -1.594572 |      0.0 |   0.0 |   0.0 |  0.0 |  -1.27302 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.170873 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-19-08 |  0.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.0 | 0.0 |                   8.5 |                     14.487046 |       199.71349 |    0.0 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                         0.0 |                          0.0 |            17.619414 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.132224 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.541567 | -1.541567 |      0.0 |   0.0 |   0.0 |  0.0 | -1.409342 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.303098 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-20-08 |  0.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.0 | 0.0 |                  11.3 |                     14.416048 |      233.771902 |    0.0 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                         0.0 |                          0.0 |            18.722517 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.365956 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.939383 | -1.939383 |      0.0 |   0.0 |   0.0 |  0.0 | -1.573427 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.669054 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.0 | 0.0 |                  12.4 |                     14.344752 |      246.121286 |    0.0 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                         0.0 |                          0.0 |             16.95292 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.111688 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.826811 | -1.826811 |      0.0 |   0.0 |   0.0 |  0.0 | -1.715123 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.780741 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.0 | 0.0 |                   0.1 |                     14.273175 |        90.58551 |   1.88 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                         0.0 |                          0.0 |            19.305027 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  -0.21427 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.073812 |  1.073812 |      0.0 |   0.0 |   0.0 |  0.0 | -1.020458 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -0.686471 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-23-08 |  0.3 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.0 | 0.0 |                   6.7 |                     14.201333 |      171.944993 |  0.282 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                         0.0 |                          0.0 |            19.945708 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.557521 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.80221 |  -0.80221 |      0.0 |   0.0 |   0.0 |  0.0 | -0.526689 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -0.961992 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-24-08 |  0.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.0 | 0.0 |                  10.4 |                     14.129243 |      216.549813 |    0.0 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                         0.0 |                          0.0 |            23.000686 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.793059 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.672232 | -1.672232 |      0.0 |   0.0 |   0.0 |  0.0 | -0.879173 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.755051 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-25-08 |  0.1 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.0 | 0.0 |                   6.5 |                     14.056918 |       166.91683 |  0.094 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                         0.0 |                          0.0 |            22.484088 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.226008 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.280102 | -1.280102 |      0.0 |   0.0 |   0.0 |  0.0 | -1.148095 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.887059 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-26-08 | 10.5 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.0 | 0.0 |                   4.9 |                     13.984374 |      145.985611 |   9.87 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                         0.0 |                          0.0 |            21.103195 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  1.077096 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  8.792904 |  8.792904 |      0.0 |   0.0 |   0.0 |  0.0 |    1.3836 |      0.0 |   0.0 |   0.0 |  0.0 |   1.3836 | 0.016014 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  5.522245 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.016014 |
    | 1997-27-08 |  6.5 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.0 | 0.0 |                   6.6 |                     13.911624 |       165.54389 |   6.11 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                         0.0 |                          0.0 |            21.241626 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   1.09756 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   5.01244 |   5.01244 |      0.0 |   0.0 |   0.0 |  0.0 |   3.86595 |      0.0 |   0.0 |   0.0 |  0.0 |  3.86595 | 0.044745 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  6.668735 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.044745 |
    | 1997-28-08 | 15.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.0 | 0.0 |                   0.3 |                     13.838682 |       87.693346 |   14.1 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                         0.0 |                          0.0 |            20.510475 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.794859 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 13.305141 |      12.0 | 1.305141 |   0.0 |   0.0 |  0.0 |  6.199583 |  0.82777 |   0.0 |   0.0 |  0.0 | 7.027354 | 0.081335 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | 12.469152 | 0.477371 |  0.0 |  0.0 | 0.0 |   0.0 | 0.081335 |
    | 1997-29-08 |  9.2 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.0 | 0.0 |                   0.1 |                     13.765561 |       84.356885 |  8.648 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                         0.0 |                          0.0 |            16.420805 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.607776 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  8.040224 |  8.040224 |      0.0 |   0.0 |   0.0 |  0.0 |  7.885403 | 0.438667 |   0.0 |   0.0 |  0.0 |  8.32407 | 0.096343 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | 12.623973 | 0.038704 |  0.0 |  0.0 | 0.0 |   0.0 | 0.096343 |
    | 1997-30-08 |  0.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.0 | 0.0 |                   5.0 |                     13.692273 |      142.309202 |    0.0 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                         0.0 |                          0.0 |            15.195625 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.90065 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.90065 |  -0.90065 |      0.0 |   0.0 |   0.0 |  0.0 |  5.655812 | 0.035566 |   0.0 |   0.0 |  0.0 | 5.691378 | 0.065872 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   6.06751 | 0.003138 |  0.0 |  0.0 | 0.0 |   0.0 | 0.065872 |
    | 1997-31-08 |  0.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.0 | 0.0 |                   3.8 |                     13.618831 |      126.722078 |    0.0 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                         0.0 |                          0.0 |            16.432781 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.912078 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.912078 | -0.912078 |      0.0 |   0.0 |   0.0 |  0.0 |  2.595008 | 0.002884 |   0.0 |   0.0 |  0.0 | 2.597892 | 0.030068 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  2.560423 | 0.000254 |  0.0 |  0.0 | 0.0 |   0.0 | 0.030068 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_ic_vers_daily:

sealed surfaces
---------------

.. integration-test::

    >>> lnk(VERS)
    >>> negq(True)
    >>> model.aetmodel.parameters.control.surfaceresistance.vers_aug = 500
    >>> model.aetmodel.parameters.control.cropheight.vers_aug = 5.0
    >>> model.aetmodel.parameters.control.albedo.vers_aug = 0.10
    >>> lai.vers_aug = 10.0
    >>> test("lland_knauf_ic_vers_daily",
    ...      axis1=(inputs.nied, fluxes.qah, states.bowa))
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow |   nbes | sbes | snowintmax | snowintrate | nbesinz | sbesinz | wniedinz | actualalbedoinz | wadainz | schmpotinz | schminz | gefrpotinz | gefrinz | evsinz |   evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatinz | wlatsnow | wsensinz | wsenssnow | wsurfinz | wsurf | sff | fvg |   wada |    qdb | qib1 | qib2 | qbb | qkap |   qdgz |  qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 |    qdga2 | qiga1 | qiga2 | qbga |      qah |       qa | inzp | stinz | sinz | esnowinz | asinz | wats | waes | esnow | taus |       ebdn | bowa |      sdg1 |     sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.0 | 0.0 |                   6.3 |                     15.690785 |       190.25149 | 12.502 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                         0.0 |                          0.0 |            15.797694 | 11.733333 |      -26.2 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 10.502 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 10.502 | 10.502 |  0.0 |  0.0 | 0.0 |  0.0 | 10.502 | 10.502 |   0.0 |   0.0 |   0.0 |  0.0 | 2.703704 |      0.0 |   0.0 |   0.0 |  0.0 | 2.703704 | 0.031293 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |      373.3 |  0.0 |  7.798296 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.031293 |
    | 1997-02-08 |  0.1 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.0 | 0.0 |                   1.7 |                     15.628821 |      127.660782 |  0.094 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                         0.0 |                          0.0 |            16.970501 | 18.443733 |     6.8624 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.094 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 3.637039 |      0.0 |   0.0 |   0.0 |  0.0 | 3.637039 | 0.042095 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |   363.5376 |  0.0 |  4.161257 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.042095 |
    | 1997-03-08 |  0.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.0 | 0.0 |                   4.5 |                     15.566134 |      164.401298 |    0.0 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                         0.0 |                          0.0 |            18.022179 | 15.632162 | -19.607027 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.940764 |      0.0 |   0.0 |   0.0 |  0.0 | 1.940764 | 0.022463 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 380.244627 |  0.0 |  2.220493 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.022463 |
    | 1997-04-08 |  0.1 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.0 | 0.0 |                  12.4 |                     15.502754 |      269.020502 |  0.094 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                         0.0 |                          0.0 |            15.840774 | 20.443786 |  15.862716 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.094 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.035613 |      0.0 |   0.0 |   0.0 |  0.0 | 1.035613 | 0.011986 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 361.481911 |  0.0 |  1.184879 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.011986 |
    | 1997-05-08 |  0.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.0 | 0.0 |                  13.9 |                     15.438712 |      287.848667 |    0.0 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                         0.0 |                          0.0 |            15.977013 | 15.040124 | -20.759257 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.552615 |      0.0 |   0.0 |   0.0 |  0.0 | 0.552615 | 0.006396 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 379.341169 |  0.0 |  0.632265 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.006396 |
    | 1997-06-08 |  0.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.0 | 0.0 |                  13.0 |                     15.374034 |      274.674263 |    0.0 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                         0.0 |                          0.0 |            18.131496 |  20.18359 |   1.701539 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.294881 |      0.0 |   0.0 |   0.0 |  0.0 | 0.294881 | 0.003413 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 374.739629 |  0.0 |  0.337383 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.003413 |
    | 1997-07-08 |  0.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.0 | 0.0 |                  13.8 |                      15.30875 |      284.047892 |    0.0 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                         0.0 |                          0.0 |            18.084931 | 18.858347 | -11.649921 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.157352 |      0.0 |   0.0 |   0.0 |  0.0 | 0.157352 | 0.001821 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  383.48955 |  0.0 |  0.180032 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001821 |
    | 1997-08-08 |  0.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.0 | 0.0 |                  12.3 |                     15.242885 |      262.971263 |    0.0 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                         0.0 |                          0.0 |            19.830469 | 21.378324 |   2.869942 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.083965 |      0.0 |   0.0 |   0.0 |  0.0 | 0.083965 | 0.000972 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 377.719608 |  0.0 |  0.096067 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000972 |
    | 1997-09-08 |  0.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.176467 |      272.256414 |    0.0 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                         0.0 |                          0.0 |            19.660875 |  19.71658 | -15.500518 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.044804 |      0.0 |   0.0 |   0.0 |  0.0 | 0.044804 | 0.000519 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 390.320126 |  0.0 |  0.051262 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000519 |
    | 1997-10-08 |  0.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.0 | 0.0 |                  12.8 |                     15.109521 |      267.019218 |    0.0 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                         0.0 |                          0.0 |            19.275317 |  23.34553 |   2.073177 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.023908 |      0.0 |   0.0 |   0.0 |  0.0 | 0.023908 | 0.000277 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 385.346949 |  0.0 |  0.027354 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000277 |
    | 1997-11-08 |  0.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.042071 |      269.626547 |    0.0 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                         0.0 |                          0.0 |            19.420506 | 21.913255 |  -7.120473 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.012758 |      0.0 |   0.0 |   0.0 |  0.0 | 0.012758 | 0.000148 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 389.567421 |  0.0 |  0.014596 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000148 |
    | 1997-12-08 |  0.1 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.0 | 0.0 |                  13.3 |                     14.974141 |      270.878291 |  0.094 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                         0.0 |                          0.0 |             18.79766 | 23.128751 |  -0.427496 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.094 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006808 |      0.0 |   0.0 |   0.0 |  0.0 | 0.006808 | 0.000079 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 387.094917 |  0.0 |  0.007789 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000079 |
    | 1997-13-08 |  0.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.0 | 0.0 |                  12.7 |                     14.905755 |      261.697164 |    0.0 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                         0.0 |                          0.0 |              19.6521 | 22.416669 |  -4.699983 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003633 |      0.0 |   0.0 |   0.0 |  0.0 | 0.003633 | 0.000042 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |   388.8949 |  0.0 |  0.004156 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000042 |
    | 1997-14-08 |  0.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.0 | 0.0 |                  10.2 |                     14.836934 |      227.962116 |    0.0 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                         0.0 |                          0.0 |            19.656892 | 22.935065 |   7.410388 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001938 |      0.0 |   0.0 |   0.0 |  0.0 | 0.001938 | 0.000022 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 378.584513 |  0.0 |  0.002218 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000022 |
    | 1997-15-08 |  2.7 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.0 | 0.0 |                   9.4 |                       14.7677 |      216.356321 |  2.538 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                         0.0 |                          0.0 |            20.028753 | 19.965673 |  -5.005962 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.538 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.538 |  0.538 |  0.0 |  0.0 | 0.0 |  0.0 |  0.538 |  0.538 |   0.0 |   0.0 |   0.0 |  0.0 | 0.139541 |      0.0 |   0.0 |   0.0 |  0.0 | 0.139541 | 0.001615 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 380.690475 |  0.0 |  0.400677 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001615 |
    | 1997-16-08 |  0.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.0 | 0.0 |                  10.3 |                     14.698075 |      226.637904 |    0.0 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                         0.0 |                          0.0 |            17.936694 |  20.57219 |   6.433141 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.186871 |      0.0 |   0.0 |   0.0 |  0.0 | 0.186871 | 0.002163 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 371.357334 |  0.0 |  0.213806 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002163 |
    | 1997-17-08 |  0.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.0 | 0.0 |                  11.1 |                     14.628078 |      235.528503 |    0.0 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                         0.0 |                          0.0 |            18.906499 | 17.884246 | -18.694526 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.099717 |      0.0 |   0.0 |   0.0 |  0.0 | 0.099717 | 0.001154 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 387.151861 |  0.0 |  0.114089 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001154 |
    | 1997-18-08 |  0.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.0 | 0.0 |                  11.0 |                     14.557728 |      232.849428 |    0.0 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                         0.0 |                          0.0 |            18.089927 | 22.433069 |   4.998415 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.05321 |      0.0 |   0.0 |   0.0 |  0.0 |  0.05321 | 0.000616 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 379.253445 |  0.0 |  0.060879 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000616 |
    | 1997-19-08 |  0.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.0 | 0.0 |                   8.5 |                     14.487046 |       199.71349 |    0.0 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                         0.0 |                          0.0 |            17.619414 | 20.158326 |  -5.650046 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.028393 |      0.0 |   0.0 |   0.0 |  0.0 | 0.028393 | 0.000329 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 382.003492 |  0.0 |  0.032486 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000329 |
    | 1997-20-08 |  0.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.0 | 0.0 |                  11.3 |                     14.416048 |      233.771902 |    0.0 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                         0.0 |                          0.0 |            18.722517 | 20.950339 |  -8.697966 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.015151 |      0.0 |   0.0 |   0.0 |  0.0 | 0.015151 | 0.000175 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 387.801458 |  0.0 |  0.017335 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000175 |
    | 1997-21-08 |  0.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.0 | 0.0 |                  12.4 |                     14.344752 |      246.121286 |    0.0 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                         0.0 |                          0.0 |             16.95292 | 22.620153 |    0.72092 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.008085 |      0.0 |   0.0 |   0.0 |  0.0 | 0.008085 | 0.000094 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 384.180539 |  0.0 |   0.00925 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000094 |
    | 1997-22-08 |  2.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.0 | 0.0 |                   0.1 |                     14.273175 |        90.58551 |   1.88 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                         0.0 |                          0.0 |            19.305027 | 21.577328 |  11.863971 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  1.88 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004314 |      0.0 |   0.0 |   0.0 |  0.0 | 0.004314 |  0.00005 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 369.416568 |  0.0 |  0.004936 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00005 |
    | 1997-23-08 |  0.3 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.0 | 0.0 |                   6.7 |                     14.201333 |      171.944993 |  0.282 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                         0.0 |                          0.0 |            19.945708 | 17.325305 | -17.848171 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.282 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002302 |      0.0 |   0.0 |   0.0 |  0.0 | 0.002302 | 0.000027 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 384.364739 |  0.0 |  0.002634 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000027 |
    | 1997-24-08 |  0.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.0 | 0.0 |                  10.4 |                     14.129243 |      216.549813 |    0.0 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                         0.0 |                          0.0 |            23.000686 | 21.630378 | -14.817732 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001228 |      0.0 |   0.0 |   0.0 |  0.0 | 0.001228 | 0.000014 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  396.28247 |  0.0 |  0.001405 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000014 |
    | 1997-25-08 |  0.1 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.0 | 0.0 |                   6.5 |                     14.056918 |       166.91683 |  0.094 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                         0.0 |                          0.0 |            22.484088 | 25.062685 |   8.176109 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.094 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000655 |      0.0 |   0.0 |   0.0 |  0.0 | 0.000655 | 0.000008 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 385.206362 |  0.0 |   0.00075 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000008 |
    | 1997-26-08 | 10.5 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.0 | 0.0 |                   4.9 |                     13.984374 |      145.985611 |   9.87 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                         0.0 |                          0.0 |            21.103195 | 21.872765 |   4.636593 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   7.87 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |   7.87 |   7.87 |  0.0 |  0.0 | 0.0 |  0.0 |   7.87 |   7.87 |   0.0 |   0.0 |   0.0 |  0.0 | 2.026454 |      0.0 |   0.0 |   0.0 |  0.0 | 2.026454 | 0.023454 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 377.669769 |  0.0 |  5.844296 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.023454 |
    | 1997-27-08 |  6.5 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.0 | 0.0 |                   6.6 |                     13.911624 |       165.54389 |   6.11 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                         0.0 |                          0.0 |            21.241626 | 19.702227 |   -7.18664 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   4.11 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |   4.11 |   4.11 |  0.0 |  0.0 | 0.0 |  0.0 |   4.11 |   4.11 |   0.0 |   0.0 |   0.0 |  0.0 |  3.78382 |      0.0 |   0.0 |   0.0 |  0.0 |  3.78382 | 0.043794 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 381.956408 |  0.0 |  6.170475 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.043794 |
    | 1997-28-08 | 15.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.0 | 0.0 |                   0.3 |                     13.838682 |       87.693346 |   14.1 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                         0.0 |                          0.0 |            20.510475 | 20.936779 |   4.420674 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   12.1 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |   12.1 |   12.1 |  0.0 |  0.0 | 0.0 |  0.0 |   12.1 |   12.0 |   0.1 |   0.0 |   0.0 |  0.0 | 5.967201 | 0.063424 |   0.0 |   0.0 |  0.0 | 6.030624 | 0.069799 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 374.635735 |  0.0 | 12.203275 | 0.036576 |  0.0 |  0.0 | 0.0 |   0.0 | 0.069799 |
    | 1997-29-08 |  9.2 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.0 | 0.0 |                   0.1 |                     13.765561 |       84.356885 |  8.648 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                         0.0 |                          0.0 |            16.420805 | 18.828425 |   16.37055 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  6.648 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  6.648 |  6.648 |  0.0 |  0.0 | 0.0 |  0.0 |  6.648 |  6.648 |   0.0 |   0.0 |   0.0 |  0.0 | 7.402977 | 0.033611 |   0.0 |   0.0 |  0.0 | 7.436588 | 0.086072 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 355.365185 |  0.0 | 11.448297 | 0.002965 |  0.0 |  0.0 | 0.0 |   0.0 | 0.086072 |
    | 1997-30-08 |  0.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.0 | 0.0 |                   5.0 |                     13.692273 |      142.309202 |    0.0 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                         0.0 |                          0.0 |            15.195625 | 13.278507 |  -22.32896 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 5.339359 | 0.002725 |   0.0 |   0.0 |  0.0 | 5.342084 |  0.06183 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 374.794145 |  0.0 |  6.108938 |  0.00024 |  0.0 |  0.0 | 0.0 |   0.0 |  0.06183 |
    | 1997-31-08 |  0.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.0 | 0.0 |                   3.8 |                     13.618831 |      126.722078 |    0.0 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                         0.0 |                          0.0 |            16.432781 | 18.874047 |   4.044283 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 2.849141 | 0.000221 |   0.0 |   0.0 |  0.0 | 2.849362 | 0.032979 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 367.849862 |  0.0 |  3.259797 | 0.000019 |  0.0 |  0.0 | 0.0 |   0.0 | 0.032979 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_ic_acker_winter_daily:

acre (winter)
-------------

>>> pub.timegrids = "2010-12-01", "2011-01-01", "1d"
>>> lnk(ACKER)
>>> model.aetmodel.parameters.control.cropheight.acker_dec = 0.05
>>> model.aetmodel.parameters.control.albedo.acker_dec = 0.1
>>> model.aetmodel.parameters.control.leafareaindex.acker_dec = 0.3
>>> lai.acker_dec = 0.3
>>> model.aetmodel.parameters.control.averagesoilheatflux.dec = -8.583333333333
>>> wg2z.dec = 8.583333333333
>>> test = IntegrationTest(land)
>>> test.dateformat = "%Y-%d-%m"

>>> test.inits = ((states.inzp, 0.0),
...               (states.stinz, 0.0),
...               (states.sinz, 0.0),
...               (states.esnowinz, 0.0),
...               (states.asinz, 0.0),
...               (states.wats, 0.0),
...               (states.waes, 0.0),
...               (states.esnow, 0.0),
...               (states.taus, 0.0),
...               (states.ebdn, 0.0),
...               (states.bowa, 72.0),
...               (states.sdg1, 0.0),
...               (states.sdg2, 0.0),
...               (states.sig1, 0.0),
...               (states.sig2, 0.0),
...               (states.sbg, 0.0))

>>> inputs.nied.series = (
...     0.0, 11.1, 0.4, 1.2, 6.4, 0.0, 0.0, 0.0, 0.7, 3.3, 9.4, 0.2, 2.9, 0.0, 0.0,
...     6.7, 0.0, 0.1, 0.0, 0.2, 0.2, 2.4, 7.0, 0.2, 0.0, 1.2, 0.9, 0.0, 0.0, 0.2, 0.9)
>>> inputs.teml.series = (
...     -5.8, -3.0, -5.6, -3.8, 0.2, -1.1, -2.9, -2.9, -2.1, -3.1, 5.1, 0.2, -3.5,
...     -3.9, -6.3, -3.5, -7.3, -5.8, -7.9, -5.7, -5.6, -4.8, -1.1, -2.9, -9.1, -8.6,
...     -5.3, -7.7, -9.2, -7.4, 1.7)
>>> inputs.windspeed.series = (
...     6.9, 5.2, 2.0, 4.3, 4.0, 2.4, 1.9, 1.8, 2.7, 3.0, 7.4, 5.7, 2.2, 3.1, 2.8, 3.7,
...     0.9, 1.3, 1.9, 2.4, 2.3, 4.6, 8.6, 7.6, 3.8, 1.5, 1.3, 3.0, 2.7, 1.4, 4.2)
>>> inputs.relativehumidity.series = (
...     70.3, 85.0, 95.9, 90.6, 96.5, 99.0, 99.2, 99.1, 96.9, 95.0, 90.6, 78.7, 90.5,
...     88.1, 92.8, 98.1, 96.6, 97.7, 95.8, 97.5, 95.0, 93.8, 94.9, 85.4, 87.2, 92.7,
...     97.8, 95.4, 94.9, 96.0, 96.5)

>>> model.radiationmodel.sequences.inputs.sunshineduration.series = (
...     3.5, 0.0, 0.1, 3.7, 0.0, 0.1, 0.0, 0.0, 4.4, 0.1, 0.2, 0.7, 0.0, 1.2, 3.0, 0.0,
...     0.0, 0.0, 0.0, 0.2, 0.8, 0.0, 0.0, 0.0, 3.2, 4.3, 0.0, 3.4, 0.0, 0.0, 4.0)

>>> aetinputs.atmosphericpressure.series = (
...     1015.0, 1007.0, 1005.0, 1008.0, 996.0, 998.0, 1004.0, 1006.0, 1009.0, 1018.0,
...     1004.0, 1012.0, 1020.0, 1025.0, 1023.0, 1000.0, 991.0, 997.0, 997.0, 997.0,
...     1004.0, 1007.0, 1004.0, 1008.0, 1018.0, 1019.0, 1017.0, 1022.0, 1024.0, 1021.0,
...     1016.0)
>>> aetinputs.windspeed.series = inputs.windspeed.series
>>> aetinputs.relativehumidity.series = inputs.relativehumidity.series

>>> inlet.sequences.sim.series = 0.0

>>> model.radiationmodel.sequences.inputs.possiblesunshineduration.series = (
...     7.64102946, 7.6096444, 7.57982377, 7.55159603, 7.52498861, 7.50002776,
...     7.47673851, 7.45514453, 7.43526809, 7.41712991, 7.40074913, 7.38614323,
...     7.3733279, 7.36231706, 7.35312273, 7.34575502, 7.34022207, 7.33653002,
...     7.33468296, 7.33468296, 7.33653002, 7.34022207, 7.34575502, 7.35312273,
...     7.36231706, 7.3733279, 7.38614323, 7.40074913, 7.41712991, 7.43526809,
...     7.45514453)
>>> model.radiationmodel.sequences.inputs.globalradiation.series = (
...     31.9427981, 9.8738984, 16.6834998, 31.8074275, 9.5325105, 16.1414844,
...     9.3414287, 9.2567644, 33.405815, 15.5911584, 15.8915442, 17.8235189, 8.941324,
...     19.6680679, 26.8315815, 8.8377384, 8.817416, 8.8041854, 8.7980439, 15.4647459,
...     17.8795412, 8.8221612, 8.8443961, 8.8737425, 27.7596417, 32.3287553, 9.0045823,
...     28.9823209, 9.1276454, 9.1999895, 32.0622345)

.. integration-test::

    >>> conditions_acker_winter = test(
    ...     "lland_knauf_ic_acker_winter_daily",
    ...     axis1=(inputs.nied, fluxes.wada), axis2=(states.waes, states.wats),
    ...     get_conditions="2010-12-10")
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |         tz |         wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow |   nbes |    sbes | snowintmax | snowintrate | nbesinz | sbesinz | wniedinz | actualalbedoinz | wadainz | schmpotinz | schminz | gefrpotinz | gefrinz | evsinz | evi |     evb |       evs |     wnied | tempssurface | actualalbedo |   schmpot |      schm |  gefrpot |     gefr | wlatinz |   wlatsnow | wsensinz |  wsenssnow | wsurfinz |       wsurf |      sff |      fvg |      wada |      qdb | qib1 | qib2 | qbb | qkap |     qdgz |    qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga |      qah |       qa | inzp | stinz | sinz | esnowinz | asinz |      wats |      waes |     esnow |     taus |       ebdn |      bowa |     sdg1 | sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-01-12 |  0.0 | -5.8 |             70.3 |       6.9 |                 7.641029 |              3.5 | 0.0 | 0.0 |                   3.5 |                      7.641029 |       31.942798 |    0.0 | -5.8 |     5.53098 |            5.53098 |                 3.958114 |                         0.0 |                          0.0 |             2.782554 |        0.0 |       34.8 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 | 0.55374 |       0.0 |       0.0 |          nan |          nan |       0.0 |       0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |       0.0 |      nan | -26.216667 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-02-12 | 11.1 | -3.0 |             85.0 |       5.2 |                 7.609644 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.609644 |        9.873898 | 10.434 | -3.0 |    4.168274 |           4.168274 |                 4.893489 |                         0.0 |                     3.818807 |             4.159465 |    -7.5504 |   -7.70501 |                      0.0 |                  1.229506 |                     0.0 |                27.857608 |             0.0 |       -26.628102 | 10.374 |  10.374 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.168357 | -0.752835 |    -6.266232 |     0.875479 |       0.0 |       0.0 | 0.406773 |      0.0 |     0.0 |  -5.197724 |      0.0 | -28.315743 |      0.0 |   -6.885365 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 10.542357 | 10.542357 |  -1.57248 | 0.084415 |  -9.928323 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-03-12 |  0.4 | -5.6 |             95.9 |       2.0 |                 7.579824 |              0.1 | 0.0 | 0.0 |                   0.1 |                      7.579824 |         16.6835 |  0.376 | -5.6 |    1.603182 |           1.603182 |                 4.019239 |                         0.0 |                      3.57079 |              3.85445 |  -2.859357 |  25.653408 |                      0.0 |                  2.321007 |                     0.0 |                37.521585 |             0.0 |       -35.200578 |  0.376 |   0.376 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.073821 | -0.050934 |    -7.134925 |      0.86088 |       0.0 |       0.0 | 0.487468 |      0.0 |     0.0 |  -2.279085 |      0.0 |  -7.007074 |      0.0 |    25.91442 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 10.992177 | 10.992177 | -1.884427 |   0.1418 | -26.998397 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-04-12 |  1.2 | -3.8 |             90.6 |       4.3 |                 7.551596 |              3.7 | 0.0 | 0.0 |                   3.7 |                      7.551596 |       31.807427 |  1.128 | -3.8 |    3.446842 |           3.446842 |                 4.608124 |                         0.0 |                     3.236114 |              4.17496 |  -7.775538 |    3.72366 |                      0.0 |                   4.73374 |                     0.0 |                55.888201 |             0.0 |       -51.154461 |  1.128 |   1.128 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.402208 | -0.103687 |    -8.396148 |     0.851175 |       0.0 |       0.0 | 0.636779 |      0.0 |     0.0 | -12.417475 |      0.0 | -34.539816 |      0.0 |     4.19717 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 12.522386 | 12.522386 | -2.461624 | 0.183432 | -22.138724 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-05-12 |  6.4 |  0.2 |             96.5 |       4.0 |                 7.524989 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.524989 |        9.532511 |  6.016 |  0.2 |    3.206365 |           3.206365 |                 6.197477 |                         0.0 |                      4.91255 |             5.980565 |  -6.375953 |  -20.56744 |                      0.0 |                  1.460936 |                     0.0 |                 15.88459 |             0.0 |       -14.423654 |  6.016 |  2.7072 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.434119 |  0.045165 |    -2.948046 |     0.846742 |       0.0 |       0.0 | 0.403221 | 0.403221 |     0.0 | -13.402657 |      0.0 | -22.446147 |      0.0 |  -21.425149 | 0.977326 |  0.28655 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 15.998885 | 18.972504 |       0.0 | 0.203476 |   7.012049 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-06-12 |  0.0 | -1.1 |             99.0 |       2.4 |                 7.500028 |              0.1 | 0.0 | 0.0 |                   0.1 |                      7.500028 |       16.141484 |    0.0 | -1.1 |    1.923819 |           1.923819 |                 5.634754 |                         0.0 |                     5.151567 |             5.578406 |        0.0 |  13.867616 |                      0.0 |                  3.011222 |                     0.0 |                28.072188 |             0.0 |       -25.060966 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.123565 |       0.0 |    -2.311269 |     0.813448 |       0.0 |       0.0 | 0.317541 | 0.317541 |     0.0 |   -3.81486 |      0.0 |  -6.150959 |      0.0 |   15.095147 | 0.994413 | 0.296657 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 16.422692 | 19.096069 |       0.0 | 0.378868 |   1.727766 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-07-12 |  0.0 | -2.9 |             99.2 |       1.9 |                 7.476739 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.476739 |        9.341429 |    0.0 | -2.9 |    1.523023 |           1.523023 |                 4.930236 |                         0.0 |                     4.623648 |             4.890794 |        0.0 |   22.53221 |                      0.0 |                  1.983477 |                     0.0 |                32.375282 |             0.0 |       -30.391805 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 |  -0.06757 |       0.0 |    -3.755368 |     0.787669 |       0.0 |       0.0 | 0.511768 | 0.511768 |     0.0 |  -2.086102 |      0.0 |   -3.79513 |      0.0 |   24.510573 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 16.994381 | 19.163639 |       0.0 |  0.55426 | -12.221111 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-08-12 |  0.0 | -2.9 |             99.1 |       1.8 |                 7.455145 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.455145 |        9.256764 |    0.0 | -2.9 |    1.442864 |           1.442864 |                 4.930236 |                         0.0 |                     4.223622 |             4.885864 |   -3.51968 |   8.583604 |                      0.0 |                  2.155738 |                     0.0 |                27.150377 |             0.0 |       -24.994639 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 |  -0.16266 |       0.0 |    -4.950281 |     0.767118 |       0.0 |       0.0 | 0.661033 | 0.661033 |     0.0 |   -5.02185 |      0.0 |  -8.833803 |      0.0 |   11.138986 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 17.805273 | 19.326299 |       0.0 | 0.729652 | -12.221381 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-09-12 |  0.7 | -2.1 |             96.9 |       2.7 |                 7.435268 |              4.4 | 0.0 | 0.0 |                   4.4 |                      7.435268 |       33.405815 |  0.658 | -2.1 |    2.164296 |           2.164296 |                 5.233109 |                         0.0 |                     3.897182 |             5.070883 |  -3.519758 |  14.894088 |                      0.0 |                  8.184668 |                     0.0 |                58.769843 |             0.0 |       -50.585175 |  0.658 |   0.658 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.365518 | -0.033425 |    -6.002106 |     0.754993 |       0.0 |       0.0 | 0.807885 | 0.807885 |     0.0 | -11.284727 |      0.0 | -21.316713 |      0.0 |   17.983735 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 19.623632 | 20.349817 |       0.0 |  0.85302 | -18.532136 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-10-12 |  3.3 | -3.1 |             95.0 |       3.0 |                  7.41713 |              0.1 | 0.0 | 0.0 |                   0.1 |                       7.41713 |       15.591158 |  3.102 | -3.1 |    2.404774 |           2.404774 |                 4.856984 |                         0.0 |                     4.026976 |             4.614135 |  -5.337255 |   1.425738 |                      0.0 |                  3.649886 |                     0.0 |                28.618996 |             0.0 |        -24.96911 |  3.102 |   3.102 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.195734 | -0.232614 |    -5.574878 |       0.7659 |       0.0 |       0.0 | 0.843525 | 0.726185 |     0.0 |  -6.042946 |      0.0 | -14.472191 |      0.0 |    4.453973 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 23.647551 | 23.647551 | -0.453606 | 0.741291 |  -11.37454 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-11-12 |  9.4 |  5.1 |             90.6 |       7.4 |                 7.400749 |              0.2 | 0.0 | 0.0 |                   0.2 |                      7.400749 |       15.891544 |  8.836 |  5.1 |    5.931775 |           5.931775 |                 8.790172 |                         0.0 |                       6.1078 |             7.963896 |  -3.275868 | -19.655205 |                      0.0 |                  3.997567 |                     0.0 |                -0.754591 |             0.0 |         4.752158 |  8.836 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -1.215856 |  2.183707 |          0.0 |     0.748447 | 21.462327 | 21.462327 |      0.0 |      0.0 |     0.0 | -37.537452 |      0.0 | -58.603285 |      0.0 | -100.892896 |  0.94547 | 0.268174 | 30.457573 | 9.778899 |  0.0 |  0.0 | 0.0 |  0.0 | 9.778899 | 9.778899 |   0.0 |   0.0 |   0.0 |  0.0 | 2.517544 |   0.0 |   0.0 |   0.0 |  0.0 | 2.517544 | 0.029138 | 0.06 |   0.0 |  0.0 |      0.0 |   nan |  2.267017 |  3.241834 |       0.0 | 0.927231 |  16.863998 | 92.124934 | 7.261355 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.029138 |
    | 2010-12-12 |  0.2 |  0.2 |             78.7 |       5.7 |                 7.386143 |              0.7 | 0.0 | 0.0 |                   0.7 |                      7.386143 |       17.823519 |  0.188 |  0.2 |     4.56907 |            4.56907 |                 6.197477 |                         0.0 |                     5.176451 |             4.877414 |        0.0 |  13.478921 |                      0.0 |                  4.716622 |                     0.0 |                36.312063 |             0.0 |       -31.595441 |  0.188 |  0.0846 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 |  0.158718 |  0.000762 |    -2.246487 |     0.735371 |       0.0 |       0.0 | 0.061533 | 0.061533 |     0.0 |   4.900158 |      0.0 | -22.778045 |      0.0 |   13.717554 |   0.9613 | 0.277229 |  0.067022 | 0.022382 |  0.0 |  0.0 | 0.0 |  0.0 | 0.022382 | 0.022382 |   0.0 |   0.0 |   0.0 |  0.0 | 3.392378 |   0.0 |   0.0 |   0.0 |  0.0 | 3.392378 | 0.039264 | 0.06 |   0.0 |  0.0 |      0.0 |   nan |  2.299254 |  3.204093 |       0.0 | 1.094821 |  11.968411 | 92.169574 | 3.891359 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.039264 |
    | 2010-13-12 |  2.9 | -3.5 |             90.5 |       2.2 |                 7.373328 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.373328 |        8.941324 |  2.726 | -3.5 |    1.763501 |           1.763501 |                 4.713362 |                         0.0 |                     4.320066 |             4.265593 |        0.0 |  27.919864 |                      0.0 |                  2.237089 |                     0.0 |                35.794391 |             0.0 |       -33.557302 |  2.726 |   2.726 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 |  0.014973 | -0.230795 |    -4.653311 |     0.749803 |       0.0 |       0.0 | 0.199105 | 0.199105 |     0.0 |   0.462261 |      0.0 |  -5.560804 |      0.0 |   28.458759 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.814887 |   0.0 |   0.0 |   0.0 |  0.0 | 1.814887 | 0.021006 | 0.06 |   0.0 |  0.0 |      0.0 |   nan |  5.211168 |   5.91512 |       0.0 | 0.911369 |   -7.36812 | 92.169574 | 2.076472 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.021006 |
    | 2010-14-12 |  0.0 | -3.9 |             88.1 |       3.1 |                 7.362317 |              1.2 | 0.0 | 0.0 |                   1.2 |                      7.362317 |       19.668068 |    0.0 | -3.9 |    2.484933 |           2.484933 |                  4.57351 |                         0.0 |                     3.838228 |             4.029262 |  -2.122018 |  24.469911 |                      0.0 |                  5.193321 |                     0.0 |                46.412117 |             0.0 |       -41.218796 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.065079 |       0.0 |    -6.200337 |     0.735952 |       0.0 |       0.0 | 0.256899 | 0.256899 |     0.0 |  -2.009214 |      0.0 | -13.746566 |      0.0 |   25.463016 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.968444 |   0.0 |   0.0 |   0.0 |  0.0 | 0.968444 | 0.011209 | 0.06 |   0.0 |  0.0 |      0.0 |   nan |  5.528228 |    5.9802 |       0.0 | 1.086761 | -23.254697 | 92.169574 | 1.108028 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.011209 |
    | 2010-15-12 |  0.0 | -6.3 |             92.8 |       2.8 |                 7.353123 |              3.0 | 0.0 | 0.0 |                   3.0 |                      7.353123 |       26.831581 |    0.0 | -6.3 |    2.244455 |           2.244455 |                 3.808888 |                         0.0 |                     2.826082 |             3.534648 |  -6.697353 |  20.473897 |                      0.0 |                  7.398865 |                     0.0 |                57.719027 |             0.0 |       -50.320161 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.225845 |       0.0 |   -10.109669 |     0.724248 |       0.0 |       0.0 | 0.406999 | 0.406999 |     0.0 |  -6.972561 |      0.0 |  -21.30035 |      0.0 |   22.047251 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.516772 |   0.0 |   0.0 |   0.0 |  0.0 | 0.516772 | 0.005981 | 0.06 |   0.0 |  0.0 |      0.0 |   nan |  6.159374 |  6.206044 |       0.0 | 1.262153 | -35.145261 | 92.169574 | 0.591256 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.005981 |
    | 2010-16-12 |  6.7 | -3.5 |             98.1 |       3.7 |                 7.345755 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.345755 |        8.837738 |  6.298 | -3.5 |    2.965888 |           2.965888 |                 4.713362 |                         0.0 |                     3.578581 |             4.623808 | -10.121835 | -18.089989 |                      0.0 |                  1.897885 |                     0.0 |                22.170845 |             0.0 |        -20.27296 |  6.298 |   6.298 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 |  -0.40193 | -0.533216 |    -7.106837 |     0.785252 |       0.0 |       0.0 | 0.558149 | 0.046671 |     0.0 | -12.408877 |      0.0 |  -24.32963 |      0.0 |  -16.465547 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.275755 |   0.0 |   0.0 |   0.0 |  0.0 | 0.275755 | 0.003192 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 12.905974 | 12.905974 |  -1.97724 | 0.573017 |  -8.471938 | 92.169574 | 0.315501 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.003192 |
    | 2010-17-12 |  0.0 | -7.3 |             96.6 |       0.9 |                 7.340222 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.340222 |        8.817416 |    0.0 | -7.3 |    0.721432 |           0.721432 |                  3.52531 |                         0.0 |                     3.261149 |             3.405449 |  -2.439918 |  35.147834 |                      0.0 |                  1.958046 |                     0.0 |                41.667909 |             0.0 |       -39.709863 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.025948 |       0.0 |    -8.297891 |     0.777934 |       0.0 |       0.0 | 0.670129 |      0.0 |     0.0 |  -0.801088 |      0.0 |  -3.147637 |      0.0 |   35.761138 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.147146 |   0.0 |   0.0 |   0.0 |  0.0 | 0.147146 | 0.001703 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 12.931922 | 12.931922 | -2.590544 | 0.632681 | -35.036439 | 92.169574 | 0.168355 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001703 |
    | 2010-18-12 |  0.1 | -5.8 |             97.7 |       1.3 |                  7.33653 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.33653 |        8.804185 |  0.094 | -5.8 |    1.042069 |           1.042069 |                 3.958114 |                         0.0 |                     2.837121 |             3.867077 | -10.090494 |  -0.178602 |                      0.0 |                  2.001401 |                     0.0 |                24.662372 |             0.0 |       -22.660971 |  0.094 |   0.094 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.215326 | -0.013188 |   -10.060727 |     0.772676 |       0.0 |       0.0 | 0.820045 |      0.0 |     0.0 |  -6.647818 |      0.0 | -15.625407 |      0.0 |    0.387746 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.078519 |   0.0 |   0.0 |   0.0 |  0.0 | 0.078519 | 0.000909 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 13.241248 | 13.241248 | -3.170081 | 0.678422 | -26.274503 | 92.169574 | 0.089836 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000909 |
    | 2010-19-12 |  0.0 | -7.9 |             95.8 |       1.9 |                 7.334683 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.334683 |        8.798044 |    0.0 | -7.9 |    1.523023 |           1.523023 |                 3.364296 |                         0.0 |                     2.722791 |             3.222996 |  -7.567057 |  18.053896 |                      0.0 |                   2.04414 |                     0.0 |                 36.09466 |             0.0 |        -34.05052 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.126518 |       0.0 |    -10.57604 |      0.76766 |       0.0 |       0.0 | 0.876298 |      0.0 |     0.0 |  -3.906012 |      0.0 | -11.873153 |      0.0 |   18.271356 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.041898 |   0.0 |   0.0 |   0.0 |  0.0 | 0.041898 | 0.000485 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 13.367765 | 13.367765 | -3.387541 | 0.724518 | -35.745066 | 92.169574 | 0.047937 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000485 |
    | 2010-20-12 |  0.2 | -5.7 |             97.5 |       2.4 |                 7.334683 |              0.2 | 0.0 | 0.0 |                   0.2 |                      7.334683 |       15.464746 |  0.188 | -5.7 |    1.923819 |           1.923819 |                 3.988573 |                         0.0 |                     2.930412 |             3.888858 | -10.294579 |  -3.845125 |                      0.0 |                  3.642756 |                     0.0 |                28.193275 |             0.0 |       -24.550519 |  0.188 |   0.188 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 |  -0.27746 | -0.025922 |    -9.653725 |     0.764448 |       0.0 |       0.0 | 0.818877 |      0.0 |     0.0 |  -8.566089 |      0.0 | -20.077451 |      0.0 |   -4.093021 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.022357 |   0.0 |   0.0 |   0.0 |  0.0 | 0.022357 | 0.000259 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 13.833225 | 13.833225 | -3.165567 | 0.755386 | -23.316607 | 92.169574 |  0.02558 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000259 |
    | 2010-21-12 |  0.2 | -5.6 |             95.0 |       2.3 |                  7.33653 |              0.8 | 0.0 | 0.0 |                   0.8 |                       7.33653 |       17.879541 |  0.188 | -5.6 |     1.84366 |            1.84366 |                 4.019239 |                         0.0 |                      3.12843 |             3.818277 |  -6.715183 |  12.668619 |                      0.0 |                  4.271628 |                     0.0 |                38.724024 |             0.0 |       -34.452396 |  0.188 |   0.188 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 |  -0.19466 | -0.025467 |     -8.82662 |     0.761089 |       0.0 |       0.0 | 0.774426 |      0.0 |     0.0 |  -6.009778 |      0.0 | -15.971301 |      0.0 |   12.471317 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.01193 |   0.0 |   0.0 |   0.0 |  0.0 |  0.01193 | 0.000138 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 14.215885 | 14.215885 | -2.993732 | 0.788873 | -27.401894 | 92.169574 |  0.01365 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000138 |
    | 2010-22-12 |  2.4 | -4.8 |             93.8 |       4.6 |                 7.340222 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.340222 |        8.822161 |  2.256 | -4.8 |     3.68732 |            3.68732 |                 4.272186 |                         0.0 |                      3.44223 |             4.007311 |  -7.891745 |  -1.711598 |                      0.0 |                  1.990066 |                     0.0 |                30.080567 |             0.0 |       -28.090501 |  2.256 |   2.256 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.254479 | -0.261947 |    -7.606479 |     0.774424 |       0.0 |       0.0 | 0.784019 |      0.0 |     0.0 |  -7.856587 |      0.0 | -22.170375 |      0.0 |   -1.936461 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006366 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006366 | 0.000074 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 16.726363 | 16.726363 | -3.030816 | 0.662934 | -17.106962 | 92.169574 | 0.007284 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000074 |
    | 2010-23-12 |  7.0 | -1.1 |             94.9 |       8.6 |                 7.345755 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.345755 |        8.844396 |   6.58 | -1.1 |    6.893685 |           6.893685 |                 5.634754 |                         0.0 |                     4.876556 |             5.347381 |  -4.926805 | -11.283011 |                      0.0 |                  1.665079 |                     0.0 |                25.120919 |             0.0 |        -23.45584 |   6.58 |  5.0995 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.349728 | -0.214608 |    -3.046303 |     0.811736 |       0.0 |       0.0 | 0.472584 | 0.472584 |     0.0 |  -10.79726 |      0.0 | -25.360128 |      0.0 |  -12.701548 | 0.991077 |  0.29467 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003397 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003397 | 0.000039 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 22.633051 | 23.656092 |       0.0 | 0.389279 |   2.759382 | 92.169574 | 0.003887 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000039 |
    | 2010-24-12 |  0.2 | -2.9 |             85.4 |       7.6 |                 7.353123 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.353123 |        8.873743 |  0.188 | -2.9 |    6.092093 |           6.092093 |                 4.930236 |                         0.0 |                     4.510219 |             4.210422 |        0.0 |  24.507545 |                      0.0 |                  1.886144 |                     0.0 |                 36.55523 |             0.0 |       -34.669086 |  0.188 |   0.188 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 |   0.20077 | -0.013188 |    -4.084591 |     0.787447 |       0.0 |       0.0 | 0.635671 | 0.635671 |     0.0 |   6.198411 |      0.0 | -13.915802 |      0.0 |   26.951695 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001813 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001813 | 0.000021 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 23.259214 | 23.643322 |       0.0 | 0.555966 | -13.164829 | 92.169574 | 0.002074 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000021 |
    | 2010-25-12 |  0.0 | -9.1 |             87.2 |       3.8 |                 7.362317 |              3.2 | 0.0 | 0.0 |                   3.2 |                      7.362317 |       27.759642 |    0.0 | -9.1 |    3.046047 |           3.046047 |                 3.061705 |                         0.0 |                     2.608187 |             2.669806 |  -3.791471 |  43.924699 |                      0.0 |                  6.469714 |                     0.0 |                71.430435 |             0.0 |        -64.96072 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.024146 |       0.0 |   -11.112254 |     0.766938 |       0.0 |       0.0 | 1.670829 | 0.384108 |     0.0 |  -0.745452 |      0.0 | -13.831579 |      0.0 |   50.383689 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000967 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000967 | 0.000011 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 23.667468 | 23.667468 | -4.974127 | 0.731358 | -48.506195 | 92.169574 | 0.001107 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000011 |
    | 2010-26-12 |  1.2 | -8.6 |             92.7 |       1.5 |                 7.373328 |              4.3 | 0.0 | 0.0 |                   4.3 |                      7.373328 |       32.328755 |  1.128 | -8.6 |    1.202387 |           1.202387 |                 3.184716 |                         0.0 |                     1.819064 |             2.952232 | -13.969784 |   9.219314 |                      0.0 |                  7.422688 |                     0.0 |                55.658664 |             0.0 |       -48.235975 |  1.128 |   1.128 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.253474 | -0.234661 |   -15.506336 |       0.7704 |       0.0 |       0.0 | 2.405921 |      0.0 |     0.0 |  -7.825569 |      0.0 | -27.099214 |      0.0 |   13.311192 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000516 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000516 | 0.000006 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 25.048942 | 25.048942 | -9.300666 | 0.699031 | -49.142175 | 92.169574 | 0.000591 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000006 |
    | 2010-27-12 |  0.9 | -5.3 |             97.8 |       1.3 |                 7.386143 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.386143 |        9.004582 |  0.846 | -5.3 |    1.042069 |           1.042069 |                 4.112496 |                         0.0 |                     2.537355 |             4.022021 | -14.152947 |  -16.19232 |                      0.0 |                  2.040733 |                     0.0 |                15.766318 |             0.0 |       -13.725584 |  0.846 |   0.846 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.310389 | -0.108462 |   -11.454227 |     0.773367 |       0.0 |       0.0 | 1.856011 |      0.0 |     0.0 |   -9.58272 |      0.0 | -22.569455 |      0.0 |  -18.426591 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000275 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000275 | 0.000003 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 26.205331 | 26.205331 | -7.174857 | 0.672265 | -24.366522 | 92.169574 | 0.000315 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000003 |
    | 2010-28-12 |  0.0 | -7.7 |             95.4 |       3.0 |                 7.400749 |              3.4 | 0.0 | 0.0 |                   3.4 |                      7.400749 |       28.982321 |    0.0 | -7.7 |    2.404774 |           2.404774 |                 3.417227 |                         0.0 |                     2.553035 |             3.260035 |  -7.017558 |  26.161471 |                      0.0 |                   6.70143 |                     0.0 |                61.683256 |             0.0 |       -54.981825 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.235684 |       0.0 |   -11.377804 |     0.768775 |       0.0 |       0.0 | 1.865726 |      0.0 |     0.0 |  -7.276335 |      0.0 | -21.506464 |      0.0 |   26.199027 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000147 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000147 | 0.000002 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 26.441015 | 26.441015 | -7.212414 |  0.71405 |  -41.94466 | 92.169574 | 0.000168 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000002 |
    | 2010-29-12 |  0.0 | -9.2 |             94.9 |       2.7 |                  7.41713 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.41713 |        9.127645 |    0.0 | -9.2 |    2.164296 |           2.164296 |                 3.037615 |                         0.0 |                     2.252914 |             2.882696 | -12.080062 |   5.033832 |                      0.0 |                   2.15057 |                     0.0 |                34.606803 |             0.0 |       -32.456233 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.196129 |       0.0 |   -12.919034 |     0.764389 |       0.0 |       0.0 | 2.137509 |      0.0 |     0.0 |  -6.055144 |      0.0 | -20.316614 |      0.0 |    6.084474 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000078 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000078 | 0.000001 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 26.637144 | 26.637144 | -8.263056 | 0.755957 | -38.395158 | 92.169574 |  0.00009 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000001 |
    | 2010-30-12 |  0.2 | -7.4 |             96.0 |       1.4 |                 7.435268 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.435268 |        9.199989 |  0.188 | -7.4 |    1.122228 |           1.122228 |                 3.498009 |                         0.0 |                     2.475478 |             3.358088 | -11.057806 |   4.213054 |                      0.0 |                  2.189463 |                     0.0 |                28.181399 |             0.0 |       -25.991936 |  0.188 |   0.188 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.190974 | -0.033653 |   -11.759981 |     0.762015 |       0.0 |       0.0 | 1.974006 |      0.0 |     0.0 |  -5.896006 |      0.0 | -16.548589 |      0.0 |    3.547341 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000042 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000042 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 27.016118 | 27.016118 | -7.630996 | 0.779513 | -34.024879 | 92.169574 | 0.000048 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-31-12 |  0.9 |  1.7 |             96.5 |       4.2 |                 7.455145 |              4.0 | 0.0 | 0.0 |                   4.0 |                      7.455145 |       32.062235 |  0.846 |  1.7 |    3.366683 |           3.366683 |                 6.907942 |                         0.0 |                     4.469848 |             6.666164 |  -9.799165 | -33.573784 |                      0.0 |                  7.749055 |                     0.0 |                41.685254 |             0.0 |       -33.936199 |  0.846 | 0.06345 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.0 |     0.0 | -0.924858 |  0.067075 |    -4.203534 |     0.758312 |       0.0 |       0.0 | 0.753524 | 0.753524 |     0.0 |  -28.55338 |      0.0 | -43.607597 |      0.0 |  -38.224778 | 0.973704 |  0.28443 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000022 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000022 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 28.756986 | 28.786976 |       0.0 | 0.817531 |   8.132239 | 92.169574 | 0.000026 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_ic_nadelw_winter_daily:

conifers (winter)
-----------------

.. integration-test::

    >>> lnk(NADELW)
    >>> model.aetmodel.parameters.control.surfaceresistance.nadelw = 56.0
    >>> model.aetmodel.parameters.control.cropheight.nadelw = 10.0
    >>> model.aetmodel.parameters.control.albedo.nadelw = 0.12
    >>> model.aetmodel.parameters.control.leafareaindex.nadelw = 11.0
    >>> lai.nadelw = 11.0
    >>> conditions_nadelw_winter = test(
    ...     "lland_knauf_ic_nadelw_winter_daily",
    ...     axis1=(inputs.nied, fluxes.wada), axis2=(states.waes, states.wats),
    ...     get_conditions="2010-12-10")
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow |  nbes |    sbes | snowintmax | snowintrate |  nbesinz |  sbesinz |  wniedinz | actualalbedoinz |  wadainz | schmpotinz |  schminz | gefrpotinz |  gefrinz |   evsinz | evi |       evb |       evs |     wnied | tempssurface | actualalbedo |   schmpot |      schm |  gefrpot |     gefr |   wlatinz |   wlatsnow |   wsensinz |  wsenssnow |   wsurfinz |      wsurf |      sff |      fvg |      wada |      qdb | qib1 | qib2 | qbb | qkap |     qdgz |    qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga |      qah |       qa | inzp |    stinz |     sinz |  esnowinz |    asinz |      wats |      waes |     esnow |     taus |       ebdn |      bowa |     sdg1 | sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-01-12 |  0.0 | -5.8 |             70.3 |       6.9 |                 7.641029 |              3.5 | 0.0 | 0.0 |                   3.5 |                      7.641029 |       31.942798 |    0.0 | -5.8 |     5.53098 |           2.449434 |                 3.958114 |                         0.0 |                          0.0 |             2.782554 |       0.0 |       34.8 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |     0.0 |       24.5 |        0.42 |      0.0 |      0.0 |       0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.0 |  0.674992 |       0.0 |       0.0 |          nan |          nan |       0.0 |       0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |        0.0 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |      0.0 |      0.0 |       0.0 |      nan |       0.0 |       0.0 |       0.0 |      nan | -26.216667 | 71.325008 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-02-12 | 11.1 | -3.0 |             85.0 |       5.2 |                 7.609644 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.609644 |        9.873898 | 10.434 | -3.0 |    4.168274 |            1.84595 |                 4.893489 |                    4.893489 |                     4.000777 |             4.159465 |   -7.5504 | -11.341661 |                 1.001169 |                  0.219033 |               17.929265 |                 3.494169 |      -16.928095 |        -3.275135 | 8.234 |   8.234 |       24.5 |        0.42 |  3.45828 |  3.45828 | -0.250965 |        0.875479 |      0.0 |        0.0 |      0.0 |   0.165137 |      0.0 | 0.207278 | 0.0 |  0.389762 | -0.044812 | -0.597537 |    -5.660123 |     0.880553 |       0.0 |       0.0 | 0.169147 |      0.0 |  6.399359 |   -1.38348 | -22.940042 | -13.176974 |   0.387412 | -11.285319 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  2.2 | 3.251002 | 3.251002 | -0.638377 | 0.084415 |  4.820532 |  4.820532 | -0.653879 | 0.065799 |  -6.291673 | 70.935246 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-03-12 |  0.4 | -5.6 |             95.9 |       2.0 |                 7.579824 |              0.1 | 0.0 | 0.0 |                   0.1 |                      7.579824 |         16.6835 |  0.376 | -5.6 |    1.603182 |           0.709981 |                 4.019239 |                    4.019239 |                     4.160243 |              3.85445 | -1.812002 |  20.020333 |                 1.869927 |                  0.418817 |                17.78068 |                17.258471 |      -15.910754 |       -16.839654 | 0.376 |   0.376 |       24.5 |    0.429753 | 0.161587 | 0.161587 | -0.021889 |        0.862355 |      0.0 |        0.0 |      0.0 |   0.232496 |      0.0 |  0.02946 | 0.0 |  0.051032 |  0.054667 | -0.050934 |    -5.148724 |     0.864827 |       0.0 |       0.0 | 0.162216 |      0.0 |   0.90952 |   1.687763 | -16.581769 |   1.415188 |   0.238504 |  19.942605 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  2.2 | 3.383129 | 3.383129 | -0.898771 | 0.135729 |  4.980277 |  4.980277 | -0.627086 | 0.125697 | -17.728672 | 70.884214 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-04-12 |  1.2 | -3.8 |             90.6 |       4.3 |                 7.551596 |              3.7 | 0.0 | 0.0 |                   3.7 |                      7.551596 |       31.807427 |  1.128 | -3.8 |    3.446842 |           1.526459 |                 4.608124 |                    4.608124 |                     4.012494 |              4.17496 | -5.105858 |   3.096474 |                 3.803097 |                  0.869388 |               34.929625 |                13.363685 |      -31.126528 |       -12.494297 | 1.128 |   1.128 |       24.5 |    0.430149 | 0.485209 | 0.485209 | -0.044601 |        0.853164 |      0.0 |        0.0 |      0.0 |   0.279301 |      0.0 | 0.109697 | 0.0 |   0.14346 | -0.041144 | -0.103687 |    -5.621937 |     0.852823 |       0.0 |       0.0 | 0.197815 |      0.0 |  3.386694 |  -1.270243 | -34.376888 |  -8.093651 |   0.136334 |   3.130403 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  2.2 | 3.758641 | 3.758641 | -1.079705 | 0.174653 |  5.664212 |  5.664212 | -0.764701 | 0.176149 | -12.241813 | 70.740754 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-05-12 |  6.4 |  0.2 |             96.5 |       4.0 |                 7.524989 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.524989 |        9.532511 |  6.016 |  0.2 |    3.206365 |           1.419962 |                 6.197477 |                    6.197477 |                     5.324062 |             5.980565 | -3.525642 |  -9.947345 |                 1.212963 |                  0.276378 |                 12.7506 |                 3.874587 |      -11.537636 |         -3.59821 | 6.016 |  2.7072 |       49.0 |    0.431276 | 2.594556 |  1.16755 |  0.019479 |        0.843734 |      0.0 |        0.0 |      0.0 |   0.146071 | 0.146071 | 0.052825 | 0.0 |  0.097263 | -0.159879 |  0.045165 |    -1.867751 |     0.843883 |       0.0 |       0.0 | 0.128253 | 0.128253 |  1.630875 |  -4.935991 | -13.664067 |  -8.833307 |  -0.495555 | -10.171089 | 0.979665 | 0.287923 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  2.2 | 5.030087 | 6.300372 |       0.0 | 0.217465 |  7.461137 |  9.245535 |       0.0 | 0.216766 |   6.288865 | 70.643491 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-06-12 |  0.0 | -1.1 |             99.0 |       2.4 |                 7.500028 |              0.1 | 0.0 | 0.0 |                   0.1 |                      7.500028 |       16.141484 |    0.0 | -1.1 |    1.923819 |           0.851977 |                 5.634754 |                    5.634754 |                     5.469703 |             5.578406 |       0.0 |   9.018379 |                 2.482152 |                  0.565764 |               14.665301 |                11.984392 |      -12.183149 |       -11.418628 |   0.0 |     0.0 |     47.775 |    0.438901 |      0.0 |      0.0 |       0.0 |        0.811154 |      0.0 |        0.0 |      0.0 |   0.216736 | 0.216736 | 0.010803 | 0.0 |  0.012457 | -0.020841 |       0.0 |    -1.503063 |     0.811267 |       0.0 |       0.0 | 0.103795 | 0.103795 |   0.33353 |  -0.643435 | -11.678834 |  -1.355567 |   0.837845 |   9.419626 | 0.981071 |  0.28875 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  2.2 | 5.237827 | 6.289569 |       0.0 | 0.392857 |  7.581986 |  9.266377 |       0.0 | 0.392158 |   5.853819 | 70.631034 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-07-12 |  0.0 | -2.9 |             99.2 |       1.9 |                 7.476739 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.476739 |        9.341429 |    0.0 | -2.9 |    1.523023 |           0.674482 |                 4.930236 |                    4.930236 |                     5.029253 |             4.890794 |       0.0 |  15.802737 |                 1.628868 |                  0.371341 |               15.077555 |                15.302045 |      -13.448687 |       -14.930704 |   0.0 |     0.0 |     25.725 |    0.438869 |      0.0 |      0.0 |       0.0 |        0.785861 |      0.0 |        0.0 |      0.0 |   0.318812 | 0.318812 | 0.006923 | 0.0 |  0.000144 |  0.024304 |       0.0 |    -2.633789 |     0.785951 |       0.0 |       0.0 | 0.180569 | 0.180569 |  0.213749 |   0.750357 | -12.429991 |   0.819708 |   1.232445 |  16.500769 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  2.2 | 5.550522 | 6.282645 |       0.0 | 0.568249 |  7.742194 |  9.242072 |       0.0 |  0.56755 |  -1.365584 | 70.630889 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-08-12 |  0.0 | -2.9 |             99.1 |       1.8 |                 7.455145 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.455145 |        9.256764 |    0.0 | -2.9 |    1.442864 |           0.638983 |                 4.930236 |                    4.930236 |                      4.97365 |             4.885864 | -0.393288 |  14.336484 |                   1.7664 |                  0.402739 |               15.004299 |                14.641138 |      -13.237899 |       -14.238399 |   0.0 |     0.0 |     25.725 |    0.438848 |      0.0 |      0.0 |       0.0 |        0.765656 |      0.0 |        0.0 |      0.0 |    0.30591 |  0.30591 | 0.007645 | 0.0 |  0.002204 |  0.015125 |       0.0 |    -2.782702 |     0.765729 |       0.0 |       0.0 | 0.187132 | 0.187132 |  0.236032 |    0.46697 | -12.291363 |   0.354517 |   1.182568 |  15.059886 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  2.2 | 5.849306 |    6.275 |       0.0 | 0.743641 |  7.916349 |  9.226947 |       0.0 | 0.742942 |  -7.118735 | 70.628686 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-09-12 |  0.7 | -2.1 |             96.9 |       2.7 |                 7.435268 |              4.4 | 0.0 | 0.0 |                   4.4 |                      7.435268 |       33.405815 |  0.658 | -2.1 |    2.164296 |           0.958474 |                 5.233109 |                    5.233109 |                      4.77755 |             5.070883 | -2.050196 |    7.61832 |                 6.760354 |                  1.524918 |               34.687643 |                16.120955 |      -27.927288 |       -14.596037 | 0.658 |   0.658 |     35.525 |    0.438825 | 0.288747 | 0.288747 | -0.014668 |        0.751475 |      0.0 |        0.0 |      0.0 |   0.429708 | 0.425694 | 0.032679 | 0.0 | -0.052391 | -0.059088 | -0.033425 |    -3.319916 |     0.754202 |       0.0 |       0.0 |  0.22667 |  0.22667 |  1.008896 |  -1.824252 | -27.289714 |  -4.310643 |    1.64647 |   8.461141 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  2.2 | 6.531068 | 6.531068 | -0.015514 | 0.892179 |  8.564686 |  9.655288 |       0.0 | 0.861685 |  -6.153721 | 70.681077 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-10-12 |  3.3 | -3.1 |             95.0 |       3.0 |                  7.41713 |              0.1 | 0.0 | 0.0 |                   0.1 |                       7.41713 |       15.591158 |  3.102 | -3.1 |    2.404774 |           1.064971 |                 4.856984 |                    4.856984 |                     4.708644 |             4.614135 | -1.772272 |  10.446299 |                  3.17831 |                  0.674576 |               17.810271 |                12.864381 |      -14.631961 |       -12.189805 | 3.102 |   3.102 |       24.5 |    0.439593 | 1.363618 | 1.363618 | -0.102256 |        0.749654 |      0.0 |        0.0 |      0.0 |    0.35279 |      0.0 | 0.051278 | 0.0 |  0.107502 |  0.019956 | -0.232614 |    -3.513322 |     0.767026 |       0.0 |       0.0 | 0.274539 | 0.274539 |  1.583124 |   0.616102 | -14.969059 |  -1.530924 |   1.246026 |  11.274983 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  2.2 | 7.843408 | 7.843408 | -1.363795 | 0.913105 | 10.559081 | 11.373714 |       0.0 |  0.73052 |  -8.016687 | 70.573575 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-11-12 |  9.4 |  5.1 |             90.6 |       7.4 |                 7.400749 |              0.2 | 0.0 | 0.0 |                   0.2 |                      7.400749 |       15.891544 |  8.836 |  5.1 |    5.931775 |           2.626929 |                 8.790172 |                    8.790172 |                       6.1078 |             7.963896 | -2.308806 | -13.852836 |                 3.357472 |                  0.722715 |               -0.614453 |               -11.576952 |        3.971925 |        12.299667 | 8.836 |     0.0 |       49.0 |     0.44353 | 3.919033 |      0.0 |  0.833512 |        0.740541 | 9.934016 |   6.740382 | 6.740382 |        0.0 |      0.0 | 0.292189 | 0.0 |  0.472765 | -0.656354 |  -0.56575 |          0.0 |     0.755119 | 12.877382 | 10.559081 |      0.0 |      0.0 |  9.020814 | -20.263798 | -31.635741 | -31.635741 | -26.586852 | -64.199205 | 0.953374 | 0.272677 | 26.881051 | 8.695463 |  0.0 |  0.0 | 0.0 |  0.0 | 8.695463 | 8.695463 |   0.0 |   0.0 |   0.0 |  0.0 | 2.238617 |   0.0 |   0.0 |   0.0 |  0.0 | 2.238617 |  0.02591 |  2.2 | 1.074292 | 1.536237 |       0.0 | 1.025195 |       0.0 |       0.0 |       0.0 | 0.851648 |  14.419482 | 88.286398 | 6.456846 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.02591 |
    | 2010-12-12 |  0.2 |  0.2 |             78.7 |       5.7 |                 7.386143 |              0.7 | 0.0 | 0.0 |                   0.7 |                      7.386143 |       17.823519 |  0.188 |  0.2 |     4.56907 |           2.023445 |                 6.197477 |                    6.197477 |                     5.565605 |             4.877414 |       0.0 |   7.605462 |                 3.942123 |                  0.857985 |               19.469698 |                 9.809681 |      -15.527574 |        -8.951696 | 0.188 |  0.0846 |       49.0 |    0.424609 | 0.079826 | 0.035922 |  0.000323 |        0.728381 | 0.028458 |        0.0 |      0.0 |   0.065172 | 0.065172 | 0.394139 | 0.0 |  0.725773 |   0.06961 |  0.000762 |    -1.267577 |     0.740796 |       0.0 |       0.0 | 0.000719 | 0.000719 | 12.168366 |   6.343759 | -27.443679 |  -7.686453 |   0.252262 |   7.609002 | 0.950212 | 0.270871 |  0.067022 | 0.021807 |  0.0 |  0.0 | 0.0 |  0.0 | 0.021807 | 0.021807 |   0.0 |   0.0 |   0.0 |  0.0 | 3.017016 |   0.0 |   0.0 |   0.0 |  0.0 | 3.017016 | 0.034919 |  2.2 | 0.883584 | 1.193466 |       0.0 | 1.196942 |       0.0 |       0.0 |       0.0 | 1.021868 |  15.397353 |  87.60584 | 3.461637 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.034919 |
    | 2010-13-12 |  2.9 | -3.5 |             90.5 |       2.2 |                 7.373328 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.373328 |        8.941324 |  2.726 | -3.5 |    1.763501 |           0.780979 |                 4.713362 |                    4.713362 |                     4.799895 |             4.265593 |       0.0 |  19.546376 |                 1.964804 |                  0.399209 |               15.898175 |                15.995542 |      -13.933371 |       -15.596333 | 2.726 |   2.726 |       24.5 |     0.42358 |  1.15468 |  1.15468 |  -0.09776 |        0.730139 |      0.0 |        0.0 |      0.0 |   0.140414 | 0.140414 | 0.082949 | 0.0 |  0.179863 |  0.098979 | -0.230795 |    -3.257729 |     0.759589 |       0.0 |       0.0 | 0.032032 |      0.0 |  2.560899 |   3.055801 | -16.049228 |   0.787274 |   0.445042 |  19.439408 | 0.985662 | 0.291459 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.614469 |   0.0 |   0.0 |   0.0 |  0.0 | 1.614469 | 0.018686 |  2.2 | 2.101716 | 2.265198 |       0.0 | 1.170338 |  1.472341 |  1.472341 | -0.123826 | 0.804233 |    4.43431 | 87.425977 | 1.847168 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.018686 |
    | 2010-14-12 |  0.0 | -3.9 |             88.1 |       3.1 |                 7.362317 |              1.2 | 0.0 | 0.0 |                   1.2 |                      7.362317 |       19.668068 |    0.0 | -3.9 |    2.484933 |            1.10047 |                  4.57351 |                     4.57351 |                     4.635498 |             4.029262 |       0.0 |  22.328328 |                 4.495746 |                  0.904881 |                 24.5834 |                 18.55758 |      -20.087654 |       -17.652699 |   0.0 |     0.0 |       24.5 |    0.426796 |      0.0 |      0.0 |       0.0 |        0.719287 |      0.0 |        0.0 |      0.0 |   0.152651 | 0.152651 | 0.116681 | 0.0 |  0.231474 |  0.129971 |       0.0 |    -3.721388 |     0.752267 |       0.0 |       0.0 | 0.034286 |      0.0 |  3.602334 |   4.012627 | -23.099877 |   0.671716 |   0.590111 |  22.337041 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.861499 |   0.0 |   0.0 |   0.0 |  0.0 | 0.861499 | 0.009971 |  2.2 | 2.138244 | 2.148516 |       0.0 |  1.34573 |   1.34237 |   1.34237 |  -0.13254 |  0.88322 |  -9.310684 | 87.194503 | 0.985669 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.009971 |
    | 2010-15-12 |  0.0 | -6.3 |             92.8 |       2.8 |                 7.353123 |              3.0 | 0.0 | 0.0 |                   3.0 |                      7.353123 |       26.831581 |    0.0 | -6.3 |    2.244455 |           0.993973 |                 3.808888 |                    3.808888 |                     3.819843 |             3.534648 | -2.681477 |  21.487389 |                 6.337302 |                  1.265733 |               33.634876 |                20.887899 |      -27.297574 |       -19.622166 |   0.0 |     0.0 |       24.5 |    0.426446 |      0.0 |      0.0 |       0.0 |        0.709944 |      0.0 |        0.0 |      0.0 |   0.191547 | 0.010273 |  0.05613 | 0.0 |  0.056208 |  0.058373 |       0.0 |    -6.262709 |      0.74599 |       0.0 |       0.0 | 0.052606 |      0.0 |   1.73293 |   1.802155 | -28.290032 |   0.133889 |   0.740472 |   21.55821 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.459705 |   0.0 |   0.0 |   0.0 |  0.0 | 0.459705 | 0.005321 |  2.2 | 2.092386 | 2.092386 | -0.700761 | 1.521122 |  1.283997 |  1.283997 | -0.203361 | 0.956641 |  -22.21474 | 87.138295 | 0.525964 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.005321 |
    | 2010-16-12 |  6.7 | -3.5 |             98.1 |       3.7 |                 7.345755 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.345755 |        8.837738 |  6.298 | -3.5 |    2.965888 |           1.313465 |                 4.713362 |                    4.713362 |                      4.08012 |             4.623808 | -6.397845 |  -5.966306 |                 1.939612 |                  0.315596 |               17.117087 |                  5.98469 |      -15.177474 |        -5.669094 | 6.298 |   6.298 |       24.5 |    0.426277 | 2.684694 | 2.684694 | -0.227298 |        0.730476 |      0.0 |        0.0 |      0.0 |   0.220502 |      0.0 | 0.020939 | 0.0 |  0.033418 | -0.127124 | -0.533216 |    -5.403461 |     0.807715 |       0.0 |       0.0 | 0.165588 |      0.0 |  0.646464 |  -3.924729 | -15.899594 |  -7.807127 |  -0.075656 |  -6.062762 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.245304 |   0.0 |   0.0 |   0.0 |  0.0 | 0.245304 | 0.002839 |  2.2 |  4.75614 |  4.75614 | -0.852403 | 1.165301 |  5.024428 |  5.024428 |  -0.64012 | 0.414364 |  -7.665101 | 87.104877 |  0.28066 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002839 |
    | 2010-17-12 |  0.0 | -7.3 |             96.6 |       0.9 |                 7.340222 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.340222 |        8.817416 |    0.0 | -7.3 |    0.721432 |           0.319491 |                  3.52531 |                     3.52531 |                     3.829309 |             3.405449 | -2.207549 |  24.138067 |                 1.961018 |                  0.330952 |                17.20791 |                20.027269 |      -15.246892 |       -19.696317 |   0.0 |     0.0 |       24.5 |    0.434268 |      0.0 |      0.0 |       0.0 |        0.726874 |      0.0 |        0.0 |      0.0 |   0.396187 |      0.0 | 0.017159 | 0.0 |  0.013151 |  0.060678 |       0.0 |     -6.23056 |     0.797895 |       0.0 |       0.0 | 0.195891 |      0.0 |  0.529746 |    1.87333 | -15.097483 |   2.685562 |   0.679155 |  24.255209 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.130897 |   0.0 |   0.0 |   0.0 |  0.0 | 0.130897 | 0.001515 |  2.2 | 4.738981 | 4.738981 | -1.531558 | 1.220283 |   4.96375 |   4.96375 | -0.757262 | 0.479605 | -23.219835 | 87.091725 | 0.149763 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001515 |
    | 2010-18-12 |  0.1 | -5.8 |             97.7 |       1.3 |                  7.33653 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.33653 |        8.804185 |  0.094 | -5.8 |    1.042069 |           0.461488 |                 3.958114 |                    3.958114 |                     3.550629 |             3.867077 | -6.687312 |   3.123354 |                 1.972296 |                  0.342949 |               16.169634 |                 8.951016 |      -14.197337 |        -8.608066 | 0.094 |   0.094 |       24.5 |    0.434217 | 0.040816 | 0.040816 | -0.005727 |         0.72489 |      0.0 |        0.0 |      0.0 |   0.335942 |      0.0 | 0.014212 | 0.0 |  0.012143 |   -0.0494 | -0.013188 |    -7.207871 |     0.790253 |       0.0 |       0.0 | 0.226279 |      0.0 |  0.438755 |  -1.525138 | -14.874714 |  -3.855287 |  -0.238621 |   3.227641 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.069848 |   0.0 |   0.0 |   0.0 |  0.0 | 0.069848 | 0.000808 |  2.2 | 4.765586 | 4.765586 | -1.298664 | 1.251767 |  5.066333 |  5.066333 | -0.874738 |  0.53469 | -17.759856 | 87.079583 | 0.079915 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000808 |
    | 2010-19-12 |  0.0 | -7.9 |             95.8 |       1.9 |                 7.334683 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.334683 |        8.798044 |    0.0 | -7.9 |    1.523023 |           0.674482 |                 3.364296 |                    3.364296 |                     3.377786 |             3.222996 | -5.114838 |  16.403556 |                 1.989245 |                  0.354537 |               19.248216 |                15.848624 |      -17.258971 |       -15.494087 |   0.0 |     0.0 |       24.5 |    0.434297 |      0.0 |      0.0 |       0.0 |        0.722332 |      0.0 |        0.0 |      0.0 |   0.407467 |      0.0 | 0.024803 | 0.0 |  0.043279 |  0.027171 |       0.0 |    -7.848764 |     0.783015 |       0.0 |       0.0 | 0.248825 |      0.0 |  0.765755 |   0.838862 | -17.748227 |   0.157763 |   0.276499 |  16.490713 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.037272 |   0.0 |   0.0 |   0.0 |  0.0 | 0.037272 | 0.000431 |  2.2 | 4.740783 | 4.740783 | -1.575163 | 1.293705 |  5.039162 |  5.039162 | -0.961895 | 0.590788 | -25.580078 | 87.036303 | 0.042644 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000431 |
    | 2010-20-12 |  0.2 | -5.7 |             97.5 |       2.4 |                 7.334683 |              0.2 | 0.0 | 0.0 |                   0.2 |                      7.334683 |       15.464746 |  0.188 | -5.7 |    1.923819 |           0.851977 |                 3.988573 |                    3.988573 |                      3.50984 |             3.888858 | -7.367063 |  -0.062903 |                 3.515347 |                  0.637749 |               19.423467 |                 8.317516 |       -15.90812 |        -7.679767 | 0.188 |   0.188 |       24.5 |    0.434222 | 0.081634 | 0.081634 | -0.011256 |        0.720843 |      0.0 |        0.0 |      0.0 |   0.323065 |      0.0 | 0.019118 | 0.0 |  0.029842 | -0.072667 | -0.025922 |    -7.356579 |     0.777944 |       0.0 |       0.0 | 0.236868 |      0.0 |  0.590226 |  -2.243473 | -16.835878 |  -5.571345 |  -0.337532 |  -0.135051 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.019889 |   0.0 |   0.0 |   0.0 |  0.0 | 0.019889 |  0.00023 |  2.2 | 4.803299 | 4.803299 | -1.248887 | 1.318854 |  5.218195 |  5.218195 | -0.915669 | 0.632595 | -16.933842 | 87.006461 | 0.022755 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00023 |
    | 2010-21-12 |  0.2 | -5.6 |             95.0 |       2.3 |                  7.33653 |              0.8 | 0.0 | 0.0 |                   0.8 |                       7.33653 |       17.879541 |  0.188 | -5.6 |     1.84366 |           0.816478 |                 4.019239 |                    4.019239 |                     3.765689 |             3.818277 | -4.876947 |   9.426214 |                  4.09197 |                  0.753961 |               22.540759 |                13.179295 |      -18.448789 |       -12.425334 | 0.188 |   0.188 |       24.5 |     0.43441 | 0.081669 | 0.081669 | -0.011063 |         0.71894 |      0.0 |        0.0 |      0.0 |   0.351632 |      0.0 | 0.037879 | 0.0 |  0.057855 | -0.009912 | -0.025467 |    -6.447982 |     0.772936 |       0.0 |       0.0 | 0.214835 |      0.0 |  1.169438 |  -0.306022 | -19.518859 |  -2.803738 |   0.099369 |   9.315574 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.010613 |   0.0 |   0.0 |   0.0 |  0.0 | 0.010613 | 0.000123 |  2.2 | 4.847089 | 4.847089 | -1.359319 | 1.351802 |  5.334439 |  5.334439 | -0.830496 | 0.676098 | -17.776723 | 86.948606 | 0.012142 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000123 |
    | 2010-22-12 |  2.4 | -4.8 |             93.8 |       4.6 |                 7.340222 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.340222 |        8.822161 |  2.256 | -4.8 |     3.68732 |           1.632956 |                 4.272186 |                    4.272186 |                     3.934375 |             4.007311 | -5.119696 |   4.552284 |                 1.981596 |                  0.353494 |               19.415762 |                10.319812 |      -17.434166 |        -9.966318 | 2.256 |   2.256 |       24.5 |    0.434541 | 0.980325 | 0.980325 | -0.113827 |        0.724156 |      0.0 |        0.0 |      0.0 |   0.331372 |      0.0 | 0.069651 | 0.0 |  0.133482 | -0.019179 | -0.261947 |     -5.87841 |     0.784244 |       0.0 |       0.0 | 0.243147 |      0.0 |  2.150367 |   -0.59212 | -19.776679 |  -4.974414 |  -0.192146 |   4.399784 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005663 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005663 | 0.000066 |  2.2 | 5.757763 | 5.757763 | -1.280999 | 1.263645 |  6.629293 |  6.629293 | -0.939943 | 0.580974 | -13.745673 | 86.815125 | 0.006479 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000066 |
    | 2010-23-12 |  7.0 | -1.1 |             94.9 |       8.6 |                 7.345755 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.345755 |        8.844396 |   6.58 | -1.1 |    6.893685 |           3.052917 |                 5.634754 |                    5.634754 |                     5.018803 |             5.347381 | -3.958754 |  -7.782537 |                 1.878109 |                  0.300256 |               17.299788 |                 6.815137 |      -15.421679 |        -6.514881 |  6.58 |  5.0995 |     47.775 |    0.437273 | 2.877258 | 2.229875 | -0.093843 |        0.739219 |      0.0 |        0.0 |      0.0 |   0.227608 | 0.227608 | 0.112787 | 0.0 |  0.167586 | -0.128959 | -0.214608 |    -2.661664 |     0.817199 |       0.0 |       0.0 | 0.186004 | 0.186004 |  3.482096 |  -3.981384 | -19.398744 |  -10.75154 |  -0.494969 |  -8.218043 | 0.991528 | 0.294938 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003022 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003022 | 0.000035 |  2.2 | 8.107942 | 8.522235 |       0.0 |  1.04255 |  9.805804 | 10.460993 |       0.0 | 0.356595 |   2.620197 | 86.647539 | 0.003457 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000035 |
    | 2010-24-12 |  0.2 | -2.9 |             85.4 |       7.6 |                 7.353123 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.353123 |        8.873743 |  0.188 | -2.9 |    6.092093 |           2.697927 |                 4.930236 |                    4.930236 |                     4.849838 |             4.210422 |       0.0 |  18.717925 |                 1.967612 |                  0.342891 |               20.121642 |                14.173388 |       -18.15403 |       -13.830497 | 0.188 |   0.188 |     25.725 |    0.445567 | 0.083767 | 0.083767 | -0.005876 |        0.727695 |      0.0 |        0.0 |      0.0 |   0.384253 | 0.384253 | 0.259203 | 0.0 |   0.42204 |  0.230252 | -0.013188 |    -3.119654 |     0.791933 |       0.0 |       0.0 | 0.219077 | 0.219077 |  8.002438 |   7.108622 | -24.676922 |  -1.387486 |   1.479546 |  19.551633 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001613 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001613 | 0.000019 |  2.2 | 8.317664 | 8.346798 |       0.0 | 1.207507 |  9.908367 | 10.334975 |       0.0 | 0.522234 |  -7.514394 | 86.225499 | 0.001845 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000019 |
    | 2010-25-12 |  0.0 | -9.1 |             87.2 |       3.8 |                 7.362317 |              3.2 | 0.0 | 0.0 |                   3.2 |                      7.362317 |       27.759642 |    0.0 | -9.1 |    3.046047 |           1.348964 |                 3.061705 |                    3.061705 |                     3.380113 |             2.669806 | -2.164145 |  34.054801 |                 6.392673 |                  1.182881 |               39.487127 |                26.840603 |      -33.094453 |       -25.657722 |   0.0 |     0.0 |       24.5 |     0.44504 |      0.0 |      0.0 |       0.0 |        0.717192 |      0.0 |        0.0 |      0.0 |    0.88702 | 0.029135 | 0.092902 | 0.0 |  0.149835 |  0.168382 |       0.0 |    -7.839946 |     0.770553 |       0.0 |       0.0 | 0.528014 | 0.426608 |  2.868178 |   5.198506 | -32.533642 |   5.239736 |   3.428989 |  36.095964 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00086 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00086 |  0.00001 |  2.2 | 8.253897 | 8.253897 | -3.316362 | 1.382899 | 10.166593 | 10.166593 | -0.392009 | 0.697626 | -32.985862 | 86.075664 | 0.000984 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00001 |
    | 2010-26-12 |  1.2 | -8.6 |             92.7 |       1.5 |                 7.373328 |              4.3 | 0.0 | 0.0 |                   4.3 |                      7.373328 |       32.328755 |  1.128 | -8.6 |    1.202387 |           0.532486 |                 3.184716 |                    3.184716 |                     2.763681 |             2.952232 | -9.499928 |   5.337832 |                 7.386314 |                  1.386953 |               35.492318 |                14.860836 |      -28.106004 |       -13.473883 | 1.128 |   1.128 |       24.5 |    0.444762 | 0.501691 | 0.501691 | -0.104368 |        0.719417 |      0.0 |        0.0 |      0.0 |    1.02325 |      0.0 | 0.037798 | 0.0 | -0.044608 | -0.030655 | -0.234661 |   -10.389567 |     0.768992 |       0.0 |       0.0 | 0.701674 |      0.0 |   1.16695 |  -0.946426 | -28.738067 |  -5.103804 |   0.534888 |   7.423653 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000459 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000459 | 0.000005 |  2.2 |  8.71779 |  8.71779 | -3.955618 | 1.343461 | 10.823557 | 10.823557 |  -2.71249 | 0.712034 | -29.740361 | 86.120272 | 0.000525 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000005 |
    | 2010-27-12 |  0.9 | -5.3 |             97.8 |       1.3 |                 7.386143 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.386143 |        9.004582 |  0.846 | -5.3 |    1.042069 |           0.461488 |                 4.112496 |                    4.112496 |                     3.406464 |             4.022021 | -8.565224 |  -4.948671 |                 2.046813 |                    0.3835 |                15.02886 |                 4.377783 |      -12.982048 |        -3.994283 | 0.846 |   0.846 |       24.5 |    0.446153 | 0.377446 | 0.377446 | -0.048391 |         0.72085 |      0.0 |        0.0 |      0.0 |   0.614373 |      0.0 | 0.014124 | 0.0 |   0.01334 | -0.096093 | -0.108462 |    -7.740445 |     0.770672 |       0.0 |       0.0 | 0.546942 |      0.0 |  0.436048 |  -2.966706 | -15.047098 |  -6.682867 |  -1.629003 |   -5.65529 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000245 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000245 | 0.000003 |  2.2 | 9.081112 | 9.081112 | -2.375006 | 1.318735 | 11.388204 | 11.388204 | -2.114334 | 0.696539 | -16.208356 | 86.106931 |  0.00028 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000003 |
    | 2010-28-12 |  0.0 | -7.7 |             95.4 |       3.0 |                 7.400749 |              3.4 | 0.0 | 0.0 |                   3.4 |                      7.400749 |       28.982321 |    0.0 | -7.7 |    2.404774 |           1.064971 |                 3.417227 |                    3.417227 |                     3.371846 |             3.260035 | -4.668007 |  19.219765 |                 6.646645 |                  1.265126 |               36.290071 |                20.444526 |      -29.643426 |         -19.1794 |   0.0 |     0.0 |       24.5 |    0.447243 |      0.0 |      0.0 |       0.0 |        0.718361 |      0.0 |        0.0 |      0.0 |   0.891611 |      0.0 | 0.033192 | 0.0 |  0.001185 |  0.023609 |       0.0 |    -7.871301 |     0.764953 |       0.0 |       0.0 | 0.560921 |      0.0 |  1.024731 |   0.728896 | -29.596426 |   -0.63449 |   1.071731 |  19.273806 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000131 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000131 | 0.000002 |  2.2 |  9.04792 |  9.04792 | -3.446737 | 1.362006 | 11.364595 | 11.364595 | -2.168375 | 0.750461 | -26.844788 | 86.105746 |  0.00015 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000002 |
    | 2010-29-12 |  0.0 | -9.2 |             94.9 |       2.7 |                  7.41713 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.41713 |        9.127645 |    0.0 | -9.2 |    2.164296 |           0.958474 |                 3.037615 |                    3.037615 |                     2.925149 |             2.882696 | -7.731299 |  11.670462 |                 2.106037 |                  0.407418 |               21.481317 |                13.988897 |      -19.375281 |       -13.581479 |   0.0 |     0.0 |       24.5 |    0.447144 |      0.0 |      0.0 |       0.0 |        0.716645 |      0.0 |        0.0 |      0.0 |   0.849376 |      0.0 | 0.031207 | 0.0 |  0.064362 |  0.008552 |       0.0 |    -9.676376 |     0.759655 |       0.0 |       0.0 | 0.688124 |      0.0 |  0.963447 |   0.264019 | -20.501997 |  -1.683303 |  -0.163269 |  12.162196 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00007 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00007 | 0.000001 |  2.2 | 9.016714 | 9.016714 | -3.283468 | 1.392793 | 11.356043 | 11.356043 | -2.660108 |  0.80356 | -29.931917 | 86.041384 |  0.00008 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000001 |
    | 2010-30-12 |  0.2 | -7.4 |             96.0 |       1.4 |                 7.435268 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.435268 |        9.199989 |  0.188 | -7.4 |    1.122228 |           0.496987 |                 3.498009 |                    3.498009 |                     3.078393 |             3.358088 | -8.620392 |   2.464458 |                 2.131208 |                   0.41578 |               17.732181 |                 8.625328 |      -15.600973 |        -8.209548 | 0.188 |   0.188 |       24.5 |     0.44705 | 0.084045 | 0.084045 | -0.015045 |        0.715513 |      0.0 |        0.0 |      0.0 |   0.760586 |      0.0 | 0.022296 | 0.0 |  0.035231 | -0.044568 | -0.033653 |    -9.031135 |      0.75665 |       0.0 |       0.0 |  0.64763 |      0.0 |   0.68834 |  -1.375968 | -16.647598 |  -4.559314 |  -0.358285 |   2.274266 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000037 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000037 |      0.0 |  2.2 | 9.078463 | 9.078463 | -2.940227 | 1.413544 | 11.504566 | 11.504566 | -2.503568 | 0.835128 | -23.813042 | 86.006153 | 0.000043 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-31-12 |  0.9 |  1.7 |             96.5 |       4.2 |                 7.455145 |              4.0 | 0.0 | 0.0 |                   4.0 |                      7.455145 |       32.062235 |  0.846 |  1.7 |    3.366683 |            1.49096 |                 6.907942 |                    6.907942 |                     5.142434 |             6.666164 | -6.858156 |  -27.13824 |                 7.476212 |                   1.47449 |               32.041784 |                  2.05765 |      -24.565572 |        -0.583161 | 0.846 | 0.06345 |       49.0 |    0.447235 | 0.378361 | 0.028377 |  0.029998 |        0.713641 |      0.0 |        0.0 |      0.0 |   0.290571 | 0.290571 | 0.060446 | 0.0 |  0.027375 | -0.380944 |  0.067075 |    -2.335116 |      0.75237 |       0.0 |       0.0 | 0.181279 | 0.181279 |  1.866174 | -11.760972 | -28.218705 | -17.696145 |  -1.786959 | -28.873957 | 0.961493 | 0.277341 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00002 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00002 |      0.0 |  2.2 | 9.337344 | 9.396378 |       0.0 | 1.448677 | 12.093865 | 12.353148 |       0.0 | 0.882053 |  11.908531 | 85.978778 | 0.000023 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_ic_acker_winter_ktschnee_daily:

acre (snow surface temperature)
-------------------------------

.. integration-test::

    >>> lnk(ACKER)
    >>> ktschnee(5.0)
    >>> conditions_acker_winter_ktschnee = test(
    ...     "lland_knauf_ic_acker_winter_ktschnee_daily",
    ...     axis1=(inputs.nied, fluxes.wada), axis2=(states.waes, states.wats),
    ...     get_conditions="2010-12-10")
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow |   nbes |    sbes | snowintmax | snowintrate | nbesinz | sbesinz | wniedinz | actualalbedoinz | wadainz | schmpotinz | schminz | gefrpotinz | gefrinz | evsinz |  evi |      evb |       evs |     wnied | tempssurface | actualalbedo |   schmpot |      schm |  gefrpot |     gefr | wlatinz |   wlatsnow | wsensinz |  wsenssnow | wsurfinz |       wsurf |      sff |      fvg |      wada |       qdb | qib1 | qib2 | qbb | qkap |      qdgz |     qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga |      qah |       qa | inzp | stinz | sinz | esnowinz | asinz |      wats |      waes |     esnow |     taus |       ebdn |      bowa |     sdg1 | sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-01-12 |  0.0 | -5.8 |             70.3 |       6.9 |                 7.641029 |              3.5 | 0.0 | 0.0 |                   3.5 |                      7.641029 |       31.942798 |    0.0 | -5.8 |     5.53098 |            5.53098 |                 3.958114 |                         0.0 |                          0.0 |             2.782554 |       0.0 |       34.8 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |  0.55374 |       0.0 |       0.0 |          nan |          nan |       0.0 |       0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |       0.0 |      nan | -26.216667 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-02-12 | 11.1 | -3.0 |             85.0 |       5.2 |                 7.609644 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.609644 |        9.873898 | 10.434 | -3.0 |    4.168274 |           4.168274 |                 4.893489 |                         0.0 |                     3.874639 |             4.159465 |   -7.5504 |  -4.536126 |                      0.0 |                  1.229506 |                     0.0 |                28.671763 |             0.0 |       -27.442257 | 10.374 |  10.374 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.140764 | -0.752835 |     -6.07759 |     0.875479 |       0.0 |       0.0 | 0.441058 |      0.0 |     0.0 |  -4.345837 |      0.0 | -26.680365 |      0.0 |   -3.583945 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 10.514764 | 10.514764 | -1.705016 | 0.084415 | -13.097208 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-03-12 |  0.4 | -5.6 |             95.9 |       2.0 |                 7.579824 |              0.1 | 0.0 | 0.0 |                   0.1 |                      7.579824 |         16.6835 |  0.376 | -5.6 |    1.603182 |           1.603182 |                 4.019239 |                         0.0 |                     3.243359 |              3.85445 | -3.771996 |  12.646472 |                      0.0 |                  2.311234 |                     0.0 |                32.295147 |             0.0 |       -29.983913 |  0.376 |   0.376 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.159032 | -0.050934 |    -8.367644 |     0.861466 |       0.0 |       0.0 | 0.400697 |      0.0 |     0.0 |  -4.909851 |      0.0 | -12.634548 |      0.0 |   12.439513 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 11.049796 | 11.049796 | -1.548992 | 0.139381 | -17.160346 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-04-12 |  1.2 | -3.8 |             90.6 |       4.3 |                 7.551596 |              3.7 | 0.0 | 0.0 |                   3.7 |                      7.551596 |       31.807427 |  1.128 | -3.8 |    3.446842 |           3.446842 |                 4.608124 |                         0.0 |                     3.312979 |              4.17496 |  -4.94218 |     8.4831 |                      0.0 |                  4.758969 |                     0.0 |                57.151182 |             0.0 |       -52.392213 |  1.128 |   1.128 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.369279 | -0.103687 |    -8.096584 |     0.850382 |       0.0 |       0.0 | 0.484345 |      0.0 |     0.0 | -11.400835 |      0.0 | -32.288606 |      0.0 |    8.702772 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 12.547075 | 12.547075 | -1.872351 | 0.186969 | -17.060113 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-05-12 |  6.4 |  0.2 |             96.5 |       4.0 |                 7.524989 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.524989 |        9.532511 |  6.016 |  0.2 |    3.206365 |           3.206365 |                 6.197477 |                         0.0 |                     5.231858 |             5.980565 | -4.913313 |   -7.71462 |                      0.0 |                  1.485816 |                     0.0 |                19.682213 |             0.0 |       -18.196397 |  6.016 |  2.7072 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.304329 |  0.045165 |     -2.10322 |     0.844132 |       0.0 |       0.0 |  0.49672 |  0.49672 |     0.0 |  -9.395626 |      0.0 | -16.422384 |      0.0 |   -7.621612 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 16.009222 | 18.867404 |       0.0 | 0.215598 |   -0.76216 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-06-12 |  0.0 | -1.1 |             99.0 |       2.4 |                 7.500028 |              0.1 | 0.0 | 0.0 |                   0.1 |                      7.500028 |       16.141484 |    0.0 | -1.1 |    1.923819 |           1.923819 |                 5.634754 |                         0.0 |                      4.93464 |             5.578406 | -0.219502 |   6.885814 |                      0.0 |                  3.043358 |                     0.0 |                25.481561 |             0.0 |       -22.438203 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.186363 |       0.0 |    -2.888059 |     0.811457 |       0.0 |       0.0 | 0.185939 | 0.185939 |     0.0 |   -5.75364 |      0.0 |  -9.079959 |      0.0 |    7.604604 | 0.996975 | 0.298188 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 16.355129 | 19.053767 |       0.0 |  0.39099 |   0.935359 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-07-12 |  0.0 | -2.9 |             99.2 |       1.9 |                 7.476739 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.476739 |        9.341429 |    0.0 | -2.9 |    1.523023 |           1.523023 |                 4.930236 |                         0.0 |                     4.296908 |             4.890794 |       0.0 |  12.295456 |                      0.0 |                  1.998128 |                     0.0 |                28.102997 |             0.0 |       -26.104869 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.150213 |       0.0 |    -4.724078 |       0.7861 |       0.0 |       0.0 | 0.279046 | 0.279046 |     0.0 |  -4.637556 |      0.0 |  -8.093137 |      0.0 |   13.374176 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 16.765312 |  19.20398 |       0.0 | 0.566382 |  -2.776764 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-08-12 |  0.0 | -2.9 |             99.1 |       1.8 |                 7.455145 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.455145 |        9.256764 |    0.0 | -2.9 |    1.442864 |           1.442864 |                 4.930236 |                         0.0 |                     4.242152 |             4.885864 | -0.799708 |  10.437477 |                      0.0 |                  2.167471 |                     0.0 |                27.402135 |             0.0 |       -25.234664 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.158109 |       0.0 |     -4.89276 |      0.76585 |       0.0 |       0.0 | 0.344018 | 0.344018 |     0.0 |  -4.881333 |      0.0 |   -8.58597 |      0.0 |   11.767361 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 17.250194 | 19.362088 |       0.0 | 0.741774 |  -4.630907 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-09-12 |  0.7 | -2.1 |             96.9 |       2.7 |                 7.435268 |              4.4 | 0.0 | 0.0 |                   4.4 |                      7.435268 |       33.405815 |  0.658 | -2.1 |    2.164296 |           2.164296 |                 5.233109 |                         0.0 |                     3.812253 |             5.070883 | -1.333701 |  12.531717 |                      0.0 |                    8.2195 |                     0.0 |                57.533285 |             0.0 |       -49.313785 |  0.658 |   0.658 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.391967 | -0.033425 |    -6.288533 |      0.75395 |       0.0 |       0.0 | 0.474106 | 0.474106 |     0.0 | -12.101292 |      0.0 |  -22.88143 |      0.0 |   14.331063 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 18.742201 | 20.412055 |       0.0 | 0.864457 |   -8.57929 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-10-12 |  3.3 | -3.1 |             95.0 |       3.0 |                  7.41713 |              0.1 | 0.0 | 0.0 |                   0.1 |                       7.41713 |       15.591158 |  3.102 | -3.1 |    2.404774 |           2.404774 |                 4.856984 |                         0.0 |                     4.121719 |             4.614135 | -2.470836 |   6.577217 |                      0.0 |                  3.663893 |                     0.0 |                29.943065 |             0.0 |       -26.279171 |  3.102 |   3.102 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.164151 | -0.232614 |    -5.270657 |     0.765002 |       0.0 |       0.0 | 0.562244 | 0.562244 |     0.0 |  -5.067866 |      0.0 | -12.693215 |      0.0 |    8.518091 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 22.562863 | 23.678206 |       0.0 | 0.749982 |  -6.573175 |  71.44626 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-11-12 |  9.4 |  5.1 |             90.6 |       7.4 |                 7.400749 |              0.2 | 0.0 | 0.0 |                   0.2 |                      7.400749 |       15.891544 |  8.836 |  5.1 |    5.931775 |           5.931775 |                 8.790172 |                         0.0 |                       6.1078 |             7.963896 | -1.893074 | -11.358446 |                      0.0 |                  4.012333 |                     0.0 |                -0.754591 |             0.0 |         4.766924 |  8.836 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.06 | 0.338985 | -1.215856 |  2.122093 |          0.0 |     0.747518 | 23.713776 | 22.562863 |      0.0 |      0.0 |     0.0 | -37.537452 |      0.0 | -58.603285 |      0.0 | -100.907662 | 0.956772 | 0.274624 | 33.730062 | 11.067544 |  0.0 |  0.0 | 0.0 |  0.0 | 11.067544 | 11.067544 |   0.0 |   0.0 |   0.0 |  0.0 | 2.849301 |   0.0 |   0.0 |   0.0 |  0.0 | 2.849301 | 0.032978 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |       0.0 |  0.93825 |  13.368604 | 93.769793 | 8.218243 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.032978 |
    | 2010-12-12 |  0.2 |  0.2 |             78.7 |       5.7 |                 7.386143 |              0.7 | 0.0 | 0.0 |                   0.7 |                      7.386143 |       17.823519 |  0.188 |  0.2 |     4.56907 |            4.56907 |                 6.197477 |                         0.0 |                     5.049902 |             4.877414 |       0.0 |   7.031308 |                      0.0 |                   4.73515 |                     0.0 |                34.816134 |             0.0 |       -30.080984 |  0.128 |  0.0576 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.06 | 0.519628 |  0.082368 |  0.000519 |    -2.578858 |     0.734331 |       0.0 |       0.0 | 0.000786 | 0.000786 |     0.0 |   2.826474 |      0.0 | -25.872591 |      0.0 |    7.034866 | 0.951754 | 0.271751 |  0.045632 |  0.015062 |  0.0 |  0.0 | 0.0 |  0.0 |  0.015062 |  0.015062 |   0.0 |   0.0 |   0.0 |  0.0 | 3.836775 |   0.0 |   0.0 |   0.0 |  0.0 | 3.836775 | 0.044407 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |       0.0 | 1.109403 |  14.920629 | 93.280735 | 4.396529 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.044407 |
    | 2010-13-12 |  2.9 | -3.5 |             90.5 |       2.2 |                 7.373328 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.373328 |        8.941324 |  2.726 | -3.5 |    1.763501 |           1.763501 |                 4.713362 |                         0.0 |                     3.977478 |             4.265593 |       0.0 |  15.675798 |                      0.0 |                  2.222307 |                     0.0 |                31.068601 |             0.0 |       -28.846294 |  2.666 |   2.666 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.079193 | -0.225715 |    -5.736347 |     0.751457 |       0.0 |       0.0 | 0.043585 |      0.0 |     0.0 |  -2.444947 |      0.0 | -10.782775 |      0.0 |   15.618572 | 0.974687 | 0.285005 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 2.050493 |   0.0 |   0.0 |   0.0 |  0.0 | 2.050493 | 0.023733 | 0.06 |   0.0 |  0.0 |      0.0 |   nan |  2.745193 |  2.745193 | -0.168489 | 0.892385 |   7.828164 | 93.280735 | 2.346037 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.023733 |
    | 2010-14-12 |  0.0 | -3.9 |             88.1 |       3.1 |                 7.362317 |              1.2 | 0.0 | 0.0 |                   1.2 |                      7.362317 |       19.668068 |    0.0 | -3.9 |    2.484933 |           2.484933 |                  4.57351 |                         0.0 |                     3.681219 |             4.029262 |       0.0 |  18.367591 |                      0.0 |                  5.038413 |                     0.0 |                44.083316 |             0.0 |       -39.044903 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.118568 |       0.0 |    -6.741743 |     0.743828 |       0.0 |       0.0 | 0.052586 |      0.0 |     0.0 |  -3.660568 |      0.0 | -16.981948 |      0.0 |   18.402387 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.094166 |   0.0 |   0.0 |   0.0 |  0.0 | 1.094166 | 0.012664 | 0.06 |   0.0 |  0.0 |      0.0 |   nan |  2.863761 |  2.863761 | -0.203285 |  0.98328 |  -1.956093 | 93.280735 | 1.251871 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.012664 |
    | 2010-15-12 |  0.0 | -6.3 |             92.8 |       2.8 |                 7.353123 |              3.0 | 0.0 | 0.0 |                   3.0 |                      7.353123 |       26.831581 |    0.0 | -6.3 |    2.244455 |           2.244455 |                 3.808888 |                         0.0 |                     2.883255 |             3.534648 | -0.563355 |  25.279421 |                      0.0 |                  7.049141 |                     0.0 |                58.759349 |             0.0 |       -51.710208 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.207622 |       0.0 |    -9.857997 |     0.737282 |       0.0 |       0.0 | 0.085596 |      0.0 |     0.0 |   -6.40996 |      0.0 |  -19.89322 |      0.0 |   25.407028 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.583859 |   0.0 |   0.0 |   0.0 |  0.0 | 0.583859 | 0.006758 | 0.06 |   0.0 |  0.0 |      0.0 |   nan |  3.071382 |  3.071382 | -0.330893 | 1.068532 |  -18.65218 | 93.280735 | 0.668012 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.006758 |
    | 2010-16-12 |  6.7 | -3.5 |             98.1 |       3.7 |                 7.345755 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.345755 |        8.837738 |  6.298 | -3.5 |    2.965888 |           2.965888 |                 4.713362 |                         0.0 |                     3.936324 |             4.623808 | -5.371828 |   1.147589 |                      0.0 |                  1.775552 |                     0.0 |                27.481485 |             0.0 |       -25.705933 |  6.298 |   6.298 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.264364 | -0.533216 |    -5.871957 |     0.799094 |       0.0 |       0.0 | 0.326157 |      0.0 |     0.0 |  -8.161769 |      0.0 | -15.999844 |      0.0 |     1.54432 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.311554 |   0.0 |   0.0 |   0.0 |  0.0 | 0.311554 | 0.003606 | 0.06 |   0.0 |  0.0 |      0.0 |   nan |  9.633746 |  9.633746 | -1.260839 | 0.471314 | -11.216436 | 93.280735 | 0.356459 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.003606 |
    | 2010-17-12 |  0.0 | -7.3 |             96.6 |       0.9 |                 7.340222 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.340222 |        8.817416 |    0.0 | -7.3 |    0.721432 |           0.721432 |                  3.52531 |                         0.0 |                     2.799842 |             3.405449 | -3.230334 |  18.956495 |                      0.0 |                   1.85065 |                     0.0 |                33.629061 |             0.0 |       -31.778411 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.108899 |       0.0 |   -10.226692 |     0.790114 |       0.0 |       0.0 | 0.385193 |      0.0 |     0.0 |  -3.362059 |      0.0 |  -9.231639 |      0.0 |   19.184713 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.166248 |   0.0 |   0.0 |   0.0 |  0.0 | 0.166248 | 0.001924 | 0.06 |   0.0 |  0.0 |      0.0 |   nan |  9.742645 |  9.742645 | -1.489058 |  0.53573 | -21.589597 | 93.280735 |  0.19021 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001924 |
    | 2010-18-12 |  0.1 | -5.8 |             97.7 |       1.3 |                  7.33653 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.33653 |        8.804185 |  0.094 | -5.8 |    1.042069 |           1.042069 |                 3.958114 |                         0.0 |                     3.032971 |             3.867077 | -6.217804 |   8.024872 |                      0.0 |                  1.910008 |                     0.0 |                28.153948 |             0.0 |        -26.24394 |  0.094 |   0.094 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.174381 | -0.013188 |    -9.219359 |     0.783057 |       0.0 |       0.0 | 0.465048 |      0.0 |     0.0 |  -5.383712 |      0.0 | -12.539848 |      0.0 |     8.32038 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.088712 |   0.0 |   0.0 |   0.0 |  0.0 | 0.088712 | 0.001027 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 10.011026 | 10.011026 | -1.797754 | 0.590452 | -21.031136 | 93.280735 | 0.101498 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001027 |
    | 2010-19-12 |  0.0 | -7.9 |             95.8 |       1.9 |                 7.334683 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.334683 |        8.798044 |    0.0 | -7.9 |    1.523023 |           1.523023 |                 3.364296 |                         0.0 |                     2.624563 |             3.222996 | -6.056967 |  13.458957 |                      0.0 |                   1.96684 |                     0.0 |                 34.21778 |             0.0 |        -32.25094 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.151363 |       0.0 |   -11.034367 |     0.776446 |       0.0 |       0.0 | 0.519952 |      0.0 |     0.0 |  -4.673061 |      0.0 | -13.906677 |      0.0 |   13.671202 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.047338 |   0.0 |   0.0 |   0.0 |  0.0 | 0.047338 | 0.000548 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 10.162388 | 10.162388 | -2.009999 | 0.645375 |  -25.90676 | 93.280735 | 0.054161 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000548 |
    | 2010-20-12 |  0.2 | -5.7 |             97.5 |       2.4 |                 7.334683 |              0.2 | 0.0 | 0.0 |                   0.2 |                      7.334683 |       15.464746 |  0.188 | -5.7 |    1.923819 |           1.923819 |                 3.988573 |                         0.0 |                     3.098752 |             3.888858 | -7.461147 |    4.05317 |                      0.0 |                  3.527553 |                     0.0 |                31.135077 |             0.0 |       -27.607524 |  0.188 |   0.188 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.228727 | -0.025922 |    -8.947567 |     0.771897 |       0.0 |       0.0 | 0.526991 |      0.0 |     0.0 |  -7.061557 |      0.0 | -16.491506 |      0.0 |    4.054461 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02526 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02526 | 0.000292 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 10.579116 | 10.579116 | -2.037212 | 0.685419 | -21.376597 | 93.280735 | 0.028901 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000292 |
    | 2010-21-12 |  0.2 | -5.6 |             95.0 |       2.3 |                  7.33653 |              0.8 | 0.0 | 0.0 |                   0.8 |                       7.33653 |       17.879541 |  0.188 | -5.6 |     1.84366 |            1.84366 |                 4.019239 |                         0.0 |                     3.035378 |             3.818277 |  -6.15646 |   8.378078 |                      0.0 |                  4.156016 |                     0.0 |                  37.1246 |             0.0 |       -32.968584 |  0.188 |   0.188 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.220917 | -0.025467 |    -9.209318 |     0.767555 |       0.0 |       0.0 | 0.508871 |      0.0 |     0.0 |  -6.820417 |      0.0 | -17.865606 |      0.0 |    8.282561 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.013479 |   0.0 |   0.0 |   0.0 |  0.0 | 0.013479 | 0.000156 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 10.988032 | 10.988032 | -1.967163 |  0.72551 | -21.171341 | 93.280735 | 0.015422 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000156 |
    | 2010-22-12 |  2.4 | -4.8 |             93.8 |       4.6 |                 7.340222 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.340222 |        8.822161 |  2.256 | -4.8 |     3.68732 |            3.68732 |                 4.272186 |                         0.0 |                     3.528908 |             4.007311 | -6.097346 |   3.298469 |                      0.0 |                  1.944393 |                     0.0 |                  31.4403 |             0.0 |       -29.495907 |  2.256 |   2.256 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.215444 | -0.261947 |     -7.28687 |     0.779601 |       0.0 |       0.0 | 0.550873 |      0.0 |     0.0 |   -6.65146 |      0.0 | -19.645554 |      0.0 |    3.198893 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007193 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007193 | 0.000083 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 13.459476 | 13.459476 | -2.129533 | 0.618694 | -15.886477 | 93.280735 | 0.008229 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000083 |
    | 2010-23-12 |  7.0 | -1.1 |             94.9 |       8.6 |                 7.345755 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.345755 |        8.844396 |   6.58 | -1.1 |    6.893685 |           6.893685 |                 5.634754 |                         0.0 |                     4.983706 |             5.347381 | -4.575305 |  -4.758233 |                      0.0 |                  1.641862 |                     0.0 |                26.421965 |             0.0 |       -24.780102 |   6.58 |  5.0995 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.270137 | -0.214608 |    -2.755662 |     0.814361 |       0.0 |       0.0 | 0.509438 | 0.509438 |     0.0 |  -8.340017 |      0.0 | -21.573106 |      0.0 |   -5.133021 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003838 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003838 | 0.000044 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 19.325461 | 20.309613 |       0.0 | 0.373379 |  -2.544911 | 93.280735 | 0.004391 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000044 |
    | 2010-24-12 |  0.2 | -2.9 |             85.4 |       7.6 |                 7.353123 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.353123 |        8.873743 |  0.188 | -2.9 |    6.092093 |           6.092093 |                 4.930236 |                         0.0 |                     4.287029 |             4.210422 | -0.732934 |  10.281215 |                      0.0 |                  1.867694 |                     0.0 |                33.607237 |             0.0 |       -31.739544 |  0.188 |   0.188 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 |  0.051303 | -0.013188 |     -4.75437 |     0.789526 |       0.0 |       0.0 | 0.328908 | 0.328908 |     0.0 |   1.583889 |      0.0 | -21.783933 |      0.0 |     11.5395 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002048 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002048 | 0.000024 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 19.792706 | 20.446311 |       0.0 | 0.540148 |  -4.242792 | 93.280735 | 0.002343 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000024 |
    | 2010-25-12 |  0.0 | -9.1 |             87.2 |       3.8 |                 7.362317 |              3.2 | 0.0 | 0.0 |                   3.2 |                      7.362317 |       27.759642 |    0.0 | -9.1 |    3.046047 |           3.046047 |                 3.061705 |                         0.0 |                     2.334119 |             2.669806 | -1.221924 |  29.029614 |                      0.0 |                  6.423146 |                     0.0 |                65.873313 |             0.0 |       -59.450167 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.131539 |       0.0 |   -12.484807 |     0.768616 |       0.0 |       0.0 | 0.800223 | 0.653605 |     0.0 |  -4.061041 |      0.0 |  -23.26606 |      0.0 |   32.123067 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001093 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001093 | 0.000013 | 0.06 |   0.0 |  0.0 |      0.0 |   nan |  20.57785 |  20.57785 | -0.566787 |  0.71554 | -24.689073 | 93.280735 |  0.00125 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000013 |
    | 2010-26-12 |  1.2 | -8.6 |             92.7 |       1.5 |                 7.373328 |              4.3 | 0.0 | 0.0 |                   4.3 |                      7.373328 |       32.328755 |  1.128 | -8.6 |    1.202387 |           1.202387 |                 3.184716 |                         0.0 |                     1.979012 |             2.952232 | -7.110453 |  17.693718 |                      0.0 |                  7.584795 |                     0.0 |                59.607694 |             0.0 |       -52.022899 |  1.128 |   1.128 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.217696 | -0.234661 |   -14.494219 |     0.765385 |       0.0 |       0.0 | 1.366309 |      0.0 |     0.0 |  -6.720982 |      0.0 |  -23.12785 |      0.0 |   22.174067 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000583 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000583 | 0.000007 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 21.923545 | 21.923545 | -5.281797 |  0.74626 | -33.799457 | 93.280735 | 0.000667 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000007 |
    | 2010-27-12 |  0.9 | -5.3 |             97.8 |       1.3 |                 7.386143 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.386143 |        9.004582 |  0.846 | -5.3 |    1.042069 |           1.042069 |                 4.112496 |                         0.0 |                     2.963217 |             4.022021 | -9.734244 |  -0.558819 |                      0.0 |                  2.096939 |                     0.0 |                23.743729 |             0.0 |       -21.646791 |  0.846 |   0.846 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.221357 | -0.108462 |    -9.513339 |     0.767125 |       0.0 |       0.0 | 1.373666 |      0.0 |     0.0 |  -6.834013 |      0.0 | -15.451619 |      0.0 |   -0.638841 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000311 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000311 | 0.000004 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 22.990902 | 22.990902 | -5.310236 | 0.729578 | -24.657306 | 93.280735 | 0.000356 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000004 |
    | 2010-28-12 |  0.0 | -7.7 |             95.4 |       3.0 |                 7.400749 |              3.4 | 0.0 | 0.0 |                   3.4 |                      7.400749 |       28.982321 |    0.0 | -7.7 |    2.404774 |           2.404774 |                 3.417227 |                         0.0 |                     2.358487 |             3.260035 | -7.101304 |  14.350787 |                      0.0 |                  6.889543 |                     0.0 |                57.721392 |             0.0 |       -50.831849 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.300538 |       0.0 |   -12.357122 |     0.762285 |       0.0 |       0.0 | 1.365727 |      0.0 |     0.0 |  -9.278588 |      0.0 | -27.233162 |      0.0 |   14.320099 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000166 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000166 | 0.000002 | 0.06 |   0.0 |  0.0 |      0.0 |   nan |  23.29144 |  23.29144 | -5.279548 | 0.776804 |  -30.42476 | 93.280735 |  0.00019 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000002 |
    | 2010-29-12 |  0.0 | -9.2 |             94.9 |       2.7 |                  7.41713 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.41713 |        9.127645 |    0.0 | -9.2 |    2.164296 |           2.164296 |                 3.037615 |                         0.0 |                     2.327418 |             2.882696 | -8.762331 |   9.927002 |                      0.0 |                  2.212186 |                     0.0 |                36.204868 |             0.0 |       -33.992682 |    0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.172927 |       0.0 |   -12.520123 |     0.757639 |       0.0 |       0.0 |  1.51821 |      0.0 |     0.0 |  -5.338809 |      0.0 | -18.137414 |      0.0 |   10.516459 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000089 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000089 | 0.000001 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 23.464367 | 23.464367 | -5.869005 | 0.824618 | -31.768428 | 93.280735 | 0.000101 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000001 |
    | 2010-30-12 |  0.2 | -7.4 |             96.0 |       1.4 |                 7.435268 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.435268 |        9.199989 |  0.188 | -7.4 |    1.122228 |           1.122228 |                 3.498009 |                         0.0 |                     2.536678 |             3.358088 | -9.149307 |   6.330234 |                      0.0 |                  2.254799 |                     0.0 |                29.408579 |             0.0 |       -27.153781 |  0.188 |   0.188 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.177732 | -0.033653 |   -11.457537 |     0.754913 |       0.0 |       0.0 | 1.510287 |      0.0 |     0.0 |   -5.48718 |      0.0 | -15.400644 |      0.0 |    6.265956 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000047 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000047 | 0.000001 | 0.06 |   0.0 |  0.0 |      0.0 |   nan |   23.8301 |   23.8301 |  -5.83838 | 0.853891 | -29.515329 | 93.280735 | 0.000054 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000001 |
    | 2010-31-12 |  0.9 |  1.7 |             96.5 |       4.2 |                 7.455145 |              4.0 | 0.0 | 0.0 |                   4.0 |                      7.455145 |       32.062235 |  0.846 |  1.7 |    3.366683 |           3.366683 |                 6.907942 |                         0.0 |                     4.946677 |             6.666164 | -8.500415 |   -14.3022 |                      0.0 |                  7.979839 |                     0.0 |                 47.67784 |             0.0 |       -39.698001 |  0.846 | 0.06345 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.0 |      0.0 | -0.724068 |  0.067075 |    -2.855473 |     0.751114 |       0.0 |       0.0 | 0.974534 |  0.78255 |     0.0 | -22.354332 |      0.0 | -33.649879 |      0.0 |   -16.30621 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000025 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000025 |      0.0 | 0.06 |   0.0 |  0.0 |      0.0 |   nan | 25.400167 | 25.400167 | -0.742159 | 0.896287 |  -6.629796 | 93.280735 | 0.000029 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |

>>> round_(model.check_waterbalance(conditions))
0.0

>>> ktschnee(inf)

hourly simulation
_________________

>>> pub.timegrids = "1997-08-03", "1997-08-06", "1h"
>>> for parameter in control:
...     if parameter.TIME == True:
...         parameter.value = parameter.value/24
...     if parameter.TIME == False:
...         parameter.value = parameter.value*24
>>> for parameter in fixed:
...     parameter.restore()
>>> for parameter in model.aetmodel.parameters.fixed:
...     parameter.restore()

>>> with model.add_radiationmodel_v1("meteo_glob_morsim") as submodel_meteo_glob_morsim:
...     latitude(54.1)
...     longitude(9.7)
...     angstromconstant(0.25)
...     angstromfactor(0.5)
...     angstromalternative(0.15)

>>> model.aetmodel.add_radiationmodel_v1(submodel_meteo_glob_morsim)

>>> test = IntegrationTest(land)

.. _lland_knauf_ic_acker_summer_hourly:

acre (summer)
-------------

>>> inputs.nied.series = 0.0
>>> inputs.teml.series = (
...     16.9, 16.6, 16.4, 16.3, 16.0, 15.9, 16.0, 16.6, 17.4, 19.0, 20.3, 21.4, 21.3,
...     21.8, 22.9, 22.7, 22.5, 21.9, 21.4, 20.7, 19.4, 17.8, 17.0, 16.4, 15.9, 15.4,
...     15.2, 14.9, 14.3, 14.1, 14.3, 15.4, 16.8, 18.2, 19.5, 20.3, 21.0, 21.8, 21.9,
...     21.8, 22.2, 21.8, 20.9, 19.7, 17.9, 15.8, 14.8, 14.1, 13.4, 12.8, 12.3, 12.0,
...     11.7, 11.6, 12.6, 14.8, 17.1, 19.4, 20.6, 21.4, 22.5, 23.1, 23.4, 23.8, 24.0,
...     24.0, 23.7, 22.8, 21.3, 19.3, 18.3, 17.4)
>>> inputs.windspeed.series = (
...     0.8, 0.8, 0.8, 0.8, 0.8, 0.6, 0.9, 0.9, 0.9, 1.3, 1.5, 1.2, 1.3, 1.5, 1.9, 1.9,
...     2.3, 2.4, 2.5, 2.5, 2.2, 1.7, 1.7, 2.3, 2.3, 2.2, 2.3, 2.0, 2.3, 2.5, 2.4, 2.5,
...     2.5, 2.9, 3.1, 3.3, 3.3, 3.2, 2.5, 2.9, 3.6, 4.2, 4.2, 3.6, 3.2, 2.2, 2.2, 2.1,
...     1.8, 1.8, 1.6, 1.2, 1.9, 1.3, 1.6, 1.9, 3.2, 4.0, 4.6, 4.1, 4.1, 4.4, 4.6, 4.5,
...     4.7, 4.5, 4.1, 3.8, 2.5, 1.9, 2.2, 2.7)
>>> inputs.relativehumidity.series = (
...     95.1, 94.9, 95.9, 96.7, 97.2, 97.5, 97.7, 97.4, 96.8, 86.1, 76.8, 71.8, 67.5,
...     66.1, 63.4, 62.4, 61.1, 62.1, 67.0, 74.5, 81.2, 86.9, 90.1, 90.9, 88.0, 89.2,
...     88.8, 89.7, 93.0, 93.6, 94.6, 93.3, 86.9, 78.9, 75.8, 73.4, 68.3, 61.3, 60.6,
...     58.8, 57.2, 57.2, 59.9, 66.1, 71.7, 80.6, 85.9, 89.3, 91.4, 93.1, 94.6, 95.6,
...     96.2, 95.7, 95.1, 96.3, 89.9, 79.2, 73.4, 67.3, 62.2, 59.6, 55.4, 52.1, 49.5,
...     48.1, 49.4, 57.0, 65.9, 73.0, 76.7, 81.8)

>>> model.radiationmodel.sequences.inputs.sunshineduration.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.5, 0.7, 0.8, 0.5, 0.4, 0.5,
...     0.5, 0.3, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.9,
...     1.0, 1.0, 0.9, 0.8, 0.9, 0.8, 0.9, 0.9, 0.9, 1.0, 1.0, 1.0, 0.3, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.6, 0.9, 1.0, 0.9, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
...     1.0, 1.0, 1.0, 1.0, 0.5, 0.0, 0.0, 0.0)

>>> aetinputs.atmosphericpressure.series = (
...     1015.0, 1015.0, 1015.0, 1015.0, 1015.0, 1015.0, 1015.0, 1015.0, 1016.0, 1016.0,
...     1016.0, 1016.0, 1016.0, 1016.0, 1016.0, 1016.0, 1016.0, 1016.0, 1016.0, 1016.0,
...     1016.0, 1016.0, 1017.0, 1017.0, 1017.0, 1016.0, 1016.0, 1016.0, 1016.0, 1017.0,
...     1017.0, 1017.0, 1017.0, 1017.0, 1017.0, 1018.0, 1018.0, 1018.0, 1018.0, 1017.0,
...     1017.0, 1017.0, 1017.0, 1017.0, 1017.0, 1018.0, 1018.0, 1018.0, 1018.0, 1018.0,
...     1018.0, 1018.0, 1018.0, 1018.0, 1018.0, 1019.0, 1019.0, 1019.0, 1019.0, 1019.0,
...     1019.0, 1019.0, 1019.0, 1019.0, 1018.0, 1018.0, 1018.0, 1018.0, 1018.0, 1018.0,
...     1018.0, 1018.0)
>>> aetinputs.windspeed.series = inputs.windspeed.series
>>> aetinputs.relativehumidity.series = inputs.relativehumidity.series

>>> inlet.sequences.sim.series = 0.0

>>> test.inits = (
...     (logs.loggedsunshineduration,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.1, 0.2, 0.1, 0.2, 0.2, 0.3, 0.0,
...       0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]),
...     (logs.loggedpossiblesunshineduration,
...      [0.0, 0.0, 0.0, 0.0, 0.5, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
...       1.0, 1.0, 1.0, 1.0, 1.0, 0.2, 0.0, 0.0, 0.0]),
...     (model.radiationmodel.sequences.logs.loggedunadjustedglobalradiation,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 27.777778, 55.555556, 138.888889, 222.222222,
...       305.555556, 333.333333, 388.888889, 527.777778, 444.444444, 250.0,
...       222.222222, 166.666667, 111.111111, 55.555556, 27.777778, 0.0, 0.0, 0.0,
...       0.0]),
...     (model.aetmodel.sequences.logs.loggedglobalradiation,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 27.8, 55.6, 138.9, 222.2, 305.6, 333.3, 388.9,
...       527.8, 444.4, 250.0, 222.2, 166.7, 111.1, 55.6, 27.8, 0.0, 0.0, 0.0, 0.0]),
...     (model.aetmodel.sequences.logs.loggedairtemperature,
...      [[13.2], [13.2], [13.1], [12.6], [12.7], [13.0], [13.5], [14.8], [16.2],
...       [17.7], [18.8], [19.4], [20.4], [21.0], [21.5], [21.2], [20.4], [20.7],
...       [20.2], [19.7], [19.0], [18.0], [17.5], [17.1]]),
...     (model.aetmodel.sequences.logs.loggedrelativehumidity,
...      [95.1, 94.5, 94.8, 96.4, 96.6, 97.1, 97.1, 96.7, 92.2, 88.5, 81.1, 76.5, 75.1,
...       70.8, 68.9, 69.2, 75.0, 74.0, 77.4, 81.4, 85.3, 90.1, 92.3, 93.8]),
...     (model.aetmodel.sequences.logs.loggedwindspeed2m,
...      [0.8, 1.0, 1.2, 1.3, 0.9, 1.1, 1.3, 1.3, 1.9, 2.2, 1.8, 2.3, 2.4, 2.5, 2.4,
...       2.5, 2.1, 2.2, 1.7, 1.7, 1.3, 1.3, 0.7, 0.8]))

>>> for key, value in locals().copy().items():
...     if key.startswith("conditions_") and ("states" in value.get("model", {})):
...         for name in ("esnowinz", "esnow", "ebdn"):
...             value["model"]["states"][name] *= 24

.. integration-test::

    >>> test("lland_knauf_ic_acker_summer_hourly",
    ...      axis1=(fluxes.evb, fluxes.qah), axis2=states.bowa,
    ...      use_conditions=conditions_acker_summer)
    |                date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation | nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow | nbes | sbes | snowintmax | snowintrate | nbesinz | sbesinz | wniedinz | actualalbedoinz | wadainz | schmpotinz | schminz | gefrpotinz | gefrinz | evsinz |       evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatinz | wlatsnow | wsensinz | wsenssnow | wsurfinz | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga |      qah |       qa |     inzp | stinz | sinz | esnowinz | asinz | wats | waes | esnow | taus |        ebdn |      bowa |     sdg1 | sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-08-03 00:00:00 |  0.0 | 16.9 |             95.1 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 |  0.0 | 16.9 |    0.641273 |           0.641273 |                19.284227 |                         0.0 |                          0.0 |              18.3393 | 15.632162 |  -7.607027 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.002953 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007792 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007792 | 0.002165 | 0.002953 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8729.609427 | 79.623355 | 0.293878 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002165 |
    | 1997-08-03 01:00:00 |  0.0 | 16.6 |             94.9 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 |  0.0 | 16.6 |    0.641273 |           0.641273 |                18.920184 |                         0.0 |                          0.0 |            17.955254 | 15.688646 |  -5.468121 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.002673 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007591 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007591 | 0.002109 | 0.005626 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8732.177548 | 79.623355 | 0.286287 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002109 |
    | 1997-08-03 02:00:00 |  0.0 | 16.4 |             95.9 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 |  0.0 | 16.4 |    0.641273 |           0.641273 |                 18.68084 |                         0.0 |                          0.0 |            17.914926 | 15.719464 |  -4.083217 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.004096 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007395 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007395 | 0.002054 | 0.009722 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8733.360765 | 79.623355 | 0.278892 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002054 |
    | 1997-08-03 03:00:00 |  0.0 | 16.3 |             96.7 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.5 |             0.0 |  0.0 | 16.3 |    0.641273 |           0.641273 |                18.562165 |                         0.0 |                          0.0 |            17.949613 | 15.733663 |  -3.398025 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.005245 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007204 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007204 | 0.002001 | 0.014967 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  8733.85879 | 79.623355 | 0.271688 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002001 |
    | 1997-08-03 04:00:00 |  0.0 | 16.0 |             97.2 |       0.8 |                 0.429734 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |        1.943686 |  0.0 | 16.0 |    0.641273 |           0.641273 |                18.210086 |                         0.0 |                          0.0 |            17.700204 | 15.739639 |  -1.562167 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.004905 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007018 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007018 | 0.001949 | 0.019872 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8732.520957 | 79.623355 |  0.26467 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001949 |
    | 1997-08-03 05:00:00 |  0.0 | 15.9 |             97.5 |       0.6 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |       21.932441 |  0.0 | 15.9 |    0.480955 |           0.480955 |                18.094032 |                         0.0 |                          0.0 |            17.641681 | 15.723585 |  -1.058491 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.004992 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006837 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006837 | 0.001899 |  0.01488 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8730.679448 | 79.623355 | 0.257833 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001899 |
    | 1997-08-03 06:00:00 |  0.0 | 16.0 |             97.7 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |       57.256187 |  0.0 | 16.0 |    0.721432 |           0.721432 |                18.210086 |                         0.0 |                          0.0 |            17.791254 | 15.701487 |   -1.79108 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.01488 | 0.008643 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00666 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00666 |  0.00185 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8729.570528 | 79.614713 | 0.251173 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00185 |
    | 1997-08-03 07:00:00 |  0.0 | 16.6 |             97.4 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |      109.332844 |  0.0 | 16.6 |    0.721432 |           0.721432 |                18.920184 |                         0.0 |                          0.0 |            18.428259 |  15.68818 |  -5.470922 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.043948 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006488 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006488 | 0.001802 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  8732.14145 | 79.570765 | 0.244685 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001802 |
    | 1997-08-03 08:00:00 |  0.0 | 17.4 |             96.8 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |      170.949152 |  0.0 | 17.4 |    0.721432 |           0.721432 |                19.904589 |                         0.0 |                          0.0 |            19.267642 | 15.719031 | -10.085816 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.073907 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00632 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00632 | 0.001756 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8739.327265 | 79.496858 | 0.238365 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001756 |
    | 1997-08-03 09:00:00 |  0.0 | 19.0 |             86.1 |       1.3 |                      1.0 |              0.2 | 0.0 | 0.0 |                   1.5 |                     14.929734 |      311.762624 |  0.0 | 19.0 |    1.042069 |           1.042069 |                22.008543 |                         0.0 |                          0.0 |            18.949356 | 15.805261 | -19.168437 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.160936 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006157 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006157 |  0.00171 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8755.595702 | 79.335922 | 0.232208 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00171 |
    | 1997-08-03 10:00:00 |  0.0 | 20.3 |             76.8 |       1.5 |                      1.0 |              0.5 | 0.0 | 0.0 |                   1.7 |                     14.929734 |      501.583299 |  0.0 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                         0.0 |                          0.0 |             18.32333 | 16.000482 | -25.797109 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.275473 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005998 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005998 | 0.001666 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8778.492812 | 79.060449 | 0.226209 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001666 |
    | 1997-08-03 11:00:00 |  0.0 | 21.4 |             71.8 |       1.2 |                      1.0 |              0.7 | 0.0 | 0.0 |                   2.2 |                     14.929734 |      615.018727 |  0.0 | 21.4 |    0.961909 |           0.961909 |                25.528421 |                         0.0 |                          0.0 |            18.329406 | 16.275247 | -30.748518 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.348731 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005843 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005843 | 0.001623 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8806.341329 | 78.711718 | 0.220366 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001623 |
    | 1997-08-03 12:00:00 |  0.0 | 21.3 |             67.5 |       1.3 |                      1.0 |              0.8 | 0.0 | 0.0 |                   2.8 |                     14.929734 |      626.544326 |  0.0 | 21.3 |    1.042069 |           1.042069 |                 25.37251 |                         0.0 |                          0.0 |            17.126444 | 16.609429 | -28.143424 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.360693 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005692 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005692 | 0.001581 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8831.584754 | 78.351026 | 0.214674 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001581 |
    | 1997-08-03 13:00:00 |  0.0 | 21.8 |             66.1 |       1.5 |                      1.0 |              0.5 | 0.0 | 0.0 |                   3.2 |                     14.929734 |      496.133417 |  0.0 | 21.8 |    1.202387 |           1.202387 |                26.160453 |                         0.0 |                          0.0 |             17.29206 |  16.91235 | -29.325898 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.303239 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005545 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005545 |  0.00154 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8858.010651 | 78.047787 | 0.209129 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00154 |
    | 1997-08-03 14:00:00 |  0.0 | 22.9 |             63.4 |       1.9 |                      1.0 |              0.4 | 0.0 | 0.0 |                   3.4 |                     14.929734 |      419.520994 |  0.0 | 22.9 |    1.523023 |           1.523023 |                27.969419 |                         0.0 |                          0.0 |            17.732611 | 17.229461 | -34.023233 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.282369 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005402 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005402 | 0.001501 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8889.133884 | 77.765418 | 0.203727 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001501 |
    | 1997-08-03 15:00:00 |  0.0 | 22.7 |             62.4 |       1.9 |                      1.0 |              0.5 | 0.0 | 0.0 |                   3.8 |                     14.929734 |      387.887354 |  0.0 | 22.7 |    1.523023 |           1.523023 |                27.632633 |                         0.0 |                          0.0 |            17.242763 |  17.60294 |  -30.58236 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.266636 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005262 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005262 | 0.001462 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8916.816245 | 77.498782 | 0.198464 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001462 |
    | 1997-08-03 16:00:00 |  0.0 | 22.5 |             61.1 |       2.3 |                      1.0 |              0.5 | 0.0 | 0.0 |                   4.1 |                     14.929734 |      278.496873 |  0.0 | 22.5 |     1.84366 |            1.84366 |                27.299387 |                         0.0 |                          0.0 |            16.679926 | 17.935128 |  -27.38923 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.218982 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005127 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005127 | 0.001424 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8941.305475 | 77.279799 | 0.193338 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001424 |
    | 1997-08-03 17:00:00 |  0.0 | 21.9 |             62.1 |       2.4 |                      1.0 |              0.3 | 0.0 | 0.0 |                   4.4 |                     14.929734 |      137.138608 |  0.0 | 21.9 |    1.923819 |           1.923819 |                26.320577 |                         0.0 |                          0.0 |            16.345078 | 18.228999 | -22.026006 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.147975 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004994 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004994 | 0.001387 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8960.431481 | 77.131824 | 0.188344 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001387 |
    | 1997-08-03 18:00:00 |  0.0 | 21.4 |             67.0 |       2.5 |                      1.0 |              0.1 | 0.0 | 0.0 |                   4.5 |                     14.929734 |       51.080715 |  0.0 | 21.4 |    2.003978 |           2.003978 |                25.528421 |                         0.0 |                          0.0 |            17.104042 | 18.458511 | -17.648933 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.095816 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004865 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004865 | 0.001351 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8975.180414 | 77.036008 | 0.183479 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001351 |
    | 1997-08-03 19:00:00 |  0.0 | 20.7 |             74.5 |       2.5 |                      1.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.429734 |       13.632816 |  0.0 | 20.7 |    2.003978 |           2.003978 |                24.454368 |                         0.0 |                          0.0 |            18.218504 | 18.635498 |  -12.38701 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.058452 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004739 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004739 | 0.001317 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8984.667424 | 76.977556 | 0.178739 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001317 |
    | 1997-08-03 20:00:00 |  0.0 | 19.4 |             81.2 |       2.2 |                   0.1364 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |        0.185943 |  0.0 | 19.4 |    1.763501 |           1.763501 |                22.563931 |                         0.0 |                          0.0 |            18.321912 | 18.749342 |  -3.903945 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.032413 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004617 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004617 | 0.001282 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  8985.67137 | 76.945143 | 0.174122 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001282 |
    | 1997-08-03 21:00:00 |  0.0 | 17.8 |             86.9 |       1.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 |  0.0 | 17.8 |    1.362705 |           1.362705 |                20.413369 |                         0.0 |                          0.0 |            17.739217 |  18.76139 |   5.768339 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.013439 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004498 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004498 | 0.001249 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8977.003031 | 76.931705 | 0.169625 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001249 |
    | 1997-08-03 22:00:00 |  0.0 | 17.0 |             90.1 |       1.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 |  0.0 | 17.0 |    1.362705 |           1.362705 |                19.406929 |                         0.0 |                          0.0 |            17.485643 |  18.65737 |   9.944218 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.007132 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004382 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004382 | 0.001217 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8964.158813 | 76.924573 | 0.165243 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001217 |
    | 1997-08-03 23:00:00 |  0.0 | 16.4 |             90.9 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 |  0.0 | 16.4 |     1.84366 |            1.84366 |                 18.68084 |                         0.0 |                          0.0 |            16.980884 | 18.503239 |  12.619435 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.009242 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004268 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004268 | 0.001186 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8948.639378 | 76.915331 | 0.160975 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001186 |
    | 1997-08-04 00:00:00 |  0.0 | 15.9 |             88.0 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 |  0.0 | 15.9 |     1.84366 |            1.84366 |                18.094032 |                         0.0 |                          0.0 |            15.922748 | 18.317006 |  14.502035 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.015057 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004158 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004158 | 0.001155 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8931.237343 | 76.900275 | 0.156817 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001155 |
    | 1997-08-04 01:00:00 |  0.0 | 15.4 |             89.2 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 |  0.0 | 15.4 |    1.763501 |           1.763501 |                17.523416 |                         0.0 |                          0.0 |            15.630887 | 18.108181 |  16.249089 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.011515 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004051 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004051 | 0.001125 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8912.088255 | 76.888759 | 0.152766 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001125 |
    | 1997-08-04 02:00:00 |  0.0 | 15.2 |             88.8 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 |  0.0 | 15.2 |     1.84366 |            1.84366 |                17.299618 |                         0.0 |                          0.0 |            15.362061 | 17.878392 |  16.070354 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.012795 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003946 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003946 | 0.001096 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |   8893.1179 | 76.875964 |  0.14882 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001096 |
    | 1997-08-04 03:00:00 |  0.0 | 14.9 |             89.7 |       2.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 |  0.0 | 14.9 |    1.603182 |           1.603182 |                16.968612 |                         0.0 |                          0.0 |            15.220845 | 17.650748 |  16.504489 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.008851 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003844 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003844 | 0.001068 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8873.713411 | 76.867113 | 0.144976 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001068 |
    | 1997-08-04 04:00:00 |  0.0 | 14.3 |             93.0 |       2.3 |                 0.398044 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.534444 |        1.590516 |  0.0 | 14.3 |     1.84366 |            1.84366 |                16.323194 |                         0.0 |                          0.0 |             15.18057 | 17.417894 |  18.707366 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.004365 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003745 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003745 |  0.00104 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8852.106046 | 76.862748 | 0.141231 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00104 |
    | 1997-08-04 05:00:00 |  0.0 | 14.1 |             93.6 |       2.5 |                      1.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.534444 |       19.950903 |  0.0 | 14.1 |    2.003978 |           2.003978 |                16.112886 |                         0.0 |                          0.0 |            15.081661 | 17.158606 |  18.351635 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.010118 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003648 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003648 | 0.001013 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  8830.85441 |  76.85263 | 0.137583 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001013 |
    | 1997-08-04 06:00:00 |  0.0 | 14.3 |             94.6 |       2.4 |                      1.0 |              0.1 | 0.0 | 0.0 |                   4.6 |                     15.534444 |       62.985774 |  0.0 | 14.3 |    1.923819 |           1.923819 |                16.323194 |                         0.0 |                          0.0 |            15.441741 | 16.903586 |  15.621518 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.023223 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003554 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003554 | 0.000987 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8812.332893 | 76.829407 | 0.134029 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000987 |
    | 1997-08-04 07:00:00 |  0.0 | 15.4 |             93.3 |       2.5 |                      1.0 |              0.9 | 0.0 | 0.0 |                   5.5 |                     15.534444 |      285.776389 |  0.0 | 15.4 |    2.003978 |           2.003978 |                17.523416 |                         0.0 |                          0.0 |            16.349347 | 16.681328 |   7.687968 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 |   0.1075 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003462 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003462 | 0.000962 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8801.744925 | 76.721908 | 0.130567 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000962 |
    | 1997-08-04 08:00:00 |  0.0 | 16.8 |             86.9 |       2.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                   6.5 |                     15.534444 |      475.225247 |  0.0 | 16.8 |    2.003978 |           2.003978 |                19.162205 |                         0.0 |                          0.0 |            16.651956 | 16.554272 |  -1.474365 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.198377 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003373 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003373 | 0.000937 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  8800.31929 | 76.523531 | 0.127194 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000937 |
    | 1997-08-04 09:00:00 |  0.0 | 18.2 |             78.9 |       2.9 |                      1.0 |              1.0 | 0.0 | 0.0 |                   7.3 |                     15.534444 |      614.006719 |  0.0 | 18.2 |    2.324615 |           2.324615 |                20.933479 |                         0.0 |                          0.0 |            16.516515 | 16.537165 |  -9.977011 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.274475 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003286 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003286 | 0.000913 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8807.396301 | 76.249056 | 0.123909 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000913 |
    | 1997-08-04 10:00:00 |  0.0 | 19.5 |             75.8 |       3.1 |                      1.0 |              0.9 | 0.0 | 0.0 |                   7.7 |                     15.534444 |      643.806508 |  0.0 | 19.5 |    2.484933 |           2.484933 |                22.704676 |                         0.0 |                          0.0 |            17.210144 | 16.622089 | -17.267466 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 |  0.30313 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003201 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003201 | 0.000889 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8821.763767 | 75.945926 | 0.120708 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000889 |
    | 1997-08-04 11:00:00 |  0.0 | 20.3 |             73.4 |       3.3 |                      1.0 |              0.8 | 0.0 | 0.0 |                   7.8 |                     15.534444 |       600.26298 |  0.0 | 20.3 |    2.645251 |           2.645251 |                23.858503 |                         0.0 |                          0.0 |            17.512141 | 16.794499 | -21.033009 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.297686 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003118 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003118 | 0.000866 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8839.896776 |  75.64824 |  0.11759 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000866 |
    | 1997-08-04 12:00:00 |  0.0 | 21.0 |             68.3 |       3.3 |                      1.0 |              0.9 | 0.0 | 0.0 |                   7.9 |                     15.534444 |       620.51414 |  0.0 | 21.0 |    2.645251 |           2.645251 |                24.909749 |                         0.0 |                          0.0 |            17.013358 | 17.012095 | -23.927432 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.327488 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003037 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003037 | 0.000844 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8860.924208 | 75.320752 | 0.114552 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000844 |
    | 1997-08-04 13:00:00 |  0.0 | 21.8 |             61.3 |       3.2 |                      1.0 |              0.8 | 0.0 | 0.0 |                   8.2 |                     15.534444 |      599.441386 |  0.0 | 21.8 |    2.565092 |           2.565092 |                26.160453 |                         0.0 |                          0.0 |            16.036358 | 17.264424 | -27.213457 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.348705 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002959 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002959 | 0.000822 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8885.237665 | 74.972048 | 0.111593 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000822 |
    | 1997-08-04 14:00:00 |  0.0 | 21.9 |             60.6 |       2.5 |                      1.0 |              0.9 | 0.0 | 0.0 |                   8.7 |                     15.534444 |      613.551223 |  0.0 | 21.9 |    2.003978 |           2.003978 |                26.320577 |                         0.0 |                          0.0 |             15.95027 | 17.556185 | -26.062888 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.361278 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002883 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002883 | 0.000801 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8908.400553 | 74.610769 | 0.108711 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000801 |
    | 1997-08-04 15:00:00 |  0.0 | 21.8 |             58.8 |       2.9 |                      1.0 |              0.9 | 0.0 | 0.0 |                   9.1 |                     15.534444 |      517.107047 |  0.0 | 21.8 |    2.324615 |           2.324615 |                26.160453 |                         0.0 |                          0.0 |            15.382347 |  17.83414 |  -23.79516 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.320086 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002808 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002808 |  0.00078 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8929.295714 | 74.290683 | 0.105903 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00078 |
    | 1997-08-04 16:00:00 |  0.0 | 22.2 |             57.2 |       3.6 |                      1.0 |              0.9 | 0.0 | 0.0 |                   9.5 |                     15.534444 |      377.836616 |  0.0 | 22.2 |    2.885728 |           2.885728 |                26.806088 |                         0.0 |                          0.0 |            15.333082 | 18.084882 | -24.690709 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.271017 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002736 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002736 |  0.00076 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8951.086422 | 74.019667 | 0.103167 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00076 |
    | 1997-08-04 17:00:00 |  0.0 | 21.8 |             57.2 |       4.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  10.2 |                     15.534444 |      252.204639 |  0.0 | 21.8 |    3.366683 |           3.366683 |                26.160453 |                         0.0 |                          0.0 |            14.963779 |  18.34637 | -20.721778 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.220995 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002665 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002665 |  0.00074 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |   8968.9082 | 73.798672 | 0.100502 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00074 |
    | 1997-08-04 18:00:00 |  0.0 | 20.9 |             59.9 |       4.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  11.1 |                     15.534444 |      128.576715 |  0.0 | 20.9 |    3.366683 |           3.366683 |                 24.75714 |                         0.0 |                          0.0 |            14.829527 | 18.560232 |  -14.03861 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.158739 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002596 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002596 | 0.000721 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8980.046809 | 73.639933 | 0.097906 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000721 |
    | 1997-08-04 19:00:00 |  0.0 | 19.7 |             66.1 |       3.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.1 |                     15.534444 |       42.072585 |  0.0 | 19.7 |    2.885728 |           2.885728 |                22.988466 |                         0.0 |                          0.0 |            15.195376 | 18.693895 |   -6.03663 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.093951 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002529 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002529 | 0.000703 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8983.183439 | 73.545981 | 0.095377 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000703 |
    | 1997-08-04 20:00:00 |  0.0 | 17.9 |             71.7 |       3.2 |                 0.104711 |              0.3 | 0.0 | 0.0 |                  12.4 |                     15.502754 |        0.812898 |  0.0 | 17.9 |    2.565092 |           2.565092 |                20.542326 |                         0.0 |                          0.0 |            14.728848 | 18.731535 |   4.989208 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.057093 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002464 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002464 | 0.000684 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8975.294232 | 73.488889 | 0.092914 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000684 |
    | 1997-08-04 21:00:00 |  0.0 | 15.8 |             80.6 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 |  0.0 | 15.8 |    1.763501 |           1.763501 |                17.978626 |                         0.0 |                          0.0 |            14.490772 | 18.636864 |  17.021185 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.019046 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0024 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0024 | 0.000667 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8955.373047 | 73.469843 | 0.090514 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000667 |
    | 1997-08-04 22:00:00 |  0.0 | 14.8 |             85.9 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 |  0.0 | 14.8 |    1.763501 |           1.763501 |                16.859515 |                         0.0 |                          0.0 |            14.482324 |  18.39781 |  21.586859 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.007576 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002338 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002338 | 0.000649 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8930.886187 | 73.462266 | 0.088176 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000649 |
    | 1997-08-04 23:00:00 |  0.0 | 14.1 |             89.3 |       2.1 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 |  0.0 | 14.1 |    1.683342 |           1.683342 |                16.112886 |                         0.0 |                          0.0 |            14.388807 | 18.103968 |  24.023805 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.000116 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002278 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002278 | 0.000633 | 0.000116 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8903.962382 | 73.462266 | 0.085898 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000633 |
    | 1997-08-05 00:00:00 |  0.0 | 13.4 |             91.4 |       1.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 |  0.0 | 13.4 |    1.442864 |           1.442864 |                15.395377 |                         0.0 |                          0.0 |            14.071374 | 17.780882 |  26.285292 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.009023 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002219 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002219 | 0.000616 | 0.009139 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  8874.77709 | 73.462266 | 0.083679 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000616 |
    | 1997-08-05 01:00:00 |  0.0 | 12.8 |             93.1 |       1.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 |  0.0 | 12.8 |    1.442864 |           1.442864 |                14.802811 |                         0.0 |                          0.0 |            13.781417 | 17.430658 |  27.783951 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.013264 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002162 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002162 |   0.0006 | 0.022403 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  8844.09314 | 73.462266 | 0.081518 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |   0.0006 |
    | 1997-08-05 02:00:00 |  0.0 | 12.3 |             94.6 |       1.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 |  0.0 | 12.3 |    1.282546 |           1.282546 |                14.324376 |                         0.0 |                          0.0 |            13.550859 | 17.062451 |  28.574706 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.017718 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002106 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002106 | 0.000585 | 0.040121 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8812.618434 | 73.462266 | 0.079412 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000585 |
    | 1997-08-05 03:00:00 |  0.0 | 12.0 |             95.6 |       1.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 |  0.0 | 12.0 |    0.961909 |           0.961909 |                 14.04387 |                         0.0 |                          0.0 |            13.425939 | 16.684755 |  28.108527 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.020716 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002051 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002051 |  0.00057 | 0.060837 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8781.609907 | 73.462266 | 0.077361 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00057 |
    | 1997-08-05 04:00:00 |  0.0 | 11.7 |             96.2 |       1.9 |                 0.366022 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.470733 |        1.482348 |  0.0 | 11.7 |    1.523023 |           1.523023 |                13.768192 |                         0.0 |                          0.0 |            13.245001 | 16.312652 |  27.675913 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.019578 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001998 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001998 | 0.000555 | 0.080415 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8751.033993 | 73.462266 | 0.075362 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000555 |
    | 1997-08-05 05:00:00 |  0.0 | 11.6 |             95.7 |       1.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.470733 |       21.135568 |  0.0 | 11.6 |    1.042069 |           1.042069 |                 13.67736 |                         0.0 |                          0.0 |            13.089234 | 15.945741 |  26.074448 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.009263 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001947 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001947 | 0.000541 | 0.089679 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8722.059546 | 73.462266 | 0.073416 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000541 |
    | 1997-08-05 06:00:00 |  0.0 | 12.6 |             95.1 |       1.6 |                      1.0 |              0.6 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      125.883451 |  0.0 | 12.6 |    1.282546 |           1.282546 |                14.609782 |                         0.0 |                          0.0 |            13.893903 | 15.598048 |  17.988287 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.052361 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001896 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001896 | 0.000527 | 0.037317 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8701.171259 | 73.462266 | 0.071519 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000527 |
    | 1997-08-05 07:00:00 |  0.0 | 14.8 |             96.3 |       1.9 |                      1.0 |              0.9 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      305.954943 |  0.0 | 14.8 |    1.523023 |           1.523023 |                16.859515 |                         0.0 |                          0.0 |            16.235713 | 15.347388 |   3.284331 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.037317 | 0.079358 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001847 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001847 | 0.000513 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8694.986928 | 73.382908 | 0.069672 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000513 |
    | 1997-08-05 08:00:00 |  0.0 | 17.1 |             89.9 |       3.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      511.104575 |  0.0 | 17.1 |    2.565092 |           2.565092 |                19.530313 |                         0.0 |                          0.0 |            17.557751 | 15.273176 | -10.960941 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.185679 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0018 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0018 |   0.0005 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8703.047869 | 73.197229 | 0.067872 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |   0.0005 |
    | 1997-08-05 09:00:00 |  0.0 | 19.4 |             79.2 |       4.0 |                      1.0 |              0.9 | 0.0 | 0.0 |                  12.8 |                     15.470733 |       629.51924 |  0.0 | 19.4 |    3.206365 |           3.206365 |                22.563931 |                         0.0 |                          0.0 |            17.870634 | 15.369908 | -24.180553 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 |   0.2631 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001753 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001753 | 0.000487 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8724.328423 |  72.93413 | 0.066119 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000487 |
    | 1997-08-05 10:00:00 |  0.0 | 20.6 |             73.4 |       4.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.9 |                     15.470733 |       765.81659 |  0.0 | 20.6 |     3.68732 |            3.68732 |                24.304197 |                         0.0 |                          0.0 |            17.839281 | 15.625274 | -29.848354 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 |   0.3322 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001708 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001708 | 0.000474 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8751.276776 |  72.60193 | 0.064411 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000474 |
    | 1997-08-05 11:00:00 |  0.0 | 21.4 |             67.3 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.1 |                     15.470733 |      767.145843 |  0.0 | 21.4 |    3.286524 |           3.286524 |                25.528421 |                         0.0 |                          0.0 |            17.180627 | 15.948655 | -32.708072 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.369947 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001664 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001664 | 0.000462 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8781.084848 | 72.231982 | 0.062747 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000462 |
    | 1997-08-05 12:00:00 |  0.0 | 22.5 |             62.2 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.2 |                     15.470733 |      736.263854 |  0.0 | 22.5 |    3.286524 |           3.286524 |                27.299387 |                         0.0 |                          0.0 |            16.980219 | 16.306352 | -37.161891 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.389289 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001621 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001621 |  0.00045 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8815.346739 | 71.842694 | 0.061126 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00045 |
    | 1997-08-05 13:00:00 |  0.0 | 23.1 |             59.6 |       4.4 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.4 |                     15.470733 |      768.006183 |  0.0 | 23.1 |    3.527001 |           3.527001 |                28.309776 |                         0.0 |                          0.0 |            16.872626 | 16.717494 | -38.295035 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 |  0.41464 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001579 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001579 | 0.000439 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8850.741774 | 71.428054 | 0.059547 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000439 |
    | 1997-08-05 14:00:00 |  0.0 | 23.4 |             55.4 |       4.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.5 |                     15.470733 |       735.92937 |  0.0 | 23.4 |     3.68732 |            3.68732 |                28.827078 |                         0.0 |                          0.0 |            15.970201 | 17.142235 | -37.546592 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.419328 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001538 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001538 | 0.000427 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8885.388366 | 71.008726 | 0.058009 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000427 |
    | 1997-08-05 15:00:00 |  0.0 | 23.8 |             52.1 |       4.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.6 |                     15.470733 |      620.374953 |  0.0 | 23.8 |    3.607161 |           3.607161 |                29.529613 |                         0.0 |                          0.0 |            15.384928 | 17.557994 | -37.452038 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 |  0.39307 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001498 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001498 | 0.000416 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8919.940404 | 70.615656 | 0.056511 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000416 |
    | 1997-08-05 16:00:00 |  0.0 | 24.0 |             49.5 |       4.7 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      450.095263 |  0.0 | 24.0 |    3.767479 |           3.767479 |                29.886439 |                         0.0 |                          0.0 |            14.793787 | 17.972618 | -36.164291 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 |  0.33964 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00146 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00146 | 0.000405 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8953.204695 | 70.276016 | 0.055051 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000405 |
    | 1997-08-05 17:00:00 |  0.0 | 24.0 |             48.1 |       4.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      273.614141 |  0.0 | 24.0 |    3.607161 |           3.607161 |                29.886439 |                         0.0 |                          0.0 |            14.375377 |  18.37179 | -33.769262 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.275457 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001422 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001422 | 0.000395 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8984.073957 | 70.000559 | 0.053629 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000395 |
    | 1997-08-05 18:00:00 |  0.0 | 23.7 |             49.4 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      133.368104 |  0.0 | 23.7 |    3.286524 |           3.286524 |                29.352596 |                         0.0 |                          0.0 |            14.500182 | 18.742221 | -29.746675 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.207398 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001385 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001385 | 0.000385 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 9010.920632 | 69.793161 | 0.052244 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000385 |
    | 1997-08-05 19:00:00 |  0.0 | 22.8 |             57.0 |       3.8 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |       40.326502 |  0.0 | 22.8 |    3.046047 |           3.046047 |                27.800581 |                         0.0 |                          0.0 |            15.846331 | 19.064381 | -22.413715 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.133656 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00135 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00135 | 0.000375 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 9030.434346 | 69.659505 | 0.050894 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000375 |
    | 1997-08-05 20:00:00 |  0.0 | 21.3 |             65.9 |       2.5 |                 0.072689 |              0.5 | 0.0 | 0.0 |                  13.9 |                     15.438712 |        0.867122 |  0.0 | 21.3 |    2.003978 |           2.003978 |                 25.37251 |                         0.0 |                          0.0 |            16.720484 | 19.298545 | -12.008727 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.066112 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001315 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001315 | 0.000365 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 9039.543073 | 69.593394 |  0.04958 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000365 |
    | 1997-08-05 21:00:00 |  0.0 | 19.3 |             73.0 |       1.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 |  0.0 | 19.3 |    1.523023 |           1.523023 |                22.423949 |                         0.0 |                          0.0 |            16.369483 |  19.40785 |   0.647101 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 |  0.03182 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001281 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001281 | 0.000356 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 9035.995972 | 69.561574 | 0.048299 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000356 |
    | 1997-08-05 22:00:00 |  0.0 | 18.3 |             76.7 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 |  0.0 | 18.3 |    1.763501 |           1.763501 |                21.065302 |                         0.0 |                          0.0 |            16.157086 | 19.365285 |    6.39171 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 |  0.02777 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001248 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001248 | 0.000347 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 9026.704262 | 69.533804 | 0.047051 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000347 |
    | 1997-08-05 23:00:00 |  0.0 | 17.4 |             81.8 |       2.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 |  0.0 | 17.4 |    2.164296 |           2.164296 |                19.904589 |                         0.0 |                          0.0 |            16.281953 | 19.253784 |  11.122707 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.0 | 0.021514 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001215 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001215 | 0.000338 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 9012.681555 |  69.51229 | 0.045836 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000338 |

>>> round_(model.check_waterbalance(conditions_acker_summer))
0.0

.. _lland_knauf_ic_acker_heavy_rain_hourly:

acre (heavy rain)
-----------------

.. integration-test::

    >>> inputs.nied.series = 20.0 / 24.0
    >>> control.negq(False)
    >>> test("lland_knauf_ic_acker_heavy_rain_hourly",
    ...      axis1=(inputs.nied, fluxes.qah), axis2=states.bowa,
    ...      use_conditions=conditions_acker_heavy_rain)
    |                date |     nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |     nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow |     nbes | sbes | snowintmax | snowintrate | nbesinz | sbesinz | wniedinz | actualalbedoinz | wadainz | schmpotinz | schminz | gefrpotinz | gefrinz | evsinz |       evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatinz | wlatsnow | wsensinz | wsenssnow | wsurfinz | wsurf | sff | fvg |     wada |      qdb |     qib1 | qib2 |      qbb | qkap |     qdgz |    qdgz1 | qdgz2 |    qigz1 | qigz2 |     qbgz |    qdga1 | qdga2 |    qiga1 | qiga2 |     qbga |      qah |       qa |     inzp | stinz | sinz | esnowinz | asinz | wats | waes | esnow | taus |        ebdn |       bowa |     sdg1 | sdg2 |     sig1 | sig2 |      sbg | inlet |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-08-03 00:00:00 | 0.833333 | 16.9 |             95.1 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 | 0.783333 | 16.9 |    0.641273 |           0.641273 |                19.284227 |                         0.0 |                          0.0 |              18.3393 | 15.632162 |  -7.607027 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.083333 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.002953 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.007403 |      0.0 |  0.0 |      0.0 |  0.0 | 0.007403 | 0.007403 |   0.0 |      0.0 |   0.0 |      0.0 | 0.040965 |   0.0 |      0.0 |   0.0 |      0.0 | 0.040965 | 0.011379 | 0.702953 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8729.609427 | 102.562669 |   1.5486 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.011379 |
    | 1997-08-03 01:00:00 | 0.833333 | 16.6 |             94.9 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 | 0.783333 | 16.6 |    0.641273 |           0.641273 |                18.920184 |                         0.0 |                          0.0 |            17.955254 | 15.688646 |  -5.468121 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.786286 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.002673 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.786286 | 0.070171 |      0.0 |  0.0 |      0.0 |  0.0 | 0.070171 | 0.070171 |   0.0 |      0.0 |   0.0 |      0.0 | 0.040912 |   0.0 |      0.0 |   0.0 |      0.0 | 0.040912 | 0.011364 | 0.702673 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8732.177548 | 103.278784 |  1.57786 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.011364 |
    | 1997-08-03 02:00:00 | 0.833333 | 16.4 |             95.9 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 | 0.783333 | 16.4 |    0.641273 |           0.641273 |                 18.68084 |                         0.0 |                          0.0 |            17.914926 | 15.719464 |  -4.083217 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.786007 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.004096 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.786007 | 0.070721 |      0.0 |  0.0 |      0.0 |  0.0 | 0.070721 | 0.070721 |   0.0 |      0.0 |   0.0 |      0.0 | 0.041675 |   0.0 |      0.0 |   0.0 |      0.0 | 0.041675 | 0.011576 | 0.704096 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8733.360765 |  103.99407 | 1.606906 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.011576 |
    | 1997-08-03 03:00:00 | 0.833333 | 16.3 |             96.7 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.5 |             0.0 | 0.783333 | 16.3 |    0.641273 |           0.641273 |                18.562165 |                         0.0 |                          0.0 |            17.949613 | 15.733663 |  -3.398025 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.787429 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.005245 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.787429 | 0.071426 |      0.0 |  0.0 |      0.0 |  0.0 | 0.071426 | 0.071426 |   0.0 |      0.0 |   0.0 |      0.0 | 0.042434 |   0.0 |      0.0 |   0.0 |      0.0 | 0.042434 | 0.011787 | 0.705245 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  8733.85879 | 104.710073 | 1.635897 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.011787 |
    | 1997-08-03 04:00:00 | 0.833333 | 16.0 |             97.2 |       0.8 |                 0.429734 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |        1.943686 | 0.783333 | 16.0 |    0.641273 |           0.641273 |                18.210086 |                         0.0 |                          0.0 |            17.700204 | 15.739639 |  -1.562167 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.788579 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.004905 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.788579 |  0.07211 |      0.0 |  0.0 |      0.0 |  0.0 |  0.07211 |  0.07211 |   0.0 |      0.0 |   0.0 |      0.0 | 0.043192 |   0.0 |      0.0 |   0.0 |      0.0 | 0.043192 | 0.011998 | 0.704905 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8732.520957 | 105.426542 | 1.664815 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.011998 |
    | 1997-08-03 05:00:00 | 0.833333 | 15.9 |             97.5 |       0.6 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |       21.932441 | 0.783333 | 15.9 |    0.480955 |           0.480955 |                18.094032 |                         0.0 |                          0.0 |            17.641681 | 15.723585 |  -1.058491 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.788239 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.004992 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.788239 |  0.07266 |      0.0 |  0.0 |      0.0 |  0.0 |  0.07266 |  0.07266 |   0.0 |      0.0 |   0.0 |      0.0 | 0.043946 |   0.0 |      0.0 |   0.0 |      0.0 | 0.043946 | 0.012207 | 0.695008 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8730.679448 |  106.14212 | 1.693529 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012207 |
    | 1997-08-03 06:00:00 | 0.833333 | 16.0 |             97.7 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |       57.256187 | 0.783333 | 16.0 |    0.721432 |           0.721432 |                18.210086 |                         0.0 |                          0.0 |            17.791254 | 15.701487 |   -1.79108 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.778341 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.026203 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.778341 | 0.072319 |      0.0 |  0.0 |      0.0 |  0.0 | 0.072319 | 0.072319 |   0.0 |      0.0 |   0.0 |      0.0 | 0.044683 |   0.0 |      0.0 |   0.0 |      0.0 | 0.044683 | 0.012412 | 0.673797 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8729.570528 | 106.848143 | 1.721164 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012412 |
    | 1997-08-03 07:00:00 | 0.833333 | 16.6 |             97.4 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |      109.332844 | 0.783333 | 16.6 |    0.721432 |           0.721432 |                18.920184 |                         0.0 |                          0.0 |            18.428259 |  15.68818 |  -5.470922 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.75713 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.057366 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.75713 | 0.070894 |      0.0 |  0.0 |      0.0 |  0.0 | 0.070894 | 0.070894 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045379 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045379 | 0.012605 | 0.642634 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  8732.14145 |  107.53438 | 1.746679 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012605 |
    | 1997-08-03 08:00:00 | 0.833333 | 17.4 |             96.8 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |      170.949152 | 0.783333 | 17.4 |    0.721432 |           0.721432 |                19.904589 |                         0.0 |                          0.0 |            19.267642 | 15.719031 | -10.085816 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.725967 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.096008 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.725967 | 0.068482 |      0.0 |  0.0 |      0.0 |  0.0 | 0.068482 | 0.068482 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046007 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046007 |  0.01278 | 0.603992 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8739.327265 | 108.191865 | 1.769154 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 |  0.01278 |
    | 1997-08-03 09:00:00 | 0.833333 | 19.0 |             86.1 |       1.3 |                      1.0 |              0.2 | 0.0 | 0.0 |                   1.5 |                     14.929734 |      311.762624 | 0.783333 | 19.0 |    1.042069 |           1.042069 |                22.008543 |                         0.0 |                          0.0 |            18.949356 | 15.805261 | -19.168437 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.687326 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.222649 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.687326 | 0.065294 |      0.0 |  0.0 |      0.0 |  0.0 | 0.065294 | 0.065294 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046546 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046546 | 0.012929 | 0.477351 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8755.595702 | 108.813897 | 1.787903 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012929 |
    | 1997-08-03 10:00:00 | 0.833333 | 20.3 |             76.8 |       1.5 |                      1.0 |              0.5 | 0.0 | 0.0 |                   1.7 |                     14.929734 |      501.583299 | 0.783333 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                         0.0 |                          0.0 |             18.32333 | 16.000482 | -25.797109 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.560684 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.389789 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.560684 | 0.053594 |      0.0 |  0.0 |      0.0 |  0.0 | 0.053594 | 0.053594 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046878 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046878 | 0.013022 | 0.310211 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8778.492812 | 109.320987 | 1.794618 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.013022 |
    | 1997-08-03 11:00:00 | 0.833333 | 21.4 |             71.8 |       1.2 |                      1.0 |              0.7 | 0.0 | 0.0 |                   2.2 |                     14.929734 |      615.018727 | 0.783333 | 21.4 |    0.961909 |           0.961909 |                25.528421 |                         0.0 |                          0.0 |            18.329406 | 16.275247 | -30.748518 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.393544 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.46611 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.393544 | 0.037795 |      0.0 |  0.0 |      0.0 |  0.0 | 0.037795 | 0.037795 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046847 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046847 | 0.013013 |  0.23389 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8806.341329 | 109.676737 | 1.785566 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.013013 |
    | 1997-08-03 12:00:00 | 0.833333 | 21.3 |             67.5 |       1.3 |                      1.0 |              0.8 | 0.0 | 0.0 |                   2.8 |                     14.929734 |      626.544326 | 0.783333 | 21.3 |    1.042069 |           1.042069 |                 25.37251 |                         0.0 |                          0.0 |            17.126444 | 16.609429 | -28.143424 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.317224 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.490849 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.317224 | 0.030572 |      0.0 |  0.0 |      0.0 |  0.0 | 0.030572 | 0.030572 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046519 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046519 | 0.012922 | 0.209151 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8831.584754 | 109.963389 | 1.769618 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012922 |
    | 1997-08-03 13:00:00 | 0.833333 | 21.8 |             66.1 |       1.5 |                      1.0 |              0.5 | 0.0 | 0.0 |                   3.2 |                     14.929734 |      496.133417 | 0.783333 | 21.8 |    1.202387 |           1.202387 |                26.160453 |                         0.0 |                          0.0 |             17.29206 |  16.91235 | -29.325898 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.292485 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.424547 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.292485 | 0.028272 |      0.0 |  0.0 |      0.0 |  0.0 | 0.028272 | 0.028272 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046078 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046078 | 0.012799 | 0.275453 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8858.010651 | 110.227602 | 1.751813 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012799 |
    | 1997-08-03 14:00:00 | 0.833333 | 22.9 |             63.4 |       1.9 |                      1.0 |              0.4 | 0.0 | 0.0 |                   3.4 |                     14.929734 |      419.520994 | 0.783333 | 22.9 |    1.523023 |           1.523023 |                27.969419 |                         0.0 |                          0.0 |            17.732611 | 17.229461 | -34.023233 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.358786 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.415176 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.358786 | 0.034791 |      0.0 |  0.0 |      0.0 |  0.0 | 0.034791 | 0.034791 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045702 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045702 | 0.012695 | 0.284824 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8889.133884 | 110.551596 | 1.740902 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012695 |
    | 1997-08-03 15:00:00 | 0.833333 | 22.7 |             62.4 |       1.9 |                      1.0 |              0.5 | 0.0 | 0.0 |                   3.8 |                     14.929734 |      387.887354 | 0.783333 | 22.7 |    1.523023 |           1.523023 |                27.632633 |                         0.0 |                          0.0 |            17.242763 |  17.60294 |  -30.58236 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.368157 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.393194 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.368157 | 0.035827 |      0.0 |  0.0 |      0.0 |  0.0 | 0.035827 | 0.035827 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045434 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045434 |  0.01262 | 0.306806 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8916.816245 | 110.883927 | 1.731295 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 |  0.01262 |
    | 1997-08-03 16:00:00 | 0.833333 | 22.5 |             61.1 |       2.3 |                      1.0 |              0.5 | 0.0 | 0.0 |                   4.1 |                     14.929734 |      278.496873 | 0.783333 | 22.5 |     1.84366 |            1.84366 |                27.299387 |                         0.0 |                          0.0 |            16.679926 | 17.935128 |  -27.38923 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.390139 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.34217 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.390139 | 0.038106 |      0.0 |  0.0 |      0.0 |  0.0 | 0.038106 | 0.038106 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045215 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045215 |  0.01256 |  0.35783 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8941.305475 | 111.235959 | 1.724186 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 |  0.01256 |
    | 1997-08-03 17:00:00 | 0.833333 | 21.9 |             62.1 |       2.4 |                      1.0 |              0.3 | 0.0 | 0.0 |                   4.4 |                     14.929734 |      137.138608 | 0.783333 | 21.9 |    1.923819 |           1.923819 |                26.320577 |                         0.0 |                          0.0 |            16.345078 | 18.228999 | -22.026006 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.441163 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.236219 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.441163 | 0.043264 |      0.0 |  0.0 |      0.0 |  0.0 | 0.043264 | 0.043264 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045098 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045098 | 0.012527 | 0.463781 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8960.431481 | 111.633858 | 1.722352 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012527 |
    | 1997-08-03 18:00:00 | 0.833333 | 21.4 |             67.0 |       2.5 |                      1.0 |              0.1 | 0.0 | 0.0 |                   4.5 |                     14.929734 |       51.080715 | 0.783333 | 21.4 |    2.003978 |           2.003978 |                25.528421 |                         0.0 |                          0.0 |            17.104042 | 18.458511 | -17.648933 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.547114 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.156077 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.547114 | 0.053912 |      0.0 |  0.0 |      0.0 |  0.0 | 0.053912 | 0.053912 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045189 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045189 | 0.012553 | 0.543923 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8975.180414 | 112.127061 | 1.731075 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012553 |
    | 1997-08-03 19:00:00 | 0.833333 | 20.7 |             74.5 |       2.5 |                      1.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.429734 |       13.632816 | 0.783333 | 20.7 |    2.003978 |           2.003978 |                24.454368 |                         0.0 |                          0.0 |            18.218504 | 18.635498 |  -12.38701 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.627256 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.09611 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.627256 |  0.06216 |      0.0 |  0.0 |      0.0 |  0.0 |  0.06216 |  0.06216 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045521 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045521 | 0.012645 |  0.60389 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8984.667424 | 112.692157 | 1.747713 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012645 |
    | 1997-08-03 20:00:00 | 0.833333 | 19.4 |             81.2 |       2.2 |                   0.1364 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |        0.185943 | 0.783333 | 19.4 |    1.763501 |           1.763501 |                22.563931 |                         0.0 |                          0.0 |            18.321912 | 18.749342 |  -3.903945 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.687223 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.049374 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.687223 | 0.068533 |      0.0 |  0.0 |      0.0 |  0.0 | 0.068533 | 0.068533 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046034 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046034 | 0.012787 | 0.650626 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  8985.67137 | 113.310848 | 1.770212 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012787 |
    | 1997-08-03 21:00:00 | 0.833333 | 17.8 |             86.9 |       1.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.783333 | 17.8 |    1.362705 |           1.362705 |                20.413369 |                         0.0 |                          0.0 |            17.739217 |  18.76139 |   5.768339 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.733959 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.019253 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.733959 | 0.073692 |      0.0 |  0.0 |      0.0 |  0.0 | 0.073692 | 0.073692 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046682 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046682 | 0.012967 | 0.680747 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8977.003031 | 113.971116 | 1.797222 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012967 |
    | 1997-08-03 22:00:00 | 0.833333 | 17.0 |             90.1 |       1.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.783333 | 17.0 |    1.362705 |           1.362705 |                19.406929 |                         0.0 |                          0.0 |            17.485643 |  18.65737 |   9.944218 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.764081 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.010292 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.764081 | 0.077264 |      0.0 |  0.0 |      0.0 |  0.0 | 0.077264 | 0.077264 |   0.0 |      0.0 |   0.0 |      0.0 | 0.047426 |   0.0 |      0.0 |   0.0 |      0.0 | 0.047426 | 0.013174 | 0.689708 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8964.158813 | 114.657933 | 1.827059 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.013174 |
    | 1997-08-03 23:00:00 | 0.833333 | 16.4 |             90.9 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.783333 | 16.4 |     1.84366 |            1.84366 |                 18.68084 |                         0.0 |                          0.0 |            16.980884 | 18.503239 |  12.619435 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.773041 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.014655 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.773041 | 0.078739 |      0.0 |  0.0 |      0.0 |  0.0 | 0.078739 | 0.078739 |   0.0 |      0.0 |   0.0 |      0.0 | 0.048216 |   0.0 |      0.0 |   0.0 |      0.0 | 0.048216 | 0.013393 | 0.685345 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8948.639378 | 115.352235 | 1.857583 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.013393 |
    | 1997-08-04 00:00:00 | 0.833333 | 15.9 |             88.0 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.783333 | 15.9 |     1.84366 |            1.84366 |                18.094032 |                         0.0 |                          0.0 |            15.922748 | 18.317006 |  14.502035 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.768678 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.024014 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.768678 | 0.078864 |      0.0 |  0.0 |      0.0 |  0.0 | 0.078864 | 0.078864 |   0.0 |      0.0 |   0.0 |      0.0 | 0.049006 |   0.0 |      0.0 |   0.0 |      0.0 | 0.049006 | 0.013613 | 0.675986 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8931.237343 | 116.042049 | 1.887441 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.013613 |
    | 1997-08-04 01:00:00 | 0.833333 | 15.4 |             89.2 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.783333 | 15.4 |    1.763501 |           1.763501 |                17.523416 |                         0.0 |                          0.0 |            15.630887 | 18.108181 |  16.249089 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.759319 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.018207 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.759319 | 0.078462 |      0.0 |  0.0 |      0.0 |  0.0 | 0.078462 | 0.078462 |   0.0 |      0.0 |   0.0 |      0.0 | 0.049772 |   0.0 |      0.0 |   0.0 |      0.0 | 0.049772 | 0.013826 | 0.681793 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8912.088255 | 116.722906 | 1.916131 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.013826 |
    | 1997-08-04 02:00:00 | 0.833333 | 15.2 |             88.8 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.783333 | 15.2 |     1.84366 |            1.84366 |                17.299618 |                         0.0 |                          0.0 |            15.362061 | 17.878392 |  16.070354 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.765127 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.020574 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.765127 | 0.079624 |      0.0 |  0.0 |      0.0 |  0.0 | 0.079624 | 0.079624 |   0.0 |      0.0 |   0.0 |      0.0 | 0.050528 |   0.0 |      0.0 |   0.0 |      0.0 | 0.050528 | 0.014036 | 0.679426 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |   8893.1179 | 117.408409 | 1.945227 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.014036 |
    | 1997-08-04 03:00:00 | 0.833333 | 14.9 |             89.7 |       2.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.783333 | 14.9 |    1.603182 |           1.603182 |                16.968612 |                         0.0 |                          0.0 |            15.220845 | 17.650748 |  16.504489 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.762759 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.013655 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.762759 | 0.079941 |      0.0 |  0.0 |      0.0 |  0.0 | 0.079941 | 0.079941 |   0.0 |      0.0 |   0.0 |      0.0 | 0.051284 |   0.0 |      0.0 |   0.0 |      0.0 | 0.051284 | 0.014246 | 0.686345 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8873.713411 | 118.091227 | 1.973884 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.014246 |
    | 1997-08-04 04:00:00 | 0.833333 | 14.3 |             93.0 |       2.3 |                 0.398044 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.534444 |        1.590516 | 0.783333 | 14.3 |     1.84366 |            1.84366 |                16.323194 |                         0.0 |                          0.0 |             15.18057 | 17.417894 |  18.707366 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.769679 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.007264 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.769679 | 0.081237 |      0.0 |  0.0 |      0.0 |  0.0 | 0.081237 | 0.081237 |   0.0 |      0.0 |   0.0 |      0.0 | 0.052041 |   0.0 |      0.0 |   0.0 |      0.0 | 0.052041 | 0.014456 | 0.692736 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8852.106046 | 118.779668 |  2.00308 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.014456 |
    | 1997-08-04 05:00:00 | 0.833333 | 14.1 |             93.6 |       2.5 |                      1.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.534444 |       19.950903 | 0.783333 | 14.1 |    2.003978 |           2.003978 |                16.112886 |                         0.0 |                          0.0 |            15.081661 | 17.158606 |  18.351635 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.776069 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.018172 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.776069 | 0.082493 |      0.0 |  0.0 |      0.0 |  0.0 | 0.082493 | 0.082493 |   0.0 |      0.0 |   0.0 |      0.0 | 0.052811 |   0.0 |      0.0 |   0.0 |      0.0 | 0.052811 |  0.01467 | 0.681828 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  8830.85441 | 119.473244 | 2.032762 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 |  0.01467 |
    | 1997-08-04 06:00:00 | 0.833333 | 14.3 |             94.6 |       2.4 |                      1.0 |              0.1 | 0.0 | 0.0 |                   4.6 |                     15.534444 |       62.985774 | 0.783333 | 14.3 |    1.923819 |           1.923819 |                16.323194 |                         0.0 |                          0.0 |            15.441741 | 16.903586 |  15.621518 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.765161 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.04094 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.765161 | 0.081907 | 0.003897 |  0.0 | 0.000003 |  0.0 | 0.081907 | 0.081907 |   0.0 | 0.003897 |   0.0 | 0.000003 |  0.05357 |   0.0 | 0.000025 |   0.0 |      0.0 | 0.053596 | 0.014888 |  0.65906 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8812.332893 | 120.152598 | 2.061098 |  0.0 | 0.003872 |  0.0 | 0.000003 |   0.0 | 0.014888 |
    | 1997-08-04 07:00:00 | 0.833333 | 15.4 |             93.3 |       2.5 |                      1.0 |              0.9 | 0.0 | 0.0 |                   5.5 |                     15.534444 |      285.776389 | 0.783333 | 15.4 |    2.003978 |           2.003978 |                17.523416 |                         0.0 |                          0.0 |            16.349347 | 16.681328 |   7.687968 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.742393 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.189733 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.742393 | 0.080011 |  0.00392 |  0.0 | 0.000031 |  0.0 | 0.080011 | 0.080011 |   0.0 |  0.00392 |   0.0 | 0.000031 | 0.054278 |   0.0 | 0.000076 |   0.0 |      0.0 | 0.054354 | 0.015098 | 0.510267 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8801.744925 | 120.811029 | 2.086831 |  0.0 | 0.007716 |  0.0 | 0.000034 |   0.0 | 0.015098 |
    | 1997-08-04 08:00:00 | 0.833333 | 16.8 |             86.9 |       2.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                   6.5 |                     15.534444 |      475.225247 | 0.783333 | 16.8 |    2.003978 |           2.003978 |                19.162205 |                         0.0 |                          0.0 |            16.651956 | 16.554272 |  -1.474365 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.5936 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.343753 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |   0.5936 | 0.064359 | 0.003941 |  0.0 | 0.000059 |  0.0 | 0.064359 | 0.064359 |   0.0 | 0.003941 |   0.0 | 0.000059 |  0.05474 |   0.0 | 0.000126 |   0.0 |      0.0 | 0.054865 |  0.01524 | 0.356247 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  8800.31929 |  121.33627 | 2.096451 |  0.0 | 0.011531 |  0.0 | 0.000093 |   0.0 |  0.01524 |
    | 1997-08-04 09:00:00 | 0.833333 | 18.2 |             78.9 |       2.9 |                      1.0 |              1.0 | 0.0 | 0.0 |                   7.3 |                     15.534444 |      614.006719 | 0.783333 | 18.2 |    2.324615 |           2.324615 |                20.933479 |                         0.0 |                          0.0 |            16.516515 | 16.537165 |  -9.977011 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.439581 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.494205 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.439581 |  0.04788 | 0.003958 |  0.0 | 0.000081 |  0.0 |  0.04788 |  0.04788 |   0.0 | 0.003958 |   0.0 | 0.000081 | 0.054774 |   0.0 | 0.000176 |   0.0 |      0.0 |  0.05495 | 0.015264 | 0.205795 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8807.396301 | 121.723932 | 2.089557 |  0.0 | 0.015313 |  0.0 | 0.000174 |   0.0 | 0.015264 |
    | 1997-08-04 10:00:00 | 0.833333 | 19.5 |             75.8 |       3.1 |                      1.0 |              0.9 | 0.0 | 0.0 |                   7.7 |                     15.534444 |      643.806508 | 0.783333 | 19.5 |    2.484933 |           2.484933 |                22.704676 |                         0.0 |                          0.0 |            17.210144 | 16.622089 | -17.267466 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.289128 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.550473 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.289128 | 0.031594 | 0.003971 |  0.0 | 0.000097 |  0.0 | 0.031594 | 0.031594 |   0.0 | 0.003971 |   0.0 | 0.000097 | 0.054385 |   0.0 | 0.000225 |   0.0 |      0.0 |  0.05461 | 0.015169 | 0.149527 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8821.763767 | 121.977399 | 2.066766 |  0.0 | 0.019059 |  0.0 | 0.000271 |   0.0 | 0.015169 |
    | 1997-08-04 11:00:00 | 0.833333 | 20.3 |             73.4 |       3.3 |                      1.0 |              0.8 | 0.0 | 0.0 |                   7.8 |                     15.534444 |       600.26298 | 0.783333 | 20.3 |    2.645251 |           2.645251 |                23.858503 |                         0.0 |                          0.0 |            17.512141 | 16.794499 | -21.033009 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.23286 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.548409 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.23286 | 0.025504 | 0.003979 |  0.0 | 0.000107 |  0.0 | 0.025504 | 0.025504 |   0.0 | 0.003979 |   0.0 | 0.000107 | 0.053717 |   0.0 | 0.000274 |   0.0 |      0.0 | 0.053991 | 0.014997 | 0.151591 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8839.896776 | 122.180668 | 2.038553 |  0.0 | 0.022764 |  0.0 | 0.000378 |   0.0 | 0.014997 |
    | 1997-08-04 12:00:00 | 0.833333 | 21.0 |             68.3 |       3.3 |                      1.0 |              0.9 | 0.0 | 0.0 |                   7.9 |                     15.534444 |       620.51414 | 0.783333 | 21.0 |    2.645251 |           2.645251 |                24.909749 |                         0.0 |                          0.0 |            17.013358 | 17.012095 | -23.927432 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.234924 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.597585 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.234924 | 0.025783 | 0.003986 |  0.0 | 0.000116 |  0.0 | 0.025783 | 0.025783 |   0.0 | 0.003986 |   0.0 | 0.000116 | 0.052992 |   0.0 | 0.000322 |   0.0 |      0.0 | 0.053314 | 0.014809 | 0.102415 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8860.924208 | 122.385708 | 2.011344 |  0.0 | 0.026428 |  0.0 | 0.000494 |   0.0 | 0.014809 |
    | 1997-08-04 13:00:00 | 0.833333 | 21.8 |             61.3 |       3.2 |                      1.0 |              0.8 | 0.0 | 0.0 |                   8.2 |                     15.534444 |      599.441386 | 0.783333 | 21.8 |    2.565092 |           2.565092 |                26.160453 |                         0.0 |                          0.0 |            16.036358 | 17.264424 | -27.213457 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.185749 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.621828 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.185749 | 0.020423 | 0.003992 |  0.0 | 0.000124 |  0.0 | 0.020423 | 0.020423 |   0.0 | 0.003992 |   0.0 | 0.000124 |  0.05222 |   0.0 |  0.00037 |   0.0 |      0.0 | 0.052589 | 0.014608 | 0.078172 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8885.237665 | 122.546917 | 1.979547 |  0.0 | 0.030051 |  0.0 | 0.000618 |   0.0 | 0.014608 |
    | 1997-08-04 14:00:00 | 0.833333 | 21.9 |             60.6 |       2.5 |                      1.0 |              0.9 | 0.0 | 0.0 |                   8.7 |                     15.534444 |      613.551223 | 0.783333 | 21.9 |    2.003978 |           2.003978 |                26.320577 |                         0.0 |                          0.0 |             15.95027 | 17.556185 | -26.062888 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.161505 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.589213 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.161505 | 0.017784 | 0.003998 |  0.0 | 0.000131 |  0.0 | 0.017784 | 0.017784 |   0.0 | 0.003998 |   0.0 | 0.000131 | 0.051364 |   0.0 | 0.000417 |   0.0 |      0.0 | 0.051781 | 0.014384 | 0.110787 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8908.400553 | 122.686509 | 1.945967 |  0.0 | 0.033632 |  0.0 | 0.000749 |   0.0 | 0.014384 |
    | 1997-08-04 15:00:00 | 0.833333 | 21.8 |             58.8 |       2.9 |                      1.0 |              0.9 | 0.0 | 0.0 |                   9.1 |                     15.534444 |      517.107047 | 0.783333 | 21.8 |    2.324615 |           2.324615 |                26.160453 |                         0.0 |                          0.0 |            15.382347 |  17.83414 |  -23.79516 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.194121 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.551455 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.194121 | 0.021409 | 0.004002 |  0.0 | 0.000137 |  0.0 | 0.021409 | 0.021409 |   0.0 | 0.004002 |   0.0 | 0.000137 | 0.050544 |   0.0 | 0.000463 |   0.0 |      0.0 | 0.051007 | 0.014169 | 0.148545 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8929.295714 | 122.855082 | 1.916832 |  0.0 | 0.037171 |  0.0 | 0.000886 |   0.0 | 0.014169 |
    | 1997-08-04 16:00:00 | 0.833333 | 22.2 |             57.2 |       3.6 |                      1.0 |              0.9 | 0.0 | 0.0 |                   9.5 |                     15.534444 |      377.836616 | 0.783333 | 22.2 |    2.885728 |           2.885728 |                26.806088 |                         0.0 |                          0.0 |            15.333082 | 18.084882 | -24.690709 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.231878 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.505795 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.231878 |  0.02562 | 0.004008 |  0.0 | 0.000144 |  0.0 |  0.02562 |  0.02562 |   0.0 | 0.004008 |   0.0 | 0.000144 | 0.049846 |   0.0 | 0.000509 |   0.0 |      0.0 | 0.050355 | 0.013988 | 0.194205 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8951.086422 | 123.057187 | 1.892607 |  0.0 | 0.040669 |  0.0 |  0.00103 |   0.0 | 0.013988 |
    | 1997-08-04 17:00:00 | 0.833333 | 21.8 |             57.2 |       4.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  10.2 |                     15.534444 |      252.204639 | 0.783333 | 21.8 |    3.366683 |           3.366683 |                26.160453 |                         0.0 |                          0.0 |            14.963779 |  18.34637 | -20.721778 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.277538 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.44464 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.277538 | 0.030734 | 0.004014 |  0.0 | 0.000152 |  0.0 | 0.030734 | 0.030734 |   0.0 | 0.004014 |   0.0 | 0.000152 | 0.049286 |   0.0 | 0.000555 |   0.0 |      0.0 | 0.049841 | 0.013845 |  0.25536 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |   8968.9082 | 123.299825 | 1.874054 |  0.0 | 0.044129 |  0.0 | 0.001182 |   0.0 | 0.013845 |
    | 1997-08-04 18:00:00 | 0.833333 | 20.9 |             59.9 |       4.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  11.1 |                     15.534444 |      128.576715 | 0.783333 | 20.9 |    3.366683 |           3.366683 |                 24.75714 |                         0.0 |                          0.0 |            14.829527 | 18.560232 |  -14.03861 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.338693 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.32482 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.338693 | 0.037607 | 0.004022 |  0.0 | 0.000162 |  0.0 | 0.037607 | 0.037607 |   0.0 | 0.004022 |   0.0 | 0.000162 | 0.048896 |   0.0 |   0.0006 |   0.0 |      0.0 | 0.049496 | 0.013749 |  0.37518 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8980.046809 | 123.596727 | 1.862765 |  0.0 | 0.047551 |  0.0 | 0.001344 |   0.0 | 0.013749 |
    | 1997-08-04 19:00:00 | 0.833333 | 19.7 |             66.1 |       3.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.1 |                     15.534444 |       42.072585 | 0.783333 | 19.7 |    2.885728 |           2.885728 |                22.988466 |                         0.0 |                          0.0 |            15.195376 | 18.693895 |   -6.03663 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.458513 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.183135 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.458513 | 0.051088 | 0.004032 |  0.0 | 0.000175 |  0.0 | 0.051088 | 0.051088 |   0.0 | 0.004032 |   0.0 | 0.000175 |  0.04878 |   0.0 | 0.000644 |   0.0 |      0.0 | 0.049424 | 0.013729 | 0.516865 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8983.183439 | 123.999945 | 1.865073 |  0.0 | 0.050939 |  0.0 | 0.001519 |   0.0 | 0.013729 |
    | 1997-08-04 20:00:00 | 0.833333 | 17.9 |             71.7 |       3.2 |                 0.104711 |              0.3 | 0.0 | 0.0 |                  12.4 |                     15.502754 |        0.812898 | 0.783333 | 17.9 |    2.565092 |           2.565092 |                20.542326 |                         0.0 |                          0.0 |            14.728848 | 18.731535 |   4.989208 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.600198 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.100625 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.600198 | 0.067185 | 0.004045 |  0.0 | 0.000192 |  0.0 | 0.067185 | 0.067185 |   0.0 | 0.004045 |   0.0 | 0.000192 | 0.049048 |   0.0 | 0.000689 |   0.0 |      0.0 | 0.049737 | 0.013816 | 0.599375 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8975.294232 | 124.528721 |  1.88321 |  0.0 | 0.054295 |  0.0 |  0.00171 |   0.0 | 0.013816 |
    | 1997-08-04 21:00:00 | 0.833333 | 15.8 |             80.6 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.783333 | 15.8 |    1.763501 |           1.763501 |                17.978626 |                         0.0 |                          0.0 |            14.490772 | 18.636864 |  17.021185 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.682708 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.029975 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.682708 |  0.07685 | 0.004062 |  0.0 | 0.000214 |  0.0 |  0.07685 |  0.07685 |   0.0 | 0.004062 |   0.0 | 0.000214 | 0.049642 |   0.0 | 0.000732 |   0.0 |      0.0 | 0.050374 | 0.013993 | 0.670025 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8955.373047 | 125.130304 | 1.910418 |  0.0 | 0.057625 |  0.0 | 0.001924 |   0.0 | 0.013993 |
    | 1997-08-04 22:00:00 | 0.833333 | 14.8 |             85.9 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.783333 | 14.8 |    1.763501 |           1.763501 |                16.859515 |                         0.0 |                          0.0 |            14.482324 |  18.39781 |  21.586859 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.753358 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.012059 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.753358 | 0.085333 | 0.004082 |  0.0 | 0.000239 |  0.0 | 0.085333 | 0.085333 |   0.0 | 0.004082 |   0.0 | 0.000239 | 0.050455 |   0.0 | 0.000776 |   0.0 |      0.0 | 0.051231 | 0.014231 | 0.687941 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8930.886187 | 125.794008 | 1.945297 |  0.0 | 0.060931 |  0.0 | 0.002162 |   0.0 | 0.014231 |
    | 1997-08-04 23:00:00 | 0.833333 | 14.1 |             89.3 |       2.1 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.783333 | 14.1 |    1.683342 |           1.683342 |                16.112886 |                         0.0 |                          0.0 |            14.388807 | 18.103968 |  24.023805 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.771275 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.000116 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.771275 | 0.087941 | 0.004104 |  0.0 | 0.000266 |  0.0 | 0.087941 | 0.087941 |   0.0 | 0.004104 |   0.0 | 0.000266 | 0.051389 |   0.0 | 0.000819 |   0.0 |      0.0 | 0.052209 | 0.014502 | 0.700116 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8903.962382 | 126.472972 | 1.981848 |  0.0 | 0.064216 |  0.0 | 0.002428 |   0.0 | 0.014502 |
    | 1997-08-05 00:00:00 | 0.833333 | 13.4 |             91.4 |       1.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.783333 | 13.4 |    1.442864 |           1.442864 |                15.395377 |                         0.0 |                          0.0 |            14.071374 | 17.780882 |  26.285292 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.78345 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.009023 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.78345 | 0.089929 | 0.004126 |  0.0 | 0.000295 |  0.0 | 0.089929 | 0.089929 |   0.0 | 0.004126 |   0.0 | 0.000295 | 0.052359 |   0.0 | 0.000862 |   0.0 |      0.0 | 0.053221 | 0.014784 | 0.709023 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  8874.77709 | 127.162072 | 2.019418 |  0.0 |  0.06748 |  0.0 | 0.002723 |   0.0 | 0.014784 |
    | 1997-08-05 01:00:00 | 0.833333 | 12.8 |             93.1 |       1.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.783333 | 12.8 |    1.442864 |           1.442864 |                14.802811 |                         0.0 |                          0.0 |            13.781417 | 17.430658 |  27.783951 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.792356 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.013264 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.792356 | 0.091568 | 0.004148 |  0.0 | 0.000323 |  0.0 | 0.091568 | 0.091568 |   0.0 | 0.004148 |   0.0 | 0.000323 | 0.053351 |   0.0 | 0.000904 |   0.0 |      0.0 | 0.054256 | 0.015071 | 0.713264 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  8844.09314 | 127.858389 | 2.057635 |  0.0 | 0.070724 |  0.0 | 0.003046 |   0.0 | 0.015071 |
    | 1997-08-05 02:00:00 | 0.833333 | 12.3 |             94.6 |       1.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.783333 | 12.3 |    1.282546 |           1.282546 |                14.324376 |                         0.0 |                          0.0 |            13.550859 | 17.062451 |  28.574706 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.796598 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.017718 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.796598 | 0.092684 | 0.004171 |  0.0 | 0.000352 |  0.0 | 0.092684 | 0.092684 |   0.0 | 0.004171 |   0.0 | 0.000352 | 0.054353 |   0.0 | 0.000947 |   0.0 |      0.0 |   0.0553 | 0.015361 | 0.717718 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8812.618434 | 128.557779 | 2.095966 |  0.0 | 0.073948 |  0.0 | 0.003398 |   0.0 | 0.015361 |
    | 1997-08-05 03:00:00 | 0.833333 | 12.0 |             95.6 |       1.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.783333 | 12.0 |    0.961909 |           0.961909 |                 14.04387 |                         0.0 |                          0.0 |            13.425939 | 16.684755 |  28.108527 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.801051 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.020716 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.801051 | 0.093837 | 0.004194 |  0.0 | 0.000382 |  0.0 | 0.093837 | 0.093837 |   0.0 | 0.004194 |   0.0 | 0.000382 | 0.055358 |   0.0 | 0.000989 |   0.0 | 0.000001 | 0.056347 | 0.015652 | 0.720716 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8781.609907 | 129.260418 | 2.134445 |  0.0 | 0.077153 |  0.0 | 0.003779 |   0.0 | 0.015652 |
    | 1997-08-05 04:00:00 | 0.833333 | 11.7 |             96.2 |       1.9 |                 0.366022 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.470733 |        1.482348 | 0.783333 | 11.7 |    1.523023 |           1.523023 |                13.768192 |                         0.0 |                          0.0 |            13.245001 | 16.312652 |  27.675913 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.804049 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.019578 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.804049 | 0.094829 | 0.004217 |  0.0 | 0.000411 |  0.0 | 0.094829 | 0.094829 |   0.0 | 0.004217 |   0.0 | 0.000411 | 0.056365 |   0.0 |  0.00103 |   0.0 | 0.000001 | 0.057396 | 0.015943 | 0.719578 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8751.033993 | 129.965011 | 2.172909 |  0.0 |  0.08034 |  0.0 | 0.004189 |   0.0 | 0.015943 |
    | 1997-08-05 05:00:00 | 0.833333 | 11.6 |             95.7 |       1.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.470733 |       21.135568 | 0.783333 | 11.6 |    1.042069 |           1.042069 |                 13.67736 |                         0.0 |                          0.0 |            13.089234 | 15.945741 |  26.074448 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.802911 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | -0.009263 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.802911 | 0.095337 |  0.00424 |  0.0 |  0.00044 |  0.0 | 0.095337 | 0.095337 |   0.0 |  0.00424 |   0.0 |  0.00044 | 0.057365 |   0.0 | 0.001072 |   0.0 | 0.000001 | 0.058437 | 0.016233 | 0.709263 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8722.059546 | 130.667906 | 2.210881 |  0.0 | 0.083507 |  0.0 | 0.004628 |   0.0 | 0.016233 |
    | 1997-08-05 06:00:00 | 0.833333 | 12.6 |             95.1 |       1.6 |                      1.0 |              0.6 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      125.883451 | 0.783333 | 12.6 |    1.282546 |           1.282546 |                14.609782 |                         0.0 |                          0.0 |            13.893903 | 15.598048 |  17.988287 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.792597 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.052361 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.792597 | 0.094743 | 0.004263 |  0.0 | 0.000469 |  0.0 | 0.094743 | 0.094743 |   0.0 | 0.004263 |   0.0 | 0.000469 | 0.058338 |   0.0 | 0.001113 |   0.0 | 0.000001 | 0.059452 | 0.016514 | 0.647639 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8701.171259 | 131.361028 | 2.247286 |  0.0 | 0.086656 |  0.0 | 0.005097 |   0.0 | 0.016514 |
    | 1997-08-05 07:00:00 | 0.833333 | 14.8 |             96.3 |       1.9 |                      1.0 |              0.9 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      305.954943 | 0.783333 | 14.8 |    1.523023 |           1.523023 |                16.859515 |                         0.0 |                          0.0 |            16.235713 | 15.347388 |   3.284331 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.730972 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.166779 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.730972 | 0.087933 | 0.004285 |  0.0 | 0.000498 |  0.0 | 0.087933 | 0.087933 |   0.0 | 0.004285 |   0.0 | 0.000498 |  0.05919 |   0.0 | 0.001154 |   0.0 | 0.000001 | 0.060345 | 0.016763 | 0.533221 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8694.986928 | 131.999284 | 2.276028 |  0.0 | 0.089787 |  0.0 | 0.005595 |   0.0 | 0.016763 |
    | 1997-08-05 08:00:00 | 0.833333 | 17.1 |             89.9 |       3.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      511.104575 | 0.783333 | 17.1 |    2.565092 |           2.565092 |                19.530313 |                         0.0 |                          0.0 |            17.557751 | 15.273176 | -10.960941 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.616554 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.358863 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.616554 | 0.074584 | 0.004306 |  0.0 | 0.000525 |  0.0 | 0.074584 | 0.074584 |   0.0 | 0.004306 |   0.0 | 0.000525 | 0.059759 |   0.0 | 0.001195 |   0.0 | 0.000001 | 0.060955 | 0.016932 | 0.341137 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8703.047869 | 132.536423 | 2.290853 |  0.0 | 0.092898 |  0.0 | 0.006119 |   0.0 | 0.016932 |
    | 1997-08-05 09:00:00 | 0.833333 | 19.4 |             79.2 |       4.0 |                      1.0 |              0.9 | 0.0 | 0.0 |                  12.8 |                     15.470733 |       629.51924 | 0.783333 | 19.4 |    3.206365 |           3.206365 |                22.563931 |                         0.0 |                          0.0 |            17.870634 | 15.369908 | -24.180553 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.424471 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.541806 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.424471 | 0.051569 | 0.004324 |  0.0 | 0.000547 |  0.0 | 0.051569 | 0.051569 |   0.0 | 0.004324 |   0.0 | 0.000547 | 0.059844 |   0.0 | 0.001236 |   0.0 | 0.000001 |  0.06108 | 0.016967 | 0.158194 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8724.328423 | 132.904454 | 2.282578 |  0.0 | 0.095986 |  0.0 | 0.006665 |   0.0 | 0.016967 |
    | 1997-08-05 10:00:00 | 0.833333 | 20.6 |             73.4 |       4.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.9 |                     15.470733 |       765.81659 | 0.783333 | 20.6 |     3.68732 |            3.68732 |                24.304197 |                         0.0 |                          0.0 |            17.839281 | 15.625274 | -29.848354 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.241527 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.7 | 0.010395 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.241527 | 0.029423 | 0.004336 |  0.0 | 0.000563 |  0.0 | 0.029423 | 0.029423 |   0.0 | 0.004336 |   0.0 | 0.000563 | 0.059343 |   0.0 | 0.001276 |   0.0 | 0.000001 |  0.06062 | 0.016839 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8751.276776 | 133.101265 | 2.252658 |  0.0 | 0.099045 |  0.0 | 0.007227 |   0.0 | 0.016839 |
    | 1997-08-05 11:00:00 | 0.833333 | 21.4 |             67.3 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.1 |                     15.470733 |      767.145843 | 0.783333 | 21.4 |    3.286524 |           3.286524 |                25.528421 |                         0.0 |                          0.0 |            17.180627 | 15.948655 | -32.708072 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.083333 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.7 | 0.029871 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010164 | 0.004342 |  0.0 | 0.000571 |  0.0 | 0.010164 | 0.010164 |   0.0 | 0.004342 |   0.0 | 0.000571 |  0.05832 |   0.0 | 0.001316 |   0.0 | 0.000001 | 0.059637 | 0.016566 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8781.084848 |  133.13965 | 2.204502 |  0.0 | 0.102071 |  0.0 | 0.007797 |   0.0 | 0.016566 |
    | 1997-08-05 12:00:00 | 0.833333 | 22.5 |             62.2 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.2 |                     15.470733 |      736.263854 | 0.783333 | 22.5 |    3.286524 |           3.286524 |                27.299387 |                         0.0 |                          0.0 |            16.980219 | 16.306352 | -37.161891 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.083333 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.7 | 0.046934 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010168 | 0.004343 |  0.0 | 0.000572 |  0.0 | 0.010168 | 0.010168 |   0.0 | 0.004343 |   0.0 | 0.000572 | 0.057076 |   0.0 | 0.001355 |   0.0 | 0.000001 | 0.058433 | 0.016231 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8815.346739 | 133.160966 | 2.157594 |  0.0 | 0.105059 |  0.0 | 0.008368 |   0.0 | 0.016231 |
    | 1997-08-05 13:00:00 | 0.833333 | 23.1 |             59.6 |       4.4 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.4 |                     15.470733 |      768.006183 | 0.783333 | 23.1 |    3.527001 |           3.527001 |                28.309776 |                         0.0 |                          0.0 |            16.872626 | 16.717494 | -38.295035 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.083333 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.7 | 0.091262 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.083333 |  0.01017 | 0.004344 |  0.0 | 0.000573 |  0.0 |  0.01017 |  0.01017 |   0.0 | 0.004344 |   0.0 | 0.000573 | 0.055864 |   0.0 | 0.001394 |   0.0 | 0.000001 |  0.05726 | 0.015905 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8850.741774 |  133.13795 | 2.111899 |  0.0 | 0.108009 |  0.0 |  0.00894 |   0.0 | 0.015905 |
    | 1997-08-05 14:00:00 | 0.833333 | 23.4 |             55.4 |       4.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.5 |                     15.470733 |       735.92937 | 0.783333 | 23.4 |     3.68732 |            3.68732 |                28.827078 |                         0.0 |                          0.0 |            15.970201 | 17.142235 | -37.546592 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.083333 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.7 | 0.105785 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010167 | 0.004343 |  0.0 | 0.000572 |  0.0 | 0.010167 | 0.010167 |   0.0 | 0.004343 |   0.0 | 0.000572 | 0.054684 |   0.0 | 0.001432 |   0.0 | 0.000001 | 0.056118 | 0.015588 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8885.388366 | 133.100415 | 2.067382 |  0.0 |  0.11092 |  0.0 | 0.009511 |   0.0 | 0.015588 |
    | 1997-08-05 15:00:00 | 0.833333 | 23.8 |             52.1 |       4.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.6 |                     15.470733 |      620.374953 | 0.783333 | 23.8 |    3.607161 |           3.607161 |                29.529613 |                         0.0 |                          0.0 |            15.384928 | 17.557994 | -37.452038 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.083333 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.7 | 0.063817 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010164 | 0.004342 |  0.0 | 0.000571 |  0.0 | 0.010164 | 0.010164 |   0.0 | 0.004342 |   0.0 | 0.000571 | 0.053534 |   0.0 |  0.00147 |   0.0 | 0.000001 | 0.055006 | 0.015279 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8919.940404 | 133.104855 | 2.024012 |  0.0 | 0.113791 |  0.0 |  0.01008 |   0.0 | 0.015279 |
    | 1997-08-05 16:00:00 | 0.833333 | 24.0 |             49.5 |       4.7 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      450.095263 | 0.783333 | 24.0 |    3.767479 |           3.767479 |                29.886439 |                         0.0 |                          0.0 |            14.793787 | 17.972618 | -36.164291 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.083333 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |       0.7 | 0.001774 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010164 | 0.004342 |  0.0 | 0.000571 |  0.0 | 0.010164 | 0.010164 |   0.0 | 0.004342 |   0.0 | 0.000571 | 0.052414 |   0.0 | 0.001508 |   0.0 | 0.000002 | 0.053923 | 0.014979 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8953.204695 | 133.171337 | 1.981763 |  0.0 | 0.116626 |  0.0 |  0.01065 |   0.0 | 0.014979 |
    | 1997-08-05 17:00:00 | 0.833333 | 24.0 |             48.1 |       4.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      273.614141 | 0.783333 | 24.0 |    3.607161 |           3.607161 |                29.886439 |                         0.0 |                          0.0 |            14.375377 |  18.37179 | -33.769262 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.083333 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.559369 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010171 | 0.004344 |  0.0 | 0.000574 |  0.0 | 0.010171 | 0.010171 |   0.0 | 0.004344 |   0.0 | 0.000574 | 0.051323 |   0.0 | 0.001544 |   0.0 | 0.000002 | 0.052869 | 0.014686 | 0.140631 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 8984.073957 | 133.239582 | 1.940611 |  0.0 | 0.119426 |  0.0 | 0.011222 |   0.0 | 0.014686 |
    | 1997-08-05 18:00:00 | 0.833333 | 23.7 |             49.4 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      133.368104 | 0.783333 | 23.7 |    3.286524 |           3.286524 |                29.352596 |                         0.0 |                          0.0 |            14.500182 | 18.742221 | -29.746675 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.223964 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.406457 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.223964 | 0.027368 | 0.004346 |  0.0 | 0.000577 |  0.0 | 0.027368 | 0.027368 |   0.0 | 0.004346 |   0.0 | 0.000577 | 0.050483 |   0.0 | 0.001581 |   0.0 | 0.000002 | 0.052065 | 0.014463 | 0.293543 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 9010.920632 | 133.431255 | 1.917496 |  0.0 | 0.122191 |  0.0 | 0.011797 |   0.0 | 0.014463 |
    | 1997-08-05 19:00:00 | 0.833333 | 22.8 |             57.0 |       3.8 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |       40.326502 | 0.783333 | 22.8 |    3.046047 |           3.046047 |                27.800581 |                         0.0 |                          0.0 |            15.846331 | 19.064381 | -22.413715 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.376876 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.257526 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.376876 | 0.046166 | 0.004353 |  0.0 | 0.000585 |  0.0 | 0.046166 | 0.046166 |   0.0 | 0.004353 |   0.0 | 0.000585 | 0.050129 |   0.0 | 0.001617 |   0.0 | 0.000002 | 0.051748 | 0.014374 | 0.442474 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 9030.434346 | 133.757028 | 1.913532 |  0.0 | 0.124927 |  0.0 |  0.01238 |   0.0 | 0.014374 |
    | 1997-08-05 20:00:00 | 0.833333 | 21.3 |             65.9 |       2.5 |                 0.072689 |              0.5 | 0.0 | 0.0 |                  13.9 |                     15.438712 |        0.867122 | 0.783333 | 21.3 |    2.003978 |           2.003978 |                 25.37251 |                         0.0 |                          0.0 |            16.720484 | 19.298545 | -12.008727 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.525807 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.102206 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.525807 | 0.064647 | 0.004363 |  0.0 | 0.000598 |  0.0 | 0.064647 | 0.064647 |   0.0 | 0.004363 |   0.0 | 0.000598 | 0.050267 |   0.0 | 0.001652 |   0.0 | 0.000002 | 0.051921 | 0.014423 | 0.597794 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 9039.543073 | 134.213226 | 1.927912 |  0.0 | 0.127638 |  0.0 | 0.012976 |   0.0 | 0.014423 |
    | 1997-08-05 21:00:00 | 0.833333 | 19.3 |             73.0 |       1.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 | 0.783333 | 19.3 |    1.523023 |           1.523023 |                22.423949 |                         0.0 |                          0.0 |            16.369483 |  19.40785 |   0.647101 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.681127 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.046254 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.681127 | 0.084156 | 0.004378 |  0.0 | 0.000617 |  0.0 | 0.084156 | 0.084156 |   0.0 | 0.004378 |   0.0 | 0.000617 | 0.050891 |   0.0 | 0.001688 |   0.0 | 0.000002 | 0.052581 | 0.014606 | 0.653746 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 9035.995972 | 134.805202 | 1.961177 |  0.0 | 0.130328 |  0.0 | 0.013591 |   0.0 | 0.014606 |
    | 1997-08-05 22:00:00 | 0.833333 | 18.3 |             76.7 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 | 0.783333 | 18.3 |    1.763501 |           1.763501 |                21.065302 |                         0.0 |                          0.0 |            16.157086 | 19.365285 |    6.39171 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.73708 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.042508 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.73708 | 0.091597 | 0.004398 |  0.0 | 0.000642 |  0.0 | 0.091597 | 0.091597 |   0.0 | 0.004398 |   0.0 | 0.000642 | 0.051847 |   0.0 | 0.001723 |   0.0 | 0.000002 | 0.053572 | 0.014881 | 0.657492 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 9026.704262 | 135.445645 | 2.000927 |  0.0 | 0.133003 |  0.0 | 0.014231 |   0.0 | 0.014881 |
    | 1997-08-05 23:00:00 | 0.833333 | 17.4 |             81.8 |       2.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 | 0.783333 | 17.4 |    2.164296 |           2.164296 |                19.904589 |                         0.0 |                          0.0 |            16.281953 | 19.253784 |  11.122707 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.740825 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.035577 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 | 0.740825 | 0.092616 | 0.004418 |  0.0 | 0.000669 |  0.0 | 0.092616 | 0.092616 |   0.0 | 0.004418 |   0.0 | 0.000669 | 0.052887 |   0.0 | 0.001758 |   0.0 | 0.000002 | 0.054647 |  0.01518 | 0.664423 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan | 9012.681555 | 136.088768 | 2.040656 |  0.0 | 0.135664 |  0.0 | 0.014898 |   0.0 |  0.01518 |

>>> round_(model.check_waterbalance(conditions_acker_heavy_rain))
0.0

.. _lland_knauf_ic_water_hourly:

water
-----

.. integration-test::

    >>> lnk(WASSER)
    >>> inputs.nied.series = 0.05
    >>> test("lland_knauf_ic_water_hourly",
    ...      axis1=(fluxes.nkor, fluxes.evi, fluxes.qah),
    ...      use_conditions=conditions_wasser)
    |                date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |  nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |  tz |  wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow | nbes | sbes | snowintmax | snowintrate | nbesinz | sbesinz | wniedinz | actualalbedoinz | wadainz | schmpotinz | schminz | gefrpotinz | gefrinz | evsinz |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatinz | wlatsnow | wsensinz | wsenssnow | wsurfinz | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga |      qah |       qa | inzp | stinz | sinz | esnowinz | asinz | wats | waes | esnow | taus | ebdn | bowa | sdg1 | sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-08-03 00:00:00 | 0.05 | 16.9 |             95.1 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 | 0.047 | 16.9 |    0.641273 |           0.641273 |                19.284227 |                         0.0 |                          0.0 |              18.3393 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.040545 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006455 | 0.001793 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001793 |
    | 1997-08-03 01:00:00 | 0.05 | 16.6 |             94.9 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 | 0.047 | 16.6 |    0.641273 |           0.641273 |                18.920184 |                         0.0 |                          0.0 |            17.955254 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.04041 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00659 | 0.001831 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001831 |
    | 1997-08-03 02:00:00 | 0.05 | 16.4 |             95.9 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 | 0.047 | 16.4 |    0.641273 |           0.641273 |                 18.68084 |                         0.0 |                          0.0 |            17.914926 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.040055 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006945 | 0.001929 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001929 |
    | 1997-08-03 03:00:00 | 0.05 | 16.3 |             96.7 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.5 |             0.0 | 0.047 | 16.3 |    0.641273 |           0.641273 |                18.562165 |                         0.0 |                          0.0 |            17.949613 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.039459 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007541 | 0.002095 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002095 |
    | 1997-08-03 04:00:00 | 0.05 | 16.0 |             97.2 |       0.8 |                 0.429734 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |        1.943686 | 0.047 | 16.0 |    0.641273 |           0.641273 |                18.210086 |                         0.0 |                          0.0 |            17.700204 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.038271 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.008729 | 0.002425 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002425 |
    | 1997-08-03 05:00:00 | 0.05 | 15.9 |             97.5 |       0.6 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |       21.932441 | 0.047 | 15.9 |    0.480955 |           0.480955 |                18.094032 |                         0.0 |                          0.0 |            17.641681 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.036944 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.010056 | 0.002793 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002793 |
    | 1997-08-03 06:00:00 | 0.05 | 16.0 |             97.7 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |       57.256187 | 0.047 | 16.0 |    0.721432 |           0.721432 |                18.210086 |                         0.0 |                          0.0 |            17.791254 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.035268 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.011732 | 0.003259 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.003259 |
    | 1997-08-03 07:00:00 | 0.05 | 16.6 |             97.4 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |      109.332844 | 0.047 | 16.6 |    0.721432 |           0.721432 |                18.920184 |                         0.0 |                          0.0 |            18.428259 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.033685 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.013315 | 0.003699 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.003699 |
    | 1997-08-03 08:00:00 | 0.05 | 17.4 |             96.8 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |      170.949152 | 0.047 | 17.4 |    0.721432 |           0.721432 |                19.904589 |                         0.0 |                          0.0 |            19.267642 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.032054 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.014946 | 0.004152 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.004152 |
    | 1997-08-03 09:00:00 | 0.05 | 19.0 |             86.1 |       1.3 |                      1.0 |              0.2 | 0.0 | 0.0 |                   1.5 |                     14.929734 |      311.762624 | 0.047 | 19.0 |    1.042069 |           1.042069 |                22.008543 |                         0.0 |                          0.0 |            18.949356 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.031746 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.015254 | 0.004237 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.004237 |
    | 1997-08-03 10:00:00 | 0.05 | 20.3 |             76.8 |       1.5 |                      1.0 |              0.5 | 0.0 | 0.0 |                   1.7 |                     14.929734 |      501.583299 | 0.047 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                         0.0 |                          0.0 |             18.32333 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.031752 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.015248 | 0.004236 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.004236 |
    | 1997-08-03 11:00:00 | 0.05 | 21.4 |             71.8 |       1.2 |                      1.0 |              0.7 | 0.0 | 0.0 |                   2.2 |                     14.929734 |      615.018727 | 0.047 | 21.4 |    0.961909 |           0.961909 |                25.528421 |                         0.0 |                          0.0 |            18.329406 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.031812 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.015188 | 0.004219 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.004219 |
    | 1997-08-03 12:00:00 | 0.05 | 21.3 |             67.5 |       1.3 |                      1.0 |              0.8 | 0.0 | 0.0 |                   2.8 |                     14.929734 |      626.544326 | 0.047 | 21.3 |    1.042069 |           1.042069 |                 25.37251 |                         0.0 |                          0.0 |            17.126444 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.033619 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.013381 | 0.003717 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.003717 |
    | 1997-08-03 13:00:00 | 0.05 | 21.8 |             66.1 |       1.5 |                      1.0 |              0.5 | 0.0 | 0.0 |                   3.2 |                     14.929734 |      496.133417 | 0.047 | 21.8 |    1.202387 |           1.202387 |                26.160453 |                         0.0 |                          0.0 |             17.29206 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.035172 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.011828 | 0.003286 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.003286 |
    | 1997-08-03 14:00:00 | 0.05 | 22.9 |             63.4 |       1.9 |                      1.0 |              0.4 | 0.0 | 0.0 |                   3.4 |                     14.929734 |      419.520994 | 0.047 | 22.9 |    1.523023 |           1.523023 |                27.969419 |                         0.0 |                          0.0 |            17.732611 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.036854 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.010146 | 0.002818 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002818 |
    | 1997-08-03 15:00:00 | 0.05 | 22.7 |             62.4 |       1.9 |                      1.0 |              0.5 | 0.0 | 0.0 |                   3.8 |                     14.929734 |      387.887354 | 0.047 | 22.7 |    1.523023 |           1.523023 |                27.632633 |                         0.0 |                          0.0 |            17.242763 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.038963 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.008037 | 0.002233 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002233 |
    | 1997-08-03 16:00:00 | 0.05 | 22.5 |             61.1 |       2.3 |                      1.0 |              0.5 | 0.0 | 0.0 |                   4.1 |                     14.929734 |      278.496873 | 0.047 | 22.5 |     1.84366 |            1.84366 |                27.299387 |                         0.0 |                          0.0 |            16.679926 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.041278 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005722 |  0.00159 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00159 |
    | 1997-08-03 17:00:00 | 0.05 | 21.9 |             62.1 |       2.4 |                      1.0 |              0.3 | 0.0 | 0.0 |                   4.4 |                     14.929734 |      137.138608 | 0.047 | 21.9 |    1.923819 |           1.923819 |                26.320577 |                         0.0 |                          0.0 |            16.345078 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |  0.04299 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00401 | 0.001114 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001114 |
    | 1997-08-03 18:00:00 | 0.05 | 21.4 |             67.0 |       2.5 |                      1.0 |              0.1 | 0.0 | 0.0 |                   4.5 |                     14.929734 |       51.080715 | 0.047 | 21.4 |    2.003978 |           2.003978 |                25.528421 |                         0.0 |                          0.0 |            17.104042 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |   0.0444 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0026 | 0.000722 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000722 |
    | 1997-08-03 19:00:00 | 0.05 | 20.7 |             74.5 |       2.5 |                      1.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.429734 |       13.632816 | 0.047 | 20.7 |    2.003978 |           2.003978 |                24.454368 |                         0.0 |                          0.0 |            18.218504 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.046089 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000911 | 0.000253 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000253 |
    | 1997-08-03 20:00:00 | 0.05 | 19.4 |             81.2 |       2.2 |                   0.1364 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |        0.185943 | 0.047 | 19.4 |    1.763501 |           1.763501 |                22.563931 |                         0.0 |                          0.0 |            18.321912 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-03 21:00:00 | 0.05 | 17.8 |             86.9 |       1.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.047 | 17.8 |    1.362705 |           1.362705 |                20.413369 |                         0.0 |                          0.0 |            17.739217 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-03 22:00:00 | 0.05 | 17.0 |             90.1 |       1.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.047 | 17.0 |    1.362705 |           1.362705 |                19.406929 |                         0.0 |                          0.0 |            17.485643 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-03 23:00:00 | 0.05 | 16.4 |             90.9 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.047 | 16.4 |     1.84366 |            1.84366 |                 18.68084 |                         0.0 |                          0.0 |            16.980884 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 00:00:00 | 0.05 | 15.9 |             88.0 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.047 | 15.9 |     1.84366 |            1.84366 |                18.094032 |                         0.0 |                          0.0 |            15.922748 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 01:00:00 | 0.05 | 15.4 |             89.2 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.047 | 15.4 |    1.763501 |           1.763501 |                17.523416 |                         0.0 |                          0.0 |            15.630887 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 02:00:00 | 0.05 | 15.2 |             88.8 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.047 | 15.2 |     1.84366 |            1.84366 |                17.299618 |                         0.0 |                          0.0 |            15.362061 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 03:00:00 | 0.05 | 14.9 |             89.7 |       2.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.047 | 14.9 |    1.603182 |           1.603182 |                16.968612 |                         0.0 |                          0.0 |            15.220845 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 04:00:00 | 0.05 | 14.3 |             93.0 |       2.3 |                 0.398044 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.534444 |        1.590516 | 0.047 | 14.3 |     1.84366 |            1.84366 |                16.323194 |                         0.0 |                          0.0 |             15.18057 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 05:00:00 | 0.05 | 14.1 |             93.6 |       2.5 |                      1.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.534444 |       19.950903 | 0.047 | 14.1 |    2.003978 |           2.003978 |                16.112886 |                         0.0 |                          0.0 |            15.081661 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 06:00:00 | 0.05 | 14.3 |             94.6 |       2.4 |                      1.0 |              0.1 | 0.0 | 0.0 |                   4.6 |                     15.534444 |       62.985774 | 0.047 | 14.3 |    1.923819 |           1.923819 |                16.323194 |                         0.0 |                          0.0 |            15.441741 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 07:00:00 | 0.05 | 15.4 |             93.3 |       2.5 |                      1.0 |              0.9 | 0.0 | 0.0 |                   5.5 |                     15.534444 |      285.776389 | 0.047 | 15.4 |    2.003978 |           2.003978 |                17.523416 |                         0.0 |                          0.0 |            16.349347 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 08:00:00 | 0.05 | 16.8 |             86.9 |       2.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                   6.5 |                     15.534444 |      475.225247 | 0.047 | 16.8 |    2.003978 |           2.003978 |                19.162205 |                         0.0 |                          0.0 |            16.651956 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 09:00:00 | 0.05 | 18.2 |             78.9 |       2.9 |                      1.0 |              1.0 | 0.0 | 0.0 |                   7.3 |                     15.534444 |      614.006719 | 0.047 | 18.2 |    2.324615 |           2.324615 |                20.933479 |                         0.0 |                          0.0 |            16.516515 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 10:00:00 | 0.05 | 19.5 |             75.8 |       3.1 |                      1.0 |              0.9 | 0.0 | 0.0 |                   7.7 |                     15.534444 |      643.806508 | 0.047 | 19.5 |    2.484933 |           2.484933 |                22.704676 |                         0.0 |                          0.0 |            17.210144 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 11:00:00 | 0.05 | 20.3 |             73.4 |       3.3 |                      1.0 |              0.8 | 0.0 | 0.0 |                   7.8 |                     15.534444 |       600.26298 | 0.047 | 20.3 |    2.645251 |           2.645251 |                23.858503 |                         0.0 |                          0.0 |            17.512141 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 12:00:00 | 0.05 | 21.0 |             68.3 |       3.3 |                      1.0 |              0.9 | 0.0 | 0.0 |                   7.9 |                     15.534444 |       620.51414 | 0.047 | 21.0 |    2.645251 |           2.645251 |                24.909749 |                         0.0 |                          0.0 |            17.013358 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 13:00:00 | 0.05 | 21.8 |             61.3 |       3.2 |                      1.0 |              0.8 | 0.0 | 0.0 |                   8.2 |                     15.534444 |      599.441386 | 0.047 | 21.8 |    2.565092 |           2.565092 |                26.160453 |                         0.0 |                          0.0 |            16.036358 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 14:00:00 | 0.05 | 21.9 |             60.6 |       2.5 |                      1.0 |              0.9 | 0.0 | 0.0 |                   8.7 |                     15.534444 |      613.551223 | 0.047 | 21.9 |    2.003978 |           2.003978 |                26.320577 |                         0.0 |                          0.0 |             15.95027 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 15:00:00 | 0.05 | 21.8 |             58.8 |       2.9 |                      1.0 |              0.9 | 0.0 | 0.0 |                   9.1 |                     15.534444 |      517.107047 | 0.047 | 21.8 |    2.324615 |           2.324615 |                26.160453 |                         0.0 |                          0.0 |            15.382347 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 16:00:00 | 0.05 | 22.2 |             57.2 |       3.6 |                      1.0 |              0.9 | 0.0 | 0.0 |                   9.5 |                     15.534444 |      377.836616 | 0.047 | 22.2 |    2.885728 |           2.885728 |                26.806088 |                         0.0 |                          0.0 |            15.333082 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 17:00:00 | 0.05 | 21.8 |             57.2 |       4.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  10.2 |                     15.534444 |      252.204639 | 0.047 | 21.8 |    3.366683 |           3.366683 |                26.160453 |                         0.0 |                          0.0 |            14.963779 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 18:00:00 | 0.05 | 20.9 |             59.9 |       4.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  11.1 |                     15.534444 |      128.576715 | 0.047 | 20.9 |    3.366683 |           3.366683 |                 24.75714 |                         0.0 |                          0.0 |            14.829527 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 19:00:00 | 0.05 | 19.7 |             66.1 |       3.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.1 |                     15.534444 |       42.072585 | 0.047 | 19.7 |    2.885728 |           2.885728 |                22.988466 |                         0.0 |                          0.0 |            15.195376 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 20:00:00 | 0.05 | 17.9 |             71.7 |       3.2 |                 0.104711 |              0.3 | 0.0 | 0.0 |                  12.4 |                     15.502754 |        0.812898 | 0.047 | 17.9 |    2.565092 |           2.565092 |                20.542326 |                         0.0 |                          0.0 |            14.728848 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 21:00:00 | 0.05 | 15.8 |             80.6 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.047 | 15.8 |    1.763501 |           1.763501 |                17.978626 |                         0.0 |                          0.0 |            14.490772 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 22:00:00 | 0.05 | 14.8 |             85.9 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.047 | 14.8 |    1.763501 |           1.763501 |                16.859515 |                         0.0 |                          0.0 |            14.482324 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 23:00:00 | 0.05 | 14.1 |             89.3 |       2.1 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.047 | 14.1 |    1.683342 |           1.683342 |                16.112886 |                         0.0 |                          0.0 |            14.388807 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 00:00:00 | 0.05 | 13.4 |             91.4 |       1.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.047 | 13.4 |    1.442864 |           1.442864 |                15.395377 |                         0.0 |                          0.0 |            14.071374 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 01:00:00 | 0.05 | 12.8 |             93.1 |       1.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.047 | 12.8 |    1.442864 |           1.442864 |                14.802811 |                         0.0 |                          0.0 |            13.781417 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 02:00:00 | 0.05 | 12.3 |             94.6 |       1.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.047 | 12.3 |    1.282546 |           1.282546 |                14.324376 |                         0.0 |                          0.0 |            13.550859 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 03:00:00 | 0.05 | 12.0 |             95.6 |       1.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.047 | 12.0 |    0.961909 |           0.961909 |                 14.04387 |                         0.0 |                          0.0 |            13.425939 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 04:00:00 | 0.05 | 11.7 |             96.2 |       1.9 |                 0.366022 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.470733 |        1.482348 | 0.047 | 11.7 |    1.523023 |           1.523023 |                13.768192 |                         0.0 |                          0.0 |            13.245001 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 05:00:00 | 0.05 | 11.6 |             95.7 |       1.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.470733 |       21.135568 | 0.047 | 11.6 |    1.042069 |           1.042069 |                 13.67736 |                         0.0 |                          0.0 |            13.089234 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 06:00:00 | 0.05 | 12.6 |             95.1 |       1.6 |                      1.0 |              0.6 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      125.883451 | 0.047 | 12.6 |    1.282546 |           1.282546 |                14.609782 |                         0.0 |                          0.0 |            13.893903 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 07:00:00 | 0.05 | 14.8 |             96.3 |       1.9 |                      1.0 |              0.9 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      305.954943 | 0.047 | 14.8 |    1.523023 |           1.523023 |                16.859515 |                         0.0 |                          0.0 |            16.235713 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 08:00:00 | 0.05 | 17.1 |             89.9 |       3.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      511.104575 | 0.047 | 17.1 |    2.565092 |           2.565092 |                19.530313 |                         0.0 |                          0.0 |            17.557751 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 09:00:00 | 0.05 | 19.4 |             79.2 |       4.0 |                      1.0 |              0.9 | 0.0 | 0.0 |                  12.8 |                     15.470733 |       629.51924 | 0.047 | 19.4 |    3.206365 |           3.206365 |                22.563931 |                         0.0 |                          0.0 |            17.870634 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 10:00:00 | 0.05 | 20.6 |             73.4 |       4.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.9 |                     15.470733 |       765.81659 | 0.047 | 20.6 |     3.68732 |            3.68732 |                24.304197 |                         0.0 |                          0.0 |            17.839281 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 11:00:00 | 0.05 | 21.4 |             67.3 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.1 |                     15.470733 |      767.145843 | 0.047 | 21.4 |    3.286524 |           3.286524 |                25.528421 |                         0.0 |                          0.0 |            17.180627 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 12:00:00 | 0.05 | 22.5 |             62.2 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.2 |                     15.470733 |      736.263854 | 0.047 | 22.5 |    3.286524 |           3.286524 |                27.299387 |                         0.0 |                          0.0 |            16.980219 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 13:00:00 | 0.05 | 23.1 |             59.6 |       4.4 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.4 |                     15.470733 |      768.006183 | 0.047 | 23.1 |    3.527001 |           3.527001 |                28.309776 |                         0.0 |                          0.0 |            16.872626 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 14:00:00 | 0.05 | 23.4 |             55.4 |       4.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.5 |                     15.470733 |       735.92937 | 0.047 | 23.4 |     3.68732 |            3.68732 |                28.827078 |                         0.0 |                          0.0 |            15.970201 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 15:00:00 | 0.05 | 23.8 |             52.1 |       4.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.6 |                     15.470733 |      620.374953 | 0.047 | 23.8 |    3.607161 |           3.607161 |                29.529613 |                         0.0 |                          0.0 |            15.384928 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 16:00:00 | 0.05 | 24.0 |             49.5 |       4.7 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      450.095263 | 0.047 | 24.0 |    3.767479 |           3.767479 |                29.886439 |                         0.0 |                          0.0 |            14.793787 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 17:00:00 | 0.05 | 24.0 |             48.1 |       4.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      273.614141 | 0.047 | 24.0 |    3.607161 |           3.607161 |                29.886439 |                         0.0 |                          0.0 |            14.375377 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 18:00:00 | 0.05 | 23.7 |             49.4 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      133.368104 | 0.047 | 23.7 |    3.286524 |           3.286524 |                29.352596 |                         0.0 |                          0.0 |            14.500182 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 19:00:00 | 0.05 | 22.8 |             57.0 |       3.8 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |       40.326502 | 0.047 | 22.8 |    3.046047 |           3.046047 |                27.800581 |                         0.0 |                          0.0 |            15.846331 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 20:00:00 | 0.05 | 21.3 |             65.9 |       2.5 |                 0.072689 |              0.5 | 0.0 | 0.0 |                  13.9 |                     15.438712 |        0.867122 | 0.047 | 21.3 |    2.003978 |           2.003978 |                 25.37251 |                         0.0 |                          0.0 |            16.720484 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 21:00:00 | 0.05 | 19.3 |             73.0 |       1.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 | 0.047 | 19.3 |    1.523023 |           1.523023 |                22.423949 |                         0.0 |                          0.0 |            16.369483 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 22:00:00 | 0.05 | 18.3 |             76.7 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 | 0.047 | 18.3 |    1.763501 |           1.763501 |                21.065302 |                         0.0 |                          0.0 |            16.157086 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 23:00:00 | 0.05 | 17.4 |             81.8 |       2.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 | 0.047 | 17.4 |    2.164296 |           2.164296 |                19.904589 |                         0.0 |                          0.0 |            16.281953 | nan | 0.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |  0.0 |  0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |     0.0 |      0.0 |      0.0 |       0.0 |      0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |   0.0 |  0.0 |      0.0 |   nan |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |

>>> inputs.nied.series = 0.0
>>> round_(model.check_waterbalance(conditions_wasser))
0.0

.. _lland_knauf_ic_acker_winter_hourly:

acre (winter)
-------------

>>> pub.timegrids = "2010-12-10", "2010-12-13", "1h"

>>> inputs.nied.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.1, 0.0, 0.3, 0.2, 0.4, 0.7, 0.5, 1.0, 2.0, 2.1, 1.8, 1.3, 0.5, 0.2, 0.5, 0.2,
...     0.1, 0.3, 0.1, 0.1, 0.0, 0.1, 0.0, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.1,
...     0.0, 0.0, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0)
>>> inputs.teml.series = (
...     -3.7, -4.8, -5.6, -6.2, -7.3, -8.3, -8.6, -8.5, -9.5, -9.2, -6.9, -4.0, -2.8,
...     -1.4, -0.4, 0.5, 1.0, 1.1, 1.0, 1.4, 1.6, 2.0, 2.4, 2.5, 2.7, 3.2, 3.9, 4.4,
...     4.8, 4.8, 5.1, 5.1, 5.4, 5.3, 5.5, 6.1, 6.4, 6.4, 6.5, 6.3, 5.9, 5.5, 5.1, 4.8,
...     4.9, 4.9, 4.7, 3.8, 3.9, 3.7, 3.5, 3.3, 3.3, 3.0, 1.9, 1.2, 0.8, 0.8, 0.5, 0.3,
...     -0.1, -0.4, -0.6, -0.8, -1.3, -1.6, -2.0, -2.1, -2.2, -2.6, -3.1, -3.9)
>>> inputs.windspeed.series = (
...     3.4, 2.9, 2.5, 3.0, 2.5, 2.3, 2.1, 1.4, 1.1, 1.0, 1.4, 1.3, 1.3, 1.4, 2.3, 2.8,
...     3.1, 3.5, 4.4, 3.8, 5.7, 5.6, 5.8, 6.2, 5.6, 5.2, 5.7, 6.6, 7.0, 7.3, 7.2, 7.6,
...     7.3, 7.1, 7.1, 8.9, 9.5, 7.9, 9.6, 9.1, 8.7, 8.9, 7.9, 7.1, 6.4, 7.2, 7.0, 6.8,
...     6.4, 6.3, 5.7, 5.2, 4.6, 4.9, 6.3, 5.8, 6.1, 5.9, 6.4, 6.8, 7.0, 6.3, 7.3, 7.5,
...     7.0, 6.0, 4.8, 4.2, 4.9, 4.0, 3.4, 3.4)
>>> inputs.relativehumidity.series = (
...     87.5, 90.7, 89.7, 91.3, 94.2, 93.2, 96.5, 95.9, 96.8, 94.3, 96.7, 97.4, 93.9,
...     93.4, 93.3, 94.1, 94.6, 94.4, 96.9, 98.3, 98.7, 99.6, 99.4, 99.2, 99.2, 99.6,
...     99.8, 99.8, 99.0, 97.6, 97.4, 96.3, 95.2, 96.3, 95.5, 91.6, 90.0, 85.7, 82.3,
...     82.5, 82.2, 76.5, 81.7, 83.4, 82.8, 85.8, 87.6, 85.6, 85.3, 86.2, 89.3, 91.4,
...     90.9, 89.2, 85.6, 86.3, 91.2, 87.5, 84.5, 76.7, 74.8, 69.7, 66.9, 64.4, 65.4,
...     67.6, 70.2, 70.4, 69.5, 71.6, 74.5, 80.2)

>>> model.radiationmodel.sequences.inputs.sunshineduration.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.2,
...     0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0)

>>> aetinputs.atmosphericpressure.series = (
...     1017.0, 1018.0, 1018.0, 1019.0, 1020.0, 1020.0, 1021.0, 1021.0, 1022.0, 1022.0,
...     1022.0, 1022.0, 1022.0, 1021.0, 1020.0, 1020.0, 1019.0, 1018.0, 1016.0, 1015.0,
...     1014.0, 1013.0, 1011.0, 1010.0, 1008.0, 1007.0, 1006.0, 1005.0, 1005.0, 1004.0,
...     1004.0, 1004.0, 1003.0, 1003.0, 1003.0, 1002.0, 1002.0, 1002.0, 1002.0, 1003.0,
...     1003.0, 1003.0, 1003.0, 1003.0, 1003.0, 1003.0, 1003.0, 1002.0, 1002.0, 1002.0,
...     1002.0, 1003.0, 1004.0, 1005.0, 1006.0, 1008.0, 1009.0, 1010.0, 1011.0, 1012.0,
...     1013.0, 1014.0, 1015.0, 1015.0, 1017.0, 1017.0, 1018.0, 1019.0, 1019.0, 1020.0,
...     1020.0, 1020.0)
>>> aetinputs.windspeed.series = inputs.windspeed.series
>>> aetinputs.relativehumidity.series = inputs.relativehumidity.series

>>> test.inits = (
...     (logs.loggedsunshineduration,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.04, 0.25, 0.59, 0.91,
...       0.97, 1.0, 0.65, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]),
...     (logs.loggedpossiblesunshineduration,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.4, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
...       1.0, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]),
...     (model.radiationmodel.sequences.logs.loggedunadjustedglobalradiation,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 27.777778, 55.555556,
...       111.111111, 166.666667, 138.888889, 55.555556, 0.0,0.0, 0.0, 0.0, 0.0, 0.0,
...       0.0, 0.0, 0.0]),
...     (model.aetmodel.sequences.logs.loggedglobalradiation,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 27.8, 55.6, 111.1, 166.7, 138.9,
...       55.6, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]),
...     (model.aetmodel.sequences.logs.loggedairtemperature,
...      [[-2.8], [-2.6], [-2.2], [-1.8], [-1.8], [-2.4], [-1.5], [-1.4], [-0.8],
...       [-0.8], [-0.9], [0.9], [-0.8], [-1.0], [-1.2], [-1.8], [-2.8], [-3.8],
...       [-4.3], [-4.0], [-3.9], [-1.6], [-1.6], [2.6]]),
...     (model.aetmodel.sequences.logs.loggedrelativehumidity,
...      [99.4, 99.0, 99.6, 99.6, 99.3, 99.3, 99.8, 99.7, 100.0, 99.6, 99.3, 98.9,
...       96.6, 93.0, 91.4, 92.0, 94.5, 96.5, 98.2, 96.9, 98.0, 95.0, 91.8, 88.5]),
...     (model.aetmodel.sequences.logs.loggedwindspeed2m,
...      [2.2, 2.0, 2.5, 2.6, 2.3, 1.7, 2.8, 1.9, 2.6, 2.9, 3.5, 3.4, 3.5, 3.0, 3.9,
...       3.6, 2.2, 2.4, 2.5, 1.5, 2.5, 3.0, 3.2, 3.1]))

.. integration-test::

    >>> lnk(ACKER)
    >>> test("lland_knauf_ic_acker_winter_hourly",
    ...      axis1=(inputs.nied, fluxes.wada), axis2=(states.waes, states.wats),
    ...      use_conditions=conditions_acker_winter)
    |                date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |  nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow |     nbes |    sbes | snowintmax | snowintrate | nbesinz | sbesinz | wniedinz | actualalbedoinz | wadainz | schmpotinz | schminz | gefrpotinz | gefrinz | evsinz |      evi |      evb |       evs |    wnied | tempssurface | actualalbedo |  schmpot |     schm |  gefrpot |     gefr | wlatinz |   wlatsnow | wsensinz |  wsenssnow | wsurfinz |       wsurf |      sff |      fvg |     wada |      qdb | qib1 | qib2 | qbb | qkap |     qdgz |    qdgz1 |    qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 |    qdga2 | qiga1 | qiga2 | qbga |      qah |       qa |     inzp | stinz | sinz | esnowinz | asinz |      wats |      waes |       esnow |     taus |        ebdn |      bowa |     sdg1 |     sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-12-10 00:00:00 |  0.0 | -3.7 |             87.5 |       3.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -3.7 |     2.72541 |            2.72541 |                  4.64297 |                         0.0 |                     4.116122 |             4.062599 | -5.337255 |  -0.292799 |                      0.0 |                       0.0 |                     0.0 |                73.927318 |             0.0 |       -73.927318 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 |  0.000809 |      0.0 |    -5.288455 |     0.754325 |      0.0 |      0.0 | 0.697534 | 0.697534 |     0.0 |   0.599182 |      0.0 | -10.103618 |      0.0 |   64.422883 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.320358 | 20.349008 |         0.0 | 0.860328 | -435.895126 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 01:00:00 |  0.0 | -4.8 |             90.7 |       2.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -4.8 |    2.324615 |           2.324615 |                 4.272186 |                         0.0 |                     3.961569 |             3.874873 | -5.230742 |   3.347272 |                      0.0 |                       0.0 |                     0.0 |                76.610661 |             0.0 |       -76.610661 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 |  0.001178 |      0.0 |     -5.78862 |     0.753662 |      0.0 |      0.0 | 0.738127 |  0.02865 |     0.0 |   0.872698 |      0.0 |  -5.654298 |      0.0 |   71.829062 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.347831 | 20.347831 |  -65.823721 | 0.867636 | -430.659065 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 02:00:00 |  0.0 | -5.6 |             89.7 |       2.5 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -5.6 |    2.003978 |           2.003978 |                 4.019239 |                         0.0 |                     3.241256 |             3.605258 | -5.167909 |  19.248028 |                      0.0 |                       0.0 |                     0.0 |                70.157201 |             0.0 |       -70.157201 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.004502 |      0.0 |    -8.375913 |     0.753424 |      0.0 |      0.0 | 1.066474 |      0.0 |     0.0 |  -3.335424 |      0.0 | -14.452419 |      0.0 |   52.369359 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.352332 | 20.352332 |  -98.945052 | 0.870283 |  -441.32376 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 03:00:00 |  0.0 | -6.2 |             91.3 |       3.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -6.2 |    2.404774 |           2.404774 |                 3.838328 |                         0.0 |                     2.929796 |             3.504393 | -5.295885 |  26.162948 |                      0.0 |                       0.0 |                     0.0 |                67.439656 |             0.0 |       -67.439656 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.007981 |      0.0 |    -9.656376 |     0.753232 |      0.0 |      0.0 | 1.229782 |      0.0 |     0.0 |  -5.913667 |      0.0 | -20.211638 |      0.0 |   41.314351 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.360313 | 20.360313 | -114.096455 |  0.87242 | -458.903374 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 04:00:00 |  0.0 | -7.3 |             94.2 |       2.5 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -7.3 |    2.003978 |           2.003978 |                  3.52531 |                         0.0 |                     2.688706 |             3.320842 |  -5.50684 |  31.359309 |                      0.0 |                       0.0 |                     0.0 |                67.802576 |             0.0 |       -67.802576 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.007817 |      0.0 |   -10.733392 |     0.753056 |      0.0 |      0.0 | 1.367481 |      0.0 |     0.0 |  -5.792391 |      0.0 | -17.875492 |      0.0 |   44.134693 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.368131 | 20.368131 | -126.871838 | 0.874373 | -481.679351 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 05:00:00 |  0.0 | -8.3 |             93.2 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -8.3 |     1.84366 |            1.84366 |                  3.26061 |                         0.0 |                     2.483084 |             3.038889 | -5.780152 |    35.6513 |                      0.0 |                       0.0 |                     0.0 |                69.171166 |             0.0 |       -69.171166 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.006535 |      0.0 |   -11.722035 |     0.752894 |      0.0 |      0.0 | 1.494012 |      0.0 |     0.0 |   -4.84203 |      0.0 | -16.938582 |      0.0 |   47.390554 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.374665 | 20.374665 | -138.611093 | 0.876187 | -508.747317 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 06:00:00 |  0.0 | -8.6 |             96.5 |       2.1 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -8.6 |    1.683342 |           1.683342 |                 3.184716 |                         0.0 |                     2.368455 |             3.073251 | -6.104968 |  37.201561 |                      0.0 |                       0.0 |                     0.0 |                67.356137 |             0.0 |       -67.356137 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.007857 |      0.0 |   -12.305228 |     0.752742 |      0.0 |      0.0 | 1.568845 |      0.0 |     0.0 |   -5.82182 |      0.0 | -17.389919 |      0.0 |   44.144398 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.382523 | 20.382523 | -145.553929 |  0.87788 | -537.365545 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 07:00:00 |  0.0 | -8.5 |             95.9 |       1.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                  3.76 |                           7.4 |             0.0 |   0.0 | -8.5 |    1.122228 |           1.122228 |                 3.209838 |                         0.0 |                     2.300334 |             3.078235 | -6.448387 |  37.292525 |                      0.0 |                       0.0 |                     0.0 |                62.592321 |             0.0 |       -62.592321 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.007013 |      0.0 |   -12.663807 |     0.752597 |      0.0 |      0.0 | 1.615184 |      0.0 |     0.0 |  -5.196531 |      0.0 | -15.803999 |      0.0 |   41.591791 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.389536 | 20.389536 | -149.853196 | 0.879508 | -566.074736 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 08:00:00 |  0.0 | -9.5 |             96.8 |       1.1 |                 0.355232 |              0.0 | 0.0 | 0.0 |                  2.76 |                      6.755232 |         1.10761 |   0.0 | -9.5 |     0.88175 |            0.88175 |                 2.966351 |                         0.0 |                     2.207601 |             2.871427 | -6.792897 |  38.247153 |                      0.0 |                  0.274183 |                     0.0 |                61.028121 |             0.0 |       -60.753939 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.005378 |      0.0 |   -13.167422 |     0.752456 |      0.0 |      0.0 | 1.679995 |      0.0 |     0.0 |  -3.984955 |      0.0 | -12.508846 |      0.0 |   44.260137 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.394914 | 20.394914 |  -155.86618 | 0.881095 | -595.738556 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 09:00:00 |  0.0 | -9.2 |             94.3 |       1.0 |                      1.0 |              0.0 | 0.0 | 0.0 |                  1.79 |                      6.755232 |       18.663203 |   0.0 | -9.2 |    0.801591 |           0.801591 |                 3.037615 |                         0.0 |                     2.243104 |             2.864471 | -7.148863 |  34.941383 |                      0.0 |                  4.622516 |                     0.0 |                53.269158 |             0.0 |       -48.646642 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.004845 |      0.0 |   -12.972427 |     0.752319 |      0.0 |      0.0 | 1.655552 |      0.0 |     0.0 |  -3.589807 |      0.0 | -12.383164 |      0.0 |   32.673671 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.399759 | 20.399759 | -153.598467 | 0.882628 | -622.096606 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 10:00:00 |  0.0 | -6.9 |             96.7 |       1.4 |                      1.0 |              0.0 | 0.0 | 0.0 |                  0.88 |                      6.755232 |       56.267907 |   0.0 | -6.9 |    1.122228 |           1.122228 |                 3.636412 |                         0.0 |                     2.591489 |             3.516411 | -7.465159 |   22.36178 |                      0.0 |                  13.94424 |                     0.0 |                 37.72771 |             0.0 |       -23.783469 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.008339 |      0.0 |   -11.192123 |     0.752181 |      0.0 |      0.0 | 1.428688 |      0.0 |     0.0 |  -6.178658 |      0.0 | -16.291028 |      0.0 |    1.313783 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.408097 | 20.408097 | -132.550471 | 0.884182 | -635.875052 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 11:00:00 |  0.0 | -4.0 |             97.4 |       1.3 |                      1.0 |              0.1 | 0.0 | 0.0 |                  0.39 |                      6.755232 |      107.168237 |   0.0 | -4.0 |    1.042069 |           1.042069 |                 4.539126 |                         0.0 |                     3.202943 |             4.421109 | -7.630501 |   5.381266 |                      0.0 |                 26.574985 |                     0.0 |                24.904358 |             0.0 |         1.670627 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.010611 |      0.0 |    -8.527378 |     0.752026 |      0.0 |      0.0 | 1.088975 |      0.0 |     0.0 |  -7.862607 |      0.0 | -16.603299 |      0.0 |  -26.136533 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.418709 | 20.418709 | -101.032672 | 0.885939 | -632.672985 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 12:00:00 |  0.0 | -2.8 |             93.9 |       1.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                  0.14 |                      6.755232 |       91.473333 |   0.0 | -2.8 |    1.042069 |           1.042069 |                 4.967227 |                         0.0 |                     3.625473 |             4.664226 | -7.592076 |  -3.919048 |                      0.0 |                 22.700159 |                     0.0 |                21.886755 |             0.0 |         0.813404 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.009049 |      0.0 |    -6.938901 |     0.751839 |      0.0 |      0.0 | 0.886582 |      0.0 |     0.0 |  -6.704598 |      0.0 | -15.178632 |      0.0 |  -22.696634 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.427757 | 20.427757 |  -82.255086 | 0.888053 | -620.170603 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 13:00:00 |  0.0 | -1.4 |             93.4 |       1.4 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      6.755232 |       80.475535 |   0.0 | -1.4 |    1.122228 |           1.122228 |                 5.511494 |                         0.0 |                     4.003734 |             5.147736 | -7.442047 | -10.749432 |                      0.0 |                 19.987783 |                     0.0 |                17.769887 |             0.0 |         2.217896 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.010314 |      0.0 |    -5.650475 |     0.751629 |      0.0 |      0.0 |  0.72228 |      0.0 |     0.0 |  -7.642151 |      0.0 | -16.132953 |      0.0 |     -25.993 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.438071 | 20.438071 |  -67.011518 | 0.890426 | -600.837838 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 14:00:00 |  0.0 | -0.4 |             93.3 |       2.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      6.755232 |       43.120591 |   0.0 | -0.4 |     1.84366 |            1.84366 |                 5.931876 |                         0.0 |                     4.318486 |              5.53444 | -7.210054 | -15.311558 |                      0.0 |                 10.719882 |                     0.0 |                15.337726 |             0.0 |        -4.617844 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.014296 |      0.0 |    -4.658128 |     0.751398 |      0.0 |      0.0 | 0.595732 |      0.0 |     0.0 | -10.593086 |      0.0 | -21.077118 |      0.0 |   -27.05236 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.452368 | 20.452368 |  -55.270716 | 0.893056 | -576.942947 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 15:00:00 |  0.0 |  0.5 |             94.1 |       2.8 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.355232 |       10.320091 |   0.0 |  0.5 |    2.244455 |           2.244455 |                 6.334173 |                         0.0 |                     4.592753 |             5.960456 | -6.923315 | -18.473943 |                      0.0 |                  2.568208 |                     0.0 |                12.218458 |             0.0 |         -9.65025 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.018164 |      0.0 |    -3.844325 |     0.751145 |      0.0 |      0.0 | 0.491998 |      0.0 |     0.0 | -13.458731 |      0.0 |  -24.28968 |      0.0 |  -28.098161 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.470532 | 20.470532 |  -45.646498 | 0.895936 | -549.885671 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 16:00:00 |  0.1 |  1.0 |             94.6 |       3.1 |                 0.061898 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |        0.028102 | 0.094 |  1.0 |    2.484933 |           2.484933 |                 6.567919 |                         0.0 |                     4.829803 |             6.213251 | -6.598628 | -20.542414 |                      0.0 |                  0.006996 |                     0.0 |                11.447885 |             0.0 |       -11.440889 |    0.094 |  0.0235 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.019637 | 0.095635 |    -3.174892 |     0.751053 |      0.0 |      0.0 | 0.409957 |   0.0705 |     0.0 | -14.550516 |      0.0 | -24.948708 |      0.0 |  -28.058335 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.584169 | 20.584169 |  -31.494109 |  0.89699 | -520.759924 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 17:00:00 |  0.0 |  1.1 |             94.4 |       3.5 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |   0.0 |  1.1 |    2.805569 |           2.805569 |                 6.615571 |                         0.0 |                     5.064957 |             6.245099 | -6.249119 | -22.261106 |                      0.0 |                       0.0 |                     0.0 |                13.676209 |             0.0 |       -13.676209 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 |  -0.01819 |      0.0 |    -2.538935 |     0.750727 |      0.0 |      0.0 | 0.327028 |      0.0 |     0.0 | -13.477788 |      0.0 | -23.612723 |      0.0 |  -23.414302 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.602359 | 20.602359 |  -30.340914 | 0.900715 | -489.915484 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 18:00:00 |  0.3 |  1.0 |             96.9 |       4.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.282 |  1.0 |    3.527001 |           3.527001 |                 6.567919 |                         0.0 |                     5.196402 |             6.364313 | -5.878986 | -22.105464 |                      0.0 |                       0.0 |                     0.0 |                14.849938 |             0.0 |       -14.849938 |    0.282 |  0.0705 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.021203 | 0.286904 |    -2.194742 |     0.750946 |      0.0 |      0.0 | 0.289731 |   0.2115 |     0.0 | -15.710776 |      0.0 | -24.418058 |      0.0 |  -25.278896 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.905562 | 20.905562 |   -7.258078 | 0.898208 | -459.226687 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 19:00:00 |  0.2 |  1.4 |             98.3 |       3.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.188 |  1.4 |    3.046047 |           3.046047 |                 6.760362 |                         0.0 |                     5.553833 |             6.645436 |  -5.51072 | -25.286578 |                      0.0 |                       0.0 |                     0.0 |                15.472735 |             0.0 |       -15.472735 |    0.188 |  0.0282 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.017823 | 0.283107 |    -1.296291 |      0.75064 |      0.0 |      0.0 | 0.172401 |   0.1598 |     0.0 | -13.205853 |      0.0 | -18.533424 |      0.0 |  -16.266543 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 21.111385 | 21.111385 |   -1.169117 | 0.901712 | -425.356775 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 20:00:00 |  0.4 |  1.6 |             98.7 |       5.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.376 |  1.6 |     4.56907 |            4.56907 |                 6.858436 |                         0.0 |                     5.830906 |             6.769276 | -5.104281 |  -26.81969 |                      0.0 |                       0.0 |                     0.0 |                16.893295 |             0.0 |       -16.893295 |    0.376 |  0.0376 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.020752 | 0.664621 |    -0.634333 |     0.750309 |      0.0 |      0.0 | 0.086638 | 0.086638 |     0.0 | -15.376607 |      0.0 | -20.802784 |      0.0 |  -19.286096 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 21.256132 | 21.508137 |         0.0 | 0.905525 | -389.953752 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 21:00:00 |  0.7 |  2.0 |             99.6 |       5.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.658 |  2.0 |    4.488911 |           4.488911 |                 7.058357 |                         0.0 |                     6.000247 |             7.030123 | -4.679445 | -26.616782 |                      0.0 |                       0.0 |                     0.0 |                15.416111 |             0.0 |       -15.416111 |    0.658 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.022462 | 1.530508 |    -0.243315 |     0.749675 |      0.0 |      0.0 | 0.035139 | 0.035139 |     0.0 |  -16.64359 |      0.0 | -20.598693 |      0.0 |  -21.826173 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 21.312847 | 22.188599 |         0.0 |  0.91286 | -354.753637 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 22:00:00 |  0.5 |  2.4 |             99.4 |       5.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.47 |  2.4 |    4.649229 |           4.649229 |                 7.263399 |                         0.0 |                       6.1078 |             7.219819 | -4.257044 | -25.542262 |                      0.0 |                       0.0 |                     0.0 |                 13.71239 |             0.0 |        -13.71239 |     0.47 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.024931 | 1.311864 |          0.0 |     0.749046 | 0.034312 | 0.034312 |      0.0 |      0.0 |     0.0 | -18.473108 |      0.0 |  -22.65304 |      0.0 |  -27.413757 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 21.301948 | 22.683531 |         0.0 | 0.920191 | -320.628042 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 23:00:00 |  1.0 |  2.5 |             99.2 |       6.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.94 |  2.5 |    4.969866 |           4.969866 |                 7.315473 |                         0.0 |                       6.1078 |             7.256949 | -3.847536 | -23.085219 |                      0.0 |                       0.0 |                     0.0 |                13.067893 |             0.0 |       -13.067893 |     0.94 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.027164 |  2.73305 |          0.0 |      0.74842 | 0.124889 | 0.124889 |      0.0 |      0.0 |     0.0 |  -20.12751 |      0.0 | -24.879463 |      0.0 |   -31.93908 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan |  21.20141 | 23.650695 |         0.0 | 0.927543 | -288.959489 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-11 00:00:00 |  2.0 |  2.7 |             99.2 |       5.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  1.88 |  2.7 |    4.488911 |           4.488911 |                 7.420609 |                         0.0 |                       6.1078 |             7.361244 | -3.467514 | -20.805083 |                      0.0 |                       0.0 |                     0.0 |                11.601106 |             0.0 |       -11.601106 |     1.88 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.027338 | 5.903388 |          0.0 |     0.747796 | 0.199896 | 0.199896 |      0.0 |      0.0 |     0.0 |  -20.25663 |      0.0 | -24.792095 |      0.0 |  -33.447619 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 21.024002 | 25.558033 |         0.0 | 0.934934 | -259.571073 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-11 01:00:00 |  2.1 |  3.2 |             99.6 |       5.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 1.974 |  3.2 |    4.168274 |           4.168274 |                 7.689299 |                         0.0 |                       6.1078 |             7.658542 | -3.114853 | -18.689117 |                      0.0 |                       0.0 |                     0.0 |                 7.733351 |             0.0 |        -7.733351 |    1.974 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.031933 | 7.346438 |          0.0 |     0.747176 |  0.34843 |  0.34843 |      0.0 |      0.0 |     0.0 | -23.661007 |      0.0 | -27.741565 |      0.0 |  -43.669221 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.699553 | 27.563966 |         0.0 | 0.942333 | -232.298622 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-11 02:00:00 |  1.8 |  3.9 |             99.8 |       5.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 1.692 |  3.9 |     4.56907 |            4.56907 |                  8.07986 |                         0.0 |                       6.1078 |               8.0637 | -2.787583 | -16.725501 |                      0.0 |                       0.0 |                     0.0 |                 2.415944 |             0.0 |        -2.415944 |    1.692 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.043255 | 7.674404 |          0.0 |     0.746561 | 0.613232 | 0.613232 |      0.0 |      0.0 |     0.0 | -32.050365 |      0.0 | -36.310997 |      0.0 |  -65.945417 |      1.0 |      0.3 | 0.533315 |  0.18206 |  0.0 |  0.0 | 0.0 |  0.0 |  0.18206 |  0.18206 |      0.0 |   0.0 |   0.0 |  0.0 | 0.002362 |      0.0 |   0.0 |   0.0 |  0.0 | 0.002362 | 0.000656 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.116019 | 28.765907 |         0.0 | 0.949726 | -206.989788 | 71.797515 | 0.179698 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000656 |
    | 2010-12-11 03:00:00 |  1.3 |  4.4 |             99.8 |       6.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 1.222 |  4.4 |    5.290502 |           5.290502 |                  8.36943 |                         0.0 |                       6.1078 |             8.352691 | -2.483877 | -14.903265 |                      0.0 |                       0.0 |                     0.0 |                -1.360594 |             0.0 |         1.360594 |    1.222 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.055801 |      0.0 |          0.0 |     0.745958 | 0.795977 | 0.795977 |      0.0 |      0.0 |     0.0 | -41.346526 |      0.0 | -46.045135 |      0.0 |  -88.752255 |      1.0 |      0.3 | 2.362455 | 0.808317 |  0.0 |  0.0 | 0.0 |  0.0 | 0.808317 |      0.5 | 0.308317 |   0.0 |   0.0 |  0.0 | 0.011128 | 0.015589 |   0.0 |   0.0 |  0.0 | 0.026716 | 0.007421 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 19.357519 | 27.681253 |         0.0 | 0.957034 |  -183.50319 | 73.351654 |  0.66857 | 0.292728 |  0.0 |  0.0 | 0.0 |   0.0 | 0.007421 |
    | 2010-12-11 04:00:00 |  0.5 |  4.8 |             99.0 |       7.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.47 |  4.8 |    5.611139 |           5.611139 |                 8.607627 |                         0.0 |                       6.1078 |             8.521551 | -2.202038 |  -13.21223 |                      0.0 |                       0.0 |                     0.0 |                -4.039084 |             0.0 |         4.039084 |     0.47 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 |  -0.06294 |      0.0 |          0.0 |     0.745359 | 0.971746 | 0.971746 |      0.0 |      0.0 |     0.0 | -46.635995 |      0.0 | -52.693545 |      0.0 | -103.368624 |      1.0 |      0.3 | 1.862756 | 0.638926 |  0.0 |  0.0 | 0.0 |  0.0 | 0.638926 |      0.5 | 0.138926 |   0.0 |   0.0 |  0.0 | 0.023756 | 0.036118 |   0.0 |   0.0 |  0.0 | 0.059874 | 0.016632 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 18.427578 | 26.351437 |         0.0 | 0.964342 | -161.707627 | 74.575484 | 1.144815 | 0.395536 |  0.0 |  0.0 | 0.0 |   0.0 | 0.016632 |
    | 2010-12-11 05:00:00 |  0.2 |  4.8 |             97.6 |       7.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.188 |  4.8 |    5.851616 |           5.851616 |                 8.607627 |                         0.0 |                       6.1078 |             8.401044 | -1.940492 | -11.642949 |                      0.0 |                       0.0 |                     0.0 |                -3.389325 |             0.0 |         3.389325 |    0.188 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.061894 |      0.0 |          0.0 |     0.744765 | 0.993206 | 0.993206 |      0.0 |      0.0 |     0.0 | -45.860636 |      0.0 | -54.540411 |      0.0 | -103.790373 |      1.0 |      0.3 |  1.61162 | 0.553938 |  0.0 |  0.0 | 0.0 |  0.0 | 0.553938 |      0.5 | 0.053938 |   0.0 |   0.0 |  0.0 | 0.036057 | 0.042039 |   0.0 |   0.0 |  0.0 | 0.078096 | 0.021693 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 17.475322 | 24.989711 |         0.0 |  0.97165 | -141.481344 | 75.633165 | 1.608757 | 0.407436 |  0.0 |  0.0 | 0.0 |   0.0 | 0.021693 |
    | 2010-12-11 06:00:00 |  0.5 |  5.1 |             97.4 |       7.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.47 |  5.1 |    5.771457 |           5.771457 |                 8.790172 |                         0.0 |                       6.1078 |             8.561627 | -1.697776 | -10.186657 |                      0.0 |                       0.0 |                     0.0 |                -5.587229 |             0.0 |         5.587229 |     0.47 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 |  -0.06548 |      0.0 |          0.0 |     0.744175 | 1.090927 | 1.090927 |      0.0 |      0.0 |     0.0 |  -48.51811 |      0.0 | -57.295089 |      0.0 | -111.400428 |      1.0 |      0.3 | 2.034113 | 0.700814 |  0.0 |  0.0 | 0.0 |  0.0 | 0.700814 |      0.5 | 0.200814 |   0.0 |   0.0 |  0.0 | 0.048041 | 0.050648 |   0.0 |   0.0 |  0.0 | 0.098689 | 0.027414 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 16.427327 | 23.491078 |         0.0 | 0.978958 | -122.711354 | 76.966465 | 2.060716 | 0.557601 |  0.0 |  0.0 | 0.0 |   0.0 | 0.027414 |
    | 2010-12-11 07:00:00 |  0.2 |  5.1 |             96.3 |       7.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.188 |  5.1 |    6.092093 |           6.092093 |                 8.790172 |                         0.0 |                       6.1078 |             8.464935 | -1.472536 |  -8.835218 |                      0.0 |                       0.0 |                     0.0 |                -5.066448 |             0.0 |         5.066448 |    0.188 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.065772 |      0.0 |          0.0 |     0.743589 | 1.130414 | 1.130414 |      0.0 |      0.0 |     0.0 | -48.734564 |      0.0 | -59.911482 |      0.0 | -113.712494 |      1.0 |      0.3 | 1.809018 | 0.624709 |  0.0 |  0.0 | 0.0 |  0.0 | 0.624709 |      0.5 | 0.124709 |   0.0 |   0.0 |  0.0 | 0.059716 | 0.061725 |   0.0 |   0.0 |  0.0 | 0.121441 | 0.033733 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 15.339743 | 21.935832 |         0.0 | 0.986266 | -105.292803 | 78.150774 |    2.501 | 0.620585 |  0.0 |  0.0 | 0.0 |   0.0 | 0.033733 |
    | 2010-12-11 08:00:00 |  0.1 |  5.4 |             95.2 |       7.3 |                 0.347041 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.40894 |        0.900734 | 0.094 |  5.4 |    5.851616 |           5.851616 |                 8.976116 |                         0.0 |                       6.1078 |             8.545263 | -1.263514 |  -7.581082 |                      0.0 |                  0.231482 |                     0.0 |                -6.835277 |             0.0 |         7.066758 |    0.094 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.065786 |      0.0 |          0.0 |     0.743008 | 1.181192 | 1.181192 |      0.0 |      0.0 |     0.0 | -48.744743 |      0.0 | -61.357963 |      0.0 | -117.169464 |      1.0 |      0.3 |  1.78817 | 0.618883 |  0.0 |  0.0 | 0.0 |  0.0 | 0.618883 |      0.5 | 0.118883 |   0.0 |   0.0 |  0.0 | 0.071089 |  0.06769 |   0.0 |   0.0 |  0.0 | 0.138779 |  0.03855 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 14.201013 | 20.307448 |         0.0 | 0.993574 |  -89.128388 | 79.320061 | 2.929911 | 0.671779 |  0.0 |  0.0 | 0.0 |   0.0 |  0.03855 |
    | 2010-12-11 09:00:00 |  0.3 |  5.3 |             96.3 |       7.1 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.40894 |        17.26751 | 0.282 |  5.3 |    5.691298 |           5.691298 |                 8.913754 |                         0.0 |                       6.1078 |             8.583945 | -1.069541 |  -6.417244 |                      0.0 |                  4.447583 |                     0.0 |                -6.596815 |             0.0 |        11.044398 |    0.282 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.065321 |      0.0 |          0.0 |     0.742431 | 1.205998 | 1.205998 |      0.0 |      0.0 |     0.0 | -48.400444 |      0.0 | -58.862205 |      0.0 | -118.307048 |      1.0 |      0.3 | 2.012124 | 0.698077 |  0.0 |  0.0 | 0.0 |  0.0 | 0.698077 |      0.5 | 0.198077 |   0.0 |   0.0 |  0.0 | 0.082168 | 0.076782 |   0.0 |   0.0 |  0.0 |  0.15895 | 0.044153 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 13.036815 | 18.642645 |         0.0 | 1.000882 |  -74.127811 | 80.634108 | 3.347743 | 0.793074 |  0.0 |  0.0 | 0.0 |   0.0 | 0.044153 |
    | 2010-12-11 10:00:00 |  0.1 |  5.5 |             95.5 |       7.1 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.40894 |        52.61761 | 0.094 |  5.5 |    5.691298 |           5.691298 |                 9.038862 |                         0.0 |                       6.1078 |             8.632113 | -0.889534 |  -5.337202 |                      0.0 |                 13.582832 |                     0.0 |                -7.748872 |             0.0 |        21.331705 |    0.094 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.066592 |      0.0 |          0.0 |     0.741858 |  1.36261 |  1.36261 |      0.0 |      0.0 |     0.0 | -49.341985 |      0.0 | -61.083421 |      0.0 |  -131.75711 |      1.0 |      0.3 | 2.049492 | 0.712842 |  0.0 |  0.0 | 0.0 |  0.0 | 0.712842 |      0.5 | 0.212842 |   0.0 |   0.0 |  0.0 | 0.092961 | 0.089584 |   0.0 |   0.0 |  0.0 | 0.182545 | 0.050707 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 11.715906 | 16.753745 |         0.0 |  1.00819 |  -60.207275 | 81.970758 | 3.754782 | 0.916332 |  0.0 |  0.0 | 0.0 |   0.0 | 0.050707 |
    | 2010-12-11 11:00:00 |  0.1 |  6.1 |             91.6 |       8.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       80.135633 | 0.094 |  6.1 |    7.134162 |           7.134162 |                 9.423511 |                         0.0 |                       6.1078 |             8.631936 | -0.722487 |  -4.334924 |                      0.0 |                 20.731976 |                     0.0 |               -12.026271 |             0.0 |        32.758247 |    0.094 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.080429 |      0.0 |          0.0 |     0.741289 | 1.830687 | 1.830687 |      0.0 |      0.0 |     0.0 |  -59.59435 |      0.0 | -81.829421 |      0.0 | -174.182018 |      1.0 |      0.3 |  2.72445 | 0.950584 |  0.0 |  0.0 | 0.0 |  0.0 | 0.950584 |      0.5 | 0.450584 |   0.0 |   0.0 |  0.0 | 0.103475 | 0.113855 |   0.0 |   0.0 |  0.0 |  0.21733 | 0.060369 |     0.06 |   0.0 |  0.0 |      0.0 |   nan |  9.932674 | 14.203724 |         0.0 | 1.015498 |  -47.289018 | 83.744624 | 4.151307 | 1.253062 |  0.0 |  0.0 | 0.0 |   0.0 | 0.060369 |
    | 2010-12-11 12:00:00 |  0.0 |  6.4 |             90.0 |       9.5 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       80.276118 |   0.0 |  6.4 |    7.615117 |           7.615117 |                 9.621181 |                         0.0 |                       6.1078 |             8.659063 | -0.567468 |  -3.404809 |                      0.0 |                 20.813648 |                     0.0 |               -13.533625 |             0.0 |        34.347273 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.085956 |      0.0 |          0.0 |     0.740724 | 1.998446 | 1.998446 |      0.0 |      0.0 |     0.0 | -63.690154 |      0.0 | -90.778796 |      0.0 | -188.816222 |      1.0 |      0.3 | 2.875073 | 1.006673 |  0.0 |  0.0 | 0.0 |  0.0 | 1.006673 |      0.5 | 0.506673 |   0.0 |   0.0 |  0.0 | 0.113718 | 0.150158 |   0.0 |   0.0 |  0.0 | 0.263876 | 0.073299 |     0.06 |   0.0 |  0.0 |      0.0 |   nan |  7.982243 | 11.414607 |         0.0 | 1.022806 |  -35.300875 | 85.613024 | 4.537589 | 1.609577 |  0.0 |  0.0 | 0.0 |   0.0 | 0.073299 |
    | 2010-12-11 13:00:00 |  0.1 |  6.4 |             85.7 |       7.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       75.835403 | 0.094 |  6.4 |    6.332571 |           6.332571 |                 9.621181 |                         0.0 |                       6.1078 |             8.245352 | -0.423611 |  -2.541663 |                      0.0 |                  19.70479 |                     0.0 |               -11.239491 |             0.0 |        30.944281 |    0.094 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.061599 |      0.0 |          0.0 |     0.740164 | 1.634985 | 1.634985 |      0.0 |      0.0 |     0.0 | -45.642119 |      0.0 | -77.645525 |      0.0 | -154.231926 |      1.0 |      0.3 | 2.444645 |   0.8587 |  0.0 |  0.0 | 0.0 |  0.0 |   0.8587 |      0.5 |   0.3587 |   0.0 |   0.0 |  0.0 | 0.123696 |  0.17811 |   0.0 |   0.0 |  0.0 | 0.301805 | 0.083835 |     0.06 |   0.0 |  0.0 |      0.0 |   nan |  6.381511 |  9.125561 |         0.0 | 1.030114 |  -24.175879 | 87.198969 | 4.913893 | 1.790167 |  0.0 |  0.0 | 0.0 |   0.0 | 0.083835 |
    | 2010-12-11 14:00:00 |  0.0 |  6.5 |             82.3 |       9.6 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       42.227911 |   0.0 |  6.5 |    7.695276 |           7.695276 |                 9.687876 |                         0.0 |                       6.1078 |             7.973122 | -0.290111 |  -1.740663 |                      0.0 |                 10.995846 |                     0.0 |               -10.124562 |             0.0 |        21.120409 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.063414 |      0.0 |          0.0 |     0.739607 | 1.718061 | 1.718061 |      0.0 |      0.0 |     0.0 | -46.987277 |      0.0 | -93.030869 |      0.0 | -161.138555 |      1.0 |      0.3 |   2.4739 | 0.871645 |  0.0 |  0.0 | 0.0 |  0.0 | 0.871645 |      0.5 | 0.371645 |   0.0 |   0.0 |  0.0 | 0.133416 | 0.196713 |   0.0 |   0.0 |  0.0 | 0.330129 | 0.091702 |     0.06 |   0.0 |  0.0 |      0.0 |   nan |  4.695856 |  6.715075 |         0.0 | 1.037422 |  -13.851882 | 88.801224 | 5.280478 | 1.965099 |  0.0 |  0.0 | 0.0 |   0.0 | 0.091702 |
    | 2010-12-11 15:00:00 |  0.1 |  6.3 |             82.5 |       9.1 |                      1.0 |              0.1 | 0.0 | 0.0 |                   0.1 |                       7.40894 |       12.613921 | 0.094 |  6.3 |     7.29448 |            7.29448 |                 9.554889 |                         0.0 |                       6.1078 |             7.882784 | -0.166223 |  -0.997336 |                      0.0 |                  3.291545 |                     0.0 |                -7.130286 |             0.0 |        10.421831 |    0.094 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.057639 |      0.0 |          0.0 |     0.739055 |  1.49024 |  1.49024 |      0.0 |      0.0 |     0.0 |  -42.70834 |      0.0 | -86.128361 |      0.0 | -139.258532 |      1.0 |      0.3 | 2.243336 | 0.792685 |  0.0 |  0.0 | 0.0 |  0.0 | 0.792685 |      0.5 | 0.292685 |   0.0 |   0.0 |  0.0 | 0.142885 | 0.210107 |   0.0 |   0.0 |  0.0 | 0.352992 | 0.098053 |     0.06 |   0.0 |  0.0 |      0.0 |   nan |  3.233132 |  4.623378 |         0.0 |  1.04473 |   -4.271214 | 90.251875 | 5.637592 | 2.047677 |  0.0 |  0.0 | 0.0 |   0.0 | 0.098053 |
    | 2010-12-11 16:00:00 |  0.0 |  5.9 |             82.2 |       8.7 |                 0.053708 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |        0.021912 |   0.0 |  5.9 |    6.973844 |           6.973844 |                 9.293725 |                         0.0 |                       6.1078 |             7.639442 | -0.051255 |  -0.307527 |                      0.0 |                   0.00573 |                     0.0 |                -3.915548 |             0.0 |         3.921278 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.047871 |      0.0 |          0.0 |     0.738506 | 1.258029 | 1.258029 |      0.0 |      0.0 |     0.0 | -35.470286 |      0.0 | -77.633085 |      0.0 | -117.024649 | 0.999378 | 0.299627 | 1.817608 | 0.643181 |  0.0 |  0.0 | 0.0 |  0.0 | 0.643181 |      0.5 | 0.143181 |   0.0 |   0.0 |  0.0 |  0.15211 | 0.210755 |   0.0 |   0.0 |  0.0 | 0.362865 | 0.100796 |     0.06 |   0.0 |  0.0 |      0.0 |   nan |  1.995553 |  2.853641 |         0.0 | 1.052038 |    4.619647 | 91.426302 | 5.985483 | 1.980103 |  0.0 |  0.0 | 0.0 |   0.0 | 0.100796 |
    | 2010-12-11 17:00:00 |  0.0 |  5.5 |             76.5 |       8.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  5.5 |    7.134162 |           7.134162 |                 9.038862 |                         0.0 |                       6.1078 |              6.91473 |       0.0 |        0.0 |                      0.0 |                       0.0 |                     0.0 |                 2.340465 |             0.0 |        -2.340465 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.025712 |      0.0 |          0.0 |     0.737961 | 0.975359 | 0.975359 |      0.0 |      0.0 |     0.0 | -19.051448 |      0.0 | -73.780626 |      0.0 |  -90.491608 | 0.998221 | 0.298934 |  1.40733 |  0.49805 |  0.0 |  0.0 | 0.0 |  0.0 |  0.49805 |  0.49805 |      0.0 |   0.0 |   0.0 |  0.0 | 0.161071 |   0.1968 |   0.0 |   0.0 |  0.0 |  0.35787 | 0.099408 |     0.06 |   0.0 |  0.0 |      0.0 |   nan |  1.029387 |  1.472023 |         0.0 | 1.059346 |    13.20298 | 92.335582 | 6.322462 | 1.783304 |  0.0 |  0.0 | 0.0 |   0.0 | 0.099408 |
    | 2010-12-11 18:00:00 |  0.0 |  5.1 |             81.7 |       7.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  5.1 |    6.332571 |           6.332571 |                 8.790172 |                         0.0 |                       6.1078 |              7.18157 |       0.0 |        0.0 |                      0.0 |                       0.0 |                     0.0 |                 2.381726 |             0.0 |        -2.381726 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.030943 |      0.0 |          0.0 |      0.73742 | 0.888357 | 0.888357 |      0.0 |      0.0 |     0.0 | -22.927701 |      0.0 | -61.873778 |      0.0 |  -82.419753 | 0.997065 | 0.298241 | 1.297054 | 0.458963 |  0.0 |  0.0 | 0.0 |  0.0 | 0.458963 | 0.458963 |      0.0 |   0.0 |   0.0 |  0.0 | 0.169268 |  0.17724 |   0.0 |   0.0 |  0.0 | 0.346508 | 0.096252 |     0.06 |   0.0 |  0.0 |      0.0 |   nan |  0.143995 |  0.205912 |         0.0 | 1.066654 |   21.786314 | 93.173673 | 6.612157 | 1.606064 |  0.0 |  0.0 | 0.0 |   0.0 | 0.096252 |
    | 2010-12-11 19:00:00 |  0.0 |  4.8 |             83.4 |       7.1 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  4.8 |    5.691298 |           5.691298 |                 8.607627 |                         0.0 |                       6.1078 |             7.178761 |       0.0 |        0.0 |                      0.0 |                       0.0 |                     0.0 |                 3.699875 |             0.0 |        -3.699875 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |     0.06 |  0.00372 | -0.028252 |      0.0 |          0.0 |     0.736883 | 0.760344 | 0.143995 |      0.0 |      0.0 |     0.0 | -20.933744 |      0.0 | -53.309167 |      0.0 |  -70.543036 | 0.995908 |  0.29755 | 0.234165 | 0.082767 |  0.0 |  0.0 | 0.0 |  0.0 | 0.082767 | 0.082767 |      0.0 |   0.0 |   0.0 |  0.0 | 0.171871 | 0.159624 |   0.0 |   0.0 |  0.0 | 0.331496 | 0.092082 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 | 1.073962 |   30.369647 | 93.321351 | 6.523052 | 1.446439 |  0.0 |  0.0 | 0.0 |   0.0 | 0.092082 |
    | 2010-12-11 20:00:00 |  0.0 |  4.9 |             82.8 |       6.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  4.9 |    5.130184 |           5.130184 |                 8.668101 |                         0.0 |                          0.0 |             7.177187 |       0.0 |      -29.4 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.025199 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.990791 |   0.2945 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.168496 | 0.143759 |   0.0 |   0.0 |  0.0 | 0.312256 | 0.086738 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |    68.35298 | 93.296151 | 6.354556 |  1.30268 |  0.0 |  0.0 | 0.0 |   0.0 | 0.086738 |
    | 2010-12-11 21:00:00 |  0.0 |  4.9 |             85.8 |       7.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  4.9 |    5.771457 |           5.771457 |                 8.668101 |                         0.0 |                          0.0 |              7.43723 |       0.0 |      -29.4 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.021594 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.985673 | 0.291466 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.164144 | 0.129471 |   0.0 |   0.0 |  0.0 | 0.293615 |  0.08156 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  106.336314 | 93.274558 | 6.190412 | 1.173208 |  0.0 |  0.0 | 0.0 |   0.0 |  0.08156 |
    | 2010-12-11 22:00:00 |  0.1 |  4.7 |             87.6 |       7.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 | 0.094 |  4.7 |    5.611139 |           5.611139 |                 8.547525 |                         0.0 |                          0.0 |             7.487632 |       0.0 |      -28.2 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.034 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.050931 |      0.0 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.980717 | 0.288542 |    0.034 | 0.011736 |  0.0 |  0.0 | 0.0 |  0.0 | 0.011736 | 0.011736 |      0.0 |   0.0 |   0.0 |  0.0 | 0.160056 | 0.116603 |   0.0 |   0.0 |  0.0 | 0.276659 |  0.07685 | 0.009069 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  143.119647 | 93.296822 | 6.042092 | 1.056605 |  0.0 |  0.0 | 0.0 |   0.0 |  0.07685 |
    | 2010-12-11 23:00:00 |  0.1 |  3.8 |             85.6 |       6.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 | 0.094 |  3.8 |     5.45082 |            5.45082 |                 8.023017 |                         0.0 |                          0.0 |             6.867703 |       0.0 |      -22.8 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 | 0.043069 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.055782 |      0.0 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.976489 | 0.286059 | 0.043069 | 0.014768 |  0.0 |  0.0 | 0.0 |  0.0 | 0.014768 | 0.014768 |      0.0 |   0.0 |   0.0 |  0.0 | 0.156264 | 0.105014 |   0.0 |   0.0 |  0.0 | 0.261278 | 0.072577 | 0.004218 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |   174.50298 | 93.325122 | 5.900597 |  0.95159 |  0.0 |  0.0 | 0.0 |   0.0 | 0.072577 |
    | 2010-12-12 00:00:00 |  0.0 |  3.9 |             85.3 |       6.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.9 |    5.130184 |           5.130184 |                  8.07986 |                         0.0 |                          0.0 |             6.892121 |       0.0 |      -23.4 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.004218 | 0.018582 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 |  0.97218 |  0.28354 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.152417 | 0.094577 |   0.0 |   0.0 |  0.0 | 0.246995 |  0.06861 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  206.486314 | 93.306541 | 5.748179 | 0.857013 |  0.0 |  0.0 | 0.0 |   0.0 |  0.06861 |
    | 2010-12-12 01:00:00 |  0.0 |  3.7 |             86.2 |       6.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.7 |    5.050025 |           5.050025 |                 7.966527 |                         0.0 |                          0.0 |             6.867146 |       0.0 |      -22.2 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.018518 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.968033 | 0.281126 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |  0.14848 | 0.085177 |   0.0 |   0.0 |  0.0 | 0.233658 | 0.064905 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  237.269647 | 93.288022 | 5.599699 | 0.771836 |  0.0 |  0.0 | 0.0 |   0.0 | 0.064905 |
    | 2010-12-12 02:00:00 |  0.1 |  3.5 |             89.3 |       5.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 | 0.094 |  3.5 |     4.56907 |            4.56907 |                 7.854597 |                         0.0 |                          0.0 |             7.014155 |       0.0 |      -21.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |    0.034 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.033549 |      0.0 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.964047 | 0.278816 |    0.034 | 0.011432 |  0.0 |  0.0 | 0.0 |  0.0 | 0.011432 | 0.011432 |      0.0 |   0.0 |   0.0 |  0.0 | 0.144793 | 0.076712 |   0.0 |   0.0 |  0.0 | 0.221505 | 0.061529 | 0.026451 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |   266.85298 |  93.31059 | 5.466337 | 0.695124 |  0.0 |  0.0 | 0.0 |   0.0 | 0.061529 |
    | 2010-12-12 03:00:00 |  0.0 |  3.3 |             91.4 |       5.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.3 |    4.168274 |           4.168274 |                 7.744054 |                         0.0 |                          0.0 |             7.078066 |       0.0 |      -19.8 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.023802 |      0.0 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.960223 | 0.276608 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |   0.1412 | 0.069087 |   0.0 |   0.0 |  0.0 | 0.210288 | 0.058413 | 0.002648 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  295.236314 |  93.31059 | 5.325137 | 0.626037 |  0.0 |  0.0 | 0.0 |   0.0 | 0.058413 |
    | 2010-12-12 04:00:00 |  0.0 |  3.3 |             90.9 |       4.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.3 |     3.68732 |            3.68732 |                 7.744054 |                         0.0 |                          0.0 |             7.039346 |       0.0 |      -19.8 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 | 0.002648 | 0.008766 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.956399 | 0.274409 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.137553 | 0.062221 |   0.0 |   0.0 |  0.0 | 0.199774 | 0.055493 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  323.619647 | 93.301824 | 5.187584 | 0.563816 |  0.0 |  0.0 | 0.0 |   0.0 | 0.055493 |
    | 2010-12-12 05:00:00 |  0.0 |  3.0 |             89.2 |       4.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.0 |    3.927797 |           3.927797 |                 7.580812 |                         0.0 |                          0.0 |             6.762084 |       0.0 |      -18.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |  0.01221 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.952817 | 0.272358 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |    0.134 | 0.056037 |   0.0 |   0.0 |  0.0 | 0.190037 | 0.052788 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |   350.20298 | 93.289614 | 5.053584 | 0.507779 |  0.0 |  0.0 | 0.0 |   0.0 | 0.052788 |
    | 2010-12-12 06:00:00 |  0.0 |  1.9 |             85.6 |       6.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  1.9 |    5.050025 |           5.050025 |                 7.007901 |                         0.0 |                          0.0 |             5.998763 |       0.0 |      -11.4 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.017452 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.950125 | 0.270821 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.130538 | 0.050467 |   0.0 |   0.0 |  0.0 | 0.181006 | 0.050279 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  370.186314 | 93.272161 | 4.923046 | 0.457312 |  0.0 |  0.0 | 0.0 |   0.0 | 0.050279 |
    | 2010-12-12 07:00:00 |  0.0 |  1.2 |             86.3 |       5.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  1.2 |    4.649229 |           4.649229 |                 6.663527 |                         0.0 |                          0.0 |             5.750624 |       0.0 |       -7.2 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.015359 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.947998 |  0.26961 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.127167 | 0.045452 |   0.0 |   0.0 |  0.0 | 0.172618 | 0.047949 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  385.969647 | 93.256802 | 4.795879 |  0.41186 |  0.0 |  0.0 | 0.0 |   0.0 | 0.047949 |
    | 2010-12-12 08:00:00 |  0.0 |  0.8 |             91.2 |       6.1 |                 0.339738 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |        0.894145 |   0.0 |  0.8 |    4.889707 |           4.889707 |                 6.473522 |                         0.0 |                          0.0 |             5.903852 |       0.0 |       -4.8 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.009663 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.946195 | 0.268585 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.123882 | 0.040934 |   0.0 |   0.0 |  0.0 | 0.164816 | 0.045782 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |   399.35298 | 93.247139 | 4.671998 | 0.370926 |  0.0 |  0.0 | 0.0 |   0.0 | 0.045782 |
    | 2010-12-12 09:00:00 |  0.0 |  0.8 |             87.5 |       5.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |        17.71254 |   0.0 |  0.8 |    4.729388 |           4.729388 |                 6.473522 |                         0.0 |                          0.0 |             5.664332 |       0.0 |       -4.8 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.017181 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.944392 | 0.267563 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.120682 | 0.036866 |   0.0 |   0.0 |  0.0 | 0.157548 | 0.043763 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  412.736314 | 93.229958 | 4.551316 |  0.33406 |  0.0 |  0.0 | 0.0 |   0.0 | 0.043763 |
    | 2010-12-12 10:00:00 |  0.1 |  0.5 |             84.5 |       6.4 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       54.302585 | 0.094 |  0.5 |    5.130184 |           5.130184 |                 6.334173 |                         0.0 |                     5.554612 |             5.352376 |       0.0 |   7.766328 |                      0.0 |                  5.556421 |                     0.0 |                28.031613 |             0.0 |       -22.475192 |    0.034 | 0.01275 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 |  0.004904 | 0.006889 |    -1.294388 |     0.897677 |      0.0 |      0.0 | 0.000192 | 0.000192 |     0.0 |   3.633499 |      0.0 |  -18.31764 |      0.0 |    7.791051 | 0.944282 |   0.2675 | 0.015768 | 0.005136 |  0.0 |  0.0 | 0.0 |  0.0 | 0.005136 | 0.005136 |      0.0 |   0.0 |   0.0 |  0.0 | 0.117631 | 0.033202 |   0.0 |   0.0 |  0.0 | 0.150833 | 0.041898 |     0.06 |   0.0 |  0.0 |      0.0 |   nan |  0.009461 |  0.013329 |         0.0 |  0.00743 |  413.553319 | 93.240589 | 4.438821 | 0.300858 |  0.0 |  0.0 | 0.0 |   0.0 | 0.041898 |
    | 2010-12-12 11:00:00 |  0.0 |  0.3 |             76.7 |       6.8 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       81.504146 |   0.0 |  0.3 |     5.45082 |            5.45082 |                  6.24275 |                         0.0 |                     5.354685 |             4.788189 |       0.0 |  10.742078 |                      0.0 |                  8.523311 |                     0.0 |                31.005039 |             0.0 |       -22.481729 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |     0.06 | 0.008323 |  0.013329 |      0.0 |    -1.790346 |     0.895425 |      0.0 |      0.0 | 0.000193 | 0.000193 |     0.0 |  10.689492 |      0.0 | -22.411256 |      0.0 |   10.759964 | 0.944573 | 0.267665 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.114659 | 0.029902 |   0.0 |   0.0 |  0.0 |  0.14456 | 0.040156 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 | 0.014738 |  411.394575 | 93.232266 | 4.324163 | 0.270957 |  0.0 |  0.0 | 0.0 |   0.0 | 0.040156 |
    | 2010-12-12 12:00:00 |  0.0 | -0.1 |             74.8 |       7.0 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       81.665294 |   0.0 | -0.1 |    5.611139 |           5.611139 |                 6.063393 |                         0.0 |                          0.0 |             4.535418 |       0.0 |        0.6 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.042746 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.943497 | 0.267056 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.111697 |  0.02693 |   0.0 |   0.0 |  0.0 | 0.138627 | 0.038507 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  419.377908 | 93.189519 | 4.212466 | 0.244027 |  0.0 |  0.0 | 0.0 |   0.0 | 0.038507 |
    | 2010-12-12 13:00:00 |  0.0 | -0.4 |             69.7 |       6.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       77.078959 |   0.0 | -0.4 |    5.050025 |           5.050025 |                 5.931876 |                         0.0 |                          0.0 |             4.134518 |       0.0 |        2.4 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.046731 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.942664 | 0.266585 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.108812 | 0.024253 |   0.0 |   0.0 |  0.0 | 0.133065 | 0.036963 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  425.561242 | 93.142789 | 4.103654 | 0.219773 |  0.0 |  0.0 | 0.0 |   0.0 | 0.036963 |
    | 2010-12-12 14:00:00 |  0.0 | -0.6 |             66.9 |       7.3 |                      1.0 |              0.2 | 0.0 | 0.0 |                   0.3 |                      7.393446 |       60.254042 |   0.0 | -0.6 |    5.851616 |           5.851616 |                 5.845604 |                         0.0 |                          0.0 |             3.910709 |       0.0 |        3.6 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.047228 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.941992 | 0.266205 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.106001 | 0.021843 |   0.0 |   0.0 |  0.0 | 0.127844 | 0.035512 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  430.544575 | 93.095561 | 3.997653 |  0.19793 |  0.0 |  0.0 | 0.0 |   0.0 | 0.035512 |
    | 2010-12-12 15:00:00 |  0.0 | -0.8 |             64.4 |       7.5 |                      1.0 |              0.2 | 0.0 | 0.0 |                   0.4 |                      7.393446 |       14.837301 |   0.0 | -0.8 |    6.011934 |           6.011934 |                 5.760442 |                         0.0 |                          0.0 |             3.709725 |       0.0 |        4.8 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.042618 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.941483 | 0.265917 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.103263 | 0.019672 |   0.0 |   0.0 |  0.0 | 0.122935 | 0.034149 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  434.327908 | 93.052943 |  3.89439 | 0.178258 |  0.0 |  0.0 | 0.0 |   0.0 | 0.034149 |
    | 2010-12-12 16:00:00 |  0.0 | -1.3 |             65.4 |       7.0 |                 0.046405 |              0.2 | 0.0 | 0.0 |                   0.6 |                      7.386143 |        0.168856 |   0.0 | -1.3 |    5.611139 |           5.611139 |                 5.552313 |                         0.0 |                          0.0 |             3.631213 |       0.0 |        7.8 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.036657 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.941377 | 0.265857 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.100595 | 0.017717 |   0.0 |   0.0 |  0.0 | 0.118312 | 0.032865 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  435.111242 | 93.016286 | 3.793795 | 0.160541 |  0.0 |  0.0 | 0.0 |   0.0 | 0.032865 |
    | 2010-12-12 17:00:00 |  0.0 | -1.6 |             67.6 |       6.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -1.6 |    4.809547 |           4.809547 |                 5.430651 |                         0.0 |                          0.0 |              3.67112 |       0.0 |        9.6 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |  0.03163 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.941514 | 0.265935 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.097997 | 0.015956 |   0.0 |   0.0 |  0.0 | 0.113953 | 0.031654 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  434.094575 | 92.984656 | 3.695798 | 0.144585 |  0.0 |  0.0 | 0.0 |   0.0 | 0.031654 |
    | 2010-12-12 18:00:00 |  0.0 | -2.0 |             70.2 |       4.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.0 |    3.847638 |           3.847638 |                 5.272101 |                         0.0 |                          0.0 |             3.701015 |       0.0 |       12.0 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.025658 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.941975 | 0.266195 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.095466 |  0.01437 |   0.0 |   0.0 |  0.0 | 0.109836 |  0.03051 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  430.677908 | 92.958998 | 3.600332 | 0.130215 |  0.0 |  0.0 | 0.0 |   0.0 |  0.03051 |
    | 2010-12-12 19:00:00 |  0.0 | -2.1 |             70.4 |       4.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.1 |    3.366683 |           3.366683 |                 5.233109 |                         0.0 |                          0.0 |             3.684109 |       0.0 |       12.6 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |  0.02375 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.942516 | 0.266501 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |    0.093 | 0.012942 |   0.0 |   0.0 |  0.0 | 0.105942 | 0.029428 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  426.661242 | 92.935248 | 3.507333 | 0.117273 |  0.0 |  0.0 | 0.0 |   0.0 | 0.029428 |
    | 2010-12-12 20:00:00 |  0.0 | -2.2 |             69.5 |       4.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.2 |    3.927797 |           3.927797 |                 5.194372 |                         0.0 |                          0.0 |             3.610088 |       0.0 |       13.2 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |  0.02619 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.943138 | 0.266853 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.090597 | 0.011656 |   0.0 |   0.0 |  0.0 | 0.102253 | 0.028404 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  422.044575 | 92.909058 | 3.416735 | 0.105618 |  0.0 |  0.0 | 0.0 |   0.0 | 0.028404 |
    | 2010-12-12 21:00:00 |  0.0 | -2.6 |             71.6 |       4.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.6 |    3.206365 |           3.206365 |                 5.041946 |                         0.0 |                          0.0 |             3.610033 |       0.0 |       15.6 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.021424 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.944083 | 0.267388 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.088257 | 0.010497 |   0.0 |   0.0 |  0.0 | 0.098754 | 0.027432 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  415.027908 | 92.887633 | 3.328478 | 0.095121 |  0.0 |  0.0 | 0.0 |   0.0 | 0.027432 |
    | 2010-12-12 22:00:00 |  0.0 | -3.1 |             74.5 |       3.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -3.1 |     2.72541 |            2.72541 |                 4.856984 |                         0.0 |                          0.0 |             3.618453 |       0.0 |       18.6 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 | 0.016822 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.945433 | 0.268153 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.085977 | 0.009454 |   0.0 |   0.0 |  0.0 | 0.095431 | 0.026509 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  405.011242 | 92.870811 | 3.242501 | 0.085667 |  0.0 |  0.0 | 0.0 |   0.0 | 0.026509 |
    | 2010-12-12 23:00:00 |  0.0 | -3.9 |             80.2 |       3.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -3.9 |     2.72541 |            2.72541 |                  4.57351 |                         0.0 |                          0.0 |             3.667955 |       0.0 |       23.4 |                      0.0 |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |  0.01189 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |     0.0 |        0.0 |      0.0 |        0.0 |      0.0 |         0.0 | 0.947429 | 0.269286 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.083757 | 0.008514 |   0.0 |   0.0 |  0.0 | 0.092271 | 0.025631 |      0.0 |   0.0 |  0.0 |      0.0 |   nan |       0.0 |       0.0 |         0.0 |      nan |  390.194575 | 92.858921 | 3.158744 | 0.077152 |  0.0 |  0.0 | 0.0 |   0.0 | 0.025631 |

>>> round_(model.check_waterbalance(conditions_acker_winter))
0.0

.. _lland_knauf_ic_nadelw_winter_hourly:

conifers (winter)
-----------------

.. integration-test::

    >>> lnk(NADELW)
    >>> test("lland_knauf_ic_nadelw_winter_hourly",
    ...      axis1=(inputs.nied, fluxes.wada), axis2=(states.waes, states.wats),
    ...      use_conditions=conditions_nadelw_winter)
    |                date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |  nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow |  nbes |   sbes | snowintmax | snowintrate |  nbesinz |  sbesinz | wniedinz | actualalbedoinz |  wadainz | schmpotinz |  schminz | gefrpotinz |  gefrinz |   evsinz |      evi |       evb |       evs |     wnied | tempssurface | actualalbedo |  schmpot |     schm |  gefrpot |     gefr |   wlatinz |   wlatsnow |   wsensinz |  wsenssnow |   wsurfinz |       wsurf |      sff |      fvg |     wada |      qdb | qib1 | qib2 | qbb | qkap |     qdgz |    qdgz1 |    qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 |    qdga2 | qiga1 | qiga2 | qbga |      qah |       qa |     inzp |    stinz |     sinz |   esnowinz |    asinz |     wats |      waes |      esnow |     taus |        ebdn |      bowa |     sdg1 |     sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-12-10 00:00:00 |  0.0 | -3.7 |             87.5 |       3.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -3.7 |     2.72541 |           1.206967 |                  4.64297 |                     4.64297 |                     4.770504 |             4.062599 | -1.772272 |   9.403848 |                      0.0 |                       0.0 |               48.299114 |                23.930251 |      -48.299114 |       -23.930251 |   0.0 |    0.0 |       24.5 |    0.439593 |      0.0 |      0.0 |      0.0 |        0.750856 |      0.0 |        0.0 |      0.0 |   0.355784 |      0.0 | 0.005419 |      0.0 |  0.008725 |   0.00661 |       0.0 |     -3.33958 |      0.75354 |      0.0 |      0.0 | 0.224635 | 0.224635 |  4.015484 |   4.897869 | -19.678059 |   1.416865 |   32.63654 |   30.244985 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.525649 | 6.525649 | -33.008866 | 0.899234 | 8.783304 |  9.648678 |        0.0 | 0.868993 |  -148.50983 | 70.672352 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 01:00:00 |  0.0 | -4.8 |             90.7 |       2.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -4.8 |    2.324615 |           1.029472 |                 4.272186 |                    4.272186 |                     4.613345 |             3.874873 | -1.782118 |  12.017133 |                      0.0 |                       0.0 |               39.666214 |                26.684557 |      -39.666214 |       -26.684557 |   0.0 |    0.0 |       24.5 |    0.439577 |      0.0 |      0.0 |      0.0 |        0.750674 |      0.0 |        0.0 |      0.0 |   0.509252 |      0.0 | 0.003442 |      0.0 |  0.003207 |  0.006397 |       0.0 |    -3.784974 |     0.752884 |      0.0 |      0.0 | 0.249086 | 0.249086 |  2.550352 |   4.740251 | -27.978162 |   3.701959 |  14.238404 |   35.126767 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.522207 | 6.522207 |  -47.24727 |  0.90132 | 9.026401 |  9.642281 |        0.0 | 0.876301 |  -151.94363 | 70.669145 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 02:00:00 |  0.0 | -5.6 |             89.7 |       2.5 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -5.6 |    2.003978 |           0.887476 |                 4.019239 |                    4.019239 |                     4.483734 |             3.605258 | -1.823324 |  14.035154 |                      0.0 |                       0.0 |                36.46814 |                 28.62577 |       -36.46814 |        -28.62577 |   0.0 |    0.0 |       24.5 |    0.439567 |      0.0 |      0.0 |      0.0 |        0.750534 |      0.0 |        0.0 |      0.0 |   0.596683 |      0.0 | 0.003363 |      0.0 |  0.002886 |  0.007136 |       0.0 |    -4.162516 |     0.752232 |      0.0 |      0.0 | 0.267245 | 0.267245 |   2.49181 |   5.287666 | -30.848284 |   4.916141 |   8.111667 |   38.829577 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.518844 | 6.518844 | -55.358937 | 0.902928 | 9.286768 |  9.635145 |        0.0 | 0.883609 |  -157.39545 | 70.666259 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 03:00:00 |  0.0 | -6.2 |             91.3 |       3.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -6.2 |    2.404774 |           1.064971 |                 3.838328 |                    3.838328 |                     4.372555 |             3.504393 | -1.888745 |  15.632395 |                      0.0 |                       0.0 |               35.745783 |                29.705445 |      -35.745783 |       -29.705445 |   0.0 |    0.0 |       24.5 |    0.439557 |      0.0 |      0.0 |      0.0 |        0.750414 |      0.0 |        0.0 |      0.0 |   0.633219 |      0.0 | 0.002938 |      0.0 |  0.002458 |  0.007638 |       0.0 |    -4.494145 |     0.751586 |      0.0 |      0.0 | 0.280789 | 0.280789 |  2.176905 |   5.659509 | -34.532989 |    6.31841 |   3.389699 |   41.683364 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.515906 | 6.515906 | -58.748636 | 0.904314 | 9.559972 |  9.627506 |        0.0 | 0.890917 | -164.444512 | 70.663801 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 04:00:00 |  0.0 | -7.3 |             94.2 |       2.5 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -7.3 |    2.003978 |           0.887476 |                  3.52531 |                     3.52531 |                      4.22261 |             3.320842 | -1.973334 |  17.880559 |                      0.0 |                       0.0 |               35.778488 |                32.308553 |      -35.778488 |       -32.308553 |   0.0 |    0.0 |       24.5 |    0.439548 |      0.0 |      0.0 |      0.0 |        0.750301 |      0.0 |        0.0 |      0.0 |   0.683384 |      0.0 | 0.001661 |      0.0 | -0.001439 |  0.007325 |       0.0 |    -4.953427 |     0.750945 |      0.0 |      0.0 | 0.300514 | 0.067534 |   1.23072 |   5.427864 | -32.354981 |   8.025188 |   4.654227 |   45.761605 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.514245 | 6.514245 | -63.402863 | 0.905616 | 9.620181 |  9.620181 | -21.615366 | 0.898225 | -173.741738 |  70.66524 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 05:00:00 |  0.0 | -8.3 |             93.2 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -8.3 |     1.84366 |           0.816478 |                  3.26061 |                     3.26061 |                     3.766374 |             3.038889 | -2.084901 |  26.164341 |                      0.0 |                       0.0 |               36.074771 |                30.183482 |      -36.074771 |       -30.183482 |   0.0 |    0.0 |       24.5 |    0.439543 |      0.0 |      0.0 |      0.0 |        0.750197 |      0.0 |        0.0 |      0.0 |   0.737309 |      0.0 | 0.001741 |      0.0 | -0.001038 |  0.005713 |       0.0 |    -6.445624 |     0.750671 |      0.0 |      0.0 | 0.388015 |      0.0 |  1.290242 |   4.233383 | -32.362003 |   6.131242 |    5.00301 |   40.548106 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.512504 | 6.512504 | -68.405873 | 0.906812 | 9.614468 |  9.614468 | -35.999131 | 0.901359 | -191.322746 | 70.666278 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 06:00:00 |  0.0 | -8.6 |             96.5 |       2.1 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -8.6 |    1.683342 |            0.74548 |                 3.184716 |                    3.184716 |                      3.53143 |             3.073251 | -2.295873 |  29.890831 |                      0.0 |                       0.0 |               34.424201 |                27.717256 |      -34.424201 |       -27.717256 |   0.0 |    0.0 |       24.5 |    0.439538 |      0.0 |      0.0 |      0.0 |        0.750103 |      0.0 |        0.0 |      0.0 |   0.765024 |      0.0 | 0.000845 |      0.0 | -0.003968 |  0.003475 |       0.0 |    -7.277678 |     0.750457 |      0.0 |      0.0 | 0.437842 |      0.0 |  0.626352 |   2.574628 | -32.479167 |   4.221867 |   2.571387 |   34.513752 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.511659 | 6.511659 |  -70.97726 | 0.907902 | 9.610993 |  9.610993 | -40.622052 | 0.903822 | -212.630243 | 70.670245 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 07:00:00 |  0.0 | -8.5 |             95.9 |       1.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                  3.76 |                           7.4 |             0.0 |   0.0 | -8.5 |    1.122228 |           0.496987 |                 3.209838 |                    3.209838 |                     3.475879 |             3.078235 | -2.551563 |  29.580037 |                      0.0 |                       0.0 |               30.562509 |                25.900101 |      -30.562509 |       -25.900101 |   0.0 |    0.0 |       24.5 |    0.439535 |      0.0 |      0.0 |      0.0 |        0.750013 |      0.0 |        0.0 |      0.0 |   0.780448 |      0.0 | 0.000874 |      0.0 |  -0.00429 |   0.00264 |       0.0 |    -7.481569 |     0.750256 |      0.0 |      0.0 | 0.449946 |      0.0 |  0.647425 |   1.956214 |  -29.77897 |   2.846696 |   1.430964 |   30.703011 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.510785 | 6.510785 | -72.408224 | 0.908943 | 9.608353 |  9.608353 | -41.745026 | 0.906135 | -233.626947 | 70.674536 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 08:00:00 |  0.0 | -9.5 |             96.8 |       1.1 |                 0.355232 |              0.0 | 0.0 | 0.0 |                  2.76 |                      6.755232 |         1.10761 |   0.0 | -9.5 |     0.88175 |           0.390489 |                 2.966351 |                    2.966351 |                     3.350494 |             2.871427 | -2.803523 |  30.894515 |                 0.225545 |                  0.051413 |               29.067635 |                27.287337 |       -28.84209 |       -27.235924 |   0.0 |    0.0 |       24.5 |    0.439532 |      0.0 |      0.0 |      0.0 |        0.749925 |      0.0 |        0.0 |      0.0 |   0.805479 |      0.0 | 0.000592 |      0.0 | -0.003375 |  0.002987 |       0.0 |    -7.952609 |     0.750059 |      0.0 |      0.0 | 0.478144 |      0.0 |  0.438509 |   2.213105 | -26.958254 |   4.061565 |   2.322345 |   33.510595 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.510193 | 6.510193 | -74.730569 | 0.909956 | 9.605366 |  9.605366 | -44.361106 | 0.908414 | -255.938128 | 70.677911 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 09:00:00 |  0.0 | -9.2 |             94.3 |       1.0 |                      1.0 |              0.0 | 0.0 | 0.0 |                  1.79 |                      6.755232 |       18.663203 |   0.0 | -9.2 |    0.801591 |            0.35499 |                 3.037615 |                    3.037615 |                     3.350843 |             2.864471 | -3.071258 |  29.280122 |                 3.801704 |                  0.866962 |               24.708347 |                24.720938 |      -20.906643 |       -23.853975 |   0.0 |    0.0 |       24.5 |    0.439531 |      0.0 |      0.0 |      0.0 |        0.749841 |      0.0 |        0.0 |      0.0 |   0.770433 |      0.0 | 0.001056 |      0.0 | -0.000025 |  0.002967 |       0.0 |    -7.951278 |     0.749868 |      0.0 |      0.0 | 0.477915 |      0.0 |  0.782551 |   2.198233 | -24.940691 |   3.206699 |  -3.251497 |   29.258908 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.509137 | 6.509137 | -71.479072 | 0.910927 | 9.602399 |  9.602399 | -44.339892 | 0.910615 | -276.634917 | 70.677936 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 10:00:00 |  0.0 | -6.9 |             96.7 |       1.4 |                      1.0 |              0.0 | 0.0 | 0.0 |                  0.88 |                      6.755232 |       56.267907 |   0.0 | -6.9 |    1.122228 |           0.496987 |                 3.636412 |                    3.636412 |                     3.641135 |             3.516411 | -3.319619 |  21.381715 |                 11.46587 |                  2.615798 |               18.472727 |                17.369353 |       -7.006857 |       -14.753556 |   0.0 |    0.0 |       24.5 |    0.439527 |      0.0 |      0.0 |      0.0 |        0.749753 |      0.0 |        0.0 |      0.0 |   0.609423 |      0.0 | 0.000797 |      0.0 |   0.00009 |  0.000828 |       0.0 |    -6.883238 |     0.749679 |      0.0 |      0.0 | 0.413592 |      0.0 |   0.59035 |   0.613583 |  -22.53533 |   0.046852 | -14.938124 |   15.413991 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 |  6.50834 |  6.50834 | -56.540947 | 0.911957 | 9.601571 |  9.601571 | -38.372168 | 0.912817 | -289.433298 | 70.677846 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 11:00:00 |  0.0 | -4.0 |             97.4 |       1.3 |                      1.0 |              0.1 | 0.0 | 0.0 |                  0.39 |                      6.755232 |      107.168237 |   0.0 | -4.0 |    1.042069 |           0.461488 |                 4.539126 |                    4.539126 |                      4.14924 |             4.421109 |   -3.4732 |   10.26148 |                 21.84815 |                  4.986143 |               16.533721 |                10.320264 |        5.314429 |        -5.334121 |   0.0 |    0.0 |       24.5 |    0.439525 |      0.0 |      0.0 |      0.0 |        0.749636 |      0.0 |        0.0 |      0.0 |   0.392138 |      0.0 | 0.000768 |      0.0 |  0.000489 | -0.001768 |       0.0 |    -5.183446 |     0.749474 |      0.0 |      0.0 |  0.31143 |      0.0 |   0.56879 |  -1.310282 | -15.413603 |  -3.240726 | -20.159242 |    0.783114 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.507572 | 6.507572 | -36.381705 | 0.913311 | 9.603339 |  9.603339 | -28.893802 |   0.9152 | -291.111445 | 70.677357 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 12:00:00 |  0.0 | -2.8 |             93.9 |       1.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                  0.14 |                      6.755232 |       91.473333 |   0.0 | -2.8 |    1.042069 |           0.461488 |                 4.967227 |                    4.967227 |                     4.457474 |             4.664226 | -3.493337 |   4.481067 |                18.660993 |                  4.259907 |               18.012332 |                 8.427556 |        0.648661 |        -4.167649 |   0.0 |    0.0 |      26.95 |    0.439523 |      0.0 |      0.0 |      0.0 |        0.749468 |      0.0 |        0.0 |      0.0 |   0.280336 |      0.0 | 0.001971 |      0.0 |  0.001138 | -0.001345 |       0.0 |    -4.240182 |     0.749239 |      0.0 |      0.0 | 0.254804 |      0.0 |  1.460327 |   -0.99645 | -11.184395 |  -3.943765 | -10.372728 |   -0.772567 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.505602 | 6.505602 | -26.008977 | 0.915268 | 9.604684 |  9.604684 | -23.640168 | 0.917939 | -287.009179 | 70.676218 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 13:00:00 |  0.0 | -1.4 |             93.4 |       1.4 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      6.755232 |       80.475535 |   0.0 | -1.4 |    1.122228 |           0.496987 |                 5.511494 |                    5.511494 |                     4.766224 |             5.147736 |  -3.44411 |  -0.555439 |                16.430777 |                  3.751582 |               17.059386 |                 5.576071 |       -0.628609 |        -1.824489 |   0.0 |    0.0 |       44.1 |    0.439517 |      0.0 |      0.0 |      0.0 |        0.749264 |      0.0 |        0.0 |      0.0 |   0.200323 |      0.0 | 0.002415 |      0.0 |  0.001687 | -0.002533 |       0.0 |    -3.351537 |     0.748982 |      0.0 |      0.0 | 0.201431 |      0.0 |  1.789516 |  -1.876854 |   -9.84154 |  -5.454894 |  -7.423416 |   -5.507259 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.503186 | 6.503186 | -18.585561 |  0.91765 | 9.607217 |  9.607217 | -18.688347 | 0.920945 | -277.870407 | 70.674531 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 14:00:00 |  0.0 | -0.4 |             93.3 |       2.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      6.755232 |       43.120591 |   0.0 | -0.4 |     1.84366 |           0.816478 |                 5.931876 |                    5.931876 |                     5.033102 |              5.53444 | -3.334445 |  -4.265454 |                 8.812405 |                  2.012465 |               14.873485 |                 3.960708 |        -6.06108 |        -1.948243 |   0.0 |    0.0 |       49.0 |     0.43951 |      0.0 |      0.0 |      0.0 |        0.749024 |      0.0 |        0.0 |      0.0 |   0.162514 |      0.0 | 0.003121 |      0.0 |   0.00158 | -0.003937 |       0.0 |    -2.623536 |     0.748697 |      0.0 |      0.0 | 0.157719 |      0.0 |  2.312759 |   -2.91739 | -11.881737 |  -7.351821 |  -3.507898 |   -8.320968 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.500065 | 6.500065 | -15.077663 | 0.920455 | 9.611155 |  9.611155 | -14.632833 | 0.924292 |  -265.02162 | 70.672951 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 15:00:00 |  0.0 |  0.5 |             94.1 |       2.8 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.355232 |       10.320091 |   0.0 |  0.5 |    2.244455 |           0.993973 |                 6.334173 |                    6.334173 |                     5.290061 |             5.960456 | -3.180259 |  -7.356597 |                 2.111295 |                  0.482253 |               11.539149 |                 2.444218 |       -9.427854 |        -1.961966 |   0.0 |    0.0 |       49.0 |      0.4395 |      0.0 |      0.0 |      0.0 |         0.74876 |      0.0 |        0.0 |      0.0 |   0.138481 |      0.0 | 0.003187 |      0.0 |  0.001363 | -0.005717 |       0.0 |     -1.95416 |      0.74838 |      0.0 |      0.0 | 0.117526 |      0.0 |  2.361523 |  -4.236246 | -14.019087 |   -8.81131 |   -2.22971 |  -11.085591 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.496878 | 6.496878 | -12.847953 | 0.923545 | 9.616872 |  9.616872 | -10.903839 | 0.928025 |  -249.08169 | 70.671588 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 16:00:00 |  0.1 |  1.0 |             94.6 |       3.1 |                 0.061898 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |        0.028102 | 0.094 |  1.0 |    2.484933 |            1.10047 |                 6.567919 |                    6.567919 |                     5.489944 |             6.213251 |  -2.98898 |  -9.215531 |                 0.005754 |                  0.001314 |               10.405854 |                 2.180939 |        -10.4001 |        -2.179625 | 0.094 | 0.0235 |       49.0 |    0.439491 | 0.041312 | 0.010328 |  0.04203 |        0.748557 |      0.0 |        0.0 |      0.0 |   0.118269 | 0.030984 | 0.003168 |      0.0 |  0.006769 | -0.006461 |  0.095635 |    -1.453058 |     0.748204 |      0.0 |      0.0 | 0.088281 | 0.039516 |  2.347518 |  -4.787513 | -14.580793 |  -9.225346 |  -1.833175 |  -11.833234 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.535022 | 6.535022 |  -8.098112 | 0.925934 | 9.676021 |  9.676021 |  -4.524304 | 0.930102 | -231.282826 | 70.664819 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 17:00:00 |  0.0 |  1.1 |             94.4 |       3.5 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |   0.0 |  1.1 |    2.805569 |           1.242466 |                 6.615571 |                    6.615571 |                     5.659133 |             6.245099 | -2.775394 | -10.404262 |                      0.0 |                       0.0 |               11.875706 |                  3.57354 |      -11.875706 |         -3.57354 |   0.0 |    0.0 |       49.0 |    0.439605 |      0.0 |      0.0 |      0.0 |        0.748212 |      0.0 |        0.0 |      0.0 |   0.095572 |      0.0 | 0.003509 |      0.0 |  0.007884 | -0.005551 |       0.0 |     -1.04135 |     0.747731 |      0.0 |      0.0 | 0.063051 |      0.0 |  2.600263 |  -4.112764 | -13.707117 |   -8.53959 |   0.768852 |   -9.078814 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.531513 | 6.531513 |  -8.866965 | 0.930011 | 9.681572 |  9.681572 |  -5.849752 | 0.935716 | -212.295231 | 70.656935 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 18:00:00 |  0.3 |  1.0 |             96.9 |       4.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.282 |  1.0 |    3.527001 |           1.561958 |                 6.567919 |                    6.567919 |                     5.699177 |             6.364313 | -2.547543 |  -9.612254 |                      0.0 |                       0.0 |               12.252381 |                 4.311207 |      -12.252381 |        -4.311207 | 0.282 | 0.0705 |       49.0 |    0.439595 | 0.123966 | 0.030991 | 0.126121 |        0.748117 |      0.0 |        0.0 |      0.0 |   0.090842 | 0.090842 | 0.002176 |      0.0 |  0.004285 | -0.007108 |  0.286904 |      -0.9455 |     0.747834 |      0.0 |      0.0 | 0.058919 | 0.058919 |  1.612244 |  -5.266861 | -14.177315 |  -8.753064 |  -0.312689 |   -9.708718 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.651171 | 6.653302 |        0.0 |  0.93113 | 9.787064 |  9.846714 |        0.0 | 0.934486 | -194.099643 | 70.652651 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 19:00:00 |  0.2 |  1.4 |             98.3 |       3.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.188 |  1.4 |    3.046047 |           1.348964 |                 6.760362 |                    6.760362 |                     5.914887 |             6.645436 | -2.329196 | -11.340094 |                      0.0 |                       0.0 |               13.548131 |                 4.513202 |      -13.548131 |        -4.513202 | 0.188 | 0.0282 |       49.0 |     0.43996 | 0.082712 | 0.012407 | 0.124556 |        0.747598 |      0.0 |        0.0 |      0.0 |   0.044348 | 0.044348 | 0.001135 |      0.0 |  0.001535 | -0.007216 |  0.283107 |     -0.43918 |     0.747441 |      0.0 |      0.0 | 0.027761 | 0.027761 |  0.841108 |  -5.346653 | -10.150145 |  -7.647939 |   4.239094 |    -8.48139 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.706796 |  6.73488 |        0.0 |  0.93729 | 9.837746 |  9.959217 |        0.0 | 0.939171 | -174.176216 | 70.651115 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 20:00:00 |  0.4 |  1.6 |             98.7 |       5.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.376 |  1.6 |     4.56907 |           2.023445 |                 6.858436 |                    6.858436 |                     6.058417 |             6.769276 | -2.090115 |  -11.87322 |                      0.0 |                       0.0 |               13.762728 |                 4.983727 |      -13.762728 |        -4.983727 | 0.376 | 0.0376 |       49.0 |    0.440205 | 0.165517 | 0.016552 | 0.292569 |        0.747115 |      0.0 |        0.0 |      0.0 |   0.028015 | 0.028015 | 0.001109 |      0.0 |  0.001492 | -0.008844 |  0.664621 |    -0.111245 |     0.747123 |      0.0 |      0.0 | 0.007296 | 0.007296 |  0.821876 |  -6.552718 | -11.692875 |  -8.962665 |   2.891729 |  -10.531657 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.750277 | 6.899287 |        0.0 | 0.943064 | 9.874671 | 10.178544 |        0.0 | 0.942974 | -153.719663 | 70.649624 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 21:00:00 |  0.7 |  2.0 |             99.6 |       5.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.658 |  2.0 |    4.488911 |           1.987946 |                 7.058357 |                    7.058357 |                       6.1078 |             7.030123 | -1.844636 | -11.067816 |                      0.0 |                       0.0 |                12.71147 |                 3.399829 |       -12.71147 |        -3.399829 | 0.658 |    0.0 |       49.0 |    0.440698 | 0.289979 |      0.0 | 0.674492 |        0.746505 |      0.0 |        0.0 |      0.0 |   0.009603 | 0.009603 | 0.000347 |      0.0 | -0.000343 |  -0.01135 |  1.530508 |          0.0 |      0.74651 | 0.062882 | 0.062882 |      0.0 |      0.0 |  0.257434 |  -8.409795 | -11.403474 | -10.361428 |    1.56543 |  -15.371395 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.759554 | 7.188919 |        0.0 | 0.950409 | 9.822347 | 10.557915 |        0.0 | 0.950339 | -134.068514 | 70.649966 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 22:00:00 |  0.5 |  2.4 |             99.4 |       5.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.47 |  2.4 |    4.649229 |           2.058944 |                 7.263399 |                    7.263399 |                       6.1078 |             7.219819 | -1.608822 |  -9.652933 |                      0.0 |                       0.0 |               11.165804 |                 1.378924 |      -11.165804 |        -1.378924 |  0.47 |    0.0 |       49.0 |    0.441567 | 0.207536 |      0.0 | 0.579276 |        0.745899 |      0.0 |   0.018471 | 0.018471 |        0.0 |      0.0 | 0.000548 |      0.0 |   0.00012 | -0.013984 |  1.311864 |          0.0 |     0.745904 | 0.143872 | 0.143872 |      0.0 |      0.0 |  0.406082 | -10.361775 | -12.706346 | -12.706346 |  -1.134461 |  -21.689197 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.740583 | 7.395907 |        0.0 | 0.957747 | 9.690984 | 10.834363 |        0.0 | 0.957693 | -115.832247 | 70.649846 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 23:00:00 |  1.0 |  2.5 |             99.2 |       6.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.94 |  2.5 |    4.969866 |           2.200941 |                 7.315473 |                    7.315473 |                       6.1078 |             7.256949 | -1.389987 |  -8.339922 |                      0.0 |                       0.0 |               10.640998 |                 0.884234 |      -10.640998 |        -0.884234 |  0.94 |    0.0 |       49.0 |    0.442188 | 0.415656 |      0.0 | 1.208521 |        0.745296 |      0.0 |   0.040986 | 0.040986 |        0.0 |      0.0 | 0.000768 |      0.0 |  0.000639 | -0.015071 |   2.73305 |          0.0 |     0.745298 | 0.199185 | 0.199185 |      0.0 |      0.0 |  0.568725 | -11.167257 | -13.803762 | -13.803762 |  -2.594038 |  -24.086786 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.698939 | 7.810796 |        0.0 | 0.965112 | 9.504393 | 11.373778 |        0.0 | 0.965089 |  -98.908992 | 70.649208 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-11 00:00:00 |  2.0 |  2.7 |             99.2 |       5.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  1.88 |  2.7 |    4.488911 |           1.987946 |                 7.420609 |                    7.420609 |                       6.1078 |             7.361244 | -1.186908 |  -7.121447 |                      0.0 |                       0.0 |                9.446615 |                -0.139391 |       -9.446615 |         0.139391 |  1.88 |    0.0 |       49.0 |    0.443432 | 0.833653 |      0.0 | 2.617753 |        0.744694 |      0.0 |   0.071329 | 0.071329 |        0.0 |      0.0 | 0.000731 |      0.0 |  0.000535 | -0.015425 |  5.903388 |          0.0 |     0.744691 | 0.262328 | 0.262328 |      0.0 |      0.0 |  0.541292 | -11.428977 | -13.987928 | -13.987928 |  -4.000021 |  -25.556296 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      2.2 | 6.627049 | 8.643719 |        0.0 | 0.972524 | 9.253543 | 12.435549 |        0.0 |  0.97256 |  -83.204212 | 70.648673 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-11 01:00:00 |  2.1 |  3.2 |             99.6 |       5.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 1.974 |  3.2 |    4.168274 |            1.84595 |                 7.689299 |                    7.689299 |                       6.1078 |             7.658542 | -0.998451 |  -5.990703 |                      0.0 |                       0.0 |                6.297157 |                -2.742905 |       -6.297157 |         2.742905 | 1.974 |    0.0 |       49.0 |    0.445931 | 0.880268 |      0.0 |  3.09995 |        0.744096 | 0.238205 |   0.133501 | 0.133501 |        0.0 |      0.0 | 0.000362 |      0.0 | -0.000381 | -0.018246 |  5.355679 |          0.0 |     0.744091 | 0.339293 | 0.339293 |      0.0 |      0.0 |  0.268147 | -13.519673 | -15.851265 | -15.851265 |   -9.28596 |  -32.113842 |      1.0 |      0.3 | 1.020778 | 0.348096 |  0.0 |  0.0 | 0.0 |  0.0 | 0.348096 | 0.348096 |      0.0 |   0.0 |   0.0 |  0.0 | 0.004515 |      0.0 |   0.0 |   0.0 |  0.0 | 0.004515 | 0.001254 |      2.2 |   6.4933 |  9.28542 |        0.0 | 0.979938 | 8.926542 | 12.764955 |        0.0 |     0.98 |  -68.630175 | 71.321736 |  0.34358 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001254 |
    | 2010-12-11 02:00:00 |  1.8 |  3.9 |             99.8 |       5.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 1.692 |  3.9 |     4.56907 |           2.023445 |                  8.07986 |                     8.07986 |                       6.1078 |               8.0637 | -0.823562 |  -4.941373 |                      0.0 |                       0.0 |                1.967269 |                -6.386972 |       -1.967269 |         6.386972 | 1.692 |    0.0 |       49.0 |    0.447856 | 0.757773 |      0.0 |      0.0 |        0.743511 | 1.039983 |   0.197354 | 0.197354 |        0.0 |      0.0 | 0.000201 |      0.0 | -0.000728 | -0.024333 | -1.280021 |          0.0 |     0.743533 | 0.416279 | 0.416279 |      0.0 |      0.0 |  0.148961 | -18.029533 | -20.426299 | -20.426299 | -18.310069 |  -44.842804 |      1.0 |      0.3 | 2.570623 | 0.878919 |  0.0 |  0.0 | 0.0 |  0.0 | 0.878919 |      0.5 | 0.378919 |   0.0 |   0.0 |  0.0 | 0.015361 | 0.019159 |   0.0 |   0.0 |  0.0 | 0.034519 | 0.009589 |      2.2 |  6.29581 | 9.003008 |        0.0 | 0.987246 | 8.526486 | 12.192874 |        0.0 | 0.986969 |  -55.105469 | 73.014169 |  0.82822 |  0.35976 |  0.0 |  0.0 | 0.0 |   0.0 | 0.009589 |
    | 2010-12-11 03:00:00 |  1.3 |  4.4 |             99.8 |       6.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 1.222 |  4.4 |    5.290502 |           2.342937 |                  8.36943 |                     8.36943 |                       6.1078 |             8.352691 | -0.661266 |  -3.967594 |                      0.0 |                       0.0 |               -1.107912 |                -8.998234 |        1.107912 |         8.998234 | 1.222 |    0.0 |       49.0 |    0.447009 | 0.546245 |      0.0 |      0.0 |         0.74293 | 0.950566 |   0.282749 | 0.282749 |        0.0 |      0.0 | 0.000229 |      0.0 | -0.000588 | -0.030654 | -2.068991 |          0.0 |     0.742996 | 0.549367 | 0.549367 |      0.0 |      0.0 |  0.169359 | -22.713156 | -25.294274 | -25.294274 | -26.232827 |  -57.005664 |      1.0 |      0.3 | 2.413891 | 0.827844 |  0.0 |  0.0 | 0.0 |  0.0 | 0.827844 |      0.5 | 0.327844 |   0.0 |   0.0 |  0.0 |  0.02788 | 0.052332 |   0.0 |   0.0 |  0.0 | 0.080212 | 0.022281 |      2.2 | 6.012908 | 8.598459 |        0.0 | 0.994554 | 7.997173 | 11.435958 |        0.0 |  0.99372 |  -42.554542 | 74.600804 |  1.30034 | 0.635271 |  0.0 |  0.0 | 0.0 |   0.0 | 0.022281 |
    | 2010-12-11 04:00:00 |  0.5 |  4.8 |             99.0 |       7.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.47 |  4.8 |    5.611139 |           2.484933 |                 8.607627 |                    8.607627 |                       6.1078 |             8.521551 | -0.510655 |  -3.063927 |                      0.0 |                       0.0 |               -3.288969 |               -11.031041 |        3.288969 |        11.031041 |  0.47 |    0.0 |       49.0 |    0.445795 | 0.209524 |      0.0 |      0.0 |        0.742354 | 0.688311 |   0.334864 | 0.334864 |        0.0 |      0.0 | 0.001222 |      0.0 |  0.001528 | -0.034262 | -2.672783 |          0.0 |     0.742478 | 0.639867 | 0.639867 |      0.0 |      0.0 |  0.905313 | -25.386793 | -28.684284 | -28.684284 |  -31.06794 |  -65.102118 |      1.0 |      0.3 | 1.866538 | 0.641733 |  0.0 |  0.0 | 0.0 |  0.0 | 0.641733 |      0.5 | 0.141733 |   0.0 |   0.0 |  0.0 | 0.040075 | 0.070305 |   0.0 |   0.0 |  0.0 |  0.11038 | 0.030661 |      2.2 | 5.677238 |  8.11845 |        0.0 | 1.001862 | 7.379349 | 10.552469 |        0.0 |  1.00028 |  -30.907282 | 75.824082 | 1.760265 |   0.7067 |  0.0 |  0.0 | 0.0 |   0.0 | 0.030661 |
    | 2010-12-11 05:00:00 |  0.2 |  4.8 |             97.6 |       7.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.188 |  4.8 |    5.851616 |            2.59143 |                 8.607627 |                    8.607627 |                       6.1078 |             8.401044 | -0.370887 |  -2.225324 |                      0.0 |                       0.0 |               -2.759879 |               -10.910371 |        2.759879 |        10.910371 | 0.188 |    0.0 |       49.0 |    0.444355 | 0.083539 |      0.0 |      0.0 |        0.741781 | 0.546184 |   0.323648 | 0.323648 |        0.0 |      0.0 | 0.003016 |      0.0 |  0.005233 |  -0.03348 | -2.582669 |          0.0 |     0.741966 | 0.651144 | 0.651144 |      0.0 |      0.0 |  2.234705 | -24.807089 | -29.502182 | -29.502182 | -30.027356 |  -65.219643 |      1.0 |      0.3 | 1.584734 | 0.545969 |  0.0 |  0.0 | 0.0 |  0.0 | 0.545969 |      0.5 | 0.045969 |   0.0 |   0.0 |  0.0 | 0.051955 | 0.072562 |   0.0 |   0.0 |  0.0 | 0.124517 | 0.034588 |      2.2 | 5.351601 | 7.652789 |        0.0 |  1.00917 | 6.749552 |  9.651859 |        0.0 | 1.006807 |  -20.098624 | 76.857614 |  2.20831 | 0.680106 |  0.0 |  0.0 | 0.0 |   0.0 | 0.034588 |
    | 2010-12-11 06:00:00 |  0.5 |  5.1 |             97.4 |       7.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.47 |  5.1 |    5.771457 |           2.555931 |                 8.790172 |                    8.790172 |                       6.1078 |             8.561627 | -0.241183 |  -1.447101 |                      0.0 |                       0.0 |               -4.549601 |               -12.474442 |        4.549601 |        12.474442 |  0.47 |    0.0 |       49.0 |    0.442958 |  0.20819 |      0.0 |      0.0 |        0.741213 | 0.719018 |   0.357376 | 0.357376 |        0.0 |      0.0 | 0.003306 |      0.0 |   0.00585 | -0.035493 |  -3.02987 |          0.0 |     0.741472 | 0.704401 | 0.704401 |      0.0 |      0.0 |  2.449426 | -26.298897 | -31.056396 | -31.056396 | -33.156572 |  -69.829736 |      1.0 |      0.3 | 1.991825 | 0.687807 |  0.0 |  0.0 | 0.0 |  0.0 | 0.687807 |      0.5 | 0.187807 |   0.0 |   0.0 |  0.0 | 0.063528 | 0.077091 |   0.0 |   0.0 |  0.0 | 0.140619 | 0.039061 |      2.2 | 4.992067 | 7.138656 |        0.0 | 1.016478 | 6.067381 |  8.676355 |        0.0 |  1.01314 |   -10.06819 | 78.155782 | 2.644782 | 0.790823 |  0.0 |  0.0 | 0.0 |   0.0 | 0.039061 |
    | 2010-12-11 07:00:00 |  0.2 |  5.1 |             96.3 |       7.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.188 |  5.1 |    6.092093 |           2.697927 |                 8.790172 |                    8.790172 |                       6.1078 |             8.464935 | -0.120818 |   -0.72491 |                      0.0 |                       0.0 |               -4.125536 |               -12.377725 |        4.125536 |        12.377725 | 0.188 |    0.0 |       49.0 |    0.441416 | 0.082986 |      0.0 |      0.0 |        0.740649 | 0.587035 |   0.352723 | 0.352723 |        0.0 |      0.0 |  0.00488 |      0.0 |  0.008934 | -0.035367 | -2.989666 |          0.0 |     0.740988 | 0.723054 | 0.723054 |      0.0 |      0.0 |  3.615771 | -26.205129 | -32.215085 | -32.215085 |  -32.72485 |  -70.797939 |      1.0 |      0.3 | 1.730231 | 0.598806 |  0.0 |  0.0 | 0.0 |  0.0 | 0.598806 |      0.5 | 0.098806 |   0.0 |   0.0 |  0.0 | 0.074803 | 0.083594 |   0.0 |   0.0 |  0.0 | 0.158397 | 0.043999 |      2.2 | 4.636173 | 6.629727 |        0.0 | 1.023786 | 5.366111 |  7.673539 |        0.0 | 1.019391 |   -0.759947 | 79.278274 | 3.069979 | 0.806034 |  0.0 |  0.0 | 0.0 |   0.0 | 0.043999 |
    | 2010-12-11 08:00:00 |  0.1 |  5.4 |             95.2 |       7.3 |                 0.347041 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.40894 |        0.900734 | 0.094 |  5.4 |    5.851616 |            2.59143 |                 8.976116 |                    8.976116 |                       6.1078 |             8.545263 | -0.009119 |  -0.054716 |                 0.190633 |                  0.043405 |               -5.565868 |               -13.865856 |        5.756501 |        13.909261 | 0.094 |    0.0 |       49.0 |    0.439889 |  0.04135 |      0.0 |      0.0 |        0.740089 |   0.5693 |   0.369547 | 0.369547 |        0.0 |      0.0 |  0.00629 |      0.0 |  0.009363 | -0.035585 | -3.315633 |          0.0 |     0.740524 | 0.755526 | 0.755526 |      0.0 |      0.0 |  4.660744 |  -26.36717 | -33.189955 | -33.189955 | -34.285712 |  -73.466385 | 0.998939 | 0.299363 | 1.707363 | 0.591124 |  0.0 |  0.0 | 0.0 |  0.0 | 0.591124 |      0.5 | 0.091124 |   0.0 |   0.0 |  0.0 | 0.085786 | 0.084718 |   0.0 |   0.0 |  0.0 | 0.170504 | 0.047362 |      2.2 | 4.262578 | 6.095487 |        0.0 | 1.031094 | 4.631966 |  6.623712 |        0.0 | 1.025415 |    7.878103 |  80.38515 | 3.484193 |  0.81244 |  0.0 |  0.0 | 0.0 |   0.0 | 0.047362 |
    | 2010-12-11 09:00:00 |  0.3 |  5.3 |             96.3 |       7.1 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.40894 |        17.26751 | 0.282 |  5.3 |    5.691298 |           2.520432 |                 8.913754 |                    8.913754 |                       6.1078 |             8.583945 |       0.0 |        0.0 |                 3.662347 |                  0.833512 |               -5.371692 |               -13.434609 |         9.03404 |        14.268121 | 0.282 |    0.0 |       49.0 |    0.438286 | 0.123597 |      0.0 |      0.0 |        0.739533 | 0.701228 |   0.404251 | 0.404251 |        0.0 |      0.0 | 0.004726 |      0.0 |  0.002601 | -0.035482 | -3.560461 |          0.0 |     0.740082 | 0.743405 | 0.743405 |      0.0 |      0.0 |  3.501758 | -26.290556 | -31.973262 | -31.973262 | -37.505544 |  -72.531939 | 0.997782 | 0.298671 | 1.928395 | 0.667945 |  0.0 |  0.0 | 0.0 |  0.0 | 0.667945 |      0.5 | 0.167945 |   0.0 |   0.0 |  0.0 | 0.096486 | 0.089239 |   0.0 |   0.0 |  0.0 | 0.185724 |  0.05159 |      2.2 | 3.855335 | 5.513129 |        0.0 | 1.038402 | 3.909391 |   5.59043 |        0.0 | 1.031187 |   16.461436 |    81.643 | 3.887707 | 0.891146 |  0.0 |  0.0 | 0.0 |   0.0 |  0.05159 |
    | 2010-12-11 10:00:00 |  0.1 |  5.5 |             95.5 |       7.1 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.40894 |        52.61761 | 0.094 |  5.5 |    5.691298 |           2.520432 |                 9.038862 |                    9.038862 |                       6.1078 |             8.632113 |       0.0 |        0.0 |                11.183569 |                  2.543747 |               -6.309796 |               -14.422901 |       17.493365 |        16.966648 | 0.094 |    0.0 |       49.0 |    0.436539 | 0.041035 |      0.0 |      0.0 |        0.738981 | 0.754749 |   0.499629 | 0.499629 |        0.0 |      0.0 | 0.005828 |      0.0 |  0.003784 | -0.036172 | -4.565275 |          0.0 |     0.739686 | 0.780178 | 0.780178 |      0.0 |      0.0 |   4.31867 |  -26.80199 | -33.179801 | -33.179801 | -46.354495 |  -76.948439 | 0.996626 | 0.297979 | 1.930587 | 0.669072 |  0.0 |  0.0 | 0.0 |  0.0 | 0.669072 |      0.5 | 0.169072 |   0.0 |   0.0 |  0.0 | 0.106909 | 0.097118 |   0.0 |   0.0 |  0.0 | 0.204027 | 0.056674 |      2.2 | 3.352158 | 4.793586 |        0.0 |  1.04571 | 3.149461 |  4.503729 |        0.0 | 1.036383 |   25.044769 | 82.900731 | 4.280799 |   0.9631 |  0.0 |  0.0 | 0.0 |   0.0 | 0.056674 |
    | 2010-12-11 11:00:00 |  0.1 |  6.1 |             91.6 |       8.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       80.135633 | 0.094 |  6.1 |    7.134162 |           3.159415 |                 9.423511 |                    9.423511 |                       6.1078 |             8.631936 |       0.0 |        0.0 |                17.068132 |                  3.878875 |                -9.79282 |               -17.550317 |       26.860952 |        21.429192 | 0.094 |    0.0 |       49.0 |    0.434381 | 0.040832 |      0.0 |      0.0 |        0.738433 | 0.964107 |   0.647439 | 0.647439 |        0.0 |      0.0 | 0.013265 |      0.0 |  0.007618 | -0.042299 | -6.549992 |          0.0 |     0.739364 | 0.962052 | 0.962052 |      0.0 |      0.0 |  9.828916 | -31.341975 | -43.035887 | -43.035887 | -60.067923 |  -95.807053 | 0.995469 | 0.297288 | 2.405931 | 0.834637 |  0.0 |  0.0 | 0.0 |  0.0 | 0.834637 |      0.5 | 0.334637 |   0.0 |   0.0 |  0.0 | 0.117063 | 0.112641 |   0.0 |   0.0 |  0.0 | 0.229703 | 0.063806 |      2.2 | 2.697235 | 3.857046 |        0.0 | 1.053018 | 2.207953 |  3.157372 |        0.0 | 1.040635 |   33.628103 | 84.464406 | 4.663736 | 1.185096 |  0.0 |  0.0 | 0.0 |   0.0 | 0.063806 |
    | 2010-12-11 12:00:00 |  0.0 |  6.4 |             90.0 |       9.5 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       80.276118 |   0.0 |  6.4 |    7.615117 |           3.372409 |                 9.621181 |                    9.621181 |                       6.1078 |             8.659063 |       0.0 |        0.0 |                17.133629 |                  3.889627 |              -11.020237 |               -19.002428 |       28.153866 |        22.892055 |   0.0 |    0.0 |       49.0 |    0.431571 |      0.0 |      0.0 |      0.0 |        0.737888 | 0.966218 |   0.678651 | 0.678651 |        0.0 |      0.0 | 0.016902 |      0.0 |  0.009414 | -0.044819 | -7.191755 |          0.0 |     0.739099 | 1.037348 | 1.037348 |      0.0 |      0.0 | 12.523606 |  -33.20903 | -47.333467 | -47.333467 | -62.963727 | -103.434552 | 0.994313 | 0.296597 | 2.470682 | 0.858159 |  0.0 |  0.0 | 0.0 |  0.0 | 0.858159 |      0.5 | 0.358159 |   0.0 |   0.0 |  0.0 | 0.126954 | 0.135894 |   0.0 |   0.0 |  0.0 | 0.262848 | 0.073013 |      2.2 | 2.009738 | 2.873926 |        0.0 | 1.060326 | 1.187222 |  1.697727 |        0.0 | 1.044145 |   42.211436 | 86.067515 | 5.036782 | 1.407362 |  0.0 |  0.0 | 0.0 |   0.0 | 0.073013 |
    | 2010-12-11 13:00:00 |  0.1 |  6.4 |             85.7 |       7.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       75.835403 | 0.094 |  6.4 |    6.332571 |           2.804424 |                 9.621181 |                    9.621181 |                       6.1078 |             8.245352 |       0.0 |        0.0 |                  16.2192 |                  3.677548 |               -9.152157 |               -18.576375 |       25.371357 |        22.253923 | 0.094 |    0.0 |       49.0 |    0.428622 |  0.04029 |      0.0 |      0.0 |        0.737348 | 0.823325 |   0.551646 | 0.551646 |        0.0 |      0.0 |   0.0212 |      0.0 |  0.013073 | -0.032937 | -5.828281 |          0.0 |     0.738879 | 0.887582 | 0.887582 |      0.0 |      0.0 | 15.708194 | -24.404983 | -41.517304 | -41.517304 | -51.180467 |   -88.17621 | 0.993156 | 0.295908 | 2.170901 | 0.754793 |  0.0 |  0.0 | 0.0 |  0.0 | 0.754793 |      0.5 | 0.254793 |   0.0 |   0.0 |  0.0 |  0.13659 | 0.152758 |   0.0 |   0.0 |  0.0 | 0.289349 | 0.080375 |      2.2 | 1.447337 | 2.069692 |        0.0 | 1.067634 | 0.305453 |  0.436797 |        0.0 | 1.047057 |   50.794769 |  87.47055 | 5.400191 | 1.509397 |  0.0 |  0.0 | 0.0 |   0.0 | 0.080375 |
    | 2010-12-11 14:00:00 |  0.0 |  6.5 |             82.3 |       9.6 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       42.227911 |   0.0 |  6.5 |    7.695276 |           3.407908 |                 9.687876 |                    9.687876 |                       6.1078 |             7.973122 |       0.0 |        0.0 |                 9.049891 |                  2.048586 |               -8.244287 |               -18.760881 |       17.294177 |        20.809467 |   0.0 |    0.0 |       49.0 |    0.426209 |      0.0 |      0.0 |      0.0 |        0.736811 |  0.65676 |   0.466109 | 0.466109 |        0.0 |      0.0 | 0.030355 |      0.0 |  0.015898 | -0.033021 | -4.964778 |          0.0 |     0.738778 | 0.956627 | 0.305453 |      0.0 |      0.0 | 22.491862 | -24.466815 | -48.442242 | -48.442242 | -43.244558 |  -93.718524 |    0.992 | 0.295219 | 1.126578 | 0.391681 |  0.0 |  0.0 | 0.0 |  0.0 | 0.391681 | 0.391681 |      0.0 |   0.0 |   0.0 |  0.0 | 0.144572 | 0.150017 |   0.0 |   0.0 |  0.0 | 0.294589 |  0.08183 |      2.2 | 0.966837 | 1.382576 |        0.0 | 1.074942 |      0.0 |       0.0 |        0.0 | 1.048404 |   59.378103 |  88.18955 |   5.6473 |  1.35938 |  0.0 |  0.0 | 0.0 |   0.0 |  0.08183 |
    | 2010-12-11 15:00:00 |  0.1 |  6.3 |             82.5 |       9.1 |                      1.0 |              0.1 | 0.0 | 0.0 |                   0.1 |                       7.40894 |       12.613921 | 0.094 |  6.3 |     7.29448 |           3.230413 |                 9.554889 |                    9.554889 |                          0.0 |             7.882784 |       0.0 |      -37.8 |                  2.70877 |                       0.0 |                -5.80609 |                      0.0 |         8.51486 |              0.0 | 0.094 |    0.0 |       49.0 |    0.424148 |  0.03987 |      0.0 |      0.0 |        0.736279 | 0.531701 |    0.35117 |  0.35117 |        0.0 |      0.0 | 0.028472 |      0.0 |  0.015417 |       0.0 | -3.603596 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 | 21.096693 |        0.0 |  -45.16256 |        0.0 | -32.580727 |         0.0 | 0.985751 | 0.291511 | 0.585831 | 0.201855 |  0.0 |  0.0 | 0.0 |  0.0 | 0.201855 | 0.201855 |      0.0 |   0.0 |   0.0 |  0.0 | 0.148493 | 0.135107 |   0.0 |   0.0 |  0.0 |   0.2836 | 0.078778 |      2.2 | 0.602988 | 0.862273 |        0.0 |  1.08225 |      0.0 |       0.0 |        0.0 |      nan |  105.761436 | 88.558108 | 5.700662 | 1.224273 |  0.0 |  0.0 | 0.0 |   0.0 | 0.078778 |
    | 2010-12-11 16:00:00 |  0.0 |  5.9 |             82.2 |       8.7 |                 0.053708 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |        0.021912 |   0.0 |  5.9 |    6.973844 |           3.088417 |                 9.293725 |                    9.293725 |                          0.0 |             7.639442 |       0.0 |      -35.4 |                 0.004715 |                       0.0 |               -3.188375 |                      0.0 |         3.19309 |              0.0 |   0.0 |    0.0 |       49.0 |    0.422587 |      0.0 |      0.0 |      0.0 |         0.73575 |  0.35728 |   0.258008 | 0.258008 |        0.0 |      0.0 | 0.027276 |      0.0 |  0.048899 |       0.0 | -2.451551 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 | 20.210344 |        0.0 | -40.954652 |        0.0 | -23.937398 |         0.0 | 0.979825 | 0.288017 |  0.35728 | 0.122015 |  0.0 |  0.0 | 0.0 |  0.0 | 0.122015 | 0.122015 |      0.0 |   0.0 |   0.0 |  0.0 | 0.148836 | 0.121679 |   0.0 |   0.0 |  0.0 | 0.270515 | 0.075143 |      2.2 | 0.334068 | 0.477717 |        0.0 | 1.089558 |      0.0 |       0.0 |        0.0 |      nan |  149.744769 | 88.744474 | 5.673841 | 1.102595 |  0.0 |  0.0 | 0.0 |   0.0 | 0.075143 |
    | 2010-12-11 17:00:00 |  0.0 |  5.5 |             76.5 |       8.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  5.5 |    7.134162 |           3.159415 |                 9.038862 |                    9.038862 |                          0.0 |              6.91473 |       0.0 |      -33.0 |                      0.0 |                       0.0 |                1.905808 |                      0.0 |       -1.905808 |              0.0 |   0.0 |    0.0 |       49.0 |    0.421433 |      0.0 |      0.0 |      0.0 |        0.735224 | 0.150091 |   0.113409 | 0.113409 |        0.0 |      0.0 | 0.035596 |      0.0 |  0.065707 |       0.0 | -0.960059 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 | 26.375167 |        0.0 | -38.802849 |        0.0 | -10.521874 |         0.0 | 0.974222 | 0.284733 | 0.150091 | 0.050811 |  0.0 |  0.0 | 0.0 |  0.0 | 0.050811 | 0.050811 |      0.0 |   0.0 |   0.0 |  0.0 | 0.147219 | 0.109585 |   0.0 |   0.0 |  0.0 | 0.256805 | 0.071335 |      2.2 | 0.204217 |  0.29203 |        0.0 | 1.096866 |      0.0 |       0.0 |        0.0 |      nan |  191.328103 | 88.778047 | 5.577433 | 0.993009 |  0.0 |  0.0 | 0.0 |   0.0 | 0.071335 |
    | 2010-12-11 18:00:00 |  0.0 |  5.1 |             81.7 |       7.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  5.1 |    6.332571 |           2.804424 |                 8.790172 |                    8.790172 |                          0.0 |              7.18157 |       0.0 |      -30.6 |                      0.0 |                       0.0 |                1.939405 |                      0.0 |       -1.939405 |              0.0 |   0.0 |    0.0 |       49.0 |    0.420876 |      0.0 |      0.0 |      0.0 |        0.734702 | 0.180245 |   0.137736 | 0.137736 |        0.0 |      0.0 | 0.024787 |      0.0 |  0.048384 |       0.0 |  -1.06909 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 | 18.365818 |        0.0 | -33.084102 |        0.0 | -12.778878 |         0.0 | 0.968943 | 0.281655 | 0.180245 | 0.060513 |  0.0 |  0.0 | 0.0 |  0.0 | 0.060513 | 0.060513 |      0.0 |   0.0 |   0.0 |  0.0 | 0.144855 | 0.098694 |   0.0 |   0.0 |  0.0 | 0.243549 | 0.067652 |      2.2 | 0.060837 | 0.086998 |        0.0 | 1.104174 |      0.0 |       0.0 |        0.0 |      nan |  230.511436 | 88.849396 | 5.493091 | 0.894316 |  0.0 |  0.0 | 0.0 |   0.0 | 0.067652 |
    | 2010-12-11 19:00:00 |  0.0 |  4.8 |             83.4 |       7.1 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  4.8 |    5.691298 |           2.520432 |                 8.607627 |                    8.607627 |                          0.0 |             7.178761 |       0.0 |      -28.8 |                      0.0 |                       0.0 |                3.012755 |                      0.0 |       -3.012755 |              0.0 |   0.0 |    0.0 |       49.0 |    0.420261 |      0.0 |      0.0 |      0.0 |        0.734184 | 0.066523 |   0.116118 | 0.060837 |        0.0 |      0.0 | 0.020475 |  0.15874 |       0.0 |       0.0 | -0.371357 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 | 15.171037 |        0.0 | -28.956917 |        0.0 | -10.773124 |         0.0 | 0.963906 | 0.278735 | 0.066523 | 0.022154 |  0.0 |  0.0 | 0.0 |  0.0 | 0.022154 | 0.022154 |      0.0 |   0.0 |   0.0 |  0.0 | 0.142179 | 0.088885 |   0.0 |   0.0 |  0.0 | 0.231063 | 0.064184 |  2.04126 |      0.0 |      0.0 |        0.0 | 1.111482 |      0.0 |       0.0 |        0.0 |      nan |  267.894769 | 88.893765 | 5.373066 | 0.805431 |  0.0 |  0.0 | 0.0 |   0.0 | 0.064184 |
    | 2010-12-11 20:00:00 |  0.0 |  4.9 |             82.8 |       6.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  4.9 |    5.130184 |           2.271939 |                 8.668101 |                         0.0 |                          0.0 |             7.177187 |       0.0 |      -29.4 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.148995 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.958789 | 0.275783 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.138791 | 0.080051 |   0.0 |   0.0 |  0.0 | 0.218842 | 0.060789 | 1.892265 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  305.878103 | 88.893765 | 5.234275 |  0.72538 |  0.0 |  0.0 | 0.0 |   0.0 | 0.060789 |
    | 2010-12-11 21:00:00 |  0.0 |  4.9 |             85.8 |       7.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  4.9 |    5.771457 |           2.555931 |                 8.668101 |                         0.0 |                          0.0 |              7.43723 |       0.0 |      -29.4 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.137771 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.953671 | 0.272847 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.135206 | 0.072094 |   0.0 |   0.0 |  0.0 |   0.2073 | 0.057583 | 1.754494 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  343.861436 | 88.893765 |  5.09907 | 0.653286 |  0.0 |  0.0 | 0.0 |   0.0 | 0.057583 |
    | 2010-12-11 22:00:00 |  0.1 |  4.7 |             87.6 |       7.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 | 0.094 |  4.7 |    5.611139 |           2.484933 |                 8.547525 |                         0.0 |                          0.0 |             7.487632 |       0.0 |      -28.2 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.115706 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.948716 | 0.270018 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.131713 | 0.064929 |   0.0 |   0.0 |  0.0 | 0.196642 | 0.054623 | 1.732788 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  380.644769 | 88.893765 | 4.967356 | 0.588357 |  0.0 |  0.0 | 0.0 |   0.0 | 0.054623 |
    | 2010-12-11 23:00:00 |  0.1 |  3.8 |             85.6 |       6.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 | 0.094 |  3.8 |     5.45082 |           2.413935 |                 8.023017 |                         0.0 |                          0.0 |             6.867703 |       0.0 |      -22.8 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 |  0.12638 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.944487 | 0.267617 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.128311 | 0.058476 |   0.0 |   0.0 |  0.0 | 0.186787 | 0.051885 | 1.700408 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  412.028103 | 88.893765 | 4.839045 | 0.529881 |  0.0 |  0.0 | 0.0 |   0.0 | 0.051885 |
    | 2010-12-12 00:00:00 |  0.0 |  3.9 |             85.3 |       6.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.9 |    5.130184 |           2.271939 |                  8.07986 |                         0.0 |                          0.0 |             6.892121 |       0.0 |      -23.4 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.121939 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.940178 |  0.26518 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.124997 | 0.052664 |   0.0 |   0.0 |  0.0 | 0.177661 |  0.04935 | 1.578469 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  444.011436 | 88.893765 | 4.714048 | 0.477217 |  0.0 |  0.0 | 0.0 |   0.0 |  0.04935 |
    | 2010-12-12 01:00:00 |  0.0 |  3.7 |             86.2 |       6.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.7 |    5.050025 |            2.23644 |                 7.966527 |                         0.0 |                          0.0 |             6.867146 |       0.0 |      -22.2 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 |   0.1116 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.936031 | 0.262846 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.121768 |  0.04743 |   0.0 |   0.0 |  0.0 | 0.169198 | 0.046999 | 1.466869 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  474.794769 | 88.893765 |  4.59228 | 0.429787 |  0.0 |  0.0 | 0.0 |   0.0 | 0.046999 |
    | 2010-12-12 02:00:00 |  0.1 |  3.5 |             89.3 |       5.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 | 0.094 |  3.5 |     4.56907 |           2.023445 |                 7.854597 |                         0.0 |                          0.0 |             7.014155 |       0.0 |      -21.0 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.076985 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.932045 | 0.260612 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.118623 | 0.042716 |   0.0 |   0.0 |  0.0 | 0.161338 | 0.044816 | 1.483884 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  504.378103 | 88.893765 | 4.473658 | 0.387071 |  0.0 |  0.0 | 0.0 |   0.0 | 0.044816 |
    | 2010-12-12 03:00:00 |  0.0 |  3.3 |             91.4 |       5.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.3 |    4.168274 |            1.84595 |                 7.744054 |                         0.0 |                          0.0 |             7.078066 |       0.0 |      -19.8 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.055407 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.928221 | 0.258478 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.115558 |  0.03847 |   0.0 |   0.0 |  0.0 | 0.154029 | 0.042786 | 1.428477 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  532.761436 | 88.893765 | 4.358099 | 0.348601 |  0.0 |  0.0 | 0.0 |   0.0 | 0.042786 |
    | 2010-12-12 04:00:00 |  0.0 |  3.3 |             90.9 |       4.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.3 |     3.68732 |           1.632956 |                 7.744054 |                         0.0 |                          0.0 |             7.039346 |       0.0 |      -19.8 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.051904 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.924397 | 0.256353 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.112573 | 0.034647 |   0.0 |   0.0 |  0.0 |  0.14722 | 0.040895 | 1.376573 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  561.144769 | 88.893765 | 4.245526 | 0.313954 |  0.0 |  0.0 | 0.0 |   0.0 | 0.040895 |
    | 2010-12-12 05:00:00 |  0.0 |  3.0 |             89.2 |       4.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.0 |    3.927797 |           1.739453 |                 7.580812 |                         0.0 |                          0.0 |             6.762084 |       0.0 |      -18.0 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.065482 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.920815 |  0.25437 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.109666 | 0.031203 |   0.0 |   0.0 |  0.0 | 0.140869 |  0.03913 | 1.311091 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  587.728103 | 88.893765 |  4.13586 |  0.28275 |  0.0 |  0.0 | 0.0 |   0.0 |  0.03913 |
    | 2010-12-12 06:00:00 |  0.0 |  1.9 |             85.6 |       6.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  1.9 |    5.050025 |            2.23644 |                 7.007901 |                         0.0 |                          0.0 |             5.998763 |       0.0 |      -11.4 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.108634 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.918123 | 0.252885 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.106833 | 0.028102 |   0.0 |   0.0 |  0.0 | 0.134935 | 0.037482 | 1.202457 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  607.711436 | 88.893765 | 4.029028 | 0.254648 |  0.0 |  0.0 | 0.0 |   0.0 | 0.037482 |
    | 2010-12-12 07:00:00 |  0.0 |  1.2 |             86.3 |       5.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  1.2 |    4.649229 |           2.058944 |                 6.663527 |                         0.0 |                          0.0 |             5.750624 |       0.0 |       -7.2 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.092336 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.915996 | 0.251715 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.104073 | 0.025309 |   0.0 |   0.0 |  0.0 | 0.129382 |  0.03594 | 1.110122 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  623.494769 | 88.893765 | 3.924954 | 0.229339 |  0.0 |  0.0 | 0.0 |   0.0 |  0.03594 |
    | 2010-12-12 08:00:00 |  0.0 |  0.8 |             91.2 |       6.1 |                 0.339738 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |        0.894145 |   0.0 |  0.8 |    4.889707 |           2.165441 |                 6.473522 |                         0.0 |                          0.0 |             5.903852 |       0.0 |       -4.8 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.060807 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.914193 | 0.250725 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.101385 | 0.022794 |   0.0 |   0.0 |  0.0 | 0.124179 | 0.034494 | 1.049315 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  636.878103 | 88.893765 | 3.823569 | 0.206545 |  0.0 |  0.0 | 0.0 |   0.0 | 0.034494 |
    | 2010-12-12 09:00:00 |  0.0 |  0.8 |             87.5 |       5.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |        17.71254 |   0.0 |  0.8 |    4.729388 |           2.094443 |                 6.473522 |                         0.0 |                          0.0 |             5.664332 |       0.0 |       -4.8 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.093283 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 |  0.91239 | 0.249737 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.098766 | 0.020528 |   0.0 |   0.0 |  0.0 | 0.119294 | 0.033137 | 0.956032 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  650.261436 | 88.893765 | 3.724803 | 0.186017 |  0.0 |  0.0 | 0.0 |   0.0 | 0.033137 |
    | 2010-12-12 10:00:00 |  0.1 |  0.5 |             84.5 |       6.4 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       54.302585 | 0.094 |  0.5 |    5.130184 |           2.271939 |                 6.334173 |                         0.0 |                          0.0 |             5.352376 |       0.0 |       -3.0 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.140469 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.910829 | 0.248883 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.096215 | 0.018488 |   0.0 |   0.0 |  0.0 | 0.114703 | 0.031862 | 0.909563 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  661.844769 | 88.893765 | 3.628588 | 0.167529 |  0.0 |  0.0 | 0.0 |   0.0 | 0.031862 |
    | 2010-12-12 11:00:00 |  0.0 |  0.3 |             76.7 |       6.8 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       81.504146 |   0.0 |  0.3 |     5.45082 |           2.413935 |                  6.24275 |                         0.0 |                          0.0 |             4.788189 |       0.0 |       -1.8 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.221518 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 |  0.90943 | 0.248119 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |  0.09373 |  0.01665 |   0.0 |   0.0 |  0.0 |  0.11038 | 0.030661 | 0.688045 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  672.228103 | 88.893765 | 3.534859 | 0.150879 |  0.0 |  0.0 | 0.0 |   0.0 | 0.030661 |
    | 2010-12-12 12:00:00 |  0.0 | -0.1 |             74.8 |       7.0 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       81.665294 |   0.0 | -0.1 |    5.611139 |           2.484933 |                 6.063393 |                         0.0 |                          0.0 |             4.535418 |       0.0 |        0.6 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.238233 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.908355 | 0.247533 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.091308 | 0.014996 |   0.0 |   0.0 |  0.0 | 0.106304 | 0.029529 | 0.449813 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  680.211436 | 88.893765 |  3.44355 | 0.135883 |  0.0 |  0.0 | 0.0 |   0.0 | 0.029529 |
    | 2010-12-12 13:00:00 |  0.0 | -0.4 |             69.7 |       6.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       77.078959 |   0.0 | -0.4 |    5.050025 |            2.23644 |                 5.931876 |                         0.0 |                          0.0 |             4.134518 |       0.0 |        2.4 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.249511 |       0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.907522 | 0.247079 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |  0.08895 | 0.013505 |   0.0 |   0.0 |  0.0 | 0.102455 |  0.02846 | 0.200302 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  686.394769 | 88.893765 |   3.3546 | 0.122378 |  0.0 |  0.0 | 0.0 |   0.0 |  0.02846 |
    | 2010-12-12 14:00:00 |  0.0 | -0.6 |             66.9 |       7.3 |                      1.0 |              0.2 | 0.0 | 0.0 |                   0.3 |                      7.393446 |       60.254042 |   0.0 | -0.6 |    5.851616 |            2.59143 |                 5.845604 |                         0.0 |                          0.0 |             3.910709 |       0.0 |        3.6 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 | 0.200302 |  0.002791 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 |  0.90685 | 0.246713 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.086652 | 0.012163 |   0.0 |   0.0 |  0.0 | 0.098815 | 0.027449 |      0.0 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  691.378103 | 88.890973 | 3.267948 | 0.110215 |  0.0 |  0.0 | 0.0 |   0.0 | 0.027449 |
    | 2010-12-12 15:00:00 |  0.0 | -0.8 |             64.4 |       7.5 |                      1.0 |              0.2 | 0.0 | 0.0 |                   0.4 |                      7.393446 |       14.837301 |   0.0 | -0.8 |    6.011934 |           2.662428 |                 5.760442 |                         0.0 |                          0.0 |             3.709725 |       0.0 |        4.8 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       49.0 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 |      0.0 |  0.008175 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.906341 | 0.246436 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.084414 | 0.010954 |   0.0 |   0.0 |  0.0 | 0.095368 | 0.026491 |      0.0 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  695.161436 | 88.882798 | 3.183534 | 0.099261 |  0.0 |  0.0 | 0.0 |   0.0 | 0.026491 |
    | 2010-12-12 16:00:00 |  0.0 | -1.3 |             65.4 |       7.0 |                 0.046405 |              0.2 | 0.0 | 0.0 |                   0.6 |                      7.386143 |        0.168856 |   0.0 | -1.3 |    5.611139 |           2.484933 |                 5.552313 |                         0.0 |                          0.0 |             3.631213 |       0.0 |        7.8 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |     45.325 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 |      0.0 |  0.058495 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.906235 | 0.246379 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.082233 | 0.009865 |   0.0 |   0.0 |  0.0 | 0.092099 | 0.025583 |      0.0 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  695.944769 | 88.824303 | 3.101301 | 0.089395 |  0.0 |  0.0 | 0.0 |   0.0 | 0.025583 |
    | 2010-12-12 17:00:00 |  0.0 | -1.6 |             67.6 |       6.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -1.6 |    4.809547 |           2.129942 |                 5.430651 |                         0.0 |                          0.0 |              3.67112 |       0.0 |        9.6 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |      41.65 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 |      0.0 |  0.053184 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.906372 | 0.246453 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.080109 | 0.008885 |   0.0 |   0.0 |  0.0 | 0.088994 | 0.024721 |      0.0 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  694.928103 |  88.77112 | 3.021192 | 0.080511 |  0.0 |  0.0 | 0.0 |   0.0 | 0.024721 |
    | 2010-12-12 18:00:00 |  0.0 | -2.0 |             70.2 |       4.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.0 |    3.847638 |           1.703954 |                 5.272101 |                         0.0 |                          0.0 |             3.701015 |       0.0 |       12.0 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |      36.75 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 |      0.0 |  0.044429 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.906832 | 0.246703 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |  0.07804 | 0.008002 |   0.0 |   0.0 |  0.0 | 0.086042 | 0.023901 |      0.0 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  691.511436 |  88.72669 | 2.943152 | 0.072509 |  0.0 |  0.0 | 0.0 |   0.0 | 0.023901 |
    | 2010-12-12 19:00:00 |  0.0 | -2.1 |             70.4 |       4.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.1 |    3.366683 |            1.49096 |                 5.233109 |                         0.0 |                          0.0 |             3.684109 |       0.0 |       12.6 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |     35.525 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 |      0.0 |  0.041865 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.907373 | 0.246998 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.076024 | 0.007207 |   0.0 |   0.0 |  0.0 | 0.083231 |  0.02312 |      0.0 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  687.494769 | 88.684826 | 2.867128 | 0.065302 |  0.0 |  0.0 | 0.0 |   0.0 |  0.02312 |
    | 2010-12-12 20:00:00 |  0.0 | -2.2 |             69.5 |       4.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.2 |    3.927797 |           1.739453 |                 5.194372 |                         0.0 |                          0.0 |             3.610088 |       0.0 |       13.2 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       34.3 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 |      0.0 |  0.045237 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.907995 | 0.247337 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |  0.07406 |  0.00649 |   0.0 |   0.0 |  0.0 | 0.080551 | 0.022375 |      0.0 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  682.878103 | 88.639589 | 2.793067 | 0.058812 |  0.0 |  0.0 | 0.0 |   0.0 | 0.022375 |
    | 2010-12-12 21:00:00 |  0.0 | -2.6 |             71.6 |       4.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.6 |    3.206365 |           1.419962 |                 5.041946 |                         0.0 |                          0.0 |             3.610033 |       0.0 |       15.6 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       29.4 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 |      0.0 |  0.038133 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.908941 | 0.247852 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.072147 | 0.005845 |   0.0 |   0.0 |  0.0 | 0.077993 | 0.021665 |      0.0 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  675.861436 | 88.601455 |  2.72092 | 0.052967 |  0.0 |  0.0 | 0.0 |   0.0 | 0.021665 |
    | 2010-12-12 22:00:00 |  0.0 | -3.1 |             74.5 |       3.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -3.1 |     2.72541 |           1.206967 |                 4.856984 |                         0.0 |                          0.0 |             3.618453 |       0.0 |       18.6 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       24.5 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 |      0.0 |  0.030873 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 |  0.91029 | 0.248589 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.070284 | 0.005264 |   0.0 |   0.0 |  0.0 | 0.075548 | 0.020986 |      0.0 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  665.844769 | 88.570582 | 2.650636 | 0.047702 |  0.0 |  0.0 | 0.0 |   0.0 | 0.020986 |
    | 2010-12-12 23:00:00 |  0.0 | -3.9 |             80.2 |       3.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -3.9 |     2.72541 |           1.206967 |                  4.57351 |                         0.0 |                          0.0 |             3.667955 |       0.0 |       23.4 |                      nan |                       0.0 |                     0.0 |                      0.0 |             0.0 |              0.0 |   0.0 |    0.0 |       24.5 |        0.42 |      0.0 |      0.0 |      0.0 |             nan |      0.0 |        0.0 |      0.0 |        0.0 |      0.0 |      0.0 |      0.0 |  0.022339 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |        0.0 |        0.0 |        0.0 |        0.0 |         0.0 | 0.912287 |  0.24968 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.068468 | 0.004741 |   0.0 |   0.0 |  0.0 | 0.073209 | 0.020336 |      0.0 |      0.0 |      0.0 |        0.0 |      nan |      0.0 |       0.0 |        0.0 |      nan |  651.028103 | 88.548243 | 2.582168 | 0.042961 |  0.0 |  0.0 | 0.0 |   0.0 | 0.020336 |

>>> round_(model.check_waterbalance(conditions_nadelw_winter))
0.0

.. _lland_knauf_ic_acker_winter_ktschnee_hourly:

acre (snow surface temperature)
-------------------------------

.. integration-test::

    >>> lnk(ACKER)
    >>> ktschnee(5.0)
    >>> test("lland_knauf_ic_acker_winter_ktschnee_hourly",
    ...      axis1=(inputs.nied, fluxes.wada),
    ...      axis2=(states.waes, states.wats),
    ...      use_conditions=conditions_acker_winter_ktschnee)
    |                date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |  nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressureinz | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationinz | netshortwaveradiationsnow | netlongwaveradiationinz | netlongwaveradiationsnow | netradiationinz | netradiationsnow |     nbes |    sbes | snowintmax | snowintrate | nbesinz | sbesinz | wniedinz | actualalbedoinz | wadainz | schmpotinz | schminz | gefrpotinz | gefrinz | evsinz |      evi |      evb |       evs |    wnied | tempssurface | actualalbedo |  schmpot |     schm |  gefrpot |     gefr | wlatinz |   wlatsnow | wsensinz |  wsenssnow | wsurfinz |       wsurf |      sff |      fvg |     wada |      qdb | qib1 | qib2 | qbb | qkap |     qdgz |    qdgz1 |    qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 |    qdga2 | qiga1 | qiga2 | qbga |      qah |       qa |     inzp | stinz | sinz | esnowinz | asinz |      wats |      waes |      esnow |     taus |        ebdn |      bowa |     sdg1 |     sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-12-10 00:00:00 |  0.0 | -3.7 |             87.5 |       3.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -3.7 |     2.72541 |            2.72541 |                  4.64297 |                         0.0 |                     3.363108 |             4.062599 | -2.470836 |  -1.137119 |                      0.0 |                       0.0 |                     0.0 |                62.690184 |             0.0 |       -62.690184 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.010568 |      0.0 |    -7.904522 |      0.75329 |      0.0 |      0.0 | 0.315303 | 0.315303 |     0.0 |  -7.830631 |      0.0 |  -26.74352 |      0.0 |   28.116032 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 19.067371 | 20.422623 |        0.0 | 0.871765 | -196.182518 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 01:00:00 |  0.0 | -4.8 |             90.7 |       2.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -4.8 |    2.324615 |           2.324615 |                 4.272186 |                         0.0 |                     3.108746 |             3.874873 |  -2.35419 |     0.7592 |                      0.0 |                       0.0 |                     0.0 |                63.329844 |             0.0 |       -63.329844 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.010408 |      0.0 |    -8.906723 |     0.752636 |      0.0 |      0.0 | 0.338128 | 0.338128 |     0.0 |   -7.71192 |      0.0 | -23.487925 |      0.0 |   32.129999 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 19.415389 | 20.433031 |        0.0 | 0.879073 | -188.358384 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 02:00:00 |  0.0 | -5.6 |             89.7 |       2.5 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -5.6 |    2.003978 |           2.003978 |                 4.019239 |                         0.0 |                     2.893794 |             3.605258 | -2.260301 |   2.457374 |                      0.0 |                       0.0 |                     0.0 |                64.160092 |             0.0 |       -64.160092 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.008798 |      0.0 |    -9.812089 |     0.751986 |      0.0 |      0.0 | 0.358424 | 0.358424 |     0.0 |  -6.519287 |      0.0 | -21.929674 |      0.0 |   35.711131 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 19.782327 |  20.44183 |        0.0 | 0.886381 | -182.232425 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 03:00:00 |  0.0 | -6.2 |             91.3 |       3.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -6.2 |    2.404774 |           2.404774 |                 3.838328 |                         0.0 |                     2.841896 |             3.504393 | -2.186789 |   3.218369 |                      0.0 |                       0.0 |                     0.0 |                65.853061 |             0.0 |       -65.853061 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.009202 |      0.0 |   -10.039607 |     0.751342 |      0.0 |      0.0 | 0.359609 | 0.359609 |     0.0 |  -6.818315 |      0.0 | -22.452631 |      0.0 |   36.582115 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.151004 | 20.451032 |        0.0 | 0.893689 | -176.867461 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 04:00:00 |  0.0 | -7.3 |             94.2 |       2.5 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -7.3 |    2.003978 |           2.003978 |                  3.52531 |                         0.0 |                      2.62027 |             3.320842 |  -2.12241 |   4.970078 |                      0.0 |                       0.0 |                     0.0 |                66.487965 |             0.0 |       -66.487965 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.008664 |      0.0 |   -11.054742 |     0.750703 |      0.0 |      0.0 | 0.383172 | 0.300028 |     0.0 |  -6.419479 |      0.0 | -19.548556 |      0.0 |    40.51993 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |  0.0 |      0.0 |   nan | 20.459696 | 20.459696 |  -7.713912 | 0.900997 | -173.254206 |  71.44626 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 05:00:00 |  0.0 | -8.3 |             93.2 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -8.3 |     1.84366 |            1.84366 |                  3.26061 |                         0.0 |                     2.401909 |             3.038889 |  -2.07905 |   8.729642 |                      0.0 |                       0.0 |                     0.0 |                67.511852 |             0.0 |       -67.511852 |      0.0 |     0.0 |        0.0 |         0.0 |     0.0 |     0.0 |      0.0 |             nan |     0.0 |        0.0 |     0.0 |        0.0 |     0.0 |    0.0 |      0.0 |      0.0 | -0.007489 |      0.0 |   -12.132478 |     0.750191 |      0.0 |      0.0 | 0.452444 |      0.0 |     0.0 |  -5.549204 |      0.0 | -18.970213 |      0.0 |   42.992436 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |   0.0 |