# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, unused-wildcard-import
"""
The |musk.DOCNAME.long| family member |musk_mct| realises a modification of the
variable coefficient Muskingum approach of :cite:t:`ref-Cunge1969` (*MC*) proposed by
:cite:t:`ref-Todini2007` (*MCT*).

.. _`Main`: https://de.wikipedia.org/wiki/Main

Opposed to the original Muskingum method of :cite:p:`ref-McCarthy1940`, *MC* does not
rely on predetermined coefficients.  Instead, it calculates the coefficients for each
simulation step based on a water depth-discharge relationship.  Essentially, *MC* is a
kinematic wave approach that does not explicitly account for pressure gradients or
other forcings than the channel's bottom slope explicitly.  However, it adjusts its
coefficients so that the numerical error of the underlying finite difference
discretisation (the Muskingum working formula) mimics the physical diffusion of flood
waves due to pressure gradients as well as possible.   Hence, it can simulate flood
waves even in channels with mild bottom slopes without much calibration (provided that
information on the rating curve is available).  Compared to more complete solutions of
the Saint-Vernant equations, it is less computationally demanding.  On the downside, it
cannot deal with cases where, for example, effects or rapid changes in channel geometry
play a significant role.

|musk_mct| implements the Muskingum-Cunge modification of :cite:t:`ref-Todini2007`,
which solves a "mass conservation inconsistency" and a "steady-state inconsistency" of
the original formulation.  :cite:t:`ref-Todini2007` evaluates the method for simple
rectangular, triangular, and trapezoidal cross-section profiles in combination with the
Manning-Strickler equation.  |musk_mct| also allows to define more complex profiles by
selecting and configuring submodels that follow the |CrossSectionModel_V1| interface.
However, be aware that the achieved performance or accuracy might be worse than
expected if the underlying expectations are violated.

When applying |musk_mct|, one should be aware that it relies on two nested iterations.
First, it uses a root search algorithm to find the current "reference water depth" that
results in the previously determined "reference discharge".  While
:cite:t:`ref-Todini2007` uses the Newton-Raphson approach, |musk_mct| applies the
Pegasus iteration :cite:p:`ref-Dowell1972`.  By default, |musk_mct| strives for very
high accuracies.  There is a good chance you can increase the computation speed
significantly in your use case without reducing numerical precision too much (see the
documentation on method |Calc_ReferenceWaterDepth_V1| for further information).  The
second iteration addresses the "reference discharge" itself, which's initial guess can
be quite inaccurate (see |Calc_ReferenceDischarge_V1|).  :cite:t:`ref-Todini2007`
suggests refining it by starting subsequent runs of the whole *MCT* procedure
(including the Newton-Raphson or Pegasus approach described above).  |musk_mct| follows
Todini's suggestion to run everything twice.  You can deviate from this default
behaviour by changing the value of parameter |musk_mct.NmbRuns|.


Integration tests
=================

.. how_to_understand_integration_tests::

All of the following tests are equal to or similar to the examples provided by
:cite:t:`ref-Todini2007`.

The simulation period covers four days; the simulation step is 30 minutes:

>>> from hydpy import pub, Nodes, Element
>>> pub.timegrids = "2000-01-01", "2000-01-05", "30m"

The model shall retrieve its inflow from two |Node| objects (`input1` and `input2`)
and passe its outflow to node `output`.  We define these nodes and connect them to an
|Element| object that handles a |musk_mct| instance:

>>> nodes = Nodes("input1", "input2", "output")
>>> stream = Element("stream", inlets=["input1", "input2"], outlets="output")
>>> from hydpy.models.musk_mct import *
>>> parameterstep("1h")
>>> stream.model = model

The last general preparation step is to define a test function object for controlling
the following test runs:

>>> from hydpy.core.testtools import IntegrationTest
>>> test = IntegrationTest(stream)
>>> test.plotting_options.axis1 = (states.discharge,)
>>> test.dateformat = "%H:%M"

.. _musk_mct_base_example:

Base example
_____________

This example repeats the base example of :cite:t:`ref-Todini2007` dealing with a
trapezoidal profile.  A channel with a total length of 100 km is subdivided into 50
segments:

>>> nmbsegments(50)
>>> length(100.0)
>>> bottomslope(0.00025)

We use the |wq_trapeze_strickler| submodel to configure the shape of a simple trapeze:

>>> with model.add_wqmodel_v1("wq_trapeze_strickler"):
...     nmbtrapezes(1)
...     bottomlevels(0.0)
...     bottomwidths(15.0)
...     sideslopes(5.0)
...     stricklercoefficients(1.0/0.035)

|musk_mct| uses the catchment area for calculating the value of the solver parameter
|ToleranceDischarge|, which determines the accuracy of the iterative estimation of the
reference water depth (see methods |ToleranceDischarge.modify_init| and
|Calc_ReferenceWaterDepth_V1|).  We set it 25 000 km², which is approximately the
catchment size of the river `Main`_ at gauge Frankfurt-Osthafen:

>>> catchmentarea(25000.0)

This setting makes up for an error tolerance of 0.025 m³/s, which is 360 times smaller
than the lowest discharge reported for Frankfurt-Ostenhagen (9 m³/s, 1976):

>>> solver.tolerancedischarge.update()
>>> solver.tolerancedischarge
tolerancedischarge(0.025)

Besides "old" discarge values, |musk_mct| requires the Courant number and the cell
Reynolds number as initial conditions. These states must be consistent if you need good
simulation results right from the start.  If there are none available from a previous
simulation run, you can use |Model.prepare_states| to calculate them for definable
discharge values under the assumption of stationarity:

>>> model.prepare_states(discharge=100.0)
>>> test.inits = ((states.discharge, 100.0),
...               (states.courantnumber, states.courantnumber[0]),
...               (states.reynoldsnumber, states.reynoldsnumber[0]))

Node `input1` provides a constant baseflow of 100 m³/s, and node `input2` provides a
synthetic flood wave with a peak discharge of 800 m³/s, making a total peak discharge
of 900 m³/s (which is similar to the average flood discharge at Frankfurt-Ostenhagen)
that occurs 24 hours after the start of the simulation period:

>>> import numpy
>>> q_base, q_peak, t_peak, beta = 100.0, 900.0, 24.0, 16.0
>>> ts = pub.timegrids.init.to_timepoints()
>>> nodes.input1.sequences.sim.series = q_base
>>> nodes.input2.sequences.sim.series = (
...     (q_peak - q_base) * ((ts / t_peak) * numpy.exp(1.0 - ts / t_peak)) ** beta)

The following table and figure show the discharge calculated for all 51 channel segment
endpoints, including the start and endpoint of the complete channel.  Other properties
like the reference water depth, the wetted perimeter, or the Courant number represent
estimated averages for all 50 segments:

.. integration-test::

    >>> conditions = test("musk_mct_base_example", get_conditions="2000-01-01")
    |  date |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           referencewaterdepth |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             wettedarea |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             surfacewidth |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           celerity |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           correctingfactor |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           coefficient1 |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            coefficient2 |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           coefficient3 |     inflow |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             referencedischarge |    outflow |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           courantnumber |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           reynoldsnumber |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          discharge | input1 |     input2 |     output |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 00:00 | 3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 |   0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 00:30 | 3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 |   0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 01:00 | 3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 |   0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 01:30 | 3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 |   0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 02:00 | 3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 |   0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 02:30 | 3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |  52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 |   0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |        0.0 |      100.0 |
    | 03:00 | 3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 124.878655  124.878654  124.878653  124.878653  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |  52.178282   52.178282   52.178282   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 100.000003 | 100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 |   0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 | 100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |   0.000003 |      100.0 |
    | 03:30 | 3.717829  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 124.878674  124.878666  124.878661  124.878658  124.878656  124.878655  124.878654  124.878653  124.878653  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |  52.178286   52.178284   52.178283   52.178282   52.178282   52.178282   52.178282   52.178282   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 100.000029 | 100.000024  100.000016   100.00001  100.000007  100.000004  100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 |   0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.591117  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 | 100.000029  100.000019  100.000012  100.000008  100.000005  100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |   0.000029 |      100.0 |
    | 04:00 | 3.717831   3.71783  3.717829  3.717829  3.717829  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 124.878787  124.878741  124.878711  124.878691  124.878678  124.878669  124.878663  124.878659  124.878657  124.878655  124.878654  124.878653  124.878653  124.878653  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |  52.178307   52.178298   52.178293   52.178289   52.178286   52.178285   52.178283   52.178283   52.178282   52.178282   52.178282   52.178282   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.536159  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 100.000175 | 100.000147  100.000097  100.000064  100.000042  100.000028  100.000018  100.000012  100.000008  100.000005  100.000003  100.000002  100.000002  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 |   0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.591118  2.591117  2.591117  2.591117  2.591117  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 | 100.000175  100.000116  100.000076   100.00005  100.000033  100.000022  100.000014   100.00001  100.000006  100.000004  100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |   0.000175 |      100.0 |
    | 04:30 |  3.71784  3.717836  3.717834  3.717832  3.717831   3.71783  3.717829  3.717829  3.717829  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 124.879291  124.879078  124.878937  124.878842  124.878779  124.878736  124.878708  124.878689  124.878677  124.878669  124.878663  124.878659  124.878657  124.878655  124.878654  124.878653  124.878653  124.878653  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |  52.178404   52.178363   52.178336   52.178318   52.178306   52.178297   52.178292   52.178288   52.178286   52.178284   52.178283   52.178283   52.178282   52.178282   52.178282   52.178282   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.088387  1.088386  1.088386  1.088386  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 |  1.35916   1.35916  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.536159  0.536159  0.536159  0.536159  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 100.000826 | 100.000695  100.000464   100.00031  100.000207  100.000138  100.000092  100.000061  100.000041  100.000027  100.000018  100.000012  100.000008  100.000005  100.000004  100.000002  100.000002  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 | 0.720701  0.720701    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.591124  2.591121   2.59112  2.591119  2.591118  2.591117  2.591117  2.591117  2.591117  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 | 100.000826  100.000552  100.000368  100.000246  100.000164  100.000109  100.000073  100.000048  100.000032  100.000021  100.000014  100.000009  100.000006  100.000004  100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |   0.000826 |      100.0 |
    | 05:00 | 3.717876   3.71786   3.71785  3.717843  3.717838  3.717835  3.717833  3.717831   3.71783  3.717829  3.717829  3.717829  3.717829  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 124.881131  124.880327  124.879783  124.879415  124.879166  124.878998  124.878885  124.878809  124.878758  124.878723    124.8787  124.878684  124.878673  124.878666  124.878662  124.878658  124.878656  124.878655  124.878654  124.878653  124.878653  124.878653  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |  52.178756   52.178602   52.178498   52.178428    52.17838   52.178348   52.178326   52.178311   52.178302   52.178295    52.17829   52.178287   52.178285   52.178284   52.178283   52.178283   52.178282   52.178282   52.178282   52.178282   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.088392   1.08839  1.088388  1.088387  1.088387  1.088386  1.088386  1.088386  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 |  1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.536162  0.536161   0.53616  0.536159  0.536159  0.536159  0.536159  0.536159  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.201869  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 100.003194 | 100.002698  100.001823  100.001231   100.00083   100.00056  100.000377  100.000254  100.000171  100.000115  100.000077  100.000052  100.000035  100.000023  100.000016   100.00001  100.000007  100.000005  100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 | 0.720705  0.720703  0.720702  0.720701  0.720701    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.591146  2.591136   2.59113  2.591125  2.591122   2.59112  2.591119  2.591118  2.591118  2.591117  2.591117  2.591117  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 | 100.003194  100.002158  100.001458  100.000984  100.000663  100.000447  100.000301  100.000203  100.000136  100.000092  100.000062  100.000041  100.000028  100.000019  100.000012  100.000008  100.000006  100.000004  100.000003  100.000002  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |   0.003194 |      100.0 |
    | 05:30 | 3.717985  3.717936  3.717902  3.717878  3.717862  3.717851  3.717844  3.717839  3.717835  3.717833  3.717832   3.71783   3.71783  3.717829  3.717829  3.717829  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 124.886847  124.884256   124.88248  124.881265  124.880434  124.879866  124.879478  124.879214  124.879034  124.878911  124.878828  124.878771  124.878733  124.878707  124.878689  124.878677  124.878669  124.878663   124.87866  124.878657  124.878655  124.878654  124.878654  124.878653  124.878653  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |  52.179852   52.179355   52.179015   52.178782   52.178623   52.178514    52.17844   52.178389   52.178355   52.178331   52.178315   52.178304   52.178297   52.178292   52.178288   52.178286   52.178285   52.178283   52.178283   52.178282   52.178282   52.178282   52.178282   52.178282   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.088409  1.088402  1.088396  1.088393   1.08839  1.088389  1.088387  1.088387  1.088386  1.088386  1.088386  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.536171  0.536167  0.536164  0.536162  0.536161   0.53616   0.53616  0.536159  0.536159  0.536159  0.536159  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.201871   -0.20187  -0.201869  -0.201869  -0.201869  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 100.010515 |  100.00892  100.006099  100.004167  100.002844  100.001939  100.001321    100.0009  100.000612  100.000416  100.000282  100.000192   100.00013  100.000088   100.00006   100.00004  100.000027  100.000018  100.000012  100.000008  100.000006  100.000004  100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 | 0.720717  0.720711  0.720708  0.720705  0.720703  0.720702  0.720701  0.720701    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.591214  2.591183  2.591162  2.591148  2.591138  2.591131  2.591126  2.591123  2.591121  2.591119  2.591118  2.591118  2.591117  2.591117  2.591117  2.591117  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 | 100.010515  100.007193  100.004915  100.003356  100.002289   100.00156  100.001062  100.000723  100.000491  100.000334  100.000227  100.000154  100.000104  100.000071  100.000048  100.000032  100.000022  100.000015   100.00001  100.000007  100.000005  100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |   0.010515 |      100.0 |
    | 06:00 | 3.718283  3.718143  3.718046  3.717978  3.717932    3.7179  3.717877  3.717862  3.717851  3.717844  3.717839  3.717836  3.717833  3.717832  3.717831   3.71783  3.717829  3.717829  3.717829  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 124.902381  124.895076  124.890007  124.886494  124.884062   124.88238  124.881218  124.880417  124.879865  124.879484  124.879223  124.879043   124.87892  124.878835  124.878777  124.878737   124.87871  124.878692  124.878679   124.87867  124.878665  124.878661  124.878658  124.878656  124.878655  124.878654  124.878653  124.878653  124.878653  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |  52.182829   52.181429   52.180458   52.179784   52.179318   52.178996   52.178773    52.17862   52.178514   52.178441   52.178391   52.178356   52.178333   52.178316   52.178305   52.178298   52.178292   52.178289   52.178286   52.178285   52.178284   52.178283   52.178282   52.178282   52.178282   52.178282   52.178282   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.088456  1.088434  1.088419  1.088408  1.088401  1.088396  1.088393   1.08839  1.088389  1.088388  1.088387  1.088386  1.088386  1.088386  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.359156  1.359157  1.359158  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.536194  0.536183  0.536175   0.53617  0.536166  0.536164  0.536162  0.536161   0.53616   0.53616  0.536159  0.536159  0.536159  0.536159  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 |  -0.20188  -0.201876  -0.201873  -0.201872   -0.20187   -0.20187  -0.201869  -0.201869  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665707  0.665707  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 100.030315 | 100.025827  100.017877  100.012359  100.008535  100.005888  100.004057  100.002793  100.001921   100.00132  100.000906  100.000621  100.000426  100.000291  100.000199  100.000136  100.000093  100.000064  100.000043   100.00003   100.00002  100.000014  100.000009  100.000006  100.000004  100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 | 0.720749  0.720734  0.720723  0.720716  0.720711  0.720707  0.720705  0.720703  0.720702  0.720701  0.720701    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.591401  2.591313  2.591252   2.59121  2.591181  2.591161  2.591147  2.591137  2.591131  2.591126  2.591123  2.591121  2.591119  2.591118  2.591118  2.591117  2.591117  2.591117  2.591117  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 | 100.030315  100.020992  100.014519  100.010031  100.006922  100.004772  100.003286  100.002261  100.001554  100.001067  100.000732  100.000502  100.000344  100.000235  100.000161   100.00011  100.000075  100.000051  100.000035  100.000024  100.000016  100.000011  100.000007  100.000005  100.000003  100.000002  100.000002  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |   0.030315 |      100.0 |
    | 06:30 | 3.719006  3.718653  3.718406  3.718232   3.71811  3.718024  3.717965  3.717923  3.717894  3.717874   3.71786   3.71785  3.717843  3.717839  3.717835  3.717833  3.717832  3.717831   3.71783  3.717829  3.717829  3.717829  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 124.940113  124.921719  124.908788  124.899711  124.893349  124.888896  124.885784  124.883611  124.882097  124.881042  124.880308  124.879799  124.879445    124.8792   124.87903  124.878913  124.878832  124.878776  124.878737  124.878711  124.878692   124.87868  124.878671  124.878665  124.878661  124.878658  124.878656  124.878655  124.878654  124.878653  124.878653  124.878653  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |  52.190059   52.186535   52.184057   52.182317   52.181098   52.180245   52.179648   52.179232   52.178941   52.178739   52.178599   52.178501   52.178433   52.178386   52.178354   52.178331   52.178316   52.178305   52.178298   52.178293   52.178289   52.178287   52.178285   52.178284   52.178283   52.178282   52.178282   52.178282   52.178282   52.178282   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.088568  1.088513  1.088475  1.088448  1.088429  1.088415  1.088406    1.0884  1.088395  1.088392   1.08839  1.088388  1.088387  1.088387  1.088386  1.088386  1.088386  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.359148  1.359152  1.359155  1.359156  1.359158  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.536251  0.536223  0.536204   0.53619  0.536181  0.536174  0.536169  0.536166  0.536164  0.536162  0.536161   0.53616   0.53616  0.536159  0.536159  0.536159  0.536159  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.201904  -0.201893  -0.201885   -0.20188  -0.201876  -0.201873  -0.201872   -0.20187   -0.20187  -0.201869  -0.201869  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665703  0.665705  0.665706  0.665707  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 100.078179 | 100.066899  100.046877  100.032801  100.022921  100.015997   100.01115  100.007762  100.005398  100.003749  100.002601  100.001803  100.001248  100.000863  100.000597  100.000412  100.000284  100.000196  100.000135  100.000093  100.000064  100.000044   100.00003  100.000021  100.000014   100.00001  100.000007  100.000005  100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 | 0.720827  0.720789  0.720762  0.720743   0.72073  0.720721  0.720714   0.72071  0.720707  0.720705  0.720703  0.720702  0.720701  0.720701    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.591853  2.591632  2.591477  2.591369  2.591292  2.591239  2.591202  2.591176  2.591158  2.591145  2.591136   2.59113  2.591126  2.591123  2.591121  2.591119  2.591118  2.591118  2.591117  2.591117  2.591117  2.591117  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 | 100.078179  100.054809   100.03837  100.026825   100.01873  100.013061  100.009096  100.006328  100.004397  100.003052  100.002116  100.001466  100.001014  100.000701  100.000484  100.000334   100.00023  100.000159  100.000109  100.000075  100.000052  100.000035  100.000024  100.000017  100.000011  100.000008  100.000005  100.000004  100.000002  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |   0.078179 |      100.0 |
    | 07:00 | 3.720602  3.719796  3.719222  3.718814  3.718524  3.718319  3.718174  3.718071  3.717999  3.717948  3.717912  3.717887  3.717869  3.717857  3.717848  3.717842  3.717838  3.717835  3.717833  3.717831   3.71783   3.71783  3.717829  3.717829  3.717829  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 125.023431   124.98136  124.951394  124.930089  124.914969  124.904256  124.896678  124.891326  124.887551  124.884892  124.883023   124.88171  124.880789  124.880143  124.879692  124.879376  124.879156  124.879002  124.878895  124.878821  124.878769  124.878733  124.878708  124.878691  124.878679   124.87867  124.878665  124.878661  124.878658  124.878656  124.878655  124.878654  124.878653  124.878653  124.878653  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |  52.206021   52.197962    52.19222   52.188138   52.185241   52.183188   52.181736    52.18071   52.179987   52.179477   52.179119   52.178867   52.178691   52.178567   52.178481    52.17842   52.178378   52.178348   52.178328   52.178314   52.178304   52.178297   52.178292   52.178289   52.178286   52.178285   52.178284   52.178283   52.178282   52.178282   52.178282   52.178282   52.178282   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.088815   1.08869  1.088601  1.088538  1.088493  1.088461  1.088439  1.088423  1.088411  1.088404  1.088398  1.088394  1.088391  1.088389  1.088388  1.088387  1.088387  1.088386  1.088386  1.088386  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.359132   1.35914  1.359146   1.35915  1.359153  1.359156  1.359157  1.359158  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.536377  0.536314  0.536268  0.536236  0.536213  0.536197  0.536186  0.536177  0.536172  0.536168  0.536165  0.536163  0.536162  0.536161   0.53616  0.536159  0.536159  0.536159  0.536159  0.536159  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.201962  -0.201934  -0.201914    -0.2019   -0.20189  -0.201883  -0.201879  -0.201875  -0.201873  -0.201871   -0.20187  -0.201869  -0.201869  -0.201869  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665696    0.6657  0.665702  0.665704  0.665706  0.665707  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 100.183351 | 100.157608  100.111803   100.07918  100.055988  100.039529  100.027868   100.01962  100.013794  100.009686  100.006792  100.004757  100.003328  100.002325  100.001623  100.001132  100.000788  100.000548  100.000381  100.000265  100.000184  100.000127  100.000088  100.000061  100.000042  100.000029   100.00002  100.000014   100.00001  100.000007  100.000005  100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 |    0.721  0.720912   0.72085  0.720806  0.720775  0.720753  0.720737  0.720726  0.720718  0.720713  0.720709  0.720706  0.720704  0.720703  0.720702  0.720701  0.720701    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.592851  2.592347  2.591988  2.591733  2.591551  2.591423  2.591332  2.591268  2.591223  2.591191  2.591169  2.591153  2.591142  2.591134  2.591129  2.591125  2.591122   2.59112  2.591119  2.591118  2.591118  2.591117  2.591117  2.591117  2.591117  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 | 100.183351  100.130144  100.092223  100.065247  100.046091  100.032511    100.0229  100.016108  100.011315  100.007939  100.005563  100.003893  100.002721    100.0019  100.001325  100.000923  100.000643  100.000447   100.00031  100.000215  100.000149  100.000103  100.000072   100.00005  100.000034  100.000024  100.000016  100.000011  100.000008  100.000005  100.000004  100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |   0.183351 |      100.0 |
    | 07:30 | 3.723847  3.722152  3.720928  3.720046  3.719413  3.718958  3.718633    3.7184  3.718234  3.718116  3.718032  3.717972   3.71793    3.7179  3.717879  3.717864  3.717853  3.717846   3.71784  3.717837  3.717834  3.717832  3.717831   3.71783   3.71783  3.717829  3.717829  3.717829  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 125.192889  125.104349  125.040441  124.994416  124.961338  124.937613  124.920628   124.90849   124.89983  124.893661  124.889275   124.88616  124.883952  124.882388  124.881283  124.880502  124.879952  124.879564  124.879291    124.8791  124.878965  124.878871  124.878805  124.878758  124.878726  124.878704  124.878688  124.878677  124.878669  124.878664   124.87866  124.878658  124.878656  124.878655  124.878654  124.878653  124.878653  124.878653  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |   52.23847   52.221518   52.209279   52.200463   52.194126    52.18958   52.186325   52.183999    52.18234   52.181158   52.180317    52.17972   52.179297   52.178997   52.178786   52.178636    52.17853   52.178456   52.178404   52.178367   52.178341   52.178323   52.178311   52.178302   52.178296   52.178291   52.178288   52.178286   52.178285   52.178284   52.178283   52.178282   52.178282   52.178282   52.178282   52.178282   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.089318  1.089055  1.088866  1.088729  1.088631   1.08856   1.08851  1.088474  1.088448   1.08843  1.088417  1.088407  1.088401  1.088396  1.088393  1.088391  1.088389  1.088388  1.088387  1.088386  1.088386  1.088386  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.359099  1.359116  1.359129  1.359138  1.359144  1.359149  1.359152  1.359155  1.359156  1.359158  1.359158  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.536633  0.536499  0.536403  0.536333  0.536283  0.536247  0.536222  0.536203   0.53619  0.536181  0.536174   0.53617  0.536166  0.536164  0.536162  0.536161   0.53616   0.53616  0.536159  0.536159  0.536159  0.536159  0.536159  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.202092  -0.202027   -0.20198  -0.201947  -0.201924  -0.201907  -0.201895  -0.201887  -0.201881  -0.201877  -0.201874  -0.201872  -0.201871   -0.20187  -0.201869  -0.201869  -0.201869  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665685  0.665691  0.665696    0.6657  0.665702  0.665704  0.665706  0.665707  0.665707  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 100.396214 |  100.34216  100.245722  100.176129  100.126016  100.090006  100.064178  100.045689  100.032476   100.02305  100.016337  100.011562  100.008172  100.005769  100.004067  100.002863  100.002014  100.001415  100.000993  100.000696  100.000487  100.000341  100.000238  100.000166  100.000116  100.000081  100.000056  100.000039  100.000027  100.000019  100.000013  100.000009  100.000006  100.000004  100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 |  0.72135  0.721167  0.721035   0.72094  0.720871  0.720822  0.720787  0.720761  0.720744  0.720731  0.720722  0.720715  0.720711  0.720707  0.720705  0.720703  0.720702  0.720702  0.720701  0.720701    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.594879   2.59382  2.593054  2.592503  2.592107  2.591823  2.591619  2.591474   2.59137  2.591296  2.591244  2.591206   2.59118  2.591161  2.591148  2.591138  2.591132  2.591127  2.591124  2.591122   2.59112  2.591119  2.591118  2.591118  2.591117  2.591117  2.591117  2.591117  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 | 100.396214  100.284747  100.204241  100.146226  100.104505  100.074562  100.053112  100.037773  100.026824  100.019021  100.013469  100.009524  100.006726  100.004744  100.003342  100.002351  100.001652   100.00116  100.000813   100.00057  100.000399  100.000279  100.000195  100.000136  100.000095  100.000066  100.000046  100.000032  100.000022  100.000015  100.000011  100.000007  100.000005  100.000004  100.000002  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |   0.396214 |      100.0 |
    | 08:00 | 3.729985  3.726671  3.724246  3.722476  3.721187   3.72025  3.719572  3.719081  3.718727  3.718472  3.718289  3.718157  3.718063  3.717995  3.717947  3.717912  3.717888   3.71787  3.717858  3.717849  3.717843  3.717839  3.717836  3.717833  3.717832  3.717831   3.71783  3.717829  3.717829  3.717829  3.717829  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 125.513703   125.34047   125.21373  125.121253  125.053945  125.005068  124.969654  124.944045  124.925565  124.912252   124.90268  124.895809  124.890885  124.887362  124.884845  124.883049   124.88177  124.880861  124.880214  124.879756  124.879431  124.879201  124.879039  124.878924  124.878843  124.878786  124.878746  124.878718  124.878698  124.878684  124.878675  124.878668  124.878663   124.87866  124.878657  124.878656  124.878655  124.878654  124.878653  124.878653  124.878653  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |  52.299848   52.266714    52.24246   52.224755   52.211865   52.202503   52.195719   52.190813   52.187271    52.18472   52.182886   52.181569   52.180626   52.179951   52.179468   52.179124   52.178879   52.178705   52.178581   52.178493   52.178431   52.178387   52.178355   52.178333   52.178318   52.178307   52.178299   52.178294    52.17829   52.178287   52.178286   52.178284   52.178283   52.178283   52.178282   52.178282   52.178282   52.178282   52.178282   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.090269  1.089756   1.08938  1.089106  1.088906  1.088761  1.088655  1.088579  1.088524  1.088485  1.088456  1.088436  1.088421  1.088411  1.088403  1.088398  1.088394  1.088392   1.08839  1.088388  1.088387  1.088387  1.088386  1.088386  1.088386  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.359036   1.35907  1.359095  1.359113  1.359126  1.359136  1.359143  1.359148  1.359151  1.359154  1.359156  1.359157  1.359158  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.537115  0.536855  0.536664  0.536525  0.536423  0.536349  0.536296  0.536257  0.536229  0.536209  0.536195  0.536184  0.536177  0.536171  0.536168  0.536165  0.536163  0.536162  0.536161   0.53616  0.536159  0.536159  0.536159  0.536159  0.536159  0.536159  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.202358  -0.202219   -0.20212  -0.202048  -0.201996  -0.201959  -0.201933  -0.201914    -0.2019  -0.201891  -0.201884  -0.201879  -0.201876  -0.201873  -0.201872  -0.201871   -0.20187  -0.201869  -0.201869  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665668  0.665678  0.665686  0.665692  0.665696    0.6657  0.665702  0.665704  0.665706  0.665707  0.665707  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 100.797305 | 100.691784  100.502956  100.364863  100.264133  100.190833  100.137614  100.099058   100.07118  100.051063  100.036572  100.026153  100.018674  100.013315   100.00948   100.00674  100.004786  100.003394  100.002404    100.0017  100.001202  100.000848  100.000598  100.000421  100.000296  100.000208  100.000146  100.000102  100.000072   100.00005  100.000035  100.000025  100.000017  100.000012  100.000008  100.000006  100.000004  100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 | 0.722014  0.721656  0.721394  0.721202  0.721063  0.720962  0.720888  0.720835  0.720797  0.720769  0.720749  0.720735  0.720725  0.720718  0.720712  0.720709  0.720706  0.720704  0.720703  0.720702  0.720701  0.720701    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.598716  2.596645  2.595129  2.594022  2.593216  2.592631  2.592207    2.5919  2.591678  2.591519  2.591404  2.591322  2.591263  2.591221   2.59119  2.591169  2.591154  2.591143  2.591135  2.591129  2.591126  2.591123  2.591121  2.591119  2.591119  2.591118  2.591117  2.591117  2.591117  2.591117  2.591117  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 | 100.797305  100.580172  100.421218  100.305161  100.220635  100.159215  100.114682  100.082459   100.05919  100.042418  100.030351  100.021683  100.015468  100.011019  100.007838  100.005568  100.003951  100.002799  100.001981    100.0014  100.000989  100.000697  100.000491  100.000346  100.000243  100.000171   100.00012  100.000084  100.000059  100.000041  100.000029   100.00002  100.000014   100.00001  100.000007  100.000005  100.000003  100.000002  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |   0.797305 |      100.0 |
    | 08:30 | 3.740875   3.73481  3.730305  3.726972  3.724513  3.722704  3.721378  3.720407  3.719698  3.719181  3.718806  3.718533  3.718336  3.718193   3.71809  3.718016  3.717963  3.717924  3.717897  3.717877  3.717863  3.717853  3.717846  3.717841  3.717837  3.717834  3.717833  3.717831   3.71783   3.71783  3.717829  3.717829  3.717829  3.717829  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 |  126.08386   125.76616  125.530461  125.356172  125.227676  125.133201  125.063916  125.013225   124.97622  124.949264  124.929667  124.915447  124.905148  124.897702  124.892328  124.888455   124.88567  124.883668  124.882233  124.881205   124.88047  124.879945   124.87957  124.879304  124.879114  124.878979  124.878883  124.878815  124.878767  124.878733  124.878709  124.878692   124.87868  124.878672  124.878666  124.878662  124.878659  124.878657  124.878655  124.878654  124.878654  124.878653  124.878653  124.878653  124.878652  124.878652  124.878652  124.878652  124.878652  124.878652 |  52.408751   52.348096   52.303052   52.269718   52.245129   52.227043   52.213775   52.204066   52.196977   52.191812   52.188057   52.185333   52.183359   52.181932   52.180902    52.18016   52.179626   52.179243   52.178968   52.178771    52.17863   52.178529   52.178457   52.178406    52.17837   52.178344   52.178326   52.178313   52.178303   52.178297   52.178292   52.178289   52.178287   52.178285   52.178284   52.178283   52.178283   52.178282   52.178282   52.178282   52.178282   52.178282   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.091956  1.091017  1.090319  1.089803  1.089421  1.089141  1.088935  1.088785  1.088675  1.088595  1.088537  1.088494  1.088464  1.088442  1.088426  1.088414  1.088406    1.0884  1.088396  1.088393   1.08839  1.088389  1.088388  1.088387  1.088386  1.088386  1.088386  1.088386  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.358925  1.358987  1.359033  1.359067  1.359092   1.35911  1.359124  1.359134  1.359141  1.359147  1.359151  1.359153  1.359155  1.359157  1.359158  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.537969  0.537494   0.53714  0.536878  0.536685  0.536543  0.536438  0.536362  0.536306  0.536265  0.536235  0.536214  0.536198  0.536187  0.536179  0.536173  0.536169  0.536166  0.536164  0.536162  0.536161   0.53616   0.53616  0.536159  0.536159  0.536159  0.536159  0.536159  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.202862   -0.20259  -0.202392  -0.202247  -0.202142  -0.202065   -0.20201   -0.20197  -0.201941   -0.20192  -0.201905  -0.201894  -0.201887  -0.201881  -0.201877  -0.201875  -0.201873  -0.201871   -0.20187  -0.201869  -0.201869  -0.201869  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665645   0.66566  0.665672  0.665681  0.665688  0.665693  0.665697    0.6657  0.665703  0.665704  0.665706  0.665707  0.665707  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 101.507032 | 101.313893  100.967125  100.710055  100.520067  100.380056  100.277146  100.201691  100.146495  100.106207  100.076861  100.055528  100.040049  100.028839  100.020734  100.014885   100.01067  100.007638   100.00546  100.003898  100.002779  100.001979  100.001407     100.001  100.000709  100.000503  100.000356  100.000252  100.000178  100.000125  100.000088  100.000062  100.000044  100.000031  100.000022  100.000015  100.000011  100.000007  100.000005  100.000004  100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0               100.0 |      100.0 | 0.723189  0.722535  0.722048  0.721688  0.721422  0.721227  0.721083  0.720978  0.720902  0.720846  0.720805  0.720776  0.720755  0.720739  0.720728   0.72072  0.720714   0.72071  0.720707  0.720705  0.720703  0.720702  0.720702  0.720701  0.720701    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.605522  2.601732  2.598916  2.596833  2.595296  2.594165  2.593336  2.592728  2.592285  2.591962  2.591728  2.591557  2.591434  2.591345   2.59128  2.591234    2.5912  2.591176  2.591159  2.591147  2.591138  2.591132  2.591127  2.591124  2.591122   2.59112  2.591119  2.591118  2.591118  2.591117  2.591117  2.591117  2.591117  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 | 101.507032  101.110418  100.816029  100.598215  100.437529  100.319308  100.232549  100.169029  100.122627  100.088803  100.064196  100.046329  100.033381  100.024014  100.017249  100.012371   100.00886  100.006337  100.004526  100.003228    100.0023  100.001636  100.001163  100.000825  100.000585  100.000414  100.000293  100.000207  100.000146  100.000103  100.000073  100.000051  100.000036  100.000025  100.000018  100.000012  100.000009  100.000006  100.000004  100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |  100.0 |   1.507032 |      100.0 |
    | 09:00 | 3.759119  3.748656  3.740765  3.734842  3.730413  3.727112  3.724661  3.722846  3.721505  3.720517   3.71979  3.719258  3.718868  3.718583  3.718375  3.718224  3.718114  3.718034  3.717977  3.717935  3.717905  3.717883  3.717868  3.717856  3.717848  3.717843  3.717838  3.717835  3.717833  3.717832  3.717831   3.71783  3.717829  3.717829  3.717829  3.717829  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 127.041661  126.491925  126.078102  125.767849  125.536085   125.36352  125.235416   125.14058  125.070552  125.018966  124.981051  124.953245  124.932893  124.918027  124.907189    124.8993   124.89357  124.889413  124.886404  124.884228  124.882657  124.881525   124.88071  124.880125  124.879704  124.879403  124.879187  124.879033  124.878923  124.878845  124.878789  124.878749  124.878721  124.878701  124.878686  124.878676  124.878669  124.878664   124.87866  124.878658  124.878656  124.878655  124.878654  124.878653  124.878653  124.878653  124.878652  124.878652  124.878652  124.878652 |   52.59119   52.486555   52.407653   52.348419   52.304127   52.271124   52.246611   52.228456   52.215046   52.205166   52.197902   52.192575   52.188676   52.185827    52.18375   52.182238    52.18114   52.180344   52.179767    52.17935   52.179049   52.178832   52.178676   52.178564   52.178483   52.178425   52.178384   52.178354   52.178333   52.178318   52.178308     52.1783   52.178294   52.178291   52.178288   52.178286   52.178285   52.178284   52.178283   52.178282   52.178282   52.178282   52.178282   52.178282   52.178281   52.178281   52.178281   52.178281   52.178281     52.178281 | 1.094777  1.093159  1.091939  1.091022  1.090336  1.089824  1.089444  1.089163  1.088955  1.088802  1.088689  1.088607  1.088546  1.088502   1.08847  1.088446  1.088429  1.088417  1.088408  1.088402  1.088397  1.088394  1.088391  1.088389  1.088388  1.088387  1.088387  1.088386  1.088386  1.088386  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.358741  1.358846  1.358926  1.358986  1.359032  1.359065   1.35909  1.359109  1.359123  1.359133   1.35914  1.359146   1.35915  1.359153  1.359155  1.359156  1.359158  1.359158  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.539391  0.538576   0.53796  0.537496  0.537149   0.53689  0.536697  0.536554  0.536448   0.53637  0.536313  0.536271   0.53624  0.536218  0.536201   0.53619  0.536181  0.536175   0.53617  0.536167  0.536164  0.536163  0.536161  0.536161   0.53616  0.536159  0.536159  0.536159  0.536159  0.536159  0.536159  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.203758  -0.203261  -0.202891  -0.202617  -0.202415  -0.202267  -0.202158  -0.202078   -0.20202  -0.201978  -0.201947  -0.201925  -0.201909  -0.201897  -0.201889  -0.201883  -0.201879  -0.201876  -0.201873  -0.201872  -0.201871   -0.20187  -0.201869  -0.201869  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 |  0.66562  0.665638  0.665653  0.665666  0.665675  0.665683  0.665689  0.665694  0.665698  0.665701  0.665703  0.665705  0.665706  0.665707  0.665707  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 102.694853 | 102.361128   101.75973  101.307605  100.968968  100.716187  100.528076  100.388489  100.285182  100.208917  100.152746  100.111466  100.081195   100.05904  100.042858   100.03106  100.022474  100.016237  100.011713  100.008437  100.006069  100.004359  100.003127   100.00224  100.001603  100.001145  100.000818  100.000583  100.000415  100.000295   100.00021  100.000149  100.000106  100.000075  100.000053  100.000037  100.000026  100.000019  100.000013  100.000009  100.000007  100.000005  100.000003  100.000002  100.000002  100.000001  100.000001  100.000001       100.0       100.0               100.0 |      100.0 | 0.725156  0.724028  0.723177  0.722538   0.72206  0.721703  0.721438  0.721242  0.721097   0.72099  0.720912  0.720854  0.720812  0.720781  0.720759  0.720742  0.720731  0.720722  0.720716  0.720711  0.720708  0.720706  0.720704  0.720703  0.720702  0.720701  0.720701    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.616917  2.610382  2.605453  2.601752  2.598984  2.596921  2.595388  2.594253  2.593415  2.592797  2.592343   2.59201  2.591766  2.591588  2.591458  2.591364  2.591295  2.591245  2.591209  2.591183  2.591164  2.591151  2.591141  2.591134  2.591129  2.591125  2.591123  2.591121  2.591119  2.591119  2.591118  2.591117  2.591117  2.591117  2.591117  2.591117  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 | 102.694853   102.01088  101.495885  101.109626  100.820935  100.605856  100.446087  100.327728  100.240269  100.175797  100.128378  100.093577  100.068089  100.049458  100.035865  100.025966   100.01877  100.013548  100.009764  100.007027   100.00505  100.003624  100.002598   100.00186  100.001329  100.000949  100.000677  100.000482  100.000343  100.000244  100.000173  100.000123  100.000087  100.000062  100.000044  100.000031  100.000022  100.000015  100.000011  100.000008  100.000005  100.000004  100.000003  100.000002  100.000001  100.000001  100.000001       100.0       100.0       100.0       100.0 |  100.0 |   2.694853 |      100.0 |
    | 09:30 |  3.78812  3.771032  3.757931  3.747945   3.74037  3.734649  3.730345  3.727118  3.724707   3.72291  3.721574  3.720584  3.719852  3.719311  3.718913   3.71862  3.718406  3.718249  3.718134   3.71805  3.717989  3.717944  3.717912  3.717889  3.717872   3.71786  3.717851  3.717844   3.71784  3.717836  3.717834  3.717832  3.717831   3.71783   3.71783  3.717829  3.717829  3.717829  3.717829  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 128.571088  127.668895  126.979207  126.454614  126.057364  125.757732  125.532531  125.363818  125.237799  125.143926  125.074178  125.022481  124.984249  124.956036   124.93526   124.91999  124.908789  124.900588  124.894593  124.890219  124.887032  124.884715  124.883032  124.881812  124.880929  124.880291   124.87983  124.879497  124.879258  124.879086  124.878962  124.878874   124.87881  124.878765  124.878732  124.878709  124.878693  124.878681  124.878672  124.878666  124.878662  124.878659  124.878657  124.878656  124.878655  124.878654  124.878653  124.878653  124.878653  124.878652 |  52.881204   52.710321   52.579313   52.479446   52.403695   52.346486   52.303447   52.271181   52.247067   52.229096   52.215741   52.205839   52.198515    52.19311   52.189129   52.186203   52.184057   52.182485   52.181336   52.180498   52.179887   52.179443   52.179121   52.178887   52.178718   52.178595   52.178507   52.178443   52.178397   52.178364   52.178341   52.178324   52.178312   52.178303   52.178297   52.178292   52.178289   52.178287   52.178285   52.178284   52.178283   52.178283   52.178282   52.178282   52.178282   52.178282   52.178282   52.178281   52.178281     52.178281 | 1.099251  1.096616  1.094593  1.093049  1.091877  1.090992  1.090325  1.089825  1.089452  1.089173  1.088966  1.088812  1.088699  1.088615  1.088553  1.088508  1.088475   1.08845  1.088432  1.088419   1.08841  1.088403  1.088398  1.088394  1.088392   1.08839  1.088389  1.088388  1.088387  1.088386  1.088386  1.088386  1.088386  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.358452  1.358621  1.358753  1.358853   1.35893  1.358988  1.359032  1.359065   1.35909  1.359108  1.359122  1.359132   1.35914  1.359145  1.359149  1.359152  1.359155  1.359156  1.359157  1.359158  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 |  0.54163  0.540314  0.539298  0.538521  0.537929  0.537481  0.537144   0.53689    0.5367  0.536559  0.536454  0.536376  0.536318  0.536275  0.536244  0.536221  0.536204  0.536191  0.536182  0.536176  0.536171  0.536167  0.536165  0.536163  0.536162  0.536161   0.53616   0.53616  0.536159  0.536159  0.536159  0.536159  0.536159  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.205257    -0.2044  -0.203753  -0.203266  -0.202902  -0.202631  -0.202429   -0.20228  -0.202169  -0.202088  -0.202028  -0.201985  -0.201953  -0.201929  -0.201912    -0.2019  -0.201891  -0.201885   -0.20188  -0.201876  -0.201874  -0.201872  -0.201871   -0.20187  -0.201869  -0.201869  -0.201869  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665598  0.665617  0.665634  0.665648  0.665661  0.665671  0.665679  0.665686  0.665691  0.665696  0.665699  0.665701  0.665703  0.665705  0.665706  0.665707  0.665707  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 104.586356 | 104.038944   103.04839  102.292759  101.718945  101.284961   100.95793  100.712311  100.528401  100.391084  100.288827  100.212866  100.156573  100.114947  100.084233  100.061616  100.044995  100.032802  100.023875   100.01735  100.012589  100.009121  100.006599  100.004768   100.00344  100.002478  100.001783  100.001282   100.00092  100.000659  100.000472  100.000338  100.000241  100.000172  100.000123  100.000087  100.000062  100.000044  100.000031  100.000022  100.000016  100.000011  100.000008  100.000006  100.000004  100.000003  100.000002  100.000001  100.000001  100.000001               100.0 |      100.0 | 0.728275  0.726438  0.725028  0.723952  0.723135  0.722517  0.722053  0.721704  0.721443  0.721249  0.721105  0.720998  0.720918   0.72086  0.720817  0.720785  0.720762  0.720745  0.720733  0.720724  0.720717  0.720712  0.720709  0.720706  0.720704  0.720703  0.720702  0.720701  0.720701  0.720701    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.635017  2.624354  2.616175  2.609938  2.605206  2.601631  2.598941  2.596924  2.595417  2.594293  2.593458  2.592839  2.592381  2.592043  2.591795  2.591612  2.591477  2.591379  2.591307  2.591255  2.591217  2.591189  2.591169  2.591154  2.591144  2.591136   2.59113  2.591126  2.591124  2.591121   2.59112  2.591119  2.591118  2.591118  2.591117  2.591117  2.591117  2.591117  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 | 104.586356  103.466536  102.610665  101.959629  101.466496  101.094384  100.814559  100.604796  100.448009  100.331137  100.244239  100.179783  100.132081  100.096856  100.070897  100.051805  100.037791  100.027523  100.020013   100.01453  100.010533  100.007624  100.005511  100.003978  100.002868  100.002065  100.001484  100.001066  100.000765  100.000548  100.000392   100.00028    100.0002  100.000143  100.000102  100.000072  100.000051  100.000036  100.000026  100.000018  100.000013  100.000009  100.000006  100.000005  100.000003  100.000002  100.000002  100.000001  100.000001  100.000001       100.0 |  100.0 |   4.586356 |      100.0 |
    | 10:00 | 3.832038  3.805527  3.784832  3.768792  3.756437  3.746974  3.739758   3.73428  3.730137  3.727014  3.724667  3.722908  3.721593  3.720613  3.719884  3.719343  3.718943  3.718647  3.718428  3.718267  3.718149  3.718062  3.717999  3.717952  3.717918  3.717893  3.717875  3.717862  3.717853  3.717846  3.717841  3.717837  3.717835  3.717833  3.717832  3.717831   3.71783  3.717829  3.717829  3.717829  3.717829  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 130.903131  129.493085  128.397221  127.550826  126.900672  126.403657  126.025331  125.738452  125.521666  125.358365  125.235711  125.143836   125.07519  125.024022  124.985969   124.95773  124.936817  124.921361  124.909959  124.901564  124.895393  124.890866  124.887549  124.885124  124.883353  124.882062  124.881122  124.880439  124.879943  124.879583  124.879323  124.879135  124.878999  124.878901  124.878831   124.87878  124.878744  124.878718  124.878699  124.878685  124.878676  124.878669  124.878664  124.878661  124.878658  124.878656  124.878655  124.878654  124.878654  124.878653 |  53.320377    53.05527   52.848315   52.687916   52.564374   52.469736   52.397582   52.342803    52.30137   52.270138   52.246667   52.229079   52.215934   52.206134   52.198845   52.193434   52.189428   52.186466   52.184281   52.182672    52.18149   52.180622   52.179986   52.179522   52.179182   52.178935   52.178755   52.178624   52.178529    52.17846    52.17841   52.178374   52.178348   52.178329   52.178316   52.178306   52.178299   52.178294    52.17829   52.178288   52.178286   52.178285   52.178284   52.178283   52.178282   52.178282   52.178282   52.178282   52.178282     52.178282 | 1.106004  1.101931  1.098744   1.09627  1.094362  1.092899  1.091783  1.090935  1.090293  1.089809  1.089445  1.089173  1.088969  1.088817  1.088704   1.08862  1.088558  1.088512  1.088478  1.088453  1.088435  1.088421  1.088411  1.088404  1.088399  1.088395  1.088392   1.08839  1.088389  1.088388  1.088387  1.088386  1.088386  1.088386  1.088386  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.358023  1.358281  1.358484  1.358644  1.358768  1.358863  1.358936  1.358992  1.359034  1.359066   1.35909  1.359108  1.359122  1.359132  1.359139  1.359145  1.359149  1.359152  1.359154  1.359156  1.359157  1.359158  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.544976  0.542963  0.541378   0.54014  0.539182  0.538445  0.537881  0.537452  0.537127  0.536882  0.536697  0.536559  0.536455  0.536378   0.53632  0.536278  0.536246  0.536223  0.536206  0.536193  0.536184  0.536177  0.536172  0.536168  0.536165  0.536163  0.536162  0.536161   0.53616   0.53616  0.536159  0.536159  0.536159  0.536159  0.536159  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.207628  -0.206237  -0.205166  -0.204347  -0.203724  -0.203253  -0.202898  -0.202633  -0.202434  -0.202286  -0.202176  -0.202094  -0.202034  -0.201989  -0.201957  -0.201933  -0.201915  -0.201902  -0.201893  -0.201886  -0.201881  -0.201877  -0.201875  -0.201873  -0.201871   -0.20187   -0.20187  -0.201869  -0.201869  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665591  0.665604  0.665618  0.665632  0.665646  0.665657  0.665668  0.665676  0.665683  0.665689  0.665693  0.665697    0.6657  0.665702  0.665704  0.665705  0.665706  0.665707  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 107.466641 | 106.610349  105.053692  103.847863  102.918933  102.206804   101.66325  101.249987  100.936896  100.700465  100.522457   100.38881  100.288728  100.213967  100.158251   100.11682  100.086077  100.063312  100.046487  100.034076  100.024938  100.018222  100.013294  100.009684  100.007044  100.005116  100.003711  100.002688  100.001945  100.001405  100.001014  100.000731  100.000526  100.000378  100.000271  100.000195   100.00014    100.0001  100.000071  100.000051  100.000036  100.000026  100.000018  100.000013  100.000009  100.000007  100.000005  100.000003  100.000002  100.000002          100.000001 | 100.000001 |  0.73298  0.730142  0.727921  0.726197  0.724867  0.723847  0.723069  0.722478   0.72203  0.721693  0.721439  0.721249  0.721107  0.721001  0.720922  0.720864   0.72082  0.720788  0.720765  0.720747  0.720734  0.720725  0.720718  0.720713  0.720709  0.720707  0.720705  0.720703  0.720702  0.720702  0.720701  0.720701    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.662394  2.645873  2.632966  2.622956  2.615243  2.609332  2.604824  2.601401  2.598811  2.596859  2.595392  2.594292  2.593471  2.592858  2.592402  2.592064  2.591813  2.591628  2.591491  2.591391  2.591317  2.591263  2.591223  2.591194  2.591173  2.591157  2.591146  2.591138  2.591132  2.591127  2.591124  2.591122   2.59112  2.591119  2.591118  2.591118  2.591117  2.591117  2.591117  2.591117  2.591117  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 | 107.466641  105.718149  104.360507  103.312446   102.50751  101.892109  101.423529  101.068054  100.799286  100.596704  100.444443  100.330311  100.244975   100.18132  100.133946  100.098765  100.072693   100.05341  100.039176  100.028688  100.020974   100.01531  100.011159  100.008122  100.005902  100.004283  100.003104  100.002247  100.001624  100.001172  100.000845  100.000609  100.000438  100.000315  100.000226  100.000162  100.000116  100.000083  100.000059  100.000042   100.00003  100.000021  100.000015  100.000011  100.000008  100.000005  100.000004  100.000003  100.000002  100.000001  100.000001 |  100.0 |   7.466641 | 100.000001 |
    | 10:30 | 3.895577  3.856412  3.825223  3.800607   3.78133  3.766336  3.754741  3.745822  3.738991  3.733781  3.729822  3.726822  3.724558  3.722852  3.721571   3.72061  3.719893  3.719357  3.718959  3.718662  3.718443   3.71828   3.71816  3.718072  3.718006  3.717958  3.717923  3.717897  3.717879  3.717865  3.717855  3.717847  3.717842  3.717838  3.717835  3.717833  3.717832  3.717831   3.71783   3.71783  3.717829  3.717829  3.717829  3.717829  3.717828  3.717828  3.717828  3.717828  3.717828             3.717828 | 134.311281  132.205757  130.540002  129.232185  128.212241  127.421463  126.811523  126.343216  125.985123  125.712312  125.505167  125.348364  125.230007  125.140905  125.073994  125.023866  124.986396  124.958446  124.937641  124.922186  124.910725  124.902243  124.895977  124.891355  124.887952  124.885451  124.883615   124.88227  124.881286  124.880567  124.880042   124.87966  124.879382   124.87918  124.879034  124.878927   124.87885  124.878795  124.878755  124.878726  124.878705   124.87869  124.878679  124.878671  124.878666  124.878662  124.878659  124.878657  124.878656  124.878655 |  53.955775   53.564122    53.25223   53.006072   52.813302   52.663358   52.547411   52.458215   52.389908   52.337809   52.298215   52.268224   52.245575   52.228518   52.215705   52.206104   52.198926   52.193572   52.189585   52.186624   52.184428   52.182802   52.181602   52.180716   52.180064   52.179584   52.179233   52.178975   52.178786   52.178648   52.178548   52.178475   52.178421   52.178383   52.178354   52.178334   52.178319   52.178309   52.178301   52.178295   52.178291   52.178289   52.178287   52.178285   52.178284   52.178283   52.178283   52.178282   52.178282     52.178282 | 1.115726   1.10974  1.104958  1.101174  1.098204  1.095891    1.0941  1.092721  1.091664  1.090857  1.090244  1.089779  1.089428  1.089164  1.088965  1.088817  1.088705  1.088622   1.08856  1.088514   1.08848  1.088455  1.088437  1.088423  1.088413  1.088405    1.0884  1.088396  1.088393  1.088391  1.088389  1.088388  1.088387  1.088387  1.088386  1.088386  1.088386  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.357422   1.35779  1.358089  1.358329  1.358519  1.358668  1.358785  1.358875  1.358944  1.358997  1.359038  1.359068  1.359091  1.359109  1.359122  1.359132  1.359139  1.359145  1.359149  1.359152  1.359154  1.359156  1.359157  1.359158  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 |  0.54972  0.546809  0.544461  0.542587  0.541108   0.53995   0.53905  0.538355  0.537821  0.537413  0.537102  0.536867  0.536689  0.536554  0.536453  0.536378  0.536321  0.536279  0.536247  0.536224  0.536207  0.536194  0.536184  0.536178  0.536172  0.536169  0.536166  0.536164  0.536162  0.536161   0.53616   0.53616  0.536159  0.536159  0.536159  0.536159  0.536159  0.536159  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.211181  -0.209046  -0.207368   -0.20606  -0.205048   -0.20427  -0.203676  -0.203225  -0.202883  -0.202625  -0.202431  -0.202286  -0.202178  -0.202097  -0.202037  -0.201992  -0.201959  -0.201935  -0.201917  -0.201904  -0.201894  -0.201887  -0.201882  -0.201878  -0.201875  -0.201873  -0.201872  -0.201871   -0.20187  -0.201869  -0.201869  -0.201869  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 |  0.66562  0.665614  0.665617  0.665625  0.665635  0.665646  0.665656  0.665665  0.665674  0.665681  0.665686  0.665691  0.665695  0.665698  0.665701  0.665703  0.665704  0.665705  0.665706  0.665707  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 111.678559 |  110.39641  108.053502  106.208913  104.766294  103.644666  102.777139  102.109253  101.597199   101.20609   100.90838  100.682476  100.511559  100.382595  100.285536  100.212666  100.158082  100.117285  100.086857  100.064209  100.047384   100.03491  100.025677  100.018857  100.013826  100.010123    100.0074  100.005402  100.003938  100.002867  100.002084  100.001513  100.001097  100.000795  100.000575  100.000415    100.0003  100.000216  100.000156  100.000112   100.00008  100.000058  100.000041   100.00003  100.000021  100.000015  100.000011  100.000008  100.000005  100.000004          100.000003 | 100.000002 |  0.73975  0.735582  0.732251  0.729614  0.727545  0.725933  0.724684  0.723723  0.722986  0.722424  0.721996  0.721672  0.721427  0.721243  0.721104  0.721001  0.720923  0.720865  0.720822   0.72079  0.720766  0.720749  0.720736  0.720726  0.720719  0.720714   0.72071  0.720707  0.720705  0.720704  0.720703  0.720702  0.720701  0.720701    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.701933  2.677571  2.658148  2.642805  2.630781  2.621423  2.614183  2.608612  2.604344  2.601089  2.598614  2.596739  2.595324  2.594257  2.593456  2.592856  2.592407  2.592072  2.591823  2.591638  2.591501  2.591399  2.591324  2.591268  2.591228  2.591198  2.591176   2.59116  2.591148  2.591139  2.591133  2.591128  2.591125  2.591123  2.591121   2.59112  2.591119  2.591118  2.591117  2.591117  2.591117  2.591117  2.591117  2.591116  2.591116  2.591116  2.591116  2.591116  2.591116        2.591116 | 111.678559  109.065161   107.00162  105.383639  104.122842   103.14574  102.392167  101.813505  101.370888  101.033531  100.777235  100.583107  100.436477  100.326014  100.243003  100.180768  100.134214  100.099463  100.073578  100.054334  100.040056  100.029481  100.021663  100.015893  100.011642  100.008516   100.00622  100.004537  100.003304  100.002403  100.001746  100.001267  100.000918  100.000664   100.00048  100.000347   100.00025   100.00018  100.000129  100.000093  100.000067  100.000048  100.000034  100.000025  100.000018  100.000013  100.000009  100.000006  100.000005  100.000003  100.000002 |  100.0 |  11.678559 | 100.000002 |
    | 11:00 |  3.98361  3.928411  3.883484  3.847312  3.818466  3.795652   3.77774  3.763766  3.752923  3.744552  3.738116  3.733187  3.729426  3.726564  3.724394  3.722752  3.721513   3.72058   3.71988  3.719355  3.718962  3.718669   3.71845  3.718288  3.718167  3.718078  3.718012  3.717963  3.717927  3.717901  3.717881  3.717867  3.717856  3.717849  3.717843  3.717839  3.717836  3.717834  3.717832  3.717831   3.71783   3.71783  3.717829  3.717829  3.717829  3.717829  3.717828  3.717828  3.717828             3.717828 | 139.099913  136.088203  133.659492  131.718737  130.180387   128.96965    128.0227  127.286137  126.716006  126.276612  125.939292  125.681246  125.484479  125.334882  125.221462  125.135692  125.070989  125.022292  124.985721  124.958315  124.937819   124.92252  124.911123  124.902648  124.896356  124.891694  124.888245  124.885698   124.88382  124.882437   124.88142  124.880674  124.880127  124.879726  124.879434   124.87922  124.879064  124.878951  124.878868  124.878808  124.878765  124.878734  124.878711  124.878694  124.878682  124.878674  124.878668  124.878663   124.87866  124.878658 |  54.836104   54.284105   53.834838   53.473122   53.184657    52.95652   52.777401   52.637655   52.529231   52.445517   52.381159   52.331873   52.294259   52.265645    52.24394    52.22752    52.21513   52.205803   52.198797   52.193547   52.189619   52.186688   52.184504    52.18288   52.181674   52.180781    52.18012   52.179632   52.179272   52.179007   52.178812   52.178669   52.178564   52.178487   52.178431    52.17839    52.17836   52.178339   52.178323   52.178311   52.178303   52.178297   52.178293   52.178289   52.178287   52.178285   52.178284   52.178283   52.178283     52.178282 | 1.129104  1.120728   1.11388  1.108346   1.10392  1.100411  1.097651  1.095494  1.093819  1.092525  1.091529  1.090765  1.090183  1.089739  1.089403  1.089149  1.088956  1.088812  1.088703  1.088622  1.088561  1.088515  1.088482  1.088456  1.088438  1.088424  1.088414  1.088406    1.0884  1.088396  1.088393  1.088391  1.088389  1.088388  1.088387  1.088387  1.088386  1.088386  1.088386  1.088386  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.356624   1.35712  1.357535  1.357877  1.358155  1.358378  1.358555  1.358694  1.358803  1.358888  1.358953  1.359003  1.359042  1.359071  1.359093   1.35911  1.359123  1.359132  1.359139  1.359145  1.359149  1.359152  1.359154  1.359156  1.359157  1.359158  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.556112  0.552128  0.548826  0.546127  0.543948  0.542208  0.540832  0.539751  0.538909  0.538256  0.537753  0.537367  0.537071  0.536846  0.536676  0.536546  0.536449  0.536375   0.53632  0.536279  0.536248  0.536225  0.536207  0.536195  0.536185  0.536178  0.536173  0.536169  0.536166  0.536164  0.536162  0.536161  0.536161   0.53616  0.536159  0.536159  0.536159  0.536159  0.536159  0.536159  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.216238  -0.213139  -0.210643  -0.208656  -0.207089  -0.205863  -0.204912  -0.204179  -0.203616  -0.203186  -0.202858  -0.202611  -0.202423  -0.202282  -0.202176  -0.202097  -0.202038  -0.201994  -0.201961  -0.201936  -0.201918  -0.201905  -0.201895  -0.201888  -0.201882  -0.201878  -0.201876  -0.201873  -0.201872  -0.201871   -0.20187  -0.201869  -0.201869  -0.201869  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665713  0.665668  0.665646  0.665637  0.665637  0.665642  0.665649  0.665657  0.665665  0.665672  0.665679  0.665685   0.66569  0.665694  0.665697    0.6657  0.665702  0.665703  0.665705  0.665706  0.665707  0.665707  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 117.614836 | 115.771398  112.383429  109.669789  107.513372  105.811738  104.477296  103.436562  102.628863  102.004761  101.524425  101.156061  100.874493  100.659922  100.496866  100.373286  100.279859  100.209393  100.156367   100.11655  100.086714  100.064402  100.047749  100.035343  100.026118   100.01927  100.014195  100.010442  100.007669  100.005625  100.004119  100.003013    100.0022  100.001605  100.001169  100.000851  100.000618  100.000449  100.000325  100.000236   100.00017  100.000123  100.000089  100.000064  100.000046  100.000033  100.000024  100.000017  100.000012  100.000009          100.000006 | 100.000005 | 0.749061  0.743232  0.738465  0.734611  0.731528  0.729083  0.727159  0.725656  0.724488  0.723586  0.722892   0.72236  0.721953  0.721644   0.72141  0.721232  0.721098  0.720997  0.720922  0.720865  0.720822  0.720791  0.720767  0.720749  0.720736  0.720727   0.72072  0.720714   0.72071  0.720707  0.720705  0.720704  0.720703  0.720702  0.720701  0.720701    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.756584  2.722333  2.694414  2.671906  2.653938  2.639715  2.628541  2.619818  2.613048  2.607819  2.603798  2.600718  2.598367  2.596578  2.595221  2.594195   2.59342  2.592837  2.592399  2.592071  2.591825  2.591642  2.591505  2.591404  2.591328  2.591273  2.591231  2.591201  2.591178  2.591162  2.591149   2.59114  2.591134  2.591129  2.591126  2.591123  2.591121   2.59112  2.591119  2.591118  2.591118  2.591117  2.591117  2.591117  2.591117  2.591117  2.591116  2.591116  2.591116        2.591116 | 117.614836   113.86395  110.849084  108.445876  106.544334  105.049545  103.881289  102.972916  102.269845  101.727911  101.311737  100.993222  100.750211  100.565344  100.425089  100.318953  100.238829  100.178481  100.133129  100.099119  100.073665  100.054653  100.040479  100.029932  100.022097  100.016288  100.011987  100.008809  100.006464  100.004737  100.003466  100.002533  100.001849  100.001347  100.000981  100.000713  100.000518  100.000375  100.000272  100.000197  100.000142  100.000103  100.000074  100.000053  100.000038  100.000028   100.00002  100.000014   100.00001  100.000007  100.000005 |  100.0 |  17.614836 | 100.000005 |
    | 11:30 | 4.100633  4.026293  3.964352  3.913389  3.871928  3.838536  3.811879  3.790764   3.77415  3.761156  3.751045  3.743213  3.737172   3.73253  3.728973  3.726257  3.724189  3.722618  3.721428  3.720528  3.719849  3.719338  3.718955  3.718667  3.718452  3.718291  3.718171  3.718082  3.718016  3.717966   3.71793  3.717903  3.717883  3.717869  3.717858   3.71785  3.717844   3.71784  3.717837  3.717834  3.717833  3.717831   3.71783   3.71783  3.717829  3.717829  3.717829  3.717829  3.717828             3.717828 | 145.585456  141.449571  138.045735  135.273877  133.038053  131.249837  129.830314  128.710901   127.83329  127.148789  126.617349  126.206432  125.889878  125.646837   125.46081  125.318831  125.210757  125.128697  125.066538   125.01956  124.984131  124.957467  124.937441  124.922428  124.911195  124.902805   124.89655  124.891894  124.888435  124.885869  124.883968  124.882562  124.881524  124.880759  124.880196  124.879781  124.879477  124.879254  124.879091  124.878972  124.878884  124.878821  124.878774  124.878741  124.878716  124.878698  124.878686  124.878676  124.878669  124.878665 |  56.006331    55.26293   54.643524   54.133885    53.71928   53.385361   53.118794   52.907637     52.7415   52.611556   52.510446   52.432134   52.371725   52.325297   52.289733   52.262574   52.241891   52.226181   52.214277   52.205279   52.198492   52.193384   52.189547    52.18667   52.184518    52.18291   52.181711   52.180819   52.180156   52.179664     52.1793   52.179031   52.178832   52.178685   52.178577   52.178498   52.178439   52.178397   52.178365   52.178343   52.178326   52.178314   52.178305   52.178298   52.178294    52.17829   52.178288   52.178286   52.178285     52.178284 |  1.14673  1.135553  1.126186  1.118441  1.112114  1.107001  1.102908  1.099658  1.097097  1.095091  1.093529  1.092317  1.091383  1.090664  1.090113  1.089692  1.089371  1.089128  1.088943  1.088804  1.088698  1.088619   1.08856  1.088515  1.088482  1.088457  1.088438  1.088424  1.088414  1.088406  1.088401  1.088397  1.088394  1.088391   1.08839  1.088388  1.088387  1.088387  1.088386  1.088386  1.088386  1.088386  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 | 1.355622  1.356251  1.356795  1.357258  1.357643  1.357961  1.358219  1.358426   1.35859   1.35872  1.358822  1.358901  1.358963   1.35901  1.359046  1.359074  1.359095  1.359111  1.359124  1.359133   1.35914  1.359145  1.359149  1.359152  1.359154  1.359156  1.359157  1.359158  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.564301  0.559139  0.554731   0.55103  0.547968  0.545466  0.543447  0.541833  0.540555  0.539549  0.538762  0.538151  0.537679  0.537315  0.537036  0.536822   0.53666  0.536536  0.536442  0.536371  0.536318  0.536277  0.536247  0.536224  0.536208  0.536195  0.536185  0.536178  0.536173  0.536169  0.536166  0.536164  0.536163  0.536161  0.536161   0.53616   0.53616  0.536159  0.536159  0.536159  0.536159  0.536159  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158  0.536158      0.536158 |  -0.22308  -0.218819  -0.215296  -0.212422  -0.210106   -0.20826  -0.206801  -0.205657  -0.204766  -0.204077  -0.203545  -0.203138  -0.202827   -0.20259  -0.202411  -0.202275  -0.202172  -0.202095  -0.202037  -0.201994  -0.201961  -0.201937  -0.201919  -0.201906  -0.201896  -0.201888  -0.201883  -0.201879  -0.201876  -0.201874  -0.201872  -0.201871   -0.20187  -0.201869  -0.201869  -0.201869  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.665908  0.665799  0.665729  0.665688  0.665665  0.665655  0.665653  0.665656  0.665661  0.665666  0.665673  0.665678  0.665684  0.665688  0.665692  0.665696  0.665698  0.665701  0.665703  0.665704  0.665705  0.665706  0.665707  0.665707  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 125.703736 | 123.151741  118.432011  114.582651  111.471716  108.978126   106.99398   105.42546  104.192662  103.228707  102.478425  101.896861  101.447758  101.102127  100.836961   100.63412  100.479374  100.361624   100.27224  100.204546  100.153392  100.114819  100.085791   100.06399  100.047648  100.035421  100.026289   100.01948  100.014413  100.010648  100.007855  100.005786  100.004256  100.003126  100.002293   100.00168  100.001229  100.000898  100.000656  100.000478  100.000348  100.000253  100.000184  100.000133  100.000097   100.00007  100.000051  100.000037  100.000026  100.000019          100.000014 | 100.000011 | 0.761316  0.753546  0.747031   0.74164  0.737235  0.733674  0.730822  0.728558  0.726773  0.725375  0.724286  0.723442   0.72279  0.722289  0.721904  0.721611  0.721387  0.721218  0.721089  0.720992  0.720918  0.720863  0.720821   0.72079  0.720767   0.72075  0.720737  0.720727   0.72072  0.720715  0.720711  0.720708  0.720706  0.720704  0.720703  0.720702  0.720701  0.720701  0.720701    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 2.829006  2.783028  2.744641  2.713002  2.687226  2.666441  2.649833  2.636666    2.6263  2.618189  2.611875  2.606983  2.603208  2.600307  2.598084  2.596386  2.595093  2.594111  2.593367  2.592804   2.59238  2.592061  2.591821  2.591641  2.591506  2.591406  2.591331  2.591275  2.591233  2.591203   2.59118  2.591163  2.591151  2.591141  2.591135   2.59113  2.591126  2.591123  2.591122   2.59112  2.591119  2.591118  2.591118  2.591117  2.591117  2.591117  2.591117  2.591117  2.591116        2.591116 | 125.703736  120.520097  116.275013  112.831663  110.062627  107.852966  106.101732   104.72224  103.641435  102.798713  102.144455  101.638486  101.248576  100.949077  100.719712  100.544549  100.411128  100.309754   100.23291  100.174792  100.130931  100.097898   100.07307  100.054445  100.040499  100.030077  100.022301  100.016509  100.012203  100.009007  100.006638  100.004886  100.003591  100.002635  100.001932  100.001414  100.001034  100.000755   100.00055  100.000401  100.000292  100.000212  100.000154  100.000112  100.000081  100.000058  100.000042   100.00003  100.000022  100.000016  100.000011 |  100.0 |  25.703736 | 100.000011 |
    | 12:00 | 4.250162  4.154331  4.072497  4.003585    3.9463  3.899236  3.860973  3.830156  3.805537  3.786011  3.770622   3.75856  3.749151  3.741844  3.736191  3.731832  3.728483  3.725916  3.723954  3.722458  3.721321  3.720458  3.719805  3.719311  3.718939  3.718658  3.718448   3.71829  3.718172  3.718083  3.718017  3.717968  3.717932  3.717905  3.717885   3.71787  3.717859  3.717851  3.717845   3.71784  3.717837  3.717835  3.717833  3.717832  3.717831   3.71783  3.717829  3.717829  3.717829             3.717829 | 154.071843  148.607303  144.013609  140.197247  137.060919  134.508731  132.450178  130.802799  129.493615  128.459579  127.647291  127.012268  126.517954  126.134656  125.838477  125.610348  125.435151  125.300971  125.198469  125.120356  125.060966  125.015909  124.981799  124.956028  124.936596   124.92197  124.910984  124.902745  124.896577  124.891968  124.888529  124.885968  124.884063  124.882648  124.881599  124.880823  124.880249  124.879825  124.879513  124.879283  124.879114  124.878989  124.878898  124.878832  124.878783  124.878747  124.878721  124.878702  124.878688  124.878678 |  57.501625   56.543311   55.724969   55.035851      54.463   53.992357   53.609734   53.301557    53.05537   52.860113   52.706222     52.5856   52.491514   52.418442   52.361909   52.318323   52.284826   52.259156   52.239539   52.224584    52.21321    52.20458   52.198046   52.193108   52.189385   52.186583   52.184477   52.182898   52.181717   52.180833   52.180174   52.179683   52.179318   52.179047   52.178846   52.178697   52.178587   52.178506   52.178446   52.178402    52.17837   52.178346   52.178329   52.178316   52.178306     52.1783   52.178295   52.178291   52.178288     52.178286 | 1.169001  1.154759  1.142508  1.132125  1.123447  1.116284  1.110438  1.105715  1.101932  1.098926  1.096553   1.09469  1.093236  1.092106  1.091231  1.090556  1.090037  1.089639  1.089335  1.089103  1.088927  1.088793  1.088692  1.088615  1.088557  1.088514  1.088481  1.088457  1.088438  1.088425  1.088414  1.088407  1.088401  1.088397  1.088394  1.088391   1.08839  1.088389  1.088388  1.088387  1.088386  1.088386  1.088386  1.088386  1.088385  1.088385  1.088385  1.088385  1.088385  1.088385 |  1.35443  1.355183  1.355857  1.356448  1.356958  1.357388  1.357747  1.358042  1.358281  1.358473  1.358626  1.358746  1.358841  1.358915  1.358973  1.359017  1.359051  1.359078  1.359098  1.359113  1.359125  1.359134   1.35914  1.359145  1.359149  1.359152  1.359154  1.359156  1.359157  1.359158  1.359159  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.574285  0.567946  0.562363  0.557534  0.553428   0.54999   0.54715  0.544834  0.542964  0.541468  0.540282  0.539347  0.538615  0.538044  0.537602   0.53726  0.536997  0.536795  0.536641  0.536523  0.536434  0.536366  0.536314  0.536275  0.536246  0.536224  0.536207  0.536195  0.536185  0.536178  0.536173  0.536169  0.536166  0.536164  0.536163  0.536162  0.536161   0.53616   0.53616  0.536159  0.536159  0.536159  0.536159  0.536159  0.536159  0.536158  0.536158  0.536158  0.536158      0.536158 | -0.231883  -0.226336  -0.221613  -0.217656  -0.214389  -0.211727   -0.20958  -0.207867  -0.206511  -0.205446  -0.204614  -0.203968  -0.203469  -0.203085  -0.202791  -0.202566  -0.202394  -0.202264  -0.202165  -0.202091  -0.202035  -0.201992  -0.201961  -0.201937  -0.201919  -0.201906  -0.201896  -0.201889  -0.201883  -0.201879  -0.201876  -0.201874  -0.201872  -0.201871   -0.20187   -0.20187  -0.201869  -0.201869  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.666248  0.666043    0.6659  0.665803   0.66574  0.665702   0.66568   0.66567  0.665666  0.665667   0.66567  0.665674  0.665679  0.665684  0.665688  0.665691  0.665695  0.665698    0.6657  0.665702  0.665703  0.665705  0.665706  0.665706  0.665707  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 136.388647 | 132.978491   126.62916  121.352562  117.012066  113.474906  110.616766  108.324831  106.499393  105.054275  103.916384  103.024698  102.328948  101.788184  101.369363  101.046032  100.797167  100.606148  100.459913  100.348238  100.263156  100.198478  100.149417   100.11228  100.084224   100.06307   100.04715  100.035191  100.026223   100.01951  100.014493  100.010751  100.007963  100.005889   100.00435  100.003208  100.002363  100.001738  100.001277  100.000937  100.000687  100.000502  100.000367  100.000268  100.000196  100.000142  100.000104  100.000075  100.000055   100.00004          100.000029 | 100.000024 | 0.776785  0.766895  0.758382  0.751162  0.745124  0.740138  0.736068  0.732778  0.730143  0.728048  0.726394  0.725096  0.724082  0.723294  0.722684  0.722213  0.721851  0.721574  0.721362    0.7212  0.721077  0.720984  0.720913   0.72086   0.72082  0.720789  0.720767   0.72075  0.720737  0.720727   0.72072  0.720715  0.720711  0.720708  0.720706  0.720704  0.720703  0.720702  0.720701  0.720701  0.720701    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 |  2.92119  2.862155  2.811616  2.768963   2.73344  2.704207   2.68041  2.661221  2.645879  2.633702  2.624099  2.616568  2.610692  2.606127  2.602595  2.599871  2.597777  2.596173  2.594946  2.594011    2.5933  2.592761  2.592352  2.592043  2.591811  2.591635  2.591504  2.591405  2.591331  2.591276  2.591235  2.591204  2.591181  2.591164  2.591152  2.591142  2.591135   2.59113  2.591127  2.591124  2.591122   2.59112  2.591119  2.591118  2.591118  2.591117  2.591117  2.591117  2.591117        2.591117 | 136.388647  129.473624  123.700239  118.930984  115.029589  111.866393  109.322164  107.290364  105.678076  104.405896  103.407119  102.626509  102.018873  101.547615  101.183345  100.902642  100.686952  100.521661  100.395312  100.298961  100.225653  100.169999  100.127836  100.095959  100.071906  100.053791  100.040173  100.029954  100.022299  100.016575  100.012302  100.009117  100.006746  100.004985  100.003679  100.002711  100.001995  100.001467  100.001077  100.000789  100.000578  100.000423  100.000309  100.000225  100.000164   100.00012  100.000087  100.000063  100.000046  100.000033  100.000024 |  100.0 |  36.388647 | 100.000024 |
    | 12:30 | 4.434196  4.315692  4.211945  4.122449  4.046325  3.982425   3.92943  3.885959  3.850647   3.82221  3.799483   3.78144  3.767199  3.756017  3.747277  3.740473  3.735195  3.731114  3.727969  3.725551  3.723698   3.72228  3.721198  3.720375  3.719749  3.719274  3.718915  3.718644  3.718439  3.718285  3.718169  3.718083  3.718018  3.717969  3.717933  3.717906  3.717886  3.717871  3.717859  3.717851  3.717845  3.717841  3.717837  3.717835  3.717833  3.717832  3.717831   3.71783   3.71783             3.717829 | 164.823421  157.861374  151.881587  146.809646  142.558632  139.034923  136.143577  133.792785  131.897123  130.379591  129.172585  128.218032  127.466938  126.878591  126.419578  126.062768  125.786317  125.572777  125.408293  125.281927  125.185084   125.11104  125.054553  125.011551  124.978882  124.954112  124.935367  124.921208  124.910533  124.902497  124.896459   124.89193  124.888539  124.886002  124.884109  124.882698  124.881647  124.880867  124.880287  124.879858   124.87954  124.879305  124.879132  124.879004   124.87891  124.878841   124.87879  124.878753  124.878726  124.878706 |  59.341962   58.156921   57.119451   56.224487   55.463255   54.824251   54.294305   53.859593   53.506471   53.222099   52.994827   52.814398   52.671992   52.560173    52.47277   52.404726   52.351947   52.311142   52.279689   52.255512   52.236976     52.2228   52.211982   52.203745   52.197487   52.192741    52.18915   52.186437   52.184391   52.182851   52.181694   52.180826   52.180176    52.17969   52.179327   52.179057   52.178855   52.178706   52.178595   52.178512   52.178452   52.178407   52.178373   52.178349   52.178331   52.178318   52.178308   52.178301   52.178295     52.178292 | 1.196045  1.178676  1.163335  1.149996  1.138572  1.128925  1.120883  1.114258  1.108857  1.104495  1.101001  1.098221  1.096024  1.094297  1.092946  1.091893  1.091076  1.090444  1.089957  1.089582  1.089295  1.089075  1.088908   1.08878  1.088683  1.088609  1.088554  1.088512   1.08848  1.088456  1.088438  1.088425  1.088414  1.088407  1.088401  1.088397  1.088394  1.088392   1.08839  1.088389  1.088388  1.088387  1.088386  1.088386  1.088386  1.088386  1.088385  1.088385  1.088385  1.088385 | 1.353087  1.353937  1.354726  1.355442  1.356079  1.356635  1.357111  1.357512  1.357845  1.358118   1.35834  1.358518   1.35866  1.358772   1.35886  1.358929  1.358983  1.359024  1.359057  1.359081    1.3591  1.359115  1.359126  1.359134  1.359141  1.359146  1.359149  1.359152  1.359154  1.359156  1.359157  1.359158  1.359159  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161  1.359161  1.359161  1.359161  1.359161          1.359161 | 0.585894  0.578501  0.571782   0.56579  0.560543  0.556028  0.552203  0.549009  0.546377  0.544232  0.542501  0.541117  0.540017  0.539149  0.538469  0.537937  0.537524  0.537204  0.536957  0.536767  0.536621  0.536509  0.536424  0.536359   0.53631  0.536272  0.536244  0.536223  0.536207  0.536194  0.536185  0.536178  0.536173  0.536169  0.536167  0.536164  0.536163  0.536162  0.536161   0.53616   0.53616  0.536159  0.536159  0.536159  0.536159  0.536159  0.536159  0.536158  0.536158      0.536158 | -0.242675  -0.235823  -0.229809  -0.224625  -0.220227  -0.216554  -0.213526  -0.211059  -0.209072  -0.207484  -0.206226  -0.205235   -0.20446  -0.203857  -0.203389  -0.203028  -0.202751  -0.202538  -0.202375  -0.202251  -0.202157  -0.202085  -0.202031   -0.20199   -0.20196  -0.201936  -0.201919  -0.201906  -0.201896  -0.201889  -0.201883  -0.201879  -0.201876  -0.201874  -0.201872  -0.201871   -0.20187   -0.20187  -0.201869  -0.201869  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.666782  0.666447  0.666197  0.666016  0.665889  0.665803  0.665747  0.665712  0.665691  0.665681  0.665677  0.665676  0.665678  0.665681  0.665684  0.665688  0.665691  0.665694  0.665697  0.665699  0.665701  0.665703  0.665704  0.665705  0.665706  0.665707  0.665707  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 150.102726 | 145.693656   137.42686  130.424262  124.557563  119.693089  115.698021  112.445491  109.818287   107.71113  106.031701  104.700669  103.651025  102.826978  102.182639   101.68065   101.29086  100.989117  100.756195  100.576873  100.439163  100.333658  100.253009  100.191495  100.144672  100.109104  100.082139  100.061733  100.046321    100.0347  100.025954  100.019382  100.014452  100.010761     100.008   100.00594  100.004403   100.00326  100.002411   100.00178  100.001313  100.000967  100.000711  100.000522  100.000383  100.000281  100.000206   100.00015   100.00011   100.00008          100.000058 | 100.000049 | 0.795544  0.783499  0.772851  0.763586  0.755645  0.748936   0.74334  0.738728  0.734967  0.731928  0.729494  0.727557  0.726026  0.724822   0.72388  0.723146  0.722576  0.722136  0.721796  0.721535  0.721334  0.721181  0.721064  0.720975  0.720907  0.720856  0.720817  0.720788  0.720766  0.720749  0.720737  0.720727   0.72072  0.720715  0.720711  0.720708  0.720706  0.720704  0.720703  0.720702  0.720701  0.720701  0.720701    0.7207    0.7207    0.7207    0.7207    0.7207    0.7207         0.7207 | 3.034138   2.96147  2.897666  2.842479  2.795427  2.755849  2.722967  2.695953  2.673982  2.656271  2.642104  2.630849  2.621962   2.61498  2.609521   2.60527  2.601972  2.599422  2.597456  2.595945  2.594786    2.5939  2.593223  2.592708  2.592317   2.59202  2.591796  2.591626  2.591498  2.591402   2.59133  2.591275  2.591235  2.591204  2.591182  2.591165  2.591152  2.591143  2.591136  2.591131  2.591127  2.591124  2.591122   2.59112  2.591119  2.591119  2.591118  2.591117  2.591117        2.591117 | 150.102726  141.176813  133.577782  127.181401  121.854721  117.462807  113.874519  110.966949  108.628387  106.759899  105.275774  104.103117   103.18089  102.458652  101.895174   101.45707   101.11752  100.855122  100.652897  100.497444  100.378235   100.28703  100.217404  100.164364  100.124042  100.093447  100.070279  100.052767  100.039554  100.029602   100.02212  100.016504  100.012295  100.009146  100.006794   100.00504  100.003733  100.002762   100.00204  100.001505  100.001109  100.000816    100.0006  100.000441  100.000323  100.000237  100.000173  100.000126  100.000092  100.000067  100.000049 |  100.0 |  50.102726 | 100.000049 |
    | 13:00 | 4.652878  4.511914  4.385471  4.273721  4.176392  4.092813  4.021996  3.962734  3.913701  3.873543  3.840953  3.814715  3.793741  3.777077  3.763909  3.753554  3.745445  3.739119  3.734202   3.73039  3.727444  3.725174  3.723428  3.722089  3.721064  3.720281  3.719684   3.71923  3.718886  3.718624  3.718427  3.718277  3.718165   3.71808  3.718016  3.717969  3.717933  3.717906  3.717886  3.717871   3.71786  3.717852  3.717846  3.717841  3.717838  3.717835  3.717833  3.717832  3.717831              3.71783 | 178.039511  169.465579  161.943847  155.429273  149.857128  145.147791  141.212209  137.957305  135.290772  133.124839  131.378898  129.981002  128.868455  127.987691  127.293694  126.749148  126.323478  125.991864  125.734328  125.534894  125.380866  125.262202  125.170999  125.101058  125.047538  125.006667  124.975518  124.951823  124.933833  124.920198  124.909883  124.902092  124.896218  124.891796  124.888473   124.88598  124.884111  124.882714   124.88167  124.880892  124.880312   124.87988   124.87956  124.879322  124.879146  124.879016   124.87892  124.878849  124.878796  124.878758 |  61.528776   60.119145    58.85471    57.73721   56.763919   55.928131   55.219962   54.627338   54.137006   53.735433   53.409531   53.147155   52.937407   52.770767   52.639091    52.53554   52.454452   52.391195   52.342015   52.303899   52.274442   52.251737    52.23428   52.220888   52.210638    52.20281   52.196842   52.192303   52.188856   52.186243   52.184266   52.182773   52.181648     52.1808   52.180163   52.179686   52.179328    52.17906    52.17886   52.178711   52.178599   52.178517   52.178455    52.17841   52.178376   52.178351   52.178333   52.178319   52.178309     52.178302 |  1.22769  1.207351  1.188923  1.172485  1.158048  1.145557  1.134905  1.125941  1.118488  1.112361  1.107371  1.103344  1.100117  1.097549  1.095516  1.093917  1.092663  1.091684  1.090923  1.090332  1.089876  1.089524  1.089253  1.089046  1.088887  1.088765  1.088673  1.088602  1.088549  1.088509  1.088478  1.088455  1.088437  1.088424  1.088414  1.088407  1.088401  1.088397  1.088394  1.088392   1.08839  1.088389  1.088388  1.088387  1.088387  1.088386  1.088386  1.088386  1.088385  1.088385 | 1.351648  1.352558  1.353431  1.354251  1.355006  1.355687  1.356288   1.35681  1.357255  1.357628  1.357938  1.358191  1.358396  1.358561  1.358693  1.358797  1.358879  1.358943  1.358993  1.359032  1.359062  1.359085  1.359103  1.359117  1.359127  1.359135  1.359141  1.359146   1.35915  1.359152  1.359154  1.359156  1.359157  1.359158  1.359159  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161  1.359161  1.359161          1.359161 | 0.598808  0.590588  0.582889  0.575811  0.569424  0.563764  0.558836  0.554615  0.551053  0.548088  0.545649  0.543663  0.542062   0.54078  0.539762  0.538958  0.538326  0.537831  0.537446  0.537147  0.536916  0.536737    0.5366  0.536494  0.536413  0.536352  0.536305  0.536269  0.536242  0.536221  0.536206  0.536194  0.536185  0.536178  0.536173  0.536169  0.536167  0.536164  0.536163  0.536162  0.536161   0.53616   0.53616  0.536159  0.536159  0.536159  0.536159  0.536159  0.536159      0.536158 | -0.255302  -0.247256  -0.239975  -0.233507  -0.227862  -0.223017  -0.218924  -0.215513  -0.212707  -0.210424  -0.208584  -0.207114  -0.205948  -0.205028  -0.204307  -0.203745  -0.203308   -0.20297  -0.202709  -0.202508  -0.202354  -0.202237  -0.202147  -0.202079  -0.202027  -0.201987  -0.201958  -0.201935  -0.201918  -0.201905  -0.201896  -0.201889  -0.201883  -0.201879  -0.201876  -0.201874  -0.201872  -0.201871   -0.20187   -0.20187  -0.201869  -0.201869  -0.201869  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.667548  0.667056  0.666668  0.666371   0.66615   0.66599  0.665878  0.665801  0.665751  0.665719    0.6657   0.66569  0.665685  0.665683  0.665684  0.665686  0.665689  0.665692  0.665694  0.665697  0.665699  0.665701  0.665702  0.665704  0.665705  0.665706  0.665706  0.665707  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 167.240349 | 161.711715  151.272237  142.259772  134.567702   128.07447   122.65011  118.162547  114.483072  111.490611  109.074652  107.136875  105.591689  104.365954  103.398136   102.63714  102.041014   101.57563  101.213448  100.932397  100.714887   100.54698  100.417671  100.318315  100.242138  100.183856  100.139355  100.105442  100.079647  100.060063  100.045221  100.033993  100.025513  100.019119  100.014306  100.010689  100.007975  100.005942  100.004421  100.003285  100.002437  100.001806  100.001337  100.000988   100.00073  100.000538  100.000396  100.000291  100.000214  100.000157          100.000115 | 100.000097 | 0.817462  0.803379  0.790606  0.779203  0.769179  0.760501  0.753096   0.74686  0.741673  0.737407  0.733932  0.731126  0.728878  0.727088  0.725672  0.724556  0.723682     0.723  0.722469  0.722057  0.721739  0.721494  0.721305   0.72116   0.72105  0.720965    0.7209  0.720851  0.720814  0.720786  0.720764  0.720748  0.720736  0.720727   0.72072  0.720715  0.720711  0.720708  0.720706  0.720704  0.720703  0.720702  0.720702  0.720701  0.720701    0.7207    0.7207    0.7207    0.7207         0.7207 | 3.167677  3.081677  3.004286   2.93568  2.875759  2.824174  2.780367  2.743637  2.713196  2.688231  2.667947    2.6516  2.638523  2.628127  2.619908  2.613442  2.608377  2.604425  2.601352  2.598969  2.597128  2.595709  2.594617   2.59378  2.593139   2.59265  2.592277  2.591993  2.591777  2.591614   2.59149  2.591397  2.591327  2.591274  2.591234  2.591204  2.591182  2.591165  2.591152  2.591143  2.591136  2.591131  2.591127  2.591124  2.591122  2.591121  2.591119  2.591119  2.591118        2.591118 | 167.240349  156.065508  146.367479  138.048506  130.992611  125.072104  120.154205  116.107075  112.804772  110.130849  107.980591  106.262028  104.895977  103.815405  102.964368  102.296727  101.774815  101.368143  101.052207  100.807442  100.618304  100.472509  100.360384  100.274343   100.20846  100.158114  100.119717   100.09049  100.068284  100.051444  100.038694  100.029059   100.02179  100.016314  100.012196  100.009105  100.006787  100.005053  100.003756  100.002789  100.002068  100.001531  100.001132  100.000836  100.000617  100.000455  100.000335  100.000246  100.000181  100.000132  100.000097 |  100.0 |  67.240349 | 100.000097 |
    | 13:30 | 4.904444  4.742605  4.594088  4.459717  4.339892  4.234571  4.143298  4.065263  3.999388  3.944427  3.899056  3.861959   3.83188  3.807674  3.788322  3.772937  3.760769  3.751188  3.743673    3.7378  3.733225  3.729671  3.726918  3.724791  3.723151   3.72189  3.720922   3.72018  3.719613  3.719181  3.718852  3.718601  3.718411  3.718267  3.718158  3.718076  3.718014  3.717967  3.717932  3.717906  3.717886  3.717871   3.71786  3.717852  3.717846  3.717841  3.717838  3.717835  3.717833             3.717832 | 193.834501  183.600558  174.439545  166.341158  159.271694  153.176519  147.984054  143.610741  139.966339  136.958914  134.499049  132.502995  130.894737  129.607043  128.581727  127.769336  127.128459  126.624861  126.230527  125.922736    125.6832  125.497287  125.353357  125.242195  125.156534  125.090666  125.040121  125.001413  124.971825  124.949251  124.932059   124.91899  124.909071  124.901556  124.895872   124.89158  124.888343  124.885906  124.884075    124.8827   124.88167  124.880899  124.880323  124.879893  124.879572  124.879334  124.879156  124.879025  124.878927  124.878855 |  64.044438   62.426045    60.94088   59.597174    58.39892   57.345709   56.432979   55.652626   54.993879   54.444268   53.990564   53.619585   53.318803   53.076745   52.883216   52.729372   52.607691   52.511877   52.436729   52.377999   52.332246   52.296709    52.26918   52.247908    52.23151   52.218898   52.209218   52.201803   52.196135    52.19181   52.188516   52.186011   52.184111   52.182671   52.181581   52.180759   52.180139   52.179672   52.179321   52.179057    52.17886   52.178712   52.178602   52.178519   52.178458   52.178412   52.178378   52.178353   52.178334      52.17832 |  1.26348  1.240528  1.219233  1.199765  1.182236  1.166691  1.153112  1.141421  1.131491  1.123162  1.116256  1.110589   1.10598  1.102261  1.099282   1.09691  1.095031  1.093551  1.092389   1.09148  1.090771  1.090221  1.089794  1.089465   1.08921  1.089015  1.088865   1.08875  1.088662  1.088595  1.088544  1.088505  1.088475  1.088453  1.088436  1.088423  1.088414  1.088407  1.088401  1.088397  1.088394  1.088392   1.08839  1.088389  1.088388  1.088387  1.088387  1.088386  1.088386  1.088386 | 1.350174  1.351102   1.35202  1.352911  1.353759   1.35455  1.355272  1.355918  1.356485  1.356975   1.35739  1.357738  1.358025   1.35826   1.35845  1.358603  1.358724  1.358821  1.358897  1.358956  1.359003  1.359039  1.359067  1.359089  1.359106  1.359119  1.359129  1.359136  1.359142  1.359147   1.35915  1.359153  1.359155  1.359156  1.359157  1.359158  1.359159  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359161          1.359161 | 0.612605  0.603852  0.595424  0.587449  0.580035  0.573268  0.567203  0.561862  0.557236  0.553293  0.549977  0.547224  0.544964  0.543127  0.541646  0.540461  0.539519  0.538774  0.538187  0.537728   0.53737  0.537091  0.536874  0.536707  0.536578  0.536479  0.536402  0.536344  0.536299  0.536265  0.536239  0.536219  0.536204  0.536193  0.536184  0.536178  0.536173  0.536169  0.536167  0.536164  0.536163  0.536162  0.536161   0.53616   0.53616  0.536159  0.536159  0.536159  0.536159      0.536159 | -0.269452  -0.260436  -0.252034  -0.244345  -0.237434  -0.231333  -0.226036  -0.221511  -0.217701  -0.214536  -0.211936  -0.209823   -0.20812   -0.20676   -0.20568  -0.204827  -0.204157  -0.203634  -0.203227   -0.20291  -0.202666  -0.202477  -0.202332  -0.202221  -0.202136  -0.202071  -0.202022  -0.201984  -0.201955  -0.201934  -0.201917  -0.201905  -0.201896  -0.201889  -0.201883  -0.201879  -0.201876  -0.201874  -0.201873  -0.201871   -0.20187   -0.20187  -0.201869  -0.201869  -0.201869  -0.201868  -0.201868  -0.201868  -0.201868     -0.201868 | 0.668576  0.667909  0.667357  0.666914  0.666566  0.666302  0.666107  0.665965  0.665866  0.665798  0.665753  0.665724  0.665706  0.665696  0.665691  0.665689   0.66569  0.665691  0.665693  0.665695  0.665697  0.665699    0.6657  0.665702  0.665703  0.665704  0.665705  0.665706  0.665707  0.665707  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 188.127569 | 181.388537  168.574811  157.307237  147.511772   139.09169  131.932884  125.909965  120.892496   116.75072  113.360322  110.605956  108.383484  106.601063  105.179283  104.050638  103.158548  102.456162  101.905076  101.474078  101.137989  100.876624  100.673885     100.517  100.395874  100.302559  100.230821  100.175781  100.133634  100.101421  100.076847  100.058132  100.043906  100.033109   100.02493  100.018743   100.01407  100.010548  100.007896  100.005902  100.004406  100.003285  100.002446  100.001818   100.00135  100.001002  100.000742  100.000549  100.000406  100.000299          100.000221 | 100.000187 | 0.842212  0.826345  0.811608  0.798122  0.785968  0.775181  0.765751  0.757626  0.750721  0.744926  0.740119  0.736173  0.732963  0.730372  0.728296  0.726643  0.725334  0.724301  0.723491  0.722857  0.722364   0.72198  0.721682  0.721452  0.721275  0.721139  0.721034  0.720954  0.720893  0.720846   0.72081  0.720783  0.720763  0.720747  0.720735  0.720726   0.72072  0.720715  0.720711  0.720708  0.720706  0.720704  0.720703  0.720702  0.720702  0.720701  0.720701    0.7207    0.7207         0.7207 | 3.320475  3.222272  3.131845  3.049759  2.976328  2.911596  2.855348  2.807142  2.766363  2.732277  2.704096  2.681023  2.662296  2.647212  2.635143  2.625543  2.617947  2.611964   2.60727    2.6036  2.600741   2.59852  2.596799  2.595469  2.594444  2.593656  2.593051  2.592587  2.592233  2.591962  2.591756    2.5916  2.591481  2.591391  2.591323  2.591271  2.591232  2.591203  2.591181  2.591165  2.591152  2.591143  2.591136  2.591131  2.591127  2.591124  2.591122  2.591121   2.59112        2.591119 | 188.127569  174.526271  162.503058  151.996314  142.919284  135.164843  128.611388  123.129211  118.586652  114.855454   111.81495  109.354927  107.377238  105.796334  104.538991  103.543496  102.758501  102.141753  101.658794   101.28175  100.988218  100.760298  100.583759  100.447336  100.342145   100.26121  100.199063  100.151438   100.11501    100.0872  100.066006  100.049884  100.037641  100.028358  100.021333  100.016024  100.012019  100.009002  100.006733  100.005029  100.003751  100.002794  100.002079  100.001544  100.001146  100.000849  100.000629  100.000465  100.000343  100.000253  100.000187 |  100.0 |  88.127569 | 100.000187 |
    | 14:00 | 5.185446  5.005433  4.836794   4.68085  4.538602  4.410661  4.297215  4.198027   4.11248  4.039649  3.978385  3.927418  3.885439  3.851168  3.823409  3.801079  3.783226  3.769028  3.757789  3.748929  3.741971  3.736524  3.732274  3.728965  3.726397  3.724408  3.722872  3.721687  3.720775  3.720075  3.719538  3.719128  3.718814  3.718575  3.718393  3.718255   3.71815   3.71807   3.71801  3.717965  3.717931  3.717905  3.717886  3.717871   3.71786  3.717852  3.717846  3.717841  3.717838             3.717835 | 212.225932  200.353261  189.524779  179.764544  171.073573  163.429584  156.788497  151.087553  146.249683  142.188529  138.813508  136.034352  133.764756  131.924979  130.443403  129.257218  128.312409  127.563287  126.971733  126.506303  126.141311   125.85594  125.633437  125.460395  125.326142  125.222219  125.141947   125.08007  125.032468  124.995918  124.967905  124.946474  124.930107  124.917629  124.908132  124.900916  124.895441  124.891294  124.888158   124.88579  124.884005   124.88266   124.88165  124.880891  124.880322  124.879896  124.879578   124.87934  124.879163  124.879031 |  66.854459   65.054325   63.367938   61.808502    60.38602   59.106613   57.972148   56.980269   56.124804   55.396485   54.783849   54.274184   53.854388   53.511677   53.234087   53.010795   52.832265   52.690281   52.577891   52.489295   52.419712   52.365244   52.322736   52.289654   52.263973   52.244085   52.228718   52.216869   52.207752   52.200751   52.195384   52.191278   52.188142   52.185751   52.183931   52.182548   52.181499   52.180704   52.180103   52.179649   52.179307    52.17905   52.178856    52.17871   52.178601    52.17852   52.178459   52.178413   52.178379     52.178354 | 1.302742  1.277674  1.253918  1.231701  1.211218  1.192608  1.175953  1.161267  1.148504  1.137567  1.128313  1.120577  1.114178  1.108937  1.104679  1.101246  1.098497  1.096307  1.094571  1.093202  1.092125  1.091282  1.090624  1.090111  1.089714  1.089405  1.089167  1.088983  1.088842  1.088733   1.08865  1.088587  1.088538  1.088501  1.088473  1.088451  1.088435  1.088423  1.088413  1.088406  1.088401  1.088397  1.088394  1.088392   1.08839  1.088389  1.088388  1.088387  1.088387  1.088386 | 1.348723  1.349631  1.350553  1.351475   1.35238  1.353252  1.354075  1.354835  1.355524  1.356136   1.35667  1.357129  1.357517   1.35784  1.358107  1.358324    1.3585  1.358642  1.358754  1.358843  1.358914  1.358969  1.359013  1.359046  1.359073  1.359093  1.359109  1.359121   1.35913  1.359137  1.359143  1.359147   1.35915  1.359153  1.359155  1.359156  1.359157  1.359158  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916           1.35916 | 0.626833  0.617854  0.608998  0.600395  0.592173  0.584449  0.577322  0.570862  0.565111  0.560076  0.555739  0.552056  0.548971  0.546416  0.544323  0.542623  0.541254  0.540159  0.539287  0.538598  0.538054  0.537628  0.537295  0.537035  0.536833  0.536677  0.536556  0.536463  0.536391  0.536336  0.536293  0.536261  0.536236  0.536217  0.536203  0.536192  0.536184  0.536177  0.536173  0.536169  0.536166  0.536164  0.536163  0.536162  0.536161   0.53616   0.53616  0.536159  0.536159      0.536159 | -0.284696   -0.27501  -0.265738   -0.25701  -0.248935  -0.241596  -0.235043  -0.229292  -0.224326  -0.220101  -0.216556  -0.213619  -0.211211  -0.209256  -0.207681  -0.206423  -0.205423  -0.204633  -0.204012  -0.203526  -0.203146  -0.202851  -0.202623  -0.202446   -0.20231  -0.202205  -0.202125  -0.202063  -0.202016   -0.20198  -0.201953  -0.201932  -0.201916  -0.201904  -0.201895  -0.201888  -0.201883  -0.201879  -0.201876  -0.201874  -0.201873  -0.201871   -0.20187   -0.20187  -0.201869  -0.201869  -0.201869  -0.201868  -0.201868     -0.201868 | 0.669873  0.669028    0.6683  0.667687  0.667185  0.666786  0.666475   0.66624  0.666067  0.665942  0.665854  0.665794  0.665754  0.665728  0.665711  0.665702  0.665697  0.665694  0.665694  0.665695  0.665696  0.665697  0.665699    0.6657  0.665702  0.665703  0.665704  0.665705  0.665706  0.665707  0.665707  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 212.993962 | 204.990599  189.671339  175.963815  163.832994  153.216772  144.028948  136.163738   129.50135  123.914021  119.271883  115.448125  112.323077  109.787055  107.742019  106.102187   104.79386  103.754685  102.932592  102.284577  101.775447  101.376631  101.065089  100.822347  100.633667  100.487342  100.374111  100.286671  100.219282  100.167448  100.127651  100.097154  100.073823  100.056007  100.042425  100.032087  100.024233  100.018274   100.01376  100.010347  100.007769  100.005826  100.004363  100.003263  100.002437  100.001818  100.001354  100.001008  100.000749  100.000556          100.000412 | 100.000349 | 0.869317  0.852016  0.835603  0.820238  0.806057  0.793161  0.781609  0.771415  0.762549  0.754946   0.74851  0.743127  0.738673  0.735022  0.732057  0.729665  0.727749  0.726223  0.725013  0.724058  0.723308   0.72272  0.722261  0.721904  0.721626  0.721411  0.721245  0.721117  0.721018  0.720943  0.720885   0.72084  0.720806   0.72078  0.720761  0.720746  0.720734  0.720726  0.720719  0.720714  0.720711  0.720708  0.720706  0.720704  0.720703  0.720702  0.720702  0.720701  0.720701         0.7207 | 3.490217  3.381586  3.279467  3.184709  3.097981  3.019724   2.95012  2.889092  2.836324  2.791295  2.753344  2.721717   2.69563  2.674306  2.657018    2.6431  2.631964  2.623103  2.616087  2.610553  2.606207  2.602803  2.600147  2.598079  2.596474   2.59523   2.59427  2.593529  2.592959  2.592521  2.592186  2.591929  2.591733  2.591583   2.59147  2.591383  2.591317  2.591268   2.59123  2.591202   2.59118  2.591164  2.591152  2.591143  2.591136  2.591131  2.591127  2.591124  2.591122        2.591121 | 212.993962  196.862863  182.355112  169.449781  158.097711  148.223648  139.730091  132.502505  126.415323  121.338079  117.141064  113.700022  110.899629  108.635711  106.816314  105.361848  104.204564  103.287593  102.563763  101.994334  101.547757  101.198523  100.926135  100.714209  100.549707  100.422299  100.323829   100.24788  100.189414  100.144494  100.110044  100.083672  100.063519  100.048144  100.036435  100.027532  100.020774  100.015652  100.011776  100.008847  100.006638  100.004973  100.003721  100.002781  100.002075  100.001547  100.001151  100.000856  100.000636  100.000472  100.000349 |  100.0 | 112.993962 | 100.000349 |
    | 14:30 | 5.491152  5.296408  5.110636  4.935443  4.772257  4.622234  4.486187  4.364523  4.257231  4.163901  4.083776  4.015834  3.958878  3.911625  3.872787   3.84113  3.815513  3.794917   3.77845   3.76535  3.754973  3.746786  3.740347    3.7353  3.731355   3.72828  3.725888  3.724031  3.722594  3.721483  3.720627  3.719967  3.719461  3.719072  3.718774  3.718547  3.718373  3.718241   3.71814  3.718064  3.718006  3.717962  3.717929  3.717904  3.717885  3.717871   3.71786  3.717852  3.717846             3.717841 | 233.131042  219.705796  207.252564  195.824656  185.456014  176.158765  167.922157   160.71313  154.478554  149.148886  144.642794   140.87214  137.746739  135.178436  133.084223  131.388337  130.023386  128.930713  128.060177   127.36955  126.823721  126.393794  126.056198   125.79185  125.585394  125.424541  125.299504  125.202515  125.127438  125.069437  125.024713  124.990289  124.963842  124.943558  124.928027  124.916156  124.907096  124.900193  124.894942  124.890954  124.887929  124.885638  124.883906  124.882598  124.881612  124.880869  124.880311  124.879891  124.879577  124.879341 |  69.911522   67.964078   66.106363   64.354434   62.722566   61.222343   59.861867   58.645227   57.572312   56.639012   55.837764   55.158343   54.588779   54.116252   53.727874   53.411298   53.155129   52.949167   52.784501   52.653499   52.549733   52.467856   52.403473   52.353004   52.313554   52.282797   52.258876   52.240313    52.22594   52.214833   52.206266   52.199672   52.194605   52.190719   52.187743   52.185468   52.183732    52.18241   52.181403   52.180639   52.180059    52.17962   52.179288   52.179038   52.178849   52.178706   52.178599   52.178519   52.178459     52.178413 | 1.344664   1.31805   1.29236  1.267848  1.244753  1.223287  1.203616  1.185852  1.170047  1.156187  1.144202  1.133975  1.125356  1.118172  1.112245  1.107398  1.103466  1.100298   1.09776  1.095739  1.094136   1.09287  1.091874  1.091093  1.090482  1.090005  1.089635  1.089347  1.089124  1.088952  1.088819  1.088717  1.088638  1.088578  1.088532  1.088497   1.08847  1.088449  1.088433  1.088422  1.088413  1.088406  1.088401  1.088397  1.088394  1.088392   1.08839  1.088389  1.088388  1.088387 | 1.347342    1.3482  1.349091  1.350004  1.350926   1.35184  1.352731  1.353581  1.354376  1.355106  1.355762  1.356342  1.356844  1.357274  1.357635  1.357936  1.358183  1.358385  1.358548  1.358678  1.358782  1.358865   1.35893  1.358982  1.359022  1.359053  1.359078  1.359097  1.359112  1.359123  1.359132  1.359139  1.359144  1.359148  1.359151  1.359153  1.359155  1.359156  1.359157  1.359158  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916           1.35916 | 0.641069  0.632141  0.623159  0.614233  0.605488  0.597052  0.589047  0.581583  0.574744  0.568588  0.563142  0.558401  0.554337  0.550901  0.548032  0.545662  0.543724  0.542152  0.540886  0.539874  0.539068   0.53843  0.537927  0.537532  0.537223  0.536981  0.536793  0.536647  0.536534  0.536446  0.536379  0.536327  0.536287  0.536256  0.536233  0.536215  0.536201  0.536191  0.536183  0.536177  0.536172  0.536169  0.536166  0.536164  0.536163  0.536162  0.536161   0.53616   0.53616      0.536159 | -0.300565  -0.290531  -0.280698  -0.271203  -0.262177  -0.253742  -0.245993  -0.238999  -0.232794  -0.227378  -0.222725  -0.218783  -0.215486  -0.212761  -0.210531  -0.208723  -0.207268  -0.206104  -0.205179  -0.204448  -0.203872  -0.203421  -0.203068  -0.202793   -0.20258  -0.202415  -0.202287  -0.202188  -0.202113  -0.202054   -0.20201  -0.201976   -0.20195   -0.20193  -0.201915  -0.201903  -0.201894  -0.201888  -0.201883  -0.201879  -0.201876  -0.201874  -0.201872  -0.201871   -0.20187   -0.20187  -0.201869  -0.201869  -0.201869     -0.201868 | 0.671426  0.670417  0.669514  0.668723  0.668048  0.667487  0.667031  0.666671  0.666394  0.666185  0.666031  0.665921  0.665843  0.665789  0.665753   0.66573  0.665715  0.665706  0.665701  0.665698  0.665697  0.665698  0.665698  0.665699  0.665701  0.665702  0.665703  0.665704  0.665705  0.665706  0.665706  0.665707  0.665707  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 241.948195 | 232.667532  214.792455  198.537301  183.907422  170.880492  159.406865  149.411655  140.798425   133.45415  127.254946  122.071945  117.776757  114.246046  111.364981  109.029474  107.147326  105.638454  104.434451  103.477699  102.720251  102.122599  101.652469  101.283687  100.995154  100.769953  100.594584  100.458314  100.352646  100.270869  100.207703  100.159003  100.121524  100.092731  100.070649  100.053743  100.040821   100.03096  100.023446  100.017731  100.013389  100.010097  100.007604  100.005718  100.004295  100.003221  100.002413  100.001805  100.001349  100.001007           100.00075 | 100.000638 | 0.898211  0.879873  0.862154  0.845229  0.829266  0.814414  0.800791  0.788477  0.777511  0.767887  0.759559  0.752449  0.746453  0.741453  0.737327  0.733951  0.731212  0.729004  0.727236  0.725827  0.724709  0.723827  0.723132  0.722588  0.722162  0.721829  0.721571   0.72137  0.721215  0.721095  0.721002  0.720931  0.720876  0.720834  0.720802  0.720777  0.720759  0.720744  0.720733  0.720725  0.720719  0.720714  0.720711  0.720708  0.720706  0.720704  0.720703  0.720702  0.720702       0.720701 | 3.673884  3.556996  3.445118  3.339247   3.24029  3.149006   3.06595   2.99144  2.925539  2.868057  2.818589  2.776551  2.741245  2.711907  2.687761  2.668057  2.652097  2.639256  2.628984  2.620807  2.614328  2.609214  2.605192  2.602038  2.599573   2.59765  2.596155  2.594995  2.594096  2.593402  2.592866  2.592454  2.592137  2.591894  2.591708  2.591566  2.591457  2.591374  2.591311  2.591264  2.591227    2.5912  2.591179  2.591164  2.591152  2.591143  2.591136  2.591131  2.591127        2.591125 | 241.948195  223.265786  206.194716  190.754186  176.935898  164.703545  153.994023  144.720383  136.776288  130.041536  124.388048  119.685701  115.807446  112.633372  110.053543  107.969635  106.295557  104.957274  103.892079  103.047541  102.380288  101.854776  101.442092  101.118877  100.866363  100.669545  100.516476  100.397683  100.305675  100.234552  100.179676  100.137413  100.104923  100.079988  100.060884  100.046273  100.035115  100.026609  100.020134  100.015213  100.011478  100.008648  100.006508   100.00489   100.00367   100.00275  100.002059  100.001539  100.001149  100.000857  100.000638 |  100.0 | 141.948195 | 100.000638 |
    | 15:00 | 5.816019  5.610333  5.411061  5.219888  5.038455  4.868277  4.710651  4.566567  4.436649  4.321112   4.21976  4.132028  4.057039  3.993695  3.940764  3.896967  3.861042  3.831802  3.808164  3.789169  3.773983  3.761899  3.752322  3.744759  3.738805  3.734132  3.730474  3.727618  3.725393  3.723663  3.722321  3.721281  3.720478  3.719859  3.719381  3.719014  3.718733  3.718517  3.718352  3.718226   3.71813  3.718057  3.718001  3.717959  3.717926  3.717902  3.717884   3.71787   3.71786             3.717852 | 256.370661  241.534161  227.563825  214.534455  202.506968  191.524781  181.610905   172.76619   164.96903  158.176711   152.32829  147.348675  143.153387  139.653406  136.759586   134.38626  132.453864  130.890564  129.633029  128.626518  127.824488  127.187909  126.684424  126.287465  125.975393  125.730703  125.539313  125.389955  125.273649  125.183265  125.113164  125.058896  125.016962  124.984615  124.959707  124.940559  124.925864  124.914603  124.905989  124.899409   124.89439  124.890569  124.887663  124.885456  124.883783  124.882516  124.881559  124.880835   124.88029  124.879879 |  73.160189   71.103328   69.110611   67.198877    65.38455   63.682773   62.106506   60.665672   59.366494   58.211118   57.197603   56.320276   55.570385   54.936947   54.407644   53.969669   53.610421    53.31802   53.081641   52.891685   52.739831   52.618991   52.523218   52.447586   52.388051   52.341323   52.304744   52.276181   52.253928   52.236628   52.223206   52.212814   52.204782   52.198585   52.193813   52.190145   52.187329   52.185171    52.18352   52.182259   52.181297   52.180565   52.180008   52.179585   52.179265   52.179022   52.178838     52.1787   52.178595     52.178517 | 1.388382  1.360798  1.333757  1.307505  1.282295  1.258373  1.235966  1.215262  1.196403  1.179473  1.164495  1.151429  1.140184   1.13063  1.122606  1.115938  1.110449  1.105968  1.102336  1.099412  1.097071  1.095206  1.093726  1.092557  1.091635  1.090912  1.090345  1.089903  1.089558   1.08929  1.089082  1.088921  1.088796    1.0887  1.088626  1.088569  1.088525  1.088492  1.088466  1.088447  1.088432   1.08842  1.088412  1.088405    1.0884  1.088397  1.088394  1.088392   1.08839  1.088389 | 1.346064  1.346853  1.347686  1.348558  1.349458  1.350375  1.351293  1.352197   1.35307  1.353897  1.354665  1.355364  1.355988  1.356535  1.357008  1.357409  1.357746  1.358026  1.358255  1.358442  1.358592  1.358713  1.358809  1.358886  1.358946  1.358994  1.359031   1.35906  1.359083  1.359101  1.359114  1.359125  1.359133   1.35914  1.359145  1.359148  1.359151  1.359153  1.359155  1.359156  1.359157  1.359158  1.359159  1.359159   1.35916   1.35916   1.35916   1.35916   1.35916           1.35916 | 0.654955  0.646302  0.637455  0.628499  0.619537  0.610686  0.602072  0.593819  0.586044  0.578845  0.572296   0.56644   0.56129  0.556832  0.553027  0.549823  0.547156  0.544958  0.543164  0.541711  0.540542  0.539606  0.538862  0.538272  0.537807  0.537441  0.537154  0.536929  0.536754  0.536618  0.536512  0.536431  0.536367  0.536318  0.536281  0.536252   0.53623  0.536213    0.5362   0.53619  0.536182  0.536176  0.536172  0.536169  0.536166  0.536164  0.536163  0.536162  0.536161       0.53616 | -0.316604  -0.306523  -0.296448  -0.286502  -0.276818  -0.267529  -0.258762  -0.250626  -0.243204  -0.236548  -0.230679  -0.225584  -0.221228  -0.217551  -0.214487   -0.21196  -0.209896  -0.208224  -0.206879  -0.205803  -0.204948  -0.204272  -0.203739   -0.20332  -0.202993  -0.202737  -0.202538  -0.202384  -0.202264  -0.202172    -0.2021  -0.202045  -0.202003  -0.201971  -0.201946  -0.201927  -0.201913  -0.201902  -0.201894  -0.201887  -0.201882  -0.201879  -0.201876  -0.201874  -0.201872  -0.201871   -0.20187   -0.20187  -0.201869     -0.201869 | 0.673206  0.672057  0.670996  0.670035  0.669182  0.668444   0.66782  0.667306  0.666893  0.666569  0.666321  0.666135  0.665999  0.665901  0.665832  0.665785  0.665753  0.665731  0.665717  0.665709  0.665704  0.665701    0.6657    0.6657  0.665701  0.665701  0.665702  0.665703  0.665704  0.665705  0.665706  0.665706  0.665707  0.665707  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 |  274.95935 | 264.430599  244.035018  225.211739  208.003529  192.428073  178.476126  166.111112  155.270336  145.867836  137.798666  130.944188  125.177833  120.370742  116.396771  113.136498  110.480072  108.328922  106.596446  105.207928  104.099878   103.21905  102.521267  101.970216  101.536282  101.195468  100.928442  100.719705  100.556886  100.430143  100.331676  100.255323  100.196225  100.150564  100.115346   100.08823  100.067385  100.051388  100.039131  100.029754  100.022592   100.01713   100.01297  100.009807  100.007406  100.005585  100.004206  100.003163  100.002376  100.001783          100.001336 |  100.00114 | 0.928295  0.909319  0.890698  0.872602  0.855206  0.838683  0.823188  0.808858  0.795792  0.784052  0.773656  0.764581  0.756766  0.750122  0.744539  0.739898  0.736075  0.732954  0.730424  0.728387  0.726755  0.725455  0.724424  0.723608  0.722966  0.722462  0.722067  0.721758  0.721517  0.721331  0.721185  0.721073  0.720986  0.720919  0.720868  0.720828  0.720797  0.720774  0.720756  0.720743  0.720732  0.720724  0.720718  0.720714   0.72071  0.720708  0.720706  0.720704  0.720703       0.720702 | 3.868052  3.745231  3.625859  3.510959  3.401542  3.298558  3.202842  3.115054   3.03564  2.964799  2.902478  2.848392  2.802055  2.762834  2.730004  2.702797  2.680453  2.662247  2.647517  2.635671  2.626196  2.618653  2.612672  2.607948  2.604228  2.601308  2.599022  2.597237  2.595846  2.594764  2.593925  2.593275  2.592773  2.592386  2.592087  2.591858  2.591682  2.591547  2.591444  2.591365  2.591305  2.591259  2.591224  2.591198  2.591178  2.591163  2.591151  2.591142  2.591136        2.591131 |  274.95935  253.788045  234.162402  216.137285   199.74369   184.98623  171.841922  160.260553  150.166742    141.4636   134.03767  127.764633  122.515181  118.160473  114.576721  111.648633  109.271667  107.353167  105.812603  104.581124  103.600689  102.822938  102.207976  101.723165  101.341996  101.043061  100.809168  100.626567  100.484308  100.373701  100.287867  100.221382  100.169977  100.130301   100.09973  100.076215  100.058157  100.044312  100.033713  100.025612  100.019431   100.01472  100.011137  100.008414  100.006348  100.004784    100.0036  100.002706  100.002031  100.001522   100.00114 |  100.0 |  174.95935 |  100.00114 |
    | 15:30 |  6.15413  5.941307  5.732419  5.529068   5.33292  5.145634  4.968792  4.803806  4.651831  4.513692  4.389826  4.280264  4.184642  4.102257  4.032133  3.973114  3.923949  3.883367  3.850144  3.823141  3.801332  3.783815  3.769813  3.758669  3.749832  3.742849  3.737346  3.733023  3.729634  3.726984  3.724916  3.723306  3.722054  3.721083  3.720331   3.71975  3.719302  3.718956  3.718691  3.718486   3.71833   3.71821  3.718118  3.718048  3.717995  3.717955  3.717924    3.7179  3.717883             3.717869 | 281.678565  265.615274  250.289427  235.789017  222.193948  209.572244  197.976334  187.439819  177.975134  169.572478   162.20028  155.807248  150.325792  145.676411   141.77248  138.524896  135.846099  133.653219  131.870231  130.429177  129.270621  128.343513  127.604665     127.018  126.553695  126.187314   125.89898  125.672629  125.495345  125.356791  125.248726  125.164604  125.099243   125.04855  125.009302  124.978966  124.955558  124.937524  124.923654  124.913001  124.904833   124.89858  124.893799   124.89015  124.887368   124.88525   124.88364  124.882419  124.881493  124.880791 |  76.541305   74.413073   72.324191   70.290685   68.329196   66.456338   64.687918   63.038055   61.518312   60.136923   58.898265   57.802638   56.846423   56.022569    55.32133   54.731142   54.239487   53.833673   53.501445   53.231415   53.013323   52.838152   52.698134    52.58669   52.498323   52.428487   52.373463   52.330226   52.296337   52.269837   52.249158   52.233055   52.220541   52.210832   52.203314   52.197503   52.193018   52.189563   52.186905   52.184864   52.183299     52.1821   52.181184   52.180485   52.179952   52.179546   52.179237   52.179003   52.178826     52.178691 | 1.433046  1.405028   1.37721  1.349809  1.323064  1.297223  1.272536  1.249238   1.22754  1.207609   1.18956  1.173451  1.159276  1.146973  1.136434  1.127514  1.120049  1.113862   1.10878  1.104638  1.101285  1.098588  1.096428  1.094707  1.093341  1.092261  1.091409   1.09074  1.090215  1.089804  1.089484  1.089234   1.08904   1.08889  1.088773  1.088683  1.088614   1.08856  1.088519  1.088487  1.088463  1.088444   1.08843  1.088419  1.088411  1.088405    1.0884  1.088396  1.088393  1.088391 | 1.344905  1.345615  1.346376  1.347184  1.348034  1.348917  1.349825  1.350742  1.351655  1.352546    1.3534  1.354202  1.354941  1.355608  1.356201  1.356717  1.357161  1.357536   1.35785  1.358109  1.358322  1.358494  1.358634  1.358745  1.358834  1.358905  1.358961  1.359005   1.35904  1.359067  1.359088  1.359104  1.359117  1.359127  1.359135  1.359141  1.359145  1.359149  1.359152  1.359154  1.359155  1.359157  1.359158  1.359158  1.359159  1.359159   1.35916   1.35916   1.35916           1.35916 | 0.668223  0.660004  0.651494  0.642752  0.633852  0.624887  0.615968  0.607213  0.598748  0.590694   0.58316  0.576233  0.569974  0.564412  0.559549  0.555361  0.551803  0.548817  0.546339  0.544303  0.542643    0.5413  0.540219  0.539356  0.538668  0.538123  0.537693  0.537354  0.537088  0.536879  0.536717   0.53659  0.536491  0.536415  0.536356   0.53631  0.536275  0.536247  0.536226   0.53621  0.536198  0.536188  0.536181  0.536176  0.536171  0.536168  0.536166  0.536164  0.536163      0.536162 | -0.332421  -0.322542  -0.312511  -0.302427  -0.292403  -0.282567  -0.273051  -0.263985  -0.255487  -0.247655  -0.240558  -0.234235  -0.228691  -0.223904  -0.219828  -0.216402  -0.213555  -0.211213  -0.209303  -0.207757  -0.206514  -0.205521  -0.204731  -0.204105  -0.203612  -0.203224   -0.20292  -0.202682  -0.202497  -0.202353  -0.202241  -0.202155  -0.202088  -0.202037  -0.201997  -0.201966  -0.201943  -0.201925  -0.201911  -0.201901  -0.201893  -0.201887  -0.201882  -0.201879  -0.201876  -0.201874  -0.201872  -0.201871   -0.20187      -0.20187 | 0.675167  0.673912  0.672721   0.67161  0.670592  0.669678  0.668876  0.668188  0.667613  0.667143  0.666768  0.666477  0.666256  0.666091   0.66597  0.665883  0.665822   0.66578  0.665751  0.665732  0.665719  0.665711  0.665707  0.665704  0.665703  0.665702  0.665702  0.665703  0.665704  0.665704  0.665705  0.665706  0.665706  0.665707  0.665707  0.665708  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 311.845554 | 300.138987  277.342974  256.021361   236.24848  218.078227  201.540826  186.640485  173.354306  161.632684  151.401316  142.564733  135.011056  128.617495   123.25605  118.798833  115.122594  112.112173    109.6628  107.681311  106.086471   104.80862  103.788853  102.977958  102.335223  101.827259  101.426875   101.11206  100.865093  100.671768  100.520742  100.402988  100.311349  100.240162  100.184958  100.142223  100.109196  100.083712  100.064081  100.048982  100.037387  100.028497   100.02169  100.016486  100.012514  100.009486  100.007181   100.00543    100.0041  100.003092          100.002329 | 100.001993 | 0.958983  0.939738  0.920611  0.901754  0.883329  0.865509  0.848467  0.832368  0.817358  0.803557  0.791048  0.779874  0.770033  0.761485  0.754159  0.747955  0.742759  0.738452  0.734913  0.732028  0.729692  0.727812  0.726307  0.725107  0.724155  0.723402  0.722809  0.722342  0.721976  0.721689  0.721466  0.721292  0.721157  0.721052   0.72097  0.720908  0.720859  0.720822  0.720793  0.720771  0.720754  0.720741  0.720731  0.720723  0.720718  0.720713   0.72071  0.720707  0.720706       0.720704 |  4.06916  3.942683  3.818178  3.696597  3.578941  3.466224  3.359428  3.259448  3.167039  3.082764  3.006956  2.939702  2.880845  2.830009  2.786643  2.750075  2.719562  2.694341  2.673669  2.656851  2.643257  2.632331  2.623594  2.616636  2.611117  2.606755  2.603317  2.600615  2.598497   2.59684  2.595548  2.594541  2.593758  2.593152  2.592681  2.592318  2.592038  2.591822  2.591655  2.591528   2.59143  2.591355  2.591298  2.591254  2.591221  2.591195  2.591176  2.591161   2.59115        2.591142 | 311.845554  288.329231  266.245927  245.679473  226.695067  209.335647  193.618999   179.53593  167.049869  156.098053  146.594362  138.433579  131.496681  125.656616  120.783968  116.751992  113.440644   110.73946  108.549267  106.782892  105.365075  104.231821  103.329409  102.613226  102.046561  101.599435  101.247523  100.971201   100.75471  100.585447  100.453371  100.350508  100.270543  100.208489  100.160418  100.123241  100.094537   100.07241  100.055381  100.042296  100.032256  100.024565  100.018682  100.014188  100.010761  100.008151  100.006166  100.004658  100.003514  100.002648  100.001993 |  100.0 | 211.845554 | 100.001993 |
    | 16:00 | 6.499555  6.283185    6.0685  5.856924  5.650004  5.449372  5.256691  5.073587  4.901566  4.741927   4.59568  4.463477  4.345576  4.241833  4.151734  4.074454  4.008937  3.953986  3.908342  3.870754  3.840038  3.815106  3.794987  3.778836  3.765927  3.755652    3.7475  3.741054  3.735971  3.731973  3.728835  3.726378  3.724459  3.722961  3.721796   3.72089  3.720188  3.719644  3.719223  3.718898  3.718648  3.718455  3.718307  3.718194  3.718107   3.71804  3.717989   3.71795  3.717921             3.717898 | 308.714417  291.639808  275.160954   259.37168   244.36281  230.218847  217.014346  204.810209  193.650224   183.55827  174.536561  166.565283  159.603778  153.593218  148.460468  144.122668  140.491954  137.479831  135.000792  132.975006  131.330049  130.001763  128.934444  128.080524  127.399945  126.859364  126.431289  126.093239  125.826953  125.617683  125.453576  125.325148  125.224835  125.146628  125.085762  125.038475  125.001798  124.973397  124.951439   124.93449  124.921427  124.911374  124.903649  124.897722   124.89318  124.889706  124.887051  124.885026  124.883482  124.882308 |  79.995552   77.831845   75.684999   73.569244   71.500043   69.493719   67.566907   65.735867   64.015658   62.419271   60.956798   59.634769   58.455757   57.418328   56.517337   55.744537   55.089374   54.539863   54.083416   53.707543   53.400384   53.151061   52.949871   52.788355   52.659272   52.556515   52.475002   52.410541   52.359708   52.319725    52.28835   52.263782   52.244585   52.229614   52.217959   52.208902   52.201877   52.196436   52.192229   52.188982   52.186478   52.184552   52.183072   52.181936   52.181066     52.1804   52.179891   52.179503   52.179207     52.178982 | 1.477863  1.449883  1.421811   1.39383  1.366144   1.33898  1.312583    1.2872  1.263074  1.240432  1.219463  1.200312  1.183071  1.167767  1.154372  1.142802  1.132934  1.124614  1.117672  1.111934  1.107231  1.103404  1.100309   1.09782  1.095828  1.094241  1.092981  1.091983  1.091196  1.090577  1.090091  1.089711  1.089413  1.089181     1.089   1.08886  1.088751  1.088667  1.088601  1.088551  1.088512  1.088482  1.088459  1.088442  1.088428  1.088418   1.08841  1.088404  1.088399  1.088396 | 1.343872  1.344503  1.345183  1.345915  1.346695   1.34752  1.348384  1.349278   1.35019  1.351106  1.352009  1.352885  1.353718  1.354494  1.355204   1.35584  1.356402  1.356888  1.357304  1.357655  1.357947  1.358187  1.358384  1.358544  1.358673  1.358776  1.358858  1.358923  1.358975  1.359016  1.359048  1.359073  1.359092  1.359108   1.35912  1.359129  1.359136  1.359142  1.359146   1.35915  1.359152  1.359154  1.359156  1.359157  1.359158  1.359158  1.359159  1.359159   1.35916           1.35916 | 0.680687  0.673001  0.664968  0.656621  0.648008  0.639193   0.63026  0.621309  0.612453  0.603814  0.595517  0.587677  0.580393  0.573742  0.567771  0.562498  0.557914  0.553984   0.55066   0.54788   0.54558  0.543693  0.542157  0.540916  0.539919  0.539121  0.538486  0.537983  0.537585  0.537271  0.537025  0.536832  0.536681  0.536563  0.536471    0.5364  0.536344  0.536301  0.536268  0.536243  0.536223  0.536208  0.536196  0.536187   0.53618  0.536175  0.536171  0.536168  0.536166      0.536164 | -0.347707  -0.338215  -0.328458  -0.318505  -0.308444  -0.298379  -0.288429  -0.278724  -0.269395  -0.260567  -0.252348  -0.244824  -0.238051  -0.232053  -0.226824  -0.222331  -0.218522  -0.215331  -0.212687  -0.210517   -0.20875  -0.207321  -0.206173  -0.205256  -0.204526  -0.203948  -0.203491  -0.203132   -0.20285   -0.20263  -0.202458  -0.202324  -0.202219  -0.202139  -0.202076  -0.202028   -0.20199  -0.201962  -0.201939  -0.201922  -0.201909  -0.201899  -0.201892  -0.201886  -0.201882  -0.201878  -0.201876  -0.201874  -0.201872     -0.201871 | 0.677259  0.675933  0.674647  0.673417  0.672258  0.671186  0.670212  0.669346  0.668594  0.667954  0.667425  0.666996  0.666657  0.666396  0.666198  0.666051  0.665944  0.665867  0.665813  0.665775   0.66575  0.665732  0.665721  0.665713  0.665709  0.665706  0.665705  0.665704  0.665704  0.665704  0.665705  0.665705  0.665706  0.665706  0.665707  0.665707  0.665708  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 352.270855 | 339.494892   314.49817  290.835416  268.605369  247.891828  228.759854   211.25232  195.387177  181.155743  168.522354  157.425533  147.780731  139.484432  132.419269  126.459627  121.477176   117.34583  113.945776  111.166387  108.908014  107.082783  105.614595  104.438555  103.500036  102.753557  102.161599  101.693449  101.324133  101.033456  100.805165  100.626233  100.486258  100.376961  100.291769  100.225481  100.173988  100.134054  100.103133  100.079229  100.060778  100.046559  100.035616  100.027207  100.020756  100.015813  100.012031  100.009141  100.006937  100.005257           100.00398 | 100.003416 | 0.989735  0.970541  0.951268   0.93204  0.912998  0.894296  0.876104  0.858592  0.841931  0.826278  0.811767  0.798502  0.786548  0.775928  0.766626  0.758586  0.751725  0.745936  0.741105   0.73711  0.733834  0.731168  0.729012  0.727277  0.725889  0.724782  0.723904  0.723209   0.72266  0.722228  0.721889  0.721624  0.721417  0.721255  0.721129  0.721031  0.720955  0.720896   0.72085  0.720815  0.720788  0.720767  0.720751  0.720739   0.72073  0.720723  0.720717  0.720713   0.72071       0.720707 | 4.273711  4.145684  4.018315  3.892433   3.76895  3.648839  3.533109  3.422759  3.318731   3.22186  3.132816  3.052059  2.979816  2.916065  2.860552  2.812825  2.772279   2.73821  2.709866  2.686496  2.667377  2.651844    2.6393  2.629224  2.621168  2.614752  2.609661  2.605634  2.602457  2.599959  2.597997  2.596462  2.595262  2.594326  2.593597  2.593031  2.592592  2.592251  2.591988  2.591785  2.591629  2.591508  2.591416  2.591345   2.59129  2.591249  2.591217  2.591193  2.591174         2.59116 | 352.270855  326.628958  302.268626  279.295253  257.801259  237.862181  219.532951  202.844697  187.802414   174.38385  162.539881  152.196482  143.258236  135.613086  129.137864  123.704023  119.183018   115.45088  112.391708  109.900004  107.881911  106.255545  104.950645  103.907759  103.077178  102.417746  101.895671  101.483407  101.158627  100.903332   100.70307  100.546288  100.423776  100.328214  100.253805  100.195965  100.151078    100.1163  100.089397  100.068618  100.052594  100.040255  100.030769  100.023485  100.017902  100.013628   100.01036  100.007866  100.005964  100.004517  100.003416 |  100.0 | 252.270855 | 100.003416 |
    | 16:30 | 6.846601  6.629925   6.41299  6.196998  5.983295  5.773367  5.568801  5.371248  5.182358  5.003708  4.836718  4.682562  4.542097  4.415805  4.303771  4.205687  4.120904  4.048494  3.987339  3.936212   3.89386   3.85906   3.83067  3.807655  3.789098  3.774208   3.76231  3.752837   3.74532  3.739372  3.734677  3.730982  3.728079  3.725803  3.724022  3.722632  3.721547  3.720704  3.720048  3.719539  3.719145  3.718841  3.718605  3.718424  3.718284  3.718177  3.718094  3.718031  3.717983             3.717946 | 337.078735  319.228363   301.82708  284.968858  268.748541  253.259328  238.589746  224.820242  212.019546  200.241107  189.519948  179.870339  171.284669   163.73377  157.168773  151.524327  146.722796  142.678925  139.304424  136.512007  134.218604  132.347611  130.830227  129.606006  128.622814  127.836365  127.209526  126.711482  126.316893  126.005074  125.759242  125.565856  125.414037   125.29508  125.202041  125.129402  125.072785  125.028729  124.994502  124.967952  124.947389  124.931488  124.919209  124.909741  124.902452  124.896847  124.892544  124.889245  124.886719  124.884787 |  83.466009   81.299245   79.129903   76.969976   74.832953   72.733669   70.688011    68.71248   66.823581   65.037083   63.367176   61.825616   60.420968   59.158054   58.037707   57.056871   56.209038   55.484939   54.873386   54.362121   53.938596   53.590598   53.306703    53.07655   52.890985   52.742083   52.623099    52.52837   52.453197   52.393716   52.346775   52.309819   52.280788   52.258029   52.240222   52.226316   52.215474   52.207036   52.200479   52.195393   52.191453   52.188406   52.186053   52.184239   52.182842   52.181768   52.180944   52.180311   52.179827     52.179457 | 1.522125  1.494577  1.466706  1.438651  1.410581  1.382689  1.355188  1.328315  1.302314  1.277433  1.253907  1.231946  1.211724   1.19336  1.176919  1.162405  1.149765  1.138899  1.129668  1.121914  1.115464  1.110145  1.105794  1.102258  1.099402  1.097106  1.095269  1.093806  1.092643  1.091723  1.090996  1.090424  1.089974  1.089621  1.089345   1.08913  1.088962  1.088831  1.088729   1.08865  1.088589  1.088542  1.088506  1.088477  1.088456  1.088439  1.088426  1.088417  1.088409  1.088403 | 1.342963  1.343517  1.344118  1.344769   1.34547  1.346222   1.34702  1.347862  1.348738   1.34964  1.350553  1.351465  1.352357  1.353216  1.354026  1.354775  1.355455   1.35606  1.356591  1.357049  1.357438  1.357765  1.358037   1.35826  1.358442   1.35859  1.358709  1.358804   1.35888   1.35894  1.358988  1.359026  1.359055  1.359079  1.359097  1.359111  1.359122  1.359131  1.359138  1.359143  1.359147   1.35915  1.359153  1.359154  1.359156  1.359157  1.359158  1.359158  1.359159          1.359159 | 0.692235  0.685133  0.677659  0.669827   0.66166  0.653199  0.644498  0.635629  0.626683  0.617766  0.608994  0.600492  0.592379  0.584766  0.577741  0.571369  0.565685  0.560695  0.556379  0.552696  0.549593  0.547007  0.544873  0.543125  0.541706  0.540559  0.539638  0.538902  0.538316  0.537851  0.537483  0.537194  0.536965  0.536787  0.536646  0.536537  0.536452  0.536385  0.536333  0.536293  0.536262  0.536238   0.53622  0.536205  0.536194  0.536186  0.536179  0.536174   0.53617      0.536168 | -0.362235  -0.353255  -0.343938  -0.334327  -0.324479  -0.314473  -0.304403   -0.29438  -0.284527  -0.274974   -0.26585  -0.257273  -0.249341  -0.242128  -0.235676  -0.229996  -0.225071  -0.220859  -0.217302  -0.214333  -0.211879  -0.209869  -0.208235  -0.206915  -0.205855  -0.205008  -0.204334    -0.2038  -0.203378  -0.203045  -0.202784   -0.20258   -0.20242  -0.202295  -0.202198  -0.202123  -0.202064  -0.202019  -0.201984  -0.201957  -0.201936   -0.20192  -0.201908  -0.201898  -0.201891  -0.201885  -0.201881  -0.201878  -0.201875     -0.201874 | 0.679431  0.678067  0.676721  0.675408  0.674142   0.67294  0.671816  0.670785  0.669856  0.669038  0.668333  0.667741  0.667254  0.666864  0.666558  0.666323  0.666146  0.666016  0.665921  0.665853  0.665805  0.665771  0.665748  0.665732  0.665721  0.665715   0.66571  0.665708  0.665706  0.665705  0.665705  0.665705  0.665706  0.665706  0.665706  0.665707  0.665707  0.665707  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 395.750601 | 382.047624  355.121347  329.354621  304.863387  281.754038  260.119704  240.036492  221.559904  204.721776  189.528058  175.957758  163.963316  153.472509  144.391838   136.61111  130.008811  124.457693   119.83007  116.002367  112.858648  110.293017  108.210949   106.52972   105.17814  104.095805   103.23208  102.544942  101.999811  101.568435  101.227869  100.959577  100.748648  100.583134  100.453494  100.352129  100.273007  100.211348  100.163376   100.12611  100.097205   100.07482   100.05751  100.044144  100.033839  100.025904  100.019804   100.01512  100.011529   100.00878          100.006678 | 100.005749 | 1.020067  1.001193  0.982083  0.962832  0.943554   0.92438  0.905457  0.886948  0.869022  0.851849  0.835595  0.820408  0.806408  0.793683   0.78228  0.772206  0.763426  0.755872  0.749453  0.744058  0.739567  0.735864  0.732833   0.73037   0.72838   0.72678    0.7255  0.724479  0.723669  0.723027   0.72252  0.722121  0.721808  0.721562  0.721369  0.721219  0.721102  0.721011   0.72094  0.720885  0.720842  0.720809  0.720784  0.720764  0.720749  0.720737  0.720728  0.720722  0.720716       0.720712 | 4.478406  4.350696   4.22253  4.094593  3.967665  3.842614  3.720383  3.601962  3.488357  3.380544   3.27942   3.18575  3.100115  3.022875  2.954148  2.893811  2.841525  2.796769  2.758895  2.727177  2.700865  2.679219  2.661542  2.647199  2.635627  2.626337  2.618909  2.612994  2.608298  2.604582  2.601649  2.599339  2.597525  2.596102  2.594989   2.59412  2.593442  2.592914  2.592504  2.592186   2.59194  2.591749  2.591602  2.591489  2.591401  2.591334  2.591283  2.591243  2.591213         2.59119 | 395.750601  368.269771  341.888623  316.727089   292.89739  270.500404  249.621863  230.328596   212.66511   196.65084  182.278417  169.513261  158.294712  148.538719  140.141936  132.986858  126.947463  121.894814  117.702083   114.24865  111.423066  109.124881   107.26547  105.768038  104.567061  103.607346  102.842909  102.235783  101.754863   101.37483  101.075187  100.839418   100.65427  100.509143  100.395589  100.306891  100.237724  100.183874  100.142015  100.109529  100.084354  100.064875  100.049825  100.038214  100.029269  100.022388  100.017103  100.013048  100.009941  100.007565  100.005749 |  100.0 | 295.750601 | 100.005749 |
    | 17:00 | 7.189981  6.975837  6.759814  6.542882  6.326162  6.110927  5.898592  5.690692  5.488848  5.294716  5.109921  4.935977  4.774203  4.625643  4.490996  4.370575  4.264297  4.171706  4.092032  4.024263  3.967232    3.9197  3.880425   3.84822   3.82199  3.800749  3.783637  3.769912  3.758946  3.750215  3.743284  3.737797  3.733464   3.73005  3.727365  3.725258  3.723608  3.722317  3.721309  3.720524  3.719913  3.719438   3.71907  3.718784  3.718563  3.718393  3.718261   3.71816  3.718082             3.718022 | 366.328806   347.94908  329.872668  312.189771  294.994081  278.381082  262.445835  247.280267  232.969997  219.590851  207.205278  195.858975   185.57811   176.36751  168.210174   161.06827  154.885599  149.591271  145.104132  141.337423  138.203142   135.61573  133.494866  131.767304  130.367863  129.239705  128.334102  127.609865  127.032587  126.573801  126.210151  125.922603  125.695732  125.517101  125.376723  125.266607  125.180378  125.112967  125.060353  125.019352   124.98745  124.962664  124.943436  124.928542   124.91702   124.90812  124.901255  124.895967  124.891899  124.888774 |  86.899805   84.758372   82.598144   80.428822   78.261623   76.109274   73.985922   71.906921   69.888482   67.947163    66.09921   64.359766   62.742029    61.25643   59.909961   58.705753    57.64297   56.717065   55.920324   55.242633   54.672322   54.196998    53.80425   53.482203   53.219895   53.007491    52.83637   52.699121   52.589464   52.502152   52.432843   52.377973   52.334641   52.300497    52.27365    52.25258   52.236075   52.223169   52.213093    52.20524   52.199128    52.19438   52.190696   52.187842   52.185634   52.183929   52.182613     52.1816    52.18082     52.180221 | 1.565216  1.538422  1.511125   1.48343  1.455465  1.427384  1.399366  1.371613  1.344351  1.317817   1.29226  1.267923   1.24503  1.223777  1.204314   1.18674  1.171092   1.15735   1.14544  1.135247  1.126623  1.119402  1.113412  1.108485  1.104461  1.101196   1.09856  1.096443   1.09475    1.0934  1.092328  1.091479  1.090808  1.090279  1.089863  1.089537  1.089281  1.089081  1.088925  1.088803  1.088708  1.088635  1.088578  1.088533  1.088499  1.088473  1.088452  1.088437  1.088424  1.088415 | 1.342171  1.342654   1.34318  1.343752  1.344373  1.345044  1.345766  1.346536  1.347352  1.348208  1.349095  1.350002  1.350915  1.351819  1.352698  1.353537  1.354322  1.355044  1.355693  1.356269   1.35677    1.3572  1.357564  1.357868  1.358121  1.358328  1.358496  1.358633  1.358743   1.35883  1.358901  1.358956     1.359  1.359035  1.359063  1.359084  1.359101  1.359114  1.359125  1.359133  1.359139  1.359144  1.359148  1.359151  1.359153  1.359155  1.359156  1.359157  1.359158          1.359159 | 0.702814   0.69631  0.689432   0.68218  0.674559  0.666589  0.658301  0.649739  0.640966  0.632062  0.623123  0.614261  0.605595  0.597248  0.589336  0.581961  0.575202  0.569111   0.56371  0.558996  0.554939  0.551492  0.548599  0.546195  0.544215  0.542598  0.541286  0.540227  0.539377  0.538698  0.538157  0.537728  0.537388   0.53712  0.536909  0.536744  0.536614  0.536512  0.536433  0.536371  0.536323  0.536285  0.536256  0.536234  0.536216  0.536203  0.536192  0.536184  0.536178      0.536174 | -0.375855  -0.367462  -0.358692  -0.349567  -0.340122  -0.330406  -0.320486  -0.310447   -0.30039  -0.290432  -0.280697  -0.271317  -0.262415  -0.254101  -0.246464  -0.239564   -0.23343   -0.22806  -0.223427  -0.219483  -0.216165  -0.213404  -0.211128  -0.209266  -0.207755  -0.206536  -0.205558  -0.204776  -0.204154  -0.203661  -0.203271  -0.202963  -0.202722  -0.202532  -0.202384  -0.202268  -0.202178  -0.202107  -0.202053   -0.20201  -0.201977  -0.201952  -0.201932  -0.201917  -0.201906  -0.201897   -0.20189  -0.201885  -0.201881     -0.201878 | 0.681635  0.680262  0.678889  0.677528  0.676192  0.674895  0.673654  0.672482  0.671395  0.670405  0.669522  0.668751  0.668093  0.667545  0.667099  0.666745  0.666469  0.666258  0.666101  0.665985    0.6659   0.66584  0.665797  0.665767  0.665746  0.665732  0.665722  0.665716  0.665711  0.665709  0.665707  0.665707  0.665706  0.665706  0.665706  0.665707  0.665707  0.665707  0.665707  0.665708  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 441.664962 | 427.206229  398.682633  371.118565  344.640702  319.370874   295.42284  272.898761  251.885504  232.451006  214.640966  198.476193  183.950936  171.032489  159.662273  149.758419  141.219726  133.930626  127.766684  122.600096  118.304674  114.759953  111.854222   109.48645  107.567201  106.018746  104.774573  103.778514  102.983659  102.351191  101.849242   101.45182  101.137843  100.890293  100.695487  100.542464   100.42247  100.328531  100.255108  100.197811  100.153166  100.118432  100.091449  100.070517  100.054303  100.041761  100.032074  100.024602  100.018846  100.014418          100.011016 | 100.009511 | 1.049564  1.031227  1.012532  0.993552  0.974371  0.955096  0.935846  0.916761  0.897995  0.879713  0.862085  0.845281  0.829458  0.814753  0.801275  0.789092  0.778236  0.768695   0.76042  0.753333  0.747334  0.742309  0.738139  0.734708  0.731905   0.72963  0.727793  0.726318  0.725137  0.724197  0.723449  0.722857  0.722389  0.722021  0.721731  0.721503  0.721325  0.721185  0.721076  0.720991  0.720925  0.720874  0.720834  0.720803  0.720779  0.720761  0.720746  0.720736  0.720727       0.720721 | 4.680222  4.554441  4.427289  4.299309  4.171141  4.043515  3.917254  3.793261  3.672503  3.555979  3.444686   3.33957  3.241473  3.151083  3.068891  2.995153  2.929885   2.87287  2.823691  2.781771  2.746427  2.716923  2.692511  2.672472  2.656134  2.642894   2.63222  2.623655  2.616809  2.611357  2.607027  2.603599  2.600891  2.598757  2.597079  2.595762   2.59473  2.593923  2.593293  2.592802   2.59242  2.592123  2.591892  2.591714  2.591576  2.591469  2.591387  2.591324  2.591275        2.591238 | 441.664962  412.688919   384.60814  357.551057  331.642788  307.002115  283.738069  261.946269  241.705181  223.072555  206.082341  190.742431  177.033538  164.909501   154.29913  145.109552  137.230803  130.541223  124.913117  120.218133  116.331894  113.137594  110.528463  108.409135  106.696105  105.317466  104.212176  103.329032   102.62551  102.066594   101.62365  101.273403  100.997032  100.779378  100.608282  100.474021  100.368843  100.286581  100.222345  100.172262  100.133273  100.102966  100.079442   100.06121  100.047098  100.036192  100.027775  100.021288  100.016295  100.012457  100.009511 |  100.0 | 341.664962 | 100.009511 |
    | 17:30 | 7.524907  7.315729  7.103354   6.88854  6.672185  6.455334  6.239182   6.02507  5.814472  5.608966  5.410195  5.219809  5.039391  4.870382  4.713989  4.571116  4.442301  4.327693  4.227049  4.139779  4.065008  4.001655  3.948521  3.904364  3.867965  3.838176  3.813948  3.794351  3.778575  3.765926  3.755822  3.747776  3.741387  3.736327  3.732328  3.729175  3.726693  3.724743  3.723215  3.722018  3.721082  3.720352  3.719783   3.71934  3.718996  3.718729  3.718522  3.718362  3.718239             3.718143 | 395.994739  377.335421  358.838477  340.587982   322.67302  305.186694  288.224703   271.88343  256.257527  241.437016  227.504001  214.529144  202.568183  191.658806  181.818285  173.042229  165.304735  158.560025  152.745457  147.785557  143.596578  140.091058  137.181918  134.785762  132.825262   131.23062   129.94024  128.900778   128.06675  127.399883  126.868314  126.445756  126.110684  125.845582  125.636275  125.471336  125.341598  125.239722  125.159857  125.097346  125.048493  125.010372  124.980669  124.957558  124.939602   124.92567  124.914876  124.906523   124.90007   124.89509 |  90.249071   88.157294   86.033537   83.885396   81.721848    79.55334   77.391822   75.250705   73.144723   71.089664   69.101954   67.198087   65.393912   63.703816   62.139888   60.711157   59.423015   58.276929   57.270491   56.397794   55.650081   55.016554   54.485212   54.043642   53.679654   53.381761   53.139484   52.943513   52.785746    52.65926   52.558218   52.477758   52.413869   52.363266   52.323279   52.291746    52.26693   52.247435   52.232147   52.220177   52.210821    52.20352   52.197829   52.193402   52.189961   52.187292   52.185223   52.183623   52.182386     52.181432 | 1.606618  1.580832  1.554409  1.527424  1.499973  1.472169  1.444155  1.416094  1.388176  1.360614  1.333638  1.307494  1.282426  1.258671  1.236443  1.215919  1.197227  1.180442  1.165576  1.152587  1.141383  1.131834  1.123784  1.117066  1.111508  1.106945  1.103226  1.100211   1.09778  1.095828  1.094267  1.093023  1.092035  1.091252  1.090632  1.090144  1.089759  1.089457   1.08922  1.089035   1.08889  1.088776  1.088688   1.08862  1.088566  1.088525  1.088493  1.088468  1.088449  1.088434 | 1.341486  1.341904  1.342362  1.342861  1.343405  1.343997  1.344637  1.345328  1.346069  1.346858   1.34769  1.348558  1.349454  1.350363  1.351273  1.352168  1.353031  1.353849  1.354608  1.355301   1.35592  1.356465  1.356938  1.357341  1.357681  1.357964  1.358199   1.35839  1.358546  1.358673  1.358774  1.358855   1.35892  1.358971  1.359012  1.359044   1.35907   1.35909  1.359105  1.359118  1.359127  1.359135   1.35914  1.359145  1.359149  1.359151  1.359153  1.359155  1.359156          1.359157 |  0.71241  0.706496  0.700219  0.693571  0.686546  0.679148  0.671389  0.663291  0.654892  0.646243  0.637416  0.628495  0.619584  0.610798  0.602258  0.594085  0.586389  0.579263  0.572775  0.566965  0.561844  0.557397  0.553589  0.550368  0.547672  0.545439  0.543605  0.542109  0.540896  0.539918  0.539134  0.538508  0.538009  0.537613  0.537299  0.537052  0.536857  0.536703  0.536583  0.536489  0.536415  0.536357  0.536312  0.536278   0.53625  0.536229  0.536213    0.5362  0.536191      0.536183 |  -0.38848  -0.380707  -0.372543  -0.363995  -0.355077  -0.345818  -0.336258  -0.326455  -0.316482  -0.306431  -0.296409  -0.286537  -0.276943  -0.267754   -0.25909  -0.251051  -0.243715  -0.237127  -0.231306  -0.226237  -0.221886  -0.218197  -0.215105  -0.212539  -0.210429  -0.208707   -0.20731  -0.206184  -0.205281   -0.20456  -0.203986   -0.20353   -0.20317  -0.202886  -0.202663  -0.202487  -0.202349  -0.202242  -0.202158  -0.202092  -0.202041  -0.202002  -0.201971  -0.201947  -0.201929  -0.201915  -0.201904  -0.201895  -0.201889     -0.201884 | 0.683829  0.682471  0.681098  0.679721  0.678349  0.676995  0.675672  0.674395   0.67318  0.672042  0.670993  0.670047  0.669209  0.668485  0.667873  0.667368   0.66696  0.666638  0.666389  0.666201   0.66606  0.665957  0.665882  0.665828   0.66579  0.665763  0.665744  0.665731  0.665722  0.665716  0.665712   0.66571  0.665708  0.665708  0.665707  0.665707  0.665707  0.665707  0.665707  0.665708  0.665708  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 489.279607 | 474.259572  444.520281   415.52256  387.398474  360.278984    334.2914  309.556408  286.184728  264.273538  243.902833  225.131952  207.996586  192.506573  178.644801  166.367477  155.605888  146.269615  138.250962  131.430204  125.681126  120.876329  116.891859  113.610861  110.926118  108.741538  106.972705  105.546715  104.401513  103.484915  102.753489  102.171392  101.709261  101.343183  101.053785  100.825441  100.645594  100.504184  100.393179  100.306178  100.238095  100.184897  100.143389   100.11105   100.08589  100.066343  100.051177  100.039427  100.030336  100.023312          100.017891 | 100.015487 | 1.077877  1.060246  1.042169  1.023696  1.004891   0.98583   0.96661  0.947341  0.928153  0.909192  0.890616  0.872594  0.855297  0.838888  0.823519  0.809313  0.796363  0.784724  0.774407  0.765386    0.7576  0.750959  0.745359  0.740683  0.736813  0.733635  0.731044  0.728943  0.727249  0.725889  0.724801  0.723934  0.723245  0.722698  0.722267  0.721926  0.721658  0.721447  0.721282  0.721153  0.721052  0.720973  0.720911  0.720863  0.720826  0.720797  0.720775  0.720757  0.720744       0.720734 | 4.876459  4.753967  4.629371  4.503091  4.375628  4.247572  4.119606  3.992505   3.86713  3.744413   3.62534  3.510912  3.402108  3.299834  3.204872   3.11783    3.0391  2.968839  2.906966  2.853177  2.806985  2.767768  2.734818  2.707395  2.684761  2.666217  2.651122  2.638904  2.629062  2.621167  2.614858  2.609833  2.605841   2.60268  2.600181   2.59821  2.596659   2.59544  2.594484  2.593736  2.593151  2.592694  2.592339  2.592062  2.591847   2.59168   2.59155   2.59145  2.591373        2.591313 | 489.279607   459.19787  429.791513  401.192625  373.533373  346.943681  321.548464  297.464444  274.796652  253.634755  234.049421   216.08899  199.776772  185.109304  172.055871  160.559496  150.539475  141.895292   134.51161  128.263823  123.023634   118.66414  115.064027  112.110682  109.702173  107.748206  106.170261  104.901112  103.883957  103.071325  102.423914  101.909432  101.501524  101.178794  100.923954  100.723092   100.56505  100.440905  100.343543  100.267305  100.207697  100.161161  100.124882  100.096641  100.074687  100.057645  100.044433  100.034206  100.026299  100.020194  100.015487 |  100.0 | 389.279607 | 100.015487 |
    | 18:00 |  7.84714   7.64498   7.43857  7.228489  7.015433  6.800229  6.583847  6.367404  6.152162  5.939527  5.731024  5.528264  5.332902  5.146566  4.970789  4.806922   4.65605  4.518931  4.395943  4.287072  4.191929  4.109802  4.039727  3.980571  3.931115  3.890126  3.856413  3.828871  3.806502  3.788426  3.773884   3.76223  3.752921  3.745508   3.73962  3.734955  3.731266  3.728355  3.726062  3.724259  3.722844  3.721734  3.720866  3.720188  3.719659  3.719246  3.718925  3.718676  3.718482             3.718332 | 425.595105    406.9033  388.240203  369.682603  351.312971  333.219023  315.492941    298.2302  281.527941  265.482846  250.188534    235.7325  222.192735  209.634219  198.105583  187.636299   178.23477  169.887673  162.560741  156.201004  150.740262  146.099375  142.192863  138.933293  136.235054  134.017275  132.205807  130.734348  129.544828   128.58727  127.819279  127.205322  126.715911  126.326781  126.018099  125.773749  125.580701   125.42846  125.308605    125.2144  125.140472  125.082544  125.037219  125.001807  124.974178  124.952651  124.935902  124.922888  124.912789  124.904963 |  93.471397     91.4498   89.385704    87.28489   85.154327   83.002292   80.838474   78.674036   76.521623   74.395275   72.310239   70.282644   68.329018   66.465663   64.707895   63.069216   61.560502   60.189313   58.959434   57.870719   56.919287   56.098017   55.397268    54.80571   54.311151   53.901257   53.564131   53.288713   53.065022   52.884264   52.738843     52.6223   52.529213   52.455082   52.396202   52.349546   52.312656   52.283546   52.260617   52.242588   52.228435   52.217343   52.208662   52.201879   52.196586   52.192461   52.189252   52.186758   52.184823     52.183323 | 1.645901  1.621317  1.596003  1.570007  1.543396  1.516253  1.488682   1.46081  1.432788  1.404793  1.377023    1.3497  1.323061  1.297352  1.272816  1.249681  1.228145  1.208368  1.190456  1.174456   1.16036  1.148103  1.137578  1.128644  1.121139  1.114894   1.10974  1.105518  1.102081  1.099298  1.097056  1.095257  1.093819  1.092672  1.091761  1.091039  1.090468  1.090017  1.089662  1.089382  1.089163  1.088991  1.088856  1.088751  1.088669  1.088605  1.088555  1.088517  1.088487  1.088463 | 1.340896  1.341258  1.341655  1.342088  1.342562  1.343078   1.34364  1.344251  1.344911  1.345622  1.346381  1.347187  1.348034  1.348913  1.349814  1.350724  1.351628  1.352511  1.353356   1.35415  1.354883  1.355546  1.356135  1.356651  1.357095  1.357473   1.35779  1.358054  1.358271  1.358449  1.358593   1.35871  1.358803  1.358878  1.358938  1.358985  1.359023  1.359053  1.359076  1.359095  1.359109   1.35912  1.359129  1.359136  1.359142  1.359146  1.359149  1.359152  1.359154          1.359155 | 0.721039  0.715691      0.71  0.703952  0.697536  0.690744  0.683577   0.67604  0.668149  0.659933  0.651436  0.642716  0.633851  0.624933  0.616071  0.607382  0.598988  0.591006  0.583539  0.576671  0.570458  0.564928  0.560082  0.555895  0.552325  0.549318  0.546809  0.544737  0.543038  0.541654  0.540534  0.539632  0.538909  0.538331  0.537871  0.537505  0.537216  0.536987  0.536807  0.536665  0.536554  0.536466  0.536398  0.536344  0.536303   0.53627  0.536245  0.536225   0.53621      0.536198 | -0.400067  -0.392921  -0.385386  -0.377459  -0.369143  -0.360447  -0.351392  -0.342011  -0.332351  -0.322477   -0.31247  -0.302427  -0.292462    -0.2827  -0.273267  -0.264288  -0.255876  -0.248122   -0.24109  -0.234815    -0.2293  -0.224524  -0.220442  -0.216996  -0.214117  -0.211735   -0.20978  -0.208187  -0.206896  -0.205857  -0.205024  -0.204358  -0.203829  -0.203409  -0.203076  -0.202814  -0.202607  -0.202444  -0.202317  -0.202217  -0.202139  -0.202078  -0.202031  -0.201994  -0.201965  -0.201943  -0.201926  -0.201912  -0.201902     -0.201894 | 0.685975  0.684651  0.683303  0.681936  0.680559  0.679181  0.677813  0.676469  0.675162  0.673908  0.672722  0.671619  0.670611  0.669708  0.668917  0.668239  0.667671  0.667206  0.666834  0.666542  0.666318   0.66615  0.666024  0.665932  0.665865  0.665817  0.665783  0.665759  0.665742   0.66573  0.665722  0.665717  0.665713  0.665711  0.665709  0.665708  0.665708  0.665708  0.665708  0.665708  0.665708  0.665708  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709  0.665709      0.665709 |  537.77215 | 522.402417  491.865982  461.841985  432.463717   403.86612  376.183942  349.549433  324.089615  299.923153  277.156923  255.882419  236.172196  218.076622  201.621225  186.804979  173.599801  161.951477  151.782069  142.993686  135.473309  129.098207   123.74142  119.276812    115.5833  112.548036  110.068497  108.053554   106.42371   105.11071   104.05673  103.213335  102.540337  102.004656  101.579239  101.242089  100.975405  100.764835  100.598854  100.468231  100.365593  100.285065  100.221976  100.172621  100.134063  100.103983  100.080548  100.062316  100.048149  100.037156          100.028637 | 100.024851 | 1.104717  1.087923   1.07062  1.052842  1.034632  1.016045  0.997152  0.978039  0.958807  0.939576  0.920483  0.901679  0.883328  0.865599  0.848661  0.832674  0.817777  0.804083  0.791669  0.780571  0.770785  0.762271  0.754954   0.74874  0.743518  0.739171  0.735582  0.732641  0.730246  0.728307  0.726745  0.725491  0.724488  0.723689  0.723054   0.72255  0.722152  0.721838   0.72159  0.721395  0.721242  0.721122  0.721028  0.720955  0.720898  0.720853  0.720818  0.720791   0.72077       0.720754 | 5.064747  4.946676  4.825928  4.702814  4.577718  4.451098    4.3235  4.195556  4.067992  3.941624  3.817345  3.696116   3.57893  3.466786  3.360637   3.26134  3.169609  3.085965  3.010705  2.943887  2.885335  2.834669  2.791344  2.754699  2.724013  2.698544  2.677572  2.660421  2.646481  2.635208  2.626135  2.618859  2.613047  2.608416  2.604738  2.601822  2.599517  2.597697  2.596264  2.595137  2.594252  2.593559  2.593016  2.592592  2.592261  2.592003  2.591802  2.591646  2.591525        2.591432 |  537.77215  507.008032   476.69011  446.950464  417.923662  389.745127  362.549074  336.465985  311.619664  288.123925  266.079034  245.568058  226.653384  209.373664  193.741533  179.742419  167.334709  156.451417  147.003339  138.883475  131.972304  126.143397  121.268818  117.223863  113.890799  111.161493  108.938949  107.137894  105.684644  104.516443  103.580487  102.832804  102.237084  101.763559  101.387972  101.090654  100.855727   100.67042  100.524491  100.409753  100.319676  100.249065  100.193792  100.150587  100.116861  100.090572  100.070107  100.054197  100.041845  100.032267  100.024851 |  100.0 |  437.77215 | 100.024851 |
    | 18:30 | 8.152996  7.959567  7.761059  7.557902  7.350623  7.139858  6.926368  6.711048  6.494936  6.279216  6.065219  5.854405  5.648342  5.448669  5.257035  5.075038  4.904144  4.745601  4.600367  4.469046  4.351857   4.24863  4.158843  4.081675   4.01609  3.960917  3.914929   3.87691  3.845705  3.820254  3.799609  3.782941  3.769541  3.758805  3.750231  3.743402  3.737977  3.733676  3.730274  3.727587   3.72547  3.723803  3.722494  3.721466  3.720662  3.720032   3.71954  3.719156  3.718857             3.718624 | 454.651678  436.167045  417.586072  398.977945  380.417613   361.98574  343.768413  325.856555  308.344976  291.331005  274.912663  259.186338  244.243991  230.169974  217.037585   204.90563  193.815289  183.787665   174.82239  166.897562  159.971146  153.983758  148.862513   144.52549  140.886269  137.858093  135.357294  133.305812  131.632818  130.275508  129.179265  128.297349  127.590291  127.025134  126.574607   126.21632  125.932004  125.706833  125.528829  125.388353  125.277672  125.190602  125.122206  125.068557  125.026535  124.993664  124.967987  124.947955  124.932349  124.920206 |  96.529962   94.595671    92.61059    90.57902   88.506227   86.398581   84.263683   82.110481   79.949356    77.79216   75.652186   73.544046   71.483423   69.486686   67.570346   65.750381   64.041438   62.456011    61.00367   59.690462   58.518569   57.486304   56.588429   55.816752   55.160904   54.609174   54.149292   53.769101   53.457052   53.202539   52.996088   52.829414   52.695406   52.588047   52.502306    52.43402   52.379768   52.336762    52.30274   52.275875   52.254698   52.238032   52.224938   52.214664   52.206615   52.200319     52.1954   52.191562   52.188571     52.186244 | 1.682715  1.659485  1.635458  1.610665  1.585149  1.558968  1.532196  1.504924  1.477269  1.449367   1.42138  1.393494   1.36592  1.338885   1.31263  1.287402  1.263438  1.240956  1.220138  1.201123  1.183993  1.168774  1.155432  1.143886  1.134014  1.125665  1.118676  1.112875    1.1081  1.104195   1.10102  1.098453  1.096386  1.094728  1.093403  1.092347  1.091507  1.090841  1.090314  1.089898   1.08957  1.089311  1.089109  1.088949  1.088824  1.088727  1.088651  1.088591  1.088545  1.088509 | 1.340391  1.340705  1.341047  1.341422  1.341832  1.342281   1.34277  1.343304  1.343885  1.344515  1.345194  1.345924  1.346701  1.347523  1.348383  1.349271  1.350176  1.351084  1.351979  1.352847  1.353672  1.354442  1.355147   1.35578  1.356339  1.356826  1.357243  1.357597  1.357892  1.358137  1.358339  1.358503  1.358636  1.358744   1.35883  1.358899  1.358954  1.358998  1.359033   1.35906  1.359082  1.359099  1.359113  1.359123  1.359131  1.359138  1.359143  1.359147   1.35915          1.359152 | 0.728736  0.723922  0.718788  0.713319    0.7075  0.701318  0.694764  0.687833  0.680527  0.672856  0.664842  0.656519  0.647937  0.639162  0.630277  0.621382  0.612589  0.604018  0.595789  0.588014  0.580788  0.574185  0.568249  0.562997  0.558419  0.554484  0.551143  0.548338  0.546006  0.544084  0.542511  0.541232  0.540198  0.539366  0.538699  0.538166  0.537742  0.537405  0.537138  0.536927   0.53676  0.536629  0.536526  0.536445  0.536382  0.536332  0.536293  0.536263  0.536239      0.536221 | -0.410609  -0.404076   -0.39717  -0.389878  -0.382195   -0.37412  -0.365657  -0.356821  -0.347639  -0.338148  -0.328403  -0.318476  -0.308453   -0.29844  -0.288555  -0.278923  -0.269672  -0.260921  -0.252774  -0.245312  -0.238588  -0.232622  -0.227408  -0.222915  -0.219092  -0.215876  -0.213197  -0.210987  -0.209177  -0.207704  -0.206512  -0.205553  -0.204785  -0.204171  -0.203683  -0.203295  -0.202988  -0.202745  -0.202554  -0.202404  -0.202286  -0.202193  -0.202121  -0.202064   -0.20202  -0.201986  -0.201959  -0.201938  -0.201922      -0.20191 | 0.688045   0.68677  0.685464  0.684129  0.682771  0.681398  0.680018  0.678642  0.677281   0.67595  0.674663  0.673434   0.67228  0.671214  0.670248   0.66939  0.668645  0.668011  0.667486  0.667059   0.66672  0.666457  0.666255  0.666104  0.665992   0.66591  0.665851  0.665808  0.665777  0.665756   0.66574   0.66573  0.665722  0.665717  0.665714  0.665711   0.66571  0.665709  0.665708  0.665708  0.665708  0.665708  0.665708  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709      0.665709 | 586.262607 | 570.765738  539.874884  509.261995   479.05851  449.399391  420.421899   392.26396  365.062114  338.949012  314.050518  290.482445  268.347062  247.729517  228.694416  211.282824  195.510014  181.364263  168.806964   157.77419  148.179706  139.919225  132.875526  126.923951  121.937742  117.792777  114.371376   111.56502  109.276005  107.418151  105.916756  104.708022  103.738141  102.962197  102.343032  101.850124  101.458559  101.148104  100.902402  100.708274   100.55514  100.434527  100.339668   100.26517  100.206745  100.160987  100.125198  100.097243  100.075436  100.058448          100.045229 | 100.039343 | 1.129852  1.113994  1.097584  1.080643  1.063199  1.045289  1.026963  1.008284  0.989327  0.970187  0.950972   0.93181  0.912844   0.89423  0.876136  0.858732  0.842182   0.82664  0.812234  0.799063  0.787188  0.776627  0.767363   0.75934  0.752476  0.746668  0.741804  0.737765  0.734439  0.731719  0.729507  0.727718  0.726278  0.725122  0.724198  0.723462  0.722877  0.722412  0.722045  0.721755  0.721526  0.721346  0.721204  0.721093  0.721006  0.720938  0.720885  0.720843  0.720811       0.720786 | 5.243045  5.130331  5.014494  4.895761  4.774415  4.650805  4.525348  4.398545  4.270981  4.143333  4.016365  3.890932  3.767956  3.648418  3.533316  3.423635  3.320293  3.224093  3.135674  3.055466   2.98367  2.920248  2.864938   2.81729   2.77671   2.74251  2.713959  2.690325  2.670906  2.655052  2.642183  2.631786  2.623423  2.616721  2.611366    2.6071  2.603711  2.601023  2.598897  2.597218  2.595894  2.594852  2.594033  2.593391  2.592888  2.592494  2.592187  2.591947   2.59176        2.591614 | 586.262607  555.260904  524.472222   494.02599  464.055683  434.697801  406.090449  378.371503  351.676333  326.135092  301.869617  278.990007  257.591039  237.748581  219.516302  202.922942  187.970491  174.633574  162.860265  152.574402  143.679314  136.062645  129.601819  124.169617  119.639327  115.889091  112.805176  110.284144  108.233975  106.574336   105.23621  104.161099  103.299971  102.612124   102.06403  101.628257  101.282483  101.008629   100.79211  100.621202  100.486506  100.380507  100.297212   100.23185  100.180631  100.140548  100.109222  100.084772  100.065714  100.050878  100.039343 |  100.0 | 486.262607 | 100.039343 |
    | 19:00 | 8.439345  8.256063  8.067054  7.872631  7.673182  7.469187  7.261224   7.04999  6.836308  6.621137  6.405584  6.190899  5.978472  5.769814   5.56653  5.370268  5.182667  5.005278  4.839486  4.686428  4.546922  4.421419   4.30998   4.21229  4.127703  4.055307  3.994007  3.942607  3.899885  3.864648  3.835783  3.812275  3.793229  3.777865  3.765519  3.755632  3.747736  3.741447  3.736449  3.732486  3.729349  3.726871  3.724916  3.723376  3.722165  3.721214  3.720468  3.719884  3.719427              3.71907 | 482.702924  464.653845  446.392611  427.981025  409.486357  390.981546  372.545245  354.261658  336.220115  318.514329  301.241269  284.499607   268.38768  253.000981  238.429199  224.752928  212.040217  200.343242  189.695437  180.109462  171.576344  164.066026  157.529348  151.901309  147.105205  143.057162  139.670552  136.859868  134.543774  132.647257  131.102886  129.851342  128.841361  128.029303  127.378472  126.858327  126.443665  126.113833   125.85201  125.644559   125.48047  125.350891  125.248719  125.168275  125.105029  125.055372  125.016435  124.985946  124.962101  124.943477 |  99.393453   97.560632    95.67054   93.726306   91.731822   89.691866   87.612242   85.499902   83.363075    81.21137   79.055837   76.908986   74.784715    72.69814   70.665295   68.702682   66.826674   65.052785   63.394864   61.864281   60.469223   59.214192   58.099802   57.122904    56.27703   55.553067   54.940068   54.426072   53.998847   53.646483   53.357827   53.122753   52.932289   52.778652   52.655194   52.556318    52.47736    52.41447   52.364494   52.324862   52.293493   52.268708   52.249157   52.233758   52.221649   52.212139   52.204681    52.19884   52.194272     52.190704 | 1.716787  1.695022  1.672416  1.648986  1.624759  1.599772  1.574074   1.54773  1.520823  1.493454  1.465749  1.437855  1.409944  1.382214  1.354881  1.328181  1.302357  1.277653  1.254299    1.2325  1.212422   1.19418  1.177834  1.163386  1.150782  1.139924  1.130677  1.122886  1.116383     1.111  1.106578  1.102968  1.100038   1.09767  1.095765  1.094238  1.093017  1.092044  1.091271  1.090657  1.090171  1.089787  1.089484  1.089245  1.089058   1.08891  1.088794  1.088704  1.088633  1.088578 | 1.339961  1.340232  1.340528  1.340852  1.341206  1.341594  1.342019  1.342482  1.342989   1.34354  1.344139  1.344788  1.345487  1.346235   1.34703  1.347866  1.348737  1.349631  1.350538  1.351441  1.352326  1.353177   1.35398  1.354723  1.355399  1.356002  1.356532  1.356991  1.357382  1.357712  1.357987  1.358215  1.358401  1.358553  1.358677  1.358776  1.358855  1.358919   1.35897   1.35901  1.359042  1.359068  1.359088  1.359104  1.359116  1.359126  1.359133  1.359139  1.359144          1.359148 | 0.735545  0.731229  0.726619  0.721698   0.71645  0.710859  0.704912  0.698597  0.691906  0.684838  0.677397    0.6696  0.661471  0.653052  0.644398  0.635584  0.626698  0.617846  0.609143   0.60071  0.592664  0.585111  0.578137  0.571805  0.566147   0.56117  0.556854  0.553161  0.550038  0.547425  0.545259  0.543477  0.542022  0.540841  0.539887   0.53912  0.538505  0.538013  0.537622  0.537312  0.537065  0.536871  0.536717  0.536596    0.5365  0.536425  0.536366   0.53632  0.536284      0.536256 | -0.420119  -0.414178  -0.407881  -0.401217  -0.394171  -0.386737  -0.378909  -0.370689  -0.362086   -0.35312   -0.34382  -0.334232  -0.324418  -0.314455  -0.304438  -0.294476  -0.284693  -0.275213  -0.266163  -0.257655  -0.249785  -0.242622  -0.236205  -0.230546  -0.225625  -0.221405  -0.217829  -0.214831  -0.212342  -0.210292  -0.208617  -0.207256  -0.206156  -0.205272  -0.204563  -0.203997  -0.203546  -0.203189  -0.202906  -0.202682  -0.202505  -0.202366  -0.202257  -0.202171  -0.202104  -0.202052   -0.20201  -0.201978  -0.201954     -0.201934 | 0.690015  0.688801  0.687548  0.686261  0.684942  0.683597  0.682233  0.680855  0.679476  0.678104  0.676754  0.675439  0.674174  0.672974  0.671854  0.670827  0.669904  0.669091  0.668391  0.667802  0.667316  0.666926  0.666618   0.66638  0.666199  0.666064  0.665964  0.665891  0.665838    0.6658  0.665772  0.665753  0.665739  0.665729  0.665722  0.665717  0.665714  0.665712   0.66571  0.665709  0.665709  0.665709  0.665708  0.665708  0.665708  0.665708  0.665709  0.665709  0.665709      0.665709 | 633.846018 | 618.449354   587.65833   556.91044  526.333106  496.058405  466.222368  436.963933   408.42353  380.741244  354.054558  328.495681  304.188479  281.245124  259.762549  239.818943  221.470492  204.748696  189.658539  176.177839  164.257969  153.826042  144.788476  137.035644  130.447204   124.89757  120.261038  116.416157  113.249088  110.655882  108.543728  106.831334  105.448651  104.336147  103.443808  102.730026  102.160464  101.706975  101.346622    101.0608  100.834476  100.655551  100.514311  100.402981  100.315348  100.246463  100.192387  100.149991  100.116795  100.090836          100.070562 | 100.061517 | 1.153099  1.138251  1.122822  1.106824  1.090275  1.073197  1.055624  1.037598  1.019175  1.000423  0.981426  0.962285  0.943116  0.924053  0.905246  0.886856  0.869051  0.852001  0.835866  0.820791  0.806891  0.794251  0.782915  0.772887  0.764132  0.756585  0.750155  0.744734  0.740207   0.73646   0.73338  0.730865  0.728823  0.727173  0.725845   0.72478  0.723929  0.723251  0.722712  0.722284  0.721945  0.721677  0.721466  0.721299  0.721169  0.721066  0.720985  0.720922  0.720873       0.720834 | 5.409633  5.303042  5.192984  5.079622  4.963159  4.843851  4.722013  4.598025  4.472345  4.345511  4.218148  4.090975  3.964795  3.840495  3.719024  3.601374  3.488543  3.381493    3.2811  3.188104  3.103062  3.026313  2.957962  2.897878  2.845723  2.800984  2.763028  2.731148  2.704611  2.682697  2.664726  2.650079  2.638204  2.628619  2.620913  2.614739  2.609808  2.605879  2.602756   2.60028  2.598319   2.59677  2.595547  2.594585  2.593828  2.593233  2.592767  2.592402  2.592116        2.591893 | 633.846018  603.060736  572.255878  541.556394   511.09217   480.99749   451.41017  422.470342  394.318834  367.095134  340.934918  315.967162  292.310913  270.071787  249.338384   230.17882  212.637666  196.733588  182.458027  169.775178  158.623442  148.918354  140.556806  133.422181   127.38992  122.332956  118.126565   114.65228  111.800703  109.473235  107.582826  106.053972  104.822145  103.832891   103.04073  102.408015  101.903807  101.502832  101.184557  100.932363  100.732857  100.575274  100.450987    100.3531  100.276111   100.21564  100.168205  100.131043  100.101968  100.079249  100.061517 |  100.0 | 533.846018 | 100.061517 |
    | 19:30 | 8.703587  8.531614  8.353409  8.169195  7.979253  7.783936  7.583673  7.378993   7.17053  6.959035  6.745396  6.530637  6.315928  6.102583  5.892046  5.685873  5.485693  5.293158  5.109878  4.937346  4.776852  4.629412    4.4957   4.37601  4.270253  4.177976  4.098424  4.030607  3.973388  3.925558  3.885907  3.853276  3.826591   3.80489  3.787326  3.773169  3.761798  3.752694  3.745424  3.739634  3.735031  3.731379  3.728488  3.726202  3.724398  3.722975  3.721856  3.720976  3.720286             3.719744 | 509.315914  491.916425  474.198355  456.216656  438.031232  419.707295  401.315609  382.932623  364.640409   346.52639  328.682772  311.205642  294.193668   277.74634  261.961749  246.933887  232.749552  219.484984   207.20245  195.947087  185.744353   176.59846  168.492081  161.387488   155.22909  149.947073  145.461752  141.688087    138.5399  135.933425   133.78999  132.037797   130.61286  129.459291  128.529078  127.781536   127.18259  126.703964   126.32238  126.018801   125.77774  125.586656  125.435433   125.31594  125.221655  125.147365  125.088906  125.042967  125.006911  124.978648 | 102.035868  100.316143   98.534091    96.69195   94.792535   92.839355   90.836734   88.789934   86.705295   84.590353   82.453959   80.306369   78.159282    76.02583   73.920464   71.858735   69.856933   67.931581   66.098782   64.373455    62.76852   61.294121   59.956998   58.760104   57.702528   56.779763   55.984239   55.306073   54.733883   54.255585   53.859074   53.532756    53.26591     53.0489   52.873259   52.731686    52.61798   52.526939   52.454243   52.396336   52.350308   52.313795    52.28488   52.262021   52.243977   52.229755   52.218561   52.209763   52.202856     52.197442 | 1.747905  1.727687  1.706601  1.684653  1.661857  1.638237  1.613822  1.588655  1.562793  1.536309  1.509293  1.481858  1.454137  1.426289  1.398497  1.370966  1.343922  1.317603  1.292254  1.268115  1.245407  1.224319  1.204998  1.187537  1.171972  1.158284  1.146399  1.136204  1.127556  1.120294   1.11425  1.109259  1.105168  1.101833  1.099129  1.096946   1.09519  1.093784  1.092659  1.091763  1.091051  1.090485  1.090038  1.089683  1.089404  1.089183   1.08901  1.088873  1.088766  1.088682 | 1.339597   1.33983  1.340086  1.340366  1.340672  1.341007  1.341373  1.341775  1.342213  1.342693  1.343216  1.343786  1.344404  1.345071  1.345789  1.346555  1.347365  1.348215  1.349095  1.349994  1.350899  1.351795  1.352666  1.353498  1.354277  1.354994   1.35564  1.356214  1.356715  1.357146  1.357512   1.35782  1.358076  1.358287   1.35846    1.3586  1.358714  1.358805  1.358879  1.358938  1.358984  1.359022  1.359051  1.359075  1.359093  1.359108  1.359119  1.359128  1.359135          1.359141 | 0.741516  0.737663   0.73354  0.729131   0.72442   0.71939  0.714026  0.708312  0.702235  0.695787  0.688961  0.681759   0.67419  0.666272  0.658038  0.649535  0.640825  0.631988  0.623121  0.614332   0.60574  0.597465  0.589619    0.5823  0.575587  0.569529  0.564149  0.559442  0.555381   0.55192  0.549005  0.546574  0.544564  0.542915  0.541569  0.540479  0.539598  0.538891  0.538324  0.537872  0.537511  0.537225  0.536998  0.536818  0.536676  0.536564  0.536476  0.536407  0.536352      0.536309 | -0.428627  -0.423248  -0.417536  -0.411476  -0.405055  -0.398258  -0.391075    -0.3835   -0.37553   -0.36717  -0.358434  -0.349344  -0.339939  -0.330268    -0.3204   -0.31042  -0.300428  -0.290541   -0.28088  -0.271574  -0.262742  -0.254492  -0.246909  -0.240049   -0.23394  -0.228582  -0.223947  -0.219989  -0.216649  -0.213857  -0.211546  -0.209647  -0.208098  -0.206841  -0.205826   -0.20501  -0.204357  -0.203835   -0.20342   -0.20309  -0.202829  -0.202622  -0.202459  -0.202331   -0.20223   -0.20215  -0.202088  -0.202039  -0.202001     -0.201971 | 0.691868   0.69072  0.689532  0.688304  0.687039   0.68574  0.684412  0.683059  0.681689   0.68031  0.678933  0.677569  0.676232  0.674936  0.673696  0.672527  0.671444  0.670458  0.669579  0.668811  0.668156  0.667609  0.667162  0.666805  0.666526  0.666311  0.666149  0.666028  0.665939  0.665874  0.665826  0.665792  0.665767   0.66575  0.665737  0.665728  0.665722  0.665717  0.665714  0.665712  0.665711   0.66571  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709  0.665709      0.665709 | 679.625318 | 664.555012  634.317386   603.89196  573.400675    542.9707  512.734128  482.827412  453.390487  424.565545  396.495419  369.321557  343.181577  318.206421  294.517159  272.221552  251.410512  232.154695  214.501462  198.472537  184.062662  171.239499  159.944968  150.098021  141.598685  134.333038  128.178639  123.009901  118.702933   115.13951  112.209993  109.815171  107.867144  106.289423  105.016452  103.992763   103.17193  102.515441  101.991588   101.57443  101.242856  100.979758  100.771329  100.606456  100.476224  100.373497  100.292572  100.228905  100.178879   100.13962           100.10885 |   100.0951 | 1.174319  1.160534  1.146151  1.131174  1.115614  1.099482    1.0828  1.065596  1.047906   1.02978  1.011277  0.992474   0.97346  0.954344  0.935249  0.916316    0.8977  0.879565  0.862081  0.845414  0.829719  0.815129  0.801748  0.789645  0.778847  0.769343  0.761086  0.753999  0.747984   0.74293  0.738722  0.735247  0.732397  0.730074  0.728189  0.726668  0.725444  0.724464   0.72368  0.723055  0.722559  0.722164  0.721852  0.721605   0.72141  0.721256  0.721135   0.72104  0.720965       0.720907 | 5.563093  5.463247  5.359671  5.252477   5.14181  5.027852  4.910833  4.791037  4.668808  4.544561  4.418792  4.292076   4.16508  4.038559  3.913355  3.790383  3.670612  3.555042   3.44466  3.340398  3.243082   3.15338  3.071766  2.998486  2.933547  2.876736   2.82764  2.785699  2.750245  2.720562  2.695921  2.675619  2.659001  2.645476  2.634522  2.625688   2.61859  2.612905  2.608364  2.604746   2.60187  2.599588  2.597781  2.596352  2.595224  2.594335  2.593635  2.593085  2.592653        2.592314 | 679.625318  649.507798  619.142613  588.649035  558.151652    527.7802  497.669136  467.956936  438.785049  410.296493  382.634041  355.937985  330.343483   305.97752  282.955558   261.37799  241.326599  222.861243  206.017072  190.802589  177.198852  165.160069  154.615665  145.473774  137.625879  130.952166  125.327086  120.624586  116.722591  113.506479  110.871442  108.723801  106.981433  105.573524   104.43985  103.529788    102.8012   102.21929  101.755522  101.386625   101.09371  100.861509  100.677721  100.532462  100.417816  100.327454  100.256324  100.200404  100.156498  100.122067    100.0951 |  100.0 | 579.625318 |   100.0951 |
    | 20:00 | 8.943624  8.783908  8.617563  8.444748  8.265663  8.080562  7.889758  7.693642   7.49269  7.287477   7.07869  6.867142  6.653779   6.43969  6.226109  6.014405  5.806074  5.602706  5.405946  5.217437  5.038748  4.871299  4.716276  4.574562  4.446683   4.33278  4.232615  4.145614  4.070923  4.007492  3.954152  3.909694  3.872928  3.842732  3.818079  3.798056  3.781866  3.768825  3.758355  3.749975  3.743283  3.737952  3.733714  3.730351  3.727686  3.725579  3.723914  3.722601  3.721567             3.720753 | 534.096426  517.543773  500.575387  483.240058   465.59091  447.685822  429.587807  411.365304  393.092374  374.848733  356.719603  338.795306  321.170562  303.943418  287.213765  271.081389  255.643554  240.992147  227.210459  214.369795  202.526143  191.717248  181.960431  173.251534  165.565198  158.856584  153.064384  148.114787  143.925935  140.412345  137.488877  135.073932  133.091764  131.473903   130.15982  129.096989  128.240526  127.552564  127.001494  126.561164  126.210089  125.930727  125.708827  125.532859  125.393529  125.283368   125.19639  125.127807  125.073797  125.031317 | 104.436241  102.839076  101.175628   99.447479   97.656634   95.805618   93.897583   91.936424   89.926901   87.874767   85.786899   83.671418    81.53779   79.396904   77.261085   75.144047   73.060736   71.027058   69.059461   67.174369   65.387483   63.712989   62.162759   60.745623   59.466831   58.327795   57.326152    56.45614   55.709233   55.074921   54.541521   54.096938   53.729278   53.427316    53.18079    52.98056   52.818657   52.688246   52.583552   52.499746   52.432831   52.379524   52.337143    52.30351   52.276865   52.255788   52.239141    52.22601   52.215668     52.207531 | 1.775914  1.757304  1.737807  1.717426  1.696166  1.674037  1.651057  1.627254  1.602662  1.577331  1.551324  1.524722  1.497624  1.470152  1.442451  1.414688  1.387056  1.359769  1.333058  1.307166  1.282336  1.258801   1.23677  1.216416  1.197866   1.18119  1.166401  1.153458  1.142272  1.132716  1.124639  1.117878  1.112267  1.107644   1.10386  1.100781  1.098287  1.096275  1.094659  1.093363  1.092328  1.091503  1.090847  1.090326  1.089913  1.089587  1.089329  1.089125  1.088965  1.088839 |  1.33929  1.339491  1.339712  1.339953  1.340217  1.340506  1.340823  1.341169  1.341548  1.341963  1.342417  1.342913  1.343454  1.344041  1.344678  1.345364    1.3461  1.346883  1.347709   1.34857  1.349457  1.350358  1.351259  1.352145  1.353001  1.353811  1.354565  1.355253   1.35587  1.356414  1.356887  1.357292  1.357634  1.357921  1.358158  1.358354  1.358514  1.358644  1.358748  1.358833  1.358901  1.358955  1.358998  1.359032  1.359059  1.359081  1.359098  1.359112  1.359122           1.35913 | 0.746698  0.743274  0.739604   0.73567  0.731459  0.726954  0.722139  0.716998  0.711515  0.705677  0.699472  0.692891  0.685932    0.6786  0.670906  0.662876  0.654547  0.645973  0.637222  0.628381  0.619552  0.610847  0.602386  0.594287  0.586656  0.579585   0.57314  0.567359  0.562254  0.557811  0.553996  0.550759  0.548042  0.545782  0.543919  0.542392  0.541149  0.540143  0.539331  0.538679  0.538157   0.53774  0.537408  0.537144  0.536935  0.536769  0.536638  0.536535  0.536453      0.536389 |  -0.43617  -0.431322  -0.426164  -0.420681  -0.414858  -0.408678   -0.40213  -0.395201  -0.387881  -0.380167  -0.372059  -0.363565  -0.354702  -0.345499  -0.335999  -0.326259  -0.316354  -0.306376  -0.296432  -0.286639  -0.277124  -0.268011  -0.259415  -0.251436  -0.244147  -0.237593  -0.231791  -0.226728  -0.222369  -0.218663  -0.215547  -0.212951  -0.210807  -0.209049  -0.207616  -0.206456   -0.20552  -0.204768  -0.204166  -0.203685  -0.203302  -0.202998  -0.202757  -0.202567  -0.202416  -0.202297  -0.202204   -0.20213  -0.202073     -0.202028 |  0.69359  0.692515  0.691396  0.690236  0.689035  0.687795  0.686518  0.685209  0.683872  0.682513   0.68114  0.679763  0.678391  0.677037  0.675716  0.674442   0.67323  0.672095   0.67105  0.670107  0.669272   0.66855  0.667938  0.667432  0.667022  0.666696  0.666444   0.66625  0.666105  0.665996  0.665917  0.665858  0.665816  0.665785  0.665763  0.665747  0.665735  0.665727  0.665721  0.665717  0.665714  0.665712  0.665711   0.66571   0.66571  0.665709  0.665709  0.665709  0.665709      0.665709 | 722.742733 | 708.218133  678.975348  649.321351  619.371669  589.247215  559.074282   528.98434  499.113599  469.602278  440.593564  412.232203  384.662695  358.027083  332.462327  308.097307  285.049516  263.421573   243.29772  224.740554  207.788261   192.45266  178.718363  166.543256  155.860439  146.581532  138.601145   131.80207  126.060736  121.252399  117.255646  113.955949  111.248142   109.03786  107.242092  105.789032  104.617444  103.675727  102.920835  102.317154  101.835425  101.451752  101.146711  100.904577  100.712668  100.560781  100.440733  100.345973   100.27127  100.212451          100.166194 | 100.145491 | 1.193411  1.180727  1.167435  1.153535  1.139031  1.123928  1.108239  1.091979  1.075172  1.057851  1.040058  1.021845  1.003281  0.984447   0.96544  0.946375  0.927383   0.90861  0.890216  0.872368  0.855235  0.838978  0.823745  0.809657  0.796806  0.785243   0.77498  0.765991  0.758217  0.751573  0.745954  0.741248  0.737341  0.734122  0.731486  0.729341  0.727603  0.726201  0.725074  0.724171  0.723449  0.722874  0.722416  0.722053  0.721765  0.721538  0.721358  0.721216  0.721104       0.721016 | 5.702291  5.609692  5.513161  5.412773  5.308628  5.200854  5.089615  4.975114  4.857607  4.737405  4.614885  4.490497  4.364771  4.238322  4.111856  3.986164  3.862122  3.740669   3.62279  3.509484  3.401719   3.30039  3.206263  3.119933  3.041782  2.971962  2.910392  2.856777  2.810643  2.771384  2.738313  2.710707  2.687848  2.669054  2.653697  2.641214  2.631115  2.622976   2.61644  2.611206  2.607026  2.603696  2.601047  2.598945   2.59728  2.595962  2.594921    2.5941  2.593454        2.592945 | 722.742733  693.730479  664.250376  634.415266  604.343331  574.158186  543.988788  513.969122  484.237632  454.936354  426.209697  398.202844  371.059756  344.920747   319.91966  296.180692  273.814953  252.916911   233.56093  215.798158  199.654072  185.126992  172.187844  160.781358  150.828734  142.231632  134.877144  128.643276  123.404418  119.036292  115.420046  112.445268  110.011919  108.031272  106.426048   105.12997  104.086923   103.24991  102.579925  102.044835  101.618335  101.279006   101.00948   100.79573  100.626459  100.492596  100.386876  100.303488  100.237797   100.18611  100.145491 |  100.0 | 622.742733 | 100.145491 |
    | 20:30 | 9.157834  9.011135  8.857496  8.697029  8.529875  8.356212  8.176267  7.990321  7.798724  7.601905  7.400384  7.194787  6.985854  6.774457  6.561603  6.348442  6.136267  5.926503  5.720688  5.520438  5.327404  5.143206   4.96936  4.807198  4.657792  4.521886  4.399854  4.291685  4.197007  4.115131  4.045121  3.985879  3.936219   3.89494  3.860881   3.83296  3.810199  3.791734  3.776816  3.764808  3.755171  3.747458  3.741301  3.736395  3.732493  3.729396  3.726941  3.724998  3.723463             3.722251 | 556.697087  541.169798  525.138651  508.647015  491.741933  474.474578  456.900685  439.080946  421.081337  402.973367  384.834192  366.746571  348.798601  331.083191  313.697208  296.740236  280.312905  264.514758  249.441653  235.182755  221.817245  209.410935  198.013084  187.653748  178.342033  170.065568  162.791368  156.468089  151.029457  146.398466  142.491854  139.224359  136.512396  134.276887  132.445211  130.952303  129.741064  128.762242  127.973953  127.341004  126.834111  126.429103  126.106165  125.849141  125.644922  125.482911   125.35457  125.253042  125.172829  125.109537 | 106.578336   105.11135  103.574963  101.970291  100.298747    98.56212   96.762667   94.903208    92.98724   91.019049   89.003842   86.947866   84.858541    82.74457   80.616029   78.484423   76.362675   74.265033   72.206877   70.204381   68.274043   66.432061   64.693599   63.071982   61.577923   60.218862   58.998537   57.916852   56.970072   56.151307   55.451213   54.858793   54.362192     53.9494   53.608807   53.329598   53.101989    52.91734   52.768163   52.648078    52.55171   52.474585   52.413007   52.363946   52.324931    52.29396   52.269412   52.249984    52.23463     52.222512 |  1.80071  1.783749  1.765892  1.747136  1.727482  1.706933  1.685498   1.66319  1.640032  1.616053  1.591295  1.565815  1.539682  1.512984  1.485831  1.458354  1.430707  1.403067  1.375638  1.348639  1.322307  1.296886  1.272615   1.24972  1.228395  1.208797  1.191028  1.175137  1.161115  1.148901  1.138391  1.129447  1.121915  1.115628  1.110424  1.106145  1.102649  1.099808  1.097508  1.095655  1.094167  1.092974  1.092021  1.091262  1.090658  1.090178  1.089798  1.089497  1.089259  1.089071 | 1.339034  1.339207  1.339397  1.339605  1.339833  1.340082  1.340355  1.340653  1.340981  1.341339  1.341731  1.342161   1.34263  1.343143  1.343701  1.344307  1.344962  1.345667  1.346421   1.34722  1.348059  1.348929  1.349822  1.350723  1.351618  1.352491  1.353328  1.354116  1.354843  1.355502  1.356089  1.356604  1.357049  1.357428  1.357748  1.358015  1.358235  1.358416  1.358564  1.358684   1.35878  1.358858  1.358921  1.358971   1.35901  1.359042  1.359067  1.359087  1.359103          1.359115 |  0.75114  0.748116  0.744864  0.741371  0.737623  0.733606  0.729303  0.724699  0.719778  0.714523   0.70892  0.702956  0.696621  0.689909   0.68282  0.675361   0.66755  0.659416  0.651003   0.64237  0.633595  0.624768  0.615997  0.607397  0.599088  0.591182  0.583781  0.576967  0.570795  0.565292  0.560459  0.556274  0.552697  0.549673  0.547144  0.545046  0.543319  0.541908   0.54076  0.539832  0.539084  0.538483  0.538002  0.537618  0.537312  0.537069  0.536876  0.536723  0.536602      0.536507 |  -0.44279  -0.438443  -0.433807  -0.428869  -0.423613  -0.418023  -0.412085  -0.405785  -0.399109  -0.392047  -0.384592  -0.376741  -0.368498  -0.359875  -0.350893  -0.341586  -0.332004   -0.32221  -0.312287  -0.302331  -0.292454  -0.282779   -0.27343  -0.264529  -0.256186  -0.248488  -0.241499  -0.235253  -0.229753  -0.224979  -0.220888  -0.217423  -0.214519  -0.212107   -0.21012  -0.208493   -0.20717  -0.206099  -0.205236  -0.204543  -0.203988  -0.203545  -0.203193  -0.202913  -0.202691  -0.202515  -0.202376  -0.202266   -0.20218     -0.202112 | 0.695172  0.694172  0.693128  0.692041   0.69091  0.689738  0.688525  0.687274  0.685987  0.684669  0.683325  0.681962  0.680588  0.679213   0.67785  0.676509  0.675207  0.673957  0.672776  0.671677  0.670673  0.669773  0.668984  0.668306  0.667737  0.667269  0.666894  0.666598   0.66637  0.666196  0.666065  0.665968  0.665897  0.665844  0.665806  0.665779  0.665759  0.665744  0.665734  0.665726  0.665721  0.665717  0.665715  0.665713  0.665711   0.66571   0.66571  0.665709  0.665709      0.665709 | 762.408181 | 748.636706  720.807656  692.354598  663.386083  634.015926  604.363404  574.553318  544.715878  514.986376  485.504598   456.41395  427.860239  399.990087  372.948944  346.878696  321.914872  298.183514  275.797784  254.854454   235.43049  217.579962  201.331596  186.687252  173.621605  162.083224   151.99707  143.268304   135.78708  129.433887  124.084928   119.61706  115.911928  112.859083  110.358034  108.319312  106.664733  105.327035  104.249121  103.383056  102.688972  102.133966   101.69106  101.338248  101.057669  100.834872  100.658212  100.518321  100.407691  100.320309          100.251373 | 100.220473 | 1.210305  1.198749   1.18658  1.173795  1.160394  1.146378  1.131751  1.116524  1.100708  1.084325  1.067401  1.049974  1.032089  1.013806  0.995198  0.976354  0.957377   0.93839   0.91953  0.900948  0.882808  0.865277  0.848522  0.832701   0.81795   0.80438  0.792066  0.781043   0.77131  0.762825  0.755519  0.749299  0.744058  0.739682  0.736058  0.733078  0.730642  0.728662   0.72706  0.725768  0.724731  0.723899  0.723235  0.722706  0.722285   0.72195  0.721685  0.721475  0.721309       0.721178 | 5.826356  5.741407  5.652367  5.559288  5.462236  5.361301  5.256595  5.148262  5.036486  4.921494  4.803566  4.683042  4.560331  4.435916  4.310365  4.184332  4.058558  3.933871  3.811174  3.691428  3.575627   3.46476  3.359772  3.261508   3.17067   3.08777  3.013101  2.946722  2.888464   2.83796  2.794682   2.75799  2.727182  2.701537  2.680352  2.662968  2.648785  2.637271  2.627964  2.620469  2.614452  2.609635  2.605788  2.602722  2.600284  2.598348  2.596814  2.595599  2.594639        2.593882 | 762.408181  734.914669  706.743967  678.002042  648.799994  619.254318  589.487046  559.625729  529.803227  500.157268  470.829726  441.965589  413.711568  386.214316  359.618245  334.062922  309.680096    286.5904  264.899855  244.696334  226.046231  208.991592  193.548041  179.703778  167.419923  156.632313  147.254718  139.183269  132.301667  126.486706   121.61356  117.560407  114.212091  111.462701  109.217079  107.391433  105.913218  104.720524  103.761148  102.991514  102.375552   101.88361  101.491456  101.179382  100.931425  100.734699  100.578835  100.455506  100.358044  100.281118  100.220473 |  100.0 | 662.408181 | 100.220473 |
    | 21:00 | 9.345031  9.211955  9.071689  8.924314   8.76993  8.608662  8.440671  8.266155  8.085367   7.89862  7.706298   7.50887  7.306903   7.10107  6.892169  6.681128  6.469015   6.25704  6.046551  5.839022  5.636027  5.439201  5.250186  5.070568  4.901795  4.745101  4.601433  4.471388   4.35519  4.252684  4.163368  4.086454   4.02094   3.96569  3.919514  3.881228  3.849705  3.823908  3.802907  3.785887  3.772148  3.761094  3.752227  3.745131  3.739466  3.734952  3.731361   3.72851  3.726249             3.724458 | 576.823462  562.479881  547.553048  532.081624  516.107307  499.675278  482.834654   465.63892  448.146339  430.420303  412.529628  394.548731  376.557676  358.642036  340.892515  323.404291  306.276002  289.608333  273.502162  258.056237  243.364414   229.51253  216.575062  204.611806  193.664888  183.756454  174.887398  167.037383  160.166269  154.216869  149.118706  144.792364  141.153894  138.118862  135.605681  133.538084   131.84672  130.469962  129.354089  128.453017  127.727734  127.145576   126.67944  126.307009  126.010018  125.773597  125.585691  125.436565  125.318377   125.22483 | 108.450308  107.119548  105.716891  104.243141    102.6993  101.086624   99.406705    97.66155   95.853674   93.986201   92.062982   90.088704   88.069027   86.010701   83.921691   81.811282    79.69015   77.570398    75.46551   73.390222   71.360271   69.392007   67.501861   65.705678   64.017949   62.451014   61.014326   59.713882   58.551903    57.52684   56.633684   55.864544     55.2094   54.656905   54.195144   53.812282    53.49705   53.239076   53.029065   52.858872   52.721482   52.610945   52.522269   52.451312    52.39466   52.349517    52.31361   52.285096   52.262487     52.244584 | 1.822229  1.806945   1.79076   1.77367   1.75567  1.736761  1.716944  1.696224  1.674613  1.652129  1.628796   1.60465  1.579738  1.554123  1.527882  1.501113  1.473932  1.446481  1.418923  1.391447  1.364262  1.337595  1.311686  1.286779  1.263107  1.240884  1.220292  1.201464  1.184483  1.169374  1.156107  1.144604  1.134746  1.126389  1.119374  1.113535  1.108713  1.104756  1.101528  1.098907  1.096788  1.095082  1.093711  1.092614  1.091738  1.091039  1.090483  1.090041  1.089691  1.089413 | 1.338823  1.338971  1.339135  1.339314   1.33951  1.339724  1.339959  1.340217  1.340499  1.340807  1.341146  1.341517  1.341923  1.342367  1.342852  1.343382  1.343958  1.344582  1.345256   1.34598   1.34675  1.347564  1.348415  1.349293  1.350189  1.351087  1.351973  1.352831  1.353648  1.354411   1.35511   1.35574  1.356297  1.356783  1.357201  1.357556  1.357854  1.358102  1.358306  1.358474   1.35861  1.358721   1.35881  1.358882  1.358939  1.358985  1.359022  1.359051  1.359074          1.359092 | 0.754891  0.752237  0.749374  0.746289   0.74297  0.739404  0.735576  0.731471  0.727073  0.722367  0.717336  0.711965  0.706241   0.70015  0.693685  0.686843  0.679626  0.672045  0.664123  0.655894  0.647409  0.638734   0.62995  0.621157  0.612465   0.60399  0.595851  0.588155  0.580998  0.574449  0.568553  0.563327  0.558762  0.554828  0.551479  0.548659  0.546306  0.544361  0.542763  0.541459    0.5404  0.539544  0.538855  0.538301  0.537858  0.537505  0.537223  0.536999  0.536822      0.536681 | -0.448531  -0.444655  -0.440511  -0.436086  -0.431364  -0.426332  -0.420974  -0.415275  -0.409221  -0.402797  -0.395993  -0.388799  -0.381209  -0.373224   -0.36485  -0.356103   -0.34701  -0.337611   -0.32796  -0.318128  -0.308204  -0.298291  -0.288504  -0.278967  -0.269805  -0.261134  -0.253056   -0.24565  -0.238966  -0.233027  -0.227825  -0.223332  -0.219498  -0.216263  -0.213561  -0.211322  -0.209482  -0.207978  -0.206757  -0.205769  -0.204973  -0.204335  -0.203824  -0.203416  -0.203091  -0.202833  -0.202629  -0.202466  -0.202338     -0.202237 | 0.696608  0.695684  0.694717  0.693706  0.692651  0.691552  0.690411  0.689227  0.688003  0.686741  0.685445  0.684119   0.68277  0.681405  0.680033  0.678664  0.677311  0.675986  0.674706  0.673484  0.672336  0.671275  0.670312  0.669457  0.668713  0.668079  0.667551  0.667121  0.666778   0.66651  0.666303  0.666147   0.66603  0.665943  0.665879  0.665832  0.665798  0.665773  0.665755  0.665742  0.665733  0.665726  0.665721  0.665717  0.665715  0.665713  0.665711  0.665711   0.66571       0.66571 | 797.923464 | 785.096097  759.067949  732.216245  704.642249   676.45219  647.757589  618.675504  589.328644  559.845334  530.359289  501.009154  471.937783  443.291202  415.217224  387.863686  361.376285  335.896008  311.556199  288.479303  266.773417  246.528805  227.814591  210.675915  195.131832  181.174263  168.768232  157.853513  148.347671  140.150297  133.148051  127.220052  122.243112  118.096367   114.66501  111.842972  109.534572  107.655242  106.131525  104.900552  103.909171  103.112916  102.474905  101.964764  101.557634  101.233266  100.975238  100.770277  100.607689  100.478879          100.376955 | 100.331203 | 1.224961  1.214552  1.203527  1.191881  1.179613  1.166721  1.153206  1.139071  1.124322   1.10897  1.093032  1.076531  1.059498  1.041974   1.02401  1.005672  0.987039  0.968206  0.949285  0.930402  0.911702  0.893342  0.875485  0.858302  0.841954  0.826591  0.812341    0.7993  0.787527  0.777044  0.767832  0.759838  0.752985  0.747172   0.74229  0.738225  0.734866   0.73211  0.729861  0.728035  0.726558  0.725369  0.724413  0.723648  0.723037   0.72255  0.722162  0.721854   0.72161       0.721417 | 5.934663  5.857679   5.77648    5.6911  5.601584  5.507993  5.410403  5.308914  5.203654  5.094784  4.982508  4.867076  4.748793   4.62803  4.505227  4.380902   4.25566  4.130191  4.005273  3.881764  3.760594   3.64274  3.529195  3.420936   3.31887   3.22379  3.136324  3.056899  2.985716  2.922742  2.867729  2.820244  2.779713  2.745471  2.716808  2.693011  2.673396  2.657329  2.644239  2.633624  2.625051  2.618151  2.612613  2.608181  2.604641   2.60182  2.599576  2.597794  2.596381        2.595262 | 797.923464  772.329174  745.861719  718.619959  690.707528  662.233208  633.311228  604.061444  574.609394  545.086176  515.628118    486.3762  457.475187  429.072432  401.316309  374.354257  348.330412  323.382844  299.640442  277.219523  256.220306  236.723444   218.78686  202.443177  187.698061  174.529743  162.889945  152.706259  143.885867  136.320318  129.890901    124.4741  119.946643   116.18975  113.092359  110.553262  108.482246  106.800392  105.439756  104.342615  103.460477  102.752989  102.186826  101.734649  101.374146  101.087194  100.859125  100.678107   100.53462  100.421025  100.331203 |  100.0 | 697.923464 | 100.331203 |
    | 21:30 | 9.504436  9.385452  9.259074  9.125364  8.984394  8.836255  8.681057  8.518942  8.350087  8.174715  7.993104    7.8056  7.612622  7.414681  7.212389  7.006472  6.797781  6.587303  6.376164  6.165632  5.957108  5.752108  5.552233  5.359126  5.174411  4.999624  4.836133  4.685063  4.547224  4.423068  4.312668   4.21573  4.131637  4.059513  3.998303   3.94685  3.903966  3.868493  3.839343  3.815526   3.79616  3.780482  3.767834  3.757663  3.749506  3.742979  3.737769  3.733616  3.730312             3.727688 | 594.238098  581.215344  567.538362  553.241771  538.362623  522.940837  507.019634  490.645983  473.871029  456.750512  439.345139  421.720912  403.949357  386.107648  368.278571  350.550287  333.015836  315.772336  298.919809  282.559596  266.792311  251.715344  237.419952  223.988037  211.488801   199.97554  189.482908  180.024998  171.594579  164.163678  157.685547  152.097846  147.326669   143.29094    139.9067  137.090871  134.764245  132.853592  131.292929   130.02406  128.996572  128.167433  127.500363  126.965087  126.536549   126.19416  125.921095  125.703675   125.53082  125.393585 | 110.044364  108.854522   107.59074  106.253637  104.843943   103.36255  101.810573  100.189419   98.500866   96.747146   94.931042   93.055995   91.126215   89.146806   87.123885   85.064715   82.977809   80.873028   78.761642   76.656323   74.571082   72.521079    70.52233   68.591258   66.744108   64.996237    63.36133   61.850626   60.472238   59.230681    58.12668     57.1573   56.316369   55.595133   54.983034   54.468499    54.03966   53.684931   53.393432   53.155256   52.961603   52.804817   52.678338   52.576627   52.495057   52.429793   52.377685   52.336159    52.30312     52.276875 | 1.840446  1.826858  1.812365  1.796963  1.780648  1.763415  1.745263  1.726192  1.706206  1.685313  1.663526  1.640866  1.617363  1.593059  1.568005  1.542271  1.515943  1.489125  1.461944  1.434551   1.40712  1.379846  1.352947  1.326656  1.301213  1.276861  1.253824  1.232305  1.212465  1.194421   1.17823  1.163897   1.15137  1.140557  1.131327   1.12353  1.117005  1.111588  1.107124  1.103468  1.100489  1.098074  1.096122  1.094552  1.093291  1.092281  1.091475  1.090832   1.09032  1.089913 | 1.338651  1.338778  1.338918  1.339072   1.33924  1.339424  1.339627  1.339848  1.340091  1.340357  1.340649  1.340969  1.341319  1.341703  1.342123  1.342582  1.343084  1.343631  1.344225  1.344868  1.345561  1.346301  1.347088  1.347916  1.348777  1.349661  1.350557  1.351449  1.352324  1.353165   1.35396  1.354696  1.355367  1.355967  1.356495  1.356953  1.357345  1.357676  1.357953  1.358183  1.358373  1.358527  1.358653  1.358755  1.358838  1.358904  1.358957  1.358999  1.359033          1.359059 | 0.757992  0.755685  0.753184  0.750478  0.747556  0.744407  0.741018  0.737374  0.733462  0.729266   0.72477  0.719958  0.714815  0.709325  0.703477  0.697259  0.690665  0.683694  0.676352  0.668655  0.660629  0.652316  0.643772  0.635069  0.626296  0.617556  0.608963  0.600633  0.592682  0.585212  0.578308  0.572031  0.566414  0.561462  0.557159  0.553468  0.550338  0.547712  0.545527  0.543725  0.542247  0.541043  0.540066  0.539277  0.538643  0.538133  0.537726    0.5374  0.537141      0.536935 | -0.453435  -0.450005  -0.446324   -0.44238  -0.438161  -0.433653  -0.428841  -0.423712  -0.418249  -0.412438  -0.406265  -0.399717  -0.392784  -0.385457  -0.377734  -0.369618  -0.361117  -0.352254  -0.343058  -0.333577  -0.323872  -0.314019  -0.304113  -0.294263  -0.284588  -0.275211  -0.266255  -0.257831  -0.250031  -0.242924  -0.236547  -0.230913  -0.226004  -0.221783  -0.218196   -0.21518  -0.212668  -0.210593   -0.20889  -0.207501  -0.206374  -0.205463   -0.20473  -0.204142  -0.203672  -0.203296  -0.202997   -0.20276  -0.202571     -0.202422 | 0.697894  0.697048  0.696158  0.695225  0.694248  0.693227  0.692161  0.691051  0.689898  0.688703  0.687468  0.686197  0.684893  0.683562   0.68221  0.680844  0.679476  0.678115  0.676775  0.675469  0.674213  0.673022  0.671909  0.670888  0.669969  0.669159  0.668459  0.667868  0.667381  0.666986  0.666674   0.66643  0.666244  0.666104  0.665999  0.665921  0.665863  0.665821  0.665791  0.665768  0.665752   0.66574  0.665731  0.665725   0.66572  0.665717  0.665715  0.665713  0.665712      0.665711 | 828.701418 |   816.9889  793.109331  768.222123  742.421108  715.804675  688.476186  660.544327  632.123381  603.333393  574.300207  545.155324  516.035568  487.082493  458.441524  430.260755  402.689407  375.875879  349.965419  325.097392  301.402208  278.997978  257.987032  238.452498  220.455154  204.030864  189.188885  175.911311  164.153873   153.84815  144.905101  137.219651  130.675902  125.152492  120.527603  116.683242  113.508554  110.902082  108.773029  107.041683  105.639198  104.506921  103.595458  102.863613  102.277302  101.808513  101.434352  101.136197  100.898958  100.710445          100.560843 | 100.493595 | 1.237366  1.228113  1.218243  1.207752  1.196636  1.184892  1.172518  1.159514  1.145882  1.131625  1.116753  1.101278  1.085221  1.068607  1.051472  1.033862  1.015832  0.997456  0.978817  0.960017  0.941175  0.922424  0.903914  0.885805  0.868262  0.851454  0.835538  0.820655  0.806921  0.794418   0.78319  0.773241  0.764541  0.757025  0.750607  0.745182  0.740641  0.736869   0.73376  0.731213  0.729137  0.727454  0.726094  0.724999   0.72412  0.723416  0.722854  0.722406  0.722049       0.721765 | 6.026813  5.958037  5.884943  5.807559  5.725915   5.64005  5.550018  5.455885  5.357739  5.255691  5.149885    5.0405   4.92776  4.811939  4.693369  4.572451  4.449656   4.32554  4.200741  4.075985  3.952086  3.829929  3.710466  3.594683  3.483568  3.378074  3.279066  3.187273  3.103246  3.027323  2.959613  2.899997  2.848151  2.803586   2.76569  2.733781  2.707147  2.685089  2.666944  2.652105  2.640032  2.630252  2.622358  2.616008  2.610913  2.606836  2.603581  2.600986  2.598921         2.59728 | 828.701418  805.346262  780.937487  755.566733  729.329986  702.328005  674.666717  646.457533  617.817575  588.869785  559.742886  530.571153   501.49396  472.655072  444.201619  416.282743  389.047856  362.644506   337.21584  312.897692  289.815339   268.08005  247.785566  229.004739  211.786581  196.154045   182.10282  169.601406  158.592616  148.996508  140.714546   133.63465  127.636628  122.597499  118.396232  114.917588  112.054906  109.711828  107.803089  106.254555  105.002723   103.99387  103.183019  102.532843  102.012583  101.597049  101.265714  101.001916  100.792184  100.625656  100.493595 |  100.0 | 728.701418 | 100.493595 |
    | 22:00 | 9.635642  9.531101  9.418994  9.299375    9.1723  9.037835   8.89606  8.747073  8.590997  8.427989  8.258248  8.082018  7.899607  7.711391  7.517828  7.319472  7.116981  6.911133  6.702834  6.493129  6.283201  6.074372   5.86809  5.665909  5.469448  5.280347  5.100197  4.930473   4.77245  4.627132  4.495188  4.376921  4.272257  4.180774   4.10175  4.034236  3.977137  3.929285  3.889506  3.856673  3.829741  3.807768  3.789922  3.775486  3.763847  3.754492  3.746991   3.74099  3.736199              3.73238 | 608.762608   597.17591   584.87214  571.882458  558.239913  543.979849  529.140312  513.762479   497.89109  481.574877  464.866981   447.82534  430.513042    412.9986  395.356136  377.665433  360.011811   342.48579  325.182463  308.200552  291.641066  275.605534  260.193781  245.501268   231.61605  218.615499     206.563  195.504915  185.468155  176.458703  168.461381   161.44099  155.344775   150.10597  145.647998  141.888846  138.745153  136.135666  133.983867  132.219735  130.780705  129.611981  128.666363  127.903745  127.290445  126.798435  126.404575  126.089885  125.838881  125.638981 |  111.35642  110.311007  109.189939  107.993746  106.722998  105.378352  103.960599  102.470725  100.909969   99.279895   97.582476   95.820179   93.996068   92.113908   90.178283    88.19472    86.16981    84.11133   82.028344    79.93129   77.832007   75.743717   73.680904   71.659091   69.694483   67.803466    66.00197    64.30473   62.724502   61.271315   59.951878   58.769208   57.722574   56.807741   56.017497   55.342361   54.771371   54.292848   53.895059   53.566731   53.297412   53.077675   52.899218   52.754857   52.638474   52.544921    52.46991   52.409901   52.361986     52.323796 | 1.855369  1.843484  1.830694  1.816993  1.802377  1.786842  1.770383  1.752997  1.734683  1.715443  1.695282  1.674212  1.652248  1.629416  1.605749  1.581295  1.556112  1.530275  1.503879  1.477037  1.449885  1.422583  1.395314  1.368283  1.341713  1.315841  1.290907  1.267148   1.24478  1.223991  1.204923   1.18767  1.172268    1.1587  1.146897  1.136751  1.128124  1.120861  1.114799   1.10978  1.105651  1.102275  1.099528  1.097303  1.095507  1.094062  1.092902  1.091973  1.091232   1.09064 | 1.338516  1.338624  1.338742  1.338873  1.339017  1.339175  1.339349  1.339539  1.339748  1.339977  1.340229  1.340504  1.340806  1.341137  1.341499  1.341897  1.342332  1.342807  1.343326   1.34389  1.344503  1.345164  1.345875  1.346633  1.347434  1.348274  1.349143  1.350031  1.350925  1.351809   1.35267  1.353492  1.354262  1.354971  1.355613  1.356183  1.356682  1.357112  1.357479  1.357788  1.358046  1.358259  1.358434  1.358577  1.358693  1.358787  1.358863  1.358924  1.358973          1.359011 | 0.760483  0.758502   0.75634  0.753987  0.751435  0.748672  0.745688  0.742471  0.739007  0.735282  0.731282   0.72699  0.722392  0.717472  0.712214  0.706604   0.70063  0.694285  0.687562  0.680464  0.673001  0.665194  0.657073  0.648687  0.640097  0.631383  0.622639  0.613973  0.605499  0.597334  0.589588  0.582357  0.575717  0.569716  0.564377  0.559697  0.555649  0.552192  0.549272  0.546829  0.544802  0.543134  0.541769  0.540658  0.539757  0.539031  0.538446  0.537978  0.537603      0.537303 | -0.457545  -0.454535  -0.451291  -0.447802  -0.444055  -0.440039  -0.435741  -0.431146  -0.426241   -0.42101  -0.415439  -0.409513   -0.40322  -0.396546  -0.389483  -0.382024  -0.374169  -0.365924  -0.357302  -0.348329  -0.339041  -0.329491  -0.319746   -0.30989  -0.300022  -0.290257  -0.280714  -0.271519  -0.262788  -0.254625  -0.247114   -0.24031  -0.234242  -0.228909  -0.224286  -0.220328  -0.216977   -0.21417  -0.211838  -0.209916  -0.208341  -0.207059  -0.206019   -0.20518  -0.204505  -0.203964  -0.203531  -0.203185   -0.20291     -0.202691 | 0.699028  0.698259  0.697448  0.696594  0.695696  0.694753  0.693766  0.692734  0.691657  0.690537  0.689373  0.688168  0.686923  0.685644  0.684333  0.682998  0.681644   0.68028  0.678918  0.677568  0.676243  0.674959  0.673731  0.672572  0.671497  0.670518  0.669643  0.668878  0.668223  0.667674  0.667224  0.666863  0.666579  0.666359  0.666191  0.666065  0.665971  0.665901  0.665849  0.665812  0.665784  0.665764  0.665749  0.665738   0.66573  0.665724   0.66572  0.665717  0.665714      0.665713 |  854.27953 | 843.829311  822.400281  799.796835  776.105316  751.416181  725.824452  699.430125  672.338547  644.660711  616.513465  588.019611  559.307841  530.512507  501.773158  473.233832  445.042035  417.347394  390.299939  364.047993  338.735666  314.499984  291.467691  269.751847  249.448346  230.632596  213.356582  197.646629  183.502146  170.895602   159.77388  150.061027  141.662222  134.468634   128.36272   123.22346  118.931092  115.371004  112.436621  110.031251  108.069009  106.474962  105.184725  104.143685   103.30601   102.63358  102.094932  101.664251  101.320471  101.046472          100.828393 |  100.73023 | 1.247525  1.239434  1.230726  1.221396  1.211441  1.200857  1.189641  1.177791  1.165304  1.152182  1.138428  1.124048  1.109052  1.093456  1.077283  1.060563  1.043334  1.025648  1.007567  0.989168  0.970542  0.951798  0.933061   0.91447  0.896179   0.87835  0.861151  0.844746  0.829285  0.814902  0.801697  0.789737  0.779053  0.769633  0.761432  0.754379  0.748379  0.743324  0.739105   0.73561  0.732734  0.730382  0.728468  0.726917  0.725665  0.724658  0.723849  0.723202  0.722685       0.722272 | 6.102608   6.04222  5.977429  5.908257  5.834729  5.756873  5.674723  5.588324  5.497736  5.403032  5.304313  5.201702   5.09536  4.985483  4.872317   4.75616  4.637373  4.516385  4.393702  4.269913  4.145693  4.021803  3.899086  3.778455  3.660876  3.547338  3.438819  3.336238  3.240408   3.15199  3.071453  2.999044   2.93478   2.87846  2.829695  2.787945   2.75257  2.722876  2.698158  2.677733  2.660963  2.647269  2.636141  2.627134  2.619869  2.614028  2.609343  2.605594    2.6026        2.600213 |  854.27953  833.456799  811.417315  788.245466  764.029518  738.862085  712.840558  686.067507  658.651018  630.704956  602.349127  573.709301  544.917079   516.10956  487.428761  459.020771  431.034579  403.620547  376.928504  351.105441  326.292811  302.623478  280.218376  259.183022  239.604039  221.545946  205.048476  190.124743  176.760534   164.91494  154.522434  145.496305  137.733195  131.118328  125.530925  120.849308   116.95529  113.737587  111.094177  108.933631  107.175593  105.750594  104.599401  103.672099  102.927027  102.329695  101.851742  101.469975  101.165515  100.923057   100.73023 |  100.0 |  754.27953 |  100.73023 |
    | 22:30 | 9.738576  9.648725   9.55116  9.445932  9.333092  9.212692  9.084791  8.949459  8.806782  8.656868  8.499855  8.335915  8.165267  7.988183  7.804996  7.616119  7.422047  7.223377  7.020817  6.815195  6.607475  6.398755  6.190277  5.983412  5.779651  5.580576  5.387817  5.202995  5.027661  4.863211  4.710817  4.571351   4.44534  4.332935  4.233924  4.147764  4.073642  4.010552  3.957369  3.912925  3.876069   3.84571  3.820848   3.80059  3.784155   3.77087  3.760165  3.751563  3.744668             3.739152 | 620.277911  610.220312  599.390663  587.817172  575.529452  562.558888  548.939021  534.705947  519.898728  504.559812  488.735454  472.476138  455.836971  438.878051  421.664779  404.268095    386.7646  369.236538  351.771575  334.462352  317.405727  300.701673  284.451768   268.75725  253.716616  239.422813  225.960094  213.400725  201.801769   191.20228  181.621233  173.056532  165.485322  158.865661  153.139439  148.236203  144.077452  140.580907  137.664355  135.248768  133.260572  131.633068  130.307127  129.231291  128.361463  127.660338  127.096686  126.644593  126.282712   125.99356 | 112.385756  111.487247  110.511598  109.459323  108.330924  107.126924  105.847912  104.494588  103.067815  101.568677   99.998545   98.359152   96.652674   94.881827   93.049963   91.161186   89.220469   87.233771   85.208166   83.151951   81.074747   78.987553   76.902766   74.834117   72.796513   70.805764   68.878167   67.029952   65.276607   63.632111   62.108169   60.713513   59.453397   58.329351   57.339243   56.477642   55.736425   55.105518   54.573685   54.129247   53.760687   53.457098   53.208482   53.005904   52.841549   52.708697   52.601651   52.515635    52.44668     52.391518 | 1.867031  1.856853  1.845768  1.833772  1.820861   1.80703  1.792275  1.776592  1.759976  1.742425  1.723939  1.704523  1.684183  1.662933  1.640793  1.617791  1.593967  1.569372  1.544072   1.51815  1.491707  1.464866  1.437773  1.410597  1.383528   1.35678  1.330581   1.30517  1.280786  1.257657  1.235989  1.215953   1.19767  1.181213  1.166595  1.153779   1.14268  1.133178  1.125127   1.11837  1.112747    1.1081  1.104286  1.101171   1.09864  1.096591  1.094938  1.093609  1.092543  1.091689 | 1.338413  1.338503  1.338603  1.338713  1.338836  1.338971  1.339119  1.339283  1.339462  1.339659  1.339875  1.340112  1.340372  1.340657   1.34097  1.341312  1.341688  1.342099  1.342549  1.343041  1.343577  1.344159   1.34479   1.34547  1.346198  1.346972  1.347788  1.348639  1.349514  1.350404  1.351292  1.352166   1.35301   1.35381  1.354555  1.355236  1.355847  1.356388  1.356858  1.357262  1.357605  1.357892  1.358132  1.358329  1.358491  1.358623   1.35873  1.358817  1.358887          1.358943 |   0.7624  0.760728  0.758885  0.756863  0.754655  0.752252  0.749644  0.746821  0.743771  0.740481  0.736938  0.733128  0.729036  0.724645  0.719942   0.71491  0.709534  0.703801  0.697701  0.691227  0.684377  0.677155  0.669576  0.661665  0.653459  0.645011  0.636391  0.627684  0.618988  0.610416  0.602081  0.594099  0.586574  0.579595  0.573225  0.567504  0.562442  0.558028  0.554228  0.550996  0.548276  0.546006  0.544129  0.542586  0.541326  0.540301  0.539472  0.538803  0.538265      0.537834 | -0.460898   -0.45829   -0.45546  -0.452399  -0.449097  -0.445543  -0.441726  -0.437633  -0.433251  -0.428566  -0.423563  -0.418227  -0.412545  -0.406502  -0.400085  -0.393284   -0.38609    -0.3785  -0.370516  -0.362146  -0.353409  -0.344334  -0.334964  -0.325358  -0.315589  -0.305748  -0.295939   -0.28628  -0.276892  -0.267897  -0.259408   -0.25152  -0.244306  -0.237811   -0.23205  -0.227013  -0.222668  -0.218963  -0.215838  -0.213228  -0.211066  -0.209287  -0.207832  -0.206649  -0.205691  -0.204919  -0.204297  -0.203799  -0.203401     -0.203083 |  0.70001  0.699319  0.698585   0.69781  0.696991  0.696128  0.695221  0.694269  0.693272  0.692229  0.691142   0.69001  0.688836  0.687621  0.686368  0.685082  0.683766  0.682427  0.681073  0.679714   0.67836  0.677023  0.675717  0.674457  0.673259  0.672135    0.6711  0.670164  0.669334  0.668614  0.668002  0.667494  0.667081  0.666752  0.666494  0.666295  0.666144   0.66603  0.665946  0.665883  0.665837  0.665803  0.665778  0.665759  0.665746  0.665736  0.665729  0.665723  0.665719      0.665716 | 874.327871 | 865.261866  846.534985  826.485696  805.192852  782.738975  759.210705  734.699246  709.300787  683.116876  656.254748  628.827567  600.954576  572.761106  544.378439  515.943457  487.598075  459.488388  431.763511   404.57407  378.070319  352.399862  327.704993  304.119687  281.766312   260.75218  241.166117  223.075281  206.522483  191.524323   178.07041  166.123874  155.623277  146.485857  138.611865  131.889621  126.200804  121.425504  117.446617  114.153342  111.443631   109.22566  107.418427   105.95167  104.765308  103.808574  103.039003   102.42137  101.926654  101.531088          101.215299 |  101.07295 | 1.255463  1.248535  1.240989  1.232821   1.22403   1.21461  1.204559  1.193872  1.182548  1.170583  1.157978  1.144733  1.130854  1.116348  1.101228  1.085513  1.069228  1.052407  1.035094  1.017344  0.999225  0.980821  0.962229  0.943564  0.924957  0.906553   0.88851  0.870992  0.854164  0.838188  0.823205  0.809336   0.79667  0.785259  0.775115  0.766214  0.758501  0.751894  0.746293  0.741591  0.737676   0.73444  0.731783  0.729613  0.727849  0.726421  0.725269  0.724342  0.723598       0.723003 | 6.162039  6.110163  6.053809  5.993001  5.927759  5.858106  5.784067  5.705671  5.622959  5.535978  5.444796  5.349497   5.25019  5.147015  5.040148  4.929804  4.816252  4.699815  4.580882  4.459913  4.337447  4.214108  4.090606  3.967734  3.846363   3.72743  3.611909  3.500787   3.39502  3.295487  3.202943  3.117974   3.04096  2.972058  2.911198  2.858103  2.812324  2.773279  2.740308  2.712714  2.689802  2.670908  2.655422  2.642795  2.632543  2.624253   2.61757  2.612199  2.607891        2.604445 | 874.327871  856.279772  836.870575  816.177403  794.280814  771.265259  747.219535  722.237221  696.417077  669.863405  642.686338  615.002042  586.932806  558.606987  530.158777  501.727758    473.4582  445.498069  417.997705  391.108131   364.97898  339.756022  315.578317  292.575039  270.862066  250.538486  231.683209  214.351951  198.574871  184.355165   171.66888  160.466115  150.673642  142.198793  134.934292  128.763583  123.566132  119.222261  115.617143  112.643788  110.204974   108.21423  106.596031  105.285427  104.227278  103.375276  102.690885  102.142271  101.703305  101.352648   101.07295 |  100.0 | 774.327871 |  101.07295 |
    | 23:00 | 9.813468  9.738463   9.65561  9.564967  9.466583  9.360505  9.246782  9.125465  8.996616   8.86031  8.716641  8.565729  8.407724  8.242818   8.07125  7.893317  7.709386  7.519901  7.325401  7.126524  6.924029  6.718798  6.511846  6.304331  6.097545  5.892908  5.691946  5.496261  5.307481  5.127202  4.956914  4.797928  4.651297  4.517758  4.397688  4.291094   4.19763  4.116645  4.047249   3.98839  3.938927  3.897702  3.863592  3.835548  3.812617  3.793954  3.778826  3.766606  3.756765             3.748859 | 628.722832  620.265221  610.988222  600.917511  590.079743   578.50289  566.216586  553.252497  539.644702  525.430088  510.648762  495.344458  479.564961  463.362497  446.794124  429.922059  412.813961  395.543112  378.188481  360.834616  343.571339  326.493166  309.698414   293.28794  277.363458  262.025414  247.370426  233.488335  220.459002  208.349035   197.20872  187.069492  177.942286   169.81705  162.663604  156.433822  151.064924  146.483515  142.609884   139.36211  136.659615   134.42593  132.590614  131.090367  129.869472  128.879726  128.080021  127.435712  126.917886  126.502613 | 113.134684  112.384627  111.556104  110.649673  109.665833  108.605054  107.467817  106.254647  104.966157  103.603097  102.166409  100.657286   99.077239   97.428178   95.712499   93.933174   92.093861   90.199015   88.254006   86.265244   84.240292   82.187976   80.118464   78.043314   75.975451   73.929076   71.919459   69.962609   68.074812   66.272021   64.569144   62.979281   61.512972   60.177579   58.976878   57.910935   56.976298   56.166452   55.472495   54.883898   54.389266   53.977019   53.635923   53.355481   53.126165   52.939536    52.78826   52.666063   52.567649     52.488592 | 1.875492  1.867018  1.857634  1.847339  1.836129  1.824002  1.810952  1.796975  1.782065  1.766219  1.749434  1.731708  1.713043  1.693443  1.672919  1.651488  1.629172  1.606004  1.582029  1.557304  1.531901  1.505911  1.479444  1.452631  1.425628  1.398611  1.371782  1.345358  1.319572  1.294663  1.270867  1.248403  1.227463  1.208198  1.190711   1.17505  1.161208  1.149128  1.138711  1.129827  1.122326   1.11605  1.110839  1.106542  1.103021  1.100149  1.097818  1.095933  1.094413  1.093191 |  1.33834  1.338414  1.338496  1.338588  1.338691  1.338806  1.338932  1.339071  1.339225  1.339394  1.339579  1.339783  1.340007  1.340252  1.340521  1.340817   1.34114  1.341495  1.341884   1.34231  1.342776  1.343284  1.343838  1.344439  1.345088  1.345786  1.346531  1.347321  1.348149  1.349009  1.349888  1.350776  1.351658  1.352519  1.353344   1.35412  1.354838   1.35549  1.356071  1.356582  1.357024  1.357403  1.357722   1.35799  1.358212  1.358394  1.358544  1.358666  1.358764          1.358844 | 0.763774  0.762398  0.760857  0.759148  0.757263  0.755195  0.752937   0.75048  0.747812  0.744924  0.741803  0.738437  0.734811  0.730912  0.726723  0.722231  0.717418  0.712271  0.706775  0.700918  0.694691  0.688088  0.681112   0.67377   0.66608  0.658073  0.649792  0.641296  0.632662  0.623979  0.615352  0.606893  0.598717  0.590936  0.583647  0.576929  0.570836  0.565395  0.560608  0.556453  0.552894  0.549877  0.547346  0.545241  0.543503  0.542078  0.540915  0.539971  0.539208      0.538592 | -0.463533  -0.461308  -0.458872  -0.456217  -0.453335  -0.450216  -0.446852  -0.443229  -0.439337  -0.435163  -0.430693  -0.425913  -0.420809  -0.415365  -0.409569  -0.403405  -0.396864  -0.389934  -0.382609  -0.374889  -0.366777  -0.358286   -0.34944  -0.340272  -0.330833  -0.321185  -0.311409  -0.301602  -0.291873  -0.282341  -0.273129  -0.264354  -0.256122  -0.248521  -0.241611  -0.235425  -0.229969  -0.225223  -0.221146  -0.217685  -0.214775  -0.212352  -0.210348  -0.208704  -0.207361   -0.20627  -0.205388  -0.204677  -0.204105     -0.203647 | 0.700842  0.700227  0.699571  0.698874  0.698134  0.697351  0.696524  0.695652  0.694736  0.693774  0.692766  0.691713  0.690615  0.689473  0.688289  0.687065  0.685804  0.684511  0.683191  0.681851    0.6805  0.679146  0.677803  0.676482  0.675198  0.673965  0.672798  0.671712  0.670718  0.669827  0.669042  0.668367  0.667798  0.667329   0.66695   0.66665  0.666417  0.666237  0.666101     0.666  0.665924  0.665867  0.665826  0.665795  0.665772  0.665756  0.665743  0.665734  0.665728      0.665723 | 888.651531 | 881.064673   865.23817  847.961189  829.305117  809.344583  788.157896  765.827492  742.440374   718.08852  692.869281  666.885714  640.246865  613.067953   585.47045  557.582018   529.53627   501.47232   473.53409   445.86932  418.628264  391.962017  366.020471  340.949876  316.890034  293.971174  272.310583  252.009166  233.148091  215.785792  199.955603    185.6643  172.891828  161.592358  151.696733  143.116152  135.746811  129.475069   124.18265  119.751444  116.067537  113.024282  110.524342  108.480802  106.817479  105.468648  104.378352  103.499484  102.792753  102.225641          101.771412 | 101.566343 | 1.261221  1.255454  1.249067  1.242058  1.234427  1.226169  1.217281   1.20776  1.197602  1.186804  1.175362  1.163276  1.150545  1.137173  1.123166  1.108533  1.093289  1.077457  1.061065   1.04415  1.026761   1.00896  0.990818  0.972427  0.953889  0.935327  0.916877  0.898689  0.880922  0.863743  0.847315  0.831791  0.817305  0.803966  0.791846  0.780983  0.771374  0.762983  0.755742  0.749564  0.744345  0.739976  0.736347  0.733355  0.730901  0.728901  0.727276  0.725962  0.724902        0.72405 | 6.205263  6.161974  6.114139  6.061786  6.004937  5.943612  5.877831  5.807617  5.732995  5.653998  5.570668  5.483061  5.391252  5.295334  5.195429  5.091691  4.984312   4.87353  4.759635  4.642976  4.523972  4.403113  4.280974  4.158211  4.035567  3.913868   3.79401  3.676945  3.563653   3.45511  3.352242   3.25588  3.166714  3.085248  3.011774  2.946358  2.888848  2.838895  2.795999  2.759546  2.728863  2.703254   2.68204   2.66458  2.650292  2.638656  2.629218  2.621592  2.615447         2.61051 | 888.651531  873.566327  856.995594  839.009202  819.680035  799.084435   777.30265  754.419275  730.523682  705.710426  680.079614  653.737222   626.79533  599.372267  571.592626  543.587121   515.49226  487.449778  459.605812  432.109768  405.112835  378.766141  353.218508  328.613828  305.088073  282.766022  261.757805  242.155441  224.029582  207.426737  192.367277  178.844493  166.824954  156.250266  147.040205  139.096994  132.310344  126.562795  121.734834  117.709397  114.375458  111.630575   109.38243  107.549485   106.06094  104.856202  103.884029   103.10153    102.4731  101.969389  101.566343 |  100.0 | 788.651531 | 101.566343 |
    | 23:30 | 9.860821  9.800732  9.732675  9.656713  9.572903  9.481292  9.381925  9.274844  9.160097  9.037736  8.907827   8.77045  8.625707  8.473731  8.314688  8.148786   7.97629  7.797523  7.612883  7.422852  7.228009   7.02904  6.826751  6.622079  6.416095  6.210008  6.005161  5.803016   5.60513  5.413113  5.228583  5.053094  4.888065  4.734707  4.593947  4.466379  4.352232  4.251372   4.16333  4.087359  4.022504   3.96768  3.921742  3.883551  3.852018  3.826136  3.805001   3.78782  3.773904              3.76267 | 634.091275  627.282738  619.614911   611.11128  601.795943  591.693914  580.831441  569.236337   556.93833  543.969429  530.364305  516.160686  501.399758  486.126575   470.39046  454.245392   437.75036  420.969674  403.973193   386.83646  369.640693  352.472595  335.423935  318.590846  302.072785  285.971109   270.38721   255.42023  241.164339  227.705686  215.119161  203.465197    192.7869  183.107859  174.430952  166.738387    159.9931  154.141386  149.116514  144.842881  141.240244  138.227606  135.726442  133.663118  131.970465  130.588608  129.465196  128.555182  127.820316  127.228477 | 113.608211  113.007322  112.326748  111.567135  110.729033  109.812924  108.819248  107.748442  106.600969  105.377363  104.078269  102.704497  101.257075   99.737312   98.146876   96.487864     94.7629    92.97523   91.128831   89.228522   87.280089   85.290397   83.267513   81.220791   79.160948   77.100079    75.05161   73.030162   71.051297   69.131134   67.285832   65.530939   63.880654    62.34707    60.93947   59.663789   58.522321   57.513718   56.633297   55.873586   55.225039   54.676797   54.217422   53.835512   53.520177   53.261357   53.050013   52.878196    52.73904       52.6267 | 1.880831  1.874054  1.866364  1.857759  1.848241  1.837807  1.826454  1.814177  1.800971  1.786831  1.771752  1.755731  1.738765  1.720853     1.702  1.682212    1.6615  1.639886  1.617395  1.594066  1.569948  1.545103  1.519613  1.493575  1.467107   1.44035  1.413468  1.386648  1.360096  1.334037  1.308706   1.28434  1.261169  1.239402  1.219213  1.200735  1.184049   1.16918  1.156101  1.144739  1.134982  1.126691  1.119713   1.11389  1.109067  1.105098   1.10185  1.099205  1.097059  1.095325 | 1.338295  1.338353  1.338419  1.338495   1.33858  1.338676  1.338782    1.3389  1.339031  1.339175  1.339334  1.339509  1.339701  1.339912  1.340144  1.340398  1.340677  1.340983  1.341318  1.341686  1.342089   1.34253  1.343012  1.343538  1.344109  1.344728  1.345396  1.346112  1.346873  1.347677  1.348517  1.349383  1.350265  1.351149  1.352021  1.352866   1.35367  1.354421  1.355111  1.355732  1.356284  1.356766  1.357181  1.357534  1.357832   1.35808  1.358286  1.358455  1.358593          1.358705 | 0.764635  0.763542  0.762291  0.760878  0.759298  0.757547  0.755616  0.753499  0.751186   0.74867  0.745939  0.742981  0.739786  0.736339  0.732626  0.728633  0.724345  0.719746  0.714822  0.709556  0.703938  0.697954  0.691599  0.684869  0.677769   0.67031  0.662516  0.654422  0.646077  0.637549  0.628917  0.620277  0.611738  0.603414  0.595416  0.587852  0.580812  0.574363   0.56855  0.563389  0.558872  0.554971  0.551642  0.548831   0.54648   0.54453  0.542923  0.541607  0.540536      0.539666 | -0.465482  -0.463626  -0.461567  -0.459299  -0.456815  -0.454108   -0.45117  -0.447989  -0.444557  -0.440861   -0.43689   -0.43263  -0.428068  -0.423189  -0.417979  -0.412424   -0.40651  -0.400224  -0.393555  -0.386495   -0.37904  -0.371191  -0.362955  -0.354349    -0.3454   -0.33615  -0.326652  -0.316978  -0.307214  -0.297461  -0.287833   -0.27845  -0.269433  -0.260896  -0.252936   -0.24563  -0.239029  -0.233153  -0.227998  -0.223535  -0.219718  -0.216489  -0.213783  -0.211536  -0.209682  -0.208163  -0.206925   -0.20592  -0.205107     -0.204453 | 0.701525  0.700986  0.700407  0.699787  0.699125  0.698421  0.697673  0.696882  0.696047  0.695166   0.69424  0.693267  0.692249  0.691186  0.690077  0.688925  0.687732  0.686499  0.685231  0.683933  0.682611  0.681271  0.679924  0.678579  0.677248  0.675946  0.674686  0.673482  0.672351  0.671304  0.670353  0.669507  0.668768  0.668137   0.66761  0.667178  0.666832  0.666559  0.666347  0.666186  0.666064  0.665972  0.665904  0.665853  0.665816  0.665788  0.665767  0.665752  0.665741      0.665733 | 897.187983 | 891.147552  878.364804  864.024233  848.189792  830.928283   812.30978  792.408058   771.30102   749.07113  725.805825  701.597906  676.545894  650.754331  624.334007  597.402092  570.082143  542.503957  514.803237  487.121036   459.60293  432.397898   405.65686  379.530854  354.168824  329.715033  306.306122  284.067881  263.111847  243.531884   225.40096  208.768384  193.657785  180.066112  167.963875  157.296756  147.988545  139.945216  133.059789  127.217517  122.300938  118.194363  114.787514   111.97817  109.673825  107.792462  106.262621  105.022957  104.021456  103.214471          102.565699 |  102.27229 | 1.264854  1.260242  1.255008  1.249152  1.242673  1.235569  1.227839  1.219478  1.210482  1.200849  1.190574  1.179655  1.168088  1.155873  1.143012  1.129508   1.11537  1.100609  1.085243  1.069296  1.052801    1.0358  1.018346  1.000506  0.982358  0.963998  0.945537  0.927102  0.908835  0.890891   0.87343  0.856618  0.840615  0.825565  0.811594  0.798794  0.787226  0.776909  0.767828  0.759933  0.753149  0.747382  0.742526  0.738472  0.735113  0.732348  0.730085  0.728242  0.726747       0.725538 | 6.232586  6.197914  6.158633  6.114776  6.066371  6.013438  5.955997  5.894067  5.827666  5.756815  5.681543  5.601886  5.517889  5.429617  5.337149  5.240593  5.140082  5.035785  4.927913  4.816724  4.702532  4.585714  4.466718  4.346066  4.224366  4.102308  3.980668  3.860299  3.742118  3.627091  3.516194  3.410384  3.310552  3.217471  3.131759  3.053835    2.9839  2.921934  2.867705  2.820803  2.780681  2.746705  2.718192  2.694456  2.674835  2.658717  2.645545   2.63483  2.626147        2.619134 | 897.187983  885.198679  871.620137   856.51498  839.948468  821.988922  802.708144   782.18185  760.490105  737.717738  713.954757  689.296724  663.845085  637.707442  610.997743  583.836352  556.349997  528.671536  500.939521  473.297522  445.893166  418.876855  392.400133  366.613672  341.664874  317.695093   294.83653  273.208876  252.915851  234.041803  216.648639  200.773334  186.426344  173.591166  162.225244  152.262274    143.6158  136.183802  129.853876  124.508486  120.029846  116.304048  113.224219  110.692652  108.621979  106.935533  105.567101  104.460269  103.567502  102.849128   102.27229 |  100.0 | 797.187983 |  102.27229 |
    | 00:00 | 9.881373  9.836196  9.782936  9.721669  9.652461  9.575365  9.490426  9.397686  9.297183   9.18896   9.07306  8.949539  8.818466  8.679929  8.534039  8.380942  8.220819  8.053901  7.880475  7.700894  7.515589  7.325079  7.129988  6.931047  6.729114  6.525179  6.320366  6.115937   5.91328  5.713895  5.519358  5.331283  5.151261  4.980793  4.821214  4.673623  4.538813  4.417233  4.308965  4.213739  4.130975  4.059843  3.999338  3.948352  3.905748   3.87041  3.841288  3.817423   3.79796             3.782153 | 636.428225  631.296671  625.273185  618.379328  610.636963  602.068524  592.697309  582.547773  571.645851  560.019286   547.69798  534.714355  521.103741  506.904758   492.15972  476.915035  461.221589  445.135119  428.716543  412.032229  395.154187  378.160137  361.133426  344.162745  327.341597   310.76746  294.540594  278.762444  263.533613  248.951402   235.10696   222.08216  209.946366  198.753362  188.538741  179.318102   171.08634  163.818235  157.470346  151.984053  147.289399  143.309288  139.963576  137.172696  134.860536  132.956506  131.396797  130.124958  129.091926  128.255698 | 113.813727  113.361958  112.829357  112.216695  111.524613  110.753648  109.904259  108.976857  107.971834  106.889596  105.730599  104.495393  103.184664  101.799289  100.340393   98.809416   97.208188    95.53901   93.804749   92.008938   90.155886   88.250795   86.299876   84.310467   82.291141   80.251786   78.203656   76.159365   74.132802    72.13895   70.193584   68.312833    66.51261   64.807926   63.212141    61.73623   60.388135   59.172331   58.089646   57.137388    56.30975   55.598433   54.993377    54.48352   54.057476   53.704098   53.412882   53.174234   52.979605     52.821529 | 1.883146  1.878055  1.872045  1.865118  1.857277  1.848521  1.838849  1.828257  1.816742  1.804297  1.790919  1.776601   1.76134  1.745131  1.727973  1.709868  1.690819  1.670836  1.649935  1.628137  1.605475  1.581989  1.557736  1.532785  1.507223  1.481157  1.454713  1.428041  1.401315  1.374727  1.348492  1.322839  1.298004   1.27422  1.251709  1.230666  1.211248  1.193569  1.177685  1.163601  1.151271  1.140606  1.131483  1.123758  1.117277  1.111882  1.107423   1.10376  1.100766  1.098331 | 1.338276  1.338319   1.33837   1.33843  1.338499  1.338578  1.338666  1.338765  1.338875  1.338998  1.339133  1.339283  1.339447  1.339628  1.339827  1.340046  1.340286  1.340549  1.340839  1.341156  1.341504  1.341885  1.342303  1.342759  1.343258  1.343801   1.34439  1.345028  1.345714  1.346447  1.347224  1.348041   1.34889  1.349761  1.350642   1.35152  1.352379  1.353206  1.353987  1.354712  1.355372  1.355964  1.356486  1.356939  1.357328  1.357658  1.357935  1.358165  1.358355          1.358511 | 0.765006  0.764188  0.763216  0.762087  0.760798  0.759345  0.757722  0.755924  0.753943  0.751772  0.749402  0.746823  0.744023  0.740993  0.737718  0.734186  0.730382  0.726292    0.7219  0.717192  0.712152  0.706766  0.701022  0.694911  0.688428  0.681571   0.67435   0.66678  0.658888  0.650717  0.642322  0.633776  0.625164  0.616588  0.608157  0.599986  0.592185  0.584854  0.578072    0.5719  0.566369  0.561485  0.557233  0.553577  0.550469  0.547854  0.545674  0.543869  0.542385      0.541172 | -0.466776  -0.465278  -0.463582  -0.461685  -0.459581  -0.457265  -0.454729  -0.451965  -0.448965  -0.445718  -0.442213   -0.43844  -0.434384  -0.430034  -0.425375  -0.420393  -0.415074  -0.409404  -0.403369  -0.396958  -0.390161   -0.38297  -0.375385  -0.367408  -0.359052  -0.350336  -0.341295  -0.331972  -0.322431  -0.312746  -0.303012  -0.293334  -0.283828  -0.274615  -0.265813   -0.25753  -0.249854  -0.242852  -0.236562  -0.230995  -0.226135  -0.221946  -0.218379  -0.215372   -0.21286  -0.210778  -0.209064  -0.207662  -0.206521     -0.205596 | 0.702062  0.701599  0.701095  0.700551  0.699967   0.69934  0.698672   0.69796  0.697205  0.696405   0.69556   0.69467  0.693733  0.692751  0.691722  0.690648  0.689529  0.688368  0.687165  0.685925  0.684651  0.683349  0.682026  0.680688  0.679347  0.678012  0.676698  0.675416  0.674183  0.673012  0.671917  0.670912  0.670005  0.669204   0.66851  0.667923  0.667435  0.667039  0.666724  0.666476  0.666285   0.66614   0.66603  0.665947  0.665886   0.66584  0.665807  0.665782  0.665763      0.665749 |      900.0 | 895.545692  885.895215  874.600785   861.71894  847.308733   831.43212  814.154359  795.544424   775.67542  754.625001  732.475779  709.315717  685.238496  660.343831  634.737734   608.53269  581.847725  554.808343  527.546294  500.199137   472.90957  445.824476  419.093664  392.868254  367.298692   342.53239  318.710989  295.967304  274.422016   254.18026  235.328269  217.930328  202.026287  187.629935  174.728482  163.283334  153.232232   144.49264  136.966151  130.543474  125.109582  120.548529  116.747591  113.600496  111.009653  108.887441  107.156694   105.75055   104.61187          103.692391 | 103.275704 | 1.266429  1.262965  1.258875  1.254161  1.248824  1.242863  1.236278  1.229067  1.221224  1.212748  1.203635  1.193879  1.183478  1.172428  1.160729  1.148379  1.135383  1.121744  1.107472  1.092583  1.077095  1.061037  1.044446  1.027367  1.009859  0.991993  0.973855  0.955547  0.937185  0.918903  0.900847  0.883174  0.866048  0.849631  0.834076  0.819521  0.806078  0.793827  0.782811  0.773036  0.764472   0.75706  0.750716  0.745341   0.74083  0.737073  0.733968  0.731416  0.729331       0.727634 | 6.244442  6.218378  6.187643   6.15228  6.112321  6.067792  6.018716   5.96511  5.906989  5.844371  5.777273  5.705718  5.629735  5.549363  5.464655  5.375681   5.28253   5.18532  5.084198   4.97935  4.871006  4.759446  4.645009    4.5281  4.409195  4.288851  4.167707  4.046488  3.925999  3.807118  3.690781  3.577958  3.469616  3.366687  3.270014  3.180309   3.09811  3.023749  2.957338   2.89877  2.847743   2.80379  2.766331  2.734713  2.708254  2.686281  2.668155  2.653288  2.641155        2.631294 |      900.0  891.184497  880.697261  868.593659  854.931385  839.770832  823.175493   805.21237  785.952381  765.470789  743.847609  721.168016  697.522729  673.008353   647.72768  621.789909  595.310781  568.412585  541.224023  513.879886  486.520511  459.290986  432.340046  405.818655  379.878208  354.668363  330.334486  307.014742  284.836883  263.914844  244.345295  226.204369  209.544799  194.393783  180.751832  168.592854  157.865619  148.496568  140.393819  133.451993  127.557435  122.593317  118.444212  114.999822   112.15771  109.825026  107.919343  106.368767  105.111519  104.095172  103.275704 |  100.0 |      800.0 | 103.275704 |
    | 00:30 |  9.87607  9.845729  9.807196  9.760563  9.705905  9.643286  9.572757  9.494362  9.408139  9.314122  9.212347  9.102852  8.985682  8.860895  8.728565  8.588787  8.441685  8.287417  8.126182  7.958234  7.783885  7.603518  7.417599  7.226688   7.03145  6.832668  6.631249  6.428235  6.224806   6.02227  5.822059  5.625701  5.434787  5.250918  5.075649  4.910414  4.756453  4.614742  4.485934  4.370329  4.267869  4.178157  4.100512  4.034035  3.977682  3.930338  3.890877  3.858217   3.83135             3.809365 | 635.824817  632.377864  628.013435  622.751367  616.611531  609.614087  601.779735  593.129994  583.687479  573.476201  562.521887  550.852303  538.497611  525.490731  511.867716  497.668145  482.935512  467.717618  452.066944  436.040998   419.70262  403.120214  386.367882  369.525434   352.67822  335.916747   319.33602  303.034571  287.113102  271.672723  256.812751    242.6281  229.206323  216.624441  204.945768  194.217022  184.466024  175.700323  167.907012  161.053834  155.091552  149.957322  145.578687  141.877742  138.775027  136.192841  134.057782  132.302461  130.866477   129.69676 | 113.760698  113.457293  113.071962  112.605627   112.05905  111.432858   110.72757  109.943621   109.08139  108.141223   107.12347  106.028515  104.856818   103.60895  102.285651  100.887873   99.416851   97.874166   96.261825   94.582345   92.838852   91.035182   89.175992   87.266882   85.314503   83.326676   81.312486   79.282352   77.248055   75.222699   73.220591   71.257014   69.347866   67.509176   65.756485   64.104138    62.56453   61.147416   59.859337   58.703294   57.678688   56.781568   56.005122   55.340354   54.776825   54.303378   53.908771   53.582173   53.313502     53.093646 | 1.882548   1.87913  1.874784  1.869517  1.863334  1.856236  1.848225  1.839298  1.829453  1.818685   1.80699  1.794363  1.780797  1.766288   1.75083  1.734423  1.717064  1.698756  1.679506  1.659324  1.638231   1.61625  1.593419  1.569784  1.545406  1.520362  1.494747  1.468674  1.442281  1.415725  1.389187   1.36287  1.336993  1.311787  1.287487  1.264322  1.242502  1.222208  1.203579  1.186704   1.17162   1.15831  1.146712  1.136721  1.128206  1.121021  1.115008  1.110016  1.105898  1.102521 | 1.338281   1.33831  1.338347  1.338392  1.338446  1.338508   1.33858  1.338662  1.338754  1.338857  1.338971  1.339098  1.339238  1.339393  1.339564  1.339751  1.339958  1.340184  1.340433  1.340707  1.341007  1.341336  1.341697  1.342092  1.342525  1.342998  1.343513  1.344074  1.344682  1.345338  1.346042  1.346791  1.347583  1.348411  1.349268  1.350141  1.351019  1.351888  1.352732  1.353539  1.354295  1.354992  1.355623  1.356184  1.356677  1.357102  1.357466  1.357773   1.35803          1.358243 | 0.764911  0.764361   0.76366  0.762805  0.761795  0.760626  0.759296  0.757798  0.756128   0.75428  0.752245  0.750016  0.747583  0.744937  0.742065  0.738957  0.735599  0.731978  0.728079  0.723888  0.719389  0.714567  0.709408  0.703899  0.698028  0.691789  0.685178  0.678197  0.670858  0.663182  0.655202  0.646965  0.638534  0.629985  0.621412  0.612919  0.604618   0.59662  0.589032  0.581946  0.575433  0.569542  0.564292  0.559683  0.555688  0.552269  0.549373  0.546944  0.544924      0.543256 | -0.467444  -0.466294   -0.46495  -0.463411  -0.461671  -0.459728  -0.457575  -0.455206  -0.452612  -0.449787   -0.44672  -0.443402   -0.43982  -0.435964   -0.43182  -0.427375  -0.422615  -0.417526  -0.412094  -0.406306  -0.400147  -0.393609  -0.386681   -0.37936  -0.371646  -0.363545  -0.355072  -0.346254  -0.337128  -0.327746  -0.318174  -0.308498  -0.298813   -0.28923  -0.279868  -0.270846  -0.262277  -0.254263  -0.246882  -0.240189  -0.234211  -0.228948  -0.224376  -0.220454  -0.217126  -0.214329     -0.212  -0.210074  -0.208492     -0.207199 | 0.702458  0.702069   0.70164  0.701172  0.700663  0.700113  0.699522  0.698889  0.698213  0.697493  0.696729   0.69592  0.695065  0.694165  0.693218  0.692225  0.691185  0.690101  0.688972  0.687801   0.68659  0.685343  0.684064   0.68276  0.681436  0.680103   0.67877  0.677448  0.676152  0.674894   0.67369  0.672553  0.671498  0.670535  0.669674  0.668919   0.66827  0.667724  0.667275  0.666913  0.666626  0.666401  0.666229  0.666098  0.665999  0.665925   0.66587  0.665829  0.665799      0.665776 | 897.264896 | 894.409558  887.926349  879.734443  869.882983  858.423344  845.409487  830.898325  814.950103  797.628793  779.002493  759.143834  738.130384  716.045037  692.976384  669.019052  644.273985  618.848661  592.857217  566.420453  539.665689  512.726446  485.741908  458.856128  432.216951  405.974601  380.279922  355.282237  331.126833  307.952101   285.88637  265.044559   245.52478  227.405094  210.740681  195.561682  181.872003  169.649301  158.846299  149.393422  141.202592  134.171869  128.190509  123.143973  118.918469  115.404706  112.500711  110.113659  108.160827  106.569807          105.278185 | 104.691406 | 1.266023  1.263697  1.260739  1.257155  1.252946  1.248115  1.242661  1.236584  1.229881  1.222548  1.214583  1.205981  1.196738   1.18685  1.176314  1.165127  1.153288  1.140798  1.127661  1.113884  1.099478   1.08446  1.068853  1.052689  1.036007  1.018859  1.001309  0.983433  0.965323  0.947087  0.928848  0.910745  0.892927  0.875555  0.858791  0.842793   0.82771  0.813668  0.800765  0.789067  0.778603  0.769362  0.761304  0.754358  0.748436  0.743436  0.739251  0.735774  0.732906       0.730553 | 6.241383  6.223878  6.201644  6.174731  6.143179  6.107022  6.066286  6.020991  5.971153  5.916787  5.857906  5.794524   5.72666  5.654337  5.577586  5.496452  5.410993  5.321285  5.227429  5.129554  5.027822  4.922437  4.813648  4.701757  4.587131  4.470201  4.351477  4.231547  4.111083   3.99084  3.871653  3.754421  3.640092  3.529635  3.424003  3.324091  3.230688  3.144438  3.065795  2.995002  2.932081  2.876847  2.828931  2.787821  2.752908   2.72353  2.699011  2.678694  2.661965        2.648265 | 897.264896  891.647486  884.297236  875.262253  864.592703  852.341154  838.562941  823.316534  806.663931  788.671052  769.408146  748.950197  727.377324  704.775162  681.235226  656.855218  631.739291  605.998219  579.749473   553.11716  526.231802   499.22992  472.253377  445.448457  418.964629  392.952972  367.564228  342.946476  319.242434  296.586425  275.101089  254.893953  236.054048   218.64879  202.721393  188.289118  175.342597   163.84646  153.741335  144.947132  137.367384  130.894233  125.413605  120.810097  116.971197  113.790597  111.170503  109.022985  107.270507  105.845821  104.691406 |  100.0 | 797.264896 | 104.691406 |
    | 01:00 | 9.846037  9.830391  9.806448  9.774316  9.734086  9.685832  9.629616  9.565486  9.493484  9.413644  9.325995  9.230568  9.127395  9.016511  8.897965  8.771816  8.638145  8.497057  8.348686  8.193206  8.030837  7.861854  7.686596  7.505481  7.319008  7.127781  6.932509  6.734024  6.533286  6.331389  6.129562  5.929162  5.731655  5.538593  5.351567  5.172159  5.001874  4.842065  4.693865  4.558118  4.435335  4.325667   4.22892  4.144582  4.071883  4.009869  3.957469  3.913569  3.877067             3.846916 | 632.412743  630.638826  627.928835  624.301018  619.773443  614.364221  608.091743  600.974926  593.033463   584.28809  574.760871  564.475492  553.457576  541.735021   529.33834  516.301032  502.659958  488.455719   473.73305   458.54119  442.934248  426.971521  410.717768  394.243398   377.62455  360.943032  344.286057  327.745759  311.418402  295.403258  279.801092  264.712226  250.234182   236.45893  223.469848  211.338552  200.121826  189.858964  180.569837  172.253991   164.89098  158.441989  152.852616  148.056506  143.979421  140.543286  137.669849  135.283657  133.314246   131.69754 | 113.460367  113.303912   113.06448  112.743161   112.34086  111.858323  111.296158  110.654862  109.934841  109.136437  108.259953  107.305684  106.273946   105.16511  103.979646  102.718161  101.381454   99.970568   98.486857   96.932058   95.308368   93.618537   91.865964   90.054805   88.190085   86.277811   84.325092    82.34024   80.332858   78.313889   76.295621   74.291618   72.316552   70.385926    68.51567   66.721594    65.01874   63.420653   61.938653   60.581184   59.353345   58.256671   57.289199   56.445816   55.718834   55.098691   54.574692   54.135692    53.77067     53.469157 | 1.879165  1.877401    1.8747  1.871071  1.866523   1.86106  1.854685  1.847398  1.839198  1.830082  1.820047  1.809087  1.797197  1.784372  1.770605  1.755891  1.740226  1.723609   1.70604  1.687522  1.668065  1.647682  1.626395  1.604233  1.581238  1.557461  1.532969  1.507848  1.482198  1.456143  1.429828   1.40342  1.377107    1.3511   1.32562  1.300901  1.277176  1.254664  1.233565  1.214041  1.196211  1.180144  1.165853  1.153304  1.142416  1.133075  1.125142  1.118468  1.112899  1.108285 | 1.338309  1.338324  1.338347  1.338378  1.338418  1.338466  1.338522  1.338588  1.338663  1.338748  1.338844   1.33895  1.339069  1.339201  1.339346  1.339507  1.339684  1.339879  1.340093  1.340328  1.340587  1.340871  1.341182  1.341523  1.341898  1.342308  1.342756  1.343245  1.343779  1.344358  1.344984  1.345658  1.346379  1.347144   1.34795  1.348788  1.349649  1.350523  1.351395  1.352252   1.35308  1.353864  1.354594  1.355262  1.355862  1.356394  1.356857  1.357256  1.357595          1.357881 | 0.764367  0.764083  0.763646  0.763058  0.762317  0.761422   0.76037  0.759158  0.757781  0.756236  0.754515  0.752612  0.750519  0.748228  0.745729  0.743011  0.740064  0.736874  0.733429  0.729715  0.725717   0.72142   0.71681  0.711871   0.70659  0.700956  0.694957  0.688589   0.68185  0.674748  0.667296  0.659521  0.651462  0.643172  0.634719  0.626186  0.617672  0.609282  0.601129  0.593323  0.585964  0.579135  0.572898  0.567289  0.562321   0.55798  0.554236  0.551043   0.54835      0.546097 | -0.467511  -0.466703  -0.465703   -0.46451  -0.463122  -0.461537   -0.45975  -0.457755  -0.455548   -0.45312  -0.450465  -0.447573  -0.444435  -0.441041  -0.437377  -0.433434  -0.429197  -0.424653  -0.419788  -0.414588   -0.40904  -0.403129  -0.396845  -0.390177  -0.383119  -0.375667  -0.367825  -0.359602  -0.351019  -0.342106  -0.332906  -0.323477  -0.313892  -0.304241  -0.294626  -0.285161  -0.275963  -0.267151  -0.258833    -0.2511  -0.244022  -0.237642  -0.231975  -0.227012  -0.222721  -0.219054  -0.215955  -0.213358  -0.211201     -0.209421 | 0.702716    0.7024  0.702045  0.701651  0.701217  0.700743  0.700228  0.699672  0.699073  0.698432  0.697748  0.697019  0.696246  0.695427  0.694562  0.693651  0.692693  0.691689  0.690639  0.689544  0.688406  0.687226  0.686007  0.684754  0.683471  0.682166  0.680844  0.679517  0.678194  0.676888  0.675612  0.674381  0.673208  0.672108  0.671094  0.670176   0.66936   0.66865  0.668046  0.667541  0.667129  0.666798  0.666537  0.666334  0.666179  0.666061  0.665972  0.665906  0.665856      0.665819 | 889.260928 | 887.991891  884.659952  879.575838  872.781306  864.320124  854.238381   842.58482  829.411179  814.772556  798.727778  781.339791   762.67605  742.808912   721.81603   699.78072  676.792311   652.94645  628.345352  603.097965   577.32004   551.13407  524.669066  498.060143  471.447873  444.977376  418.797102  393.057275  367.907981  343.496877  319.966538  297.451469   276.07487  255.945266   237.15316  219.767955  203.835375  189.375691  176.382976  164.825625  154.648185  145.774459   138.11164  131.555118  125.993514  121.313497  117.403988  114.159523   111.48265   109.28539          107.489875 | 106.671706 |  1.26372   1.26252  1.260681  1.258212  1.255117  1.251399  1.247059  1.242099  1.236516  1.230309  1.223475  1.216011  1.207912  1.199174  1.189792  1.179763  1.169084  1.157752  1.145768  1.133133  1.119852  1.105934  1.091392  1.076247  1.060523  1.044257  1.027493  1.010287  0.992707  0.974837  0.956774  0.938632  0.920541  0.902642  0.885091  0.868047  0.851672  0.836119  0.821527  0.808012  0.795659  0.784517    0.7746  0.765884  0.758318  0.751822  0.746304  0.741659  0.737782       0.734569 | 6.224056  6.215028  6.201212  6.182669  6.159448   6.13159  6.099128  6.062086  6.020483  5.974336  5.923654   5.86845  5.808734   5.74452  5.675827  5.602678   5.52511   5.44317  5.356924  5.266456  5.171878  5.073333  4.970997  4.865092  4.755888  4.643714   4.52896  4.412089   4.29364  4.174235  4.054578  3.935454  3.817722    3.7023  3.590143  3.482212  3.379435  3.282664   3.19263  3.109897  3.034835  2.967596  2.908117   2.85614  2.811236  2.772856  2.740371  2.713114  2.690423        2.671659 | 889.260928  886.814967  882.596333  876.645852  869.006203  859.722228   848.84125  836.413413  822.492033  807.133966  790.399988  772.355188  753.069357  732.617391  711.079665  688.542402  665.098005  640.845331  615.889917  590.344095   564.32701  537.964485  511.388712  484.737735  458.154683  431.786714  405.783647  380.296225  355.474026  331.462981  308.402541  286.422546   265.63987  246.155016  228.048823  211.379558  196.180648  182.459351   170.19659  159.348123  149.847052  141.607533  134.529376  128.503107  123.415021  119.151786  115.604288   112.67051  110.257443  108.282085  106.671706 |  100.0 | 789.260928 | 106.671706 |
    | 01:30 | 9.792548  9.791394  9.781841  9.764017  9.738027  9.703961  9.661889  9.611871  9.553953  9.488173  9.414561  9.333143  9.243942  9.146984  9.042299  8.929922  8.809903  8.682308  8.547223  8.404762  8.255075  8.098353  7.934835  7.764821   7.58868  7.406862  7.219907  7.028459  6.833276  6.635238  6.435357   6.23478  6.034784  5.836767  5.642225  5.452725  5.269849  5.095147  4.930059  4.775846  4.633524  4.503798  4.387032  4.283236  4.192085  4.112963  4.045027  3.987276  3.938629              3.89798 |  626.35823  626.227895  625.149687  623.140349  620.216264  616.393665  611.688851  606.118405   599.69942  592.449737  584.388191  575.534882  565.911453  555.541386  544.450321  532.666385  520.220548  507.146978  493.483422   479.27158  464.557487  449.391875  433.830512  417.934504  401.770523  385.410954  368.933911  352.423092  335.967429  319.660476  303.599489  287.884146  272.614873  257.890731  243.806918  230.451895    217.9043  206.229816  195.478266  185.681237  176.850568  168.977934  162.035714  155.979082  150.749147  146.276765  142.486601  139.301012  136.643414  134.440916 | 112.925483   112.91394  112.818411  112.640166  112.380271  112.039606  111.618892  111.118712  110.539533  109.881731  109.145608  108.331425   107.43942  106.469844  105.422988  104.299222  103.099035   101.82308  100.472227   99.047623   97.550755   95.983527   94.348345   92.648206     90.8868    89.06862   87.199072   85.284593   83.332758   81.352379   79.353574   77.347805   75.347843   73.367667   71.422254   69.527246   67.698493    65.95147   64.300586   62.758464   61.335237   60.037977   58.870317   57.832358   56.920848   56.129629   55.450266   54.872764   54.386288     53.979795 |  1.87313     1.873  1.871921  1.869907  1.866969  1.863114  1.858346   1.85267  1.846086  1.838592  1.830187  1.820866  1.810625  1.799458  1.787359  1.774322   1.76034   1.74541  1.729527  1.712692  1.694904  1.676171  1.656502  1.635915  1.614435  1.592095   1.56894  1.545031  1.520439  1.495256  1.469593  1.443581  1.417374  1.391147  1.365096  1.339437  1.314396  1.290204   1.26709  1.245264   1.22491  1.206173  1.189151   1.17389  1.160383  1.148577  1.138377  1.129659  1.122281  1.116092 | 1.338361  1.338362  1.338371  1.338388  1.338414  1.338448   1.33849   1.33854    1.3386  1.338669  1.338747  1.338836  1.338935  1.339046   1.33917  1.339307  1.339458  1.339625  1.339809  1.340011  1.340233  1.340478  1.340746  1.341041  1.341364  1.341718  1.342107  1.342532  1.342996  1.343503  1.344054  1.344651  1.345296  1.345988  1.346726  1.347506  1.348323  1.349169  1.350034  1.350905  1.351769  1.352612  1.353419  1.354179  1.354882   1.35552   1.35609  1.356592  1.357027            1.3574 | 0.763392  0.763371  0.763196  0.762869   0.76239  0.761759  0.760975  0.760036  0.758938  0.757679  0.756254  0.754656   0.75288  0.750919  0.748765  0.746408  0.743838  0.741045  0.738018  0.734742  0.731206  0.727394  0.723293  0.718887  0.714162  0.709104  0.703699  0.697936  0.691808  0.685311  0.678447  0.671226  0.663668  0.655802  0.647675  0.639345  0.630886  0.622388  0.613951  0.605685  0.597701  0.590103  0.582986  0.576424  0.570468  0.565144  0.560452  0.556374  0.552872      0.549897 | -0.467001   -0.46653  -0.465867  -0.465013  -0.463967  -0.462727  -0.461291  -0.459655  -0.457815  -0.455764  -0.453498  -0.451008  -0.448286  -0.445323  -0.442109  -0.438634  -0.434884  -0.430849  -0.426515  -0.421867  -0.416893  -0.411579   -0.40591  -0.399875  -0.393462  -0.386663  -0.379473  -0.371891  -0.363924  -0.355584  -0.346897  -0.337898  -0.328635  -0.319173  -0.309592  -0.299986  -0.290462  -0.281135  -0.272123   -0.26354  -0.255487  -0.248047  -0.241278  -0.235212  -0.229854  -0.225184  -0.221165  -0.217744  -0.214862     -0.212454 |  0.70284  0.702597  0.702316  0.701995  0.701635  0.701234  0.700794  0.700313  0.699791  0.699227   0.69862   0.69797  0.697277  0.696539  0.695755  0.694926  0.694051  0.693129   0.69216  0.691145  0.690084  0.688979  0.687831  0.686642  0.685417  0.684159  0.682874  0.681568  0.680251  0.678931  0.677621  0.676333   0.67508  0.673877  0.672739  0.671678  0.670706  0.669833  0.669064  0.668399  0.667837  0.667372  0.666994  0.666693  0.666457  0.666274  0.666134  0.666028  0.665948      0.665888 | 876.351729 | 876.632637  876.388511  874.369597  870.610273  865.146762  858.017406  849.262953  838.926863  827.055634  813.699137  798.910976  782.748853  765.274949  746.556309  726.665221  705.679597  683.683319  660.766568  637.026091  612.565417  587.494972  561.932086  536.000868  509.831889  483.561679  457.331963  431.288628  405.580367  380.356977  355.767299  331.956774  309.064667  287.220975  266.543138  247.132674  229.071944  212.421261   197.21663  183.468374  171.160892  160.253698  150.683757     142.369  135.212729  129.108514  123.945122  119.611079  115.998511  113.006098          110.541067 | 109.413639 | 1.259614  1.259525  1.258791   1.25742   1.25542  1.252796  1.249551  1.245688  1.241205  1.236104  1.230381  1.224033  1.217059  1.209452  1.201209  1.192325  1.182796  1.172618  1.161789  1.150306   1.13817  1.125385  1.111957  1.097896  1.083219  1.067948  1.052112   1.03575  1.018912  1.001658  0.984063  0.966216   0.94822  0.930196  0.912277  0.894611  0.877354  0.860666  0.844705   0.82962  0.815538  0.802563  0.790765  0.780178  0.770801    0.7626  0.755509  0.749447  0.744313       0.740005 | 6.193191  6.192525  6.187011  6.176724  6.161723  6.142056  6.117765  6.088879  6.055423  6.017414  5.974866  5.927788  5.876188  5.820075  5.759458  5.694349  5.624769  5.550744  5.472313   5.38953  5.302467  5.211218  5.115908   5.01669  4.913761  4.807359  4.697779  4.585373  4.470559  4.353831  4.235759  4.116997  3.998279   3.88042    3.7643  3.650849  3.541025  3.435772  3.335987  3.242471  3.155886  3.076716  3.005243  2.941529  2.885431  2.836621  2.794623  2.758856  2.728678        2.703426 | 876.351729  877.003313  875.863265  872.965108   868.34407  862.037358  854.084437  844.527324  833.410909  820.783278  806.696069  791.204828  774.369388  756.254253  736.928977  716.468554  694.953778  672.471588  649.115369  624.985202  600.188028   574.83773  549.055068   522.96748  496.708675  470.418014  444.239624  418.321211  392.812544  367.863578  343.622202  320.231624  297.827415  276.534289   256.46273  237.705627   220.33513  204.399995   189.92368  176.903478  165.310872  155.093245  146.176866  138.470954  131.872466  126.271146  121.554388  117.611511  114.337198  111.633956  109.413639 |  100.0 | 776.351729 | 109.413639 |
    | 02:00 | 9.717003  9.730077  9.734656  9.730887  9.718892  9.698775  9.670619  9.634495  9.590457  9.538548  9.478803  9.411248  9.335904  9.252789  9.161923  9.063326  8.957027  8.843064  8.721488  8.592371  8.455807  8.311924  8.160884  8.002897  7.838224  7.667194  7.490205  7.307742  7.120385  6.928824  6.733862  6.536432  6.337597  6.138553  5.940622   5.74524  5.553927  5.368251  5.189779  5.020012  4.860315  4.711846   4.57549  4.451812   4.34103  4.243015  4.157323  4.083245  4.019877             3.966188 | 617.855812  619.323171  619.837509  619.414091  618.067687  615.812765   612.66369  608.634916  603.741188  597.997759  591.420601  584.026646  575.834028  566.862355   557.13298   546.66931  535.497117  523.644871  511.144096  498.029726   484.34048  470.119234  455.413386  440.275213  424.762181  408.937219  392.868908  376.631571  360.305224  343.975343  327.732411   311.67118  295.889609  280.487426  265.564285  251.217499  237.539395  224.614363  212.515749  201.302814  191.018048  181.685135  173.307886  165.870346  159.338161  153.661106    148.7765  144.613126  141.095209  138.146078 | 112.170033  112.300772  112.346563  112.308868   112.18892  111.987746  111.706194  111.344952  110.904571  110.385484  109.788032  109.112478  108.359035   107.52789  106.619227  105.633263  104.570275  103.430641  102.214881  100.923706   99.558072   98.119237   96.608839   95.028965   93.382245   91.671939   89.902048   88.077417   86.203854   84.288237   82.338619   80.364318   78.375967   76.385526   74.406221   72.452398   70.539265   68.682511   66.897795   65.200125   63.603152   62.118457   60.754899   59.518123   58.410301    57.43015   56.573227    55.83245    55.19877     54.661884 |  1.86459   1.86607  1.866588  1.866161  1.864804  1.862526  1.859336  1.855239  1.850237  1.844332  1.837523  1.829808  1.821183  1.811642  1.801181  1.789793  1.777471  1.764209  1.750002  1.734844  1.718734  1.701671  1.683659  1.664704  1.644821  1.624029  1.602356  1.579842  1.556537  1.532505  1.507827  1.482602  1.456948  1.431006  1.404938  1.378926  1.353176  1.327905  1.303342  1.279716  1.257248  1.236137   1.21655  1.198614  1.182403  1.167942  1.155206  1.144122  1.134585  1.126465 | 1.338435  1.338422  1.338417  1.338421  1.338433  1.338453  1.338481  1.338517  1.338562  1.338616  1.338678  1.338751  1.338833  1.338925  1.339029  1.339145  1.339273  1.339416  1.339573  1.339746  1.339938  1.340148  1.340379  1.340633  1.340912  1.341218  1.341553  1.341921  1.342324  1.342765  1.343246   1.34377  1.344339  1.344955  1.345618  1.346327  1.347081  1.347875  1.348702  1.349554   1.35042  1.351286  1.352139  1.352966  1.353751  1.354485  1.355159  1.355767  1.356307          1.356779 | 0.762001  0.762243  0.762327  0.762258  0.762036  0.761662  0.761138  0.760461  0.759631  0.758645  0.757499  0.756189  0.754711  0.753058  0.751224  0.749201   0.74698  0.744554   0.74191  0.739038  0.735926  0.732561  0.728929  0.725016  0.720808  0.716289  0.711446  0.706265  0.700733  0.694841  0.688583  0.681958  0.674971  0.667636  0.659977   0.65203  0.643846  0.635491  0.627044  0.618599  0.610261  0.602139  0.594341  0.586968  0.580106  0.573819  0.568147  0.563105  0.558687      0.554864 | -0.465934  -0.465799  -0.465469  -0.464947  -0.464235  -0.463331  -0.462234  -0.460943  -0.459454  -0.457763  -0.455865  -0.453754  -0.451424  -0.448867  -0.446074  -0.443035  -0.439741   -0.43618   -0.43234  -0.428209  -0.423772  -0.419018  -0.413931  -0.408497  -0.402706  -0.396543  -0.389999  -0.383068  -0.375746  -0.368036  -0.359946  -0.351494   -0.34271  -0.333634  -0.324322  -0.314843  -0.305283  -0.295742   -0.28633  -0.277163  -0.268356   -0.26002  -0.252246  -0.245108  -0.238653    -0.2329  -0.227846  -0.223462  -0.219705     -0.216519 | 0.702835  0.702666  0.702456  0.702208   0.70192  0.701592  0.701225  0.700818   0.70037  0.699881  0.699351  0.698778  0.698163  0.697504    0.6968  0.696052  0.695258  0.694418  0.693532  0.692599  0.691619  0.690593  0.689521  0.688406  0.687249  0.686052  0.684821  0.683558  0.682272  0.680968  0.679656  0.678347  0.677052  0.675784  0.674557  0.673385  0.672283  0.671263  0.670336  0.669508  0.668785  0.668165  0.667645  0.667217  0.666872  0.666598  0.666384  0.666219  0.666093      0.665998 | 858.969595 | 860.742648  863.479776  864.439706  863.649445  861.137735  856.935289  851.075036   843.59239  834.525527  823.915692  811.807514  798.249345  783.293608  766.997169  749.421704  730.634078  710.706714  689.717958  667.752403  644.901195  621.262264  596.940497  572.047804  546.703062   521.03191  495.166352  469.244144   443.40792  417.804026  392.581025  367.887851  343.871594  320.674931  298.433216  277.271317  257.300298  238.614104  221.286461  205.368235  190.885527  177.838742  166.202858  155.928985   146.94717   139.17027  132.498548  126.824575  122.037996  118.029777          114.695666 |  113.16387 | 1.253801  1.254808  1.255161  1.254871  1.253947  1.252397  1.250225  1.247436  1.244031  1.240012  1.235376  1.230123  1.224249  1.217751  1.210626  1.202867  1.194471  1.185433  1.175749  1.165415  1.154428  1.142788  1.130496  1.117557  1.103979  1.089775  1.074963  1.059569  1.043625  1.027175  1.010273  0.992984  0.975389  0.957583  0.939675  0.921792  0.904072  0.886666   0.86973  0.853426  0.837904  0.823307   0.80975  0.797324  0.786084   0.77605  0.767205  0.759504  0.752873       0.747224 | 6.149586  6.157133  6.159777  6.157601  6.150677  6.139062  6.122806  6.101946   6.07651  6.046523  6.011999   5.97295  5.929385  5.881307  5.828722  5.771636  5.710057  5.643998   5.57348  5.498533    5.4192  5.335542  5.247637  5.155593  5.059543  4.959659  4.856152  4.749284  4.639372  4.526792  4.411993  4.295498  4.177911  4.059914  3.942275   3.82583   3.71148  3.600162  3.492827  3.390398  3.293731  3.203569  3.120499  3.044921  2.977026  2.916792     2.864   2.81826  2.779055         2.74578 | 858.969595  862.602048  864.444126  864.522021  862.863549  859.498374  854.458248  847.777269  839.492152  829.642523  818.271228  805.424661  791.153108   775.51111   758.55783   740.35743  720.979447   700.49916  678.997944  656.563582  633.290544  609.280192  584.640906  559.488099  533.944095  508.137839    482.2044  456.284243  430.522216  405.066234  380.065612  355.669044  332.022203  309.264992  287.528484  266.931644  247.577959  229.552159  212.917269  197.712251  183.950505  171.619495  160.681639  151.076529  142.724352  135.530239  129.389155  124.190858  119.824507  116.182576   113.16387 |  100.0 | 758.969595 |  113.16387 |
    | 02:30 |   9.6209  9.647882   9.66628  9.676258  9.677958  9.671498  9.656975   9.63447  9.604048   9.56576  9.519647  9.465737  9.404054  9.334613  9.257429  9.172512  9.079879  8.979547  8.871545  8.755912  8.632704     8.502  8.363905  8.218557  8.066137  7.906873  7.741053   7.56903  7.391238  7.208198  7.020533  6.828975   6.63438   6.43773  6.240138  6.042849  5.847227  5.654736  5.466914  5.285323  5.111498  4.946881  4.792747  4.650141  4.519813  4.402183  4.297323  4.204975  4.124585             4.055367 | 607.122099  610.126405  612.179035  613.293761  613.483763  612.761807  611.140426  608.632095  605.249411  601.005276  595.913095  589.986977  583.241955  575.694215  567.361347  558.262604  548.419192  537.854566  526.594747  514.668665    502.1085  488.950049   475.23309  461.001752  446.304868  431.196307  415.735271  399.986525  384.020546  367.913553  351.747379  335.609151  319.590718  303.787785  288.298702  273.222862  258.658696  244.701256  231.439449  218.953031  207.309531   196.56135  186.743344  177.871181  169.940753  162.928804  156.794774  151.483692  146.929806   143.06052 | 111.209001  111.478824  111.662799  111.762584  111.779583  111.714977  111.569747  111.344699   111.04048  110.657605   110.19647  109.657373  109.040539  108.346132  107.574286  106.725124  105.798789  104.795474  103.715452  102.559121  101.327045  100.020003   98.639048    97.18557   95.661368   94.068731   92.410527     90.6903   88.912378   87.081979   85.205326   83.289753   81.343803     79.3773   77.401383   75.428491   73.472266   71.547363   69.669139   67.853229    66.11498   64.468806   62.927473   61.501411   60.198132   59.021827   57.973231   57.049749   56.245854     55.553671 | 1.853695  1.856758  1.858844  1.859975  1.860168  1.859436  1.857789  1.855236  1.851781  1.847429   1.84218  1.836033  1.828986  1.821035  1.812176  1.802402  1.791707  1.780085  1.767528  1.754031  1.739587  1.724193  1.707846  1.690549  1.672306  1.653126  1.633026  1.612028  1.590167  1.567484  1.544036  1.519895  1.495147  1.469899   1.44428  1.418436  1.392539   1.36678  1.341368  1.316525   1.29248  1.269457  1.247667  1.227297  1.208496  1.191369  1.175969  1.162299  1.150316  1.139933 | 1.338531  1.338504  1.338485  1.338475  1.338474   1.33848  1.338495  1.338517  1.338548  1.338588  1.338635  1.338692  1.338758  1.338834   1.33892  1.339017  1.339125  1.339246   1.33938  1.339528  1.339692  1.339872  1.340071  1.340289   1.34053  1.340793  1.341083  1.341401   1.34175  1.342132   1.34255  1.343007  1.343505  1.344047  1.344635  1.345269   1.34595  1.346676  1.347445  1.348251  1.349087  1.349943  1.350806  1.351665  1.352505  1.353312  1.354074   1.35478  1.355424          1.356002 | 0.760206  0.760713  0.761057  0.761243  0.761275  0.761154  0.760883  0.760461  0.759888  0.759163  0.758283  0.757246  0.756049  0.754685  0.753151  0.751439  0.749543  0.747454  0.745165  0.742664  0.739942  0.736987  0.733787  0.730328  0.726597  0.722579   0.71826  0.713625   0.70866  0.703353  0.697693   0.69167  0.685283  0.678531  0.671424   0.66398  0.656228   0.64821  0.639983  0.631618  0.623201   0.61483   0.60661  0.598651  0.591059  0.583925  0.577328  0.571322  0.565935      0.561174 | -0.464331  -0.464531  -0.464532  -0.464339  -0.463953  -0.463377   -0.46261  -0.461652  -0.460501  -0.459154  -0.457608  -0.455858  -0.453898  -0.451723  -0.449325  -0.446696  -0.443827  -0.440709   -0.43733  -0.433679  -0.429744  -0.425512   -0.42097  -0.416103  -0.410899  -0.405344  -0.399425  -0.393133  -0.386457  -0.379393  -0.371939  -0.364102  -0.355894  -0.347336  -0.338464  -0.329323  -0.319974  -0.310495  -0.300976  -0.291521  -0.282242  -0.273255  -0.264673  -0.256599  -0.249116  -0.242287  -0.236147  -0.230706  -0.225949     -0.221843 | 0.702707  0.702609  0.702472  0.702294  0.702078  0.701822  0.701526  0.701191  0.700816  0.700401  0.699945  0.699447  0.698908  0.698326  0.697701  0.697032  0.696319  0.695559  0.694755  0.693903  0.693005  0.692061  0.691069  0.690032   0.68895  0.687825  0.686659  0.685455  0.684218  0.682953  0.681667  0.680367  0.679063  0.677766  0.676488  0.675243  0.674044  0.672906  0.671842  0.670865  0.669982  0.669201  0.668523  0.667947  0.667467  0.667075   0.66676  0.666512  0.666319       0.66617 |   837.5984 | 840.786902   846.36061  850.174014   852.24676  852.600178  851.257475  848.243937  843.587158  837.317279  829.467248  820.073097  809.174248  796.813824  783.038991  767.901302  751.457063  733.767698  714.900119  694.927086  673.927548  651.986955  629.197527  605.658459  581.476039  556.763664  531.641717  506.237282  480.683656  455.119626  429.688478  404.536696  379.812328  355.662992   332.23352  309.663256  288.083049  267.612027  248.354288  230.395668  213.800831  198.610916  184.842027  172.484778  161.505081  151.846207  143.432015  136.171114  129.961575  124.695761          120.264864 | 118.218296 | 1.246386   1.24847   1.24989   1.25066  1.250791  1.250293  1.249172  1.247434  1.245083   1.24212  1.238546  1.234361  1.229563  1.224148  1.218115  1.211457  1.204172  1.196253  1.187696  1.178496  1.168648  1.158151  1.147001  1.135198  1.122747  1.109651  1.095922  1.081575   1.06663  1.051115   1.03507  1.018539  1.001583  0.984273  0.966695   0.94895  0.931153  0.913436  0.895941  0.878822  0.862236   0.84634  0.831282   0.81719  0.804172  0.792302   0.78162  0.772132  0.763808       0.756592 | 6.094094  6.109677    6.1203  6.126062  6.127044  6.123313  6.114927  6.101931  6.084361  6.062244  6.035602  6.004448  5.968791  5.928638  5.883991  5.834852  5.781222  5.723106  5.660511  5.593452  5.521951  5.446042  5.365774  5.281213  5.192449  5.099598  5.002809  4.902269  4.798208   4.69091  4.580715  4.468027  4.353325  4.237162  4.120173  4.003072  3.886654  3.771777  3.659356  3.550331  3.445637  3.346169  3.252734   3.16601  3.086504  3.014528  2.950186  2.893372  2.843798        2.801021 |   837.5984  844.057384  848.746556  851.684769  852.892462  852.391851  850.207126  846.364667  840.893272  833.824412  825.192498  815.035173  803.393617  790.312878  775.842215  760.035453  742.951351  724.653971  705.213044  684.704325  663.209924  640.818598  617.625996  593.734826   569.25493  544.303236  519.003568  493.486267  467.887597  442.348901  417.015461  392.035048  367.556106  343.725584  320.686397  298.574558  277.516032  257.623422  238.992643  221.699772  205.798339  191.317313   178.26005  166.604425   156.30426   147.29202  139.482612  132.777941  127.071818  122.254762  118.218296 |  100.0 |   737.5984 | 118.218296 |
    | 03:00 | 9.505814   9.54633   9.57818  9.601548  9.616593  9.623447  9.622222  9.613012  9.595892  9.570921  9.538147  9.497606  9.449323  9.393317  9.329598  9.258174  9.179054  9.092242  8.997752    8.8956  8.785815   8.66844  8.543536   8.41119  8.271519  8.124676  7.970858  7.810316  7.643364  7.470384  7.291846  7.108309  6.920439  6.729017  6.534946  6.339259  6.143119  5.947816   5.75475  5.565411  5.381341  5.204087  5.035139  4.875867  4.727448  4.590798  4.466531  4.354918  4.255893             4.169077 | 594.389717  598.857028  602.380334  604.971855  606.643163  607.405341  607.269152  606.245189  604.344042  601.576452  597.953489  593.486723  588.188416  582.071725  575.150912  567.441581  558.960925  549.727991  539.763967  529.092485  517.739938  505.735822  493.113084   479.90848  466.162938  451.921917  437.235748  422.159949  406.755487  391.088974  375.232763  359.264912  343.268978  327.333589  311.551758  296.019874  280.836338  266.099799   251.90699  238.350171  225.514279  213.473889  202.290218  192.008426  182.655516  174.239124  166.747431  160.150299  154.401547  149.442154 | 110.058141    110.4633  110.781798  111.015481  111.165927  111.234468  111.222224  111.130121  110.958915  110.709209  110.381474  109.976063  109.493234  108.933165  108.295975  107.581744  106.790536  105.922424  104.977518  103.955999  102.858149  101.684396  100.435361   99.111904   97.715192   96.246758   94.708579   93.103163   91.433636   89.703843   87.918458   86.083089   84.204392   82.290168   80.349456   78.392586    76.43119   74.478158     72.5475   70.654111   68.813411   67.040866   65.351391   63.758674   62.274476   60.907984   59.665305   58.549176   57.558934     56.690767 | 1.840603  1.845218  1.848841  1.851497  1.853206  1.853985  1.853846    1.8528  1.850855  1.848016  1.844287  1.839668  1.834159  1.827757   1.82046  1.812261  1.803156  1.793137  1.782197  1.770329  1.757527  1.743783  1.729093  1.713453  1.696863  1.679325  1.660846  1.641438   1.62112  1.599919  1.577872  1.555028  1.531448  1.507211  1.482411  1.457164  1.431604   1.40589  1.380199   1.35473  1.329696  1.305321  1.281831  1.259446  1.238365  1.218759  1.200757  1.184443  1.169849  1.156958 |  1.33865  1.338608  1.338575  1.338551  1.338535  1.338529   1.33853  1.338539  1.338557  1.338582  1.338616  1.338659   1.33871   1.33877   1.33884  1.338919  1.339009   1.33911  1.339224  1.339349  1.339489  1.339643  1.339814  1.340002  1.340208  1.340436  1.340686   1.34096  1.341262  1.341592  1.341954  1.342351  1.342785  1.343258  1.343774  1.344334   1.34494  1.345593  1.346292  1.347034  1.347817  1.348634  1.349476  1.350333  1.351192  1.352041  1.352864   1.35365  1.354387          1.355065 | 0.758018  0.758793  0.759398  0.759841  0.760125  0.760254  0.760231  0.760057  0.759734  0.759261  0.758637   0.75786  0.756929  0.755839  0.754586  0.753166  0.751572  0.749798  0.747836  0.745678  0.743316  0.740738  0.737934  0.734892  0.731599  0.728042  0.724208  0.720081  0.715647  0.710893  0.705804  0.700369  0.694578  0.688424  0.681907  0.675031  0.667808  0.660262  0.652426  0.644349  0.636094  0.627737  0.619368   0.61109   0.60301  0.595233  0.587861   0.58098  0.574657      0.568935 | -0.462209  -0.462745  -0.463077   -0.46321  -0.463148  -0.462894  -0.462449  -0.461815  -0.460991  -0.459975  -0.458765  -0.457359  -0.455751  -0.453938  -0.451912  -0.449668  -0.447198  -0.444493  -0.441545  -0.438342  -0.434874  -0.431129  -0.427094  -0.422758  -0.418105  -0.413123  -0.407798  -0.402118   -0.39607  -0.389645  -0.382835  -0.375638  -0.368054  -0.360093  -0.351771  -0.343114  -0.334163  -0.324969    -0.3156  -0.306138   -0.29668  -0.287331  -0.278207  -0.269421  -0.261081  -0.253283  -0.246101  -0.239586  -0.233762     -0.228629 |  0.70246  0.702434  0.702368  0.702261  0.702114  0.701929  0.701703  0.701439  0.701135  0.700791  0.700408  0.699984  0.699519  0.699012  0.698463  0.697871  0.697236  0.696556  0.695831  0.695061  0.694244  0.693381  0.692472  0.691515  0.690512  0.689464  0.688372  0.687237  0.686063  0.684853  0.683612  0.682345   0.68106  0.679765  0.678471  0.677188   0.67593   0.67471  0.673542   0.67244  0.671417  0.670483  0.669647  0.668912  0.668279  0.667745  0.667303  0.666945  0.666659      0.666434 | 812.756821 | 817.267945  825.500862  832.008558  836.803345  839.899215  841.311994  841.059509  839.161764  835.641143   830.52263  823.834044  815.606302  805.873695  794.674195  782.049766  768.046705  752.715988  736.113632  718.301055  699.345434  679.320059  658.304651  636.385655  613.656475  590.217641  566.176882  541.649079  516.756075  491.626304  466.394203  441.199385  416.185515   391.49888  367.286612  343.694546  320.864733  298.932614  278.023937  258.251496   239.71186  222.482273  206.617972  192.150172  179.084986  167.403468  157.062916  147.999401  140.131377   133.36405          127.594129 | 124.912751 | 1.237473  1.240615  1.243081  1.244889  1.246053  1.246582  1.246488  1.245776  1.244452  1.242519   1.23998  1.236836  1.233085  1.228726  1.223757  1.218173  1.211971  1.205146  1.197692  1.189605  1.180879  1.171509  1.161492  1.150825  1.139507  1.127538  1.114923  1.101669  1.087788  1.073297  1.058222  1.042593  1.026452   1.00985  0.992853  0.975537  0.957993   0.94033  0.922667  0.905142    0.8879  0.871096  0.854886  0.839424  0.824848  0.811279   0.79881  0.787499  0.777373       0.768423 | 6.027609  6.051019  6.069419  6.082917  6.091606  6.095565  6.094858  6.089538   6.07965  6.065225  6.046291  6.022866  5.994961  5.962584  5.925738  5.884423  5.838638  5.788381  5.733653  5.674456  5.610798  5.542695  5.470171  5.393267  5.312035  5.226551  5.136914  5.043253  4.945731  4.844551  4.739966   4.63228  4.521859  4.409137   4.29462  4.178894  4.062625  3.946556  3.831506  3.718354  3.608022  3.501444  3.399538  3.303159  3.213057  3.129838  3.053927  2.985548  2.924716        2.871249 | 812.756821   821.85586  829.223841  834.872272  838.814268  841.064701  841.640349  840.560079  837.845029  833.518823  827.607792  820.141232  811.151667  800.675142  788.751533  775.424874    760.7437  744.761405    727.5366  709.133478  689.622173  669.079096  647.587244  625.236475  602.123709  578.353055  554.035834  529.290471  504.242216  479.022686  453.769162  428.623626  403.731495  379.240024  355.296352  332.045186  309.626134  288.170729  267.799222  248.617267  230.712667  214.152411  198.980238   185.21502  172.850181  161.854356  152.173327  143.733166  136.444336  130.206375  124.912751 |  100.0 | 712.756821 | 124.912751 |
    | 03:30 | 9.373376  9.426998  9.471884  9.508236  9.536228  9.556009  9.567705  9.571419  9.567239  9.555235  9.535461  9.507961  9.472764  9.429892  9.379359  9.321171  9.255331   9.18184  9.100699   9.01191  8.915483  8.811436  8.699799  8.580619  8.453965  8.319933  8.178652  8.030292   7.87507  7.713259  7.545201  7.371312  7.192096  7.008154  6.820197  6.629053  6.435675  6.241146  6.046678    5.8536  5.663345  5.477418  5.297359  5.124684  4.960829  4.807079  4.664497   4.53387  4.415666             4.310013 |  579.90155  585.746449  590.661157  594.656249  597.741629  599.926681  601.220415  601.631603  601.168922  599.841094  597.657031  594.625992   590.75774  586.062723  580.552251  574.238704  567.135742  559.258544  550.624055  541.251253  531.161442  520.378557  508.929483  496.844395  484.157112  470.905446  457.131565  442.882337  428.209656  413.170735  397.828333  382.250912  366.512672  350.693445  334.878395  319.157489  303.624672  288.376723  273.511729  259.127158  245.317549  232.171834  219.770423  208.182184  197.461582  187.646221  178.755115  170.787953   163.72551  157.531264 | 108.733762  109.269982  109.718837  110.082355  110.362279  110.560091  110.677045  110.714191  110.672392  110.552349  110.354613  110.079607  109.727639  109.298923  108.793589  108.211709  107.553312  106.818401  106.006986  105.119099  104.154831   103.11436  101.997988   100.80619   99.539652   98.199333   96.786524   95.302921   93.750697   92.132593   90.452013   88.713123   86.920961   85.081543   83.201971   81.290527   79.356748   77.411462   75.466778   73.535999   71.633449   69.774184   67.973586   66.246839   64.608294   63.070789    61.64497   60.338703   59.156658     58.100132 | 1.825475  1.831609  1.836734  1.840879  1.844068   1.84632   1.84765  1.848073  1.847597  1.846231  1.843981  1.840848  1.836834  1.831939   1.82616  1.819494  1.811935  1.803477  1.794114  1.783838  1.772643  1.760519  1.747461  1.733461  1.718516  1.702624  1.685783  1.667999  1.649281  1.629643  1.609108  1.587706   1.56548  1.542483  1.518783  1.494466  1.469634  1.444411   1.41894  1.393387  1.367938  1.342797  1.318181  1.294313  1.271417  1.249703  1.229357  1.210533   1.19334  1.177839 | 1.338792  1.338734  1.338686  1.338647  1.338618  1.338598  1.338586  1.338582  1.338586  1.338598  1.338619  1.338648  1.338685  1.338731  1.338785  1.338849  1.338922  1.339006    1.3391  1.339206  1.339325  1.339456  1.339602  1.339763   1.33994  1.340136  1.340351  1.340588  1.340848  1.341133  1.341447   1.34179  1.342167  1.342579  1.343029  1.343519  1.344053  1.344631  1.345256  1.345927  1.346643  1.347401  1.348196  1.349021  1.349867  1.350723  1.351576  1.352412  1.353217          1.353979 | 0.755448  0.756496  0.757365  0.758065    0.7586  0.758977    0.7592   0.75927  0.759191  0.758963  0.758586  0.758059  0.757382  0.756552  0.755566  0.754419  0.753109  0.751628  0.749972  0.748132  0.746101  0.743871  0.741432  0.738773  0.735883   0.73275  0.729361  0.725703  0.721761  0.717522   0.71297  0.708092  0.702876  0.697311  0.691388  0.685104  0.678459  0.671461  0.664128  0.656487  0.648577  0.640454  0.632186  0.623855  0.615555  0.607391  0.599469  0.591893  0.584757      0.578139 | -0.459585   -0.46046  -0.461123  -0.461582   -0.46184  -0.461904  -0.461777  -0.461459  -0.460952  -0.460257  -0.459372  -0.458295  -0.457023  -0.455554  -0.453881     -0.452  -0.449905  -0.447588  -0.445042  -0.442257  -0.439225  -0.435933  -0.432372  -0.428528   -0.42439  -0.419944  -0.415177  -0.410076  -0.404627  -0.398818  -0.392638  -0.386079  -0.379135  -0.371805  -0.364093  -0.356011  -0.347581  -0.338834  -0.329815  -0.320582  -0.311209  -0.301783  -0.292405  -0.283185  -0.274236   -0.26567  -0.257589   -0.25008  -0.243206     -0.237008 | 0.702101  0.702146  0.702149  0.702112  0.702035  0.701918  0.701762  0.701567  0.701332  0.701059  0.700746  0.700393       0.7  0.699566  0.699091  0.698574  0.698015  0.697412  0.696766  0.696075  0.695338  0.694556  0.693728  0.692853  0.691931  0.690963  0.689948  0.688889  0.687786  0.686643  0.685461  0.684245  0.683001  0.681734  0.680452  0.679164  0.677882  0.676616   0.67538  0.674188  0.673054   0.67199  0.671008  0.670119  0.669329   0.66864  0.668051  0.667558  0.667153      0.666827 | 784.982427 | 790.710094  801.397827  810.412313  817.758568  823.443338   827.47521  829.864731  830.624555  829.769591  827.317189  823.287334  817.702865  810.589714   801.97717  791.898162  780.389558  767.492491  753.252693  737.720844  720.952927  703.010587  683.961481  663.879611  642.845633  620.947116  598.278751   574.94247  551.047472  526.710112  502.053635  477.207715  452.307768  427.494007  402.910189  378.702053  355.015392  331.993778  309.775944  288.492859  268.264578  249.196984  231.378593  214.877627  199.739607  185.985719  173.612191  162.590847  152.870913  144.381975          137.037896 |  133.60186 | 1.227172  1.231349  1.234838  1.237661  1.239832  1.241365   1.24227  1.242558  1.242234  1.241305  1.239772  1.237639  1.234907  1.231574  1.227639  1.223099   1.21795   1.21219  1.205811   1.19881  1.191181  1.182918  1.174017  1.164471  1.154279  1.143437  1.131946  1.119807  1.107026  1.093612  1.079579  1.064947  1.049744  1.034006  1.017778  1.001117  0.984091  0.966785  0.949296  0.931736  0.914233  0.896925  0.879963  0.863502  0.847695  0.832689  0.818616  0.805583  0.793668       0.782918 | 5.951054  5.982056     6.008  6.029008  6.045182  6.056611  6.063367  6.065513  6.063098  6.056163  6.044739  6.028849  6.008509  5.983729  5.954513  5.920864  5.882778  5.840251  5.793278  5.741855  5.685981  5.625658  5.560894  5.491709   5.41813    5.3402  5.257983  5.171561  5.081044  4.986574  4.888331  4.786536  4.681463  4.573439  4.462858  4.350181  4.235947   4.12077  4.005348  3.890451  3.776922  3.665653  3.557568  3.453591   3.35461  3.261435  3.174752  3.095091  3.022789        2.957982 | 784.982427  796.508674    806.3595  814.539083   821.05328  825.909728  829.117963  830.689545  830.638212  828.980045  825.733654   820.92039  814.564569  806.693727  797.338891  786.534872   774.32058  760.739347   745.83928  729.673605  712.301035  693.786118  674.199585  653.618678  632.127432  609.816917  586.785411  563.138476  538.988922  514.456634  489.668212  464.756412  439.859339  415.119362  390.681709  366.692737  343.297825  320.638928  298.851788  278.062868  258.386114  239.919664  222.742723  206.912815  192.463681  179.404087  167.717744  157.364491  148.282714  140.392879   133.60186 |  100.0 | 684.982427 |  133.60186 |
    | 04:00 | 9.225254  9.291504   9.34896  9.397843  9.438343  9.470621  9.494817  9.511046  9.519406  9.519977  9.512821   9.49799  9.475519  9.445436  9.407756  9.362488  9.309634  9.249192  9.181157  9.105523  9.022287  8.931448  8.833016  8.727009   8.61346  8.492422  8.363974  8.228221  8.085306  7.935418  7.778794  7.615735  7.446611  7.271871  7.092057  6.907815  6.719901   6.52919  6.336688  6.143524  5.950956   5.76035  5.573164   5.39091  5.215113   5.04725  4.888687   4.74061  4.603964             4.479397 | 563.905408  571.032753  577.249664  582.564939  586.986729  590.522666  593.179993  594.965682  595.886559   595.94942  595.161158  593.528894  591.060115   587.76282   583.64568  578.718208  572.990948  566.475672  559.185599  551.135631   542.34261  532.825583  522.606101  511.708523  500.160346  487.992537   475.23989  461.941373  448.140478  433.885559  419.230135  404.233157  388.959199  373.478563  357.867258  342.206803  326.583841  311.089485  295.818363  280.867319   266.33373  252.313427  238.898237  226.173219  214.213714  203.082394  192.826577  183.476074   175.04188  167.515914 | 107.252544  107.915036    108.4896  108.978433  109.383429  109.706214  109.948169  110.110461  110.194061  110.199766  110.128212  109.979898  109.755193  109.454358  109.077558  108.624878  108.096341  107.491923  106.811572  106.055234  105.222869  104.314484   103.33016  102.270086  101.134598   99.924225   98.639737   97.282205   95.853062   94.354179   92.787945   91.157354   89.466105   87.718705   85.920575   84.078154   82.199007   80.291903   78.366876   76.435243    74.50956   72.603502   70.731639   68.909102    67.15113   65.472497   63.886865   62.406101    61.03964     59.793965 | 1.808476   1.81609  1.822679  1.828275  1.832905   1.83659   1.83935    1.8412  1.842152  1.842217  1.841402  1.839711  1.837149  1.833715  1.829409  1.824229   1.81817  1.811229  1.803398  1.794672  1.785041  1.774499  1.763037  1.750648  1.737325  1.723062  1.707855  1.691702  1.674606  1.656572  1.637612  1.617744  1.596993  1.575395  1.552996  1.529857  1.506051  1.481672   1.45683  1.431657  1.406306  1.380948  1.355778  1.331004  1.306845  1.283524  1.261257  1.240244  1.220656  1.202629 | 1.338956  1.338882  1.338818  1.338765  1.338721  1.338687  1.338662  1.338644  1.338636  1.338635  1.338643  1.338658  1.338682  1.338714  1.338754  1.338803  1.338862  1.338929  1.339007  1.339095  1.339194  1.339305  1.339428  1.339566  1.339718  1.339886  1.340071  1.340274  1.340499  1.340745  1.341016  1.341313  1.341639  1.341996  1.342387  1.342815  1.343282   1.34379  1.344342  1.344939  1.345582  1.346271  1.347003  1.347775  1.348581  1.349413  1.350261  1.351113  1.351957          1.352777 | 0.752505  0.753831  0.754968  0.755927  0.756716  0.757341  0.757807  0.758118  0.758279   0.75829  0.758153  0.757868  0.757435  0.756854  0.756121  0.755234  0.754191  0.752986  0.751614  0.750071  0.748348   0.74644  0.744337  0.742031  0.739512  0.736768  0.733788  0.730561  0.727072  0.723308  0.719255  0.714899  0.710226  0.705223  0.699877   0.69418  0.688125  0.681709  0.674938  0.667823  0.660386  0.652659  0.644688  0.636533  0.628269  0.619982  0.611771  0.603741  0.595997      0.588639 | -0.456474  -0.457693   -0.45869  -0.459474  -0.460053  -0.460432  -0.460617   -0.46061  -0.460414  -0.460031  -0.459459    -0.4587  -0.457751  -0.456609  -0.455272  -0.453736  -0.451995  -0.450043  -0.447875  -0.445481  -0.442855  -0.439986  -0.436865  -0.433481  -0.429822  -0.425876   -0.42163  -0.417071  -0.412186  -0.406961  -0.401385  -0.395444   -0.38913  -0.382436  -0.375357  -0.367895  -0.360057   -0.35186  -0.343328    -0.3345  -0.325425  -0.316168  -0.306809  -0.297439  -0.288163  -0.279092  -0.270339  -0.262012  -0.254204     -0.246994 | 0.701634  0.701749  0.701822  0.701854  0.701845  0.701797  0.701708  0.701581  0.701414  0.701209  0.700964  0.700681  0.700357  0.699994  0.699591  0.699147  0.698661  0.698134  0.697563  0.696949  0.696291  0.695588   0.69484  0.694046  0.693205  0.692317  0.691383  0.690403  0.689377  0.688307  0.687195  0.686042  0.684854  0.683633  0.682387  0.681121  0.679844  0.678566  0.677297  0.676051   0.67484  0.673678  0.672579  0.671555  0.670617  0.669774   0.66903  0.668386   0.66784      0.667387 | 754.817086 | 761.644889  774.561774   785.87282  795.575811  803.670355  810.157951   815.04207  818.328251  820.024218  820.140019  818.688181  815.683888  811.145179  805.093176  797.552323  788.550656  778.120087  766.296713   753.12114   738.63882    722.9004  705.962076  687.885943  668.740336  648.600149   627.54712  605.670068   583.06506  559.835491  536.092047  511.952536  487.541532  462.989827  438.433635  414.013527  389.873048   366.15701  343.009432  320.571127  298.976965  278.352866  258.812614   240.45463  223.358896  207.584248  193.166283  180.116147  168.420393  158.042048          148.922886 |  144.62564 | 1.215595   1.22078  1.225268  1.229079  1.232231   1.23474  1.236619  1.237879  1.238527  1.238572  1.238017  1.236866  1.235121  1.232783  1.229851  1.226323  1.222197   1.21747  1.212136  1.206191   1.19963  1.192446  1.184635   1.17619  1.167106  1.157379  1.147006  1.135985  1.124317  1.112005  1.099056  1.085481  1.071297  1.056527  1.041202  1.025362  1.009056  0.992346  0.975308   0.95803  0.940615  0.923183  0.905863  0.888801  0.872147  0.856055  0.840675  0.826148  0.812593       0.800107 | 5.865375  5.903704  5.936935  5.965201  5.988614  6.007271  6.021254  6.030632  6.035463  6.035792  6.031658  6.023087  6.010101  5.992714  5.970932  5.944758  5.914191  5.879226  5.839855  5.796071  5.747866  5.695234  5.638172  5.576683  5.510778  5.440477  5.365814  5.286838  5.203618  5.116248   5.02485   4.92958  4.830634  4.728255  4.622736  4.514432  4.403763  4.291221  4.177372  4.062865  3.948424  3.834848  3.722994  3.613766  3.508084  3.406854  3.310928  3.221059  3.137867        3.061797 | 754.817086  768.537161  780.652864  791.161084  800.060505  807.351664  813.037025  817.121066  819.610386  820.513834  819.842651  817.610643  813.834363  808.533329  801.730254    793.4513  783.726359  772.589346  760.078519  746.236809  731.112163  714.757902  697.233068  678.602777  658.938549  638.318619  616.828197  594.559676  571.612766  548.094521  524.119243  499.808229  475.289332    450.6963  426.167852  401.846468  377.876851  354.404049  331.571214  309.517025   288.37279  268.259319  249.283673  231.535949  215.086315   199.98253  186.248218  173.882136  162.858604  153.129203   144.62564 |  100.0 | 654.817086 |  144.62564 |
    | 04:30 | 9.063136  9.141484     9.211  9.271919  9.324445  9.368752   9.40499  9.433287   9.45375  9.466468  9.471513  9.468943    9.4588  9.441118  9.415916  9.383206  9.342992  9.295272  9.240038  9.177279  9.106984  9.029142  8.943745  8.850794  8.750296  8.642272  8.526762  8.403823  8.273544  8.136045  7.991483  7.840066  7.682056  7.517779  7.347637  7.172117  6.991801  6.807377  6.619648   6.42954  6.238107  6.046527  5.856097  5.668217  5.484359  5.306036  5.134746  4.971911  4.818818             4.676544 | 546.649201  554.955869   562.37758    568.9212  574.593005  579.398798  583.344013  586.433814  588.673202   590.06711  590.620509  590.338518  589.226517  587.290274  584.536074  580.970869  576.602428  571.439516  565.492077  558.771437  551.290527   543.06412   534.10909  524.444689  514.092839  503.078449  491.429741  479.178587  466.360858  453.016772  439.191234  424.934153  410.300731  395.351694  380.153443  364.778104  349.303434  333.812544    318.3934  303.138047  288.141515  273.500362  259.310835  245.666639  232.656366  220.360654  208.849254  198.178191  188.387311  179.498488 | 105.631359  106.414836  107.109998   107.71919  108.244446  108.687515  109.049898  109.332869  109.537501  109.664681  109.715132  109.689427  109.588003  109.411176  109.159157   108.83206  108.429925  107.952723   107.40038   106.77279  106.069838  105.291417  104.437454  103.507941   102.50296  101.422724  100.267616   99.038234   97.735445   96.360445   94.914829   93.400659   91.820557    90.17779   88.476375   86.721174   84.918011   83.073768   81.196478   79.295403   77.381072   75.465272   73.560973   71.682165   69.843592   68.060364   66.347457   64.719115   63.188181     61.765441 | 1.789771  1.798823  1.806835  1.813841  1.819869  1.824946  1.829093  1.832327  1.834664  1.836116  1.836692  1.836398  1.835241  1.833222  1.830342  1.826601  1.821995  1.816522  1.810176  1.802951   1.79484  1.785835  1.775928   1.76511  1.753374  1.740711  1.727115   1.71258  1.697104  1.680686   1.66333  1.645044  1.625841  1.605743   1.58478  1.562991  1.540429  1.517159  1.493264  1.468843  1.444015   1.41892   1.39372  1.368593  1.343741  1.319373  1.295711  1.272974  1.251369  1.231084 | 1.339145  1.339053  1.338973  1.338904  1.338845  1.338797  1.338757  1.338727  1.338705  1.338691  1.338686  1.338689    1.3387  1.338718  1.338746  1.338781  1.338825  1.338878   1.33894  1.339011  1.339093  1.339186   1.33929  1.339406  1.339535  1.339679  1.339837  1.340012  1.340205  1.340418  1.340652  1.340908   1.34119    1.3415  1.341839   1.34221  1.342616   1.34306  1.343544   1.34407  1.344641  1.345257  1.345918  1.346624  1.347371  1.348156  1.348971  1.349808  1.350656          1.351502 | 0.749197  0.750807  0.752218   0.75344  0.754484  0.755357  0.756067  0.756618  0.757015  0.757261  0.757358  0.757308  0.757112   0.75677   0.75628  0.755641   0.75485  0.753905  0.752802  0.751536  0.750101  0.748491    0.7467   0.74472  0.742541  0.740156  0.737552   0.73472  0.731648  0.728321  0.724729  0.720856  0.716688  0.712212  0.707415  0.702283  0.696806  0.690975  0.684787  0.678243  0.671349  0.664122  0.656588  0.648785  0.640765  0.632594  0.624351  0.616129  0.608028      0.600152 | -0.452891  -0.454458  -0.455793  -0.456905  -0.457804  -0.458498  -0.458992  -0.459292  -0.459401  -0.459322  -0.459056  -0.458604  -0.457966   -0.45714  -0.456124  -0.454915   -0.45351  -0.451903   -0.45009  -0.448064  -0.445818  -0.443344  -0.440634  -0.437677  -0.434464  -0.430983  -0.427222  -0.423169  -0.418812  -0.414136  -0.409129  -0.403778   -0.39807  -0.391996  -0.385546  -0.378715  -0.371502  -0.363909   -0.35595  -0.347643  -0.339018   -0.33012  -0.321003  -0.311738   -0.30241  -0.293117  -0.283964  -0.275064  -0.266527     -0.258454 | 0.701065  0.701251  0.701393  0.701493  0.701551  0.701569  0.701547  0.701486  0.701386  0.701247   0.70107  0.700853  0.700598  0.700304   0.69997  0.699596  0.699182  0.698727   0.69823  0.697691  0.697108  0.696483  0.695812  0.695097  0.694336  0.693529  0.692675  0.691775  0.690829  0.689836  0.688799  0.687719  0.686597  0.685436  0.684242  0.683018  0.681771  0.680507  0.679237   0.67797  0.676718  0.675494   0.67431   0.67318  0.672119  0.671136  0.670243  0.669446  0.668749       0.66815 |  722.79402 | 730.598093   745.50288  758.883087  770.729355  781.034286  789.792466    797.0005  802.657069  806.763004  809.321385  810.337655  809.819761  807.778315  804.226773  799.181646  792.662723  784.693326  775.300579  764.515703  752.374329  738.916821   724.18862   708.24059  691.129368   672.91771  653.674818  633.476642  612.406145  590.553507  568.016252  544.899284  521.314786  497.381976  473.226674  448.980649  424.780714  400.767533  377.084109  353.873938  331.278804  309.436249  288.476729  268.520541  249.674633  232.029451  215.656013  200.603473  186.897388  174.538955          163.505363 | 158.267707 | 1.202852   1.20902  1.214477  1.219249  1.223354  1.226812  1.229635  1.231838  1.233429  1.234418   1.23481   1.23461  1.233822  1.232447  1.230486  1.227938  1.224802  1.221075  1.216753  1.211832  1.206306  1.200171   1.19342  1.186048  1.178048  1.169415  1.160143   1.15023  1.139671  1.128467  1.116619  1.104132  1.091014  1.077279  1.062946  1.048042    1.0326  1.016666  1.000293  0.983548  0.966513  0.949282  0.931964  0.914683  0.897575  0.880785  0.864466   0.84877  0.833841       0.819811 | 5.771526  5.816891  5.857126  5.892375  5.922757   5.94838  5.969333  5.985691  5.997519   6.00487  6.007786    6.0063  6.000438  5.990218  5.975649  5.956738  5.933484  5.905884  5.873929  5.837611  5.796917  5.751837   5.70236   5.64848  5.590194  5.527506  5.460427  5.388984  5.313213  5.233172  5.148939  5.060617  4.968343  4.872288  4.772665  4.669739  4.563826  4.455308  4.344631  4.232318  4.118968  4.005258   3.89194  3.779833  3.669812  3.562784  3.459659  3.361315   3.26856        3.182087 |  722.79402  738.459753   752.60598  765.222366  776.300536  785.834079  793.818582  800.251671  805.133083  808.464744  810.250884  810.498158  809.215799  806.415788   802.11305  796.325669  789.075128  780.386571  770.289086  758.816008   746.00524  731.899585  716.547095  700.001418   682.32215  663.575174  643.832978  623.174943   601.68758  579.464704  556.607523  533.224609   509.43174  485.351569   461.11309  436.850871  412.704015  388.814821  365.327102  342.384173  320.126472  298.688862  278.197649  258.767411  240.497774  223.470306  207.745763   193.36192  180.332264  168.645741  158.267707 |  100.0 |  622.79402 | 158.267707 |
    | 05:00 | 8.888712  8.978581  9.059601   9.13202  9.196051  9.251882  9.299671  9.339557  9.371654  9.396061  9.412857  9.422108  9.423863  9.418161  9.405026  9.384475  9.356515  9.321144  9.278354  9.228134  9.170467  9.105337  9.032725  8.952617  8.865005  8.769886  8.667271  8.557184   8.43967  8.314796  8.182661  8.043399  7.897186   7.74425  7.584878  7.419427  7.248331  7.072114  6.891402  6.706926  6.519539  6.330217   6.14006  5.950292  5.762247  5.577349  5.397081  5.222943  5.056392             4.898788 | 528.376671  537.753274  546.275878  553.949227  560.777585  566.764821  571.914493  576.229927  579.714299  582.370711   584.20228  585.212221  585.403945  584.781156  583.347964  581.109006  578.069572  574.235755  569.614608  564.214313  558.044375  551.115828  543.441459   535.03605  525.916644  516.102824  505.617011  494.484781  482.735195  470.401138   457.51966  444.132322   430.28552   416.03079  401.425067  386.530879  371.416457  356.155719  340.828099  325.518177  310.315066  295.311502  280.602605  266.284267  252.451156   239.19435  226.598666  214.739786  203.681377   193.47242 | 103.887119  104.785808  105.596011  106.320198  106.960515  107.518819  107.996712  108.395565   108.71654  108.960609  109.128574   109.22108  109.238633  109.181606   109.05026  108.844752  108.565148  108.211437  107.783543  107.281342  106.704674  106.053367  105.327248  104.526174   103.65005  102.698863  101.672711  100.571843     99.3967   98.147964   96.826614    95.43399    93.97186     92.4425   90.848783   89.194269   87.483308   85.721143   83.914015   82.069261   80.195395   78.302171   76.400603   74.502922    72.62247   70.773491   68.970815   67.229426   65.563919     63.987877 | 1.769528  1.779973  1.789362  1.797731  1.805114  1.811538  1.817027  1.821602  1.825278  1.828071  1.829992   1.83105   1.83125  1.830598  1.829097  1.826746  1.823545  1.819491   1.81458  1.808807  1.802166   1.79465   1.78625  1.776959  1.766767  1.755665  1.743646  1.730701  1.716825  1.702013  1.686262  1.669575  1.651955  1.633414  1.613969  1.593644  1.572473  1.550501  1.527786    1.5044  1.480432  1.455991  1.431204  1.406218  1.381202  1.356344  1.331847  1.307927  1.284801  1.262683 | 1.339358  1.339247  1.339149  1.339064   1.33899  1.338926  1.338873  1.338829  1.338793  1.338767  1.338749  1.338739  1.338737  1.338743  1.338757   1.33878   1.33881  1.338849  1.338897  1.338953  1.339019  1.339095  1.339181  1.339279  1.339388   1.33951  1.339645  1.339795  1.339961  1.340143  1.340345  1.340566   1.34081  1.341078  1.341371  1.341693  1.342046  1.342432  1.342854  1.343315  1.343817  1.344361   1.34495  1.345584  1.346263  1.346986  1.347748  1.348544  1.349366          1.350205 | 0.745532  0.747434  0.749124  0.750614  0.751916   0.75304  0.753993  0.754783  0.755414  0.755892   0.75622    0.7564  0.756435  0.756324  0.756068  0.755666  0.755117  0.754419  0.753569  0.752563  0.751397  0.750067  0.748566  0.746887  0.745025  0.742969  0.740712  0.738243  0.735553  0.732629  0.729459   0.72603   0.72233  0.718344  0.714059   0.70946  0.704534  0.699271  0.693661  0.687697  0.681377  0.674705  0.667693   0.66036  0.652738  0.644871  0.636816  0.628646  0.620444      0.612306 |  -0.44885  -0.450773  -0.452449  -0.453892  -0.455112   -0.45612  -0.456922  -0.457526  -0.457936  -0.458156  -0.458189  -0.458036  -0.457698  -0.457176  -0.456469  -0.455573  -0.454488  -0.453209  -0.451732  -0.450053  -0.448164   -0.44606  -0.443734  -0.441176  -0.438377  -0.435329  -0.432019  -0.428437  -0.424571  -0.420407  -0.415934  -0.411138  -0.406005  -0.400524  -0.394684  -0.388473  -0.381886  -0.374918  -0.367571  -0.359851  -0.351774  -0.343364  -0.334656  -0.325699  -0.316554  -0.307299  -0.298023  -0.288827   -0.27982     -0.271111 | 0.700401  0.700656  0.700866  0.701033  0.701158  0.701242  0.701286  0.701289  0.701254   0.70118  0.701068  0.700917  0.700727    0.7005  0.700233  0.699927  0.699582  0.699197  0.698771  0.698304  0.697795  0.697244   0.69665  0.696012  0.695329    0.6946  0.693826  0.693006   0.69214  0.691227  0.690268  0.689263  0.688215  0.687124  0.685993  0.684826  0.683626    0.6824  0.681153  0.679894  0.678633   0.67738  0.676146  0.674946  0.673792  0.672697  0.671675  0.670736  0.669888      0.669138 | 689.426711 | 698.078496  714.719825  729.930003  743.692649   755.99352  766.820486  776.163519  784.014708  790.368294  795.220728  798.570747  800.419477  800.770553  799.630266  797.007726  792.915058  787.367611  780.384196  771.987344  762.203585  751.063749  738.603279  724.862567  709.887291  693.728767   676.44429  658.097473   638.75856  618.504713  597.420248  575.596806  553.133439  530.136588  506.719926  483.004028  459.115856  435.187993   411.35763  387.765241  364.552933  341.862468  319.832923  298.598042  278.283306  259.002826  240.856177  223.925348  208.272029  193.935476          180.931185 | 174.713131 | 1.189058  1.196177  1.202574  1.208276  1.213305   1.21768  1.221419  1.224534  1.227038   1.22894  1.230248  1.230968  1.231105  1.230661  1.229638  1.228037  1.225857  1.223097  1.219752   1.21582  1.211297  1.206176  1.200454  1.194122  1.187177   1.17961  1.171416  1.162589  1.153125  1.143021  1.132273  1.120882  1.108852  1.096188  1.082901  1.069007  1.054528  1.039494  1.023944  1.007924  0.991496  0.974732  0.957718  0.940555  0.923357  0.906253  0.889382  0.872893  0.856936        0.84166 | 5.670463  5.722546  5.769479  5.811412  5.848476  5.880782  5.908429  5.931497  5.950058   5.96417  5.973881  5.979229  5.980244  5.976947  5.969354  5.957472  5.941304  5.920848  5.896097  5.867041  5.833668  5.795963  5.753912  5.707502   5.65672  5.601559  5.542016  5.478099  5.409821  5.337212  5.260317  5.179199  5.093947  5.004677  4.911537  4.814717  4.714451  4.611022  4.504774  4.396114  4.285519  4.173541  4.060809  3.948029  3.835979  3.725498   3.61747  3.512798  3.412376        3.317048 | 689.426711  706.780669  722.712105  737.203565  750.239752  761.807488  771.895698  780.495407   787.59977  793.204117  797.306019  799.905379  801.004542  800.608429  798.724692  795.363894  790.539709  784.269151  776.572813  767.475147  757.004745  745.194651  732.082687  717.711788   702.13035  685.392578  667.558827  648.695934  628.877519  608.184252  586.704063  564.532272  541.771636  518.532261  494.931371  471.092896  447.146839    423.2284  399.476816   376.03388  353.042138  330.642719  308.972844  288.163013  268.333959  249.593466  232.033202  215.725772  200.722212   187.05019  174.713131 |  100.0 | 589.426711 | 174.713131 |
    | 05:30 | 8.703663  8.804429  8.896355  8.979698   9.05468  9.121495  9.180312  9.231276   9.27451   9.31012  9.338191  9.358797  9.371992   9.37782  9.376313  9.367491  9.351364  9.327934  9.297194  9.259132  9.213732  9.160971  9.100827  9.033277    8.9583  8.875878  8.786001  8.688668  8.583893  8.471704  8.352153  8.225318  8.091311   7.95028  7.802421  7.647987   7.48729  7.320718  7.148742  6.971926  6.790937  6.606555  6.419681  6.231339  6.042677  5.854959  5.669552  5.487898  5.311479              5.14177 | 509.323711  519.656267  529.170993  537.870307  545.756297  552.830777  559.095344  564.551434  569.200376  573.043456  576.081974  578.317315   579.75102  580.384868  580.220961  579.261826  577.510517  574.970739  571.646978  567.544643  562.670234   557.03151  550.637688  543.499653  535.630186  527.044213  517.759076  507.794811   497.17446  485.924384  474.074593  461.659089  448.716201  435.288921  421.425214  407.178304  392.606901  377.775357   362.75372  347.617653  332.448176  317.331195  302.356764  287.618041  273.209897  259.227145  245.762403  232.903603  220.731246  209.315541 | 102.036632  103.044288  103.963551  104.796976  105.546795   106.21495  106.803122  107.312761  107.745104  108.101199  108.381915  108.587966  108.719917  108.778203  108.763134  108.674912  108.513641  108.279337  107.971939  107.591323  107.137317   106.60971  106.008272  105.332773  104.582999  103.758779  102.860009  101.886683  100.838927   99.717038    98.52153   97.253184   95.913107   94.502796   93.024213   91.479867   89.872899   88.207183   86.487423    84.71926   82.909369   81.065553   79.196814   77.313394   75.426772   73.549595   71.695523   69.878982   68.114792       66.4177 | 1.747914  1.759701  1.770417  1.780103  1.788792  1.796516  1.803301  1.809169  1.814138  1.818226  1.821445  1.823806  1.825317  1.825984  1.825812  1.824802  1.822955  1.820269  1.816743  1.812371   1.80715  1.801071  1.794129  1.786314  1.777619  1.768033  1.757548  1.746156  1.733847  1.720614  1.706451  1.691356  1.675326  1.658365   1.64048  1.621685  1.601998  1.581449  1.560077  1.537931  1.515075  1.491589   1.46757  1.443133  1.418413  1.393568  1.368773  1.344221  1.320121  1.296687 | 1.339597  1.339465  1.339348  1.339246  1.339155  1.339076  1.339008   1.33895  1.338901  1.338861   1.33883  1.338807  1.338793  1.338787  1.338788  1.338798  1.338816  1.338841  1.338875  1.338918  1.338969   1.33903    1.3391  1.339181  1.339272  1.339374  1.339489  1.339616  1.339758  1.339915  1.340088  1.340279  1.340489   1.34072  1.340974  1.341253  1.341559  1.341894  1.342261  1.342663  1.343101  1.343579  1.344099  1.344662  1.345269  1.345922  1.346618  1.347356  1.348131          1.348937 | 0.741517   0.74372  0.745695  0.747457  0.749022  0.750398  0.751597  0.752626  0.753492    0.7542  0.754756  0.755162  0.755421  0.755535  0.755506  0.755333  0.755015  0.754553  0.753944  0.753185  0.752273  0.751204  0.749974  0.748577  0.747007  0.745257   0.74332  0.741186  0.738847  0.736292  0.733511  0.730491   0.72722  0.723686  0.719875  0.715772  0.711365   0.70664  0.701585  0.696189  0.690444  0.684346  0.677895  0.671099  0.663973  0.656542  0.648842  0.640923  0.632849      0.624697 | -0.444366   -0.44665  -0.448673   -0.45045  -0.451994  -0.453317  -0.454427  -0.455332   -0.45604  -0.456555   -0.45688   -0.45702  -0.456976  -0.456748  -0.456338  -0.455744  -0.454964  -0.453998   -0.45284  -0.451488  -0.449937  -0.448181  -0.446215   -0.44403  -0.441619  -0.438973  -0.436083  -0.432939  -0.429529  -0.425842  -0.421866  -0.417588  -0.412995  -0.408075  -0.402813    -0.3972  -0.391223  -0.384875   -0.37815  -0.371046  -0.363567  -0.355723  -0.347534  -0.339029  -0.330248  -0.321245   -0.31209  -0.302863  -0.293659     -0.284582 | 0.699646  0.699971  0.700249  0.700482  0.700672  0.700821  0.700928  0.700996  0.701024  0.701013  0.700964  0.700877  0.700752  0.700589  0.700387  0.700147  0.699869  0.699551  0.699194  0.698797  0.698359   0.69788  0.697358  0.696795  0.696187  0.695536   0.69484  0.694098  0.693311  0.692478  0.691598  0.690672    0.6897  0.688684  0.687625  0.686524  0.685385  0.684211  0.683007  0.681779  0.680535  0.679283  0.678032  0.676795  0.675583  0.674409  0.673287  0.672229  0.671248      0.670354 | 655.199767 | 664.568615  682.690174  699.484436  714.928153   729.00043  741.682645   752.95839  762.813448  771.235784  778.215567  783.745208  787.819423  790.435322  791.592513  791.293233  789.542505   786.34831  781.721789  775.677465   768.23349  759.411907  749.238941  737.745305  724.966519  710.943249  695.721645  679.353692  661.897543  643.417852  623.986071  603.680713  582.587563   560.79981  538.418087  515.550391  492.311857  468.824349   445.21584  421.619548  398.172796  375.015554  352.288664   330.13171  308.680569   288.06465  268.403904  249.805702  232.361732  216.145102          201.207876 |  194.01565 | 1.174325  1.182361  1.189665  1.196265  1.202186  1.207448   1.21207  1.216067  1.219451  1.222235  1.224428  1.226035  1.227064  1.227519  1.227401  1.226713  1.225456  1.223627  1.221225  1.218248  1.214691  1.210551  1.205822  1.200497  1.194572   1.18804  1.180893  1.173127  1.164734   1.15571  1.146049  1.135749  1.124809  1.113229  1.101015  1.088174  1.074719  1.060668  1.046047   1.03089  1.015238  0.999145  0.982675  0.965908  0.948934   0.93186  0.914807  0.897906  0.881301       0.865139 | 5.563137  5.621594  5.674894  5.723193  5.766629  5.805319  5.839366   5.86886  5.893874  5.914472  5.930708  5.942624  5.950254  5.953624  5.952753  5.947651  5.938326  5.924775  5.906995  5.884977  5.858707  5.828171  5.793352  5.754232  5.710794  5.663023  5.610906  5.554435   5.49361  5.428438   5.35894  5.285148  5.207116  5.124915  5.038644  4.948433  4.854446   4.75689  4.656019  4.552141  4.445624  4.336904  4.226487  4.114956   4.00297  3.891261  3.780631  3.671933  3.566056        3.463894 | 655.199767  673.980446  691.445944  707.571823  722.336041  735.718861  747.702783  758.272502  767.414892  775.119014  781.376146  786.179833  789.525963  791.412862  791.841415  790.815209  788.340693   784.42737  779.088009  772.338875  764.199988  754.695399  743.853487  731.707277  718.294762  703.659252  687.849713  670.921114  652.934764  633.958628  614.067612  593.343804  571.876652  549.763058  527.107362  504.021202    480.6232  457.038466  433.397859     409.837  386.494979  363.512747  341.031167  319.188719  298.118883  277.947245  258.788398  240.742789  223.893646  208.304235   194.01565 |  100.0 | 555.199767 |  194.01565 |
    | 06:00 | 8.509648  8.620641  8.722833  8.816487   8.90183  8.979061  9.048354   9.10986  9.163708   9.21001  9.248857  9.280326  9.304481  9.321368  9.331026  9.333477  9.328737  9.316811  9.297694  9.271377  9.237841  9.197066  9.149024  9.093688  9.031029   8.96102  8.883635  8.798856  8.706673  8.607087  8.500115  8.385794  8.264182  8.135369  7.999481  7.856685  7.707198  7.551295  7.389321  7.221696  7.048928  6.871622   6.69049  6.506361  6.320184   6.13303  5.946092  5.760672  5.578162             5.400013 | 489.715297  500.886822  511.281613  520.899531   529.74031  537.803574  545.088861  551.595647  557.323383  562.271527  566.439585  569.827158  572.433997  574.260058  575.305575  575.571131  575.057746  573.766973  571.701006  568.862798  565.256199  560.886099  555.758595  549.881172    543.2629   535.91465  527.849331  519.082141  509.630842  499.516049   488.76154  477.394571  465.446214  452.951689  439.950696  426.487743  412.612437  398.379743  383.850181  369.089934  354.170843  339.170249  324.170646  309.259095  294.526361  280.065723  265.971432  252.336785  239.251853  226.800892 | 100.096483  101.206405  102.228334  103.164871  104.018297  104.790608   105.48354    106.0986  106.637084  107.100096  107.488565  107.803261  108.044805  108.213683  108.310256  108.334771  108.287372  108.168107  107.976943  107.713769  107.378415  106.970659  106.490243  105.936884  105.310294  104.610196  103.836345  102.988557  102.066727  101.070871  100.001154   98.857936   97.641816   96.353691    94.99481   93.566847   92.071976   90.512954   88.893215   87.216963   85.489279   83.716217   81.904902   80.063612   78.201836   76.330298    74.46092   72.606719   70.781615     69.000129 | 1.725096  1.738169  1.750159  1.761109  1.771054  1.780029   1.78806  1.795172  1.801387  1.806721   1.81119  1.814806  1.817579  1.819516  1.820624  1.820905  1.820361  1.818994    1.8168  1.813778  1.809924  1.805231  1.799693  1.793304  1.786054  1.777934  1.768936   1.75905  1.748266  1.736576   1.72397  1.710443  1.695989  1.680606  1.664293  1.647056  1.628905  1.609855   1.58993  1.569163  1.547597  1.525288  1.502306  1.478739  1.454689  1.430282  1.405662  1.380991  1.356454  1.332248 | 1.339861  1.339708  1.339571   1.33945  1.339342  1.339246  1.339163   1.33909  1.339027  1.338974   1.33893  1.338894  1.338867  1.338849  1.338838  1.338835  1.338841  1.338854  1.338875  1.338904  1.338942  1.338989  1.339044  1.339109  1.339183  1.339268  1.339364  1.339472  1.339593  1.339726  1.339875  1.340039   1.34022  1.340419  1.340638   1.34088  1.341144  1.341435  1.341754  1.342103  1.342485  1.342902  1.343358  1.343853  1.344391  1.344973  1.345599  1.346269  1.346982          1.347735 | 0.737162  0.739672  0.741939  0.743981  0.745811  0.747444   0.74889   0.75016   0.75126  0.752198  0.752979  0.753608  0.754088  0.754423  0.754614  0.754663  0.754569  0.754333  0.753954  0.753429  0.752758  0.751936  0.750961  0.749827   0.74853  0.747065  0.745423  0.743599  0.741584  0.739369  0.736944    0.7343  0.731424  0.728305   0.72493  0.721286   0.71736  0.713138  0.708606  0.703752  0.698564  0.693033  0.687154  0.680922  0.674343  0.667427  0.660194  0.652672  0.644906       0.63695 | -0.439453  -0.442106  -0.444481  -0.446596  -0.448466  -0.450105  -0.451523   -0.45273  -0.453733  -0.454539  -0.455154   -0.45558  -0.455822  -0.455881  -0.455758  -0.455455   -0.45497  -0.454302   -0.45345  -0.452409  -0.451178   -0.44975  -0.448122  -0.446287  -0.444239  -0.441969  -0.439471  -0.436734  -0.433749  -0.430505  -0.426991  -0.423196  -0.419107   -0.41471  -0.409994  -0.404946  -0.399552  -0.393803  -0.387689  -0.381201  -0.374338  -0.367098   -0.35949  -0.351527  -0.343233  -0.334642    -0.3258  -0.316767  -0.307618     -0.298439 | 0.698809  0.699201  0.699546  0.699845  0.700099  0.700311  0.700482  0.700611  0.700702  0.700753  0.700766   0.70074  0.700677  0.700576  0.700438  0.700262  0.700047  0.699795  0.699504  0.699174  0.698804  0.698395  0.697944  0.697452  0.696918  0.696341  0.695721  0.695056  0.694346  0.693591   0.69279  0.691943   0.69105  0.690111  0.689127  0.688099  0.687028  0.685918  0.684771  0.683592  0.682386  0.681158  0.679918  0.678674  0.677436  0.676217  0.675028  0.673884  0.672796      0.671778 | 620.561767 | 630.517326  649.862603   667.99308  684.878838  700.492554  714.809364  727.806763  739.464531  749.764688  758.691476  766.231363   772.37307  777.107622  780.428423  782.331348  782.814865  781.880172  779.531366  775.775625  770.623421  764.088754    756.1894  746.947194  736.388322  724.543633  711.448967  697.145493  681.680051  665.105496  647.481029  628.872514  609.352759  589.001759  567.906866   546.16289  523.872085  501.144003  478.095198  454.848725  431.533424   408.28294  385.234455  362.527104  340.300057  318.690267  297.829895  277.843473  258.844865  240.934172          224.194722 | 216.081698 | 1.158766  1.167682  1.175856   1.18332  1.190099  1.196214  1.201687  1.206533  1.210766    1.2144  1.217444  1.219906  1.221795  1.223114  1.223868   1.22406   1.22369  1.222758  1.221264  1.219206  1.216581  1.213384  1.209612  1.205259   1.20032  1.194787  1.188655  1.181917  1.174566  1.166595  1.157999  1.148772  1.138911  1.128412  1.117277  1.105506  1.093107  1.080089  1.066468  1.052264  1.037507  1.022233   1.00649  0.990335  0.973839  0.957086  0.940173  0.923212  0.906328       0.889658 | 5.450486  5.514947  5.574261  5.628587  5.678067  5.722824  5.762965  5.798582  5.829756  5.856554  5.879032  5.897238   5.91121  5.920978  5.926564  5.927981   5.92524  5.918342  5.907285  5.892061  5.872658  5.849063  5.821256  5.789219   5.75293   5.71237   5.66752  5.618362  5.564883  5.507078  5.444947  5.378501  5.307765  5.232779  5.153601  5.070316  4.983033  4.891896  4.797086  4.698829    4.5974  4.493133  4.386423  4.277732  4.167599  4.056636   3.94553  3.835039  3.725984        3.619229 | 620.561767  640.508365  659.255677  676.772458  693.030133  708.002645  721.666336  733.999861  744.984132  754.602285   762.83967  769.683875  775.124755  779.154505  781.767735  782.961582  782.735837    781.0931  778.038954  773.582165  767.734898  760.512968  751.936102  742.028223  730.817758  718.337956  704.627224  689.729465  673.694428  656.578042  638.442748    619.3578  599.399529  578.651558  557.204941  535.158216  512.617329  489.695428  466.512468  443.194616  419.873416  396.684667  373.767012  351.260191  329.302953  308.030637  287.572454  268.048513  249.566718   232.21966  216.081698 |  100.0 | 520.561767 | 216.081698 |
    | 06:30 | 8.308292  8.428797  8.540578  8.643895  8.738977  8.826026  8.905217  8.976705  9.040621  9.097082  9.146185  9.188009  9.222623  9.250079  9.270417  9.283668  9.289848  9.288967  9.281023  9.266009  9.243909    9.2147  9.178357  9.134848   9.08414    9.0262  8.960993  8.888488   8.80866  8.721488  8.626964  8.525094  8.415898   8.29942   8.17573  8.044931  7.907164  7.762616  7.611527  7.454199  7.291008  7.122408  6.948948  6.771275  6.590149  6.406445  6.221161  6.035416  5.850442             5.667574 | 469.762998  481.655022  492.816014  503.243043  512.933298  521.884066  530.092716  537.556688  544.273506  550.240779  555.456224  559.917693  563.623199  566.570961   568.75945  570.187443  570.854094  570.759001  569.902299   568.28475   565.90786  562.773995  558.886522  554.249959  548.870146  542.754429  535.911863  528.353437  520.092312  511.144079  501.527044  491.262516  480.375122  468.893125   456.84876  444.278565  431.223702  417.730273  403.849591   389.63841  375.159082  360.479616  345.673601  330.819973  316.002561  301.309388  286.831665  272.662463  258.895008  245.620618 |  98.082924   99.287967  100.405778  101.438951  102.389775   103.26026  104.052171  104.767045  105.406215  105.970824  106.461845  106.880091  107.226228  107.500787  107.704174  107.836677   107.89848  107.889666  107.810231   107.66009  107.439086  107.147001  106.783568   106.34848  105.841405     105.262  104.609929  103.884882  103.086596  102.214879  101.269644  100.250937   99.158976   97.994196   96.757301   95.449313   94.071643    92.62616   91.115267   89.541991   87.910077   86.224082   84.489479   82.712753   80.901491   79.064453   77.211614   75.354159   73.504423     71.675745 |  1.70124  1.715538  1.728744  1.740902  1.752049  1.762222  1.771448  1.779755  1.787165  1.793696  1.799366  1.804188  1.808173  1.811331  1.813668   1.81519    1.8159  1.815798  1.814886  1.813162  1.810621  1.807261  1.803076  1.798058    1.7922  1.785494  1.777931  1.769501  1.760195  1.750002  1.738912  1.726918  1.714011  1.700184  1.685434  1.669759  1.653161  1.635647   1.61723  1.597928  1.577768   1.55679   1.53504   1.51258  1.489489   1.46586  1.441805  1.417457  1.392967  1.368507 | 1.340153  1.339976  1.339818  1.339676   1.33955  1.339437  1.339337  1.339249  1.339172  1.339105  1.339047  1.338999  1.338959  1.338928  1.338905  1.338891  1.338884  1.338885  1.338894   1.33891  1.338935  1.338968   1.33901   1.33906   1.33912  1.339189  1.339268  1.339358   1.33946  1.339573  1.339699   1.33984  1.339995  1.340166  1.340356  1.340564  1.340793  1.341045  1.341321  1.341624  1.341956   1.34232  1.342717  1.343151  1.343623  1.344137  1.344693  1.345294  1.345938          1.346626 | 0.732474  0.735301  0.737867  0.740192  0.742294  0.744187  0.745883  0.747395   0.74873  0.749897  0.750903  0.751753  0.752452  0.753003   0.75341  0.753675  0.753798   0.75378  0.753622  0.753322   0.75288  0.752292  0.751557  0.750672  0.749631   0.74843  0.747064  0.745527  0.743811   0.74191  0.739814  0.737514  0.735001  0.732264   0.72929  0.726069  0.722586  0.718829  0.714785  0.710439   0.70578  0.700794  0.695472  0.689806  0.683791  0.677428  0.670723  0.663692  0.656358      0.648758 | -0.434126  -0.437154  -0.439886  -0.442344  -0.444544  -0.446501  -0.448228  -0.449736  -0.451033  -0.452129  -0.453028  -0.453737  -0.454259  -0.454598  -0.454756  -0.454735  -0.454534  -0.454154  -0.453593   -0.45285  -0.451922  -0.450806  -0.449497  -0.447992  -0.446284  -0.444367  -0.442234  -0.439877  -0.437287  -0.434455  -0.431372  -0.428025  -0.424404  -0.420496   -0.41629  -0.411771  -0.406929  -0.401749  -0.396221  -0.390334   -0.38408  -0.377452  -0.370451  -0.363078  -0.355344  -0.347267  -0.338876  -0.330209   -0.32132     -0.312273 | 0.697894  0.698354  0.698764  0.699127  0.699445  0.699719  0.699951  0.700142  0.700293  0.700404  0.700477  0.700512  0.700509  0.700469  0.700392  0.700277  0.700125  0.699935  0.699708  0.699443  0.699139  0.698796  0.698413   0.69799  0.697527  0.697022  0.696474  0.695884  0.695249  0.694571  0.693847  0.693078  0.692263  0.691402  0.690496  0.689544  0.688547  0.687508  0.686427  0.685308  0.684154  0.682971  0.681762  0.680537  0.679303  0.678069  0.676847  0.675648  0.674486      0.673372 | 585.920022 | 596.334382  616.650944  635.872055  653.961361  670.885374  686.613272  701.116758  714.369932  726.349217  737.033294  746.403078  754.441709  761.134568  766.469321  770.435981   773.02699  774.237331  774.064655  772.509435  769.575144  765.268447  759.599429  752.581833  744.233328  734.575792  723.635613  711.444012  698.037373  683.457581  667.752368  650.975646  633.187838  614.456172  594.854951  574.465762  553.377615  531.686995  509.497796  486.921109   464.07484   441.08312  418.075484  395.185768  372.550718  350.308269  328.595497  307.546238  287.288418  267.941142          249.611646 | 240.673729 | 1.142493  1.152248  1.161255  1.169545  1.177145  1.184079  1.190367  1.196028  1.201077  1.205527  1.209389  1.212674  1.215388  1.217539  1.219131  1.220168  1.220651  1.220582  1.219961  1.218786  1.217056  1.214767  1.211916  1.208498  1.204508  1.199939  1.194785  1.189041  1.182697  1.175749  1.168188  1.160009  1.151206  1.141773  1.131708  1.121008  1.109675  1.097713  1.085129  1.071936  1.058151  1.043798   1.02891  1.013529  0.997705  0.981503  0.964997  0.948277  0.931447       0.914624 | 5.333429  5.403501  5.468453  5.528448  5.583628  5.634119  5.680031  5.721459  5.758486  5.791184  5.819612  5.843821  5.863853  5.879739  5.891506  5.899171  5.902746  5.902236  5.897641  5.888955  5.876169  5.859268  5.838235  5.813049   5.78369  5.750133  5.712355  5.670333  5.624047   5.57348  5.518619  5.459458  5.396003  5.328267  5.256281  5.180092  5.099768  5.015402  4.927119  4.835077  4.739475  4.640559  4.538628  4.434041  4.327219  4.218657  4.108922  3.998654   3.88857        3.779449 | 585.920022  606.776721  626.556768  645.222337   662.73857   679.07336  694.197189  708.082987  720.706036  732.043904  742.076395  750.785535  758.155579  764.173038  768.826732  772.107863  774.010111  774.529756  773.665814  771.420207  767.797947  762.807344  756.460241  748.772268  739.763114  729.456825  717.882113  705.072684  691.067572  675.911485  659.655149  642.355634  624.076679  604.888973  584.870397  564.106211  542.689149  520.719422  498.304588  475.559262  452.604643  429.567816  406.580802  383.779318  361.301231  339.284672  317.865821  297.176373  277.340721  258.472947  240.673729 |  100.0 | 485.920022 | 240.673729 |
    | 07:00 | 8.101178  8.230437  8.351091  8.463391  8.567563  8.663805  8.752292  8.833178  8.906598  8.972669  9.031488  9.083142  9.127699  9.165216  9.195738  9.219296  9.235913  9.245601  9.248362   9.24419  9.233072  9.214988   9.18991  9.157809  9.118648  9.072391  9.018996  8.958426  8.890643  8.815614  8.733311  8.643717  8.546825  8.442643  8.331201  8.212552  8.086776  7.953991  7.814356  7.668079  7.515425  7.356725  7.192388  7.022906  6.848868  6.670966  6.490007  6.306915  6.122735             5.938629 | 449.663064  462.157038  473.969952  485.095832  495.529128  505.264639  514.297445  522.622871  530.236455  537.133929  543.311219  548.764438  553.489907  557.484171  560.744024  563.266552  565.049177   566.08971  566.386419  565.938104  564.744184  562.804798  560.120914   556.69446  552.528463  547.627206  541.996402  535.643388   528.57733  520.809456  512.353298  503.224963  493.443413  483.030764  472.012603  460.418306  448.281373  435.639752  422.536153  409.018343  395.139399    380.9579  366.538041  351.949632  337.267956  322.573431  307.951061    293.4896  279.280418   265.41601 |  96.011777   97.304372   98.510908   99.633913   100.67563  101.638048  102.522919  103.331783  104.065984  104.726685  105.314882  105.831417  106.276988  106.652161  106.957377  107.192962  107.359133  107.456011  107.483619  107.441901  107.330721  107.149876  106.899103   106.57809  106.186483  105.723905  105.189962  104.584262  103.906432  103.156139  102.333113  101.437169  100.468245   99.426432   98.312014   97.125517    95.86776   94.539912   93.143562   91.680788   90.154245   88.567251   86.923879    85.22906   83.488677   81.709661    79.90007   78.069149   76.227347      74.38629 | 1.676509  1.691967  1.706325  1.719631  1.731924  1.743239  1.753607  1.763056  1.771609  1.779287  1.786107  1.792085  1.797233  1.801561  1.805078   1.80779  1.809702  1.810816  1.811133  1.810654  1.809375  1.807294  1.804407  1.800707  1.796188  1.790841   1.78466  1.777633  1.769752  1.761007  1.751386  1.740881  1.729481  1.717177  1.703963  1.689832  1.674782  1.658813  1.641928  1.624137  1.605454  1.585904  1.565516  1.544334  1.522412  1.499817  1.476635  1.452967  1.428933  1.404674 | 1.340473  1.340271   1.34009  1.339927  1.339781   1.33965  1.339533  1.339428  1.339336  1.339254  1.339183  1.339121  1.339069  1.339025   1.33899  1.338963  1.338944  1.338933   1.33893  1.338935  1.338948  1.338968  1.338997  1.339034  1.339079  1.339134  1.339198  1.339272  1.339356  1.339451  1.339557  1.339677  1.339809  1.339956  1.340119  1.340299  1.340496  1.340714  1.340953  1.341216  1.341504   1.34182  1.342166  1.342545  1.342958  1.343409  1.343899  1.344431  1.345006          1.345625 | 0.727464  0.730614  0.733486  0.736101  0.738478  0.740635  0.742585  0.744341  0.745913   0.74731   0.74854   0.74961  0.750525  0.751291   0.75191  0.752385  0.752719  0.752914  0.752969  0.752885  0.752662  0.752298  0.751792   0.75114   0.75034  0.749388   0.74828   0.74701  0.745573  0.743962  0.742169  0.740188  0.738009  0.735622  0.733017  0.730183  0.727108   0.72378  0.720186  0.716313  0.712147  0.707675  0.702885  0.697766  0.692308  0.686505  0.680356  0.673863  0.667038      0.659898 | -0.428399  -0.431809  -0.434905  -0.437709  -0.440242   -0.44252  -0.444557  -0.446366  -0.447958  -0.449341  -0.450523  -0.451511  -0.452309  -0.452923  -0.453354  -0.453607  -0.453681  -0.453579  -0.453299  -0.452841  -0.452203  -0.451383  -0.450378  -0.449184  -0.447797  -0.446211   -0.44442  -0.442419  -0.440198   -0.43775  -0.435066  -0.432136   -0.42895  -0.425497  -0.421764  -0.417741  -0.413413   -0.40877  -0.403797  -0.398483  -0.392818  -0.386791  -0.380396  -0.373628   -0.36649  -0.358986  -0.351131  -0.342948  -0.334468     -0.325739 | 0.696909  0.697435   0.69791  0.698336  0.698715   0.69905  0.699342  0.699593  0.699803  0.699973  0.700105  0.700198  0.700254  0.700273  0.700254  0.700199  0.700107  0.699978  0.699812  0.699609  0.699368  0.699089  0.698772  0.698416   0.69802  0.697584  0.697107  0.696588  0.696026  0.695422  0.694774  0.694081  0.693343   0.69256   0.69173  0.690856  0.689935   0.68897  0.687962  0.686911   0.68582  0.684693  0.683534  0.682347  0.681139  0.679917   0.67869  0.677469  0.676265      0.675089 |  551.63713 | 562.386699  583.429946  603.502172   622.56087  640.566681  657.483147  673.276505  687.915528  701.371399   713.61762   724.62995  734.386361   742.86703  750.054345  755.932941  760.489752  763.714088  765.597733  766.135068  765.323211  763.162184    759.6551  754.808373  748.631951   741.13957  732.349025  722.282469  710.966719  698.433582  684.720194  669.869353  653.929868  636.956886  619.012207  600.164568  580.489887  560.071442  538.999971  517.373675  495.298091  472.885805  450.255989  427.533712  404.849001  382.335626   360.12957  338.367179  317.182984  296.707208          277.063006 | 267.430129 | 1.125616  1.136166  1.145963  1.155039  1.163423  1.171138  1.178207  1.184648  1.190477  1.195709  1.200356  1.204429  1.207936  1.210884   1.21328  1.215128   1.21643  1.217189  1.217405  1.217078  1.216207   1.21479  1.212823  1.210303  1.207224  1.203582   1.19937  1.194582  1.189211  1.183251  1.176693   1.16953  1.161757  1.153365  1.144351  1.134709  1.124437  1.113535  1.102003  1.089849  1.077081  1.063714  1.049769  1.035273  1.020263  1.004784  0.988892  0.972657  0.956159       0.939494 | 5.212864  5.288128  5.358322  5.423608  5.484127  5.540005  5.591352  5.638266  5.680831   5.71912  5.753196  5.783112  5.808911  5.830629  5.848294  5.861928  5.871543  5.877148  5.878746  5.876332  5.869899  5.859434  5.844921  5.826341  5.803671  5.776886   5.74596  5.710868  5.671583  5.628081   5.58034  5.528344  5.472081  5.411549  5.346754  5.277717  5.204474  5.127079  5.045611  4.960175   4.87091   4.77799  4.681634  4.582109  4.479739  4.374908  4.268068  4.159741  4.050524        3.941088 |  551.63713  573.156838   593.72748  613.304912  631.848282  649.319747  665.684246  680.909326  694.964997  707.823625  719.459855  729.850561   738.97482  746.813909   753.35133   758.57285  762.466567  765.022999  766.235193  766.098857  764.612512  761.777671  757.599036  752.084721  745.246493  737.100039  727.665247  716.966511  705.033049  691.899234  677.604934  662.195854  645.723874  628.247376  609.831545  590.548639  570.478202  549.707214  528.330147  506.448909  484.172648  461.617388  438.905458  416.164693  393.527363  371.128812  349.105769  327.594337  306.727653  286.633239  267.430129 |  100.0 |  451.63713 | 267.430129 |
    | 07:30 | 7.889835  8.027052  8.155828    8.2764  8.388984  8.493771  8.590931  8.680615  8.762955  8.838068  8.906052  8.966995  9.020966  9.068026   9.10822  9.141583  9.168142   9.18791  9.200892  9.207087  9.206482   9.19906  9.184796  9.163658  9.135612  9.100618  9.058633  9.009613  8.953515  8.890294  8.819912  8.742333  8.657531  8.565489  8.466206  8.359695  8.245993  8.125164  7.997301  7.862539  7.721056  7.573083  7.418912  7.258905  7.093507  6.923247  6.748758  6.570779  6.390163             6.207883 | 429.595039  442.573638  454.925053  466.640014  477.710036  488.127275   497.88441  506.974558  515.391216  523.128203  530.179638   536.53991  542.203673  547.165847  551.421623  554.966486   557.79624  559.907048  561.295475  561.958549  561.893823   561.09946  559.574316  557.318049  554.331234  550.615493  546.173642  541.009856  535.129845  528.541057  521.252888   513.27692  504.627177  495.320391  485.376291  474.817913  463.671906  451.968865   439.74365  427.035708  413.889376  400.354153  386.484924   372.34212  357.991775  343.505464  328.960075  314.437381  300.023366  285.807265 |  93.898353   95.270524   96.558278   97.764003   98.889841   99.937708  100.909307  101.806145  102.629549  103.380676  104.060524  104.669949  105.209664  105.680258  106.082197  106.415834  106.681417  106.879095  107.008923  107.070869  107.064824  106.990603  106.847959  106.636584  106.356122  106.006178  105.586329  105.096133  104.535147  103.902941  103.199117  102.423329   101.57531  100.654895   99.662058   98.596948   97.459931   96.251635   94.973012   93.625393   92.210561   90.730827   89.189116   87.589054   85.935066   84.232472   82.487584   80.707792   78.901631     77.078825 | 1.651067   1.66761  1.683054  1.697444  1.710821  1.723221  1.734675  1.745211  1.754855  1.763626  1.771546  1.778628  1.784888  1.790336  1.794983  1.798835  1.801898  1.804176  1.805672  1.806385  1.806315  1.805461  1.803818  1.801381  1.798146  1.794105   1.78925  1.783572  1.777063  1.769712  1.761508  1.752442  1.742503   1.73168  1.719964  1.707347  1.693822  1.679384  1.664031  1.647765  1.630592  1.612525   1.59358  1.573786  1.553177  1.531802  1.509721  1.487008  1.463755  1.440073 | 1.340823  1.340593  1.340387  1.340201  1.340034  1.339884  1.339748  1.339627  1.339519  1.339422  1.339336  1.339261  1.339195  1.339139  1.339092  1.339053  1.339022  1.338999  1.338984  1.338977  1.338978  1.338986  1.339003  1.339027   1.33906  1.339101   1.33915  1.339209  1.339278  1.339356  1.339445  1.339546  1.339658  1.339783  1.339923  1.340077  1.340247  1.340435  1.340642  1.340869  1.341119  1.341394  1.341694  1.342024  1.342384  1.342778  1.343208  1.343676  1.344184          1.344735 | 0.722141  0.725622  0.728805  0.731716  0.734374  0.736799  0.739005  0.741008  0.742818  0.744446  0.745901   0.74719  0.748321  0.749298  0.750126  0.750809   0.75135  0.751751  0.752014  0.752139  0.752127  0.751977  0.751688  0.751259  0.750687   0.74997  0.749104  0.748084  0.746906  0.745565  0.744055  0.742367  0.740496  0.738431  0.736165  0.733688  0.730988  0.728054  0.724875  0.721438  0.717729  0.713736  0.709445  0.704844  0.699921  0.694666  0.689071  0.683133   0.67685      0.670231 |  -0.42229  -0.426087  -0.429551  -0.432707  -0.435576  -0.438177  -0.440526  -0.442638  -0.444523  -0.446192  -0.447655  -0.448919   -0.44999  -0.450873  -0.451574  -0.452094  -0.452436  -0.452603  -0.452594   -0.45241   -0.45205  -0.451514  -0.450798  -0.449899  -0.448816  -0.447542  -0.446074  -0.444405  -0.442529  -0.440439  -0.438128  -0.435586  -0.432805  -0.429774  -0.426482  -0.422918  -0.419071  -0.414928  -0.410476  -0.405703  -0.400597  -0.395147  -0.389342  -0.383173  -0.376637   -0.36973  -0.362456  -0.354825  -0.346855     -0.338573 | 0.695861  0.696451  0.696988  0.697476  0.697916  0.698311  0.698662   0.69897  0.699238  0.699465  0.699654  0.699804  0.699917  0.699992  0.700031  0.700033  0.699999  0.699929  0.699822  0.699679  0.699499  0.699282  0.699027  0.698735  0.698404  0.698034  0.697624  0.697174  0.696683   0.69615  0.695574  0.694955  0.694293  0.693585  0.692833  0.692035  0.691192  0.690303  0.689369  0.688391  0.687371  0.686309  0.685209  0.684075  0.682911  0.681722  0.680515  0.679299  0.678082      0.676876 | 518.029154 | 528.996258   550.53263  571.225767  591.027329  609.892427  627.779312  644.649117  660.465653  675.195245  688.806608  701.270744  712.560879  722.652419   731.52293  739.152144  745.521984  750.616613  754.422502  756.928522  758.126059   758.00914  756.574598  753.822245  749.755072  744.379473  737.705487  729.747064  720.522345  710.053969  698.369388  685.501197   671.48747  656.372108  640.205166  623.043186   604.94949  585.994454  566.255716  545.818328  524.774812  503.225103  481.276363  459.042615  436.644183  414.206905  391.861078  369.740113  347.978876   326.71171          306.070129 | 295.895863 | 1.108245  1.119541  1.130083  1.139904   1.14903  1.157488  1.165299  1.172483  1.179057  1.185036  1.190434   1.19526  1.199526  1.203238  1.206403  1.209028  1.211114  1.212666  1.213685   1.21417  1.214123  1.213541  1.212422  1.210762  1.208558  1.205805  1.202497  1.198629  1.194193  1.189184  1.183593  1.177413  1.170636  1.163256  1.155266   1.14666  1.137431  1.127578  1.117097   1.10599   1.09426  1.081914  1.068964  1.055427  1.041326  1.026694   1.01157  0.996004  0.980059       0.963807 | 5.089659  5.169673  5.244693  5.314875  5.380357  5.441261  5.497697  5.549761  5.597538  5.641101  5.680515  5.715833  5.747102  5.774358  5.797633  5.816949  5.832322  5.843764  5.851277  5.854862  5.854512  5.850217  5.841961  5.829727  5.813492  5.793231  5.768918  5.740525  5.708022   5.67138  5.630573  5.585574  5.536363  5.482922  5.425243  5.363326  5.297181  5.226835  5.152331  5.073732  4.991129  4.904639  4.814416  4.720652  4.623587  4.523511  4.420772  4.315783  4.209024        4.101047 | 518.029154  539.976699  561.105958  581.366789  600.712676   619.10038  636.489657  652.843032  668.125613  682.304953  695.350933  707.235686  717.933536  727.420971  735.676638  742.681356  748.418151   752.87232  756.031506  757.885805  758.427884  757.653126  755.559803  752.149257  747.426117  741.398528  734.078407  725.481718  715.628762  704.544493  692.258837  678.807032  664.229964  648.574514  631.893885  614.247921  595.703394  576.334248  556.221787   535.45478  514.129474  492.349479  470.225497  447.874872  425.420927  402.992052   380.72051  358.740947  337.188571  316.197013  295.895863 |  100.0 | 418.029154 | 295.895863 |
    | 08:00 | 7.675738  7.820077   7.95619  8.084295  8.204589  8.317251  8.422441  8.520304  8.610967  8.694542  8.771127  8.840807  8.903654  8.959726  9.009072  9.051729  9.087723  9.117073  9.139785   9.15586  9.165288  9.168054  9.164134    9.1535  9.136116  9.111942  9.080935  9.043049  8.998235  8.946443  8.887626  8.821737  8.748736  8.668587  8.581265  8.486757  8.385066  8.276212  8.160243  8.037233  7.907291   7.77057  7.627268  7.477642  7.322013  7.160777  6.994414  6.823495  6.648697             6.470806 | 409.720867  423.069176  435.847684  448.043584  459.645265  470.642085  481.024207   490.78246  499.908235  508.393405  516.230264  523.411479  529.930062  535.779344  540.952971  545.444904  549.249427  552.361168  554.775131   556.48673  557.491842  557.786866  557.368792  556.235284  554.384773  551.816567  548.530972  544.529428  539.814662  534.390856  528.263834  521.441268  513.932897  505.750773  496.909516  487.426594  477.322617  466.621638   455.35148  443.544047  431.235655  418.467333  405.285116  391.740292  377.889594  363.795313  349.525304  335.152847  320.756333  306.418728 |  91.757383    93.20077   94.561904   95.842953   97.045893    98.17251   99.224413   100.20304  101.109667  101.945417  102.711271  103.408073  104.036538   104.59726   105.09072  105.517288  105.877233  106.170727  106.397851  106.558597   106.65288  106.680539  106.641342  106.534997  106.361156   106.11942  105.809354  105.430492  104.982347   104.46443  103.876257  103.217369  102.487355  101.685867   100.81265   99.867572   98.850657   97.762123   96.602431   95.372328   94.072914   92.705699   91.272681   89.776421   88.220133   86.607772   84.944135    83.23495   81.486972     79.708058 | 1.625071  1.642622  1.659078  1.674485  1.688882  1.702305  1.714786  1.726353  1.737032  1.746844   1.75581  1.763946  1.771267  1.777784  1.783509  1.788451  1.792614  1.796006  1.798627  1.800482  1.801569  1.801888  1.801436   1.80021  1.798204  1.795413   1.79183  1.787446  1.782253  1.776241  1.769401  1.761721  1.753191    1.7438  1.733537  1.722393  1.710357  1.697422  1.683582  1.668834  1.653176  1.636614  1.619154  1.600812   1.58161  1.561578  1.540757  1.519201  1.496975  1.474163 | 1.341202  1.340943   1.34071    1.3405  1.340311   1.34014  1.339985  1.339846  1.339721  1.339609  1.339508  1.339419  1.339339   1.33927   1.33921  1.339159  1.339116  1.339081  1.339055  1.339036  1.339025  1.339022  1.339026  1.339039  1.339059  1.339087  1.339124  1.339169  1.339223  1.339286  1.339359  1.339443  1.339537  1.339643  1.339762  1.339894   1.34004  1.340201   1.34038  1.340576  1.340793  1.341031  1.341292  1.341578  1.341892  1.342235   1.34261   1.34302  1.343467          1.343953 | 0.716519  0.720336  0.723836  0.727047   0.72999  0.732687  0.735153  0.737405  0.739456  0.741316  0.742996  0.744505   0.74585  0.747037  0.748073   0.74896  0.749704  0.750308  0.750772    0.7511  0.751292  0.751348  0.751269  0.751052  0.750697  0.750202  0.749565   0.74878  0.747846  0.746757  0.745508  0.744094  0.742507  0.740741  0.738788  0.736638  0.734283  0.731711  0.728913  0.725877  0.722589  0.719039  0.715212  0.711096  0.706677  0.701944  0.696886  0.691494  0.685762      0.679688 | -0.415816  -0.420006  -0.423842  -0.427353  -0.430561  -0.433488   -0.43615  -0.438564  -0.440743  -0.442699  -0.444441  -0.445979   -0.44732  -0.448469  -0.449433  -0.450215   -0.45082  -0.451248  -0.451502  -0.451583  -0.451491  -0.451226  -0.450786   -0.45017  -0.449374  -0.448396  -0.447232  -0.445877  -0.444325  -0.442571  -0.440607  -0.438427  -0.436022  -0.433384  -0.430501  -0.427365  -0.423964  -0.420287  -0.416321  -0.412055  -0.407475  -0.402571  -0.397329  -0.391739  -0.385792  -0.379481  -0.372803  -0.365758  -0.358352     -0.350599 | 0.694755  0.695408  0.696006  0.696554  0.697053  0.697506  0.697914  0.698279  0.698603  0.698886   0.69913  0.699335  0.699503  0.699634  0.699728  0.699786  0.699808  0.699794  0.699744  0.699659  0.699537  0.699379  0.699185  0.698954  0.698685  0.698378  0.698033  0.697649  0.697225  0.696761  0.696256  0.695708  0.695118  0.694484  0.693807  0.693084  0.692317  0.691505  0.690647  0.689744  0.688797  0.687806  0.686774  0.685702  0.684594  0.683454  0.682286  0.681096  0.679893      0.678684 | 485.365238 | 496.439435  518.249068   539.34494   559.67325  579.183957    597.8304  615.568991  632.358972  648.162204  662.943015  676.668063   689.30625  700.828641  711.208428  720.420904  728.443467  735.255637  740.839103  745.177786   748.25792  750.068164   750.59972  749.846486  747.805223  744.475748  739.861142  733.967987   726.80662  718.391407  708.741035  697.878824  685.833052  672.637285  658.330718  642.958516  626.572145  609.229686  590.996117  571.943563  552.151477  531.706752  510.703726  489.244075  467.436543  445.396503  423.245297  401.109338  379.118934  357.406819          336.106362 | 325.557379 | 1.090488  1.102477  1.113716  1.124234  1.134061   1.14322  1.151734  1.159624  1.166906  1.173596  1.179709  1.185254  1.190243  1.194685  1.198586  1.201953   1.20479    1.2071  1.208886  1.210149   1.21089  1.211107  1.210799  1.209964  1.208598  1.206696  1.204255  1.201268   1.19773  1.193634  1.188972  1.183738  1.177923  1.171521  1.164523  1.156923  1.148713  1.139888  1.130444  1.120377  1.109686  1.098373  1.086444  1.073908  1.060778  1.047074  1.032825  1.018064  1.002836       0.987197 | 4.964652  5.048951  5.128362  5.203029  5.273083  5.338641  5.399807  5.456676  5.509331  5.557844  5.602279   5.64269  5.679125  5.711621  5.740211   5.76492  5.785765  5.802759  5.815908  5.825213   5.83067  5.832271  5.830002  5.823847  5.813783  5.799788  5.781834  5.759892  5.733933  5.703924  5.669833  5.631631  5.589289   5.54278  5.492084  5.437185  5.378078  5.314765  5.247264  5.175606  5.099842  5.020047  4.936322  4.848799  4.757649  4.663083  4.565362    4.4648  4.361772        4.256718 | 485.365238  507.519985  528.988731  549.715706  569.649104  588.740668  606.945351  624.221043  640.528345  655.830394  670.092714  683.283109  695.371577   706.33026  716.133402  724.757347  732.180546  738.383589  743.349261  747.062616  749.511067  750.684508  750.575447  749.179163  746.493889  742.521012  737.265297  730.735127  722.942776  713.904684  703.641765  692.179723  679.549383  665.787026  650.934734  635.040725   618.15968  600.353046  581.689304  562.244189  542.100842  521.349878  500.089341   478.42452  456.467611  434.337172  412.157354  390.056877  368.167711  346.623453  325.557379 |  100.0 | 385.365238 | 325.557379 |
    | 08:30 | 7.460295  7.610884  7.753521  7.888393  8.015673   8.13552  8.248082   8.35349  8.451864   8.54331  8.627921  8.705779  8.776953  8.841501  8.899472  8.950903  8.995823  9.034249  9.066192  9.091653  9.110625  9.123096  9.129043  9.128439  9.121251  9.107439   9.08696  9.059767  9.025809  8.985032  8.937385  8.882813  8.821265  8.752693  8.677057  8.594322  8.504464  8.407476  8.303366  8.192164  8.073926  7.948742  7.816737  7.678082  7.533001  7.381776   7.22476  7.062384  6.895165             6.723719 | 390.184427  403.791006  416.888256  429.459602  441.490121  452.966239  463.875493   474.20634  483.948013    493.0904  501.623956  509.539625  516.828791  523.483234  529.495101  534.856892  539.561453  543.601978  546.972024  549.665532  551.676862  553.000832  553.632773  553.568591  552.804842  551.338817  549.168644  546.293399  542.713235  538.429522  533.445009  527.763995  521.392529  514.338614  506.612441  498.226636  489.196525  479.540418  469.279905  458.440167  447.050291  435.143591  422.757922  409.935973  396.725543  383.179753  369.357211  355.322067  341.143956  326.897784 |  89.602949   91.108837    92.53521   93.883929   95.156726   96.355201   97.480818   98.534901   99.518643  100.433102  101.279214   102.05779  102.769528  103.415012  103.994721  104.509032  104.958225  105.342487  105.661916  105.916527  106.106254  106.230959   106.29043  106.284391  106.212508  106.074391  105.869603   105.59767  105.258086  104.850324  104.373848  103.828127  103.212647  102.526934   101.77057  100.943215  100.044643   99.074761   98.033658   96.921635    95.73926   94.487416   93.167368   91.780823   90.330011   88.817763   87.247603   85.623836   83.951647     82.237192 | 1.598678  1.617151  1.634542  1.650892  1.666241  1.680624  1.694071   1.70661  1.718268  1.729066  1.739025  1.748161  1.756491  1.764027   1.77078   1.77676  1.781973  1.786427  1.790124  1.793069  1.795261  1.796701  1.797388  1.797318  1.796488  1.794893  1.792526  1.789381  1.785449  1.780722  1.775189  1.768841  1.761666  1.753654  1.744794  1.735074  1.724484  1.713013  1.700654  1.687398  1.673241   1.65818  1.642217  1.625357  1.607612  1.588999  1.569544  1.549283  1.528261  1.506537 | 1.341612  1.341322  1.341061  1.340825  1.340612  1.340419  1.340244  1.340086  1.339943  1.339814  1.339698  1.339594    1.3395  1.339418  1.339345  1.339281  1.339226  1.339179  1.339141  1.339111  1.339089  1.339074  1.339067  1.339068  1.339076  1.339093  1.339117  1.339149   1.33919  1.339239  1.339298  1.339365  1.339443  1.339532  1.339632  1.339744  1.339869  1.340007  1.340161   1.34033  1.340517  1.340723  1.340949  1.341198   1.34147  1.341769  1.342096  1.342454  1.342845          1.343272 |  0.71061  0.714767  0.718589  0.722104  0.725337  0.728309  0.731038  0.733542  0.735835  0.737929  0.739835  0.741564  0.743123   0.74452  0.745761  0.746851  0.747796  0.748597   0.74926  0.749785  0.750175  0.750431  0.750553   0.75054  0.750393   0.75011  0.749689  0.749127  0.748422  0.747569  0.746566  0.745406  0.744084  0.742594  0.740929  0.739082  0.737044  0.734805  0.732358  0.729689   0.72679  0.723647  0.720248  0.716582  0.712633  0.708391  0.703842  0.698975   0.69378       0.68825 | -0.408997  -0.413583  -0.417794  -0.421663  -0.425212  -0.428466  -0.431444  -0.434162  -0.436635  -0.438876  -0.440897  -0.442707  -0.444315  -0.445728  -0.446951  -0.447991  -0.448851  -0.449535  -0.450045  -0.450383   -0.45055  -0.450547  -0.450372  -0.450025  -0.449504  -0.448807  -0.447931  -0.446871  -0.445625  -0.444185  -0.442548  -0.440706  -0.438653  -0.436379  -0.433878  -0.431139  -0.428153  -0.424909  -0.421395  -0.417601  -0.413513   -0.40912   -0.40441   -0.39937   -0.39399  -0.388258  -0.382169  -0.375715  -0.368895     -0.361714 |   0.6936  0.694313  0.694971  0.695577  0.696133  0.696642  0.697105  0.697525  0.697903  0.698241  0.698538  0.698797  0.699018  0.699203   0.69935  0.699462  0.699538  0.699579  0.699584  0.699554  0.699488  0.699387  0.699251  0.699078  0.698869  0.698624   0.69834   0.69802   0.69766  0.697262  0.696823  0.696344  0.695824  0.695262  0.694656  0.694008  0.693315  0.692578  0.691796  0.690968  0.690096  0.689179  0.688219  0.687216  0.686172  0.685091  0.683976   0.68283  0.681659      0.680471 |  453.86844 | 464.947556   486.82638  508.121027  528.772653  548.726477  567.931339  586.339343  603.905565  620.587817  636.346449  651.144195  664.946051  677.719174  689.432825   700.05831   709.56897  717.940172  725.149328   731.17594  736.001651  739.610331  741.988171  743.123803  743.008443  741.636049  739.003504  735.110817  729.961351  723.562066  715.923782   707.06147  696.994545   685.74719  673.348683  659.833732  645.242819   629.62253  613.025887  595.512643   577.14955  558.010582  538.177084  517.737835  496.789013  475.434009  453.783092  431.952872  410.065544  388.247872          366.629891 | 355.876475 | 1.072449  1.085076  1.096958  1.108126  1.118607  1.128425  1.137601  1.146157   1.15411  1.161474  1.168265  1.174494  1.180173  1.185309  1.189912  1.193987   1.19754  1.200574  1.203093  1.205099  1.206593  1.207574  1.208042  1.207994  1.207429  1.206342   1.20473  1.202587  1.199908  1.196687  1.192916   1.18859    1.1837  1.178239  1.172199  1.165571  1.158349  1.150525  1.142093  1.133048  1.123385  1.113102  1.102201  1.090683  1.078556  1.065831  1.052525   1.03866  1.024269       1.009389 | 4.838645  4.926743  5.010091  5.088817   5.16304  5.232867  5.298397  5.359718  5.416909  5.470041  5.519175  5.564365  5.605658  5.643092  5.676701  5.706509  5.732536  5.754795  5.773296   5.78804  5.799026  5.806246  5.809689  5.809339  5.805177  5.797181  5.785323  5.769575  5.749906  5.726284  5.698674  5.667043  5.631357  5.591585  5.547696  5.499666  5.447474  5.391107  5.330563  5.265849  5.196987  5.124018  5.047001  4.966021  4.881194  4.792666  4.700626  4.605306  4.506988        4.406014 |  453.86844  476.026352  497.630449  518.619751  538.937579   558.53107  577.350786   595.35039  612.486387  628.717904  644.006524  658.316141  671.612857  683.864895  695.042549  705.118148  714.066042  721.862615  728.486311  733.917684  738.139466  741.136658  742.896637  743.409286  742.667144  740.665579  737.402978  732.880961  727.104619  720.082764   711.82821  702.358061  691.694025  679.862735  666.896088  652.831578  637.712641  621.588988  604.516917  586.559602  567.787345   548.27776  528.115891  507.394231   486.21262  464.677993  442.903953  421.010133  399.121322  377.366315  355.876475 |  100.0 |  353.86844 | 355.876475 |
    | 09:00 | 7.244844  7.400779  7.549098  7.689948  7.823469  7.949797  8.069056  8.181365  8.286829  8.385545  8.477599  8.563067  8.642014  8.714497  8.780561  8.840243  8.893572  8.940566  8.981236  9.015586   9.04361  9.065297  9.080627  9.089574  9.092106  9.088187  9.077773  9.060816  9.037266  9.007069  8.970167  8.926503   8.87602  8.818661  8.754373  8.683107  8.604823  8.519489  8.427084  8.327607  8.221072  8.107521  7.987021  7.859677  7.725632   7.58508  7.438269   7.28551  7.127189             6.963774 |  371.11146  384.869291  398.180901   411.02574  423.385415  435.243291  446.584181  457.394101  467.660073   477.36997  486.512391  495.076561  503.052249   510.42971  517.199637  523.353125  528.881651  533.777061  538.031566  541.637752  544.588598  546.877499  548.498305  549.445366  549.713591   549.29851  548.196362  546.404178  543.919892  540.742454  536.871971  532.309848  527.058959  521.123827  514.510824   507.22839  499.287272  490.700773  481.485028  471.659291  461.246235  450.272264  438.767826  426.767734  414.311462  401.443433  388.213262  374.675943  360.891966  346.927321 |  87.448437   89.007785   90.490983   91.899482   93.234695   94.497967   95.690562   96.813646   97.868286   98.855447   99.775988  100.630667  101.420141  102.144967  102.805607   103.40243  103.935716  104.405657  104.812362  105.155861  105.436104   105.65297  105.806267  105.895738  105.921064  105.881869  105.777726  105.608161  105.372662  105.070686  104.701669  104.265032    103.7602   103.18661   102.54373  101.831075  101.048233  100.194887   99.270844   98.276069   97.210723   96.075206   94.870209   93.596766   92.256324   90.850804    89.38269   87.855101   86.271892     84.637736 |  1.57204  1.591344  1.609586  1.626803  1.643033  1.658307  1.672656  1.686107  1.698686  1.710414  1.721311  1.731394  1.740681  1.749183  1.756913   1.76388  1.770093  1.775559  1.780281  1.784265  1.787511  1.790021  1.791794  1.792828  1.793121  1.792668  1.791464  1.789502  1.786776  1.783277  1.778996  1.773924   1.76805  1.761362  1.753851  1.745504  1.736309  1.726257  1.715336  1.703536  1.690849   1.67727  1.662793  1.647419  1.631149  1.613994  1.595966  1.577087  1.557387  1.536905 | 1.342053  1.341731  1.341439  1.341176  1.340937  1.340721  1.340525  1.340347  1.340185  1.340039  1.339906  1.339787  1.339679  1.339582  1.339496  1.339419  1.339352  1.339294  1.339244  1.339202  1.339168  1.339142  1.339124  1.339114  1.339111  1.339115  1.339128  1.339148  1.339176  1.339212  1.339257  1.339311  1.339374  1.339447   1.33953  1.339624  1.339729  1.339847  1.339979  1.340124  1.340285  1.340463  1.340659  1.340874  1.341111  1.341371  1.341656  1.341968  1.342309          1.342682 | 0.704432  0.708931  0.713077  0.716899  0.720424  0.723674  0.726669  0.729427  0.731964  0.734294  0.736428  0.738377   0.74015  0.741756  0.743202  0.744493  0.745635  0.746633   0.74749  0.748209  0.748792  0.749241  0.749558  0.749743  0.749795  0.749714  0.749499  0.749149   0.74866  0.748031  0.747257  0.746335   0.74526  0.744028  0.742631  0.741063  0.739318  0.737387  0.735261  0.732932  0.730388   0.72762  0.724616  0.721364  0.717851  0.714064  0.709991   0.70562  0.700938      0.695935 | -0.401855  -0.406837  -0.411425  -0.415653  -0.419546  -0.423128  -0.426422  -0.429444  -0.432212  -0.434739  -0.437037  -0.439118  -0.440991  -0.442664  -0.444145  -0.445438   -0.44655  -0.447484  -0.448244  -0.448831  -0.449249  -0.449498  -0.449579  -0.449491  -0.449233  -0.448804  -0.448201  -0.447423  -0.446464  -0.445322  -0.443991  -0.442466  -0.440741  -0.438809  -0.436663  -0.434294  -0.431693  -0.428851  -0.425757  -0.422402  -0.418772  -0.414857  -0.410644  -0.406122  -0.401278    -0.3961  -0.390579  -0.384704   -0.37847     -0.371873 | 0.692403  0.693173  0.693888  0.694549  0.695161  0.695724  0.696241  0.696714  0.697145  0.697534  0.697884  0.698195  0.698468  0.698704  0.698903  0.699067  0.699195  0.699288  0.699346  0.699369  0.699358  0.699312  0.699231  0.699114  0.698963  0.698775  0.698551  0.698291  0.697993  0.697658  0.697284   0.69687  0.696417  0.695923  0.695388   0.69481   0.69419  0.693526  0.692818  0.692065  0.691268  0.690426  0.689539  0.688609  0.687635   0.68662  0.685566  0.684476  0.683354      0.682204 | 423.717566 | 434.708461  456.469778  477.775127  498.561091  518.768599  538.342433  557.230822  575.385107  592.759469  609.310703  624.998038  639.782987  653.629231  666.502527  678.370644  689.203318  698.972231  707.651007  715.215231  721.642484  726.912398  731.006731  733.909462    735.6069  736.087822  735.343621  733.368485  730.159588  725.717307  720.045455  713.151544  705.047051  695.747719  685.273864  673.650698  660.908662  647.083768  632.217931  616.359309  599.562609  581.889377  563.408245  544.195119  524.333294  503.913472  483.033663  461.798935  440.320992  418.717547          397.111453 | 386.319679 | 1.054232  1.067435  1.079905  1.091671  1.102758  1.113189  1.122986  1.132167  1.140751  1.148752  1.156185  1.163062  1.169394  1.175191   1.18046  1.185209  1.189444  1.193168  1.196387  1.199101  1.201313  1.203023  1.204231  1.204935  1.205135  1.204826  1.204006  1.202669  1.200812  1.198428  1.195511  1.192054  1.188051  1.183493  1.178373  1.172682  1.166413  1.159558  1.152109   1.14406  1.135403  1.126135  1.116252  1.105754  1.094641  1.082918  1.070595  1.057684  1.044207       1.030188 | 4.712406  4.803796  4.890611  4.972955  5.050931  5.124633   5.19415  5.259562  5.320943  5.378357  5.431864  5.481516  5.527356  5.569424  5.607751  5.642363   5.67328  5.700518  5.724085  5.743985  5.760218  5.772778  5.781655  5.786836  5.788303  5.786033  5.780002  5.770183  5.756543  5.739051  5.717671  5.692368  5.663105  5.629848  5.592559  5.551208  5.505764  5.456202  5.402506  5.344664  5.282677  5.216559  5.146337  5.072062  4.993802  4.911655   4.82575  4.736252  4.643366        4.547347 | 423.717566  445.692732  467.244659  488.307632  508.820556   528.72641  547.971801  566.506595  584.283621  601.258423  617.389061   632.63594  646.961688  660.331044  672.710787   684.06968  694.378435  703.609704  711.738085  718.740149  724.594485  729.281766  732.784831  735.088793  736.181153  736.051953  734.693933  732.102716  728.277015  723.218858  716.933832  709.431351  700.724941   690.83254  679.776819  667.585507  654.291731  639.934358  624.558326  608.214976  590.962351  572.865469  553.996545  534.435145  514.268265  493.590292  472.502842  451.114431   429.53996  407.899975  386.319679 |  100.0 | 323.717566 | 386.319679 |
    | 09:30 | 7.030648  7.190997  7.344133  7.490152  7.629154   7.76124  7.886511  8.005064  8.116988  8.222366  8.321274   8.41378  8.499943  8.579815  8.653437  8.720845  8.782066  8.837117  8.886011   8.92875  8.965331  8.995743  9.019969  9.037983  9.049757  9.055254  9.054433  9.047247  9.033646  9.013575  8.986975  8.953788  8.913951  8.867403  8.814083  8.753932  8.686896  8.612928  8.531987  8.444044  8.349086  8.247115  8.138156   8.02226  7.899512  7.770029  7.633978  7.491574  7.343091             7.188874 | 352.609816  366.417127  379.843479  392.864181  405.457231  417.602827  429.282979    440.4812  451.182264  461.372001  471.037147  480.165204  488.744347  496.763331  504.211433  511.078393  517.354381  523.029969  528.096111  532.544141  536.365772  539.553114  542.098689  543.995465  545.236898  545.816981  545.730306  544.972137  543.538494   541.42625  538.633243    535.1584  531.001877  526.165213  520.651505  514.465595  507.614282  500.106541  491.953771  483.170054  473.772435  463.781209  453.220231  442.117221  430.504083  418.417208  405.897769   392.99198  379.751321  366.232683 |  85.306485   86.909968   88.441334   89.901522   91.291536   92.612399   93.865114   95.050639   96.169877    97.22366   98.212743   99.137803   99.999435  100.798148  101.534372  102.208453  102.820658  103.371173  103.860109  104.287501  104.653311  104.957431  105.199685  105.379833  105.497573  105.552544  105.544333  105.472474   105.33646  105.135746  104.869752  104.537878   104.13951  103.674029  103.140827  102.539319  101.868963  101.129278  100.319865    99.44044   98.490856   97.471145   96.381557   95.222605   93.995115   92.700292    91.33978   89.915736   88.430913     86.888743 | 1.545305  1.565343  1.584347   1.60235  1.619384   1.63548  1.650665  1.664965  1.678404  1.691004  1.702783   1.71376   1.72395  1.733367  1.742022  1.749926  1.757089  1.763516  1.769213  1.774185  1.778435  1.781964  1.784772  1.786859  1.788222  1.788859  1.788764  1.787932  1.786357  1.784031  1.780947  1.777095  1.772464  1.767046  1.760828  1.753799  1.745948  1.737262  1.727731  1.717343  1.706087  1.693955  1.680939  1.667034  1.652236  1.636548  1.619974  1.602525  1.584218  1.565079 | 1.342527  1.342169  1.341846  1.341553  1.341288  1.341047  1.340828  1.340629  1.340448  1.340283  1.340134  1.339998  1.339875  1.339764  1.339664  1.339574  1.339494  1.339423  1.339361  1.339308  1.339263  1.339226  1.339197  1.339175  1.339161  1.339154  1.339155  1.339164   1.33918  1.339204  1.339237  1.339277  1.339327  1.339385  1.339453   1.33953  1.339619  1.339718   1.33983  1.339954  1.340093  1.340246  1.340415  1.340601  1.340806  1.341032  1.341279  1.341551  1.341848          1.342174 | 0.698004  0.702844  0.707314  0.711445  0.715263  0.718793  0.722056   0.72507  0.727853   0.73042  0.732782  0.734952   0.73694  0.738755  0.740405  0.741896  0.743234  0.744426  0.745474  0.746383  0.747155  0.747794    0.7483  0.748675   0.74892  0.749034  0.749017  0.748868  0.748585  0.748167   0.74761  0.746912  0.746069  0.745076  0.743929  0.742621  0.741147    0.7395  0.737671  0.735654  0.733438  0.731015  0.728373  0.725502   0.72239  0.719025  0.715394  0.711484  0.707284      0.702781 | -0.394414   -0.39979  -0.404755  -0.409341  -0.413577   -0.41749  -0.421099  -0.424427  -0.427489    -0.4303  -0.432876  -0.435227  -0.437363  -0.439295  -0.441029  -0.442573  -0.443932  -0.445111  -0.446115  -0.446947  -0.447609  -0.448103   -0.44843  -0.448591  -0.448586  -0.448413  -0.448072  -0.447561  -0.446876  -0.446014  -0.444973  -0.443746   -0.44233  -0.440717  -0.438902  -0.436878  -0.434636  -0.432168  -0.429464  -0.426516  -0.423312  -0.419842  -0.416093  -0.412054  -0.407713  -0.403058  -0.398077  -0.392759  -0.387094     -0.381075 |  0.69117  0.691995  0.692764  0.693479  0.694143  0.694758  0.695327  0.695852  0.696333  0.696773  0.697173  0.697534  0.697857  0.698142  0.698392  0.698606  0.698784  0.698928  0.699037  0.699111  0.699152  0.699158   0.69913  0.699068  0.698971  0.698839  0.698672   0.69847  0.698231  0.697956  0.697643  0.697293  0.696904  0.696475  0.696007  0.695498  0.694947  0.694354  0.693718  0.693039  0.692315  0.691547  0.690734  0.689877  0.688976  0.688032  0.687046  0.686019  0.684956      0.683858 | 395.049792 | 405.868877  427.344437  448.489494  469.236568  489.522794  509.289365  528.481087  547.046007   564.93511  582.102064  598.503016  614.096419  628.842897  642.705134  655.647791  667.637441  678.642533  688.633365  697.582088  705.462717  712.251169  717.925312  722.465038  725.852352  728.071478  729.108985  728.953937  727.598057  725.035912   721.26512  716.286581  710.104719   702.72775  694.167971  684.442056  673.571376  661.582328  648.506672  634.381865  619.251403  603.165143    586.1796  568.358223   549.77161   530.49767   510.62169  490.236308  469.441342  448.343473          427.055729 | 416.381526 | 1.035938  1.049651   1.06265  1.074959  1.086602  1.097599  1.107971  1.117735   1.12691  1.135509  1.143546  1.151035  1.157985  1.164407  1.170309  1.175698   1.18058  1.184961  1.188844  1.192233  1.195129  1.197533  1.199447  1.200869  1.201797  1.202231  1.202166    1.2016  1.200526  1.198942   1.19684  1.194215   1.19106  1.187367  1.183129  1.178338  1.172985  1.167063  1.160564  1.153478    1.1458  1.137523   1.12864  1.119148  1.109044  1.098328  1.087004  1.075078  1.062561        1.04947 | 4.586659  4.680818  4.770612  4.856121  4.937424  5.014599   5.08772  5.156856  5.222073   5.28343  5.340981  5.394773  5.444847  5.491242  5.533987  5.573107  5.608624   5.64055  5.668898   5.69367  5.714869  5.732489  5.746523  5.756958  5.763778  5.766962  5.766486  5.762324  5.754446  5.742819   5.72741   5.70818  5.685093   5.65811  5.627193  5.592303  5.553407  5.510469  5.463463  5.412363  5.357156  5.297834  5.234402  5.166879  5.095304  5.019731  4.940244  4.856953  4.770001        4.679572 | 395.049792  416.675449  438.005432  458.969823  479.503612  499.546099  519.040394  537.933015  556.173553  573.714397  590.510507  606.519224  621.700121   636.01488  649.427193  661.902691  673.408895  683.915181  693.392773  701.814747  709.156058  715.393582  720.506178  724.474767  727.282431  728.914532  729.358841  728.605705  726.648213  723.482398  719.107454   713.52597  706.744189  698.772285  689.624655   679.32023  667.882798   655.34134  641.730366  627.090255  611.467587  594.915462  577.493783   559.26951  540.316851  520.717384  500.560084  479.941229  458.964169  437.738914  416.381526 |  100.0 | 295.049792 | 416.381526 |
    | 10:00 | 6.818894  6.982699  7.139767  7.290128  7.433834  7.570947  7.701535  7.825667  7.943414  8.054841  8.160011   8.25898  8.351799  8.438512  8.519157  8.593765   8.66236  8.724958   8.78157    8.8322  8.876843   8.91549  8.948124  8.974723  8.995257  9.009693   9.01799  9.020103  9.015983  9.005576  8.988823  8.965664  8.936035  8.899871  8.857107  8.807678   8.75152  8.688574  8.618785  8.542108  8.458503  8.367947  8.270432  8.165968  8.054591  7.936364  7.811386  7.679796  7.541779             7.397577 | 334.769979  348.530953  361.977874  375.081784  387.816976  400.160402  412.091209  423.590365  434.640363  445.224978  455.329076  464.938448  474.039687  482.620078  490.667517  498.170436  505.117754  511.498835  517.303455  522.521784  527.144376  531.162168  534.566488  537.349073  539.502096  541.018198   541.89054  542.112854  541.679512  540.585601  538.827019  536.400571  533.304092  529.536574  525.098316  519.991082  514.218285  507.785183  500.699091  492.969616  484.608908  475.631925  466.056714  455.904713  445.201049  433.974853  422.259566  410.093241  397.518812  384.584345 |  83.188939   84.826995   86.397671   87.901284   89.338343   90.709471   92.015348   93.256674   94.434142   95.548415   96.600111   97.589799   98.517987   99.385117  100.191568  100.937648  101.623595   102.24958  102.815705     103.322  103.768432  104.154901  104.481241  104.747227   104.95257  105.096927  105.179897  105.201032  105.159832  105.055757  104.888228  104.656636  104.360346  103.998709  103.571069  103.076775  102.515197  101.885738  101.187854  100.421075    99.58503   98.679474   97.704321   96.659683    95.54591   94.363643   93.113862   91.797956   90.417787     88.975766 | 1.518618  1.539285  1.558957  1.577659   1.59542  1.612263  1.628215  1.643299  1.657537  1.670949  1.683554  1.695369  1.706409  1.716688  1.726218  1.735008  1.743069  1.750408  1.757031  1.762942  1.768146  1.772643  1.776437  1.779525  1.781908  1.783581  1.784543  1.784788  1.784311  1.783104  1.781161  1.778474  1.775032  1.770826  1.765846   1.76008  1.753517  1.746145  1.737951  1.728924  1.719053  1.708326  1.696734  1.684267  1.670919  1.656687  1.641568  1.625566  1.608688  1.590949 | 1.343032  1.342638  1.342281  1.341958  1.341664  1.341398  1.341155  1.340933  1.340732  1.340548   1.34038  1.340227  1.340088  1.339962  1.339848  1.339745  1.339652  1.339568  1.339494  1.339429  1.339373  1.339325  1.339284  1.339252  1.339227  1.339209  1.339199  1.339197  1.339201  1.339214  1.339234  1.339263  1.339299  1.339344  1.339398  1.339461  1.339534  1.339617   1.33971  1.339816  1.339934  1.340065   1.34021  1.340371  1.340548  1.340744  1.340958  1.341194  1.341453          1.341737 | 0.691346  0.696523  0.701316  0.705754  0.709866  0.713677  0.717209  0.720481  0.723512  0.726316  0.728907  0.731299  0.733502  0.735526  0.737379  0.739069  0.740603  0.741986  0.743224   0.74432  0.745278  0.746102  0.746793  0.747353  0.747784  0.748086  0.748259  0.748303  0.748217     0.748  0.747649  0.747162  0.746537  0.745769  0.744855   0.74379  0.742568  0.741184  0.739631  0.737902  0.735988  0.733882  0.731573  0.729053  0.726309  0.723332  0.720109  0.716627  0.712876       0.70884 | -0.386701  -0.392466  -0.397803  -0.402746  -0.407325  -0.411566  -0.415492  -0.419123  -0.422479  -0.425576  -0.428427  -0.431047  -0.433445  -0.435633  -0.437619   -0.43941  -0.441013  -0.442434  -0.443677  -0.444747  -0.445647   -0.44638  -0.446946  -0.447348  -0.447586  -0.447661   -0.44757  -0.447313  -0.446889  -0.446294  -0.445526  -0.444581  -0.443454  -0.442142  -0.440637  -0.438935  -0.437028  -0.434909  -0.432568  -0.429999  -0.427191  -0.424133  -0.420816  -0.417227  -0.413356   -0.40919  -0.404718  -0.399927  -0.394806     -0.389345 |  0.68991  0.690786  0.691606  0.692371  0.693085   0.69375  0.694369  0.694942  0.695473  0.695961  0.696409  0.696819   0.69719  0.697523  0.697821  0.698083   0.69831  0.698502   0.69866  0.698784  0.698875  0.698932  0.698955  0.698944    0.6989  0.698821  0.698709  0.698562  0.698379  0.698162  0.697908  0.697617   0.69729  0.696924   0.69652  0.696076  0.695592  0.695067  0.694501  0.693892  0.693241  0.692545  0.691806  0.691022  0.690195  0.689323  0.688408   0.68745  0.686451      0.685414 | 367.963873 | 378.537392  399.578015    420.4096  440.961174  461.166598  480.963975  500.295162  519.105367  537.342819   554.95849  571.905868  588.140766  603.621169  618.307106  632.160548  645.145333  657.227108   668.37329  678.553052  687.737313  695.898764  703.011891  709.053036  714.000452  717.834398  720.537238  722.093559  722.490317   721.71699  719.765759   716.63171  712.313046  706.811329  700.131738  692.283344  683.279405  673.137676  661.880731  649.536297   636.13759  621.723655  606.339697   590.03739   572.87517  554.918481  536.239965  516.919583  497.044644  476.709711          456.016366 | 445.601377 | 1.017665  1.031817  1.045281  1.058076  1.070221  1.081735  1.092636   1.10294  1.112664  1.121821  1.130425  1.138488   1.14602  1.153032  1.159532  1.165526  1.171023  1.176026  1.180541   1.18457  1.188116  1.191182  1.193767  1.195871  1.197495  1.198635  1.199291  1.199457  1.199132   1.19831  1.196986  1.195155   1.19281  1.189943  1.186549  1.182619  1.178145  1.173119  1.167533  1.161378  1.154645  1.147328  1.139419  1.130911    1.1218  1.112083  1.101757  1.090826  1.079292       1.067164 |  4.46209  4.558475   4.65075  4.738959  4.823154   4.90339  4.979725  5.052214  5.120911  5.185868  5.247129  5.304739  5.358734  5.409148   5.45601  5.499343  5.539166  5.575494  5.608336  5.637699  5.663583  5.685985  5.704898  5.720311  5.732208  5.740571  5.745377  5.746601  5.744215  5.738186   5.72848  5.715061  5.697892  5.676932   5.65214  5.623478  5.590904   5.55438   5.51387  5.469342  5.420767  5.368124  5.311402  5.250598  5.185721  5.116799  5.043877  4.967022  4.886329        4.801921 | 367.963873  389.092965  410.049659   430.76042  451.156867  471.175126  490.755297  509.841011  528.379068  546.319137  563.613505   580.21687  596.086168  611.180438  625.460704  638.889891  651.432753   663.05583  673.727421  683.417571  692.098077  699.742517   706.32629  711.826673  716.222897  719.496241  721.630148  722.610346  722.425006  721.064908  718.523625  714.797739  709.887063  703.794894  696.528275  688.098286  678.520344   667.81452  656.005866  643.124756  629.207223  614.295294  598.437318  581.688267  564.110008  545.771529  526.749097  507.126342  486.994228  466.450901  445.601377 |  100.0 | 267.963873 | 445.601377 |
    | 10:30 | 6.610682   6.77697  6.937067  7.090931  7.238555  7.379952  7.515153  7.644195  7.767124  7.883984  7.994819  8.099674  8.198588  8.291595  8.378728  8.460011  8.535464  8.605101   8.66893  8.726953  8.779166  8.825559  8.866117  8.900816  8.929631  8.952528  8.969468  8.980408  8.985299  8.984089   8.97672  8.963131  8.943259  8.917037  8.884396  8.845269  8.799585  8.747278  8.688283   8.62254  8.549995  8.470604  8.384333  8.291163  8.191093  8.084142  7.970358  7.849818  7.722638             7.588977 |  317.66581  331.291187  344.670508  357.770495  370.561682  383.017729  395.114886  406.831554  418.147931  429.045733  439.507959  449.518704     459.063   468.12669  476.696324  484.759072  492.302655  499.315289  505.785641  511.702797  517.056243  521.835847  526.031857   529.63491  532.636038  535.026699  536.798802  537.944752  538.457499  538.330598  537.558286   536.13556  534.058276  531.323257   527.92842  523.872908  519.157247  513.783515   507.75553  501.079053   493.76201  485.814733  477.250213  468.084378  458.336376  448.028875   437.18837  425.845486  414.035285  401.797551 |   81.10682   82.769703   84.370671   85.909312   87.385546   88.799519   90.151526   91.441955   92.671239   93.839835   94.948192    95.99674   96.985875   97.915953   98.787279   99.600108  100.354637  101.051006  101.689295  102.269526  102.791658   103.25559  103.661165  104.008164  104.296312  104.525279   104.69468  104.804079  104.852992  104.840889  104.767198  104.631311  104.432588  104.170366  103.843962  103.452686  102.995849  102.472778  101.882828  101.225397  100.499951   99.706041   98.843332   97.911631   96.910926   95.841419   94.703576    93.49818   92.226383     90.889774 | 1.492117  1.513303  1.533544  1.552855  1.571258  1.588773  1.605421  1.621222  1.636195  1.650359  1.663732  1.676329  1.688165  1.699253  1.709605  1.719231  1.728141  1.736342   1.74384  1.750642   1.75675  1.762167  1.766896  1.770936  1.774288  1.776948  1.778915  1.780185  1.780753  1.780612  1.779757   1.77818  1.775871  1.772823  1.769025  1.764466  1.759135  1.753021  1.746111  1.738392  1.729854  1.720484   1.71027  1.699202   1.68727  1.674466  1.660786  1.646225  1.630785  1.614471 | 1.343568  1.343136  1.342745   1.34239  1.342067  1.341773  1.341505   1.34126  1.341037  1.340833  1.340646  1.340476   1.34032  1.340178  1.340049  1.339931  1.339825  1.339729  1.339643  1.339566  1.339498  1.339438  1.339386  1.339343  1.339307  1.339279  1.339258  1.339245  1.339239   1.33924  1.339249  1.339266   1.33929  1.339323  1.339363  1.339413  1.339471  1.339539  1.339617  1.339705  1.339805  1.339916  1.340041  1.340179  1.340332  1.340501  1.340687  1.340891  1.341116          1.341363 | 0.684485  0.689991    0.6951  0.699843  0.704248  0.708339  0.712139   0.71567  0.718948  0.721991  0.724813  0.727427  0.729845  0.732078  0.734134  0.736023  0.737751  0.739324  0.740749   0.74203  0.743171  0.744176  0.745048   0.74579  0.746402  0.746886  0.747242  0.747472  0.747575   0.74755  0.747395  0.747109   0.74669  0.746134  0.745439  0.744601  0.743615  0.742475  0.741178  0.739715  0.738081  0.736267  0.734265  0.732067  0.729663  0.727043  0.724195  0.721108  0.717771       0.71417 | -0.378745   -0.38489  -0.390594   -0.39589  -0.400808  -0.405375  -0.409615   -0.41355  -0.417199  -0.420579  -0.423705  -0.426592  -0.429251  -0.431693  -0.433928  -0.435964  -0.437808  -0.439467  -0.440945  -0.442249  -0.443382  -0.444347  -0.445146  -0.445782  -0.446255  -0.446567  -0.446717  -0.446705  -0.446529  -0.446188   -0.44568  -0.445002  -0.444149  -0.443119  -0.441907  -0.440507  -0.438913  -0.437119  -0.435117    -0.4329  -0.430459  -0.427786  -0.424869  -0.421699  -0.418265  -0.414555  -0.410559  -0.406262  -0.401655     -0.396726 | 0.688631  0.689553   0.69042  0.691233  0.691994  0.692706  0.693372  0.693992  0.694569  0.695104  0.695599  0.696054  0.696472  0.696852  0.697196  0.697504  0.697778  0.698017  0.698222  0.698394  0.698532  0.698637  0.698709  0.698748  0.698754  0.698727  0.698666  0.698572  0.698444  0.698281  0.698083   0.69785  0.697581  0.697275  0.696932  0.696551  0.696131  0.695672  0.695173  0.694632   0.69405  0.693425  0.692757  0.692046  0.691291  0.690492  0.689649  0.688762  0.687833      0.686862 | 342.523746 | 352.787857   373.26364  393.646724   413.86326  433.844397  453.525667  472.846465  491.749627  510.181072  528.089508  545.426186  562.144694  578.200786  593.552243  608.158756  621.981836  634.984744  647.132439  658.391545  668.730334  678.118725  686.528304  693.932352  700.305894  705.625766  709.870694  713.021392  715.060678  715.973612   715.74764  714.372771  711.841769  708.150356   703.29745  697.285407  690.120295  681.812174  672.375403  661.828954  650.196741  637.507957  623.797408  609.105852  593.480319  576.974417  559.648597  541.570377  522.814505  503.463032          483.605293 | 473.574309 | 0.999507  1.014024  1.027886  1.041106  1.053698  1.065677  1.077058  1.087857  1.098087  1.107762  1.116893  1.125493  1.133571  1.141138    1.1482  1.154767  1.160843  1.166436  1.171548  1.176185  1.180349  1.184042  1.187265  1.190018  1.192302  1.194115  1.195456  1.196321  1.196708  1.196612  1.196029  1.194954  1.193382  1.191304  1.188716  1.185609  1.181975  1.177807  1.173096  1.167834  1.162012  1.155621  1.148654  1.141103   1.13296  1.124222  1.114882  1.104938  1.094392       1.083244 | 4.339339  4.437396  4.531641  4.622075  4.708717  4.791598  4.870751  4.946217  5.018035  5.086245  5.150884  5.211988  5.269589  5.323716  5.374394  5.421643  5.465483  5.505925  5.542979  5.576651  5.606942  5.633848  5.657364   5.67748  5.694181   5.70745  5.717266  5.723605  5.726439  5.725737  5.721468  5.713594  5.702078  5.686881  5.667961  5.645277  5.618785  5.588443  5.554211   5.51605  5.473923  5.427799  5.377652  5.323464  5.265225  5.202939  5.136622  5.066309  4.992053        4.913934 | 342.523746  363.029072  383.479832  403.799517   423.91696  443.765653  463.283184  482.410773  501.092886  519.276907  536.912879  553.953272  570.352797  586.068258  601.058419  615.283903  628.707112  641.292169  653.004873  663.812673  673.684663  682.591591  690.505877  697.401661  703.254854  708.043211  711.746424  714.346226  715.826515    716.1735   715.37586  713.424925  710.314878  706.042971   700.60977  694.019409  686.279868  677.403271   667.40619  656.309973  644.141076    630.9314   616.71863  601.546566  585.465438  568.532198  550.810766  532.372229  513.294965  493.664677  473.574309 |  100.0 | 242.523746 | 473.574309 |
    | 11:00 | 6.407028  6.574813  6.737027  6.893544   7.04429  7.189224  7.328329  7.461609  7.589075  7.710748  7.826654  7.936818  8.041266  8.140023   8.23311  8.320546  8.402343  8.478513   8.54906  8.613983  8.673278  8.726933  8.774932  8.817255  8.853874  8.884757  8.909867  8.929162  8.942595  8.950114  8.951663  8.947183   8.93661  8.919876  8.896914  8.867651  8.832016  8.789936  8.741339  8.686158  8.624327  8.555787  8.480484  8.398379  8.309442   8.21366   8.11104  8.001613   7.88544             7.762612 |  301.35547  314.763031   327.99305  341.007944   353.77449  366.263059  378.446998  390.302138  401.806392  412.939423  423.682382  434.017683  443.928826  453.400244  462.417179  470.965583  479.032027  486.603637  493.668036  500.213299  506.227922  511.700794  516.621189  520.978752  524.763505  527.965861  530.576636  532.587083  533.988927  534.774408  534.936343   534.46819  533.364121  531.619122  529.229086   526.19093  522.502732  518.163865  513.175168  507.539114  501.260012  494.344212  486.800341  478.639543  469.875744  460.525934   450.61045  440.153284  429.182383  417.729951 |  79.070281   80.748131   82.370268   83.935445   85.442904   86.892239   88.283294   89.616085   90.890747   92.107483   93.266541   94.368181   95.412664   96.400233   97.331103   98.205456   99.023434   99.785133    100.4906  101.139834   101.73278   102.26933  102.749325   103.17255  103.538737  103.847567  104.098668  104.291618  104.425948   104.50114  104.516634  104.471833  104.366098  104.198764  103.969138  103.676509  103.320156  102.899355  102.413394  101.861584  101.243273  100.557865   99.804844   98.983791   98.094418   97.136598     96.1104   95.016134   93.854396     92.626125 | 1.465935  1.487525   1.50823  1.528056  1.547016  1.565122  1.582391  1.598839  1.614483  1.629337  1.643419  1.656741  1.669319  1.681163  1.692285  1.702696  1.712405  1.721419  1.729744  1.737387  1.744351  1.750639  1.756255  1.761198  1.765469  1.769067  1.771989  1.774233  1.775794  1.776668  1.776848  1.776327  1.775099  1.773154  1.770482  1.767075  1.762921  1.758008  1.752326  1.745861  1.738603  1.730537  1.721652  1.711935  1.701376  1.689965  1.677691   1.66455  1.650535  1.635647 | 1.344135  1.343665  1.343238  1.342849  1.342495  1.342173  1.341878  1.341609  1.341363  1.341138  1.340932  1.340743   1.34057  1.340412  1.340267  1.340135  1.340015  1.339905  1.339806  1.339717  1.339637  1.339566  1.339503  1.339449  1.339402  1.339363  1.339332  1.339308  1.339291  1.339282   1.33928  1.339285  1.339298  1.339319  1.339348  1.339385   1.33943  1.339484  1.339547   1.33962  1.339703  1.339797  1.339902   1.34002  1.340151  1.340297  1.340458  1.340635   1.34083          1.341045 | 0.677448   0.68327  0.688687  0.693729  0.698422  0.702791  0.706859  0.710647  0.714173  0.717455  0.720507  0.723344  0.725977  0.728419  0.730678  0.732765  0.734686  0.736449  0.738059  0.739523  0.740845  0.742029  0.743079  0.743997  0.744786  0.745447  0.745982  0.746392  0.746676  0.746835  0.746867  0.746773  0.746549  0.746195  0.745706  0.745081  0.744316  0.743405  0.742345  0.741131  0.739755  0.738212  0.736494  0.734593  0.732502   0.73021  0.727707  0.724984  0.722028      0.718829 | -0.370581  -0.377092  -0.383152  -0.388793  -0.394045  -0.398935  -0.403486  -0.407722  -0.411662  -0.415324  -0.418724  -0.421876  -0.424794  -0.427489   -0.42997  -0.432248   -0.43433  -0.436224  -0.437935  -0.439468  -0.440829  -0.442021  -0.443047   -0.44391  -0.444612  -0.445153  -0.445535  -0.445758  -0.445821  -0.445723  -0.445462  -0.445037  -0.444445  -0.443682  -0.442744  -0.441628  -0.440328  -0.438839  -0.437153  -0.435265  -0.433167   -0.43085  -0.428306  -0.425525  -0.422497  -0.419212  -0.415658  -0.411824  -0.407698     -0.403269 | 0.687339  0.688304  0.689213   0.69007  0.690875  0.691632  0.692341  0.693006  0.693627  0.694207  0.694746  0.695245  0.695707  0.696132   0.69652  0.696873  0.697192  0.697476  0.697726  0.697943  0.698128  0.698279  0.698399  0.698485   0.69854  0.698561   0.69855  0.698507   0.69843  0.698319  0.698175  0.697997  0.697784  0.697535   0.69725  0.696929  0.696571  0.696174  0.695738  0.695263  0.694748  0.694191  0.693593  0.692952  0.692268  0.691541   0.69077  0.689955  0.689097      0.688196 | 318.762374 | 328.663042  348.463205  368.280889  388.040029   407.66966  427.103282  446.278334  465.135749  483.619587  501.676722  519.256585  536.310946  552.793729  568.660861  583.870145  598.381162  612.155183  625.155108  637.345424  648.692171   659.16293  668.726827  677.354547  685.018365  691.692196  697.351655  701.974137  705.538905  708.027211  709.422415  709.710136  708.878413  706.917891  703.822021  699.587281   694.21342   687.70371  680.065232  671.309163  661.451093   650.51134  638.515287  625.493717  611.483152  596.526179  580.671766  563.975548  546.500078  528.315018          509.497263 | 499.956953 | 0.981554  0.996359  1.010548  1.024129  1.037109    1.0495  1.061312  1.072559  1.083252  1.093403  1.103022   1.11212  1.120707  1.128792  1.136383  1.143487   1.15011  1.156259  1.161936  1.167148  1.171896  1.176184  1.180012  1.183381  1.186292  1.188744  1.190736  1.192265  1.193329  1.193924  1.194047  1.193692  1.192855  1.191529  1.189709  1.187387  1.184555  1.181207  1.177333  1.172926  1.167977  1.162477  1.156417   1.14979  1.142586  1.134799  1.126423  1.117452  1.107882       1.097713 | 4.219002  4.318165  4.413858  4.506035  4.594676  4.679777  4.761351  4.839414  4.913991  4.985108   5.05279  5.117065  5.177957   5.23549  5.289684  5.340557  5.388124  5.432396   5.47338  5.511083  5.545503   5.57664  5.604486  5.629033  5.650266  5.668171  5.682726  5.693909  5.701694  5.706051  5.706949  5.704353  5.698225  5.688527  5.675218  5.658254  5.637592  5.613188  5.584998  5.552978  5.517088  5.477287   5.43354  5.385819  5.334098  5.278362  5.218609  5.154844  5.087094          5.0154 | 318.762374  338.536372  358.367162   378.17597  397.889464  417.439067  436.760375  455.792683  474.478578  492.763602  510.595974  527.926347  544.707613  560.894734  576.444609  591.315952   605.46921  618.866484  631.471477   643.24946  654.167243   664.19318  673.297167  681.450676  688.626788  694.800247  699.947535  704.046954  707.078729  709.025129  709.870599  709.601922  708.208387  705.681982   702.01761  697.213312  691.270525  684.194345  675.993815   666.68223  656.277448  644.802225  632.284546  618.757966  604.261941   588.84216  572.550841  555.447004  537.596702  519.073178  499.956953 |  100.0 | 218.762374 | 499.956953 |
    | 11:30 | 6.208858   6.37715  6.540563   6.69888   6.85195  6.999667  7.141967  7.278807  7.410166  7.536036  7.656416  7.771315   7.88074  7.984703  8.083214  8.176284  8.263917  8.346119  8.422889  8.494225  8.560117  8.620555   8.67552  8.724991  8.768942  8.807342  8.840153  8.867336  8.888844  8.904627  8.914631  8.918797  8.917062   8.90936  8.895623  8.875777  8.849751  8.817467  8.778852   8.73383  8.682328  8.624278  8.559612  8.488275  8.410216  8.325398  8.233797   8.13541  8.030252             7.918365 | 285.882459  298.997426  312.003262  324.858191  337.525307  349.971734  362.167954  374.087258  385.705293  396.999698  407.949802  418.536369  428.741399   438.54795  447.939998  456.902319  465.420385  473.480286  481.068659  488.172635  494.779794  500.878129  506.456024  511.502236  516.005887  519.956466  523.343835  526.158245  528.390364  530.031307  531.072678  531.506623  531.325893  530.523912  529.094866  527.033792  524.336694  521.000659  517.023996  512.406391  507.149068  501.254979  494.729004  487.578171  479.811888  471.442204  462.484067  452.955618  442.878473  432.278034 |   77.08858   78.771496   80.405629   81.988803   83.519496   84.996675   86.419668    87.78807    89.10166   90.360356   91.564163   92.713146   93.807398   94.847029   95.832145   96.762836   97.639171   98.461189   99.228893   99.942247  100.601172  101.205546  101.755199  102.249913  102.689424  103.073417  103.401531  103.673357  103.888437  104.046269  104.146308  104.187967  104.170619  104.093603  103.956228  103.757775  103.497507  103.174673   102.78852  102.338301  101.823285  101.242775  100.596124   99.882748   99.102158   98.253977   97.337975   96.354099   95.302515     94.183654 |   1.4402  1.462072  1.483132  1.503375  1.522801  1.541417  1.559231  1.576256  1.592502  1.607984  1.622714  1.636704  1.649967  1.662514  1.674355    1.6855  1.695958  1.705735  1.714839  1.723275  1.731047  1.738159  1.744614  1.750412  1.755555  1.760041   1.76387  1.767038  1.769543   1.77138  1.772544  1.773028  1.772826   1.77193  1.770332  1.768021  1.764989  1.761223  1.756713  1.751447  1.745412  1.738597  1.730987  1.722572  1.713338  1.703273  1.692367   1.68061  1.667995  1.654514 | 1.344732  1.344222  1.343759  1.343336   1.34295  1.342598  1.342276  1.341982  1.341712  1.341464  1.341237  1.341029  1.340838  1.340663  1.340502  1.340355   1.34022  1.340097  1.339985  1.339883  1.339791  1.339708  1.339634  1.339568  1.339511  1.339461  1.339419  1.339385  1.339358  1.339338  1.339326  1.339321  1.339323  1.339332  1.339349  1.339374  1.339407  1.339448  1.339498  1.339557  1.339625  1.339703  1.339792  1.339891  1.340003  1.340128  1.340266  1.340419  1.340588          1.340774 | 0.670267  0.676387    0.6821  0.687431  0.692406  0.697049  0.701382  0.705425  0.709198  0.712717  0.715999  0.719058  0.721907  0.724558   0.72702  0.729304  0.731417  0.733369  0.735164  0.736809   0.73831  0.739671  0.740895  0.741987  0.742949  0.743783  0.744491  0.745075  0.745534  0.745871  0.746083  0.746172  0.746135  0.745971  0.745679  0.745255  0.744697  0.744002  0.743164  0.742181  0.741046  0.739754  0.738298  0.736673  0.734869   0.73288  0.730695  0.728306  0.725702      0.722873 | -0.362243  -0.369102  -0.375504   -0.38148  -0.387058  -0.392264  -0.397122  -0.401656  -0.405884  -0.409825  -0.413497  -0.416913  -0.420087  -0.423033  -0.425759  -0.428277  -0.430594  -0.432719  -0.434659  -0.436418  -0.438003  -0.439417  -0.440665   -0.44175  -0.442673  -0.443438  -0.444044  -0.444493  -0.444785   -0.44492  -0.444897  -0.444713  -0.444368  -0.443858  -0.443181  -0.442333  -0.441309  -0.440106  -0.438716  -0.437136  -0.435357  -0.433373  -0.431175  -0.428756  -0.426106  -0.423215  -0.420073  -0.416669  -0.412993     -0.409031 | 0.686043  0.687045  0.687993  0.688889  0.689734  0.690532  0.691283  0.691989  0.692653  0.693274  0.693855  0.694397  0.694901  0.695368  0.695799  0.696195  0.696556  0.696884  0.697178  0.697439  0.697667  0.697864  0.698028   0.69816   0.69826  0.698329  0.698366   0.69837  0.698343  0.698282  0.698189  0.698063  0.697903  0.697709   0.69748  0.697216  0.696916  0.696579  0.696204  0.695792   0.69534  0.694849  0.694318  0.693745   0.69313  0.692473  0.691773  0.691029  0.690242      0.689412 |  296.68568 | 306.178412  325.210834  344.364012  363.560377  382.727474  401.797335  420.705953  439.392838  457.800636  475.874816    493.5634  510.816735  527.587304  543.829562  559.499806  574.556057  588.957976  602.666787  615.645222  627.857483  639.269217  649.847501  659.560853  668.379243  676.274129  683.218498  689.186931  694.155675  698.102735  701.007978  702.853257  703.622553  703.302128  701.880705  699.349658   695.70323  690.938759  685.056933  678.062059  669.962346   660.77021   650.50259  639.181274  626.833235  613.490967  599.192818  583.983316  567.913464  551.041014          533.430692 | 524.469322 | 0.963895  0.978904  0.993347  1.007222   1.02053  1.033277  1.045469  1.057116  1.068226   1.07881  1.088877  1.098435  1.107494  1.116062  1.124146  1.131753  1.138889   1.14556  1.151771  1.157524  1.162825  1.167675  1.172076  1.176029  1.179534  1.182593  1.185202  1.187362  1.189069  1.190321  1.191114  1.191444  1.191306  1.190696  1.189607  1.188032  1.185965  1.183398  1.180324  1.176734   1.17262  1.167973  1.162784  1.157045  1.150747   1.14388  1.136439  1.128415  1.119804       1.110599 | 4.101626  4.201323  4.297938   4.39137  4.481553   4.56845  4.652042  4.732322  4.809294  4.882969   4.95336  5.020482  5.084352  5.144987  5.202399  5.256604  5.307612  5.355431  5.400068  5.441525  5.479803  5.514898  5.546804  5.575513  5.601011  5.623283  5.642311  5.658071   5.67054  5.679689  5.685487  5.687902  5.686896  5.682432  5.674469  5.662965  5.647875  5.629155  5.606759  5.580641  5.550756  5.517059  5.479509  5.438067    5.3927  5.343379  5.290083  5.232802  5.171537        5.106302 |  296.68568  315.639889  334.754887  353.950392  373.151572  392.288367  411.294916   430.10908  448.672029  466.927898    484.8235  502.308074  519.333083  535.852034  551.820338  567.195181  581.935425  596.001531   609.35549  621.960779  633.782325  644.786492  654.941076  664.215314  672.579909   680.00707  686.470563   691.94578   696.40982  699.841586  702.221905  703.533651  703.761901  702.894097  700.920231  697.833048  693.628274  688.304849  681.865191  674.315476   665.66593  655.931138  645.130367  633.287903   620.43338  606.602122  591.835474  576.181113   559.69334  542.433328  524.469322 |  100.0 |  196.68568 | 524.469322 |
    | 12:00 | 6.017004  6.184817  6.348515  6.507779  6.662372  6.812122  6.956905  7.096631  7.231239  7.360689  7.484952  7.604012   7.71786   7.82649  7.929901   8.02809  8.121055  8.208794  8.291299  8.368564  8.440578  8.507324  8.568786  8.624939  8.675757  8.721209  8.761259  8.795867  8.824989  8.848575  8.866573  8.878925   8.88557  8.886443  8.881476  8.870596   8.85373  8.830803  8.801735  8.766449  8.724868  8.676914  8.622513  8.561597  8.494101  8.419969  8.339156  8.251629  8.157373              8.05639 |  271.27676  284.032089  296.745946  309.372585  321.871563  334.206863  346.346177  358.260315   369.92271  381.309019  392.396786   403.16517  413.594708  423.667125  433.365175  442.672502   451.57353  460.053364  468.097714  475.692826  482.825433  489.482703  495.652213  501.321919   506.48014  511.115549  515.217171  518.774389  521.776959  524.215029  526.079172  527.360424  528.050328  528.140995  527.625171   526.49631  524.748672  522.377415   519.37872  515.749911  511.489604  506.597865  501.076384  494.928667  488.160249  480.778912  472.794938  464.221359  455.074237  445.372947 |  75.170042   76.848174   78.485151   80.077785   81.623717    83.12122   84.569046   85.966309   87.312394   88.606887   89.849517    91.04012   92.178599   93.264905   94.299011   95.280901   96.210553   97.087936   97.912993   98.685645   99.405778  100.073243  100.687856  101.249387  101.757569  102.212088  102.612589   102.95867  103.249887  103.485751  103.665729   103.78925  103.855701   103.86443  103.814755   103.70596  103.537305  103.308026   103.01735  102.664493  102.248678  101.769137  101.225134   100.61597   99.941007   99.199689   98.391558   97.516292   96.573727       95.5639 |  1.41503   1.43706  1.458364  1.478921  1.498721   1.51776  1.536041  1.553568  1.570349  1.586394  1.601711  1.616311  1.630203  1.643399  1.655907  1.667735  1.678891  1.689384  1.699218  1.708399  1.716933  1.724821  1.732068  1.738674  1.744641  1.749969  1.754656  1.758701  1.762101  1.764852  1.766949  1.768388  1.769162  1.769263  1.768685  1.767418  1.765453  1.762779  1.759386  1.755263  1.750398  1.744777  1.738389  1.731221   1.72326  1.714493  1.704908  1.694493  1.683239  1.671135 | 1.345356  1.344808  1.344307  1.343849  1.343431  1.343049  1.342698  1.342377  1.342082  1.341812  1.341563  1.341335  1.341125  1.340932  1.340754  1.340591  1.340442  1.340304  1.340179  1.340064  1.339959  1.339864  1.339779  1.339702  1.339634  1.339573  1.339521  1.339476  1.339439  1.339409  1.339386   1.33937  1.339362  1.339361  1.339367  1.339381  1.339402  1.339431  1.339469  1.339514  1.339569  1.339632  1.339705  1.339789  1.339883  1.339989  1.340107  1.340239  1.340384          1.340545 | 0.662977  0.669373  0.675363  0.680971  0.686219  0.691128   0.69572  0.700016  0.704033  0.707789    0.7113   0.71458  0.717644  0.720503  0.723167  0.725648  0.727954  0.730092  0.732071  0.733896  0.735574  0.737109  0.738506  0.739769    0.7409  0.741904  0.742781  0.743534  0.744164  0.744672  0.745058  0.745323  0.745465  0.745483  0.745377  0.745144  0.744783   0.74429  0.743661  0.742894  0.741984  0.740926  0.739714  0.738343  0.736806  0.735096  0.733204  0.731123  0.728843      0.726354 | -0.353772  -0.360954   -0.36768  -0.373977   -0.37987  -0.385384  -0.390543  -0.395368   -0.39988  -0.404098  -0.408038  -0.411715  -0.415144  -0.418337  -0.421307  -0.424062  -0.426612  -0.428966  -0.431131  -0.433112  -0.434917   -0.43655  -0.438015  -0.439316  -0.440456  -0.441437  -0.442261  -0.442929  -0.443443  -0.443801  -0.444005  -0.444053  -0.443943  -0.443675  -0.443244   -0.44265  -0.441887  -0.440953  -0.439842  -0.438549   -0.43707  -0.435396  -0.433522  -0.431439   -0.42914  -0.426616  -0.423857  -0.420853  -0.417594     -0.414069 | 0.684751  0.685784  0.686765  0.687696  0.688578  0.689413  0.690202  0.690947   0.69165  0.692311  0.692932  0.693514  0.694058  0.694566  0.695038  0.695474  0.695876  0.696245  0.696581  0.696884  0.697155  0.697394  0.697601  0.697777  0.697921  0.698035  0.698117  0.698168  0.698187  0.698175  0.698131  0.698054  0.697945  0.697803  0.697627  0.697417  0.697172  0.696892  0.696576  0.696224  0.695833  0.695405  0.694937   0.69443  0.693882  0.693292  0.692661  0.691987  0.691271      0.690511 | 276.276442 |   285.3259  303.516397  321.923161  340.467883  359.077271  377.682468  396.218543  414.624057  432.840691  450.812929  468.487783  485.814565  502.744691  519.231512  535.230176  550.697508  565.591913  579.873299  593.503006  606.443766  618.659664  630.116117   640.77987  650.618998  659.602927  667.702464   674.88984  681.138772  686.424534   690.72404  694.015952  696.280793  697.501084  697.661494  696.749006  694.753109  691.665997  687.482797  682.201809  675.824766  668.357114  659.808304  650.192103  639.526915  627.836113  615.148375  601.498014  586.925317  571.476844          555.205728 |  546.89363 |  0.94661  0.961739   0.97636   0.99046  1.004033  1.017077  1.029596  1.041594  1.053076  1.064049  1.074522  1.084501  1.093994  1.103008   1.11155  1.119626  1.127242  1.134403  1.141114  1.147378  1.153199  1.158579  1.163521  1.168026  1.172095  1.175727  1.178922  1.181679  1.183996  1.185871  1.187301  1.188282  1.188809  1.188878  1.188484   1.18762  1.186281  1.184459  1.182146  1.179336  1.176019  1.172187  1.167832  1.162944  1.157514  1.151535  1.144996   1.13789  1.130209       1.121948 | 3.987709  4.087367  4.184374   4.27857  4.369839  4.458102  4.543308  4.625422  4.704427  4.780312  4.853078  4.922726  4.989262  5.052694   5.11303  5.170278  5.224443   5.27553  5.323544  5.368484  5.410349  5.449136  5.484837  5.517443  5.546942  5.573318  5.596554  5.616629  5.633517  5.647194  5.657629   5.66479  5.668642  5.669148  5.666268  5.659961  5.650183  5.636889  5.620032  5.599565  5.575441  5.547613  5.516035  5.480662  5.441452   5.39837  5.351381  5.300461  5.245592        5.186769 | 276.276442   294.34072  312.661655   331.15828  349.754957  368.380755  386.968903  405.456322  423.783225  441.892772  459.730777  477.245459  494.387231  511.108517    527.3636  543.108498  558.300848  572.899826  586.866072  600.161634  612.749925  624.595703  635.665048  645.925367  655.345405  663.895266  671.546458  678.271936  684.046174  688.845241  692.646894  695.430693   697.17812  697.872725  697.500287  696.048988  693.509609  689.875746  685.144044  679.314442  672.390451  664.379433  655.292908  645.146869  633.962108  621.764552    608.5856  594.462456  579.438454  563.563357   546.89363 |  100.0 | 176.276442 |  546.89363 |
    | 12:30 | 5.832203  5.998568  6.161644  6.321006  6.476328  6.627362   6.77392  6.915861  7.053079  7.185495  7.313051  7.435706  7.553427  7.666191  7.773981  7.876782  7.974581  8.067367  8.155126  8.237846   8.31551    8.3881  8.455594  8.517969  8.575197  8.627245  8.674079  8.715657  8.751938  8.782873   8.80841  8.828494  8.843064  8.852057  8.855406   8.85304  8.844886  8.830868  8.810906  8.784922  8.752835  8.714563  8.670028  8.619151  8.561858  8.498081  8.427758  8.350836  8.267274             8.177046 | 257.556023  269.892613  282.253963  294.590694  306.859064  319.020085  331.038784  342.883582  354.525775    365.9391  377.099376  387.984202  398.572703   408.84532  418.783629  428.370193  437.588435   446.42253  454.857312  462.878205  470.471153  477.622568  484.319292  490.548559  496.297971   501.55548  506.309376   510.54829  514.261191  517.437402  520.066617  522.138932  523.644873  524.575448  524.922192  524.677235  523.833374  522.384152  520.323961  517.648143  514.353119  510.436517  505.897327  500.736068  494.954969  488.558168  481.551927  473.944872  465.748233  456.976116 |  73.322033    74.98568   76.616442   78.210063   79.763283   81.273622   82.739203    84.15861   85.530787   86.854948   88.130514   89.357059    90.53427   91.661914   92.739811   93.767819   94.745811   95.673667   96.551262   97.378458   98.155097   98.880996   99.555943  100.179694   100.75197  101.272452  101.740786  102.156575  102.519383  102.828732  103.084103  103.284939  103.430641  103.520573  103.554062  103.530405  103.448864  103.308678  103.109065  102.849224  102.528349  102.145633  101.700278  101.191508  100.618584   99.980815   99.277583   98.508362   97.672743     96.770462 | 1.390539  1.412598   1.43403  1.454796  1.474874   1.49425  1.512916  1.530871  1.548117  1.564658    1.5805   1.59565  1.610116  1.623906  1.637028  1.649488  1.661295  1.672453   1.68297   1.69285  1.702098  1.710717  1.718709  1.726078  1.732823  1.738945  1.744445  1.749319  1.753566  1.757183  1.760166   1.76251  1.764209  1.765258  1.765648  1.765372  1.764422  1.762787  1.760457  1.757422  1.753671  1.749191  1.743969  1.737994  1.731252   1.72373  1.715415  1.706295  1.696358  1.685591 | 1.346005  1.345418  1.344881  1.344389  1.343937  1.343524  1.343144  1.342796  1.342475  1.342181   1.34191  1.341661  1.341431  1.341219  1.341024  1.340845   1.34068  1.340528  1.340388   1.34026  1.340142  1.340035  1.339938   1.33985   1.33977  1.339699  1.339636  1.339581  1.339533  1.339493   1.33946  1.339434  1.339416  1.339404    1.3394  1.339403  1.339413  1.339431  1.339457   1.33949  1.339532  1.339582  1.339641   1.33971  1.339788  1.339878  1.339978   1.34009  1.340215          1.340354 | 0.655616  0.662258  0.668505  0.674373  0.679881  0.685047  0.689892  0.694434  0.698691  0.702681  0.706418  0.709919  0.713196  0.716262  0.719129  0.721805  0.724302  0.726627  0.728788  0.730792  0.732646  0.734353  0.735921  0.737352  0.738651   0.73982  0.740863  0.741782  0.742577  0.743252  0.743806   0.74424  0.744554  0.744747  0.744819  0.744768  0.744593  0.744291   0.74386  0.743296  0.742597  0.741757  0.740773  0.739639  0.738349  0.736898  0.735277   0.73348  0.731498      0.729322 | -0.345211  -0.352686  -0.359712  -0.366311  -0.372505  -0.378316  -0.383766  -0.388877  -0.393668  -0.398157  -0.402361  -0.406297  -0.409977  -0.413416  -0.416626  -0.419615  -0.422396  -0.424976  -0.427363  -0.429564  -0.431586  -0.433433  -0.435111  -0.436624  -0.437975  -0.439167  -0.440202  -0.441083  -0.441811  -0.442385  -0.442808  -0.443077  -0.443193  -0.443155   -0.44296  -0.442606  -0.442091  -0.441411  -0.440562   -0.43954  -0.438341  -0.436958  -0.435386  -0.433617  -0.431645  -0.429462  -0.427059  -0.424427  -0.421557     -0.418437 | 0.683471  0.684528  0.685537  0.686498  0.687412  0.688281  0.689105  0.689885  0.690624  0.691322   0.69198    0.6926  0.693183  0.693729  0.694239  0.694714  0.695156  0.695564   0.69594  0.696283  0.696594  0.696874  0.697122   0.69734  0.697527  0.697683  0.697809  0.697904  0.697968  0.698002  0.698004  0.697975  0.697914  0.697822  0.697696  0.697538  0.697347  0.697121   0.69686  0.696564  0.696232  0.695863  0.695457  0.695012  0.694528  0.694004   0.69344  0.692834  0.692186      0.691496 | 257.498082 |  266.07758  283.368962  300.963801  318.783844  336.755645  354.810047  372.881723  390.908765  408.832328  426.596318   444.14713  461.433413  478.405874  495.017112  511.221468  526.974912  542.234932  556.960451  571.111763  584.650468  597.539441  609.742793  621.225863   631.95521  641.898618  651.025118   659.30502  666.709956  673.212934  678.788411  683.412375  687.062443  689.717975    691.3602  691.972363  691.539885  690.050546  687.494675  683.865372  679.158737  673.374126  666.514417  658.586298  649.600567   639.57245  628.521929  616.474068  603.459356  589.514034          574.680419 | 567.070969 | 0.929778  0.944939  0.959659  0.973912  0.987685  1.000968  1.013759  1.026056  1.037863  1.049182  1.060019  1.070379  1.080268  1.089692  1.098656  1.107167  1.115229  1.122847  1.130026  1.136769  1.143079  1.148959  1.154411  1.159436  1.164036  1.168211   1.17196  1.175283  1.178179  1.180644  1.182678  1.184275  1.185434  1.186148  1.186414  1.186226  1.185578  1.184464  1.182876  1.180808   1.17825  1.175196  1.171636  1.167562  1.162965  1.157835  1.152164  1.145942  1.139162       1.131815 | 3.877699  3.976747  4.073618  4.168086  4.259982  4.349183  4.435599  4.519166  4.599839  4.677589  4.752397   4.82425  4.893143  4.959073  5.022039  5.082043  5.139085  5.193166  5.244285   5.29244  5.337628  5.379843  5.419077   5.45532   5.48856  5.518782  5.545968  5.570097  5.591147  5.609092  5.623903   5.63555  5.643999  5.649213  5.651155  5.649783  5.645055  5.636926  5.625351   5.61028  5.591667  5.569462  5.543616  5.514082  5.480813  5.443765  5.402897  5.358174  5.309565        5.257048 | 257.498082  274.619595   292.08488  309.813152  327.728697  345.760343  363.840961  381.907033  399.898269  417.757278  435.429274  452.861836   470.00469  486.809531  503.229865  519.220877   534.73932  549.743422  564.192808  578.048437  591.272556  603.828666  615.681494  626.796988  637.142317  646.685881  655.397342  663.247656  670.209126  676.255465   681.36187  685.505115  688.663656   690.81775  691.949592  692.043469  691.085928  689.065968   685.97524  681.808278  676.562737  670.239655  662.843733  654.383626  644.872256  634.327132  622.770679  610.230578  596.740097  582.338421  567.070969 |  100.0 | 157.498082 | 567.070969 |
    | 13:00 | 5.655093  5.819066  5.980632  6.139258  6.294522  6.446098   6.59373  6.737219  6.876411  7.011186  7.141453   7.26714   7.38819  7.504561  7.616216  7.723127   7.82527  7.922621   8.01516  8.102866  8.185719  8.263694  8.336768  8.404914  8.468101  8.526298  8.579468  8.627571  8.670564  8.708401   8.74103  8.768397  8.790444  8.807109  8.818325  8.824024  8.824133  8.818577  8.807277  8.790155  8.767126  8.738109  8.703021  8.661778    8.6143  8.560508  8.500331  8.433699  8.360554             8.280848 | 244.726788  256.593605  268.549283  280.541284  292.522899  304.452406  316.292346  328.008885  339.571286  350.951453  362.123551  373.063683   383.74962  394.160569  404.276984  414.080396   423.55328  432.678929  441.441359  449.825224  457.815739  465.398622  472.560047  479.286595  485.565229  491.383263  496.728347  501.588454  505.951879  509.807239  513.143485  515.949917  518.216209  519.932443  521.089147  521.677344   521.68861  521.115144  519.949845    518.1864   515.81939  512.844399  509.258149  505.058635   500.24529  494.819154  488.783066   482.14187  474.902639  467.074913 |  71.550931   73.190656   74.806321   76.392576   77.945224   79.460985   80.937302    82.37219   83.764108   85.111862    86.41453   87.671396   88.881901   90.045607   91.162161   92.231274   93.252698    94.22621   95.151601   96.028665   96.857187   97.636942   98.367682   99.049139   99.681014  100.262981  100.794677  101.275708  101.705642  102.084008  102.410301  102.683973  102.904442  103.071086  103.183249  103.240239   103.24133  103.185769  103.072775  102.901545  102.671261  102.381092  102.030206  101.617777  101.142997  100.605085  100.003307   99.336989   98.605541     97.808477 | 1.366828  1.388788   1.41023  1.431098  1.451357  1.470979  1.489948  1.508254  1.525893  1.542863  1.559167  1.574808  1.589791   1.60412  1.617803  1.630845  1.643251  1.655028   1.66618  1.676712  1.686628  1.695931  1.704624   1.71271  1.720188   1.72706  1.733326  1.738984  1.744032  1.748469   1.75229  1.755491  1.758068  1.760014  1.761323  1.761988  1.762001  1.761353  1.760034  1.758034  1.755342  1.751948  1.747838  1.743001  1.737424  1.731093  1.723996  1.716119  1.707449  1.697974 | 1.346675  1.346053   1.34548  1.344953  1.344468  1.344023  1.343614  1.343237  1.342891  1.342572  1.342277  1.342006  1.341756  1.341525  1.341312  1.341116  1.340934  1.340767  1.340613  1.340471   1.34034   1.34022  1.340111  1.340011   1.33992  1.339838  1.339764  1.339698  1.339641   1.33959  1.339547  1.339512  1.339483  1.339462  1.339447   1.33944   1.33944  1.339447  1.339461  1.339483  1.339513  1.339551  1.339597  1.339652  1.339717   1.33979  1.339874  1.339969  1.340076          1.340194 | 0.648225   0.65508  0.661556  0.667662  0.673413  0.678824  0.683912  0.688693  0.693186  0.697405  0.701366  0.705085  0.708574  0.711846  0.714912  0.717784  0.720471  0.722981  0.725324  0.727506  0.729533  0.731412  0.733148  0.734746  0.736209  0.737542  0.738747  0.739828  0.740785  0.741622  0.742339  0.742937  0.743416  0.743778   0.74402  0.744143  0.744146  0.744026  0.743781   0.74341  0.742909  0.742275  0.741503   0.74059   0.73953  0.738319  0.736949  0.735415  0.733708      0.731822 | -0.336605  -0.344337  -0.351635  -0.358514   -0.36499  -0.371084  -0.376815  -0.382202  -0.387264  -0.392019  -0.396483  -0.400672  -0.404601  -0.408282  -0.411728   -0.41495  -0.417958  -0.420762  -0.423368  -0.425785   -0.42802  -0.430079  -0.431966  -0.433686  -0.435244  -0.436642  -0.437883  -0.438971  -0.439906   -0.44069  -0.441323  -0.441806  -0.442139   -0.44232   -0.44235  -0.442225  -0.441945  -0.441506  -0.440905  -0.440139  -0.439204  -0.438094  -0.436804   -0.43533  -0.433663  -0.431798  -0.429726   -0.42744   -0.42493     -0.422188 | 0.682209  0.683284  0.684315    0.6853  0.686242   0.68714  0.687995  0.688808   0.68958  0.690312  0.691005   0.69166  0.692279  0.692861  0.693408   0.69392  0.694399  0.694845  0.695258   0.69564   0.69599  0.696308  0.696597  0.696854  0.697082  0.697279  0.697446  0.697583  0.697691  0.697768  0.697814  0.697831  0.697816  0.697771  0.697694  0.697585  0.697443  0.697269  0.697061   0.69682  0.696543  0.696231  0.695883  0.695498  0.695075  0.694614  0.694113  0.693573  0.692991      0.692369 | 240.298231 | 248.389159  264.740134  281.472955  298.510263  315.779159   333.21079  350.739961  368.304769  385.846273  403.308207  420.636719  437.780153  454.688854  471.314995  487.612442  503.536623  519.044426  534.094111  548.645232  562.658584   576.09615  588.921067  601.097604  612.591148  623.368202  633.396393  642.644495   651.08246  658.681456  665.413928  671.253663  676.175869  680.157269  683.176214  685.212797  686.249002  686.268852  685.258583  683.206835  680.104855  675.946727  670.729612  664.454011  657.124042  648.747732  639.337331  628.909627  617.486279  605.094151          591.765644 | 584.896578 | 0.913469  0.928574  0.943312  0.957646  0.971552  0.985013   0.99802  1.010565  1.022647  1.034267  1.045425  1.056126  1.066372  1.076169  1.085521  1.094432  1.102907   1.11095  1.118565  1.125754  1.132522  1.138871  1.144802  1.150318  1.155419  1.160106  1.164379  1.168237  1.171679  1.174704  1.177309  1.179491  1.181247  1.182574  1.183466   1.18392  1.183928  1.183486  1.182587  1.181224   1.17939  1.177076  1.174274  1.170976  1.167173  1.162856  1.158016  1.152643  1.146729       1.140265 |  3.77199  3.869867  3.966079  4.060332    4.1524   4.24211  4.329333  4.413971  4.495952  4.575221   4.65174  4.725481  4.796423  4.864553  4.929861  4.992339  5.051982  5.108784  5.162741  5.213848  5.262097  5.307482  5.349993  5.389618  5.426345  5.460159   5.49104  5.518971  5.543928  5.565886  5.584819  5.600695  5.613483  5.623148  5.629653  5.632958  5.633021  5.629799  5.623246  5.613315  5.599958  5.583124  5.562764  5.538828  5.511266  5.480029  5.445072  5.406351  5.363825        5.317462 | 240.298231  256.440293   273.00399  289.909628   307.08218  324.450874  341.948785  359.512452  377.081536  394.598503  412.008357    429.2584  446.298023  463.078526  479.552965  495.676016  511.403864  526.694104  541.505661  555.798725  569.534694  582.676137  595.186762  607.031399  618.175992  628.587603  638.234426  647.085812  655.112309  662.285707  668.579103  673.966969  678.425245  681.931435  684.464727  686.006118  686.538566  686.047152   684.51926  681.944771  678.316289  673.629365  667.882754  661.078684  653.223145  644.326185   634.40223  623.470409  611.554887  598.685197  584.896578 |  100.0 | 140.298231 | 584.896578 |
    | 13:30 | 5.486211  5.646885   5.80608  5.963154   6.11759  6.268978   6.41699  6.561369  6.701907  6.838441  6.970842  7.099005  7.222848  7.342304  7.457319  7.567847   7.67385  7.775293  7.872146   7.96438  8.051966  8.134878  8.213086   8.28656  8.355267  8.419174  8.478242  8.532432    8.5817  8.626001  8.665284  8.699495  8.728578  8.752472  8.771113  8.784433  8.792362  8.794824  8.791744   8.78304  8.768632  8.748436  8.722366  8.690337  8.652264  8.608063  8.557653  8.500956    8.4379              8.36842 | 232.785701  244.139814  255.644051  267.243315  278.888362  290.535058  302.143686  313.678339  325.106382  336.397996  347.525779  358.464417  369.190393  379.681743  389.917854   399.87928  409.547599  418.905278  427.935565  436.622398  444.950319  452.904411  460.470238  467.633797  474.381477  480.700031  486.576547   491.99843  496.953396  501.429462  505.414948  508.898488  511.869043  514.315924  516.228822  517.597841  518.413549  518.667029  518.349941  517.454601  515.974056  513.902192  511.233829  507.964852  504.092343  499.614728  494.531944  488.845623  482.559284  475.678551 |  69.862107   71.468848   73.060804   74.631537   76.175897   77.689775   79.169904   80.613689   82.019069    83.38441   84.708415   85.990048   87.228481   88.423045   89.573194   90.678474   91.738498   92.752927   93.721456   94.643795   95.519665   96.348784   97.130864   97.865601   98.552674   99.191737   99.782418  100.324317  100.817002  101.260008  101.652835   101.99495   102.28578  102.524721  102.711131  102.844333  102.923617  102.948242  102.917437  102.830404  102.686324  102.484359   102.22366  101.903371  101.522642  101.080634  100.576532  100.009562      99.379     98.684198 | 1.343992  1.365724  1.387057  1.407919  1.428258  1.448035  1.467222  1.485801  1.503761  1.521093  1.537794  1.553865  1.569306  1.584121  1.598312  1.611884  1.624841  1.637187  1.648927  1.660065  1.670604  1.680547  1.689896  1.698654  1.706821  1.714399  1.721387  1.727784  1.733589  1.738799  1.743413  1.747425  1.750832  1.753628  1.755808  1.757365  1.758292  1.758579  1.758219  1.757203  1.755518  1.753156  1.750105  1.746351  1.741884   1.73669  1.730757   1.72407  1.716616  1.708382 | 1.347363  1.346707    1.3461   1.34554  1.345023  1.344546  1.344107  1.343702  1.343328  1.342983  1.342665  1.342372    1.3421   1.34185  1.341618  1.341403  1.341205  1.341022  1.340853  1.340697  1.340552   1.34042  1.340298  1.340186  1.340083   1.33999  1.339906  1.339829  1.339761  1.339701  1.339648  1.339602  1.339564  1.339532  1.339508  1.339491  1.339481  1.339477  1.339481  1.339493  1.339511  1.339538  1.339572  1.339614  1.339665  1.339725  1.339794  1.339873  1.339963          1.340064 | 0.640848  0.647874  0.654547  0.660868  0.666842  0.672482    0.6778  0.682812  0.687531  0.691974  0.696155  0.700088  0.703786  0.707261  0.710527  0.713592  0.716468  0.719163  0.721686  0.724044  0.726244  0.728293  0.730196  0.731958  0.733584  0.735078  0.736442  0.737682  0.738798  0.739792  0.740668  0.741425  0.742066  0.742589  0.742996  0.743286  0.743458  0.743511  0.743445  0.743256  0.742942  0.742501  0.741929  0.741223  0.740378  0.739391  0.738254  0.736963  0.735512      0.733893 | -0.328002   -0.33595  -0.343485  -0.350616  -0.357354  -0.363713   -0.36971  -0.375362  -0.380686    -0.3857  -0.390417  -0.394856  -0.399028  -0.402948  -0.406628  -0.410078  -0.413311  -0.416334  -0.419157  -0.421788  -0.424233  -0.426499  -0.428591  -0.430515  -0.432275  -0.433875  -0.435318  -0.436607  -0.437743   -0.43873  -0.439567  -0.440256  -0.440798  -0.441191  -0.441435   -0.44153  -0.441474  -0.441264  -0.440898  -0.440374  -0.439687  -0.438834  -0.437811  -0.436611   -0.43523  -0.433662    -0.4319  -0.429936  -0.427762      -0.42537 | 0.680974  0.682059  0.683105   0.68411  0.685074  0.685997  0.686879  0.687721  0.688523  0.689286  0.690011  0.690699  0.691351  0.691967  0.692548  0.693095   0.69361  0.694091   0.69454  0.694958  0.695345  0.695701  0.696027  0.696323  0.696589  0.696825  0.697032   0.69721  0.697358  0.697477  0.697566  0.697625  0.697655  0.697654  0.697623  0.697561  0.697468  0.697343  0.697185  0.696995  0.696771  0.696514  0.696221  0.695893  0.695529  0.695127  0.694688   0.69421  0.693692      0.693135 | 224.612057 | 232.203236  247.587184  263.422219  279.632737  296.147101  312.897375  329.819025  346.850621  363.933548  381.011753  398.031503  414.941174  431.691072  448.233263  464.521437  480.510783  496.157882  511.420622  526.258117  540.630644  554.499595  567.827434  580.577671  592.714837   604.20448  615.013162  625.108474  634.459052  643.034611  650.805987   657.74519  663.825469  669.021386  673.308913   676.66553  679.070345  680.504228  680.949957  680.392386  678.818624  676.218237  672.583462  667.909444  662.194487  655.440324  647.652403  638.840189  629.017478  618.202719          606.419352 | 600.314285 | 0.897748  0.912708  0.927383  0.941724  0.955696  0.969272  0.982437  0.995177  1.007486   1.01936  1.030796  1.041797  1.052362  1.062495  1.072198  1.081476   1.09033  1.098765  1.106784   1.11439  1.121585  1.128372  1.134753  1.140729  1.146301   1.15147  1.156237    1.1606  1.164558  1.168111  1.171257  1.173992  1.176315  1.178221  1.179707  1.180769    1.1814  1.181596  1.181351  1.180658   1.17951  1.177899  1.175819   1.17326  1.170215  1.166673  1.162627  1.158066  1.152983       1.147366 | 3.670921  3.767084  3.862125  3.955681  4.047469  4.137265  4.224895  4.310226  4.393154    4.4736  4.551503  4.626816  4.699504   4.76954  4.836904  4.901577  4.963548  5.022805  5.079338  5.133137  5.184192  5.232493  5.278029  5.320786  5.360751  5.397908  5.432238  5.463722  5.492337   5.51806  5.540863  5.560718  5.577594  5.591457   5.60227  5.609997  5.614595  5.616024  5.614237  5.609189  5.600831  5.589115   5.57399  5.555404  5.533306  5.507645  5.478371  5.445436  5.408792        5.368399 | 224.612057  239.752837  255.383497  271.426379  287.807919  304.458403  321.311673  338.304819  355.377892  372.473622  389.537172  406.515917  423.359244  440.018378  456.446231  472.597274  488.427417  503.893917  518.955293  533.571256  547.702653  561.311421  574.360554  586.814076  598.637028  609.795463  620.256454  629.988107  638.959588   647.14116   654.50423  661.021407  666.666575  671.414976  675.243303  678.129818  680.054471  680.999043  680.947302  679.885177  677.800949  674.685456  670.532322  665.338198  659.103025  651.830312  643.527426  634.205909  623.881789  612.575913  600.314285 |  100.0 | 124.612057 | 600.314285 |
    | 14:00 |  5.32599   5.48251   5.63851  5.793242  5.946098  6.096583  6.244297  6.388918  6.530184  6.667885  6.801851  6.931943  7.058051   7.18008  7.297957  7.411616  7.521004  7.626073  7.726782  7.823094  7.914971  8.002379  8.085285  8.163654  8.237451  8.306637  8.371176  8.431025  8.486141  8.536478  8.581985  8.622611  8.658298  8.688989  8.714621  8.735126  8.750437   8.76048  8.765179  8.764457  8.758231  8.746419  8.728935  8.705693  8.676604  8.641582   8.60054  8.553393   8.50006             8.440466 | 221.720713  232.527259  243.541639   254.70693  265.971896  277.290381  288.620711  299.925123  311.169258  322.321706  333.353612  344.238333  354.951143  365.468975   375.77021  385.834486  395.642536  405.176055  414.417576  423.350372  431.958368  440.226064  448.138472  455.681062  462.839715  469.600688  475.950578  481.876303   487.36508  492.404416  496.982103  501.086219  504.705131  507.827513  510.442364  512.539033  514.107259  515.137205  515.619515  515.545369  514.906556  513.695547  511.905593  509.530818  506.566341    503.0084  498.854494   494.10354  488.756048  482.814304 |  68.259902   69.825104   71.385102   72.932425   74.460983   75.965832   77.442974   78.889178   80.301838   81.678847   83.018505   84.319432   85.580505   86.800804   87.979567   89.116159   90.210037   91.260731   92.267825   93.230936   94.149707   95.023793   95.852853   96.636542   97.374505   98.066374    98.71176   99.310252   99.861412  100.364776   100.81985  101.226105  101.582984  101.889893  102.146205  102.351261  102.504367  102.604796  102.651791  102.644568  102.582314  102.464193  102.289354  102.056927   101.76604  101.415817  101.005395  100.533929  100.000605     99.404658 | 1.322113  1.343489  1.364596  1.385343  1.405663  1.425502  1.444822  1.463594    1.4818  1.499424  1.516459  1.532898   1.54874  1.563983   1.57863  1.592681  1.606139  1.619008  1.631289  1.642987  1.654104  1.664642  1.674604   1.68399  1.692803  1.701043  1.708709  1.715802   1.72232  1.728261  1.733622  1.738401  1.742593  1.746193  1.749197  1.751599   1.75339  1.754565  1.755115   1.75503  1.754302   1.75292  1.750874  1.748151  1.744741   1.74063  1.735806  1.730255  1.723964  1.716919 | 1.348065  1.347379  1.346741  1.346148  1.345599  1.345091  1.344622  1.344188  1.343787  1.343417  1.343074  1.342757  1.342464  1.342193  1.341942  1.341709  1.341493  1.341294  1.341109  1.340938   1.34078  1.340634  1.340499  1.340374   1.34026  1.340156   1.34006  1.339973  1.339894  1.339824  1.339761  1.339705  1.339657  1.339616  1.339582  1.339555  1.339535  1.339522  1.339516  1.339517  1.339525   1.33954  1.339563  1.339594  1.339633   1.33968  1.339735    1.3398  1.339875           1.33996 | 0.633528  0.640682  0.647516   0.65402  0.660194  0.666043  0.671577  0.676806  0.681744  0.686403  0.690797  0.694939  0.698843   0.70252  0.705981  0.709239  0.712302   0.71518  0.717881  0.720414  0.722786  0.725003  0.727071  0.728996  0.730783  0.732435  0.733957  0.735353  0.736624  0.737774  0.738804  0.739717  0.740513  0.741193  0.741759  0.742209  0.742545  0.742764  0.742867  0.742851  0.742715  0.742457  0.742073  0.741562  0.740919   0.74014  0.739222  0.738158  0.736943      0.735571 | -0.319453  -0.327568  -0.335302  -0.342654  -0.349627   -0.35623  -0.362477   -0.36838  -0.373955  -0.379217  -0.384182  -0.388862  -0.393273  -0.397426  -0.401336  -0.405012  -0.408465  -0.411706  -0.414742  -0.417583  -0.420235  -0.422705     -0.425  -0.427124  -0.429082  -0.430879  -0.432519  -0.434004  -0.435337   -0.43652  -0.437556  -0.438444  -0.439187  -0.439784  -0.440235   -0.44054  -0.440698  -0.440706  -0.440564  -0.440269  -0.439818  -0.439208  -0.438435  -0.437494  -0.436381  -0.435091  -0.433618  -0.431954  -0.430094     -0.428028 | 0.679771  0.680859  0.681914  0.682933  0.683914  0.684857  0.685762  0.686628  0.687457  0.688248  0.689002   0.68972  0.690403   0.69105  0.691664  0.692244  0.692791  0.693307   0.69379  0.694243  0.694665  0.695056  0.695418   0.69575  0.696053  0.696327  0.696572  0.696788  0.696975  0.697134  0.697263  0.697364  0.697435  0.697477   0.69749  0.697473  0.697425  0.697347  0.697237  0.697096  0.696922  0.696716  0.696476  0.696202  0.695894  0.695549  0.695168   0.69475  0.694294      0.693799 | 210.365288 | 217.452286  231.855953  246.770575  262.123174  277.844105   293.86696  310.128392  326.567903  343.127618  359.752069  376.387984  392.984098  409.490982  425.860887  442.047614  458.006387  473.693759  489.067512  504.086589  518.711023  532.901889  546.621254  559.832147  572.498537  584.585311  596.058275  606.884153  617.030599   626.46622  635.160608  643.084378  650.209222   656.50797  661.954668   666.52466  670.194689  672.943012  674.749525  675.595905  675.465772   674.34486  672.221207  669.085369  664.930642  659.753306  653.552889  646.332441  638.098829  628.863048          618.640532 |  613.31055 | 0.882674  0.897402  0.911932  0.926205  0.940173  0.953803  0.967067  0.979948  0.992434  1.004515  1.016186  1.027444  1.038289   1.04872   1.05874  1.068349  1.077549  1.086344  1.094736  1.102727  1.110319  1.117515  1.124315  1.130722  1.136736  1.142359  1.147589  1.152428  1.156873  1.160925  1.164581   1.16784  1.170698  1.173153  1.175201  1.176838  1.178059   1.17886  1.179234  1.179177  1.178681  1.177739  1.176344  1.174488  1.172162  1.169359   1.16607  1.162285  1.157994        1.15319 | 3.574776  3.668704  3.762078  3.854469  3.945533  4.034995  4.122637  4.208287  4.291807   4.37309  4.452052  4.528627   4.60276  4.674412  4.743548  4.810143  4.874175  4.935624  4.994474  5.050712  5.104322  5.155291  5.203606  5.249251   5.29221  5.332467  5.370002  5.404797  5.436828  5.466072  5.492502  5.516091  5.536809  5.554622  5.569495  5.581393  5.590276  5.596102  5.598828  5.598409  5.594798  5.587945  5.577801  5.564315  5.547433  5.527105  5.503276  5.475897  5.444915        5.410284 | 210.365288  224.496437  239.175863  254.328899  269.884234  285.773875   301.93299  318.299715  334.814918  351.421983  368.066592  384.696522  401.261468  417.712881  434.003818  450.088822  465.923806   481.46596  496.673665  511.506425  525.924806   539.89039  553.365734   566.31434   578.70064  590.489982  601.648626   612.14376  621.943505  631.016953  639.334193  646.866363  653.585708  659.465643  664.480836  668.607304  671.822514  674.105504  675.437021  675.799668   675.17807  673.559059  670.931873  667.288371  662.623272  656.934404  650.222976  642.493862  633.755904  624.022224   613.31055 |  100.0 | 110.365288 |  613.31055 |
    | 14:30 | 5.174762  5.326336  5.478362  5.629999   5.78055  5.929437  6.076189  6.220418  6.361805  6.500091  6.635064  6.766548  6.894399  7.018499  7.138748  7.255061  7.367369  7.475609  7.579727  7.679674  7.775406  7.866881  7.954059    8.0369  8.115365  8.189414  8.259005  8.324096  8.384643  8.440597  8.491911  8.538531  8.580402  8.617467  8.649663  8.676927  8.699191  8.716384  8.728432  8.735257   8.73678  8.732919  8.723589  8.708702  8.688171  8.661906  8.629819   8.59182  8.547822             8.497741 | 211.512255  221.744314  232.237671  242.934433  253.782009  264.732701  275.743231  286.774253  297.789891  308.757309  319.646325  330.429075  341.079712  351.574145  361.889816  372.005507  381.901167   391.55777  400.957188  410.082082  418.915808  427.442335  435.646179  443.512337  451.026239  458.173705  464.940907  471.314341  477.280801  482.827366  487.941386  492.610473   496.82251  500.565645  503.828312  506.599246  508.867506  510.622509  511.854067  512.552437   512.70837  512.313184  511.358831  509.837986  507.744141  505.071714  501.816168  497.974149  493.543632  488.524085 |  66.747622    68.26336   69.783619   71.299991   72.805496   74.294374   75.761894   77.204178    78.61805   80.000914   81.350639    82.66548   83.943995   85.184992   86.387478   87.550615    88.67369   89.756088   90.797267   91.796741   92.754063   93.668814   94.540592   95.369003   96.153652   96.894139   97.590051   98.240963   98.846426   99.405972   99.919106  100.385305  100.804019  101.174665  101.496632  101.769273  101.991912   102.16384  102.284316  102.352571  102.367805  102.329193  102.235887   102.08702   101.88171  101.619065   101.29819  100.918199   100.47822     99.977406 | 1.301262  1.322161  1.342925  1.363449  1.383648  1.403456  1.422822  1.441708  1.460085  1.477932  1.495234   1.51198  1.528164  1.543781   1.55883  1.573309  1.587219  1.600562  1.613338  1.625551  1.637201  1.648291  1.658821  1.668794   1.67821  1.687069  1.695372  1.703119  1.710307  1.716935  1.723001  1.728503  1.733436  1.737796  1.741579  1.744779  1.747389  1.749404  1.750815  1.751614  1.751792   1.75134  1.750248  1.748504  1.746097  1.743016  1.739248   1.73478  1.729598   1.72369 | 1.348775  1.348064  1.347397  1.346774  1.346195  1.345657  1.345158  1.344696  1.344268  1.343871  1.343503  1.343163  1.342847  1.342555  1.342283  1.342032  1.341798  1.341582  1.341381  1.341195  1.341022  1.340862  1.340714  1.340577  1.340451  1.340334  1.340227   1.34013   1.34004  1.339959  1.339886  1.339821  1.339763  1.339712  1.339669  1.339632  1.339602   1.33958  1.339564  1.339555  1.339553  1.339558   1.33957   1.33959  1.339617  1.339652  1.339695  1.339747  1.339808          1.339878 | 0.626313  0.633545  0.640497   0.64715  0.653496  0.659532  0.665263  0.670696  0.675839  0.680705  0.685306  0.689652  0.693756   0.69763  0.701285  0.704732   0.70798  0.711039  0.713918  0.716624  0.719166   0.72155  0.723782  0.725868  0.727814  0.729623    0.7313  0.732849  0.734273  0.735574  0.736756   0.73782  0.738768  0.739601  0.740321  0.740926  0.741419  0.741798  0.742062  0.742212  0.742246  0.742161  0.741956  0.741628  0.741175  0.740593  0.739878  0.739025  0.738031       0.73689 | -0.311008  -0.319237  -0.327126  -0.334662   -0.34184  -0.348664  -0.355139  -0.361278  -0.367091  -0.372591  -0.377792  -0.382707  -0.387349  -0.391732  -0.395866  -0.399763  -0.403434  -0.406888  -0.410134  -0.413182  -0.416038  -0.418709  -0.421202  -0.423523  -0.425676  -0.427667  -0.429499  -0.431175    -0.4327  -0.434075  -0.435303  -0.436385  -0.437323  -0.438116  -0.438766  -0.439273  -0.439636  -0.439853  -0.439925  -0.439848  -0.439621   -0.43924  -0.438704  -0.438007  -0.437147  -0.436117  -0.434915  -0.433532  -0.431964     -0.430203 | 0.678608  0.679692  0.680748  0.681774  0.682766  0.683725  0.684648  0.685535  0.686387  0.687202  0.687983  0.688728  0.689439  0.690115  0.690759   0.69137  0.691948  0.692495  0.693011  0.693496  0.693952  0.694377  0.694773   0.69514  0.695478  0.695787  0.696068  0.696321  0.696545  0.696742   0.69691   0.69705  0.697161  0.697244  0.697298  0.697323  0.697319  0.697285  0.697221  0.697127  0.697001  0.696844  0.696654  0.696432  0.696176  0.695886   0.69556  0.695199  0.694802      0.694367 |  197.47692 | 204.061346  217.483488  231.466974  245.942327  260.842616  276.103572  291.663553  307.463447  323.446515  339.558233  355.746116  371.959562  388.149695  404.269225  420.272323  436.114512  451.752559  467.144396  482.249039  497.026527  511.437862  525.444968  539.010654  552.098583  564.673253   576.69999  588.144937  598.975068  609.158193  618.662984  627.459006  635.516756  642.807715  649.304406  654.980466  659.810731  663.771329  666.839789  668.995163  670.218164  670.491313  669.799113  668.128226  665.467678  661.809076  657.146842  651.478466  644.804776  637.130224          628.463183 | 623.908461 | 0.868296  0.882706  0.897013  0.911143  0.925039  0.938657  0.951962   0.96493  0.977541  0.989782  1.001643  1.013118  1.024203  1.034895  1.045194    1.0551  1.064614  1.073737   1.08247  1.090815  1.098775  1.106349   1.11354  1.120349  1.126777  1.132824   1.13849  1.143775  1.148679    1.1532  1.157338   1.16109  1.164454  1.167427  1.170007  1.172188  1.173968  1.175341  1.176303  1.176848   1.17697  1.176662  1.175917  1.174728  1.173087  1.170986  1.168417   1.16537  1.161837       1.157808 | 3.483779  3.574984  3.666217  3.756989  3.846898  3.935617  4.022882   4.10848  4.192241  4.274027  4.353728  4.431255  4.506538  4.579519  4.650152  4.718398  4.784226  4.847609  4.908525  4.966952  5.022871  5.076266  5.127119  5.175412  5.221128  5.264249  5.304754  5.342622  5.377833   5.41036   5.44018  5.467264  5.491583  5.513105  5.531796  5.547621  5.560542  5.570519  5.577509  5.581469  5.582353  5.580113  5.574699  5.566061  5.554147  5.538903  5.520276  5.498213  5.472661        5.443567 |  197.47692  210.602151  224.324105  238.572059  253.277744  268.375546  283.802524   299.49833  315.405079  331.467181  347.631174  363.845556  380.060624   396.22833  412.302146  428.236946  443.988899  459.515376  474.774871  489.726926  504.332075  518.551792  532.348452  545.685296  558.526411   570.83671  582.581927  593.728618   604.24417  614.096819  623.255671  631.690748  639.373022  646.274478  652.368174  657.628324  662.030381  665.551144  668.168866  669.863392  670.616294  670.411038  669.233151   667.07042    663.9131  659.754137  654.589413  648.418012   641.24249  633.069172  623.908461 |  100.0 |   97.47692 | 623.908461 |
    | 15:00 | 5.032754  5.178664  5.325995  5.473826  5.621379  5.768001  5.913147  6.056366  6.197282  6.335585  6.471017  6.603365  6.732451  6.858127  6.980268  7.098769   7.21354  7.324503  7.431592  7.534744  7.633906  7.729028  7.820061  7.906962  7.989685  8.068187  8.142424  8.212351  8.277921  8.339087  8.395798  8.448004  8.495648  8.538674  8.577022  8.610628  8.639426  8.663347  8.682319  8.696267  8.705112  8.708773  8.707167  8.700208  8.687808  8.669877  8.646324  8.617059  8.581991              8.54103 | 202.134363  211.772752  221.721006  231.921217  242.320169  252.869176  263.523777  274.243363  284.990785  295.731972  306.435569  317.072612  327.616234  338.041405  348.324706  358.444126  368.378889  378.109307  387.616642  396.882992  405.891197  414.624745  423.067703  431.204648   439.02061  446.501028  453.631705  460.398778  466.788684  472.788141  478.384133  483.563893  488.314902  492.624887  496.481825  499.873952  502.789782  505.218129  507.148135  508.569308  509.471562  509.845275  509.681342  508.971249  507.707154  505.881973  503.489486  500.524448   496.98272  492.861408 |  65.327538   66.786638   68.259945   69.738256   71.213786   72.680008   74.131475   75.563664   76.972824   78.355851   79.710171   81.033649   82.324508   83.581266   84.802678   85.987688   87.135399   88.245035   89.315916    90.34744    91.33906   92.290275   93.200612   94.069618    94.89685   95.681872    96.42424   97.123507   97.779209   98.390868   98.957984   99.480037   99.956481  100.386741  100.770216  101.106276  101.394258  101.633472  101.823193   101.96267  102.051121  102.087734  102.071675  102.002083  101.878079  101.698768  101.463243  101.170593  100.819911    100.410299 | 1.281498  1.301803  1.322114  1.342308  1.362288  1.381971  1.401297  1.420215  1.438688  1.456687   1.47419  1.491181  1.507648  1.523582  1.538979  1.553835  1.568149  1.581918  1.595143  1.607825  1.619965  1.631563   1.64262  1.653137  1.663114  1.672552   1.68145  1.689808  1.697625    1.7049  1.711629  1.717811  1.723443   1.72852  1.733038  1.736992  1.740377  1.743185  1.745411  1.747047  1.748083  1.748512  1.748324  1.747509  1.746055  1.743952  1.741187  1.737748  1.733623  1.728797 | 1.349488  1.348756  1.348065  1.347416  1.346809  1.346242  1.345714  1.345224  1.344768  1.344345  1.343952  1.343588  1.343249  1.342935  1.342643  1.342372   1.34212  1.341886  1.341669  1.341467  1.341279  1.341105  1.340943  1.340793  1.340655  1.340526  1.340408  1.340299  1.340199  1.340107  1.340024  1.339949  1.339881  1.339821  1.339768  1.339721  1.339682   1.33965  1.339625  1.339606  1.339595   1.33959  1.339592  1.339601  1.339618  1.339642  1.339673  1.339713  1.339761          1.339817 | 0.619247  0.626503  0.633529  0.640293  0.646779  0.652976  0.658883  0.664501  0.669837  0.674899  0.679696  0.684238  0.688537  0.692603  0.696448   0.70008  0.703511  0.706749  0.709803  0.712681  0.715392  0.717941  0.720335  0.722581  0.724683  0.726647  0.728478  0.730178  0.731752  0.733203  0.734533  0.735746  0.736842  0.737824  0.738692  0.739448  0.740092  0.740625  0.741046  0.741354  0.741549   0.74163  0.741595  0.741441  0.741167   0.74077  0.740246  0.739592  0.738804      0.737877 | -0.302719  -0.311004  -0.318999  -0.326678  -0.334028  -0.341043  -0.347725  -0.354079  -0.360114  -0.365839  -0.371266  -0.376407  -0.381274  -0.385878  -0.390231  -0.394344  -0.398228  -0.401892  -0.405345  -0.408595  -0.411651  -0.414521  -0.417209  -0.419723  -0.422067  -0.424248  -0.426268  -0.428133  -0.429845  -0.431407  -0.432822  -0.434092  -0.435219  -0.436203  -0.437045  -0.437746  -0.438306  -0.438724  -0.438999  -0.439131  -0.439117  -0.438955  -0.438642  -0.438176  -0.437554   -0.43677  -0.435822  -0.434703  -0.433408     -0.431932 |  0.67749  0.678561  0.679612  0.680639  0.681638  0.682606  0.683543  0.684447  0.685317  0.686154  0.686957  0.687727  0.688463  0.689166  0.689837  0.690476  0.691084  0.691661  0.692207  0.692723  0.693209  0.693667  0.694095  0.694495  0.694866  0.695209  0.695525  0.695812  0.696072  0.696305   0.69651  0.696687  0.696837  0.696958  0.697052  0.697118  0.697155  0.697163  0.697143  0.697092  0.697012  0.696901  0.696759  0.696586   0.69638  0.696142  0.695869  0.695563  0.695221      0.694844 | 185.861607 |  191.95039  204.400392   217.45267  231.042094   245.10516  259.580335  274.408195   289.53145  304.894883  320.445254  336.131179  351.903002  367.712665  383.513591  399.260567  414.909643  430.418036  445.744051  460.847004  475.687161  490.225681  504.424572  518.246651  531.655514  544.615513  557.091741  569.050022  580.456913  591.279708  601.486453  611.045969  619.927882  628.102661  635.541671  642.217224  648.102655  653.172396  657.402072  660.768599  663.250307  664.827064  665.480428  665.193801  663.952609  661.744496  658.559527  654.390423    649.2328  643.085429          635.950517 | 632.161886 | 0.854656  0.868669  0.882674  0.896588  0.910344  0.923886  0.937173  0.950172  0.962857   0.97521  0.987216  0.998865   1.01015  1.021065  1.031608  1.041777   1.05157  1.060989  1.070032  1.078702  1.086998  1.094923  1.102476  1.109659  1.116471  1.122915  1.128989  1.134693  1.140027   1.14499  1.149581  1.153798  1.157639  1.161102  1.164183  1.166879  1.169187  1.171102  1.172619  1.173734  1.174441  1.174734  1.174605  1.174049  1.173058  1.171624  1.169739  1.167395  1.164582       1.161291 | 3.398096   3.48613  3.574778  3.663498  3.751835  3.839412  3.925919  4.011104   4.09476  4.176719  4.256843   4.33502  4.411161   4.48519  4.557045  4.626678  4.694044  4.759109  4.821841  4.882213  4.940202  4.995786  5.048942   5.09965  5.147891  5.193644  5.236888  5.277601   5.31576  5.351341  5.384319  5.414665  5.442352  5.467348   5.48962  5.509134  5.525853  5.539739  5.550751  5.558845  5.563978  5.566103  5.565171  5.561132  5.553936  5.543529  5.529858  5.512868  5.492506        5.468716 | 185.861607  197.995248  210.764143   224.10243   237.94566  252.231246  266.898683  281.889616  297.147809   312.61906   328.25109  343.993408   359.79719   375.61515  391.401419  407.111441  422.701874  438.130497   453.35614  468.338609   483.03863  497.417797  511.438532  525.064048  538.258325  550.986088  563.212798  574.904643  586.028546  596.552173   606.44395  615.673091  624.209632  632.024476  639.089443  645.377338  650.862021  655.518493  659.322996  662.253119  664.287923   665.40808  665.596023  664.836116  663.114838  660.420984  656.745881  652.083627  646.431341  639.789432  632.161886 |  100.0 |  85.861607 | 632.161886 |
    | 15:30 |  4.90009  5.039705  5.181686  5.325052  5.468955  5.612673  5.755596  5.897208  6.037078  6.174842  6.310199  6.442894  6.572716  6.699484  6.823049   6.94328  7.060068  7.173319  7.282949  7.388885  7.491062   7.58942  7.683904  7.774462  7.861044  7.943602  8.022088  8.096455  8.166653  8.232635  8.294348  8.351741  8.404759  8.453345  8.497441  8.536982  8.571907  8.602146  8.627629  8.648282   8.66403  8.674793   8.68049  8.681034  8.676341   8.66632  8.650881  8.629932  8.603381             8.571134 | 193.555751   202.58874   211.97467  221.656652  231.581653   241.70062  251.968389  262.343457  272.787688  283.265995  293.746026  304.197858  314.593717   324.90773  335.115694  345.194879   355.12385  364.882309  374.450962  383.811398  392.945987  401.837785  410.470457  418.828205  426.895711  434.658082  442.100806  449.209712  455.970938  462.370905  468.396295  474.034035  479.271284   484.09543  488.494084  492.455088  495.966524  499.016726  501.594304  503.688168  505.287562  506.382108   506.96185  507.017313  506.539567  505.520304  503.951921  501.827617  499.141502  495.888717 |  64.000899   65.397055   66.816865   68.250517   69.689548   71.126735   72.555963   73.972084   75.370775    76.74842   78.101988   79.428944    80.72716   81.994845   83.230486   84.432799   85.600683    86.73319    87.82949   88.888852   89.910621   90.894201   91.839039   92.744618   93.610439   94.436019   95.220881   95.964547   96.666534   97.326348   97.943483   98.517413   99.047593   99.533455   99.974405  100.369825  100.719067  101.021456  101.276286  101.482823  101.640303  101.747934  101.804897  101.810345  101.763409  101.663199   101.50881  101.299321  101.033806    100.711342 | 1.262866   1.28247  1.302221  1.321984  1.341646  1.361114  1.380312  1.399182  1.417676  1.435756  1.453393  1.470566  1.487256  1.503452  1.519144  1.534326  1.548993  1.563141  1.576769  1.589876  1.602462  1.614525  1.626067  1.637086  1.647584   1.65756  1.667013  1.675943  1.684349  1.692229  1.699581  1.706403  1.712691  1.718443  1.723654   1.72832  1.732436  1.735994  1.738991  1.741417  1.743266  1.744528  1.745197  1.745261   1.74471  1.743534  1.741722  1.739261   1.73614  1.732345 | 1.350198  1.349452  1.348742  1.348069  1.347437  1.346843  1.346289  1.345771  1.345288  1.344839  1.344421  1.344032  1.343671  1.343334  1.343021   1.34273   1.34246  1.342208  1.341973  1.341755  1.341552  1.341363  1.341187  1.341024  1.340872  1.340731  1.340601  1.340481   1.34037  1.340268  1.340174  1.340089  1.340011  1.339941  1.339878  1.339823  1.339775  1.339733  1.339698   1.33967  1.339649  1.339635  1.339627  1.339627  1.339633  1.339646  1.339667  1.339695  1.339731          1.339776 | 0.612375    0.6196   0.62665  0.633485  0.640075  0.646404  0.652461  0.658245  0.663756     0.669  0.673982  0.678712  0.683198  0.687451   0.69148  0.695295  0.698904  0.702318  0.705545  0.708593   0.71147  0.714183  0.716738  0.719141  0.721399  0.723516  0.725498  0.727347  0.729069  0.730667  0.732143  0.733501  0.734742  0.735869  0.736883  0.737785  0.738576  0.739258  0.739829   0.74029   0.74064  0.740879  0.741005  0.741017  0.740913  0.740691  0.740348   0.73988  0.739285      0.738559 | -0.294634  -0.302916  -0.310965  -0.318743  -0.326226  -0.333401  -0.340263   -0.34681  -0.353048  -0.358983  -0.364623  -0.369979   -0.37506  -0.379879  -0.384445  -0.388768   -0.39286  -0.396729  -0.400384  -0.403834  -0.407087  -0.410151  -0.413031  -0.415735  -0.418267  -0.420634  -0.422839  -0.424888  -0.426783  -0.428528  -0.430125  -0.431578  -0.432888  -0.434057  -0.435085  -0.435974  -0.436724  -0.437334  -0.437806  -0.438136  -0.438326  -0.438372  -0.438272  -0.438025  -0.437628  -0.437077  -0.436367  -0.435496  -0.434459     -0.433249 | 0.676423  0.677474  0.678512  0.679533  0.680532  0.681506  0.682451  0.683367  0.684253  0.685107  0.685929   0.68672  0.687479  0.688206  0.688903  0.689568  0.690202  0.690806  0.691381  0.691926  0.692442  0.692929  0.693388  0.693819  0.694222  0.694597  0.694945  0.695266   0.69556  0.695827  0.696067   0.69628  0.696466  0.696624  0.696756   0.69686  0.696937  0.696985  0.697006  0.696997   0.69696  0.696894  0.696797   0.69667  0.696512  0.696323  0.696101  0.695846  0.695557      0.695234 | 175.431719 | 181.036389  192.532895  204.663291  217.367591  230.586404  244.261514  258.336212  272.755454  287.465913  302.415961  317.555605  332.836401  348.211361  363.634849   379.06249  394.451076  409.758483  424.943592  439.966223   454.78707  469.367654   483.67027  497.657953  511.294442   524.54416  537.372191  549.744271  561.626778   572.98674  583.791837  594.010419  603.611527  612.564925  620.841136  628.411491  635.248184  641.324338   646.61408   651.09263  654.736402  657.523109  659.431895  660.443471  660.540265  659.706599  657.928868   655.19574  651.498382  646.830686          641.189529 | 638.149957 | 0.841787  0.855327  0.868957  0.882585  0.896132  0.909536  0.922745   0.93572  0.948427  0.960844  0.972949   0.98473  0.996174  1.007275  1.018025  1.028422  1.038462  1.048144  1.057467  1.066431  1.075036  1.083281  1.091168  1.098696  1.105867  1.112679  1.119134   1.12523  1.130967  1.136345  1.141361  1.146016  1.150306  1.154229  1.157783  1.160965  1.163772  1.166199  1.168242  1.169896  1.171156  1.172018  1.172473  1.172517  1.172141   1.17134  1.170104  1.168426  1.166298        1.16371 | 3.317836  3.402296  3.487951  3.574212  3.660579  3.746629   3.83201  3.916428  3.999641  4.081449  4.161685  4.240215  4.316927  4.391726  4.464537  4.535295  4.603946  4.670444   4.73475   4.79683  4.856654  4.914193  4.969423   5.02232   5.07286  5.121021  5.166779  5.210113  5.250997  5.289407  5.325318  5.358701  5.389528   5.41777  5.443393  5.466365  5.486649  5.504209  5.519005  5.530995  5.540136  5.546383  5.549689  5.550005  5.547281  5.541464  5.532503  5.520342  5.504926        5.486201 | 175.431719  186.597275  198.426867  210.860358  223.837853  237.300423  251.190536  265.452283  280.031471  294.875633  309.933973  325.157297  340.497911  355.909529   371.34717  386.767064  402.126565  417.384072  432.498951  447.431474  462.142762  476.594734  490.750063  504.572145  518.025066  531.073584  543.683111  555.819704  567.450066  578.541547  589.062157  598.980585  608.266224  616.889206  624.820445  632.031686  638.495569  644.185696  649.076717  653.144419  656.365834  658.719354  660.184866  660.743897  660.379773  659.077798  656.825446  653.612574  649.431641  644.277961  638.149957 |  100.0 |  75.431719 | 638.149957 |
    | 16:00 | 4.776796  4.909583  5.045636  5.183935  5.323582  5.463796  5.603905  5.743337  5.881603  6.018291  6.153053  6.285593  6.415663  6.543053  6.667581  6.789096  6.907466  7.022577  7.134331   7.24264  7.347427  7.448622  7.546161  7.639985  7.730037  7.816264  7.898614  7.977036  8.051479  8.121892  8.188222  8.250417  8.308421  8.362177  8.411628  8.456712  8.497365  8.533521  8.565112  8.592066  8.614308  8.631762  8.644346  8.651979  8.654574  8.652044    8.6443  8.631249  8.612799             8.588856 | 185.740833   194.16377  202.976735  212.124926   221.55635  231.222274  241.077365  251.079639  261.190296   271.37349  281.596065  291.827297  302.038637   312.20347  322.296902  332.295557  342.177405    351.9216  361.508347  370.918776  380.134833  389.139187  397.915149  406.446591  414.717891   422.71387  430.419746  437.821091  444.903797   451.65404   458.05826  464.103139  469.775581  475.062708  479.951849  484.430537  488.486517   492.10775   495.28243  497.998999  500.246174   502.01298  503.288784  504.063341   504.32685  504.070013  503.284105  501.961056   500.09354  497.675078 |  62.767959    64.09583   65.456357   66.839349   68.235819    69.63796   71.039055   72.433368   73.816028   75.182909   76.530525   77.855931   79.156634   80.430525   81.675811   82.890959   84.074658   85.225771   86.343309   87.426401   88.474271   89.486221   90.461611   91.399846   92.300367    93.16264   93.986142   94.770364   95.514794   96.218921   96.882223   97.504168   98.084207   98.621773    99.11628   99.567117   99.973648  100.335213  100.651123  100.920662  101.143084  101.317617   101.44346  101.519785  101.545738  101.520443  101.442999  101.312492  101.127992     100.88856 | 1.245399  1.264205  1.283298  1.302533  1.321783  1.340944  1.359931  1.378672   1.39711    1.4152  1.432905  1.450196  1.467051  1.483452  1.499385  1.514842  1.529813  1.544293  1.558278  1.571766  1.584754  1.597241  1.609226  1.620707  1.631685  1.642159  1.652128   1.66159  1.670546  1.678992  1.686927  1.694349  1.701255  1.707641  1.713505  1.718841  1.723645  1.727912  1.731635  1.734809  1.737425  1.739476  1.740955  1.741851  1.742156  1.741859  1.740949  1.739416  1.737247  1.734431 |   1.3509  1.350146  1.349422  1.348731  1.348076  1.347459  1.346879  1.346335  1.345826  1.345352  1.344909  1.344495  1.344111  1.343752  1.343418  1.343106  1.342816  1.342545  1.342293  1.342058  1.341839  1.341635  1.341445  1.341268  1.341103   1.34095  1.340808  1.340676  1.340553   1.34044  1.340336   1.34024  1.340153  1.340073  1.340001  1.339936  1.339878  1.339828  1.339784  1.339747  1.339716  1.339693  1.339676  1.339666  1.339662  1.339665  1.339676  1.339693  1.339719          1.339751 | 0.605737  0.612876    0.6199   0.62676  0.633416  0.639843  0.646024  0.651951  0.657618  0.663027  0.668182  0.673089  0.677754  0.682185  0.686393  0.690384  0.694169  0.697756  0.701152  0.704368  0.707409  0.710283  0.712996  0.715556  0.717968  0.720236  0.722367  0.724364  0.726232  0.727974  0.729594  0.731094  0.732478  0.733746  0.734902  0.735947  0.736881  0.737706  0.738423  0.739031  0.739531  0.739921  0.740202  0.740372   0.74043  0.740374  0.740201   0.73991  0.739497      0.738959 | -0.286802  -0.295019  -0.303066  -0.310894   -0.31847   -0.32577  -0.332781  -0.339497  -0.345917  -0.352043  -0.357882   -0.36344  -0.368726   -0.37375  -0.378521  -0.383048  -0.387342  -0.391411  -0.395264   -0.39891  -0.402356   -0.40561  -0.408679  -0.411569  -0.414285  -0.416835  -0.419222   -0.42145  -0.423524  -0.425448  -0.427224  -0.428855  -0.430344  -0.431691    -0.4329  -0.433971  -0.434904  -0.435701   -0.43636  -0.436882  -0.437266  -0.437511  -0.437615  -0.437576  -0.437392   -0.43706  -0.436577   -0.43594  -0.435143     -0.434184 | 0.675409  0.676433  0.677453  0.678462  0.679455  0.680428  0.681377  0.682301  0.683197  0.684065  0.684904  0.685712  0.686491   0.68724  0.687958  0.688647  0.689306  0.689936  0.690536  0.691108  0.691652  0.692167  0.692655  0.693115  0.693548  0.693954  0.694333  0.694686  0.695012  0.695311  0.695584  0.695831  0.696052  0.696246  0.696413  0.696554  0.696668  0.696755  0.696814  0.696846   0.69685  0.696825  0.696772  0.696689  0.696576  0.696433  0.696259  0.696053  0.695815      0.695544 | 166.099094 | 171.235081  181.804644   193.03065  204.858981  217.235001  230.104368  243.413557  257.110181  271.143157  285.462777  300.020714  314.769986  329.664899  344.660975  359.714876  374.784332  389.828063   404.80571  419.677779  434.405575  448.951158  463.277294  477.347421  491.125612  504.576552  517.665515  530.358352  542.621479  554.421875  565.727086  576.505233  586.725028  596.355796  605.367505  613.730805  621.417068  628.398448   634.64794  640.139455  644.847902  648.749286  651.820809  654.040994  655.389817  655.848853  655.401437  654.032844  651.730481  648.484096          644.286008 | 641.971983 | 0.829713  0.842712  0.855899  0.869172  0.882446  0.895649  0.908722  0.921617  0.934295  0.946727  0.958886  0.970756  0.982319  0.993566  1.004488  1.015078  1.025331  1.035245  1.044817  1.054045  1.062928  1.071466  1.079659  1.087506  1.095007  1.102162   1.10897  1.115431  1.121545  1.127311  1.132726  1.137791  1.142503  1.146861  1.150861  1.154501  1.157777  1.160687  1.163226   1.16539  1.167174  1.168573  1.169581  1.170192    1.1704  1.170197  1.169577  1.168532  1.167053       1.165132 | 3.243046  3.323586  3.405878  3.489305  3.573329  3.657486  3.741385  3.824693  3.907134  3.988474  4.068519   4.14711   4.22411  4.299409  4.372911  4.444539  4.514226  4.581916  4.647559  4.711113  4.772542  4.831812  4.888892  4.943756  4.996375  5.046725  5.094781  5.140517  5.183908   5.22493  5.263555  5.299756  5.333504   5.36477  5.393521  5.419726  5.443349  5.464354  5.482704  5.498356  5.511271  5.521404   5.52871   5.53314  5.534647  5.533178  5.528683  5.521106  5.510395        5.496492 | 166.099094  176.327832  187.239944   198.78178  210.898691     223.536  236.639659  250.156643  264.035183   278.22487  292.676688  307.342992  322.177459  337.135022  352.171793  367.244985  382.312841  397.334559  412.270227  427.080765  441.727868   456.17396  470.382151  484.316205  497.940507  511.220042  524.120374  536.607638  548.648533  560.210319  571.260826  581.768466  591.702251  601.031817  609.727463  617.760186  625.101735  631.724663  637.602405  642.709346  647.020915  650.513685  653.165484  654.955525  655.864539  655.874935  654.970968  653.138924  650.367318   646.64712  641.971983 |  100.0 |  66.099094 | 641.971983 |
    | 16:30 | 4.662803  4.788331  4.917961  5.050661  5.185499   5.32165   5.45839  5.595094  5.731223  5.866316  5.999978  6.131875   6.26172   6.38927  6.514316  6.636679  6.756205  6.872761  6.986231  7.096513  7.203516  7.307159  7.407368  7.504077  7.597221  7.686742  7.772582  7.854686  7.933001  8.007472  8.078045  8.144667  8.207281   8.26583  8.320256  8.370499  8.416496  8.458182  8.495489  8.528349  8.556687   8.58043  8.599498  8.613811  8.623285  8.627835  8.627372  8.621805  8.611043             8.594991 | 178.650675  186.465533  194.701133  203.305811   212.22951  221.424536  230.845967  240.451799  250.202927  260.063026  269.998363  279.977589   289.97152  299.952919  309.896293  319.777704  329.574595  339.265636  348.830584  358.250159  367.505934  376.580237  385.456062  394.116998  402.547158  410.731121  418.653877  426.300788  433.657542   440.71012  447.444773  453.847988  459.906478   465.60716  470.937147  475.883742  480.434433  484.576901  488.299021  491.588879  494.434786  496.825303  498.749268  500.195836  501.154514  501.615215  501.568316  501.004724  499.915948  498.294189 |  61.628025    62.88331   64.179612   65.506612   66.854994   68.216499   69.583901   70.950941    72.31223   73.663156   74.999782   76.318751   77.617204   78.892702   80.143159   81.366787   82.562049    83.72761    84.86231   85.965128   87.035158   88.071589   89.073684   90.040768   90.972211   91.867418   92.725819   93.546864    94.33001    95.07472   95.780454   96.446668   97.072805   97.658298   98.202561   98.704989   99.164957   99.581816   99.954892  100.283486  100.566872  100.804296  100.994977  101.138107  101.232852  101.278351   101.27372  101.218054  101.110429    100.949907 | 1.229114  1.247039  1.265386     1.284  1.302749  1.321517  1.340208  1.358741  1.377049  1.395079  1.412784  1.430131  1.447091   1.46364  1.479761   1.49544  1.510667  1.525432  1.539729  1.553553  1.566901   1.57977  1.592157  1.604061   1.61548  1.626413  1.636858  1.646814  1.656281  1.665255  1.673735  1.681719  1.689203  1.696185  1.702662  1.708629  1.714082  1.719015  1.723424  1.727302  1.730643  1.733439  1.735683  1.737366   1.73848  1.739015   1.73896  1.738306  1.737041  1.735153 | 1.351587  1.350832    1.3501  1.349396  1.348723  1.348085  1.347482  1.346914  1.346381  1.345881  1.345414  1.344977  1.344568  1.344187  1.343831  1.343499  1.343189    1.3429  1.342629  1.342377  1.342142  1.341922  1.341717  1.341526  1.341348  1.341182  1.341027  1.340883  1.340749  1.340625   1.34051  1.340404  1.340307  1.340217  1.340135  1.340061  1.339994  1.339934  1.339881  1.339835  1.339796  1.339763  1.339737  1.339717  1.339704  1.339698  1.339699  1.339706  1.339721          1.339743 | 0.599372  0.606369  0.613316  0.620154  0.626836  0.633326    0.6396  0.645642  0.651444  0.657001  0.662313  0.667384  0.672216  0.676819  0.681197   0.68536  0.689315   0.69307  0.696633  0.700012  0.703215  0.706248  0.709118  0.711832  0.714395  0.716814  0.719092  0.721235  0.723246  0.725131  0.726892  0.728533  0.730055  0.731463  0.732758  0.733941  0.735015  0.735981  0.736838  0.737589  0.738232  0.738769  0.739198  0.739519  0.739732  0.739833  0.739823  0.739699  0.739457      0.739097 | -0.279266  -0.287355  -0.295344  -0.303173  -0.310796  -0.318183  -0.325311  -0.332167  -0.338746  -0.345044  -0.351064   -0.35681  -0.362289  -0.367507  -0.372474  -0.377198  -0.381687   -0.38595  -0.389996  -0.393832  -0.397467  -0.400908  -0.404162  -0.407234  -0.410132  -0.412861  -0.415425   -0.41783   -0.42008  -0.422178  -0.424129  -0.425934  -0.427596  -0.429119  -0.430503   -0.43175   -0.43286  -0.433836  -0.434677  -0.435383  -0.435954  -0.436389  -0.436687  -0.436847  -0.436866  -0.436743  -0.436474  -0.436057  -0.435489     -0.434765 | 0.674454  0.675445  0.676439   0.67743  0.678411  0.679377  0.680325  0.681252  0.682155  0.683033  0.683884  0.684707  0.685502  0.686269  0.687008  0.687718  0.688399  0.689052  0.689677  0.690274  0.690843  0.691385    0.6919  0.692387  0.692849  0.693283  0.693692  0.694074  0.694431  0.694761  0.695066  0.695345  0.695598  0.695825  0.696027  0.696203  0.696352  0.696475  0.696572  0.696642  0.696685    0.6967  0.696688  0.696647  0.696577  0.696478   0.69635   0.69619     0.696      0.695778 | 157.776495 | 162.462441  172.138219  182.484301  193.453069  204.995216  217.060787  229.599895  242.563196  255.902187  269.569372  283.518346  297.703822   312.08161  326.608583  341.242623  355.942571  370.668156  385.379948  400.039296  414.608276  429.049649  443.326814  457.403773  471.245102  484.815916  498.081857  511.009071  523.564198  535.714365  547.427187  558.670771  569.413722  579.625165  589.274762  598.332745  606.769949   614.55786  621.668659  628.075291  633.751528  638.672055  642.812554  646.149811  648.661826  650.327943   651.12899  651.047432  650.067544  648.175593          645.360045 | 643.742834 | 0.818447  0.830847  0.843528  0.856384  0.869322  0.882263  0.895142  0.907903  0.920501  0.932899  0.945067  0.956982  0.968624   0.97998  0.991036  1.001784  1.012218  1.022332  1.032121  1.041583  1.050717   1.05952   1.06799  1.076129  1.083933  1.091404   1.09854  1.105341  1.111806  1.117933  1.123722  1.129172   1.13428  1.139045  1.143464  1.147534  1.151254  1.154619  1.157626  1.160271  1.162549  1.164456  1.165986  1.167134  1.167894  1.168258  1.168221  1.167775  1.166912       1.165625 |  3.17372  3.250052   3.32866  3.408914  3.490248  3.572167  3.654245  3.736114  3.817463  3.898027  3.977585   4.05595  4.132964  4.208495  4.282432  4.354681  4.425161  4.493804  4.560551  4.625353  4.688163  4.748943  4.807656   4.86427  4.918755  4.971082  5.021222   5.06915  5.114838   5.15826  5.199388  5.238194   5.27465  5.308725  5.340389  5.369609  5.396351   5.42058  5.442259   5.46135   5.47781  5.491599  5.502672  5.510982  5.516483  5.519124  5.518856  5.515624  5.509375        5.500054 | 157.776495  167.106073  177.129349  187.799808  199.068651  210.886007   223.20179  235.966278  249.130483  262.646368  276.466966  290.546429  304.840021  319.304095  333.896041  348.574236  363.297979  378.027434  392.723573  407.348121   421.86351  436.232828  450.419786  464.388679   478.10436  491.532212  504.638133  517.388517   529.75025  541.690702  553.177732  564.179692  574.665443  584.604371  593.966415  602.722097  610.842564  618.299636  625.065858  631.114568  636.419971  640.957226  644.702536  647.633262   649.72804  650.966916  651.331491  650.805086  649.372919  647.022299  643.742834 |  100.0 |  57.776495 | 643.742834 |
    | 17:00 | 4.557952  4.675901  4.798705  4.925347  5.054883  5.186458  5.319312  5.452773  5.586256  5.719256  5.851333  5.982114  6.111275  6.238538  6.363666  6.486453  6.606721  6.724317  6.839106  6.950971  7.059806  7.165519  7.268026  7.367249  7.463118  7.555567  7.644534  7.729959  7.811784  7.889952  7.964406  8.035092  8.101951  8.164927  8.223961  8.278992  8.329959  8.376798  8.419443  8.457825  8.491874  8.521516  8.546674  8.567271  8.583225  8.594451  8.600863  8.602373   8.59889             8.590319 | 172.243889  179.458735   187.11841   195.17539  203.582431  212.293627   221.26508  230.455258  239.825147  249.338254  258.960515  268.660157  278.407515  288.174858  297.936199  307.667125  317.344628  326.946956  336.453475  345.844548  355.101418  364.206115  373.141363  381.890504  390.437426  398.766504  406.862544  414.710731  422.296594  429.605964  436.624942  443.339875  449.737329  455.804075   461.52707  466.893447  471.890513  476.505742  480.726776  484.541439  487.937737  490.903885  493.428319  495.499729  497.107088  498.239694  498.887214  499.039741  498.687855  497.822693 |  60.579516   61.759005   62.987048   64.253465   65.548826   66.864584    68.19312   69.527729   70.862564   72.192556   73.513334   74.821141   76.112747   77.385381   78.636658   79.864526    81.06721   82.243171   83.391064   84.509709   85.598063   86.655192   87.680256   88.672488   89.631181   90.555674   91.445344   92.299592   93.117839   93.899517   94.644064   95.350918   96.019512   96.649271   97.239608   97.789923   98.299594   98.767985   99.194433   99.578255   99.918741  100.215157  100.466743   100.67271  100.832246  100.944509  101.008635  101.023734  100.988896    100.903191 | 1.214017  1.230992  1.248514  1.266426   1.28459  1.302882  1.321196  1.339444  1.357548  1.375446  1.393086  1.410425   1.42743  1.444071  1.460326  1.476178   1.49161  1.506613  1.521177  1.535294   1.54896  1.562169  1.574918  1.587204  1.599025  1.610379  1.621263  1.631676  1.641616  1.651081  1.660068  1.668576  1.676602  1.684142  1.691194  1.697753  1.703815  1.709376  1.714431  1.718973  1.722997  1.726496  1.729463   1.73189  1.733768  1.735089  1.735844  1.736021  1.735611  1.734603 | 1.352254  1.351506  1.350772   1.35006  1.349374  1.348719  1.348096  1.347506   1.34695  1.346426  1.345935  1.345475  1.345043   1.34464  1.344262  1.343909  1.343579   1.34327  1.342982  1.342712   1.34246  1.342225  1.342004  1.341799  1.341606  1.341427  1.341259  1.341103  1.340958  1.340822  1.340697   1.34058  1.340472  1.340372  1.340281  1.340197  1.340121  1.340052   1.33999  1.339935  1.339886  1.339845  1.339809  1.339781  1.339759  1.339744  1.339735  1.339733  1.339738          1.339749 | 0.593313  0.600115  0.606935  0.613704  0.620368  0.626882  0.633216  0.639347  0.645258  0.650943  0.656394  0.661614  0.666603  0.671365  0.675906  0.680233  0.684351   0.68827  0.691995  0.695535  0.698896  0.702086   0.70511  0.707976  0.710689  0.713255  0.715679  0.717965  0.720119  0.722144  0.724044  0.725823  0.727483  0.729027  0.730458  0.731778  0.732987  0.734089  0.735084  0.735972  0.736755  0.737433  0.738005  0.738472  0.738832  0.739085  0.739229  0.739263  0.739185      0.738992 | -0.272066  -0.279968  -0.287842  -0.295618  -0.303243  -0.310675  -0.317884   -0.32485   -0.33156  -0.338008   -0.34419  -0.350108  -0.355765  -0.361167   -0.36632  -0.371231  -0.375908  -0.380359  -0.384591  -0.388614  -0.392433  -0.396056   -0.39949  -0.402742  -0.405817  -0.408721   -0.41146  -0.414038   -0.41646  -0.418729   -0.42085  -0.422825  -0.424657   -0.42635  -0.427904  -0.429322  -0.430605  -0.431755  -0.432771  -0.433654  -0.434405  -0.435023  -0.435507  -0.435856  -0.436069  -0.436145   -0.43608  -0.435872  -0.435519     -0.435017 |  0.67356   0.67451  0.675473  0.676439  0.677403  0.678358    0.6793  0.680225   0.68113  0.682014  0.682873  0.683708  0.684517  0.685299  0.686055  0.686783  0.687484  0.688158  0.688805  0.689425  0.690018  0.690584  0.691124  0.691638  0.692126  0.692588  0.693024  0.693434  0.693819  0.694179  0.694514  0.694824  0.695108  0.695367  0.695601   0.69581  0.695994  0.696151  0.696283  0.696389  0.696469  0.696522  0.696549  0.696548  0.696519  0.696462  0.696376  0.696261  0.696117      0.695941 | 150.378786 | 154.635896  163.456402  172.952867  183.084668   193.80833  205.078748  216.850087  229.076409  241.712108  254.712175  268.032372  281.629317  295.460517  309.484373  323.660153   337.94796  352.308686  366.703968  381.096142  395.448198  409.723735  423.886927  437.902486  451.735632  465.352065  478.717943  491.799868  504.564868   516.98039  529.014297  540.634866  551.810795  562.511215  572.705701  582.364303  591.457565  599.956565  607.832955  615.059014  621.607699  627.452718  632.568605  636.930804  640.515765   643.30106  645.265495   646.38925  646.654028  646.043213          644.542053 | 643.588839 | 0.807996  0.819747  0.831867  0.844247  0.856791  0.869413  0.882042  0.894615  0.907081  0.919397  0.931529  0.943446  0.955127  0.966552  0.977706  0.988579  0.999159  1.009441  1.019417  1.029085   1.03844  1.047479  1.056201  1.064604  1.072686  1.080447  1.087886     1.095   1.10179  1.108255  1.114392  1.120201   1.12568  1.130826  1.135638  1.140114   1.14425  1.148044  1.151492   1.15459  1.157335  1.159721  1.161745    1.1634  1.164681  1.165581  1.166096  1.166217  1.165938        1.16525 | 3.109794  3.181694   3.25635  3.333132  3.411463  3.490825  3.570762  3.650877  3.730827  3.810318    3.8891  3.966961  4.043719  4.119223  4.193342  4.265967  4.337001  4.406366  4.473992  4.539818  4.603792  4.665866     4.726  4.784156  4.840298  4.894395  4.946415   4.99633   5.04411  5.089727  5.133153  5.174358  5.213315  5.249992  5.284359  5.316383  5.346033  5.373272  5.398064  5.420373  5.440159   5.45738  5.471995  5.483957  5.493222  5.499741  5.503465  5.504341  5.502318        5.497342 | 150.378786  158.851944  168.020668  177.846069  188.285711  199.295012  210.828294  222.839525  235.282835  248.112856  261.284927  274.755219  288.480788   302.41959  316.530464  330.773106  345.108023  359.496492  373.900511  388.282752  402.606522  416.835715  430.934784    444.8687  458.602931  472.103411  485.336524  498.269087   510.86834  523.101938   534.93795  546.344862  557.291584  567.747465   577.68231  587.066407  595.870558  604.066115   611.62503  618.519902  624.724048  630.211563  634.957414  638.937518  642.128857  644.509583  646.059152  646.758464  646.590013  645.538065  643.588839 |  100.0 |  50.378786 | 643.588839 |
    | 17:30 | 4.462004  4.572163  4.687834  4.808038  4.931844  5.058387   5.18688  5.316617  5.446976  5.577407  5.707435  5.836644  5.964676   6.09122  6.216007  6.338806  6.459414  6.577656  6.693379  6.806447  6.916742  7.024157  7.128597  7.229976  7.328214  7.423238   7.51498  7.603374  7.688359  7.769873  7.847859  7.922257  7.993009  8.060058  8.123343  8.182805  8.238381  8.290009  8.337624  8.381158  8.420543  8.455706  8.486573  8.513068  8.535111  8.552621  8.565513  8.573701  8.577094             8.575602 | 166.477454  173.105845  180.196427  187.706708  195.593095  203.812213  212.321816  221.081366  230.052361  239.198482  248.485605  257.881746  267.356937  276.883089  286.433839   295.98439  305.511361  314.992646  324.407281  333.735318  342.957722  352.056266  361.013443  369.812386  378.436797  386.870882  395.099296  403.107089  410.879665  418.402742  425.662313  432.644625  439.336145  445.723541  451.793665   457.53354  462.930347   467.97142  472.644244  476.936455  480.835847  484.330381    487.4082  490.057648  492.267297  494.025974    495.3228  496.147234  496.489122  496.338757 |  59.620039   60.721635   61.878337   63.080379   64.318441   65.583872   66.868799   68.166174   69.469758   70.774075   72.074351   73.366443    74.64676   75.912198   77.160072   78.388059   79.594141   80.776562   81.933788   83.064471   84.167419    85.24157   86.285971   87.299758   88.282138    89.23238   90.149797    91.03374   91.883586   92.698732   93.478587   94.222569   94.930095    95.60058   96.233431   96.828048   97.383813   97.900094   98.376241   98.811584   99.205428   99.557057    99.86573  100.130679  100.351113  100.526213  100.655134  100.737008   100.77094    100.756018 | 1.200098   1.21607  1.232701  1.249839  1.267341  1.285079   1.30294  1.320827  1.338654  1.356352  1.373861   1.39113  1.408121  1.424797  1.441133  1.457105  1.472695  1.487889  1.502674  1.517041  1.530982  1.544491  1.557562  1.570192  1.582377  1.594114    1.6054  1.616232   1.62661  1.636529  1.645988  1.654984  1.663514  1.671576  1.679165   1.68628  1.692914  1.699064  1.704726  1.709893  1.714561  1.718722  1.722371  1.725499    1.7281  1.730164  1.731683  1.732647  1.733046  1.732871 | 1.352896  1.352161  1.351433  1.350718  1.350024  1.349356  1.348717  1.348108  1.347531  1.346986  1.346472  1.345989  1.345535  1.345109   1.34471  1.344336  1.343985  1.343657   1.34335  1.343063  1.342794  1.342542  1.342306  1.342085  1.341879  1.341686  1.341505  1.341336  1.341179  1.341032  1.340895  1.340767  1.340649  1.340539  1.340438  1.340345  1.340259  1.340181  1.340109  1.340045  1.339988  1.339938  1.339894  1.339856  1.339826  1.339801  1.339783  1.339772  1.339767          1.339769 | 0.587587  0.594146  0.600789  0.607443  0.614045  0.620544  0.626903   0.63309  0.639085  0.644873  0.650445  0.655797  0.660928  0.665838  0.670532  0.675015  0.679291  0.683367  0.687249  0.690945  0.694461  0.697804   0.70098  0.703995  0.706856  0.709567  0.712135  0.714563  0.716857   0.71902  0.721058  0.722972  0.724767  0.726445   0.72801  0.729462  0.730805   0.73204  0.733168  0.734191  0.735109  0.735923  0.736634   0.73724  0.737743   0.73814  0.738432  0.738617  0.738694       0.73866 | -0.265236  -0.272893  -0.280596  -0.288268  -0.295846   -0.30328  -0.310532  -0.317575  -0.324388   -0.33096  -0.337283  -0.343354  -0.349174  -0.354746  -0.360073  -0.365162  -0.370018  -0.374649  -0.379062  -0.383264  -0.387262  -0.391063  -0.394674  -0.398101  -0.401349  -0.404426  -0.407335  -0.410083  -0.412673   -0.41511  -0.417397  -0.419538  -0.421537  -0.423395  -0.425115    -0.4267   -0.42815  -0.429468  -0.430654  -0.431709  -0.432633  -0.433427  -0.434089   -0.43462  -0.435019  -0.435283  -0.435412  -0.435404  -0.435255     -0.434963 | 0.672727  0.673633  0.674558  0.675495  0.676436  0.677374  0.678305  0.679223  0.680127  0.681012  0.681876  0.682719  0.683538  0.684332  0.685102  0.685847  0.686565  0.687258  0.687924  0.688565   0.68918  0.689769  0.690332   0.69087  0.691383   0.69187  0.692332  0.692769  0.693182  0.693569  0.693932  0.694271  0.694585  0.694874  0.695139   0.69538  0.695595  0.695786  0.695951  0.696092  0.696207  0.696296  0.696359  0.696396  0.696406  0.696389  0.696344  0.696271  0.696169      0.696038 | 143.823868 | 147.675291  155.683213  164.365128  173.687757  183.613853  194.103564  205.115476  216.607397  228.536908  240.861749  253.540072  266.530599  279.792713    293.2865  306.972761     320.813  334.769402  348.804798  362.882636  376.966937  391.022264  405.013687  418.906752  432.667448   446.26219  459.657788  472.821434  485.720689  498.323468  510.598036  522.513006  534.037338   545.14035  555.791725  565.961529  575.620233  584.738738  593.288414  601.241133  608.569322  615.246015  621.244919  626.540486  631.107993  634.923637   637.96464  640.209359  641.637422  642.229864          641.969284 | 641.644175 | 0.798353  0.809417   0.82093  0.832783  0.844879  0.857128  0.869453  0.881787  0.894071  0.906258  0.918307  0.930184  0.941862  0.953319  0.964535  0.975496  0.986191  0.996608  1.006742  1.016585  1.026132  1.035381  1.044327  1.052968  1.061303  1.069328  1.077044  1.084448  1.091539  1.098316  1.104776   1.11092  1.116745  1.122248  1.127429  1.132285  1.136812  1.141009  1.144872  1.148397  1.151581  1.154419  1.156908  1.159042  1.160815  1.162223  1.163258  1.163916  1.164188       1.164069 | 3.051157  3.118468  3.188958  3.262016  3.337066   3.41358  3.491079  3.569143  3.647401  3.725532   3.80326  3.880346  3.956587   4.03181  4.105865  4.178626  4.249983  4.319843  4.388126   4.45476  4.519684  4.582844  4.644193  4.703685  4.761283  4.816949   4.87065  4.922353  4.972026  5.019641  5.065166  5.108572  5.149829  5.188907  5.225775  5.260401  5.292752  5.322793  5.350491  5.375807  5.398704  5.419141  5.437079  5.452473  5.465278  5.475449  5.482936  5.487691  5.489662        5.488795 | 143.823868   151.48719  159.840164  168.851837  178.486528  188.705356  199.467438  210.730788  222.452973  234.591565   247.10445  259.950031  273.087341  286.476104  300.076763  313.850474  327.759086  341.765117  355.831713  369.922618  384.002134  398.035085  411.986785  425.823011  439.509969  453.014277   466.30294   479.34334  492.103214  504.550654  516.654099  528.382336    539.7045  550.590087  561.008969  570.931406  580.328077  589.170107  597.429104  605.077206  612.087126  618.432217  624.086538  629.024928  633.223099   636.65773  639.306578  641.148599  642.164082  642.334801  641.644175 |  100.0 |  43.823868 | 641.644175 |
    | 18:00 | 4.374648  4.476919  4.585244  4.698716  4.816435  4.937546  5.061251  5.186824  5.313609  5.441026  5.568561  5.695762  5.822237  5.947643  6.071682  6.194093   6.31465  6.433156  6.549437  6.663341  6.774733  6.883493  6.989514    7.0927  7.192961  7.290217  7.384393  7.475416  7.563222  7.647744  7.728922  7.806693  7.880998  7.951777   8.01897  8.082515  8.142352  8.198417  8.250647  8.298975  8.343333  8.383652  8.419858  8.451878  8.479633  8.503045  8.522031  8.536507  8.546385             8.551576 | 161.307468  167.367785  173.900995  180.870377  188.236746  195.959966   204.00007  212.318049  220.876361  229.639219  238.572738  247.644955   256.82579  266.086951  275.401819  284.745315  294.093771  303.424797  312.717156  321.950648  331.106002  340.164777  349.109274  357.922456   366.58787   375.08959  383.412151    391.5405  399.459952  407.156144  414.614999  421.822697  428.765645  435.430452  441.803908  447.872972  453.624751  459.046496  464.125593  468.849561  473.206054  477.182863  480.767928  483.949348  486.715403  489.054575  490.955576  492.407381  493.399275  493.920892 |  58.746484   59.769187   60.852444   61.987156   63.164348   64.375456   65.612509   66.868236   68.136093   69.410261   70.685605    71.95762   73.222372   74.476433   75.716817   76.940927     78.1465   79.331557   80.494367   81.633406   82.747326    83.83493   84.895144   85.926999   86.929612   87.902172   88.843925   89.754164   90.632219   91.477445   92.289219   93.066933   93.809983   94.517771   95.189696    95.82515   96.423519   96.984174   97.506471   97.989751   98.433333   98.836518   99.198581   99.518777   99.796333  100.030453  100.220315  100.365072  100.463852    100.515759 | 1.187337  1.202268  1.217958  1.234259  1.251031  1.268143  1.285479  1.302932  1.320413  1.337843  1.355156  1.372294  1.389211  1.405867   1.42223  1.438272  1.453971  1.469309  1.484271  1.498844  1.513019  1.526787  1.540142  1.553077  1.565588   1.57767  1.589322  1.600538  1.611317  1.621655   1.63155  1.640999  1.649998  1.658546  1.666638  1.674271  1.681442  1.688145  1.694376  1.700131  1.705404  1.710189   1.71448   1.71827  1.721551  1.724316  1.726557  1.728264  1.729429  1.730041 | 1.353508  1.352794  1.352077  1.351366   1.35067  1.349993  1.349341  1.348717  1.348122  1.347556  1.347022  1.346517  1.346041  1.345594  1.345173  1.344778  1.344408   1.34406  1.343734  1.343429  1.343142  1.342874  1.342622  1.342386  1.342165  1.341958  1.341764  1.341582  1.341412  1.341253  1.341105  1.340967  1.340838  1.340718  1.340606  1.340503  1.340408   1.34032   1.34024  1.340167  1.340101  1.340042  1.339989  1.339943  1.339904   1.33987  1.339844  1.339824   1.33981          1.339803 | 0.582215  0.588489   0.59491  0.601402  0.607898  0.614343  0.620689    0.6269  0.632949  0.638816  0.644487  0.649953  0.655209  0.660255   0.66509  0.669718  0.674143   0.67837  0.682404  0.686251  0.689918   0.69341  0.696735  0.699897  0.702902  0.705757  0.708465  0.711033  0.713465  0.715766  0.717938  0.719986  0.721914  0.723724   0.72542  0.727003  0.728476  0.729841    0.7311  0.732253  0.733303  0.734249  0.735093  0.735835  0.736474  0.737011  0.737445  0.737774  0.737999      0.738116 | -0.258801  -0.266163  -0.273643  -0.281159  -0.288642  -0.296033  -0.303288  -0.310372  -0.317257  -0.323926  -0.330366   -0.33657  -0.342536  -0.348262  -0.353751  -0.359006  -0.364032  -0.368835  -0.373421  -0.377796  -0.381968  -0.385941  -0.389724  -0.393321  -0.396739  -0.399984   -0.40306  -0.405974  -0.408729  -0.411329   -0.41378  -0.416083  -0.418244  -0.420264  -0.422146  -0.423893  -0.425506  -0.426987  -0.428338  -0.429558   -0.43065  -0.431614  -0.432448  -0.433154   -0.43373  -0.434176   -0.43449  -0.434671  -0.434716     -0.434624 | 0.671956  0.672813  0.673697  0.674599  0.675512  0.676429  0.677343  0.678251  0.679147   0.68003  0.680896  0.681742  0.682568  0.683372  0.684154  0.684911  0.685645  0.686354  0.687038  0.687697  0.688332  0.688941  0.689526  0.690086  0.690622  0.691133   0.69162  0.692082   0.69252  0.692934  0.693324   0.69369  0.694032   0.69435  0.694644  0.694914   0.69516  0.695382  0.695579  0.695752  0.695901  0.696024  0.696122  0.696195  0.696242  0.696262  0.696256  0.696223  0.696162      0.696073 | 138.033385 | 141.503641  148.744745  156.650926  165.196435  174.350538  184.078952  194.345012  205.110567   216.33666  227.984017  240.013389  252.385788  265.062632  278.005841  291.177878  304.541773  318.061115  331.700041  345.423215  359.195797  372.983416  386.752143  400.468462  414.099245  427.611726  440.973484  454.152422  467.116753  479.834988  492.275931  504.408669  516.202576   527.62731  538.652828  549.249391  559.387582  569.038328  578.172925  586.763073   594.78091  602.199062  608.990693  615.129566  620.590112   625.34751  629.377773  632.657849  635.165728   636.88057          637.782834 |  638.04775 | 0.789506  0.799857  0.810725  0.822007  0.833607  0.845433  0.857404  0.869448  0.881502  0.893513  0.905435  0.917229  0.928865  0.940314  0.951555  0.962571  0.973346  0.983868  0.994128  1.004117   1.01383   1.02326  1.032403  1.041257  1.049818  1.058084  1.066052  1.073721  1.081088  1.088153  1.094914  1.101369  1.107515  1.113352  1.118877  1.124088  1.128983  1.133558   1.13781  1.141737  1.145334  1.148599  1.151526  1.154111  1.156349  1.158235  1.159763  1.160927  1.161721       1.162139 | 2.997649  3.060281  3.126449   3.19558  3.267112  3.340518  3.415309  3.491045  3.567335  3.643833  3.720238  3.796288  3.871758  3.946452  4.020204  4.092868  4.164322  4.234456  4.303179   4.37041  4.436078   4.50012  4.562482  4.623113   4.68197  4.739011  4.794199  4.847497  4.898872  4.948291  4.995724  5.041138  5.084503  5.125788  5.164962  5.201992  5.236846   5.26949   5.29989  5.328009  5.353811  5.377256  5.398306  5.416917  5.433046   5.44665   5.45768  5.466089  5.471827        5.474842 | 138.033385  144.936151  152.515642   160.74896  169.607432  179.058193  189.065485  199.591686  210.598081   222.04544  233.894421  246.105852  258.640915  271.461258  284.529062  297.807069  311.258591  324.847495  338.538185  352.295581  366.085084  379.872552  393.624267  407.306915  420.887554  434.333595  447.612783   460.69318   473.54315  486.131353  498.426734  510.398524  522.016237  533.249678  544.068953  554.444479  564.347004  573.747632  582.617852   590.92957  598.655157  605.767492   612.24002   618.04682  623.162673   627.56315  631.224704  634.124771  636.241892  637.555838   638.04775 |  100.0 |  38.033385 |  638.04775 |
    | 18:30 | 4.295512  4.389899  4.490767  4.597297  4.708646  4.823987  4.942531   5.06354  5.186341  5.310326  5.434948  5.559727  5.684236  5.808102  5.930997  6.052634  6.172762  6.291161  6.407637  6.522019  6.634157  6.743916  6.851177  6.955831  7.057781  7.156936  7.253215   7.34654  7.436838  7.524041  7.608083  7.688899  7.766427  7.840606  7.911373  7.978668  8.042429  8.102592  8.159094   8.21187  8.260852  8.305973  8.347161  8.384344  8.417446   8.44639  8.471097  8.491483  8.507464             8.518952 | 156.689822  162.204565  168.196467  174.635128  181.486413  188.714069  196.281015  204.150306  212.285812  220.652668  229.217533  237.948726  246.816257  255.791797  264.848605  273.961432  283.106412  292.260945  301.403585  310.513932  319.572528  328.560762  337.460778  346.255404  354.928067  363.462738  371.843866  380.056324  388.085365  395.916579  403.535847  410.929318   418.08337  424.984587   431.61974  437.975758  444.039723  449.798849  455.240475   460.35206   465.12118  469.535525  473.582908   477.25127  480.528694  483.403418  485.863862  487.898649  489.496643  490.646979 |  57.955124   58.898992   59.907673   60.972966   62.086458   63.239872   64.425308   65.635403   66.863415   68.103255   69.349482    70.59727   71.842363   73.081023   74.309973   75.526344   76.727624   77.911609   79.076366   80.220188   81.341567   82.439161   83.511769   84.558312   85.577809   86.569364    87.53215   88.465397    89.36838    90.24041   91.080826   91.888989   92.664272   93.406058   94.113733   94.786682   95.424287   96.025918   96.590939   97.118696   97.608522   98.059729   98.471611   98.843439   99.174462   99.463905   99.710968   99.914829  100.074636    100.189518 | 1.175702  1.189571  1.204281  1.219696  1.235679  1.252102  1.268845  1.285798  1.302866  1.319962  1.337015  1.353961  1.370746  1.387326  1.403663  1.419724  1.435484   1.45092  1.466014  1.480751  1.495118  1.509105  1.522704  1.535906  1.548706  1.561099  1.573079  1.584644   1.59579  1.606512  1.616808  1.626676   1.63611  1.645109  1.653669  1.661787  1.669458  1.676679  1.683445  1.689751  1.695592  1.700964  1.705859  1.710271  1.714194   1.71762  1.720542  1.722951  1.724838  1.726194 | 1.354087  1.353399    1.3527  1.351999  1.351306  1.350626  1.349966   1.34933  1.348719  1.348137  1.347583  1.347058  1.346561  1.346093  1.345652  1.345236  1.344846  1.344479  1.344134   1.34381  1.343506   1.34322  1.342952  1.342701  1.342465  1.342243  1.342036  1.341841  1.341658  1.341487  1.341327  1.341178  1.341038  1.340907  1.340786  1.340673  1.340568  1.340471  1.340382    1.3403  1.340225  1.340157  1.340095  1.340041  1.339993  1.339951  1.339916  1.339887  1.339864          1.339848 | 0.577212  0.583164  0.589322  0.595611  0.601959  0.608307  0.614603  0.620804  0.626877  0.632795  0.638541    0.6441  0.649465   0.65463  0.659594  0.664357  0.668922  0.673291   0.67747  0.681463  0.685275  0.688913  0.692382  0.695687  0.698835   0.70183  0.704678  0.707383  0.709951  0.712386  0.714691  0.716871   0.71893   0.72087  0.722694  0.724405  0.726006  0.727499  0.728885  0.730166  0.731344  0.732419  0.733393  0.734266  0.735037  0.735708  0.736278  0.736746  0.737112      0.737374 | -0.252783  -0.259804  -0.267012  -0.274324  -0.281663  -0.288968  -0.296184  -0.303271  -0.310195  -0.316931  -0.323463  -0.329778   -0.33587  -0.341734   -0.34737  -0.352779  -0.357964  -0.362929   -0.36768  -0.372221  -0.376559    -0.3807  -0.384649  -0.388412  -0.391995  -0.395404  -0.398644  -0.401719  -0.404636  -0.407397  -0.410006  -0.412469  -0.414788  -0.416966  -0.419006  -0.420911  -0.422683  -0.424323  -0.425833  -0.427215  -0.428469  -0.429596  -0.430596   -0.43147  -0.432217  -0.432837  -0.433328   -0.43369  -0.433921     -0.434018 | 0.671248  0.672053  0.672891  0.673754  0.674634  0.675524  0.676418   0.67731  0.678196  0.679072  0.679935  0.680782  0.681612  0.682422  0.683212   0.68398  0.684726  0.685449  0.686148  0.686824  0.687476  0.688105  0.688709   0.68929  0.689847   0.69038  0.690889  0.691375  0.691837  0.692275  0.692691  0.693082  0.693451  0.693796  0.694118  0.694416  0.694691  0.694943   0.69517  0.695374  0.695554   0.69571  0.695842  0.695948   0.69603  0.696087  0.696117  0.696122    0.6961      0.696051 | 132.933235 | 136.047689  142.569805  149.741881  157.545711  165.957223  174.947941  184.486203  194.538162  205.068565  216.041342  227.420038  239.168124  251.249205  263.627163  276.266245   289.13111  302.186855  315.399018  328.733567   342.15689  355.635773  369.137374  382.629205  396.079111  409.455241  422.726038  435.860215  448.826743  461.594841  474.133961  486.413789  498.404235  510.075438  521.397768  532.341832  542.878488  552.978861   562.61436  571.756708  580.377969  588.450587  595.947431   602.84184  609.107685  614.719435  619.652228  623.881958  627.385368  630.140156          632.125091 |  632.94054 | 0.781435  0.791056  0.801252  0.811928   0.82299  0.834348  0.845918  0.857625  0.869402  0.881191  0.892942  0.904612  0.916164  0.927569  0.938799  0.949834  0.960657  0.971252  0.981608  0.991715  1.001564  1.011148  1.020463  1.029504  1.038266  1.046747  1.054943  1.062853  1.070474  1.077804  1.084841  1.091584   1.09803  1.104176  1.110023  1.115565  1.120803  1.125732   1.13035  1.134654   1.13864  1.142305  1.145645  1.148655  1.151331  1.153668  1.155661  1.157304   1.15859       1.159515 | 2.949073  3.006999   3.06875    3.1338  3.201621  3.271695  3.343535  3.416691  3.490755  3.565361  3.640188  3.714951  3.789403   3.86333  3.936545  4.008887  4.080215  4.150408  4.219362  4.286983  4.353192  4.417918  4.481098  4.542676  4.602602  4.660829  4.717315  4.772022  4.824913  4.875952  4.925106  4.972342  5.017628  5.060933  5.102224  5.141468  5.178634  5.213688  5.246595  5.277321  5.305828   5.33208  5.356037  5.377659  5.396904  5.413728  5.428086  5.439932  5.449217        5.455891 | 132.933235  139.126367  145.977135  153.470612  161.585236  170.294367  179.567629  189.372026  199.672819  210.434207  221.619827  233.193113  245.117553  257.356859  269.875076  282.636646  295.606441  308.749777  322.032411  335.420525  348.880708  362.379939   375.88556  389.365253   402.78702  416.119162  429.330261  442.389162  455.264963  467.927001  480.344847  492.488298   504.32738  515.832343  526.973676  537.722106  548.048619  557.924478   567.32124  576.210789   584.56537  592.357626  599.560646   606.14802  612.093898  617.373061  621.961003  625.834015  628.969286  631.345017   632.94054 |  100.0 |  32.933235 |  632.94054 |
    | 19:00 | 4.224173  4.310781  4.404173  4.503637   4.60841  4.717712  4.830775   4.94687  5.065314  5.185479  5.306799  5.428763  5.550917  5.672858   5.79423  5.914721  6.034054  6.151987  6.268305  6.382819  6.495362  6.605785  6.713954  6.819749  6.923063  7.023797  7.121859  7.217166   7.30964  7.399207  7.485796   7.56934  7.649774  7.727033  7.801055  7.871777  7.939136   8.00307  8.063514  8.120403  8.173673  8.223255  8.269081  8.311078  8.349174  8.383293  8.413357  8.439288  8.461001             8.478411 | 152.580804  157.575855  163.046271  168.968315   175.31339  182.049686  189.143564  196.560659  204.266715  212.228176  220.412581  228.788798  237.327149  245.999448  254.778977  263.640436  272.559855  281.514505  290.482799  299.444192  308.379088  317.268744  326.095192  334.841152  343.489968  352.025535  360.432242  368.694919  376.798785  384.729405  392.472651  400.014665  407.341828  414.440731  421.298153  427.901034  434.236463   440.29166  446.053961  451.510813  456.649766  461.458468  465.924671  470.036226  473.781098  477.147373  480.123274  482.697181  484.857658  486.593477 |  57.241734   58.107806   59.041726   60.036375   61.084104   62.177116   63.307751   64.468699   65.653136   66.854794   68.067993   69.287632    70.50917   71.728578   72.942303    74.14721   75.340541   76.519867   77.683048   78.828192   79.953623   81.057849   82.139539   83.197494   84.230632   85.237965   86.218588    87.17166     88.0964   88.992068    89.85796   90.693403   91.497741   92.270335   93.010553   93.717771   94.391362   95.030696   95.635136   96.204035   96.736732   97.232553   97.690805   98.110777   98.491735   98.832927   99.133574   99.392875   99.610005     99.784115 | 1.165149  1.177952   1.19166   1.20615  1.221297  1.236975  1.253065  1.269455  1.286046  1.302746  1.319478  1.336172  1.352769  1.369217  1.385475  1.401506  1.417278  1.432766  1.447947  1.462805  1.477324  1.491491  1.505294  1.518726  1.531779  1.544446  1.556721    1.5686  1.580077   1.59115  1.601814  1.612066  1.621903   1.63132  1.640315  1.648883  1.657021  1.664725  1.671991  1.678813  1.685188   1.69111  1.696573  1.701571  1.706098  1.710147   1.71371   1.71678  1.719348  1.721407 | 1.354631  1.353974  1.353298  1.352613  1.351928  1.351251  1.350588  1.349943  1.349321  1.348723  1.348153  1.347609  1.347094  1.346606  1.346144  1.345709  1.345298  1.344912  1.344548  1.344206  1.343884  1.343581  1.343297   1.34303  1.342778  1.342542  1.342321  1.342113  1.341917  1.341734  1.341562  1.341401   1.34125  1.341108  1.340977  1.340853  1.340739  1.340632  1.340534  1.340443  1.340359  1.340282  1.340212  1.340149  1.340092  1.340042  1.339999  1.339961   1.33993          1.339905 | 0.572586  0.578188  0.584048  0.590094  0.596254  0.602466  0.608674  0.614829  0.620893  0.626835   0.63263   0.63826  0.643714  0.648982   0.65406  0.658946  0.663639  0.668142  0.672458  0.676589  0.680542   0.68432  0.687929  0.691374   0.69466  0.697793  0.700777  0.703618   0.70632  0.708887  0.711323  0.713633   0.71582  0.717888  0.719839  0.721676  0.723403   0.72502  0.726531  0.727937   0.72924  0.730441  0.731541  0.732541   0.73344  0.734241  0.734942  0.735544  0.736046      0.736446 | -0.247193  -0.253837  -0.260731  -0.267791  -0.274942  -0.282115  -0.289251  -0.296302   -0.30323  -0.310003  -0.316599  -0.323001  -0.329198  -0.335181  -0.340948  -0.346497  -0.351828  -0.356946  -0.361852  -0.366551  -0.371049   -0.37535   -0.37946  -0.383384  -0.387127  -0.390696  -0.394095  -0.397329  -0.400403   -0.40332  -0.406086  -0.408704  -0.411178  -0.413511  -0.415705  -0.417764   -0.41969  -0.421485  -0.423151  -0.424688    -0.4261  -0.427386  -0.428546  -0.429582  -0.430494  -0.431281  -0.431942  -0.432477  -0.432884     -0.433163 | 0.670601  0.671352  0.672141  0.672961  0.673805  0.674664  0.675533  0.676405  0.677276  0.678141  0.678998  0.679842  0.680672  0.681485   0.68228  0.683056  0.683811  0.684546  0.685258  0.685948  0.686616  0.687261  0.687884  0.688483  0.689059  0.689613  0.690143  0.690651  0.691135  0.691597  0.692036  0.692452  0.692846  0.693216  0.693564  0.693889  0.694192  0.694471  0.694728  0.694961  0.695171  0.695358  0.695521   0.69566  0.695775  0.695865  0.695931  0.695971  0.695986      0.695975 | 128.453911 | 131.238302  137.090401  143.571957  150.672128   158.37353  166.653616  175.485921  184.841113  194.687854  204.993475  215.724492  226.846991  238.326899  250.130184  262.222981  274.571676  287.142959  299.903848  312.821697  325.864195  338.999355  352.195499  365.421242  378.645474  391.837339  404.966223  418.001734   430.91369  443.672104   456.24718  468.609299   480.72902  492.577072  504.124362  515.341973  526.201177  536.673442   546.73045  556.344119   565.48662  574.130414  582.248286  589.813383  596.799262  603.179949  608.929997   614.02456  618.439471  622.151331          625.137612 | 626.463357 | 0.774111  0.782997  0.792504  0.802546  0.813036  0.823886  0.835013  0.846339  0.857795   0.86932  0.880857  0.892361  0.903791  0.915112  0.926296  0.937316  0.948154  0.958791  0.969212  0.979407  0.989365  0.999077  1.008537  1.017739  1.026678   1.03535  1.043751  1.051879   1.05973  1.067302  1.074593  1.081601  1.088323  1.094757  1.100902  1.106754  1.112311  1.117571  1.122532  1.127189   1.13154  1.135581  1.139309  1.142719  1.145808   1.14857     1.151  1.153095  1.154847        1.15625 | 2.905195  2.958453  3.015747  3.076617  3.140577  3.207135  3.275814  3.346162  3.417762  3.490236  3.563242  3.636478  3.709677  3.782605  3.855058  3.926856  3.997844  4.067887  4.136866  4.204678  4.271232  4.336449  4.400257  4.462594  4.523403  4.582632  4.640236  4.696171  4.750397  4.802876  4.853572   4.90245  4.949478  4.994621  5.037847  5.079123  5.118416  5.155694  5.190921  5.224063  5.255084  5.283948  5.310616   5.33505  5.357207  5.377048  5.394527  5.409599  5.422219        5.432336 | 128.453911  133.989029  140.157436  146.951895  154.357903  162.355133  170.918765  180.020637  189.630205    199.7153  210.242724  221.178687   232.48913  244.139955  256.097183  268.327055  280.796099  293.471161  306.319426  319.308416  332.405985  345.580303  358.799843  372.033362  385.249881   398.41867  411.509229  424.491273  437.334721  450.009685  462.486458   474.73551  486.727485  498.433202  509.823653  520.870013  531.543648   541.81613  551.659252   561.04505  569.945833  578.334214  586.183145  593.465967  600.156453  606.228877  611.658071  616.419506  620.489372  623.844679  626.463357 |  100.0 |  28.453911 | 626.463357 |
    | 19:30 |  4.16017   4.23919  4.325176   4.41754  4.515608  4.618666   4.72599  4.836869  4.950625  5.066622  5.184278   5.30306   5.42249  5.542139  5.661626  5.780613  5.898799  6.015919  6.131739   6.24605  6.358669   6.46943  6.578186  6.684806  6.789169  6.891169  6.990706  7.087688  7.182031  7.273656  7.362487  7.448453  7.531486  7.611518  7.688485  7.762323  7.832968  7.900356  7.964423  8.025106  8.082338  8.136053  8.186183  8.232658  8.275407  8.314356   8.34943   8.38055  8.407637             8.430607 | 148.937618  153.441496  158.413403  163.836371  169.687671  175.940374   182.56477  189.529556  196.802798  204.352646  212.147853  220.158124  228.354328  236.708614  245.194446  253.786596  262.461099  271.195181  279.967182  288.756473  297.543368  306.309038  315.035435  323.705212  332.301652  340.808606  349.210432  357.491937  365.638333  373.635187  381.468381  389.124078  396.588687  403.848832  410.891327  417.703155  424.271443  430.583449  436.626546  442.388207  447.856001  453.017585    457.8607   462.37317  466.542908   470.35792  473.806316  476.876325  479.556309  481.834793 |  56.601699   57.391898   58.251764   59.175395   60.156076   61.186661   62.259902   63.368692   64.506247   65.666224   66.842779     68.0306   69.224898   70.421391   71.616261   72.806126   73.987985   75.159189   76.317388   77.460503   78.586687   79.694296   80.781859   81.848056   82.891695   83.911692   84.907059   85.876881   86.820313   87.736559    88.62487   89.484533   90.314859   91.115183   91.884855   92.623232   93.329678   94.003558   94.644233   95.251058   95.823379    96.36053   96.861829   97.326581    97.75407    98.14356   98.494296   98.805498   99.076365     99.306072 |  1.15563  1.167376  1.180072  1.193613  1.207886  1.222773  1.238157  1.253928  1.269983  1.286228   1.30258  1.318964  1.335316   1.35158  1.367707  1.383657  1.399393  1.414887  1.430113   1.44505  1.459679  1.473986  1.487957  1.501582  1.514851  1.527756  1.540291   1.55245  1.564226  1.575617  1.586616   1.59722  1.607426  1.617229  1.626625  1.635612  1.644184  1.652338  1.660071  1.667376   1.67425  1.680687  1.686683  1.692231  1.697326  1.701961  1.706128  1.709821  1.713032  1.715753 | 1.355136  1.354515  1.353868  1.353204  1.352533  1.351863  1.351201  1.350553  1.349923  1.349314  1.348729   1.34817  1.347637   1.34713   1.34665  1.346195  1.345765   1.34536  1.344977  1.344616  1.344277  1.343957  1.343656  1.343373  1.343106  1.342855  1.342619  1.342397  1.342189  1.341992  1.341808  1.341635  1.341473  1.341321  1.341178  1.341045  1.340921  1.340805  1.340697  1.340596  1.340503  1.340418  1.340339  1.340268  1.340203  1.340144  1.340092  1.340046  1.340007          1.339974 | 0.568338  0.573569  0.579103  0.584872  0.590808  0.596846  0.602928  0.609002  0.615025  0.620959  0.626776  0.632454  0.637975  0.643328  0.648504  0.653498  0.658309  0.662935  0.667378  0.671641  0.675727   0.67964  0.683385  0.686965  0.690387  0.693654  0.696772  0.699745  0.702578  0.705275   0.70784  0.710278  0.712591  0.714785   0.71686  0.718822  0.720671  0.722412  0.724045  0.725573  0.726998  0.728322  0.729544  0.730668  0.731692  0.732618  0.733447  0.734177  0.734809      0.735343 | -0.242038  -0.248276  -0.254819  -0.261588  -0.268507  -0.275505  -0.282519  -0.289496  -0.296392  -0.303169    -0.3098  -0.316262   -0.32254  -0.328623  -0.334503  -0.340177  -0.345642  -0.350899   -0.35595  -0.360799  -0.365448  -0.369902  -0.374167  -0.378246  -0.382145  -0.385869  -0.389423  -0.392811  -0.396038  -0.399109  -0.402027  -0.404797  -0.407422  -0.409906  -0.412251   -0.41446  -0.416537  -0.418483  -0.420299  -0.421989  -0.423553  -0.424993  -0.426309  -0.427502  -0.428572  -0.429519  -0.430344  -0.431045  -0.431622     -0.432074 | 0.670014  0.670709  0.671447  0.672222  0.673025  0.673849  0.674689  0.675537  0.676389   0.67724  0.678086  0.678924   0.67975  0.680563  0.681361  0.682142  0.682904  0.683647   0.68437  0.685073  0.685754  0.686414  0.687052  0.687668  0.688262  0.688834  0.689384  0.689912  0.690417  0.690901  0.691362  0.691801  0.692218  0.692613  0.692985  0.693336  0.693664   0.69397  0.694254  0.694515  0.694754   0.69497  0.695162  0.695332  0.695478  0.695601    0.6957  0.695774  0.695824      0.695848 | 124.530701 | 127.010719    132.2421  138.077884  144.514263  151.540408   159.13973  167.291068  175.969762  185.148563  194.798373  204.888839  215.388805  226.266647  237.490526  249.028563  260.848956  272.920065  285.210457  297.688936  310.324554  323.086613  335.944657  348.868464  361.828031  374.793562   387.73545  400.624267  413.430747   426.12578  438.680395  451.065757   463.25316  475.214025  486.919894  498.342437  509.453456   520.22489  530.628829  540.637528  550.223426  559.359168  568.017637  576.171984   583.79567  590.862506  597.346713  603.222974  608.466506  613.053132          616.959373 | 618.754992 |   0.7675  0.775656  0.784467  0.793858  0.803749  0.814058  0.824704   0.83561  0.846704  0.857922  0.869205  0.880503  0.891772  0.902973  0.914073  0.925045  0.935865  0.946512  0.956969  0.967224  0.977262  0.987076  0.996655  1.005993  1.015084  1.023923  1.032506  1.040828  1.048887  1.056679  1.064201  1.071452  1.078429  1.085129  1.091549  1.097689  1.103544  1.109114  1.114393  1.119381  1.124074  1.128468   1.13256  1.136346  1.139823  1.142985  1.145828  1.148348  1.150538       1.152394 | 2.865755  2.914438  2.967293  3.023936  3.083933  3.146829   3.21217  3.279511  3.348434  3.418552  3.489512  3.560994  3.632715  3.704422  3.775894  3.846935  3.917375  3.987063  4.055869  4.123676  4.190384  4.255904  4.320156   4.38307  4.444583  4.504637  4.563182   4.62017  4.675557  4.729302  4.781366  4.831713  4.880307  4.927114    4.9721  5.015232  5.056475  5.095797  5.133163  5.168538  5.201889  5.233178  5.262368  5.289421  5.314297  5.336957  5.357356  5.375453  5.391201        5.404554 | 124.530701  129.459284  134.992485  141.130297  147.865063  155.182738  163.064133  171.486067  180.422359  189.844652  199.723079  210.026775   220.72427  231.783775  243.173387  254.861238  266.815589  279.004891  291.397824  303.963313  316.670535  329.488913  342.388112  355.338022  368.308747  381.270588  394.194032  407.049735   419.80851   432.44132  444.919264  457.213572  469.295602  481.136834  492.708876   503.98346  514.932455  525.527872  535.741877  545.546814  554.915216  563.819841  572.233695  580.130072  587.482596  594.265268  600.452522  606.019289  610.941069  615.194008  618.754992 |  100.0 |  24.530701 | 618.754992 |
    | 20:00 | 4.103014  4.174715  4.253449  4.338754  4.430065  4.526748  4.628135   4.73355  4.842332   4.95385  5.067512  5.182773  5.299134  5.416145  5.533401  5.650541  5.767241  5.883215  5.998209  6.111993  6.224368  6.335153  6.444186  6.551323  6.656433  6.759399  6.860111  6.958471  7.054387  7.147774  7.238552  7.326645   7.41198  7.494489  7.574104  7.650758  7.724386  7.794925   7.86231  7.926475  7.987357  8.044889  8.099003  8.149633  8.196707  8.240153  8.279899  8.315867  8.347982             8.376161 | 145.718828   149.76196  154.260864  159.205235  164.578348  170.358465  176.520199  183.035739  189.875891  197.010897  204.411071  212.047246  219.891086  227.915279   236.09364  244.401154  252.813967  261.309353  269.865657  278.462225  287.079333  295.698113  304.300471  312.869027  321.387034  329.838326  338.207251  346.478621   354.63766  362.669958  370.561433  378.298287   385.86698   393.25419  400.446795   407.43184  414.196521  420.728162  427.014201  433.042174  438.799707  444.274503  449.454341  454.327065  458.880593  463.102912  466.982088  470.506273  473.663719  476.442793 |   56.03014   56.747151   57.534488   58.387539   59.300649   60.267481   61.281351   62.335502   63.423322     64.5385    65.67512   66.827726   67.991336   69.161446   70.334009   71.505406   72.672411   73.832155   74.982085   76.119935   77.243684    78.35153   79.441862   80.513232   81.564335   82.593986   83.601107   84.584706   85.543867   86.477738   87.385517   88.266447   89.119805   89.944893   90.741038   91.507578   92.243864   92.949251   93.623096   94.264752   94.873569   95.448887   95.990035   96.496328   96.967066   97.401531   97.798987   98.158675   98.479817     98.761611 | 1.147087  1.157798  1.169487  1.182068  1.195442  1.209501  1.224135  1.239236  1.254702  1.270436  1.286352  1.302372  1.318425   1.33445  1.350396  1.366216   1.38187  1.397324  1.412551  1.427524  1.442224  1.456631  1.470732  1.484513  1.497963  1.511072  1.523833  1.536238  1.548281  1.559956  1.571258  1.582183  1.592726  1.602883   1.61265  1.622023  1.630998  1.639571  1.647737  1.655493  1.662833  1.669754  1.676249  1.682313   1.68794  1.693125  1.697861   1.70214  1.705956  1.709301 | 1.355602   1.35502  1.354405  1.353768  1.353116  1.352459  1.351803  1.351156  1.350522  1.349905  1.349309  1.348736  1.348188  1.347664  1.347166  1.346693  1.346245  1.345821   1.34542  1.345041  1.344684  1.344347  1.344029  1.343729  1.343447  1.343181   1.34293  1.342695  1.342472  1.342263  1.342067  1.341882  1.341708  1.341545  1.341392  1.341248  1.341113  1.340987   1.34087   1.34076  1.340658  1.340564  1.340477  1.340396  1.340323  1.340256  1.340196  1.340142  1.340094          1.340053 | 0.564464  0.569312  0.574498  0.579963  0.585641  0.591471  0.597391  0.603349  0.609296  0.615192  0.621004  0.626703  0.632269  0.637687  0.642943  0.648031  0.652945  0.657683  0.662244   0.66663  0.670842  0.674883  0.678757  0.682468  0.686021  0.689419  0.692667  0.695769  0.698731  0.701556  0.704248  0.706811  0.709249  0.711565  0.713764  0.715847  0.717817  0.719678  0.721432   0.72308  0.724625  0.726068   0.72741  0.728654    0.7298  0.730848  0.731799  0.732654  0.733412      0.734074 | -0.237315  -0.243127  -0.249291  -0.255734  -0.262382  -0.269165  -0.276017  -0.282881  -0.289707  -0.296455   -0.30309  -0.309585   -0.31592  -0.322081  -0.328055  -0.333836   -0.33942  -0.344804   -0.34999  -0.354977  -0.359769  -0.364369  -0.368781  -0.373009  -0.377058  -0.380932  -0.384636  -0.388174  -0.391551  -0.394771  -0.397838  -0.400756  -0.403529   -0.40616  -0.408652  -0.411009  -0.413232  -0.415325  -0.417289  -0.419126  -0.420839  -0.422428  -0.423894  -0.425239  -0.426462  -0.427565  -0.428547  -0.429408  -0.430148     -0.430765 | 0.669483  0.670123  0.670809  0.671535  0.672295  0.673082  0.673889  0.674709  0.675539  0.676371  0.677203  0.678031  0.678851   0.67966  0.680458   0.68124  0.682007  0.682756  0.683488    0.6842  0.684893  0.685565  0.686217  0.686848  0.687458  0.688047  0.688615  0.689161  0.689686  0.690189  0.690671  0.691132  0.691571  0.691988  0.692384  0.692758  0.693111  0.693442  0.693752  0.694039  0.694305  0.694548   0.69477  0.694968  0.695145  0.695298  0.695428  0.695534  0.695617      0.695675 | 121.103761 | 123.304688  127.964252  133.199479  139.013098  145.400579  152.351196  159.849121  167.874458  176.404176  185.412893  194.873524  204.757799   215.03666  225.680563  236.659699  247.944157  259.504032  271.309501  283.330872  295.538612  307.903358  320.395927  332.987307  345.648651   358.35127  371.066619  383.766289   396.42199  409.005544  421.488875  433.844001  446.043026  458.058138  469.861604  481.425772  492.723069  503.726013  514.407215  524.739391  534.695379  544.248154  553.370854  562.036801  570.219541  577.892872  585.030896   591.60806  597.599219  602.979695          607.725353 | 609.950719 | 0.761564  0.769006  0.777122  0.785852  0.795126  0.804868  0.815001  0.825451  0.836145  0.847017  0.858007  0.869061  0.880131  0.891175  0.902158  0.913047  0.923816  0.934442  0.944906  0.955192  0.965284  0.975171  0.984844  0.994294  1.003513  1.012496  1.021237  1.029731  1.037975  1.045965  1.053697   1.06117  1.068379  1.075323  1.081999  1.088405  1.094537  1.100393  1.105971  1.111268   1.11628  1.121005  1.125438  1.129577  1.133418  1.136956  1.140188  1.143108  1.145711       1.147993 | 2.830476  2.874724  2.923211  2.975628  3.031607  3.090739  3.152601  3.216766  3.282824  3.350386  3.419089  3.488605  3.558633  3.628908   3.69919  3.769269  3.838958  3.908094  3.976533  4.044146  4.110823  4.176463  4.240979  4.304293  4.366335  4.427042  4.486357  4.544228  4.600608   4.65545  4.708715  4.760363  4.810357   4.85866  4.905236  4.950053  4.993074  5.034268  5.073599  5.111032  5.146534  5.180068  5.211598  5.241086  5.268494  5.293783  5.316911  5.337836  5.356514        5.372901 | 121.103761  125.476426  130.421653  135.946046  142.048432  148.720886  155.949836   163.71715   172.00113  180.777372  190.019483  199.699661  209.789144   220.25856   231.07818  242.218105  253.648401  265.339186  277.260692  289.383299  301.677556  314.114189  326.664101  339.298364  351.988213  364.705033  377.420351  390.105817  402.733202  415.274381  427.701326    439.9861  452.100849  464.017801  475.709263  487.147622  498.305349  509.155006  519.669252   529.82086  539.582732  548.927916  557.829636  566.261317  574.196619   581.60948  588.474161  594.765296  600.457955  605.527713  609.950719 |  100.0 |  21.103761 | 609.950719 |
    | 20:30 | 4.052202  4.116916  4.188621  4.266988  4.351564  4.441808  4.537123  4.636881  4.740453   4.84722  4.956594  5.068023  5.180995  5.295043  5.409742  5.524709    5.6396  5.754108  5.867958  5.980905  6.092728  6.203232  6.312242    6.4196  6.525163  6.628802  6.730401  6.829851  6.927054  7.021919  7.114359  7.204294  7.291648  7.376347  7.458322  7.537503  7.613826  7.687223  7.757629   7.82498  7.889211  7.950255  8.008047  8.062519  8.113602  8.161226  8.205319  8.245807  8.282615             8.315665 |  142.88472  146.498743  150.552065  155.040731  159.953977  165.275413  170.984263  177.056566  183.466242  190.185998  197.188041  204.444632  211.928488  219.613051  227.472666  235.482675  243.619461  251.860441  260.184045  268.569661  276.997584  285.448945  293.905653  302.350319  310.766201  319.137138  327.447495  335.682107  343.826232  351.865504  359.785892   367.57366  375.215336  382.697674  390.007633  397.132343   404.05909   410.77529  417.268471  423.526262  429.536374  435.286592  440.764768  445.958811  450.856687  455.446417  459.716079  463.653813   467.24783  470.486421 |  55.522017   56.169163   56.886214   57.669876   58.515635    59.41808   60.371229   61.368814   62.404526   63.472198   64.565942    65.68023   66.809953   67.950431   69.097419   70.247089   71.396003   72.541084   73.679583   74.809045   75.927279   77.032324   78.122424      79.196   80.251629   81.288024   82.304009   83.298512   84.270544   85.219188   86.143588   87.042939   87.916476   88.763469   89.583216   90.375035   91.138257   91.872225    92.57629     93.2498   93.892105   94.502549   95.080468   95.625186   96.136017   96.612258   97.053189   97.458074   97.826155     98.156652 | 1.139457  1.149168  1.159868   1.17149   1.18395  1.197155  1.211004  1.225393  1.240221  1.255394  1.270822  1.286424  1.302125  1.317862  1.333576  1.349218  1.364743  1.380113  1.395297  1.410266  1.424995  1.439465  1.453659  1.467559  1.481155  1.494434  1.507387  1.520005  1.532282   1.54421  1.555784  1.566998  1.577848  1.588328  1.598435  1.608164  1.617511  1.626471  1.635041  1.643216  1.650991  1.658362  1.665324  1.671871  1.677998    1.6837  1.688969  1.693799  1.698184  1.702116 | 1.356029  1.355488  1.354909  1.354302  1.353675  1.353035   1.35239  1.351748  1.351115  1.350494   1.34989  1.349307  1.348745  1.348207  1.347692  1.347202  1.346736  1.346294  1.345875  1.345479  1.345104   1.34475  1.344415  1.344099  1.343801   1.34352  1.343255  1.343005  1.342769  1.342547  1.342337   1.34214  1.341955   1.34178  1.341616  1.341462  1.341317  1.341181  1.341054  1.340935  1.340824   1.34072  1.340624  1.340535  1.340453  1.340378   1.34031  1.340248  1.340192          1.340142 | 0.560953  0.565413  0.570238  0.575376   0.58077  0.586359  0.592085  0.597893  0.603732  0.609559  0.615335  0.621029  0.626617  0.632077  0.637395  0.642559  0.647562  0.652399  0.657068  0.661566  0.665896  0.670058  0.674056  0.677892  0.681571  0.685095   0.68847  0.691699  0.694786  0.697735  0.700551  0.703238  0.705798  0.708236  0.710555  0.712758  0.714847  0.716826  0.718698  0.720463  0.722125  0.723685  0.725145  0.726507   0.72777  0.728937  0.730008  0.730984  0.731864      0.732649 | -0.233017  -0.238392  -0.244156  -0.250244  -0.256588  -0.263118   -0.26977  -0.276483  -0.283204  -0.289887  -0.296494  -0.302994  -0.309361  -0.315575  -0.321623  -0.327494   -0.33318  -0.338677  -0.343984  -0.349099  -0.354024  -0.358762  -0.363314  -0.367684  -0.371876  -0.375895  -0.379744  -0.383427   -0.38695  -0.390315  -0.393527  -0.396589  -0.399506  -0.402281  -0.404917  -0.407417  -0.409783   -0.41202  -0.414127  -0.416109  -0.417966    -0.4197  -0.421312  -0.422804  -0.424176  -0.425429  -0.426563  -0.427578  -0.428474      -0.42925 | 0.669006  0.669591  0.670225  0.670902  0.671617  0.672363  0.673133  0.673923  0.674726  0.675537  0.676351  0.677166  0.677976  0.678779  0.679572  0.680354  0.681122  0.681875  0.682613  0.683332  0.684034  0.684717  0.685381  0.686025   0.68665  0.687254  0.687837  0.688401  0.688943  0.689465  0.689966  0.690447  0.690906  0.691345  0.691763  0.692159  0.692535   0.69289  0.693224  0.693536  0.693827  0.694097  0.694345  0.694572  0.694776  0.694958  0.695118  0.695255  0.695369       0.69546 | 118.118088 | 120.064494  124.200134  128.879856  134.112308  139.898871   146.23448  153.108572  160.506046  168.408168  176.793378  185.637977  194.916697  204.603146  214.670169  225.090108   235.83501  246.876766  258.187213  269.738211  281.501682   293.44964  305.554208  317.787622   330.12223   342.53049  354.984957  367.458283  379.923198  392.352508  404.719081  416.995844  429.155771  441.171882  453.017237  464.664936  476.088117  487.259961  498.153695  508.742602  519.000028    528.8994  538.414241  547.518191  556.185033  564.388724  572.103429  579.303561  585.963831  592.059297          597.565427 | 600.181103 |  0.75626  0.763011  0.770444  0.778512  0.787158  0.796313  0.805909  0.815872  0.826132  0.836623  0.847284  0.858056  0.868891  0.879743  0.890573  0.901347  0.912033  0.922608  0.933049  0.943336  0.953455   0.96339  0.973132  0.982668  0.991992  1.001095  1.009971  1.018615  1.027022  1.035189   1.04311  1.050783  1.058205  1.065372  1.072283  1.078933  1.085321  1.091444  1.097299  1.102883  1.108194  1.113227   1.11798   1.12245  1.126633  1.130524   1.13412  1.137416  1.140408       1.143091 | 2.799062  2.839062  2.883296  2.931538  2.983489  3.038797  3.097076   3.15793  3.220962  3.285789  3.352046  3.419398  3.487534  3.556174  3.625066  3.693984  3.762729  3.831122  3.899005  3.966241  4.032705  4.098289  4.162896  4.226438  4.288841  4.350033  4.409952  4.468542  4.525751  4.581529  4.635833  4.688619   4.73985  4.789486   4.83749  4.883828  4.928463  4.971362   5.01249  5.051813  5.089295  5.124901  5.158595  5.190341  5.220101  5.247837  5.273508  5.297073  5.318491        5.337718 | 118.118088  121.983988  126.387922  131.342357  136.852113  142.915105  149.523254  156.663472  164.318606  172.468304  181.089764  190.158364  199.648163  209.532302   219.78331  230.373335  241.274309  252.458072  263.896458  275.561345  287.424695  299.458573  311.635156  323.926738   336.30572  348.744613  361.216022  373.692639  386.147235  398.552649   410.88178   423.10758  435.203046  447.141221  458.895183  470.438051  481.742983  492.783183  503.531902  513.962452  524.048215   533.76266  543.079364  551.972029  560.414519  568.380883  575.845398  582.782614  589.167399     594.975  600.181103 |  100.0 |  18.118088 | 600.181103 |
    | 21:00 | 4.007224  4.065334  4.130296  4.201911  4.279843  4.363653  4.452823  4.546789  4.644962  4.746752  4.851582  4.958901  5.068191  5.178974  5.290809  5.403295  5.516069  5.628804  5.741207  5.853014  5.963989  6.073921  6.182617  6.289909  6.395641  6.499673  6.601879  6.702143  6.800357  6.896424  6.990251  7.081753   7.17085  7.257465  7.341523  7.422955  7.501692  7.577665   7.65081   7.72106   7.78835  7.852615  7.913788  7.971803  8.026592  8.078086  8.126215  8.170906  8.212087             8.249681 |  140.39759  143.614704  147.251179   151.30892  155.782911  160.662111  165.930524  171.568306  177.552816  183.859555  190.462949  197.336984  204.455687  211.793481  219.325428  227.027386  234.876101  242.849242  250.925407  259.084101  267.305691  275.571361  283.863056  292.163418  300.455734  308.723877  316.952248  325.125727  333.229626  341.249638  349.171803  356.982462  364.668226  372.215944   379.61267  386.845642  393.902252  400.770026  407.436608  413.889739  420.117242  426.107013  431.847008  437.325234  442.529743  447.448633  452.070041  456.382148  460.373182  464.031426 |  55.072242   55.653338   56.302962   57.019106   57.798427   58.636526   59.528233   60.467893   61.449624    62.46752   63.515817   64.589006   65.681913   66.789742   67.908089   69.032947   70.160687   71.288041   72.412072   73.530144   74.639894   75.739205   76.826175   77.899091    78.95641   79.996735   81.018794   82.021427   83.003569   83.964235   84.902509   85.817535   86.708503   87.574647   88.415233   89.229551   90.016915   90.776652   91.508099     92.2106   92.883501   93.526148   94.137878   94.718027   95.265917   95.780858   96.262146   96.709063   97.120871     97.496813 | 1.132675  1.141432   1.15117  1.161844  1.173389  1.185725  1.198761  1.212403  1.226554   1.24112  1.256012  1.271146  1.286447  1.301846   1.31728  1.332696  1.348046  1.363288  1.378387  1.393309   1.40803  1.422524  1.436772  1.450757  1.464464  1.477879  1.490991  1.503791  1.516269  1.528419  1.540234  1.551707  1.562833  1.573607  1.584024  1.594079  1.603768  1.613086  1.622029  1.630593  1.638773  1.646564  1.653961   1.66096  1.667555   1.67374  1.679509  1.684857  1.689777  1.694261 | 1.356417  1.355918  1.355378  1.354805  1.354205  1.353587  1.352959  1.352327  1.351698  1.351077  1.350469  1.349878  1.349306  1.348755  1.348226   1.34772  1.347238  1.346779  1.346343  1.345929  1.345537  1.345166  1.344815  1.344482  1.344169  1.343872  1.343592  1.343327  1.343078  1.342842   1.34262  1.342411  1.342213  1.342027  1.341851  1.341686  1.341531  1.341385  1.341248  1.341119  1.340999  1.340887  1.340782  1.340684  1.340594   1.34051  1.340433  1.340363  1.340299          1.340242 | 0.557792  0.561867  0.566323  0.571119  0.576206  0.581528   0.58703  0.592656  0.598356  0.604082  0.609793  0.615455  0.621038  0.626518  0.631877    0.6371  0.642177  0.647099  0.651862  0.656463  0.660901  0.665176   0.66929  0.673246  0.677045  0.680691  0.684188  0.687539  0.690749   0.69382  0.696757  0.699565  0.702245  0.704802  0.707239  0.709559  0.711766  0.713861  0.715848  0.717729  0.719506  0.721181  0.722756  0.724232   0.72561  0.726893   0.72808  0.729173  0.730172      0.731076 |  -0.22913  -0.234066  -0.239417  -0.245129  -0.251139  -0.257385  -0.263801  -0.270326  -0.276906  -0.283491  -0.290039  -0.296512  -0.302883  -0.309128  -0.315227  -0.321167  -0.326938  -0.332533  -0.337948  -0.343179  -0.348227  -0.353092  -0.357775  -0.362281   -0.36661  -0.370767  -0.374755  -0.378579  -0.382242  -0.385748  -0.389101  -0.392304  -0.395362  -0.398277  -0.401053  -0.403693  -0.406199  -0.408575  -0.410823  -0.412944  -0.414942  -0.416817  -0.418571  -0.420206  -0.421722   -0.42312  -0.424401  -0.425565  -0.426611     -0.427541 |  0.66858  0.669111  0.669693  0.670321  0.670989  0.671692  0.672424   0.67318  0.673953  0.674739  0.675533   0.67633  0.677127   0.67792  0.678707  0.679485  0.680252  0.681007  0.681747  0.682472  0.683181  0.683873  0.684546  0.685202  0.685838  0.686456  0.687054  0.687633  0.688191   0.68873  0.689249  0.689748  0.690227  0.690685  0.691123  0.691541  0.691939  0.692316  0.692672  0.693008  0.693324  0.693618  0.693892  0.694144  0.694375  0.694585  0.694773  0.694939  0.695082      0.695204 | 115.523427 | 117.238921  120.897004  125.065566  129.758459  134.982473  140.737897  147.019279  153.816262  161.114439  168.896141  177.141147  185.827282  194.930918  204.427368  214.291202  224.496486  235.016959  245.826169  256.897561  268.204547  279.720551   291.41903  303.273496  315.257519   327.34473  339.508817  351.723522  363.962632   376.19997  388.409389  400.564767  412.639996  424.608979  436.445623  448.123843  459.617551  470.900663  481.947103  492.730802   503.22571  513.405806  523.245108  532.717694  541.797717  550.459434  558.677231  566.425659  573.679469  580.413658          586.603522 | 589.571078 | 0.751544  0.757633  0.764401  0.771816   0.77983  0.788388  0.797426  0.806878  0.816675  0.826753   0.83705  0.847507  0.858073  0.868698  0.879342  0.889967   0.90054  0.911033   0.92142  0.931682    0.9418  0.951758  0.961542  0.971141  0.980545  0.989745  0.998735  1.007507  1.016056  1.024377  1.032467   1.04032  1.047934  1.055304  1.062429  1.069305  1.075928  1.082297  1.088409   1.09426  1.099848   1.10517  1.110222  1.115001  1.119503  1.123726  1.127664  1.131314  1.134671       1.137731 | 2.771217  2.807185  2.847323  2.891484  2.939442  2.990906  3.045539  3.102979  3.162854  3.224791  3.288434  3.353442    3.4195  3.486316  3.553628  3.621197   3.68881  3.756274  3.823422  3.890101  3.956178  4.021534  4.086061  4.149666  4.212265   4.27378  4.334144  4.393293  4.451173  4.507729  4.562915  4.616684  4.668995  4.719807  4.769083  4.816784  4.862874  4.907319  4.950083  4.991131  5.030429  5.067941  5.103632  5.137465  5.169404  5.199411  5.227448  5.253473  5.277447        5.299328 | 115.523427  118.929757  122.837984  127.265606  132.222833  137.713023  143.733372  150.275745  157.327532   164.87249  172.891495  181.363198  190.264575  199.571369  209.258448  219.300071  229.670099  240.342141  251.289673  262.486108  273.904853  285.519341  297.303054   309.22953  321.272371  333.405235   345.60184  357.835951  370.081374  382.311951  394.501547   406.62405  418.653356  430.563374  442.328014  453.921189  465.316814  476.488806  487.411092   498.05761  508.402322  518.419221  528.082351  537.365821  546.243831  554.690693  562.680865  570.188987  577.189922  583.658801  589.571078 |  100.0 |  15.523427 | 589.571078 |
    | 21:30 | 3.967576  4.019499  4.078054  4.143163  4.214606  4.292049  4.375065   4.46316    4.5558   4.65243  4.752498  4.855462  4.960808  5.068049  5.176735  5.286449  5.396814  5.507485  5.618151  5.728531  5.838372  5.947447  6.055552  6.162501  6.268127  6.372281  6.474825  6.575634  6.674593  6.771596  6.866546  6.959351  7.049926  7.138191  7.224067  7.307482  7.388363  7.466643  7.542253  7.615127  7.685199  7.752403  7.816673  7.877944  7.936147  7.991216   8.04308  8.091671  8.136916             8.178741 | 138.221951  141.074338  144.323446  147.976417  152.033607  156.489185  161.331972  166.546411   172.11355  178.011974  184.218627  190.709501  197.460197  204.446345  211.643922  219.029468  226.580229  234.274239  242.090361  250.008286  258.008521  266.072351  274.181798  282.319572   290.46902   298.61407  306.739183  314.829305  322.869816  330.846487   338.74544   346.55311  354.256208  361.841688  369.296718  376.608653  383.765009  390.753441  397.561724   404.17773  410.589418  416.784815  422.752006  428.479124  433.954341  439.165863  444.101927  448.750797  453.100769  457.140171 |  54.675763   55.194988   55.780542   56.431625    57.14606   57.920495   58.750655   59.631604   60.557997     61.5243   62.524975   63.554622    64.60808   65.680491   66.767346   67.864493   68.968142   70.074851   71.181509   72.285308    73.38372   74.474472   75.555516   76.625005   77.681274   78.722814   79.748252   80.756338   81.745925   82.715958   83.665458   84.593512   85.499264   86.381906   87.240669   88.074815   88.883633   89.666431   90.422533   91.151273   91.851992   92.524031   93.166733   93.779435   94.361469   94.912155   95.430805   95.916714   96.369162     96.787414 | 1.126675  1.134528  1.143343  1.153092   1.16373  1.175191  1.187398  1.200266  1.213706  1.227626  1.241939  1.256561  1.271414  1.286427  1.301535   1.31668  1.331811  1.346881  1.361852  1.376689   1.39136  1.405841  1.420108  1.434142  1.447924  1.461442  1.474681   1.48763   1.50028  1.512621  1.524646  1.536349  1.547722   1.55876  1.569458   1.57981  1.589812  1.599459  1.608746   1.61767  1.626225  1.634406  1.642209  1.649629   1.65666  1.663298  1.669536  1.675369  1.680791  1.685794 | 1.356767   1.35631   1.35581  1.355273  1.354705  1.354113  1.353505  1.352888  1.352268  1.351651  1.351043  1.350447  1.349868  1.349307  1.348766  1.348246  1.347749  1.347274  1.346822  1.346391  1.345982  1.345594  1.345227  1.344878  1.344549  1.344237  1.343942  1.343663  1.343399   1.34315  1.342915  1.342692  1.342483  1.342285  1.342098  1.341922  1.341756  1.341599  1.341452  1.341314  1.341184  1.341063  1.340949  1.340843  1.340744  1.340652  1.340567  1.340488  1.340416          1.340351 | 0.554963   0.55866  0.562747  0.567193  0.571957   0.57699  0.582241  0.587657  0.593187  0.598782  0.604399  0.610001  0.615554  0.621031  0.626409  0.631671  0.636804  0.641796   0.64664  0.651332  0.655868  0.660247   0.66447  0.668537  0.672451  0.676214  0.679828  0.683298  0.686626  0.689816  0.692872  0.695797  0.698595  0.701268  0.703822  0.706257  0.708578  0.710788  0.712889  0.714883  0.716772   0.71856  0.720247  0.721835  0.723327  0.724722  0.726023  0.727229  0.728343      0.729364 | -0.225636  -0.230137   -0.23507  -0.240392  -0.246048  -0.251981  -0.258129  -0.264434  -0.270839  -0.277292  -0.283747  -0.290165  -0.296511   -0.30276  -0.308887  -0.314876  -0.320714   -0.32639  -0.331897  -0.337231   -0.34239  -0.347372  -0.352178   -0.35681  -0.361269  -0.365558   -0.36968  -0.373639  -0.377437   -0.38108  -0.384569  -0.387909  -0.391103  -0.394155  -0.397067  -0.399844  -0.402487  -0.404999  -0.407383  -0.409642  -0.411776  -0.413789  -0.415681  -0.417454   -0.41911  -0.420649  -0.422072  -0.423379  -0.424572     -0.425649 | 0.668201  0.668681  0.669212   0.66979   0.67041  0.671069  0.671761   0.67248  0.673221  0.673979  0.674749  0.675526  0.676307  0.677088  0.677865  0.678637    0.6794  0.680153  0.680894  0.681622  0.682335  0.683033  0.683715   0.68438  0.685027  0.685656  0.686267  0.686859  0.687433  0.687987  0.688522  0.689038  0.689534  0.690011  0.690468  0.690906  0.691324  0.691722    0.6921  0.692458  0.692796  0.693114  0.693412  0.693689  0.693945  0.694181  0.694395  0.694589  0.694761      0.694911 | 113.274094 | 114.781134  118.006087  121.706665  125.901139  130.601119  135.811847  141.532732  147.758047  154.477675  161.677869   169.34194  177.450888  185.983921    194.9189  204.232687  213.901426  223.900755  234.205968  244.792134  255.634184  266.706971  277.985311  289.444009  301.057879  312.801744  324.650445  336.578836  348.561782  360.574151  372.590812  384.586622  396.536428  408.415052  420.197297  431.857934  443.371703  454.713316  465.857451  476.778761  487.451873  497.851401  507.951953   517.72814  527.154599  536.206006  544.857099   553.08271  560.857789  568.157447          574.956992 | 578.239266 | 0.747371  0.752833  0.758962  0.765737  0.773125  0.781081  0.789549   0.79847   0.80778  0.817417   0.82732   0.83743  0.847693  0.858059  0.868484  0.878929  0.889357  0.899738  0.910044  0.920253  0.930342  0.940296  0.950098  0.959735  0.969197  0.978472  0.987552  0.996431  1.005101  1.013557  1.021794  1.029807  1.037592  1.045146  1.052466  1.059547  1.066387  1.072983  1.079331   1.08543  1.091276  1.096865  1.102195  1.107263  1.112065  1.116597  1.120856  1.124838  1.128539       1.131953 |  2.74664   2.77882  2.815051  2.855263  2.899304  2.946945  2.997905  3.051864   3.10848  3.167402  3.228283  3.290787  3.354596  3.419414  3.484967  3.551007  3.617308  3.683668  3.749904  3.815855  3.881375  3.946335   4.01062  4.074126  4.136761  4.198442  4.259094  4.318649  4.377047  4.434229  4.490145  4.544746  4.597987  4.649825  4.700219  4.749132  4.796525  4.842361  4.886606  4.929224   4.97018  5.009438  5.046964  5.082721  5.116673  5.148783  5.179014  5.207326   5.23368        5.258035 | 113.274094  116.265714  119.722278  123.665436  128.110098  133.064577  138.531029  144.506095  150.981647  157.945553  165.382414  173.274228  181.600965  190.341054  199.471768  208.969543  218.810217  228.969219  239.421701  250.142641  261.106918  272.289355  283.664758  295.207932  306.893693  318.696872  330.592317   342.55489  354.559456  366.580887  378.594046  390.573788  402.494947  414.332337  426.060745  437.654929  449.089613  460.339492   471.37923  482.183466  492.726818  502.983894    512.9293  522.537656  531.783611  540.641867    549.0872  557.094494  564.638767  571.695218  578.239266 |  100.0 |  13.274094 | 578.239266 |
    | 22:00 | 3.932765  3.978941  4.031463  4.090361  4.155527  4.226731  4.303642  4.385843  4.472865  4.564201  4.659331  4.757733  4.858901  4.962351  5.067625  5.174299   5.28198   5.39031  5.498962  5.607639  5.716073   5.82402  5.931263  6.037602   6.14286  6.246873  6.349495   6.45059  6.550037  6.647721  6.743538  6.837391   6.92919  7.018848  7.106286  7.191426  7.274195  7.354521  7.432334  7.507568  7.580154  7.650028  7.717122  7.781372   7.84271  7.901069  7.956383  8.008581  8.057593             8.103349 | 136.324682  138.843997  141.735441  145.010668   148.67491  152.727264  157.161284  161.965759  167.125596  172.622696  178.436782  184.546122  190.928138  197.559899  204.418497  211.481328  218.726286  226.131889  233.677364  241.342673  249.108526  256.956366  264.868338  272.827255  280.816551  288.820239  296.822861  304.809445  312.765457  320.676761  328.529578  336.310445  344.006185  351.603871  359.090796  366.454446  373.682474  380.762676  387.682971  394.431381  400.996015  407.365049  413.526721   419.46931  425.181133  430.650534   435.86588  440.815555  445.487961  449.871517 |  54.327651   54.789414   55.314635   55.903608   56.555267   57.267314   58.036417   58.858433   59.728652   60.642014   61.593308   62.577332   63.589014   64.623509   65.676251   66.742989   67.819803   68.903104   69.989623   71.076392   72.160727   73.240203   74.312629   75.376025   76.428601   77.468734    78.49495   79.505905   80.500367   81.477207   82.435378    83.37391   84.291896   85.188482    86.06286   86.914262   87.741948   88.545206   89.323342   90.075677   90.801543   91.500279   92.171223   92.813718   93.427098   94.010695   94.563828   95.085809   95.575934     96.033486 |  1.12139  1.128397  1.136333   1.14519  1.154938  1.165529    1.1769  1.188977  1.201679   1.21492  1.228616  1.242685  1.257048  1.271631  1.286368  1.301198  1.316065  1.330922  1.345724  1.360434  1.375019  1.389449  1.403698  1.417745   1.43157  1.445157  1.458491  1.471558  1.484348  1.496851  1.509057   1.52096  1.532551  1.543825  1.554775  1.565396  1.575683  1.585631  1.595235   1.60449  1.613391  1.621934  1.630114  1.637925  1.645364  1.652425  1.659101  1.665388   1.67128   1.67677 |  1.35708  1.356666  1.356207  1.355707  1.355173  1.354611  1.354027  1.353428  1.352821  1.352213  1.351608  1.351012  1.350428   1.34986  1.349309  1.348778  1.348267  1.347778   1.34731  1.346864  1.346439  1.346035  1.345651  1.345287  1.344941  1.344614  1.344304   1.34401  1.343733   1.34347  1.343221  1.342986  1.342764  1.342554  1.342355  1.342168  1.341991  1.341825  1.341667  1.341519   1.34138  1.341249  1.341126  1.341011  1.340904  1.340803   1.34071  1.340623  1.340543           1.34047 | 0.552446  0.555778  0.559502  0.563595  0.568026  0.572754  0.577732  0.582912  0.588244   0.59368  0.599175  0.604688  0.610185  0.615634  0.621009  0.626291   0.63146  0.636506  0.641417  0.646186  0.650809  0.655282  0.659605  0.663777  0.667798  0.671672  0.675398  0.678982  0.682424   0.68573    0.6889   0.69194  0.694853  0.697641  0.700308  0.702857   0.70529  0.707612  0.709824  0.711929  0.713929  0.715827  0.717624  0.719323  0.720924   0.72243  0.723841  0.725159  0.726384      0.727518 | -0.222513  -0.226592  -0.231109  -0.236032  -0.241319  -0.246917  -0.252772  -0.258826  -0.265023  -0.271312  -0.277643  -0.283974  -0.290268  -0.296493  -0.302624   -0.30864  -0.314524  -0.320263  -0.325847   -0.33127  -0.336527  -0.341615  -0.346534  -0.351283  -0.355864  -0.360278  -0.364527  -0.368615  -0.372544  -0.376318  -0.379939  -0.383412  -0.386738  -0.389923  -0.392969  -0.395878  -0.398654  -0.401299  -0.403816  -0.406208  -0.408476  -0.410622  -0.412649  -0.414557  -0.416348  -0.418024  -0.419585  -0.421031  -0.422364     -0.423584 | 0.667865  0.668297  0.668778  0.669307  0.669881  0.670495  0.671145  0.671825  0.672532  0.673259  0.674002  0.674756  0.675518  0.676283  0.677048   0.67781  0.678567  0.679316  0.680055  0.680784    0.6815  0.682202   0.68289  0.683562  0.684217  0.684856  0.685478  0.686083  0.686669  0.687237  0.687787  0.688319  0.688831  0.689325    0.6898  0.690256  0.690693  0.691111  0.691509  0.691888  0.692247  0.692587  0.692908  0.693208  0.693488  0.693748  0.693988  0.694208  0.694406      0.694583 | 111.328777 | 112.648524  115.482516  118.756727  122.493033  126.707214  131.408976  136.602269  142.285793  148.453634   155.09594  162.199586  169.748792  177.725672  186.110694  194.883075  204.021091  213.502323  223.303848  233.402389  243.774418  254.396239  265.244042  276.293941  287.522001  298.904253    310.4167  322.035324  333.736083  345.494907  357.287695  369.090312  380.878579  392.628271  404.315112  415.914768  427.402849  438.754902  449.946414   460.95281   471.74946  482.311678  492.614734  502.633859   512.34426  521.721133  530.739679   539.37513  547.602771  555.397967          562.736206 | 566.297499 | 0.743693  0.748569  0.754088  0.760246  0.767019  0.774374  0.782267  0.790644  0.799448  0.808621  0.818103  0.827836  0.837766  0.847842  0.858018  0.868252  0.878505  0.888744  0.898941  0.909068  0.919104  0.929028  0.938823  0.948475   0.95797  0.967297  0.976447  0.985411  0.994181  1.002751  1.011115  1.019269  1.027207  1.034925   1.04242  1.049687  1.056724  1.063528  1.070095  1.076422  1.082506  1.088344  1.093933   1.09927  1.104351  1.109172  1.113732  1.118024  1.122046       1.125794 | 2.725037  2.753688  2.786228  2.822657  2.862891  2.906769  2.954067  3.004513  3.057801  3.113609  3.171605  3.231465  3.292872   3.35553  3.419158  3.483499  3.548319  3.613404  3.678562  3.743618  3.808417  3.872821  3.936703  3.999953   4.06247  4.124164  4.184954  4.244766  4.303534  4.361195  4.417695  4.472982  4.527007  4.579724  4.631092   4.68107  4.729618  4.776698  4.822276  4.866313  4.908775  4.949626  4.988831  5.026355  5.062161  5.096213  5.128474  5.158906  5.187471        5.214129 | 111.328777  113.947931  116.994969    120.4948  124.466298  128.922161  133.869105  139.308297  145.235948  151.643993  158.520765  165.851653  173.619684  181.806032  190.390444   199.35159  208.667336  218.314967  228.271346  238.513045  249.016433  259.757744  270.713119  281.858643  293.170359  304.624282  316.196406  327.862697  339.599103  351.381537  363.185884  374.987987  386.763646  398.488611  410.138578  421.689185  433.116011  444.394573  455.500327  466.408669  477.094939  487.534428  497.702382  507.574018  517.124532  526.329119  535.162988   543.60139  551.619643  559.193162  566.297499 |  100.0 |  11.328777 | 566.297499 |
    | 22:30 | 3.902316  3.943198  3.990087  4.043107  4.102255  4.167402  4.238311  4.314652  4.396025   4.48198  4.572038  4.665707  4.762499  4.861935  4.963561  5.066945  5.171687  5.277417  5.383792  5.490504  5.597267  5.703827  5.809949  5.915422  6.020056  6.123676  6.226125  6.327258  6.426943  6.525061  6.621499  6.716155  6.808931  6.899738  6.988491  7.075109  7.159516  7.241638  7.321403  7.398742  7.473587  7.545871  7.615528  7.682492  7.746696  7.808075   7.86656  7.922086  7.974582             8.023978 | 134.675101  136.892049  139.455283  142.380198  145.676306  149.347235  153.391061   157.80088  162.565534  167.670407  173.098214  178.829735  184.844453    191.1211  197.638088  204.373844  211.307061  218.416876  225.682983  233.085704  240.606025  248.225604  255.926757  263.692439  271.506206  279.352178  287.215003  295.079808  302.932161  310.758034  318.543755  326.275979  333.941652  341.527973  349.022372  356.412476  363.686083  370.831143  377.835731   384.68803  391.376312  397.888919  404.214255  410.340766  416.256932   421.95126   427.41227  432.628496  437.588478  442.280763 |  54.023162   54.431985   54.900871   55.431074    56.02255   56.674021   57.383109   58.146518   58.960247   59.819797   60.720378   61.657073   62.624988   63.619352   64.635608   65.669452   66.716874   67.774166   68.837923   69.905036   70.972674   72.038268   73.099488   74.154223   75.200559    76.23676   77.261245   78.272576   79.269434   80.250612   81.214993   82.161546   83.089308   83.997378   84.884907    85.75109    86.59516   87.416376   88.214027   88.987418   89.735869   90.458711    91.15528   91.824917    92.46696   93.080746   93.665604   94.220857   94.745816     95.239778 | 1.116753  1.122976  1.130084  1.138088  1.146973  1.156708  1.167245  1.178522  1.190468  1.203004  1.216052  1.229531  1.243364  1.257477  1.271801  1.286273  1.300836  1.315437  1.330031  1.344575  1.359035  1.373376  1.387573  1.401599  1.415433  1.429056  1.442453  1.455607  1.468507  1.481142    1.4935  1.505574  1.517356  1.528838  1.540013  1.550876   1.56142  1.571642  1.581534  1.591093  1.600313   1.60919  1.617719  1.625894  1.633712  1.641166  1.648252  1.654963  1.661295   1.66724 |  1.35736  1.356986  1.356567  1.356107  1.355609  1.355078  1.354522  1.353945  1.353356  1.352759  1.352162  1.351569  1.350984  1.350411  1.349853  1.349312   1.34879  1.348288  1.347806  1.347345  1.346905  1.346486  1.346086  1.345706  1.345346  1.345003  1.344678   1.34437  1.344078  1.343801  1.343539  1.343291  1.343057  1.342834  1.342624  1.342426  1.342238   1.34206  1.341893  1.341735  1.341586  1.341445  1.341313  1.341189  1.341073  1.340964  1.340862  1.340768   1.34068          1.340598 | 0.550217  0.553203  0.556574  0.560318  0.564412  0.568823  0.573512  0.578435  0.583544  0.588794  0.594139  0.599537  0.604951  0.610347  0.615697  0.620975  0.626163  0.631244  0.636206  0.641039  0.645737  0.650293  0.654706  0.658974  0.663096  0.667073  0.670907  0.674599  0.678151  0.681567   0.68485  0.688001  0.691025  0.693925  0.696703  0.699363  0.701907  0.704338   0.70666  0.708874  0.710982  0.712988  0.714893    0.7167  0.718409  0.720022  0.721542  0.722968  0.724302      0.725545 | -0.219738   -0.22341   -0.22752  -0.232045  -0.236953    -0.2422  -0.247739  -0.253516  -0.259477   -0.26557  -0.271747  -0.277961  -0.284173  -0.290349  -0.296459  -0.302479  -0.308388   -0.31417  -0.319815  -0.325311  -0.330652  -0.335835  -0.340855  -0.345712  -0.350406  -0.354937  -0.359306  -0.363517  -0.367571   -0.37147  -0.375219   -0.37882  -0.382275  -0.385589  -0.388763  -0.391802  -0.394707  -0.397482  -0.400129   -0.40265  -0.405048  -0.407325  -0.409482  -0.411522  -0.413445  -0.415253  -0.416948  -0.418529  -0.419998     -0.421356 |  0.66757  0.667955   0.66839  0.668871  0.669398  0.669967  0.670574  0.671215  0.671885  0.672579  0.673293  0.674021  0.674761  0.675507  0.676257  0.677007  0.677755  0.678498  0.679233   0.67996  0.680676   0.68138  0.682072  0.682749  0.683412  0.684059   0.68469  0.685305  0.685903  0.686483  0.687047  0.687592   0.68812  0.688629   0.68912  0.689593  0.690048  0.690484  0.690901    0.6913  0.691679   0.69204  0.692381  0.692704  0.693007   0.69329  0.693554  0.693798  0.694021      0.694225 | 109.650285 | 110.802511  113.285216  116.172809  119.489955  123.255912  127.484306  132.183224  137.355532  142.999362  149.108686  155.673928  162.682555  170.119627  177.968277  186.210128  194.825636  203.794365  213.095215  222.706585  232.606512  242.772767  253.182922  263.814411  274.644558  285.650606  296.809728  308.099039  319.495595  330.976398  342.518391  354.098455  365.693405  377.279985  388.834861  400.334623  411.755776  423.074737  434.267839  445.311323  456.181346  466.853978  477.305211  487.510959  497.447076  507.089356  516.413557   525.39541  534.010643  542.235002          550.044282 | 553.850514 | 0.740465  0.744796  0.749742  0.755308  0.761485  0.768249  0.775566  0.783392  0.791677  0.800367  0.809405  0.818736  0.828305  0.838063   0.84796  0.857953  0.868002  0.878072   0.88813   0.89815  0.908105  0.917974  0.927738   0.93738  0.946886  0.956244  0.965441  0.974469  0.983318  0.991983  1.000455  1.008729    1.0168  1.024664  1.032315  1.039751  1.046967   1.05396  1.060726  1.067263  1.073567  1.079635  1.085464   1.09105  1.096391  1.101483  1.106323  1.110906  1.115229       1.119289 | 2.706121  2.731514  2.760598  2.793435  2.830007  2.870215  2.913897   2.96083  3.010754  3.063376  3.118391  3.175488  3.234361  3.294712  3.356262  3.418747  3.481926  3.545575  3.609492  3.673493  3.737414  3.801104  3.864431  3.927273  3.989523  4.051082  4.111864  4.171788  4.230782  4.288781  4.345723  4.401554  4.456222  4.509679   4.56188  4.612781  4.662343  4.710525  4.757291  4.802603  4.846426  4.888723  4.929459  4.968598  5.006105  5.041944  5.076079  5.108472  5.139085        5.167881 | 109.650285  111.936415  114.613875  117.709968  121.246765   125.24073  129.702665  134.637936  140.046896  145.925432  152.265578  159.056123  166.283189  173.930748  181.981076  190.415126   199.21284  208.353395    217.8154  227.577041  237.616196  247.910522  258.437507  269.174523  280.098847  291.187683  302.418173  313.767402  325.212398  336.730135  348.297525  359.891417  371.488592  383.065755  394.599536  406.066482  417.443056  428.705632  439.830497   450.79385  461.571803  472.140386  482.475549  492.553175  502.349084  511.839048  520.998806  529.804082  538.230606  546.254142  553.850514 |  100.0 |   9.650285 | 553.850514 |
    | 23:00 | 3.875779   3.91182  3.953491  4.000997  4.054424  4.113741  4.178804  4.249368  4.325111  4.405645  4.490543  4.579348  4.671598   4.76683  4.864596  4.964465  5.066033  5.168918  5.272771  5.377268  5.482112  5.587034  5.691787  5.796148  5.899912  6.002896  6.104929   6.20586  6.305546   6.40386  6.500682  6.595902  6.689419  6.781138  6.870969  6.958827  7.044634  7.128311  7.209786  7.288986  7.365844   7.44029  7.512258  7.581682  7.648495  7.712632  7.774026   7.83261  7.888316             7.941077 | 133.244993  135.188985  137.452813  140.054822  143.008149  146.320466  149.994058  154.026174  158.409581  163.133233  168.183001  173.542371  179.193106  185.115812   191.29042  197.696571  204.313916  211.122343  218.102129   225.23405  232.499448  239.880256  247.359017  254.918867  262.543516  270.217224  277.924762  285.651379   293.38276  301.104995  308.804535  316.468161  324.082949  331.636238  339.115599  346.508807  353.803819  360.988742  368.051818  374.981401  381.765935  388.393941  394.854001  401.134741  407.224823   413.11293  418.787761  424.238019   429.45241  434.419636 |  53.757789   54.118201   54.534909   55.009967   55.544243   56.137415   56.788037   57.493682   58.251108   59.056453   59.905426   60.793482   61.715979   62.668303   63.645961   64.644655   65.660325   66.689181   67.727709   68.772676   69.821121    70.87034   71.917872   72.961478   73.999124   75.028958   76.049295   77.058598   78.055462   79.038598   80.006817   80.959022   81.894194   82.811381   83.709689   84.588274   85.446336   86.283109   87.097855   87.889863   88.658438   89.402902   90.122583    90.81682   91.484952   92.126319   92.740257   93.326097   93.883162     94.410766 | 1.112702  1.118202  1.124538  1.131734  1.139791  1.148693  1.158407  1.168883  1.180062  1.191875  1.204249  1.217107  1.230375   1.24398  1.257853  1.271928  1.286146  1.300452  1.314798  1.329139  1.343435  1.357653   1.37176  1.385731  1.399541  1.413169  1.426597  1.439809  1.452789  1.465526  1.478008  1.490226   1.50217  1.513832  1.525205  1.536283  1.547059  1.557527  1.567682  1.577518  1.587031  1.596215  1.605066  1.613578  1.621747  1.629567  1.637033  1.644141  1.650883  1.657255 | 1.357607  1.357272  1.356893  1.356471   1.35601  1.355514  1.354987  1.354437  1.353868  1.353288  1.352702  1.352115  1.351532  1.350958  1.350396  1.349848  1.349317  1.348804   1.34831  1.347835  1.347381  1.346947  1.346532  1.346137  1.345761  1.345404  1.345064  1.344741  1.344435  1.344145  1.343869  1.343608  1.343361  1.343126  1.342904  1.342694  1.342495  1.342306  1.342128   1.34196  1.341801  1.341652   1.34151  1.341377  1.341252  1.341135  1.341024  1.340921  1.340825          1.340736 | 0.548254  0.550915  0.553948  0.557351  0.561108  0.565197  0.569585  0.574233  0.579099  0.584139  0.589309  0.594566  0.599872   0.60519  0.610489  0.615743  0.620929  0.626028  0.631024  0.635906  0.640664  0.645292  0.649785  0.654139  0.658354  0.662427  0.666361  0.670156  0.673814  0.677336  0.680726  0.683986  0.687118  0.690126  0.693013   0.69578  0.698433  0.700972    0.7034  0.705721  0.707936  0.710048  0.712059  0.713971  0.715786  0.717505   0.71913  0.720662  0.722102      0.723452 | -0.217285  -0.220572  -0.224287  -0.228419  -0.232946  -0.237833  -0.243039  -0.248517  -0.254216  -0.260087  -0.266079  -0.272148   -0.27825  -0.284349  -0.290411  -0.296411  -0.302323   -0.30813  -0.313816  -0.319369  -0.324781  -0.330044  -0.335154  -0.340108  -0.344906  -0.349545  -0.354028  -0.358354  -0.362526  -0.366547  -0.370417  -0.374141  -0.377721  -0.381159  -0.384459  -0.387623  -0.390654  -0.393555  -0.396328  -0.398976  -0.401501  -0.403904  -0.406189  -0.408356  -0.410408  -0.412345   -0.41417  -0.415882  -0.417483     -0.418973 |  0.66731  0.667653  0.668043  0.668479   0.66896  0.669484  0.670048  0.670648   0.67128   0.67194  0.672622  0.673323  0.674038  0.674763  0.675495  0.676231  0.676967    0.6777  0.678429  0.679152  0.679866   0.68057  0.681264  0.681944  0.682612  0.683266  0.683904  0.684528  0.685136  0.685727  0.686302   0.68686  0.687401  0.687925  0.688431   0.68892  0.689391  0.689844  0.690278  0.690695  0.691094  0.691474  0.691836  0.692179  0.692503  0.692808  0.693094  0.693361  0.693609      0.693837 | 108.205287 |  109.20832  111.376768  113.915382  116.850836  120.205156  123.995303  128.233044   132.92508  138.073371  143.675592  149.725659  156.214282  163.129487   170.45711  178.181225   186.28451  194.748556  203.554111  212.681282  222.109688  231.818577  241.786918  251.993466  262.416809  273.035401  283.827587  294.771613  305.845638  317.027737  328.295898  339.628025  351.001934  362.395348  373.785891  385.151089  396.468363  407.715024  418.868277   429.90521  440.802803  451.537924   462.08733  472.427674  482.535513  492.387308  501.959443  511.228231  520.169934   528.76078          536.976984 | 540.995797 | 0.737645  0.741474  0.745884   0.75089  0.756493  0.762681  0.769429  0.776703   0.78446  0.792653  0.801229  0.810136   0.81932  0.828732  0.838323  0.848047  0.857865  0.867737  0.877631  0.887516  0.897364  0.907154  0.916862  0.926472  0.935966  0.945331  0.954555  0.963626  0.972535  0.981273  0.989834   0.99821  1.006396  1.014386  1.022176  1.029762  1.037138  1.044303  1.051251  1.057979  1.064485  1.070765  1.076815  1.082634  1.088216  1.093559   1.09866  1.103515   1.10812       1.112471 | 2.689621  2.712027  2.737902  2.767359  2.800437  2.837101  2.877245  2.920701  2.967253  3.016649  3.068612  3.122852  3.179075  3.236992  3.296325  3.356809  3.418196  3.480257  3.542781  3.605575  3.668465  3.731292  3.793914  3.856201  3.918038   3.97932  4.039951  4.099847  4.158929  4.217127  4.274376  4.330615  4.385791  4.439851  4.492749  4.544438  4.594877  4.644024  4.691841   4.73829  4.783333  4.826934  4.869058  4.909668   4.94873  4.986208  5.022065  5.056266  5.088773        5.119548 | 108.205287  110.194932  112.540372  115.270486   118.40979  121.977856  125.989051  130.452548   135.37257  140.748798  146.576882  152.848985  159.554335  166.679739  174.210041  182.128525  190.417249  199.057323  208.029124  217.312478   226.88679  236.731146  246.824393   257.14519   267.67205  278.383368  289.257434   300.27245  311.406531  322.637709  333.943931  345.303056  356.692855  368.091002  379.475074  390.822544  402.110779  413.317038  424.418466    435.3921   446.21486   456.86356  467.314903  477.545492  487.531834  497.250347  506.677378  515.789208  524.562078  532.972203  540.995797 |  100.0 |   8.205287 | 540.995797 |
    | 23:30 |  3.85273  3.884375  3.921249  3.963622  4.011658  4.065411  4.124827  4.189747  4.259927  4.335048  4.414739  4.498589  4.586169  4.677038  4.770761  4.866913  4.965091  5.064908  5.166007  5.268055  5.370743   5.47379  5.576937  5.679948  5.782608  5.884721  5.986107  6.086603   6.18606   6.28434  6.381317  6.476874  6.570905  6.663307  6.753987  6.842857  6.929833  7.014835  7.097787  7.178615  7.257249  7.333619  7.407657  7.479297  7.548473  7.615117  7.679166  7.740551  7.799207             7.855066 | 132.008587  133.707479  135.699711  138.005812  140.641862  143.619021  146.943384  150.616106  154.633773  158.988922  163.670671  168.665372  173.957247  179.528973  185.362192  191.437937  197.736981  204.240099  210.928274  217.782838   224.78557  231.918751  239.165203  246.508289  253.931913  261.420501  268.958974   276.53272  284.127557  291.729708  299.325755  306.902617  314.447507   321.94791  329.391546  336.766347   344.06043  351.262071  358.359683  365.341797   372.19704  378.914118  385.481802  391.888909  398.124293   404.17683   410.03541  415.688927   421.12627  426.336323 |  53.527299   53.843752   54.212491   54.636217   55.116579   55.654114   56.248268    56.89747   57.599266   58.350479   59.147387   59.985894   60.861687   61.770377   62.707606   63.669135   64.650906   65.649082   66.660074   67.680549   68.707433   69.737902   70.769372   71.799483   72.826082   73.847207   74.861068   75.866029   76.860596   77.843395   78.813166   79.768743   80.709046    81.63307   82.539875   83.428574   84.298331   85.148349   85.977867   86.786151   87.572489   88.336189   89.076574   89.792974   90.484727   91.151174   91.791657   92.405511   92.992072     93.550662 | 1.109176  1.114016  1.119638  1.126075  1.133345  1.141443  1.150352  1.160035  1.170445  1.181524  1.193204  1.205417  1.218091  1.231155   1.24454  1.258181  1.272016  1.285989  1.300049  1.314148  1.328246  1.342304  1.356288   1.37017  1.383923  1.397524  1.410953  1.424191  1.437222  1.450033  1.462611  1.474945  1.487024   1.49884  1.510385  1.521651  1.532632  1.543321  1.553713  1.563801   1.57358  1.583046  1.592193  1.601015  1.609509  1.617669  1.625489  1.632965   1.64009   1.64686 | 1.357825  1.357527  1.357186  1.356802  1.356378  1.355917  1.355423    1.3549  1.354356  1.353795  1.353224  1.352647  1.352071  1.351499  1.350935  1.350383  1.349845  1.349323  1.348818  1.348332  1.347864  1.347416  1.346988  1.346578  1.346187  1.345815  1.345461  1.345124  1.344804  1.344499   1.34421  1.343936  1.343676  1.343429  1.343195  1.342973  1.342762  1.342563  1.342374  1.342196  1.342027  1.341868  1.341717  1.341575   1.34144  1.341314  1.341196  1.341084   1.34098          1.340882 | 0.546533  0.548892  0.551606  0.554679  0.558106  0.561872  0.565951  0.570313  0.574919  0.579728    0.5847  0.589792  0.594964   0.60018  0.605406  0.610613  0.615776  0.620873  0.625886  0.630801  0.635606  0.640292  0.644853  0.649283  0.653581  0.657744   0.66177  0.665662  0.669419  0.673043  0.676536    0.6799  0.683137   0.68625  0.689242  0.692116  0.694873  0.697517  0.700051  0.702476  0.704796  0.707012  0.709126  0.711142   0.71306  0.714882  0.716611  0.718246   0.71979      0.721244 | -0.215126  -0.218052  -0.221391  -0.225141   -0.22929  -0.233812  -0.238675  -0.243837  -0.249253  -0.254876  -0.260658  -0.266553  -0.272517  -0.278512  -0.284501  -0.290456  -0.296349  -0.302159  -0.307868  -0.313461  -0.318927  -0.324257  -0.329444  -0.334485  -0.339375  -0.344114  -0.348701  -0.353136   -0.35742  -0.361555  -0.365542  -0.369384  -0.373083  -0.376642  -0.380063  -0.383348  -0.386502  -0.389525  -0.392421  -0.395192   -0.39784  -0.400367  -0.402776  -0.405067  -0.407244  -0.409307  -0.411257  -0.413097  -0.414826     -0.416445 | 0.667083  0.667387  0.667735  0.668128  0.668565  0.669045  0.669566  0.670125  0.670718  0.671341   0.67199  0.672661   0.67335  0.674052  0.674763  0.675482  0.676204  0.676926  0.677646  0.678362  0.679072  0.679774  0.680467  0.681149   0.68182  0.682478  0.683123  0.683753  0.684369   0.68497  0.685555  0.686125  0.686678  0.687214  0.687734  0.688237  0.688723  0.689192  0.689643  0.690077  0.690493  0.690892  0.691272  0.691634  0.691979  0.692305  0.692612  0.692901  0.693171      0.693422 | 106.964036 |  107.83474  109.723235  111.948232  114.537679  117.515684   120.90193  124.711364  128.954138  133.635769  138.757455   144.31651  150.306838  156.719436  163.542862  170.763671  178.366805  186.335911  194.653621  203.301771  212.261581  221.513792  231.038773  240.816606   250.82714  261.050041   271.46482  282.050852  292.787392  303.653585  314.628463  325.690953   336.81987  347.993919  359.191689   370.39165  381.572149  392.711407  403.787515  414.778434  425.661989  436.415876  447.017654  457.444754  467.674479  477.684011  487.450417  496.950659  506.161606  515.060048          523.622716 | 527.823549 | 0.735189  0.738559  0.742473  0.746953   0.75201  0.757642  0.763833   0.77056  0.777787  0.785474  0.793574  0.802039  0.810817   0.81986  0.829119  0.838549  0.848108  0.857756  0.867459  0.877183    0.8869  0.896585  0.906214  0.915768  0.925228   0.93458  0.943809  0.952902   0.96185  0.970644  0.979274  0.987733  0.996015  1.004114  1.012025  1.019742  1.027262   1.03458  1.041692  1.048596  1.055286   1.06176  1.068015  1.074047  1.079853  1.085429  1.090773  1.095881  1.100748       1.105372 | 2.675279  2.694968  2.717885  2.744187  2.773964  2.807233  2.843946   2.88399  2.927196  2.973353   3.02222  3.073532  3.127013  3.182386   3.23938   3.29773  3.357187  3.417517  3.478503  3.539944  3.601657  3.663477   3.72525  3.786841  3.848126  3.908991  3.969335  4.029067  4.088103  4.146367  4.203789  4.260305  4.315857   4.37039  4.423854    4.4762  4.527385  4.577365  4.626101  4.673552  4.719681  4.764451  4.807825  4.849768  4.890245  4.929219  4.966655  5.002516  5.036768        5.069372 | 106.964036  108.690799  110.739252   113.13911  115.916605  119.093761  122.687947  126.711701  131.172791  136.074474  141.415879  147.192483  153.396602  160.017887  167.043784  174.459944  182.250579   190.39876  198.886666  207.695783  216.807055  226.201013  235.857861   245.75755  255.879828  266.204275  276.710329  287.377302  298.184392  309.110686  320.135164  331.236696  342.394044  353.585855  364.790658  375.986865  387.152758  398.266495  409.306102  420.249473  431.074368  441.758413  452.279104  462.613806  472.739759  482.634086  492.273799  501.635809  510.696943  519.433955  527.823549 |  100.0 |   6.964036 | 527.823549 |
    | 00:00 | 3.832776  3.860455  3.892951  3.930581  3.973575  4.022062  4.076068  4.135519  4.200248  4.270009  4.344491  4.423336  4.506153  4.592532  4.682059  4.774319  4.868914  4.965459  5.063591   5.16297  5.263279  5.364226   5.46554  5.566975  5.668305  5.769321  5.869836  5.969675   6.06868  6.166705  6.263617  6.359292  6.453616  6.546483  6.637794  6.727455  6.815379  6.901484  6.985689   7.06792  7.148104  7.226169  7.302047  7.375669   7.44697  7.515883  7.582343  7.646284  7.707639             7.766343 |  130.94249  132.422396  134.169578  136.206034  138.550104  141.215847  144.212698  147.545395  151.214147  155.215006  159.540368  164.179546   169.11937  174.344757  179.839237  185.585411  191.565328  197.760801  204.153643  210.725846  217.459718  224.337963   231.34374  238.460694   245.67296   252.96516  260.322393  267.730206  275.174567  282.641844  290.118764  297.592387  305.050077  312.479466  319.868433  327.205068  334.477656  341.674644  348.784623  355.796306  362.698508  369.480127  376.130128   382.63753  388.991385  395.180775  401.194791  407.022531  412.653086  418.075536 |  53.327758   53.604551   53.929505   54.305807   54.735748   55.220621   55.760685   56.355194   57.002482   57.700088   58.444909    59.23336    60.06153   60.925324   61.820585   62.743193    63.68914    64.65459   65.635911     66.6297   67.632791   68.642256     69.6554   70.669752   71.683047   72.693213   73.698357   74.696748   75.686798   76.667052   77.636172   78.592924   79.536165   80.464833   81.377937   82.274549   83.153792   84.014837   84.856894   85.679205   86.481039    87.26169   88.020467   88.756693   89.469703   90.158835   90.823432   91.462837   92.076391     92.663427 | 1.106117  1.110359  1.115325  1.121058  1.127584  1.134915  1.143045   1.15195  1.161597  1.171936  1.182912   1.19446  1.206515  1.219009  1.231874  1.245046  1.258464  1.272068  1.285805  1.299628  1.313491  1.327354  1.341181  1.354941  1.368605  1.382148  1.395547  1.408781  1.421835  1.434692  1.447337   1.45976  1.471948  1.483892  1.495583  1.507012  1.518173  1.529058  1.539661  1.549976  1.559997  1.569719  1.579137  1.588244  1.597038  1.605511  1.613659  1.621477  1.628959    1.6361 | 1.358016  1.357752  1.357447    1.3571  1.356713  1.356288  1.355827  1.355335  1.354818  1.354279  1.353726  1.353163  1.352596   1.35203  1.351468  1.350914  1.350372  1.349843  1.349329  1.348833  1.348354  1.347893  1.347451  1.347028  1.346623  1.346237  1.345869  1.345517  1.345183  1.344865  1.344563  1.344275  1.344002  1.343742  1.343496  1.343262  1.343041   1.34283  1.342631  1.342442  1.342263  1.342093  1.341933  1.341781  1.341638  1.341503  1.341376  1.341256  1.341144          1.341038 | 0.545032  0.547112  0.549526  0.552286  0.555394  0.558841   0.56261  0.566677  0.571009  0.575571  0.580324  0.585228  0.590244  0.595334  0.600463  0.605601   0.61072  0.615795  0.620806  0.625737  0.630574  0.635305  0.639921  0.644418  0.648789  0.653031  0.657143  0.661125  0.664975  0.668695  0.672286   0.67575  0.679088  0.682303  0.685397  0.688373  0.691233   0.69398  0.696616  0.699144  0.701565  0.703883    0.7061  0.708217  0.710236   0.71216  0.713989  0.715726  0.717372      0.718927 | -0.213236  -0.215826   -0.21881  -0.222194  -0.225973  -0.230132  -0.234645  -0.239479  -0.244596   -0.24995  -0.255497  -0.261193  -0.266993  -0.272857  -0.278748  -0.284634  -0.290484  -0.296276  -0.301988  -0.307602  -0.313106  -0.318487  -0.323739  -0.328853  -0.333826  -0.338655  -0.343337  -0.347873  -0.352261  -0.356504  -0.360602  -0.364556   -0.36837  -0.372044  -0.375582  -0.378985  -0.382257  -0.385399  -0.388414  -0.391305  -0.394073   -0.39672   -0.39925  -0.401662   -0.40396  -0.406145  -0.408218  -0.410181  -0.412035      -0.41378 | 0.666885  0.667153  0.667462  0.667815   0.66821  0.668648  0.669127  0.669645  0.670198  0.670784  0.671398  0.672037  0.672697  0.673373  0.674063  0.674762  0.675467  0.676175  0.676885  0.677592  0.678296  0.678993  0.679684  0.680366  0.681038  0.681698  0.682347  0.682983  0.683606  0.684214  0.684808  0.685387  0.685951  0.686499  0.687032  0.687548  0.688047  0.688531  0.688997  0.689447  0.689879  0.690295  0.690693  0.691073  0.691437  0.691782   0.69211  0.692419   0.69271      0.692983 | 105.900084 | 106.653874  108.293973  110.238325  112.515493  115.151005   118.16666  121.580057  125.404373  129.648357  134.316507  139.409387  144.924024  150.854347  157.191638  163.924958  171.041533  178.527104  186.366211  194.542446  203.038646  211.837056  220.919456  230.267252  239.861558  249.683244   259.71298  269.931265  280.318441  290.854713  301.520152  312.294698  323.158165  334.090235  345.070459  356.078253  367.092894  378.093518  389.059114  399.968526  410.800445  421.533411  432.145811  442.615882  452.921708   463.04123  472.952243  482.632411  492.059267  501.210232          510.062621 | 514.416746 | 0.733058  0.736013  0.739471  0.743462  0.748003  0.753102  0.758755  0.764944  0.771644  0.778822  0.786437  0.794445  0.802799  0.811452  0.820357  0.829469  0.838745  0.848144   0.85763  0.867168   0.87673  0.886286  0.895812  0.905287  0.914691  0.924007   0.93322  0.942317  0.951284  0.960113  0.968794  0.977318  0.985678  0.993868  1.001882  1.009714   1.01736  1.024815  1.032074  1.039135  1.045993  1.052645  1.059086  1.065315  1.071327   1.07712  1.082689  1.088032  1.093144       1.098023 | 2.662853  2.680087    2.7003   2.72368   2.75036  2.780407  2.813823  2.850547   2.89046  2.933396  2.979149  3.027486  3.078155  3.130894  3.185443  3.241542  3.298943   3.35741  3.416722  3.476672  3.537071  3.597744  3.658531  3.719289  3.779885  3.840199  3.900124  3.959561   4.01842  4.076621  4.134088  4.190754  4.246555  4.301435  4.355338  4.408216   4.46002  4.510707  4.560233  4.608558  4.655644  4.701452  4.745946  4.789089  4.830845  4.871179  4.910055  4.947438  4.983291        5.017579 | 105.900084  107.394663   109.17857  111.281706  113.731343  116.551312  119.761411  123.377062  127.409206    131.8644  136.745077  142.049917  147.774276  153.910639  160.449065  167.377598  174.682635   182.34924  190.361421  198.702342  207.354504   216.29989  225.520069  234.996282  244.709508  254.640507  264.769857  275.077974  285.545128  296.151455  306.876959  317.701518  328.604879   339.56666  350.566344  361.583278  372.596671  383.585586  394.528939  405.405497  416.193878  426.872544  437.419807  447.813826  458.032614  468.054037  477.855822  487.415565   496.71074   505.71871  514.416746 |  100.0 |   5.900084 | 514.416746 |
    | 00:30 | 3.815554  3.839677  3.868202  3.901482  3.939796  3.983337  4.032204  4.086398  4.145829  4.210323  4.279637  4.353466  4.431466  4.513264   4.59847   4.68669  4.777533  4.870622  4.965592  5.062098  5.159819  5.258453   5.35772  5.457363  5.557146  5.656851  5.756279  5.855246  5.953586  6.051144  6.147779   6.24336  6.337767  6.430887  6.522616  6.612857  6.701517  6.788511  6.873757  6.957174   7.03869  7.118231  7.195726  7.271107  7.344307  7.415259  7.483898  7.550158  7.613973             7.675278 | 130.025594  131.310769   132.83798  134.630044   136.70691  139.084936  141.776388  144.789189  148.126911   151.78897  155.770999  160.065315  164.661453  169.546711  174.706665  180.125642  185.787129  191.674118  197.769379  204.055682  210.515955  217.133406  223.891601  230.774514  237.766559  244.852591  252.017911  259.248249  266.529742  273.848914  281.192649  288.548159  295.902961  303.244846   310.56185  317.842231  325.074441  332.247104  339.348991  346.369002  353.296141  360.119504  366.828257   373.41162  379.858857  386.159257  392.302126  398.276773  404.072502  409.678605 |  53.155544   53.396773   53.682023   54.014821   54.397961   54.833372   55.322037   55.863976   56.458287   57.103235   57.796367   58.534659   59.314661   60.132639   60.984697   61.866896   62.775334   63.706219   64.655917   65.620985   66.598191   67.584526   68.577197   69.573632   70.571461   71.568511   72.562788   73.552464   74.535863   75.511445   76.477794   77.433605    78.37767    79.30887   80.226162   81.128568   82.015174   82.885114   83.737565   84.571745   85.386901   86.182307    86.95726   87.711073   88.443073   89.152595   89.838981   90.501577   91.139728     91.752777 | 1.103472  1.107176  1.111544  1.116626  1.122459  1.129063  1.136445  1.144595   1.15349  1.163094  1.173359   1.18423  1.195647  1.207547  1.219865  1.232538  1.245504  1.258705  1.272086  1.285597  1.299191  1.312826  1.326463  1.340069  1.353612  1.367065  1.380404  1.393606  1.406654  1.419528  1.432215  1.444699   1.45697  1.469016  1.480828  1.492395  1.503711  1.514767  1.525558  1.536075  1.546314  1.556268  1.565932    1.5753  1.584368   1.59313  1.601581  1.609716  1.617529  1.625015 | 1.358183   1.35795  1.357679  1.357368  1.357017  1.356627    1.3562  1.355741  1.355252  1.354739  1.354207  1.353661  1.353107  1.352549  1.351992   1.35144  1.350896  1.350362  1.349842  1.349337  1.348848  1.348376  1.347922  1.347486  1.347068  1.346668  1.346286  1.345921  1.345573  1.345241  1.344925  1.344625  1.344339  1.344067  1.343808  1.343562  1.343329  1.343108  1.342897  1.342698  1.342508  1.342329  1.342159  1.341998  1.341846  1.341702  1.341566  1.341437  1.341316          1.341203 | 0.543727  0.545552   0.54769  0.550155  0.552957  0.556093  0.559554  0.563323  0.567373  0.571675  0.576193  0.580889  0.585727  0.590668  0.595679  0.600725  0.605778  0.610811  0.615802  0.620731  0.625583  0.630344  0.635004  0.639554  0.643987    0.6483   0.65249  0.656553   0.66049    0.6643  0.667984  0.671542  0.674977   0.67829  0.681483  0.684559  0.687518  0.690365  0.693101  0.695729   0.69825  0.700668  0.702984    0.7052  0.707319  0.709342   0.71127  0.713106  0.714851      0.716506 | -0.211587   -0.21387  -0.216523  -0.219558   -0.22298  -0.226782  -0.230945  -0.235445  -0.240249  -0.245318   -0.25061  -0.256083  -0.261694  -0.267403   -0.27317  -0.278962  -0.284747  -0.290498  -0.296192  -0.301809  -0.307333  -0.312751  -0.318051  -0.323227   -0.32827  -0.333178  -0.337946  -0.342574  -0.347059  -0.351403  -0.355605  -0.359666  -0.363588  -0.367374  -0.371024  -0.374541  -0.377926  -0.381184  -0.384314  -0.387321  -0.390206   -0.39297  -0.395617  -0.398147  -0.400564  -0.402867   -0.40506  -0.407143  -0.409117     -0.410983 | 0.666714  0.666949  0.667222  0.667537  0.667892  0.668289  0.668727  0.669205  0.669719  0.670267  0.670846  0.671452  0.672081   0.67273  0.673395  0.674072  0.674758  0.675451  0.676147  0.676843  0.677538   0.67823  0.678916  0.679595  0.680267  0.680928  0.681579  0.682219  0.682847  0.683462  0.684063   0.68465  0.685223  0.685782  0.686325  0.686853  0.687365  0.687861  0.688342  0.688806  0.689254  0.689685    0.6901  0.690497  0.690878  0.691242  0.691588  0.691917  0.692228      0.692522 |  104.99001 | 105.640887  107.061431  108.755669  110.752196  113.077356  115.754471  118.803258  122.239462  126.074701  130.316503  134.968493  140.030701   145.49993  151.370162  157.632964  164.277868  171.292729  178.664025  186.377122  194.416497  202.765909  211.408555  220.327171  229.504129  238.921501  248.561109  258.404561  268.433279  278.628517  288.971371  299.442785  310.023561   320.69435  331.435658  342.227843   353.05111  363.885508   374.71093  385.507104  396.253595  406.929802  417.514955  427.988114  438.328171  448.513853  458.523717  468.336166  477.929445  487.281654          496.370756 | 500.851295 | 0.731216  0.733796  0.736838  0.740377  0.744437  0.749032  0.754166  0.759832  0.766014  0.772684  0.779809  0.787351  0.795268  0.803514  0.812045  0.820816  0.829785  0.838912  0.848157  0.857486  0.866867  0.876271  0.885672  0.895046  0.904372  0.913631  0.922808  0.931887  0.940854    0.9497  0.958412  0.966983  0.975404  0.983667  0.991767  0.999697  1.007452  1.015027  1.022418  1.029619  1.036628  1.043441  1.050054  1.056462  1.062664  1.068656  1.074434  1.079994  1.085334       1.090449 | 2.652123  2.667152  2.684908  2.705603  2.729402  2.756414  2.786686  2.820208  2.856908  2.896666  2.939315  2.984656  3.032465    3.0825  3.134515  3.188261  3.243494  3.299978   3.35749   3.41582  3.474773  3.534167  3.593837  3.653629  3.713406   3.77304  3.832417  3.891431  3.949988  4.008001   4.06539  4.122082  4.178011  4.233114  4.287336  4.340623  4.392925  4.444195  4.494389  4.543466  4.591385  4.638106  4.683592  4.727807  4.770714  4.812277  4.852461   4.89123  4.928549        4.964383 |  104.99001  106.280272  107.829468   109.66713  111.820966     114.316  117.173889  120.412438  124.045348  128.082162  132.528389  137.385765  142.652598  148.324169  154.393146  160.849977  167.683265  174.880098  182.426333  190.306835   198.50568  207.006308  215.791659  224.844266  234.146334  243.679796   253.42636  263.367534  273.484652  283.758886  294.171253  304.702627  315.333732  326.045149  336.817309  347.630495  358.464833  369.300297  380.116696  390.893678  401.610727  412.247155  422.782109  433.194565  443.463329  453.567045  463.484191  473.193088  482.671908  491.898677  500.851295 |  100.0 |    4.99001 | 500.851295 |
    | 01:00 | 3.800735  3.821687  3.846634  3.875949   3.90995   3.94888  3.992899   4.04208  4.096402  4.155765  4.219989  4.288831  4.361998  4.439156   4.51995  4.604008  4.690958  4.780428  4.872059  4.965508  5.060446  5.156567  5.253584  5.351231  5.449261  5.547448  5.645583  5.743473  5.840942  5.937829  6.033982  6.129266  6.223551  6.316721  6.408666  6.499282  6.588475  6.676153  6.762232   6.84663  6.929269  7.010074  7.088975    7.1659  7.240782  7.313553  7.384148  7.452501  7.518546              7.58222 | 129.238958  130.351743  131.682449  133.254147  135.087795  137.201463  139.609718   142.32323  145.348601  148.688398  152.341378  156.302843  160.565085  165.117882  169.948986  175.044598  180.389788  185.968868    191.7657  197.763942  203.947247   210.29941  216.804476   223.44681  230.211149  237.082621  244.046762  251.089506  258.197178  265.356477  272.554451  279.778477  287.016229  294.255662  301.484975  308.692595  315.867147   322.99743    330.0724  337.081143  344.012859   350.85684  357.602455  364.239134  370.756352  377.143612  383.390438   389.48636  395.420904  401.183582 |   53.00735   53.216866   53.466335   53.759492     54.0995   54.488799   54.928994   55.420796   55.964024   56.557652   57.199891   57.888314   58.619977   59.391562   60.199499   61.040085   61.909577   62.804278   63.720593   64.655076   65.604458   66.565668   67.535839   68.512307   69.492611   70.474481   71.455827    72.43473   73.409424   74.378287   75.339824   76.292657   77.235514   78.167213   79.086658   79.992824   80.884751   81.761535   82.622321   83.466298   84.292688   85.100745    85.88975   86.659002    87.40782   88.135533   88.841481   89.525009   90.185465     90.822198 | 1.101193  1.104415  1.108242  1.112728  1.117917  1.123838   1.13051  1.137933  1.146096  1.154973  1.164529  1.174716  1.185482  1.196769  1.208516  1.220663  1.233149  1.245916  1.258908  1.272074  1.285366   1.29874  1.312155  1.325574  1.338965  1.352299  1.365548   1.37869  1.391703  1.404568  1.417268  1.429789  1.442117   1.45424  1.466147  1.477828  1.489275  1.500479  1.511432  1.522129  1.532561  1.542724  1.552611  1.562216  1.571535  1.580562  1.589292  1.597718  1.605838  1.613644 | 1.358328  1.358124  1.357883  1.357606  1.357289  1.356934  1.356542  1.356115  1.355657  1.355172  1.354663  1.354138  1.353599  1.353053  1.352504  1.351957  1.351414  1.350879  1.350354  1.349843  1.349346  1.348864  1.348399  1.347951  1.347521  1.347108  1.346713  1.346334  1.345973  1.345628  1.345299  1.344985  1.344686  1.344402  1.344131  1.343873  1.343628  1.343395  1.343174  1.342963  1.342764  1.342574  1.342394  1.342224  1.342062  1.341909  1.341764  1.341627  1.341498          1.341376 | 0.542597  0.544192  0.546076  0.548267  0.550778  0.553615  0.556775  0.560246   0.56401  0.568042  0.572311  0.576784  0.581424  0.586197  0.591067     0.596  0.600965  0.605936  0.610887  0.615797  0.620648  0.625424  0.630112  0.634703  0.639188  0.643562  0.647819  0.651956  0.655972  0.659866  0.663637  0.667285  0.670812  0.674218  0.677506  0.680677  0.683734  0.686678  0.689511  0.692236  0.694855   0.69737  0.699784  0.702097  0.704313  0.706433  0.708458  0.710392  0.712234      0.713986 | -0.210155  -0.212158  -0.214504  -0.217213  -0.220295  -0.223749  -0.227567  -0.231731  -0.236214  -0.240985  -0.246005  -0.251236  -0.256635  -0.262165  -0.267784  -0.273458  -0.279154  -0.284843  -0.290498  -0.296098  -0.301625  -0.307062  -0.312396  -0.317619  -0.322721  -0.327696   -0.33254   -0.33725  -0.341824  -0.346261   -0.35056  -0.354722  -0.358748  -0.362638  -0.366396  -0.370021  -0.373517  -0.376886  -0.380128  -0.383247  -0.386245  -0.389123  -0.391884  -0.394529   -0.39706   -0.39948  -0.401788  -0.403988  -0.406079     -0.408064 | 0.666565  0.666771  0.667012  0.667291  0.667609  0.667968  0.668366  0.668804  0.669279  0.669789  0.670332  0.670904  0.671501  0.672121   0.67276  0.673413  0.674079  0.674753  0.675434  0.676117  0.676802  0.677485  0.678165   0.67884  0.679508  0.680169  0.680821  0.681463  0.682094  0.682714  0.683321  0.683915  0.684496  0.685063  0.685616  0.686154  0.686678  0.687186  0.687679  0.688157  0.688618  0.689064  0.689494  0.689908  0.690305  0.690686   0.69105  0.691397  0.691727       0.69204 | 104.213141 | 104.773748  106.000939  107.473146  109.218505  111.263631  113.632815  116.347365  119.425123  122.880184   126.72279  130.959404    135.5929   140.62286  146.045917  151.856126  158.045333   164.60352  171.519124  178.779309  186.370205  194.277104  202.484623  210.976835  219.737368  228.749488   237.99616  247.460093  257.123774  266.969492  276.979354  287.135296  297.419091  307.812348  318.296518  328.852889  339.462585  350.106567  360.765621  371.420364  382.051236  392.638498   403.16223  413.602329  423.938508  434.150296  444.217042  454.117912  463.831897  473.337815          482.614323 | 487.196237 | 0.729628  0.731872  0.734539  0.737663  0.741275  0.745397  0.750038  0.755201  0.760875  0.767044   0.77368  0.780751   0.78822  0.796045  0.804186  0.812597  0.821239   0.83007  0.839052  0.848148  0.857327  0.866556  0.875808  0.885059  0.894286  0.903468  0.912588  0.921629  0.930578  0.939421  0.948147  0.956747  0.965211  0.973531  0.981699  0.989711  0.997558  1.005237  1.012742  1.020069  1.027213  1.034171  1.040938  1.047511  1.053887  1.060061  1.066031  1.071793  1.077343       1.082679 | 2.642885  2.655945  2.671483  2.689727  2.710865   2.73504  2.762341  2.792799  2.826391  2.863038  2.902618  2.944967   2.98989  3.037173  3.086586  3.137893  3.190859  3.245252  3.300849  3.357439  3.414823  3.472813  3.531238  3.589939  3.648771    3.7076  3.766305  3.824774  3.882907  3.940611  3.997801  4.054401  4.110338  4.165548   4.21997  4.273549  4.326231  4.377967  4.428712  4.478421  4.527053  4.574567  4.620926   4.66609  4.710024  4.752691  4.794055  4.834083  4.872737        4.909983 | 104.213141  105.324238  106.665989  108.267098   110.15518  112.355895  114.892203   117.78379   121.04668  124.693048  128.731211  133.165771   137.99787  143.225514  148.843948  154.846023  161.222564  167.962697  175.054154  182.483521   190.23646  198.297887  206.652113  215.282964  224.173867  233.307923  242.667958  252.236562  261.996115  271.928813  282.016675  292.241552  302.585135  313.028954  323.554379  334.142617  344.774709  355.431531  366.093785  376.741999  387.356525  397.917534  408.405015  418.798772  429.078427  439.223419  449.213002  459.026256  468.642083  478.039218  487.196237 |  100.0 |   4.213141 | 487.196237 |
    | 01:30 | 3.788018  3.806157  3.827897  3.853624  3.883676  3.918337   3.95782  4.002254  4.051689  4.106088   4.16534  4.229261  4.297613   4.37011  4.446431  4.526237  4.609174  4.694887  4.783025  4.873246  4.965224  5.058647  5.153224  5.248681  5.344763  5.441235  5.537879  5.634495  5.730896  5.826914   5.92239   6.01718  6.111149  6.204173  6.296137  6.386934  6.476462  6.564628  6.651343  6.736521  6.820083  6.901952  6.982054  7.060317  7.136673  7.211052   7.28339   7.35362  7.421677             7.487495 | 128.565676  129.526487  130.682455  132.056426  133.669834  135.541901  137.688971  140.123997  142.856232  145.891119  149.230363  152.872172  156.811592  161.040936  165.550232  170.327677   175.36006  180.633149  186.132023  191.841348  197.745607  203.829274  210.076945  216.473443   223.00388    229.6537  236.408707  243.255067  250.179316  257.168341   264.20937  271.289949  278.397922   285.52141   292.64878   299.76863  306.869757  313.941143  320.971924  327.951376  334.868892  341.713966  348.476169  355.145141  361.710569  368.162177   374.48971  380.682921  386.731563  392.625377 |  52.880181   53.061566   53.278974   53.536236    53.83676   54.183374   54.578196    55.02254   55.516886   56.060881   56.653396   57.292612   57.976132   58.701096   59.464314   60.262372   61.091744   61.948874   62.830251   63.732464    64.65224   65.586473    66.53224   67.486805   68.447627   69.412348    70.37879   71.344946   72.308964    73.26914     74.2239   75.171796   76.111487    77.04173   77.961372   78.869339   79.764623   80.646282   81.513425   82.365208   83.200828   84.019517   84.820536   85.603171   86.366726   87.110525   87.833901   88.536198   89.216766     89.874955 | 1.099235  1.102028  1.105368  1.109313  1.113909  1.119195  1.125195  1.131924  1.139379  1.147547  1.156401  1.165904  1.176011  1.186672  1.197829  1.209427  1.221407  1.233711  1.246285  1.259076  1.272035  1.285115  1.298276  1.311479  1.324688  1.337872  1.351003  1.364055  1.377006  1.389834  1.402522  1.415054  1.427413  1.439588  1.451567  1.463338  1.474891  1.486219  1.497313  1.508165  1.518769  1.529117  1.539204  1.549024  1.558571  1.567839  1.576824   1.58552  1.593921  1.602023 | 1.358453  1.358275  1.358063  1.357817  1.357533  1.357212  1.356854   1.35646  1.356033  1.355577  1.355095  1.354591  1.354072  1.353541  1.353003  1.352463  1.351924   1.35139  1.350863  1.350348  1.349844  1.349355   1.34888  1.348422   1.34798  1.347556  1.347148  1.346757  1.346382  1.346024  1.345682  1.345355  1.345044  1.344747  1.344464  1.344194  1.343937  1.343693   1.34346  1.343239  1.343029  1.342829  1.342639  1.342459  1.342288  1.342126  1.341972  1.341826  1.341689          1.341558 | 0.541623  0.543011  0.544663    0.5466   0.54884  0.551393  0.554261   0.55744  0.560917  0.564674  0.568685   0.57292  0.577347  0.581932  0.586641   0.59144  0.596298  0.601187  0.606079  0.610951  0.615783  0.620557   0.62526  0.629878  0.634403  0.638826  0.643141  0.647343  0.651431    0.6554  0.659252  0.662984  0.666598  0.670093  0.673472  0.676736  0.679885  0.682923  0.685851  0.688671  0.691385  0.693995  0.696503  0.698912  0.701223  0.703438  0.705558  0.707586  0.709523      0.711371 | -0.208917  -0.210666  -0.212732  -0.215137  -0.217897  -0.221018  -0.224499  -0.228329  -0.232489  -0.236953  -0.241688  -0.246659  -0.251828  -0.257156  -0.262605  -0.268139  -0.273723  -0.279327  -0.284922  -0.290486  -0.295996  -0.301435  -0.306788  -0.312043  -0.317189   -0.32222   -0.32713  -0.331912  -0.336566  -0.341087  -0.345476  -0.349732  -0.353855  -0.357846  -0.361705  -0.365435  -0.369037  -0.372512  -0.375862   -0.37909  -0.382197  -0.385186  -0.388057  -0.390814  -0.393457  -0.395988   -0.39841  -0.400723  -0.402928     -0.405028 | 0.666437  0.666616  0.666828  0.667074  0.667358   0.66768  0.668041  0.668441  0.668878   0.66935  0.669857  0.670394  0.670959  0.671548  0.672158  0.672787   0.67343  0.674084  0.674747  0.675415  0.676087   0.67676  0.677432  0.678101  0.678765  0.679423  0.680074  0.680716  0.681349  0.681972  0.682583  0.683183   0.68377  0.684345  0.684906  0.685453  0.685987  0.686506  0.687011    0.6875  0.687975  0.688435  0.688879  0.689307  0.689719  0.690116  0.690497  0.690861  0.691209       0.69154 | 103.551298 | 104.032989  105.090494  106.366341  107.887805  109.681299  111.771576  114.181029  116.929129  120.032032  123.502357  127.349135  131.577895  136.190867  141.187257  146.563571  152.313953  158.430524   164.90369  171.722431   178.87455  186.346882  194.125479  202.195749  210.542578  219.150424  228.003384  237.085257  246.379581  255.869665   265.53861  275.369328  285.344546  295.446817   305.65852   315.96186  326.338869  336.771404  347.241142  357.729579  368.218024  378.687601   389.11924  399.493678  409.791458  419.992925  430.078227  440.027317  449.819954  459.435706          468.853953 | 473.514016 | 0.728264  0.730209  0.732537  0.735284  0.738485  0.742165  0.746341  0.751022  0.756207  0.761884  0.768036  0.774635   0.78165  0.789045  0.796781  0.804817  0.813113  0.821628  0.830326  0.839168  0.848121  0.857153  0.866236  0.875342  0.884448  0.893533  0.902576   0.91156  0.920471  0.929293  0.938015  0.946626  0.955115  0.963475  0.971696  0.979772  0.987697  0.995464  1.003068  1.010504  1.017768  1.024855  1.031762  1.038483  1.045017  1.051358  1.057505  1.063452  1.069197       1.074736 | 2.634953  2.646265  2.659814  2.675835  2.694533  2.716076  2.740588  2.768138  2.798745  2.832374  2.868943  2.908327  2.950363  2.994866  3.041627  3.090427  3.141043   3.19325  3.246829  3.301569  3.357268  3.413736  3.470798  3.528288  3.586054  3.643958  3.701871  3.759677  3.817267  3.874545  3.931421  3.987813  4.043644  4.098847  4.153356  4.207113  4.260061  4.312151  4.363332   4.41356   4.46279  4.510982  4.558095  4.604092  4.648934  4.692585  4.735009  4.776171  4.816035        4.854566 | 103.551298  104.505809  105.664878  107.056027  108.706329  110.641577  112.885522  115.459234  118.380613  121.664087  125.320477  129.357026  133.777546  138.582675  143.770175  149.335282  155.271037  161.568622  168.217657   175.20647  182.522326  190.151625  198.080059  206.292748  214.774339  223.509093  232.480944   241.67355  251.070326  260.654474  270.408996  280.316708  290.360248  300.522083  310.784504   321.12963  331.539406  341.995599  352.479797  362.973402  373.457633  383.913517  394.321887  404.663384  414.918452  425.067337  435.090091  444.966567  454.676429  464.199148  473.514016 |  100.0 |   3.551298 | 473.514016 |
    | 02:00 | 3.777134  3.792789  3.811673  3.834168  3.860629  3.891365   3.92663  3.966605  4.011396  4.061031  4.115461  4.174563  4.238156  4.306001  4.377824  4.453315   4.53215  4.613991  4.698501  4.785346  4.874202  4.964757  5.056716    5.1498  5.243749  5.338319  5.433285  5.528437  5.623582  5.718542  5.813152  5.907259  6.000723  6.093414  6.185209  6.275998  6.365673  6.454137  6.541297  6.627065  6.711358  6.794096  6.875204  6.954608  7.032238  7.108025  7.181901  7.253801  7.323659              7.39141 | 127.990726  128.818092  129.819351   131.01674  132.431699  134.084104  135.991564  138.168844   140.62744  143.375346  146.416991  149.753342  153.382147  157.298263  161.494056  165.959813  170.684159  175.654442  180.857082  186.277873   191.90223  197.715399  203.702616  209.849224  216.140771  222.563062  229.102207  235.744638  242.477119  249.286748  256.160944  263.087437  270.054245  277.049661  284.062224  291.080705  298.094077  305.091499  312.062294  318.995924  325.881978  332.710145  339.470205  346.152004  352.745445  359.240469  365.627045  371.895153  378.034775  384.035887 |  52.771342   52.927893    53.11673    53.34168   53.606287   53.913654     54.2663   54.666049   55.113962   55.610313   56.154606   56.745633   57.381556   58.060014   58.778237   59.533153   60.321498   61.139912   61.985011    62.85346   63.742016   64.647567   65.567159   66.498004   67.437493   68.383194    69.33285   70.284371   71.235822   72.185421   73.131518   74.072591   75.007232   75.934137   76.852095   77.759977   78.656732   79.541373   80.412971    81.27065   82.113577    82.94096   83.752039   84.546083   85.322382    86.08025   86.819012   87.538009   88.236588     88.914103 | 1.097557   1.09997  1.102876  1.106331  1.110385  1.115083  1.120457  1.126528  1.133305  1.140785  1.148951  1.157775  1.167222  1.177248  1.187802  1.198833  1.210284    1.2221  1.234228  1.246615  1.259211  1.271969  1.284846  1.297801  1.310799  1.323804  1.336788  1.349723  1.362585   1.37535     1.388  1.400516  1.412883  1.425085  1.437111  1.448948  1.460586  1.472015  1.483226  1.494212  1.504964  1.515476   1.52574  1.535752  1.545505  1.554993   1.56421  1.573152  1.581813  1.590188 | 1.358561  1.358406  1.358221  1.358003   1.35775  1.357462  1.357136  1.356775   1.35638  1.355954    1.3555  1.355021  1.354523  1.354009  1.353485  1.352955  1.352423  1.351893  1.351368   1.35085  1.350342  1.349847  1.349365  1.348897  1.348445  1.348009   1.34759  1.347187    1.3468  1.346429  1.346075  1.345735  1.345411  1.345102  1.344806  1.344525  1.344256     1.344  1.343757  1.343525  1.343304  1.343094  1.342894  1.342704  1.342523  1.342352  1.342189  1.342034  1.341888           1.34175 | 0.540785  0.541989  0.543432  0.545137  0.547125  0.549409  0.551998  0.554893  0.558088  0.561568  0.565314  0.569302  0.573502  0.577883  0.582413   0.58706  0.591791  0.596577   0.60139  0.606206  0.611002  0.615759   0.62046  0.625092  0.629643  0.634103  0.638466  0.642724  0.646874  0.650913  0.654838  0.658648  0.662343  0.665922  0.669387  0.672739  0.675978  0.679106  0.682125  0.685037  0.687843  0.690546  0.693147  0.695649  0.698052   0.70036  0.702574  0.704695  0.706725      0.708665 | -0.207849   -0.20937  -0.211181  -0.213307  -0.215766  -0.218572  -0.221728  -0.225231  -0.229069  -0.233221  -0.237662  -0.242361  -0.247282   -0.25239  -0.257648  -0.263019  -0.268469  -0.273966  -0.279481  -0.284987  -0.290462  -0.295885   -0.30124  -0.306512   -0.31169  -0.316764  -0.321726  -0.326571  -0.331294  -0.335892  -0.340363  -0.344705  -0.348919  -0.353004   -0.35696  -0.360789  -0.364491  -0.368069  -0.371523  -0.374856  -0.378069  -0.381164  -0.384143  -0.387007  -0.389759    -0.3924  -0.394931  -0.397354  -0.399671     -0.401882 | 0.666326  0.666482  0.666667  0.666885  0.667137  0.667425   0.66775  0.668113  0.668513  0.668948  0.669418  0.669921  0.670452   0.67101  0.671591  0.672193  0.672811  0.673444  0.674087  0.674739  0.675396  0.676057  0.676719  0.677379  0.678037  0.678691  0.679339   0.67998  0.680613  0.681238  0.681852  0.682456  0.683048  0.683629  0.684197  0.684752  0.685294  0.685823  0.686338  0.686839  0.687325  0.687797  0.688254  0.688696  0.689123  0.689534   0.68993  0.690311  0.690675      0.691023 | 102.988541 | 103.401463  104.310548  105.413364  106.736011  108.304302  110.143003  112.275122  114.721301  117.499338  120.623867  124.106192   127.95427  132.172816  136.763501   141.72522  147.054386  152.745247  158.790188  165.180018  171.904227  178.951208  186.308457  193.962725  201.900157  210.106396  218.566669  227.265851  236.188518  245.318983  254.641325  264.139412  273.796909  283.597291  293.523849  303.559689  313.687733  323.890721  334.151207  344.451554  354.773934  365.100322  375.412497  385.692033  395.920303   406.07847  416.147493   426.10812  435.940893  445.626147          455.144013 |  459.86077 | 0.727094  0.728776    0.7308  0.733207  0.736031  0.739302  0.743043  0.747268  0.751983  0.757184  0.762859   0.76899   0.77555  0.782508  0.789829  0.797476   0.80541  0.813593  0.821986  0.830554  0.839261  0.848076  0.856967  0.865908  0.874873  0.883839  0.892786  0.901695  0.910548  0.919332  0.928032  0.936636  0.945134  0.953517  0.961774  0.969899  0.977885  0.985724  0.993412  1.000942  1.008311  1.015512  1.022543  1.029398  1.036075  1.042568  1.048876  1.054993  1.060917       1.066644 | 2.628163  2.637929  2.649704   2.66372  2.680195  2.699314  2.721227  2.746038  2.773802  2.804524  2.838163  2.874631  2.913801  2.955517  2.999596   3.04584  3.094039  3.143979  3.195449  3.248239  3.302148  3.356985   3.41257  3.468734   3.52532  3.582183   3.63919  3.696217  3.753152  3.809892  3.866341  3.922413  3.978028  4.033112  4.087599  4.141424   4.19453  4.246863  4.298371  4.349008  4.398726  4.447485  4.495241  4.541957  4.587593  4.632113   4.67548  4.717659  4.758613        4.798309 | 102.988541  103.806634   104.80539  106.010881  107.449272  109.146061  111.125319  113.409027  116.016517  118.964077  122.264701   125.92801  129.960299  134.364697  139.141409  144.288007  149.799735  155.669829  161.889812  168.449767  175.338579  182.544143  190.053542  197.853192  205.928961  214.266261  222.850129  231.665276  240.696139  249.926906  259.341546  268.923822  278.657302  288.525368  298.511216  308.597862  318.768136  329.004687  339.289972   349.60626  359.935626  370.259948  380.560901  390.819962  401.018397  411.137268  421.157429  431.059525  440.823992  450.431064   459.86077 |  100.0 |   2.988541 |  459.86077 |
    | 02:30 | 3.767843  3.781315  3.797665  3.817268  3.840481  3.867632  3.899002  3.934816  3.975228  4.020323   4.07011  4.124527  4.183448  4.246687  4.314015  4.385161   4.45983   4.53771  4.618481   4.70182   4.78741  4.874942   4.96412  5.054663  5.146305  5.238796  5.331904  5.425413   5.51912  5.612841  5.706403  5.799647  5.892425  5.984601  6.076047  6.166646  6.256287  6.344867   6.43229  6.518463  6.603302  6.686724  6.768651  6.849007   6.92772   7.00472   7.07994  7.153311   7.22477              7.29425 | 127.500825  128.211452  129.076298  130.116687  131.353681  132.807361  134.496132  136.436105  138.640602  141.119809    143.8806  146.926514  150.257887  153.872086  157.763833  161.925571  166.347852  171.019714  175.929033  181.062841  186.407597  191.949415  197.674252  203.568051  209.616852  215.806876  222.124579  228.556692  235.090238  241.712543  248.411234  255.174236   261.98975  268.846247  275.732443  282.637282  289.549915  296.459681  303.356085  310.228778  317.067539  323.862258  330.602914  337.279564  343.882322  350.401349  356.826836   363.14899  369.358024  375.444148 |  52.678425   52.813152   52.976655   53.172679   53.404809   53.676319   53.990023   54.348156   54.752279   55.203226   55.701095   56.245269   56.834477   57.466875   58.140147   58.851605   59.598297   60.377101   61.184807   62.018198   62.874096   63.749418   64.641202   65.546632    66.46305   67.387963   68.319043   69.254125   70.191201   71.128411   72.064032   72.996471   73.924252   74.846008    75.76047   76.666457   77.562867   78.448669   79.322895   80.184634   81.033023   81.867241   82.686506   83.490067     84.2772   85.047204   85.799398   86.533114   87.247696     87.942498 | 1.096124  1.098202  1.100721  1.103736  1.107299  1.111457  1.116248  1.121702  1.127835  1.134653  1.142149  1.150307  1.159098  1.168486  1.178429  1.188877  1.199781  1.211089  1.222746  1.234703  1.246908  1.259316   1.27188  1.284559  1.297316  1.310115  1.322924  1.335715   1.34846  1.361136  1.373722  1.386198  1.398547  1.410754  1.422804  1.434684  1.446383  1.457891  1.469197  1.480294  1.491173  1.501826  1.512247  1.522429  1.532366  1.542052   1.55148  1.560647  1.569545   1.57817 | 1.358653  1.358519  1.358358  1.358166  1.357942  1.357684  1.357391  1.357062  1.356699  1.356303  1.355877  1.355425   1.35495  1.354457   1.35395  1.353433  1.352911  1.352387  1.351865  1.351347  1.350838  1.350338   1.34985  1.349375  1.348914  1.348469  1.348038  1.347624  1.347225  1.346843  1.346476  1.346124  1.345788  1.345466  1.345159  1.344865  1.344585  1.344318  1.344063   1.34382  1.343588  1.343368  1.343157  1.342958  1.342767  1.342587  1.342415  1.342251  1.342096          1.341949 | 0.540067  0.541107  0.542362  0.543857  0.545613  0.547647  0.549973  0.552595  0.555512  0.558718  0.562198  0.565931  0.569894  0.574058  0.578394   0.58287  0.587455   0.59212  0.596836  0.601576  0.606318  0.611041  0.615726  0.620356   0.62492  0.629406  0.633804  0.638108  0.642312  0.646411  0.650402  0.654283  0.658053  0.661711  0.665258  0.668693  0.672017  0.675232  0.678339   0.68134  0.684236  0.687029   0.68972  0.692312  0.694807  0.697205  0.699509  0.701721  0.703842      0.705874 | -0.206931  -0.208249   -0.20983    -0.2117  -0.213881   -0.21639  -0.219237  -0.222423  -0.225944  -0.229785  -0.233927  -0.238343  -0.243004  -0.247875  -0.252923  -0.258112  -0.263407  -0.268776  -0.274189  -0.279618  -0.285038  -0.290427  -0.295767   -0.30104  -0.306234  -0.311338  -0.316341  -0.321237   -0.32602  -0.330685  -0.335229  -0.339651  -0.343948   -0.34812  -0.352167  -0.356089  -0.359888  -0.363563  -0.367117  -0.370551  -0.373866  -0.377064  -0.380147  -0.383116  -0.385973   -0.38872  -0.391357  -0.393888  -0.396312     -0.398632 | 0.666232  0.666367  0.666528  0.666719  0.666941  0.667198   0.66749  0.667818  0.668182  0.668582  0.669016  0.669484  0.669981  0.670507  0.671058  0.671632  0.672225  0.672834  0.673456  0.674089   0.67473  0.675377  0.676026  0.676677  0.677327  0.677975  0.678618  0.679257  0.679888  0.680513  0.681128  0.681735  0.682331  0.682916   0.68349  0.684052  0.684602  0.685139  0.685663  0.686173   0.68667  0.687153  0.687622  0.688077  0.688517  0.688942  0.689352  0.689748  0.690127      0.690492 |  102.51094 | 102.864117  103.643795  104.594665  105.741418  107.108944  108.721635   110.60269  112.773489   115.25306  118.057676  121.200599  124.691957  128.538756  132.744996   137.31187  142.238009  147.519766    153.1515  159.125856  165.434021  172.065963  179.010628  186.256121  193.789848  201.598638  209.668838  217.986393  226.536902  235.305668  244.277731  253.437894  262.770741  272.260653  281.891807  291.648193  301.513605  311.471648  321.505734   331.59908  341.734706   351.89543  362.063866   372.22242  382.353289  392.438454  402.459683  412.398525  422.236311  431.954153           441.53295 | 446.286663 | 0.726095  0.727544  0.729299  0.731399  0.733882  0.736777  0.740114   0.74391  0.748177   0.75292  0.758132  0.763802  0.769909  0.776427  0.783327  0.790574  0.798133  0.805968   0.81404  0.822315  0.830757  0.839333  0.848014  0.856769  0.865573  0.874402  0.883233  0.892047  0.900825  0.909551  0.918212  0.926793  0.935283  0.943672   0.95195  0.960108  0.968139  0.976035  0.983792  0.991401  0.998859  1.006161  1.013301  1.020275   1.02708  1.033711  1.040165  1.046437  1.052525       1.058425 | 2.622363  2.630772  2.640971  2.653191  2.667652  2.684553  2.704062   2.72631  2.751386   2.77933  2.810139  2.843761  2.880107  2.919051  2.960439  3.004094  3.049826  3.097435  3.146716  3.197468  3.249492  3.302596    3.3566   3.41133  3.466627  3.522339  3.578329  3.634468  3.690637  3.746729  3.802643  3.858288  3.913579  3.968439  4.022797  4.076585  4.129743  4.182214  4.233944  4.284883  4.334983  4.384201  4.432494  4.479821  4.526142   4.57142  4.615619  4.658701  4.700631        4.741375 |  102.51094  103.210545  104.069086  105.110997  106.361256  107.844699  109.585308  111.605535  113.925717    116.5636  119.534005  122.848643  126.516066  130.541734  134.928183  139.675251  144.780354   150.23877  156.043926  162.187673  168.660526  175.451891  182.550249  189.943321  197.618192  205.561428  213.759155  222.197129  230.860792  239.735306  248.805586  258.056322  267.471992  277.036873  286.735048  296.550408  306.466653  316.467292   326.53564  336.654815  346.807738  356.977124  367.145487  377.295126  387.408133  397.466381  407.451529  417.345019  427.128071   436.78169  446.286663 |  100.0 |    2.51094 | 446.286663 |
    | 03:00 | 3.759929  3.771492  3.785606  3.802633  3.822925   3.84682  3.874618  3.906574  3.942884  3.983682  4.029032  4.078926  4.133294  4.192004  4.254872  4.321671   4.39214  4.465995  4.542939  4.622663  4.704862  4.789233  4.875483  4.963328    5.0525  5.142746  5.233827   5.32552  5.417618  5.509926  5.602267  5.694473  5.786391  5.877878  5.968801  6.059035  6.148467  6.236989  6.324499  6.410902   6.49611  6.580037  6.662602  6.743729  6.823342  6.901371  6.977747    7.0524  7.125267             7.196281 | 127.084277  127.693138  128.438169  129.339559  130.417679  131.692434    133.1826  134.905197  136.874952  139.103868  141.600953  144.372082  147.420022  150.744561  154.342757  158.209237  162.336547  166.715501  171.335536  176.185028  181.251583  186.522282   191.98389  197.623025   203.42629  209.380376  215.472132  221.688624  228.017164  234.445331  240.960978  247.552236  254.207503  260.915431  267.664919  274.445088  281.245267  288.054974  294.863897  301.661872   308.43887  315.184975  321.890368   328.54531  335.140127  341.665196  348.110929  354.467759   360.72613  366.876485 |  52.599292    52.71492   52.856063   53.026325   53.229255   53.468203   53.746181   54.065737   54.428844   54.836825   55.290316   55.789261   56.332943   56.920042    57.54872   58.216705   58.921396   59.659953   60.429386   61.226633   62.048623   62.892334   63.754826   64.633277   65.525001   66.427461    67.33827   68.255201   69.176176   70.099263   71.022669   71.944734   72.863915   73.778782   74.688007   75.590355   76.484674   77.369887   78.244987   79.109023   79.961099   80.800368   81.626021   82.437286   83.233422   84.013713   84.777465   85.524003   86.252667     86.962806 | 1.094902  1.096687  1.098864  1.101485  1.104605  1.108271  1.112525  1.117402  1.122928  1.129115  1.135966  1.143474  1.151618  1.160371  1.169698  1.179556  1.189899  1.200679  1.211845  1.223348  1.235138  1.247168  1.259392  1.271768  1.284257  1.296822  1.309429  1.322049  1.334651  1.347213   1.35971  1.372121  1.384429  1.396615  1.408666  1.420567  1.432305  1.443869  1.455249  1.466436   1.47742  1.488194   1.49875  1.509081  1.519181  1.529044  1.538662  1.548032  1.557147  1.566001 | 1.358733  1.358617  1.358477  1.358309  1.358112  1.357882  1.357618  1.357321  1.356989  1.356624  1.356228  1.355803  1.355354  1.354883  1.354395  1.353893  1.353383  1.352868  1.352352  1.351838  1.351329  1.350827  1.350335  1.349854  1.349386  1.348932  1.348492  1.348067  1.347658  1.347264  1.346885  1.346522  1.346173   1.34584   1.34552  1.345215  1.344923  1.344644  1.344378  1.344124  1.343882  1.343651  1.343431  1.343221  1.343021   1.34283  1.342649  1.342477  1.342313          1.342158 | 0.539453  0.540349  0.541437  0.542742  0.544286   0.54609  0.548168   0.55053  0.553181  0.556117  0.559331  0.562807  0.566526  0.570463  0.574591  0.578881  0.583303  0.587829  0.592429  0.597077  0.601747  0.606418   0.61107  0.615684  0.620247  0.624745  0.629168  0.633506  0.637754  0.641904  0.645953  0.649898  0.653737  0.657468   0.66109  0.664604   0.66801  0.671308  0.674499  0.677585  0.680568  0.683447  0.686227  0.688907  0.691489  0.693976   0.69637   0.69867   0.70088      0.703001 | -0.206145  -0.207283  -0.208658  -0.210296  -0.212222  -0.214455  -0.217009  -0.219892  -0.223104  -0.226637  -0.230479  -0.234607  -0.238997  -0.243619  -0.248441  -0.253429   -0.25855  -0.263771  -0.269062  -0.274393  -0.279739  -0.285076  -0.290383  -0.295641  -0.300837  -0.305956  -0.310987  -0.315922  -0.320754  -0.325476  -0.330085  -0.334577   -0.33895  -0.343203  -0.347335  -0.351345  -0.355234  -0.359002  -0.362651  -0.366182  -0.369595  -0.372892  -0.376075  -0.379146  -0.382105  -0.384954  -0.387695   -0.39033  -0.392859     -0.395284 | 0.666151  0.666267  0.666407  0.666574   0.66677  0.666998  0.667258  0.667554  0.667884  0.668249  0.668649  0.669082  0.669546  0.670039  0.670559  0.671104   0.67167  0.672254  0.672854  0.673466  0.674089   0.67472  0.675356  0.675995  0.676635  0.677275  0.677913  0.678547  0.679176  0.679798  0.680414  0.681021   0.68162  0.682208  0.682787  0.683354   0.68391  0.684454  0.684986  0.685506  0.686012  0.686505  0.686985  0.687451  0.687903  0.688341  0.688765  0.689174  0.689568      0.689947 | 102.106358 | 102.407781  103.074971  103.892852  104.884543  106.073766  107.484211  109.138899  111.059547  113.266015  115.775846  118.603925  121.762274  125.259968  129.103171  133.295256     137.837   142.72682  147.961031   153.53411  159.438951  165.667103  172.208979  179.054043  186.190968  193.607768   201.29191  209.230398  217.409845  225.816529  234.436434   243.25528   252.25855  261.431506  270.759196  280.226468  289.817969  299.518148  309.311257  319.181348  329.112271  339.087672  349.090988  359.105444  369.114051  379.099602  389.044667  398.931595  408.742506  418.459298          428.063638 | 432.836227 | 0.725243  0.726488  0.728005  0.729832  0.732005  0.734558  0.737521  0.740917  0.744763  0.749068  0.753833  0.759053  0.764713  0.770793  0.777268  0.784109  0.791283  0.798756  0.806492  0.814457  0.822616  0.830936  0.839386  0.847937  0.856561  0.865233  0.873929  0.882629  0.891314  0.899966  0.908569   0.91711  0.925576  0.933955  0.942237  0.950413  0.958474  0.966413  0.974223  0.981898  0.989431  0.996817  1.004053  1.011132  1.018051  1.024805  1.031391  1.037805  1.044042       1.050101 | 2.617423  2.624641  2.633449  2.644068  2.656717    2.6716  2.688899  2.708767  2.731319  2.756628  2.784723   2.81559  2.849173   2.88538  2.924086   2.96514  3.008372  3.053599  3.100628  3.149266  3.199319  3.250599  3.302924   3.35612  3.410024  3.464481  3.519348  3.574493  3.629791   3.68513  3.740404  3.795518  3.850382  3.904915  3.959041   4.01269  4.065798  4.118305  4.170154  4.221293  4.271674  4.321249  4.369974  4.417808   4.46471   4.51064  4.555563   4.59944  4.642237        4.683918 | 102.106358  102.703346  103.439639  104.337919  105.421771  106.715078  108.241374  110.023193  112.081471   114.43502  117.100133  120.090304  123.416092  127.085094  131.102037  135.468938  140.185323  145.248489  150.653759  156.394752  162.463629  168.851315  175.547702  182.541819  189.821975  197.375885  205.190762  213.253396  221.550221  230.067355  238.790645  247.705685  256.797844  266.052274  275.453922  284.987534  294.637657  304.388642  314.224641  324.129606  334.087285  344.081221  354.094746  364.110979  374.112822  384.082957  394.003845   403.85772  413.626592  423.292243  432.836227 |  100.0 |   2.106358 | 432.836227 |
    | 03:30 | 3.753205  3.763103  3.775252  3.789996  3.807677  3.828631  3.853172  3.881574  3.914068  3.950828  3.991964  4.037523  4.087485  4.141771  4.200244  4.262723  4.328986  4.398779  4.471831  4.547855  4.626558  4.707647  4.790833  4.875838  4.962391  5.050236  5.139131  5.228847   5.31917    5.4099  5.500853  5.591856  5.682747  5.773377  5.863608  5.953312  6.042367  6.130661  6.218089  6.304553   6.38996   6.47422  6.557252  6.638975  6.719313  6.798193  6.875545    6.9513   7.02539             7.097752 | 126.730822  127.251264  127.891444  128.670265  129.607167  130.721562  132.032226  133.556689  135.310674  137.307626   139.55835  142.070789  144.849936  147.897874   151.21393  154.794906  158.635373  162.727991   167.06384  171.632739  176.423538  181.424379  186.622928  192.006558  197.562503  203.277983  209.140295  215.136878  221.255367  227.483621  233.809741   240.22208  246.709239  253.260069  259.863649  266.509283  273.186481  279.884938  286.594524  293.305261  300.007306  306.690936  313.346531  319.964553  326.535537  333.050073  339.498789  345.872345  352.161411  358.356665 |  52.532052    52.63103   52.752525   52.899956   53.076768   53.286314   53.531715   53.815739   54.140682   54.508279   54.919641   55.375227   55.874849   56.417705   57.002444   57.627234   58.289857   58.987794   59.718312   60.478548   61.265576   62.076466   62.908335   63.758381   64.623912   65.502364   66.391309   67.288465   68.191696   69.099004   70.008534   70.918556   71.827465   72.733771   73.636085   74.533118   75.423667    76.30661   77.180895   78.045533   78.899595   79.742202   80.572518   81.389748   82.193131   82.981934   83.755452   84.512998   85.253904     85.977516 | 1.093863  1.095392  1.097267   1.09954  1.102261  1.105481  1.109244  1.113588  1.118544  1.124134  1.130368  1.137246  1.144758  1.152884  1.161596  1.170859  1.180632  1.190871  1.201528  1.212557  1.223908  1.235536  1.247395  1.259442  1.271637  1.283941   1.29632  1.308742  1.321177  1.333598  1.345981  1.358304  1.370546   1.38269  1.394719  1.406617  1.418373  1.429972  1.441405   1.45266  1.463729  1.474603  1.485273  1.495734  1.505976  1.515995  1.525783  1.535336  1.544646  1.553708 |   1.3588  1.358701  1.358579  1.358433   1.35826  1.358056  1.357821  1.357553  1.357251  1.356917  1.356551  1.356155  1.355731  1.355285  1.354818  1.354335   1.35384  1.353336  1.352829   1.35232  1.351813  1.351312  1.350818  1.350333  1.349859  1.349398   1.34895  1.348516  1.348096  1.347692  1.347302  1.346927  1.346567  1.346222  1.345891  1.345574  1.345271  1.344981  1.344703  1.344438  1.344185  1.343943  1.343713  1.343493  1.343283  1.343083  1.342893  1.342711  1.342539          1.342375 | 0.538931    0.5397   0.54064  0.541774  0.543127  0.544719  0.546566  0.548684   0.55108  0.553756  0.556708  0.559927  0.563397  0.567099  0.571009    0.5751  0.579345  0.583715  0.588182  0.592719  0.597301  0.601903  0.606506  0.611089  0.615636  0.620133  0.624567  0.628929   0.63321  0.637402  0.641501  0.645502  0.649402  0.653199  0.656892  0.660479  0.663961  0.667338   0.67061  0.673778  0.676843  0.679807  0.682671  0.685437  0.688105  0.690679  0.693158  0.695545  0.697842       0.70005 | -0.205473  -0.206453  -0.207645  -0.209074  -0.210767  -0.212744  -0.215025   -0.21762  -0.220535  -0.223769  -0.227312  -0.231151  -0.235263  -0.239625  -0.244207  -0.248979  -0.253908  -0.258963  -0.264113  -0.269327   -0.27458  -0.279846  -0.285102  -0.290329   -0.29551   -0.30063  -0.305676  -0.310638  -0.315507  -0.320276   -0.32494  -0.329494  -0.333935  -0.338262  -0.342471  -0.346563  -0.350537  -0.354393  -0.358132  -0.361755  -0.365262  -0.368654  -0.371934  -0.375102   -0.37816  -0.381108   -0.38395  -0.386685  -0.389316     -0.391843 | 0.666082  0.666182  0.666303  0.666448  0.666621  0.666822  0.667054  0.667318  0.667616  0.667948  0.668314  0.668713  0.669144  0.669605  0.670094  0.670609  0.671147  0.671705  0.672281  0.672872  0.673475  0.674088  0.674709  0.675335  0.675964  0.676594  0.677224  0.677852  0.678476  0.679096   0.67971  0.680317  0.680916  0.681507  0.682088   0.68266  0.683221  0.683771   0.68431  0.684837  0.685352  0.685854  0.686343   0.68682  0.687283  0.687732  0.688168   0.68859  0.688998      0.689391 | 101.764251 | 102.020965  102.590658   103.29251  104.147973  105.179415  106.409571   107.86096  109.555283  111.512861   113.75214  116.289296  119.137947  122.308996  125.810576  129.648103  133.824399  138.339882  143.192792  148.379425  153.894382  159.730799  165.880566  172.334514  179.082591  186.114001  193.417326  200.980626  208.791519  216.837244  225.104713  233.580546  242.251106  251.102514  260.120667   269.29125  278.599738  288.031404  297.571318  307.204346  316.915148  326.688181  336.507686  346.357694  356.222015  366.084242  375.927739  385.735646  395.490871  405.176089          414.773742 |  419.54871 | 0.724519  0.725585  0.726892  0.728476  0.730372  0.732615  0.735236  0.738261  0.741712  0.745603   0.74994  0.754723  0.759946  0.765593  0.771644  0.778074  0.784856  0.791957  0.799344  0.806984  0.814844  0.822891  0.831093  0.839421  0.847846  0.856343  0.864886  0.873455  0.882028  0.890588  0.899118  0.907602  0.916027   0.92438  0.932651  0.940829  0.948906  0.956872  0.964722  0.972446  0.980041  0.987499  0.994815  1.001985  1.009004  1.015868  1.022572  1.029113  1.035487       1.041689 | 2.613224  2.619404  2.626988  2.636187  2.647213  2.660272  2.675554  2.693226  2.713424  2.736249  2.761761  2.789979   2.82088  2.854404  2.890457  2.928916  2.969632  3.012442  3.057169   3.10363  3.151639  3.201013  3.251571   3.30314  3.355554  3.408657  3.462301  3.516351  3.570677  3.625161  3.679695  3.734176  3.788513  3.842619  3.896414  3.949825  4.002784  4.055229  4.107099  4.158341  4.208903  4.258736  4.307795  4.356035  4.403416  4.449898  4.495442  4.540011  4.583569         4.62608 | 101.764251   102.27261  102.902644  103.675228   104.61241  105.736898  107.071486  108.638447  110.458941  112.552477  114.936462  117.625858  120.632956  123.967277   127.63557  131.641912  135.987867    140.6727  145.693613  151.045988  156.723634  162.719007  169.023417  175.627212  182.519931  189.690436  197.127025  204.817515  212.749321  220.909504  229.284821  237.861758   246.62655  255.565203  264.663506  273.907036  283.281168  292.771071  302.361714  312.037862  321.784072  331.584695  341.423866  351.285509  361.153325  371.010793  380.841166  390.627469  400.352494  409.998799   419.54871 |  100.0 |   1.764251 |  419.54871 |
    | 04:00 | 3.747504  3.755956  3.766386  3.779114  3.794471  3.812785  3.834372  3.859521  3.888487  3.921476   3.95864   4.00007  4.045797   4.09579  4.149964  4.208179  4.270256  4.335976  4.405097  4.477355  4.552477  4.630181  4.710189  4.792225  4.876022  4.961322  5.047881  5.135466  5.223859  5.312855  5.402261    5.4919  5.581603  5.671217  5.760596  5.849607  5.938124  6.026029  6.113214  6.199575  6.285016  6.369446  6.452778  6.534931  6.615826  6.695388  6.773544  6.850226  6.925365             6.998895 | 126.431497  126.875356    127.4241  128.095249  128.907141  129.878445  131.027622  132.372353  133.928982  135.712021  137.733732  140.003829  142.529297  145.314342  148.360453   151.66656   155.22926  159.043095  163.100856  167.393884  171.912364  176.645593  181.582223  186.710463  192.018254   197.49341  203.123725  208.897063  214.801417  220.824955   226.95605  233.183295  239.495513  245.881753  252.331291  258.833615  265.378414  271.955565  278.555119  285.167278  291.782387  298.390913  304.983427  311.550592  318.083146  324.571889  331.007664  337.381349  343.683842  349.906047 |  52.475041   52.559558   52.663859   52.791145   52.944715   53.127854    53.34372   53.595215   53.884874   54.214762   54.586396   55.000696   55.457965   55.957902   56.499638   57.081794   57.702558   58.359763   59.050971   59.773553   60.524766    61.30181   62.101888   62.922248   63.760216    64.61322    65.47881   66.354663   67.238593   68.128548   69.022612   69.918995   70.816031   71.712168   72.605963    73.49607   74.381236    75.26029   76.132138    76.99575   77.850162   78.694461   79.527785   80.349311   81.158259   81.953876   82.735441   83.502257   84.253646     84.988946 | 1.092981  1.094288  1.095899  1.097863  1.100229  1.103047  1.106362  1.110216  1.114644  1.119672   1.12532  1.131594  1.138492  1.146004  1.154108  1.162775  1.171973   1.18166  1.191795  1.202332  1.213225  1.224429    1.2359  1.247593  1.259468  1.271486  1.283612  1.295811  1.308053   1.32031  1.332555  1.344765  1.356919  1.368997  1.380981  1.392856  1.404607   1.41622  1.427684  1.438988  1.450121  1.461075   1.47184  1.482409  1.492775   1.50293  1.512868  1.522583  1.532069   1.54132 | 1.358858  1.358773  1.358668  1.358541  1.358389   1.35821  1.358001  1.357761  1.357488  1.357184  1.356847  1.356479  1.356084  1.355662  1.355218  1.354755  1.354278  1.353788  1.353292  1.352791   1.35229  1.351791  1.351297   1.35081  1.350332  1.349865   1.34941  1.348968  1.348539  1.348125  1.347725   1.34734  1.346969  1.346612   1.34627  1.345942  1.345627  1.345326  1.345037  1.344761  1.344497  1.344245  1.344004  1.343774  1.343555  1.343345  1.343145  1.342955  1.342773            1.3426 | 0.538486  0.539145  0.539954  0.540938  0.542118  0.543516  0.545152  0.547042  0.549196  0.551622   0.55432  0.557285  0.560506  0.563968  0.567652  0.571533  0.575587  0.579787  0.584105  0.588515  0.592991  0.597509  0.602046  0.606582  0.611098   0.61558  0.620014  0.624387   0.62869  0.632914  0.637053  0.641102  0.645056  0.648912  0.652669  0.656324  0.659877  0.663328  0.666676  0.669922  0.673068  0.676112  0.679058  0.681907  0.684659  0.687316  0.689879  0.692351  0.694733      0.697025 | -0.204901  -0.205742  -0.206771  -0.208014  -0.209496   -0.21124  -0.213267  -0.215591  -0.218224  -0.221167  -0.224418  -0.227968  -0.231801  -0.235896  -0.240228   -0.24477  -0.249492  -0.254363  -0.259354  -0.264433  -0.269574   -0.27475  -0.279939  -0.285117  -0.290268  -0.295373   -0.30042  -0.305396   -0.31029  -0.315095  -0.319804  -0.324411  -0.328912  -0.333303  -0.337584  -0.341751  -0.345804  -0.349742  -0.353566  -0.357276  -0.360873  -0.364357  -0.367729  -0.370991  -0.374144  -0.377188  -0.380127   -0.38296  -0.385689     -0.388316 | 0.666023  0.666109  0.666213   0.66634   0.66649  0.666667  0.666873  0.667109  0.667377  0.667678  0.668011  0.668377  0.668776  0.669204  0.669662  0.670147  0.670656  0.671187  0.671738  0.672306  0.672888  0.673482  0.674086  0.674697  0.675313  0.675932  0.676553  0.677173  0.677792  0.678407  0.679018  0.679623  0.680222  0.680813  0.681397  0.681971  0.682536  0.683091  0.683635  0.684169  0.684691  0.685201  0.685699  0.686185  0.686658  0.687118  0.687565  0.687998  0.688418      0.688825 |  101.47548 | 101.693674  102.179097  102.780025    103.5162  104.408511  105.478542  106.748051  108.238409  109.970062  111.962013  114.231396  116.793126   119.65967   122.84092  126.344162  130.174145  134.333209  138.821464     143.637  148.776116  154.233537  160.002633  166.075616  172.443713  179.097322  186.026145  193.219292  200.665381  208.352601  216.268777  224.401416  232.737738  241.264705   249.96904  258.837242  267.855593  277.010165  286.286822  295.671223  305.148819  314.704851   324.32435   333.99213  343.692788  353.410698  363.130006  372.834631  382.508258  392.134337          401.696078 |  406.45844 | 0.723904  0.724815  0.725938  0.727307  0.728956  0.730919  0.733229  0.735913  0.738997  0.742497  0.746428  0.750792   0.75559  0.760812  0.766442  0.772463  0.778847  0.785569  0.792597  0.799901  0.807447  0.815205  0.823143   0.83123  0.839439  0.847742  0.856115  0.864535   0.87298  0.881431   0.88987   0.89828  0.906648  0.914961  0.923205  0.931371  0.939448  0.947427  0.955301  0.963063  0.970704   0.97822  0.985604  0.992852  0.999958  1.006917  1.013726  1.020381  1.026877        1.03321 | 2.609663  2.614942  2.621454  2.629398  2.638979  2.650397  2.663847  2.679506  2.697525  2.718026  2.741096  2.766784  2.795099  2.826012   2.85946  2.895345  2.933548  2.973923  3.016313  3.060548  3.106451  3.153847  3.202559  3.252416  3.303251  3.354907  3.407234  3.460092  3.513348  3.566881  3.620577   3.67433  3.728042  3.781625  3.834993  3.888071  3.940786  3.993074   4.04487   4.09612  4.146768  4.196763   4.24606  4.294611  4.342375   4.38931  4.435377  4.480538  4.524756        4.567996 |  101.47548  101.907495  102.445435  103.108366  103.916715  104.891855  106.055606  107.429686  109.035153  110.891864  113.017997  115.429659  118.140593     121.162  124.502466  128.167983  132.162053  136.485846  141.138404  146.116859  151.416664   157.03181  162.955037  169.178017  175.691526   182.48558  189.549557  196.872299   204.44219  212.247225  220.275058  228.513042  236.948262  245.567558  254.357535  263.304583  272.394877  281.614384  290.948865  300.383876  309.904764  319.496664  329.144501   338.83298  348.546587  358.269584  367.986005  377.679652  387.334093  396.932659   406.45844 |  100.0 |    1.47548 |  406.45844 |
    | 04:30 | 3.742681   3.74988  3.758811   3.76977  3.783068  3.799023  3.817946  3.840135  3.865858  3.895348  3.928789  3.966314  4.007997  4.053855  4.103847  4.157882   4.21582  4.277482  4.342657  4.411109  4.482584  4.556821   4.63355  4.712504  4.793421  4.876045   4.96013  5.045442  5.131758  5.218869  5.306578    5.3947  5.483063  5.571506  5.659879  5.748042  5.835865  5.923226  6.010011  6.096113  6.181432  6.265874  6.349348   6.43177   6.51306   6.59314  6.671937  6.749379  6.825397             6.899926 | 126.178495  126.556222  127.025488  127.602394  128.304056  129.148213  130.152753  131.335198  132.712175  134.298907  136.108774  138.152949  140.440151  142.976503  145.765515  148.808154  152.103011   155.64652  159.433221  163.456044  167.706584   172.17538  176.852154  181.726034  186.785741  192.019749  197.416411  202.964062  208.651095  214.466023  220.397516  226.434429   232.56582  238.780955  245.069308  251.420559   257.82458  264.271431  270.751341  277.254694  283.772017  290.293963  296.811295  303.314869  309.795624  316.244562  322.652739  329.011247  335.311204  341.543743 |  52.426805   52.498804   52.588114   52.697703   52.830683   52.990228   53.179461   53.401348   53.658583   53.953481   54.287894   54.663141   55.079969   55.538546   56.038472   56.578822   57.158203   57.774825   58.426573   59.111089   59.825845   60.568206   61.335496   62.125041    62.93421    63.76045   64.601302   65.454421   66.317584   67.188693   68.065779   68.946998   69.830627   70.715056   71.598786   72.480419   73.358651   74.232261   75.100112   75.961134   76.814324   77.658736   78.493477   79.317699   80.130596   80.931398   81.719366   82.493787   83.253973     83.999255 | 1.092235  1.093349  1.094729  1.096421  1.098472   1.10093   1.10384  1.107246  1.111185  1.115691  1.120785  1.126484  1.132792  1.139705  1.147211  1.155289   1.16391   1.17304  1.182642  1.192674  1.203092  1.213853  1.224914  1.236231  1.247763  1.259471  1.271319  1.283271  1.295296  1.307364  1.319448  1.331522  1.343564  1.355554  1.367472  1.379302  1.391027  1.402633  1.414108   1.42544  1.436617  1.447631  1.458472   1.46913    1.4796  1.489872  1.499941  1.509799  1.519442  1.528861 | 1.358907  1.358834  1.358744  1.358634  1.358502  1.358345   1.35816  1.357946  1.357701  1.357425  1.357117  1.356778   1.35641  1.356015  1.355595  1.355155  1.354696  1.354223   1.35374  1.353249  1.352755  1.352261  1.351769  1.351283  1.350803  1.350332  1.349872  1.349423  1.348986  1.348563  1.348154  1.347758  1.347377   1.34701  1.346656  1.346317  1.345991  1.345679   1.34538  1.345093  1.344818  1.344556  1.344304  1.344064  1.343835  1.343615  1.343406  1.343206  1.343016          1.342834 |  0.53811  0.538672  0.539367  0.540216  0.541242  0.542466  0.543909  0.545587  0.547515  0.549703  0.552156  0.554872  0.557847  0.561068  0.564521  0.568184  0.572037  0.576054  0.580209  0.584476   0.58883  0.593247  0.597702  0.602175  0.606647    0.6111  0.615519   0.61989  0.624204   0.62845  0.632619  0.636707  0.640707  0.644616   0.64843  0.652147  0.655766  0.659285  0.662705  0.666025  0.669246  0.672369  0.675393  0.678321  0.681154  0.683892  0.686538  0.689092  0.691555      0.693931 | -0.204415  -0.205135  -0.206021  -0.207097  -0.208389  -0.209922  -0.211715  -0.213788  -0.216153  -0.218819  -0.221787  -0.225052  -0.228605   -0.23243  -0.236505  -0.240807  -0.245309  -0.249981  -0.254796  -0.259722  -0.264734  -0.269803  -0.274906  -0.280019  -0.285122  -0.290199  -0.295231  -0.300207  -0.305115  -0.309944  -0.314687  -0.319337  -0.323889  -0.328338  -0.332681  -0.336916  -0.341042  -0.345057  -0.348961  -0.352753  -0.356434  -0.360005  -0.363466  -0.366818  -0.370062    -0.3732  -0.376232   -0.37916  -0.381985     -0.384707 | 0.665973  0.666046  0.666137  0.666246  0.666377  0.666532  0.666714  0.666924  0.667163  0.667435  0.667737  0.668072  0.668438  0.668836  0.669262  0.669716  0.670196    0.6707  0.671225  0.671769  0.672329  0.672903  0.673488  0.674083  0.674684  0.675291  0.675901  0.676512  0.677123  0.677732  0.678338  0.678941  0.679538  0.680129  0.680712  0.681288  0.681856  0.682414  0.682963  0.683502   0.68403  0.684547  0.685053  0.685547  0.686029  0.686498  0.686956    0.6874  0.687831       0.68825 | 101.232151 | 101.417242  101.830021  102.343419  102.975465  103.745505   104.67383  105.781227  107.088478  108.615852  110.382592  112.406469  114.703394  117.287128  120.169085  123.358241  126.861123   130.68189  134.822452  139.282654  144.060468  149.152205  154.552719  160.255604  166.253377  172.537637  179.099205  185.928248  193.014372  200.346714     207.914  215.704606  223.706593  231.907745  240.295591  248.857419  257.580296  266.451068  275.456373  284.582636  293.816075  303.142697  312.548296  322.018452  331.538526  341.093657  350.668756  360.248507  369.817359  379.359522          388.858968 | 393.595174 | 0.723384  0.724161  0.725123  0.726302  0.727732  0.729444  0.731472  0.733845  0.736589  0.739726  0.743272  0.747237  0.751626  0.756433  0.761651  0.767263   0.77325  0.779588   0.78625  0.793207  0.800428  0.807882   0.81554  0.823372  0.831348  0.839441  0.847627   0.85588   0.86418  0.872505  0.880836  0.889158  0.897453  0.905709  0.913912   0.92205  0.930113  0.938091  0.945976   0.95376  0.961435  0.968995  0.976434  0.983746  0.990925  0.997968  1.004869  1.011624  1.018229        1.02468 |  2.60665  2.611147  2.616725  2.623567  2.631865  2.641817  2.653614  2.667437  2.683449   2.70179  2.722568  2.745857  2.771696  2.800085   2.83099  2.864344  2.900051  2.937991  2.978024  3.019996  3.063746  3.109103    3.1559  3.203968  3.253142  3.303265  3.354186  3.405761  3.457856  3.510344  3.563108  3.616039  3.669034     3.722  3.774849  3.827501  3.879881  3.931919  3.983551  4.034716  4.085358  4.135425  4.184867  4.233636   4.28169  4.328984   4.37548  4.421138   4.46592         4.50979 | 101.232151   101.59857  102.056914  102.624476   103.32003  104.163508  105.175578  106.377169  107.788947  109.430802  111.321362  113.477561  115.914305  118.644224  121.677527  125.021959  128.682833  132.663144  136.963719   141.58341  146.519301  151.766919  157.320437  163.172864  169.316216  175.741668  182.439682  189.400122  196.612337  204.065241  211.747374  219.646943  227.751863  236.049784  244.528111  253.174018  261.974459  270.916176  279.985699   289.16935   298.45324  307.823273  317.265134  326.764296  336.306009    345.8753  355.456967  365.035576   374.59546  384.120709  393.595174 |  100.0 |   1.232151 | 393.595174 |
    | 05:00 | 3.738608  3.744727  3.752356  3.761766  3.773248  3.787105  3.803639  3.823147  3.845908  3.872171  3.902147  3.936001  3.973847  4.015745  4.061699  4.111661  4.165534  4.223178  4.284416  4.349043   4.41683  4.487534  4.560903  4.636679  4.714606  4.794435   4.87592  4.958825  5.042927  5.128013   5.21388  5.300341  5.387216  5.474341  5.561559  5.648727  5.735707  5.822374   5.90861  5.994303  6.079348  6.163649  6.247112   6.32965   6.41118  6.491623  6.570902  6.648945  6.725682             6.801045 | 125.965045  126.285823  126.686221  127.180918  127.785744  128.517371  129.392917  130.429485   131.64368  133.051117  134.665961  136.500537  138.565017  140.867215  143.412479  146.203696  149.241381  152.523835  156.047366  159.806531  163.794404  168.002835  172.422698  177.044116  181.856669  186.849557  192.011754   197.33212  202.799493  208.402768  214.130942  219.973159  225.918728  231.957146  238.078097  244.271457   250.52729  256.835835  263.187503  269.572858  275.982607  282.407585  288.838742  295.267125  301.683865  308.080168  314.447291  320.776538  327.059243  333.286759 |  52.386075   52.447273   52.523561   52.617662   52.732484   52.871045   53.036387   53.231473   53.459083   53.721712    54.02147    54.36001   54.738472    55.15745    55.61699    56.11661   56.655341   57.231781   57.844164   58.490432   59.168303   59.875343   60.609025   61.366785   62.146065    62.94435   63.759196   64.588253   65.429274   66.280128   67.138803   68.003406   68.872161   69.743408   70.615593   71.487266   72.357071   73.223744   74.086099   74.943026   75.793483   76.636491   77.471123   78.296504   79.111803   79.916227   80.709019   81.489452   82.256823     83.010452 | 1.091605  1.092552  1.093731  1.095185  1.096958  1.099094   1.10164  1.104639  1.108131  1.112151  1.116728  1.121882  1.127625  1.133962  1.140885  1.148382   1.15643  1.165002  1.174064   1.18358   1.19351  1.203811  1.214443  1.225363  1.236531  1.247907  1.259454  1.271136   1.28292  1.294776  1.306675   1.31859  1.330499  1.342379  1.354209  1.365972   1.37765  1.389229  1.400695  1.412035  1.423238  1.434292  1.445188  1.455918  1.466472  1.476843  1.487024  1.497007  1.506787  1.516357 | 1.358948  1.358886  1.358809  1.358714  1.358599  1.358462  1.358299  1.358109   1.35789  1.357641  1.357361  1.357051  1.356711  1.356343  1.355948  1.355531  1.355093  1.354639  1.354171  1.353693  1.353209  1.352722  1.352235   1.35175   1.35127  1.350797  1.350333  1.349879  1.349436  1.349005  1.348587  1.348182  1.347791  1.347414   1.34705    1.3467  1.346364  1.346041  1.345731  1.345433  1.345148  1.344875  1.344613  1.344363  1.344123  1.343895  1.343676  1.343467  1.343267          1.343076 | 0.537791   0.53827  0.538865  0.539596  0.540485  0.541552  0.542819  0.544303  0.546021  0.547986  0.550204  0.552681  0.555413  0.558395  0.561614  0.565055  0.568698  0.572521    0.5765  0.580611  0.584829  0.589128  0.593487  0.597881  0.602293  0.606702  0.611093  0.615451  0.619763  0.624019  0.628209  0.632326  0.636364  0.640316  0.644181  0.647953  0.651632  0.655215  0.658702  0.662091  0.665384   0.66858   0.67168  0.674685  0.677595  0.680412  0.683137  0.685771  0.688315      0.690771 | -0.204003  -0.204617  -0.205377  -0.206307   -0.20743  -0.208771  -0.210351  -0.212191  -0.214307  -0.216709  -0.219405  -0.222394  -0.225671  -0.229224  -0.233038  -0.237093  -0.241363  -0.245824  -0.250448  -0.255206  -0.260071  -0.265016  -0.270015  -0.275046  -0.280087  -0.285118  -0.290122  -0.295085  -0.299992  -0.304834    -0.3096  -0.314282  -0.318875  -0.323373  -0.327772  -0.332068   -0.33626  -0.340345  -0.344322  -0.348192  -0.351953  -0.355606  -0.359152   -0.36259  -0.365922  -0.369148  -0.372271   -0.37529  -0.378207     -0.381023 |  0.66593  0.665993  0.666071  0.666165  0.666279  0.666415  0.666574   0.66676  0.666974  0.667217  0.667491  0.667796  0.668131  0.668498  0.668893  0.669317  0.669768  0.670244  0.670742  0.671261  0.671797   0.67235  0.672915  0.673492  0.674078  0.674671  0.675268  0.675869  0.676471  0.677073  0.677673  0.678271  0.678865  0.679454  0.680037  0.680613  0.681182  0.681743  0.682295  0.682838  0.683371  0.683894  0.684406  0.684908  0.685397  0.685876  0.686342  0.686796  0.687238      0.687668 | 101.027458 |  101.18417  101.534486  101.972181  102.513608  103.176544  103.979894   104.94332  106.086806  107.430182  108.992644  110.792303  112.845768   115.16782  117.771155  120.666228  123.861185  127.361875  131.171923  135.292867  139.724322  144.464165   149.50873     154.853  160.490785  166.414891  172.617268  179.089135  185.821093  192.803214  200.025119  207.476041  215.144868  223.020186  231.090309  239.343296  247.766973  256.348939  265.076579  273.937061  282.917346  292.004182  301.184106  310.443437  319.768282  329.144523  338.557818  347.993595  357.437052  366.873147          376.286599 | 380.984449 | 0.722945  0.723605  0.724427  0.725441  0.726676  0.728165  0.729939  0.732029  0.734461  0.737261  0.740447  0.744035  0.748032  0.752439  0.757254  0.762464  0.768056  0.774008  0.780299  0.786901  0.793786  0.800926  0.808291  0.815851  0.823579  0.831447  0.839429    0.8475  0.855638  0.863821  0.872029  0.880245  0.888453  0.896637  0.904783  0.912879  0.920914  0.928877  0.936759  0.944552  0.952248  0.959839  0.967319  0.974682  0.981922  0.989035  0.996015  1.002858   1.00956       1.016116 | 2.604105  2.607928  2.612694   2.61857  2.625738  2.634383  2.644695  2.656855  2.671032  2.687377  2.706016  2.727046  2.750529  2.776495  2.804937  2.835816  2.869063  2.904582  2.942253  2.981943  3.023501  3.066772  3.111595  3.157807  3.205246  3.253758  3.303189  3.353396  3.404242  3.455597   3.50734  3.559359  3.611547  3.663807  3.716047  3.768184  3.820139  3.871839  3.923217   3.97421  4.024758  4.074807  4.124306  4.173204  4.221458  4.269022  4.315855  4.361918  4.407171        4.451578 | 101.027458  101.337652  101.727386  102.212243  102.809354   103.53715  104.415028  105.462934  106.700909  108.148599  109.824777  111.746909  113.930772  116.390164  119.136693  122.179669  125.526076  129.180625  133.145856  137.422299  142.008645  146.901948  152.097816  157.590596  163.373556  169.439035  175.778583  182.383084  189.242851  196.347713   203.68708      211.25  219.025203  227.001129  235.165961  243.507637  252.013867  260.672141  269.469734  278.393711  287.430926   296.56802  305.791425  315.087354    324.4418  333.840536  343.269106  352.712823  362.156767  371.585776  380.984449 |  100.0 |   1.027458 | 380.984449 |
    | 05:30 | 3.735175  3.740366  3.746867  3.754927  3.764814  3.776812  3.791214  3.808309  3.828378  3.851682  3.878451  3.908876  3.943105  3.981236  4.023314  4.069333  4.119237  4.172928  4.230263   4.29107  4.355148  4.422276   4.49222  4.564737  4.639582  4.716509  4.795278  4.875656  4.957416  5.040344  5.124235  5.208896  5.294144   5.37981  5.465732  5.551761  5.637756  5.723585  5.809127  5.894266  5.978893  6.062908  6.146213   6.22872  6.310341  6.390996  6.470606  6.549098  6.626399              6.70244 | 125.785285  126.057154  126.398055  126.821272  127.341328  127.973743  128.734711  129.640714  130.708075  131.952509  133.388661  135.029703  136.886977  138.969736  141.284969  143.837327  146.629146  149.660542  152.929574  156.432451  160.163766  164.116744  168.283478  172.655165  177.222313  181.974921  186.902646  191.994927    197.2411  202.630479  208.152426  213.796396  219.551978  225.408911  231.357104  237.386638  243.487771  249.650929  255.866704  262.125837  268.419216  274.737855  281.072885   287.41554  293.757142  300.089087  306.402836  312.689896  318.941811  325.150148 |   52.35175   52.403655   52.468668   52.549267   52.648139   52.768123   52.912137   53.083088   53.283783   53.516821   53.784507   54.088761   54.431053   54.812359   55.233137   55.693326   56.192374   56.729277   57.302631   57.910699   58.551476   59.222756   59.922196   60.647369   61.395816    62.16509   62.952783   63.756557   64.574159   65.403437   66.242347   67.088955   67.941442   68.798098   69.657319   70.517606   71.377555   72.235854   73.091272   73.942659   74.788932   75.629076   76.462133   77.287197   78.103411    78.90996   79.706065    80.49098   81.263991     82.024405 | 1.091073  1.091877  1.092883  1.094129  1.095656  1.097508  1.099727  1.102359  1.105442  1.109015  1.113111  1.117753  1.122961  1.128745  1.135104  1.142033  1.149516  1.157532  1.166053  1.175046  1.184477  1.194305  1.204492  1.214997  1.225781  1.236803  1.248027  1.259416  1.270938  1.282559  1.294251  1.305986  1.317739  1.329486  1.341207  1.352883  1.364494  1.376026  1.387463  1.398792      1.41  1.421076   1.43201  1.442791  1.453412  1.463863  1.474137  1.484228  1.494127  1.503829 | 1.358983   1.35893  1.358864  1.358783  1.358684  1.358564  1.358421  1.358254  1.358059  1.357835  1.357582  1.357299  1.356987  1.356645  1.356277  1.355884  1.355469  1.355034  1.354584  1.354121  1.353649  1.353171   1.35269   1.35221  1.351732  1.351258  1.350792  1.350334  1.349886  1.349449  1.349024  1.348611  1.348211  1.347824  1.347451   1.34709  1.346744   1.34641  1.346089  1.345781  1.345486  1.345202   1.34493   1.34467  1.344421  1.344182  1.343954  1.343735  1.343526          1.343327 | 0.537522  0.537929  0.538437  0.539065  0.539832   0.54076  0.541868  0.543175  0.544699  0.546455  0.548452  0.550699  0.553197  0.555942  0.558929  0.562144  0.565571  0.569192  0.572985  0.576928  0.580995  0.585163   0.58941  0.593711  0.598048  0.602399  0.606748  0.611079  0.615378  0.619632  0.623832  0.627969  0.632034  0.636023   0.63993  0.643751  0.647483  0.651124  0.654673  0.658127  0.661487  0.664753  0.667925  0.671003  0.673987   0.67688  0.679681  0.682392  0.685015      0.687549 | -0.203654  -0.204177  -0.204828  -0.205628  -0.206601   -0.20777  -0.209157  -0.210783  -0.212667  -0.214822  -0.217259  -0.219981  -0.222988  -0.226273  -0.229825  -0.233627  -0.237659  -0.241898  -0.246318  -0.250893  -0.255596    -0.2604   -0.26528  -0.270212  -0.275174  -0.280144  -0.285105   -0.29004  -0.294934  -0.299775  -0.304553  -0.309257  -0.313881  -0.318419  -0.322864  -0.327214  -0.331464  -0.335613  -0.339659    -0.3436  -0.347436  -0.351166  -0.354791  -0.358312  -0.361727  -0.365039  -0.368248  -0.371355  -0.374361     -0.377268 | 0.665895  0.665948  0.666014  0.666096  0.666194  0.666312  0.666452  0.666616  0.666807  0.667024  0.667271  0.667547  0.667853  0.668189  0.668555  0.668949  0.669371  0.669818  0.670289  0.670782  0.671294  0.671824  0.672369  0.672927  0.673495  0.674072  0.674656  0.675245  0.675837   0.67643  0.677023  0.677615  0.678205   0.67879  0.679371  0.679947  0.680516  0.681078  0.681633  0.682178  0.682715  0.683243  0.683761  0.684268  0.684765  0.685251  0.685726  0.686189   0.68664      0.687079 | 100.855551 |  100.98799   101.28473  101.657124  102.119917  102.689325  103.382823  104.218841  105.216394  106.394661  107.772542  109.368208  111.198691  113.279519   115.62442  118.245112  121.151176  124.350007  127.846838  131.644824  135.745173  140.147301   144.84901  149.846668  155.135384  160.709178  166.561131  172.683521  179.067942  185.705402  192.586411  199.701047  207.039012   214.58968   222.34213  230.285172   238.40737  246.697052  255.142326  263.731081  272.450993  281.289529   290.23394  299.271268  308.388336   317.57175  326.807892  336.082917  345.382749  354.693078          363.999354 | 368.647924 | 0.722574  0.723134  0.723836  0.724704  0.725769   0.72706  0.728607   0.73044  0.732588  0.735077  0.737929  0.741162  0.744786   0.74881  0.753234  0.758051  0.763252  0.768821  0.774738   0.78098  0.787523  0.794338  0.801398  0.808675   0.81614  0.823767   0.83153  0.839403  0.847363  0.855388  0.863458  0.871554  0.879658  0.887755   0.89583   0.90387  0.911862  0.919797  0.927663  0.935451  0.943154  0.950763  0.958272  0.965673  0.972962  0.980133   0.98718  0.994098  1.000884       1.007533 |  2.60196  2.605203  2.609265  2.614299  2.620473  2.627962  2.636947  2.647607  2.660114  2.674626  2.691283  2.710198  2.731456  2.755111  2.781182  2.809658  2.840495  2.873622  2.908943  2.946342  2.985688  3.026836  3.069637  3.113935  3.159575  3.206403   3.25427  3.303029  3.352543  3.402681  3.453318  3.504339  3.555633  3.607101  3.658646  3.710182  3.761626  3.812903  3.863941  3.914675  3.965044  4.014991   4.06446  4.113402  4.161769  4.209516    4.2566  4.302979  4.348615        4.393469 | 100.855551  101.117665  101.448408  101.861742  102.373194  102.999679  103.759241  104.670712   105.75331  107.026198  108.508033  110.216526  112.168049  114.377302  116.857057  119.617989   122.66859   126.01516  129.661863  133.610844  137.862367  142.414993  147.265758  152.410353    157.8433  163.558108  169.547423  175.803151  182.316567   189.07841  196.078955  203.308079   210.75531  218.409865  226.260681  234.296441  242.505588  250.876336  259.396682   268.05441  276.837088  285.732077  294.726524  303.807362  312.961305  322.174849  331.434264   340.72559  350.034635   359.34697  368.647924 |  100.0 |   0.855551 | 368.647924 |
    | 06:00 | 3.732287  3.736681  3.742209  3.749096  3.757587  3.767948  3.780454  3.795387  3.813023  3.833629  3.857447  3.884688  3.915529  3.950098  3.988478  4.030703  4.076759  4.126582  4.180071  4.237086  4.297455  4.360983  4.427459  4.496654  4.568337  4.642272  4.718224  4.795962  4.875263   4.95591  5.037697  5.120429  5.203918   5.28799   5.37248  5.457234  5.542106  5.626961   5.71167  5.796116  5.880185  5.963773  6.046781  6.129114  6.210684  6.291407  6.371204  6.449998  6.527715             6.604285 | 125.634162  125.864133  126.153775  126.515033  126.961131  127.506391  128.165985  128.955606  129.891091  130.987996  132.261172  133.724345  135.389749  137.267817  139.366954    141.6934   144.25118   147.04214  150.066047  153.320757  156.802405  160.505636  164.423823  168.549303   172.87358  177.387522  182.081532   186.94569  191.969876  197.143873  202.457442  207.900384  213.462586  219.134054  224.904935  230.765528   236.70629  242.717839  248.790949  254.916542  261.085682   267.28956  273.519486  279.766874  286.023233  292.280147   298.52927  304.762308  310.971009   317.14715 |  52.322875   52.366809    52.42209   52.490958   52.575875   52.679482   52.804542   52.953868   53.130234   53.336291   53.574466   53.846884   54.155286   54.500976    54.88478   55.307034   55.767586   56.265823   56.800713   57.370856   57.974547   58.609835   59.274585   59.966541   60.683372   61.422719   62.182237    62.95962   63.752628   64.559101   65.376975   66.204287    67.03918   67.879902   68.724804    69.57234   70.421061   71.269606   72.116704    72.96116   73.801854   74.637733   75.467806   76.291136   77.106839   77.914074   78.712041   79.499976   80.277146     81.042847 | 1.090626  1.091306  1.092162  1.093227   1.09454   1.09614  1.098069   1.10037  1.103083  1.106248  1.109898  1.114064  1.118767  1.124023  1.129841  1.136218  1.143148  1.150614  1.158595  1.167064  1.175988  1.185333  1.195062  1.205136  1.215517  1.226167  1.237048  1.248124  1.259361  1.270726  1.282189  1.293722  1.305298  1.316892  1.328484  1.340051  1.351575   1.36304  1.374429  1.385727  1.396922  1.408001  1.418954  1.429769  1.440438  1.450952  1.461302  1.471482  1.481483  1.491299 | 1.359012  1.358968  1.358911  1.358842  1.358756  1.358652  1.358528   1.35838  1.358208  1.358008  1.357781  1.357524  1.357238  1.356924  1.356581  1.356213  1.355821  1.355408  1.354977  1.354531  1.354073  1.353607  1.353135   1.35266  1.352186  1.351715  1.351248  1.350788  1.350336  1.349894  1.349463  1.349043  1.348635  1.348239  1.347857  1.347487   1.34713  1.346786  1.346456  1.346137  1.345832  1.345538  1.345256  1.344986  1.344726  1.344478   1.34424  1.344012  1.343794          1.343585 | 0.537296   0.53764  0.538073  0.538611  0.539272  0.540075  0.541041  0.542188  0.543534  0.545096  0.546887  0.548915  0.551187  0.553703   0.55646  0.559449  0.562658  0.566071  0.569669  0.573432  0.577337  0.581361  0.585481  0.589676  0.593922  0.598202  0.602495  0.606785  0.611058  0.615299  0.619498  0.623644  0.627728  0.631744  0.635686  0.639548  0.643327  0.647019  0.650624  0.654138  0.657561  0.660892  0.664132  0.667279  0.670335    0.6733  0.676175  0.678961  0.681659      0.684269 |  -0.20336  -0.203804   -0.20436  -0.205047  -0.205887  -0.206902  -0.208115  -0.209547  -0.211217  -0.213142  -0.215334  -0.217801  -0.220547  -0.223569   -0.22686  -0.230408  -0.234196  -0.238205  -0.242411   -0.24679  -0.251317  -0.255966  -0.260711  -0.265528  -0.270395  -0.275289  -0.280191  -0.285084  -0.289951  -0.294779  -0.299556  -0.304272  -0.308917  -0.313484  -0.317967  -0.322362  -0.326664  -0.330869  -0.334976  -0.338983  -0.342888  -0.346691  -0.350392  -0.353989  -0.357485  -0.360878   -0.36417  -0.367362  -0.370454     -0.373447 | 0.665864   0.66591  0.665966  0.666036  0.666121  0.666224  0.666346  0.666491  0.666659  0.666853  0.667074  0.667323  0.667601  0.667908  0.668245   0.66861  0.669003  0.669422  0.669865  0.670332   0.67082  0.671326  0.671849  0.672387  0.672937  0.673497  0.674066  0.674641  0.675221  0.675805  0.676389  0.676974  0.677558  0.678139  0.678717   0.67929  0.679859  0.680421  0.680976  0.681524  0.682064  0.682595  0.683117   0.68363  0.684133  0.684625  0.685107  0.685578  0.686038      0.686487 | 100.711411 | 100.823137  101.074029  101.390242  101.784989  102.272971  102.870219  103.593865  104.461837  105.492491  106.704212  108.114991  109.742022  111.601318  113.707391  116.072996  118.708954  121.624043  124.824979   128.31644  132.101163  136.180062  140.552385  145.215874  150.166939  155.400815  160.911721  166.693001  172.737243  179.036393  185.581843  192.364512  199.374906  206.603172  214.039139   221.67235  229.492084  237.487378  245.647039  253.959651  262.413585  270.996997  279.697833  288.503827    297.4025  306.381154  315.426873  324.526517  333.666717  342.833873          352.014148 | 356.603706 | 0.722262  0.722737  0.723333  0.724076  0.724991  0.726106  0.727451  0.729054  0.730945   0.73315  0.735692  0.738593  0.741867  0.745525  0.749573  0.754009  0.758827  0.764016   0.76956   0.77544  0.781634  0.788117  0.794862  0.801844  0.809035  0.816408  0.823937  0.831597  0.839365  0.847217  0.855133  0.863093  0.871079  0.879075  0.887064  0.895033   0.90297  0.910861  0.918698  0.926469  0.934166   0.94178  0.949305  0.956733  0.964058  0.971274  0.978376  0.985358  0.992216       0.998946 | 2.600155  2.602901  2.606355  2.610657  2.615961  2.622429  2.630234  2.639549  2.650546  2.663385  2.678214  2.695162  2.714331  2.735796    2.7596  2.785757   2.81425   2.84503  2.878026  2.913142  2.950266  2.989268  3.030011  3.072348  3.116133  3.161214  3.207446  3.254685  3.302791  3.351632  3.401083  3.451023  3.501341  3.551933    3.6027  3.653552  3.704402  3.755172  3.805788  3.856182  3.906289   3.95605  4.005408   4.05431  4.102708  4.150554  4.197804  4.244415  4.290348        4.335563 | 100.711411  100.932502  101.212649  101.564295  102.001427   102.53946  103.195052  103.985822  104.930019  106.046127  107.352452  108.866696   110.60556  112.584389  114.816879  117.314857  120.088144  123.144484  126.489561  130.127051  134.058743  138.284675  142.803295  147.611634  152.705472  158.079495  163.727448  169.642264  175.816185  182.240854  188.907409  195.806546  202.928579  210.263485  217.800943  225.530358  233.440882  241.521436  249.760711  258.147185   266.66912  275.314568  284.071367  292.927146  301.869317  310.885075  319.961391  329.085013  338.242458  347.420008  356.603706 |  100.0 |   0.711411 | 356.603706 |
    | 06:30 | 3.729863  3.733574  3.738265  3.744136  3.751411  3.760334  3.771163  3.784166  3.799615  3.817773   3.83889  3.863193  3.890877  3.922099  3.956974  3.995572  4.037915  4.083981  4.133702  4.186973  4.243656  4.303582  4.366563  4.432392  4.500851  4.571716  4.644762  4.719762  4.796496   4.87475  4.954315  5.034995  5.116599  5.198951   5.28188  5.365228  5.448845  5.532592  5.616337  5.699956  5.783334   5.86636  5.948934  6.030957  6.112339  6.192994  6.272837  6.351792  6.429783             6.506739 | 125.507328  125.701496  125.947093  126.254803  126.636581  127.105543  127.675774  128.362068  129.179601  130.143563  131.268758  132.569204  134.057755  135.745766  137.642825   139.75656   142.09253  144.654195  147.442964  150.458305  153.697899  157.157834  160.832812  164.716363  168.801054  173.078687  177.540472  182.177191  186.979324  191.937167  197.040922  202.280771  207.646929   213.12969  218.719455  224.406754  230.182255  236.036773  241.961268  247.946843  253.984738   260.06632  266.183074  272.326592  278.488561   284.66075     290.835   297.00321  303.157325  309.289324 |  52.298629   52.335742   52.382649   52.441358   52.514109   52.603335   52.711626   52.841663   52.996151   53.177733   53.388905   53.631932   53.908767   54.220986    54.56974    54.95572   55.379153   55.839806   56.337015   56.869729   57.436556   58.035822   58.665631   59.323918   60.008508   60.717162   61.447615   62.197619   62.964962   63.747497   64.543152   65.349946   66.165993   66.989505   67.818796   68.652277   69.488453   70.325923   71.163371   71.999562   72.833336   73.663603   74.489338   75.309573   76.123395   76.929936   77.728373   78.517923   79.297834     80.067387 | 1.090251  1.090825  1.091552   1.09246  1.093585  1.094964  1.096636  1.098642  1.101021  1.103813  1.107055  1.110778  1.115008  1.119767  1.125067  1.130914  1.137305  1.144232  1.151679  1.159623  1.168037  1.176891  1.186152  1.195782  1.205745  1.216005  1.226525  1.237268    1.2482  1.259288  1.270502  1.281811  1.293189  1.304611  1.316052  1.327491  1.338909  1.350287  1.361608  1.372857   1.38402  1.395084  1.406038   1.41687   1.42757  1.438128  1.448537  1.458788  1.468874  1.478787 | 1.359037  1.358999  1.358952  1.358892  1.358818  1.358729   1.35862  1.358491  1.358339  1.358161  1.357958  1.357726  1.357466  1.357178  1.356862  1.356519  1.356151  1.355761   1.35535  1.354922   1.35448  1.354027  1.353566    1.3531  1.352632  1.352164  1.351699  1.351239  1.350785  1.350339  1.349903  1.349477  1.349062  1.348658  1.348267  1.347889  1.347523   1.34717  1.346829  1.346501  1.346185  1.345881  1.345589  1.345309   1.34504  1.344782  1.344534  1.344297   1.34407          1.343852 | 0.537106  0.537397  0.537764  0.538224  0.538791  0.539485  0.540324  0.541327  0.542511  0.543895  0.545493  0.547316  0.549373  0.551668    0.5542  0.556965  0.559955  0.563156  0.566553  0.570128  0.573861  0.577729   0.58171  0.585783  0.589927   0.59412  0.598344  0.602581  0.606815  0.611031  0.615217  0.619361  0.623454  0.627488  0.631456  0.635351  0.639169  0.642907  0.646562   0.65013  0.653611  0.657003  0.660306  0.663519  0.666643  0.669677  0.672623  0.675481  0.678251      0.680935 | -0.203112  -0.203489  -0.203962   -0.20455  -0.205273  -0.206152  -0.207209  -0.208465   -0.20994  -0.211652  -0.213615  -0.215841  -0.218337  -0.221103  -0.224138  -0.227432  -0.230974  -0.234747  -0.238731  -0.242904  -0.247242  -0.251722  -0.256317  -0.261005  -0.265761  -0.270563  -0.275392  -0.280228  -0.285054  -0.289857  -0.294621  -0.299335  -0.303991  -0.308578   -0.31309  -0.317521  -0.321866  -0.326121  -0.330283  -0.334349  -0.338318  -0.342188  -0.345958  -0.349629    -0.3532  -0.356671  -0.360042  -0.363315  -0.366489     -0.369565 | 0.665839  0.665877  0.665926  0.665985  0.666059  0.666148  0.666254  0.666381  0.666529  0.666701  0.666898  0.667122  0.667374  0.667654  0.667963  0.668299  0.668664  0.669055  0.669471  0.669911  0.670373  0.670855  0.671356  0.671872  0.672403  0.672945  0.673498  0.674059  0.674626  0.675198  0.675772  0.676349  0.676925  0.677501  0.678074  0.678645  0.679211  0.679772  0.680327  0.680876  0.681418  0.681951  0.682477  0.682994  0.683502     0.684  0.684488  0.684966  0.685434       0.68589 | 100.590745 | 100.684832   100.89658  101.164575  101.500596   101.91789  102.431066  103.055929  103.809235  104.708389  105.771082  107.014919  108.457013  110.113622  111.999798  114.129107    116.5134  119.162671  122.084969  125.286389  128.771112  132.541498  136.598205  140.940334  145.565584  150.470411  155.650177  161.099292  166.811347  172.779224  178.995201  185.451032  192.138018  199.047071  206.168755  213.493328  221.010767  228.710797  236.582902  244.616338  252.800145  261.123147  269.573957  278.140977    286.8124  295.576202  304.420144  313.331767  322.298386  331.307091          340.344737 | 344.866663 |    0.722  0.722401  0.722908  0.723541  0.724325  0.725287  0.726452   0.72785  0.729508  0.731454  0.733712  0.736305   0.73925  0.742563  0.746252  0.750319  0.754765   0.75958  0.764755  0.770273  0.776116  0.782261  0.788684  0.795361  0.802266  0.809373  0.816655  0.824089   0.83165  0.839315  0.847062  0.854872  0.862726  0.870605  0.878495   0.88638  0.894247  0.902082  0.909876  0.917617  0.925295  0.932903  0.940431  0.947873  0.955223  0.962472  0.969617  0.976651   0.98357       0.990368 |  2.59864   2.60096  2.603891  2.607559  2.612103  2.617675  2.624436   2.63255  2.642186  2.653506  2.666662  2.681791   2.69901  2.718413  2.740063  2.763997  2.790222  2.818714  2.849424   2.88228  2.917186  2.954031   2.99269  3.033031  3.074914  3.118195  3.162731  3.208382  3.255009   3.30248  3.350667   3.39945  3.448713   3.49835  3.548259  3.598345  3.648522  3.698705   3.74882  3.798793  3.848558  3.898053   3.94722  3.996003  4.044351  4.092216  4.139552  4.186314  4.232462        4.277955 | 100.590745  100.776908  101.013762  101.312339  101.685166  102.146207  102.710724  103.395073  104.216425   105.19243  106.340845  107.679135  109.224088  110.991447  112.995593  115.249286  117.763485  120.547222  123.607567  126.949641  130.576686  134.490173  138.689941  143.174349  147.940431  152.984055  158.300069  163.882443  169.724382  175.818441  182.156615  188.730413  195.530928  202.548885  209.774686  217.198441  224.809997  232.598953  240.554676  248.666313  256.922794  265.312837   273.82495  282.447431  291.168363  299.975617  308.856845  317.799476  326.790713  335.817529  344.866663 |  100.0 |   0.590745 | 344.866663 |
    | 07:00 |  3.72783   3.73096  3.734932  3.739926  3.746144  3.753808  3.763159  3.774451  3.787941   3.80389   3.82255  3.844154  3.868914  3.897009  3.928581  3.963732  4.002517  4.044951  4.091003  4.140601  4.193641  4.249983  4.309463  4.371898  4.437088  4.504822  4.574886  4.647062  4.721134   4.79689  4.874125  4.952638   5.03224   5.11275  5.193995  5.275813  5.358051  5.440564  5.523216   5.60588  5.688437  5.770773  5.852783  5.934366  6.015428   6.09588  6.175638  6.254619  6.332749             6.409952 | 125.401056  125.564697  125.772547  126.034103  126.360112    126.7625  127.254238  127.849141  128.561599  129.406261  130.397675  131.549909  132.876185  134.388528   136.09747  138.011818  140.138484  142.482409  145.046545  147.831914  150.837721  154.061503  157.499319  161.145937  164.995044  169.039433  173.271186  177.681843  182.262541  187.004138  191.897323  196.932693  202.100824  207.392324  212.797869  218.308235  223.914314  229.607123  235.377817   241.21768   247.11813  253.070707  259.067071  265.098985  271.158313  277.237002  283.327074  289.420615   295.50976  301.586685 |  52.278304   52.309597   52.349316   52.399256   52.461436   52.538082   52.631595   52.744505    52.87941   53.038903   53.225497    53.44154   53.689139   53.970089   54.285812   54.637317   55.025173    55.44951   55.910025   56.406013   56.936407   57.499827   58.094633   58.718981   59.370876   60.048219   60.748858   61.470618    62.21134   62.968903   63.741246   64.526381   65.322404   66.127502   66.939954   67.758134   68.580509   69.405637    70.23216   71.058804    71.88437   72.707731   73.527827   74.343659   75.154283   75.958805   76.756377   77.546195   78.327487     79.099518 | 1.089936   1.09042  1.091036  1.091809  1.092771  1.093956  1.095401  1.097143  1.099223  1.101679  1.104547  1.107862  1.111653  1.115944  1.120753  1.126092  1.131964  1.138365  1.145286   1.15271  1.160615  1.168974  1.177758  1.186934  1.196467  1.206322  1.216463  1.226855  1.237464  1.248256    1.2592  1.270266  1.281426  1.292654  1.303925  1.315217  1.326508   1.33778  1.349015  1.360197  1.371311  1.382342  1.393279  1.404109  1.414823  1.425409   1.43586  1.446166   1.45632  1.466313 | 1.359058  1.359026  1.358985  1.358935  1.358872  1.358794    1.3587  1.358587  1.358454  1.358297  1.358115  1.357907  1.357672  1.357409  1.357119  1.356801  1.356458  1.356091  1.355702  1.355294   1.35487  1.354432  1.353984  1.353528  1.353068  1.352605  1.352144  1.351685  1.351231  1.350783  1.350343  1.349912  1.349491  1.349081  1.348682  1.348296  1.347921  1.347558  1.347208  1.346871  1.346545  1.346232   1.34593   1.34564  1.345361  1.345094  1.344837   1.34459  1.344354          1.344127 | 0.536946  0.537192  0.537503  0.537894   0.53838  0.538978  0.539704  0.540578  0.541617  0.542838  0.544258  0.545889  0.547743  0.549826  0.552141  0.554687  0.557459  0.560447   0.56364   0.56702  0.570571  0.574273  0.578104  0.582044  0.586071  0.590164  0.594306  0.598476  0.602657  0.606836  0.610997  0.615129  0.619221  0.623263  0.627248  0.631168  0.635019  0.638795  0.642493   0.64611  0.649643  0.653091  0.656453  0.659728  0.662916  0.666016   0.66903  0.671956  0.674797      0.677552 | -0.202904  -0.203223  -0.203624  -0.204127  -0.204747  -0.205506  -0.206424  -0.207522  -0.208819  -0.210335  -0.212086  -0.214086  -0.216343  -0.218864  -0.221649  -0.224693  -0.227989  -0.231523  -0.235279  -0.239237  -0.243377  -0.247675  -0.252108  -0.256651  -0.261282  -0.265978  -0.270719  -0.275484  -0.280256  -0.285018  -0.289757  -0.294459  -0.299113   -0.30371  -0.308241    -0.3127   -0.31708  -0.321376  -0.325586  -0.329705  -0.333731  -0.337663  -0.341498  -0.345237  -0.348879  -0.352423  -0.355869  -0.359219  -0.362472     -0.365628 | 0.665817   0.66585  0.665891  0.665942  0.666005  0.666082  0.666174  0.666285  0.666415  0.666567  0.666743  0.666944  0.667171  0.667425  0.667706  0.668015  0.668352  0.668715  0.669104  0.669518  0.669954  0.670412  0.670889  0.671384  0.671894  0.672418  0.672953  0.673498  0.674051   0.67461  0.675174   0.67574  0.676309  0.676877  0.677445  0.678011  0.678574  0.679133  0.679687  0.680235  0.680778  0.681313  0.681841  0.682361  0.682873  0.683376  0.683869  0.684354  0.684828      0.685291 | 100.489885 | 100.568984  100.747384  100.974092  101.259562   101.61566  102.055618  102.593916  103.246094  104.028507  104.958021  106.051668  107.326283  108.798136  110.482592    112.3938  114.544439  116.945526  119.606291  122.534118  125.734536  129.211281  132.966373  137.000244  141.311872  145.898923  150.757905  155.884302  161.272711  166.916955  172.810198  178.945028  185.313538  191.907394  198.717885  205.735971   212.95231  220.357296  227.941069  235.693536  243.604382  251.663075   259.85887  268.180814  276.617744  285.158287  293.790856  302.503648  311.284641  320.121589          329.002018 | 333.448727 | 0.721781  0.722119  0.722548  0.723087  0.723758  0.724584  0.725591  0.726806  0.728255  0.729966  0.731965  0.734274  0.736914  0.739902   0.74325  0.746965   0.75105  0.755501  0.760313  0.765471  0.770962  0.776766  0.782862  0.789227  0.795836  0.802665   0.80969  0.816884  0.824225  0.831688  0.839254  0.846899  0.854606  0.862356  0.870133   0.87792  0.885703   0.89347  0.901207  0.908905  0.916553  0.924141  0.931661  0.939106  0.946467  0.953739  0.960915   0.96799  0.974957       0.981813 |  2.59737  2.599325  2.601808  2.604928  2.608813    2.6136   2.61944  2.626488  2.634905  2.644852  2.656482  2.669939   2.68535  2.702823  2.722439  2.744255  2.768301  2.794576  2.823054  2.853683  2.886389  2.921078  2.957643  2.995962  3.035907  3.077341  3.120129  3.164132  3.209216  3.255248  3.302101  3.349652  3.397786  3.446391  3.495365   3.54461  3.594035  3.643555   3.69309  3.742566  3.791912  3.841065  3.889963   3.93855  3.986771  4.034577   4.08192  4.128755  4.175039        4.220731 | 100.489885  100.646374  100.846266  101.099299  101.416638  101.810851  102.295835  102.886659  103.599345  104.450594  105.457452  106.636954  108.005753  109.579759  111.373807  113.401373  115.674353  118.202896  120.995318  124.058071  127.395767  131.011257  134.905732  139.078858  143.528916  148.252955  153.246934  158.505857  164.023904  169.794543  175.810624   182.06447  188.547943  195.252507  202.169277  209.289051  216.602351  224.099437   231.77033  239.604823   247.59249  255.722692   263.98458  272.367096  280.858972  289.448728  298.124671  306.874889  315.687249  324.549392  333.448727 |  100.0 |   0.489885 | 333.448727 |
    | 07:30 |  3.72613  3.728763   3.73212  3.736359  3.741662  3.748229  3.756284   3.76606  3.777806  3.791771  3.808204  3.827345  3.849412  3.874602  3.903079  3.934972  3.970368  4.009314  4.051813  4.097829  4.147288  4.200082  4.256076  4.315108     4.377  4.441557  4.508579  4.577857  4.649183   4.72235  4.797153  4.873395  4.950885   5.02944  5.108885  5.189055  5.269792   5.35095  5.432387  5.513974  5.595586  5.677107  5.758428  5.839445  5.920061  6.000184  6.079727  6.158606  6.236743             6.314062 | 125.312153  125.449827  125.625402  125.847286   126.12508  126.469561  126.892584  127.406944  128.026157  128.764194   129.63517  130.652997  131.831032  133.181732  134.716339  136.444618  138.374647  140.512687  142.863113  145.428421  148.209289  151.204693  154.412055  157.827416  161.445622  165.260512  169.265099  173.451736  177.812271  182.338181  187.020683  191.850831  196.819597  201.917931  207.136809   212.46727  217.900442   223.42756  229.039975  234.729161  240.486714  246.304348  252.173892  258.087283  264.036554  270.013826  276.011298  282.021234  288.035955  294.047826 |  52.261296   52.287633   52.321201   52.363591   52.416616   52.482294   52.562836   52.660601   52.778055   52.917709   53.082044   53.273445   53.494118    53.74602   54.030795   54.349723   54.703683   55.093137   55.518126   55.978285   56.472876   57.000823    57.56076   58.151082   58.769996   59.415572   60.085788   60.778571   61.491832   62.223497   62.971531   63.733952   64.508852     65.2944   66.088851   66.890548   67.697923   68.509497   69.323874   70.139741   70.955861   71.771073   72.584281   73.394453   74.200614   75.001843   75.797269   76.586061   77.367429      78.14062 | 1.089672   1.09008    1.0906  1.091257  1.092077  1.093093  1.094338  1.095849  1.097661  1.099813  1.102342  1.105283  1.108668  1.112522   1.11687  1.121725  1.127098  1.132991  1.139398  1.146309  1.153708  1.161572  1.169876   1.17859  1.187682  1.197119  1.206867  1.216892   1.22716  1.237637  1.248293  1.259097   1.27002  1.281035  1.292116  1.303241  1.314388  1.325536  1.336666  1.347762  1.358808  1.369789  1.380691  1.391503  1.402214  1.412812  1.423287  1.433632  1.443837  1.453895 | 1.359075  1.359048  1.359014  1.358971  1.358917  1.358851  1.358769  1.358671  1.358554  1.358416  1.358255  1.358069  1.357857  1.357618  1.357353   1.35706  1.356742  1.356399  1.356032  1.355645   1.35524  1.354819  1.354385  1.353942  1.353491  1.353037   1.35258  1.352125  1.351672  1.351223  1.350781  1.350347  1.349921  1.349505    1.3491  1.348706  1.348324  1.347953  1.347594  1.347247  1.346912  1.346589  1.346278  1.345978   1.34569  1.345413  1.345147  1.344891  1.344645           1.34441 | 0.536812  0.537019  0.537283  0.537615   0.53803  0.538543   0.53917  0.539929  0.540837  0.541911  0.543167  0.544621  0.546284  0.548166  0.550273  0.552606  0.555164   0.55794  0.560926  0.564108  0.567472  0.570998  0.574669  0.578464  0.582362  0.586344  0.590389  0.594479  0.598597  0.602725   0.60685  0.610959  0.615038  0.619079  0.623071  0.627008  0.630883   0.63469  0.638425  0.642084  0.645664  0.649163  0.652579  0.655911  0.659159  0.662321  0.665399  0.668391  0.671299      0.674122 | -0.202729  -0.202998  -0.203339  -0.203767  -0.204298  -0.204951  -0.205746  -0.206702  -0.207839  -0.209176  -0.210732   -0.21252  -0.214553  -0.216839  -0.219383  -0.222183  -0.225235   -0.22853  -0.232055  -0.235793  -0.239726  -0.243832   -0.24809  -0.252476  -0.256968  -0.261544  -0.266182  -0.270862  -0.275566  -0.280275  -0.284976  -0.289653  -0.294295   -0.29889   -0.30343  -0.307907  -0.312313  -0.316643  -0.320893  -0.325058  -0.329135  -0.333122  -0.337017  -0.340819  -0.344527   -0.34814  -0.351657   -0.35508  -0.358408     -0.361641 | 0.665799  0.665827  0.665862  0.665905  0.665959  0.666025  0.666105  0.666202  0.666316   0.66645  0.666606  0.666785  0.666989  0.667218  0.667474  0.667757  0.668067  0.668403  0.668765  0.669152  0.669563  0.669996  0.670449  0.670921   0.67141  0.671914  0.672431  0.672959  0.673497  0.674042  0.674593  0.675149  0.675708  0.676269   0.67683   0.67739  0.677948  0.678504  0.679056  0.679603  0.680145  0.680681   0.68121  0.681733  0.682248  0.682754  0.683252  0.683741  0.684221      0.684692 | 100.405708 | 100.472098  100.622146  100.813583  101.055643  101.358904   101.73528   102.19795  102.761225  103.440352  104.251265  105.210282  106.333779  107.637843  109.137933  110.848565  112.783037  114.953202  117.369304   120.03987  122.971671   126.16973  129.637372  133.376319  137.386801  141.667691  146.216634  151.030193  156.103973  161.432747  167.010565   172.83085  178.886487  185.169889  191.673065   198.38766  205.305005   212.41614  219.711847  227.182662  234.818892  242.610626  250.547739  258.619897  266.816559  275.126973  283.540183  292.045016  300.630088  309.283795          317.994307 | 322.359175 | 0.721597  0.721882  0.722244  0.722702  0.723274  0.723983  0.724851  0.725903  0.727166  0.728666  0.730429  0.732478  0.734835   0.73752  0.740546  0.743926  0.747665  0.751764   0.75622  0.761024  0.766165  0.771627  0.777392  0.783439  0.789745  0.796288  0.803043  0.809987  0.817095  0.824345  0.831714  0.839182  0.846729  0.854336  0.861985  0.869661  0.877348  0.885033  0.892702  0.900344  0.907949  0.915505  0.923006  0.930441  0.937803  0.945086  0.952282  0.959385  0.966391       0.973294 | 2.596306  2.597953  2.600051    2.6027  2.606013  2.610116  2.615146  2.621251  2.628581  2.637294  2.647542   2.65947  2.673213  2.688889  2.706596  2.726407  2.748372  2.772511  2.798821  2.827272  2.857808  2.890358  2.924827   2.96111  2.999091  3.038643  3.079637  3.121942  3.165425  3.209956  3.255408  3.301659  3.348591  3.396093  3.444059  3.492389  3.540989   3.58977  3.638651  3.687555  3.736408  3.785144  3.833699  3.882015  3.930035  3.977707  4.024983  4.071815  4.118159        4.163973 | 100.405708  100.537037  100.705441  100.919482  101.189059  101.525429  101.941164  102.450052  103.066934  103.807476   104.68789  105.724617  106.933984  108.331856  109.933304  111.752305  113.801491  116.091948  118.633083  121.432552  124.496242  127.828309  131.431248  135.306002  139.452085  143.867719  148.549972  153.494893  158.697638  164.152591  169.853463  175.793387  181.964994  188.360479  194.971657  201.790006  208.806704  216.012656  223.398516  230.954702  238.671409   246.53862  254.546101  262.683417   270.93992  279.304757  287.766865  296.314966  304.937565  313.622948  322.359175 |  100.0 |   0.405708 | 322.359175 |
    | 08:00 | 3.724709  3.726921  3.729753  3.733344  3.737856  3.743471   3.75039  3.758833   3.76903  3.781221  3.795649   3.81255   3.83215  3.854657  3.880251  3.909083  3.941268  3.976882  4.015962  4.058502  4.104463  4.153767  4.206306  4.261945  4.320527  4.381878  4.445812  4.512132  4.580641  4.651136  4.723418  4.797293  4.872568  4.949062  5.026598  5.105007  5.184131  5.263817  5.343923  5.424314  5.504863   5.58545  5.665962  5.746292  5.826341  5.906012  5.985217  6.063869  6.141889             6.219197 | 125.237898  125.353528  125.501574   125.68944  125.925665  126.219926   126.58299  127.026614  127.563383  128.206492  128.969484  129.865934  130.909133  132.111746  133.485503  135.040908  136.787004  138.731201  140.879158  143.234737  145.800023   148.57539  151.559616  154.750026  158.142661  161.732447  165.513376  169.478674  173.620952  177.932353  182.404674   187.02947  191.798143  196.702018  201.732397  206.880604  212.138022  217.496112  222.946434  228.480656  234.090555  239.768021  245.505055   251.29376  257.126338  262.995078   268.89235  274.810593  280.742306  286.680035 |  52.247086   52.269212   52.297528   52.333439   52.378558   52.434707   52.503903   52.588328   52.690299   52.812213   52.956488   53.125499   53.321503   53.546568    53.80251   54.090832   54.412683   54.768824   55.159615   55.585023   56.044629   56.537667   57.063056   57.619446   58.205268    58.81878   59.458116   60.121323   60.806406   61.511357   62.234183   62.972926   63.725684   64.490622   65.265978   66.050073   66.841308   67.638171   68.439234   69.243145   70.048634   70.854502    71.65962   72.462923   73.263407   74.060121   74.852168   75.638693   76.418886     77.191973 | 1.089452  1.089795  1.090233   1.09079  1.091488  1.092357  1.093428  1.094732  1.096307  1.098188   1.10041  1.103011  1.106021  1.109471  1.113386  1.117785  1.122682  1.128085  1.133994  1.140404  1.147304  1.154675  1.162497  1.170744  1.179387  1.188396  1.197739  1.207382  1.217294   1.22744   1.23779  1.248313   1.25898  1.269763  1.280637  1.291577   1.30256  1.313565  1.324573  1.335565  1.346526  1.357439   1.36829  1.379067  1.389758  1.400351  1.410835  1.421202  1.431443  1.441549 |  1.35909  1.359067  1.359038  1.359002  1.358956  1.358899  1.358829  1.358744  1.358642   1.35852  1.358378  1.358212  1.358022  1.357807  1.357565  1.357297  1.357003  1.356684  1.356341  1.355976   1.35559  1.355188   1.35477  1.354341  1.353902  1.353457  1.353007  1.352556  1.352107  1.351659  1.351217   1.35078  1.350351  1.349931   1.34952   1.34912   1.34873  1.348351  1.347984  1.347629  1.347285  1.346953  1.346633  1.346324  1.346026   1.34574  1.345464  1.345199  1.344944            1.3447 |   0.5367  0.536874  0.537097  0.537379  0.537732  0.538172  0.538711  0.539368  0.540159    0.5411  0.542208  0.543498  0.544985  0.546678  0.548586  0.550714  0.553063  0.555631   0.55841  0.561392  0.564563  0.567908   0.57141   0.57505  0.578808  0.582666  0.586603  0.590601  0.594642  0.598708  0.602785  0.606858  0.610915  0.614943  0.618934  0.622879  0.626769  0.630599  0.634364  0.638058  0.641679  0.645223  0.648689  0.652074  0.655377  0.658598  0.661736   0.66479  0.667762      0.670651 | -0.202583  -0.202809  -0.203098  -0.203461  -0.203915  -0.204476  -0.205162  -0.205992  -0.206985  -0.208161  -0.209537  -0.211129  -0.212953  -0.215017   -0.21733  -0.219894  -0.222707  -0.225765  -0.229057  -0.232571  -0.236291  -0.240197  -0.244269  -0.248487  -0.252828   -0.25727  -0.261792  -0.266373  -0.270994  -0.275638  -0.280287  -0.284927  -0.289544  -0.294128  -0.298666  -0.303151  -0.307575   -0.31193  -0.316212  -0.320415  -0.324537  -0.328573  -0.332522  -0.336381   -0.34015  -0.343827  -0.347412  -0.350903  -0.354303     -0.357609 | 0.665784  0.665807  0.665837  0.665874   0.66592  0.665977  0.666046   0.66613  0.666229  0.666347  0.666485  0.666644  0.666827  0.667033  0.667265  0.667523  0.667807  0.668117  0.668453  0.668814  0.669199  0.669606  0.670036  0.670485  0.670951  0.671435  0.671933  0.672443  0.672964  0.673495  0.674033  0.674577  0.675125  0.675676  0.676229  0.676782  0.677335  0.677886  0.678435   0.67898   0.67952  0.680056  0.680586   0.68111  0.681626  0.682136  0.682638  0.683131  0.683616      0.684091 |  100.33556 | 100.391192  100.517185  100.678558  100.883429  101.141185  101.462497  101.859289  102.344651  102.932694  103.638348  104.477107  105.464742  106.616977  107.949175  109.476017  111.211215  113.167266  115.355251  117.784697  120.463485  123.397829  126.592282  130.049806  133.771851  137.758473  142.008451  146.519422  151.288006  156.309926  161.580124  167.092864  172.841819   178.82015  185.020576  191.435424  198.056681  204.876025  211.884857  219.074326  226.435342  233.958592  241.634548   249.45347  257.405415  265.480231  273.667563  281.956847   290.33731  298.797965          307.327606 | 311.604896 | 0.721444  0.721683  0.721989  0.722376  0.722863  0.723469  0.724215  0.725125  0.726223  0.727533  0.729083  0.730894  0.732991  0.735395  0.738121  0.741184  0.744592  0.748352  0.752462  0.756919  0.761715  0.766837  0.772269  0.777994  0.783992  0.790241  0.796718    0.8034  0.810265  0.817289   0.82445  0.831728  0.839102  0.846552  0.854062  0.861613  0.869191  0.876781  0.884369  0.891943  0.899493  0.907006  0.914475  0.921889  0.929241  0.936522  0.943728   0.95085  0.957883       0.964821 | 2.595418  2.596801  2.598571  2.600816  2.603635  2.607143  2.611466  2.616738  2.623104  2.630713  2.639713   2.65025  2.662464  2.676478  2.692403  2.710327  2.730316  2.752411  2.776629   2.80296   2.83137  2.861806  2.894191  2.928437  2.964438  3.002082  3.041247  3.081809   3.12364  3.166614  3.210605  3.255493  3.301158  3.347488  3.394376   3.44172  3.489423  3.537394   3.58555  3.633809  3.682097  3.730344  3.778485  3.826457  3.874204   3.92167  3.968806  4.015562  4.061893        4.107756 |  100.33556  100.445596  100.587233  100.767965  100.996534  101.282967  101.638577  102.075902  102.608588  103.251219  104.019081  104.927891  105.993485  107.231496  108.657031  110.284361  112.126654  114.195747  116.501977  119.054065  121.859061  124.922338  128.247635  131.837129  135.691542  139.810256  144.191446  148.832204   153.72867  158.876148  164.269213  169.901811   175.76734  181.858722  188.168467  194.688719  201.411303   208.32775  215.429331   222.70707  230.151763  237.753987  245.504106  253.392279  261.408458  269.542389  277.783614  286.121465  294.545062  303.043311  311.604896 |  100.0 |    0.33556 | 311.604896 |
    | 08:30 | 3.723523  3.725379  3.727763  3.730799  3.734631   3.73942  3.745351  3.752623  3.761452  3.772064  3.784692  3.799568   3.81692   3.83696  3.859883  3.885855  3.915016  3.947467  3.983273  4.022462  4.065023   4.11091  4.160045  4.212318  4.267597  4.325729  4.386543  4.449861  4.515492  4.583246  4.652929  4.724348  4.797317  4.871651  4.947175  5.023718   5.10112  5.179225  5.257888  5.336971  5.416343  5.495881  5.575468  5.654996   5.73436  5.813462  5.892211  5.970517  6.048299             6.125476 | 125.175971  125.272927  125.397541  125.556311   125.75679  126.007623  126.318527  126.700232  127.164363  127.723275  128.389824  129.177111   130.09818   131.16571  132.391702  133.787191  135.361986  137.124459  139.081398   141.23791  143.597393  146.161562   148.93052  151.902874  155.075872  158.445565  162.006965  165.754215  169.680745  173.779415  178.042648  182.462543  187.030976   191.73968   196.58031  201.544502   206.62391  211.810241  217.095273  222.470875  227.929012  233.461753  239.061268  244.719829  250.429802  256.183642  261.973885  267.793137  273.634071  279.489407 |  52.235232    52.25379   52.277632   52.307994   52.346307   52.394203   52.453508   52.526228   52.614516   52.720636   52.846916   52.995681   53.169198     53.3696   53.598825   53.858554   54.150159   54.474665   54.832727   55.224616   55.650228   56.109101   56.600445   57.123178   57.675969   58.257285   58.865434   59.498608   60.154924   60.832461   61.529285   62.243481   62.973165    63.71651   64.471747   65.237183   66.011198   66.792251   67.578883   68.369712   69.163431    69.95881   70.754684   71.549959   72.343597   73.134621   73.922106   74.705172   75.482988     76.254758 | 1.089268  1.089556  1.089925  1.090396  1.090989   1.09173  1.092648  1.093773  1.095137  1.096775  1.098723  1.101014  1.103682  1.106759  1.110271  1.114242  1.118689  1.123624  1.129053  1.134975  1.141385  1.148269  1.155611   1.16339   1.17158  1.180153   1.18908  1.198329   1.20787   1.21767  1.227697  1.237923  1.248316   1.25885  1.269498  1.280234  1.291036   1.30188  1.312748  1.323619  1.334478  1.345306   1.35609  1.366815   1.37747  1.388041  1.398519  1.408893  1.419154  1.429292 | 1.359102  1.359083  1.359059  1.359028  1.358989   1.35894   1.35888  1.358806  1.358717  1.358611  1.358486  1.358339   1.35817  1.357976  1.357757  1.357513  1.357243  1.356947  1.356627  1.356284   1.35592  1.355537  1.355137  1.354723  1.354298  1.353864  1.353423  1.352979  1.352534   1.35209  1.351648  1.351211   1.35078  1.350357  1.349941  1.349535  1.349139  1.348754  1.348379  1.348016  1.347663  1.347323  1.346994  1.346676  1.346369  1.346073  1.345789  1.345515  1.345251          1.344997 | 0.536607  0.536753  0.536941  0.537179   0.53748  0.537855  0.538318  0.538885  0.539572  0.540393  0.541367  0.542508  0.543831  0.545348  0.547069  0.549002  0.551149  0.553513  0.556088  0.558869  0.561845  0.565004   0.56833  0.571806  0.575415  0.579138  0.582956   0.58685  0.590801  0.594793   0.59881  0.602837  0.606859  0.610866  0.614845  0.618788  0.622685   0.62653  0.630317   0.63404  0.637696  0.641279  0.644789  0.648221  0.651575   0.65485  0.658045  0.661158   0.66419      0.667141 | -0.202461  -0.202651  -0.202894  -0.203203  -0.203589  -0.204069   -0.20466  -0.205378  -0.206243  -0.207273  -0.208486  -0.209899  -0.211527  -0.213383  -0.215477  -0.217814  -0.220396   -0.22322  -0.226282   -0.22957  -0.233072  -0.236772  -0.240651   -0.24469  -0.248868  -0.253164  -0.257556  -0.262025  -0.266551  -0.271115    -0.2757  -0.280291  -0.284872  -0.289432  -0.293958  -0.298441  -0.302873  -0.307244   -0.31155  -0.315785  -0.319943  -0.324023  -0.328019   -0.33193  -0.335755  -0.339491  -0.343138  -0.346695  -0.350161     -0.353537 | 0.665771  0.665791  0.665816  0.665847  0.665887  0.665935  0.665995  0.666067  0.666154  0.666258  0.666379   0.66652  0.666683  0.666868  0.667077  0.667311  0.667571  0.667855  0.668166  0.668501  0.668861  0.669243  0.669648  0.670074  0.670518   0.67098  0.671458   0.67195  0.672454  0.672968  0.673492  0.674023  0.674559    0.6751  0.675644   0.67619  0.676736  0.677281  0.677825  0.678367  0.678905  0.679439  0.679969  0.680493  0.681011  0.681522  0.682026  0.682523  0.683012      0.683492 | 100.277186 | 100.323731  100.429356  100.565154  100.738239  100.956902  101.230652  101.570219  101.987504  102.495478  103.108023  103.839731  104.705647  105.720991  106.900853  108.259892  109.812043  111.570258  113.546281  115.750474  118.191697   120.87724    123.8128   127.00251  130.448997  134.153471  138.115831  142.334782  146.807957  151.532034  156.502852  161.715519  167.164498    172.8437  178.746549   184.86605  191.194836  197.725209   204.44918   211.35849  218.444636  225.698882  233.112274  240.675646  248.379627  256.214642  264.170914  272.238462  280.407098  288.666429          297.005849 | 301.190638 | 0.721315  0.721516  0.721774  0.722102  0.722515  0.723032  0.723672  0.724456  0.725407  0.726549  0.727906  0.729503  0.731362  0.733505  0.735952  0.738717  0.741813  0.745247  0.749025  0.753144  0.757601  0.762386  0.767487  0.772889  0.778575  0.784523  0.790715  0.797127  0.803738  0.810525  0.817467  0.824542   0.83173  0.839012  0.846369  0.853783   0.86124  0.868722  0.876217  0.883712  0.891194  0.898653  0.906078   0.91346   0.92079   0.92806  0.935264  0.942393  0.949442       0.956406 | 2.594677  2.595837  2.597328  2.599225   2.60162  2.604613  2.608318   2.61286  2.618373  2.624998  2.632878  2.642157  2.652974  2.665459  2.679731  2.695888  2.714013  2.734163  2.756373  2.780655  2.806993  2.835353  2.865678  2.897894  2.931913  2.967632  3.004942  3.043725  3.083862   3.12523  3.167706  3.211171  3.255507  3.300602  3.346346  3.392636  3.439374  3.486467  3.533828  3.581373  3.629027  3.676715  3.724371  3.771931  3.819335  3.866526  3.913451  3.960062   4.00631        4.052151 | 100.277186  100.369238  100.488165  100.640507  100.833948  101.077382  101.380932  101.755933  102.214854  102.771173  103.439187   104.23379  105.170195  106.263642  107.529087  108.980905  110.632609  112.496602  114.583979  116.904379  119.465889  122.275002  125.336623  128.654112  132.229361  136.062893  140.153977   144.50075  149.100336  153.948963  159.042076  164.374437  169.940212   175.73305   181.74615  187.972317  194.404009  201.033374  207.852282  214.852345   222.02494   229.36122  236.852123   244.48838  252.260516  260.158856   268.17352  276.294425  284.511282  292.813591  301.190638 |  100.0 |   0.277186 | 301.190638 |
    | 09:00 | 3.722536   3.72409  3.726094  3.728656  3.731903   3.73598  3.741051  3.747299  3.754924  3.764135  3.775156   3.78821  3.803522  3.821307  3.841767  3.865083   3.89141  3.920874  3.953565  3.989539  4.028815  4.071377  4.117175  4.166127  4.218126  4.273041  4.330722  4.391005  4.453714  4.518669  4.585683   4.65457  4.725147  4.797232  4.870649  4.945228  5.020805  5.097225  5.174339  5.252006  5.330093  5.408472  5.487024  5.565637  5.644204  5.722625  5.800803  5.878649  5.956078             6.033007 | 125.124403   125.20557  125.310284  125.444222   125.61404   125.82742  126.093082  126.420756  126.821106  127.305603  127.886345  128.575838  129.386737  130.331558  131.422386  132.670582  134.086522  135.679359  137.456841   139.42518  141.588978  143.951209  146.513254  149.274979  152.234842  155.390033  158.736618  162.269702  165.983576  169.871866  173.927671  178.143679   182.51228   187.02565  191.675833  196.454802  201.354506  206.366912  211.484032  216.697948  222.000818  227.384894   232.84252  238.366134  243.948266  249.581532  255.258631  260.972334  266.715476  272.480951 |  52.225358   52.240898   52.260938   52.286561   52.319029   52.359798   52.410511   52.472994   52.549235   52.641353   52.751558   52.882102   53.035222   53.213073   53.417672   53.650831   53.914102   54.208737   54.535647   54.895388    55.28815    55.71377   56.171746   56.661271   57.181263   57.730414   58.307224    58.91005   59.537144   60.186687   60.856827   61.545703   62.251471   62.972319   63.706488   64.452277   65.208053   65.972254   66.743394   67.520063   68.300925   69.084715    69.87024    70.65637   71.442042   72.226246   73.008031   73.786494   74.560777     75.330067 | 1.089115  1.089356  1.089667  1.090064  1.090567  1.091198  1.091983   1.09295  1.094128  1.095551  1.097252  1.099265  1.101622  1.104356  1.107496  1.111067   1.11509  1.119581   1.12455  1.130001  1.135934   1.14234  1.149207  1.156518  1.164252  1.172384  1.180888  1.189733  1.198891   1.20833  1.218021  1.227933  1.238037  1.248304  1.258709  1.269224  1.279827  1.290493  1.301204  1.311937  1.322676  1.333403  1.344103   1.35476  1.365363  1.375897  1.386353  1.396718  1.406983   1.41714 | 1.359112  1.359096  1.359076   1.35905  1.359016  1.358975  1.358923   1.35886  1.358783   1.35869   1.35858  1.358451    1.3583  1.358127   1.35793  1.357708  1.357461  1.357189  1.356892  1.356572   1.35623  1.355867  1.355485  1.355088  1.354678  1.354257  1.353827  1.353391  1.352953  1.352513  1.352074  1.351638  1.351206  1.350781  1.350362  1.349952   1.34955  1.349159  1.348777  1.348407  1.348047  1.347698   1.34736  1.347034  1.346718  1.346414   1.34612  1.345837  1.345564          1.345302 | 0.536529  0.536652  0.536809  0.537011  0.537266  0.537585  0.537982   0.53847  0.539064   0.53978  0.540632  0.541637   0.54281  0.544164   0.54571  0.547457  0.549412  0.551578  0.553954  0.556535  0.559316  0.562285  0.565431  0.568737  0.572189  0.575767  0.579455  0.583233  0.587084   0.59099  0.594935  0.598904  0.602881  0.606855  0.610812  0.614744  0.618639  0.622491  0.626292  0.630037   0.63372  0.637337  0.640884  0.644359   0.64776  0.651084  0.654331  0.657499  0.660588      0.663599 | -0.202359  -0.202518  -0.202723  -0.202984  -0.203313  -0.203723   -0.20423   -0.20485  -0.205601  -0.206499  -0.207565  -0.208814  -0.210262  -0.211925  -0.213812  -0.215932  -0.218291  -0.220889  -0.223722  -0.226786  -0.230068  -0.233557  -0.237236  -0.241088  -0.245094  -0.249232  -0.253484  -0.257829  -0.262246  -0.266718  -0.271226  -0.275755  -0.280288  -0.284812  -0.289315  -0.293787  -0.298216  -0.302595  -0.306916  -0.311174  -0.315362  -0.319477  -0.323515  -0.327473  -0.331347  -0.335137  -0.338842  -0.342459  -0.345988     -0.349429 | 0.665761  0.665777  0.665798  0.665825  0.665858    0.6659  0.665951  0.666014  0.666089  0.666179  0.666286  0.666411  0.666555  0.666721  0.666909  0.667121  0.667357  0.667617  0.667903  0.668213  0.668548  0.668906  0.669286  0.669688   0.67011   0.67055  0.671007   0.67148  0.671966  0.672464  0.672972  0.673488  0.674012  0.674542  0.675075  0.675612   0.67615  0.676689  0.677228  0.677765    0.6783  0.678832   0.67936  0.679883  0.680401  0.680914   0.68142  0.681919  0.682411      0.682895 | 100.228681 | 100.267562  100.355973  100.470061  100.616036  100.801192  101.033965  101.323961  101.681935  102.119736  102.650184  103.286914  104.044159  104.936513  105.978646  107.185027  108.569636  110.145698  111.925444  113.919909  116.138787  118.590321   121.28125  124.216804  127.400729  130.835351  134.521663  138.459427  142.647285  147.082872  151.762928  156.683406  161.839567  167.226071   172.83705   178.66618  184.706734  190.951633  197.393482  204.024603   210.83706  217.822678  224.973057  232.279584  239.733438  247.325596  255.046836  262.887735  270.838668   278.88981          287.031127 | 291.119246 | 0.721209  0.721377  0.721593   0.72187  0.722221  0.722661  0.723208  0.723882  0.724704  0.725696  0.726882  0.728284  0.729927  0.731832  0.734019  0.736506  0.739307  0.742433  0.745892  0.749685  0.753811  0.758265  0.763038  0.768117  0.773488  0.779133  0.785033  0.791168  0.797516  0.804057  0.810768   0.81763  0.824621  0.831722  0.838914   0.84618  0.853502  0.860865  0.868255  0.875658  0.883062  0.890454  0.897824  0.905163   0.91246  0.919708  0.926899  0.934025  0.941081       0.948059 |  2.59406  2.595031  2.596284  2.597886  2.599915  2.602463  2.605632  2.609535  2.614297  2.620049  2.626928  2.635073  2.644622  2.655708  2.668453  2.682967  2.699342  2.717652  2.737947  2.760258  2.784588  2.810922  2.839221  2.869429  2.901471   2.93526  2.970698  3.007677  3.046084  3.085803  3.126716  3.168706  3.211657  3.255456  3.299994  3.345167  3.390876  3.437024  3.483523  3.530289   3.57724  3.624304  3.671408  3.718488  3.765481  3.812329  3.858977  3.905374  3.951471        3.997221 | 100.228681  100.305569  100.405268  100.533462  100.696881  100.903377  101.161969  101.482845  101.877332  102.357801  102.937529  103.630515  104.451242   105.41442    106.5347  107.826382   109.30314  110.977764  112.861935  114.966056  117.299117  119.868619  122.680544  125.739368  129.048106  132.608393  136.420576  140.483828  144.796255  149.355015  154.156427  159.196072  164.468888  169.969251  175.691049   181.62774  187.772412  194.117816  200.656408  207.380376  214.281658  221.351964  228.582785  235.965403  243.490893  251.150134  258.933802  266.832374  274.836126  282.935129  291.119246 |  100.0 |   0.228681 | 291.119246 |
    | 09:30 | 3.721715  3.723014  3.724695  3.726854    3.7296  3.733063   3.73739  3.742746  3.749314  3.757289   3.76688  3.778301  3.791772  3.807505  3.825706  3.846566  3.870253  3.896911  3.926654  3.959561   3.99568  4.035023  4.077568  4.123261   4.17202  4.223738  4.278286  4.335517  4.395272  4.457382   4.52167  4.587959  4.656069  4.725823  4.797045  4.869568  4.943226  5.017862  5.093327  5.169475  5.246172  5.323287  5.400698   5.47829  5.555952  5.633582  5.711082  5.788357  5.865322             5.941891 | 125.081522  125.149366  125.237213  125.350008  125.493588  125.674751  125.901281   126.18195  126.526476  126.945429    127.4501  128.052319  128.764229  129.598037  130.565739  131.678843  132.948098  134.383253  135.992845   137.78404  139.762517  141.932415  144.296322  146.855322  149.609067  152.555889  155.692932  159.016291   162.52116  166.201977   170.05256  174.066234  178.235941  182.554343  187.013906  191.606968  196.325799  201.162648  206.109776  211.159489   216.30415  221.536199  226.848156  232.232629  237.682309  243.189976  248.748484  254.350764  259.989813  265.658686 |  52.217147   52.230138   52.246955   52.268539   52.296001   52.330632   52.373902   52.427464   52.493138   52.572888   52.668795   52.783012   52.917715   53.075048   53.257063   53.465661   53.702532   53.969112   54.266536   54.595612   54.956804   55.350233    55.77568    56.23261   56.720202   57.237381   57.782858    58.35517    58.95272   59.573816   60.216702   60.879592   61.560692   62.258227   62.970454   63.695678   64.432259   65.178623   65.933266   66.694751   67.461715   68.232866   69.006979   69.782896   70.559522   71.335822   72.110815   72.883573   73.653216     74.418907 | 1.088988  1.089189   1.08945  1.089784   1.09021  1.090746  1.091416  1.092245  1.093261  1.094494  1.095975  1.097737  1.099813  1.102235  1.105032  1.108232  1.111858  1.115929   1.12046   1.12546   1.13093   1.13687   1.14327  1.150118  1.157396  1.165085  1.173159  1.181593  1.190358  1.199425  1.208765  1.218349  1.228148  1.238133  1.248278  1.258556  1.268943  1.279415  1.289951   1.30053  1.311133  1.321742  1.332341  1.342915   1.35345  1.363932   1.37435  1.384691  1.394946  1.405106 | 1.359121  1.359107   1.35909  1.359068   1.35904  1.359005   1.35896  1.358906   1.35884  1.358759  1.358663  1.358549  1.358416  1.358261  1.358085  1.357884  1.357659   1.35741  1.357136  1.356838  1.356518  1.356176  1.355815  1.355436  1.355041  1.354634  1.354217  1.353792  1.353361  1.352928  1.352493  1.352059  1.351629  1.351202  1.350782  1.350368  1.349963  1.349566  1.349178  1.348801  1.348434  1.348078  1.347732  1.347397  1.347073   1.34676  1.346458  1.346166  1.345885          1.345614 | 0.536465  0.536567  0.536699  0.536869  0.537085  0.537357  0.537696  0.538115  0.538628  0.539248  0.539992  0.540875  0.541911  0.543114  0.544497  0.546071  0.547843  0.549819     0.552  0.554386  0.556973  0.559752  0.562714  0.565845  0.569132  0.572558  0.576106  0.579758  0.583498  0.587307  0.591169  0.595068  0.598989  0.602919  0.606845  0.610755  0.614639  0.618489  0.622297  0.626055  0.629759  0.633402  0.636981  0.640493  0.643935  0.647304  0.650599  0.653818  0.656961      0.660027 | -0.202274  -0.202408   -0.20258    -0.2028  -0.203079  -0.203428  -0.203862  -0.204396  -0.205046  -0.205828  -0.206761  -0.207861  -0.209144  -0.210627  -0.212321  -0.214237  -0.216383  -0.218761  -0.221373  -0.224214  -0.227277  -0.230553  -0.234028  -0.237686   -0.24151  -0.245482  -0.249582   -0.25379  -0.258087  -0.262454  -0.266873  -0.271327  -0.275801  -0.280278  -0.284747  -0.289196  -0.293613   -0.29799  -0.302318   -0.30659  -0.310801  -0.314945  -0.319017  -0.323014  -0.326933  -0.330772  -0.334529  -0.338202   -0.34179     -0.345292 | 0.665752  0.665766  0.665783  0.665806  0.665834   0.66587  0.665914  0.665968  0.666033  0.666112  0.666205  0.666315  0.666443   0.66659  0.666759   0.66695  0.667163  0.667401  0.667663  0.667949  0.668259  0.668593  0.668949  0.669328  0.669727  0.670145  0.670581  0.671033    0.6715   0.67198  0.672472  0.672974  0.673484  0.674001  0.674523   0.67505   0.67558  0.676111  0.676644  0.677175  0.677706  0.678234   0.67876  0.679281  0.679799  0.680311  0.680818  0.681319  0.681813        0.6823 | 100.188431 | 100.220863  100.294751  100.390446  100.513349  100.669852  100.867406   101.11457  101.421015  101.797498  102.255784   102.80852   103.46907  104.251303  105.169351  106.237345  107.469147  108.878086  110.476705  112.276556  114.288011  116.520136  118.980604  121.675649  124.610069  127.787256  131.209265  134.876896  138.789789  142.946532   147.34477  151.981306  156.852202  161.952868  167.278149   172.82239  178.579503  184.543018  190.706126  197.061719  203.602413  210.320578  217.208351   224.25765  231.460185  238.807463  246.290792  253.901286  261.629858  269.467226          277.403908 | 281.391872 |  0.72112   0.72126  0.721442  0.721675  0.721972  0.722346  0.722813  0.723391  0.724099  0.724959  0.725991   0.72722  0.728666  0.730354  0.732302  0.734531  0.737057  0.739892  0.743046  0.746525  0.750331  0.754462  0.758911   0.76367  0.768727  0.774066  0.779671  0.785522  0.791601  0.797886  0.804358  0.810996  0.817779  0.824688  0.831704  0.838808  0.845985  0.853218  0.860491   0.86779  0.875104  0.882418  0.889722  0.897006   0.90426  0.911476  0.918643  0.925756  0.932808       0.939791 | 2.593546  2.594359   2.59541  2.596759  2.598476   2.60064  2.603344  2.606691  2.610793  2.615774  2.621762  2.628891  2.637295  2.647105  2.658449  2.671441  2.686184  2.702762  2.721242  2.741668  2.764064  2.788432   2.81475  2.842979  2.873063  2.904926  2.938485  2.973641  3.010292  3.048328  3.087637  3.128105  3.169619  3.212067  3.255342  3.299339  3.343956  3.389097  3.434671  3.480592  3.526778  3.573151  3.619639  3.666173  3.712691  3.759131  3.805437  3.851555  3.897434        3.943028 | 100.188431  100.252558  100.336007  100.443703  100.581519  100.756361  100.976224  101.250223  101.588587  102.002603  102.504516  103.107384  103.824886   104.67109  105.660204  106.806298   108.12304  109.623431  111.319572  113.222471  115.341879  117.686182  120.262341  123.075868  126.130844  129.429975  132.974661  136.765095  140.800363  145.078551  149.596856  154.351683  159.338746  164.553152  169.989475  175.641829  181.503922  187.569104  193.830406  200.280575    206.9121  213.717228  220.687985  227.816181  235.093425  242.511122  250.060483  257.732522  265.518055    273.4077  281.391872 |  100.0 |   0.188431 | 281.391872 |
    | 10:00 | 3.721033  3.722117  3.723526  3.725341   3.72766  3.730595   3.73428  3.738861  3.744504  3.751391  3.759715   3.76968  3.781495   3.79537   3.81151  3.830111  3.851351  3.875388  3.902354  3.932353  3.965455  4.001698  4.041088  4.083599  4.129174   4.17773   4.22916  4.283338   4.34012  4.399353  4.460871  4.524505  4.590084  4.657433  4.726382  4.796763  4.868413  4.941174  5.014893  5.089426  5.164633  5.240384  5.316553   5.39302  5.469675   5.54641  5.623126  5.699725  5.776119             5.852221 | 125.045916  125.102538  125.176118  125.270945  125.392125  125.545644  125.738417  125.978312  126.274127   126.63554  127.073008  127.597624   128.22093  128.954703  129.810703  130.800415  131.934789  133.223988  134.677168  136.302287  138.105967  140.093397  142.268288  144.632882   147.18799  149.933079  152.866372   155.98498  159.285031  162.761815  166.409911  170.223321  174.195587  178.319894  182.589165  186.996136  191.533425  196.193586  200.969149  205.852658  210.836693  215.913889   221.07695  226.318654  231.631857  237.009497  242.444587  247.930211  253.459524  259.025737 |  52.210328   52.221171    52.23526    52.25341   52.276596   52.305954   52.342797   52.388608   52.445043   52.513911    52.59715   52.696798   52.814947   52.953697   53.115102    53.30111    53.51351   53.753881   54.023544   54.323528   54.654545   55.016979   55.410881   55.835989   56.291738   56.777298     57.2916   57.833378   58.401204   58.993528   59.608709   60.245053   60.900835   61.574328    62.26382   62.967632   63.684131   64.411736   65.148929   65.894257   66.646334   67.403841   68.165526   68.930204   69.696751   70.464104   71.231255   71.997252   72.761188     73.522206 | 1.088882   1.08905  1.089269   1.08955  1.089909  1.090364  1.090935  1.091644  1.092517  1.093582  1.094869  1.096407   1.09823  1.100368  1.102851  1.105708  1.108965  1.112643  1.116759  1.121327  1.126353   1.13184  1.137783  1.144175  1.151002  1.158247  1.165889  1.173905  1.182269  1.190955  1.199933  1.209176  1.218656  1.228344  1.238213  1.248238  1.258393  1.268654  1.278999  1.289407  1.299858  1.310334  1.320818  1.331292  1.341744  1.352158  1.362523  1.372826  1.383056  1.393204 | 1.359128  1.359116  1.359102  1.359083   1.35906   1.35903  1.358992  1.358945  1.358888  1.358819  1.358735  1.358635  1.358517   1.35838  1.358222  1.358042  1.357839  1.357611   1.35736  1.357084  1.356786  1.356465  1.356124  1.355764  1.355387  1.354996  1.354592  1.354179  1.353758  1.353332  1.352904  1.352474  1.352046   1.35162  1.351199  1.350783  1.350375  1.349974  1.349581  1.349198  1.348825  1.348461  1.348108  1.347766  1.347434  1.347112  1.346802  1.346502  1.346212          1.345932 | 0.536411  0.536496  0.536607   0.53675  0.536933  0.537163  0.537452  0.537811  0.538252  0.538789  0.539437  0.540209  0.541121  0.542187  0.543419   0.54483   0.54643  0.548225   0.55022  0.552415  0.554811    0.5574  0.560177   0.56313  0.566247  0.569513  0.572913  0.576431  0.580049   0.58375  0.587518  0.591337  0.595192  0.599067   0.60295  0.606829  0.610693  0.614532  0.618338  0.622102  0.625819  0.629482  0.633087   0.63663  0.640107  0.643516  0.646854   0.65012  0.653313       0.65643 | -0.202203  -0.202315   -0.20246  -0.202645  -0.202881  -0.203178  -0.203549  -0.204007  -0.204568  -0.205247   -0.20606  -0.207026  -0.208159  -0.209477  -0.210992  -0.212716  -0.214659  -0.216828  -0.219224  -0.221847  -0.224694  -0.227757  -0.231024  -0.234484  -0.238121  -0.241917  -0.245855  -0.249917  -0.254082  -0.258333  -0.262651  -0.267018   -0.27142  -0.275839  -0.280262  -0.284678  -0.289073  -0.293438  -0.297764  -0.302042  -0.306267  -0.310432  -0.314532  -0.318562   -0.32252  -0.326402  -0.330205  -0.333929  -0.337571      -0.34113 | 0.665745  0.665756  0.665771   0.66579  0.665814  0.665844  0.665882  0.665928  0.665985  0.666053  0.666134  0.666231  0.666344  0.666475  0.666625  0.666796   0.66699  0.667206  0.667445  0.667708  0.667994  0.668304  0.668637  0.668992  0.669368  0.669763  0.670178   0.67061  0.671058  0.671519  0.671994   0.67248  0.672975  0.673479  0.673989  0.674505  0.675025  0.675548  0.676073  0.676598  0.677123  0.677647  0.678169  0.678689  0.679204  0.679716  0.680223  0.680724   0.68122      0.681709 | 100.155078 |  100.18209  100.243749   100.32389  100.427197  100.559251  100.726608  100.936857  101.198653  101.521708  101.916752  102.395443  102.970234  103.654204  104.460843  105.403825   106.49675   107.75289  109.184946  110.804817  112.623408  114.650472   116.89449  119.362604  122.060582  124.992825  128.162405   131.57113  135.219625  139.107423  143.233071  147.594225  152.187751   157.00982  162.055984  167.321265  172.800208  178.486951  184.375265  190.458601  196.730117  203.182714  209.809048  216.601552  223.552446  230.653747  237.897275  245.274652  252.777311  260.396487          268.123224 | 272.008178 | 0.721046  0.721163  0.721316  0.721512  0.721762   0.72208  0.722477  0.722972  0.723581  0.724323   0.72522  0.726293  0.727563  0.729053  0.730783  0.732773  0.735042  0.737603   0.74047  0.743649  0.747147  0.750964  0.755097   0.75954  0.764285  0.769318  0.774624  0.780188  0.785991  0.792015  0.798239  0.804643  0.811208  0.817914  0.824743  0.831676  0.838696  0.845786  0.852931  0.860116  0.867327  0.874553  0.881781     0.889    0.8962  0.903371  0.910506  0.917595  0.924632        0.93161 |  2.59312  2.593798  2.594679  2.595813  2.597263  2.599098    2.6014  2.604263  2.607789  2.612091  2.617289   2.62351  2.630883  2.639539  2.649602  2.661193   2.67442  2.689378  2.706145  2.724781  2.745324  2.767793  2.792185  2.818478   2.84663  2.876583  2.908264   2.94159  2.976467  3.012793  3.050463  3.089369    3.1294  3.170449  3.212408  3.255171  3.298639  3.342714  3.387302  3.432317  3.477674  3.523295  3.569104  3.615031   3.66101  3.706979  3.752879  3.798655  3.844254        3.889628 | 100.155078  100.208484  100.278224  100.368554  100.484583  100.632359   100.81894  101.052445   101.34206  101.698026  102.131564  102.654771  103.280465  104.021993  104.893004  105.907204  107.078101  108.418747  109.941501  111.657816  113.578058  115.711372  118.065585  120.647155  123.461161  126.511328  129.800074  133.328591  137.096933  141.104113  145.348203   149.82644  154.535318  159.470674  164.627779  170.001396  175.585855  181.375095  187.362719  193.542023  199.906029   206.44751  213.159007  220.032844  227.061137  234.235804  241.548563  248.990943  256.554278  264.229708  272.008178 |  100.0 |   0.155078 | 272.008178 |
    | 10:30 | 3.720467  3.721371  3.722549  3.724073  3.726027  3.728511  3.731642  3.735552   3.74039  3.746323  3.753528  3.762198  3.772532  3.784731     3.799  3.815533  3.834516  3.856117  3.880484  3.907737  3.937969  3.971244  4.007592  4.047011  4.089473  4.134918  4.183261  4.234399  4.288205  4.344541  4.403255  4.464191  4.527182  4.592063  4.658668  4.726831  4.796391   4.86719  4.939075    5.0119  5.085525  5.159816  5.234644   5.30989  5.385437  5.461178  5.537008   5.61283   5.68855             5.764082 | 125.016392  125.063577  125.125111  125.204704  125.306801  125.436656  125.600384  125.805005  126.058443  126.369501  126.747797  127.203652  127.747943  128.391918  129.146979  130.024444    131.0353  132.189957  133.498018  134.968078  136.607553  138.422557  140.417825   142.59668  144.961043  147.511484  150.247305  153.166638   156.26657   159.54327  162.992121  166.607843  170.384615  174.316187  178.395976  182.617149  186.972704  191.455522  196.058427  200.774216  205.595699  210.515719  215.527169  220.623004  225.796246  231.039993  236.347411  241.711741  247.126287  252.584415 |  52.204673    52.21371   52.225494   52.240732   52.260272   52.285114   52.316419   52.355516   52.403901   52.463225   52.535283   52.621983   52.725315   52.847312   52.989995   53.155328   53.345159   53.561172   53.804836   54.077366   54.379693    54.71244   55.075916   55.470114    55.89473   56.349176   56.832615   57.343986   57.882047   58.445405   59.032554   59.641905    60.27182   60.920635   61.586683   62.268314   62.963911   63.671897   64.390749   65.119001    65.85525   66.598156   67.346443   68.098899   68.854375    69.61178    70.37008   71.128298   71.885504     72.640817 | 1.088794  1.088934  1.089117  1.089353  1.089656  1.090041  1.090526  1.091132  1.091881  1.092798  1.093913  1.095252  1.096847  1.098729  1.100926  1.103469  1.106384  1.109695  1.113421   1.11758  1.122181  1.127231  1.132731  1.138675  1.145056  1.151861   1.15907  1.166666  1.174623  1.182919  1.191525  1.200416  1.209564  1.218941  1.228521  1.238277  1.248185   1.25822  1.268359   1.27858  1.288864   1.29919  1.309542  1.319903  1.330256  1.340588  1.350885  1.361135  1.371326  1.381447 | 1.359133  1.359124  1.359112  1.359096  1.359076  1.359051  1.359019  1.358979   1.35893   1.35887  1.358797   1.35871  1.358607  1.358485  1.358345  1.358183     1.358  1.357793  1.357563   1.35731  1.357033  1.356734  1.356414  1.356073  1.355715   1.35534  1.354952  1.354552  1.354142  1.353726  1.353305  1.352881  1.352456  1.352033  1.351612  1.351196  1.350786  1.350382  1.349985  1.349597  1.349218  1.348848  1.348488  1.348139  1.347799   1.34747  1.347151  1.346843  1.346545          1.346257 | 0.536366  0.536438   0.53653   0.53665  0.536804  0.536999  0.537245  0.537552  0.537931  0.538394  0.538956   0.53963  0.540429   0.54137  0.542464  0.543725  0.545163  0.546787  0.548603  0.550615  0.552824  0.555227  0.557818  0.560591  0.563534  0.566636  0.569882  0.573256  0.576744  0.580328  0.583992   0.58772  0.591496  0.595307  0.599137  0.602975  0.606809  0.610628  0.614423  0.618185  0.621907  0.625583  0.629207  0.632775  0.636282  0.639725  0.643102   0.64641  0.649648      0.652814 | -0.202144  -0.202238  -0.202359  -0.202515  -0.202714  -0.202967  -0.203283  -0.203675  -0.204157  -0.204745  -0.205453  -0.206297  -0.207295  -0.208461  -0.209811  -0.211356  -0.213109  -0.215078  -0.217267  -0.219679  -0.222313  -0.225164  -0.228224  -0.231482  -0.234926  -0.238541   -0.24231  -0.246215  -0.250238  -0.254361  -0.258566  -0.262836  -0.267154  -0.271503   -0.27587  -0.280241  -0.284604  -0.288947  -0.293261  -0.297537  -0.301768  -0.305946  -0.310066  -0.314123  -0.318112  -0.322031  -0.325877  -0.329646  -0.333337     -0.336949 | 0.665738  0.665748  0.665761  0.665777  0.665797  0.665823  0.665855  0.665895  0.665943  0.666002  0.666073  0.666157  0.666257  0.666373  0.666507   0.66666  0.666834  0.667029  0.667247  0.667488  0.667752  0.668038  0.668348  0.668679  0.669032  0.669406  0.669799   0.67021  0.670638  0.671081  0.671537  0.672007  0.672487  0.672976  0.673473  0.673977  0.674487     0.675  0.675516  0.676034  0.676553  0.677072   0.67759  0.678105  0.678619  0.679128  0.679634  0.680136  0.680632      0.681122 | 100.127477 | 100.149943  100.201322  100.268334   100.35503  100.466266  100.607788  100.786299  101.009507  101.286137   101.62592  102.039534  102.538507   103.13508  103.842031  104.672472  105.639616  106.756544  108.035956  109.489951  111.129814  112.965843  115.007204  117.261837   119.73639  122.436201  125.365307  128.526487  131.921325  135.550287  139.412816  143.507422  147.831779  152.382816  157.156807  162.149445  167.355916   172.77096  178.388925  184.203814   190.20932   196.39886  202.765599  209.302466  216.002176  222.857236  229.859955  237.002447  244.276637   251.67426           259.18686 | 262.966516 | 0.720985  0.721083   0.72121  0.721375  0.721586  0.721854  0.722193  0.722615  0.723137  0.723777  0.724554  0.725487  0.726599  0.727911  0.729442  0.731214  0.733244   0.73555  0.738145  0.741041  0.744243  0.747757  0.751583  0.755717  0.760153  0.764881   0.76989  0.775164  0.780687  0.786442  0.792411  0.798573  0.804911  0.811406  0.818037  0.824787  0.831639  0.838576  0.845582  0.852641  0.859741  0.866867  0.874007   0.88115  0.888285  0.895403  0.902494   0.90955  0.916563       0.923526 | 2.592766  2.593331  2.594068  2.595021  2.596242  2.597795  2.599752  2.602195  2.605219  2.608925  2.613426   2.61884   2.62529  2.632903  2.641802   2.65211  2.663937  2.677387  2.692547   2.70949  2.728268  2.748915  2.771445  2.795851  2.822108  2.850175  2.879992  2.911489  2.944582  2.979179  3.015184  3.052494  3.091004  3.130608  3.171202  3.212681  3.254945  3.297897  3.341443  3.385494  3.429963  3.474771   3.51984  3.565099  3.610479  3.655916  3.701349  3.746722   3.79198        3.837073 | 100.127477   100.17189  100.230087  100.305734  100.403259  100.527942   100.68599    100.8846  101.131985  101.437384  101.811021  102.264029  102.808337  103.456505  104.221539  105.116666  106.155099  107.349795  108.713216  110.257105  111.992301  113.928568  116.074484  118.437353  121.023172   123.83662  126.881092  130.158749  133.670594  137.416556  141.395584  145.605747   150.04432  154.707884  159.592399  164.693287  170.005496  175.523557  181.241633  187.153562  193.252893  199.532909  205.986652  212.606942  219.386387  226.317391  233.392167  240.602733  247.940918  255.398363  262.966516 |  100.0 |   0.127477 | 262.966516 |
    | 11:00 | 3.719999  3.720751  3.721735  3.723012  3.724656  3.726754  3.729409  3.732739  3.736878  3.741977  3.748199  3.755722  3.764735   3.77543  3.788006  3.802656  3.819567  3.838916   3.86086  3.885536  3.913055  3.943501  3.976929  4.013362  4.052795  4.095194  4.140497  4.188621   4.23946  4.292894  4.348785  4.406988  4.467348  4.529708  4.593906  4.659782  4.727177  4.795934  4.865902  4.936934  5.008887  5.081626  5.155023  5.228952  5.303298  5.377947  5.452796  5.527742  5.602691             5.677553 | 124.991944  125.031209  125.082591   125.14929  125.235166  125.344808  125.483607  125.657793  125.874467  126.141592  126.467954   126.86309  127.337166  127.900831   128.56503  129.340787  130.238986  131.270127    132.4441  133.769976  135.255818  136.908533  138.733763  140.735824  142.917681  145.280966  147.826031  150.552031  153.457023  156.538081  159.791418   163.21251  166.796215  170.536884  174.428465  178.464594  182.638676  186.943951  191.373556  195.920567  200.578036  205.339028  210.196632  215.143989  220.174291  225.280796  230.456827  235.695776  240.991096  246.336302 |  52.199989   52.207511   52.217352   52.230124   52.246563   52.267544   52.294093   52.327391   52.368782   52.419766   52.481988   52.557224   52.647349   52.754304   52.880059   53.026557   53.195674   53.389161     53.6086   53.855357   54.130549   54.435013   54.769291   55.133624   55.527954   55.951938    56.40497   56.886208   57.394603   57.928936   58.487848   59.069876   59.673481   60.297079   60.939062   61.597824   62.271771   62.959344   63.659022   64.369335   65.088868   65.816264   66.550227   67.289522   68.032976   68.779473   69.527955   70.277418    71.02691     71.775525 | 1.088722  1.088838  1.088991  1.089189  1.089444  1.089769   1.09018  1.090696  1.091337  1.092126  1.093089  1.094252  1.095644  1.097295  1.099233  1.101489  1.104089  1.107059  1.110421  1.114193   1.11839  1.123022  1.128092  1.133602  1.139546  1.145914  1.152694  1.159868  1.167416  1.175315  1.183542  1.192071  1.200876   1.20993  1.219207  1.228681  1.238327   1.24812  1.258038  1.268057  1.278158   1.28832  1.298526  1.308757  1.318997  1.329232  1.339447  1.349629  1.359767  1.369848 | 1.359138  1.359131   1.35912  1.359107   1.35909  1.359069  1.359042  1.359008  1.358966  1.358914  1.358851  1.358775  1.358684  1.358578  1.358453  1.358309  1.358144  1.357957  1.357748  1.357516  1.357261  1.356983  1.356684  1.356363  1.356024  1.355667  1.355295  1.354909  1.354513  1.354107  1.353695  1.353278  1.352859   1.35244  1.352021  1.351606  1.351194  1.350788  1.350389  1.349997  1.349613  1.349238  1.348872  1.348515  1.348169  1.347832  1.347506   1.34719  1.346884          1.346587 |  0.53633  0.536389  0.536466  0.536567  0.536696  0.536861   0.53707  0.537331  0.537656  0.538055  0.538541  0.539127  0.539826  0.540653  0.541622  0.542744  0.544032  0.545495  0.547142  0.548978  0.551006  0.553225  0.555634  0.558227  0.560995  0.563927  0.567013  0.570238  0.573587  0.577045  0.580595  0.584222  0.587911  0.591646  0.595414  0.599201  0.602994  0.606784  0.610559  0.614311  0.618031  0.621712  0.625348  0.628934  0.632465  0.635937  0.639348  0.642693  0.645972      0.649182 | -0.202095  -0.202173  -0.202275  -0.202406  -0.202574  -0.202788  -0.203056  -0.203392  -0.203806  -0.204313  -0.204927  -0.205663  -0.206538  -0.207567  -0.208765  -0.210145  -0.211721  -0.213501  -0.215493  -0.217701  -0.220128   -0.22277  -0.225623  -0.228679  -0.231927  -0.235355  -0.238947  -0.242688   -0.24656  -0.250546  -0.254627  -0.258788  -0.263011  -0.267279  -0.271578  -0.275894  -0.280214  -0.284526  -0.288818  -0.293083   -0.29731  -0.301494  -0.305627  -0.309703  -0.313718  -0.317668  -0.321549  -0.325359  -0.329094     -0.332754 | 0.665733  0.665741  0.665752  0.665765  0.665783  0.665804  0.665832  0.665866  0.665908  0.665959   0.66602  0.666094  0.666181  0.666283  0.666402  0.666539  0.666695  0.666871  0.667069  0.667288   0.66753  0.667794  0.668081   0.66839  0.668721  0.669072  0.669443  0.669833   0.67024  0.670664  0.671102  0.671554  0.672018  0.672492  0.672976  0.673467  0.673965  0.674468  0.674975  0.675485  0.675996  0.676509  0.677021  0.677533  0.678042   0.67855  0.679054  0.679554   0.68005      0.680541 | 100.104666 | 100.123325  100.166076  100.222027  100.294669  100.388215  100.507683   100.65897  100.848911  101.085306  101.376936  101.733524  102.165674  102.684765  103.302809  104.032278  104.885896  105.876424  107.016433  108.318071  109.792861  111.451505  113.303724  115.358141  117.622184  120.102048   122.80267  125.727754   128.87981  132.260218  135.869306  139.706433  143.770082  148.057948  152.567024  157.293682   162.23375  167.382572  172.735072    178.2858  184.028978  189.958529  196.068114  202.351148  208.800823  215.410119  222.171815  229.078499  236.122567  243.296229           250.59151 | 254.264101 | 0.720934  0.721016  0.721122   0.72126  0.721438  0.721665  0.721951  0.722311  0.722758  0.723308  0.723979   0.72479  0.725761  0.726911  0.728262  0.729834  0.731646  0.733714  0.736056  0.738683  0.741605  0.744828  0.748357  0.752189  0.756322  0.760749   0.76546  0.770443  0.775684  0.781167  0.786874  0.792789  0.798892  0.805165   0.81159  0.818148  0.824822  0.831595   0.83845  0.845374   0.85235  0.859366  0.866408  0.873465  0.880526   0.88758  0.894617   0.90163  0.908609       0.915547 | 2.592474  2.592944  2.593559  2.594358  2.595385  2.596697  2.598356  2.600438  2.603024   2.60621  2.610097  2.614796  2.620423  2.627099  2.634946  2.644082  2.654624  2.666678  2.680339  2.695689  2.712794  2.731702   2.75244  2.775019  2.799429  2.825644  2.853618  2.883295  2.914604  2.947463  2.981784  3.017471  3.054426  3.092547  3.131732   3.17188  3.212891  3.254668  3.297117  3.340147  3.383672   3.42761  3.471882  3.516414  3.561136  3.605983   3.65089  3.695801  3.740658         3.78541 | 100.104666  100.141549  100.190043  100.253296  100.335137  100.440159    100.5738  100.742413  100.953308  101.214779  101.536089  101.927424  102.399809  102.964984   103.63524  104.423236  105.341775  106.403588  107.621094  109.006186  110.570024  112.322862    114.2739  116.431184  118.801534  121.390517  124.202445  127.240412  130.506342  134.001068   137.72441  141.675263  145.851692  150.251019  154.869905   159.70443  164.750164  170.002226  175.455341  181.103882  186.941914  192.963223   199.16134  205.529564  212.060977  218.748459  225.584688  232.562157  239.673168  246.909839  254.264101 |  100.0 |   0.104666 | 254.264101 |
    | 11:30 | 3.719612  3.720237  3.721057  3.722126  3.723507  3.725276  3.727523  3.730353  3.733887  3.738258  3.743618  3.750131   3.75797  3.767321  3.778372  3.791314  3.806333  3.823609  3.843307  3.865575  3.890541  3.918307  3.948947   3.98251  4.019011  4.058442  4.100765  4.145916  4.193813  4.244351  4.297411   4.35286  4.410557  4.470351   4.53209  4.595619  4.660781  4.727425  4.795398  4.864555  4.934753  5.005856  5.077732  5.150255  5.223308  5.296775  5.370548  5.444525  5.518608             5.592705 | 124.971726  125.004354  125.047196  125.103004  125.175117  125.267535  125.384986  125.532979  125.717845  125.946748   126.22767  126.569361  126.981256  127.473355  128.056065   128.74002  129.535872  130.454071  131.504649  132.696996  134.039674  135.540245  137.205137  139.039557   141.04743  143.231393  145.592816  148.131856  150.847539  153.737857   156.79988  160.029869  163.423395  166.975448  170.680545  174.532825  178.526134  182.654101  186.910199  191.287805  195.780235  200.380786  205.082762   209.87949  214.764343  219.730742  224.772168  229.882162  235.054326   240.28232 |  52.196116   52.202367   52.210573   52.221261   52.235068   52.252758    52.27523   52.303533   52.338866   52.382583   52.436184   52.501307   52.579703    52.67321   52.783722    52.91314   53.063334   53.236091    53.43307   53.655754   53.905412   54.183068   54.489474   54.825096   55.190113   55.584421   56.007645   56.459163   56.938131   57.443513   57.974111   58.528603   59.105566   59.703509   60.320899   60.956185   61.607813   62.274248   62.953983    63.64555   64.347531   65.058556   65.777316   66.502555   67.233079   67.967748   68.705483   69.445253   70.186085      70.92705 | 1.088662  1.088759  1.088886  1.089051  1.089266   1.08954  1.089888  1.090326  1.090874  1.091551   1.09238  1.093387  1.094599  1.096043  1.097748  1.099743  1.102055   1.10471  1.107732  1.111142  1.114957   1.11919  1.123849  1.128937  1.134455  1.140395  1.146749  1.153503   1.16064   1.16814  1.175982  1.184141  1.192593  1.201313  1.210275  1.219454  1.228824  1.238362  1.248044  1.257847  1.267751  1.277734  1.287778  1.297864  1.307977    1.3181  1.328219   1.33832  1.348391  1.358419 | 1.359142  1.359136  1.359127  1.359116  1.359102  1.359084  1.359061  1.359032  1.358996  1.358952  1.358897  1.358831  1.358752  1.358659  1.358548   1.35842  1.358273  1.358105  1.357915  1.357704  1.357469  1.357213  1.356934  1.356634  1.356314  1.355976  1.355621  1.355251  1.354868  1.354475  1.354073  1.353665  1.353253  1.352839  1.352424   1.35201  1.351599  1.351193  1.350791  1.350396  1.350009  1.349629  1.349257  1.348895  1.348542  1.348199  1.347865  1.347541  1.347228          1.346924 | 0.536299  0.536348  0.536413  0.536497  0.536606  0.536745  0.536922  0.537144  0.537421  0.537764  0.538183  0.538691  0.539301  0.540026   0.54088  0.541876  0.543025  0.544338  0.545826  0.547494  0.549348   0.55139   0.55362  0.556034  0.558626  0.561388   0.56431  0.567379  0.570583  0.573906  0.577334  0.580851  0.584442  0.588092  0.591787  0.595513  0.599257  0.603008  0.606755  0.610487  0.614197  0.617876  0.621517  0.625114  0.628663  0.632158  0.635597  0.638974  0.642289      0.645539 | -0.202055   -0.20212  -0.202205  -0.202315  -0.202456  -0.202637  -0.202865   -0.20315  -0.203505  -0.203941  -0.204472  -0.205113  -0.205878  -0.206783  -0.207842  -0.209071  -0.210481  -0.212084  -0.213889  -0.215903  -0.218129  -0.220568  -0.223217  -0.226072  -0.229123   -0.23236  -0.235771   -0.23934  -0.243053  -0.246892  -0.250841  -0.254882  -0.258999  -0.263175  -0.267396  -0.271646  -0.275912  -0.280182  -0.284444  -0.288687  -0.292903  -0.297084  -0.301222   -0.30531  -0.309344  -0.313318  -0.317229  -0.321073  -0.324848      -0.32855 | 0.665729  0.665736  0.665745  0.665756  0.665771  0.665789  0.665812  0.665841  0.665877  0.665921  0.665974  0.666038  0.666115  0.666205   0.66631  0.666431  0.666571  0.666729  0.666908  0.667107  0.667328  0.667571  0.667836  0.668123  0.668431  0.668761   0.66911  0.669479  0.669866   0.67027  0.670689  0.671123   0.67157  0.672029  0.672497  0.672975   0.67346  0.673952  0.674449  0.674949  0.675453  0.675958  0.676465  0.676971  0.677476   0.67798  0.678482   0.67898  0.679475      0.679966 | 100.085838 | 100.101314  100.136836  100.183484  100.244257    100.3228  100.423481  100.551469  100.712798  100.914415  101.164198  101.470956  101.844386  102.294999  102.834007  103.473183  104.224679  105.100835  106.113966  107.276143  108.598985  110.093461  111.769719  113.636942  115.703235  117.975556  120.459672  123.160155  126.080401  129.222675  132.588171   136.17709  139.988722  144.021528  148.273225   152.74087  157.420933  162.309368  167.401673  172.692946  178.177929   183.85105  189.706453  195.738029  201.939433  208.304104  214.825275  221.495985  228.309084  235.257239          242.332932 | 245.897158 | 0.720893   0.72096  0.721049  0.721164  0.721314  0.721505  0.721748  0.722053  0.722435  0.722907  0.723485  0.724187  0.725032  0.726039  0.727227  0.728617  0.730228  0.732078  0.734183  0.736558  0.739215  0.742161  0.745404  0.748945  0.752782  0.756913   0.76133  0.766023   0.77098  0.776187  0.781629   0.78729  0.793151  0.799195  0.805404   0.81176  0.818247  0.824846  0.831542  0.838319  0.845162  0.852057  0.858991  0.865952  0.872927  0.879908  0.886882  0.893841  0.900777       0.907681 | 2.592231  2.592622  2.593135  2.593804  2.594667  2.595773  2.597177  2.598946  2.601155  2.603887  2.607236  2.611304    2.6162  2.622038  2.628935  2.637009  2.646375  2.657142  2.669412  2.683273  2.698802  2.716057  2.735082    2.7559  2.778518  2.802923  2.829086  2.856962  2.886495  2.917614  2.950239  2.984285  3.019658  3.056263  3.094002  3.132776  3.172488  3.213042  3.254342    3.2963  3.338827   3.38184  3.425259  3.469008  3.513016  3.557215   3.60154  3.645931  3.690331        3.734684 | 100.085838  100.116426  100.156776  100.209588  100.278161  100.366477  100.479285  100.622171  100.801615  101.025025  101.300751  101.638061  102.047082  102.538711  103.124485  103.816416  104.626808  105.568048  106.652388   107.89173   109.29742  110.880062  112.649356  114.613972  116.781459  119.158186  121.749321   124.55884  127.589558  130.843186  134.320401  138.020926  141.943612  146.086526  150.447033  155.021877  159.807252  164.798868  169.992009  175.381584  180.962167  186.728037  192.673204  198.791435  205.076275  211.521054  218.118908   224.86278  231.745427  238.759422  245.897158 |  100.0 |   0.085838 | 245.897158 |
    | 12:00 | 3.719292   3.71981  3.720494  3.721387  3.722545  3.724033  3.725932  3.728333  3.731343  3.735084  3.739691  3.745314  3.752116  3.760269  3.769953  3.781353  3.794651  3.810028  3.827653  3.847684   3.87026  3.895497   3.92349  3.954306  3.987986   4.02454  4.063954  4.106189   4.15118  4.198844  4.249077  4.301764  4.356774  4.413969  4.473206  4.534335  4.597207  4.661671   4.72758  4.794787  4.863152  4.932537  5.002809  5.073842  5.145515  5.217711  5.290321  5.363239  5.436365             5.509605 | 124.955028  124.982104  125.017773  125.064398  125.124859  125.202628  125.301839  125.427344  125.584762  125.780505  126.021784  126.316581  126.673592  127.102135  127.612027  128.213427  128.916655  129.731988  130.669459  131.738639  132.948441  134.306943  135.821234  137.497296  139.339925   141.35269  143.537926  145.896763  148.429183    151.1341  154.009454  157.052316  160.258999  163.625167   167.14594  170.815995  174.629653   178.58096  182.663759  186.871747  191.198526  195.637644  200.182628   204.82701  209.564343  214.388222  219.292286   224.27023  229.315801  234.422803 |  52.192917   52.198104   52.204937   52.213868   52.225446   52.240335   52.259322   52.283333   52.313433   52.350837   52.396905   52.453137   52.521156   52.602687   52.699531   52.813526   52.946512    53.10028   53.276535   53.476843   53.702596   53.954971   54.234903   54.543065   54.879855   55.245396   55.639541   56.061888   56.511801   56.988437   57.490774   58.017638   58.567738   59.139693   59.732058    60.34335   60.972068   61.616712   62.275799   62.947875   63.631522   64.325367   65.028091   65.738422   66.455149   67.177112   67.903209   68.632387    69.36365     70.096049 | 1.088612  1.088692  1.088798  1.088937  1.089116  1.089347  1.089641  1.090014   1.09048  1.091059  1.091772  1.092642  1.093694  1.094954   1.09645  1.098208  1.100257  1.102623  1.105331  1.108403  1.111859  1.115713  1.119979  1.124662  1.129766  1.135289  1.141224  1.147562  1.154289  1.161388   1.16884  1.176623  1.184715  1.193092  1.201728    1.2106  1.219683  1.228952  1.238384  1.247957  1.257649  1.267439  1.277307  1.287235  1.297206  1.307204  1.317213  1.327219  1.337208  1.347169 | 1.359146   1.35914  1.359133  1.359124  1.359112  1.359097  1.359077  1.359053  1.359022  1.358984  1.358937   1.35888  1.358811  1.358729  1.358632  1.358519  1.358388  1.358237  1.358066  1.357873  1.357659  1.357423  1.357165  1.356886  1.356586  1.356266  1.355929  1.355576  1.355209  1.354829  1.354439  1.354041  1.353637  1.353229  1.352819  1.352409     1.352  1.351594  1.351192  1.350795  1.350404  1.350021  1.349645  1.349277  1.348918  1.348569  1.348228  1.347898  1.347577          1.347265 | 0.536274  0.536315  0.536369  0.536439   0.53653  0.536647  0.536797  0.536985  0.537222  0.537515  0.537876  0.538316  0.538846   0.53948   0.54023   0.54111  0.542131  0.543306  0.544645  0.546155  0.547843  0.549714  0.551769  0.554007  0.556425  0.559015  0.561771  0.564681  0.567734  0.570916  0.574214  0.577612  0.581097  0.584652  0.588265   0.59192  0.595605  0.599308  0.603017  0.606721  0.610412  0.614081  0.617719  0.621322  0.624881  0.628394  0.631854  0.635259  0.638605       0.64189 | -0.202022  -0.202076  -0.202146  -0.202238  -0.202357  -0.202509  -0.202703  -0.202946  -0.203249  -0.203623  -0.204081  -0.204637  -0.205304  -0.206097  -0.207031   -0.20812  -0.209378  -0.210816  -0.212446  -0.214275  -0.216309  -0.218551  -0.221001  -0.223656   -0.22651  -0.229555  -0.232781  -0.236174   -0.23972  -0.243405  -0.247211  -0.251123  -0.255125  -0.259199   -0.26333  -0.267504  -0.271706  -0.275924  -0.280145  -0.284358  -0.288554  -0.292723  -0.296857   -0.30095  -0.304996  -0.308988  -0.312922  -0.316795  -0.320603     -0.324343 | 0.665726  0.665731  0.665739  0.665748   0.66576  0.665776  0.665796   0.66582  0.665851  0.665889  0.665935   0.66599  0.666057  0.666136  0.666228  0.666336  0.666461  0.666603  0.666764  0.666944  0.667146  0.667368  0.667612  0.667877  0.668164  0.668471  0.668799  0.669147  0.669513  0.669897  0.670297  0.670713  0.671143  0.671585  0.672038  0.672502  0.672974  0.673453  0.673939   0.67443  0.674924  0.675422  0.675921  0.676421  0.676921  0.677421  0.677919  0.678415  0.678908      0.679398 | 100.070317 | 100.083136  100.112612  100.151447  100.202216  100.268059  100.352769  100.460859  100.597638  100.769263  100.982775  101.246112  101.568093  101.958367  102.427334  102.986027  103.645964  104.418977  105.317023  106.351974   107.53542  108.878467  110.391557  112.084312  113.965404  116.042458  118.321988  120.809369  123.508831  126.423487  129.555378  132.905531  136.474033  140.260111  144.262211  148.478075  152.904823  157.539021  162.376743  167.413635  172.644958  178.065639  183.670302  189.453301  195.408745  201.530517  207.812289  214.247538  220.829549  227.551423          234.406082 | 237.861066 | 0.720858  0.720914  0.720988  0.721084   0.72121  0.721371  0.721576  0.721835   0.72216  0.722564  0.723062  0.723668  0.724401   0.72528  0.726322  0.727548  0.728975  0.730624  0.732511  0.734651  0.737057  0.739741  0.742711   0.74597  0.749521  0.753362  0.757489  0.761895  0.766569  0.771499  0.776673  0.782075  0.787688  0.793496  0.799483  0.805629  0.811919  0.818335  0.824861  0.831482  0.838182  0.844946  0.851762  0.858617  0.865498  0.872394  0.879296  0.886192  0.893075       0.899936 | 2.592031  2.592356  2.592783  2.593341  2.594065  2.594996  2.596183  2.597684  2.599565  2.601903  2.604782  2.608295  2.612543  2.617635  2.623681  2.630795  2.639091  2.648678  2.659662  2.672137  2.686188  2.701883  2.719277  2.738408  2.759295  2.781941  2.806332  2.832436   2.86021  2.889594  2.920521  2.952913  2.986686  3.021749  3.058009  3.095373  3.133745   3.17303  3.213136  3.253972  3.295449  3.337485  3.379998  3.422911   3.46615  3.509647  3.553334  3.597151  3.641037        3.684937 | 100.070317  100.095651  100.129179  100.173209  100.230576  100.304725  100.399789   100.52066  100.673058  100.863574  101.099698  101.389815  101.743178  102.169836  102.680543  103.286617  103.999783  104.831992  105.795214  106.901236  108.161459  109.586705  111.187047  112.971662  114.948722  117.125312  119.507379  122.099724   124.90601  127.928798  131.169605  134.628967  138.306519  142.201076   146.31071  150.632833  155.164268  159.901319   164.83983  169.975244  175.302642  180.816791  186.512172  192.383011  198.423298  204.626807  210.987111  217.497587  224.151427  230.941643  237.861066 |  100.0 |   0.070317 | 237.861066 |
    | 12:30 | 3.719028  3.719458  3.720026  3.720771   3.72174  3.722991  3.724593  3.726626  3.729185  3.732379  3.736329  3.741174  3.747061  3.754151  3.762615  3.772627  3.784368  3.798013  3.813735  3.831696  3.852044  3.874909  3.900401  3.928604  3.959578  3.993357  4.029949  4.069334   4.11147  4.156293  4.203718  4.253644  4.305958  4.360532  4.417232  4.475919  4.536449  4.598677  4.662458  4.727648  4.794106  4.861697  4.930288  4.999749   5.06996  5.140802  5.212162  5.283935  5.356017             5.428312 | 124.941255  124.963693  124.993349  125.032245  125.082858  125.148195  125.231855  125.338097  125.471887  125.638938  125.845731  126.099504  126.408223  126.780513  127.225562  127.752992  128.372706  129.094708  129.928907  130.884923  131.971885   133.19825  134.571635  136.098681  137.784953  139.634868  141.651661  143.837393  146.192976  148.718235  151.411984  154.272122  157.295728  160.479168  163.818199  167.308069  170.943607  174.719309  178.629414  182.667968  186.828878  191.105965  195.492994  199.983713   204.57187  209.251236  214.015615  218.858855  223.774853  228.757556 |  52.190278   52.194577   52.200258   52.207709   52.217403   52.229914   52.245929    52.26626   52.291852   52.323788   52.363295   52.411736   52.470606    52.54151   52.626146   52.726273   52.843676    52.98013   53.137352   53.316962   53.520442   53.749093   54.004006   54.286035   54.595779   54.933572   55.299487   55.693338     56.1147   56.562927   57.037178   57.536445   58.059578   58.605319   59.172325   59.759195   60.364494   60.986771   61.624575   62.276475   62.941064   63.616973   64.302876   64.997494     65.6996   66.408017   67.121623   67.839348   68.560171     69.283123 | 1.088571  1.088638  1.088726  1.088841  1.088992  1.089186  1.089434  1.089749  1.090146   1.09064  1.091252  1.092002  1.092913  1.094009  1.095317  1.096862  1.098673  1.100774  1.103193  1.105951  1.109071  1.112569  1.116461  1.120757  1.125462  1.130579  1.136105  1.142033  1.148353  1.155052  1.162113  1.169516  1.177242  1.185267  1.193568  1.202123  1.210906  1.219895  1.229065  1.238394   1.24786  1.257443  1.267122  1.276878  1.286694  1.296552  1.306437  1.316334   1.32623   1.33611 | 1.359148  1.359144  1.359138   1.35913   1.35912  1.359108  1.359091   1.35907  1.359044  1.359011  1.358971  1.358922  1.358862  1.358791  1.358706  1.358606  1.358489  1.358354  1.358201  1.358027  1.357832  1.357616  1.357378  1.357118  1.356838  1.356538   1.35622  1.355884  1.355532  1.355167   1.35479  1.354404   1.35401   1.35361  1.353206  1.352801  1.352395  1.351991  1.351589  1.351191  1.350799  1.350412  1.350033  1.349661  1.349297  1.348942  1.348595  1.348258   1.34793          1.347611 | 0.536253  0.536287  0.536332   0.53639  0.536467  0.536565  0.536691  0.536851  0.537052  0.537303  0.537613  0.537992  0.538452  0.539004  0.539662  0.540437  0.541342  0.542389  0.543588   0.54495  0.546482  0.548189  0.550076  0.552142  0.554388  0.556807  0.559396  0.562144  0.565042  0.568078  0.571239  0.574511   0.57788  0.581333  0.584853  0.588429  0.592045  0.595691  0.599353  0.603021  0.606684  0.610334  0.613963  0.617563  0.621127   0.62465  0.628126  0.631553  0.634925       0.63824 | -0.201994  -0.202039  -0.202098  -0.202175  -0.202274  -0.202403  -0.202566  -0.202772  -0.203031  -0.203351  -0.203746  -0.204226  -0.204806  -0.205499   -0.20632  -0.207282    -0.2084  -0.209686  -0.211151  -0.212806  -0.214657   -0.21671  -0.218966  -0.221426  -0.224085  -0.226938  -0.229976  -0.233189  -0.236564  -0.240088  -0.243744  -0.247519  -0.251395  -0.255356  -0.259389  -0.263476  -0.267604   -0.27176   -0.27593  -0.280104   -0.28427  -0.288418  -0.292542  -0.296631  -0.300681  -0.304684  -0.308635  -0.312531  -0.316366     -0.320138 | 0.665723  0.665727  0.665734  0.665741  0.665752  0.665765  0.665782  0.665803  0.665829  0.665861  0.665901  0.665949  0.666007  0.666076  0.666157  0.666253  0.666363   0.66649  0.666635  0.666798   0.66698  0.667183  0.667407  0.667652  0.667917  0.668204   0.66851  0.668837  0.669183  0.669546  0.669927  0.670324  0.670736  0.671161  0.671598  0.672047  0.672505  0.672972  0.673445  0.673925   0.67441  0.674899   0.67539  0.675884  0.676378  0.676872  0.677366  0.677859  0.678349      0.678837 | 100.057539 | 100.068142  100.092568  100.124855  100.167204  100.222318  100.293475  100.384608  100.500369  100.646193  100.828346  101.053946  101.330973  101.668237  102.075323  102.562504  103.140617  103.820923  104.614928  105.534207  106.590201  107.794031  109.156309  110.686977  112.395156  114.289038  116.375795   118.66153   121.15125  123.848872  126.757248  129.878211  133.212638  136.760514  140.521011  144.492563  148.672943  153.059331  157.648383  162.436294  167.418846  172.591461  177.949237  183.486988  189.199266  195.080387  201.124452  207.325356  213.676804  220.172317          226.805236 | 230.150482 | 0.720829  0.720876  0.720937  0.721018  0.721123  0.721258  0.721431  0.721651  0.721927  0.722272  0.722699  0.723222  0.723856  0.724621  0.725532   0.72661  0.727871  0.729336  0.731021  0.732943  0.735116  0.737552  0.740262  0.743252  0.746527  0.750086   0.75393  0.758051  0.762444  0.767098  0.772002  0.777142  0.782504  0.788071  0.793827  0.799756  0.805841  0.812066  0.818413  0.824868  0.831415  0.838039  0.844727  0.851466  0.858243  0.865046  0.871865   0.87869  0.885511       0.892319 | 2.591866  2.592135   2.59249  2.592956  2.593562  2.594344  2.595346  2.596617  2.598216  2.600212  2.602681  2.605708  2.609386  2.613815    2.6191   2.62535  2.632676  2.641187  2.650989  2.662181  2.674852   2.68908  2.704931  2.722452  2.741678  2.762624   2.78529  2.809659  2.835698  2.863364  2.892597  2.923331   2.95549  2.988991  3.023747  3.059669  3.096665  3.134642  3.173509  3.213177  3.253558  3.294567  3.336124  3.378148  3.420567  3.463309  3.506306  3.549494  3.592814        3.636207 | 100.057539  100.078493  100.106314   100.14297  100.190892  100.253051   100.33303  100.435103  100.564298  100.726455  100.928261  101.177268   101.48188  101.851314  102.295527  102.825112  103.451159  104.185103  105.038536   106.02302  107.149888  108.430057  109.873847  111.490826  113.289678  115.278107   117.46276  119.849195  122.441872  125.244163  128.258399  131.485916  134.927125  138.581585  142.448077  146.524686  150.808867  155.297522  159.987061  164.873458  169.952303  175.218847  180.668036  186.294546  192.092805  198.057018  204.181178  210.459084  216.884347  223.450395  230.150482 |  100.0 |   0.057539 | 230.150482 |
    | 13:00 |  3.71881  3.719166  3.719638  3.720259  3.721069  3.722119  3.723467  3.725186  3.727357  3.730079  3.733459  3.737623  3.742706  3.748857  3.756234  3.765005   3.77534  3.787413  3.801395  3.817451  3.835733  3.856383  3.879521   3.90525  3.933645  3.964761  3.998625   4.03524  4.074584  4.116612  4.161258   4.20844  4.258058  4.309999  4.364141  4.420352  4.478498  4.538439  4.600035  4.663146  4.727632  4.793359  4.860194  4.928009  4.996679  5.066086  5.136116  5.206661  5.277616             5.348882 | 124.929909  124.948478  124.973101  125.005502  125.047808  125.102614  125.173045  125.262823  125.376322  125.518609  125.695482  125.913471  126.179829  126.502489  126.889988  127.351373  127.896062  128.533697  129.273963   130.12641  131.100253  132.204196  133.446257  134.833617  136.372497   138.06807  139.924397  141.944411  144.129915  146.481627  148.999229  151.681452  154.526158   157.53044  160.690721  164.002849  167.462195  171.063737  174.802138  178.671819  182.667022  186.781856  191.010347  195.346469  199.784179  204.317433  208.940206  213.646505  218.430377  223.285911 |  52.188104   52.191662   52.196379   52.202587    52.21069   52.221186   52.234671   52.251856   52.273573   52.300786   52.334593    52.37623    52.42706   52.488568   52.562342   52.650047     52.7534   52.874133   53.013954   53.174507   53.357334    53.56383   53.795215   54.052496   54.336451   54.647611   54.986252     55.3524   55.745837   56.166116   56.612583   57.084403    57.58058    58.09999   58.641405   59.203522   59.784981   60.384391    61.00035   61.631456   62.276323   62.933593   63.601941   64.280085   64.966788   65.660861   66.361164   67.066609   67.776158     68.488818 | 1.088537  1.088593  1.088666  1.088762  1.088888   1.08905  1.089259  1.089526  1.089862  1.090284  1.090808  1.091452  1.092239   1.09319  1.094331  1.095686  1.097281  1.099142  1.101295  1.103764  1.106571  1.109735  1.113274  1.117201  1.121524  1.126248  1.131376  1.136902  1.142822  1.149123  1.155792  1.162814  1.170169  1.177837  1.185796  1.194024  1.202498  1.211194   1.22009  1.229163  1.238392  1.247754   1.25723  1.266801  1.276448  1.286153  1.295902  1.305677  1.315465  1.325252 |  1.35915  1.359147  1.359142  1.359136  1.359127  1.359116  1.359103  1.359085  1.359063  1.359035     1.359  1.358958  1.358906  1.358844   1.35877  1.358682  1.358579  1.358459  1.358321  1.358165  1.357988  1.357791  1.357572  1.357333  1.357072  1.356792  1.356492  1.356174   1.35584   1.35549  1.355127  1.354753   1.35437   1.35398  1.353584  1.353184  1.352783  1.352382  1.351982  1.351585  1.351191  1.350803  1.350421  1.350045  1.349677  1.349317  1.348965  1.348622  1.348287          1.347962 | 0.536236  0.536264  0.536301   0.53635  0.536414  0.536497  0.536603  0.536738  0.536909  0.537123  0.537388  0.537714  0.538112  0.538592  0.539166  0.539847  0.540646  0.541576  0.542648  0.543871  0.545255  0.546807  0.548532  0.550433   0.55251  0.554761  0.557182  0.559767  0.562507  0.565392  0.568411  0.571551  0.574798  0.578138  0.581559  0.585045  0.588584  0.592163  0.595769  0.599392   0.60302  0.606643  0.610254  0.613844  0.617405  0.620932  0.624419  0.627861  0.631254      0.634594 | -0.201971  -0.202008  -0.202057  -0.202122  -0.202205  -0.202313  -0.202451  -0.202626  -0.202845  -0.203119  -0.203458  -0.203872  -0.204375  -0.204979  -0.205698  -0.206546  -0.207536  -0.208681  -0.209994  -0.211486  -0.213164  -0.215036  -0.217106  -0.219375  -0.221843  -0.224505  -0.227356  -0.230387  -0.233587  -0.236943  -0.240443  -0.244072  -0.247814  -0.251655  -0.255578  -0.259569  -0.263613  -0.267697  -0.271807  -0.275931  -0.280058  -0.284178  -0.288281  -0.292359  -0.296405  -0.300412  -0.304374  -0.308286  -0.312143     -0.315942 |  0.66572  0.665724  0.665729  0.665736  0.665745  0.665756   0.66577  0.665788   0.66581  0.665838  0.665872  0.665913  0.665964  0.666024  0.666095  0.666179  0.666277   0.66639  0.666519  0.666666  0.666832  0.667016  0.667221  0.667445  0.667691  0.667956  0.668242  0.668548  0.668873  0.669217  0.669578  0.669956   0.67035  0.670757  0.671178  0.671611  0.672055  0.672508  0.672969  0.673437  0.673912  0.674391  0.674873  0.675359  0.675847  0.676335  0.676824  0.677312  0.677799      0.678284 | 100.047032 | 100.055791  100.076005  100.102811  100.138087  100.184151  100.243832  100.320543  100.418347  100.542026  100.697131   100.89002  101.127875  101.418699  101.771275  102.195109  102.700331  103.297576  103.997837  104.812296  105.752148  106.828413  108.051758  109.432323  110.979568  112.702144  114.607787  116.703248  118.994242  121.485437  124.180458  127.081917  130.191459  133.509821  137.036897  140.771814  144.712999  148.858251  153.204814  157.749434  162.488415  167.417675  172.532785  177.829013  183.301346  188.944527  194.753072  200.721285  206.843275  213.112967          219.524103 | 222.759457 | 0.720806  0.720844  0.720895  0.720962   0.72105  0.721164  0.721309  0.721495   0.72173  0.722024  0.722389  0.722838  0.723387   0.72405  0.724845   0.72579  0.726901  0.728199  0.729699  0.731419  0.733374  0.735579  0.738043  0.740777  0.743786  0.747074  0.750641  0.754484    0.7586  0.762979  0.767613  0.772489  0.777595  0.782917  0.788438  0.794143  0.800016  0.806041  0.812201  0.818481  0.824867  0.831341  0.837892  0.844506  0.851169   0.85787  0.864597  0.871341  0.878091       0.884837 |  2.59173  2.591953  2.592248  2.592636  2.593143  2.593799  2.594642  2.595716  2.597074  2.598775  2.600888   2.60349  2.606666  2.610508  2.615116  2.620592  2.627043  2.634576  2.643296  2.653305  2.664695  2.677552  2.691949  2.707944  2.725583  2.744894  2.765889  2.788566  2.812905  2.838874  2.866426  2.895506  2.926046  2.957972  2.991205  3.025658  3.061246   3.09788   3.13547  3.173928  3.213168  3.253104  3.293656  3.334744  3.376292  3.418228  3.460484  3.502993  3.545694        3.588528 | 100.047032   100.06434  100.087395   100.11787  100.157844  100.209872  100.277054  100.363108  100.472441  100.610206  100.782348  100.995636  101.257667  101.576846  101.962339  102.423992  102.972227  103.617899  104.372143  105.246199  106.251226  107.398116  108.697318  110.158672  111.791267  113.603322  115.602096  117.793832  120.183724  122.775913  125.573507  128.578621  131.792428  135.215227  138.846506  142.685023  146.728873   150.97556   155.42206  160.064886  164.900134   169.92354  175.130511  180.516167  186.075368  191.802736  197.692678  203.739398  209.936909  216.279043  222.759457 |  100.0 |   0.047032 | 222.759457 |
    | 13:30 | 3.718631  3.718926  3.719317  3.719833   3.72051  3.721389  3.722522  3.723972  3.725812  3.728126  3.731013  3.734584  3.738963  3.744286    3.7507  3.758362  3.767436  3.778088  3.790486  3.804794  3.821171  3.839761  3.860698  3.884093  3.910042  3.938614  3.969856   4.00379  4.040415  4.079706  4.121617  4.166081  4.213016  4.262323  4.313893  4.367606  4.423334  4.480947   4.54031  4.601286   4.66374  4.727539   4.79255  4.858646  4.925702  4.993599  5.062222   5.13146  5.201207             5.271363 | 124.920573  124.935921  124.956338  124.983291  125.018602  125.064503  125.123701  125.199438  125.295551  125.416519  125.567501  125.754362   125.98367  126.262677  126.599267   127.00188  127.479406  128.041055  128.696201  129.454217  130.324292  131.315253  132.435394  133.692312  135.092775  136.642609  138.346618  140.208536  142.231011  144.415616  146.762888  149.272383  151.942758  154.771848  157.756764  160.893987  164.179456   167.60866   171.17672  174.878462   178.70848  182.661202   186.73093  190.911885  195.198243  199.584156  204.063781  208.631287  213.280876  218.006781 |  52.186315   52.189256   52.193168   52.198332   52.205096   52.213888   52.225224   52.239724   52.258119   52.281262   52.310133   52.345843   52.389631    52.44286   52.507003   52.583625   52.674359   52.780878   52.904858   53.047944   53.211708   53.397613   53.606976   53.840935   54.100421    54.38614   54.698559   55.037903   55.404153   55.797064    56.21617   56.660812    57.13016   57.623233   58.138931   58.676058   59.233345   59.809474   60.403099   61.012861   61.637404   62.275389     62.9255   63.586458   64.257022   64.935993   65.622219   66.314597    67.01207      67.71363 |  1.08851  1.088555  1.088616  1.088696  1.088801  1.088937  1.089113  1.089338  1.089623  1.089981  1.090429  1.090982   1.09166  1.092483  1.093475   1.09466  1.096061  1.097704  1.099615  1.101818  1.104335  1.107188  1.110396  1.113973  1.117931  1.122279  1.127021  1.132156  1.137682  1.143591  1.149872  1.156511  1.163494    1.1708  1.178411  1.186305   1.19446  1.202854  1.211465  1.220271  1.229249  1.238378  1.247639  1.257011  1.266476  1.276016  1.285614  1.295255  1.304923  1.314604 | 1.359152  1.359149  1.359145   1.35914  1.359133  1.359124  1.359112  1.359097  1.359079  1.359055  1.359025  1.358989  1.358944   1.35889  1.358826  1.358748  1.358657  1.358551  1.358429  1.358288  1.358129  1.357949   1.35775  1.357529  1.357289  1.357027  1.356746  1.356447   1.35613  1.355797  1.355449  1.355089  1.354718  1.354338  1.353951  1.353559  1.353164  1.352767   1.35237  1.351974  1.351581  1.351192  1.350808   1.35043  1.350058  1.349693  1.349336  1.348988  1.348648          1.348316 | 0.536222  0.536245  0.536276  0.536316   0.53637  0.536439  0.536528  0.536643  0.536787  0.536969  0.537196  0.537476  0.537819  0.538235  0.538736  0.539332  0.540035  0.540858  0.541812  0.542907  0.544153  0.545559   0.54713  0.548871  0.550785  0.552871  0.555127  0.557549   0.56013  0.562861  0.565733  0.568734  0.571852  0.575074  0.578386  0.581775  0.585228  0.588732  0.592274  0.595842  0.599426  0.603015  0.606599  0.610171  0.613723  0.617247  0.620738  0.624189  0.627597      0.630958 | -0.201953  -0.201983  -0.202024  -0.202077  -0.202147  -0.202238  -0.202354  -0.202502  -0.202688  -0.202922  -0.203212  -0.203569  -0.204003  -0.204528  -0.205156    -0.2059  -0.206775  -0.207792  -0.208964  -0.210303  -0.211819   -0.21352  -0.215411  -0.217497  -0.219778  -0.222253  -0.224917  -0.227764  -0.230787  -0.233973  -0.237311  -0.240787  -0.244388  -0.248099  -0.251904  -0.255789   -0.25974  -0.263742  -0.267782  -0.271847  -0.275926  -0.280008  -0.284083  -0.288142  -0.292177   -0.29618  -0.300145  -0.304066  -0.307939     -0.311759 | 0.665718  0.665722  0.665726  0.665731  0.665739  0.665748   0.66576  0.665775  0.665794  0.665818  0.665847  0.665883  0.665926  0.665979  0.666041  0.666115  0.666201  0.666301  0.666417  0.666549  0.666698  0.666865  0.667051  0.667257  0.667483  0.667729  0.667994   0.66828  0.668585  0.668909   0.66925  0.669609  0.669984  0.670374  0.670778  0.671195  0.671623  0.672062   0.67251  0.672966  0.673429  0.673897  0.674371  0.674848  0.675328   0.67581  0.676293  0.676776  0.677259       0.67774 | 100.038402 | 100.045629  100.062336  100.084562  100.113905  100.152349  100.202329  100.266798  100.349293  100.454007  100.585838  100.750441  100.954252  101.204502  101.509199  101.877086  102.317574   102.84064  103.456705  104.176492  105.010856   105.97062  107.066393  108.308403  109.706336  111.269196  113.005184  114.921611  117.024828  119.320192  121.812054  124.503768  127.397726  130.495399  133.797398  137.303536  141.012899   144.92391  149.034402  153.341675  157.842563  162.533479  167.410467  172.469244  177.705234  183.113595   188.68925  194.426903  200.321054  206.366017          212.555921 | 215.681533 | 0.720787  0.720818  0.720861  0.720916   0.72099  0.721085  0.721207  0.721364  0.721563  0.721813  0.722125   0.72251  0.722983  0.723557  0.724249  0.725074  0.726051  0.727197  0.728528  0.730063  0.731817  0.733805  0.736039  0.738529  0.741285  0.744311  0.747611  0.751184  0.755026  0.759134  0.763499  0.768112  0.772961  0.778033  0.783315   0.78879  0.794445  0.800263  0.806228  0.812326  0.818541  0.824857  0.831262  0.837741  0.844281  0.850871  0.857498  0.864151  0.870821       0.877497 | 2.591619  2.591802  2.592047   2.59237  2.592793  2.593343  2.594051  2.594958  2.596108  2.597554  2.599359  2.601591  2.604327  2.607653  2.611659  2.616445  2.622109  2.628758  2.636493  2.645416  2.655623  2.667204  2.680238  2.694792  2.710922  2.728668  2.748054  2.769089  2.791769  2.816072  2.841964  2.869401  2.898324  2.928669  2.960364   2.99333  3.027485  3.062744  3.099022  3.136233   3.17429  3.213111  3.252613  3.292717  3.333347   3.37443  3.415895  3.457676  3.499709        3.541933 | 100.038402  100.052681  100.071761  100.097063  100.130361  100.173844  100.230189  100.302621  100.394988   100.51182  100.658383  100.840716  101.065654  101.340821   101.67461  102.076118  102.555069  103.121701  103.786627  104.560682  105.454755  106.479609  107.645709  108.963055  110.441026  112.088256  113.912524  115.920674  118.118569  120.511062  123.102001  125.894247  128.889719  132.089442  135.493608  139.101649  142.912301  146.923673  151.133316  155.538281  160.135177  164.920221  169.889284  175.037926  180.361431  185.854833   191.51294  197.330352  203.301471  209.420517  215.681533 |  100.0 |   0.038402 | 215.681533 |
    | 14:00 | 3.718484  3.718727  3.719051   3.71948  3.720044  3.720779  3.721731  3.722952  3.724508  3.726472  3.728933  3.731989  3.735753  3.740349  3.745912  3.752589  3.760533  3.769905  3.780867  3.793582  3.808206  3.824892  3.843777  3.864985  3.888623  3.914776  3.943509  3.974862  4.008853  4.045476  4.084705   4.12649  4.170766   4.21745  4.266445  4.317646  4.370933  4.426185  4.483273  4.542067  4.602435  4.664247  4.727371  4.791682  4.857056  4.923371  4.990513  5.058369  5.126832               5.1958 | 124.912901  124.925571  124.942477  124.964868  124.994298  125.032683  125.082362   125.14615    125.2274  125.330054  125.458688  125.618543  125.815538  126.056265  126.347959  126.698445   127.11605  127.609502  128.187792  128.860029  129.635272   130.52236  131.529742  132.665316   133.93628   135.34901  136.908957  138.620583  140.487317  142.511544  144.694624   147.03693  149.537909  152.196149  155.009468     157.975  161.089282  164.348343  167.747789  171.282877  174.948588  178.739685  182.650771  186.676333   190.81078  195.048475  199.383764  203.810989  208.324508  212.918706 |  52.184845   52.187272   52.190512   52.194802    52.20044   52.207793   52.217308   52.229522   52.245076   52.264721   52.289327    52.31989   52.357528   52.403486   52.459119   52.525888   52.605332   52.699052   52.808672   52.935816   53.082063   53.248917   53.437766   53.649849   53.886228   54.147763   54.435091   54.748622   55.088532   55.454764   55.847045   56.264897   56.707655   57.174496   57.664455   58.176456   58.709332    59.26185   59.832732   60.420672   61.024354   61.642467   62.273714   62.916823   63.570556    64.23371   64.905125   65.583685   66.268319     66.958003 | 1.088487  1.088524  1.088575  1.088641  1.088729  1.088843   1.08899   1.08918  1.089421  1.089725  1.090106   1.09058  1.091163  1.091874  1.092735  1.093767  1.094995  1.096442  1.098133  1.100092  1.102343  1.104907  1.107804  1.111052  1.114664  1.118652  1.123023  1.127779  1.132921  1.138444  1.144341  1.150601   1.15721  1.164152  1.171409  1.178963  1.186792  1.194876  1.203192  1.211719  1.220436  1.229321  1.238354  1.247516  1.256786  1.266148  1.275583  1.285076  1.294612  1.304175 | 1.359154  1.359151  1.359148  1.359144  1.359138   1.35913   1.35912  1.359108  1.359092  1.359072  1.359047  1.359015  1.358977   1.35893  1.358874  1.358807  1.358727  1.358633  1.358524  1.358398  1.358255  1.358092  1.357911  1.357709  1.357487  1.357245  1.356983  1.356702  1.356403  1.356086  1.355755  1.355409  1.355051  1.354683  1.354306  1.353923  1.353535  1.353144  1.352751  1.352358  1.351967  1.351578  1.351193  1.350813  1.350439  1.350071   1.34971  1.349356  1.349011          1.348674 |  0.53621  0.536229  0.536255  0.536289  0.536333  0.536391  0.536466  0.536562  0.536685  0.536839  0.537033  0.537273  0.537568  0.537928  0.538362  0.538883    0.5395  0.540226  0.541073  0.542049  0.543167  0.544435  0.545861   0.54745  0.549206  0.551132  0.553226  0.555486  0.557908  0.560484  0.563206  0.566064  0.569048  0.572144  0.575341  0.578625  0.581983  0.585403  0.588872  0.592377  0.595909  0.599454  0.603005  0.606552  0.610086  0.613601  0.617088  0.620543  0.623961      0.627335 | -0.201937  -0.201962  -0.201996  -0.202041  -0.202099  -0.202175  -0.202272  -0.202397  -0.202555  -0.202754  -0.203002  -0.203308  -0.203683  -0.204138  -0.204685  -0.205337  -0.206106  -0.207006  -0.208049  -0.209248  -0.210612  -0.212151  -0.213873  -0.215783  -0.217883  -0.220174  -0.222654   -0.22532  -0.228163  -0.231176  -0.234348  -0.237667   -0.24112  -0.244693  -0.248372  -0.252143  -0.255991  -0.259902  -0.263863   -0.26786  -0.271882  -0.275917  -0.279955  -0.283986  -0.288001  -0.291993  -0.295955  -0.299879  -0.303761     -0.307596 | 0.665717  0.665719  0.665723  0.665728  0.665734  0.665741  0.665751  0.665764   0.66578  0.665801  0.665826  0.665856  0.665894  0.665939  0.665994  0.666058  0.666135  0.666223  0.666326  0.666444  0.666578  0.666729  0.666898  0.667086  0.667293   0.66752  0.667766  0.668032  0.668317  0.668621  0.668943  0.669282  0.669639  0.670011  0.670397  0.670798   0.67121  0.671634  0.672068  0.672511  0.672962   0.67342  0.673883  0.674351  0.674823  0.675297  0.675774  0.676251  0.676729      0.677206 | 100.031322 | 100.037278  100.051069  100.069473  100.093848  100.125888  100.167682  100.221778  100.291248  100.379754  100.491604  100.631805  100.806103     101.021  101.283759  101.602379   101.98555  102.442571  102.983258  103.617812  104.356679  105.210394  106.189412   107.30394  108.563785  109.978196  111.555741  113.304201   115.23048  117.340559  119.639457   122.13123  124.818983  127.704903  130.790302  134.075677  137.560769  141.244626  145.125675  149.201777  153.470295  157.928142  162.571839   167.39755  172.401131  177.578154  182.923942  188.433588  194.101976  199.923794          205.893546 | 208.909841 | 0.720771  0.720797  0.720832  0.720878  0.720939  0.721019  0.721122  0.721254  0.721422  0.721634    0.7219   0.72223  0.722636  0.723133  0.723733  0.724452  0.725308  0.726317  0.727495  0.728861  0.730429  0.732215  0.734234  0.736495  0.739011  0.741787  0.744829  0.748139  0.751716  0.755556  0.759656  0.764006  0.768597  0.773418  0.778456  0.783698  0.789129  0.794733  0.800497  0.806404  0.812441  0.818591  0.824841  0.831177  0.837585  0.844055  0.850572  0.857126  0.863707       0.870305 | 2.591527  2.591678  2.591881  2.592149  2.592502  2.592962  2.593556   2.59432  2.595292   2.59652  2.598059  2.599969  2.602321  2.605193  2.608668  2.612839    2.6178  2.623651  2.630492  2.638424  2.647543  2.657942  2.669704  2.682906  2.697609  2.713864  2.731707  2.751159  2.772226  2.794901  2.819161  2.844973  2.872289  2.901054  2.931204  2.962669  2.995371  3.029231  3.064167  3.100095  3.136934  3.174599  3.213009  3.252086  3.291753  3.331936  3.372563  3.413568  3.454885        3.496452 | 100.031322  100.043088  100.058858  100.079837  100.107535  100.143825  100.191009  100.251879  100.329783  100.428691  100.553248  100.708821   100.90153  101.138259   101.42665  101.775063  102.192515  102.688595  103.273349  103.957141  104.750507  105.663988   106.70796  107.892474  109.227096  110.720769  112.381694  114.217232  116.233839   118.43702  120.831313  123.420291   126.20659  129.191946  132.377248  135.762595  139.347367  143.130282  147.109468   151.28252  155.646563  160.198299  164.934061  169.849849  174.941367  180.204058  185.633122  191.223543  196.970104  202.867398  208.909841 |  100.0 |   0.031322 | 208.909841 |
    | 14:30 | 3.718364  3.718564  3.718832  3.719188  3.719657  3.720271  3.721069  3.722096  3.723409  3.725073  3.727166  3.729777  3.733005  3.736964  3.741778  3.747582   3.75452  3.762744  3.772409  3.783675  3.796697  3.811628   3.82861  3.847776  3.869242  3.893107  3.919451   3.94833   3.97978  4.013815  4.050425  4.089581  4.131233  4.175315  4.221746   4.27043  4.321262  4.374128  4.428909  4.485481  4.543716  4.603488  4.664669  4.727134   4.79076  4.855426  4.921017   4.98742  5.054527             5.122234 | 124.906603  124.917049  124.931031  124.949606  124.974101  125.006155  125.047782  125.101419  125.169989  125.256949   125.36634  125.502825  125.671713  125.878966  126.131188  126.435584  126.799906   127.23236  127.741502  128.336101  129.024999  129.816944  130.720435  131.743553  132.893816  134.178042  135.602234  137.171494  138.889961  140.760779  142.786093  144.967066  147.303925  149.796014  152.441866  155.239287  158.185435  161.276907  164.509819   167.87989   171.38251  175.012808  178.765706  182.635979  186.618286  190.707218  194.897318  199.183113  203.559125  208.019892 |  52.183638    52.18564   52.188319   52.191878   52.196571   52.202712   52.210685   52.220957   52.234086   52.250732   52.271663   52.297768   52.330051   52.369641   52.417781    52.47582   52.545201   52.627438   52.724093   52.836749   52.966971   53.116277   53.286102   53.477762   53.692423   53.931075   54.194508   54.483299   54.797803    55.13815   55.504251   55.895808    56.31233   56.753152   57.217456   57.704296   58.212616   58.741281   59.289092   59.854806   60.437159   61.034877   61.646688   62.271338   62.907597   63.554263   64.210173   64.874203   65.545271     66.222336 | 1.088468  1.088499  1.088541  1.088596  1.088669  1.088764  1.088888  1.089047   1.08925  1.089508  1.089833  1.090237  1.090737   1.09135  1.092095  1.092993  1.094066  1.095336  1.096829  1.098566  1.100572  1.102869  1.105478  1.108417  1.111703  1.115349  1.119364  1.123755  1.128524   1.13367  1.139189  1.145073   1.15131  1.157887  1.164789  1.171998  1.179496  1.187261  1.195274  1.203513  1.211958  1.220588  1.229382  1.238321  1.247385  1.256556  1.265816   1.27515   1.28454  1.293973 | 1.359155  1.359153   1.35915  1.359147  1.359142  1.359135  1.359127  1.359117  1.359103  1.359086  1.359065  1.359038  1.359005  1.358965  1.358916  1.358857  1.358787  1.358704  1.358608  1.358496  1.358367  1.358221  1.358057  1.357873  1.357669  1.357445  1.357202  1.356939  1.356658  1.356359  1.356044  1.355714   1.35537  1.355015   1.35465  1.354276  1.353896  1.353512  1.353125  1.352736  1.352347   1.35196  1.351575  1.351194  1.350818  1.350448  1.350083  1.349726  1.349376          1.349034 | 0.536201  0.536216  0.536237  0.536266  0.536303  0.536351  0.536414  0.536495  0.536598  0.536729  0.536894  0.537099  0.537352  0.537663  0.538039  0.538493  0.539033  0.539672   0.54042  0.541289  0.542288  0.543428  0.544717  0.546162  0.547767  0.549538  0.551474  0.553575  0.555838  0.558259  0.560829  0.563542  0.566386  0.569352  0.572427  0.575599  0.578855  0.582183   0.58557  0.589005  0.592475  0.595969  0.599478  0.602992  0.606502  0.609999  0.613477  0.616929   0.62035      0.623733 | -0.201924  -0.201945  -0.201973   -0.20201  -0.202059  -0.202122  -0.202204  -0.202309  -0.202443  -0.202612  -0.202824  -0.203086  -0.203408  -0.203802  -0.204277  -0.204846  -0.205521  -0.206315   -0.20724  -0.208309  -0.209532  -0.210921  -0.212482  -0.214224   -0.21615  -0.218263  -0.220563  -0.223049  -0.225714  -0.228552  -0.231555  -0.234712  -0.238012  -0.241441  -0.244987  -0.248636  -0.252372  -0.256184  -0.260056  -0.263976  -0.267932  -0.271911  -0.275903  -0.279898  -0.283886  -0.287859  -0.291809   -0.29573  -0.299615     -0.303458 | 0.665715  0.665718  0.665721  0.665724  0.665729  0.665736  0.665744  0.665755  0.665769  0.665786  0.665807  0.665834  0.665866  0.665906  0.665953  0.666009  0.666076  0.666155  0.666246  0.666351  0.666471  0.666607   0.66676  0.666931  0.667121  0.667329  0.667556  0.667803  0.668068  0.668353  0.668655  0.668976  0.669314  0.669667  0.670036   0.67042  0.670816  0.671225  0.671645  0.672074  0.672512  0.672958  0.673411  0.673869  0.674331  0.674798  0.675267  0.675738   0.67621      0.676682 | 100.025521 | 100.030423  100.041793  100.057012  100.077234  100.103899  100.138798  100.184122  100.242532  100.317214  100.411947  100.531147  100.679922  100.864093  101.090216  101.365574  101.698142   102.09654  102.569951  103.128014  103.780707  104.538201  105.410708  106.408323  107.540867  108.817735  110.247762  111.839102  113.599132   115.53438  117.650473  119.952119  122.443098  125.126283  128.003673  131.076434  134.344959  137.808921  141.467343  145.318652  149.360743  153.591034  158.006523  162.603828  167.379231   172.32872   177.44801  182.732579  188.177683  193.778378          199.529529 | 202.437175 | 0.720758  0.720779  0.720808  0.720847  0.720897  0.720964   0.72105  0.721161  0.721303  0.721483  0.721709  0.721991   0.72234  0.722767  0.723287  0.723913  0.724661  0.725546  0.726586  0.727797  0.729195  0.730795  0.732613   0.73466  0.736949  0.739488  0.742283  0.745339  0.748657  0.752237  0.756074  0.760164  0.764499  0.769068  0.773861  0.778865  0.784067  0.789453  0.795009  0.800719   0.80657  0.812546  0.818633  0.824817  0.831086  0.837426  0.843825  0.850272  0.856756       0.863266 | 2.591451  2.591576  2.591744  2.591966   2.59226  2.592644  2.593142  2.593785  2.594605  2.595646  2.596954  2.598586  2.600604  2.603078  2.606086  2.609712  2.614045   2.61918  2.625214  2.632244  2.640367  2.649676  2.660258  2.672195  2.685555  2.700397  2.716768  2.734699  2.754208    2.7753  2.797963  2.822175  2.847901  2.875094    2.9037  2.933654  2.964889   2.99733  3.030899  3.065517  3.101102  3.137575  3.174856  3.212865  3.251526  3.290765   3.33051  3.370693  3.411248        3.452111 | 100.025521  100.035204  100.048223  100.065594  100.088602  100.118847  100.158302  100.209376  100.274977  100.358569  100.464236  100.596729  100.761505  100.964753  101.213394  101.515069  101.878094  102.311393   102.82441   103.42699  104.129247  104.941415  105.873693  106.936082   108.13823  109.489288   110.99778  112.671496  114.517406  116.541598  118.749243  121.144584  123.730941  126.510739  129.485547  132.656129  136.022502  139.583997  143.339322  147.286621  151.423537  155.747264  160.254597  164.941976  169.805527  174.841093  180.044265  185.410403   190.93466  196.611992  202.437175 |  100.0 |   0.025521 | 202.437175 |
    | 15:00 | 3.718265   3.71843  3.718651  3.718946  3.719336  3.719848  3.720515  3.721378  3.722484  3.723892  3.725669  3.727895  3.730658  3.734061  3.738217   3.74325  3.749294  3.756492  3.764991  3.774946  3.786508  3.799829  3.815055  3.832323  3.851757  3.873467  3.897546  3.924064  3.953076   3.98461  4.018677  4.055263  4.094338   4.13585  4.179734  4.225909   4.27428  4.324746  4.377196  4.431512  4.487575  4.545261  4.604447  4.665011   4.72683  4.789785  4.853761  4.918643  4.984324             5.050698 |  124.90144  124.910041   124.92159   124.93698  124.957339  124.984069  125.018897  125.063929  125.121703  125.195241    125.2881  125.404411  125.548915  125.726979  125.944596  126.208367  126.525458  126.903534  127.350673  127.875251  128.485814  129.190935  129.999058  130.918347  131.956529  133.120758  134.417489  135.852374  137.430183  139.154755   141.02897  143.054748  145.233075  147.564041  150.046902  152.680143  155.461563  158.388347  161.457151  164.664179  168.005259  171.475907  175.071395  178.786801  182.617058  186.556996  190.601375  194.744912  198.982306  203.308252 |  52.182648   52.184297    52.18651   52.189459    52.19336   52.198481   52.205152   52.213778   52.224841   52.238921   52.256693   52.278946    52.30658   52.340611   52.382172   52.432503   52.492944   52.564919   52.649914   52.749455   52.865076    52.99829   53.150552   53.323231   53.517573   53.734674   53.975455   54.240644   54.530759   54.846104   55.186768   55.552632   55.943378   56.358503   56.797342   57.259085   57.742803   58.247463   58.771958   59.315121   59.875748   60.452611   61.044475    61.65011     62.2683   62.897853   63.537607   64.186433   64.843243     65.506985 | 1.088453  1.088478  1.088513  1.088558  1.088619  1.088698  1.088802  1.088935  1.089107  1.089325  1.089601  1.089946  1.090374  1.090901  1.091544  1.092323  1.093258  1.094371  1.095684   1.09722  1.099002  1.101054  1.103396  1.106047  1.109027  1.112349  1.116026  1.120066  1.124476  1.129255  1.134404  1.139917  1.145786     1.152  1.158545  1.165407  1.172567  1.180008   1.18771  1.195653  1.203817  1.212182  1.220726  1.229431  1.238277  1.247246   1.25632  1.265482  1.274716  1.284006 | 1.359156  1.359154  1.359152  1.359149  1.359145   1.35914  1.359133  1.359124  1.359113  1.359098   1.35908  1.359057  1.359029  1.358994  1.358952  1.358901   1.35884  1.358767  1.358682  1.358583  1.358468  1.358337  1.358188  1.358021  1.357835  1.357629  1.357404   1.35716  1.356897  1.356616  1.356317  1.356003  1.355674  1.355333   1.35498  1.354617  1.354247  1.353871   1.35349  1.353106  1.352722  1.352337  1.351954  1.351573  1.351196  1.350824  1.350457  1.350096  1.349742          1.349396 | 0.536193  0.536206  0.536223  0.536246  0.536277  0.536318   0.53637  0.536438  0.536525  0.536636  0.536776  0.536951  0.537168  0.537435  0.537761  0.538154  0.538626  0.539186  0.539846  0.540616  0.541507  0.542528  0.543689  0.544998   0.54646  0.548082  0.549865  0.551811  0.553918  0.556184  0.558602  0.561166  0.563869  0.566699  0.569647    0.5727  0.575847  0.579076  0.582374   0.58573  0.589131  0.592566  0.596025  0.599498  0.602975  0.606448   0.60991  0.613352   0.61677      0.620156 | -0.201914  -0.201931  -0.201954  -0.201985  -0.202026  -0.202078  -0.202147  -0.202236  -0.202349  -0.202492  -0.202672  -0.202896  -0.203173  -0.203512  -0.203924  -0.204419   -0.20501  -0.205708  -0.206526  -0.207476   -0.20857  -0.209817  -0.211228  -0.212811  -0.214571  -0.216513  -0.218638  -0.220946  -0.223435  -0.226099  -0.228932  -0.231924  -0.235067  -0.238347  -0.241753  -0.245271  -0.248889  -0.252592  -0.256368  -0.260202  -0.264082  -0.267997  -0.271935  -0.275885  -0.279838  -0.283784  -0.287716  -0.291625  -0.295506     -0.299352 | 0.665714  0.665716  0.665719  0.665722  0.665726  0.665731  0.665739  0.665748  0.665759  0.665774  0.665792  0.665815  0.665842  0.665876  0.665918  0.665967  0.666025  0.666094  0.666175  0.666268  0.666376  0.666498  0.666636  0.666791  0.666964  0.667154  0.667364  0.667592  0.667838  0.668104  0.668387  0.668689  0.669008  0.669344  0.669695  0.670061  0.670441  0.670834  0.671239  0.671654  0.672079  0.672513  0.672953  0.673401  0.673854  0.674311  0.674772  0.675236  0.675702      0.676169 | 100.020773 | 100.024803  100.034165  100.046736  100.063489  100.085652  100.114751   100.15267  100.201705  100.264622  100.344722  100.445888  100.572641  100.730175  100.924379  101.161848   101.44987  101.796385  102.209933  102.699564  103.274741   103.94521  104.720867  105.611604  106.627167  107.776999  109.070102  110.514912  112.119189   113.88993  115.833305  117.954613  120.258267  122.747793  125.425849  128.294257  131.354051   134.60553  138.048308  141.681382  145.503185  149.511645  153.704238  158.078039  162.629761  167.355802  172.252271  177.315024  182.539685  187.921667          193.456187 | 196.256068 | 0.720747  0.720765  0.720789  0.720821  0.720863  0.720918   0.72099  0.721083  0.721203  0.721355  0.721547  0.721788  0.722086  0.722454  0.722903  0.723446  0.724097  0.724873  0.725788  0.726859  0.728101  0.729531  0.731163   0.73301  0.735085  0.737399  0.739959  0.742771   0.74584  0.749166  0.752747  0.756581   0.76066  0.764978  0.769525   0.77429   0.77926  0.784423  0.789765  0.795272   0.80093  0.806724  0.812641  0.818667  0.824787   0.83099  0.837263  0.843594  0.849973       0.856387 | 2.591389  2.591492  2.591631  2.591815  2.592059  2.592379  2.592796  2.593336  2.594027  2.594908  2.596019   2.59741  2.599137  2.601264  2.603861  2.607006  2.610781  2.615276  2.620583  2.626797  2.634011   2.64232  2.651812  2.662571  2.674673  2.688183  2.703156  2.719633  2.737644  2.757203  2.778311  2.800956  2.825115  2.850751  2.877818  2.906262  2.936022  2.967029  2.999211  3.032493  3.066797  3.102046   3.13816  3.175064   3.21268  3.250934  3.289755  3.329073  3.368821        3.408936 | 100.020773  100.028733  100.039466  100.053832   100.07292  100.098091  100.131036  100.173827   100.22898  100.299511  100.388999  100.501632  100.642259  100.816415  101.030345  101.290994  101.605989  101.983589  102.432615  102.962355  103.582448  104.302755  105.133212  106.083681  107.163794  108.382813  109.749489  111.271949  112.957593  114.813018  116.843968  119.055303  121.450988  124.034109  126.806895  129.770763  132.926362  136.273633  139.811866  143.539759  147.455479  151.556714  155.840725  160.304395  164.944271  169.756596  174.737346  179.882254  185.186834  190.646398  196.256068 |  100.0 |   0.020773 | 196.256068 |
    | 15:30 | 3.718184  3.718319  3.718502  3.718746   3.71907  3.719496  3.720054  3.720777  3.721708  3.722896  3.724403  3.726296  3.728657  3.731576  3.735156  3.739511  3.744764  3.751047  3.758502  3.767274  3.777511  3.789362  3.802974  3.818486  3.836027  3.855717  3.877658  3.901935  3.928616  3.957747  3.989353   4.02344  4.059993  4.098978  4.140345  4.184026  4.229943  4.278002  4.328104  4.380141  4.433999   4.48956  4.546707  4.605319  4.665277  4.726463  4.788762  4.852061  4.916251             4.981226 | 124.897211  124.904286  124.913813  124.926548  124.943447  124.965705  124.994803  125.032553  125.081154  125.143239  125.221926  125.320863  125.444266  125.596943  125.784308  126.012373  126.287727  126.617482  127.009212  127.470854    128.0106   128.63677  129.357671  130.181448  131.115938  132.168528  133.346022  134.654529  136.099368  137.684999   139.41498  141.291943  143.317603  145.492781  147.817446  150.290772  152.911207  155.676545  158.584001  161.630291  164.811705  168.124175  171.563341  175.124612  178.803212  182.594231  186.492658  190.493416  194.591388  198.781438 |  52.181838   52.183194   52.185019    52.18746   52.190698   52.194962   52.200537   52.207768   52.217077   52.228965   52.244029   52.262963   52.286569   52.315761   52.351563   52.395109   52.447636   52.510472   52.585019   52.672736   52.775108   52.893623   53.029741   53.184857   53.360273   53.557171   53.776579   54.019354   54.286162   54.577468    54.89353   55.234399   55.599929   55.989781   56.403448   56.840263   57.299425   57.780022   58.281043    58.80141   59.339987   59.895605   60.467072   61.053192   61.652772   62.264634   62.887623   63.520613   64.162511     64.812258 |  1.08844  1.088461   1.08849  1.088527  1.088578  1.088644   1.08873  1.088842  1.088987  1.089171  1.089404  1.089698  1.090064  1.090516   1.09107  1.091744  1.092557  1.093529  1.094681  1.096036  1.097616  1.099442  1.101538  1.103923  1.106616  1.109633   1.11299  1.116695  1.120759  1.125184  1.129973  1.135123  1.140629  1.146482  1.152671  1.159184  1.166005  1.173117  1.180502  1.188142  1.196016  1.204106  1.212391  1.220852  1.229469  1.238225  1.247101   1.25608  1.265145  1.274281 | 1.359157  1.359156  1.359154  1.359151  1.359148  1.359143  1.359138   1.35913  1.359121  1.359108  1.359093  1.359074   1.35905   1.35902  1.358983  1.358939  1.358886  1.358822  1.358747  1.358659  1.358557   1.35844  1.358306  1.358155  1.357985  1.357797   1.35759  1.357363  1.357118  1.356855  1.356574  1.356276  1.355963  1.355636  1.355296  1.354946  1.354586  1.354219  1.353846  1.353469  1.353089  1.352708  1.352327  1.351948  1.351572  1.351198   1.35083  1.350467   1.35011          1.349759 | 0.536186  0.536197  0.536211  0.536231  0.536256   0.53629  0.536334  0.536391  0.536464  0.536558  0.536676  0.536825  0.537011   0.53724  0.537521  0.537862  0.538273  0.538763  0.539343  0.540023  0.540814  0.541726  0.542768   0.54395  0.545277  0.546757  0.548394  0.550189  0.552143  0.554256  0.556522  0.558938  0.561496  0.564187  0.567003  0.569932  0.572964  0.576087  0.579289  0.582558  0.585882   0.58925  0.592651  0.596075  0.599513  0.602955  0.606392  0.609819  0.613227       0.61661 | -0.201905   -0.20192  -0.201939  -0.201964  -0.201998  -0.202042    -0.2021  -0.202174  -0.202269   -0.20239  -0.202543  -0.202734  -0.202972  -0.203263  -0.203619  -0.204049  -0.204565  -0.205177  -0.205899   -0.20674  -0.207714  -0.208832  -0.210102  -0.211535  -0.213138  -0.214915  -0.216872  -0.219008  -0.221323  -0.223814  -0.226476  -0.229303  -0.232284  -0.235411  -0.238672  -0.242054  -0.245546  -0.249133  -0.252803  -0.256543   -0.26034  -0.264182  -0.268057  -0.271954  -0.275863  -0.279775   -0.28368  -0.287571  -0.291441     -0.295282 | 0.665713  0.665715  0.665717   0.66572  0.665723  0.665728  0.665734  0.665741  0.665751  0.665763  0.665779  0.665798  0.665822  0.665851  0.665887   0.66593  0.665981  0.666042  0.666113  0.666195  0.666291    0.6664  0.666525  0.666665  0.666822  0.666996  0.667188  0.667398  0.667626  0.667873  0.668138  0.668421  0.668722  0.669039  0.669373  0.669722  0.670085  0.670462  0.670851  0.671252  0.671663  0.672083  0.672512  0.672948  0.673391  0.673839  0.674291  0.674747  0.675206      0.675666 | 100.016892 |   100.0202  100.027901   100.03827  100.052133  100.070529  100.094759  100.126437   100.16754  100.220462  100.288078  100.373791  100.481589  100.616084  100.782547  100.986924  101.235837  101.536566  101.897005    102.3256  102.831265  103.423276  104.111145  104.904493  105.812902   106.84577  108.012177  109.320747  110.779534  112.395925  114.176556  116.127259  118.253027  120.557995  123.045453  125.717857  128.576871  131.623405   134.85767  138.279231  141.887061  145.679602  149.654815  153.810233  158.143005  162.649938  167.327536  172.172029  177.179406  182.345429           187.66566 | 190.358854 | 0.720738  0.720753  0.720773  0.720799  0.720834   0.72088   0.72094  0.721019  0.721119  0.721248   0.72141  0.721615   0.72187  0.722186  0.722572  0.723042  0.723609  0.724286  0.725089  0.726034  0.727135  0.728408  0.729868   0.73153  0.733406  0.735508  0.737845  0.740425  0.743254  0.746333  0.749665  0.753247  0.757075  0.761144  0.765445  0.769969  0.774705  0.779642  0.784766  0.790064  0.795524   0.80113  0.806869  0.812728  0.818693  0.824751   0.83089  0.837097  0.843362       0.849672 | 2.591339  2.591423  2.591538   2.59169  2.591893  2.592159  2.592508   2.59296  2.593542  2.594285  2.595227  2.596411  2.597886  2.599711  2.601948  2.604669  2.607951  2.611876  2.616532  2.622008  2.628398  2.635792  2.644281   2.65395  2.664878  2.677138   2.69079  2.705885   2.72246  2.740542  2.760143   2.78126  2.803882  2.827982  2.853524  2.880464  2.908746   2.93831  2.969091  3.001017  3.034015  3.068011  3.102929  3.138692  3.175226  3.212457  3.250313  3.288725  3.327624        3.366947 | 100.016892  100.023427  100.032265  100.044131  100.059945  100.080866  100.108336  100.144135  100.190433   100.24985  100.325508   100.42109  100.540886  100.689829  100.873524  101.098258  101.370989  101.699317  102.091433  102.556043  103.102273  103.739552  104.477486  105.325715  106.293769  107.390926  108.626071  110.007575  111.543183  113.239926   115.10405  117.140976   119.35527   121.75064  124.329953  127.095258  130.047828  133.188211  136.516278  140.031283   143.73192  147.616376  151.682382   155.92727  160.348005  164.941234  169.703316  174.630354  179.718216  184.962561  190.358854 |  100.0 |   0.016892 | 190.358854 |
    | 16:00 | 3.718118  3.718229  3.718379  3.718581  3.718849  3.719204  3.719669  3.720275  3.721056  3.722059  3.723333  3.724941  3.726954  3.729453  3.732531   3.73629  3.740844  3.746317  3.752838  3.760548  3.769588  3.780102  3.792236  3.806129  3.821916   3.83972  3.859653  3.881812  3.906276  3.933105  3.962342  3.994009  4.028106  4.064616  4.103505   4.14472  4.188195  4.233852    4.2816   4.33134  4.382969  4.436374  4.491442  4.548059  4.606107  4.665471  4.726037  4.787694  4.850331             4.913842 | 124.893752  124.899564  124.907413  124.917938  124.931947  124.950457  124.974733  125.006332  125.047153  125.099484  125.166049  125.250058  125.355244  125.485898  125.646887   125.84366   126.08224  126.369189   126.71156  127.116822  127.592767    128.1474  128.788813  129.525049  130.363958  131.313063  132.379419  133.569497  134.889077  136.343164  137.935926  139.670655  141.549756  143.574754  145.746317  148.064305  150.527821  153.135279  155.884473  158.772653  161.796592  164.952665  168.236904  171.645071  175.172704   178.81517  182.567706  186.425455  190.383496   194.43687 |  52.181175   52.182289   52.183793    52.18581   52.188494   52.192041   52.196692   52.202746   52.210565   52.220587   52.233332   52.249413   52.269541   52.294531   52.325307   52.362899   52.408442   52.463166   52.528385   52.605479   52.695876   52.801023   52.922361   53.061295   53.219162   53.397203   53.596533   53.818119   54.062756   54.331053   54.623425   54.940086   55.281056    55.64616   56.035046   56.447197   56.881952   57.338517   57.815997   58.313404   58.829685   59.363737   59.914423   60.480587   61.061069   61.654711   62.260374   62.876936   63.503306     64.138424 |  1.08843  1.088447  1.088471  1.088502  1.088543  1.088598  1.088671  1.088764  1.088886  1.089041  1.089239  1.089488    1.0898  1.090187  1.090664  1.091246  1.091951  1.092798  1.093806  1.094997  1.096393  1.098015  1.099885  1.102023   1.10445  1.107182  1.110236  1.113624  1.117357  1.121442  1.125881  1.130677  1.135827  1.141324   1.14716  1.153325  1.159805  1.166585  1.173648  1.180978  1.188556  1.196363  1.204379  1.212586  1.220965  1.229497  1.238164  1.246949  1.255835  1.264805 | 1.359158  1.359156  1.359155  1.359153   1.35915  1.359146  1.359142  1.359135  1.359127  1.359117  1.359104  1.359088  1.359067  1.359041   1.35901  1.358972  1.358925   1.35887  1.358804  1.358726  1.358636  1.358531  1.358411  1.358275  1.358121   1.35795   1.35776  1.357551  1.357323  1.357077  1.356814  1.356533  1.356236  1.355924  1.355598  1.355261  1.354913  1.354556  1.354192  1.353822  1.353449  1.353073  1.352696  1.352319  1.351943   1.35157  1.351201  1.350836  1.350477          1.350123 | 0.536181   0.53619  0.536202  0.536218  0.536239  0.536267  0.536304  0.536351  0.536413  0.536492  0.536592  0.536719  0.536877  0.537073  0.537315   0.53761  0.537966  0.538394  0.538902  0.539501  0.540202  0.541014  0.541946  0.543009   0.54421  0.545556  0.547052  0.548702  0.550508   0.55247  0.554587  0.556854  0.559266  0.561817  0.564497  0.567298   0.57021   0.57322  0.576319  0.579494  0.582734  0.586027  0.589363  0.592731  0.596121  0.599524  0.602931  0.606334  0.609726        0.6131 | -0.201898   -0.20191  -0.201926  -0.201947  -0.201975  -0.202012   -0.20206  -0.202122  -0.202202  -0.202305  -0.202434  -0.202597    -0.2028   -0.20305  -0.203357   -0.20373  -0.204179  -0.204714  -0.205348  -0.206092  -0.206957  -0.207954  -0.209095  -0.210387  -0.211841  -0.213463  -0.215257  -0.217226  -0.219372  -0.221693  -0.224186  -0.226845  -0.229664  -0.232634  -0.235745  -0.238986  -0.242346   -0.24581  -0.249368  -0.253005   -0.25671  -0.260471  -0.264275  -0.268111  -0.271968  -0.275837  -0.279708  -0.283574  -0.287425     -0.291256 | 0.665713  0.665714  0.665716  0.665718  0.665721  0.665725  0.665729  0.665736  0.665744  0.665755  0.665768  0.665784  0.665805   0.66583  0.665861  0.665898  0.665942  0.665995  0.666058  0.666131  0.666216  0.666314  0.666425  0.666552  0.666694  0.666852  0.667028  0.667221  0.667432  0.667661  0.667907  0.668172  0.668454  0.668754  0.669069  0.669401  0.669747  0.670108  0.670481  0.670867  0.671264  0.671671  0.672087  0.672512  0.672943  0.673381  0.673824  0.674271  0.674722      0.675176 | 100.013722 | 100.016435  100.022761  100.031305  100.042761   100.05801   100.07816  100.104587   100.13899  100.183437  100.240424  100.312923  100.404439  100.519056  100.661468  100.837015  101.051687  101.312121  101.625579  101.999895  102.443416  102.964909  103.573458  104.278342  105.088907  106.014427  107.063968  108.246253  109.569544  111.041533  112.669247  114.458983  116.416257  118.545769  120.851399  123.336214  126.002486  128.851728  131.884739  135.101651   138.50198  142.084685  145.848217   149.79057  153.909331  158.201722  162.664641  167.294693  172.088225   177.04135          182.149965 | 184.737719 | 0.720731  0.720743  0.720759  0.720781   0.72081  0.720848  0.720899  0.720964  0.721049  0.721157  0.721295  0.721469  0.721686  0.721956  0.722289  0.722694  0.723186  0.723776  0.724479   0.72531  0.726283  0.727413  0.728716  0.730206  0.731897    0.7338  0.735927  0.738287  0.740886  0.743729  0.746818  0.750155  0.753736  0.757558  0.761615  0.765899  0.770399  0.775107   0.78001  0.785096  0.790352  0.795764   0.80132  0.807005  0.812807  0.818712  0.824709  0.830785  0.836928       0.843127 | 2.591297  2.591367  2.591461  2.591587  2.591755  2.591977  2.592267  2.592646  2.593135  2.593761  2.594558  2.595564  2.596822  2.598384  2.600307  2.602657  2.605502  2.608921  2.612995  2.617809  2.623453  2.630015  2.637584  2.646248  2.656088  2.667178  2.679588  2.693373  2.708582  2.725248  2.743393  2.763028  2.784149  2.806741  2.830778  2.856224  2.883033  2.911152  2.940522  2.971077   3.00275  3.035469  3.069162  3.103754  3.139172  3.175344  3.212198  3.249664  3.287675        3.326165 | 100.013722  100.019081   100.02635  100.036139  100.049225   100.06659  100.089464   100.11937  100.158177  100.208152  100.272014  100.352988  100.454856  100.581998  100.739425    100.9328  101.168441  101.453305  101.794957  102.201508  102.681541  103.244011  103.898136  104.653266  105.518748  106.503789   107.61732  108.867866  110.263432  111.811399  113.518448  115.390496  117.432656  119.649221  122.043659  124.618632  127.376022  130.316973  133.441936  136.750721  140.242551  143.916116  147.769626  151.800858  156.007205  160.385718  164.933138  169.645935  174.520331  179.552327  184.737719 |  100.0 |   0.013722 | 184.737719 |
    | 16:30 | 3.718063  3.718155  3.718279  3.718445  3.718667  3.718962  3.719349  3.719855  3.720511  3.721355  3.722431  3.723795  3.725508  3.727643  3.730283  3.733521  3.737461  3.742216  3.747908  3.754666  3.762628  3.771931  3.782717  3.795126  3.809292  3.825344  3.843399  3.863564  3.885928  3.910565  3.937531  3.966863  3.998578  4.032675  4.069135   4.10792  4.148978  4.192244   4.23764  4.285077  4.334459  4.385683  4.438641  4.493225   4.54932  4.606814  4.665596  4.725555  4.786582             4.848571 | 124.890926  124.895694  124.902154  124.910842   124.92244  124.937812  124.958037   124.98445  125.018685  125.062724  125.118942  125.190152  125.279649  125.391244  125.529295  125.698719  125.904991  126.154133  126.452673  126.807592  127.226248  127.716283   128.28551  128.941795  129.692917  130.546442  131.509583  132.589081  133.791085  135.121063  136.583722  138.182955  139.921808  141.802474  143.826299  145.993815  148.304781  150.758237   153.35257  156.085579  158.954547  161.956307  165.087314  168.343702  171.721345  175.215908  178.822894  182.537682  186.355561   190.27176 |  52.180633   52.181547   52.182785    52.18445   52.186673   52.189618   52.193493   52.198554   52.205112   52.213547   52.224313   52.237946   52.255076   52.276428   52.302829   52.335212   52.374611   52.422158   52.479076   52.546663   52.626276    52.71931   52.827173    52.95126   53.092922    53.25344   53.433994   53.635638   53.859277   54.105649   54.375311    54.66863   54.985781   55.326752   55.691346   56.079197   56.489783   56.922445   57.376401    57.85077   58.344588   58.856828   59.386415   59.932245   60.493197   61.068144   61.655964   62.255551   62.865819     63.485709 | 1.088422  1.088436  1.088455  1.088481  1.088515  1.088561  1.088621  1.088699  1.088801  1.088932  1.089099   1.08931  1.089576  1.089907  1.090316  1.090817  1.091427  1.092163  1.093044  1.094089  1.095319  1.096755  1.098418   1.10033   1.10251  1.104976  1.107746  1.110834  1.114253  1.118011  1.122114  1.126567  1.131368  1.136516  1.142003  1.147821   1.15396  1.160407  1.167146  1.174162  1.181437  1.188954  1.196694  1.204638  1.212769  1.221067  1.229515  1.238095  1.246791  1.255586 | 1.359158  1.359157  1.359156  1.359154  1.359152  1.359149  1.359145   1.35914  1.359133  1.359124  1.359113  1.359099  1.359082   1.35906  1.359033     1.359   1.35896  1.358911  1.358854  1.358786  1.358706  1.358613  1.358505  1.358383  1.358244  1.358088  1.357914  1.357723  1.357512  1.357284  1.357037  1.356773  1.356493  1.356196  1.355886  1.355562  1.355226  1.354881  1.354527  1.354166    1.3538  1.353429  1.353057  1.352683   1.35231  1.351939   1.35157  1.351204  1.350843          1.350486 | 0.536177  0.536184  0.536194  0.536207  0.536224  0.536248  0.536278  0.536318   0.53637  0.536436  0.536521  0.536629  0.536763  0.536931  0.537139  0.537393  0.537702  0.538074  0.538518  0.539044  0.539663  0.540383  0.541215  0.542168   0.54325   0.54447  0.545833  0.547344  0.549007  0.550823  0.552792  0.554912  0.557179  0.559587   0.56213  0.564799  0.567585  0.570479  0.573468  0.576543  0.579691  0.582902  0.586166   0.58947  0.592805  0.596162  0.599531  0.602904  0.606273      0.609631 | -0.201893  -0.201902  -0.201915  -0.201933  -0.201956  -0.201986  -0.202027  -0.202079  -0.202146  -0.202232  -0.202342  -0.202481  -0.202654  -0.202868  -0.203132  -0.203454  -0.203844  -0.204311  -0.204867  -0.205522  -0.206287  -0.207175  -0.208195  -0.209358  -0.210672  -0.212146  -0.213785  -0.215594  -0.217576   -0.21973  -0.222056   -0.22455  -0.227206  -0.230017  -0.232976  -0.236071  -0.239292  -0.242628  -0.246066  -0.249594  -0.253199   -0.25687  -0.260595  -0.264361  -0.268159  -0.271977  -0.275807  -0.279639  -0.283465     -0.287278 | 0.665712  0.665713  0.665714  0.665716  0.665719  0.665722  0.665726  0.665731  0.665738  0.665747  0.665758  0.665772   0.66579  0.665811  0.665838   0.66587  0.665909  0.665955   0.66601  0.666075   0.66615  0.666237  0.666337   0.66645  0.666578  0.666722  0.666882  0.667059  0.667253  0.667465  0.667694  0.667941  0.668205  0.668487  0.668785  0.669099  0.669428  0.669772   0.67013    0.6705  0.670882  0.671275  0.671678   0.67209   0.67251  0.672937   0.67337  0.673808  0.674251      0.674697 | 100.011136 | 100.013359  100.018549  100.025581  100.035037  100.047661  100.064395  100.086412  100.115166   100.15244  100.200393  100.261615  100.339178   100.43668   100.55829  100.708782  100.893551   101.11862  101.390633   101.71682  102.104949  102.563255  103.100354  103.725133  104.446639  105.273944   106.21602  107.281608  108.479091  109.816381   111.30082    112.9391  114.737194  116.700318    118.8329   121.13858  123.620217   126.27991  129.119037   132.13829  135.337731  138.716836  142.274548  146.009329  149.919209  154.001826  158.254475  162.674137  167.257518  172.001076          176.901041 | 179.384754 | 0.720725  0.720735  0.720748  0.720766   0.72079  0.720822  0.720864  0.720919   0.72099  0.721081  0.721197  0.721345   0.72153  0.721761  0.722046  0.722396  0.722821  0.723334  0.723948  0.724676  0.725534  0.726535  0.727694  0.729026  0.730545  0.732264  0.734193  0.736344  0.738724  0.741341  0.744197  0.747295  0.750636  0.754215  0.758031  0.762075   0.76634  0.770818  0.775497  0.780367  0.785414  0.790627  0.795993  0.801499  0.807131  0.812877  0.818725  0.824661  0.830675       0.836755 | 2.591263   2.59132  2.591398  2.591502  2.591641  2.591825  2.592067  2.592384  2.592794  2.593321  2.593994  2.594847  2.595918  2.597252  2.598902  2.600926  2.603388  2.606359  2.609915  2.614136  2.619108  2.624915  2.631646  2.639387   2.64822  2.658223  2.669469  2.682021  2.695933  2.711247  2.727996  2.746198   2.76586  2.786978  2.809535  2.833505  2.858851  2.885528  2.913484  2.942659  2.972992  3.004414  3.036858  3.070251  3.104524  3.139604   3.17542  3.211905  3.248989        3.286608 | 100.011136  100.015526  100.021498  100.029564  100.040378  100.054773  100.073794  100.098743  100.131224  100.173194  100.227012  100.295497   100.38197  100.490305  100.624964  100.791028  100.994206  101.240837  101.537868  101.892814  102.313698  102.808973   103.38742  104.058041   104.82993  105.712146  106.713582  107.842831  109.108072  110.516958  112.076526  113.793124  115.672356  117.719048  119.937236  122.330163  124.900302   127.64938   130.57842  133.687787  136.977233  140.445956  144.092645  147.915532   151.91244  156.080824  160.417813  164.920242  169.584685  174.407481  179.384754 |  100.0 |   0.011136 | 179.384754 |
    | 17:00 | 3.718019  3.718094  3.718196  3.718333  3.718517  3.718761  3.719084  3.719506  3.720055  3.720764  3.721672  3.722826  3.724281  3.726102  3.728362  3.731146  3.734547  3.738669  3.743624  3.749535  3.756529  3.764739  3.774301  3.785353  3.798029   3.81246  3.828766  3.847062  3.867446  3.890004  3.914802  3.941893  3.971309  4.003062   4.03715  4.073551  4.112226  4.153124  4.196178  4.241312  4.288438  4.337464  4.388288  4.440806  4.494911  4.550494  4.607445  4.665656   4.72502              4.78543 | 124.888618  124.892527  124.897837  124.904999   124.91459   124.92734  124.944167  124.966214  124.994883  125.031888  125.079291   125.13955  125.215564  125.310705   125.42886  125.574442  125.752409  125.968255  126.227989  126.538096  126.905477  127.337374  127.841272  128.424792  129.095571  129.861133  130.728762  131.705374  132.797404  134.010697  135.350423  136.821008  138.426085  140.168471  142.050161  144.072338  146.235406  148.539034  150.982207  153.563289  156.280088   159.12992  162.109679  165.215899   168.44481  171.792397  175.254446   178.82659  182.504344  186.283138 |  52.180191    52.18094   52.181958   52.183331   52.185168   52.187612   52.190836    52.19506   52.200552   52.207641    52.21672   52.228259   52.242811   52.261019   52.283623    52.31146    52.34547   52.386688   52.436245   52.495351   52.565288   52.647388   52.743013   52.853532   52.980293   53.124596   53.287665   53.470623   53.674464   53.900037   54.148024   54.418932   54.713085   55.030623   55.371502   55.735507    56.12226   56.531236   56.961778   57.413115   57.884383   58.374638    58.88288   59.408063   59.949113   60.504941   61.074454   61.656563   62.250196     62.854298 | 1.088415  1.088426  1.088442  1.088463  1.088492   1.08853   1.08858  1.088645   1.08873   1.08884  1.088981   1.08916  1.089386  1.089668  1.090018  1.090449  1.090976  1.091614  1.092381  1.093295  1.094376  1.095645   1.09712  1.098825  1.100777  1.102997  1.105502  1.108308  1.111428  1.114875  1.118656  1.122777  1.127241  1.132046   1.13719  1.142666  1.148466  1.154579  1.160992   1.16769  1.174658  1.181879  1.189335  1.197009  1.204883  1.212938  1.221158  1.229523  1.238019  1.246627 | 1.359159  1.359158  1.359157  1.359155  1.359153  1.359151  1.359148  1.359143  1.359138   1.35913  1.359121  1.359109  1.359094  1.359076  1.359053  1.359024  1.358989  1.358947  1.358897  1.358837  1.358767  1.358684  1.358589  1.358479  1.358354  1.358213  1.358055  1.357879  1.357686  1.357474  1.357245  1.356998  1.356734  1.356453  1.356158  1.355848  1.355526  1.355193   1.35485  1.354498  1.354141  1.353778  1.353411  1.353042  1.352672  1.352303  1.351935  1.351569  1.351207          1.350849 | 0.536173  0.536179  0.536187  0.536198  0.536213  0.536232  0.536257  0.536291  0.536334   0.53639  0.536461  0.536552  0.536667   0.53681  0.536988  0.537206  0.537473  0.537796  0.538184  0.538645  0.539189  0.539826  0.540566  0.541418   0.54239  0.543491  0.544729  0.546108  0.547634  0.549308  0.551134  0.553109  0.555231  0.557497  0.559901  0.562436  0.565093  0.567865   0.57074  0.573708  0.576759  0.579881  0.583064  0.586298  0.589571  0.592874  0.596199  0.599534  0.602874       0.60621 | -0.201888  -0.201896  -0.201907  -0.201921   -0.20194  -0.201966  -0.201999  -0.202043  -0.202099  -0.202172  -0.202265  -0.202382  -0.202529  -0.202713  -0.202939  -0.203217  -0.203554  -0.203961  -0.204447  -0.205022  -0.205698  -0.206485  -0.207395  -0.208437  -0.209621  -0.210956  -0.212449  -0.214105  -0.215928  -0.217921  -0.220083  -0.222413  -0.224906  -0.227558  -0.230362  -0.233308  -0.236387  -0.239588  -0.242901  -0.246312  -0.249811  -0.253385  -0.257022  -0.260712  -0.264442  -0.268202  -0.271983  -0.275774  -0.279568     -0.283356 | 0.665712  0.665712  0.665714  0.665715  0.665717   0.66572  0.665723  0.665728  0.665734  0.665741   0.66575  0.665762  0.665777  0.665796  0.665819  0.665846   0.66588   0.66592  0.665968  0.666025  0.666092  0.666169  0.666258  0.666359  0.666475  0.666605  0.666751  0.666912   0.66709  0.667285  0.667497  0.667727  0.667974  0.668238  0.668518  0.668815  0.669127  0.669454  0.669796  0.670151  0.670518  0.670897  0.671286  0.671685  0.672093  0.672509  0.672931  0.673359  0.673793      0.674231 | 100.009028 | 100.010847  100.015102  100.020881  100.028677  100.039116  100.052995  100.071313  100.095313  100.126525  100.166816  100.218433   100.28406   100.36686   100.47052   100.59929   100.75801  100.952121  101.187674  101.471304  101.810202  102.212059  102.684992   103.23746  103.878152  104.615878  105.459445  106.417528  107.498551  108.710566  110.061154  111.557322  113.205442   115.01118   116.97947  119.114485   121.41964  123.897599  126.550303  129.378999  132.384288  135.566164  138.924067   142.45693  146.163227   150.04102  154.088002  158.301534  162.678681  167.216244          171.910786 | 174.291992 |  0.72072  0.720728  0.720739  0.720754  0.720774  0.720801  0.720835  0.720881   0.72094  0.721017  0.721115   0.72124  0.721397  0.721594  0.721838  0.722139  0.722506  0.722951  0.723486  0.724123  0.724877  0.725761   0.72679  0.727977  0.729338  0.730885   0.73263  0.734584  0.736758  0.739158   0.74179  0.744658  0.747764  0.751107  0.754684  0.758492  0.762523   0.76677  0.771224  0.775875  0.780711  0.785721  0.790892  0.796212  0.801669  0.807249   0.81294  0.818731  0.824608       0.830562 | 2.591236  2.591283  2.591346  2.591432  2.591547    2.5917  2.591901  2.592165  2.592509  2.592952   2.59352  2.594241  2.595151  2.596289  2.597702  2.599442  2.601567  2.604143  2.607239  2.610932    2.6153  2.620426  2.626395  2.633291  2.641197  2.650194  2.660356   2.67175  2.684437  2.698467   2.71388  2.730704  2.748955  2.768638  2.789748  2.812266  2.836165  2.861409  2.887952  2.915743  2.944725  2.974836  3.006011  3.038183  3.071283  3.105241  3.139988  3.175457  3.211579        3.248289 | 100.009028  100.012621  100.017521  100.024159  100.033086  100.045004    100.0608  100.081585  100.108732  100.143927  100.189212   100.24704   100.32032  100.412466  100.527439  100.669775  100.844613    101.0577  101.315386  101.624596  101.992787  102.427887  102.938208  103.532355  104.219111  105.007319  105.905754  106.922999  108.067324  109.346573  110.768064  112.338504  114.063926  115.949638  118.000198  120.219399  122.610275  125.175121  127.915522  130.832388  133.926005  137.196077  140.641775  144.261792  148.054383  152.017415  156.148403  160.444553  164.902789  169.519786  174.291992 |  100.0 |   0.009028 | 174.291992 |
    | 17:30 | 3.717983  3.718044  3.718128  3.718241  3.718393  3.718595  3.718863  3.719215  3.719675  3.720269  3.721034  3.722009  3.723243  3.724793  3.726724  3.729112  3.732042  3.735607  3.739912  3.745069  3.751197  3.758424  3.766879  3.776696  3.788007  3.800943  3.815629  3.832181  3.850707  3.871299  3.894038  3.918987  3.946191  3.975679  4.007462  4.041532  4.077866  4.116426  4.157159  4.199998   4.24487  4.291687   4.34036  4.390788  4.442872  4.496507  4.551586  4.608003  4.665654             4.724434 | 124.886737  124.889937  124.894297  124.900195  124.908115  124.918677  124.932659  124.951036  124.975011  125.006058  125.045966  125.096876  125.161327  125.242296  125.343232  125.468084  125.621318  125.807923    126.0334  126.303739  126.625373  127.005116  127.450086  127.967609  128.565112  129.250006  130.029561  130.910789  131.900318  133.004289  134.228255  135.577105  137.054997  138.665323  140.410684  142.292889  144.312972   146.47122  148.767221  151.199911  153.767636  156.468213  159.298998  162.256941  165.338653  168.540459   171.85845  175.288529  178.826454   182.46787 |  52.179831   52.180444    52.18128    52.18241   52.183928   52.185952   52.188631   52.192152   52.196745   52.202693   52.210337   52.220087   52.232428   52.247927   52.267243   52.291124    52.32042   52.356074   52.399122   52.450689   52.511974    52.58424   52.668793   52.766961   52.880074   53.009434   53.156291   53.321813   53.507068   53.712994   53.940385   54.189871   54.461913   54.756794   55.074619   55.415321   55.778665   56.164263   56.571587   56.999984   57.448696   57.916874   58.403595   58.907884   59.428723   59.965066   60.515857   61.080034   61.656541     62.244336 | 1.088409  1.088419  1.088432  1.088449  1.088473  1.088504  1.088546    1.0886  1.088671  1.088764  1.088882  1.089033  1.089225  1.089465  1.089764  1.090134  1.090588   1.09114  1.091807  1.092605  1.093552  1.094669  1.095975   1.09749  1.099234  1.101225  1.103484  1.106026  1.108866  1.112018  1.115491  1.119293   1.12343  1.127903  1.132711   1.13785  1.143315  1.149095  1.155181   1.16156  1.168217  1.175137  1.182304  1.189701  1.197311  1.205115  1.213096  1.221238  1.229523  1.237935 | 1.359159  1.359158  1.359157  1.359156  1.359155  1.359153   1.35915  1.359146  1.359142  1.359135  1.359128  1.359118  1.359105  1.359089  1.359069  1.359045  1.359015  1.358979  1.358935  1.358883  1.358821  1.358748  1.358663  1.358565  1.358453  1.358326  1.358182  1.358022  1.357845   1.35765  1.357437  1.357206  1.356959  1.356695  1.356415   1.35612  1.355812  1.355492   1.35516   1.35482  1.354471  1.354116  1.353756  1.353393  1.353028  1.352661  1.352295  1.351931  1.351569          1.351211 | 0.536171  0.536175  0.536182  0.536191  0.536203  0.536219   0.53624  0.536268  0.536304  0.536351  0.536411  0.536488  0.536585  0.536707  0.536859  0.537047  0.537277  0.537556  0.537893  0.538296  0.538774  0.539337  0.539992  0.540751  0.541622  0.542613  0.543732  0.544987  0.546382  0.547921  0.549607   0.55144   0.55342  0.555545  0.557809  0.560208  0.562734   0.56538  0.568136  0.570993   0.57394  0.576967  0.580064   0.58322  0.586424  0.589667  0.592939  0.596231  0.599534      0.602842 | -0.201884  -0.201891  -0.201899  -0.201911  -0.201927  -0.201948  -0.201976  -0.202013   -0.20206  -0.202121  -0.202199  -0.202298  -0.202424   -0.20258  -0.202774  -0.203013  -0.203305  -0.203658  -0.204081  -0.204585   -0.20518  -0.205877  -0.206685  -0.207617   -0.20868  -0.209885  -0.211239   -0.21275  -0.214422  -0.216258  -0.218261   -0.22043  -0.222763  -0.225256  -0.227903  -0.230698  -0.233631  -0.236694  -0.239875  -0.243165   -0.24655   -0.25002  -0.253563  -0.257168  -0.260822  -0.264517   -0.26824  -0.271984  -0.275737     -0.279493 | 0.665711  0.665712  0.665713  0.665714  0.665716  0.665718  0.665721  0.665725  0.665729  0.665736  0.665744  0.665754  0.665767  0.665782  0.665802  0.665826  0.665855   0.66589  0.665932  0.665982   0.66604  0.666109  0.666188  0.666279  0.666382    0.6665  0.666632  0.666779  0.666942  0.667121  0.667317  0.667529  0.667759  0.668006  0.668269  0.668549  0.668844  0.669155   0.66948  0.669819  0.670171  0.670535   0.67091  0.671296  0.671691  0.672095  0.672507  0.672925  0.673348      0.673777 | 100.007313 |   100.0088  100.012283  100.017028  100.023447  100.032069  100.043565  100.058786   100.07879   100.10489  100.138692  100.182145  100.237583  100.307779  100.395983  100.505965  100.642048  100.809129   101.01269  101.258794  101.554061  101.905633  102.321116  102.808504  103.376091  104.032368  104.785912  105.645263  106.618808  107.714664  108.940566  110.303771  111.810974  113.468237  115.280939  117.253747  119.390594  121.694683  124.168499  126.813833  129.631815  132.622957  135.787194  139.123934  142.632101  146.310184  150.156278  154.168126  158.343156  162.678514          167.171089 | 169.451444 | 0.720716  0.720723  0.720732  0.720744  0.720761  0.720783  0.720812   0.72085  0.720899  0.720964  0.721046  0.721152  0.721285  0.721453  0.721661  0.721919  0.722236  0.722621  0.723085  0.723642  0.724302  0.725081  0.725991  0.727047  0.728262   0.72965  0.731224  0.732995  0.734973  0.737168  0.739586  0.742234  0.745113  0.748225  0.751569  0.755143  0.758942   0.76296  0.767188  0.771618   0.77624  0.781043  0.786016  0.791146  0.796421  0.801829  0.807358  0.812996   0.81873       0.824551 | 2.591213  2.591251  2.591304  2.591374  2.591469  2.591596  2.591763  2.591984  2.592271  2.592643  2.593121   2.59373  2.594502  2.595471  2.596678  2.598171  2.600002   2.60223   2.60492  2.608142   2.61197  2.616483  2.621762  2.627889  2.634947  2.643015   2.65217  2.662483  2.674019  2.686835  2.700976   2.71648  2.733371  2.751665  2.771364   2.79246  2.814934  2.838759  2.863898  2.890306  2.917932  2.946722  2.976613  3.007544  3.039448  3.072258  3.105907  3.140329  3.175456        3.211223 | 100.007313  100.010249  100.014266  100.019723  100.027082  100.036937  100.050038  100.067331  100.089989   100.11946  100.157507  100.206259  100.268257  100.346499  100.444487  100.566257  100.716416  100.900152  101.123239  101.392026   101.71341  102.094779  102.543957  103.069113  103.678672  104.381198  105.185282  106.099423  107.131902  108.290676  109.583266   111.01667  112.597281  114.330835  116.222359  118.276158  120.495797  122.884116  125.443244  128.174631  131.079089  134.156827  137.407504  140.830274  144.423831  148.186455  152.116055  156.210207  160.466189  164.881011  169.451444 |  100.0 |   0.007313 | 169.451444 |
    | 18:00 | 3.717954  3.718004  3.718072  3.718165  3.718291  3.718458   3.71868  3.718973  3.719357  3.719855  3.720498  3.721321  3.722365  3.723682   3.72533  3.727375  3.729893   3.73297  3.736702   3.74119  3.746548  3.752893   3.76035  3.769047  3.779113  3.790678  3.803866  3.818798  3.835586  3.854331  3.875121  3.898031  3.923118  3.950425  3.979976  4.011778  4.045822  4.082084  4.120523  4.161086   4.20371  4.248318  4.294828   4.34315  4.393188  4.444843  4.498014  4.552598  4.608492             4.665593 | 124.885205  124.887822  124.891398  124.896248  124.902782   124.91152  124.923123   124.93842  124.958441  124.984453  125.017999  125.060942  125.115498  125.184286  125.270357  125.377228  125.508906    125.6699  125.865222  126.100373  126.381315   126.71442  127.106409  127.564272  128.095173  128.706343  129.404969  130.198074  131.092403  132.094306  133.209643  134.443689  135.801062  137.285671   138.90068  140.648486  142.530729  144.548303  146.701388  148.989497  151.411525  153.965804  156.650165  159.461999  162.398315    165.4558   168.63087  171.919717  175.318359  178.822672 |  52.179537   52.180039   52.180724   52.181654   52.182906    52.18458   52.186804   52.189735   52.193571   52.198554    52.20498   52.213206   52.223653   52.236823   52.253298   52.273746    52.29893   52.329705   52.367017   52.411902   52.465477   52.528929     52.6035   52.690468    52.79113   52.906775   53.038659   53.187982   53.355862   53.543311   53.751213   53.980309   54.231183   54.504251   54.799759   55.117781   55.458224   55.820839   56.205229   56.610864   57.037098   57.483181   57.948281   58.431498   58.931878   59.448431   59.980142   60.525981   61.084918     61.655928 | 1.088405  1.088412  1.088423  1.088437  1.088457  1.088483  1.088517  1.088563  1.088622  1.088699  1.088799  1.088927  1.089089  1.089293  1.089548  1.089865  1.090255  1.090732   1.09131  1.092004  1.092833  1.093814  1.094967   1.09631  1.097863  1.099645  1.101675  1.103971  1.106548  1.109421  1.112602    1.1161  1.119922  1.124073  1.128554  1.133363  1.138496  1.143948  1.149708  1.155767  1.162111  1.168728  1.175601  1.182714  1.190053  1.197598  1.205334  1.213243  1.221308  1.229514 | 1.359159  1.359159  1.359158  1.359157  1.359156  1.359154  1.359152  1.359149  1.359145   1.35914  1.359133  1.359125  1.359114    1.3591  1.359084  1.359063  1.359037  1.359005  1.358967  1.358922  1.358868  1.358803  1.358728  1.358641  1.358541  1.358427  1.358297  1.358152  1.357989   1.35781  1.357614    1.3574  1.357169  1.356921  1.356657  1.356377  1.356084  1.355777  1.355458  1.355129  1.354791  1.354445  1.354093  1.353736  1.353376  1.353014  1.352651  1.352289  1.351928           1.35157 | 0.536168  0.536172  0.536178  0.536185  0.536195  0.536208  0.536226  0.536249  0.536279  0.536318  0.536369  0.536434  0.536516   0.53662  0.536749   0.53691  0.537108   0.53735  0.537642  0.537993  0.538412  0.538906  0.539486   0.54016  0.540937  0.541827  0.542836  0.543973  0.545244  0.546653  0.548205  0.549901  0.551742  0.553727  0.555852  0.558115  0.560508  0.563025  0.565659    0.5684  0.571238  0.574164  0.577169   0.58024  0.583368  0.586544  0.589757  0.592998  0.596259      0.599531 | -0.201881  -0.201886  -0.201894  -0.201903  -0.201916  -0.201934  -0.201957  -0.201987  -0.202027  -0.202078  -0.202144  -0.202228  -0.202334  -0.202468  -0.202634  -0.202839   -0.20309  -0.203396  -0.203764  -0.204204  -0.204727  -0.205341  -0.206058  -0.206887  -0.207839  -0.208924  -0.210149  -0.211522  -0.213049  -0.214736  -0.216585  -0.218597  -0.220772  -0.223107  -0.225598   -0.22824  -0.231026  -0.233946  -0.236993  -0.240154   -0.24342   -0.24678  -0.250222  -0.253734  -0.257306  -0.260927  -0.264586  -0.268274  -0.271981     -0.275698 | 0.665711  0.665711  0.665712  0.665713  0.665715  0.665716  0.665719  0.665722  0.665726  0.665731  0.665738  0.665747  0.665757  0.665771  0.665788  0.665808  0.665833  0.665864    0.6659  0.665944  0.665995  0.666056  0.666126  0.666207    0.6663  0.666405  0.666524  0.666658  0.666806  0.666971  0.667151  0.667348  0.667561  0.667791  0.668037    0.6683  0.668578  0.668872  0.669181  0.669504  0.669841   0.67019  0.670551  0.670923  0.671306  0.671697  0.672097  0.672504  0.672918      0.673337 | 100.005918 | 100.007132  100.009981  100.013872  100.019152  100.026264  100.035775  100.048405  100.065057  100.086851  100.115169  100.151693  100.198452  100.257865  100.332788  100.426556  100.543014  100.686552  100.862115  101.075216  101.331923   101.63883  102.003023  102.432015  102.933669  103.516115  104.187643  104.956597  105.831256  106.819726  107.929825  109.168984  110.544152  112.061719  113.727459  115.546476  117.523187  119.661299  121.963816  124.433055  127.070668  129.877677  132.854512  136.001055  139.316684  142.800319  146.450463  150.265243  154.242454  158.379587          162.673863 | 164.855127 | 0.720713  0.720719  0.720726  0.720736   0.72075  0.720768  0.720792  0.720824  0.720865  0.720919  0.720988  0.721077   0.72119  0.721333  0.721511  0.721732  0.722004  0.722336  0.722739  0.723223  0.723801  0.724485  0.725288  0.726225  0.727307  0.728549  0.729964  0.731563  0.733359   0.73536  0.737575   0.74001  0.742671   0.74556  0.748678  0.752023  0.755593  0.759382  0.763386  0.767595  0.772001  0.776595  0.781365    0.7863  0.791389   0.79662  0.801981  0.807459  0.813045       0.818724 | 2.591195  2.591226  2.591269  2.591327  2.591405   2.59151  2.591649  2.591832  2.592072  2.592384  2.592786    2.5933  2.593953  2.594776  2.595806  2.597085  2.598659  2.600582  2.602914  2.605719  2.609066  2.613029  2.617686  2.623115  2.629397  2.636612  2.644837  2.654145  2.664604  2.676275  2.689212  2.703458  2.719046  2.735999  2.754329  2.774038  2.795115  2.817541  2.841289   2.86632  2.892592  2.920054  2.948652  2.978326  3.009015  3.040654   3.07318  3.106525  3.140626        3.175418 | 100.005918  100.008315  100.011604  100.016084  100.022145  100.030284  100.041137  100.055505   100.07439  100.099033  100.130952  100.171989  100.224358  100.290683  100.374049  100.478037  100.606756  100.764871  100.957612  101.190772  101.470691  101.804225  102.198688  102.661793  103.201563  103.826236   104.54416  105.363678  106.293015  107.340162  108.512772  109.818056  111.262706  112.852817   114.59384  116.490541  118.546983   120.76652  123.151808  125.704823  128.426891  131.318728  134.380478   137.61176  141.011708  144.579024  148.312011  152.208622  156.266488  160.482958  164.855127 |  100.0 |   0.005918 | 164.855127 |
    | 18:30 |  3.71793  3.717971  3.718027  3.718103  3.718206  3.718345  3.718529  3.718773  3.719093   3.71951  3.720049  3.720742  3.721625  3.722742  3.724145  3.725892  3.728053  3.730704  3.733931  3.737829  3.742502   3.74806   3.75462  3.762305  3.771239   3.78155  3.793361  3.806794  3.821965  3.838979  3.857933   3.87891   3.90198  3.927196  3.954595  3.984199  4.016012  4.050023  4.086205  4.124518   4.16491  4.207315   4.25166  4.297864  4.345838   4.39549  4.446723  4.499438  4.553535             4.608914 | 124.883958  124.886096  124.889025   124.89301  124.898393  124.905614  124.915231  124.927948  124.944644  124.966406  124.994563  125.030729  125.076836  125.135177  125.208442  125.299753   125.41269  125.551311   125.72016  125.924265  126.169119  126.460646  126.805152  127.209259  127.679823  128.223845   128.84836  129.560336  130.366551  131.273491   132.28724  133.413385  134.656936  136.022258  137.513021  139.132174  140.881926  142.763758  144.778433  146.926037  149.206014  151.617221  154.157982  156.826142  159.619131  162.534015  165.567554  168.716252  171.976401  175.344126 |  52.179298   52.179708   52.180269   52.181033   52.182065   52.183448   52.185291   52.187728   52.190927   52.195097   52.200491   52.207419    52.21625   52.227421   52.241448   52.258923    52.28053   52.307038   52.339308    52.37829   52.425017   52.480596   52.546199   52.623048   52.712394   52.815499   52.933611   53.067944   53.219649   53.389791   53.579332   53.789104   54.019799   54.271956   54.545948   54.841986   55.160117   55.500227    55.86205   56.245184   56.649098    57.07315   57.516603   57.978641   58.458384   58.954901   59.467227   59.994375   60.535345     61.089136 | 1.088401  1.088407  1.088416  1.088428  1.088444  1.088465  1.088494  1.088532  1.088581  1.088646  1.088729  1.088837  1.088974  1.089147  1.089364  1.089635   1.08997  1.090381  1.090881  1.091484  1.092207  1.093067  1.094081  1.095269  1.096648  1.098238  1.100058  1.102126  1.104457  1.107069  1.109973  1.113181  1.116702  1.120543  1.124706  1.129193  1.134002  1.139129  1.144566  1.150306  1.156337  1.162647  1.169222  1.176048  1.183109  1.190389  1.197872   1.20554  1.213378  1.221369 |  1.35916  1.359159  1.359159  1.359158  1.359157  1.359155  1.359153  1.359151  1.359148  1.359143  1.359138  1.359131  1.359122   1.35911  1.359096  1.359078  1.359056  1.359029  1.358996  1.358956  1.358909  1.358852  1.358786  1.358709  1.358619  1.358517    1.3584  1.358268  1.358121  1.357957  1.357776  1.357578  1.357363  1.357131  1.356883  1.356619   1.35634  1.356048  1.355742  1.355425  1.355098  1.354762  1.354419   1.35407  1.353716   1.35336  1.353001  1.352642  1.352283          1.351925 | 0.536166   0.53617  0.536174   0.53618  0.536188  0.536199  0.536214  0.536233  0.536258  0.536291  0.536333  0.536388  0.536458  0.536546  0.536656  0.536794  0.536963  0.537172  0.537425   0.53773  0.538096   0.53853  0.539041  0.539638   0.54033  0.541125  0.542033   0.54306  0.544214    0.5455  0.546923  0.548487  0.550192   0.55204  0.554028  0.556154  0.558414  0.560801  0.563309   0.56593  0.568656  0.571476  0.574382  0.577363  0.580409  0.583511  0.586659  0.589842  0.593053      0.596283 | -0.201879  -0.201883  -0.201889  -0.201897  -0.201908  -0.201922  -0.201941  -0.201967     -0.202  -0.202043  -0.202098  -0.202169  -0.202259  -0.202372  -0.202514   -0.20269  -0.202906   -0.20317  -0.203489  -0.203873   -0.20433  -0.204871  -0.205504  -0.206241  -0.207091  -0.208063  -0.209168  -0.210412  -0.211803  -0.213347  -0.215047  -0.216907  -0.218928  -0.221108  -0.223444  -0.225934   -0.22857  -0.231346  -0.234253  -0.237283  -0.240425  -0.243668  -0.247002  -0.250416  -0.253898  -0.257438  -0.261026  -0.264651  -0.268303     -0.271974 | 0.665711  0.665711  0.665712  0.665713  0.665714  0.665715  0.665717   0.66572  0.665723  0.665728  0.665733  0.665741   0.66575  0.665761  0.665776  0.665793  0.665815  0.665841  0.665873  0.665911  0.665956  0.666009  0.666071  0.666143  0.666226  0.666321  0.666428  0.666549  0.666684  0.666834  0.666999  0.667181  0.667378  0.667592  0.667822  0.668068   0.66833  0.668607    0.6689  0.669207  0.669528  0.669862  0.670209  0.670567  0.670936  0.671314  0.671702  0.672098  0.672501      0.672911 | 100.004785 | 100.005775  100.008102   100.01129  100.015628  100.021487  100.029346  100.039814  100.053657  100.071831  100.095522  100.126177  100.165554   100.21576  100.279297  100.359102  100.458586  100.581665  100.732787   100.91694  101.139657     101.407  101.725535  102.102279  102.544647  103.060368  103.657404  104.343842  105.127796  106.017291  107.020157  108.143924  109.395728  110.782223  112.309511  113.983089  115.807805  117.787835  119.926675  122.227146  124.691404  127.320973  130.116774  133.079164  136.207974  139.502559  142.961833  146.584315  150.368167   154.31123          158.411057 | 160.495087 | 0.720711  0.720715  0.720721  0.720729  0.720741  0.720756  0.720775  0.720802  0.720836  0.720881   0.72094  0.721015   0.72111  0.721231  0.721383  0.721571  0.721805  0.722091   0.72244  0.722861  0.723365  0.723964  0.724671  0.725499   0.72646  0.727569  0.728837  0.730278  0.731902  0.733721  0.735744  0.737978   0.74043  0.743103     0.746  0.749122  0.752467  0.756032  0.759812  0.763801  0.767991  0.772373  0.776938  0.781676  0.786574  0.791623   0.79681  0.802124  0.807553       0.813087 |  2.59118  2.591205  2.591241  2.591288  2.591353  2.591439  2.591555  2.591707  2.591907  2.592168  2.592505  2.592938   2.59349  2.594189  2.595066  2.596158  2.597509  2.599166  2.601182  2.603618  2.606538   2.61001  2.614107  2.618906  2.624484  2.630918  2.638286  2.646663  2.656118  2.666717  2.678517  2.691569  2.705912  2.721578  2.738587  2.756947   2.77666  2.797714  2.820089  2.843756  2.868678  2.894813   2.92211  2.950517  2.979976  3.010426  3.041805   3.07405  3.107097        3.140884 | 100.004785   100.00674   100.00943  100.013104  100.018089  100.024803  100.033782  100.045705  100.061426  100.082003  100.108742  100.143234    100.1874  100.243531  100.314337  100.402981  100.513119  100.648926  100.815119  101.016958  101.260249  101.551313  101.896957  102.304416  102.781283  103.335432  103.974913  104.707859  105.542369    106.4864   107.54766  108.733507  110.050859  111.506111  113.105077  114.852938   116.75421  118.812733   121.03166  123.413474   125.96001  128.672478  131.551507   134.59718  137.809077  141.186323  144.727622  148.431304  152.295361  156.317484  160.495087 |  100.0 |   0.004785 | 160.495087 |
    | 19:00 |  3.71791  3.717944   3.71799  3.718052  3.718137  3.718252  3.718404  3.718606  3.718873  3.719221  3.719673  3.720256  3.721001  3.721947  3.723139   3.72463   3.72648  3.728759  3.731544  3.734923  3.738989  3.743846  3.749603  3.756377  3.764287  3.773455  3.784004  3.796056  3.809727  3.825127  3.842358  3.861511  3.882666  3.905885  3.931218    3.9587  3.988348  4.020164  4.054135  4.090232  4.128415  4.168632  4.210817    4.2549  4.300799  4.348429  4.397699  4.448514   4.50078             4.554398 | 124.882944  124.884689  124.887086  124.890357  124.894787  124.900746  124.908707  124.919265   124.93317  124.951351  124.974952  125.005364  125.044269  125.093668  125.155926  125.233803  125.330486  125.449611  125.595281  125.772069  125.985009  126.239579   126.54166  126.897483  127.313566  127.796631   128.35351  128.991047  129.715989  130.534879   131.45395  132.479028   133.61544  134.867944  136.240662  137.737041  139.359826  141.111053   142.99205  145.003466  147.145293  149.416918  151.817166  154.344351   156.99634  159.770597  162.664246   165.67412  168.796807  172.028695 |  52.179104   52.179438   52.179898   52.180524   52.181373   52.182516   52.184041   52.186064   52.188729   52.192212   52.196734    52.20256   52.210012   52.219473   52.231394   52.246302   52.264804   52.287592   52.315443   52.349225   52.389886   52.438455    52.49603   52.563768   52.642866   52.734549   52.840044    52.96056   53.097267   53.251268   53.423581   53.615115   53.826655   54.058846   54.312183   54.587002   54.883481    55.20164   55.541345   55.902319   56.284153   56.686316   57.108172   57.548997   58.007989   58.484288   58.976986   59.485144   60.007801     60.543983 | 1.088398  1.088403   1.08841   1.08842  1.088433  1.088451  1.088474  1.088506  1.088547  1.088601  1.088671  1.088762  1.088877  1.089024  1.089209   1.08944  1.089726   1.09008  1.090511  1.091034  1.091664  1.092415  1.093306  1.094353  1.095575   1.09699  1.098617  1.100473  1.102577  1.104943  1.107587   1.11052  1.113755  1.117297  1.121155  1.125329  1.129821  1.134629  1.139747   1.14517  1.150889  1.156892  1.163167  1.169702  1.176481   1.18349  1.190713  1.198133  1.205735  1.213503 |  1.35916  1.359159  1.359159  1.359158  1.359157  1.359156  1.359155  1.359153   1.35915  1.359146  1.359142  1.359136  1.359128  1.359118  1.359106  1.359091  1.359072  1.359048   1.35902  1.358986  1.358944  1.358895  1.358837  1.358768  1.358689  1.358597  1.358493  1.358374   1.35824   1.35809  1.357924  1.357742  1.357543  1.357327  1.357095  1.356846  1.356583  1.356304  1.356013  1.355709  1.355393  1.355068  1.354735  1.354394  1.354048  1.353698  1.353344  1.352989  1.352633          1.352277 | 0.536165  0.536167  0.536171  0.536176  0.536183  0.536192  0.536204   0.53622  0.536241  0.536268  0.536304   0.53635  0.536409  0.536483  0.536577  0.536694   0.53684  0.537019  0.537238  0.537503  0.537821  0.538201   0.53865  0.539177  0.539791  0.540501  0.541314  0.542239  0.543283  0.544453  0.545754  0.547191  0.548765  0.550479  0.552333  0.554324   0.55645  0.558707  0.561088  0.563586  0.566195  0.568905  0.571707  0.574592  0.577551  0.580572  0.583648  0.586768  0.589923      0.593104 | -0.201877   -0.20188  -0.201885  -0.201892    -0.2019  -0.201912  -0.201928  -0.201949  -0.201977  -0.202013  -0.202059  -0.202119  -0.202195  -0.202291  -0.202412  -0.202562  -0.202748  -0.202975  -0.203252  -0.203586  -0.203985  -0.204459  -0.205018   -0.20567  -0.206426  -0.207296  -0.208288  -0.209412  -0.210675  -0.212083  -0.213642  -0.215355  -0.217226  -0.219254  -0.221438  -0.223776  -0.226262  -0.228892  -0.231658  -0.234552  -0.237565  -0.240687  -0.243907  -0.247216  -0.250602  -0.254055  -0.257564  -0.261119   -0.26471     -0.268328 |  0.66571  0.665711  0.665711  0.665712  0.665713  0.665714  0.665716  0.665718  0.665721  0.665725  0.665729  0.665735  0.665743  0.665753  0.665765   0.66578  0.665799  0.665822  0.665849  0.665882  0.665922  0.665968  0.666023  0.666087  0.666161  0.666246  0.666342  0.666451  0.666574   0.66671  0.666861  0.667028   0.66721  0.667408  0.667622  0.667852  0.668098  0.668359  0.668635  0.668927  0.669232  0.669551  0.669883  0.670227  0.670582  0.670947  0.671323  0.671707  0.672099      0.672498 | 100.003865 | 100.004672  100.006571   100.00918  100.012739  100.017561  100.024048  100.032713  100.044206  100.059342  100.079133  100.104825  100.137936  100.180296   100.23409  100.301896   100.38673  100.492075   100.62191  100.780735   100.97357  101.205964  101.483965  101.814098   102.20331  102.658908  103.188486  103.799832  104.500836  105.299381  106.203243  107.219984  108.356857  109.620711  111.017919   112.55431  114.235116  116.064941  118.047737  120.186802  122.484779  124.943681  127.564909  130.349292  133.297117  136.408172  139.681789   143.11688  146.711981  150.465286          154.374685 | 156.363423 | 0.720709  0.720712  0.720717  0.720724  0.720733  0.720745  0.720762  0.720784  0.720813   0.72085  0.720899  0.720962  0.721043  0.721145  0.721274  0.721435  0.721635  0.721881  0.722182  0.722547  0.722986   0.72351  0.724131  0.724861  0.725712  0.726699  0.727832  0.729126  0.730592   0.73224  0.734082  0.736125  0.738378  0.740844  0.743529  0.746434  0.749559  0.752903  0.756462  0.760232  0.764206  0.768376  0.772735  0.777271  0.781976  0.786838  0.791847  0.796991  0.802259        0.80764 | 2.591168  2.591189  2.591217  2.591257   2.59131  2.591381  2.591476  2.591603  2.591769  2.591987   2.59227  2.592634    2.5931  2.593692  2.594437  2.595369  2.596526   2.59795  2.599691  2.601802  2.604343  2.607377  2.610974  2.615205  2.620143  2.625866  2.632449  2.639967  2.648491  2.658088  2.668821  2.680744  2.693904  2.708339  2.724076  2.741134   2.75952  2.779232  2.800258  2.822577  2.846161  2.870973   2.89697  2.924103   2.95232  2.981565  3.011779  3.042901   3.07487        3.107624 | 100.003865  100.005458  100.007655  100.010666  100.014761  100.020293  100.027713  100.037595  100.050663  100.067823  100.090191  100.119139   100.15633   100.20376  100.263798  100.339232  100.433301  100.549731  100.692755  100.867133  101.078151  101.331615  101.633822  101.991522  102.411862  102.902315  103.470598  104.124576  104.872167  105.721227  106.679454  107.754282  108.952786  110.281598  111.746833  113.354035   115.10813    117.0134   119.07347  121.291308  123.669236  126.208954  128.911566   131.77762  134.807143  137.999684  141.354353  144.869865  148.544573  152.376512  156.363423 |  100.0 |   0.003865 | 156.363423 |
    | 19:30 | 3.717895  3.717922  3.717959  3.718011  3.718081  3.718175  3.718301  3.718469   3.71869  3.718981  3.719359  3.719849  3.720477  3.721276  3.722288  3.723557  3.725139  3.727094  3.729493  3.732414  3.735945   3.74018  3.745221  3.751177  3.758162  3.766293  3.775691  3.786474   3.79876  3.812661  3.828282  3.845721  3.865064  3.886385  3.909744  3.935186  3.962742  3.992425  4.024236   4.05816  4.094167  4.132216  4.172255  4.214219  4.258039  4.303636  4.350925  4.399817  4.450222             4.502045 | 124.882121  124.883544  124.885503  124.888184  124.891826  124.896739   124.90332  124.912076  124.923641   124.93881  124.958563    124.9841  125.016877  125.058637  125.111454  125.177758  125.260376  125.362555  125.487982  125.640796  125.825593  126.047409  126.311695  126.624284  126.991328  127.419236   127.91459  128.484058  129.134289  129.871816   130.70295  131.633681  132.669585  133.815742  135.076665   136.45625  137.957731  139.583664  141.335918  143.215685  145.223502  147.359282  149.622356   152.01152  154.525088  157.160943  159.916588  162.789204  165.775693  168.872725 |  52.178946   52.179219   52.179594   52.180108   52.180806   52.181748   52.183009   52.184687   52.186903   52.189809   52.193594   52.198487   52.204765   52.212764   52.222879   52.235574   52.251388   52.270939   52.294929   52.324143   52.359449   52.401795   52.452206   52.511767   52.581618   52.662935   52.756912   52.864744     52.9876   53.126607   53.282821   53.457213   53.650645   53.863855   54.097443   54.351863   54.627416    54.92425    55.24236   55.581595   55.941666   56.322159   56.722545   57.142195   57.580394   58.036358   58.509245    58.99817   59.502217     60.020451 | 1.088395    1.0884  1.088405  1.088413  1.088424  1.088439  1.088458  1.088484  1.088519  1.088564  1.088623  1.088698  1.088796   1.08892  1.089077  1.089273  1.089519  1.089821  1.090193  1.090646  1.091192  1.091848  1.092628  1.093549  1.094629  1.095885  1.097335  1.098997  1.100889  1.103028  1.105427  1.108102  1.111064  1.114323  1.117886  1.121758  1.125942  1.130438  1.135243  1.140353   1.14576  1.151457  1.157431  1.163672  1.170166  1.176899  1.183857  1.191023  1.198382  1.205919 |  1.35916   1.35916  1.359159  1.359159  1.359158  1.359157  1.359156  1.359154  1.359152  1.359149  1.359145   1.35914  1.359133  1.359125  1.359115  1.359102  1.359086  1.359065  1.359041  1.359011  1.358975  1.358932  1.358881  1.358821   1.35875  1.358669  1.358575  1.358468  1.358347  1.358211   1.35806  1.357892  1.357708  1.357508  1.357291  1.357058   1.35681  1.356547  1.356269  1.355978  1.355676  1.355362  1.355039  1.354708   1.35437  1.354027  1.353679  1.353329  1.352977          1.352624 | 0.536164  0.536166  0.536169  0.536173  0.536178  0.536186  0.536196  0.536209  0.536226  0.536249  0.536279  0.536318  0.536367   0.53643   0.53651   0.53661  0.536734  0.536888  0.537077  0.537306  0.537583  0.537914  0.538308  0.538773  0.539316  0.539947  0.540673  0.541504  0.542446  0.543507  0.544692  0.546007  0.547456  0.549041  0.550763  0.552622  0.554615  0.556741  0.558994  0.561368  0.563857  0.566453  0.569147  0.571932  0.574796  0.577732  0.580729  0.583779  0.586872      0.589998 | -0.201875  -0.201878  -0.201882  -0.201887  -0.201894  -0.201904  -0.201917  -0.201935  -0.201958  -0.201988  -0.202027  -0.202077  -0.202141  -0.202223  -0.202325  -0.202454  -0.202613  -0.202808  -0.203047  -0.203337  -0.203685    -0.2041  -0.204591  -0.205167  -0.205838  -0.206613  -0.207502  -0.208514  -0.209656  -0.210937  -0.212361  -0.213934   -0.21566  -0.217541  -0.219575  -0.221763  -0.224101  -0.226584  -0.229207  -0.231963  -0.234843  -0.237839  -0.240941  -0.244139  -0.247423  -0.250782  -0.254205  -0.257683  -0.261206     -0.264764 |  0.66571  0.665711  0.665711  0.665712  0.665712  0.665713  0.665715  0.665717  0.665719  0.665722  0.665726  0.665731  0.665738  0.665746  0.665756  0.665769  0.665785  0.665805  0.665829  0.665857  0.665892  0.665933  0.665981  0.666037  0.666103  0.666179  0.666265  0.666363  0.666474  0.666598  0.666736  0.666889  0.667056  0.667239  0.667438  0.667652  0.667882  0.668127  0.668387  0.668663  0.668953  0.669256  0.669573  0.669903  0.670244  0.670596  0.670958   0.67133  0.671711      0.672099 | 100.003119 | 100.003776  100.005324  100.007457  100.010375  100.014338  100.019686   100.02685   100.03638  100.048969  100.065481  100.086983  100.114785  100.150471  100.195943   100.25346  100.325677  100.415681  100.527023   100.66374  100.830372  101.031972  101.274089  101.562754  101.904442  102.306023  102.774694  103.317907   103.94328  104.658501  105.471229  106.388994  107.419099  108.568527  109.843855  111.251185  112.796083  114.483533  116.317907  118.302946  120.441758  122.736824   125.19002  127.802635  130.575409  133.508568   136.60186  139.854596  143.265689  146.833692          150.556831 | 152.452299 | 0.720707   0.72071  0.720714  0.720719  0.720727  0.720737  0.720751  0.720769  0.720793  0.720824  0.720865  0.720918  0.720986  0.721073  0.721182  0.721319   0.72149  0.721701   0.72196  0.722276  0.722657  0.723114  0.723658    0.7243  0.725053  0.725928  0.726939  0.728098  0.729416  0.730906  0.732578  0.734441  0.736504  0.738773  0.741254  0.743949  0.746861  0.749988   0.75333  0.756883  0.760642  0.764601  0.768751  0.773085  0.777594  0.782266  0.787092  0.792062  0.797163       0.802386 | 2.591158  2.591175  2.591198   2.59123  2.591274  2.591333  2.591412  2.591517  2.591655  2.591837  2.592074   2.59238  2.592772  2.593272  2.593905  2.594698  2.595687  2.596909  2.598409  2.600235  2.602441  2.605087  2.608237  2.611957  2.616319  2.621396  2.627262   2.63399  2.641653  2.650319  2.660054  2.670915  2.682955  2.696218  2.710737   2.72654  2.743641  2.762047  2.781753  2.802748  2.825009  2.848507  2.873207  2.899065  2.926034  2.954063  2.983097  3.013077  3.043946        3.075644 | 100.003119  100.004416  100.006209  100.008673  100.012034  100.016586   100.02271  100.030891  100.041741  100.056031  100.074717  100.098979   100.13025  100.170264   100.22109  100.285175  100.365379  100.465015  100.587868  100.738227  100.920886  101.141149  101.404814  101.718147  102.087834  102.520929  103.024779  103.606943  104.275102  105.036957   105.90013  106.872064  107.959928  109.170521  110.510205  111.984826  113.599671  115.359425  117.268147  119.329262   121.54556  123.919216  126.451802  129.144325  131.997257  135.010575  138.183798  141.516026  145.005984  148.652051  152.452299 |  100.0 |   0.003119 | 152.452299 |
    | 20:00 | 3.717882  3.717904  3.717935  3.717977  3.718034  3.718112  3.718216  3.718355  3.718539  3.718781  3.719097  3.719508  3.720036  3.720711  3.721568  3.722647  3.723996  3.725671  3.727733  3.730254  3.733313  3.736997  3.741401  3.746626  3.752779  3.759973  3.768324  3.777947  3.788958  3.801471  3.815594  3.831429  3.849067  3.868591  3.890069  3.913558  3.939099  3.966719   3.99643  4.028229  4.062099  4.098011  4.135923  4.175781  4.217525  4.261083  4.306378  4.353329  4.401848             4.451848 | 124.881454  124.882612  124.884212  124.886407  124.889397  124.893443  124.898878  124.906129  124.915737  124.928375  124.944885  124.966298  124.993871   125.02912  125.073853  125.130209  125.200684  125.288166  125.395957  125.527788  125.687833  125.880698  126.111415  126.385405  126.708445  127.086602  127.526173    128.0336  128.615382  129.277979  130.027712  130.870666  131.812595  132.858835  134.014229   135.28306  136.669004  138.175097  139.803716  141.556577  143.434741  145.438645  147.568126  149.822468  152.200443  154.700363  157.320132  160.057292  162.909078   165.87246 |  52.178818    52.17904   52.179347   52.179768   52.180341   52.181116   52.182158   52.183547   52.185388    52.18781   52.190973   52.195076   52.200358   52.207111   52.215678    52.22647   52.239963   52.256706   52.277329   52.302541   52.333131   52.369972   52.414009   52.466257   52.527792   52.599734   52.683237   52.779466    52.88958   53.014711   53.155943   53.314288   53.490671   53.685908   53.900692   54.135582   54.390993   54.667193     54.9643   55.282289   55.620993   55.980112   56.359228   56.757813   57.175247   57.610826   58.063781   58.533288   59.018485     59.518478 | 1.088393  1.088397  1.088402  1.088408  1.088417  1.088429  1.088445  1.088467  1.088495  1.088533  1.088582  1.088645  1.088727  1.088832  1.088965  1.089132  1.089341  1.089601  1.089921  1.090311  1.090785  1.091355  1.092037  1.092845  1.093797  1.094909  1.096198  1.097683   1.09938  1.101307  1.103479   1.10591  1.108615  1.111603  1.114885  1.118467  1.122353  1.126545  1.131044  1.135845  1.140945  1.146337  1.152011  1.157957  1.164163  1.170616  1.177303   1.18421   1.19132  1.198619 |  1.35916   1.35916  1.359159  1.359159  1.359158  1.359158  1.359157  1.359155  1.359153  1.359151  1.359148  1.359143  1.359138  1.359131  1.359122  1.359111  1.359097   1.35908  1.359059  1.359033  1.359002  1.358964   1.35892  1.358867  1.358805  1.358732  1.358649  1.358553  1.358444  1.358321  1.358183  1.358029   1.35786  1.357675  1.357474  1.357256  1.357023  1.356774  1.356511  1.356235  1.355945  1.355644  1.355332  1.355011  1.354683  1.354347  1.354007  1.353662  1.353315          1.352966 | 0.536163  0.536164  0.536167   0.53617  0.536175  0.536181  0.536189    0.5362  0.536214  0.536233  0.536258  0.536291  0.536332  0.536386  0.536453  0.536538  0.536644  0.536776  0.536938  0.537136  0.537376  0.537665   0.53801  0.538418  0.538898  0.539457  0.540104  0.540847  0.541695  0.542653   0.54373   0.54493  0.546259  0.547719  0.549313  0.551043  0.552906  0.554902  0.557026  0.559275  0.561642  0.564121  0.566704  0.569383  0.572149  0.574994  0.577907  0.580881  0.583905       0.58697 | -0.201873  -0.201876  -0.201879  -0.201884   -0.20189  -0.201898  -0.201909  -0.201923  -0.201942  -0.201967     -0.202  -0.202042  -0.202096  -0.202165  -0.202252  -0.202361  -0.202497  -0.202665  -0.202871  -0.203122  -0.203424  -0.203787  -0.204217  -0.204724  -0.205318  -0.206008  -0.206802   -0.20771   -0.20874    -0.2099  -0.211198  -0.212638  -0.214225  -0.215962  -0.217851  -0.219892  -0.222082  -0.224419  -0.226899  -0.229515   -0.23226  -0.235127  -0.238106  -0.241188  -0.244364  -0.247623  -0.250955  -0.254349  -0.257797     -0.261289 |  0.66571   0.66571  0.665711  0.665711  0.665712  0.665713  0.665714  0.665715  0.665717   0.66572  0.665723  0.665728  0.665733   0.66574  0.665749   0.66576  0.665774  0.665791  0.665811  0.665836  0.665866  0.665902  0.665944  0.665994  0.666052  0.666119  0.666196  0.666284  0.666384  0.666497  0.666622  0.666762  0.666915  0.667084  0.667268  0.667467  0.667681  0.667911  0.668156  0.668415   0.66869  0.668978   0.66928  0.669595  0.669922   0.67026   0.67061  0.670969  0.671337      0.671714 | 100.002515 | 100.003049   100.00431  100.006052  100.008441  100.011695  100.016099  100.022015  100.029907  100.040365  100.054122  100.072094  100.095404  100.125423  100.163801  100.212512  100.273886  100.350651   100.44596  100.563427  100.707139  100.881676  101.092106   101.34398    101.6433  101.996487  102.410326  102.891902  103.448522  104.087631  104.816718  105.643222  106.574433  107.617401  108.778848  110.065091  111.481971  113.034806  114.728342  116.566731  118.553515  120.691625  122.983389  125.430554  128.034304    130.7953  133.713709  136.789242  140.021194   143.40848          146.949668 | 148.753956 | 0.720705  0.720708  0.720711  0.720716  0.720722   0.72073  0.720742  0.720757  0.720777  0.720803  0.720837  0.720881  0.720938  0.721011  0.721104  0.721221  0.721367  0.721548   0.72177  0.722043  0.722373  0.722771  0.723246   0.72381  0.724473  0.725248  0.726147  0.727182  0.728364  0.729707   0.73122  0.732914  0.734798   0.73688  0.739165  0.741658  0.744363   0.74728   0.75041  0.753749  0.757295  0.761043  0.764986  0.769116  0.773426  0.777906  0.782546  0.787337  0.792268       0.797328 |  2.59115  2.591164  2.591183  2.591209  2.591245  2.591293  2.591359  2.591446  2.591561  2.591712   2.59191  2.592166  2.592497  2.592919  2.593454  2.594129  2.594973  2.596019  2.597309  2.598884  2.600796  2.603099   2.60585  2.609114  2.612958   2.61745  2.622664  2.628669   2.63554  2.643344  2.652148  2.662014  2.672998  2.685149  2.698508  2.713107  2.728969  2.746108  2.764529  2.784225  2.805185  2.827384  2.850795  2.875381    2.9011  2.927906  2.955748  2.984572  3.014322        3.044941 | 100.002515  100.003569  100.005032  100.007046  100.009801  100.013543  100.018592  100.025355  100.034352  100.046238  100.061828  100.082132  100.108387  100.142094  100.185054  100.239408  100.307675  100.392788  100.498122  100.627523  100.785323  100.976346  101.205906  101.479788  101.804216  102.185808  102.631519  103.148568  103.744354  104.426372  105.202111  106.078963  107.064123    108.1645  109.386633   110.73662  112.220053  113.841973  115.606836  117.518491  119.580174  121.794511  124.163533  126.688698   129.37092  132.210602  135.207674  138.361629   141.67156  145.136201  148.753956 |  100.0 |   0.002515 | 148.753956 |
    | 20:30 | 3.717871   3.71789  3.717915  3.717949  3.717996   3.71806  3.718146  3.718261  3.718413  3.718615  3.718879  3.719222  3.719666  3.720235   3.72096  3.721876  3.723025  3.724456  3.726226  3.728397  3.731042   3.73424  3.738079  3.742652   3.74806  3.754409   3.76181  3.770375  3.780219  3.791453  3.804187  3.818526  3.834565  3.852394  3.872089  3.893716  3.917326  3.942957  3.970633  4.000364  4.032144  4.065955  4.101767  4.139538  4.179215  4.220736  4.264032  4.309029  4.355645             4.403796 | 124.880912  124.881855   124.88316  124.884956  124.887408  124.890736  124.895219  124.901217  124.909188  124.919705  124.933486  124.951415  124.974577  125.004284  125.042112  125.089933  125.149946  125.224712  125.317174  125.430684  125.569015  125.736363  125.937345  126.176978  126.460648  126.794069  127.183222  127.634286  128.153561   128.74738  129.422013  130.183575  131.037932  131.990608  133.046709  134.210849  135.487093  136.878911  138.389149  140.020018  141.773086  143.649298  145.648995  147.771947  150.017393  152.384086   154.87034  157.474079  160.192886  163.024047 |  52.178714   52.178895   52.179145   52.179489   52.179959   52.180597   52.181456   52.182606   52.184133   52.186149   52.188789   52.192225   52.196662   52.202353   52.209599   52.218758   52.230249   52.244562   52.262257   52.283972   52.310423   52.342404   52.380788   52.426516   52.480596    52.54409     52.6181   52.703754    52.80219   52.914531   53.041873   53.185257   53.345652    53.52394   53.720891   53.937158   54.173258   54.429571   54.706334   55.003639    55.32144   55.659554   56.017675    56.39538   56.792146   57.207357   57.640323   58.090288   58.556449     59.037962 | 1.088392  1.088395  1.088398  1.088404  1.088411  1.088421  1.088434  1.088452  1.088476  1.088507  1.088548  1.088601   1.08867  1.088758  1.088871  1.089013  1.089191  1.089413  1.089687  1.090023  1.090433  1.090929  1.091523  1.092231  1.093067  1.094049  1.095192  1.096515  1.098033  1.099764  1.101725  1.103929  1.106392  1.109124  1.112138  1.115442   1.11904  1.122939  1.127138  1.131638  1.136436  1.141525    1.1469  1.152551  1.158468  1.164639  1.171053  1.177694   1.18455  1.191605 |  1.35916   1.35916   1.35916  1.359159  1.359159  1.359158  1.359157  1.359156  1.359155  1.359152   1.35915  1.359146  1.359142  1.359136  1.359128  1.359119  1.359107  1.359093  1.359074  1.359052  1.359025  1.358993  1.358953  1.358907  1.358852  1.358788  1.358714  1.358628   1.35853  1.358419  1.358294  1.358154  1.357999  1.357829  1.357642   1.35744  1.357222  1.356988   1.35674  1.356477  1.356201  1.355912  1.355613  1.355303  1.354984  1.354658  1.354325  1.353987  1.353645          1.353301 | 0.536162  0.536163  0.536165  0.536168  0.536172  0.536177  0.536183  0.536192  0.536204   0.53622  0.536241  0.536268  0.536303  0.536348  0.536405  0.536477  0.536568  0.536681   0.53682  0.536991  0.537198  0.537449   0.53775  0.538108   0.53853  0.539024  0.539599  0.540263  0.541023  0.541886  0.542861  0.543953  0.545167  0.546508  0.547979  0.549583  0.551319  0.553186  0.555183  0.557306   0.55955   0.56191  0.564378  0.566948  0.569612   0.57236  0.575185  0.578076  0.581026      0.584025 | -0.201872  -0.201874  -0.201877  -0.201881  -0.201886  -0.201892  -0.201901  -0.201913  -0.201929   -0.20195  -0.201977  -0.202013  -0.202058  -0.202116   -0.20219  -0.202283  -0.202399  -0.202543   -0.20272  -0.202937  -0.203199  -0.203514  -0.203891  -0.204337  -0.204861  -0.205472  -0.206179  -0.206992  -0.207918  -0.208966  -0.210144  -0.211458  -0.212913  -0.214513   -0.21626  -0.218157  -0.220203  -0.222396  -0.224732  -0.227208  -0.229816   -0.23255  -0.235403  -0.238365  -0.241428  -0.244581  -0.247816  -0.251121  -0.254487     -0.257906 |  0.66571   0.66571   0.66571  0.665711  0.665711  0.665712  0.665713  0.665714  0.665716  0.665718  0.665721  0.665725  0.665729  0.665735  0.665743  0.665752  0.665764  0.665778  0.665796  0.665817  0.665844  0.665875  0.665912  0.665956  0.666007  0.666067  0.666135  0.666214  0.666304  0.666405  0.666519  0.666646  0.666787  0.666942  0.667111  0.667296  0.667495   0.66771  0.667939  0.668183  0.668442  0.668716  0.669003  0.669303  0.669615   0.66994  0.670276  0.670622  0.670979      0.671344 | 100.002026 |  100.00246  100.003486  100.004907  100.006861   100.00953  100.013152  100.018032  100.024561  100.033236  100.044684  100.059685  100.079202  100.104417   100.13676  100.177948  100.230022  100.295383  100.376826  100.477569  100.601279  100.752092  100.934616  101.153934  101.415584  101.725538  102.090154   102.51613  103.010433  103.580223  104.232772  104.975375   105.81525  106.759455  107.814795   108.98774  110.284355  111.710236  113.270458  114.969546  116.811442    118.7995  120.936485  123.224581  125.665413  128.260069  131.009135  133.912725  136.970516   140.18179          143.545462 | 145.260727 | 0.720704  0.720706  0.720709  0.720713  0.720718  0.720725  0.720734  0.720746  0.720763  0.720785  0.720813   0.72085  0.720898   0.72096  0.721038  0.721137  0.721262  0.721416  0.721607  0.721842  0.722128  0.722473  0.722888  0.723381  0.723964  0.724649  0.725446  0.726367  0.727426  0.728632  0.729998  0.731534   0.73325  0.735153  0.737252  0.739552  0.742057  0.744771  0.747693  0.750823   0.75416  0.757698  0.761434  0.765361  0.769471  0.773757  0.778209  0.782817  0.787573       0.792465 | 2.591143  2.591155   2.59117  2.591192  2.591221  2.591261  2.591315  2.591387  2.591482  2.591608  2.591773  2.591988  2.592266  2.592621  2.593074  2.593647  2.594365   2.59526  2.596366  2.597724  2.599377  2.601376  2.603774  2.606632   2.61001  2.613976  2.618597  2.623944  2.630087  2.637096  2.645037  2.653975  2.663968  2.675069  2.687326  2.700775  2.715448  2.731364  2.748536  2.766966  2.786649  2.807569  2.829705  2.853026  2.877498  2.903078   2.92972  2.957376  2.985993        3.015516 | 100.002026  100.002883  100.004074  100.005718  100.007975  100.011048  100.015205   100.02079  100.028242  100.038115  100.051104  100.068073  100.090086  100.118439  100.154694  100.200721  100.258731  100.331312  100.421464  100.532622  100.668684  100.834021  101.033479  101.272376  101.556476  101.891955  102.285358  102.743537  103.273577   103.88272  104.578272  105.367517  106.257618  107.255529  108.367911  109.601049   110.96079  112.452481  114.080932   115.85038  117.764476  119.826278  122.038255  124.402308  126.919784  129.591513  132.417833  135.398632  138.533379  141.821163  145.260727 |  100.0 |   0.002026 | 145.260727 |
    | 21:00 | 3.717863  3.717878  3.717898  3.717926  3.717965  3.718017  3.718088  3.718183  3.718309  3.718477  3.718697  3.718984  3.719357  3.719836  3.720447  3.721223    3.7222  3.723422  3.724937  3.726804  3.729087  3.731857  3.735196  3.739189  3.743931  3.749521  3.756065   3.76367  3.772447  3.782507  3.793958  3.806907  3.821453   3.83769  3.855701  3.875558  3.897324  3.921047   3.94676  3.974484  4.004228  4.035982   4.06973  4.105438  4.143064  4.182557  4.223855  4.266891  4.311591             4.357876 | 124.880474   124.88124  124.882304  124.883771  124.885781  124.888514  124.892208  124.897164  124.903769  124.912509  124.923997  124.938989  124.958417  124.983417  125.015356   125.05587  125.106889  125.170676  125.249846  125.347399  125.466729  125.611644   125.78636  125.995495  126.244047  126.537362  126.881085  127.281105  127.743483  128.274378  128.879953  129.566294  130.339312   131.20466  132.167642  133.233142  134.405556  135.688738  137.085964  138.599902  140.232604  141.985504  143.859435  145.854652  147.970862  150.207265  152.562598  155.035179  157.622953  160.323542 |  52.178631   52.178777   52.178981   52.179262   52.179647   52.180171   52.180879   52.181829   52.183095    52.18477   52.186971   52.189844   52.193566   52.198356   52.204474   52.212234   52.222005   52.234218   52.249373    52.26804   52.290865   52.318571   52.351955   52.391888   52.439307   52.495212   52.560648   52.636699   52.724469   52.825066   52.939579   53.069067    53.21453   53.376898   53.557006   53.755584   53.973244   54.210467   54.467598   54.744845   55.042275   55.359824   55.697295   56.054376   56.430641   56.825569   57.238553   57.668913   58.115911     58.578758 |  1.08839  1.088393  1.088396    1.0884  1.088406  1.088414  1.088425   1.08844   1.08846  1.088486   1.08852  1.088564  1.088622  1.088696  1.088791  1.088912  1.089063  1.089252  1.089487  1.089777   1.09013  1.090559  1.091076  1.091695  1.092428  1.093293  1.094305   1.09548  1.096834  1.098386   1.10015  1.102143  1.104379  1.106871  1.109631  1.112669  1.115992  1.119607  1.123517  1.127722  1.132222  1.137014  1.142092   1.14745  1.153078  1.158966  1.165102  1.171475  1.178072  1.184877 |  1.35916   1.35916   1.35916   1.35916  1.359159  1.359159  1.359158  1.359157  1.359156  1.359154  1.359152  1.359149  1.359145   1.35914  1.359134  1.359126  1.359116  1.359103  1.359088  1.359068  1.359045  1.359017  1.358983  1.358942  1.358894  1.358837  1.358772  1.358695  1.358607  1.358507  1.358394  1.358267  1.358126  1.357969  1.357797  1.357609  1.357406  1.357187  1.356954  1.356705  1.356443  1.356168  1.355881  1.355582  1.355274  1.354957  1.354633  1.354303  1.353968          1.353629 | 0.536161  0.536162  0.536164  0.536166  0.536169  0.536173  0.536179  0.536186  0.536196  0.536209  0.536227   0.53625  0.536279  0.536317  0.536365  0.536426  0.536503  0.536599  0.536718  0.536865  0.537045  0.537262  0.537524  0.537837  0.538208  0.538644  0.539153  0.539744  0.540423  0.541199  0.542078  0.543068  0.544175  0.545403  0.546756  0.548238  0.549849  0.551591  0.553461  0.555459   0.55758  0.559819  0.562171   0.56463  0.567187  0.569835  0.572565   0.57537   0.57824      0.581166 | -0.201871  -0.201873  -0.201875  -0.201878  -0.201882  -0.201888  -0.201895  -0.201905  -0.201918  -0.201936  -0.201958  -0.201988  -0.202026  -0.202075  -0.202138  -0.202217  -0.202316  -0.202439  -0.202591  -0.202777  -0.203004  -0.203278  -0.203607  -0.203998  -0.204459  -0.204999  -0.205627  -0.206353  -0.207183  -0.208127  -0.209193  -0.210388  -0.211717  -0.213186  -0.214798  -0.216555   -0.21846   -0.22051  -0.222704  -0.225039   -0.22751   -0.23011  -0.232834  -0.235672  -0.238618  -0.241661  -0.244792  -0.248002  -0.251281      -0.25462 |  0.66571   0.66571   0.66571  0.665711  0.665711  0.665712  0.665712  0.665713  0.665715  0.665717  0.665719  0.665722  0.665726  0.665731  0.665737  0.665745  0.665755  0.665768  0.665783  0.665802  0.665824  0.665851  0.665884  0.665922  0.665967   0.66602  0.666081  0.666152  0.666232  0.666324  0.666427  0.666542   0.66667  0.666812  0.666968  0.667138  0.667323  0.667523  0.667738  0.667967  0.668211  0.668469  0.668741  0.669026  0.669325  0.669635  0.669958  0.670291  0.670635      0.670988 | 100.001631 | 100.001983  100.002817  100.003975  100.005571  100.007759  100.010734  100.014754  100.020149  100.027338  100.036852  100.049356  100.065676  100.086825  100.114042  100.148815  100.192929  100.248489  100.317963  100.404209  100.510506  100.640571  100.798579  100.989163  101.217411  101.488846    101.8094  102.185365  102.623345  103.130187  103.712905  104.378597  105.134362  105.987208  106.943963  108.011194   109.19513  110.501593  111.935942  113.503027  115.207156  117.052076  119.040962  121.176423  123.460507  125.894726  128.480079   131.21708  134.105795  137.145873          140.336581 | 141.965038 | 0.720703  0.720705  0.720707   0.72071  0.720714   0.72072  0.720728  0.720738  0.720752   0.72077  0.720794  0.720825  0.720865  0.720917  0.720983  0.721067  0.721172  0.721304  0.721468   0.72167  0.721917  0.722216  0.722576  0.723007  0.723519  0.724122  0.724827  0.725646   0.72659  0.727672  0.728901   0.73029  0.731847  0.733583  0.735506  0.737621  0.739935  0.742452  0.745173  0.748099  0.751229  0.754562  0.758093  0.761816  0.765727  0.769817  0.774078  0.778502  0.783079         0.7878 | 2.591138  2.591147   2.59116  2.591178  2.591202  2.591234  2.591279  2.591338  2.591417  2.591522   2.59166  2.591839  2.592072  2.592371  2.592754  2.593239   2.59385  2.594614  2.595561  2.596728  2.598155  2.599886  2.601973  2.604468  2.607431  2.610923   2.61501  2.619758  2.625237  2.631515  2.638658  2.646733  2.655799  2.665914  2.677128  2.689484  2.703019  2.717759  2.733724  2.750924   2.76936  2.789025  2.809903  2.831972  2.855202  2.879558  2.904998  2.931479   2.95895        2.987362 | 100.001631  100.002326  100.003295  100.004637  100.006482  100.009003  100.012423   100.01703  100.023194  100.031385  100.042193  100.056357  100.074787  100.098601  100.129152  100.168068  100.217282  100.279073  100.356095  100.451409  100.568509  100.711336  100.884296   101.09225  101.340512  101.634817  101.981292    102.3864  102.856888  103.399705  104.021933  104.730694  105.533067  106.435992   107.44619   108.57008  109.813705  111.182668  112.682087  114.316547   116.09008  118.006149  120.067644  122.276889  124.635658    127.1452  129.806261  132.619125  135.583636  138.699246  141.965038 |  100.0 |   0.001631 | 141.965038 |
    | 21:30 | 3.717856  3.717868  3.717885  3.717908  3.717939  3.717982   3.71804  3.718119  3.718224  3.718363  3.718546  3.718786  3.719098    3.7195  3.720016  3.720672  3.721502  3.722542  3.723838   3.72544  3.727405  3.729801  3.732698  3.736178  3.740326  3.745237  3.751009  3.757745  3.765552  3.774537  3.784808  3.796471  3.809628  3.824375  3.840801  3.858986  3.878998  3.900894  3.924721  3.950507  3.978273  4.008022  4.039745  4.073423  4.109023  4.146503  4.185811  4.226886  4.269663             4.314068 |  124.88012  124.880742  124.881608  124.882805   124.88445  124.886694  124.889733  124.893823   124.89929  124.906545  124.916109  124.928628  124.944902   124.96591  124.992837  125.027106  125.070409  125.124736  125.192407  125.276091  125.378837   125.50408  125.655656  125.837793  126.055107  126.312571  126.615483  126.969418  127.380169  127.853678  128.395956  129.013005  129.710726  130.494834   131.37077  132.343628  133.418078  134.598306   135.88797  137.290159  138.807372  140.441513  142.193891  144.065234  146.055716  148.164988  150.392217  152.736124  155.195034  157.766915 |  52.178563   52.178682   52.178848   52.179077   52.179392   52.179822   52.180405   52.181189   52.182236   52.183627   52.185459   52.187858   52.190977   52.195002    52.20016   52.206725   52.215019   52.225422   52.238378   52.254395   52.274054   52.298008   52.326983   52.361779   52.403265   52.452373   52.510091   52.577451   52.655516   52.745365   52.848076   52.964706   53.096276   53.243748   53.408009   53.589855   53.789976   54.008945   54.247206   54.505075   54.782729   55.080217   55.397453   55.734233   56.090234   56.465031   56.858107   57.268861   57.696626     58.140677 | 1.088389  1.088391  1.088394  1.088397  1.088402  1.088409  1.088418   1.08843  1.088446  1.088468  1.088496  1.088534  1.088582  1.088644  1.088724  1.088826  1.088955  1.089116  1.089317  1.089565   1.08987  1.090241   1.09069  1.091229  1.091871  1.092631  1.093523  1.094564   1.09577  1.097157   1.09874  1.100537  1.102561  1.104827  1.107348  1.110134  1.113194  1.116537  1.120166  1.124086  1.128296  1.132796  1.137581  1.142647  1.147987  1.153591   1.15945  1.165552  1.171885  1.178436 |  1.35916   1.35916   1.35916   1.35916  1.359159  1.359159  1.359158  1.359158  1.359156  1.359155  1.359153  1.359151  1.359148  1.359143  1.359138  1.359131  1.359123  1.359112  1.359099  1.359082  1.359062  1.359038  1.359008  1.358973  1.358931  1.358881  1.358822  1.358755  1.358676  1.358587  1.358485   1.35837  1.358241  1.358098  1.357939  1.357766  1.357577  1.357373  1.357154   1.35692  1.356672   1.35641  1.356136  1.355849  1.355553  1.355246  1.354932   1.35461  1.354282           1.35395 | 0.536161  0.536161  0.536163  0.536165  0.536167   0.53617  0.536175  0.536181   0.53619    0.5362  0.536215  0.536234  0.536258   0.53629  0.536331  0.536383  0.536448   0.53653  0.536632  0.536758  0.536913  0.537101  0.537328  0.537601  0.537926   0.53831   0.53876  0.539284  0.539889  0.540584  0.541376  0.542271  0.543276  0.544396  0.545637  0.547002  0.548493  0.550112  0.551859  0.553733   0.55573  0.557849  0.560083  0.562427  0.564875  0.567419  0.570051  0.572764  0.575549      0.578397 | -0.201871  -0.201872  -0.201874  -0.201876   -0.20188  -0.201884   -0.20189  -0.201898  -0.201909  -0.201924  -0.201943  -0.201968     -0.202  -0.202041  -0.202094   -0.20216  -0.202245   -0.20235   -0.20248   -0.20264  -0.202836  -0.203074   -0.20336  -0.203702  -0.204107  -0.204583   -0.20514  -0.205785  -0.206527  -0.207375  -0.208337   -0.20942   -0.21063  -0.211974  -0.213457   -0.21508  -0.216847  -0.218758  -0.220812  -0.223007   -0.22534  -0.227805  -0.230398   -0.23311  -0.235934  -0.238863  -0.241886  -0.244996  -0.248182     -0.251435 |  0.66571   0.66571   0.66571   0.66571  0.665711  0.665711  0.665712  0.665713  0.665714  0.665715  0.665717   0.66572  0.665723  0.665727  0.665733   0.66574  0.665748  0.665759  0.665772  0.665788  0.665807  0.665831  0.665859  0.665893  0.665933  0.665979  0.666033  0.666096  0.666168   0.66625  0.666343  0.666448  0.666565  0.666694  0.666837  0.666994  0.667165  0.667351  0.667551  0.667765  0.667994  0.668238  0.668495  0.668766   0.66905  0.669346  0.669655  0.669975  0.670306      0.670647 | 100.001312 | 100.001597  100.002274  100.003217   100.00452   100.00631  100.008752  100.012061  100.016513  100.022463   100.03036   100.04077  100.054397  100.072113  100.094982  100.124297  100.161608  100.208761  100.267926  100.341634  100.432804  100.544768  100.681291  100.846579  101.045284  101.282494  101.563711  101.894818  102.282041  102.731884  103.251071  103.846467  104.525001  105.293579  106.158998  107.127865  108.206518  109.400951  110.716755   112.15906  113.732502  115.441185  117.288669  119.277961  121.411522  123.691273   126.11862  128.694477  131.419296  134.293095          137.315496 | 138.859416 | 0.720703  0.720704  0.720706  0.720708  0.720712  0.720716  0.720723  0.720731  0.720742  0.720757  0.720777  0.720803  0.720837   0.72088  0.720936  0.721007  0.721097  0.721209  0.721349  0.721523  0.721735  0.721994  0.722307  0.722682   0.72313   0.72366  0.724282  0.725008  0.725848  0.726815  0.727919  0.729171  0.730581   0.73216  0.733916  0.735856  0.737987  0.740315  0.742841  0.745569  0.748498  0.751628  0.754956  0.758478   0.76219  0.766084  0.770153   0.77439  0.778786       0.783332 | 2.591134  2.591141  2.591152  2.591166  2.591186  2.591213  2.591249  2.591298  2.591364   2.59145  2.591565  2.591715   2.59191  2.592162  2.592484  2.592895  2.593413  2.594064  2.594874  2.595875  2.597104  2.598601  2.600412  2.602587  2.605179  2.608247  2.611852  2.616059  2.620933  2.626541   2.63295  2.640225  2.648429  2.657619  2.667851  2.679172  2.691623  2.705238  2.720041   2.73605  2.753274  2.771711  2.791354  2.812187  2.834187  2.857324  2.881563  2.906864  2.933183        2.960471 | 100.001312  100.001875  100.002663  100.003756  100.005265   100.00733   100.01014  100.013935  100.019028  100.025816  100.034798  100.046605  100.062015  100.081989  100.107696  100.140548  100.182234  100.234753  100.300445   100.38203  100.482626  100.605776  100.755465  100.936122   101.15262  101.410262   101.71475  102.072152  102.488851  102.971479  103.526852   104.16189  104.883534  105.698658  106.613989  107.636019  108.770933   110.02454  111.402214  112.908848  114.548818  116.325961  118.243559  120.304346  122.510507  124.863701  127.365081  130.015319  132.814644  135.762866  138.859416 |  100.0 |   0.001312 | 138.859416 |
    | 22:00 | 3.717851   3.71786  3.717874  3.717893  3.717918  3.717954  3.718002  3.718066  3.718153  3.718268   3.71842   3.71862  3.718881  3.719219  3.719653  3.720208  3.720911  3.721796  3.722901  3.724273  3.725963   3.72803   3.73054  3.733565  3.737187  3.741491   3.74657  3.752522  3.759449  3.767454  3.776643   3.78712  3.798989  3.812348  3.827289  3.843897  3.862248  3.882406  3.904425  3.928347    3.9542  3.981999  4.011747  4.043434  4.077038  4.112527  4.149857  4.188978  4.229831             4.272349 | 124.879833  124.880338  124.881042  124.882018  124.883363  124.885203  124.887701  124.891073  124.895592  124.901607  124.909559      124.92  124.933613  124.951242  124.973909  125.002852  125.039548  125.085743  125.143484  125.215145  125.303448  125.411483  125.542723  125.701026  125.890629  126.116138  126.382495  126.694949  127.058998  127.480336  127.964783  128.518208  129.146448  129.855223  130.650055  131.536185  132.518499  133.601463  134.789065  136.084772  137.491497  139.011581  140.646788  142.398308  144.266773  146.252285  148.354441  150.572377  152.904804  155.350054 |  52.178508   52.178604   52.178739   52.178926   52.179184   52.179537   52.180016   52.180662   52.181528    52.18268   52.184204   52.186205   52.188814   52.192191   52.196534   52.202079   52.209108   52.217955   52.229012   52.242731    52.25963   52.280299   52.305396   52.335652   52.371868    52.41491   52.465702   52.525222   52.594486   52.674536   52.766426   52.871204   52.989895   53.123483   53.272893   53.438972   53.622476   53.824058   54.044253   54.283473   54.542002   54.819993   55.117472   55.434341   55.770382   56.125268   56.498574   56.889784   57.298308     57.723488 | 1.088389   1.08839  1.088392  1.088395  1.088399  1.088405  1.088412  1.088422  1.088435  1.088453  1.088477  1.088508  1.088548  1.088601  1.088668  1.088754  1.088863     1.089  1.089172  1.089384  1.089646  1.089967  1.090355  1.090824  1.091385  1.092051  1.092837  1.093757  1.094827  1.096064  1.097482  1.099097  1.100925   1.10298  1.105275  1.107823  1.110633  1.113715  1.117075  1.120718  1.124646   1.12886  1.133358  1.138137   1.14319  1.148511  1.154092  1.159921  1.165989  1.172282 |  1.35916   1.35916   1.35916   1.35916   1.35916  1.359159  1.359159  1.359158  1.359157  1.359156  1.359154  1.359152   1.35915  1.359146  1.359142  1.359136  1.359129   1.35912  1.359108  1.359094  1.359077  1.359056   1.35903  1.358999  1.358963  1.358919  1.358867  1.358807  1.358737  1.358657  1.358566  1.358462  1.358345  1.358214  1.358069   1.35791  1.357735  1.357545   1.35734  1.357121  1.356887  1.356639  1.356377  1.356104  1.355819  1.355524  1.355219  1.354907  1.354587          1.354262 |  0.53616  0.536161  0.536162  0.536163  0.536165  0.536168  0.536172  0.536177  0.536184  0.536193  0.536205  0.536221  0.536241  0.536268  0.536302  0.536346  0.536401  0.536471  0.536558  0.536666  0.536799  0.536962  0.537159  0.537396   0.53768  0.538017  0.538414  0.538878  0.539416  0.540037  0.540747  0.541554  0.542464  0.543483  0.544617   0.54587  0.547246  0.548746  0.550372  0.552124     0.554  0.555997  0.558112  0.560341  0.562677  0.565114  0.567645  0.570262  0.572957      0.575723 |  -0.20187  -0.201871  -0.201873  -0.201875  -0.201877  -0.201881  -0.201886  -0.201893  -0.201902  -0.201914   -0.20193   -0.20195  -0.201977  -0.202012  -0.202056  -0.202113  -0.202184  -0.202274  -0.202385  -0.202523  -0.202692  -0.202898  -0.203146  -0.203444  -0.203799  -0.204218   -0.20471  -0.205282  -0.205944  -0.206704  -0.207569  -0.208547  -0.209647  -0.210873  -0.212231  -0.213725   -0.21536  -0.217135  -0.219052  -0.221109  -0.223305  -0.225635  -0.228095  -0.230679  -0.233379   -0.23619  -0.239101  -0.242105  -0.245193     -0.248355 |  0.66571   0.66571   0.66571   0.66571   0.66571  0.665711  0.665711  0.665712  0.665713  0.665714  0.665716  0.665718  0.665721  0.665724  0.665729  0.665735  0.665742  0.665751  0.665762  0.665776  0.665793  0.665813  0.665838  0.665867  0.665902  0.665943  0.665991  0.666047  0.666111  0.666185  0.666268  0.666363  0.666469  0.666587  0.666718  0.666862   0.66702  0.667192  0.667378  0.667578  0.667792  0.668021  0.668264   0.66852   0.66879  0.669072  0.669367  0.669674  0.669991       0.67032 | 100.001054 | 100.001285  100.001835  100.002601  100.003664  100.005128   100.00713  100.009849  100.013519  100.018438  100.024985   100.03364  100.045005  100.059824  100.079014  100.103691  100.135201  100.175156  100.225459  100.288345  100.366404  100.462611  100.580349  100.723423  100.896067  101.102945  101.349138   101.64012  101.981725  102.380103  102.841658  103.372989  103.980813  104.671886  105.452926  106.330526  107.311077  108.400692  109.605143  110.929796  112.379566  113.958881  115.671651  117.521262   119.51056  121.641866  123.916983  126.337219  128.903406  131.615938          134.474791 | 135.936495 | 0.720702  0.720703  0.720705  0.720707  0.720709  0.720713  0.720718  0.720725  0.720735  0.720747  0.720764  0.720785  0.720814   0.72085  0.720897  0.720957  0.721033  0.721129  0.721248  0.721396  0.721579  0.721802  0.722074    0.7224  0.722791  0.723256  0.723804  0.724445  0.725191  0.726053  0.727041  0.728167  0.729441  0.730873  0.732472  0.734246  0.736204   0.73835  0.740689  0.743225  0.745959  0.748891   0.75202  0.755343  0.758856  0.762554  0.766431   0.77048  0.774693       0.779062 |  2.59113  2.591136  2.591145  2.591157  2.591173  2.591195  2.591225  2.591265  2.591319  2.591391  2.591487  2.591612  2.591775  2.591986  2.592258  2.592604  2.593044  2.593597  2.594288  2.595146  2.596202  2.597494  2.599063  2.600954  2.603217  2.605906   2.60908  2.612797  2.617123   2.62212  2.627856  2.634393  2.641796  2.650125  2.659435  2.669779  2.681202  2.693743  2.707432  2.722293  2.738342  2.755584  2.774019  2.793637  2.814422  2.836351  2.859394  2.883516  2.908677        2.934834 | 100.001054  100.001511   100.00215  100.003041  100.004272  100.005962  100.008268  100.011392  100.015595  100.021213  100.028669  100.038497  100.051365  100.068095  100.089695  100.117388  100.152643  100.197209  100.253146  100.322859  100.409122  100.515111  100.644413  100.801049  100.989469  101.214549  101.481574  101.796211  102.164464  102.592626   103.08722  103.654922  104.302493  105.036692  105.864196   106.79152  107.824935  108.970402  110.233504  111.619394  113.132752  114.777754  116.558051  118.476758  120.536457  122.739205  125.086552  127.579561  130.218836  133.004554  135.936495 |  100.0 |   0.001054 | 135.936495 |
    | 22:30 | 3.717846  3.717854  3.717865   3.71788  3.717901   3.71793   3.71797  3.718023  3.718094   3.71819  3.718316  3.718483  3.718701  3.718984  3.719349  3.719817  3.720411  3.721163  3.722105  3.723278  3.724728  3.726508  3.728677  3.731303  3.734458  3.738222  3.742681  3.747928  3.754059  3.761174  3.769374  3.778763  3.789443  3.801513  3.815067  3.830195  3.846977  3.865486  3.885782  3.907916  3.931927  3.957838  3.985664  4.015405   4.04705  4.080576   4.11595  4.153129  4.192062             4.232692 | 124.879602  124.880011  124.880583  124.881378  124.882477  124.883983  124.886035  124.888811  124.892543  124.897523  124.904127  124.912823  124.924196  124.938969  124.958023  124.982432   125.01348  125.052697  125.101884  125.163142  125.238896  125.331914  125.445329  125.582642   125.74773  125.944834  126.178543  126.453767  126.775696  127.149751  127.581526  128.076716  128.641047  129.280193  129.999699  130.804897  131.700833  132.692195  133.783252  134.977801  136.279129  137.689982  139.212552   140.84847  142.598817  144.464134  146.444455   148.53933  150.747869  153.068773 |  52.178463   52.178542   52.178651   52.178804   52.179014   52.179303   52.179696   52.180228   52.180943   52.181898   52.183163    52.18483   52.187009    52.18984   52.193491   52.198167   52.204115   52.211626   52.221046   52.232776   52.247277   52.265077   52.286772   52.313027   52.344576   52.382217   52.426814   52.479285   52.540593   52.611738   52.693743   52.787634   52.894432   53.015129   53.150672   53.301951   53.469773   53.654859   53.857823   54.079164   54.319265   54.578381   54.856641   55.154052     55.4705   55.805759   56.159497   56.531289   56.920623     57.326917 | 1.088388  1.088389  1.088391  1.088393  1.088396  1.088401  1.088407  1.088415  1.088426  1.088441  1.088461  1.088487   1.08852  1.088564  1.088621  1.088693  1.088786  1.088902  1.089048   1.08923  1.089455  1.089731  1.090067  1.090474  1.090962  1.091545  1.092235  1.093047  1.093995  1.095094   1.09636  1.097809  1.099455  1.101313  1.103398  1.105721  1.108295  1.111128  1.114231  1.117607  1.121262  1.125198  1.129415   1.13391  1.138681  1.143721  1.149024   1.15458   1.16038  1.166413 |  1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359159  1.359159  1.359158  1.359157  1.359156  1.359154  1.359152  1.359149  1.359145   1.35914  1.359134  1.359126  1.359117  1.359105   1.35909  1.359071  1.359049  1.359022   1.35899  1.358952  1.358907  1.358854  1.358792   1.35872  1.358638  1.358545  1.358439   1.35832  1.358188  1.358041   1.35788  1.357704  1.357514  1.357308  1.357088  1.356854  1.356606  1.356346  1.356073  1.355789  1.355496  1.355193  1.354882          1.354565 |  0.53616   0.53616  0.536161  0.536162  0.536164  0.536166  0.536169  0.536174  0.536179  0.536187  0.536197   0.53621  0.536227  0.536249  0.536278  0.536315  0.536362  0.536421  0.536495  0.536588  0.536702  0.536842  0.537013  0.537219  0.537466  0.537761   0.53811   0.53852  0.538997   0.53955  0.540185   0.54091  0.541732  0.542657   0.54369  0.544837  0.546102  0.547487  0.548996  0.550629  0.552384  0.554262  0.556259  0.558371  0.560594  0.562921  0.565347  0.567865  0.570467      0.573144 |  -0.20187  -0.201871  -0.201872  -0.201873  -0.201876  -0.201879  -0.201883  -0.201888  -0.201896  -0.201906  -0.201919  -0.201936  -0.201959  -0.201988  -0.202025  -0.202073  -0.202134   -0.20221  -0.202305  -0.202423  -0.202568  -0.202746  -0.202961   -0.20322   -0.20353  -0.203898  -0.204332  -0.204838  -0.205427  -0.206105  -0.206881  -0.207763  -0.208758  -0.209873  -0.211114  -0.212486  -0.213992  -0.215636   -0.21742  -0.219342  -0.221402  -0.223597  -0.225924  -0.228378  -0.230953  -0.233642  -0.236438  -0.239333  -0.242318     -0.245384 |  0.66571   0.66571   0.66571   0.66571   0.66571  0.665711  0.665711  0.665712  0.665712  0.665713  0.665715  0.665717  0.665719  0.665722  0.665726  0.665731  0.665737  0.665744  0.665754  0.665766   0.66578  0.665798  0.665819  0.665845  0.665876  0.665912  0.665954  0.666003  0.666061  0.666126  0.666201  0.666287  0.666382   0.66649  0.666609  0.666741  0.666887  0.667045  0.667218  0.667404  0.667604  0.667819  0.668047  0.668289  0.668545  0.668813  0.669094  0.669387  0.669692      0.670007 | 100.000846 | 100.001034  100.001479  100.002102  100.002967  100.004163  100.005802  100.008036  100.011058  100.015119   100.02054  100.027728  100.037193  100.049573  100.065653  100.086396  100.112969  100.146772  100.189474  100.243038  100.309757  100.392278  100.493631  100.617242  100.766951  100.947017  101.162108  101.417294  101.718016  102.070053  102.479475  102.952584  103.495852  104.115847  104.819156  105.612311  106.501704  107.493518  108.593649  109.807652   111.14068   112.59744  114.182162  115.898576  117.749897  119.738822   121.86754  124.137743  126.550644  129.107004          131.807159 | 133.189015 | 0.720702  0.720702  0.720704  0.720705  0.720708  0.720711  0.720715  0.720721  0.720728  0.720739  0.720752   0.72077  0.720794  0.720825  0.720864  0.720915  0.720979   0.72106  0.721162  0.721289  0.721446  0.721638  0.721872  0.722156  0.722497  0.722903  0.723384   0.72395  0.724611  0.725377   0.72626  0.727269  0.728417  0.729711  0.731164  0.732782  0.734575  0.736549  0.738709   0.74106  0.743604  0.746343  0.749277  0.752404  0.755721  0.759225   0.76291  0.766771  0.770799       0.774988 | 2.591128  2.591133  2.591139  2.591149  2.591162   2.59118  2.591205  2.591238  2.591283  2.591342  2.591422  2.591526  2.591662  2.591839  2.592067   2.59236  2.592732  2.593201   2.59379  2.594523   2.59543  2.596543  2.597899   2.59954  2.601512  2.603864   2.60665  2.609928  2.613757    2.6182  2.623319  2.629179  2.635842   2.64337  2.651819  2.661246  2.671697  2.683217  2.695843  2.709602  2.724516  2.740599  2.757856  2.776284  2.795875  2.816609  2.838465  2.861412  2.885416        2.910438 | 100.000846  100.001216  100.001735  100.002459  100.003463  100.004845  100.006736  100.009304  100.012769  100.017413  100.023594  100.031766  100.042497  100.056491  100.074615  100.097925  100.127696  100.165455  100.213008  100.272476  100.346321  100.437374   100.54886  100.684408  100.848068  101.044305  101.277994  101.554398  101.879138  102.258154  102.697646  103.204023  103.783822  104.443646  105.190073  106.029588  106.968499  108.012863  109.168424  110.440548  111.834177  113.353787  115.003364  116.786382  118.705798  120.764051  122.963077  125.304324   127.78877  130.416957  133.189015 |  100.0 |   0.000846 | 133.189015 |
    | 23:00 | 3.717843  3.717849  3.717858   3.71787  3.717888  3.717911  3.717943  3.717987  3.718046  3.718125   3.71823  3.718369  3.718551  3.718787  3.719094  3.719488   3.71999  3.720627  3.721428  3.722429  3.723671  3.725202  3.727074  3.729347   3.73209  3.735375  3.739282  3.743897  3.749311  3.755619   3.76292  3.771312  3.780898  3.791775  3.804039  3.817783  3.833091   3.85004  3.868699  3.889126  3.911367  3.935458  3.961421  3.989268  4.018996  4.050594  4.084038  4.119294   4.15632             4.195065 | 124.879415  124.879746  124.880211  124.880858  124.881754  124.882987   124.88467  124.886953  124.890031   124.89415  124.899628  124.906862  124.916351  124.928713  124.944708  124.965263  124.991493  125.024734  125.066566  125.118843  125.183717  125.263661  125.361487  125.480364  125.623819  125.795742  126.000372  126.242279  126.526332  126.857662  127.241609  127.683663  128.189396  128.764389  129.414159  130.144075  130.959287   131.86465  132.864663  133.963403  135.164488  136.471032  137.885623  139.410311  141.046605   142.79548  144.657396  146.632322  148.719767  150.918815 |  52.178428   52.178491    52.17858   52.178704   52.178876   52.179112   52.179435   52.179872   52.180462   52.181252   52.182301   52.183687   52.185506   52.187875    52.19094   52.194878   52.199903    52.20627   52.214283   52.224294   52.236715   52.252016   52.270735   52.293473   52.320898   52.353747   52.392819    52.43897    52.49311   52.556191   52.629195   52.713122   52.808976   52.917745   53.040392   53.177829   53.330908   53.500402   53.686993   53.891262   54.113674   54.354582   54.614215   54.892679   55.189964   55.505942   55.840379   56.192939   56.563198     56.950648 | 1.088387  1.088388   1.08839  1.088392  1.088394  1.088398  1.088403   1.08841  1.088419  1.088431  1.088447  1.088469  1.088497  1.088534  1.088581  1.088642   1.08872  1.088819  1.088943  1.089099  1.089291  1.089528  1.089818  1.090171  1.090595  1.091104  1.091709  1.092423  1.093261  1.094236  1.095364  1.096659  1.098138  1.099814  1.101702  1.103815  1.106165  1.108764   1.11162  1.114741  1.118133  1.121799  1.125742   1.12996  1.134453  1.139215  1.144241  1.149524  1.155056  1.160826 |  1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359159  1.359159  1.359158  1.359158  1.359156  1.359155  1.359153  1.359151  1.359148  1.359143  1.359138  1.359132  1.359124  1.359113  1.359101  1.359085  1.359066  1.359042  1.359014  1.358981  1.358941  1.358894   1.35884  1.358776  1.358703  1.358619  1.358523  1.358416  1.358295  1.358161  1.358013  1.357851  1.357674  1.357482  1.357276  1.357056  1.356822  1.356574  1.356314  1.356043   1.35576  1.355468  1.355167          1.354859 | 0.536159   0.53616  0.536161  0.536162  0.536163  0.536165  0.536167  0.536171  0.536176  0.536182   0.53619  0.536201  0.536215  0.536234  0.536258  0.536289  0.536329  0.536379  0.536442  0.536521  0.536619  0.536739  0.536886  0.537065   0.53728  0.537538  0.537844  0.538205  0.538627  0.539119  0.539685  0.540335  0.541075  0.541911   0.54285  0.543896  0.545056  0.546332  0.547727  0.549244  0.550882  0.552641   0.55452  0.556516  0.558625  0.560841   0.56316  0.565575  0.568079      0.570666 | -0.201869   -0.20187  -0.201871  -0.201872  -0.201874  -0.201877   -0.20188  -0.201885  -0.201891  -0.201899   -0.20191  -0.201924  -0.201943  -0.201967  -0.201999  -0.202039   -0.20209  -0.202155  -0.202236  -0.202337  -0.202462  -0.202615  -0.202801  -0.203026  -0.203297  -0.203619     -0.204  -0.204447  -0.204969  -0.205573  -0.206267   -0.20706  -0.207958  -0.208969  -0.210099  -0.211354  -0.212739  -0.214257   -0.21591    -0.2177  -0.219627   -0.22169  -0.223885  -0.226208  -0.228656  -0.231222  -0.233899  -0.236681  -0.239558     -0.242524 |  0.66571   0.66571   0.66571   0.66571   0.66571   0.66571  0.665711  0.665711  0.665712  0.665713  0.665714  0.665715  0.665717   0.66572  0.665723  0.665727  0.665732  0.665739  0.665747  0.665757   0.66577  0.665785  0.665804  0.665826  0.665852  0.665884  0.665921  0.665965  0.666016  0.666074  0.666142  0.666218  0.666305  0.666402  0.666511  0.666631  0.666765  0.666911   0.66707  0.667243   0.66743  0.667631  0.667845  0.668073  0.668314  0.668569  0.668836  0.669115  0.669407      0.669709 | 100.000679 |  100.00083  100.001191  100.001696  100.002401  100.003377  100.004718   100.00655  100.009035  100.012385  100.016869  100.022831  100.030704  100.041033   100.05449  100.071902  100.094278  100.122834  100.159025  100.204576  100.261507  100.332168   100.41926  100.525859  100.655435  100.811857    100.9994  101.222735  101.486914  101.797341  102.159734  102.580081  103.064579  103.619572  104.251479  104.966721  105.771646  106.672451  107.675114  108.785325  110.008429  111.349372  112.812665  114.402351  116.121983  117.974618  119.962812  122.088629  124.353655  126.759017          129.305406 | 130.609826 | 0.720701  0.720702  0.720703  0.720704  0.720706  0.720709  0.720712  0.720717  0.720723  0.720732  0.720743  0.720758  0.720778  0.720803  0.720837  0.720879  0.720933  0.721002  0.721089  0.721197  0.721331  0.721497  0.721699  0.721945  0.722241  0.722596  0.723017  0.723515  0.724099  0.724779  0.725565  0.726468  0.727499  0.728667  0.729982  0.731455  0.733092  0.734902  0.736891  0.739064  0.741426  0.743978  0.746721  0.749656  0.752781  0.756092  0.759586  0.763258  0.767101       0.771109 | 2.591125  2.591129  2.591135  2.591143  2.591153  2.591168  2.591188  2.591216  2.591253  2.591302  2.591368  2.591454  2.591568  2.591716  2.591908  2.592154  2.592468  2.592866  2.593367  2.593993   2.59477  2.595726  2.596896  2.598318  2.600032  2.602085  2.604526   2.60741  2.610792  2.614731   2.61929  2.624529  2.630511  2.637296  2.644945  2.653512  2.663049  2.673604  2.685217  2.697922  2.711746  2.726709  2.742822   2.76009  2.778509  2.798068  2.818749   2.84053   2.86338        2.887266 | 100.000679  100.000978  100.001398  100.001987  100.002805  100.003934  100.005482  100.007591  100.010444  100.014279  100.019397  100.026184  100.035121  100.046812  100.061998  100.081591  100.106695  100.138637  100.178999  100.229644  100.292751  100.370837  100.466786  100.583867  100.725747  100.896499  101.100598  101.342913  101.628678  101.963467  102.353149   102.80383  103.321801  103.913463  104.585257  105.343588  106.194749  107.144848  108.199735  109.364943  110.645632  112.046539  113.571948  115.225663   117.01099  118.930734  120.987203  123.182218  125.517129  127.992838  130.609826 |  100.0 |   0.000679 | 130.609826 |
    | 23:30 |  3.71784  3.717845  3.717852  3.717862  3.717876  3.717896  3.717922  3.717958  3.718007  3.718072  3.718159  3.718274  3.718425  3.718623   3.71888  3.719212  3.719636  3.720174  3.720855  3.721708   3.72277  3.724083  3.725695   3.72766   3.73004    3.7329  3.736316  3.740367  3.745137  3.750717    3.7572  3.764684  3.773266  3.783044  3.794113  3.806567  3.820494  3.835975  3.853085  3.871887  3.892437  3.914778  3.938942  3.964951  3.992811  4.022522  4.054068  4.087426  4.122561             4.159432 | 124.879264  124.879532  124.879909  124.880435  124.881166  124.882173  124.883552  124.885429  124.887964  124.891367  124.895905  124.901916  124.909822  124.920154  124.933563  124.950848  124.972975  125.001107  125.036628  125.081167  125.136629  125.205215  125.289441  125.392162  125.516577  125.666235  125.845035  126.057207  126.307298  126.600135  126.940785  127.334502  127.786673  128.302744  128.888156  129.548266  130.288275  131.113154  132.027575   133.03585  134.141879  135.349104  136.660474   138.07843  139.604887  141.241238  142.988361  144.846638  146.815981  148.895859 |  52.178399    52.17845   52.178522   52.178623   52.178763   52.178956    52.17922    52.17958   52.180066   52.180718   52.181588    52.18274   52.184255   52.186235   52.188804   52.192116   52.196355   52.201745   52.208549   52.217079   52.227699    52.24083    52.25695   52.276603   52.300397   52.329004   52.363162   52.403665   52.451368   52.507168   52.572005   52.646843    52.73266   52.830435   52.941129    53.06567   53.204939   53.359751   53.530846    53.71887   53.924369    54.14778   54.389424   54.649507   54.928114   55.225219   55.540681   55.874259   56.225614     56.594321 | 1.088387  1.088388  1.088389   1.08839  1.088393  1.088396    1.0884  1.088405  1.088413  1.088423  1.088436  1.088454  1.088478  1.088508  1.088548    1.0886  1.088665  1.088749  1.088854  1.088987  1.089151  1.089355  1.089605  1.089909  1.090278  1.090721   1.09125  1.091877  1.092615  1.093478   1.09448  1.095636  1.096961  1.098469  1.100174  1.102091  1.104231  1.106608   1.10923  1.112107  1.115246  1.118652  1.122329  1.126277  1.130496  1.134985  1.139737  1.144749  1.150013   1.15552 |  1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359159  1.359159  1.359158  1.359157  1.359156  1.359154  1.359152   1.35915  1.359146  1.359142  1.359136  1.359129  1.359121   1.35911  1.359096   1.35908   1.35906  1.359035  1.359006  1.358971   1.35893  1.358882  1.358825   1.35876  1.358685  1.358599  1.358502  1.358393  1.358271  1.358135  1.357986  1.357822  1.357644  1.357452  1.357245  1.357024   1.35679  1.356543  1.356284  1.356013  1.355732  1.355441          1.355142 | 0.536159   0.53616   0.53616  0.536161  0.536162  0.536164  0.536166  0.536169  0.536172  0.536178  0.536184  0.536193  0.536205  0.536221  0.536241  0.536267  0.536301  0.536343  0.536397  0.536464  0.536548  0.536651  0.536778  0.536933   0.53712  0.537344  0.537612  0.537929  0.538302  0.538737  0.539241  0.539822  0.540486   0.54124   0.54209  0.543042  0.544102  0.545273   0.54656  0.547964  0.549488  0.551132  0.552895  0.554774  0.556769  0.558873  0.561083  0.563393  0.565797      0.568289 | -0.201869   -0.20187   -0.20187  -0.201871  -0.201873  -0.201875  -0.201878  -0.201881  -0.201887  -0.201893  -0.201902  -0.201914   -0.20193  -0.201951  -0.201977  -0.202011  -0.202054  -0.202109  -0.202178  -0.202264  -0.202371  -0.202503  -0.202663  -0.202859  -0.203094  -0.203375  -0.203709  -0.204104  -0.204565  -0.205102  -0.205721  -0.206431  -0.207239  -0.208153   -0.20918  -0.210325  -0.211593   -0.21299  -0.214519  -0.216181  -0.217978  -0.219909  -0.221973  -0.224167  -0.226487  -0.228928  -0.231484   -0.23415  -0.236917     -0.239777 |  0.66571   0.66571   0.66571   0.66571   0.66571   0.66571  0.665711  0.665711  0.665711  0.665712  0.665713  0.665714  0.665716  0.665718  0.665721  0.665724  0.665729  0.665734  0.665741   0.66575  0.665761  0.665774   0.66579  0.665809  0.665832   0.66586  0.665893  0.665931  0.665976  0.666028  0.666088  0.666157  0.666235  0.666323  0.666422  0.666532  0.666653  0.666788  0.666935  0.667095  0.667269  0.667456  0.667656   0.66787  0.668098  0.668338  0.668592  0.668858  0.669136      0.669426 | 100.000544 | 100.000667  100.000958  100.001368  100.001941  100.002736  100.003832  100.005334  100.007376  100.010135   100.01384  100.018779  100.025321  100.033927  100.045173  100.059769  100.078585  100.102673  100.133301  100.171976  100.220477  100.280879  100.355586   100.44735  100.559291  100.694915  100.858118  101.053187  101.284787   101.55795  101.878036    102.2507  102.681846  103.177562  103.744063  104.387622  105.114494  105.930847  106.842688  107.855798  108.975664  110.207429  111.555845  113.025233  114.619455    116.3419  118.195473  120.182596  122.305217  124.564821          126.962454 | 128.191889 | 0.720701  0.720701  0.720702  0.720703  0.720705  0.720707   0.72071  0.720714  0.720719  0.720726  0.720735  0.720748  0.720764  0.720786  0.720813  0.720849  0.720895  0.720953  0.721027  0.721119  0.721234  0.721376   0.72155  0.721763   0.72202  0.722329  0.722697  0.723134  0.723649  0.724251  0.724949  0.725755  0.726678  0.727729  0.728918  0.730253  0.731745    0.7334  0.735227  0.737231  0.739416  0.741787  0.744346  0.747094  0.750029  0.753151  0.756456   0.75994  0.763598       0.767424 | 2.591124  2.591127  2.591131  2.591138  2.591146  2.591158  2.591175  2.591197  2.591228  2.591269  2.591323  2.591395   2.59149  2.591614  2.591774  2.591981  2.592246  2.592583  2.593009  2.593542  2.594206  2.595027  2.596035  2.597263   2.59875  2.600538  2.602673  2.605204  2.608184   2.61167  2.615719  2.620392  2.625748   2.63185  2.638755  2.646521  2.655201  2.664846  2.675499    2.6872   2.69998  2.713865  2.728872  2.745011  2.762286  2.780692  2.800217  2.820843  2.842547          2.8653 | 100.000544  100.000786  100.001126  100.001604   100.00227  100.003191  100.004458  100.006188  100.008535  100.011697  100.015931   100.02156  100.028995  100.038748  100.051456  100.067901  100.089039  100.116021  100.150226  100.193289  100.247129   100.31398  100.396412  100.497357  100.620124  100.768413  100.946316  101.158314   101.40926  101.704361  102.049136  102.449378    102.9111  103.440471  104.043756  104.727238   105.49715  106.359597  107.320492  108.385486  109.559908  110.848718   112.25646  113.787233  115.444667  117.231909  119.151623  121.205989  123.396721  125.725078  128.191889 |  100.0 |   0.000544 | 128.191889 |
    | 00:00 | 3.717838  3.717842  3.717848  3.717856  3.717867  3.717883  3.717905  3.717934  3.717974  3.718028    3.7181  3.718195  3.718322  3.718487  3.718702   3.71898  3.719337  3.719793  3.720369  3.721095  3.722002  3.723126  3.724512  3.726208  3.728268  3.730754  3.733734  3.737281  3.741475    3.7464  3.752145  3.758802  3.766467  3.775234    3.7852  3.796457  3.809095  3.823199  3.838847   3.85611  3.875048  3.895714  3.918148  3.942379  3.968426  3.996295  4.025983  4.057473  4.090741             4.125754 | 124.879143   124.87936  124.879665  124.880093  124.880688   124.88151  124.882639  124.884179  124.886265  124.889074  124.892829  124.897817  124.904397  124.913021  124.924246  124.938761    124.9574  124.981173  125.011288  125.049174  125.096511  125.155251  125.227639  125.316236  125.423932  125.553954  125.709869  125.895578  126.115301  126.373556  126.675123  127.025002  127.428362  127.890482  128.416685   129.01227  129.682439  130.432228  131.266434   132.18955  133.205713  134.318647  135.531629  136.847453  138.268417  139.796311  141.432417  143.177523  145.031939  146.995523 |  52.178375   52.178417   52.178475   52.178557   52.178671   52.178829   52.179045   52.179341    52.17974   52.180279   52.180998   52.181954   52.183215   52.184868   52.187019     52.1898   52.193371   52.197926   52.203695   52.210952   52.220018   52.231265   52.245122   52.262077    52.28268   52.307543   52.337342   52.372813    52.41475   52.463998   52.521448   52.588022   52.664668   52.752342   52.851998   52.964567   53.090948   53.231988   53.388469   53.561096   53.750481    53.95714   54.181478   54.423791   54.684261   54.962953   55.259826   55.574729   55.907413     56.257537 | 1.088387  1.088387  1.088388  1.088389  1.088391  1.088394  1.088397  1.088401  1.088408  1.088416  1.088427  1.088442  1.088462  1.088487  1.088521  1.088564  1.088619   1.08869  1.088779  1.088892  1.089032  1.089207  1.089421  1.089684  1.090003  1.090389   1.09085  1.091399  1.092048   1.09281  1.093699  1.094728  1.095911  1.097264  1.098801  1.100535  1.102479  1.104647  1.107048  1.109693  1.112591  1.115746  1.119166  1.122851  1.126804  1.131023  1.135507   1.14025  1.145247  1.150491 |  1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359159  1.359159  1.359159  1.359158  1.359157  1.359155  1.359154  1.359152  1.359149  1.359145   1.35914  1.359134  1.359127  1.359118  1.359106  1.359092  1.359075  1.359053  1.359028  1.358998  1.358962  1.358919  1.358869  1.358811  1.358744  1.358667   1.35858  1.358481   1.35837  1.358246  1.358109  1.357958  1.357793  1.357614  1.357421  1.357214  1.356993  1.356759  1.356513  1.356254  1.355984  1.355704          1.355415 | 0.536159  0.536159   0.53616   0.53616  0.536161  0.536163  0.536164  0.536167   0.53617  0.536174   0.53618  0.536187  0.536197   0.53621  0.536227  0.536249  0.536277  0.536313  0.536359  0.536416  0.536487  0.536576  0.536685  0.536818   0.53698  0.537176  0.537409  0.537687  0.538016    0.5384  0.538848  0.539366   0.53996  0.540638  0.541406   0.54227  0.543235  0.544307   0.54549  0.546787    0.5482   0.54973  0.551379  0.553144  0.555024  0.557016  0.559117   0.56132  0.563622      0.566014 | -0.201869  -0.201869   -0.20187  -0.201871  -0.201872  -0.201874  -0.201876  -0.201879  -0.201883  -0.201889  -0.201896  -0.201906  -0.201919  -0.201936  -0.201959  -0.201987  -0.202024   -0.20207  -0.202129  -0.202202  -0.202294  -0.202406  -0.202545  -0.202714  -0.202918  -0.203163  -0.203456  -0.203802  -0.204209  -0.204685  -0.205236   -0.20587  -0.206596   -0.20742  -0.208349   -0.20939  -0.210549  -0.211831   -0.21324  -0.214779  -0.216449  -0.218252  -0.220186  -0.222251  -0.224444  -0.226759  -0.229194  -0.231741  -0.234394     -0.237147 |  0.66571   0.66571   0.66571   0.66571   0.66571   0.66571   0.66571  0.665711  0.665711  0.665712  0.665712  0.665713  0.665715  0.665717  0.665719  0.665722  0.665726   0.66573  0.665736  0.665744  0.665753  0.665764  0.665778  0.665795  0.665815  0.665839  0.665868  0.665902  0.665941  0.665988  0.666041  0.666102  0.666172  0.666252  0.666341  0.666441  0.666552  0.666675  0.666811  0.666959  0.667119  0.667293  0.667481  0.667681  0.667895  0.668122  0.668362  0.668615   0.66888      0.669156 | 100.000436 | 100.000535   100.00077  100.001103  100.001568  100.002215   100.00311  100.004339  100.006015  100.008286  100.011343   100.01543  100.020859  100.028022  100.037408  100.049628  100.065427  100.085718  100.111599  100.144386  100.185638  100.237187  100.301161  100.380015  100.476547  100.593919  100.735669  100.905713  101.108346  101.348224  101.630351  101.960042  102.342886  102.784695  103.291452  103.869243  104.524191  105.262392  106.089835  107.012343  108.035504  109.164611  110.404614  111.760076  113.235134  114.833484  116.558354  118.412508   120.39824  122.517387          124.771341 | 125.928277 | 0.720701  0.720701  0.720702  0.720703  0.720704  0.720706  0.720708  0.720711  0.720715  0.720721  0.720729  0.720739  0.720753  0.720771  0.720794  0.720824  0.720863  0.720912  0.720974  0.721053  0.721151  0.721273  0.721422  0.721606  0.721828  0.722097  0.722419  0.722801  0.723254  0.723785  0.724405  0.725122  0.725947   0.72689  0.727961  0.729169  0.730524  0.732034  0.733707  0.735549  0.737567  0.739764  0.742145   0.74471   0.74746  0.750395  0.753514  0.756812  0.760285       0.763929 | 2.591122  2.591125  2.591128  2.591134  2.591141   2.59115  2.591164  2.591182  2.591207  2.591241  2.591286  2.591346  2.591425  2.591528  2.591663  2.591836   2.59206  2.592345  2.592705  2.593159  2.593726  2.594429  2.595295  2.596355  2.597643  2.599197  2.601059  2.603276  2.605896  2.608973  2.612562  2.616719  2.621504  2.626977  2.633195  2.640217  2.648097  2.656887  2.666634  2.677382  2.689167  2.702017  2.715958  2.731005  2.747167  2.764445  2.782835  2.802323  2.822892        2.844518 | 100.000436  100.000631  100.000906  100.001294  100.001835  100.002586  100.003621  100.005039  100.006968  100.009573  100.013071  100.017734  100.023911  100.032038  100.042658  100.056444  100.074217  100.096975  100.125919  100.162477  100.208338  100.265474   100.33617  100.423047  100.529081  100.657619  100.812387  100.997492  101.217415  101.476992  101.781393  102.136081  102.546771  103.019378  103.559952  104.174619  104.869507  105.650678  106.524056  107.495364  108.570057  109.753272  111.049775  112.463925  113.999643  115.660396  117.449179  119.368521  121.420484  123.606678  125.928277 |  100.0 |   0.000436 | 125.928277 |
    | 00:30 | 3.717836  3.717839  3.717844   3.71785   3.71786  3.717873   3.71789  3.717914  3.717947  3.717992  3.718051   3.71813  3.718235  3.718373  3.718553  3.718786  3.719086  3.719471  3.719959  3.720576  3.721348   3.72231  3.723499  3.724959   3.72674  3.728896  3.731491  3.734591   3.73827  3.742607  3.747685  3.753594  3.760423  3.768266  3.777216  3.787365  3.798805  3.811621  3.825896  3.841705  3.859114  3.878182  3.898957  3.921477  3.945768  3.971848   3.99972   4.02938   4.06081             4.093986 | 124.879046  124.879221  124.879468  124.879815  124.880299  124.880969  124.881893  124.883155  124.884871  124.887186  124.890289  124.894424  124.899893  124.907082  124.916468   124.92864  124.944319  124.964379  124.989872  125.022049  125.062385  125.112608  125.174713  125.250992  125.344043  125.456789   125.59248  125.754697   125.94734  126.174615  126.441005  126.751242  127.110254  127.523122   127.99502  128.531145  129.136658  129.816608  130.575867  131.419065  132.350525   133.37421  134.493679   135.71205   137.03197  138.455601  139.984614  141.620189  143.363029  145.213376 |  52.178357    52.17839   52.178438   52.178504   52.178597   52.178725   52.178902   52.179144   52.179473   52.179917   52.180512   52.181304   52.182352    52.18373   52.185528   52.187861   52.190865   52.194708   52.199592   52.205756   52.213482     52.2231   52.234991   52.249592   52.267398   52.288964   52.314908   52.345907   52.382696   52.426065   52.476853   52.535938   52.604231   52.682658   52.772155    52.87365   52.988047   53.116214   53.258965   53.417051   53.591142    53.78182   53.989569   54.214767   54.457684    54.71848   54.997202   55.293795   55.608098     55.939856 | 1.088386  1.088387  1.088387  1.088388   1.08839  1.088392  1.088395  1.088398  1.088404   1.08841   1.08842  1.088432  1.088448   1.08847  1.088497  1.088534   1.08858   1.08864  1.088716  1.088811  1.088931   1.08908  1.089264  1.089491  1.089767  1.090101  1.090503  1.090983  1.091552  1.092224  1.093009  1.093923  1.094978  1.096189   1.09757  1.099135  1.100896  1.102868  1.105061  1.107486  1.110153   1.11307  1.116241  1.119672  1.123366  1.127323  1.131541  1.136019  1.140752  1.145733 |  1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359159  1.359159  1.359158  1.359157  1.359156  1.359155  1.359153  1.359151  1.359148  1.359144  1.359139  1.359132  1.359124  1.359115  1.359102  1.359087  1.359069  1.359047  1.359021  1.358989  1.358951  1.358907  1.358856  1.358796  1.358728  1.358649   1.35856  1.358459  1.358347  1.358221  1.358083  1.357931  1.357765  1.357585  1.357391  1.357184  1.356963  1.356729  1.356483  1.356225  1.355956          1.355677 | 0.536159  0.536159   0.53616   0.53616  0.536161  0.536162  0.536163  0.536165  0.536168  0.536171  0.536176  0.536182   0.53619  0.536201  0.536215  0.536234  0.536258  0.536288  0.536326  0.536375  0.536436  0.536512  0.536605   0.53672   0.53686   0.53703  0.537233  0.537477  0.537765  0.538104  0.538501  0.538961  0.539492    0.5401  0.540791  0.541572  0.542449  0.543428  0.544512  0.545705  0.547011  0.548432  0.549969  0.551622   0.55339   0.55527   0.55726  0.559356  0.561552      0.563844 | -0.201869  -0.201869  -0.201869   -0.20187  -0.201871  -0.201872  -0.201874  -0.201877   -0.20188  -0.201885  -0.201891  -0.201899   -0.20191  -0.201925  -0.201943  -0.201967  -0.201998  -0.202037  -0.202087  -0.202149  -0.202227  -0.202324  -0.202443  -0.202589  -0.202766  -0.202979  -0.203235  -0.203538  -0.203897  -0.204317  -0.204806  -0.205372  -0.206021  -0.206762  -0.207601  -0.208545  -0.209601  -0.210774  -0.212068  -0.213488  -0.215036  -0.216714  -0.218522   -0.22046  -0.222525  -0.224716  -0.227027  -0.229454  -0.231991     -0.234633 |  0.66571   0.66571   0.66571   0.66571   0.66571   0.66571   0.66571   0.66571  0.665711  0.665711  0.665712  0.665713  0.665714  0.665715  0.665717   0.66572  0.665723  0.665727  0.665732  0.665738  0.665746  0.665756  0.665768  0.665782    0.6658  0.665821  0.665846  0.665876  0.665911  0.665952  0.665999  0.666054  0.666117  0.666188  0.666269  0.666359  0.666461  0.666573  0.666697  0.666833  0.666982  0.667144  0.667318  0.667506  0.667706   0.66792  0.668146  0.668386  0.668637      0.668901 | 100.000349 | 100.000429  100.000619  100.000888  100.001265  100.001792  100.002522  100.003527  100.004901  100.006768  100.009288  100.012666  100.017166  100.023119  100.030944  100.041161   100.05441  100.071478  100.093316  100.121069  100.156102  100.200024  100.254716   100.32236  100.405457  100.506848  100.629734   100.77768  100.954617  101.164843  101.413003  101.704067  102.043301  102.436224  102.888558  103.406175  103.995031  104.661108  105.410336  106.248536  107.181348  108.214174   109.35212  110.599949  111.962043  113.442367  115.044452  116.771379  118.625774  120.609811          122.725222 |  123.81218 |   0.7207  0.720701  0.720701  0.720702  0.720703  0.720704  0.720706  0.720709  0.720713  0.720717  0.720724  0.720732  0.720744  0.720759  0.720778  0.720803  0.720836  0.720877   0.72093  0.720997   0.72108  0.721184  0.721313  0.721471  0.721663  0.721896  0.722176  0.722511  0.722908  0.723376  0.723924  0.724561  0.725296   0.72614  0.727103  0.728193  0.729421  0.730795  0.732323  0.734012   0.73587  0.737901  0.740109  0.742497  0.745068  0.747821  0.750756   0.75387  0.757161       0.760623 | 2.591121  2.591123  2.591126   2.59113  2.591136  2.591144  2.591155   2.59117  2.591191  2.591219  2.591256  2.591305  2.591371  2.591457  2.591569  2.591715  2.591903  2.592143  2.592449  2.592834  2.593317  2.593918  2.594662  2.595575  2.596688  2.598036  2.599657  2.601595  2.603894  2.606603  2.609776  2.613467  2.617731  2.622627  2.628213  2.634546  2.641681  2.649672  2.658568  2.668414  2.679252  2.691116  2.704033  2.718026  2.733109  2.749289  2.766568  2.784938  2.804387        2.824897 | 100.000349  100.000506  100.000729  100.001043  100.001482  100.002094  100.002939    100.0041  100.005683  100.007827  100.010714  100.014573  100.019699  100.026462  100.035326  100.046866   100.06179  100.080958  100.105414  100.136403  100.175405  100.224159  100.284687  100.359327  100.450743  100.561954   100.69634  100.857649  101.049999  101.277865  101.546062  101.859718  102.224239  102.645258  103.128589  103.680165  104.305971  105.011982  105.804094  106.688053  107.669397  108.753395  109.944994  111.248775  112.668921  114.209184  115.872872   117.66284  119.581487  121.630762   123.81218 |  100.0 |   0.000349 |  123.81218 |
    | 01:00 | 3.717834  3.717837  3.717841  3.717846  3.717854  3.717864  3.717879  3.717898  3.717925  3.717962  3.718011  3.718077  3.718164  3.718278  3.718429  3.718624  3.718876    3.7192  3.719613  3.720136  3.720793  3.721614  3.722633  3.723888  3.725424  3.727291  3.729545  3.732249  3.735469   3.73928   3.74376  3.748992  3.755063  3.762062   3.77008  3.779209  3.789538  3.801156  3.814145  3.828586  3.844548  3.862098  3.881288  3.902165  3.924765  3.949111  3.975217  4.003087  4.032714              4.06408 | 124.878967  124.879108  124.879308   124.87959  124.879983   124.88053  124.881284  124.882318  124.883727  124.885633  124.888195  124.891618  124.896159  124.902145  124.909982  124.920176  124.933347   124.95025  124.971799  124.999084  125.033401  125.076271  125.129464  125.195021  125.275274  125.372856  125.490721  125.632138  125.800696  126.000291  126.235109  126.509599  126.828436  127.196479  127.618718  128.100218  128.646055  129.261249  129.950703  130.719129  131.570992   132.51045  133.541303  134.666949  135.890355  137.214027  138.639999   140.16983  141.804603  143.544942 |  52.178342   52.178369   52.178407   52.178461   52.178536   52.178641   52.178786   52.178984   52.179254   52.179619    52.18011   52.180766   52.181636   52.182784   52.184285   52.186239   52.188763   52.192001    52.19613   52.201357   52.207931   52.216141   52.226327   52.238878   52.254239    52.27291   52.295453   52.322488   52.354693   52.392803   52.437603   52.489923    52.55063   52.620619     52.7008   52.792086   52.895379   53.011555   53.141453   53.285857   53.445485   53.620975    53.81288   54.021653   54.247646   54.491105   54.752169    55.03087   55.327137     55.640802 | 1.088386  1.088386  1.088387  1.088388  1.088389  1.088391  1.088393  1.088396    1.0884  1.088406  1.088413  1.088424  1.088437  1.088455  1.088478  1.088508  1.088548  1.088598  1.088662  1.088743  1.088845  1.088972   1.08913  1.089325  1.089563  1.089852  1.090201   1.09062  1.091119  1.091709  1.092402  1.093211   1.09415  1.095231  1.096469  1.097877  1.099469  1.101258  1.103256  1.105474  1.107922   1.11061  1.113544   1.11673  1.120173  1.123874  1.127833   1.13205  1.136521  1.141243 | 1.359161   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359159  1.359159  1.359158  1.359157  1.359156  1.359154  1.359152   1.35915  1.359146  1.359142  1.359137   1.35913  1.359122  1.359111  1.359098  1.359083  1.359063   1.35904  1.359013   1.35898  1.358941  1.358896  1.358843  1.358782  1.358711  1.358631   1.35854  1.358438  1.358324  1.358197  1.358057  1.357903  1.357736  1.357556  1.357361  1.357153  1.356932  1.356699  1.356453  1.356196          1.355928 | 0.536159  0.536159  0.536159   0.53616   0.53616  0.536161  0.536162  0.536164  0.536166  0.536169  0.536173  0.536178  0.536185  0.536194  0.536206  0.536221  0.536241  0.536267  0.536299   0.53634  0.536392  0.536457  0.536537  0.536636  0.536757  0.536904  0.537081  0.537293  0.537545  0.537844  0.538194  0.538603  0.539075  0.539619   0.54024  0.540945  0.541739  0.542629   0.54362  0.544715  0.545919  0.547234  0.548663  0.550206  0.551862  0.553632  0.555512  0.557499   0.55959      0.561779 | -0.201868  -0.201869  -0.201869   -0.20187   -0.20187  -0.201872  -0.201873  -0.201875  -0.201878  -0.201882  -0.201887  -0.201894  -0.201903  -0.201915   -0.20193   -0.20195  -0.201976  -0.202009  -0.202052  -0.202105  -0.202171  -0.202254  -0.202356  -0.202482  -0.202635   -0.20282  -0.203042  -0.203308  -0.203623  -0.203993  -0.204426  -0.204929  -0.205509  -0.206173  -0.206929  -0.207783  -0.208741  -0.209811  -0.210997  -0.212303  -0.213734  -0.215291  -0.216976  -0.218789  -0.220729  -0.222795  -0.224983  -0.227289  -0.229709     -0.232236 |  0.66571   0.66571   0.66571   0.66571   0.66571   0.66571   0.66571   0.66571  0.665711  0.665711  0.665711  0.665712  0.665713  0.665714  0.665716  0.665718  0.665721  0.665724  0.665728  0.665734   0.66574  0.665749  0.665759  0.665772  0.665787  0.665805  0.665827  0.665853  0.665884   0.66592  0.665962  0.666011  0.666067  0.666131  0.666203  0.666285  0.666377   0.66648  0.666594  0.666719  0.666856  0.667005  0.667167  0.667342   0.66753   0.66773  0.667944   0.66817  0.668408      0.668659 | 100.000279 | 100.000343  100.000497  100.000714   100.00102  100.001449  100.002043  100.002864   100.00399  100.005523  100.007598  100.010387  100.014112  100.019055   100.02557  100.034101  100.045197  100.059534  100.077935  100.101393  100.131099  100.168463  100.215145  100.273075  100.344482  100.431913   100.53825  100.666726  100.820929  101.004804  101.222646  101.479082  101.779047  102.127755   102.53065  102.993362  103.521654  104.121352  104.798294  105.558252  106.406879  107.349639  108.391753  109.538147  110.793404  112.161732  113.646931  115.252377  116.981007  118.835319          120.817374 | 121.836899 |   0.7207  0.720701  0.720701  0.720702  0.720702  0.720704  0.720705  0.720707   0.72071  0.720714  0.720719  0.720727  0.720736  0.720748  0.720765  0.720786  0.720813  0.720848  0.720893  0.720949   0.72102  0.721109  0.721219  0.721355  0.721521  0.721723  0.721966  0.722258  0.722606  0.723017  0.723501  0.724065  0.724719  0.725473  0.726336  0.727317  0.728427  0.729673  0.731065   0.73261  0.734316  0.736188  0.738231  0.740449  0.742846  0.745421  0.748176   0.75111  0.754219       0.757502 |  2.59112  2.591122  2.591124  2.591127  2.591132  2.591139  2.591148   2.59116  2.591177    2.5912  2.591231  2.591272  2.591326  2.591398  2.591492  2.591614  2.591772  2.591974  2.592232  2.592559   2.59297  2.593483   2.59412  2.594905  2.595865  2.597032  2.598441  2.600131  2.602144  2.604525  2.607324  2.610593  2.614384  2.618754   2.62376  2.629457  2.635901  2.643147  2.651245  2.660243  2.670185  2.681109  2.693048  2.706027  2.720068  2.735183  2.751379  2.768654  2.787002         2.80641 | 100.000279  100.000406  100.000585   100.00084  100.001197  100.001694  100.002384  100.003333  100.004631  100.006394  100.008773  100.011963  100.016211  100.021833  100.029222   100.03887  100.051384  100.067507  100.088141   100.11437  100.147488  100.189022   100.24076  100.304776  100.383452  100.479498  100.595969  100.736273  100.904178  101.103806  101.339625  101.616422  101.939283  102.313549  102.744771  103.238662  103.801032  104.437734  105.154588  105.957325  106.851522  107.842534   108.93545  110.135036  111.445695   112.87144  114.415863  116.082121  117.872934  119.790579  121.836899 |  100.0 |   0.000279 | 121.836899 |
    | 01:30 | 3.717833  3.717835  3.717838  3.717843  3.717849  3.717857  3.717869  3.717885  3.717907  3.717937  3.717978  3.718032  3.718104    3.7182  3.718325  3.718488    3.7187  3.718973  3.719321  3.719764  3.720322  3.721022  3.721893   3.72297  3.724293  3.725906  3.727861  3.730214  3.733028   3.73637  3.740313  3.744936   3.75032  3.756551  3.763718  3.771908  3.781212  3.791717  3.803508  3.816666  3.831265  3.847376  3.865059  3.884366  3.905339  3.928011  3.952406  3.978533  4.006396             4.035986 | 124.878904  124.879018   124.87918  124.879407  124.879727  124.880172  124.880787  124.881634  124.882789  124.884357  124.886471  124.889301  124.893067  124.898044   124.90458  124.913107  124.924156  124.938379  124.956568  124.979671  125.008822  125.045357  125.090841  125.147087  125.216179  125.300484  125.402671  125.525717  125.672909  125.847839  126.054397  126.296744   126.57929  126.906653  127.283618  127.715085   128.20601  128.761346  129.385978  130.084661  130.861954  131.722162  132.669284   133.70696  134.838437  136.066536   137.39363  138.821633  140.351995   141.98571 |   52.17833   52.178351   52.178382   52.178426   52.178487   52.178573   52.178691   52.178853   52.179074   52.179375    52.17978   52.180322   52.181044   52.181998    52.18325   52.184884   52.187001   52.189727   52.193212   52.197638   52.203222   52.210221   52.218932   52.229702   52.242928   52.259063   52.278613   52.302145    52.33028   52.363697   52.403129   52.449355   52.503198   52.565512   52.637177   52.719083   52.812122    52.91717   53.035079   53.166655   53.312654   53.473762   53.650589   53.843655   54.053388   54.280114   54.524055   54.785333   55.063962     55.359861 | 1.088386  1.088386  1.088387  1.088387  1.088388   1.08839  1.088391  1.088394  1.088397  1.088402  1.088408  1.088417  1.088428  1.088443  1.088462  1.088487   1.08852  1.088563  1.088617  1.088685  1.088772   1.08888  1.089015  1.089182  1.089387  1.089637   1.08994  1.090305  1.090741  1.091258  1.091869  1.092584  1.093417   1.09438  1.095487  1.096751  1.098186  1.099805   1.10162  1.103643  1.105885  1.108356  1.111063  1.114014  1.117214  1.120667  1.124374  1.128335   1.13255  1.137015 | 1.359161   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359159  1.359159  1.359158  1.359158  1.359157  1.359155  1.359154  1.359152  1.359149  1.359145  1.359141  1.359135  1.359128  1.359119  1.359108  1.359094  1.359078  1.359058  1.359034  1.359005  1.358971  1.358931  1.358884  1.358829  1.358767  1.358695  1.358613   1.35852  1.358416  1.358301  1.358172  1.358031  1.357876  1.357708  1.357527  1.357332  1.357124  1.356903  1.356669  1.356424          1.356168 | 0.536159  0.536159  0.536159  0.536159   0.53616  0.536161  0.536162  0.536163  0.536165  0.536167   0.53617  0.536174   0.53618  0.536188  0.536198   0.53621  0.536227  0.536249  0.536276  0.536311  0.536355   0.53641  0.536479  0.536564  0.536668  0.536795  0.536948  0.537133  0.537354  0.537616  0.537925  0.538286  0.538706  0.539191  0.539747  0.540381  0.541099  0.541907  0.542809  0.543811  0.544918  0.546132  0.547456  0.548891  0.550439  0.552099   0.55387  0.555749  0.557734      0.559819 | -0.201868  -0.201869  -0.201869  -0.201869   -0.20187  -0.201871  -0.201872  -0.201874  -0.201876  -0.201879  -0.201884  -0.201889  -0.201897  -0.201907   -0.20192  -0.201936  -0.201958  -0.201986  -0.202022  -0.202067  -0.202123  -0.202194  -0.202282   -0.20239  -0.202522  -0.202682  -0.202876  -0.203107  -0.203383  -0.203709  -0.204092  -0.204538  -0.205054  -0.205648  -0.206326  -0.207096  -0.207965  -0.208938  -0.210021  -0.211219  -0.212537  -0.213978  -0.215543  -0.217234  -0.219052  -0.220994  -0.223059  -0.225245  -0.227546     -0.229958 |  0.66571   0.66571   0.66571   0.66571   0.66571   0.66571   0.66571   0.66571   0.66571  0.665711  0.665711  0.665712  0.665712  0.665714  0.665715  0.665717  0.665719  0.665722  0.665725   0.66573  0.665736  0.665743  0.665751  0.665762  0.665775  0.665791   0.66581  0.665833   0.66586  0.665892  0.665929  0.665973  0.666023   0.66608  0.666145  0.666219  0.666302  0.666396  0.666499  0.666614   0.66674  0.666878  0.667028  0.667191  0.667366  0.667554  0.667754  0.667967  0.668193       0.66843 | 100.000223 | 100.000275  100.000398  100.000574  100.000822   100.00117  100.001654  100.002324  100.003245  100.004503  100.006209   100.00851   100.01159  100.015689  100.021107  100.028221  100.037502  100.049529  100.065012  100.084812  100.109963  100.141701  100.181481  100.231011  100.292269   100.36753  100.459382  100.570745  100.704881  100.865397  101.056247  101.281719  101.546418  101.855241  102.213345  102.626099   103.09904  103.637817  104.248131  104.935675  105.706069  106.564798  107.517157  108.568192  109.722653  110.984952  112.359129  113.848828  115.457275  117.187273          119.041196 | 119.995856 |   0.7207    0.7207  0.720701  0.720701  0.720702  0.720703  0.720704  0.720706  0.720708  0.720711  0.720716  0.720722   0.72073   0.72074  0.720753  0.720771  0.720794  0.720823  0.720861  0.720909  0.720969  0.721045  0.721139  0.721256  0.721399  0.721573  0.721784  0.722038  0.722342  0.722703  0.723129  0.723627  0.724208  0.724879  0.725651  0.726532  0.727532  0.728661  0.729925  0.731335  0.732897  0.734618  0.736503  0.738558  0.740786  0.743189  0.745769  0.748526  0.751457       0.754562 | 2.591119  2.591121  2.591123  2.591125  2.591129  2.591134  2.591142  2.591152  2.591166  2.591185   2.59121  2.591244  2.591289  2.591349  2.591427  2.591529  2.591661  2.591832   2.59205  2.592327  2.592676  2.593113  2.593658  2.594331  2.595158  2.596167  2.597389   2.59886  2.600618  2.602706   2.60517  2.608058  2.611422  2.615313  2.619788  2.624901  2.630707  2.637261  2.644613  2.652815  2.661912  2.671946  2.682952  2.694962     2.708  2.722085  2.737228  2.753435  2.770704        2.789027 | 100.000223  100.000325   100.00047  100.000676  100.000965   100.00137  100.001931  100.002707   100.00377  100.005218  100.007177  100.009811  100.013328  100.017995  100.024147  100.032203  100.042683  100.056226   100.07361  100.095778  100.123857  100.159186  100.203339  100.258152  100.325747  100.408549  100.509309  100.631115  100.777401  100.951949  101.158883  101.402656  101.688027  102.020033  102.403949  102.845243  103.349523  103.922482  104.569833   105.29725  106.110303  107.014398  108.014719  109.116177  110.323364  111.640515  113.071478   114.61969   116.28817  118.079504  119.995856 |  100.0 |   0.000223 | 119.995856 |
    | 02:00 | 3.717832  3.717834  3.717836   3.71784  3.717845  3.717852  3.717861  3.717875  3.717893  3.717917  3.717951  3.717996  3.718055  3.718135  3.718239  3.718375  3.718553  3.718782  3.719076   3.71945  3.719924  3.720519  3.721263  3.722185  3.723322  3.724714  3.726406  3.728451  3.730904  3.733828  3.737291  3.741367  3.746132  3.751667  3.758057  3.765389  3.773749  3.783225  3.793901  3.805861  3.819181  3.833934  3.850187  3.867997  3.887414  3.908477  3.931217  3.955654  3.981798             4.009649 | 124.878854  124.878945  124.879076   124.87926  124.879519  124.879881  124.880383  124.881075  124.882022   124.88331  124.885051  124.887389  124.890509  124.894643  124.900087   124.90721  124.916467  124.928419  124.943749  124.963283  124.988008  125.019096  125.057925  125.106102  125.165483  125.238187   125.32662   125.43348  125.561764  125.714773  125.896102  126.109625  126.359479   126.65003  126.985839  127.371614  127.812162  128.312331  128.876953   129.51078  130.218421  131.004286  131.872526  132.826984   133.87115  135.008124  136.240588  137.570789  139.000524  140.531144 |   52.17832   52.178338   52.178363   52.178398   52.178448   52.178517   52.178613   52.178746   52.178927   52.179174   52.179508   52.179956   52.180554   52.181346   52.182389   52.183754   52.185528   52.187818   52.190756   52.194498   52.199235    52.20519   52.212628   52.221854   52.233224   52.247141   52.264064   52.284506   52.309036   52.338279   52.372913   52.413667   52.461315    52.51667   52.580574   52.653891   52.737494    52.83225   52.939013   53.058605   53.191808   53.339345   53.501874   53.679975    53.87414   54.084771    54.31217   54.556537   54.817976     55.096487 | 1.088386  1.088386  1.088386  1.088387  1.088388  1.088389   1.08839  1.088392  1.088395  1.088399  1.088404  1.088411   1.08842  1.088433  1.088449   1.08847  1.088497  1.088533  1.088579  1.088637   1.08871  1.088802  1.088918  1.089061  1.089237  1.089453  1.089715  1.090032  1.090412  1.090865  1.091401  1.092032  1.092769  1.093625  1.094613  1.095745  1.097035  1.098497  1.100141  1.101982   1.10403  1.106295  1.108786  1.111513   1.11448  1.117693  1.121154  1.124867   1.12883  1.133041 | 1.359161  1.359161   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916   1.35916  1.359159  1.359159  1.359158  1.359157  1.359156  1.359155  1.359153  1.359151  1.359148  1.359144  1.359139  1.359133  1.359125  1.359116  1.359104   1.35909  1.359073  1.359052  1.359027  1.358997  1.358961   1.35892  1.358872  1.358816  1.358751  1.358678  1.358594    1.3585  1.358395  1.358278  1.358148  1.358005   1.35785  1.357681  1.357498  1.357303  1.357095  1.356874   1.35664          1.356396 | 0.536159  0.536159  0.536159  0.536159   0.53616   0.53616  0.536161  0.536162  0.536163  0.536165  