"""Initialize command for Context Engine."""

import os
from pathlib import Path
from context_engine.core.config import ContextConfig
from context_engine.core.logger import setup_logger

logger = setup_logger(__name__)

def init_command() -> int:
    """Initialize Context Engine in the current directory."""
    try:
        project_root = Path.cwd()
        logger.info(f"Initializing Context Engine in {project_root}")
        
        # Create directory structure
        directories = [
            "context_engine/config",
            "context_engine/embeddings_db",
            "context_engine/summaries",
            "context_engine/logs",
            "context_engine/logs/errors",
            "team_context",
            ".context_payload"
        ]
        
        for dir_path in directories:
            full_path = project_root / dir_path
            full_path.mkdir(parents=True, exist_ok=True)
            logger.info(f"Created directory: {dir_path}")
        
        # Create default configuration
        config = ContextConfig.default()
        config.project.name = project_root.name
        
        config_path = project_root / "context_engine" / "config" / "context.yml"
        config.save_to_file(config_path)
        logger.info(f"Created configuration file: {config_path}")
        
        # Create .gitignore entries
        gitignore_path = project_root / ".gitignore"
        gitignore_entries = [
            "# Context Engine",
            "context_engine/",
            ".context_payload/",
            ""
        ]
        
        if gitignore_path.exists():
            with open(gitignore_path, 'r', encoding='utf-8') as f:
                existing_content = f.read()
            
            # Check if entries already exist
            if "context_engine/" not in existing_content:
                with open(gitignore_path, 'a', encoding='utf-8') as f:
                    f.write("\n" + "\n".join(gitignore_entries))
                logger.info("Added Context Engine entries to .gitignore")
            else:
                logger.info(".gitignore already contains Context Engine entries")
        else:
            with open(gitignore_path, 'w', encoding='utf-8') as f:
                f.write("\n".join(gitignore_entries))
            logger.info("Created .gitignore with Context Engine entries")
        
        # Create initial sync.json
        sync_path = project_root / "context_engine" / "sync.json"
        with open(sync_path, 'w', encoding='utf-8') as f:
            f.write('{"files": {}, "last_sync": null}')
        logger.info("Created initial sync.json")
        
        # Create README for team_context
        readme_path = project_root / "team_context" / "README.md"
        readme_content = """# Team Context

This directory contains shared context information for the team:

- `latest_changes.md` - Recent changes and updates
- `adr_list.md` - Architecture Decision Records
- `dependency_map.json` - Project dependency mapping
- `conflict_warnings.md` - Merge conflict warnings
- `active_scopes.md` - Currently active development scopes

These files are automatically generated by Context Engine and should be committed to version control.
"""
        
        with open(readme_path, 'w', encoding='utf-8') as f:
            f.write(readme_content)
        logger.info("Created team_context/README.md")
        
        logger.info("Context Engine initialization complete!")
        logger.info("")
        logger.info("Next steps:")
        logger.info("1. Review and customize context_engine/config/context.yml")
        logger.info("2. Run 'context-engine reindex --all' to build initial index")
        logger.info("3. Run 'context-engine status' to check system status")
        
        return 0
        
    except Exception as e:
        logger.error(f"Failed to initialize Context Engine: {e}")
        return 1