"""Data tools for AI agents.

This module provides data processing and manipulation tools organized into logical submodules:

- structures: Data structure manipulation and transformation
- json_tools: JSON serialization, compression, and validation
- csv_tools: CSV file processing, parsing, and cleaning
- validation: Data validation and schema checking
- transform: Data transformation, cleaning, and normalization
- object_serialization: Object serialization and deserialization
- config_processing: Configuration file processing (YAML, TOML, INI)
- binary_processing: Binary data handling and encoding
- archive_processing: Archive file creation and extraction (ZIP, TAR)
"""

from typing import List

# Import all functions from submodules
from .archive_processing import (
    add_to_archive,
    create_tar_archive,
    create_zip_archive,
    extract_tar_archive,
    extract_zip_archive,
    list_archive_contents,
    validate_archive_integrity,
)
from .binary_processing import (
    decode_binary_data,
    encode_binary_data,
    extract_binary_metadata,
    read_binary_file,
    validate_binary_format,
    write_binary_file,
)
from .config_processing import (
    merge_config_files,
    read_ini_file,
    read_toml_file,
    read_yaml_file,
    validate_config_schema,
    write_ini_file,
    write_toml_file,
    write_yaml_file,
)
from .csv_tools import (
    clean_csv_data,
    csv_to_dict_list,
    detect_csv_delimiter,
    dict_list_to_csv,
    read_csv_file,
    validate_csv_structure,
    write_csv_file,
)
from .json_tools import (
    compress_json_data,
    decompress_json_data,
    safe_json_deserialize,
    safe_json_serialize,
    validate_json_string,
)
from .object_serialization import (
    deserialize_object,
    sanitize_for_serialization,
    serialize_object,
    validate_pickle_safety,
)
from .structures import (
    compare_data_structures,
    extract_keys,
    flatten_dict,
    get_nested_value,
    merge_dicts,
    remove_empty_values,
    rename_keys,
    safe_get,
    set_nested_value,
    unflatten_dict,
)
from .transform import (
    apply_data_transformations,
    clean_data,
    convert_data_types,
    deduplicate_records,
    normalize_data,
    pivot_data,
    rename_fields,
    transform_data,
)
from .validation import (
    aggregate_validation_errors,
    check_required_fields,
    create_validation_report,
    validate_data_types,
    validate_range,
    validate_schema,
)

# Re-export all functions at module level for convenience
__all__: List[str] = [
    # Data structures
    "flatten_dict",
    "unflatten_dict",
    "get_nested_value",
    "set_nested_value",
    "merge_dicts",
    "compare_data_structures",
    "safe_get",
    "remove_empty_values",
    "extract_keys",
    "rename_keys",
    # JSON processing
    "safe_json_serialize",
    "safe_json_deserialize",
    "validate_json_string",
    "compress_json_data",
    "decompress_json_data",
    # CSV processing
    "read_csv_file",
    "write_csv_file",
    "csv_to_dict_list",
    "dict_list_to_csv",
    "detect_csv_delimiter",
    "validate_csv_structure",
    "clean_csv_data",
    # Data transformation
    "transform_data",
    "rename_fields",
    "convert_data_types",
    "apply_data_transformations",
    "clean_data",
    "deduplicate_records",
    "normalize_data",
    "pivot_data",
    # Validation
    "validate_schema",
    "check_required_fields",
    "validate_data_types",
    "validate_range",
    "aggregate_validation_errors",
    "create_validation_report",
    # Object serialization
    "serialize_object",
    "deserialize_object",
    "sanitize_for_serialization",
    "validate_pickle_safety",
    # Configuration processing
    "read_yaml_file",
    "write_yaml_file",
    "read_toml_file",
    "write_toml_file",
    "read_ini_file",
    "write_ini_file",
    "validate_config_schema",
    "merge_config_files",
    # Binary processing
    "read_binary_file",
    "write_binary_file",
    "encode_binary_data",
    "decode_binary_data",
    "validate_binary_format",
    "extract_binary_metadata",
    # Archive processing
    "create_zip_archive",
    "extract_zip_archive",
    "list_archive_contents",
    "add_to_archive",
    "create_tar_archive",
    "extract_tar_archive",
    "validate_archive_integrity",
]
