"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScheduleTargetBase = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const helpers_internal_1 = require("aws-cdk-lib/core/lib/helpers-internal");
const util_1 = require("./util");
/**
 * Base class for Schedule Targets
 */
class ScheduleTargetBase {
    constructor(baseProps, targetArn) {
        this.baseProps = baseProps;
        this.targetArn = targetArn;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_scheduler_targets_alpha_ScheduleTargetBaseProps(baseProps);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ScheduleTargetBase);
            }
            throw error;
        }
    }
    bindBaseTargetConfig(_schedule) {
        const role = this.baseProps.role ?? this.singletonScheduleRole(_schedule, this.targetArn);
        this.addTargetActionToRole(_schedule, role);
        if (this.baseProps.deadLetterQueue) {
            this.addToDeadLetterQueueResourcePolicy(_schedule, this.baseProps.deadLetterQueue);
        }
        return {
            arn: this.targetArn,
            role: role,
            deadLetterConfig: this.baseProps.deadLetterQueue ? {
                arn: this.baseProps.deadLetterQueue.queueArn,
            } : undefined,
            retryPolicy: this.renderRetryPolicy(this.baseProps.maxEventAge, this.baseProps.retryAttempts),
            input: this.baseProps.input,
        };
    }
    /**
     * Create a return a Schedule Target Configuration for the given schedule
     * @param schedule
     * @returns a Schedule Target Configuration
     */
    bind(schedule) {
        return this.bindBaseTargetConfig(schedule);
    }
    /**
     * Obtain the Role for the EventBridge Scheduler event
     *
     * If a role already exists, it will be returned. This ensures that if multiple
     * events have the same target, they will share a role.
     */
    singletonScheduleRole(schedule, targetArn) {
        const stack = aws_cdk_lib_1.Stack.of(schedule);
        const arn = aws_cdk_lib_1.Token.isUnresolved(targetArn) ? stack.resolve(targetArn).toString() : targetArn;
        const hash = (0, helpers_internal_1.md5hash)(arn).slice(0, 6);
        const id = 'SchedulerRoleForTarget-' + hash;
        const existingRole = stack.node.tryFindChild(id);
        const principal = new iam.ServicePrincipal('scheduler.amazonaws.com', {
            conditions: {
                StringEquals: {
                    'aws:SourceAccount': schedule.env.account,
                },
            },
        });
        if (existingRole) {
            existingRole.assumeRolePolicy?.addStatements(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                principals: [principal],
                actions: ['sts:AssumeRole'],
            }));
            return existingRole;
        }
        const role = new iam.Role(stack, id, {
            roleName: aws_cdk_lib_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: principal,
        });
        return role;
    }
    /**
     * Allow a schedule to send events with failed invocation to an Amazon SQS queue.
     * @param schedule schedule to add DLQ to
     * @param queue the DLQ
     */
    addToDeadLetterQueueResourcePolicy(schedule, queue) {
        if (!(0, util_1.sameEnvDimension)(schedule.env.region, queue.env.region)) {
            throw new Error(`Cannot assign Dead Letter Queue in region ${queue.env.region} to the schedule ${aws_cdk_lib_1.Names.nodeUniqueId(schedule.node)} in region ${schedule.env.region}. Both the queue and the schedule must be in the same region.`);
        }
        // Skip Resource Policy creation if the Queue is not in the same account.
        // There is no way to add a target onto an imported schedule, so we can assume we will run the following code only
        // in the account where the schedule is created.
        if ((0, util_1.sameEnvDimension)(schedule.env.account, queue.env.account)) {
            const policyStatementId = `AllowSchedule${aws_cdk_lib_1.Names.nodeUniqueId(schedule.node)}`;
            queue.addToResourcePolicy(new iam.PolicyStatement({
                sid: policyStatementId,
                principals: [new iam.ServicePrincipal('scheduler.amazonaws.com')],
                effect: iam.Effect.ALLOW,
                actions: ['sqs:SendMessage'],
                resources: [queue.queueArn],
            }));
        }
        else {
            aws_cdk_lib_1.Annotations.of(schedule).addWarning(`Cannot add a resource policy to your dead letter queue associated with schedule ${schedule.scheduleName} because the queue is in a different account. You must add the resource policy manually to the dead letter queue in account ${queue.env.account}.`);
        }
    }
    renderRetryPolicy(maximumEventAge, maximumRetryAttempts) {
        const maxMaxAge = aws_cdk_lib_1.Duration.days(1).toSeconds();
        const minMaxAge = aws_cdk_lib_1.Duration.minutes(15).toSeconds();
        let maxAge = maxMaxAge;
        if (maximumEventAge) {
            maxAge = maximumEventAge.toSeconds({ integral: true });
            if (maxAge > maxMaxAge) {
                throw new Error('Maximum event age is 1 day');
            }
            if (maxAge < minMaxAge) {
                throw new Error('Minimum event age is 15 minutes');
            }
        }
        ;
        let maxAttempts = 185;
        if (typeof maximumRetryAttempts != 'undefined') {
            if (maximumRetryAttempts < 0) {
                throw Error('Number of retry attempts should be greater or equal than 0');
            }
            if (maximumRetryAttempts > 185) {
                throw Error('Number of retry attempts should be less or equal than 185');
            }
            maxAttempts = maximumRetryAttempts;
        }
        return {
            maximumEventAgeInSeconds: maxAge,
            maximumRetryAttempts: maxAttempts,
        };
    }
}
exports.ScheduleTargetBase = ScheduleTargetBase;
_a = JSII_RTTI_SYMBOL_1;
ScheduleTargetBase[_a] = { fqn: "@aws-cdk/aws-scheduler-targets-alpha.ScheduleTargetBase", version: "2.150.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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