from pyelasticache_client.client.consistenthasher import ConsistentHash
from pyelasticache_client.client.hash import HashClient

from .test_client_hash import TestHashClient

import unittest
import pytest

NUM_KEYS = 10
NUM_RESULTS = 5

class TestConsistentHashClient(TestHashClient, unittest.TestCase):
        
    def make_client(self, *mock_socket_values, **kwargs):
        current_port = 11012
        client = HashClient(hasher=ConsistentHash, **kwargs)
        ip = '127.0.0.1'

        for vals in mock_socket_values:
            s = '%s:%s' % (ip, current_port)
            c = self.make_client_pool(
                (ip, current_port),
                vals
            )
            client.clients[s] = c
            client.hasher.add_node(s)
            current_port += 1

        return client
        
    def hash_keys(self, hasher):

        res = []
        for key in range(ord('a'), ord('a') + NUM_KEYS):
            res.append(hasher.get_node(chr(key)))
        return res


    def test_empty_nodes(self):
        # no nodes
        ###########
        
        hasher = ConsistentHash()
        res = self.hash_keys(hasher)

        # all keys return None
        for i in range(0, NUM_RESULTS):
            assert res[i] is None


    def test_add_10_nodes(self):
        # add 10 nodes
        ###########
        
        hasher = ConsistentHash()
        for i in range(1,11):
            node = '172.31.0.%i' % i + ':3000'
            hasher.add_node(node)

        res = self.hash_keys(hasher)
        
        # no keys return None
        for i in range(0, NUM_RESULTS):
            assert res[i] is not None
        
        
    def test_remove_first_5_nodes(self):
        # remove first 5 nodes
        ###########
        
        hasher = ConsistentHash()
        
        for i in range(1,11):
            node = '172.31.0.%i' % i + ':3000'
            hasher.add_node(node)
            
        res1 = self.hash_keys(hasher)
        
        for i in range(1,6):
            node = '172.31.0.%i' % i + ':3000'
            hasher.remove_node(node)

        res2 = self.hash_keys(hasher)

        # keys on deleted nodes change assigned node
        # keys on kept nodes keep same node
        for i in range(0, NUM_RESULTS):
            init_node = int(res1[i][9:10])
            curr_node = int(res2[i][9:10])
            if init_node in range(1,6):
                assert init_node != curr_node
            elif init_node in range(6,11):
                assert init_node == curr_node
        
        
    def test_add_first_5_nodes(self):
        # add removed nodes again
        ###########
        
        hasher = ConsistentHash()
        
        for i in range(1,11):
            node = '172.31.0.%i' % i + ':3000'
            hasher.add_node(node)
            
        res1 = self.hash_keys(hasher)
        
        for i in range(1,6):
            node = '172.31.0.%i' % i + ':3000'
            hasher.remove_node(node)
        for i in range(1,6):
            node = '172.31.0.%i' % i + ':3000'
            hasher.add_node(node)

        res2 = self.hash_keys(hasher)

        # all keys go back to initial node assignments
        for i in range(0, NUM_RESULTS):
            init_node = int(res1[i][9:10])
            curr_node = int(res2[i][9:10])
            assert init_node == curr_node
              
                
    def test_remove_last_5_nodes(self):
        # remove last 5 nodes
        ###########
        
        hasher = ConsistentHash()
        
        for i in range(1,11):
            node = '172.31.0.%i' % i + ':3000'
            hasher.add_node(node)
            
        res1 = self.hash_keys(hasher)
        
        for i in range(6,11):
            node = '172.31.0.%i' % i + ':3000'
            hasher.remove_node(node)

        res2 = self.hash_keys(hasher)

        # keys on deleted nodes change assigned node
        # keys on kept nodes keep same node
        for i in range(0, NUM_RESULTS):
            init_node = int(res1[i][9:10])
            curr_node = int(res2[i][9:10])
            if init_node in range(1,6):
                assert init_node == curr_node
            elif init_node in range(6,11):
                assert init_node != curr_node
                
    def test_ketama_hashing(self):
        # test adapted from AWS implementation of Ketama hash:
        # https://github.com/masneyb/aws-elasticache-cluster-client-memcached-for-java/blob/master/src/test/java/net/spy/memcached/KetamaNodeLocatorTest.java
        ##############
        
        servers = [ "10.0.1.1:11211", "10.0.1.2:11211",
                    "10.0.1.3:11211", "10.0.1.4:11211", "10.0.1.5:11211",
                    "10.0.1.6:11211", "10.0.1.7:11211", "10.0.1.8:11211" ]
        
        expected = [    [ "0", 0 ], [ "233", 6 ],
                        [ "466", 2 ], [ "699", 0 ],
                        [ "932", 5 ], [ "1165", 1 ],
                        [ "1398", 0 ], [ "1631", 5 ],
                        [ "1864", 4 ], [ "2097", 2 ],
                        [ "2330", 6 ], [ "2563", 2 ],
                        [ "2796", 5 ], [ "3029", 0 ],
                        [ "3262", 1 ], [ "3495", 2 ],
                        [ "3728", 7 ], [ "3961", 3 ],
                        [ "4194", 3 ], [ "4427", 2 ],
                        [ "4660", 3 ], [ "4893", 6 ],
                        [ "5126", 3 ], [ "5359", 1 ],
                        [ "5592", 1 ], [ "5825", 2 ],
                        [ "6058", 1 ], [ "6291", 6 ],
                        [ "6524", 4 ], [ "6757", 4 ],
                        [ "6990", 0 ], [ "7223", 4 ],
                        [ "7456", 3 ], [ "7689", 1 ],
                        [ "7922", 4 ], [ "8155", 4 ],
                        [ "8388", 0 ], [ "8621", 1 ],
                        [ "8854", 1 ], [ "9087", 0 ],
                        [ "9320", 5 ], [ "9553", 2 ],
                        [ "9786", 1 ], [ "10019", 4 ],
                        [ "10252", 0 ], [ "10485", 4 ],
                        [ "10718", 4 ], [ "10951", 1 ],
                        [ "11184", 4 ], [ "11417", 2 ],
                        [ "11650", 7 ], [ "11883", 1 ],
                        [ "12116", 1 ], [ "12349", 6 ],
                        [ "12582", 4 ], [ "12815", 2 ],
                        [ "13048", 7 ], [ "13281", 5 ],
                        [ "13514", 2 ], [ "13747", 5 ],
                        [ "13980", 5 ], [ "14213", 7 ],
                        [ "14446", 1 ], [ "14679", 2 ],
                        [ "14912", 6 ], [ "15145", 0 ],
                        [ "15378", 3 ], [ "15611", 0 ],
                        [ "15844", 2 ], [ "16077", 2 ],
                        [ "16310", 4 ], [ "16543", 4 ],
                        [ "16776", 3 ], [ "17009", 0 ],
                        [ "17242", 3 ], [ "17475", 6 ],
                        [ "17708", 5 ], [ "17941", 1 ],
                        [ "28892", 6 ], [ "29125", 3 ],
                        [ "29358", 6 ], [ "29591", 4 ],
                        [ "29824", 6 ], [ "30057", 6 ],
                        [ "30290", 2 ], [ "30523", 7 ],
                        [ "30756", 2 ], [ "30989", 3 ],
                        [ "31222", 5 ], [ "31455", 0 ],
                        [ "31688", 1 ], [ "31921", 1 ],
                        [ "32154", 7 ], [ "32387", 0 ],
                        [ "32620", 2 ], [ "32853", 1 ],
                        [ "33086", 6 ], [ "33319", 3 ],
                        [ "33552", 4 ], [ "33785", 2 ],
                        [ "34018", 2 ], [ "34251", 5 ],
                        [ "34484", 6 ], [ "34717", 0 ],
                        [ "34950", 2 ], [ "35183", 5 ],
                        [ "35416", 7 ], [ "35649", 4 ],
                        [ "35882", 6 ], [ "36115", 1 ],
                        [ "36348", 4 ], [ "36581", 6 ],
                        [ "36814", 7 ], [ "37047", 7 ],
                        [ "37280", 7 ], [ "42173", 7 ],
                        [ "42406", 2 ], [ "47998", 1 ],
                        [ "48231", 4 ], [ "48464", 4 ],
                        [ "48697", 2 ], [ "48930", 0 ],
                        [ "49163", 1 ], [ "49396", 7 ],
                        [ "49629", 0 ], [ "49862", 7 ],
                        [ "50095", 4 ], [ "50328", 1 ],
                        [ "50561", 4 ], [ "50794", 6 ],
                        [ "51027", 2 ], [ "51260", 4 ],
                        [ "51493", 2 ], [ "51726", 7 ],
                        [ "51959", 1 ], [ "52192", 7 ],
                        [ "56153", 1 ], [ "56386", 5 ],
                        [ "56619", 7 ], [ "56852", 5 ],
                        [ "57085", 1 ], [ "57318", 6 ],
                        [ "57551", 7 ], [ "57784", 3 ],
                        [ "58017", 5 ], [ "58250", 7 ],
                        [ "58483", 7 ], [ "58716", 5 ],
                        [ "58949", 6 ], [ "59182", 2 ],
                        [ "59415", 1 ], [ "59648", 6 ],
                        [ "59881", 7 ], [ "60114", 7 ],
                        [ "60347", 2 ], [ "60580", 5 ],
                        [ "60813", 7 ], [ "61046", 5 ],
                        [ "61279", 6 ], [ "61512", 4 ],
                        [ "61745", 6 ], [ "61978", 7 ],
                        [ "62211", 6 ], [ "62444", 0 ],
                        [ "62677", 6 ], [ "62910", 2 ],
                        [ "63143", 1 ], [ "63376", 1 ],
                        [ "63609", 5 ], [ "63842", 1 ],
                        [ "64075", 4 ], [ "64308", 5 ],
                        [ "64541", 4 ], [ "64774", 3 ],
                        [ "65007", 6 ], [ "65240", 6 ],
                        [ "65473", 5 ], [ "65706", 7 ],
                        [ "65939", 3 ], [ "66172", 0 ],
                        [ "66405", 1 ], [ "66638", 5 ],
                        [ "66871", 4 ], [ "67104", 1 ],
                        [ "67337", 7 ], [ "67570", 7 ],
                        [ "67803", 4 ], [ "68036", 7 ],
                        [ "68269", 3 ], [ "68502", 6 ],
                        [ "68735", 0 ], [ "68968", 5 ],
                        [ "69201", 5 ], [ "69434", 5 ],
                        [ "69667", 2 ], [ "69900", 1 ],
                        [ "70133", 7 ], [ "70366", 1 ],
                        [ "70599", 1 ], [ "70832", 0 ],
                        [ "71065", 4 ], [ "71298", 1 ],
                        [ "71531", 1 ], [ "71764", 4 ],
                        [ "71997", 4 ], [ "72230", 1 ],
                        [ "72463", 6 ], [ "72696", 5 ],
                        [ "72929", 3 ], [ "73162", 3 ],
                        [ "73395", 6 ], [ "73628", 6 ],
                        [ "73861", 0 ], [ "74094", 5 ],
                        [ "74327", 0 ], [ "74560", 5 ],
                        [ "74793", 0 ], [ "75026", 4 ],
                        [ "75259", 4 ], [ "75492", 7 ],
                        [ "75725", 6 ], [ "75958", 3 ],
                        [ "76191", 4 ], [ "76424", 4 ],
                        [ "76657", 1 ], [ "76890", 6 ],
                        [ "77123", 3 ], [ "77356", 1 ],
                        [ "77589", 5 ], [ "77822", 0 ],
                        [ "78055", 5 ], [ "78288", 6 ],
                        [ "78521", 6 ], [ "78754", 4 ],
                        [ "78987", 5 ], [ "79220", 3 ],
                        [ "79453", 5 ], [ "79686", 3 ],
                        [ "79919", 2 ], [ "80152", 1 ],
                        [ "80385", 5 ], [ "80618", 4 ],
                        [ "80851", 6 ], [ "81084", 7 ],
                        [ "81317", 4 ], [ "81550", 7 ],
                        [ "81783", 3 ], [ "82016", 7 ],
                        [ "82249", 4 ], [ "82482", 4 ],
                        [ "82715", 4 ], [ "82948", 4 ],
                        [ "83181", 0 ], [ "83414", 0 ],
                        [ "83647", 1 ], [ "83880", 1 ],
                        [ "84113", 5 ], [ "84346", 5 ],
                        [ "84579", 4 ], [ "84812", 7 ],
                        [ "85045", 5 ], [ "85278", 6 ],
                        [ "85511", 1 ], [ "85744", 0 ],
                        [ "85977", 5 ], [ "86210", 6 ],
                        [ "86443", 3 ], [ "86676", 2 ],
                        [ "86909", 0 ], [ "87142", 7 ],
                        [ "87375", 0 ], [ "87608", 6 ],
                        [ "87841", 0 ], [ "88074", 0 ],
                        [ "88307", 6 ], [ "88540", 3 ],
                        [ "88773", 4 ], [ "89006", 1 ],
                        [ "89239", 0 ], [ "89472", 5 ],
                        [ "89705", 1 ], [ "89938", 1 ],
                        [ "90171", 6 ], [ "90404", 4 ],
                        [ "90637", 7 ], [ "90870", 7 ],
                        [ "91103", 6 ], [ "91336", 4 ],
                        [ "91569", 7 ], [ "91802", 1 ],
                        [ "92035", 7 ], [ "92268", 3 ],
                        [ "92501", 5 ], [ "92734", 1 ],
                        [ "92967", 5 ], [ "93200", 0 ],
                        [ "93433", 1 ], [ "93666", 5 ],
                        [ "93899", 1 ], [ "94132", 1 ],
                        [ "103685", 1 ], [ "103918", 6 ],
                        [ "104151", 4 ], [ "104384", 0 ],
                        [ "104617", 2 ], [ "104850", 2 ],
                        [ "105083", 6 ], [ "105316", 1 ],
                        [ "105549", 2 ], [ "105782", 1 ],
                        [ "106015", 3 ], [ "106248", 2 ],
                        [ "106481", 3 ], [ "106714", 7 ],
                        [ "106947", 7 ], [ "107180", 5 ],
                        [ "107413", 6 ], [ "107646", 4 ],
                        [ "107879", 7 ], [ "108112", 0 ],
                        [ "108345", 7 ], [ "108578", 4 ],
                        [ "112539", 7 ], [ "112772", 0 ],
                        [ "113005", 4 ], [ "113238", 3 ],
                        [ "116034", 7 ], [ "116267", 0 ],
                        [ "116500", 5 ], [ "116733", 0 ],
                        [ "116966", 5 ], [ "117199", 3 ],
                        [ "117432", 0 ], [ "117665", 1 ],
                        [ "117898", 5 ], [ "118131", 2 ],
                        [ "118364", 1 ], [ "118597", 4 ],
                        [ "118830", 4 ], [ "119063", 2 ],
                        [ "119296", 5 ], [ "119529", 0 ],
                        [ "119762", 5 ], [ "119995", 6 ],
                        [ "120228", 1 ], [ "120461", 1 ],
                        [ "124888", 2 ], [ "125121", 5 ],
                        [ "125354", 4 ], [ "125587", 1 ],
                        [ "125820", 2 ], [ "126053", 4 ],
                        [ "126286", 4 ], [ "126519", 1 ],
                        [ "126752", 5 ], [ "126985", 6 ],
                        [ "127218", 5 ], [ "127451", 6 ],
                        [ "127684", 5 ], [ "127917", 6 ],
                        [ "128150", 5 ], [ "128383", 0 ],
                        [ "128616", 3 ], [ "128849", 2 ],
                        [ "129082", 4 ], [ "129315", 7 ],
                        [ "129548", 5 ], [ "129781", 5 ],
                        [ "130014", 5 ], [ "130247", 4 ],
                        [ "130480", 5 ], [ "130713", 1 ],
                        [ "130946", 4 ], [ "131179", 4 ],
                        [ "131412", 6 ], [ "131645", 1 ],
                        [ "131878", 5 ], [ "132111", 4 ],
                        [ "132344", 7 ], [ "132577", 0 ],
                        [ "132810", 0 ], [ "133043", 6 ],
                        [ "133276", 3 ], [ "133509", 7 ],
                        [ "133742", 2 ], [ "133975", 4 ],
                        [ "134208", 0 ], [ "134441", 7 ],
                        [ "134674", 6 ], [ "134907", 3 ],
                        [ "135140", 2 ], [ "135373", 4 ],
                        [ "135606", 6 ], [ "135839", 7 ],
                        [ "136072", 7 ], [ "136305", 6 ],
                        [ "136538", 0 ], [ "136771", 0 ],
                        [ "137004", 1 ], [ "137237", 1 ],
                        [ "137470", 1 ], [ "137703", 2 ],
                        [ "137936", 0 ], [ "138169", 5 ],
                        [ "138402", 7 ], [ "138635", 6 ],
                        [ "138868", 1 ], [ "139101", 3 ],
                        [ "139334", 6 ], [ "139567", 5 ],
                        [ "139800", 0 ], [ "140033", 7 ],
                        [ "140266", 6 ], [ "140499", 2 ],
                        [ "140732", 1 ], [ "140965", 6 ],
                        [ "141198", 5 ], [ "141431", 6 ],
                        [ "141664", 4 ], [ "141897", 4 ],
                        [ "142130", 4 ], [ "142363", 3 ],
                        [ "142596", 7 ], [ "142829", 1 ],
                        [ "143062", 1 ], [ "143295", 3 ],
                        [ "143528", 7 ], [ "143761", 1 ],
                        [ "143994", 4 ], [ "144227", 2 ],
                        [ "144460", 1 ], [ "152149", 1 ],
                        [ "152382", 1 ], [ "152615", 2 ],
                        [ "152848", 4 ], [ "153081", 0 ],
                        [ "153314", 0 ], [ "153547", 0 ],
                        [ "153780", 0 ], [ "154013", 1 ],
                        [ "154246", 0 ], [ "154479", 7 ],
                        [ "154712", 0 ], [ "154945", 0 ],
                        [ "155178", 7 ], [ "155411", 7 ],
                        [ "155644", 7 ], [ "155877", 1 ],
                        [ "156110", 2 ], [ "156343", 0 ],
                        [ "156576", 6 ], [ "156809", 6 ],
                        [ "157042", 0 ], [ "157275", 6 ],
                        [ "157508", 1 ], [ "157741", 6 ],
                        [ "157974", 4 ], [ "158207", 4 ],
                        [ "158440", 3 ], [ "158673", 2 ],
                        [ "158906", 2 ], [ "159139", 7 ],
                        [ "159372", 5 ], [ "159605", 2 ],
                        [ "159838", 3 ], [ "160071", 1 ],
                        [ "160304", 3 ], [ "160537", 5 ],
                        [ "160770", 4 ], [ "161003", 2 ],
                        [ "161236", 6 ], [ "161469", 4 ],
                        [ "161702", 6 ], [ "161935", 7 ],
                        [ "162168", 7 ], [ "162401", 7 ],
                        [ "162634", 7 ], [ "162867", 5 ],
                        [ "163100", 7 ], [ "163333", 6 ],
                        [ "163566", 6 ], [ "163799", 2 ],
                        [ "164032", 5 ], [ "164265", 7 ],
                        [ "169158", 6 ], [ "169391", 5 ],
                        [ "169624", 5 ], [ "169857", 5 ],
                        [ "170090", 7 ], [ "170323", 3 ],
                        [ "170556", 1 ], [ "170789", 7 ],
                        [ "171022", 7 ], [ "171255", 3 ],
                        [ "171488", 6 ], [ "171721", 3 ],
                        [ "171954", 4 ], [ "172187", 3 ],
                        [ "172420", 2 ], [ "172653", 1 ],
                        [ "172886", 0 ], [ "173119", 7 ],
                        [ "173352", 4 ], [ "176614", 2 ],
                        [ "176847", 7 ], [ "177080", 0 ],
                        [ "177313", 3 ], [ "177546", 1 ],
                        [ "177779", 7 ], [ "178012", 5 ],
                        [ "178245", 2 ], [ "178478", 6 ],
                        [ "178711", 2 ], [ "178944", 7 ],
                        [ "179177", 0 ], [ "179410", 5 ],
                        [ "179643", 0 ], [ "179876", 7 ],
                        [ "180109", 2 ], [ "180342", 1 ],
                        [ "180575", 6 ], [ "180808", 7 ],
                        [ "181041", 4 ], [ "181274", 5 ],
                        [ "181507", 7 ], [ "181740", 5 ],
                        [ "181973", 4 ], [ "182206", 3 ],
                        [ "182439", 1 ], [ "182672", 7 ],
                        [ "182905", 7 ], [ "183138", 3 ],
                        [ "183371", 7 ], [ "183604", 6 ],
                        [ "183837", 6 ], [ "184070", 3 ],
                        [ "184303", 6 ], [ "184536", 0 ],
                        [ "184769", 2 ], [ "185002", 0 ],
                        [ "185235", 5 ], [ "185468", 6 ],
                        [ "185701", 3 ], [ "185934", 0 ],
                        [ "186167", 7 ], [ "186400", 7 ],
                        [ "186633", 0 ], [ "186866", 4 ],
                        [ "187099", 7 ], [ "187332", 0 ],
                        [ "187565", 4 ], [ "187798", 0 ],
                        [ "188031", 7 ], [ "188264", 4 ],
                        [ "188497", 1 ], [ "188730", 5 ],
                        [ "196419", 7 ], [ "196652", 6 ],
                        [ "196885", 0 ], [ "197118", 5 ],
                        [ "197351", 0 ], [ "197584", 0 ],
                        [ "197817", 7 ], [ "198050", 5 ],
                        [ "198283", 2 ], [ "198516", 7 ],
                        [ "198749", 5 ], [ "198982", 1 ],
                        [ "199215", 3 ], [ "199448", 4 ],
                        [ "199681", 5 ], [ "199914", 5 ],
                        [ "200147", 2 ], [ "200380", 3 ],
                        [ "200613", 0 ], [ "200846", 5 ],
                        [ "201079", 6 ], [ "201312", 6 ],
                        [ "201545", 0 ], [ "201778", 0 ],
                        [ "202011", 6 ], [ "202244", 6 ],
                        [ "202477", 5 ], [ "202710", 0 ],
                        [ "202943", 0 ], [ "203176", 0 ],
                        [ "203409", 2 ], [ "203642", 4 ],
                        [ "203875", 0 ], [ "204108", 7 ],
                        [ "204341", 0 ], [ "204574", 3 ],
                        [ "204807", 2 ], [ "205040", 6 ],
                        [ "205273", 1 ], [ "205506", 5 ],
                        [ "205739", 1 ], [ "205972", 5 ],
                        [ "206205", 5 ], [ "206438", 5 ],
                        [ "206671", 6 ], [ "206904", 5 ],
                        [ "207137", 6 ], [ "207370", 4 ],
                        [ "207603", 0 ], [ "207836", 4 ],
                        [ "208069", 5 ], [ "208302", 7 ],
                        [ "208535", 4 ], [ "208768", 2 ],
                        [ "209001", 1 ], [ "209234", 2 ],
                        [ "209467", 3 ], [ "209700", 5 ],
                        [ "209933", 4 ], [ "210166", 0 ],
                        [ "210399", 7 ], [ "210632", 3 ],
                        [ "210865", 7 ], [ "211098", 7 ],
                        [ "211331", 2 ], [ "211564", 5 ],
                        [ "211797", 7 ], [ "212030", 7 ],
                        [ "212263", 6 ], [ "212496", 7 ],
                        [ "212729", 0 ], [ "212962", 6 ],
                        [ "213195", 5 ], [ "213428", 1 ],
                        [ "213661", 7 ], [ "213894", 4 ],
                        [ "214127", 2 ], [ "214360", 0 ],
                        [ "214593", 1 ], [ "214826", 6 ],
                        [ "215059", 6 ], [ "215292", 1 ],
                        [ "215525", 0 ], [ "215758", 7 ],
                        [ "215991", 2 ], [ "216224", 7 ],
                        [ "216457", 2 ], [ "216690", 6 ],
                        [ "216923", 1 ], [ "217156", 1 ],
                        [ "217389", 1 ], [ "217622", 3 ],
                        [ "217855", 2 ], [ "218088", 6 ],
                        [ "218321", 7 ], [ "218554", 6 ],
                        [ "218787", 1 ], [ "219020", 7 ],
                        [ "219253", 7 ], [ "219486", 4 ],
                        [ "219719", 1 ], [ "219952", 2 ],
                        [ "220185", 6 ], [ "220418", 6 ],
                        [ "220651", 6 ], [ "220884", 4 ],
                        [ "221117", 4 ], [ "221350", 7 ],
                        [ "221583", 6 ], [ "221816", 6 ],
                        [ "222049", 7 ], [ "222282", 1 ],
                        [ "222515", 7 ], [ "222748", 7 ],
                        [ "222981", 5 ], [ "223214", 1 ],
                        [ "223447", 3 ], [ "223680", 1 ],
                        [ "223913", 7 ], [ "224146", 7 ],
                        [ "224379", 2 ], [ "224612", 4 ],
                        [ "224845", 7 ], [ "225078", 7 ],
                        [ "225311", 2 ], [ "225544", 4 ],
                        [ "225777", 0 ], [ "226010", 5 ],
                        [ "226243", 6 ], [ "226476", 1 ],
                        [ "226709", 2 ], [ "226942", 4 ],
                        [ "227175", 2 ], [ "227408", 0 ],
                        [ "227641", 7 ], [ "227874", 3 ],
                        [ "228107", 5 ], [ "228340", 3 ],
                        [ "228573", 0 ], [ "228806", 2 ],
                        [ "238825", 1 ], [ "239058", 4 ],
                        [ "239291", 4 ], [ "239524", 5 ],
                        [ "239757", 4 ], [ "239990", 6 ],
                        [ "240223", 6 ], [ "240456", 4 ],
                        [ "240689", 4 ], [ "240922", 0 ],
                        [ "241155", 0 ], [ "241388", 6 ],
                        [ "241621", 0 ], [ "241854", 7 ],
                        [ "242087", 5 ], [ "242320", 7 ],
                        [ "242553", 1 ], [ "242786", 4 ],
                        [ "243019", 5 ], [ "243252", 3 ],
                        [ "243485", 7 ], [ "243718", 6 ],
                        [ "243951", 4 ], [ "249310", 4 ],
                        [ "249543", 7 ], [ "249776", 6 ],
                        [ "250009", 2 ], [ "250242", 7 ],
                        [ "250475", 3 ], [ "250708", 7 ],
                        [ "250941", 0 ], [ "251174", 4 ],
                        [ "251407", 7 ], [ "251640", 0 ],
                        [ "251873", 1 ], [ "252106", 4 ],
                        [ "252339", 1 ], [ "252572", 4 ],
                        [ "252805", 7 ], [ "253038", 4 ],
                        [ "253271", 6 ], [ "253504", 3 ],
                        [ "253737", 2 ], [ "253970", 6 ],
                        [ "254203", 5 ], [ "254436", 1 ],
                        [ "254669", 4 ], [ "254902", 7 ],
                        [ "255135", 4 ], [ "255368", 2 ],
                        [ "255601", 3 ], [ "255834", 0 ],
                        [ "256067", 7 ], [ "260960", 4 ],
                        [ "261193", 4 ], [ "261426", 1 ],
                        [ "261659", 1 ], [ "261892", 1 ],
                        [ "262125", 7 ], [ "262358", 5 ],
                        [ "262591", 7 ], [ "262824", 0 ],
                        [ "263057", 4 ], [ "263290", 0 ],
                        [ "263523", 2 ], [ "263756", 1 ],
                        [ "263989", 4 ], [ "264222", 4 ],
                        [ "264455", 4 ], [ "264688", 4 ],
                        [ "264921", 5 ], [ "265154", 6 ],
                        [ "265387", 6 ], [ "265620", 3 ],
                        [ "265853", 4 ], [ "266086", 5 ],
                        [ "266319", 0 ], [ "266552", 1 ],
                        [ "266785", 6 ], [ "267018", 0 ],
                        [ "267251", 5 ], [ "267484", 7 ],
                        [ "267717", 2 ], [ "267950", 1 ],
                        [ "268183", 4 ], [ "268416", 5 ],
                        [ "268649", 4 ], [ "268882", 5 ],
                        [ "269115", 4 ], [ "269348", 1 ],
                        [ "269581", 3 ], [ "269814", 5 ],
                        [ "270047", 1 ], [ "270280", 0 ],
                        [ "270513", 7 ], [ "270746", 5 ],
                        [ "270979", 3 ], [ "271212", 5 ],
                        [ "271445", 7 ], [ "271678", 6 ],
                        [ "271911", 2 ], [ "272144", 7 ],
                        [ "272377", 6 ], [ "272610", 0 ],
                        [ "272843", 7 ], [ "273076", 7 ],
                        [ "273309", 0 ], [ "273542", 7 ],
                        [ "273775", 4 ], [ "274008", 6 ],
                        [ "274241", 6 ], [ "274474", 5 ],
                        [ "274707", 5 ], [ "274940", 7 ],
                        [ "275173", 3 ], [ "275406", 5 ],
                        [ "275639", 1 ], [ "275872", 2 ],
                        [ "276105", 4 ], [ "276338", 1 ],
                        [ "276571", 3 ], [ "276804", 3 ],
                        [ "277037", 5 ], [ "277270", 5 ],
                        [ "277503", 0 ], [ "277736", 5 ],
                        [ "277969", 6 ], [ "278202", 2 ],
                        [ "278435", 5 ], [ "278668", 2 ],
                        [ "278901", 0 ], [ "279134", 6 ],
                        [ "279367", 3 ], [ "279600", 5 ],
                        [ "279833", 7 ], [ "280066", 6 ],
                        [ "280299", 7 ], [ "280532", 4 ],
                        [ "280765", 7 ], [ "280998", 0 ],
                        [ "281231", 4 ], [ "281464", 7 ],
                        [ "281697", 1 ], [ "281930", 6 ],
                        [ "282163", 4 ], [ "282396", 3 ],
                        [ "282629", 6 ], [ "282862", 6 ],
                        [ "283095", 4 ], [ "283328", 1 ],
                        [ "283561", 0 ], [ "283794", 7 ],
                        [ "284027", 1 ], [ "284260", 5 ],
                        [ "284493", 1 ], [ "284726", 1 ],
                        [ "284959", 5 ], [ "285192", 2 ],
                        [ "285425", 5 ], [ "285658", 1 ],
                        [ "285891", 7 ], [ "286124", 2 ],
                        [ "286357", 3 ], [ "286590", 4 ],
                        [ "286823", 6 ], [ "287056", 5 ],
                        [ "287289", 2 ], [ "287522", 2 ],
                        [ "287755", 2 ], [ "287988", 5 ],
                        [ "288221", 7 ], [ "288454", 3 ],
                        [ "288687", 2 ], [ "288920", 4 ],
                        [ "289153", 7 ], [ "289386", 6 ],
                        [ "289619", 7 ], [ "289852", 7 ],
                        [ "290085", 2 ], [ "290318", 6 ],
                        [ "290551", 7 ], [ "290784", 6 ],
                        [ "291017", 4 ], [ "291250", 7 ],
                        [ "291483", 2 ], [ "291716", 2 ],
                        [ "291949", 0 ], [ "292182", 7 ],
                        [ "292415", 4 ], [ "292648", 7 ],
                        [ "292881", 4 ], [ "293114", 6 ],
                        [ "293347", 4 ], [ "293580", 4 ],
                        [ "293813", 4 ], [ "294046", 2 ],
                        [ "294279", 5 ], [ "294512", 7 ],
                        [ "294745", 6 ], [ "294978", 6 ],
                        [ "295211", 1 ], [ "295444", 4 ],
                        [ "295677", 6 ], [ "295910", 2 ],
                        [ "296143", 6 ], [ "296376", 3 ],
                        [ "296609", 2 ], [ "296842", 5 ],
                        [ "297075", 1 ], [ "297308", 3 ],
                        [ "297541", 7 ], [ "297774", 2 ],
                        [ "298007", 2 ], [ "298240", 4 ],
                        [ "298473", 3 ], [ "298706", 4 ],
                        [ "298939", 6 ], [ "303599", 5 ],
                        [ "303832", 3 ], [ "304065", 2 ],
                        [ "304298", 0 ], [ "304531", 0 ],
                        [ "304764", 7 ], [ "304997", 6 ],
                        [ "305230", 5 ], [ "305463", 1 ],
                        [ "305696", 6 ], [ "305929", 3 ],
                        [ "306162", 3 ], [ "306395", 3 ],
                        [ "306628", 7 ], [ "306861", 6 ],
                        [ "307094", 3 ], [ "307327", 7 ],
                        [ "307560", 1 ], [ "307793", 7 ],
                        [ "308026", 4 ], [ "308259", 5 ],
                        [ "308492", 1 ], [ "308725", 5 ],
                        [ "308958", 2 ], [ "309191", 6 ],
                        [ "309424", 6 ], [ "309657", 7 ],
                        [ "309890", 7 ], [ "310123", 0 ],
                        [ "310356", 2 ], [ "310589", 4 ],
                        [ "310822", 7 ], [ "311055", 2 ],
                        [ "311288", 7 ], [ "311521", 7 ],
                        [ "311754", 2 ], [ "311987", 6 ],
                        [ "312220", 4 ], [ "312453", 1 ],
                        [ "312686", 1 ], [ "312919", 0 ],
                        [ "313152", 7 ], [ "313385", 1 ],
                        [ "313618", 7 ], [ "313851", 3 ],
                        [ "314084", 3 ], [ "314317", 5 ],
                        [ "314550", 4 ], [ "314783", 0 ],
                        [ "315016", 0 ], [ "315249", 5 ],
                        [ "315482", 3 ], [ "315715", 1 ],
                        [ "315948", 6 ], [ "316181", 1 ],
                        [ "316414", 7 ], [ "316647", 6 ],
                        [ "316880", 6 ], [ "317113", 7 ],
                        [ "317346", 6 ], [ "317579", 3 ],
                        [ "317812", 1 ], [ "318045", 0 ],
                        [ "318278", 5 ], [ "318511", 5 ],
                        [ "318744", 0 ], [ "318977", 4 ],
                        [ "319210", 1 ], [ "319443", 3 ],
                        [ "319676", 6 ], [ "319909", 2 ],
                        [ "320142", 7 ], [ "320375", 0 ],
                        [ "320608", 7 ], [ "320841", 0 ],
                        [ "321074", 4 ], [ "321307", 1 ],
                        [ "321540", 2 ], [ "321773", 5 ],
                        [ "322006", 2 ], [ "322239", 6 ],
                        [ "322472", 5 ], [ "322705", 7 ],
                        [ "322938", 3 ], [ "323171", 6 ],
                        [ "323404", 1 ], [ "323637", 3 ],
                        [ "323870", 1 ], [ "324103", 6 ],
                        [ "324336", 2 ], [ "324569", 1 ],
                        [ "324802", 2 ], [ "325035", 7 ],
                        [ "325268", 4 ], [ "325501", 5 ],
                        [ "325734", 2 ], [ "325967", 4 ],
                        [ "326200", 0 ], [ "326433", 7 ],
                        [ "326666", 3 ], [ "326899", 1 ],
                        [ "327132", 3 ], [ "327365", 0 ],
                        [ "327598", 2 ], [ "327831", 5 ],
                        [ "328064", 3 ], [ "328297", 5 ],
                        [ "328530", 7 ], [ "328763", 1 ],
                        [ "328996", 0 ], [ "329229", 2 ],
                        [ "329462", 6 ], [ "329695", 5 ],
                        [ "329928", 0 ], [ "330161", 6 ],
                        [ "330394", 0 ], [ "330627", 2 ],
                        [ "330860", 7 ], [ "331093", 1 ],
                        [ "331326", 5 ], [ "331559", 3 ],
                        [ "331792", 6 ], [ "332025", 7 ],
                        [ "332258", 1 ], [ "332491", 4 ],
                        [ "332724", 5 ], [ "332957", 3 ],
                        [ "333190", 7 ], [ "333423", 6 ],
                        [ "333656", 6 ], [ "333889", 0 ],
                        [ "334122", 4 ], [ "334355", 3 ],
                        [ "334588", 7 ], [ "334821", 2 ],
                        [ "335054", 2 ], [ "335287", 6 ],
                        [ "335520", 7 ], [ "335753", 1 ],
                        [ "335986", 2 ], [ "336219", 1 ],
                        [ "336452", 7 ], [ "336685", 1 ],
                        [ "336918", 6 ], [ "337151", 3 ],
                        [ "337384", 5 ], [ "337617", 0 ],
                        [ "337850", 1 ], [ "338083", 6 ],
                        [ "338316", 7 ], [ "338549", 4 ],
                        [ "338782", 7 ], [ "339015", 7 ],
                        [ "339248", 4 ], [ "339481", 7 ],
                        [ "339714", 2 ], [ "339947", 6 ],
                        [ "340180", 4 ], [ "340413", 4 ],
                        [ "340646", 7 ], [ "340879", 2 ],
                        [ "341112", 5 ], [ "341345", 7 ],
                        [ "341578", 6 ], [ "341811", 1 ],
                        [ "342044", 0 ], [ "342277", 3 ],
                        [ "342510", 4 ], [ "342743", 4 ],
                        [ "342976", 3 ], [ "343209", 6 ],
                        [ "343442", 7 ], [ "343675", 0 ],
                        [ "343908", 1 ], [ "348801", 1 ],
                        [ "349034", 6 ], [ "349267", 4 ],
                        [ "349500", 6 ], [ "349733", 7 ],
                        [ "349966", 2 ], [ "350199", 7 ],
                        [ "350432", 3 ], [ "350665", 4 ],
                        [ "350898", 1 ], [ "351131", 4 ],
                        [ "351364", 7 ], [ "351597", 2 ],
                        [ "351830", 4 ], [ "352063", 2 ],
                        [ "352296", 0 ], [ "352529", 3 ],
                        [ "352762", 4 ], [ "352995", 0 ],
                        [ "353228", 7 ], [ "357888", 4 ],
                        [ "358121", 7 ], [ "358354", 0 ],
                        [ "358587", 0 ], [ "358820", 6 ],
                        [ "359053", 3 ], [ "359286", 1 ],
                        [ "359519", 2 ], [ "359752", 0 ],
                        [ "359985", 0 ], [ "360218", 3 ],
                        [ "360451", 4 ], [ "360684", 1 ],
                        [ "360917", 3 ], [ "361150", 0 ],
                        [ "361383", 1 ], [ "361616", 2 ],
                        [ "361849", 2 ], [ "362082", 1 ],
                        [ "362315", 3 ], [ "362548", 4 ],
                        [ "362781", 5 ], [ "363014", 6 ],
                        [ "363247", 7 ], [ "363480", 6 ],
                        [ "363713", 3 ], [ "363946", 3 ],
                        [ "364179", 1 ], [ "364412", 4 ],
                        [ "364645", 7 ], [ "364878", 0 ],
                        [ "365111", 5 ], [ "365344", 4 ],
                        [ "365577", 5 ], [ "365810", 7 ],
                        [ "366043", 4 ], [ "366276", 3 ],
                        [ "366509", 5 ], [ "366742", 6 ],
                        [ "374431", 3 ], [ "374664", 7 ],
                        [ "374897", 5 ], [ "375130", 2 ],
                        [ "375363", 3 ], [ "375596", 7 ],
                        [ "375829", 0 ], [ "376062", 2 ],
                        [ "376295", 0 ], [ "376528", 0 ],
                        [ "376761", 3 ], [ "376994", 6 ],
                        [ "377227", 1 ], [ "377460", 7 ],
                        [ "377693", 0 ], [ "377926", 4 ],
                        [ "378159", 0 ], [ "378392", 7 ],
                        [ "378625", 6 ], [ "378858", 3 ],
                        [ "379091", 2 ], [ "379324", 7 ],
                        [ "379557", 1 ], [ "379790", 1 ],
                        [ "380023", 7 ], [ "380256", 5 ],
                        [ "380489", 1 ], [ "380722", 7 ],
                        [ "380955", 4 ], [ "381188", 7 ],
                        [ "381421", 7 ], [ "381654", 3 ],
                        [ "381887", 6 ], [ "382120", 6 ],
                        [ "382353", 2 ], [ "382586", 5 ],
                        [ "382819", 1 ], [ "383052", 7 ],
                        [ "383285", 6 ], [ "383518", 7 ],
                        [ "383751", 1 ], [ "383984", 5 ],
                        [ "384217", 5 ], [ "384450", 2 ],
                        [ "384683", 7 ], [ "384916", 2 ],
                        [ "385149", 0 ], [ "385382", 1 ],
                        [ "385615", 0 ], [ "385848", 4 ],
                        [ "386081", 4 ], [ "386314", 5 ],
                        [ "386547", 2 ], [ "386780", 3 ],
                        [ "387013", 6 ], [ "387246", 7 ],
                        [ "387479", 2 ], [ "387712", 7 ],
                        [ "387945", 7 ], [ "388178", 5 ],
                        [ "388411", 0 ], [ "388644", 7 ],
                        [ "388877", 6 ], [ "389110", 0 ],
                        [ "389343", 0 ], [ "389576", 7 ],
                        [ "389809", 1 ], [ "390042", 4 ],
                        [ "390275", 6 ], [ "390508", 1 ],
                        [ "390741", 1 ], [ "390974", 3 ],
                        [ "391207", 2 ], [ "391440", 3 ],
                        [ "391673", 3 ], [ "391906", 6 ],
                        [ "392139", 2 ], [ "392372", 5 ],
                        [ "392605", 5 ], [ "392838", 5 ],
                        [ "393071", 1 ], [ "393304", 1 ],
                        [ "393537", 1 ], [ "393770", 3 ],
                        [ "394003", 0 ], [ "394236", 5 ],
                        [ "394469", 3 ], [ "394702", 5 ],
                        [ "394935", 6 ], [ "395168", 3 ],
                        [ "395401", 2 ], [ "395634", 3 ],
                        [ "395867", 0 ], [ "396100", 5 ],
                        [ "396333", 3 ], [ "396566", 7 ],
                        [ "396799", 2 ], [ "397032", 3 ],
                        [ "397265", 2 ], [ "397498", 4 ],
                        [ "397731", 0 ], [ "397964", 1 ],
                        [ "398197", 7 ], [ "398430", 6 ],
                        [ "398663", 0 ], [ "398896", 4 ],
                        [ "399129", 4 ], [ "399362", 7 ],
                        [ "399595", 7 ], [ "399828", 6 ],
                        [ "400061", 5 ], [ "400294", 7 ],
                        [ "400527", 1 ], [ "400760", 6 ],
                        [ "400993", 6 ], [ "401226", 0 ],
                        [ "401459", 0 ], [ "401692", 1 ],
                        [ "401925", 0 ], [ "402158", 0 ],
                        [ "402391", 4 ], [ "402624", 0 ],
                        [ "402857", 7 ], [ "403090", 0 ],
                        [ "403323", 4 ], [ "403556", 5 ],
                        [ "403789", 5 ], [ "404022", 1 ],
                        [ "404255", 3 ], [ "404488", 0 ],
                        [ "404721", 7 ], [ "404954", 0 ],
                        [ "405187", 6 ], [ "405420", 6 ],
                        [ "405653", 6 ], [ "405886", 0 ],
                        [ "406119", 2 ], [ "406352", 4 ],
                        [ "411944", 0 ], [ "412177", 7 ],
                        [ "412410", 2 ], [ "412643", 6 ],
                        [ "412876", 3 ], [ "413109", 7 ],
                        [ "413342", 6 ], [ "413575", 7 ],
                        [ "413808", 6 ], [ "414041", 0 ],
                        [ "414274", 4 ], [ "414507", 0 ],
                        [ "414740", 7 ], [ "414973", 1 ],
                        [ "415206", 7 ], [ "415439", 5 ],
                        [ "415672", 0 ], [ "415905", 6 ],
                        [ "416138", 5 ], [ "416371", 7 ],
                        [ "416604", 3 ], [ "416837", 3 ],
                        [ "417070", 3 ], [ "417303", 5 ],
                        [ "417536", 5 ], [ "417769", 7 ],
                        [ "418002", 2 ], [ "418235", 7 ],
                        [ "418468", 3 ], [ "418701", 3 ],
                        [ "418934", 6 ], [ "419167", 5 ],
                        [ "419400", 2 ], [ "419633", 4 ],
                        [ "419866", 7 ], [ "420099", 5 ],
                        [ "420332", 2 ], [ "420565", 4 ],
                        [ "420798", 7 ], [ "421031", 7 ],
                        [ "421264", 5 ], [ "426856", 2 ],
                        [ "427089", 4 ], [ "427322", 0 ],
                        [ "427555", 5 ], [ "427788", 4 ],
                        [ "428021", 7 ], [ "428254", 2 ],
                        [ "428487", 6 ], [ "428720", 3 ],
                        [ "428953", 3 ], [ "429186", 7 ],
                        [ "429419", 5 ], [ "429652", 7 ],
                        [ "429885", 5 ], [ "430118", 4 ],
                        [ "430351", 3 ], [ "430584", 5 ],
                        [ "430817", 4 ], [ "431050", 2 ],
                        [ "431283", 7 ], [ "431516", 0 ],
                        [ "431749", 3 ], [ "431982", 4 ],
                        [ "432215", 5 ], [ "432448", 5 ],
                        [ "432681", 3 ], [ "432914", 1 ],
                        [ "433147", 6 ], [ "433380", 5 ],
                        [ "433613", 0 ], [ "433846", 2 ],
                        [ "434079", 0 ], [ "434312", 1 ],
                        [ "434545", 3 ], [ "434778", 2 ],
                        [ "435011", 4 ], [ "435244", 1 ],
                        [ "435477", 3 ], [ "435710", 4 ],
                        [ "435943", 5 ], [ "436176", 7 ],
                        [ "436409", 4 ], [ "436642", 0 ],
                        [ "436875", 7 ], [ "437108", 5 ],
                        [ "437341", 1 ], [ "437574", 0 ],
                        [ "437807", 7 ], [ "438040", 6 ],
                        [ "438273", 2 ], [ "438506", 3 ],
                        [ "438739", 7 ], [ "438972", 6 ],
                        [ "439205", 7 ], [ "439438", 0 ],
                        [ "439671", 2 ], [ "439904", 2 ],
                        [ "440137", 3 ], [ "440370", 2 ],
                        [ "457612", 7 ], [ "457845", 2 ],
                        [ "458078", 0 ], [ "458311", 2 ],
                        [ "458544", 6 ], [ "458777", 6 ],
                        [ "459010", 4 ], [ "459243", 5 ],
                        [ "459476", 5 ], [ "459709", 5 ],
                        [ "459942", 0 ], [ "460175", 0 ],
                        [ "460408", 6 ], [ "460641", 4 ],
                        [ "460874", 4 ], [ "461107", 2 ],
                        [ "461340", 3 ], [ "461573", 6 ],
                        [ "461806", 6 ], [ "462039", 3 ],
                        [ "462272", 0 ], [ "462505", 7 ],
                        [ "462738", 4 ], [ "462971", 2 ],
                        [ "463204", 7 ], [ "463437", 7 ],
                        [ "463670", 4 ], [ "463903", 0 ],
                        [ "464136", 4 ], [ "464369", 5 ],
                        [ "464602", 6 ], [ "464835", 1 ],
                        [ "465068", 1 ], [ "465301", 0 ],
                        [ "465534", 7 ], [ "465767", 5 ],
                        [ "466000", 0 ], [ "466233", 5 ],
                        [ "466466", 2 ], [ "466699", 5 ],
                        [ "466932", 3 ], [ "467165", 3 ],
                        [ "467398", 1 ], [ "467631", 6 ],
                        [ "467864", 2 ], [ "468097", 5 ],
                        [ "468330", 0 ], [ "468563", 0 ],
                        [ "468796", 6 ], [ "469029", 5 ],
                        [ "469262", 2 ], [ "469495", 5 ],
                        [ "469728", 5 ], [ "469961", 3 ],
                        [ "470194", 4 ], [ "470427", 5 ],
                        [ "470660", 0 ], [ "470893", 3 ],
                        [ "471126", 6 ], [ "471359", 6 ],
                        [ "471592", 0 ], [ "471825", 4 ],
                        [ "472058", 5 ], [ "472291", 6 ],
                        [ "472524", 7 ], [ "472757", 1 ],
                        [ "472990", 1 ], [ "473223", 6 ],
                        [ "473456", 6 ], [ "473689", 4 ],
                        [ "473922", 6 ], [ "474155", 5 ],
                        [ "474388", 7 ], [ "474621", 2 ],
                        [ "474854", 0 ], [ "475087", 0 ],
                        [ "475320", 3 ], [ "475553", 5 ],
                        [ "475786", 3 ], [ "476019", 4 ],
                        [ "476252", 0 ], [ "476485", 3 ],
                        [ "476718", 5 ], [ "476951", 7 ],
                        [ "477184", 7 ], [ "477417", 2 ],
                        [ "477650", 0 ], [ "477883", 0 ],
                        [ "478116", 1 ], [ "478349", 4 ],
                        [ "478582", 3 ], [ "478815", 3 ],
                        [ "479048", 0 ], [ "479281", 0 ],
                        [ "479514", 1 ], [ "479747", 4 ],
                        [ "479980", 6 ], [ "480213", 2 ],
                        [ "480446", 0 ], [ "480679", 3 ],
                        [ "480912", 5 ], [ "481145", 1 ],
                        [ "481378", 0 ], [ "481611", 6 ],
                        [ "481844", 2 ], [ "482077", 3 ],
                        [ "482310", 3 ], [ "482543", 2 ],
                        [ "482776", 7 ], [ "483009", 6 ],
                        [ "483242", 5 ], [ "483475", 2 ],
                        [ "483708", 4 ], [ "483941", 7 ],
                        [ "484174", 6 ], [ "484407", 1 ],
                        [ "484640", 7 ], [ "484873", 7 ],
                        [ "485106", 3 ], [ "485339", 1 ],
                        [ "485572", 5 ], [ "485805", 4 ],
                        [ "486038", 2 ], [ "486271", 4 ],
                        [ "486504", 0 ], [ "486737", 4 ],
                        [ "486970", 2 ], [ "487203", 1 ],
                        [ "487436", 7 ], [ "487669", 2 ],
                        [ "487902", 6 ], [ "488135", 1 ],
                        [ "488368", 6 ], [ "488601", 7 ],
                        [ "488834", 7 ], [ "489067", 4 ],
                        [ "489300", 3 ], [ "489533", 4 ],
                        [ "489766", 4 ], [ "489999", 2 ],
                        [ "490232", 3 ], [ "490465", 1 ],
                        [ "490698", 0 ], [ "490931", 1 ],
                        [ "491164", 4 ], [ "491397", 4 ],
                        [ "491630", 1 ], [ "491863", 0 ],
                        [ "492096", 0 ], [ "492329", 4 ],
                        [ "492562", 6 ], [ "492795", 2 ],
                        [ "493028", 0 ], [ "493261", 1 ],
                        [ "493494", 2 ], [ "493727", 5 ],
                        [ "493960", 4 ], [ "494193", 5 ],
                        [ "494426", 5 ], [ "494659", 3 ],
                        [ "494892", 3 ], [ "495125", 0 ],
                        [ "495358", 2 ], [ "495591", 5 ],
                        [ "495824", 4 ], [ "496057", 6 ],
                        [ "496290", 4 ], [ "496523", 4 ],
                        [ "496756", 0 ], [ "496989", 2 ],
                        [ "497222", 7 ], [ "497455", 5 ],
                        [ "497688", 6 ], [ "497921", 4 ],
                        [ "498154", 3 ], [ "498387", 2 ],
                        [ "498620", 2 ], [ "498853", 2 ],
                        [ "499086", 0 ], [ "499319", 7 ],
                        [ "499552", 2 ], [ "499785", 1 ],
                        [ "500018", 4 ], [ "500251", 7 ],
                        [ "500484", 6 ], [ "500717", 3 ],
                        [ "500950", 4 ], [ "501183", 3 ],
                        [ "501416", 3 ], [ "501649", 1 ],
                        [ "501882", 0 ], [ "502115", 2 ],
                        [ "502348", 2 ], [ "502581", 7 ],
                        [ "502814", 0 ], [ "503047", 7 ],
                        [ "503280", 6 ], [ "503513", 0 ],
                        [ "503746", 0 ], [ "503979", 7 ],
                        [ "504212", 2 ], [ "504445", 1 ],
                        [ "504678", 1 ], [ "504911", 2 ],
                        [ "505144", 4 ], [ "505377", 5 ],
                        [ "505610", 0 ], [ "505843", 5 ],
                        [ "506076", 0 ], [ "506309", 1 ],
                        [ "506542", 2 ], [ "506775", 7 ],
                        [ "507008", 1 ], [ "507241", 2 ],
                        [ "507474", 5 ], [ "507707", 6 ],
                        [ "507940", 3 ], [ "508173", 4 ],
                        [ "508406", 0 ], [ "508639", 2 ],
                        [ "508872", 3 ], [ "509105", 6 ],
                        [ "509338", 1 ], [ "509571", 2 ],
                        [ "509804", 2 ], [ "510037", 6 ],
                        [ "510270", 3 ], [ "510503", 0 ],
                        [ "510736", 2 ], [ "510969", 5 ],
                        [ "511202", 1 ], [ "511435", 1 ],
                        [ "511668", 3 ], [ "511901", 5 ],
                        [ "512134", 1 ], [ "512367", 0 ],
                        [ "512600", 0 ], [ "512833", 2 ],
                        [ "513066", 2 ], [ "513299", 2 ],
                        [ "513532", 1 ], [ "513765", 7 ],
                        [ "513998", 6 ], [ "514231", 5 ],
                        [ "514464", 2 ], [ "514697", 2 ],
                        [ "514930", 5 ], [ "515163", 6 ],
                        [ "515396", 7 ], [ "515629", 4 ],
                        [ "515862", 6 ], [ "516095", 1 ],
                        [ "516328", 7 ], [ "516561", 2 ],
                        [ "516794", 4 ], [ "517027", 5 ],
                        [ "517260", 2 ], [ "517493", 4 ],
                        [ "517726", 7 ], [ "517959", 6 ],
                        [ "518192", 7 ], [ "518425", 7 ],
                        [ "518658", 7 ], [ "518891", 6 ],
                        [ "519124", 2 ], [ "519357", 5 ],
                        [ "519590", 4 ], [ "519823", 2 ],
                        [ "520056", 1 ], [ "520289", 7 ],
                        [ "520522", 4 ], [ "533337", 2 ],
                        [ "533570", 2 ], [ "533803", 1 ],
                        [ "534036", 3 ], [ "534269", 0 ],
                        [ "534502", 1 ], [ "534735", 3 ],
                        [ "534968", 7 ], [ "535201", 0 ],
                        [ "535434", 0 ], [ "535667", 2 ],
                        [ "535900", 6 ], [ "536133", 7 ],
                        [ "541026", 0 ], [ "541259", 1 ],
                        [ "541492", 5 ], [ "541725", 7 ],
                        [ "541958", 0 ], [ "542191", 5 ],
                        [ "542424", 6 ], [ "542657", 1 ],
                        [ "542890", 3 ], [ "543123", 4 ],
                        [ "543356", 2 ], [ "543589", 6 ],
                        [ "547084", 1 ], [ "547317", 2 ],
                        [ "547550", 6 ], [ "547783", 6 ],
                        [ "548016", 1 ], [ "548249", 4 ],
                        [ "548482", 6 ], [ "548715", 5 ],
                        [ "548948", 3 ], [ "549181", 2 ],
                        [ "549414", 2 ], [ "549647", 3 ],
                        [ "549880", 6 ], [ "550113", 5 ],
                        [ "550346", 3 ], [ "550579", 5 ],
                        [ "550812", 3 ], [ "551045", 4 ],
                        [ "551278", 5 ], [ "551511", 4 ],
                        [ "551744", 5 ], [ "551977", 7 ],
                        [ "552210", 1 ], [ "552443", 1 ],
                        [ "552676", 0 ], [ "552909", 3 ],
                        [ "553142", 6 ], [ "553375", 1 ],
                        [ "553608", 4 ], [ "553841", 4 ],
                        [ "554074", 4 ], [ "554307", 6 ],
                        [ "554540", 5 ], [ "554773", 2 ],
                        [ "555006", 2 ], [ "555239", 4 ],
                        [ "555472", 7 ], [ "555705", 7 ],
                        [ "555938", 5 ], [ "556171", 3 ],
                        [ "556404", 3 ], [ "556637", 7 ],
                        [ "556870", 4 ], [ "557103", 2 ],
                        [ "557336", 2 ], [ "557569", 7 ],
                        [ "557802", 0 ], [ "558035", 0 ],
                        [ "558268", 3 ], [ "558501", 2 ],
                        [ "558734", 5 ], [ "558967", 6 ],
                        [ "559200", 2 ], [ "559433", 0 ],
                        [ "559666", 3 ], [ "559899", 1 ],
                        [ "560132", 1 ], [ "560365", 5 ],
                        [ "560598", 7 ], [ "560831", 2 ],
                        [ "561064", 6 ], [ "561297", 0 ],
                        [ "561530", 6 ], [ "561763", 6 ],
                        [ "561996", 0 ], [ "562229", 7 ],
                        [ "562462", 5 ], [ "562695", 6 ],
                        [ "562928", 1 ], [ "563161", 7 ],
                        [ "563394", 7 ], [ "563627", 5 ],
                        [ "563860", 1 ], [ "564093", 4 ],
                        [ "564326", 3 ], [ "564559", 7 ],
                        [ "564792", 0 ], [ "565025", 2 ],
                        [ "565258", 6 ], [ "565491", 6 ],
                        [ "565724", 4 ], [ "565957", 3 ],
                        [ "566190", 3 ], [ "566423", 7 ],
                        [ "566656", 3 ], [ "566889", 2 ],
                        [ "567122", 0 ], [ "567355", 7 ],
                        [ "567588", 5 ], [ "567821", 2 ],
                        [ "568054", 4 ], [ "568287", 7 ],
                        [ "568520", 7 ], [ "568753", 3 ],
                        [ "568986", 4 ], [ "569219", 7 ],
                        [ "569452", 5 ], [ "569685", 0 ],
                        [ "569918", 2 ], [ "570151", 3 ],
                        [ "570384", 3 ], [ "570617", 7 ],
                        [ "570850", 4 ], [ "571083", 7 ],
                        [ "571316", 2 ], [ "571549", 2 ],
                        [ "571782", 1 ], [ "572015", 5 ],
                        [ "572248", 2 ], [ "572481", 6 ],
                        [ "572714", 5 ], [ "572947", 7 ],
                        [ "573180", 6 ], [ "573413", 0 ],
                        [ "573646", 0 ], [ "573879", 0 ],
                        [ "574112", 0 ], [ "574345", 0 ],
                        [ "574578", 5 ], [ "574811", 4 ],
                        [ "575044", 1 ], [ "575277", 0 ],
                        [ "575510", 0 ], [ "575743", 4 ],
                        [ "575976", 3 ], [ "576209", 5 ],
                        [ "576442", 4 ], [ "576675", 1 ],
                        [ "576908", 2 ], [ "577141", 6 ],
                        [ "577374", 4 ], [ "577607", 4 ],
                        [ "577840", 4 ], [ "578073", 0 ],
                        [ "578306", 1 ], [ "578539", 6 ],
                        [ "578772", 4 ], [ "579005", 5 ],
                        [ "579238", 3 ], [ "579471", 1 ],
                        [ "579704", 6 ], [ "579937", 5 ],
                        [ "580170", 3 ], [ "585063", 0 ],
                        [ "585296", 5 ], [ "585529", 3 ],
                        [ "585762", 3 ], [ "585995", 1 ],
                        [ "586228", 6 ], [ "586461", 7 ],
                        [ "586694", 2 ], [ "586927", 6 ],
                        [ "587160", 3 ], [ "587393", 7 ],
                        [ "587626", 6 ], [ "587859", 6 ],
                        [ "588092", 0 ], [ "588325", 7 ],
                        [ "588558", 2 ], [ "588791", 0 ],
                        [ "589024", 4 ], [ "589257", 3 ],
                        [ "589490", 0 ], [ "589723", 7 ],
                        [ "589956", 7 ], [ "590189", 0 ],
                        [ "590422", 7 ], [ "590655", 3 ],
                        [ "590888", 7 ], [ "591121", 0 ],
                        [ "591354", 5 ], [ "591587", 6 ],
                        [ "591820", 1 ], [ "592053", 6 ],
                        [ "592286", 7 ], [ "592519", 7 ],
                        [ "592752", 4 ], [ "592985", 4 ],
                        [ "593218", 4 ], [ "593451", 7 ],
                        [ "593684", 1 ], [ "593917", 2 ],
                        [ "594150", 4 ], [ "594383", 7 ],
                        [ "594616", 7 ], [ "594849", 1 ],
                        [ "595082", 2 ], [ "595315", 6 ],
                        [ "595548", 7 ], [ "595781", 1 ],
                        [ "596014", 4 ], [ "596247", 2 ],
                        [ "596480", 2 ], [ "596713", 2 ],
                        [ "596946", 7 ], [ "597179", 0 ],
                        [ "597412", 3 ], [ "597645", 2 ],
                        [ "597878", 3 ], [ "598111", 2 ],
                        [ "598344", 2 ], [ "598577", 6 ],
                        [ "598810", 2 ], [ "599043", 0 ],
                        [ "599276", 6 ], [ "599509", 4 ],
                        [ "599742", 6 ], [ "599975", 2 ],
                        [ "600208", 3 ], [ "600441", 0 ],
                        [ "600674", 7 ], [ "600907", 5 ],
                        [ "601140", 6 ], [ "601373", 1 ],
                        [ "601606", 4 ], [ "601839", 5 ],
                        [ "602072", 1 ], [ "602305", 2 ],
                        [ "602538", 2 ], [ "602771", 2 ],
                        [ "603004", 2 ], [ "603237", 7 ],
                        [ "603470", 4 ], [ "603703", 6 ],
                        [ "603936", 3 ], [ "604169", 6 ],
                        [ "604402", 1 ], [ "604635", 2 ],
                        [ "604868", 4 ], [ "605101", 4 ],
                        [ "614887", 1 ], [ "615120", 1 ],
                        [ "615353", 7 ], [ "615586", 5 ],
                        [ "615819", 4 ], [ "616052", 2 ],
                        [ "616285", 0 ], [ "616518", 7 ],
                        [ "616751", 4 ], [ "616984", 7 ],
                        [ "617217", 2 ], [ "617450", 3 ],
                        [ "617683", 0 ], [ "617916", 6 ],
                        [ "618149", 7 ], [ "618382", 6 ],
                        [ "618615", 6 ], [ "618848", 3 ],
                        [ "619081", 0 ], [ "619314", 7 ],
                        [ "619547", 2 ], [ "619780", 0 ],
                        [ "620013", 6 ], [ "620246", 4 ],
                        [ "620479", 2 ], [ "620712", 6 ],
                        [ "625139", 0 ], [ "625372", 7 ],
                        [ "625605", 6 ], [ "625838", 1 ],
                        [ "626071", 4 ], [ "626304", 2 ],
                        [ "626537", 4 ], [ "626770", 0 ],
                        [ "627003", 7 ], [ "627236", 3 ],
                        [ "627469", 3 ], [ "627702", 7 ],
                        [ "627935", 0 ], [ "628168", 5 ],
                        [ "628401", 3 ], [ "628634", 3 ],
                        [ "628867", 2 ], [ "629100", 4 ],
                        [ "629333", 3 ], [ "629566", 3 ],
                        [ "629799", 1 ], [ "630032", 4 ],
                        [ "630265", 7 ], [ "630498", 2 ],
                        [ "630731", 5 ], [ "630964", 5 ],
                        [ "631197", 1 ], [ "631430", 2 ],
                        [ "631663", 6 ], [ "631896", 3 ],
                        [ "632129", 1 ], [ "632362", 6 ],
                        [ "632595", 5 ], [ "632828", 2 ],
                        [ "633061", 7 ], [ "633294", 4 ],
                        [ "633527", 2 ], [ "633760", 5 ],
                        [ "633993", 6 ], [ "634226", 1 ],
                        [ "634459", 0 ], [ "634692", 1 ],
                        [ "634925", 5 ], [ "635158", 3 ],
                        [ "635391", 3 ], [ "635624", 3 ],
                        [ "635857", 4 ], [ "636090", 1 ],
                        [ "636323", 2 ], [ "636556", 0 ],
                        [ "636789", 3 ], [ "637022", 0 ],
                        [ "637255", 3 ], [ "637488", 2 ],
                        [ "637721", 2 ], [ "637954", 1 ],
                        [ "638187", 1 ], [ "638420", 4 ],
                        [ "638653", 4 ], [ "638886", 1 ],
                        [ "639119", 5 ], [ "639352", 5 ],
                        [ "639585", 1 ], [ "639818", 6 ],
                        [ "640051", 2 ], [ "640284", 2 ],
                        [ "640517", 6 ], [ "640750", 3 ],
                        [ "640983", 2 ], [ "641216", 1 ],
                        [ "641449", 0 ], [ "641682", 3 ],
                        [ "641915", 4 ], [ "642148", 0 ],
                        [ "642381", 1 ], [ "642614", 6 ],
                        [ "642847", 6 ], [ "643080", 2 ],
                        [ "643313", 5 ], [ "643546", 3 ],
                        [ "643779", 2 ], [ "644012", 4 ],
                        [ "644245", 2 ], [ "644478", 7 ],
                        [ "644711", 7 ], [ "644944", 1 ],
                        [ "645177", 7 ], [ "645410", 5 ],
                        [ "645643", 1 ], [ "645876", 3 ],
                        [ "646109", 3 ], [ "646342", 0 ],
                        [ "646575", 2 ], [ "646808", 2 ],
                        [ "647041", 3 ], [ "647274", 4 ],
                        [ "647507", 6 ], [ "647740", 0 ],
                        [ "647973", 6 ], [ "648206", 6 ],
                        [ "648439", 7 ], [ "648672", 3 ],
                        [ "648905", 3 ], [ "649138", 4 ],
                        [ "649371", 5 ], [ "649604", 2 ],
                        [ "649837", 6 ], [ "650070", 4 ],
                        [ "650303", 3 ], [ "650536", 7 ],
                        [ "650769", 7 ], [ "651002", 7 ],
                        [ "651235", 3 ], [ "651468", 0 ],
                        [ "651701", 2 ], [ "651934", 3 ],
                        [ "652167", 3 ], [ "652400", 6 ],
                        [ "652633", 3 ], [ "652866", 2 ],
                        [ "653099", 0 ], [ "653332", 7 ],
                        [ "653565", 1 ], [ "653798", 1 ],
                        [ "654031", 5 ], [ "654264", 1 ],
                        [ "654497", 3 ], [ "654730", 6 ],
                        [ "654963", 7 ], [ "655196", 0 ],
                        [ "655429", 7 ], [ "655662", 4 ],
                        [ "655895", 6 ], [ "656128", 2 ],
                        [ "656361", 7 ], [ "656594", 4 ],
                        [ "656827", 5 ], [ "657060", 2 ],
                        [ "657293", 1 ], [ "657526", 4 ],
                        [ "657759", 1 ], [ "657992", 3 ],
                        [ "658225", 4 ], [ "658458", 7 ],
                        [ "658691", 0 ], [ "658924", 3 ],
                        [ "659157", 1 ], [ "659390", 1 ],
                        [ "659623", 1 ], [ "659856", 2 ],
                        [ "660089", 7 ], [ "660322", 6 ],
                        [ "660555", 3 ], [ "660788", 6 ],
                        [ "661021", 0 ], [ "661254", 6 ],
                        [ "661487", 1 ], [ "661720", 4 ],
                        [ "661953", 6 ], [ "662186", 0 ],
                        [ "662419", 0 ], [ "662652", 2 ],
                        [ "662885", 6 ], [ "677564", 4 ],
                        [ "677797", 6 ], [ "678030", 5 ],
                        [ "678263", 7 ], [ "678496", 7 ],
                        [ "678729", 5 ], [ "678962", 0 ],
                        [ "679195", 6 ], [ "679428", 5 ],
                        [ "679661", 4 ], [ "679894", 7 ],
                        [ "680127", 4 ], [ "680360", 0 ],
                        [ "680593", 7 ], [ "680826", 6 ],
                        [ "681059", 4 ], [ "681292", 6 ],
                        [ "681525", 2 ], [ "681758", 2 ],
                        [ "685952", 3 ], [ "686185", 5 ],
                        [ "686418", 4 ], [ "686651", 2 ],
                        [ "686884", 7 ], [ "687117", 5 ],
                        [ "687350", 3 ], [ "687583", 7 ],
                        [ "687816", 6 ], [ "688049", 6 ],
                        [ "688282", 4 ], [ "688515", 5 ],
                        [ "688748", 3 ], [ "688981", 0 ],
                        [ "689214", 4 ], [ "689447", 2 ],
                        [ "689680", 7 ], [ "689913", 7 ],
                        [ "690146", 5 ], [ "690379", 7 ],
                        [ "690612", 1 ], [ "690845", 5 ],
                        [ "691078", 1 ], [ "691311", 6 ],
                        [ "691544", 7 ], [ "691777", 4 ],
                        [ "692010", 6 ], [ "692243", 3 ],
                        [ "692476", 2 ], [ "692709", 2 ],
                        [ "692942", 0 ], [ "693175", 4 ],
                        [ "693408", 4 ], [ "693641", 5 ],
                        [ "693874", 2 ], [ "694107", 2 ],
                        [ "694340", 4 ], [ "694573", 3 ],
                        [ "694806", 5 ], [ "695039", 0 ],
                        [ "695272", 5 ], [ "695505", 2 ],
                        [ "695738", 5 ], [ "695971", 7 ],
                        [ "696204", 1 ], [ "696437", 0 ],
                        [ "696670", 5 ], [ "696903", 6 ],
                        [ "697136", 1 ], [ "697369", 5 ],
                        [ "697602", 1 ], [ "697835", 3 ],
                        [ "698068", 7 ], [ "698301", 4 ],
                        [ "698534", 3 ], [ "698767", 5 ],
                        [ "699000", 6 ], [ "699233", 3 ],
                        [ "699466", 4 ], [ "699699", 2 ],
                        [ "699932", 1 ], [ "700165", 5 ],
                        [ "700398", 0 ], [ "700631", 0 ],
                        [ "700864", 2 ], [ "701097", 3 ],
                        [ "701330", 6 ], [ "701563", 5 ],
                        [ "701796", 7 ], [ "702029", 3 ],
                        [ "702262", 3 ], [ "702495", 4 ],
                        [ "702728", 6 ], [ "702961", 3 ],
                        [ "703194", 4 ], [ "703427", 6 ],
                        [ "703660", 0 ], [ "703893", 5 ],
                        [ "704126", 3 ], [ "704359", 0 ],
                        [ "704592", 3 ], [ "704825", 1 ],
                        [ "705058", 7 ], [ "705291", 5 ],
                        [ "705524", 2 ], [ "705757", 6 ],
                        [ "705990", 6 ], [ "706223", 7 ],
                        [ "706456", 6 ], [ "706689", 7 ],
                        [ "706922", 6 ], [ "707155", 7 ],
                        [ "707388", 3 ], [ "707621", 7 ],
                        [ "707854", 7 ], [ "708087", 6 ],
                        [ "708320", 4 ], [ "708553", 5 ],
                        [ "708786", 2 ], [ "709019", 6 ],
                        [ "709252", 3 ], [ "709485", 7 ],
                        [ "709718", 0 ], [ "709951", 2 ],
                        [ "710184", 0 ], [ "710417", 5 ],
                        [ "710650", 0 ], [ "710883", 6 ],
                        [ "711116", 3 ], [ "711349", 6 ],
                        [ "711582", 0 ], [ "711815", 6 ],
                        [ "712048", 5 ], [ "712281", 6 ],
                        [ "712514", 1 ], [ "712747", 2 ],
                        [ "712980", 0 ], [ "713213", 7 ],
                        [ "713446", 1 ], [ "713679", 7 ],
                        [ "713912", 2 ], [ "714145", 2 ],
                        [ "714378", 5 ], [ "714611", 7 ],
                        [ "714844", 2 ], [ "715077", 7 ],
                        [ "715310", 2 ], [ "715543", 6 ],
                        [ "715776", 1 ], [ "716009", 4 ],
                        [ "716242", 6 ], [ "716475", 7 ],
                        [ "716708", 6 ], [ "716941", 5 ],
                        [ "717174", 2 ], [ "717407", 1 ],
                        [ "717640", 2 ], [ "717873", 2 ],
                        [ "718106", 1 ], [ "718339", 7 ],
                        [ "718572", 4 ], [ "718805", 4 ],
                        [ "719038", 7 ], [ "719271", 4 ],
                        [ "719504", 2 ], [ "719737", 4 ],
                        [ "719970", 5 ], [ "720203", 6 ],
                        [ "720436", 1 ], [ "720669", 6 ],
                        [ "720902", 6 ], [ "721135", 1 ],
                        [ "721368", 0 ], [ "721601", 1 ],
                        [ "721834", 2 ], [ "722067", 4 ],
                        [ "722300", 7 ], [ "722533", 0 ],
                        [ "722766", 3 ], [ "722999", 7 ],
                        [ "723232", 7 ], [ "723465", 7 ],
                        [ "723698", 1 ], [ "723931", 5 ],
                        [ "724164", 4 ], [ "724397", 7 ],
                        [ "724630", 4 ], [ "724863", 3 ],
                        [ "725096", 3 ], [ "725329", 7 ],
                        [ "725562", 0 ], [ "729290", 6 ],
                        [ "729523", 0 ], [ "729756", 0 ],
                        [ "729989", 4 ], [ "730222", 2 ],
                        [ "730455", 2 ], [ "730688", 5 ],
                        [ "730921", 0 ], [ "731154", 3 ],
                        [ "731387", 0 ], [ "731620", 6 ],
                        [ "731853", 2 ], [ "732086", 6 ],
                        [ "732319", 6 ], [ "732552", 5 ],
                        [ "732785", 1 ], [ "733018", 4 ],
                        [ "733251", 7 ], [ "733484", 2 ],
                        [ "733717", 4 ], [ "733950", 4 ],
                        [ "734183", 0 ], [ "734416", 6 ],
                        [ "734649", 3 ], [ "734882", 7 ],
                        [ "735115", 3 ], [ "735348", 0 ],
                        [ "735581", 6 ], [ "735814", 1 ],
                        [ "736047", 0 ], [ "736280", 2 ],
                        [ "736513", 2 ], [ "736746", 5 ],
                        [ "736979", 2 ], [ "737212", 5 ],
                        [ "737445", 3 ], [ "737678", 3 ],
                        [ "737911", 3 ], [ "740474", 6 ],
                        [ "740707", 1 ], [ "740940", 0 ],
                        [ "741173", 1 ], [ "741406", 4 ],
                        [ "741639", 7 ], [ "741872", 4 ],
                        [ "742105", 5 ], [ "742338", 6 ],
                        [ "742571", 0 ], [ "742804", 6 ],
                        [ "743037", 6 ], [ "743270", 0 ],
                        [ "743503", 0 ], [ "743736", 4 ],
                        [ "743969", 5 ], [ "744202", 6 ],
                        [ "744435", 1 ], [ "744668", 1 ],
                        [ "744901", 4 ], [ "745134", 0 ],
                        [ "745367", 6 ], [ "745600", 6 ],
                        [ "745833", 6 ], [ "746066", 5 ],
                        [ "746299", 4 ], [ "746532", 2 ],
                        [ "746765", 5 ], [ "746998", 2 ],
                        [ "747231", 6 ], [ "747464", 0 ],
                        [ "747697", 5 ], [ "747930", 2 ],
                        [ "748163", 5 ], [ "748396", 6 ],
                        [ "748629", 5 ], [ "748862", 1 ],
                        [ "749095", 3 ], [ "749328", 6 ],
                        [ "749561", 3 ], [ "749794", 4 ],
                        [ "750027", 2 ], [ "750260", 5 ],
                        [ "750493", 4 ], [ "750726", 3 ],
                        [ "750959", 0 ], [ "751192", 0 ],
                        [ "751425", 5 ], [ "751658", 2 ],
                        [ "751891", 5 ], [ "752124", 5 ],
                        [ "752357", 4 ], [ "752590", 2 ],
                        [ "752823", 1 ], [ "753056", 0 ],
                        [ "753289", 6 ], [ "753522", 7 ],
                        [ "753755", 2 ], [ "753988", 1 ],
                        [ "754221", 7 ], [ "754454", 0 ],
                        [ "754687", 6 ], [ "754920", 5 ],
                        [ "755153", 1 ], [ "755386", 0 ],
                        [ "755619", 6 ], [ "755852", 2 ],
                        [ "756085", 4 ], [ "756318", 5 ],
                        [ "756551", 2 ], [ "756784", 6 ],
                        [ "757017", 6 ], [ "757250", 1 ],
                        [ "757483", 1 ], [ "757716", 2 ],
                        [ "757949", 1 ], [ "758182", 2 ],
                        [ "758415", 3 ], [ "758648", 5 ],
                        [ "758881", 5 ], [ "759114", 4 ],
                        [ "759347", 7 ], [ "759580", 7 ],
                        [ "759813", 0 ], [ "760046", 0 ],
                        [ "760279", 2 ], [ "760512", 5 ],
                        [ "760745", 7 ], [ "760978", 5 ],
                        [ "761211", 3 ], [ "761444", 6 ],
                        [ "761677", 0 ], [ "761910", 3 ],
                        [ "762143", 6 ], [ "762376", 6 ],
                        [ "762609", 1 ], [ "762842", 5 ],
                        [ "763075", 6 ], [ "763308", 2 ],
                        [ "763541", 6 ], [ "763774", 4 ],
                        [ "764007", 2 ], [ "764240", 5 ],
                        [ "764473", 3 ], [ "764706", 2 ],
                        [ "764939", 1 ], [ "765172", 7 ],
                        [ "765405", 1 ], [ "765638", 5 ],
                        [ "765871", 0 ], [ "766104", 0 ],
                        [ "766337", 4 ], [ "766570", 3 ],
                        [ "766803", 4 ], [ "767036", 5 ],
                        [ "767269", 2 ], [ "767502", 4 ],
                        [ "767735", 1 ], [ "767968", 2 ],
                        [ "768201", 6 ], [ "777521", 1 ],
                        [ "777754", 3 ], [ "777987", 1 ],
                        [ "778220", 2 ], [ "778453", 5 ],
                        [ "778686", 7 ], [ "778919", 0 ],
                        [ "779152", 0 ], [ "779385", 2 ],
                        [ "779618", 4 ], [ "779851", 7 ],
                        [ "780084", 6 ], [ "780317", 1 ],
                        [ "780550", 0 ], [ "780783", 1 ],
                        [ "781016", 2 ], [ "781249", 5 ],
                        [ "781482", 0 ], [ "781715", 6 ],
                        [ "781948", 2 ], [ "782181", 5 ],
                        [ "782414", 1 ], [ "782647", 4 ],
                        [ "782880", 4 ], [ "783113", 3 ],
                        [ "783346", 7 ], [ "783579", 7 ],
                        [ "783812", 0 ], [ "784045", 4 ],
                        [ "784278", 0 ], [ "784511", 7 ],
                        [ "784744", 7 ], [ "784977", 0 ],
                        [ "785210", 6 ], [ "785443", 1 ],
                        [ "785676", 7 ], [ "785909", 2 ],
                        [ "786142", 7 ], [ "786375", 7 ],
                        [ "786608", 1 ], [ "786841", 0 ],
                        [ "787074", 7 ], [ "787307", 6 ],
                        [ "787540", 1 ], [ "787773", 2 ],
                        [ "788006", 7 ], [ "788239", 0 ],
                        [ "788472", 1 ], [ "788705", 5 ],
                        [ "788938", 6 ], [ "789171", 6 ],
                        [ "789404", 1 ], [ "789637", 7 ],
                        [ "789870", 2 ], [ "790103", 7 ],
                        [ "790336", 5 ], [ "790569", 4 ],
                        [ "790802", 5 ], [ "791035", 6 ],
                        [ "791268", 6 ], [ "791501", 6 ],
                        [ "791734", 7 ], [ "791967", 5 ],
                        [ "792200", 2 ], [ "792433", 5 ],
                        [ "792666", 6 ], [ "792899", 1 ],
                        [ "793132", 0 ], [ "793365", 4 ],
                        [ "793598", 1 ], [ "793831", 5 ],
                        [ "794064", 2 ], [ "794297", 0 ],
                        [ "794530", 0 ], [ "794763", 6 ],
                        [ "794996", 5 ], [ "795229", 4 ],
                        [ "795462", 0 ], [ "795695", 7 ],
                        [ "795928", 3 ], [ "796161", 2 ],
                        [ "796394", 6 ], [ "796627", 4 ],
                        [ "796860", 7 ], [ "797093", 6 ],
                        [ "797326", 2 ], [ "797559", 2 ],
                        [ "797792", 7 ], [ "798025", 3 ],
                        [ "798258", 2 ], [ "798491", 2 ],
                        [ "798724", 7 ], [ "798957", 1 ],
                        [ "799190", 4 ], [ "799423", 1 ],
                        [ "799656", 5 ], [ "799889", 4 ],
                        [ "800122", 4 ], [ "800355", 2 ],
                        [ "800588", 5 ], [ "800821", 4 ],
                        [ "801054", 2 ], [ "801287", 1 ],
                        [ "801520", 1 ], [ "801753", 7 ],
                        [ "801986", 1 ], [ "802219", 0 ],
                        [ "802452", 1 ], [ "802685", 6 ],
                        [ "802918", 7 ], [ "803151", 5 ],
                        [ "803384", 6 ], [ "803617", 6 ],
                        [ "803850", 2 ], [ "804083", 7 ],
                        [ "804316", 7 ], [ "804549", 5 ],
                        [ "804782", 3 ], [ "805015", 6 ],
                        [ "805248", 4 ], [ "805481", 1 ],
                        [ "805714", 6 ], [ "805947", 5 ],
                        [ "806180", 1 ], [ "806413", 2 ],
                        [ "806646", 1 ], [ "806879", 4 ],
                        [ "807112", 7 ], [ "807345", 0 ],
                        [ "807578", 6 ], [ "807811", 4 ],
                        [ "808044", 1 ], [ "808277", 6 ],
                        [ "808510", 4 ], [ "808743", 7 ],
                        [ "811772", 4 ], [ "812005", 6 ],
                        [ "812238", 0 ], [ "812471", 1 ],
                        [ "815500", 4 ], [ "815733", 5 ],
                        [ "815966", 7 ], [ "816199", 7 ],
                        [ "816432", 1 ], [ "816665", 3 ],
                        [ "816898", 0 ], [ "817131", 0 ],
                        [ "817364", 6 ], [ "817597", 2 ],
                        [ "817830", 1 ], [ "818063", 6 ],
                        [ "818296", 5 ], [ "818529", 3 ],
                        [ "818762", 2 ], [ "818995", 1 ],
                        [ "819228", 3 ], [ "819461", 6 ],
                        [ "819694", 4 ], [ "819927", 5 ],
                        [ "820160", 4 ], [ "820393", 1 ],
                        [ "820626", 0 ], [ "820859", 5 ],
                        [ "821092", 7 ], [ "821325", 1 ],
                        [ "821558", 5 ], [ "821791", 1 ],
                        [ "822024", 0 ], [ "822257", 2 ],
                        [ "822490", 0 ], [ "822723", 5 ],
                        [ "822956", 0 ], [ "823189", 6 ],
                        [ "823422", 1 ], [ "823655", 2 ],
                        [ "823888", 6 ], [ "824121", 1 ],
                        [ "824354", 5 ], [ "824587", 6 ],
                        [ "824820", 7 ], [ "825053", 4 ],
                        [ "825286", 3 ], [ "825519", 7 ],
                        [ "825752", 7 ], [ "825985", 7 ],
                        [ "826218", 7 ], [ "826451", 3 ],
                        [ "826684", 6 ], [ "826917", 4 ],
                        [ "827150", 2 ], [ "827383", 0 ],
                        [ "827616", 7 ], [ "827849", 0 ],
                        [ "828082", 6 ], [ "828315", 6 ],
                        [ "828548", 3 ], [ "828781", 6 ],
                        [ "829014", 4 ], [ "829247", 6 ],
                        [ "829480", 1 ], [ "829713", 3 ],
                        [ "829946", 1 ], [ "830179", 0 ],
                        [ "830412", 2 ], [ "830645", 5 ],
                        [ "830878", 6 ], [ "831111", 0 ],
                        [ "831344", 6 ], [ "831577", 6 ],
                        [ "831810", 0 ], [ "832043", 7 ],
                        [ "832276", 2 ], [ "832509", 2 ],
                        [ "832742", 5 ], [ "832975", 3 ],
                        [ "833208", 0 ], [ "833441", 3 ],
                        [ "833674", 6 ], [ "833907", 2 ],
                        [ "834140", 3 ], [ "834373", 2 ],
                        [ "834606", 6 ], [ "834839", 2 ],
                        [ "835072", 3 ], [ "835305", 4 ],
                        [ "835538", 2 ], [ "835771", 2 ],
                        [ "836004", 5 ], [ "836237", 6 ],
                        [ "836470", 0 ], [ "836703", 6 ],
                        [ "836936", 1 ], [ "837169", 3 ],
                        [ "837402", 1 ], [ "837635", 5 ],
                        [ "837868", 7 ], [ "838101", 1 ],
                        [ "838334", 3 ], [ "838567", 3 ],
                        [ "838800", 6 ], [ "839033", 3 ],
                        [ "839266", 0 ], [ "839499", 4 ],
                        [ "839732", 2 ], [ "839965", 6 ],
                        [ "840198", 6 ], [ "840431", 6 ],
                        [ "840664", 5 ], [ "840897", 0 ],
                        [ "841130", 6 ], [ "841363", 1 ],
                        [ "841596", 5 ], [ "841829", 6 ],
                        [ "842062", 3 ], [ "842295", 6 ],
                        [ "842528", 6 ], [ "842761", 1 ],
                        [ "842994", 6 ], [ "843227", 6 ],
                        [ "843460", 4 ], [ "843693", 0 ],
                        [ "843926", 4 ], [ "844159", 0 ],
                        [ "844392", 5 ], [ "844625", 0 ],
                        [ "844858", 4 ], [ "845091", 7 ],
                        [ "845324", 5 ], [ "845557", 7 ],
                        [ "845790", 3 ], [ "846023", 0 ],
                        [ "846256", 1 ], [ "846489", 7 ],
                        [ "846722", 3 ], [ "846955", 1 ],
                        [ "847188", 5 ], [ "847421", 0 ],
                        [ "847654", 1 ], [ "847887", 0 ],
                        [ "848120", 7 ], [ "848353", 5 ],
                        [ "848586", 5 ], [ "848819", 1 ],
                        [ "849052", 2 ], [ "849285", 0 ],
                        [ "849518", 4 ], [ "849751", 2 ],
                        [ "849984", 5 ], [ "850217", 1 ],
                        [ "855576", 7 ], [ "855809", 1 ],
                        [ "856042", 6 ], [ "856275", 5 ],
                        [ "856508", 6 ], [ "856741", 2 ],
                        [ "856974", 1 ], [ "857207", 3 ],
                        [ "857440", 6 ], [ "857673", 4 ],
                        [ "857906", 7 ], [ "858139", 2 ],
                        [ "858372", 0 ], [ "858605", 0 ],
                        [ "858838", 5 ], [ "859071", 4 ],
                        [ "859304", 2 ], [ "859537", 1 ],
                        [ "859770", 7 ], [ "860003", 7 ],
                        [ "860236", 3 ], [ "860469", 1 ],
                        [ "860702", 6 ], [ "860935", 5 ],
                        [ "861168", 3 ], [ "861401", 1 ],
                        [ "861634", 4 ], [ "861867", 4 ],
                        [ "862100", 0 ], [ "862333", 7 ],
                        [ "862566", 3 ], [ "862799", 0 ],
                        [ "863032", 3 ], [ "863265", 2 ],
                        [ "863498", 5 ], [ "863731", 7 ],
                        [ "863964", 5 ], [ "864197", 4 ],
                        [ "864430", 5 ], [ "864663", 0 ],
                        [ "864896", 5 ], [ "865129", 2 ],
                        [ "865362", 7 ], [ "865595", 3 ],
                        [ "865828", 7 ], [ "866061", 4 ],
                        [ "866294", 2 ], [ "866527", 4 ],
                        [ "866760", 2 ], [ "866993", 1 ],
                        [ "867226", 0 ], [ "867459", 0 ],
                        [ "867692", 5 ], [ "867925", 5 ],
                        [ "868158", 3 ], [ "868391", 7 ],
                        [ "868624", 2 ], [ "868857", 4 ],
                        [ "869090", 3 ], [ "869323", 6 ],
                        [ "869556", 0 ], [ "869789", 1 ],
                        [ "870022", 0 ], [ "870255", 3 ],
                        [ "870488", 6 ], [ "870721", 0 ],
                        [ "870954", 6 ], [ "871187", 0 ],
                        [ "871420", 3 ], [ "871653", 3 ],
                        [ "871886", 7 ], [ "872119", 5 ],
                        [ "872352", 5 ], [ "872585", 0 ],
                        [ "872818", 2 ], [ "873051", 3 ],
                        [ "873284", 7 ], [ "873517", 2 ],
                        [ "873750", 7 ], [ "877711", 5 ],
                        [ "877944", 5 ], [ "878177", 5 ],
                        [ "878410", 2 ], [ "878643", 5 ],
                        [ "878876", 4 ], [ "879109", 3 ],
                        [ "879342", 6 ], [ "879575", 5 ],
                        [ "879808", 0 ], [ "880041", 6 ],
                        [ "880274", 7 ], [ "880507", 1 ],
                        [ "880740", 6 ], [ "880973", 4 ],
                        [ "881206", 7 ], [ "881439", 5 ],
                        [ "881672", 5 ], [ "881905", 1 ],
                        [ "882138", 7 ], [ "882371", 2 ],
                        [ "882604", 0 ], [ "882837", 1 ],
                        [ "883070", 7 ], [ "883303", 2 ],
                        [ "883536", 0 ], [ "883769", 4 ],
                        [ "884002", 1 ], [ "884235", 4 ],
                        [ "884468", 3 ], [ "884701", 2 ],
                        [ "884934", 6 ], [ "885167", 3 ],
                        [ "885400", 6 ], [ "885633", 7 ],
                        [ "885866", 6 ], [ "886099", 3 ],
                        [ "886332", 3 ], [ "886565", 6 ],
                        [ "886798", 2 ], [ "887031", 2 ],
                        [ "887264", 5 ], [ "887497", 6 ],
                        [ "887730", 6 ], [ "887963", 6 ],
                        [ "888196", 2 ], [ "888429", 7 ],
                        [ "888662", 4 ], [ "888895", 3 ],
                        [ "889128", 2 ], [ "889361", 2 ],
                        [ "889594", 3 ], [ "889827", 4 ],
                        [ "890060", 6 ], [ "890293", 2 ],
                        [ "890526", 0 ], [ "890759", 2 ],
                        [ "890992", 2 ], [ "891225", 0 ],
                        [ "891458", 5 ], [ "891691", 5 ],
                        [ "891924", 1 ], [ "892157", 3 ],
                        [ "892390", 3 ], [ "892623", 7 ],
                        [ "892856", 0 ], [ "893089", 6 ],
                        [ "893322", 7 ], [ "893555", 4 ],
                        [ "893788", 5 ], [ "894021", 7 ],
                        [ "894254", 5 ], [ "894487", 3 ],
                        [ "894720", 5 ], [ "894953", 5 ],
                        [ "895186", 2 ], [ "895419", 0 ],
                        [ "895652", 7 ], [ "895885", 1 ],
                        [ "896118", 6 ], [ "896351", 2 ],
                        [ "896584", 1 ], [ "896817", 0 ],
                        [ "897050", 1 ], [ "897283", 1 ],
                        [ "897516", 3 ], [ "897749", 4 ],
                        [ "897982", 5 ], [ "898215", 4 ],
                        [ "898448", 6 ], [ "898681", 1 ],
                        [ "898914", 1 ], [ "899147", 3 ],
                        [ "899380", 4 ], [ "899613", 0 ],
                        [ "899846", 1 ], [ "900079", 2 ],
                        [ "900312", 0 ], [ "900545", 5 ],
                        [ "900778", 5 ], [ "901011", 1 ],
                        [ "901244", 6 ], [ "901477", 5 ],
                        [ "901710", 1 ], [ "901943", 7 ],
                        [ "902176", 5 ], [ "902409", 6 ],
                        [ "902642", 3 ], [ "902875", 4 ],
                        [ "903108", 5 ], [ "907535", 0 ],
                        [ "907768", 2 ], [ "908001", 5 ],
                        [ "908234", 4 ], [ "908467", 1 ],
                        [ "908700", 7 ], [ "908933", 7 ],
                        [ "909166", 1 ], [ "909399", 1 ],
                        [ "909632", 6 ], [ "909865", 2 ],
                        [ "910098", 1 ], [ "910331", 5 ],
                        [ "910564", 1 ], [ "910797", 4 ],
                        [ "911030", 7 ], [ "911263", 6 ],
                        [ "911496", 1 ], [ "911729", 1 ],
                        [ "911962", 0 ], [ "912195", 4 ],
                        [ "912428", 7 ], [ "912661", 7 ],
                        [ "912894", 0 ], [ "913127", 7 ],
                        [ "913360", 6 ], [ "913593", 7 ],
                        [ "913826", 0 ], [ "914059", 1 ],
                        [ "914292", 7 ], [ "914525", 4 ],
                        [ "914758", 0 ], [ "914991", 3 ],
                        [ "915224", 6 ], [ "915457", 0 ],
                        [ "915690", 1 ], [ "915923", 0 ],
                        [ "916156", 7 ], [ "916389", 5 ],
                        [ "916622", 7 ], [ "916855", 4 ],
                        [ "917088", 5 ], [ "917321", 1 ],
                        [ "917554", 7 ], [ "917787", 2 ],
                        [ "918020", 2 ], [ "918253", 0 ],
                        [ "918486", 0 ], [ "918719", 4 ],
                        [ "918952", 0 ], [ "919185", 1 ],
                        [ "919418", 2 ], [ "919651", 5 ],
                        [ "919884", 6 ], [ "920117", 7 ],
                        [ "920350", 3 ], [ "920583", 6 ],
                        [ "920816", 1 ], [ "921049", 5 ],
                        [ "921282", 5 ], [ "921515", 1 ],
                        [ "921748", 1 ], [ "921981", 0 ],
                        [ "922214", 6 ], [ "922447", 4 ],
                        [ "922680", 6 ], [ "922913", 5 ],
                        [ "923146", 2 ], [ "923379", 0 ],
                        [ "923612", 4 ], [ "923845", 6 ],
                        [ "924078", 3 ], [ "924311", 6 ],
                        [ "924544", 1 ], [ "924777", 6 ],
                        [ "925010", 7 ], [ "925243", 7 ],
                        [ "925476", 6 ], [ "925709", 7 ],
                        [ "925942", 5 ], [ "926175", 3 ],
                        [ "926408", 2 ], [ "926641", 1 ],
                        [ "926874", 4 ], [ "927107", 2 ],
                        [ "927340", 0 ], [ "927573", 2 ],
                        [ "927806", 2 ], [ "932699", 3 ],
                        [ "932932", 6 ], [ "933165", 2 ],
                        [ "933398", 7 ], [ "933631", 4 ],
                        [ "933864", 5 ], [ "934097", 7 ],
                        [ "934330", 4 ], [ "934563", 0 ],
                        [ "934796", 3 ], [ "935029", 1 ],
                        [ "935262", 5 ], [ "935495", 2 ],
                        [ "935728", 5 ], [ "935961", 2 ],
                        [ "936194", 5 ], [ "936427", 5 ],
                        [ "936660", 4 ], [ "936893", 6 ],
                        [ "937126", 7 ], [ "937359", 5 ],
                        [ "937592", 6 ], [ "937825", 2 ],
                        [ "938058", 6 ], [ "938291", 6 ],
                        [ "938524", 1 ], [ "938757", 5 ],
                        [ "938990", 1 ], [ "939223", 6 ],
                        [ "939456", 3 ], [ "939689", 0 ],
                        [ "939922", 4 ], [ "940155", 1 ],
                        [ "940388", 0 ], [ "940621", 3 ],
                        [ "940854", 2 ], [ "941087", 3 ],
                        [ "944815", 3 ], [ "945048", 0 ],
                        [ "945281", 1 ], [ "945514", 3 ],
                        [ "945747", 4 ], [ "945980", 3 ],
                        [ "946213", 1 ], [ "946446", 1 ],
                        [ "946679", 0 ], [ "946912", 2 ],
                        [ "947145", 2 ], [ "947378", 0 ],
                        [ "947611", 2 ], [ "947844", 1 ],
                        [ "948077", 2 ], [ "948310", 6 ],
                        [ "948543", 4 ], [ "948776", 6 ],
                        [ "949009", 5 ], [ "949242", 0 ],
                        [ "949475", 3 ], [ "949708", 1 ],
                        [ "949941", 4 ], [ "950174", 0 ],
                        [ "950407", 3 ], [ "950640", 6 ],
                        [ "950873", 2 ], [ "951106", 2 ],
                        [ "951339", 0 ], [ "951572", 6 ],
                        [ "951805", 6 ], [ "952038", 0 ],
                        [ "952271", 7 ], [ "952504", 7 ],
                        [ "952737", 6 ], [ "952970", 0 ],
                        [ "953203", 1 ], [ "953436", 4 ],
                        [ "953669", 1 ], [ "953902", 3 ],
                        [ "954135", 0 ], [ "954368", 5 ],
                        [ "954601", 6 ], [ "954834", 2 ],
                        [ "955067", 5 ], [ "955300", 6 ],
                        [ "955533", 2 ], [ "955766", 0 ],
                        [ "955999", 7 ], [ "956232", 2 ],
                        [ "956465", 3 ], [ "956698", 4 ],
                        [ "956931", 1 ], [ "957164", 4 ],
                        [ "957397", 5 ], [ "957630", 6 ],
                        [ "957863", 6 ], [ "958096", 7 ],
                        [ "958329", 3 ], [ "958562", 7 ],
                        [ "958795", 3 ], [ "959028", 5 ],
                        [ "959261", 4 ], [ "959494", 0 ],
                        [ "959727", 5 ], [ "959960", 4 ],
                        [ "960193", 3 ], [ "960426", 7 ],
                        [ "960659", 1 ], [ "960892", 7 ],
                        [ "961125", 7 ], [ "961358", 1 ],
                        [ "961591", 1 ], [ "961824", 4 ],
                        [ "962057", 0 ], [ "962290", 2 ],
                        [ "962523", 5 ], [ "962756", 5 ],
                        [ "962989", 5 ], [ "963222", 3 ],
                        [ "963455", 1 ], [ "963688", 0 ],
                        [ "963921", 5 ], [ "964154", 2 ],
                        [ "964387", 0 ], [ "964620", 6 ],
                        [ "964853", 1 ], [ "965086", 4 ],
                        [ "965319", 4 ], [ "965552", 6 ],
                        [ "965785", 6 ], [ "966018", 2 ],
                        [ "966251", 5 ], [ "966484", 7 ],
                        [ "966717", 6 ], [ "966950", 5 ],
                        [ "967183", 2 ], [ "967416", 0 ],
                        [ "967649", 1 ], [ "967882", 7 ],
                        [ "968115", 6 ], [ "968348", 2 ],
                        [ "968581", 3 ], [ "968814", 3 ],
                        [ "969047", 2 ], [ "969280", 6 ],
                        [ "969513", 5 ], [ "969746", 0 ],
                        [ "969979", 3 ], [ "970212", 0 ],
                        [ "970445", 5 ], [ "970678", 0 ],
                        [ "970911", 2 ], [ "971144", 5 ],
                        [ "971377", 0 ], [ "971610", 0 ],
                        [ "971843", 3 ], [ "972076", 3 ],
                        [ "972309", 2 ], [ "976037", 0 ],
                        [ "976270", 1 ], [ "976503", 5 ],
                        [ "976736", 6 ], [ "976969", 6 ],
                        [ "977202", 6 ], [ "977435", 5 ],
                        [ "977668", 5 ], [ "977901", 1 ],
                        [ "978134", 4 ], [ "978367", 5 ],
                        [ "978600", 4 ], [ "978833", 0 ],
                        [ "979066", 4 ], [ "979299", 4 ],
                        [ "979532", 2 ], [ "979765", 3 ],
                        [ "979998", 7 ], [ "980231", 2 ],
                        [ "980464", 7 ], [ "980697", 0 ],
                        [ "980930", 1 ], [ "981163", 4 ],
                        [ "987454", 1 ], [ "987687", 1 ],
                        [ "987920", 6 ], [ "988153", 0 ],
                        [ "988386", 0 ], [ "988619", 5 ],
                        [ "988852", 2 ], [ "989085", 4 ],
                        [ "989318", 6 ], [ "989551", 4 ],
                        [ "989784", 7 ], [ "990017", 0 ],
                        [ "990250", 4 ], [ "990483", 7 ],
                        [ "990716", 4 ], [ "990949", 3 ],
                        [ "991182", 7 ], [ "991415", 2 ],
                        [ "991648", 3 ], [ "991881", 4 ],
                        [ "992114", 4 ], [ "992347", 2 ],
                        [ "992580", 6 ], [ "992813", 3 ],
                        [ "993046", 5 ], [ "993279", 2 ],
                        [ "993512", 4 ], [ "993745", 3 ],
                        [ "993978", 6 ], [ "994211", 6 ],
                        [ "994444", 4 ], [ "994677", 0 ],
                        [ "994910", 6 ], [ "995143", 6 ],
                        [ "995376", 3 ], [ "995609", 0 ],
                        [ "995842", 5 ], [ "996075", 5 ],
                        [ "996308", 5 ], [ "996541", 1 ],
                        [ "996774", 5 ], [ "997007", 6 ],
                        [ "997240", 1 ], [ "997473", 0 ],
                        [ "997706", 3 ], [ "999104", 7 ],
                        [ "999337", 3 ], [ "999570", 5 ],
                        [ "999803", 3 ] ]

        hasher = ConsistentHash(nodes = servers)
        for exp in expected:
            key = exp[0]
            server = servers[exp[1]]
            node = hasher.get_node(key)
            assert server == node