# Generated by Django 4.2.16 on 2024-11-01 12:40

from django.db import migrations, models
from aboutcode.pipeline import LoopProgress
from datetime import datetime
from datetime import timezone


class Migration(migrations.Migration):
    """
    Add url field to the VulnerabilitySeverity model and remove reference foreignkey
    relationship. Also, add new M2M 'severities' field in Vulnerability and
    AffectedByPackageRelatedVulnerability relationship.

    To achieve this following model changes and data migrations are applied in chronological order:
    - Make VulnerabilitySeverity reference field nullable (to make the migration reversible).
    - Add 'severities' field to AffectedByPackageRelatedVulnerability.
    - Add 'severities' field to Vulnerability.
    - Add 'url' field to VulnerabilitySeverity.
    - Data migration to remove corrupted SUSE scores.
    - Data migration to enable reprocessing of old suse Advisory on next import.
    - Data migration to populate new VulnerabilitySeverity url field using reference.
    - Data migration to populate Vulnerability 'severities' M2M relationship.
    - Delete VulnerabilitySeverity reference field.
    """

    def remove_inaccurate_suse_score(apps, schema_editor):
        """
        Remove inaccurate suse severity scores.
        See https://github.com/aboutcode-org/vulnerablecode/issues/1597
        """
        print(f"\nRemoving inaccurate suse severity scores.")
        VulnerabilitySeverity = apps.get_model("vulnerabilities", "VulnerabilitySeverity")
        VulnerabilitySeverity.objects.filter(
            reference__url="https://ftp.suse.com/pub/projects/security/yaml/suse-cvss-scores.yaml"
        ).delete()

    def reverse_remove_inaccurate_suse_score(apps, schema_editor):
        """Reverse data migration not needed for inaccurate severity scores."""
        pass

    def reprocess_suse_advisory_on_next_import(apps, schema_editor):
        """Clear `date_imported` on old suse advisory to enable reprocessing on next suse import."""

        print(f"\nEnable reprocess of old suse advisory on next import.")
        Advisory = apps.get_model("vulnerabilities", "Advisory")
        Advisory.objects.filter(
            created_by="vulnerabilities.importers.suse_scores.SUSESeverityScoreImporter"
        ).update(date_imported=None)

    def reverse_reprocess_suse_advisory_on_next_import(apps, schema_editor):
        """Populate `date_imported` on old suse advisory to prevent reprocessing on next suse import."""
        Advisory = apps.get_model("vulnerabilities", "Advisory")
        Advisory.objects.filter(
            created_by="vulnerabilities.importers.suse_scores.SUSESeverityScoreImporter"
        ).update(date_imported=datetime.now(timezone.utc))

    def populate_severity_model_with_url(apps, schema_editor):
        """Populate the new VulnerabilitySeverity `url` field using reference url"""

        VulnerabilitySeverity = apps.get_model("vulnerabilities", "VulnerabilitySeverity")
        chunk_size = 10000
        batch = []
        vulnerability_severity_query = VulnerabilitySeverity.objects.select_related("reference")

        print(
            f"\nPopulating {vulnerability_severity_query.count():,d} VulnerabilitySeverity"
            "`url` field using reference url."
        )
        progress = LoopProgress(
            total_iterations=vulnerability_severity_query.count(),
            progress_step=10,
            logger=print,
        )
        for severity in progress.iter(vulnerability_severity_query.iterator(chunk_size=chunk_size)):
            severity.url = severity.reference.url
            batch.append(severity)

            if len(batch) >= chunk_size:
                VulnerabilitySeverity.objects.bulk_update(
                    objs=batch,
                    fields=["url"],
                )
                batch.clear()

        VulnerabilitySeverity.objects.bulk_update(
            objs=batch,
            fields=["url"],
        )

    def reverse_populate_severity_model_reference_using_url(apps, schema_editor):
        """Reverse: Populate the reference using `url` to ensure proper reverse migration."""
        VulnerabilitySeverity = apps.get_model("vulnerabilities", "VulnerabilitySeverity")
        VulnerabilityReference = apps.get_model("vulnerabilities", "VulnerabilityReference")
        chunk_size = 10000
        batch = []
        vulnerability_severity_query = VulnerabilitySeverity.objects.select_related("reference")

        print(
            f"\nReverse: Populating {vulnerability_severity_query.count():,d} VulnerabilitySeverity"
            "`reference` relation using url."
        )
        progress = LoopProgress(
            total_iterations=vulnerability_severity_query.count(),
            progress_step=10,
            logger=print,
        )
        for severity in progress.iter(vulnerability_severity_query.iterator(chunk_size=chunk_size)):
            severity.reference = VulnerabilityReference.objects.get(url=severity.url)
            batch.append(severity)

            if len(batch) >= chunk_size:
                VulnerabilitySeverity.objects.bulk_update(
                    objs=batch,
                    fields=["reference"],
                )
                batch.clear()

        VulnerabilitySeverity.objects.bulk_update(
            objs=batch,
            fields=["reference"],
        )

    def populate_vulnerability_model_with_severities(apps, schema_editor):
        """Populate the new Vulnerability `severities` relation using referenced severity."""
        Vulnerability = apps.get_model("vulnerabilities", "Vulnerability")
        chunk_size = 10000

        vulnerability_query = Vulnerability.objects.prefetch_related(
            "references__vulnerabilityseverity_set"
        )
        print(
            f"\nPopulating {vulnerability_query.count():,d} Vulnerability `severities`"
            "relation using referenced severity."
        )
        progress = LoopProgress(
            total_iterations=vulnerability_query.count(),
            progress_step=10,
            logger=print,
        )
        for vulnerability in progress.iter(vulnerability_query.iterator(chunk_size=chunk_size)):
            references = vulnerability.references.all()
            severity_ids = [
                severity.id
                for reference in references
                for severity in reference.vulnerabilityseverity_set.all()
                if reference.vulnerabilityseverity_set.exists()
            ]
            vulnerability.severities.set(severity_ids)

    def reverse_populate_vulnerability_model_with_severities(apps, schema_editor):
        """Reverse data migration not needed for new `severities` relationship."""
        pass

    dependencies = [
        ("vulnerabilities", "0077_alter_packagechangelog_software_version_and_more"),
    ]

    operations = [
        # reference field should nullable to properly support reverse migration.
        migrations.AlterField(
            model_name="vulnerabilityseverity",
            name="reference",
            field=models.ForeignKey(
                to="vulnerabilities.vulnerabilityreference",
                on_delete=models.CASCADE,
                null=True,
            ),
        ),
        migrations.AlterModelOptions(
            name="vulnerabilityseverity",
            options={"ordering": ["url", "scoring_system", "value"]},
        ),
        migrations.AlterUniqueTogether(
            name="vulnerabilityseverity",
            unique_together=set(),
        ),
        migrations.AddField(
            model_name="affectedbypackagerelatedvulnerability",
            name="severities",
            field=models.ManyToManyField(
                related_name="affected_package_vulnerability_relations",
                to="vulnerabilities.vulnerabilityseverity",
            ),
        ),
        migrations.AddField(
            model_name="vulnerability",
            name="severities",
            field=models.ManyToManyField(
                related_name="vulnerabilities", to="vulnerabilities.vulnerabilityseverity"
            ),
        ),
        migrations.AddField(
            model_name="vulnerabilityseverity",
            name="url",
            field=models.URLField(
                help_text="URL to the vulnerability severity", max_length=1024, null=True
            ),
        ),
        migrations.RunPython(
            code=remove_inaccurate_suse_score,
            reverse_code=reverse_remove_inaccurate_suse_score,
        ),
        migrations.RunPython(
            code=reprocess_suse_advisory_on_next_import,
            reverse_code=reverse_reprocess_suse_advisory_on_next_import,
        ),
        migrations.RunPython(
            code=populate_severity_model_with_url,
            reverse_code=reverse_populate_severity_model_reference_using_url,
        ),
        migrations.RunPython(
            code=populate_vulnerability_model_with_severities,
            reverse_code=reverse_populate_vulnerability_model_with_severities,
        ),
        migrations.RemoveField(
            model_name="vulnerabilityseverity",
            name="reference",
        ),
    ]
