#
# Copyright (c) nexB Inc. and others. All rights reserved.
# VulnerableCode is a trademark of nexB Inc.
# SPDX-License-Identifier: Apache-2.0
# See http://www.apache.org/licenses/LICENSE-2.0 for the license text.
# See https://github.com/aboutcode-org/vulnerablecode for support or download.
# See https://aboutcode.org for more information about nexB OSS projects.
#

import os
from datetime import datetime
from unittest import mock

import pytest

from vulnerabilities.import_runner import associate_vulnerability_with_aliases
from vulnerabilities.improvers.vulnerability_status import VulnerabilityStatusImprover
from vulnerabilities.models import Advisory
from vulnerabilities.models import Alias
from vulnerabilities.models import Vulnerability
from vulnerabilities.models import VulnerabilityStatusType
from vulnerabilities.pipelines.nvd_importer import NVDImporterPipeline
from vulnerabilities.pipes.advisory import get_or_create_aliases

BASE_DIR = os.path.dirname(os.path.abspath(__file__))


TEST_DATA = os.path.join(
    BASE_DIR,
    "test_data/vulnerability_status_improver",
)


@pytest.mark.django_db(transaction=True)
def test_interesting_advisories():
    adv1 = Advisory.objects.create(
        unique_content_id="test-unique-content-id",
        url="https://test.url/",
        created_by=NVDImporterPipeline.pipeline_id,
        summary="1",
        date_collected=datetime.now(),
    )
    adv1.aliases.add(*get_or_create_aliases(["CVE-1"]))
    adv2 = Advisory.objects.create(
        unique_content_id="test-unique-content-id2",
        url="https://test.url/",
        created_by=NVDImporterPipeline.pipeline_id,
        summary="2",
        date_collected=datetime.now(),
    )
    adv2.aliases.add(*get_or_create_aliases(["CVE-1"]))
    advs = VulnerabilityStatusImprover().interesting_advisories
    assert len(list(advs)) == 2


@pytest.mark.django_db(transaction=True)
@mock.patch("vulnerabilities.utils.fetch_response")
def test_improver_end_to_end(mock_response):
    response = os.path.join(TEST_DATA, "CVE-2023-35866.json")
    mock_response.return_value = response
    adv = Advisory.objects.create(
        unique_content_id="test-unique-content-id",
        url="https://test.url/",
        created_by=NVDImporterPipeline.pipeline_id,
        summary="1",
        date_collected=datetime.now(),
    )
    aliases = get_or_create_aliases(aliases=["CVE-2023-35866"])
    adv.aliases.add(*aliases)
    v1 = Vulnerability.objects.create(summary="test")
    associate_vulnerability_with_aliases(aliases=aliases, vulnerability=v1)
    VulnerabilityStatusImprover().get_inferences(advisory_data=adv.to_advisory_data())
    v1 = Vulnerability.objects.get(summary="test")
    assert v1.status == VulnerabilityStatusType.DISPUTED
