"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetStaging = void 0;
const crypto = require("crypto");
const os = require("os");
const path = require("path");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs-extra");
const minimatch = require("minimatch");
const assets_1 = require("./assets");
const fs_1 = require("./fs");
const cache_1 = require("./private/cache");
const stack_1 = require("./stack");
const stage_1 = require("./stage");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("./construct-compat");
/**
 * Stages a file or directory from a location on the file system into a staging directory.
 *
 * This is controlled by the context key 'aws:cdk:asset-staging' and enabled
 * by the CLI by default in order to ensure that when the CDK app exists, all
 * assets are available for deployment. Otherwise, if an app references assets
 * in temporary locations, those will not be available when it exists (see
 * https://github.com/aws/aws-cdk/issues/1716).
 *
 * The `stagedPath` property is a stringified token that represents the location
 * of the file or directory after staging. It will be resolved only during the
 * "prepare" stage and may be either the original path or the staged path
 * depending on the context setting.
 *
 * The file/directory are staged based on their content hash (fingerprint). This
 * means that only if content was changed, copy will happen.
 */
class AssetStaging extends construct_compat_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.sourcePath = path.resolve(props.sourcePath);
        this.fingerprintOptions = props;
        const outdir = (_a = stage_1.Stage.of(this)) === null || _a === void 0 ? void 0 : _a.assetOutdir;
        if (!outdir) {
            throw new Error('unable to determine cloud assembly asset output directory. Assets must be defined indirectly within a "Stage" or an "App" scope');
        }
        this.assetOutdir = outdir;
        // Determine the hash type based on the props as props.assetHashType is
        // optional from a caller perspective.
        this.customSourceFingerprint = props.assetHash;
        this.hashType = determineHashType(props.assetHashType, this.customSourceFingerprint);
        // Decide what we're going to do, without actually doing it yet
        let stageThisAsset;
        let skip = false;
        if (props.bundling) {
            // Check if we actually have to bundle for this stack
            const bundlingStacks = (_b = this.node.tryGetContext(cxapi.BUNDLING_STACKS)) !== null && _b !== void 0 ? _b : ['*'];
            skip = !bundlingStacks.find(pattern => minimatch(stack_1.Stack.of(this).stackName, pattern));
            const bundling = props.bundling;
            stageThisAsset = () => this.stageByBundling(bundling, skip);
        }
        else {
            stageThisAsset = () => this.stageByCopying();
        }
        // Calculate a cache key from the props. This way we can check if we already
        // staged this asset and reuse the result (e.g. the same asset with the same
        // configuration is used in multiple stacks). In this case we can completely
        // skip file system and bundling operations.
        //
        // The output directory and whether this asset is skipped or not should also be
        // part of the cache key to make sure we don't accidentally return the wrong
        // staged asset from the cache.
        this.cacheKey = calculateCacheKey({
            outdir: this.assetOutdir,
            sourcePath: path.resolve(props.sourcePath),
            bundling: props.bundling,
            assetHashType: this.hashType,
            customFingerprint: this.customSourceFingerprint,
            extraHash: props.extraHash,
            exclude: props.exclude,
            ignoreMode: props.ignoreMode,
            skip,
        });
        const staged = AssetStaging.assetCache.obtain(this.cacheKey, stageThisAsset);
        this.stagedPath = staged.stagedPath;
        this.assetHash = staged.assetHash;
    }
    /**
     * Clears the asset hash cache.
     */
    static clearAssetHashCache() {
        this.assetCache.clear();
    }
    /**
     * (deprecated) A cryptographic hash of the asset.
     *
     * @deprecated see `assetHash`.
     */
    get sourceHash() {
        return this.assetHash;
    }
    /**
     * Return the path to the staged asset, relative to the Cloud Assembly (manifest) directory of the given stack.
     *
     * Only returns a relative path if the asset was staged, returns an absolute path if
     * it was not staged.
     *
     * A bundled asset might end up in the outDir and still not count as
     * "staged"; if asset staging is disabled we're technically expected to
     * reference source directories, but we don't have a source directory for the
     * bundled outputs (as the bundle output is written to a temporary
     * directory). Nevertheless, we will still return an absolute path.
     *
     * A non-obvious directory layout may look like this:
     *
     * ```
     *    CLOUD ASSEMBLY ROOT
     *      +-- asset.12345abcdef/
     *      +-- assembly-Stage
     *            +-- MyStack.template.json
     *            +-- MyStack.assets.json <- will contain { "path": "../asset.12345abcdef" }
     * ```
     */
    relativeStagedPath(stack) {
        var _a;
        const asmManifestDir = (_a = stage_1.Stage.of(stack)) === null || _a === void 0 ? void 0 : _a.outdir;
        if (!asmManifestDir) {
            return this.stagedPath;
        }
        const isOutsideAssetDir = path.relative(this.assetOutdir, this.stagedPath).startsWith('..');
        if (isOutsideAssetDir || this.stagingDisabled) {
            return this.stagedPath;
        }
        return path.relative(asmManifestDir, this.stagedPath);
    }
    /**
     * Stage the source to the target by copying
     *
     * Optionally skip if staging is disabled, in which case we pretend we did something but we don't really.
     */
    stageByCopying() {
        const assetHash = this.calculateHash(this.hashType);
        const stagedPath = this.stagingDisabled
            ? this.sourcePath
            : path.resolve(this.assetOutdir, renderAssetFilename(assetHash, path.extname(this.sourcePath)));
        this.stageAsset(this.sourcePath, stagedPath, 'copy');
        return { assetHash, stagedPath };
    }
    /**
     * Stage the source to the target by bundling
     *
     * Optionally skip, in which case we pretend we did something but we don't really.
     */
    stageByBundling(bundling, skip) {
        if (skip) {
            // We should have bundled, but didn't to save time. Still pretend to have a hash,
            // but always base it on sources.
            return {
                assetHash: this.calculateHash(assets_1.AssetHashType.SOURCE),
                stagedPath: this.sourcePath,
            };
        }
        // Try to calculate assetHash beforehand (if we can)
        let assetHash = this.hashType === assets_1.AssetHashType.SOURCE || this.hashType === assets_1.AssetHashType.CUSTOM
            ? this.calculateHash(this.hashType, bundling)
            : undefined;
        const bundleDir = this.determineBundleDir(this.assetOutdir, assetHash);
        this.bundle(bundling, bundleDir);
        // Calculate assetHash afterwards if we still must
        assetHash = assetHash !== null && assetHash !== void 0 ? assetHash : this.calculateHash(this.hashType, bundling, bundleDir);
        const stagedPath = path.resolve(this.assetOutdir, renderAssetFilename(assetHash));
        this.stageAsset(bundleDir, stagedPath, 'move');
        return { assetHash, stagedPath };
    }
    /**
     * Whether staging has been disabled
     */
    get stagingDisabled() {
        return !!this.node.tryGetContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT);
    }
    /**
     * Copies or moves the files from sourcePath to targetPath.
     *
     * Moving implies the source directory is temporary and can be trashed.
     *
     * Will not do anything if source and target are the same.
     */
    stageAsset(sourcePath, targetPath, style) {
        // Is the work already done?
        const isAlreadyStaged = fs.existsSync(targetPath);
        if (isAlreadyStaged) {
            if (style === 'move' && sourcePath !== targetPath) {
                fs.removeSync(sourcePath);
            }
            return;
        }
        // Moving can be done quickly
        if (style == 'move') {
            fs.renameSync(sourcePath, targetPath);
            return;
        }
        // Copy file/directory to staging directory
        const stat = fs.statSync(sourcePath);
        if (stat.isFile()) {
            fs.copyFileSync(sourcePath, targetPath);
        }
        else if (stat.isDirectory()) {
            fs.mkdirSync(targetPath);
            fs_1.FileSystem.copyDirectory(sourcePath, targetPath, this.fingerprintOptions);
        }
        else {
            throw new Error(`Unknown file type: ${sourcePath}`);
        }
    }
    /**
     * Determine the directory where we're going to write the bundling output
     *
     * This is the target directory where we're going to write the staged output
     * files if we can (if the hash is fully known), or a temporary directory
     * otherwise.
     */
    determineBundleDir(outdir, sourceHash) {
        if (sourceHash) {
            return path.resolve(outdir, renderAssetFilename(sourceHash));
        }
        // When the asset hash isn't known in advance, bundler outputs to an
        // intermediate directory named after the asset's cache key
        return path.resolve(outdir, `bundling-temp-${this.cacheKey}`);
    }
    /**
     * Bundles an asset to the given directory
     *
     * If the given directory already exists, assume that everything's already
     * in order and don't do anything.
     *
     * @param options Bundling options
     * @param bundleDir Where to create the bundle directory
     * @returns The fully resolved bundle output directory.
     */
    bundle(options, bundleDir) {
        var _a, _b, _c;
        if (fs.existsSync(bundleDir)) {
            return;
        }
        fs.ensureDirSync(bundleDir);
        // Chmod the bundleDir to full access.
        fs.chmodSync(bundleDir, 0o777);
        let user;
        if (options.user) {
            user = options.user;
        }
        else { // Default to current user
            const userInfo = os.userInfo();
            user = userInfo.uid !== -1 // uid is -1 on Windows
                ? `${userInfo.uid}:${userInfo.gid}`
                : '1000:1000';
        }
        // Always mount input and output dir
        const volumes = [
            {
                hostPath: this.sourcePath,
                containerPath: AssetStaging.BUNDLING_INPUT_DIR,
            },
            {
                hostPath: bundleDir,
                containerPath: AssetStaging.BUNDLING_OUTPUT_DIR,
            },
            ...(_a = options.volumes) !== null && _a !== void 0 ? _a : [],
        ];
        let localBundling;
        try {
            process.stderr.write(`Bundling asset ${this.node.path}...\n`);
            localBundling = (_b = options.local) === null || _b === void 0 ? void 0 : _b.tryBundle(bundleDir, options);
            if (!localBundling) {
                options.image.run({
                    command: options.command,
                    user,
                    volumes,
                    environment: options.environment,
                    workingDirectory: (_c = options.workingDirectory) !== null && _c !== void 0 ? _c : AssetStaging.BUNDLING_INPUT_DIR,
                });
            }
        }
        catch (err) {
            // When bundling fails, keep the bundle output for diagnosability, but
            // rename it out of the way so that the next run doesn't assume it has a
            // valid bundleDir.
            const bundleErrorDir = bundleDir + '-error';
            if (fs.existsSync(bundleErrorDir)) {
                // Remove the last bundleErrorDir.
                fs.removeSync(bundleErrorDir);
            }
            fs.renameSync(bundleDir, bundleErrorDir);
            throw new Error(`Failed to bundle asset ${this.node.path}, bundle output is located at ${bundleErrorDir}: ${err}`);
        }
        if (fs_1.FileSystem.isEmpty(bundleDir)) {
            const outputDir = localBundling ? bundleDir : AssetStaging.BUNDLING_OUTPUT_DIR;
            throw new Error(`Bundling did not produce any output. Check that content is written to ${outputDir}.`);
        }
    }
    calculateHash(hashType, bundling, outputDir) {
        var _a;
        // When bundling a CUSTOM or SOURCE asset hash type, we want the hash to include
        // the bundling configuration. We handle CUSTOM and bundled SOURCE hash types
        // as a special case to preserve existing user asset hashes in all other cases.
        if (hashType == assets_1.AssetHashType.CUSTOM || (hashType == assets_1.AssetHashType.SOURCE && bundling)) {
            const hash = crypto.createHash('sha256');
            // if asset hash is provided by user, use it, otherwise fingerprint the source.
            hash.update((_a = this.customSourceFingerprint) !== null && _a !== void 0 ? _a : fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions));
            // If we're bundling an asset, include the bundling configuration in the hash
            if (bundling) {
                hash.update(JSON.stringify(bundling));
            }
            return hash.digest('hex');
        }
        switch (hashType) {
            case assets_1.AssetHashType.SOURCE:
                return fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions);
            case assets_1.AssetHashType.BUNDLE:
            case assets_1.AssetHashType.OUTPUT:
                if (!outputDir) {
                    throw new Error(`Cannot use \`${hashType}\` hash type when \`bundling\` is not specified.`);
                }
                return fs_1.FileSystem.fingerprint(outputDir, this.fingerprintOptions);
            default:
                throw new Error('Unknown asset hash type.');
        }
    }
}
exports.AssetStaging = AssetStaging;
/**
 * (experimental) The directory inside the bundling container into which the asset sources will be mounted.
 *
 * @experimental
 */
AssetStaging.BUNDLING_INPUT_DIR = '/asset-input';
/**
 * (experimental) The directory inside the bundling container into which the bundled output should be written.
 *
 * @experimental
 */
AssetStaging.BUNDLING_OUTPUT_DIR = '/asset-output';
/**
 * Cache of asset hashes based on asset configuration to avoid repeated file
 * system and bundling operations.
 */
AssetStaging.assetCache = new cache_1.Cache();
function renderAssetFilename(assetHash, extension = '') {
    return `asset.${assetHash}${extension}`;
}
/**
 * Determines the hash type from user-given prop values.
 *
 * @param assetHashType Asset hash type construct prop
 * @param customSourceFingerprint Asset hash seed given in the construct props
 */
function determineHashType(assetHashType, customSourceFingerprint) {
    const hashType = customSourceFingerprint
        ? (assetHashType !== null && assetHashType !== void 0 ? assetHashType : assets_1.AssetHashType.CUSTOM)
        : (assetHashType !== null && assetHashType !== void 0 ? assetHashType : assets_1.AssetHashType.SOURCE);
    if (customSourceFingerprint && hashType !== assets_1.AssetHashType.CUSTOM) {
        throw new Error(`Cannot specify \`${assetHashType}\` for \`assetHashType\` when \`assetHash\` is specified. Use \`CUSTOM\` or leave \`undefined\`.`);
    }
    if (hashType === assets_1.AssetHashType.CUSTOM && !customSourceFingerprint) {
        throw new Error('`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`.');
    }
    return hashType;
}
/**
 * Calculates a cache key from the props. Normalize by sorting keys.
 */
function calculateCacheKey(props) {
    return crypto.createHash('sha256')
        .update(JSON.stringify(sortObject(props)))
        .digest('hex');
}
/**
 * Recursively sort object keys
 */
function sortObject(object) {
    if (typeof object !== 'object' || object instanceof Array) {
        return object;
    }
    const ret = {};
    for (const key of Object.keys(object).sort()) {
        ret[key] = sortObject(object[key]);
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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