# Lessons Learned: Phase 1 Days 1-3

**Date**: 2025-11-06
**Context**: Foundation implementation (scaffolding, config, encryption)
**Related**: [Devlog](../devlog/2025-11-06-days-1-3-implementation.md)

## Summary

Days 1-3 focused on establishing a solid foundation: project structure, configuration management, and credential encryption. We built 60 tests (100% passing) and implemented professional-grade security from day one.

---

## What Went Well ✅

### 1. Test-First Development

**What Happened**: Wrote tests alongside (or slightly after) implementation code

**Why It Worked**:
- Caught bugs immediately (e.g., empty string handling in encryption)
- Tests document expected behavior
- High confidence in refactoring
- 100% test pass rate throughout

**Example**:
```python
# This test caught a bug where empty strings weren't handled
def test_encrypt_empty_string(self, tmp_path: Path) -> None:
    encryptor = CredentialEncryptor(tmp_path / ".keyfile")
    encrypted = encryptor.encrypt("")
    assert encrypted == ""  # Early return, don't try to encrypt nothing
```

**Takeaway**: Keep writing tests. The time investment pays off immediately.

---

### 2. Pydantic for Configuration Validation

**What Happened**: Used Pydantic models for all configuration

**Why It Worked**:
- Validation happens automatically at config load time
- Type safety enforced at runtime
- Clear error messages when config is invalid
- Serialization/deserialization handled by Pydantic

**Example**:
```python
# This raises ValidationError with helpful message
config = GlobalConfig(log_level="INVALID")
# ValidationError: Input should be 'DEBUG', 'INFO', 'WARNING' or 'ERROR'
```

**Takeaway**: Pydantic is perfect for configuration. The upfront model definition prevents entire classes of bugs.

---

### 3. Security by Default

**What Happened**: Implemented credential encryption in Day 3 (early in project)

**Why It Worked**:
- Easier to add security early than retrofit later
- ConfigManager design already accounted for encryption
- Tests verify security properties from the start

**Contrast**: If we'd used plaintext initially:
- Would need to migrate existing configs
- Users might have committed plaintext credentials
- Breaking change to switch to encryption later

**Takeaway**: Build security in from the start. It's an architectural decision, not a feature.

---

### 4. Clear Module Boundaries

**What Happened**: Separated config, feeds, and utils into distinct modules

**Why It Worked**:
- Each module has single responsibility
- Easy to test in isolation
- Clear import structure
- New contributors know where to add code

**Example**:
```
config/  → Configuration management
feeds/   → RSS parsing and feed operations
utils/   → Shared utilities (paths, errors)
```

**Takeaway**: Organize by domain, not by file type. `models.py` becomes `config/schema.py` + `feeds/models.py`.

---

### 5. XDG Compliance via platformdirs

**What Happened**: Used `platformdirs` library for config paths

**Why It Worked**:
- Cross-platform (Linux, macOS) without manual path construction
- Respects XDG environment variables
- Well-tested library (used by many projects)

**Alternative Considered**:
```python
# Manual approach (don't do this)
config_dir = Path.home() / ".config" / "inkwell"
```

**Problems with Manual**:
- Doesn't respect $XDG_CONFIG_HOME
- Hardcoded Linux conventions
- Need platform detection for macOS

**Takeaway**: Use `platformdirs` for any CLI tool that needs config storage.

---

## What Could Be Better 🔧

### 1. Hatchling Debugging Time

**What Happened**: Spent ~20 minutes debugging Hatchling package discovery

**What We Learned**:
- Should have tested installation in first 5 minutes
- Build system should be verified before writing code
- Setuptools is boring but works

**Better Approach**:
```bash
# Day 1, Minute 1:
$ echo "[build-system]..." > pyproject.toml
$ pip install -e .
# Test immediately, before writing code
```

**Takeaway**: Test infrastructure (build, install, import) before writing application code.

---

### 2. pytest Configuration Confusion

**What Happened**: Had to debug why `pytest` command didn't find modules

**Root Cause**:
- System `pytest` wasn't using our installed package
- Needed `python3 -m pytest` instead
- Coverage plugin misconfigured

**Solution**:
```makefile
# Makefile now uses explicit python3 -m
test:
    python3 -m pytest
```

**Takeaway**: Document the "right way" to run tests in Makefile. Reduce mental overhead for contributors.

---

### 3. Dependency Installation Friction

**What Happened**: `sgmllib3k` (feedparser dependency) failed to build with default pip

**Root Cause**:
- Legacy setup.py with Debian-specific quirks
- System pip had compatibility issues

**Solution**:
```bash
pip install --use-pep517 sgmllib3k
```

**Takeaway**: Container environments have quirks. Document workarounds, but don't over-optimize for them.

---

## Technical Insights 💡

### 1. Fernet is Perfect for This Use Case

**Observation**: Many projects over-engineer credential storage

**What We Did**: Simple symmetric encryption with Fernet

**Why It's Right**:
- Threat model: casual exposure, not sophisticated attacks
- Single-user tool on trusted machine
- System keyring is overkill (can add later if needed)
- Fernet is battle-tested (Django, Twisted use it)

**When to Upgrade**:
- If handling third-party secrets (not just user's own feeds)
- If targeting enterprise environments
- If users explicitly request hardware-backed security

**Takeaway**: Match security to threat model. Don't cargo-cult "maximum security" practices.

---

### 2. Permission Checking is Critical

**Observation**: File permissions are easy to mess up

**What We Did**:
```python
def _validate_key_permissions(self) -> None:
    mode = stat.S_IMODE(self.key_path.stat().st_mode)
    if mode & (stat.S_IRGRP | stat.S_IROTH | ...):
        raise EncryptionError(f"Insecure permissions. Run: chmod 600 {self.key_path}")
```

**Why It Matters**:
- User might accidentally `chmod 644` the key file
- World-readable key file defeats encryption
- Early validation with clear error guides user to fix

**Takeaway**: Validate security assumptions with helpful error messages. Don't silently accept insecure configurations.

---

### 3. Type Hints Catch Bugs Before Runtime

**Example**:
```python
# Mypy catches this before we run code
def add_feed(self, name: str, feed_config: FeedConfig) -> None:
    pass

add_feed("test", "not-a-feed-config")  # Type error!
```

**Value**:
- IDE autocomplete works perfectly
- Refactoring is safe (mypy catches broken imports)
- New contributors understand API from type signatures

**Investment**:
- ~5% more typing initially
- Pays for itself after first refactoring

**Takeaway**: Type hints are documentation that the computer verifies. Always use them.

---

## Process Insights 🔄

### 1. Commit Frequently with Clear Messages

**What We Did**: 3 commits for Days 1-3, one per day

**Commit Structure**:
```
feat: Day X - Brief summary

Implemented:
- Feature 1
- Feature 2

Tests:
- X tests added
- All passing

Next: Day X+1
```

**Why It Works**:
- Easy to understand what changed
- Bisectable history (each commit is complete)
- ADRs reference specific commits

**Takeaway**: Commit often, but make each commit tell a story.

---

### 2. Documentation as You Go

**What We Did**: Created devlog and ADRs alongside code

**Benefits**:
- Decisions documented while fresh in mind
- Future contributors understand "why"
- Easy to onboard new team members

**When to Document**:
- ✅ Significant technical decisions → ADR
- ✅ Implementation progress → Devlog
- ✅ Surprising discoveries → Lessons Learned
- ❌ Routine code → Just write good docstrings

**Takeaway**: Document the "why" and the "what went wrong". The "how" is in the code.

---

### 3. Test Coverage as a Quality Signal

**Observation**: 60 tests for ~1000 lines of code = 1.2:1 ratio

**What This Indicates**:
- High confidence in code behavior
- Easy to refactor without fear
- New contributors can understand code through tests

**When Tests Are Less Valuable**:
- Testing trivial getters/setters
- Testing framework code (e.g., Pydantic internals)
- Tests that just restate the implementation

**Takeaway**: Write tests that verify behavior, not implementation details.

---

## Patterns to Repeat 🔁

### 1. Builder Pattern for Tests

**Pattern**:
```python
@pytest.fixture
def sample_config_dict() -> dict:
    return {"version": "1", "log_level": "INFO", ...}

def test_config(sample_config_dict):
    config = GlobalConfig(**sample_config_dict)
    # Test logic...
```

**Benefits**:
- DRY (Don't Repeat Yourself) test data
- Easy to modify fixture, all tests update
- Clear test intent (only override what's relevant)

---

### 2. Transparent Abstraction Layers

**Pattern**:
```python
# ConfigManager hides encryption from callers
manager.add_feed("name", FeedConfig(...))  # ← Plaintext goes in

# Internal: encryption happens in save_feeds()
# External: caller doesn't know/care

feed = manager.get_feed("name")  # ← Plaintext comes out
# Internal: decryption happens in load_feeds()
```

**Benefits**:
- Security layer is invisible to users
- Easy to change encryption algorithm later
- Rest of codebase stays simple

---

### 3. Fail Fast with Helpful Errors

**Pattern**:
```python
if feed_name in self.feeds:
    raise DuplicateFeedError(
        f"Feed '{feed_name}' already exists. Use update to modify it."
    )
```

**Not This**:
```python
if feed_name in self.feeds:
    raise Exception("Duplicate feed")  # ❌ Vague, no guidance
```

**Benefits**:
- Users know exactly what went wrong
- Error message suggests how to fix
- Custom exceptions can be caught by callers

---

## Patterns to Avoid 🚫

### 1. Over-Engineering Build System

**Anti-Pattern**: Spending hours debugging Hatchling

**Better**: Use boring, reliable tools (setuptools)

**When to Use New Tools**:
- When they solve a real problem you have
- When the benefits outweigh the learning curve
- Not just because they're new

---

### 2. Plaintext Credentials

**Anti-Pattern**: Storing credentials in plaintext config

**Why It's Bad**:
- Accidental exposure (screenshots, git commits)
- Bad practice in 2025
- Hard to retrofit encryption later

**Better**: Encrypt from day one, even if "it's just a dev tool"

---

### 3. Manual Path Construction

**Anti-Pattern**:
```python
config_dir = Path.home() / ".config" / "inkwell"  # ❌
```

**Better**:
```python
from platformdirs import user_config_dir
config_dir = Path(user_config_dir("inkwell"))  # ✅
```

**Why**: Respects XDG, works cross-platform, handles edge cases

---

## Questions for Future

1. **System Keyring**: Should we add optional system keyring support in v0.2?
   - User feedback will guide this
   - Easy to add without breaking changes

2. **Key Rotation**: How often should users rotate encryption keys?
   - Probably never for a dev tool
   - Can implement if users request it

3. **Config Migrations**: What's our strategy for config version upgrades?
   - Will tackle this when we actually need to migrate configs
   - Version "1" is current, no migrations yet

---

## Recommended Reading

Based on challenges we faced:

- [Cryptography Docs - Fernet](https://cryptography.io/en/latest/fernet/)
- [PEP 621 - pyproject.toml metadata](https://peps.python.org/pep-0621/)
- [platformdirs Documentation](https://platformdirs.readthedocs.io/)
- [Pydantic Validation](https://docs.pydantic.dev/latest/concepts/validators/)

---

## Action Items

For Days 4-7:

1. ✅ Continue test-first development
2. ✅ Keep documentation updated (devlogs, ADRs)
3. ✅ Commit frequently with clear messages
4. ⚠️ Watch for over-engineering (keep it simple)
5. ⚠️ Test installation early when changing build config

---

## Conclusion

Days 1-3 established a solid foundation with security, testing, and code quality baked in. The decision to use Fernet encryption, Pydantic validation, and test-first development has already paid dividends.

Key theme: **Professional-grade doesn't mean over-engineered**. We chose simple, battle-tested tools (setuptools, Fernet, Pydantic) that solve real problems without adding complexity.

Next: Day 4 will tackle RSS parsing and feed validation, building on this foundation.
