"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NotificationsResourceHandler = exports.NotificationsResourceHandlerProps = void 0;
const fs = require("fs");
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
class NotificationsResourceHandlerProps {
}
exports.NotificationsResourceHandlerProps = NotificationsResourceHandlerProps;
/**
 * A Lambda-based custom resource handler that provisions S3 bucket
 * notifications for a bucket.
 *
 * The resource property schema is:
 *
 * {
 *   BucketName: string, NotificationConfiguration: { see
 *   PutBucketNotificationConfiguration }
 * }
 *
 * For 'Delete' operations, we send an empty NotificationConfiguration as
 * required. We propagate errors and results as-is.
 *
 * Sadly, we can't use @aws-cdk/aws-lambda as it will introduce a dependency
 * cycle, so this uses raw `cdk.Resource`s.
 */
class NotificationsResourceHandler extends core_1.Construct {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.role = props.role ?? new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        });
        this.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['s3:PutBucketNotification'],
            resources: ['*'],
        }));
        const resourceType = 'AWS::Lambda::Function';
        class InLineLambda extends cdk.CfnResource {
            constructor() {
                super(...arguments);
                this.tags = new cdk.TagManager(cdk.TagType.STANDARD, resourceType);
            }
            renderProperties(properties) {
                properties.Tags = cdk.listMapper(cdk.cfnTagToCloudFormation)(this.tags.renderTags());
                delete properties.tags;
                return properties;
            }
        }
        const handlerSource = fs.readFileSync(path.join(__dirname, 'lambda/index.py'), 'utf8');
        // Removing lines that starts with '#' (comment lines) in order to fit the 4096 limit
        const handlerSourceWithoutComments = handlerSource.replace(/^ *#.*\n?/gm, '');
        if (handlerSourceWithoutComments.length > 4096) {
            throw new Error(`Source of Notifications Resource Handler is too large (${handlerSourceWithoutComments.length} > 4096)`);
        }
        const resource = new InLineLambda(this, 'Resource', {
            type: resourceType,
            properties: {
                Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)',
                Code: { ZipFile: handlerSourceWithoutComments },
                Handler: 'index.handler',
                Role: this.role.roleArn,
                Runtime: 'python3.7',
                Timeout: 300,
            },
        });
        resource.node.addDependency(this.role);
        this.functionArn = resource.getAtt('Arn').toString();
    }
    /**
     * Defines a stack-singleton lambda function with the logic for a CloudFormation custom
     * resource that provisions bucket notification configuration for a bucket.
     *
     * @returns The ARN of the custom resource lambda function.
     */
    static singleton(context, props = {}) {
        const root = cdk.Stack.of(context);
        // well-known logical id to ensure stack singletonity
        const logicalId = 'BucketNotificationsHandler050a0587b7544547bf325f094a3db834';
        let lambda = root.node.tryFindChild(logicalId);
        if (!lambda) {
            lambda = new NotificationsResourceHandler(root, logicalId, props);
        }
        return lambda;
    }
    addToRolePolicy(statement) {
        this.role.addToPrincipalPolicy(statement);
    }
}
exports.NotificationsResourceHandler = NotificationsResourceHandler;
//# sourceMappingURL=data:application/json;base64,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