"""Configuration discovery service."""

from __future__ import annotations

import json
from pathlib import Path
from typing import TYPE_CHECKING

from lazyopencode.models.customization import (
    ConfigLevel,
    Customization,
    CustomizationType,
)
from lazyopencode.services.gitignore_filter import GitignoreFilter
from lazyopencode.services.parsers import read_file_safe, strip_jsonc_comments
from lazyopencode.services.parsers.agent import AgentParser
from lazyopencode.services.parsers.command import CommandParser
from lazyopencode.services.parsers.mcp import MCPParser
from lazyopencode.services.parsers.plugin import PluginParser
from lazyopencode.services.parsers.rules import RulesParser
from lazyopencode.services.parsers.skill import SkillParser
from lazyopencode.services.parsers.tool import ToolParser

if TYPE_CHECKING:
    from lazyopencode.services.claude_code.discovery import ClaudeCodeDiscoveryService


class ConfigDiscoveryService:
    """Discovers OpenCode customizations from filesystem."""

    def __init__(
        self,
        project_root: Path | None = None,
        global_config_path: Path | None = None,
        enable_claude_code: bool = False,
    ) -> None:
        """
        Initialize discovery service.

        Args:
            project_root: Project root directory (defaults to cwd)
            global_config_path: Global config path (defaults to ~/.config/opencode)
            enable_claude_code: Enable Claude Code customization discovery
        """
        self.project_root = project_root or Path.cwd()
        self.global_config_path = global_config_path or (
            Path.home() / ".config" / "opencode"
        )
        self._enable_claude_code = enable_claude_code
        self._gitignore_filter = GitignoreFilter(project_root=self.project_root)
        self._parsers = {
            CustomizationType.COMMAND: CommandParser(),
            CustomizationType.AGENT: AgentParser(),
            CustomizationType.SKILL: SkillParser(
                gitignore_filter=self._gitignore_filter
            ),
            CustomizationType.RULES: RulesParser(),
            CustomizationType.MCP: MCPParser(),
            CustomizationType.TOOL: ToolParser(),
            CustomizationType.PLUGIN: PluginParser(),
        }
        self._cache: list[Customization] | None = None
        self._claude_code_discovery: ClaudeCodeDiscoveryService | None = None
        if enable_claude_code:
            from lazyopencode.services.claude_code.discovery import (
                ClaudeCodeDiscoveryService,
            )

            self._claude_code_discovery = ClaudeCodeDiscoveryService(self.project_root)

    @property
    def project_config_path(self) -> Path:
        """Path to project's .opencode directory."""
        return self.project_root / ".opencode"

    def discover_all(self) -> list[Customization]:
        """
        Discover all customizations from global and project levels.

        Returns:
            List of all discovered customizations (de-duplicated by path)
        """
        if self._cache is not None:
            return self._cache

        customizations: list[Customization] = []

        # Discover from global config
        customizations.extend(self._discover_level(ConfigLevel.GLOBAL))

        # Discover from project config
        customizations.extend(self._discover_level(ConfigLevel.PROJECT))

        # Discover Claude Code customizations (if enabled)
        if self._claude_code_discovery:
            customizations.extend(self._claude_code_discovery.discover_all())

        # De-duplicate by (type, name, level, source, resolved_path) tuple
        # This avoids removing items that share a source file (like multiple
        # inline definitions from the same opencode.json)
        seen: set[tuple] = set()
        unique: list[Customization] = []
        for c in customizations:
            # Create a unique key for this customization
            # Include source to allow same-named items from different sources
            key = (c.type, c.name, c.level, c.source, str(c.path.resolve()))
            if key not in seen:
                seen.add(key)
                unique.append(c)

        self._cache = unique
        return unique

    def _discover_level(self, level: ConfigLevel) -> list[Customization]:
        """Discover customizations at a specific level."""
        base_path = (
            self.global_config_path
            if level == ConfigLevel.GLOBAL
            else self.project_config_path
        )
        customizations: list[Customization] = []

        # Discover commands
        customizations.extend(self._discover_commands(base_path, level))
        customizations.extend(self._discover_inline_commands(level))

        # Discover agents
        customizations.extend(self._discover_agents(base_path, level))
        customizations.extend(self._discover_inline_agents(level))

        # Discover skills
        customizations.extend(self._discover_skills(base_path, level))

        # Discover rules (AGENTS.md)
        customizations.extend(self._discover_rules(level))

        # Discover instruction files from opencode.json
        customizations.extend(self._discover_instructions(level))

        # Discover MCPs from opencode.json
        customizations.extend(self._discover_mcps(level))

        # Discover tools
        customizations.extend(self._discover_tools(base_path, level))

        # Discover plugins
        customizations.extend(self._discover_plugins(base_path, level))

        return customizations

    def _discover_commands(
        self, base_path: Path, level: ConfigLevel
    ) -> list[Customization]:
        """Discover command customizations."""
        commands_path = base_path / "command"
        if not commands_path.exists():
            return []

        customizations = []
        parser = self._parsers[CustomizationType.COMMAND]

        for md_file in commands_path.glob("*.md"):
            if parser.can_parse(md_file):
                customizations.append(parser.parse(md_file, level))

        return customizations

    def _discover_inline_commands(self, level: ConfigLevel) -> list[Customization]:
        """Discover inline commands from opencode.json."""
        if level == ConfigLevel.GLOBAL:
            config_path = self.global_config_path / "opencode.json"
        else:
            config_path = self.project_root / "opencode.json"

        parser = self._parsers[CustomizationType.COMMAND]
        if isinstance(parser, CommandParser) and config_path.exists():
            return parser.parse_inline_commands(config_path, level)

        return []

    def _discover_agents(
        self, base_path: Path, level: ConfigLevel
    ) -> list[Customization]:
        """Discover agent customizations."""
        agents_path = base_path / "agent"
        if not agents_path.exists():
            return []

        customizations = []
        parser = self._parsers[CustomizationType.AGENT]

        for md_file in agents_path.glob("*.md"):
            if parser.can_parse(md_file):
                customizations.append(parser.parse(md_file, level))

        return customizations

    def _discover_inline_agents(self, level: ConfigLevel) -> list[Customization]:
        """Discover inline agents from opencode.json."""
        if level == ConfigLevel.GLOBAL:
            config_path = self.global_config_path / "opencode.json"
        else:
            config_path = self.project_root / "opencode.json"

        parser = self._parsers[CustomizationType.AGENT]
        if isinstance(parser, AgentParser) and config_path.exists():
            return parser.parse_inline_agents(config_path, level)

        return []

    def _discover_skills(
        self, base_path: Path, level: ConfigLevel
    ) -> list[Customization]:
        """Discover skill customizations from .opencode/skill/."""
        customizations = []

        # Discover from standard OpenCode path
        skills_path = base_path / "skill"
        customizations.extend(self._discover_skills_from_path(skills_path, level))

        # Also discover from Claude-compatible path at project level
        if level == ConfigLevel.PROJECT:
            claude_skills_path = self.project_root / ".claude" / "skills"
            customizations.extend(
                self._discover_skills_from_path(claude_skills_path, level)
            )

        return customizations

    def _discover_skills_from_path(
        self, skills_path: Path, level: ConfigLevel
    ) -> list[Customization]:
        """Discover skills from a specific directory path."""
        if not skills_path.exists():
            return []

        customizations = []
        parser = self._parsers[CustomizationType.SKILL]

        for skill_dir in skills_path.iterdir():
            if skill_dir.is_dir():
                skill_file = skill_dir / "SKILL.md"
                if parser.can_parse(skill_file):
                    customizations.append(parser.parse(skill_file, level))

        return customizations

    def _discover_rules(self, level: ConfigLevel) -> list[Customization]:
        """Discover AGENTS.md rules files."""
        customizations = []
        parser = self._parsers[CustomizationType.RULES]

        if level == ConfigLevel.GLOBAL:
            agents_md = self.global_config_path / "AGENTS.md"
        else:
            agents_md = self.project_root / "AGENTS.md"

        if parser.can_parse(agents_md):
            customizations.append(parser.parse(agents_md, level))

        return customizations

    def _discover_mcps(self, level: ConfigLevel) -> list[Customization]:
        """Discover MCP configurations from opencode.json."""
        if level == ConfigLevel.GLOBAL:
            config_path = self.global_config_path / "opencode.json"
        else:
            config_path = self.project_root / "opencode.json"

        parser = self._parsers[CustomizationType.MCP]
        # Cast to MCPParser to access parse_mcps
        if isinstance(parser, MCPParser) and parser.can_parse(config_path):
            return parser.parse_mcps(config_path, level)

        return []

    def _discover_tools(
        self, base_path: Path, level: ConfigLevel
    ) -> list[Customization]:
        """Discover tool customizations from .opencode/tool/."""
        tools_path = base_path / "tool"
        if not tools_path.exists():
            return []

        customizations = []
        parser = self._parsers[CustomizationType.TOOL]

        for tool_file in tools_path.iterdir():
            if tool_file.is_file() and parser.can_parse(tool_file):
                customizations.append(parser.parse(tool_file, level))

        return customizations

    def _discover_plugins(
        self, base_path: Path, level: ConfigLevel
    ) -> list[Customization]:
        """Discover plugin customizations from .opencode/plugin/."""
        plugins_path = base_path / "plugin"
        if not plugins_path.exists():
            return []

        customizations = []
        parser = self._parsers[CustomizationType.PLUGIN]

        for plugin_file in plugins_path.iterdir():
            if plugin_file.is_file() and parser.can_parse(plugin_file):
                customizations.append(parser.parse(plugin_file, level))

        return customizations

    def _discover_instructions(self, level: ConfigLevel) -> list[Customization]:
        """Discover instruction files from opencode.json instructions field.

        Args:
            level: Configuration level (GLOBAL or PROJECT)

        Returns:
            List of instruction file customizations
        """
        if level == ConfigLevel.GLOBAL:
            config_path = self.global_config_path / "opencode.json"
            base_dir = self.global_config_path
        else:
            config_path = self.project_root / "opencode.json"
            base_dir = self.project_root

        if not config_path.exists():
            return []

        # Parse opencode.json
        content, error = read_file_safe(config_path)
        if error or not content:
            return []

        try:
            clean_content = strip_jsonc_comments(content)
            config = json.loads(clean_content)
        except (json.JSONDecodeError, Exception):
            return []

        instructions = config.get("instructions", [])
        if not instructions:
            return []

        customizations = []
        parser = self._parsers[CustomizationType.RULES]

        for pattern in instructions:
            # Resolve glob pattern relative to base_dir
            matched_files = sorted(base_dir.glob(pattern))
            for matched_file in matched_files:
                if matched_file.is_file() and isinstance(parser, RulesParser):
                    customizations.append(
                        parser.parse_instruction(matched_file, base_dir, level)
                    )

        return customizations

    def refresh(self) -> None:
        """Clear cache and force re-discovery."""
        self._cache = None
        if self._claude_code_discovery:
            self._claude_code_discovery.refresh()

    def by_type(self, ctype: CustomizationType) -> list[Customization]:
        """Get customizations filtered by type."""
        return [c for c in self.discover_all() if c.type == ctype]

    def by_level(self, level: ConfigLevel) -> list[Customization]:
        """Get customizations filtered by level."""
        return [c for c in self.discover_all() if c.level == level]
