"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const iam = require("@aws-cdk/aws-iam");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test construct properties
// --------------------------------------------------------------
test('Test construct properties', () => {
    const stack = new core_1.Stack();
    const pattern = new lib_1.ApiGatewayToSageMakerEndpoint(stack, 'api-gateway-sagemakerendpoint', {
        endpointName: 'my-endpoint',
        resourcePath: '{my_param}',
        requestMappingTemplate: 'my-request-vtl-template'
    });
    expect(pattern.apiGateway !== null);
    expect(pattern.apiGatewayRole !== null);
    expect(pattern.apiGatewayCloudWatchRole !== null);
    expect(pattern.apiGatewayLogGroup !== null);
});
// --------------------------------------------------------------
// Test deployment w/ overwritten properties
// --------------------------------------------------------------
test('Test deployment w/ overwritten properties', () => {
    const stack = new core_1.Stack();
    const existingRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        description: 'existing role for SageMaker integration',
        inlinePolicies: {
            InvokePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        resources: [`arn:${core_1.Aws.PARTITION}:sagemaker:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:endpoint/my-endpoint`],
                        actions: ['sagemaker:InvokeEndpoint']
                    })]
            })
        }
    });
    new lib_1.ApiGatewayToSageMakerEndpoint(stack, 'api-gateway-sagemakerendpoint', {
        endpointName: 'my-endpoint',
        resourcePath: '{my_param}',
        requestMappingTemplate: 'my-request-vtl-template',
        apiGatewayProps: {
            restApiName: 'my-api',
            deployOptions: {
                methodOptions: {
                    '/*/*': {
                        throttlingRateLimit: 100,
                        throttlingBurstLimit: 25
                    }
                }
            }
        },
        apiGatewayExecutionRole: existingRole,
        resourceName: 'my-resource',
        responseMappingTemplate: 'my-response-vtl-template'
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Stage', {
        MethodSettings: [
            {
                DataTraceEnabled: false,
                HttpMethod: '*',
                LoggingLevel: 'INFO',
                ResourcePath: '/*'
            },
            {
                HttpMethod: '*',
                ResourcePath: '/*',
                ThrottlingRateLimit: 100,
                ThrottlingBurstLimit: 25
            }
        ]
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Resource', {
        PathPart: 'my-resource'
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'application/json': 'my-response-vtl-template',
                    },
                    StatusCode: '200'
                },
                {
                    StatusCode: '500',
                    SelectionPattern: '5\\d{2}'
                },
                {
                    StatusCode: '400',
                    SelectionPattern: '4\\d{2}'
                }
            ]
        },
        MethodResponses: [
            { StatusCode: '200' },
            { StatusCode: '500' },
            { StatusCode: '400' }
        ]
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        Description: 'existing role for SageMaker integration'
    });
});
//# sourceMappingURL=data:application/json;base64,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