# Copyright (c) 2022 PaddlePaddle Authors. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from __future__ import absolute_import
import logging
from .... import FastDeployModel, ModelFormat
from .... import c_lib_wrap as C


class FaceLandmark1000(FastDeployModel):
    def __init__(self,
                 model_file,
                 params_file="",
                 runtime_option=None,
                 model_format=ModelFormat.ONNX):
        """Load a face alignment model exported by FaceLandmark1000.

        :param model_file: (str)Path of model file, e.g ./FaceLandmark1000.onnx
        :param params_file: (str)Path of parameters file, if the model_fomat is ModelFormat.ONNX, this param will be ignored, can be set as empty string
        :param runtime_option: (fastdeploy.RuntimeOption)RuntimeOption for inference this model, if it's None, will use the default backend on CPU
        :param model_format: (fastdeploy.ModelForamt)Model format of the loaded model, default is ONNX
        """

        super(FaceLandmark1000, self).__init__(runtime_option)

        assert model_format == ModelFormat.ONNX, "FaceLandmark1000 only support model format of ModelFormat.ONNX now."
        self._model = C.vision.facealign.FaceLandmark1000(
            model_file, params_file, self._runtime_option, model_format)
        assert self.initialized, "FaceLandmark1000 initialize failed."

    def predict(self, input_image):
        """Detect an input image landmarks

        :param im: (numpy.ndarray)The input image data, 3-D array with layout HWC, BGR format
        :return: FaceAlignmentResult
        """

        return self._model.predict(input_image)

    @property
    def size(self):
        """
        Returns the preprocess image size, default (128, 128)
        """
        return self._model.size

    @size.setter
    def size(self, wh):
        """
        Set the preprocess image size, default (128, 128)
        """
        assert isinstance(wh, (list, tuple)),\
            "The value to set `size` must be type of tuple or list."
        assert len(wh) == 2,\
            "The value to set `size` must contatins 2 elements means [width, height], but now it contains {} elements.".format(
            len(wh))
        self._model.size = wh
