from importlib import resources
import re

def resource2path(res_uri):
    """Convenience function to convert a resource URI to a file path, using `importlib.resources`.
    
    Parameters:
        res_uri (string): Path in the format `res://package/resource`.
    """
    m = re.search(r'(?P<schema>[a-z][a-z0-9+.-]+):\/\/(?P<pkg>[a-zA-Z][a-zA-Z\.\-_]*)\/(?P<path>.*)', res_uri)
    if not m:
        raise ValueError('Invalid resource URI!')
    schema = m.group('schema')
    package_spec = m.group('pkg')
    resource_path = m.group('path')
    if schema != 'res':
        raise ValueError('Invalid URI schema! Must be res:// for this function!')
    return str(resources.files(package_spec).joinpath(resource_path))

def read_resource(res_uri):
    """Reads binary content from a resource URI-like string.

    The default mechanism is to return `importlib.resources.read_binary(package, resource)`.
    This behaviour can be overridden by the user just by redefining the 
    `witkets.core.helpers.read_resource` symbol itself.

    Parameters:
        res_uri (string): Path in the format `res://package/resource`.
    """
    try:
        package, resource = res_uri.replace('res://', '').split('/', 1)
    except:
        raise ValueError(f'Malformed resource: {res_uri}')
    return resources.read_binary(package, resource)


def read_asset(asset_path):
    """Read an asset from a file path or resource URI.
    
    The file may or not be prefixed with the protocol `file://`.
    Resources MUST explicitly indicates the protocol `res://`
    """
    if asset_path.startswith('res://'):
        return read_resource(asset_path)
    else:
        simple_path = asset_path.replace('file://', '')
        with open(simple_path, 'rb') as fd:
            contents = fd.read()
        return contents
    
def read_xbm_icon(icon_name):
    """Read and return the contents of a built-in XBM icon."""
    return resources.read_binary('witkets.data.xbm', icon_name)
