"""An image map set on a Canvas widget.

**Imperative:**

      >>> import tkinter as tk
      >>> import witkets as wtk
      >>>
      >>> def callback(name):
      >>>     print(f'Region {name} was clicked!')
      >>>
      >>> root = tk.Tk()
      >>> photo = tk.PhotoImage(file='test-imagemap.png')
      >>> map = ImageMap(root, image=photo)
      >>> map.add_rectangle((15, 20, 38, 50), callback, 'A')
      >>> map.add_rectangle((50, 230, 80, 260), callback, 'B')
      >>> root.mainloop()

**Declarative:**

Not supported for this widget.

"""

from enum import Enum
from tkinter import *
from tkinter.ttk import *


class ImageMap(Canvas):
    """Image map
    
       Options (all have default values):
          - All :code:`Canvas` widget options (notably width and height)
    """

    def __init__(self, master=None, image=None, **kw):
        if not image:
            raise Exception('Valid image not provided')
        Canvas.__init__(self, master, **kw)
        self._image = self.create_image(0, 0, image=image, anchor=NW)
        self.bind("<Button-1>", self._on_click)
        self._rects = []
        self._ellipses = []

    def add_rectangle(self, coords, callback, argument=None):
        """Add a rectangular region to the image map.
        
          Arguments:
            - coords -- Coordinates in the form (xleft, ytop, xright, ybottom)
            - callback -- Callback function to handle clicks on this region
            - argument -- Argument to the callback (defaults to None)
        """
        self._rects.append((coords, callback, argument))
        return len(self._rects) - 1

    def add_ellipse(self, coords, callback, argument=None):
        """Add an elliptical region to the image map.
        
          Arguments:
            - coords -- Coordinates of the circumscribed rectangle, \
                        in the form (xleft, ytop, xright, ybottom)
            - callback -- Callback function to handle clicks on this region
            - argument -- Argument to the callback (defaults to None)
        """
        self._ellipses.append((coords, callback, argument))
        return len(self._ellipses) - 1

    def _on_click(self, event):
        for r in self._rects:
            c, callback, argument = r
            if c[0] < event.x < c[2] and \
                    c[1] < event.y < c[3]:
                callback(argument)
                return
        for e in self._ellipses:
            c, callback, argument = e
            xc = (c[0] + c[2]) / 2
            yc = (c[1] + c[3]) / 2
            rx2 = ((c[2] - c[0]) / 2) ** 2
            ry2 = ((c[3] - c[1]) / 2) ** 2
            x = float(event.x)
            y = float(event.y)
            if (x - xc) ** 2 / rx2 + (y - yc) ** 2 / ry2 <= 1.0:
                callback(argument)
                return


if __name__ == '__main__':
    BITMAP = """
    #define xlogo64_width 64
    #define xlogo64_height 64
    static unsigned char xlogo64_bits[] = {
       0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0xfe, 0xff, 0x01, 0x00,
       0x00, 0x00, 0x00, 0xf8, 0xfc, 0xff, 0x03, 0x00, 0x00, 0x00, 0x00, 0x7c,
       0xf8, 0xff, 0x07, 0x00, 0x00, 0x00, 0x00, 0x3e, 0xf8, 0xff, 0x07, 0x00,
       0x00, 0x00, 0x00, 0x1f, 0xf0, 0xff, 0x0f, 0x00, 0x00, 0x00, 0x80, 0x0f,
       0xe0, 0xff, 0x1f, 0x00, 0x00, 0x00, 0x80, 0x0f, 0xc0, 0xff, 0x3f, 0x00,
       0x00, 0x00, 0xc0, 0x07, 0xc0, 0xff, 0x3f, 0x00, 0x00, 0x00, 0xe0, 0x03,
       0x80, 0xff, 0x7f, 0x00, 0x00, 0x00, 0xf0, 0x01, 0x00, 0xff, 0xff, 0x00,
       0x00, 0x00, 0xf8, 0x00, 0x00, 0xfe, 0xff, 0x01, 0x00, 0x00, 0xf8, 0x00,
       0x00, 0xfe, 0xff, 0x01, 0x00, 0x00, 0x7c, 0x00, 0x00, 0xfc, 0xff, 0x03,
       0x00, 0x00, 0x3e, 0x00, 0x00, 0xf8, 0xff, 0x07, 0x00, 0x00, 0x1f, 0x00,
       0x00, 0xf0, 0xff, 0x0f, 0x00, 0x80, 0x0f, 0x00, 0x00, 0xf0, 0xff, 0x0f,
       0x00, 0xc0, 0x07, 0x00, 0x00, 0xe0, 0xff, 0x1f, 0x00, 0xc0, 0x07, 0x00,
       0x00, 0xc0, 0xff, 0x3f, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x80, 0xff, 0x7f,
       0x00, 0xf0, 0x01, 0x00, 0x00, 0x80, 0xff, 0x7f, 0x00, 0xf8, 0x00, 0x00,
       0x00, 0x00, 0xff, 0xff, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xff,
       0x01, 0x7c, 0x00, 0x00, 0x00, 0x00, 0xfc, 0xff, 0x03, 0x3e, 0x00, 0x00,
       0x00, 0x00, 0xfc, 0xff, 0x03, 0x1f, 0x00, 0x00, 0x00, 0x00, 0xf8, 0xff,
       0x87, 0x0f, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xff, 0xcf, 0x07, 0x00, 0x00,
       0x00, 0x00, 0xe0, 0xff, 0xcf, 0x07, 0x00, 0x00, 0x00, 0x00, 0xe0, 0xff,
       0xe7, 0x03, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xff, 0xf3, 0x01, 0x00, 0x00,
       0x00, 0x00, 0x80, 0xff, 0xf9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
       0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xfe, 0x00, 0x00, 0x00,
       0x00, 0x00, 0x00, 0x7e, 0xfe, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e,
       0xff, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9f, 0xff, 0x07, 0x00, 0x00,
       0x00, 0x00, 0x80, 0xcf, 0xff, 0x07, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xe7,
       0xff, 0x0f, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xe7, 0xff, 0x1f, 0x00, 0x00,
       0x00, 0x00, 0xe0, 0xc3, 0xff, 0x3f, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xc1,
       0xff, 0x3f, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x80, 0xff, 0x7f, 0x00, 0x00,
       0x00, 0x00, 0x7c, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x7c, 0x00,
       0xfe, 0xff, 0x01, 0x00, 0x00, 0x00, 0x3e, 0x00, 0xfe, 0xff, 0x01, 0x00,
       0x00, 0x00, 0x1f, 0x00, 0xfc, 0xff, 0x03, 0x00, 0x00, 0x80, 0x0f, 0x00,
       0xf8, 0xff, 0x07, 0x00, 0x00, 0xc0, 0x07, 0x00, 0xf0, 0xff, 0x0f, 0x00,
       0x00, 0xe0, 0x03, 0x00, 0xf0, 0xff, 0x0f, 0x00, 0x00, 0xe0, 0x03, 0x00,
       0xe0, 0xff, 0x1f, 0x00, 0x00, 0xf0, 0x01, 0x00, 0xc0, 0xff, 0x3f, 0x00,
       0x00, 0xf8, 0x00, 0x00, 0x80, 0xff, 0x7f, 0x00, 0x00, 0x7c, 0x00, 0x00,
       0x80, 0xff, 0x7f, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00,
       0x00, 0x3e, 0x00, 0x00, 0x00, 0xfe, 0xff, 0x01, 0x00, 0x1f, 0x00, 0x00,
       0x00, 0xfc, 0xff, 0x03, 0x80, 0x0f, 0x00, 0x00, 0x00, 0xfc, 0xff, 0x03,
       0xc0, 0x07, 0x00, 0x00, 0x00, 0xf8, 0xff, 0x07, 0xe0, 0x03, 0x00, 0x00,
       0x00, 0xf0, 0xff, 0x0f, 0xe0, 0x03, 0x00, 0x00, 0x00, 0xe0, 0xff, 0x1f,
       0xf0, 0x01, 0x00, 0x00, 0x00, 0xe0, 0xff, 0x1f, 0xf8, 0x00, 0x00, 0x00,
       0x00, 0xc0, 0xff, 0x3f, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x80, 0xff, 0x7f,
       0x3e, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff};
   """


    def callback(name):
        global label
        label['text'] = 'Region ' + name + ' was clicked.'


    root = Tk()
    bitmap = BitmapImage(data=BITMAP)
    frame = Frame(root)
    label = Label(frame, text='--')
    image_map = ImageMap(frame, bitmap, width=64, height=64)
    image_map.add_rectangle((32, 0, 64, 31), callback, '1st')
    image_map.add_rectangle((0, 0, 31, 31), callback, '2nd')
    image_map.add_rectangle((0, 33, 31, 64), callback, '3rd')
    image_map.add_ellipse((32, 32, 64, 64), callback, '4th')
    label.pack()
    image_map.pack()
    frame.pack()
    root.mainloop()

# =====================================================================
# Module DocTest
# =====================================================================

if __name__ == '__main__':
    import doctest
    doctest.testmod()