import argparse
import logging
import translators as ts
from typing import List
from translator.consts import LANGUAGE_EQUIVALENCES, LANGUAGE_CODENAMES, APP_NAME

from logutils import get_logger


logger = get_logger("translator_cli")


def get_language_code(lang: str, engine: str) -> str:
    """
    Provide the desired language as a 2-letter code and then a tranlation engine.
    If the language is supported by the engine by another identificator (for example
    "spa" or "Spanish" instead of the standard "es"), this function will return the
    equivalent codification for that engine.
    Else, if the engine uses the standard code or maybe it doesn't support the
    language, the 2-letter standard code provided will be returned unchanged.
    Thus, this function should not be used to check if a language is supported by an
    engine or not.
    """
    try:
        return LANGUAGE_EQUIVALENCES[lang][engine]
    except Exception:
        return lang


def language_pair_is_available(engine: str, input_lang: str, output_lang: str) -> bool:
    if engine in ("deepl", "gpt3", "gpt4"):
        return True
    input_lang = get_language_code(input_lang, engine)
    output_lang = get_language_code(output_lang, engine)
    d = ts.get_languages(engine)
    if input_lang in d.keys():
        if output_lang in d[input_lang]:
            logger.debug(
                f"Language pair {input_lang} -> {output_lang} available for {engine}"
            )
            return True
    logger.debug(
        f"Language pair {input_lang} -> {output_lang} not available for {engine}"
    )
    return False


def build_chatgpt_translation_prompt(
    text: str, input_lang: str, output_lang: str
) -> str:
    input_lang = LANGUAGE_CODENAMES[input_lang]
    output_lang = LANGUAGE_CODENAMES[output_lang]

    prompt_v1 = (
        f"Translate the following text I am going to provide in the line below from {input_lang} to {output_lang}. "
        "Limit your response to the translated text, without adding anything else, such as additional text or "
        "surrounding quotes."
        f"{text}"
    )

    return prompt_v1


def build_chatgpt_translation_prompt_from_list(
    texts: List[str], input_lang: str, output_lang: str
) -> str:
    input_lang = LANGUAGE_CODENAMES[input_lang]
    output_lang = LANGUAGE_CODENAMES[output_lang]

    prompt_v1 = (
        "I am going to provide you with a list of strings in Python format. I want you to translate them all from "
        f"{input_lang} to {output_lang}. They are all part of the same, unique text, and correctly sorted, so you should "
        "make the translation treating the strings as part of a single document, sharing a common context."
        "For the output, I want you to keep the original format and return a Python list of strings that should "
        "be similar to the one provided as input. Both should have the exact same number of elements: if the input list "
        "contains 8 strings. Output list should be in the regular Python list format: ['a', 'b', 'c'...]. Do not include "
        "any additional text or any other addition other than the mere list of translated strings. "
        "Below this line there are several examples of the kind of input I will be providing and the output I would expect. "
        "Examples are formed by 3 lines: the first one will contain a prompt detailing the pair of languages, like "
        "'Example 2: English to Spanish'. The second line will contain the input list. The third line will contain the output "
        "list. After the examples and two linebreaks, comes the real input you have to translate, with a first line like "
        f"'Input: {input_lang} to {output_lang}' and a second line with the list of strings to be translated."
        "\nExample 1: English to Catalan"
        '\n["The spider was weaving", "a dark, intricate web", "to catch a poor old fly."]'
        '\n["L\'aranya estava teixint", "una teranyina fosca i intrincada", "per atrapar a una pobra mosca."]'
        "\nExample 2: English to Spanish"
        '\n["The spider was weaving", "a dark, intricate web", "to catch a poor old fly."]'
        '\n["La araña estaba tejiendo", "una telaraña oscura e intrincada", "para atrapar a una pobre mosca."]'
        "\nExample 3: Spanish to English"
        "\n['Hola, me llamo', 'José Bonaparte, emperador de las', 'Dos Sicilias y las Islas del', 'Verano, alla donde el sol', 'nunca se pone.']"
        "\n['Hello, my name is', 'Joseph Bonaparte, emperor of', 'the Two Sicilies and the Summer', 'Isles, where the sun', 'never sets.']"
        f"\n\nInput {input_lang} to {output_lang}"
        f"\n{texts}"
    )
    logger.debug(prompt_v1)
    return prompt_v1


def build_system_directive_batch_translation_no_context(
    input_lang: str, output_lang: str
) -> str:
    logger.debug("Building system directive for GPT context-free batch translation")
    input_lang = LANGUAGE_CODENAMES[input_lang]
    output_lang = LANGUAGE_CODENAMES[output_lang]
    prompt_v1 = (
        "In the next several prompts I am going to provide you with several texts that I need you to translate. "
        "Please, make sure that the following requirements are fulfilled:"
        f"\n- Output will be the input text translated from {input_lang} to {output_lang}."
        "\n- Output should consist in just the translated text, without any further addition of text alien to the input prompt, quotes, etc."
        "\n- Do NOT take the previous prompts in account when translating the current prompt."
        "\nBelow you have several examples, each one formed by a first line stating the translation languages, "
        "like 'Example 1: English to Spanish', a second line with the input text to be translated, and a third "
        "line with the expected output you should be generating. Here are the examples:"
        "\nExample 1: Spanish to English"
        "\nLa torre era tan alta que Miguel se asustó sólo de mirar abajo."
        "\nThe tower was so high that Miguel became scared of just looking downwards."
        "\nExample 2: English to Spanish"
        "\nINFPs are creative and compassionate individuals with a deep commitment to their personal values. They are often idealistic, seeking harmony and looking for the deeper meaning in everything they encounter."
        "\nLos INFPs son individuos creativos y compasivos con un profundo compromiso con sus valores personales. Suelen ser idealistas, y buscan la armonía y un sentido profundo en todo lo que encuentran."
        "\nExample 3: Spanish to Catalan"
        "\nLa camiseta estaba sucia de aceite y sangre."
        "\nLa samarreta estava bruta d'oli i sang."
        "\n\n"
    )
    prompt_v2 = (
        "Please forget all prior prompts. "
        "\nAs a distinguished multilingual scholar and translator, you specialize in accurately and elegantly translating texts into any language, meticulously considering all linguistic complexities and nuances. You will assist you in translating texts into my desired target language, ensuring the translation is both accurate and appropriate. I will provide you with the text I want to translate and specify the target language, and you will promptly deliver the translation."
        f"\nOrigin language will be {input_lang} and target language will be {output_lang}."
        "\nPlease remember this prompt unless I ask you to forget it."
    )
    return prompt_v1


def build_chatgpt_system_directive(input_lang: str, output_lang: str) -> str:
    input_lang = LANGUAGE_CODENAMES[input_lang]
    output_lang = LANGUAGE_CODENAMES[output_lang]
    return (
        "I am going to provide you a series of texts, one in each prompt. I want you to translate each one of them "
        f"from the original language, {input_lang}, to the target language, {output_lang}. Your response should be "
        "just the translated text, without any further additions. No additional text. No surrounding quotes, unless "
        "they were there in the original. Mind that all of these will be little fragments split from a bigger, original "
        "text. Thus, you should use the history of all previous fragments as a context that can improve your understanding "
        "and interpretation of your current translation. Subsequently, I am going to provide you an example of interaction, "
        "were you would be translating from English to Catalan. My prompts are marked with a [Me] at the start, and your "
        "responses with [GPT] for the example, but you don't need to include that tags in your responses:\n"
        "[Me] The spider was weaving"
        "[GPT] L'aranya estava teixint"
        "[Me] a dark, intricate web"
        "[GPT] una teranyina fosca i intrincada"
        "[Me] to catch a poor old fly."
        "[GPT] per atrapar a una pobra mosca."
    )


def engine_list(s: str) -> List:
    """
    (GPT4 generated code)
    Parse and validate a string as a comma-separated list of words.
    """
    engines = s.split(",")
    if not all(engine.isalpha() for engine in engines) or len(engines) < 1:
        raise argparse.ArgumentTypeError(
            "Engines must be a comma-separated list of words"
        )
    return engines


def set_logger_verbosity(verbosity):
    global logger
    for h in logger.handlers:
        logger.removeHandler(h)
    if verbosity == 0:
        logger = get_logger(APP_NAME, level=logging.WARNING)
    elif verbosity == 1:
        logger = get_logger(APP_NAME, level=logging.INFO)
    else:
        logger = get_logger(APP_NAME, level=logging.DEBUG)


class TranslationError(Exception):
    pass
