"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const destination_1 = require("../destination");
const notifications_resource_handler_1 = require("./notifications-resource-handler");
/**
 * A custom CloudFormation resource that updates bucket notifications for a
 * bucket. The reason we need it is because the AWS::S3::Bucket notification
 * configuration is defined on the bucket itself, which makes it impossible to
 * provision notifications at the same time as the target (since
 * PutBucketNotifications validates the targets).
 *
 * Since only a single BucketNotifications resource is allowed for each Bucket,
 * this construct is not exported in the public API of this module. Instead, it
 * is created just-in-time by `s3.Bucket.onEvent`, so a 1:1 relationship is
 * ensured.
 *
 * @see
 * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-s3-bucket-notificationconfig.html
 */
class BucketNotifications extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaNotifications = new Array();
        this.queueNotifications = new Array();
        this.topicNotifications = new Array();
        this.bucket = props.bucket;
    }
    /**
     * Adds a notification subscription for this bucket.
     * If this is the first notification, a BucketNotification resource is added to the stack.
     *
     * @param event The type of event
     * @param target The target construct
     * @param filters A set of S3 key filters
     */
    addNotification(event, target, ...filters) {
        const resource = this.createResourceOnce();
        // resolve target. this also provides an opportunity for the target to e.g. update
        // policies to allow this notification to happen.
        const targetProps = target.bind(this, this.bucket);
        const commonConfig = {
            Events: [event],
            Filter: renderFilters(filters),
        };
        // if the target specifies any dependencies, add them to the custom resource.
        // for example, the SNS topic policy must be created /before/ the notification resource.
        // otherwise, S3 won't be able to confirm the subscription.
        if (targetProps.dependencies) {
            resource.node.addDependency(...targetProps.dependencies);
        }
        // based on the target type, add the the correct configurations array
        switch (targetProps.type) {
            case destination_1.BucketNotificationDestinationType.LAMBDA:
                this.lambdaNotifications.push({ ...commonConfig, LambdaFunctionArn: targetProps.arn });
                break;
            case destination_1.BucketNotificationDestinationType.QUEUE:
                this.queueNotifications.push({ ...commonConfig, QueueArn: targetProps.arn });
                break;
            case destination_1.BucketNotificationDestinationType.TOPIC:
                this.topicNotifications.push({ ...commonConfig, TopicArn: targetProps.arn });
                break;
            default:
                throw new Error('Unsupported notification target type:' + destination_1.BucketNotificationDestinationType[targetProps.type]);
        }
    }
    renderNotificationConfiguration() {
        return {
            LambdaFunctionConfigurations: this.lambdaNotifications.length > 0 ? this.lambdaNotifications : undefined,
            QueueConfigurations: this.queueNotifications.length > 0 ? this.queueNotifications : undefined,
            TopicConfigurations: this.topicNotifications.length > 0 ? this.topicNotifications : undefined
        };
    }
    /**
     * Defines the bucket notifications resources in the stack only once.
     * This is called lazily as we add notifications, so that if notifications are not added,
     * there is no notifications resource.
     */
    createResourceOnce() {
        if (!this.resource) {
            const handlerArn = notifications_resource_handler_1.NotificationsResourceHandler.singleton(this);
            this.resource = new cdk.CfnResource(this, 'Resource', {
                type: 'Custom::S3BucketNotifications',
                properties: {
                    ServiceToken: handlerArn,
                    BucketName: this.bucket.bucketName,
                    NotificationConfiguration: cdk.Lazy.anyValue({ produce: () => this.renderNotificationConfiguration() })
                }
            });
        }
        return this.resource;
    }
}
exports.BucketNotifications = BucketNotifications;
function renderFilters(filters) {
    if (!filters || filters.length === 0) {
        return undefined;
    }
    const renderedRules = new Array();
    for (const rule of filters) {
        if (!rule.suffix && !rule.prefix) {
            throw new Error('NotificationKeyFilter must specify `prefix` and/or `suffix`');
        }
        if (rule.suffix) {
            renderedRules.push({ Name: 'suffix', Value: rule.suffix });
        }
        if (rule.prefix) {
            renderedRules.push({ Name: 'prefix', Value: rule.prefix });
        }
    }
    return {
        Key: {
            FilterRules: renderedRules
        }
    };
}
//# sourceMappingURL=data:application/json;base64,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