"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alias = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const kms_generated_1 = require("./kms.generated");
const REQUIRED_ALIAS_PREFIX = 'alias/';
const DISALLOWED_PREFIX = REQUIRED_ALIAS_PREFIX + 'aws/';
class AliasBase extends core_1.Resource {
    get keyArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'kms',
            // aliasName already contains the '/'
            resource: this.aliasName,
        });
    }
    get keyId() {
        return this.aliasName;
    }
    addAlias(alias) {
        return this.aliasTargetKey.addAlias(alias);
    }
    addToResourcePolicy(statement, allowNoOp) {
        return this.aliasTargetKey.addToResourcePolicy(statement, allowNoOp);
    }
    grant(grantee, ...actions) {
        return this.aliasTargetKey.grant(grantee, ...actions);
    }
    grantDecrypt(grantee) {
        return this.aliasTargetKey.grantDecrypt(grantee);
    }
    grantEncrypt(grantee) {
        return this.aliasTargetKey.grantEncrypt(grantee);
    }
    grantEncryptDecrypt(grantee) {
        return this.aliasTargetKey.grantEncryptDecrypt(grantee);
    }
}
/**
 * Defines a display name for a customer master key (CMK) in AWS Key Management
 * Service (AWS KMS). Using an alias to refer to a key can help you simplify key
 * management. For example, when rotating keys, you can just update the alias
 * mapping instead of tracking and changing key IDs. For more information, see
 * Working with Aliases in the AWS Key Management Service Developer Guide.
 *
 * You can also add an alias for a key by calling `key.addAlias(alias)`.
 *
 * @resource AWS::KMS::Alias
 */
class Alias extends AliasBase {
    constructor(scope, id, props) {
        let aliasName = props.aliasName;
        if (!core_1.Token.isUnresolved(aliasName)) {
            if (!aliasName.startsWith(REQUIRED_ALIAS_PREFIX)) {
                aliasName = REQUIRED_ALIAS_PREFIX + aliasName;
            }
            if (aliasName === REQUIRED_ALIAS_PREFIX) {
                throw new Error(`Alias must include a value after "${REQUIRED_ALIAS_PREFIX}": ${aliasName}`);
            }
            if (aliasName.toLocaleLowerCase().startsWith(DISALLOWED_PREFIX)) {
                throw new Error(`Alias cannot start with ${DISALLOWED_PREFIX}: ${aliasName}`);
            }
            if (!aliasName.match(/^[a-zA-Z0-9:/_-]{1,256}$/)) {
                throw new Error('Alias name must be between 1 and 256 characters in a-zA-Z0-9:/_-');
            }
        }
        super(scope, id, {
            physicalName: aliasName,
        });
        this.aliasTargetKey = props.targetKey;
        const resource = new kms_generated_1.CfnAlias(this, 'Resource', {
            aliasName: this.physicalName,
            targetKeyId: this.aliasTargetKey.keyArn,
        });
        this.aliasName = this.getResourceNameAttribute(resource.aliasName);
        if (props.removalPolicy) {
            resource.applyRemovalPolicy(props.removalPolicy);
        }
    }
    /**
     * Import an existing KMS Alias defined outside the CDK app.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs the properties of the referenced KMS Alias
     */
    static fromAliasAttributes(scope, id, attrs) {
        class _Alias extends AliasBase {
            get aliasName() { return attrs.aliasName; }
            get aliasTargetKey() { return attrs.aliasTargetKey; }
        }
        return new _Alias(scope, id);
    }
    /**
     * Import an existing KMS Alias defined outside the CDK app, by the alias name. This method should be used
     * instead of 'fromAliasAttributes' when the underlying KMS Key ARN is not available.
     * This Alias will not have a direct reference to the KMS Key, so addAlias and grant* methods are not supported.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param aliasName The full name of the KMS Alias (e.g., 'alias/aws/s3', 'alias/myKeyAlias').
     */
    static fromAliasName(scope, id, aliasName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.keyArn = core_1.Stack.of(this).formatArn({ service: 'kms', resource: aliasName });
                this.keyId = aliasName;
                this.aliasName = aliasName;
            }
            get aliasTargetKey() { throw new Error('Cannot access aliasTargetKey on an Alias imnported by Alias.fromAliasName().'); }
            addAlias(_alias) { throw new Error('Cannot call addAlias on an Alias imported by Alias.fromAliasName().'); }
            addToResourcePolicy(_statement, _allowNoOp) {
                return { statementAdded: false };
            }
            grant(grantee, ..._actions) { return iam.Grant.drop(grantee, ''); }
            grantDecrypt(grantee) { return iam.Grant.drop(grantee, ''); }
            grantEncrypt(grantee) { return iam.Grant.drop(grantee, ''); }
            grantEncryptDecrypt(grantee) { return iam.Grant.drop(grantee, ''); }
        }
        return new Import(scope, id);
    }
    generatePhysicalName() {
        return REQUIRED_ALIAS_PREFIX + super.generatePhysicalName();
    }
}
exports.Alias = Alias;
//# sourceMappingURL=data:application/json;base64,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