import subprocess
import os
import sys
import platform


from cdh_dav_python.cdc_admin_service.environment_logging import LoggerSingleton

# Get the currently running file name
NAMESPACE_NAME = os.path.basename(os.path.dirname(__file__))
# Get the parent folder name of the running file
SERVICE_NAME = os.path.basename(__file__)


class SphinxClient:
    """
    A class representing a Sphinx client.

    Attributes:
        None

    Methods:
        build_html: Builds the HTML documentation using Sphinx.
    """

    @staticmethod
    def build_html(doc_folder_path: str, data_product_id: str, environment: str):
        """
        Builds the HTML documentation using Sphinx.

        Args:
            doc_folder_path (str): The path to the Sphinx source directory.
            data_product_id (str): The ID of the data product.
            environment (str): The environment in which the documentation is being built.

        Returns:
            subprocess.CompletedProcess: The result of the Sphinx build command.
        """

        tracer, logger = LoggerSingleton.instance(
            NAMESPACE_NAME, SERVICE_NAME, data_product_id, environment
        ).initialize_logging_and_tracing()

        with tracer.start_as_current_span("build_html"):
            try:
                current_dir = doc_folder_path

                # Path to your Sphinx source directory (two directories up)
                sphinx_source_dir = doc_folder_path
                logger.info(f"doc_folder_path: {doc_folder_path}")
                logger.info(f"sphinx_source_dir: {sphinx_source_dir}")
                logger.info(f"current_dir: {current_dir}")

                # Path to the directory to output the HTML files
                # (two directories up and down to 'build')
                build_dir = os.path.abspath(os.path.join(current_dir, "build", "html"))
                logger.info(f"build_dir: {build_dir}")

                # Command to build Sphinx documentation
                command = ["sphinx-build", "-b", "html", sphinx_source_dir, build_dir]
                logger.info(f"command: {command}")

                # Run the Sphinx build command
                result = subprocess.run(
                    command, check=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE
                )

                logger.info("Command output: %s", result.stdout.decode())
                return result

            except subprocess.CalledProcessError as e:
                # CalledProcessError is raised when the subprocess exits with a non-zero status
                logger.error(
                    "Command '%s' failed with return code: %s", e.cmd, e.returncode
                )
                logger.error("Error output: %s", e.stderr.decode())
            except FileNotFoundError as e:
                # FileNotFoundError is raised when the command is not found
                logger.error("Command not found: %s", e.filename)
                logger.error("Full error: %s", e)
                logger.error("Current PATH: %s", os.environ.get("PATH"))
            except Exception as e:
                # Generic exception handling for any other exceptions
                logger.error("An error occurred: %s", e)

    @staticmethod
    def build_pdf(doc_folder_path: str):
        """
        Builds the PDF documentation using Sphinx.

        Args:
            doc_folder_path (str): The path to the folder containing the Sphinx documentation.

        Returns:
            CompletedProcess: The result of the Sphinx build command.
        """
        current_dir = doc_folder_path

        # Path to your Sphinx source directory (two directories up)
        sphinx_source_dir = doc_folder_path
        print(sphinx_source_dir)

        # Path to the directory to output the PDF files (two directories up and down to 'build')
        build_dir = os.path.abspath(os.path.join(current_dir, "build", "latex"))
        print(build_dir)

        # Command to build Sphinx documentation
        command = ["sphinx-build", "-b", "latex", sphinx_source_dir, build_dir]

        # Run the Sphinx build command
        result = subprocess.run(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE)

        # Navigate to the LaTeX build directory
        os.chdir(build_dir)

        # Run the make command to generate the PDF
        # Check the operating system
        is_windows = platform.system() == "Windows"

        # Run the appropriate make command based on the operating system
        if is_windows:
            result = subprocess.run(
                ["make.bat"], stdout=subprocess.PIPE, stderr=subprocess.PIPE
            )
        else:
            result = subprocess.run(
                ["make"], stdout=subprocess.PIPE, stderr=subprocess.PIPE
            )

        return result
