import inspect
from collections.abc import Callable
from typing import TYPE_CHECKING, Optional

from neuroglia.core import ModuleLoader, TypeFinder
from neuroglia.data.abstractions import AggregateRoot
from neuroglia.data.infrastructure.event_sourcing.read_model_reconciliator import (
    ReadModelConciliationOptions,
    ReadModelReconciliator,
)
from neuroglia.data.queries.generic import GetByIdQueryHandler, ListQueryHandler
from neuroglia.hosting.abstractions import ApplicationBuilderBase, HostedService
from neuroglia.mediation.mediator import RequestHandler

if TYPE_CHECKING:
    from neuroglia.data.infrastructure.event_sourcing.event_sourcing_repository import (
        EventSourcingRepositoryOptions,
    )


class DataAccessLayer:
    class WriteModel:
        """Represents a helper class used to configure an application's Write Model DAL

        Supports two configuration patterns:
        1. Simplified: Pass options directly to constructor
        2. Custom: Pass custom repository_setup function to configure()

        Examples:
            # Simple configuration with default options
            DataAccessLayer.WriteModel().configure(builder, ["domain.entities"])

            # With custom delete mode
            from neuroglia.data.infrastructure.event_sourcing.abstractions import DeleteMode
            from neuroglia.data.infrastructure.event_sourcing.event_sourcing_repository import (
                EventSourcingRepositoryOptions
            )

            DataAccessLayer.WriteModel(
                options=EventSourcingRepositoryOptions(delete_mode=DeleteMode.HARD)
            ).configure(builder, ["domain.entities"])

            # Custom factory (advanced, backwards compatible)
            def custom_setup(builder_, entity_type, key_type):
                # Custom configuration logic
                pass
            DataAccessLayer.WriteModel().configure(
                builder, ["domain.entities"], custom_setup
            )
        """

        def __init__(self, options: Optional["EventSourcingRepositoryOptions"] = None):
            """Initialize WriteModel configuration

            Args:
                options: Optional repository options (e.g., delete_mode).
                        If not provided, default options will be used.
            """
            self._options = options

        def configure(self, builder: ApplicationBuilderBase, modules: list[str], repository_setup: Optional[Callable[[ApplicationBuilderBase, type, type], None]] = None) -> ApplicationBuilderBase:
            """Configures the application's Write Model DAL, scanning for aggregate root types within the specified modules

            Args:
                builder (ApplicationBuilderBase): the application builder to configure
                modules (List[str]): a list containing the names of the modules to scan for aggregate root types
                repository_setup (Optional[Callable[[ApplicationBuilderBase, Type, Type], None]]):
                    Optional custom function to setup repositories. If provided, takes precedence over options.
                    If not provided, uses simplified configuration with options (if any).

            Returns:
                ApplicationBuilderBase: The configured builder

            Raises:
                ImportError: If EventSourcingRepository cannot be imported
            """
            # If custom setup provided, use it (backwards compatible)
            if repository_setup is not None:
                for module in [ModuleLoader.load(module_name) for module_name in modules]:
                    for aggregate_type in TypeFinder.get_types(module, lambda cls: inspect.isclass(cls) and issubclass(cls, AggregateRoot) and not cls == AggregateRoot):
                        key_type = str  # todo: reflect from DTO base type
                        repository_setup(builder, aggregate_type, key_type)
                return builder

            # Otherwise use simplified configuration with options
            return self._configure_with_options(builder, modules)

        def _configure_with_options(self, builder: ApplicationBuilderBase, modules: list[str]) -> ApplicationBuilderBase:
            """Configure repositories using simplified options pattern

            Args:
                builder: The application builder
                modules: List of module names to scan for aggregates

            Returns:
                The configured builder
            """
            from neuroglia.data.infrastructure.abstractions import Repository
            from neuroglia.data.infrastructure.event_sourcing.abstractions import (
                Aggregator,
                EventStore,
            )
            from neuroglia.data.infrastructure.event_sourcing.event_sourcing_repository import (
                EventSourcingRepository,
                EventSourcingRepositoryOptions,
            )
            from neuroglia.dependency_injection import ServiceProvider
            from neuroglia.mediation import Mediator

            # Discover and configure each aggregate type
            for module in [ModuleLoader.load(module_name) for module_name in modules]:
                for aggregate_type in TypeFinder.get_types(
                    module,
                    lambda cls: inspect.isclass(cls) and issubclass(cls, AggregateRoot) and not cls == AggregateRoot,
                ):
                    key_type = str  # todo: reflect from DTO base type

                    # Create type-specific options if global options provided
                    typed_options = None
                    if self._options:
                        typed_options = EventSourcingRepositoryOptions[aggregate_type, key_type](  # type: ignore
                            delete_mode=self._options.delete_mode,
                            soft_delete_method_name=self._options.soft_delete_method_name,
                        )

                    # Create factory function with proper closure
                    def make_factory(et, kt, opts):
                        def repository_factory(sp: ServiceProvider):
                            return EventSourcingRepository[et, kt](  # type: ignore
                                eventstore=sp.get_required_service(EventStore),
                                aggregator=sp.get_required_service(Aggregator),
                                mediator=sp.get_service(Mediator),
                                options=opts,
                            )

                        return repository_factory

                    # Register repository with factory
                    builder.services.add_singleton(
                        Repository[aggregate_type, key_type],  # type: ignore
                        implementation_factory=make_factory(aggregate_type, key_type, typed_options),
                    )

            return builder

    class ReadModel:
        """Represents a helper class used to configure an application's Read Model DAL

        Supports three configuration patterns:
        1. Simplified Sync: Pass database_name with repository_type='mongo' (default)
        2. Simplified Async: Pass database_name with repository_type='motor' for FastAPI
        3. Custom: Pass custom repository_setup function to configure()

        Examples:
            # Simple synchronous configuration (default)
            DataAccessLayer.ReadModel(database_name="myapp").configure(
                builder, ["integration.models"]
            )

            # Async configuration with Motor for FastAPI
            DataAccessLayer.ReadModel(
                database_name="myapp",
                repository_type='motor'
            ).configure(builder, ["integration.models"])

            # With custom repository mappings
            DataAccessLayer.ReadModel(
                database_name="myapp",
                repository_type='motor',
                repository_mappings={
                    TaskDtoRepository: MotorTaskDtoRepository,
                }
            ).configure(builder, ["integration.models"])

            # Custom factory (advanced, backwards compatible)
            def custom_setup(builder_, entity_type, key_type):
                # Custom configuration logic
                pass
            DataAccessLayer.ReadModel().configure(
                builder, ["integration.models"], custom_setup
            )
        """

        def __init__(
            self,
            database_name: Optional[str] = None,
            repository_type: str = "mongo",
            repository_mappings: Optional[dict[type, type]] = None,
        ):
            """Initialize ReadModel configuration

            Args:
                database_name: Optional database name for MongoDB repositories.
                              If not provided, custom repository_setup must be used.
                repository_type: Type of repository to use ('mongo' or 'motor'). Defaults to 'mongo'.
                    - 'mongo': Use MongoRepository with PyMongo (synchronous driver)
                    - 'motor': Use MotorRepository with Motor (async driver for FastAPI)
                repository_mappings: Optional mapping of abstract repository interfaces
                                    to their concrete implementations. Allows single-line
                                    registration of custom domain repositories.
                    Example: {TaskDtoRepository: MotorTaskDtoRepository}

            Example:
                ```python
                # Simplified sync configuration
                DataAccessLayer.ReadModel(database_name="myapp").configure(...)

                # Async configuration with Motor
                DataAccessLayer.ReadModel(
                    database_name="myapp",
                    repository_type='motor'
                ).configure(...)

                # With custom repository implementations
                DataAccessLayer.ReadModel(
                    database_name="myapp",
                    repository_type='motor',
                    repository_mappings={
                        TaskDtoRepository: MotorTaskDtoRepository,
                        UserDtoRepository: MotorUserDtoRepository,
                    }
                ).configure(...)
                ```
            """
            self._database_name = database_name
            self._repository_type = repository_type
            self._repository_mappings = repository_mappings or {}

            # Validate repository_type
            if repository_type not in ("mongo", "motor"):
                raise ValueError(f"Invalid repository_type '{repository_type}'. " "Must be either 'mongo' (synchronous PyMongo) or 'motor' (async Motor)")

        def configure(
            self,
            builder: ApplicationBuilderBase,
            modules: list[str],
            repository_setup: Optional[Callable[[ApplicationBuilderBase, type, type], None]] = None,
        ) -> ApplicationBuilderBase:
            """Configures the application's Read Model DAL, scanning for types marked with the 'queryable' decorator within the specified modules

            Args:
                builder (ApplicationBuilderBase): the application builder to configure
                modules (List[str]): a list containing the names of the modules to scan for types decorated with 'queryable'
                repository_setup (Optional[Callable[[ApplicationBuilderBase, Type, Type], None]]):
                    Optional custom function to setup repositories. If provided, takes precedence over database_name.
                    If not provided, uses simplified configuration with database_name.

            Returns:
                ApplicationBuilderBase: The configured builder

            Raises:
                ValueError: If consumer_group not specified in settings
                ValueError: If neither repository_setup nor database_name is provided
            """
            consumer_group = builder.settings.consumer_group
            if not consumer_group:
                raise ValueError("Cannot configure Read Model DAL: consumer group not specified in application settings")
            builder.services.add_singleton(ReadModelConciliationOptions, singleton=ReadModelConciliationOptions(consumer_group))
            builder.services.add_singleton(HostedService, ReadModelReconciliator)

            # If custom setup provided, use it (backwards compatible)
            if repository_setup is not None:
                for module in [ModuleLoader.load(module_name) for module_name in modules]:
                    for queryable_type in TypeFinder.get_types(module, lambda cls: inspect.isclass(cls) and hasattr(cls, "__queryable__")):
                        key_type = str  # todo: reflect from DTO base type
                        repository_setup(builder, queryable_type, key_type)
                        builder.services.add_transient(RequestHandler, GetByIdQueryHandler[queryable_type, key_type])  # type: ignore
                        builder.services.add_transient(RequestHandler, ListQueryHandler[queryable_type, key_type])  # type: ignore
                return builder

            # Otherwise use simplified configuration with database_name
            return self._configure_with_database_name(builder, modules)

        def _configure_with_database_name(
            self,
            builder: ApplicationBuilderBase,
            modules: list[str],
        ) -> ApplicationBuilderBase:
            """Configure repositories using simplified database_name pattern

            Args:
                builder: The application builder
                modules: List of module names to scan for queryable types

            Returns:
                The configured builder

            Raises:
                ValueError: If database_name was not provided
            """
            if not self._database_name:
                raise ValueError("Cannot configure Read Model with simplified API: " "database_name not provided. Either pass database_name to ReadModel() " "or use custom repository_setup function.")

            from neuroglia.data.infrastructure.abstractions import (
                QueryableRepository,
                Repository,
            )
            from neuroglia.dependency_injection import ServiceProvider

            # Configure based on repository type
            if self._repository_type == "mongo":
                # Synchronous MongoRepository configuration
                from pymongo import MongoClient

                from neuroglia.data.infrastructure.mongo.mongo_repository import (
                    MongoRepository,
                    MongoRepositoryOptions,
                )

                # Get MongoDB connection string
                connection_string_name = "mongo"
                connection_string = builder.settings.connection_strings.get(connection_string_name, None)
                if connection_string is None:
                    raise ValueError(f"Missing '{connection_string_name}' connection string in application settings")

                # Register MongoClient singleton (shared across all repositories)
                builder.services.try_add_singleton(MongoClient, singleton=MongoClient(connection_string))

                # Discover and configure each queryable type
                for module in [ModuleLoader.load(module_name) for module_name in modules]:
                    for queryable_type in TypeFinder.get_types(module, lambda cls: inspect.isclass(cls) and hasattr(cls, "__queryable__")):
                        key_type = str  # todo: reflect from DTO base type

                        # Register options for this entity type
                        builder.services.try_add_singleton(
                            MongoRepositoryOptions[queryable_type, key_type],  # type: ignore
                            singleton=MongoRepositoryOptions[queryable_type, key_type](self._database_name),  # type: ignore
                        )

                        # Register repository
                        builder.services.try_add_singleton(
                            Repository[queryable_type, key_type],  # type: ignore
                            MongoRepository[queryable_type, key_type],  # type: ignore
                        )

                        # Register queryable repository alias
                        def make_queryable_factory(qt, kt):
                            def queryable_factory(provider: ServiceProvider):
                                return provider.get_required_service(Repository[qt, kt])  # type: ignore

                            return queryable_factory

                        builder.services.try_add_singleton(
                            QueryableRepository[queryable_type, key_type],  # type: ignore
                            implementation_factory=make_queryable_factory(queryable_type, key_type),
                        )

                        # Register query handlers
                        builder.services.add_transient(RequestHandler, GetByIdQueryHandler[queryable_type, key_type])  # type: ignore
                        builder.services.add_transient(RequestHandler, ListQueryHandler[queryable_type, key_type])  # type: ignore

            elif self._repository_type == "motor":
                # Async MotorRepository configuration with queryable support
                from motor.motor_asyncio import AsyncIOMotorClient

                from neuroglia.data.infrastructure.mongo.motor_repository import (
                    MotorRepository,
                )
                from neuroglia.mediation import Mediator
                from neuroglia.serialization.json import JsonSerializer

                # Get MongoDB connection string
                connection_string_name = "mongo"
                connection_string = builder.settings.connection_strings.get(connection_string_name, None)
                if connection_string is None:
                    raise ValueError(f"Missing '{connection_string_name}' connection string in application settings")

                # Register AsyncIOMotorClient singleton (shared across all repositories)
                builder.services.try_add_singleton(AsyncIOMotorClient, singleton=AsyncIOMotorClient(connection_string))

                # Discover and configure each queryable type (filter by @queryable decorator)
                for module in [ModuleLoader.load(module_name) for module_name in modules]:
                    for entity_type in TypeFinder.get_types(module, lambda cls: inspect.isclass(cls) and hasattr(cls, "__queryable__")):
                        key_type = str  # todo: reflect from DTO base type

                        # Determine collection name (default to lowercase entity name)
                        collection_name = entity_type.__name__.lower()
                        if collection_name.endswith("dto"):
                            collection_name = collection_name[:-3]

                        # Factory function to create MotorRepository instance
                        def make_motor_factory(et, kt, cn):
                            def motor_factory(provider: ServiceProvider):
                                # Attempt to resolve Mediator optionally (tests may skip registration)
                                mediator = provider.get_service(Mediator)
                                if mediator is None:
                                    mediator = provider.get_required_service(Mediator)

                                return MotorRepository[et, kt](  # type: ignore
                                    client=provider.get_required_service(AsyncIOMotorClient),
                                    database_name=self._database_name,
                                    collection_name=cn,
                                    serializer=provider.get_required_service(JsonSerializer),
                                    entity_type=et,
                                    mediator=mediator,
                                )

                            return motor_factory

                        # Register MotorRepository (scoped for proper async context per request)
                        builder.services.try_add_scoped(
                            MotorRepository[entity_type, key_type],  # type: ignore
                            implementation_factory=make_motor_factory(entity_type, key_type, collection_name),
                        )

                        # Register Repository interface (handlers expect this)
                        def make_repository_factory(et, kt):
                            def repository_factory(provider: ServiceProvider):
                                return provider.get_required_service(MotorRepository[et, kt])  # type: ignore

                            return repository_factory

                        builder.services.try_add_scoped(
                            Repository[entity_type, key_type],  # type: ignore
                            implementation_factory=make_repository_factory(entity_type, key_type),
                        )

                        # Register QueryableRepository interface (for queryable support)
                        def make_queryable_factory(et, kt):
                            def queryable_factory(provider: ServiceProvider):
                                return provider.get_required_service(Repository[et, kt])  # type: ignore

                            return queryable_factory

                        builder.services.try_add_scoped(
                            QueryableRepository[entity_type, key_type],  # type: ignore
                            implementation_factory=make_queryable_factory(entity_type, key_type),
                        )

                        # Register query handlers (consistent with mongo)
                        builder.services.add_transient(RequestHandler, GetByIdQueryHandler[entity_type, key_type])  # type: ignore
                        builder.services.add_transient(RequestHandler, ListQueryHandler[entity_type, key_type])  # type: ignore

                # Register custom repository mappings
                for abstract_type, implementation_type in self._repository_mappings.items():
                    builder.services.add_scoped(abstract_type, implementation_type)

            return builder
