"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodejsFunction = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const bundling_1 = require("./bundling");
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * A Node.js Lambda function bundled using Parcel
 */
class NodejsFunction extends lambda.Function {
    constructor(scope, id, props = {}) {
        var _a, _b, _c;
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        // Entry and defaults
        const entry = path.resolve(findEntry(id, props.entry));
        const handler = (_a = props.handler) !== null && _a !== void 0 ? _a : 'handler';
        const defaultRunTime = util_1.nodeMajorVersion() >= 12
            ? lambda.Runtime.NODEJS_12_X
            : lambda.Runtime.NODEJS_10_X;
        const runtime = (_b = props.runtime) !== null && _b !== void 0 ? _b : defaultRunTime;
        // Look for the closest package.json starting in the directory of the entry
        // file. We need to restore it after bundling.
        const packageJsonManager = new package_json_manager_1.PackageJsonManager(path.dirname(entry));
        try {
            super(scope, id, {
                ...props,
                runtime,
                code: bundling_1.Bundling.parcel({
                    ...props,
                    entry,
                    runtime,
                }),
                handler: `index.${handler}`,
            });
            // Enable connection reuse for aws-sdk
            if ((_c = props.awsSdkConnectionReuse) !== null && _c !== void 0 ? _c : true) {
                this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1');
            }
        }
        finally {
            // We can only restore after the code has been bound to the function
            packageJsonManager.restore();
        }
    }
}
exports.NodejsFunction = NodejsFunction;
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(jsx?|tsx?)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    throw new Error('Cannot find entry file.');
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    const stackTrace = util_1.parseStackTrace();
    const functionIndex = stackTrace.findIndex(s => /NodejsFunction/.test(s.methodName || ''));
    if (functionIndex === -1 || !stackTrace[functionIndex + 1]) {
        throw new Error('Cannot find defining file.');
    }
    return stackTrace[functionIndex + 1].file;
}
//# sourceMappingURL=data:application/json;base64,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