# Imports

from tabulate import tabulate
from .constants import TABLE_FORMAT

__version__ = "0.7.1-d"

# Exports

__all__ = (
    "TABLE_FORMAT",
    "Row",
    "Table",
)

# Classes


class Row(object):
    """A row in tabular (command line) output."""

    def __init__(self, number=None, values=None):
        """Initialize a row.

        :param number: The row number.
        :type number: int

        :param values: The values included in the row.
        :type values: list

        """
        self.number = number
        self._values = values or list()

    def __iter__(self):
        return iter(self._values)


class Table(object):
    """A table for tabular (command line) output."""

    def __init__(self, headings=None, output_format=TABLE_FORMAT.SIMPLE):
        """Initialize a table.

        :param headings: A list of headings.
        :type headings: list[str]

        :param output_format: The output format of  the table.
        :type output_format: str

        .. note::
            Output is generated by python-tabulate. See https://bitbucket.org/astanin/python-tabulate/

        """
        self.format = output_format
        self.headings = headings
        self._rows = list()

    def __iter__(self):
        return iter(self._rows)

    def __str__(self):
        return self.to_string()

    def add(self, values):
        """Add a row to the table.

        :param values: The values of the row.
        :type values: list

        """
        row = Row(len(self._rows) + 1, values=values)
        self._rows.append(row)

    def to_string(self):
        """Get the table as string output.

        :rtype: str

        """
        return tabulate(self._rows, headers=self.headings, tablefmt=self.format)
