#ifndef __AIDGE_EXPORT_CPP_KERNELS_ACTIVATION__
#define __AIDGE_EXPORT_CPP_KERNELS_ACTIVATION__

#include <type_traits>
#include "network/typedefs.hpp"
#include "network/utils.hpp"
#include "kernels/rescaling.hpp"

template<typename Output_T, typename T,
         typename std::enable_if<std::is_floating_point<T>::value>::type* = nullptr>
__attribute__((always_inline)) inline
Output_T saturate (T value, int32_t /*sat*/)
{
    return value;
}

template<typename Output_T, typename T,
         typename std::enable_if<!std::is_floating_point<T>::value>::type* = nullptr>
__attribute__((always_inline)) inline
Output_T saturate (T value, uint32_t sat)
{
    if (std::is_unsigned<Output_T>::value) {
        return clamp(value, T(0), (T(1) << sat) - 1);
    } else {
        return clamp(value, -(T(1) << (sat - 1)), (T(1) << (sat - 1)) - 1);
    }
}

template<typename Output_T,
         typename Sum_T,
         typename Rescaling_T>
__attribute__((always_inline)) inline
Output_T activation_forward_value (Sum_T weightedSum,
                                   int output,
                                   ActivationFunction_T func,
                                   const Rescaling_T& __restrict rescaling)
{
    switch(func) {
        case Linear:
        case Saturation: {
            break;
        }
        case Rectifier: {
            if(weightedSum <= 0) weightedSum = 0;
            break;
        }
        default:
            // Unsupported activation function
            break;
    }

    // Value fixed here for now but it should be generated by
    // the export module or determined by the type of Output_T
    // For now only works for int8_t and uint8_t
    const uint32_t NB_BITS = 8;
    return saturate<Output_T>(rescaling(weightedSum, output), NB_BITS);
}


template<int NB_DATA,
         ActivationFunction_T ACTIVATION,
         typename Input_T, typename Output_T, typename Rescaling_T>
__attribute__((always_inline)) inline
void activation_forward (
    const Input_T* __restrict inputs,
    Output_T* __restrict outputs,
    const Rescaling_T& __restrict rescaling)
{
    for (int i = 0; i < NB_DATA; ++i)
    {
        outputs[i] = activation_forward_value<Output_T>(inputs[i], i, ACTIVATION, rescaling);
    }

}


#endif  // __AIDGE_EXPORT_CPP_KERNELS_ACTIVATION__
